(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 11.1' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       158,          7]
NotebookDataLength[    604003,      13521]
NotebookOptionsPosition[    499581,      11424]
NotebookOutlinePosition[    594233,      13357]
CellTagsIndexPosition[    593667,      13333]
WindowTitle->Sound: Elementary Introduction to the Wolfram Language
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell[BoxData[
 InterpretationBox[Cell[
  "Sound", "SectionInline",ExpressionUUID->
   "f911806e-0700-4223-9526-5c541c46b00f"],
  $Line = 0; Null]], "SectionDoubleDigit",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CounterAssignments->{{"Section", 11}, {"Subsection", 0}, {
   "Subsubsection", 0}, {"Exercise", 0}, {"Question", 0}},
 CellTags->{
  "p:53", "i:1"},ExpressionUUID->"c145940b-2ec1-4b3e-a40c-983b843a8511"],

Cell[TextData[{
 "In the Wolfram Language, sound works a lot like graphics, except that \
instead of having things like circles, one has sound notes. Press the play ",
 Cell[BoxData[
  GraphicsBox[
   TagBox[RasterBox[CompressedData["
1:eJxTTMoPSmNiYGAo5gASQYnljkVFiZXBAkBOaF5xZnpeaopnXklqemqRRRIz
UFANikHs/yMGbKELGAw2PqMZGLVx1MbhbeODBw/obGNHR8e2bdvoaWNDQ4Ov
r+/s2bOfPn1KTxuBoK6u7v79+/S0EQgyMjIuXbpETxuBICIiYt++ffS0EQj8
/f2XLFlCTxshoL29/dGjR/S0EQjy8/OvX79OTxuBIDY29sqVK3Sz0c/Pb+HC
hXTzY3h4+J49e/DrpaKNaWlpFy5cwK+RijZWV1ffu3ePoHXUsnH69OnwApZg
SUuhjcHBwZs3bybGa1SxMSEh4dSpUyRZR4mNJSUlt27dItU6sm3cuXPn48eP
ybCObBspAaM2ogEyWjXE20hrMLA2Dm8AACwbTbw=
     "], {{0, 38}, {38, 0}}, {0, 255},
     ColorFunction->RGBColor],
    BoxForm`ImageTag["Byte", ColorSpace -> "RGB", Interleaving -> True],
    Selectable->False],
   BaseStyle->"ImageGraphics",
   ImageSize->{16.440755208333655`, Automatic},
   ImageSizeRaw->{38, 38},
   PlotRange->{{0, 38}, {0, 38}}]], "InlineCodeText",ExpressionUUID->
  "1ea1547f-c20e-4d01-89f5-f44d6d5f41ce"],
 " button to actually play sounds. If you don\[CloseCurlyQuote]t say \
otherwise, the Wolfram Language will make the notes sound as if they were \
played on a piano. "
}], "Text",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:2",ExpressionUUID->"e672c068-a9d9-45c2-a534-2864267d6cda"],

Cell["Generate a middle C note:", "CodeCaption",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:3",ExpressionUUID->"f048db40-dfb3-45cf-aa18-4f087bdc7289"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"SoundNote", "[", "\"\<C\>\"", "]"}], "]"}]], "Input",
 Deletable->False,
 CellLabel->"In[1]:=",ExpressionUUID->"2c57b7ac-fcd7-42f3-acee-75140e007d5b"],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
     RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[RectangleBox[{0., 0.43198696602798653}, {0.95, 0.4438917279327484}],
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None,
       PlotRange->{All, All},
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
     InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, InsetBox[
     StyleBox[
      TemplateBox[{
        StyleBox[
        "\" | \"", FontSize -> 14, FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False], 
        StyleBox[
         TemplateBox[{"\"1\"", "\" s\""}, "RowDefault"], FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False]},
       "RowWithSeparator"], "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{5, -158}, {29, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       PolygonBox[
        NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
          11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {11, -157}}]], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
           Sound`MIDISequence[{
             Sound`MIDITrack[{
               Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
               Sound`MIDIEvent[
               0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
               Sound`MIDIEvent[
               0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 127], 
               
               Sound`MIDIEvent[
               48, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
         Sound`EmitSampledSound[None, "Preemptive"], 
         FEPrivate`EmitSound[None, "Preemptive"]], Method -> "Preemptive", 
       PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{29, -158}, {53, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       RectangleBox[{35, -152}, {47, -141}], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitSampledSound[
           SampledSoundList[{{0.}}, 8000], "Preemptive"]; Sound`StopMIDI[], 
         FEPrivate`EmitSound[
          SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
       "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->{159.5859375, Automatic},
   PlotRange->{{0, 360}, {-164, 0}},
   PlotRangePadding->1],
  Sound[
   SoundNote["C"]]]], "Output",
 Editable->False,
 CellEditDuplicate->False,
 Evaluatable->False,
 CellLabel->"Out[1]=",ExpressionUUID->"b8541725-c8f5-4f87-bbf1-5fc07afa3502"]
}, Open  ]],

Cell["\<\
You can specify a sequence of notes by giving them in a list.\
\>", "Text",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"f2f97578-e0cb-45b2-83c0-de9954b5d791"],

Cell["Play three notes in sequence:", "CodeCaption",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"93bc168a-9a4e-41e7-be6d-7a19090313ea"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"SoundNote", "[", "\"\<C\>\"", "]"}], ",", 
    RowBox[{"SoundNote", "[", "\"\<C\>\"", "]"}], ",", 
    RowBox[{"SoundNote", "[", "\"\<G\>\"", "]"}]}], "}"}], "]"}]], "Input",
 Deletable->False,
 CellLabel->"In[2]:=",ExpressionUUID->"6fbc38bc-ebd9-4e04-bbcd-9118d2c99edf"],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
     RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.43198696602798653}, {0.95, 0.4438917279327484}],
           RectangleBox[{1., 0.43198696602798653}, \
{1.95, 0.4438917279327484}], 
          RectangleBox[{2., 0.5153202993613198}, {2.95, 0.5272250612660818}]},
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None,
       PlotRange->{All, All},
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
     InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, InsetBox[
     StyleBox[
      TemplateBox[{
        StyleBox[
        "\" | \"", FontSize -> 14, FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False], 
        StyleBox[
         TemplateBox[{"\"3\"", "\" s\""}, "RowDefault"], FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False]},
       "RowWithSeparator"], "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{5, -158}, {29, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       PolygonBox[
        NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
          11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {11, -157}}]], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
           Sound`MIDISequence[{
             Sound`MIDITrack[{
               Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
               Sound`MIDIEvent[
               0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
               Sound`MIDIEvent[
               0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 127], 
               
               Sound`MIDIEvent[
               48, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               48, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               96, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               96, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               144, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
         Sound`EmitSampledSound[None, "Preemptive"], 
         FEPrivate`EmitSound[None, "Preemptive"]], Method -> "Preemptive", 
       PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{29, -158}, {53, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       RectangleBox[{35, -152}, {47, -141}], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitSampledSound[
           SampledSoundList[{{0.}}, 8000], "Preemptive"]; Sound`StopMIDI[], 
         FEPrivate`EmitSound[
          SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
       "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->{159.51171875, Automatic},
   PlotRange->{{0, 360}, {-164, 0}},
   PlotRangePadding->1],
  Sound[{
    SoundNote["C"], 
    SoundNote["C"], 
    SoundNote["G"]}]]], "Output",
 Editable->False,
 CellEditDuplicate->False,
 Evaluatable->False,
 CellLabel->"Out[2]=",ExpressionUUID->"9d309fcd-bc83-44eb-a7a4-a908296f0aa9"]
}, Open  ]],

Cell["\<\
Instead of giving names of notes, you can give a number to specify their \
pitch. Middle C is 0. Each semitone above middle C goes up by 1. Middle G is \
7 semitones above middle C, so it\[CloseCurlyQuote]s specified by the number \
7. (An octave is 12 semitones.)\
\>", "Text",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:4",ExpressionUUID->"10ddda16-2e39-4973-abe0-5304ecbd4e73"],

Cell["Specify the notes by numbers:", "CodeCaption",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"8523b151-2a30-4218-b3cc-e0a3ba406b37"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"SoundNote", "[", "0", "]"}], ",", 
    RowBox[{"SoundNote", "[", "0", "]"}], ",", 
    RowBox[{"SoundNote", "[", "7", "]"}]}], "}"}], "]"}]], "Input",
 Deletable->False,
 CellLabel->"In[3]:=",ExpressionUUID->"63e46507-3c3f-4b88-a448-026fa8344f5a"],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
     RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.43198696602798653}, {0.95, 0.4438917279327484}],
           RectangleBox[{1., 0.43198696602798653}, \
{1.95, 0.4438917279327484}], 
          RectangleBox[{2., 0.5153202993613198}, {2.95, 0.5272250612660818}]},
         
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None,
       PlotRange->{All, All},
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
     InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, InsetBox[
     StyleBox[
      TemplateBox[{
        StyleBox[
        "\" | \"", FontSize -> 14, FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False], 
        StyleBox[
         TemplateBox[{"\"3\"", "\" s\""}, "RowDefault"], FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False]},
       "RowWithSeparator"], "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{5, -158}, {29, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       PolygonBox[
        NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
          11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {11, -157}}]], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
           Sound`MIDISequence[{
             Sound`MIDITrack[{
               Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
               Sound`MIDIEvent[
               0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
               Sound`MIDIEvent[
               0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 127], 
               
               Sound`MIDIEvent[
               48, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               48, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               96, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               96, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               144, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
         Sound`EmitSampledSound[None, "Preemptive"], 
         FEPrivate`EmitSound[None, "Preemptive"]], Method -> "Preemptive", 
       PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{29, -158}, {53, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       RectangleBox[{35, -152}, {47, -141}], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitSampledSound[
           SampledSoundList[{{0.}}, 8000], "Preemptive"]; Sound`StopMIDI[], 
         FEPrivate`EmitSound[
          SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
       "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->{159.51171875, Automatic},
   PlotRange->{{0, 360}, {-164, 0}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[0], 
    SoundNote[0], 
    SoundNote[7]}]]], "Output",
 Editable->False,
 CellEditDuplicate->False,
 Evaluatable->False,
 CellLabel->"Out[3]=",ExpressionUUID->"cf5ec508-55d1-4ee9-b282-aafe7de13bc8"]
}, Open  ]],

Cell[TextData[{
 "Use ",
 ButtonBox["Table",
  BaseStyle->"Link"],
 " to generate a sequence of 5 notes:"
}], "CodeCaption",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"066bfb18-a215-4fef-9f4b-9adbe65949d9"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"SoundNote", "[", "n", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "5"}], "}"}]}], "]"}], "]"}]], "Input",
 Deletable->False,
 CellLabel->"In[4]:=",ExpressionUUID->"a92c2711-9be1-4ddf-8203-f621c01f12f8"],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
     RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.4438917279327484}, {0.95, 0.45579648983751037}],
           RectangleBox[{1., 0.45579648983751037}, \
{1.95, 0.4677012517422721}], 
          RectangleBox[{2., 0.4677012517422721}, {2.95, 0.4796060136470343}], 
          RectangleBox[{3., 0.4796060136470343}, {3.95, 0.49151077555179606}],
           RectangleBox[{4., 0.49151077555179606}, \
{4.95, 0.5034155374565581}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None,
       PlotRange->{All, All},
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
     InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, InsetBox[
     StyleBox[
      TemplateBox[{
        StyleBox[
        "\" | \"", FontSize -> 14, FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False], 
        StyleBox[
         TemplateBox[{"\"5\"", "\" s\""}, "RowDefault"], FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False]},
       "RowWithSeparator"], "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{5, -158}, {29, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       PolygonBox[
        NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
          11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {11, -157}}]], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
           Sound`MIDISequence[{
             Sound`MIDITrack[{
               Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
               Sound`MIDIEvent[
               0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
               Sound`MIDIEvent[
               0, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 127], 
               
               Sound`MIDIEvent[
               48, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               48, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               96, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               96, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               144, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               144, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               192, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               192, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               240, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
         Sound`EmitSampledSound[None, "Preemptive"], 
         FEPrivate`EmitSound[None, "Preemptive"]], Method -> "Preemptive", 
       PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{29, -158}, {53, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       RectangleBox[{35, -152}, {47, -141}], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitSampledSound[
           SampledSoundList[{{0.}}, 8000], "Preemptive"]; Sound`StopMIDI[], 
         FEPrivate`EmitSound[
          SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
       "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->{160.25, Automatic},
   PlotRange->{{0, 360}, {-164, 0}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[1], 
    SoundNote[2], 
    SoundNote[3], 
    SoundNote[4], 
    SoundNote[5]}]]], "Output",
 Editable->False,
 CellEditDuplicate->False,
 Evaluatable->False,
 CellTags->"LastCellOfAPage",
 CellLabel->"Out[4]=",ExpressionUUID->"ba7c44f2-f2e6-4098-b4f1-5eb13fb5cbfb"]
}, Open  ]],

Cell[TextData[{
 "If you don\[CloseCurlyQuote]t say otherwise, each note lasts 1 second. Use ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["SoundNote",
    BaseStyle->"CodeLink"], "[", 
   RowBox[{
    StyleBox["pitch", "CodeVariable"], ",", 
    StyleBox["length", "CodeVariable"]}], "]"}]], "InlineCodeText",
  ExpressionUUID->"c505097e-5b45-459a-aecb-0a3ebc3545d9"],
 " to get a different length. "
}], "Text",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->{
  "p:54", "i:5"},ExpressionUUID->"9e11890a-e096-4a6c-81ab-7f348d72ed00"],

Cell["Play each note for 0.1 seconds:", "CodeCaption",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"4dff7c30-a770-4933-b75e-459f8b834266"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{"n", ",", "0.1"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "5"}], "}"}]}], "]"}], "]"}]], "Input",
 Deletable->False,
 CellLabel->"In[5]:=",ExpressionUUID->"43f60c0a-2d47-4572-9b96-0cec892c08dd"],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
     RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {RGBColor[0.5, 0, 0.5], 
        StyleBox[{
          RectangleBox[{0., 0.4438917279327484}, \
{0.095, 0.45579648983751037}], 
          RectangleBox[{0.1, 0.45579648983751037}, \
{0.195, 0.4677012517422721}], 
          RectangleBox[{0.2, 0.4677012517422721}, \
{0.295, 0.4796060136470343}], 
          RectangleBox[{0.3, 0.4796060136470343}, \
{0.395, 0.49151077555179606}], 
          RectangleBox[{0.4, 0.49151077555179606}, \
{0.495, 0.5034155374565581}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None,
       PlotRange->{All, All},
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
     InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, InsetBox[
     StyleBox[
      TemplateBox[{
        StyleBox[
        "\" | \"", FontSize -> 14, FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False], 
        StyleBox[
         TemplateBox[{"\"0.5\"", "\" s\""}, "RowDefault"], FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False]},
       "RowWithSeparator"], "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{5, -158}, {29, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       PolygonBox[
        NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
          11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {11, -157}}]], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
           Sound`MIDISequence[{
             Sound`MIDITrack[{
               Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
               Sound`MIDIEvent[
               0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
               Sound`MIDIEvent[
               0, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 127], 
               
               Sound`MIDIEvent[
               24, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               24, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               48, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               48, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               72, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               72, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               96, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               96, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               120, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
         Sound`EmitSampledSound[None, "Preemptive"], 
         FEPrivate`EmitSound[None, "Preemptive"]], Method -> "Preemptive", 
       PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{29, -158}, {53, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       RectangleBox[{35, -152}, {47, -141}], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitSampledSound[
           SampledSoundList[{{0.}}, 8000], "Preemptive"]; Sound`StopMIDI[], 
         FEPrivate`EmitSound[
          SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
       "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->{160.25, Automatic},
   PlotRange->{{0, 360}, {-164, 0}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[1, 0.1], 
    SoundNote[2, 0.1], 
    SoundNote[3, 0.1], 
    SoundNote[4, 0.1], 
    SoundNote[5, 0.1]}]]], "Output",
 Editable->False,
 CellEditDuplicate->False,
 Evaluatable->False,
 CellLabel->"Out[5]=",ExpressionUUID->"7d724a34-5906-45f8-a44f-e2e0fe6a4f83"]
}, Open  ]],

Cell[TextData[{
 "In addition to piano, ",
 ButtonBox["SoundNote",
  BaseStyle->"Link"],
 " can handle a long list of possible instruments. The name of each \
instrument is a string."
}], "Text",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:6",ExpressionUUID->"284ee8b9-b53b-4513-8d4c-a6c2c653a86f"],

Cell["Play notes on a simulated violin:", "CodeCaption",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:7",ExpressionUUID->"3b13963b-ed9b-419e-88b0-4635b0c7e3fd"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{"n", ",", "0.1", ",", "\"\<Violin\>\""}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "5"}], "}"}]}], "]"}], "]"}]], "Input",
 Deletable->False,
 CellLabel->"In[6]:=",ExpressionUUID->"876070c2-6596-489f-8397-a8a2cbe12cba"],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
     RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[-0.1599553194190092, 0.9, 0.7], 
        StyleBox[{
          RectangleBox[{0., 0.4438917279327484}, \
{0.095, 0.45579648983751037}], 
          RectangleBox[{0.1, 0.45579648983751037}, \
{0.195, 0.4677012517422721}], 
          RectangleBox[{0.2, 0.4677012517422721}, \
{0.295, 0.4796060136470343}], 
          RectangleBox[{0.3, 0.4796060136470343}, \
{0.395, 0.49151077555179606}], 
          RectangleBox[{0.4, 0.49151077555179606}, \
{0.495, 0.5034155374565581}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None,
       PlotRange->{All, All},
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
     InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, InsetBox[
     StyleBox[
      TemplateBox[{
        StyleBox[
        "\" | \"", FontSize -> 14, FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False], 
        StyleBox[
         TemplateBox[{"\"0.5\"", "\" s\""}, "RowDefault"], FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False]},
       "RowWithSeparator"], "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{5, -158}, {29, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       PolygonBox[
        NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
          11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {11, -157}}]], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
           Sound`MIDISequence[{
             Sound`MIDITrack[{
               Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
               Sound`MIDIEvent[
               0, "ProgramCommand", "Channel" -> 0, "Value" -> 40], 
               Sound`MIDIEvent[
               0, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 127], 
               
               Sound`MIDIEvent[
               24, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               24, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               48, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               48, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               72, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               72, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               96, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               96, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               120, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
         Sound`EmitSampledSound[None, "Preemptive"], 
         FEPrivate`EmitSound[None, "Preemptive"]], Method -> "Preemptive", 
       PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{29, -158}, {53, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       RectangleBox[{35, -152}, {47, -141}], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitSampledSound[
           SampledSoundList[{{0.}}, 8000], "Preemptive"]; Sound`StopMIDI[], 
         FEPrivate`EmitSound[
          SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
       "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->{159.6171875, Automatic},
   PlotRange->{{0, 360}, {-164, 0}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[1, 0.1, "Violin"], 
    SoundNote[2, 0.1, "Violin"], 
    SoundNote[3, 0.1, "Violin"], 
    SoundNote[4, 0.1, "Violin"], 
    SoundNote[5, 0.1, "Violin"]}]]], "Output",
 Editable->False,
 CellEditDuplicate->False,
 Evaluatable->False,
 CellLabel->"Out[6]=",ExpressionUUID->"7ed6d35a-4950-4ec4-b7f1-cfc4280a065a"]
}, Open  ]],

Cell["\<\
It\[CloseCurlyQuote]s easy to make \[OpenCurlyDoubleQuote]random music\
\[CloseCurlyDoubleQuote]\[LongDash]different every time you generate it.\
\>", "Text",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:8",ExpressionUUID->"dcc3c782-19b6-43a7-a67f-312ef91b579c"],

Cell["Play a sequence of 20 notes with random pitches:", "CodeCaption",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"46bd4957-8382-433e-b179-fcc7b6d83550"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sound", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"SoundNote", "[", 
     RowBox[{
      RowBox[{"RandomInteger", "[", "12", "]"}], ",", "0.1", ",", 
      "\"\<Violin\>\""}], "]"}], ",", "20"}], "]"}], "]"}]], "Input",
 Deletable->False,
 CellLabel->"In[7]:=",ExpressionUUID->"4c8b700b-73c7-4680-9789-af3d67a5b947"],

Cell[BoxData[
 InterpretationBox[
  GraphicsBox[{
    {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
     RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
      GraphicsBox[
       {Hue[-0.1599553194190092, 0.9, 0.7], 
        StyleBox[{
          RectangleBox[{1.4, 0.43198696602798653}, \
{1.495, 0.4438917279327484}], 
          RectangleBox[{1.1, 0.4438917279327484}, \
{1.195, 0.45579648983751037}], 
          RectangleBox[{0.2, 0.45579648983751037}, \
{0.295, 0.4677012517422721}], 
          RectangleBox[{0.9, 0.45579648983751037}, \
{0.995, 0.4677012517422721}], 
          RectangleBox[{1.2, 0.4677012517422721}, \
{1.295, 0.4796060136470343}], 
          RectangleBox[{0.1, 0.4796060136470343}, \
{0.195, 0.49151077555179606}], 
          RectangleBox[{0.4, 0.4796060136470343}, \
{0.495, 0.49151077555179606}], 
          RectangleBox[{0.5, 0.4796060136470343}, \
{0.595, 0.49151077555179606}], 
          RectangleBox[{0.7, 0.4796060136470343}, \
{0.795, 0.49151077555179606}], 
          RectangleBox[{1.8, 0.4796060136470343}, \
{1.895, 0.49151077555179606}], 
          RectangleBox[{0., 0.49151077555179606}, \
{0.095, 0.5034155374565581}], 
          RectangleBox[{1.5, 0.5034155374565581}, \
{1.5950000000000002, 0.5153202993613198}], 
          RectangleBox[{1.6, 0.5153202993613198}, \
{1.695, 0.5272250612660818}], 
          RectangleBox[{0.3, 0.5391298231708438}, \
{0.395, 0.5510345850756055}], 
          RectangleBox[{1.9, 0.5391298231708438}, \
{1.995, 0.5510345850756055}], 
          RectangleBox[{0.6, 0.5510345850756055}, \
{0.695, 0.5629393469803675}], 
          RectangleBox[{1.7, 0.5510345850756055}, \
{1.795, 0.5629393469803675}], 
          RectangleBox[{0.8, 0.5629393469803675}, \
{0.895, 0.5748441088851295}], 
          RectangleBox[{1., 0.5629393469803675}, \
{1.0950000000000002, 0.5748441088851295}], 
          RectangleBox[{1.3, 0.5629393469803675}, \
{1.395, 0.5748441088851295}]},
         Antialiasing->True]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None,
       PlotRange->{All, All},
       PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
      ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
     InsetBox[
      GraphicsBox[
       {GrayLevel[0.6], RectangleBox[{0, 0}]},
       AspectRatio->NCache[
         Rational[6, 35], 0.17142857142857143`],
       Background->GrayLevel[1],
       Frame->True,
       FrameStyle->GrayLevel[0.75],
       FrameTicks->False,
       ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
      Background->RGBColor[
       0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, InsetBox[
     StyleBox[
      TemplateBox[{
        StyleBox[
        "\" | \"", FontSize -> 14, FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False], 
        StyleBox[
         TemplateBox[{"\"2\"", "\" s\""}, "RowDefault"], FontColor -> 
         RGBColor[0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
          StripOnInput -> False]},
       "RowWithSeparator"], "DialogStyle",
      StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{5, -158}, {29, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       PolygonBox[
        NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
          11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {11, -157}}]], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
           Sound`MIDISequence[{
             Sound`MIDITrack[{
               Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
               Sound`MIDIEvent[
               0, "ProgramCommand", "Channel" -> 0, "Value" -> 40], 
               Sound`MIDIEvent[
               0, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 127], 
               
               Sound`MIDIEvent[
               24, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               24, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               48, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               48, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               72, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               72, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               96, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0], 
               
               Sound`MIDIEvent[
               96, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 127], 
               Sound`MIDIEvent[
               120, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               120, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               144, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               144, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               168, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               168, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               192, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               192, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               216, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               216, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               240, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               240, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               264, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               264, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               288, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               288, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               312, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               312, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               336, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               336, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               360, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               360, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               384, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               384, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               408, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               408, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               432, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               432, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               456, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 0], 
               Sound`MIDIEvent[
               456, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                127], 
               Sound`MIDIEvent[
               480, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
         Sound`EmitSampledSound[None, "Preemptive"], 
         FEPrivate`EmitSound[None, "Preemptive"]], Method -> "Preemptive", 
       PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    TagBox[{
      {RGBColor[0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
       RectangleBox[{29, -158}, {53, -135}]}, 
      {RGBColor[0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
       RectangleBox[{35, -152}, {47, -141}], 
       StyleBox[{},
        Antialiasing->False]}},
     EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitSampledSound[
           SampledSoundList[{{0.}}, 8000], "Preemptive"]; Sound`StopMIDI[], 
         FEPrivate`EmitSound[
          SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
       "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> True}]], 
    {GrayLevel[0.75], 
     StyleBox[LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
      Antialiasing->False]}},
   ContentSelectable->False,
   ImageSize->{159.546875, Automatic},
   PlotRange->{{0, 360}, {-164, 0}},
   PlotRangePadding->1],
  Sound[{
    SoundNote[5, 0.1, "Violin"], 
    SoundNote[4, 0.1, "Violin"], 
    SoundNote[2, 0.1, "Violin"], 
    SoundNote[9, 0.1, "Violin"], 
    SoundNote[4, 0.1, "Violin"], 
    SoundNote[4, 0.1, "Violin"], 
    SoundNote[10, 0.1, "Violin"], 
    SoundNote[4, 0.1, "Violin"], 
    SoundNote[11, 0.1, "Violin"], 
    SoundNote[2, 0.1, "Violin"], 
    SoundNote[11, 0.1, "Violin"], 
    SoundNote[1, 0.1, "Violin"], 
    SoundNote[3, 0.1, "Violin"], 
    SoundNote[11, 0.1, "Violin"], 
    SoundNote[0, 0.1, "Violin"], 
    SoundNote[6, 0.1, "Violin"], 
    SoundNote[7, 0.1, "Violin"], 
    SoundNote[10, 0.1, "Violin"], 
    SoundNote[4, 0.1, "Violin"], 
    SoundNote[9, 0.1, "Violin"]}]]], "Output",
 Editable->False,
 CellEditDuplicate->False,
 Evaluatable->False,
 CellLabel->"Out[7]=",ExpressionUUID->"81b4aeea-b2c5-4f28-8121-0ccd78111707"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Vocabulary", "VocabularySection",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"c7224fc5-aa6c-478d-94ed-1d8884610388"],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["Sound",
      BaseStyle->"CodeLink"], "[", 
     RowBox[{"{", 
      StyleBox["...", "GrayPointSix"], "}"}], "]"}], " ", Cell[
    "create a sound from notes ", "VocabularyText",ExpressionUUID->
     "eef4f93d-f690-454d-b023-5509a9843eb5"]},
   {
    RowBox[{
     ButtonBox["SoundNote",
      BaseStyle->"CodeLink"], "[", "\"\<C\>\"", "]"}], " ", Cell[
    "a named note ", "VocabularyText",ExpressionUUID->
     "fe6e24be-82b5-425e-b204-930edb96c1ab"]},
   {
    RowBox[{
     ButtonBox["SoundNote",
      BaseStyle->"CodeLink"], "[", "5", "]"}], " ", Cell[
    "a note with a numbered pitch ", "VocabularyText",ExpressionUUID->
     "530166e5-bf73-4932-970b-434cb2306e33"]},
   {
    RowBox[{
     ButtonBox["SoundNote",
      BaseStyle->"CodeLink"], "[", 
     RowBox[{"5", ",", "0.1"}], "]"}], " ", Cell[
    "a note played for a specified time ", "VocabularyText",ExpressionUUID->
     "14423840-b36b-4654-b0f3-987ccbcf7277"]},
   {
    RowBox[{
     ButtonBox["SoundNote",
      BaseStyle->"CodeLink"], "[", 
     RowBox[{"5", ",", "0.1", ",", "\"\<Guitar\>\""}], "]"}], " ", Cell[
    "a note played on a certain instrument ", "VocabularyText",ExpressionUUID->
     "5992e9fb-ca24-485a-8b69-a753c0327a43"]}
  }]], "VocabularyTable",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->
  "LastCellOfAPage",ExpressionUUID->"08513e65-c161-4097-98ee-4c6cb9d4fa96"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 DynamicModuleBox[{$CellContext`open$$ = False}, 
  TagBox[GridBox[{
     {
      StyleBox["\<\"Exercises\"\>",
       StripOnInput->False,
       FontSlant->"Italic"], 
      PaneBox[
       TagBox[
        ButtonBox[
         DynamicBox[ToBoxes[
           If[$CellContext`open$$, 
            Row[{
              Style["-", "ExercisePreviewButtonTextHighlight"], 
              Style[" Preview Exercises", "ExercisePreviewButtonText"]}], 
            Row[{
              Style["+", "ExercisePreviewButtonTextHighlight"], 
              Style[" Preview Exercises", "ExercisePreviewButtonText"]}]], 
           StandardForm],
          ImageSizeCache->{115., {1., 10.}}],
         Appearance->None,
         ButtonFunction:>(
          If[$CellContext`open$$, $CellContext`open$$ = 
            False, $CellContext`open$$ = True]; NotebookFind[
            EvaluationNotebook[], "ExerciseSummaryCell", Next, CellStyle, 
            WrapAround -> True, AutoScroll -> False]; FrontEndTokenExecute[
            EvaluationNotebook[], "ExpandSelection"]; FrontEndExecute[
            FrontEndToken[
             EvaluationNotebook[], "OpenCloseGroup"]]),
         Evaluator->Automatic,
         Method->"Preemptive"],
        MouseAppearanceTag["LinkHand"]],
       Alignment->Right,
       ImageSize->Full]}
    },
    AutoDelete->False,
    GridBoxAlignment->{"Columns" -> {Left, Right}, "Rows" -> {{Center}}},
    GridBoxItemSize->{"Columns" -> {{Automatic}}, "Rows" -> {{Automatic}}}],
   "Grid"],
  DynamicModuleValues:>{}]], "ExerciseSection",
 Editable->False,
 ShowCellBracket->False,
 GeneratedCell->True,
 CellAutoOverwrite->True,
 CellChangeTimes->{3.669991951589675*^9},
 ShowStringCharacters->
  False,ExpressionUUID->"381f1676-631f-4d04-ba24-144a95c1a552"],

Cell[CellGroupData[{

Cell[BoxData[
 TagBox[GridBox[{
    {
     TagBox[GridBox[{
        {
         PaneBox[
          
          StyleBox["\<\"10 Exercises Available\"\>", "ExerciseSummaryHeading",
           
           StripOnInput->False],
          ImageMargins->{{20, 0}, {0, 10}}]},
        {
         PaneBox[
          StyleBox["\<\"with 4 extras\"\>", "ExerciseSummarySubheading",
           StripOnInput->False],
          ImageMargins->{{20, 0}, {10, 0}}]}
       },
       DefaultBaseStyle->"Column",
       GridBoxAlignment->{"Columns" -> {{Left}}},
       GridBoxItemSize->{"Columns" -> {{Automatic}}, "Rows" -> {{Automatic}}},
       
       GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{0}}}],
      "Column"], 
     PaneBox[
      ButtonBox[
       StyleBox["\<\"Get Started \[RightGuillemet]\"\>", 
        "ExerciseSummaryHeading",
        StripOnInput->False],
       Appearance->None,
       ButtonFunction:>NotebookLocate[{"EIWL2-12-sound-exercises.nb", ""}],
       Evaluator->Automatic,
       Method->"Preemptive"],
      FrameMargins->{{10, 0}, {10, 10}},
      ImageSize->Full]}
   },
   AutoDelete->False,
   GridBoxAlignment->{"Columns" -> {{Left}}},
   GridBoxItemSize->{"Columns" -> {
       Scaled[0.7], 
       Scaled[0.3]}, "Rows" -> {{Automatic}}}],
  "Grid"]], "ExerciseSummaryCell",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"8094785e-3642-4480-8539-26ee2f84007b"],

Cell[CellGroupData[{

Cell[TextData[{
 "Generate the sequence of notes with pitches 0, 4 and 7.",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "12.1", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"e957386a-cc4f-4903-b83a-bec1aabc97f5"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[
             {RGBColor[0.5, 0, 0.5], 
              
              StyleBox[{
                RectangleBox[{0., 0.43198696602798653}, \
{0.95, 0.4438917279327484}], 
                RectangleBox[{1., 0.4796060136470343}, \
{1.95, 0.49151077555179606}], 
                RectangleBox[{2., 0.5153202993613198}, \
{2.95, 0.5272250612660818}]},
               Antialiasing->True]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"3\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{159.5078125, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[{
          SoundNote[0], 
          SoundNote[4], 
          SoundNote[7]}]]], "ExerciseOutput",ExpressionUUID->
       "f7b06a80-996b-4824-9e53-0e0225f6b794"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{
                RGBColor[
                0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                RectangleBox[{0, 0}, {360, -164}]}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{
                    RGBColor[0.5, 0, 0.5], 
                    StyleBox[{{
                    
                    RectangleBox[{0., 0.43198696602798653`}, {0.95, 
                    0.4438917279327484}], 
                    
                    RectangleBox[{1., 0.4796060136470343}, {1.95, 
                    0.49151077555179606`}], 
                    
                    RectangleBox[{2., 0.5153202993613198}, {2.95, 
                    0.5272250612660818}]}}, Antialiasing -> True]}}}}}}, 
               AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{
                   GrayLevel[0.6], 
                   RectangleBox[{0, 0}]}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"3\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{5, -158}, {29, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 PolygonBox[
                  NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{29, -158}, {53, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 RectangleBox[{35, -152}, {47, -141}], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{
                GrayLevel[0.75], 
                StyleBox[
                 LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}], 
                 Antialiasing -> False]}}}}}, ContentSelectable -> False, 
           ImageSize -> {159.5078125, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[{
            SoundNote[0], 
            SoundNote[4], 
            SoundNote[7]}]]], "ExerciseOutput", Selectable -> False, Copyable -> 
        False, Editable -> False, ShowCellBracket -> False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"a7d2c89b-4b08-434e-9ba4-72bd55d94d04"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Generate 2 seconds of playing middle A on a cello.",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "12.2", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"252ce3c0-e584-433c-9da7-8e5e86586af7"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[
             {Hue[0.22538543343129763`, 0.9, 0.7], 
              StyleBox[RectangleBox[{0., 0.5391298231708438}, {1.95, 0.5510345850756055}],
               Antialiasing->True]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"2\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 42], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{160.3984375, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[
         SoundNote["A", 2, "Cello"]]]], "ExerciseOutput",ExpressionUUID->
       "c4beaa22-5921-46ac-8f08-b929b3b05801"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{
                RGBColor[
                0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                RectangleBox[{0, 0}, {360, -164}]}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{
                    Hue[0.22538543343129763`, 0.9, 0.7], 
                    StyleBox[{{
                    
                    RectangleBox[{0., 0.5391298231708438}, {1.95, 
                    0.5510345850756055}]}}, Antialiasing -> True]}}}}}}, 
               AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{
                   GrayLevel[0.6], 
                   RectangleBox[{0, 0}]}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"2\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{5, -158}, {29, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 PolygonBox[
                  NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 42], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{29, -158}, {53, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 RectangleBox[{35, -152}, {47, -141}], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{
                GrayLevel[0.75], 
                StyleBox[
                 LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}], 
                 Antialiasing -> False]}}}}}, ContentSelectable -> False, 
           ImageSize -> {160.3984375, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[
           SoundNote["A", 2, "Cello"]]]], "ExerciseOutput", Selectable -> 
        False, Copyable -> False, Editable -> False, ShowCellBracket -> 
        False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"09162a7a-7f18-491f-b9fd-621c6b4ccb61"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Generate a \[OpenCurlyDoubleQuote]riff\[CloseCurlyDoubleQuote] of notes \
from pitch 0 to pitch 48 in steps of 1, with each note lasting 0.05 \
seconds.",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "12.3", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"a0edb41f-5270-4a29-a976-592a0a91d558"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[
             {RGBColor[0.5, 0, 0.5], 
              
              StyleBox[{
                RectangleBox[{0., 0.43198696602798653}, \
{0.0475, 0.4438917279327484}], 
                RectangleBox[{0.05, 0.4438917279327484}, \
{0.0975, 0.45579648983751037}], 
                RectangleBox[{0.1, 0.45579648983751037}, \
{0.1475, 0.4677012517422721}], 
                RectangleBox[{0.15, 0.4677012517422721}, \
{0.1975, 0.4796060136470343}], 
                RectangleBox[{0.2, 0.4796060136470343}, \
{0.2475, 0.49151077555179606}], 
                RectangleBox[{0.25, 0.49151077555179606}, \
{0.2975, 0.5034155374565581}], 
                RectangleBox[{0.3, 0.5034155374565581}, \
{0.3475, 0.5153202993613198}], 
                RectangleBox[{0.35, 0.5153202993613198}, \
{0.3975, 0.5272250612660818}], 
                RectangleBox[{0.4, 0.5272250612660818}, \
{0.4475, 0.5391298231708438}], 
                RectangleBox[{0.45, 0.5391298231708438}, \
{0.4975, 0.5510345850756055}], 
                RectangleBox[{0.5, 0.5510345850756055}, \
{0.5475000000000001, 0.5629393469803675}], 
                RectangleBox[{0.55, 0.5629393469803675}, \
{0.5975, 0.5748441088851295}], 
                RectangleBox[{0.6, 0.5748441088851295}, \
{0.6475, 0.5867488707898912}], 
                RectangleBox[{0.65, 0.5867488707898912}, \
{0.6975, 0.5986536326946532}], 
                RectangleBox[{0.7, 0.5986536326946532}, \
{0.7475, 0.610558394599415}], 
                RectangleBox[{0.75, 0.610558394599415}, \
{0.7975000000000001, 0.6224631565041769}], 
                RectangleBox[{0.8, 0.6224631565041769}, \
{0.8475, 0.6343679184089389}], 
                RectangleBox[{0.85, 0.6343679184089389}, \
{0.8975, 0.646272680313701}], 
                
                RectangleBox[{0.9, 0.646272680313701}, \
{0.9475, 0.658177442218463}], 
                RectangleBox[{0.95, 0.658177442218463}, \
{0.9975, 0.6700822041232247}], 
                RectangleBox[{1., 0.6700822041232247}, \
{1.0475, 0.6819869660279864}], 
                RectangleBox[{1.05, 0.6819869660279864}, \
{1.0975000000000001, 0.6938917279327484}], 
                RectangleBox[{1.1, 0.6938917279327484}, \
{1.1475, 0.7057964898375104}], 
                RectangleBox[{1.15, 0.7057964898375104}, \
{1.1975, 0.7177012517422724}], 
                RectangleBox[{1.2, 0.7177012517422724}, \
{1.2475, 0.7296060136470344}], 
                RectangleBox[{1.25, 0.7296060136470344}, \
{1.2975, 0.7415107755517961}], 
                RectangleBox[{1.3, 0.7415107755517961}, \
{1.3475000000000001, 0.7534155374565579}], 
                RectangleBox[{1.35, 0.7534155374565579}, \
{1.3975, 0.7653202993613198}], 
                RectangleBox[{1.4, 0.7653202993613198}, \
{1.4475, 0.7772250612660818}], 
                RectangleBox[{1.45, 0.7772250612660818}, \
{1.4975, 0.7891298231708438}], 
                RectangleBox[{1.5, 0.7891298231708438}, \
{1.5475, 0.8010345850756058}], 
                RectangleBox[{1.55, 0.8010345850756058}, \
{1.5975000000000001, 0.8129393469803675}], 
                RectangleBox[{1.6, 0.8129393469803675}, \
{1.6475, 0.8248441088851293}], 
                RectangleBox[{1.65, 0.8248441088851293}, \
{1.6975, 0.8367488707898912}], 
                RectangleBox[{1.7, 0.8367488707898912}, \
{1.7475, 0.8486536326946532}], 
                RectangleBox[{1.75, 0.8486536326946532}, \
{1.7975, 0.860558394599415}], 
                RectangleBox[{1.8, 0.860558394599415}, \
{1.8475000000000001, 0.872463156504177}], 
                RectangleBox[{1.85, 0.872463156504177}, \
{1.8975, 0.884367918408939}], 
                RectangleBox[{1.9, 0.884367918408939}, \
{1.9475, 0.896272680313701}], 
                RectangleBox[{1.95, 0.896272680313701}, \
{1.9975, 0.9081774422184627}], 
                RectangleBox[{2., 0.9081774422184627}, \
{2.0475, 0.9200822041232247}], 
                RectangleBox[{2.05, 0.9200822041232247}, \
{2.0975, 0.9319869660279865}], 
                RectangleBox[{2.1, 0.9319869660279865}, \
{2.1475, 0.9438917279327484}], 
                RectangleBox[{2.15, 0.9438917279327484}, \
{2.1975000000000002, 0.9557964898375104}], 
                RectangleBox[{2.2, 0.9557964898375104}, \
{2.2475, 0.9677012517422724}], 
                RectangleBox[{2.25, 0.9677012517422724}, \
{2.2975, 0.9796060136470341}], 
                RectangleBox[{2.3, 0.9796060136470341}, \
{2.3475, 0.9915107755517961}], 
                RectangleBox[{2.35, 0.9915107755517961}, \
{2.3975, 1.003415537456558}], 
                RectangleBox[{2.4, 1.003415537456558}, \
{2.4475000000000002, 1.0153202993613202}]},
               Antialiasing->True]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"2.45\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    12, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    12, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    24, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    24, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    36, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    36, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    60, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    60, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    72, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    72, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    84, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    84, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    108, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    108, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    120, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    120, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    132, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    132, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    156, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    156, "NoteOn", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    168, "NoteOff", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    168, "NoteOn", "Note" -> 74, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    180, "NoteOff", "Note" -> 74, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    180, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    204, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    204, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    216, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    216, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    228, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    228, "NoteOn", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    240, "NoteOn", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    252, "NoteOff", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    252, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    264, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    264, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    276, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    276, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    288, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    288, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    300, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    300, "NoteOn", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    312, "NoteOff", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    312, "NoteOn", "Note" -> 86, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    324, "NoteOff", "Note" -> 86, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    324, "NoteOn", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    336, "NoteOff", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    336, "NoteOn", "Note" -> 88, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    348, "NoteOff", "Note" -> 88, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    348, "NoteOn", "Note" -> 89, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    360, "NoteOff", "Note" -> 89, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    360, "NoteOn", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    372, "NoteOff", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    372, "NoteOn", "Note" -> 91, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    384, "NoteOff", "Note" -> 91, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    384, "NoteOn", "Note" -> 92, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    396, "NoteOff", "Note" -> 92, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    396, "NoteOn", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    408, "NoteOff", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    408, "NoteOn", "Note" -> 94, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    420, "NoteOff", "Note" -> 94, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    420, "NoteOn", "Note" -> 95, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    432, "NoteOff", "Note" -> 95, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    432, "NoteOn", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    444, "NoteOff", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    444, "NoteOn", "Note" -> 97, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    456, "NoteOff", "Note" -> 97, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    456, "NoteOn", "Note" -> 98, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    468, "NoteOff", "Note" -> 98, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    468, "NoteOn", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    480, "NoteOff", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    480, "NoteOn", "Note" -> 100, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    492, "NoteOff", "Note" -> 100, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    492, "NoteOn", "Note" -> 101, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    504, "NoteOff", "Note" -> 101, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    504, "NoteOn", "Note" -> 102, "Channel" -> 0, 
                    "Velocity" -> 127], 
                    Sound`MIDIEvent[
                    516, "NoteOff", "Note" -> 102, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    516, "NoteOn", "Note" -> 103, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    528, "NoteOff", "Note" -> 103, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    528, "NoteOn", "Note" -> 104, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    540, "NoteOff", "Note" -> 104, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    540, "NoteOn", "Note" -> 105, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    552, "NoteOff", "Note" -> 105, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    552, "NoteOn", "Note" -> 106, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    564, "NoteOff", "Note" -> 106, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    564, "NoteOn", "Note" -> 107, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    576, "NoteOff", "Note" -> 107, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    576, "NoteOn", "Note" -> 108, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    588, "NoteOff", "Note" -> 108, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{160.1015625, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[{
          SoundNote[0, 0.05], 
          SoundNote[1, 0.05], 
          SoundNote[2, 0.05], 
          SoundNote[3, 0.05], 
          SoundNote[4, 0.05], 
          SoundNote[5, 0.05], 
          SoundNote[6, 0.05], 
          SoundNote[7, 0.05], 
          SoundNote[8, 0.05], 
          SoundNote[9, 0.05], 
          SoundNote[10, 0.05], 
          SoundNote[11, 0.05], 
          SoundNote[12, 0.05], 
          SoundNote[13, 0.05], 
          SoundNote[14, 0.05], 
          SoundNote[15, 0.05], 
          SoundNote[16, 0.05], 
          SoundNote[17, 0.05], 
          SoundNote[18, 0.05], 
          SoundNote[19, 0.05], 
          SoundNote[20, 0.05], 
          SoundNote[21, 0.05], 
          SoundNote[22, 0.05], 
          SoundNote[23, 0.05], 
          SoundNote[24, 0.05], 
          SoundNote[25, 0.05], 
          SoundNote[26, 0.05], 
          SoundNote[27, 0.05], 
          SoundNote[28, 0.05], 
          SoundNote[29, 0.05], 
          SoundNote[30, 0.05], 
          SoundNote[31, 0.05], 
          SoundNote[32, 0.05], 
          SoundNote[33, 0.05], 
          SoundNote[34, 0.05], 
          SoundNote[35, 0.05], 
          SoundNote[36, 0.05], 
          SoundNote[37, 0.05], 
          SoundNote[38, 0.05], 
          SoundNote[39, 0.05], 
          SoundNote[40, 0.05], 
          SoundNote[41, 0.05], 
          SoundNote[42, 0.05], 
          SoundNote[43, 0.05], 
          SoundNote[44, 0.05], 
          SoundNote[45, 0.05], 
          SoundNote[46, 0.05], 
          SoundNote[47, 0.05], 
          SoundNote[48, 0.05]}]]], "ExerciseOutput",ExpressionUUID->
       "5286d022-92c6-4c30-903c-fd04c4969e43"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{
                RGBColor[
                0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                RectangleBox[{0, 0}, {360, -164}]}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{
                    RGBColor[0.5, 0, 0.5], 
                    StyleBox[{{
                    
                    RectangleBox[{0., 0.43198696602798653`}, {0.0475, 
                    0.4438917279327484}], 
                    
                    RectangleBox[{0.05, 0.4438917279327484}, {0.0975, 
                    0.45579648983751037`}], 
                    
                    RectangleBox[{0.1, 0.45579648983751037`}, {0.1475, 
                    0.4677012517422721}], 
                    
                    RectangleBox[{0.15, 0.4677012517422721}, {0.1975, 
                    0.4796060136470343}], 
                    
                    RectangleBox[{0.2, 0.4796060136470343}, {0.2475, 
                    0.49151077555179606`}], 
                    
                    RectangleBox[{0.25, 0.49151077555179606`}, {0.2975, 
                    0.5034155374565581}], 
                    
                    RectangleBox[{0.3, 0.5034155374565581}, {0.3475, 
                    0.5153202993613198}], 
                    
                    RectangleBox[{0.35, 0.5153202993613198}, {0.3975, 
                    0.5272250612660818}], 
                    
                    RectangleBox[{0.4, 0.5272250612660818}, {0.4475, 
                    0.5391298231708438}], 
                    
                    RectangleBox[{0.45, 0.5391298231708438}, {0.4975, 
                    0.5510345850756055}], 
                    
                    RectangleBox[{0.5, 0.5510345850756055}, {
                    0.5475000000000001, 0.5629393469803675}], 
                    RectangleBox[{0.55, 0.5629393469803675}, {0.5975, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{0.6, 0.5748441088851295}, {0.6475, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{0.65, 0.5867488707898912}, {0.6975, 
                    0.5986536326946532}], 
                    
                    RectangleBox[{0.7, 0.5986536326946532}, {0.7475, 
                    0.610558394599415}], 
                    
                    RectangleBox[{0.75, 0.610558394599415}, {
                    0.7975000000000001, 0.6224631565041769}], 
                    
                    RectangleBox[{0.8, 0.6224631565041769}, {0.8475, 
                    0.6343679184089389}], 
                    
                    RectangleBox[{0.85, 0.6343679184089389}, {0.8975, 
                    0.646272680313701}], 
                    
                    RectangleBox[{0.9, 0.646272680313701}, {0.9475, 
                    0.658177442218463}], 
                    
                    RectangleBox[{0.95, 0.658177442218463}, {0.9975, 
                    0.6700822041232247}], 
                    
                    RectangleBox[{1., 0.6700822041232247}, {1.0475, 
                    0.6819869660279864}], 
                    
                    RectangleBox[{1.05, 0.6819869660279864}, {
                    1.0975000000000001`, 0.6938917279327484}], 
                    
                    RectangleBox[{1.1, 0.6938917279327484}, {1.1475, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{1.15, 0.7057964898375104}, {1.1975, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{1.2, 0.7177012517422724}, {1.2475, 
                    0.7296060136470344}], 
                    
                    RectangleBox[{1.25, 0.7296060136470344}, {1.2975, 
                    0.7415107755517961}], 
                    
                    RectangleBox[{1.3, 0.7415107755517961}, {
                    1.3475000000000001`, 0.7534155374565579}], 
                    
                    RectangleBox[{1.35, 0.7534155374565579}, {1.3975, 
                    0.7653202993613198}], 
                    
                    RectangleBox[{1.4, 0.7653202993613198}, {1.4475, 
                    0.7772250612660818}], 
                    
                    RectangleBox[{1.45, 0.7772250612660818}, {1.4975, 
                    0.7891298231708438}], 
                    
                    RectangleBox[{1.5, 0.7891298231708438}, {1.5475, 
                    0.8010345850756058}], 
                    RectangleBox[{1.55, 0.8010345850756058}, {
                    1.5975000000000001`, 0.8129393469803675}], 
                    
                    RectangleBox[{1.6, 0.8129393469803675}, {1.6475, 
                    0.8248441088851293}], 
                    
                    RectangleBox[{1.65, 0.8248441088851293}, {1.6975, 
                    0.8367488707898912}], 
                    
                    RectangleBox[{1.7, 0.8367488707898912}, {1.7475, 
                    0.8486536326946532}], 
                    
                    RectangleBox[{1.75, 0.8486536326946532}, {1.7975, 
                    0.860558394599415}], 
                    
                    RectangleBox[{1.8, 0.860558394599415}, {
                    1.8475000000000001`, 0.872463156504177}], 
                    
                    RectangleBox[{1.85, 0.872463156504177}, {1.8975, 
                    0.884367918408939}], 
                    
                    RectangleBox[{1.9, 0.884367918408939}, {1.9475, 
                    0.896272680313701}], 
                    
                    RectangleBox[{1.95, 0.896272680313701}, {1.9975, 
                    0.9081774422184627}], 
                    
                    RectangleBox[{2., 0.9081774422184627}, {2.0475, 
                    0.9200822041232247}], 
                    
                    RectangleBox[{2.05, 0.9200822041232247}, {2.0975, 
                    0.9319869660279865}], 
                    
                    RectangleBox[{2.1, 0.9319869660279865}, {2.1475, 
                    0.9438917279327484}], 
                    
                    RectangleBox[{2.15, 0.9438917279327484}, {
                    2.1975000000000002`, 0.9557964898375104}], 
                    
                    RectangleBox[{2.2, 0.9557964898375104}, {2.2475, 
                    0.9677012517422724}], 
                    
                    RectangleBox[{2.25, 0.9677012517422724}, {2.2975, 
                    0.9796060136470341}], 
                    
                    RectangleBox[{2.3, 0.9796060136470341}, {2.3475, 
                    0.9915107755517961}], 
                    
                    RectangleBox[{2.35, 0.9915107755517961}, {2.3975, 
                    1.003415537456558}], 
                    
                    RectangleBox[{2.4, 1.003415537456558}, {
                    2.4475000000000002`, 1.0153202993613202`}]}}, 
                    Antialiasing -> True]}}}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{
                   GrayLevel[0.6], 
                   RectangleBox[{0, 0}]}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"2.45\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{5, -158}, {29, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 PolygonBox[
                  NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    12, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    12, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    24, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    24, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    36, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    36, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    60, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    60, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    72, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    72, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    84, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    84, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    108, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    108, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    120, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    120, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    132, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    132, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    156, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    156, "NoteOn", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    168, "NoteOff", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    168, "NoteOn", "Note" -> 74, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    180, "NoteOff", "Note" -> 74, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    180, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    204, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    204, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    216, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    216, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    228, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    228, "NoteOn", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    240, "NoteOn", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    252, "NoteOff", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    252, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    264, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    264, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    276, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    276, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    288, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    288, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    300, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    300, "NoteOn", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    312, "NoteOff", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    312, "NoteOn", "Note" -> 86, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    324, "NoteOff", "Note" -> 86, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    324, "NoteOn", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    336, "NoteOff", "Note" -> 87, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    336, "NoteOn", "Note" -> 88, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    348, "NoteOff", "Note" -> 88, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    348, "NoteOn", "Note" -> 89, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    360, "NoteOff", "Note" -> 89, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    360, "NoteOn", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    372, "NoteOff", "Note" -> 90, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    372, "NoteOn", "Note" -> 91, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    384, "NoteOff", "Note" -> 91, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    384, "NoteOn", "Note" -> 92, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    396, "NoteOff", "Note" -> 92, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    396, "NoteOn", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    408, "NoteOff", "Note" -> 93, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    408, "NoteOn", "Note" -> 94, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    420, "NoteOff", "Note" -> 94, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    420, "NoteOn", "Note" -> 95, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    432, "NoteOff", "Note" -> 95, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    432, "NoteOn", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    444, "NoteOff", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    444, "NoteOn", "Note" -> 97, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    456, "NoteOff", "Note" -> 97, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    456, "NoteOn", "Note" -> 98, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    468, "NoteOff", "Note" -> 98, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    468, "NoteOn", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    480, "NoteOff", "Note" -> 99, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    480, "NoteOn", "Note" -> 100, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    492, "NoteOff", "Note" -> 100, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    492, "NoteOn", "Note" -> 101, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    504, "NoteOff", "Note" -> 101, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    504, "NoteOn", "Note" -> 102, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    516, "NoteOff", "Note" -> 102, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    516, "NoteOn", "Note" -> 103, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    528, "NoteOff", "Note" -> 103, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    528, "NoteOn", "Note" -> 104, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    540, "NoteOff", "Note" -> 104, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    540, "NoteOn", "Note" -> 105, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    552, "NoteOff", "Note" -> 105, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    552, "NoteOn", "Note" -> 106, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    564, "NoteOff", "Note" -> 106, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    564, "NoteOn", "Note" -> 107, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    576, "NoteOff", "Note" -> 107, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    576, "NoteOn", "Note" -> 108, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    588, "NoteOff", "Note" -> 108, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{29, -158}, {53, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 RectangleBox[{35, -152}, {47, -141}], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{
                GrayLevel[0.75], 
                StyleBox[
                 LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}], 
                 Antialiasing -> False]}}}}}, ContentSelectable -> False, 
           ImageSize -> {160.1015625, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[{
            SoundNote[0, 0.05], 
            SoundNote[1, 0.05], 
            SoundNote[2, 0.05], 
            SoundNote[3, 0.05], 
            SoundNote[4, 0.05], 
            SoundNote[5, 0.05], 
            SoundNote[6, 0.05], 
            SoundNote[7, 0.05], 
            SoundNote[8, 0.05], 
            SoundNote[9, 0.05], 
            SoundNote[10, 0.05], 
            SoundNote[11, 0.05], 
            SoundNote[12, 0.05], 
            SoundNote[13, 0.05], 
            SoundNote[14, 0.05], 
            SoundNote[15, 0.05], 
            SoundNote[16, 0.05], 
            SoundNote[17, 0.05], 
            SoundNote[18, 0.05], 
            SoundNote[19, 0.05], 
            SoundNote[20, 0.05], 
            SoundNote[21, 0.05], 
            SoundNote[22, 0.05], 
            SoundNote[23, 0.05], 
            SoundNote[24, 0.05], 
            SoundNote[25, 0.05], 
            SoundNote[26, 0.05], 
            SoundNote[27, 0.05], 
            SoundNote[28, 0.05], 
            SoundNote[29, 0.05], 
            SoundNote[30, 0.05], 
            SoundNote[31, 0.05], 
            SoundNote[32, 0.05], 
            SoundNote[33, 0.05], 
            SoundNote[34, 0.05], 
            SoundNote[35, 0.05], 
            SoundNote[36, 0.05], 
            SoundNote[37, 0.05], 
            SoundNote[38, 0.05], 
            SoundNote[39, 0.05], 
            SoundNote[40, 0.05], 
            SoundNote[41, 0.05], 
            SoundNote[42, 0.05], 
            SoundNote[43, 0.05], 
            SoundNote[44, 0.05], 
            SoundNote[45, 0.05], 
            SoundNote[46, 0.05], 
            SoundNote[47, 0.05], 
            SoundNote[48, 0.05]}]]], "ExerciseOutput", Selectable -> False, 
        Copyable -> False, Editable -> False, ShowCellBracket -> False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"e1319dbd-1806-4aea-a3f3-f6d413d59f8c"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Generate a sequence of notes going from pitch 12 down to 0 in steps of 1.",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "12.4", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"66a5c279-97fc-4402-8e4d-7e8ab08efc09"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[
             {RGBColor[0.5, 0, 0.5], 
              
              StyleBox[{
                RectangleBox[{12., 0.43198696602798653}, \
{12.95, 0.4438917279327484}], 
                RectangleBox[{11., 0.4438917279327484}, \
{11.95, 0.45579648983751037}], 
                RectangleBox[{10., 0.45579648983751037}, \
{10.95, 0.4677012517422721}], 
                RectangleBox[{9., 0.4677012517422721}, \
{9.95, 0.4796060136470343}], 
                RectangleBox[{8., 0.4796060136470343}, \
{8.95, 0.49151077555179606}], 
                RectangleBox[{7., 0.49151077555179606}, \
{7.95, 0.5034155374565581}], 
                RectangleBox[{6., 0.5034155374565581}, \
{6.95, 0.5153202993613198}], 
                RectangleBox[{5., 0.5153202993613198}, \
{5.95, 0.5272250612660818}], 
                RectangleBox[{4., 0.5272250612660818}, \
{4.95, 0.5391298231708438}], 
                RectangleBox[{3., 0.5391298231708438}, \
{3.95, 0.5510345850756055}], 
                RectangleBox[{2., 0.5510345850756055}, \
{2.95, 0.5629393469803675}], 
                RectangleBox[{1., 0.5629393469803675}, \
{1.95, 0.5748441088851295}], 
                RectangleBox[{0., 0.5748441088851295}, \
{0.95, 0.5867488707898912}]},
               Antialiasing->True]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"13\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    240, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    288, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    288, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    336, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    336, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    384, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    384, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    432, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    432, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    480, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    480, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    528, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    528, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    576, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    576, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    624, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{159.84375, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[{
          SoundNote[12], 
          SoundNote[11], 
          SoundNote[10], 
          SoundNote[9], 
          SoundNote[8], 
          SoundNote[7], 
          SoundNote[6], 
          SoundNote[5], 
          SoundNote[4], 
          SoundNote[3], 
          SoundNote[2], 
          SoundNote[1], 
          SoundNote[0]}]]], "ExerciseOutput",ExpressionUUID->
       "2f75cb11-25bf-47d9-8079-cbc59bad429e"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{
                RGBColor[
                0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                RectangleBox[{0, 0}, {360, -164}]}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{
                    RGBColor[0.5, 0, 0.5], 
                    StyleBox[{{
                    
                    RectangleBox[{12., 0.43198696602798653`}, {12.95, 
                    0.4438917279327484}], 
                    
                    RectangleBox[{11., 0.4438917279327484}, {11.95, 
                    0.45579648983751037`}], 
                    
                    RectangleBox[{10., 0.45579648983751037`}, {10.95, 
                    0.4677012517422721}], 
                    
                    RectangleBox[{9., 0.4677012517422721}, {9.95, 
                    0.4796060136470343}], 
                    
                    RectangleBox[{8., 0.4796060136470343}, {8.95, 
                    0.49151077555179606`}], 
                    
                    RectangleBox[{7., 0.49151077555179606`}, {7.95, 
                    0.5034155374565581}], 
                    
                    RectangleBox[{6., 0.5034155374565581}, {6.95, 
                    0.5153202993613198}], 
                    
                    RectangleBox[{5., 0.5153202993613198}, {5.95, 
                    0.5272250612660818}], 
                    
                    RectangleBox[{4., 0.5272250612660818}, {4.95, 
                    0.5391298231708438}], 
                    
                    RectangleBox[{3., 0.5391298231708438}, {3.95, 
                    0.5510345850756055}], 
                    
                    RectangleBox[{2., 0.5510345850756055}, {2.95, 
                    0.5629393469803675}], 
                    RectangleBox[{1., 0.5629393469803675}, {1.95, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{0., 0.5748441088851295}, {0.95, 
                    0.5867488707898912}]}}, Antialiasing -> True]}}}}}}, 
               AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{
                   GrayLevel[0.6], 
                   RectangleBox[{0, 0}]}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"13\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{5, -158}, {29, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 PolygonBox[
                  NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    240, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    288, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    288, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    336, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    336, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    384, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    384, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    432, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    432, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    480, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    480, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    528, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    528, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    576, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    576, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    624, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{29, -158}, {53, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 RectangleBox[{35, -152}, {47, -141}], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{
                GrayLevel[0.75], 
                StyleBox[
                 LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}], 
                 Antialiasing -> False]}}}}}, ContentSelectable -> False, 
           ImageSize -> {159.84375, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[{
            SoundNote[12], 
            SoundNote[11], 
            SoundNote[10], 
            SoundNote[9], 
            SoundNote[8], 
            SoundNote[7], 
            SoundNote[6], 
            SoundNote[5], 
            SoundNote[4], 
            SoundNote[3], 
            SoundNote[2], 
            SoundNote[1], 
            SoundNote[0]}]]], "ExerciseOutput", Selectable -> False, Copyable -> 
        False, Editable -> False, ShowCellBracket -> False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"03af47f8-c8d1-4fdf-be8e-eec6844623d2"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Generate a sequence of 5 notes starting with middle C, then successively \
going up by an octave at a time.",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "12.5", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"ef5aebe8-52ee-417a-86cc-b0a565b0da50"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[
             {RGBColor[0.5, 0, 0.5], 
              
              StyleBox[{
                RectangleBox[{0., 0.43198696602798653}, \
{0.95, 0.4438917279327484}], 
                RectangleBox[{1., 0.5748441088851295}, \
{1.95, 0.5867488707898912}], 
                RectangleBox[{2., 0.7177012517422724}, \
{2.95, 0.7296060136470344}], 
                RectangleBox[{3., 0.860558394599415}, \
{3.95, 0.872463156504177}], 
                RectangleBox[{4., 1.003415537456558}, \
{4.95, 1.0153202993613202}]},
               Antialiasing->True]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"5\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 108, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 108, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{159.8046875, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[{
          SoundNote[0], 
          SoundNote[12], 
          SoundNote[24], 
          SoundNote[36], 
          SoundNote[48]}]]], "ExerciseOutput",ExpressionUUID->
       "077ce984-3364-44cb-945d-d100da1968e4"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{
                RGBColor[
                0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                RectangleBox[{0, 0}, {360, -164}]}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{
                    RGBColor[0.5, 0, 0.5], 
                    StyleBox[{{
                    
                    RectangleBox[{0., 0.43198696602798653`}, {0.95, 
                    0.4438917279327484}], 
                    
                    RectangleBox[{1., 0.5748441088851295}, {1.95, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{2., 0.7177012517422724}, {2.95, 
                    0.7296060136470344}], 
                    
                    RectangleBox[{3., 0.860558394599415}, {3.95, 
                    0.872463156504177}], 
                    
                    RectangleBox[{4., 1.003415537456558}, {4.95, 
                    1.0153202993613202`}]}}, Antialiasing -> True]}}}}}}, 
               AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{
                   GrayLevel[0.6], 
                   RectangleBox[{0, 0}]}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"5\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{5, -158}, {29, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 PolygonBox[
                  NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 96, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 108, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 108, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{29, -158}, {53, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 RectangleBox[{35, -152}, {47, -141}], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{
                GrayLevel[0.75], 
                StyleBox[
                 LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}], 
                 Antialiasing -> False]}}}}}, ContentSelectable -> False, 
           ImageSize -> {159.8046875, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[{
            SoundNote[0], 
            SoundNote[12], 
            SoundNote[24], 
            SoundNote[36], 
            SoundNote[48]}]]], "ExerciseOutput", Selectable -> False, 
        Copyable -> False, Editable -> False, ShowCellBracket -> False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"87dc6a09-a089-4ed1-a893-2238968a3920"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Generate a sequence of 10 notes on a trumpet with random pitches from 0 to \
12 and duration 0.2 seconds.",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "12.6", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"25f5589c-f940-4535-87b8-3508ffa6d85e"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Sample", " ", "expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[
             {Hue[0.0027707033834458716`, 0.9, 0.7], 
              
              StyleBox[{
                RectangleBox[{0.4, 0.43198696602798653}, \
{0.59, 0.4438917279327484}], 
                RectangleBox[{1.2, 0.43198696602798653}, \
{1.39, 0.4438917279327484}], 
                RectangleBox[{0.6, 0.45579648983751037}, \
{0.79, 0.4677012517422721}], 
                RectangleBox[{0., 0.4677012517422721}, \
{0.19, 0.4796060136470343}], 
                RectangleBox[{1.4, 0.49151077555179606}, \
{1.59, 0.5034155374565581}], 
                RectangleBox[{1.8, 0.49151077555179606}, \
{1.99, 0.5034155374565581}], 
                RectangleBox[{1., 0.5034155374565581}, \
{1.19, 0.5153202993613198}], 
                RectangleBox[{0.8, 0.5153202993613198}, \
{0.99, 0.5272250612660818}], 
                RectangleBox[{0.2, 0.5510345850756055}, \
{0.39, 0.5629393469803675}], 
                RectangleBox[{1.6, 0.5510345850756055}, \
{1.79, 0.5629393469803675}]},
               Antialiasing->True]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"2\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 56], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    240, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    288, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    288, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    336, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    336, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    384, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    384, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    432, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    432, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    480, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{159.5078125, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[{
          SoundNote[3, 0.2, "Trumpet"], 
          SoundNote[10, 0.2, "Trumpet"], 
          SoundNote[0, 0.2, "Trumpet"], 
          SoundNote[2, 0.2, "Trumpet"], 
          SoundNote[7, 0.2, "Trumpet"], 
          SoundNote[6, 0.2, "Trumpet"], 
          SoundNote[0, 0.2, "Trumpet"], 
          SoundNote[5, 0.2, "Trumpet"], 
          SoundNote[10, 0.2, "Trumpet"], 
          SoundNote[5, 0.2, "Trumpet"]}]]], "ExerciseOutput",ExpressionUUID->
       "c7a877ae-ed07-4223-acc4-38c6e4d5982d"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{
                RGBColor[
                0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                RectangleBox[{0, 0}, {360, -164}]}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{
                    Hue[0.0027707033834458716`, 0.9, 0.7], 
                    StyleBox[{{
                    
                    RectangleBox[{0.4, 0.43198696602798653`}, {0.59, 
                    0.4438917279327484}], 
                    
                    RectangleBox[{1.2, 0.43198696602798653`}, {1.39, 
                    0.4438917279327484}], 
                    
                    RectangleBox[{0.6, 0.45579648983751037`}, {0.79, 
                    0.4677012517422721}], 
                    
                    RectangleBox[{0., 0.4677012517422721}, {0.19, 
                    0.4796060136470343}], 
                    
                    RectangleBox[{1.4, 0.49151077555179606`}, {1.59, 
                    0.5034155374565581}], 
                    
                    RectangleBox[{1.8, 0.49151077555179606`}, {1.99, 
                    0.5034155374565581}], 
                    
                    RectangleBox[{1., 0.5034155374565581}, {1.19, 
                    0.5153202993613198}], 
                    
                    RectangleBox[{0.8, 0.5153202993613198}, {0.99, 
                    0.5272250612660818}], 
                    
                    RectangleBox[{0.2, 0.5510345850756055}, {0.39, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{1.6, 0.5510345850756055}, {1.79, 
                    0.5629393469803675}]}}, Antialiasing -> True]}}}}}}, 
               AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{
                   GrayLevel[0.6], 
                   RectangleBox[{0, 0}]}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"2\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{5, -158}, {29, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 PolygonBox[
                  NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 56], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    240, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    288, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    288, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    336, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    336, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    384, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    384, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    432, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    432, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    480, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{29, -158}, {53, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 RectangleBox[{35, -152}, {47, -141}], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{
                GrayLevel[0.75], 
                StyleBox[
                 LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}], 
                 Antialiasing -> False]}}}}}, ContentSelectable -> False, 
           ImageSize -> {159.5078125, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[{
            SoundNote[3, 0.2, "Trumpet"], 
            SoundNote[10, 0.2, "Trumpet"], 
            SoundNote[0, 0.2, "Trumpet"], 
            SoundNote[2, 0.2, "Trumpet"], 
            SoundNote[7, 0.2, "Trumpet"], 
            SoundNote[6, 0.2, "Trumpet"], 
            SoundNote[0, 0.2, "Trumpet"], 
            SoundNote[5, 0.2, "Trumpet"], 
            SoundNote[10, 0.2, "Trumpet"], 
            SoundNote[5, 0.2, "Trumpet"]}]]], "ExerciseOutput", Selectable -> 
        False, Copyable -> False, Editable -> False, ShowCellBracket -> 
        False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"efdad4b1-7621-4b64-a190-7fddf763cb8b"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Generate a sequence of 10 notes with random pitches up to 12 and random \
durations up to 10 tenths of a second.",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "12.7", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"034286b6-c669-41d7-898c-6136d12b0e2d"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Sample", " ", "expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[
             {RGBColor[0.5, 0, 0.5], 
              
              StyleBox[{
                RectangleBox[{4.5, 0.43198696602798653}, \
{4.880000000000001, 0.4438917279327484}], 
                RectangleBox[{3.2, 0.4677012517422721}, \
{4.15, 0.4796060136470343}], 
                RectangleBox[{2.2, 0.4796060136470343}, \
{2.96, 0.49151077555179606}], 
                RectangleBox[{0., 0.49151077555179606}, \
{0.6649999999999999, 0.5034155374565581}], 
                RectangleBox[{3., 0.5034155374565581}, \
{3.1900000000000004, 0.5153202993613198}], 
                RectangleBox[{4.3, 0.5034155374565581}, \
{4.49, 0.5153202993613198}], 
                RectangleBox[{4.2, 0.5153202993613198}, \
{4.295, 0.5272250612660818}], 
                RectangleBox[{0.7, 0.5272250612660818}, \
{1.175, 0.5391298231708438}], 
                RectangleBox[{1.2, 0.5391298231708438}, \
{1.39, 0.5510345850756055}], 
                RectangleBox[{1.4, 0.5510345850756055}, \
{2.16, 0.5629393469803675}]},
               Antialiasing->True]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"4.9\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    168, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    168, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    288, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    288, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    336, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    336, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    528, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    528, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    720, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    720, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    768, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    768, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1008, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1008, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1032, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1032, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1080, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1080, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1176, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{159.69921875, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[{
          SoundNote[5, 
           Rational[7, 10]], 
          SoundNote[8, 
           Rational[1, 2]], 
          SoundNote[9, 
           Rational[1, 5]], 
          SoundNote[10, 
           Rational[4, 5]], 
          SoundNote[4, 
           Rational[4, 5]], 
          SoundNote[6, 
           Rational[1, 5]], 
          SoundNote[3, 1], 
          SoundNote[7, 
           Rational[1, 10]], 
          SoundNote[6, 
           Rational[1, 5]], 
          SoundNote[0, 
           Rational[2, 5]]}]]], "ExerciseOutput",ExpressionUUID->
       "1a97d3c2-9df8-4eaa-8abb-53f1cf27a3b6"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{0, 0}, {360, -164}]}}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{{
                    RGBColor[0.5, 0, 0.5], 
                    StyleBox[{{
                    
                    RectangleBox[{4.5, 0.43198696602798653`}, {
                    4.880000000000001, 0.4438917279327484}], 
                    
                    RectangleBox[{3.2, 0.4677012517422721}, {4.15, 
                    0.4796060136470343}], 
                    
                    RectangleBox[{2.2, 0.4796060136470343}, {2.96, 
                    0.49151077555179606`}], 
                    
                    RectangleBox[{0., 0.49151077555179606`}, {
                    0.6649999999999999, 0.5034155374565581}], 
                    
                    RectangleBox[{3., 0.5034155374565581}, {
                    3.1900000000000004`, 0.5153202993613198}], 
                    
                    RectangleBox[{4.3, 0.5034155374565581}, {4.49, 
                    0.5153202993613198}], 
                    
                    RectangleBox[{4.2, 0.5153202993613198}, {4.295, 
                    0.5272250612660818}], 
                    
                    RectangleBox[{0.7, 0.5272250612660818}, {1.175, 
                    0.5391298231708438}], 
                    
                    RectangleBox[{1.2, 0.5391298231708438}, {1.39, 
                    0.5510345850756055}], 
                    
                    RectangleBox[{1.4, 0.5510345850756055}, {2.16, 
                    0.5629393469803675}]}}, Antialiasing -> True]}}}}}}}, 
               AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{{
                    GrayLevel[0.6], 
                    RectangleBox[{0, 0}]}}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"4.9\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{{
                  RGBColor[
                  0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                  RectangleBox[{5, -158}, {29, -135}]}}}}, {{{{
                  RGBColor[
                  0.27450980392156865`, 0.5411764705882353, 
                   0.796078431372549], 
                  PolygonBox[
                   NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                  StyleBox[{}, Antialiasing -> False]}}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    168, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    168, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    288, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    288, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    336, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    336, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    528, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    528, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    720, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    720, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    768, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    768, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1008, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1008, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1032, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1032, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1080, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1080, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1176, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{{
                  RGBColor[
                  0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                  RectangleBox[{29, -158}, {53, -135}]}}}}, {{{{
                  RGBColor[
                  0.27450980392156865`, 0.5411764705882353, 
                   0.796078431372549], 
                  RectangleBox[{35, -152}, {47, -141}], 
                  StyleBox[{}, Antialiasing -> False]}}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{{
                 GrayLevel[0.75], 
                 StyleBox[
                  LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
                   Antialiasing -> False]}}}}}}, ContentSelectable -> False, 
           ImageSize -> {159.69921875, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[{
            SoundNote[5, 
             Rational[7, 10]], 
            SoundNote[8, 
             Rational[1, 2]], 
            SoundNote[9, 
             Rational[1, 5]], 
            SoundNote[10, 
             Rational[4, 5]], 
            SoundNote[4, 
             Rational[4, 5]], 
            SoundNote[6, 
             Rational[1, 5]], 
            SoundNote[3, 1], 
            SoundNote[7, 
             Rational[1, 10]], 
            SoundNote[6, 
             Rational[1, 5]], 
            SoundNote[0, 
             Rational[2, 5]]}]]], "ExerciseOutput", Selectable -> False, 
        Copyable -> False, Editable -> False, ShowCellBracket -> False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"035603dc-4464-4be3-8cf6-60ff19437d04"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Generate 0.1-second notes with pitches given by the digits of 2^31.",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "12.8", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"ca3efd76-19c3-4700-867b-e5c417fa53e3"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[
             {RGBColor[0.5, 0, 0.5], 
              
              StyleBox[{
                RectangleBox[{0.1, 0.4438917279327484}, \
{0.195, 0.45579648983751037}], 
                RectangleBox[{0., 0.45579648983751037}, \
{0.095, 0.4677012517422721}], 
                RectangleBox[{0.6, 0.4677012517422721}, \
{0.695, 0.4796060136470343}], 
                RectangleBox[{0.2, 0.4796060136470343}, \
{0.295, 0.49151077555179606}], 
                RectangleBox[{0.4, 0.4796060136470343}, \
{0.495, 0.49151077555179606}], 
                RectangleBox[{0.8, 0.4796060136470343}, \
{0.895, 0.49151077555179606}], 
                RectangleBox[{0.7, 0.5034155374565581}, \
{0.795, 0.5153202993613198}], 
                RectangleBox[{0.3, 0.5153202993613198}, \
{0.395, 0.5272250612660818}], 
                RectangleBox[{0.5, 0.5272250612660818}, \
{0.595, 0.5391298231708438}], 
                RectangleBox[{0.9, 0.5272250612660818}, \
{0.995, 0.5391298231708438}]},
               Antialiasing->True]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"1\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    24, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    24, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    72, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    72, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    120, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    120, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    168, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    168, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    216, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    216, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{159.98828125, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[{
          SoundNote[2, 0.1], 
          SoundNote[1, 0.1], 
          SoundNote[4, 0.1], 
          SoundNote[7, 0.1], 
          SoundNote[4, 0.1], 
          SoundNote[8, 0.1], 
          SoundNote[3, 0.1], 
          SoundNote[6, 0.1], 
          SoundNote[4, 0.1], 
          SoundNote[8, 0.1]}]]], "ExerciseOutput",ExpressionUUID->
       "9e131bdf-cfa1-4447-9b55-6632b23f14bd"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{
                RGBColor[
                0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                RectangleBox[{0, 0}, {360, -164}]}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{
                    RGBColor[0.5, 0, 0.5], 
                    StyleBox[{{
                    
                    RectangleBox[{0.1, 0.4438917279327484}, {0.195, 
                    0.45579648983751037`}], 
                    
                    RectangleBox[{0., 0.45579648983751037`}, {0.095, 
                    0.4677012517422721}], 
                    
                    RectangleBox[{0.6, 0.4677012517422721}, {0.695, 
                    0.4796060136470343}], 
                    
                    RectangleBox[{0.2, 0.4796060136470343}, {0.295, 
                    0.49151077555179606`}], 
                    
                    RectangleBox[{0.4, 0.4796060136470343}, {0.495, 
                    0.49151077555179606`}], 
                    
                    RectangleBox[{0.8, 0.4796060136470343}, {0.895, 
                    0.49151077555179606`}], 
                    
                    RectangleBox[{0.7, 0.5034155374565581}, {0.795, 
                    0.5153202993613198}], 
                    
                    RectangleBox[{0.3, 0.5153202993613198}, {0.395, 
                    0.5272250612660818}], 
                    
                    RectangleBox[{0.5, 0.5272250612660818}, {0.595, 
                    0.5391298231708438}], 
                    
                    RectangleBox[{0.9, 0.5272250612660818}, {0.995, 
                    0.5391298231708438}]}}, Antialiasing -> True]}}}}}}, 
               AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{
                   GrayLevel[0.6], 
                   RectangleBox[{0, 0}]}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"1\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{5, -158}, {29, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 PolygonBox[
                  NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    24, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    24, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    72, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    72, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    120, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    120, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    168, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    168, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    216, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    216, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{29, -158}, {53, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 RectangleBox[{35, -152}, {47, -141}], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{
                GrayLevel[0.75], 
                StyleBox[
                 LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}], 
                 Antialiasing -> False]}}}}}, ContentSelectable -> False, 
           ImageSize -> {159.98828125, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[{
            SoundNote[2, 0.1], 
            SoundNote[1, 0.1], 
            SoundNote[4, 0.1], 
            SoundNote[7, 0.1], 
            SoundNote[4, 0.1], 
            SoundNote[8, 0.1], 
            SoundNote[3, 0.1], 
            SoundNote[6, 0.1], 
            SoundNote[4, 0.1], 
            SoundNote[8, 0.1]}]]], "ExerciseOutput", Selectable -> False, 
        Copyable -> False, Editable -> False, ShowCellBracket -> False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"184b3ccd-317b-4b99-8dfa-378bdc480ce3"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Create a sound from the letters in CABBAGE, each playing for 0.3 seconds \
sounding like a guitar.",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "12.9", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"1ebccbe9-b4ca-4a8a-b364-fb6d48578200"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[
             {Hue[0.4073186577785305, 0.9, 0.7], 
              
              StyleBox[{
                RectangleBox[{0., 0.43198696602798653}, \
{0.285, 0.4438917279327484}], 
                RectangleBox[{1.8, 0.4796060136470343}, \
{2.085, 0.49151077555179606}], 
                RectangleBox[{1.5, 0.5153202993613198}, \
{1.7850000000000001, 0.5272250612660818}], 
                RectangleBox[{0.3, 0.5391298231708438}, \
{0.585, 0.5510345850756055}], 
                RectangleBox[{1.2, 0.5391298231708438}, \
{1.485, 0.5510345850756055}], 
                RectangleBox[{0.6, 0.5629393469803675}, \
{0.885, 0.5748441088851295}], 
                RectangleBox[{0.9, 0.5629393469803675}, \
{1.185, 0.5748441088851295}]},
               Antialiasing->True]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"2.1\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 24], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    72, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    72, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    216, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    216, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    288, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    288, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    360, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    360, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    432, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    432, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    504, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{160.21484375, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[{
          SoundNote["C", 0.3, "Guitar"], 
          SoundNote["A", 0.3, "Guitar"], 
          SoundNote["B", 0.3, "Guitar"], 
          SoundNote["B", 0.3, "Guitar"], 
          SoundNote["A", 0.3, "Guitar"], 
          SoundNote["G", 0.3, "Guitar"], 
          SoundNote["E", 0.3, "Guitar"]}]]], "ExerciseOutput",ExpressionUUID->
       "d1c22fb3-f3f8-44c9-ac29-a612fe29120c"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{
                RGBColor[
                0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                RectangleBox[{0, 0}, {360, -164}]}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{
                    Hue[0.4073186577785305, 0.9, 0.7], 
                    StyleBox[{{
                    
                    RectangleBox[{0., 0.43198696602798653`}, {0.285, 
                    0.4438917279327484}], 
                    
                    RectangleBox[{1.8, 0.4796060136470343}, {2.085, 
                    0.49151077555179606`}], 
                    
                    RectangleBox[{1.5, 0.5153202993613198}, {
                    1.7850000000000001`, 0.5272250612660818}], 
                    
                    RectangleBox[{0.3, 0.5391298231708438}, {0.585, 
                    0.5510345850756055}], 
                    
                    RectangleBox[{1.2, 0.5391298231708438}, {1.485, 
                    0.5510345850756055}], 
                    
                    RectangleBox[{0.6, 0.5629393469803675}, {0.885, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{0.9, 0.5629393469803675}, {1.185, 
                    0.5748441088851295}]}}, Antialiasing -> True]}}}}}}, 
               AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{
                   GrayLevel[0.6], 
                   RectangleBox[{0, 0}]}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"2.1\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{5, -158}, {29, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 PolygonBox[
                  NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 24], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    72, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    72, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    216, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    216, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    288, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    288, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    360, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    360, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    432, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    432, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    504, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{29, -158}, {53, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 RectangleBox[{35, -152}, {47, -141}], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{
                GrayLevel[0.75], 
                StyleBox[
                 LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}], 
                 Antialiasing -> False]}}}}}, ContentSelectable -> False, 
           ImageSize -> {160.21484375, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[{
            SoundNote["C", 0.3, "Guitar"], 
            SoundNote["A", 0.3, "Guitar"], 
            SoundNote["B", 0.3, "Guitar"], 
            SoundNote["B", 0.3, "Guitar"], 
            SoundNote["A", 0.3, "Guitar"], 
            SoundNote["G", 0.3, "Guitar"], 
            SoundNote["E", 0.3, "Guitar"]}]]], "ExerciseOutput", Selectable -> 
        False, Copyable -> False, Editable -> False, ShowCellBracket -> 
        False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"bc646827-b439-47a7-b9a0-216c6db3af38"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Generate 0.1-second notes with pitches given by the letter numbers of the \
characters in \[OpenCurlyDoubleQuote]wolfram\[CloseCurlyDoubleQuote].",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "12.10", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"3c71a8d6-cf58-43e7-8239-7dd5a46b5e8e"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[
             {RGBColor[0.5, 0, 0.5], 
              
              StyleBox[{
                RectangleBox[{0.5, 0.4438917279327484}, \
{0.595, 0.45579648983751037}], 
                RectangleBox[{0.3, 0.5034155374565581}, \
{0.395, 0.5153202993613198}], 
                RectangleBox[{0.2, 0.5748441088851295}, \
{0.295, 0.5867488707898912}], 
                RectangleBox[{0.6, 0.5867488707898912}, \
{0.695, 0.5986536326946532}], 
                RectangleBox[{0.1, 0.610558394599415}, \
{0.195, 0.6224631565041769}], 
                RectangleBox[{0.4, 0.646272680313701}, \
{0.495, 0.658177442218463}], 
                RectangleBox[{0., 0.7057964898375104}, \
{0.095, 0.7177012517422724}]},
               Antialiasing->True]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"0.7\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    24, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    24, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    72, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    72, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    120, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    120, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    168, "NoteOff", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{160.36328125, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[{
          SoundNote[23, 0.1], 
          SoundNote[15, 0.1], 
          SoundNote[12, 0.1], 
          SoundNote[6, 0.1], 
          SoundNote[18, 0.1], 
          SoundNote[1, 0.1], 
          SoundNote[13, 0.1]}]]], "ExerciseOutput",ExpressionUUID->
       "666ed3eb-8e66-4aaf-aa16-2f896b4a30cb"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{
                RGBColor[
                0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                RectangleBox[{0, 0}, {360, -164}]}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{
                    RGBColor[0.5, 0, 0.5], 
                    StyleBox[{{
                    
                    RectangleBox[{0.5, 0.4438917279327484}, {0.595, 
                    0.45579648983751037`}], 
                    
                    RectangleBox[{0.3, 0.5034155374565581}, {0.395, 
                    0.5153202993613198}], 
                    
                    RectangleBox[{0.2, 0.5748441088851295}, {0.295, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{0.6, 0.5867488707898912}, {0.695, 
                    0.5986536326946532}], 
                    
                    RectangleBox[{0.1, 0.610558394599415}, {0.195, 
                    0.6224631565041769}], 
                    
                    RectangleBox[{0.4, 0.646272680313701}, {0.495, 
                    0.658177442218463}], 
                    
                    RectangleBox[{0., 0.7057964898375104}, {0.095, 
                    0.7177012517422724}]}}, Antialiasing -> True]}}}}}}, 
               AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{
                   GrayLevel[0.6], 
                   RectangleBox[{0, 0}]}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"0.7\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{5, -158}, {29, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 PolygonBox[
                  NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    24, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    24, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    72, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    72, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    120, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    120, "NoteOn", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 61, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    168, "NoteOff", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 240]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{29, -158}, {53, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 RectangleBox[{35, -152}, {47, -141}], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{
                GrayLevel[0.75], 
                StyleBox[
                 LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}], 
                 Antialiasing -> False]}}}}}, ContentSelectable -> False, 
           ImageSize -> {160.36328125, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[{
            SoundNote[23, 0.1], 
            SoundNote[15, 0.1], 
            SoundNote[12, 0.1], 
            SoundNote[6, 0.1], 
            SoundNote[18, 0.1], 
            SoundNote[1, 0.1], 
            SoundNote[13, 0.1]}]]], "ExerciseOutput", Selectable -> False, 
        Copyable -> False, Editable -> False, ShowCellBracket -> False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"84a11213-9c37-408f-a0bb-e7d75177aac0"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Generate a sequence of three notes of 1 second of playing middle D on the \
cello, piano and guitar. ",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "+12.1", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"df9103fb-6189-453e-8aad-baa31af51787"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[{
              {RGBColor[0.5, 0, 0.5], 
               StyleBox[RectangleBox[{1., 0.45579648983751037}, {1.95, 0.4677012517422721}],
                Antialiasing->True]}, 
              {Hue[0.4073186577785305, 0.9, 0.7], 
               StyleBox[RectangleBox[{2., 0.45579648983751037}, {2.95, 0.4677012517422721}],
                Antialiasing->True]}, 
              {Hue[0.22538543343129763`, 0.9, 0.7], 
               StyleBox[RectangleBox[{0., 0.45579648983751037}, {0.95, 0.4677012517422721}],
                Antialiasing->True]}},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"3\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    0]}], 
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 1, "Value" -> 24], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 62, "Channel" -> 1, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 62, "Channel" -> 1, "Velocity" -> 
                    0]}], 
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 2, "Value" -> 42], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 62, "Channel" -> 2, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 62, "Channel" -> 2, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{159.87890625, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[{
          SoundNote["D", 1, "Cello"], 
          SoundNote["D", 1, "Piano"], 
          SoundNote["D", 1, "Guitar"]}]]], "ExerciseOutput",ExpressionUUID->
       "5565e7b5-63ab-4a45-a850-71a23f33e412"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{0, 0}, {360, -164}]}}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{{
                    RGBColor[0.5, 0, 0.5], 
                    StyleBox[{{
                    
                    RectangleBox[{1., 0.45579648983751037`}, {1.95, 
                    0.4677012517422721}]}}, Antialiasing -> True]}}}}}, {{{{{
                    Hue[0.4073186577785305, 0.9, 0.7], 
                    StyleBox[{{
                    
                    RectangleBox[{2., 0.45579648983751037`}, {2.95, 
                    0.4677012517422721}]}}, Antialiasing -> True]}}}}}, {{{{{
                    Hue[0.22538543343129763`, 0.9, 0.7], 
                    StyleBox[{{
                    
                    RectangleBox[{0., 0.45579648983751037`}, {0.95, 
                    0.4677012517422721}]}}, Antialiasing -> True]}}}}}}}, 
               AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{{
                    GrayLevel[0.6], 
                    RectangleBox[{0, 0}]}}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"3\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{{
                  RGBColor[
                  0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                  RectangleBox[{5, -158}, {29, -135}]}}}}, {{{{
                  RGBColor[
                  0.27450980392156865`, 0.5411764705882353, 
                   0.796078431372549], 
                  PolygonBox[
                   NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                  StyleBox[{}, Antialiasing -> False]}}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 62, "Channel" -> 0, "Velocity" -> 
                    0]}], 
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 1, "Value" -> 24], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 62, "Channel" -> 1, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 62, "Channel" -> 1, "Velocity" -> 
                    0]}], 
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 2, "Value" -> 42], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 62, "Channel" -> 2, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 62, "Channel" -> 2, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{{
                  RGBColor[
                  0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                  RectangleBox[{29, -158}, {53, -135}]}}}}, {{{{
                  RGBColor[
                  0.27450980392156865`, 0.5411764705882353, 
                   0.796078431372549], 
                  RectangleBox[{35, -152}, {47, -141}], 
                  StyleBox[{}, Antialiasing -> False]}}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{{
                 GrayLevel[0.75], 
                 StyleBox[
                  LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
                   Antialiasing -> False]}}}}}}, ContentSelectable -> False, 
           ImageSize -> {159.87890625, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[{
            SoundNote["D", 1, "Cello"], 
            SoundNote["D", 1, "Piano"], 
            SoundNote["D", 1, "Guitar"]}]]], "ExerciseOutput", Selectable -> 
        False, Copyable -> False, Editable -> False, ShowCellBracket -> 
        False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"77cf0aea-9781-49ff-b3c1-1699e08e71f0"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Generate a sequence of notes from pitch 0 to pitch 12, going up in steps of \
3. ",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "+12.2", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"855cfe32-1eb8-4e14-9a88-0b5192659ab6"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[
             {RGBColor[0.5, 0, 0.5], 
              
              StyleBox[{
                RectangleBox[{0., 0.43198696602798653}, \
{0.95, 0.4438917279327484}], 
                RectangleBox[{1., 0.4677012517422721}, \
{1.95, 0.4796060136470343}], 
                RectangleBox[{2., 0.5034155374565581}, \
{2.95, 0.5153202993613198}], 
                RectangleBox[{3., 0.5391298231708438}, \
{3.95, 0.5510345850756055}], 
                RectangleBox[{4., 0.5748441088851295}, \
{4.95, 0.5867488707898912}]},
               Antialiasing->True]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"5\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{159.69140625, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[{
          SoundNote[0], 
          SoundNote[3], 
          SoundNote[6], 
          SoundNote[9], 
          SoundNote[12]}]]], "ExerciseOutput",ExpressionUUID->
       "acc3b997-9025-41cc-b31b-341c15864001"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{0, 0}, {360, -164}]}}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{{
                    RGBColor[0.5, 0, 0.5], 
                    StyleBox[{{
                    
                    RectangleBox[{0., 0.43198696602798653`}, {0.95, 
                    0.4438917279327484}], 
                    
                    RectangleBox[{1., 0.4677012517422721}, {1.95, 
                    0.4796060136470343}], 
                    
                    RectangleBox[{2., 0.5034155374565581}, {2.95, 
                    0.5153202993613198}], 
                    
                    RectangleBox[{3., 0.5391298231708438}, {3.95, 
                    0.5510345850756055}], 
                    
                    RectangleBox[{4., 0.5748441088851295}, {4.95, 
                    0.5867488707898912}]}}, Antialiasing -> True]}}}}}}}, 
               AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{{
                    GrayLevel[0.6], 
                    RectangleBox[{0, 0}]}}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"5\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{{
                  RGBColor[
                  0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                  RectangleBox[{5, -158}, {29, -135}]}}}}, {{{{
                  RGBColor[
                  0.27450980392156865`, 0.5411764705882353, 
                   0.796078431372549], 
                  PolygonBox[
                   NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                  StyleBox[{}, Antialiasing -> False]}}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{{
                  RGBColor[
                  0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                  RectangleBox[{29, -158}, {53, -135}]}}}}, {{{{
                  RGBColor[
                  0.27450980392156865`, 0.5411764705882353, 
                   0.796078431372549], 
                  RectangleBox[{35, -152}, {47, -141}], 
                  StyleBox[{}, Antialiasing -> False]}}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{{
                 GrayLevel[0.75], 
                 StyleBox[
                  LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
                   Antialiasing -> False]}}}}}}, ContentSelectable -> False, 
           ImageSize -> {159.69140625, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[{
            SoundNote[0], 
            SoundNote[3], 
            SoundNote[6], 
            SoundNote[9], 
            SoundNote[12]}]]], "ExerciseOutput", Selectable -> False, 
        Copyable -> False, Editable -> False, ShowCellBracket -> False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"7a4816c5-b844-42a3-b44c-33e0ee1d3c48"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Generate a sequence of 5 notes starting with middle C, then successively \
going up by a perfect fifth (7 semitones) at a time. ",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "+12.3", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"7530991b-48c9-4a9e-b63d-f7034b186dbc"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[
             {RGBColor[0.5, 0, 0.5], 
              
              StyleBox[{
                RectangleBox[{0., 0.43198696602798653}, \
{0.95, 0.4438917279327484}], 
                RectangleBox[{1., 0.5153202993613198}, \
{1.95, 0.5272250612660818}], 
                RectangleBox[{2., 0.5986536326946532}, \
{2.95, 0.610558394599415}], 
                RectangleBox[{3., 0.6819869660279864}, \
{3.95, 0.6938917279327484}], 
                RectangleBox[{4., 0.7653202993613198}, \
{4.95, 0.7772250612660818}]},
               Antialiasing->True]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"5\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 74, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 74, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 88, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 88, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{159.734375, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[{
          SoundNote[0], 
          SoundNote[7], 
          SoundNote[14], 
          SoundNote[21], 
          SoundNote[28]}]]], "ExerciseOutput",ExpressionUUID->
       "75719b87-ba22-4ff8-aa25-cca032dbcc46"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{0, 0}, {360, -164}]}}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{{
                    RGBColor[0.5, 0, 0.5], 
                    StyleBox[{{
                    
                    RectangleBox[{0., 0.43198696602798653`}, {0.95, 
                    0.4438917279327484}], 
                    
                    RectangleBox[{1., 0.5153202993613198}, {1.95, 
                    0.5272250612660818}], 
                    
                    RectangleBox[{2., 0.5986536326946532}, {2.95, 
                    0.610558394599415}], 
                    
                    RectangleBox[{3., 0.6819869660279864}, {3.95, 
                    0.6938917279327484}], 
                    
                    RectangleBox[{4., 0.7653202993613198}, {4.95, 
                    0.7772250612660818}]}}, Antialiasing -> True]}}}}}}}, 
               AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{{
                    GrayLevel[0.6], 
                    RectangleBox[{0, 0}]}}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"5\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{{
                  RGBColor[
                  0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                  RectangleBox[{5, -158}, {29, -135}]}}}}, {{{{
                  RGBColor[
                  0.27450980392156865`, 0.5411764705882353, 
                   0.796078431372549], 
                  PolygonBox[
                   NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                  StyleBox[{}, Antialiasing -> False]}}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 60, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 74, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 74, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 88, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 88, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 48]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{{
                  RGBColor[
                  0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                  RectangleBox[{29, -158}, {53, -135}]}}}}, {{{{
                  RGBColor[
                  0.27450980392156865`, 0.5411764705882353, 
                   0.796078431372549], 
                  RectangleBox[{35, -152}, {47, -141}], 
                  StyleBox[{}, Antialiasing -> False]}}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{{
                 GrayLevel[0.75], 
                 StyleBox[
                  LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
                   Antialiasing -> False]}}}}}}, ContentSelectable -> False, 
           ImageSize -> {159.734375, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[{
            SoundNote[0], 
            SoundNote[7], 
            SoundNote[14], 
            SoundNote[21], 
            SoundNote[28]}]]], "ExerciseOutput", Selectable -> False, 
        Copyable -> False, Editable -> False, ShowCellBracket -> False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"5091d2a1-38ad-4bdf-b4fd-6af18d6fccf8"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Generate 0.02-second notes with pitches given by the lengths of the names \
of integers from 1 to 200.",
 StyleBox[ButtonBox["\[NonBreakingSpace]\[RightGuillemet]",
  Appearance->None,
  Evaluator->Automatic,
  Method->"Preemptive",
  ButtonFunction:>(SelectionMove[
     ButtonNotebook[], All, ButtonCell, AutoScroll -> False]; 
   FrontEndExecute[
     FrontEndToken[
      ButtonNotebook[], "OpenCloseGroup"]])],
  FontColor->RGBColor[1, 0.501961, 0]]
}], "Exercise",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellDingbat->Cell[
  "+12.4", "ExerciseNumber", FontColor -> 
   RGBColor[0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]],
 ShowCellBracket->
  False,ExpressionUUID->"6a6712ab-a311-4013-a552-591d8fbf3c6e"],

Cell[BoxData[
 PaneBox[GridBox[{
    {" ", 
     StyleBox[
      RowBox[{"Expected", " ", 
       RowBox[{"output", ":"}]}],
      Editable->False,
      Selectable->False,
      StripOnInput->False,
      FontColor->RGBColor[0.45, 0.45, 0.45]], " "},
    {" ", 
     InterpretationBox[Cell[BoxData[
       InterpretationBox[
        GraphicsBox[{
          {RGBColor[
           0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
           RectangleBox[{0, 0}, {360, -164}]}, {InsetBox[
            GraphicsBox[
             {RGBColor[0.5, 0, 0.5], 
              
              StyleBox[{
                RectangleBox[{0., 0.4677012517422721}, \
{0.019, 0.4796060136470343}], 
                RectangleBox[{0.02, 0.4677012517422721}, \
{0.039, 0.4796060136470343}], 
                RectangleBox[{0.1, 0.4677012517422721}, \
{0.119, 0.4796060136470343}], 
                RectangleBox[{0.18, 0.4677012517422721}, \
{0.199, 0.4796060136470343}], 
                RectangleBox[{0.06, 0.4796060136470343}, \
{0.079, 0.49151077555179606}], 
                RectangleBox[{0.08, 0.4796060136470343}, \
{0.099, 0.49151077555179606}], 
                RectangleBox[{0.16, 0.4796060136470343}, \
{0.179, 0.49151077555179606}], 
                RectangleBox[{0.04, 0.49151077555179606}, \
{0.059, 0.5034155374565581}], 
                RectangleBox[{0.12, 0.49151077555179606}, \
{0.139, 0.5034155374565581}], 
                RectangleBox[{0.14, 0.49151077555179606}, \
{0.159, 0.5034155374565581}], 
                RectangleBox[{0.78, 0.49151077555179606}, \
{0.799, 0.5034155374565581}], 
                RectangleBox[{0.98, 0.49151077555179606}, \
{0.999, 0.5034155374565581}], 
                RectangleBox[{1.18, 0.49151077555179606}, \
{1.1989999999999998, 0.5034155374565581}], 
                RectangleBox[{0.2, 0.5034155374565581}, \
{0.219, 0.5153202993613198}], 
                RectangleBox[{0.22, 0.5034155374565581}, \
{0.239, 0.5153202993613198}], 
                RectangleBox[{0.38, 0.5034155374565581}, \
{0.399, 0.5153202993613198}], 
                RectangleBox[{0.58, 0.5034155374565581}, \
{0.599, 0.5153202993613198}], 
                RectangleBox[{1.58, 0.5034155374565581}, \
{1.5990000000000002, 0.5153202993613198}], 
                
                RectangleBox[{1.78, 0.5034155374565581}, \
{1.799, 0.5153202993613198}], 
                RectangleBox[{0.28, 0.5153202993613198}, \
{0.299, 0.5272250612660818}], 
                RectangleBox[{0.3, 0.5153202993613198}, \
{0.319, 0.5272250612660818}], 
                RectangleBox[{1.38, 0.5153202993613198}, \
{1.399, 0.5272250612660818}], 
                RectangleBox[{0.24, 0.5272250612660818}, \
{0.259, 0.5391298231708438}], 
                RectangleBox[{0.26, 0.5272250612660818}, \
{0.279, 0.5391298231708438}], 
                RectangleBox[{0.34, 0.5272250612660818}, \
{0.359, 0.5391298231708438}], 
                RectangleBox[{0.36, 0.5272250612660818}, \
{0.379, 0.5391298231708438}], 
                RectangleBox[{0.32, 0.5391298231708438}, \
{0.339, 0.5510345850756055}], 
                RectangleBox[{0.8, 0.5391298231708438}, \
{0.819, 0.5510345850756055}], 
                RectangleBox[{0.82, 0.5391298231708438}, \
{0.839, 0.5510345850756055}], 
                RectangleBox[{0.9, 0.5391298231708438}, \
{0.919, 0.5510345850756055}], 
                RectangleBox[{1., 0.5391298231708438}, \
{1.0190000000000001, 0.5510345850756055}], 
                RectangleBox[{1.02, 0.5391298231708438}, \
{1.0390000000000001, 0.5510345850756055}], 
                RectangleBox[{1.1, 0.5391298231708438}, \
{1.1190000000000002, 0.5510345850756055}], 
                RectangleBox[{1.2, 0.5391298231708438}, \
{1.2189999999999999, 0.5510345850756055}], 
                RectangleBox[{1.22, 0.5391298231708438}, \
{1.2389999999999999, 0.5510345850756055}], 
                RectangleBox[{1.3, 0.5391298231708438}, \
{1.319, 0.5510345850756055}], 
                RectangleBox[{0.4, 0.5510345850756055}, \
{0.419, 0.5629393469803675}], 
                RectangleBox[{0.42, 0.5510345850756055}, \
{0.439, 0.5629393469803675}], 
                RectangleBox[{0.5, 0.5510345850756055}, \
{0.519, 0.5629393469803675}], 
                RectangleBox[{0.6, 0.5510345850756055}, \
{0.619, 0.5629393469803675}], 
                RectangleBox[{0.62, 0.5510345850756055}, \
{0.639, 0.5629393469803675}], 
                RectangleBox[{0.7, 0.5510345850756055}, \
{0.719, 0.5629393469803675}], 
                RectangleBox[{0.86, 0.5510345850756055}, \
{0.879, 0.5629393469803675}], 
                RectangleBox[{0.88, 0.5510345850756055}, \
{0.899, 0.5629393469803675}], 
                RectangleBox[{0.96, 0.5510345850756055}, \
{0.979, 0.5629393469803675}], 
                RectangleBox[{1.06, 0.5510345850756055}, \
{1.0790000000000002, 0.5629393469803675}], 
                RectangleBox[{1.08, 0.5510345850756055}, \
{1.0990000000000002, 0.5629393469803675}], 
                RectangleBox[{1.16, 0.5510345850756055}, \
{1.1789999999999998, 0.5629393469803675}], 
                RectangleBox[{1.26, 0.5510345850756055}, \
{1.279, 0.5629393469803675}], 
                RectangleBox[{1.28, 0.5510345850756055}, \
{1.299, 0.5629393469803675}], 
                RectangleBox[{1.36, 0.5510345850756055}, \
{1.379, 0.5629393469803675}], 
                RectangleBox[{1.6, 0.5510345850756055}, \
{1.6190000000000002, 0.5629393469803675}], 
                RectangleBox[{1.62, 0.5510345850756055}, \
{1.639, 0.5629393469803675}], 
                RectangleBox[{1.7, 0.5510345850756055}, \
{1.7189999999999999, 0.5629393469803675}], 
                RectangleBox[{1.8, 0.5510345850756055}, \
{1.819, 0.5629393469803675}], 
                RectangleBox[{1.82, 0.5510345850756055}, \
{1.839, 0.5629393469803675}], 
                RectangleBox[{1.9, 0.5510345850756055}, \
{1.919, 0.5629393469803675}], 
                RectangleBox[{0.46, 0.5629393469803675}, \
{0.479, 0.5748441088851295}], 
                RectangleBox[{0.48, 0.5629393469803675}, \
{0.499, 0.5748441088851295}], 
                RectangleBox[{0.56, 0.5629393469803675}, \
{0.579, 0.5748441088851295}], 
                RectangleBox[{0.66, 0.5629393469803675}, \
{0.679, 0.5748441088851295}], 
                RectangleBox[{0.68, 0.5629393469803675}, \
{0.699, 0.5748441088851295}], 
                RectangleBox[{0.76, 0.5629393469803675}, \
{0.779, 0.5748441088851295}], 
                RectangleBox[{0.84, 0.5629393469803675}, \
{0.859, 0.5748441088851295}], 
                RectangleBox[{0.92, 0.5629393469803675}, \
{0.939, 0.5748441088851295}], 
                RectangleBox[{0.94, 0.5629393469803675}, \
{0.959, 0.5748441088851295}], 
                RectangleBox[{1.04, 0.5629393469803675}, \
{1.0590000000000002, 0.5748441088851295}], 
                RectangleBox[{1.12, 0.5629393469803675}, \
{1.139, 0.5748441088851295}], 
                RectangleBox[{1.14, 0.5629393469803675}, \
{1.1589999999999998, 0.5748441088851295}], 
                RectangleBox[{1.24, 0.5629393469803675}, \
{1.259, 0.5748441088851295}], 
                RectangleBox[{1.32, 0.5629393469803675}, \
{1.339, 0.5748441088851295}], 
                RectangleBox[{1.34, 0.5629393469803675}, \
{1.359, 0.5748441088851295}], 
                RectangleBox[{1.4, 0.5629393469803675}, \
{1.419, 0.5748441088851295}], 
                RectangleBox[{1.42, 0.5629393469803675}, \
{1.439, 0.5748441088851295}], 
                RectangleBox[{1.5, 0.5629393469803675}, \
{1.5190000000000001, 0.5748441088851295}], 
                RectangleBox[{1.66, 0.5629393469803675}, \
{1.6789999999999998, 0.5748441088851295}], 
                RectangleBox[{1.68, 0.5629393469803675}, \
{1.6989999999999998, 0.5748441088851295}], 
                RectangleBox[{1.76, 0.5629393469803675}, \
{1.779, 0.5748441088851295}], 
                RectangleBox[{1.86, 0.5629393469803675}, \
{1.879, 0.5748441088851295}], 
                RectangleBox[{1.88, 0.5629393469803675}, \
{1.899, 0.5748441088851295}], 
                RectangleBox[{1.96, 0.5629393469803675}, \
{1.979, 0.5748441088851295}], 
                RectangleBox[{1.98, 0.5629393469803675}, \
{1.999, 0.5748441088851295}], 
                RectangleBox[{3.98, 0.5629393469803675}, \
{3.999, 0.5748441088851295}], 
                RectangleBox[{0.44, 0.5748441088851295}, \
{0.459, 0.5867488707898912}], 
                RectangleBox[{0.52, 0.5748441088851295}, \
{0.539, 0.5867488707898912}], 
                RectangleBox[{0.54, 0.5748441088851295}, \
{0.559, 0.5867488707898912}], 
                RectangleBox[{0.64, 0.5748441088851295}, \
{0.659, 0.5867488707898912}], 
                RectangleBox[{0.72, 0.5748441088851295}, \
{0.739, 0.5867488707898912}], 
                RectangleBox[{0.74, 0.5748441088851295}, \
{0.759, 0.5867488707898912}], 
                RectangleBox[{1.46, 0.5748441088851295}, \
{1.479, 0.5867488707898912}], 
                RectangleBox[{1.48, 0.5748441088851295}, \
{1.499, 0.5867488707898912}], 
                RectangleBox[{1.56, 0.5748441088851295}, \
{1.5790000000000002, 0.5867488707898912}], 
                RectangleBox[{1.64, 0.5748441088851295}, \
{1.6589999999999998, 0.5867488707898912}], 
                RectangleBox[{1.72, 0.5748441088851295}, \
{1.7389999999999999, 0.5867488707898912}], 
                RectangleBox[{1.74, 0.5748441088851295}, \
{1.759, 0.5867488707898912}], 
                RectangleBox[{1.84, 0.5748441088851295}, \
{1.859, 0.5867488707898912}], 
                RectangleBox[{1.92, 0.5748441088851295}, \
{1.939, 0.5867488707898912}], 
                RectangleBox[{1.94, 0.5748441088851295}, \
{1.959, 0.5867488707898912}], 
                RectangleBox[{1.44, 0.5867488707898912}, \
{1.459, 0.5986536326946532}], 
                RectangleBox[{1.52, 0.5867488707898912}, \
{1.5390000000000001, 0.5986536326946532}], 
                RectangleBox[{1.54, 0.5867488707898912}, \
{1.5590000000000002, 0.5986536326946532}], 
                RectangleBox[{2., 0.610558394599415}, \
{2.019, 0.6224631565041769}], 
                RectangleBox[{2.02, 0.610558394599415}, \
{2.039, 0.6224631565041769}], 
                RectangleBox[{2.1, 0.610558394599415}, \
{2.119, 0.6224631565041769}], 
                RectangleBox[{2.18, 0.610558394599415}, \
{2.1990000000000003, 0.6224631565041769}], 
                RectangleBox[{2.06, 0.6224631565041769}, \
{2.079, 0.6343679184089389}], 
                RectangleBox[{2.08, 0.6224631565041769}, \
{2.099, 0.6343679184089389}], 
                RectangleBox[{2.16, 0.6224631565041769}, \
{2.1790000000000003, 0.6343679184089389}], 
                RectangleBox[{2.04, 0.6343679184089389}, \
{2.059, 0.646272680313701}], 
                RectangleBox[{2.12, 0.6343679184089389}, \
{2.1390000000000002, 0.646272680313701}], 
                RectangleBox[{2.14, 0.6343679184089389}, \
{2.1590000000000003, 0.646272680313701}], 
                RectangleBox[{2.78, 0.6343679184089389}, \
{2.799, 0.646272680313701}], 
                RectangleBox[{2.98, 0.6343679184089389}, \
{2.999, 0.646272680313701}], 
                RectangleBox[{3.18, 0.6343679184089389}, \
{3.1990000000000003, 0.646272680313701}], 
                RectangleBox[{2.2, 0.646272680313701}, \
{2.2190000000000003, 0.658177442218463}], 
                RectangleBox[{2.22, 0.646272680313701}, \
{2.2390000000000003, 0.658177442218463}], 
                RectangleBox[{2.38, 0.646272680313701}, \
{2.399, 0.658177442218463}], 
                RectangleBox[{2.58, 0.646272680313701}, \
{2.599, 0.658177442218463}], 
                RectangleBox[{3.58, 0.646272680313701}, \
{3.599, 0.658177442218463}], 
                RectangleBox[{3.78, 0.646272680313701}, \
{3.799, 0.658177442218463}], 
                RectangleBox[{2.28, 0.658177442218463}, \
{2.299, 0.6700822041232247}], 
                RectangleBox[{2.3, 0.658177442218463}, \
{2.319, 0.6700822041232247}], 
                RectangleBox[{3.38, 0.658177442218463}, \
{3.399, 0.6700822041232247}], 
                RectangleBox[{2.24, 0.6700822041232247}, \
{2.259, 0.6819869660279864}], 
                RectangleBox[{2.26, 0.6700822041232247}, \
{2.279, 0.6819869660279864}], 
                RectangleBox[{2.34, 0.6700822041232247}, \
{2.359, 0.6819869660279864}], 
                RectangleBox[{2.36, 0.6700822041232247}, \
{2.379, 0.6819869660279864}], 
                RectangleBox[{2.32, 0.6819869660279864}, \
{2.339, 0.6938917279327484}], 
                RectangleBox[{2.8, 0.6819869660279864}, \
{2.819, 0.6938917279327484}], 
                RectangleBox[{2.82, 0.6819869660279864}, \
{2.839, 0.6938917279327484}], 
                RectangleBox[{2.9, 0.6819869660279864}, \
{2.919, 0.6938917279327484}], 
                RectangleBox[{3., 0.6819869660279864}, \
{3.019, 0.6938917279327484}], 
                RectangleBox[{3.02, 0.6819869660279864}, \
{3.039, 0.6938917279327484}], 
                RectangleBox[{3.1, 0.6819869660279864}, \
{3.119, 0.6938917279327484}], 
                RectangleBox[{3.2, 0.6819869660279864}, \
{3.2190000000000003, 0.6938917279327484}], 
                RectangleBox[{3.22, 0.6819869660279864}, \
{3.2390000000000003, 0.6938917279327484}], 
                RectangleBox[{3.3, 0.6819869660279864}, \
{3.319, 0.6938917279327484}], 
                RectangleBox[{2.4, 0.6938917279327484}, \
{2.419, 0.7057964898375104}], 
                RectangleBox[{2.42, 0.6938917279327484}, \
{2.439, 0.7057964898375104}], 
                RectangleBox[{2.5, 0.6938917279327484}, \
{2.519, 0.7057964898375104}], 
                RectangleBox[{2.6, 0.6938917279327484}, \
{2.619, 0.7057964898375104}], 
                RectangleBox[{2.62, 0.6938917279327484}, \
{2.6390000000000002, 0.7057964898375104}], 
                RectangleBox[{2.7, 0.6938917279327484}, \
{2.7190000000000003, 0.7057964898375104}], 
                RectangleBox[{2.86, 0.6938917279327484}, \
{2.879, 0.7057964898375104}], 
                RectangleBox[{2.88, 0.6938917279327484}, \
{2.899, 0.7057964898375104}], 
                RectangleBox[{2.96, 0.6938917279327484}, \
{2.979, 0.7057964898375104}], 
                RectangleBox[{3.06, 0.6938917279327484}, \
{3.079, 0.7057964898375104}], 
                RectangleBox[{3.08, 0.6938917279327484}, \
{3.099, 0.7057964898375104}], 
                RectangleBox[{3.16, 0.6938917279327484}, \
{3.1790000000000003, 0.7057964898375104}], 
                RectangleBox[{3.26, 0.6938917279327484}, \
{3.279, 0.7057964898375104}], 
                RectangleBox[{3.28, 0.6938917279327484}, \
{3.299, 0.7057964898375104}], 
                RectangleBox[{3.36, 0.6938917279327484}, \
{3.379, 0.7057964898375104}], 
                RectangleBox[{3.6, 0.6938917279327484}, \
{3.619, 0.7057964898375104}], 
                RectangleBox[{3.62, 0.6938917279327484}, \
{3.6390000000000002, 0.7057964898375104}], 
                RectangleBox[{3.7, 0.6938917279327484}, \
{3.7190000000000003, 0.7057964898375104}], 
                RectangleBox[{3.8, 0.6938917279327484}, \
{3.819, 0.7057964898375104}], 
                RectangleBox[{3.82, 0.6938917279327484}, \
{3.839, 0.7057964898375104}], 
                RectangleBox[{3.9, 0.6938917279327484}, \
{3.919, 0.7057964898375104}], 
                RectangleBox[{2.46, 0.7057964898375104}, \
{2.479, 0.7177012517422724}], 
                RectangleBox[{2.48, 0.7057964898375104}, \
{2.499, 0.7177012517422724}], 
                RectangleBox[{2.56, 0.7057964898375104}, \
{2.579, 0.7177012517422724}], 
                RectangleBox[{2.66, 0.7057964898375104}, \
{2.6790000000000003, 0.7177012517422724}], 
                RectangleBox[{2.68, 0.7057964898375104}, \
{2.6990000000000003, 0.7177012517422724}], 
                RectangleBox[{2.76, 0.7057964898375104}, \
{2.779, 0.7177012517422724}], 
                RectangleBox[{2.84, 0.7057964898375104}, \
{2.859, 0.7177012517422724}], 
                RectangleBox[{2.92, 0.7057964898375104}, \
{2.939, 0.7177012517422724}], 
                RectangleBox[{2.94, 0.7057964898375104}, \
{2.959, 0.7177012517422724}], 
                RectangleBox[{3.04, 0.7057964898375104}, \
{3.059, 0.7177012517422724}], 
                RectangleBox[{3.12, 0.7057964898375104}, \
{3.1390000000000002, 0.7177012517422724}], 
                RectangleBox[{3.14, 0.7057964898375104}, \
{3.1590000000000003, 0.7177012517422724}], 
                RectangleBox[{3.24, 0.7057964898375104}, \
{3.259, 0.7177012517422724}], 
                RectangleBox[{3.32, 0.7057964898375104}, \
{3.339, 0.7177012517422724}], 
                RectangleBox[{3.34, 0.7057964898375104}, \
{3.359, 0.7177012517422724}], 
                RectangleBox[{3.4, 0.7057964898375104}, \
{3.419, 0.7177012517422724}], 
                RectangleBox[{3.42, 0.7057964898375104}, \
{3.439, 0.7177012517422724}], 
                RectangleBox[{3.5, 0.7057964898375104}, \
{3.519, 0.7177012517422724}], 
                RectangleBox[{3.66, 0.7057964898375104}, \
{3.6790000000000003, 0.7177012517422724}], 
                RectangleBox[{3.68, 0.7057964898375104}, \
{3.6990000000000003, 0.7177012517422724}], 
                RectangleBox[{3.76, 0.7057964898375104}, \
{3.779, 0.7177012517422724}], 
                RectangleBox[{3.86, 0.7057964898375104}, \
{3.879, 0.7177012517422724}], 
                RectangleBox[{3.88, 0.7057964898375104}, \
{3.899, 0.7177012517422724}], 
                RectangleBox[{3.96, 0.7057964898375104}, \
{3.979, 0.7177012517422724}], 
                RectangleBox[{2.44, 0.7177012517422724}, \
{2.459, 0.7296060136470344}], 
                RectangleBox[{2.52, 0.7177012517422724}, \
{2.539, 0.7296060136470344}], 
                RectangleBox[{2.54, 0.7177012517422724}, \
{2.559, 0.7296060136470344}], 
                RectangleBox[{2.64, 0.7177012517422724}, \
{2.6590000000000003, 0.7296060136470344}], 
                RectangleBox[{2.72, 0.7177012517422724}, \
{2.7390000000000003, 0.7296060136470344}], 
                RectangleBox[{2.74, 0.7177012517422724}, \
{2.759, 0.7296060136470344}], 
                RectangleBox[{3.46, 0.7177012517422724}, \
{3.479, 0.7296060136470344}], 
                RectangleBox[{3.48, 0.7177012517422724}, \
{3.499, 0.7296060136470344}], 
                RectangleBox[{3.56, 0.7177012517422724}, \
{3.579, 0.7296060136470344}], 
                RectangleBox[{3.64, 0.7177012517422724}, \
{3.6590000000000003, 0.7296060136470344}], 
                RectangleBox[{3.72, 0.7177012517422724}, \
{3.7390000000000003, 0.7296060136470344}], 
                RectangleBox[{3.74, 0.7177012517422724}, \
{3.759, 0.7296060136470344}], 
                RectangleBox[{3.84, 0.7177012517422724}, \
{3.859, 0.7296060136470344}], 
                RectangleBox[{3.92, 0.7177012517422724}, \
{3.939, 0.7296060136470344}], 
                RectangleBox[{3.94, 0.7177012517422724}, \
{3.959, 0.7296060136470344}], 
                RectangleBox[{3.44, 0.7296060136470344}, \
{3.459, 0.7415107755517961}], 
                RectangleBox[{3.52, 0.7296060136470344}, \
{3.539, 0.7415107755517961}], 
                RectangleBox[{3.54, 0.7296060136470344}, \
{3.559, 0.7415107755517961}]},
               Antialiasing->True]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None,
             PlotRange->{All, All},
             PlotRangePadding->{Automatic, 0.05}], {180, -35}, 
            ImageScaled[{0.5, 0.5}], {350, 60},
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
           InsetBox[
            GraphicsBox[
             {GrayLevel[0.6], RectangleBox[{0, 0}]},
             AspectRatio->NCache[
               Rational[6, 35], 0.17142857142857143`],
             Background->GrayLevel[1],
             Frame->True,
             FrameStyle->GrayLevel[0.75],
             FrameTicks->False,
             ImagePadding->None], {180, -100}, ImageScaled[{0.5, 0.5}], {350, 60},
            
            
            Background->RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
          InsetBox[
           StyleBox[
            TemplateBox[{
              StyleBox[
              "\" | \"", FontSize -> 14, FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False], 
              StyleBox[
               TemplateBox[{"\"4\"", "\" s\""}, "RowDefault"], FontColor -> 
               RGBColor[
                0.5098039215686274, 0.5098039215686274, 0.5098039215686274], 
               StripOnInput -> False]},
             "RowWithSeparator"], "DialogStyle",
            StripOnInput->False], {355, -159}, Scaled[{1, 0}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{5, -158}, {29, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             PolygonBox[
              NCache[{{11, -157}, {11, -136}, {20., Rational[-293, 2]}, {
                11, -157}}, {{11, -157}, {11, -136}, {20., -146.5}, {
                11, -157}}]], 
             StyleBox[{},
              Antialiasing->False]}},
           EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                 Sound`MIDISequence[{
                   Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    24, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    24, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    72, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    72, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    120, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    120, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    168, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    168, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    216, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    216, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    240, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    264, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    264, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    288, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    288, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    312, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    312, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    336, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    336, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    360, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    360, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    384, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    384, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    408, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    408, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    432, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    432, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    456, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    456, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    480, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    480, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    504, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    504, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    528, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    528, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    552, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    552, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    576, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    576, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    600, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    600, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    624, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    624, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    648, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    648, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    672, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    672, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    696, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    696, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    720, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    720, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    744, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    744, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    768, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    768, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    792, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    792, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    816, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    816, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    840, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    840, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    864, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    864, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    888, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    888, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    912, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    912, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    936, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    936, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    960, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    960, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    984, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    984, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1008, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1008, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1032, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1032, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1056, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1056, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1080, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1080, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1104, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1104, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1128, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1128, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1152, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1152, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1176, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1176, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1200, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1200, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1224, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1224, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1248, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1248, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1272, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1272, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1296, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1296, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1320, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1320, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1344, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1344, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1368, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1368, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1392, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1392, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1416, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1416, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1440, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1440, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1464, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1464, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1488, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1488, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1512, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1512, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1536, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1536, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1560, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1560, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1584, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1584, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1608, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1608, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1632, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1632, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1656, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1656, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1680, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 0\
], 
                    Sound`MIDIEvent[
                    1680, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1704, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1704, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1728, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1728, "NoteOn", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1752, "NoteOff", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1752, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1776, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1776, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1800, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1800, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1824, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1824, "NoteOn", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1848, "NoteOff", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1848, "NoteOn", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1872, "NoteOff", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1872, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1896, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1896, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1920, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1920, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1944, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1944, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1968, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1968, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1992, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1992, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2016, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2016, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2040, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2040, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2064, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2064, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2088, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2088, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2112, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2112, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2136, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2136, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2160, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2160, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2184, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2184, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2208, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2208, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2232, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2232, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2256, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2256, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2280, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2280, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2304, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2304, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2328, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2328, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2352, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2352, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2376, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2376, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2400, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2400, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2424, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2424, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2448, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2448, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2472, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2472, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2496, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2496, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2520, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2520, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2544, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2544, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2568, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2568, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2592, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2592, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2616, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2616, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2640, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2640, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2664, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2664, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2688, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2688, "NoteOn", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2712, "NoteOff", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2712, "NoteOn", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2736, "NoteOff", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2736, "NoteOn", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2760, "NoteOff", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2760, "NoteOn", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2784, "NoteOff", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2784, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2808, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2808, "NoteOn", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2832, "NoteOff", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2832, "NoteOn", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2856, "NoteOff", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2856, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2880, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2880, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2904, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2904, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2928, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2928, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2952, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2952, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2976, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2976, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3000, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 0\
], 
                    Sound`MIDIEvent[
                    3000, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3024, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3024, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3048, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3048, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3072, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3072, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3096, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3096, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3120, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3120, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3144, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3144, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3168, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3168, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3192, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3192, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3216, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3216, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3240, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3240, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3264, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3264, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3288, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3288, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3312, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3312, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3336, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3336, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3360, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3360, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3384, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3384, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3408, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3408, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3432, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3432, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3456, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3456, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3480, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3480, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3504, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3504, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3528, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3528, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3552, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3552, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3576, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3576, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3600, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3600, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3624, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3624, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3648, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3648, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3672, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3672, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3696, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3696, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3720, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3720, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3744, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3744, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3768, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3768, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3792, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3792, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3816, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3816, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3840, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3840, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3864, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3864, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3888, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3888, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3912, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3912, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3936, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3936, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3960, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3960, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3984, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3984, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4008, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4008, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4032, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4032, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4056, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4056, "NoteOn", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4080, "NoteOff", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4080, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4104, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4104, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4128, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4128, "NoteOn", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4152, "NoteOff", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4152, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4176, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4176, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4200, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4200, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4224, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4224, "NoteOn", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4248, "NoteOff", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4248, "NoteOn", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4272, "NoteOff", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4272, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4296, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4296, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4320, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 0\
], 
                    Sound`MIDIEvent[
                    4320, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4344, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4344, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4368, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4368, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4392, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4392, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4416, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4416, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4440, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4440, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4464, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4464, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4488, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4488, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4512, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4512, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4536, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4536, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4560, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4560, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4584, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4584, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4608, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4608, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4632, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4632, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4656, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4656, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4680, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4680, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4704, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4704, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4728, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4728, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4752, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4752, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4776, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4776, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4800, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1200]]; 
               Sound`EmitSampledSound[None, "Preemptive"], 
               FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          TagBox[{
            {RGBColor[
             0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
             RectangleBox[{29, -158}, {53, -135}]}, 
            {RGBColor[
             0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
             RectangleBox[{35, -152}, {47, -141}], 
             StyleBox[{},
              Antialiasing->False]}},
           
           EventHandlerTag[{
            "MouseClicked" :> If[True, Sound`EmitSampledSound[
                 SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
               Sound`StopMIDI[], 
               FEPrivate`EmitSound[
                SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
             "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
             True}]], 
          {GrayLevel[0.75], 
           
           StyleBox[
            LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}],
            Antialiasing->False]}},
         ContentSelectable->False,
         ImageSize->{160.47265625, Automatic},
         PlotRange->{{0, 360}, {-164, 0}},
         PlotRangePadding->1],
        Sound[{
          SoundNote[3, 0.02], 
          SoundNote[3, 0.02], 
          SoundNote[5, 0.02], 
          SoundNote[4, 0.02], 
          SoundNote[4, 0.02], 
          SoundNote[3, 0.02], 
          SoundNote[5, 0.02], 
          SoundNote[5, 0.02], 
          SoundNote[4, 0.02], 
          SoundNote[3, 0.02], 
          SoundNote[6, 0.02], 
          SoundNote[6, 0.02], 
          SoundNote[8, 0.02], 
          SoundNote[8, 0.02], 
          SoundNote[7, 0.02], 
          SoundNote[7, 0.02], 
          SoundNote[9, 0.02], 
          SoundNote[8, 0.02], 
          SoundNote[8, 0.02], 
          SoundNote[6, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[6, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[5, 0.02], 
          SoundNote[9, 0.02], 
          SoundNote[9, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[9, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[5, 0.02], 
          SoundNote[9, 0.02], 
          SoundNote[9, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[9, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[5, 0.02], 
          SoundNote[9, 0.02], 
          SoundNote[9, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[9, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[7, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[13, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[13, 0.02], 
          SoundNote[13, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[6, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[6, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[10, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[12, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[11, 0.02], 
          SoundNote[15, 0.02], 
          SoundNote[15, 0.02], 
          SoundNote[17, 0.02], 
          SoundNote[16, 0.02], 
          SoundNote[16, 0.02], 
          SoundNote[15, 0.02], 
          SoundNote[17, 0.02], 
          SoundNote[17, 0.02], 
          SoundNote[16, 0.02], 
          SoundNote[15, 0.02], 
          SoundNote[18, 0.02], 
          SoundNote[18, 0.02], 
          SoundNote[20, 0.02], 
          SoundNote[20, 0.02], 
          SoundNote[19, 0.02], 
          SoundNote[19, 0.02], 
          SoundNote[21, 0.02], 
          SoundNote[20, 0.02], 
          SoundNote[20, 0.02], 
          SoundNote[18, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[18, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[17, 0.02], 
          SoundNote[21, 0.02], 
          SoundNote[21, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[21, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[17, 0.02], 
          SoundNote[21, 0.02], 
          SoundNote[21, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[21, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[17, 0.02], 
          SoundNote[21, 0.02], 
          SoundNote[21, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[21, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[19, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[25, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[25, 0.02], 
          SoundNote[25, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[18, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[18, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[22, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[24, 0.02], 
          SoundNote[23, 0.02], 
          SoundNote[11, 0.02]}]]], "ExerciseOutput",ExpressionUUID->
       "5d2c9b24-4bf2-43ad-af9f-3e5ce84b5b62"],
      Apply[TextCell, 
       Cell[
        BoxData[
         InterpretationBox[
          GraphicsBox[{{{{{
                RGBColor[
                0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                RectangleBox[{0, 0}, {360, -164}]}}}}, {
             InsetBox[
              GraphicsBox[{{{{{{
                    RGBColor[0.5, 0, 0.5], 
                    StyleBox[{{
                    
                    RectangleBox[{0., 0.4677012517422721}, {0.019, 
                    0.4796060136470343}], 
                    
                    RectangleBox[{0.02, 0.4677012517422721}, {0.039, 
                    0.4796060136470343}], 
                    
                    RectangleBox[{0.1, 0.4677012517422721}, {0.119, 
                    0.4796060136470343}], 
                    
                    RectangleBox[{0.18, 0.4677012517422721}, {0.199, 
                    0.4796060136470343}], 
                    
                    RectangleBox[{0.06, 0.4796060136470343}, {0.079, 
                    0.49151077555179606`}], 
                    
                    RectangleBox[{0.08, 0.4796060136470343}, {0.099, 
                    0.49151077555179606`}], 
                    
                    RectangleBox[{0.16, 0.4796060136470343}, {0.179, 
                    0.49151077555179606`}], 
                    
                    RectangleBox[{0.04, 0.49151077555179606`}, {0.059, 
                    0.5034155374565581}], 
                    
                    RectangleBox[{0.12, 0.49151077555179606`}, {0.139, 
                    0.5034155374565581}], 
                    
                    RectangleBox[{0.14, 0.49151077555179606`}, {0.159, 
                    0.5034155374565581}], 
                    
                    RectangleBox[{0.78, 0.49151077555179606`}, {0.799, 
                    0.5034155374565581}], 
                    
                    RectangleBox[{0.98, 0.49151077555179606`}, {0.999, 
                    0.5034155374565581}], 
                    
                    RectangleBox[{1.18, 0.49151077555179606`}, {
                    1.1989999999999998`, 0.5034155374565581}], 
                    
                    RectangleBox[{0.2, 0.5034155374565581}, {0.219, 
                    0.5153202993613198}], 
                    
                    RectangleBox[{0.22, 0.5034155374565581}, {0.239, 
                    0.5153202993613198}], 
                    
                    RectangleBox[{0.38, 0.5034155374565581}, {0.399, 
                    0.5153202993613198}], 
                    
                    RectangleBox[{0.58, 0.5034155374565581}, {0.599, 
                    0.5153202993613198}], 
                    
                    RectangleBox[{1.58, 0.5034155374565581}, {
                    1.5990000000000002`, 0.5153202993613198}], 
                    
                    RectangleBox[{1.78, 0.5034155374565581}, {1.799, 
                    0.5153202993613198}], 
                    
                    RectangleBox[{0.28, 0.5153202993613198}, {0.299, 
                    0.5272250612660818}], 
                    
                    RectangleBox[{0.3, 0.5153202993613198}, {0.319, 
                    0.5272250612660818}], 
                    
                    RectangleBox[{1.38, 0.5153202993613198}, {1.399, 
                    0.5272250612660818}], 
                    
                    RectangleBox[{0.24, 0.5272250612660818}, {0.259, 
                    0.5391298231708438}], 
                    
                    RectangleBox[{0.26, 0.5272250612660818}, {0.279, 
                    0.5391298231708438}], 
                    
                    RectangleBox[{0.34, 0.5272250612660818}, {0.359, 
                    0.5391298231708438}], 
                    
                    RectangleBox[{0.36, 0.5272250612660818}, {0.379, 
                    0.5391298231708438}], 
                    
                    RectangleBox[{0.32, 0.5391298231708438}, {0.339, 
                    0.5510345850756055}], 
                    
                    RectangleBox[{0.8, 0.5391298231708438}, {0.819, 
                    0.5510345850756055}], 
                    
                    RectangleBox[{0.82, 0.5391298231708438}, {0.839, 
                    0.5510345850756055}], 
                    
                    RectangleBox[{0.9, 0.5391298231708438}, {0.919, 
                    0.5510345850756055}], 
                    
                    RectangleBox[{1., 0.5391298231708438}, {
                    1.0190000000000001`, 0.5510345850756055}], 
                    
                    RectangleBox[{1.02, 0.5391298231708438}, {
                    1.0390000000000001`, 0.5510345850756055}], 
                    
                    RectangleBox[{1.1, 0.5391298231708438}, {
                    1.1190000000000002`, 0.5510345850756055}], 
                    
                    RectangleBox[{1.2, 0.5391298231708438}, {
                    1.2189999999999999`, 0.5510345850756055}], 
                    
                    RectangleBox[{1.22, 0.5391298231708438}, {
                    1.2389999999999999`, 0.5510345850756055}], 
                    
                    RectangleBox[{1.3, 0.5391298231708438}, {1.319, 
                    0.5510345850756055}], 
                    
                    RectangleBox[{0.4, 0.5510345850756055}, {0.419, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{0.42, 0.5510345850756055}, {0.439, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{0.5, 0.5510345850756055}, {0.519, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{0.6, 0.5510345850756055}, {0.619, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{0.62, 0.5510345850756055}, {0.639, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{0.7, 0.5510345850756055}, {0.719, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{0.86, 0.5510345850756055}, {0.879, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{0.88, 0.5510345850756055}, {0.899, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{0.96, 0.5510345850756055}, {0.979, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{1.06, 0.5510345850756055}, {
                    1.0790000000000002`, 0.5629393469803675}], 
                    
                    RectangleBox[{1.08, 0.5510345850756055}, {
                    1.0990000000000002`, 0.5629393469803675}], 
                    
                    RectangleBox[{1.16, 0.5510345850756055}, {
                    1.1789999999999998`, 0.5629393469803675}], 
                    
                    RectangleBox[{1.26, 0.5510345850756055}, {1.279, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{1.28, 0.5510345850756055}, {1.299, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{1.36, 0.5510345850756055}, {1.379, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{1.6, 0.5510345850756055}, {
                    1.6190000000000002`, 0.5629393469803675}], 
                    
                    RectangleBox[{1.62, 0.5510345850756055}, {1.639, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{1.7, 0.5510345850756055}, {
                    1.7189999999999999`, 0.5629393469803675}], 
                    
                    RectangleBox[{1.8, 0.5510345850756055}, {1.819, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{1.82, 0.5510345850756055}, {1.839, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{1.9, 0.5510345850756055}, {1.919, 
                    0.5629393469803675}], 
                    
                    RectangleBox[{0.46, 0.5629393469803675}, {0.479, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{0.48, 0.5629393469803675}, {0.499, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{0.56, 0.5629393469803675}, {0.579, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{0.66, 0.5629393469803675}, {0.679, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{0.68, 0.5629393469803675}, {0.699, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{0.76, 0.5629393469803675}, {0.779, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{0.84, 0.5629393469803675}, {0.859, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{0.92, 0.5629393469803675}, {0.939, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{0.94, 0.5629393469803675}, {0.959, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{1.04, 0.5629393469803675}, {
                    1.0590000000000002`, 0.5748441088851295}], 
                    
                    RectangleBox[{1.12, 0.5629393469803675}, {1.139, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{1.14, 0.5629393469803675}, {
                    1.1589999999999998`, 0.5748441088851295}], 
                    
                    RectangleBox[{1.24, 0.5629393469803675}, {1.259, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{1.32, 0.5629393469803675}, {1.339, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{1.34, 0.5629393469803675}, {1.359, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{1.4, 0.5629393469803675}, {1.419, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{1.42, 0.5629393469803675}, {1.439, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{1.5, 0.5629393469803675}, {
                    1.5190000000000001`, 0.5748441088851295}], 
                    
                    RectangleBox[{1.66, 0.5629393469803675}, {
                    1.6789999999999998`, 0.5748441088851295}], 
                    
                    RectangleBox[{1.68, 0.5629393469803675}, {
                    1.6989999999999998`, 0.5748441088851295}], 
                    
                    RectangleBox[{1.76, 0.5629393469803675}, {1.779, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{1.86, 0.5629393469803675}, {1.879, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{1.88, 0.5629393469803675}, {1.899, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{1.96, 0.5629393469803675}, {1.979, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{1.98, 0.5629393469803675}, {1.999, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{3.98, 0.5629393469803675}, {3.999, 
                    0.5748441088851295}], 
                    
                    RectangleBox[{0.44, 0.5748441088851295}, {0.459, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{0.52, 0.5748441088851295}, {0.539, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{0.54, 0.5748441088851295}, {0.559, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{0.64, 0.5748441088851295}, {0.659, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{0.72, 0.5748441088851295}, {0.739, 
                    0.5867488707898912}], 
                    RectangleBox[{0.74, 0.5748441088851295}, {0.759, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{1.46, 0.5748441088851295}, {1.479, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{1.48, 0.5748441088851295}, {1.499, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{1.56, 0.5748441088851295}, {
                    1.5790000000000002`, 0.5867488707898912}], 
                    
                    RectangleBox[{1.64, 0.5748441088851295}, {
                    1.6589999999999998`, 0.5867488707898912}], 
                    
                    RectangleBox[{1.72, 0.5748441088851295}, {
                    1.7389999999999999`, 0.5867488707898912}], 
                    
                    RectangleBox[{1.74, 0.5748441088851295}, {1.759, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{1.84, 0.5748441088851295}, {1.859, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{1.92, 0.5748441088851295}, {1.939, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{1.94, 0.5748441088851295}, {1.959, 
                    0.5867488707898912}], 
                    
                    RectangleBox[{1.44, 0.5867488707898912}, {1.459, 
                    0.5986536326946532}], 
                    
                    RectangleBox[{1.52, 0.5867488707898912}, {
                    1.5390000000000001`, 0.5986536326946532}], 
                    
                    RectangleBox[{1.54, 0.5867488707898912}, {
                    1.5590000000000002`, 0.5986536326946532}], 
                    
                    RectangleBox[{2., 0.610558394599415}, {2.019, 
                    0.6224631565041769}], 
                    
                    RectangleBox[{2.02, 0.610558394599415}, {2.039, 
                    0.6224631565041769}], 
                    
                    RectangleBox[{2.1, 0.610558394599415}, {2.119, 
                    0.6224631565041769}], 
                    
                    RectangleBox[{2.18, 0.610558394599415}, {
                    2.1990000000000003`, 0.6224631565041769}], 
                    
                    RectangleBox[{2.06, 0.6224631565041769}, {2.079, 
                    0.6343679184089389}], 
                    
                    RectangleBox[{2.08, 0.6224631565041769}, {2.099, 
                    0.6343679184089389}], 
                    
                    RectangleBox[{2.16, 0.6224631565041769}, {
                    2.1790000000000003`, 0.6343679184089389}], 
                    
                    RectangleBox[{2.04, 0.6343679184089389}, {2.059, 
                    0.646272680313701}], 
                    
                    RectangleBox[{2.12, 0.6343679184089389}, {
                    2.1390000000000002`, 0.646272680313701}], 
                    
                    RectangleBox[{2.14, 0.6343679184089389}, {
                    2.1590000000000003`, 0.646272680313701}], 
                    
                    RectangleBox[{2.78, 0.6343679184089389}, {2.799, 
                    0.646272680313701}], 
                    
                    RectangleBox[{2.98, 0.6343679184089389}, {2.999, 
                    0.646272680313701}], 
                    
                    RectangleBox[{3.18, 0.6343679184089389}, {
                    3.1990000000000003`, 0.646272680313701}], 
                    
                    RectangleBox[{2.2, 0.646272680313701}, {
                    2.2190000000000003`, 0.658177442218463}], 
                    
                    RectangleBox[{2.22, 0.646272680313701}, {
                    2.2390000000000003`, 0.658177442218463}], 
                    
                    RectangleBox[{2.38, 0.646272680313701}, {2.399, 
                    0.658177442218463}], 
                    
                    RectangleBox[{2.58, 0.646272680313701}, {2.599, 
                    0.658177442218463}], 
                    
                    RectangleBox[{3.58, 0.646272680313701}, {3.599, 
                    0.658177442218463}], 
                    
                    RectangleBox[{3.78, 0.646272680313701}, {3.799, 
                    0.658177442218463}], 
                    
                    RectangleBox[{2.28, 0.658177442218463}, {2.299, 
                    0.6700822041232247}], 
                    
                    RectangleBox[{2.3, 0.658177442218463}, {2.319, 
                    0.6700822041232247}], 
                    
                    RectangleBox[{3.38, 0.658177442218463}, {3.399, 
                    0.6700822041232247}], 
                    
                    RectangleBox[{2.24, 0.6700822041232247}, {2.259, 
                    0.6819869660279864}], 
                    
                    RectangleBox[{2.26, 0.6700822041232247}, {2.279, 
                    0.6819869660279864}], 
                    
                    RectangleBox[{2.34, 0.6700822041232247}, {2.359, 
                    0.6819869660279864}], 
                    
                    RectangleBox[{2.36, 0.6700822041232247}, {2.379, 
                    0.6819869660279864}], 
                    RectangleBox[{2.32, 0.6819869660279864}, {2.339, 
                    0.6938917279327484}], 
                    
                    RectangleBox[{2.8, 0.6819869660279864}, {2.819, 
                    0.6938917279327484}], 
                    
                    RectangleBox[{2.82, 0.6819869660279864}, {2.839, 
                    0.6938917279327484}], 
                    
                    RectangleBox[{2.9, 0.6819869660279864}, {2.919, 
                    0.6938917279327484}], 
                    
                    RectangleBox[{3., 0.6819869660279864}, {3.019, 
                    0.6938917279327484}], 
                    
                    RectangleBox[{3.02, 0.6819869660279864}, {3.039, 
                    0.6938917279327484}], 
                    
                    RectangleBox[{3.1, 0.6819869660279864}, {3.119, 
                    0.6938917279327484}], 
                    
                    RectangleBox[{3.2, 0.6819869660279864}, {
                    3.2190000000000003`, 0.6938917279327484}], 
                    
                    RectangleBox[{3.22, 0.6819869660279864}, {
                    3.2390000000000003`, 0.6938917279327484}], 
                    
                    RectangleBox[{3.3, 0.6819869660279864}, {3.319, 
                    0.6938917279327484}], 
                    
                    RectangleBox[{2.4, 0.6938917279327484}, {2.419, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{2.42, 0.6938917279327484}, {2.439, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{2.5, 0.6938917279327484}, {2.519, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{2.6, 0.6938917279327484}, {2.619, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{2.62, 0.6938917279327484}, {
                    2.6390000000000002`, 0.7057964898375104}], 
                    
                    RectangleBox[{2.7, 0.6938917279327484}, {
                    2.7190000000000003`, 0.7057964898375104}], 
                    
                    RectangleBox[{2.86, 0.6938917279327484}, {2.879, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{2.88, 0.6938917279327484}, {2.899, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{2.96, 0.6938917279327484}, {2.979, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{3.06, 0.6938917279327484}, {3.079, 
                    0.7057964898375104}], 
                    RectangleBox[{3.08, 0.6938917279327484}, {3.099, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{3.16, 0.6938917279327484}, {
                    3.1790000000000003`, 0.7057964898375104}], 
                    
                    RectangleBox[{3.26, 0.6938917279327484}, {3.279, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{3.28, 0.6938917279327484}, {3.299, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{3.36, 0.6938917279327484}, {3.379, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{3.6, 0.6938917279327484}, {3.619, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{3.62, 0.6938917279327484}, {
                    3.6390000000000002`, 0.7057964898375104}], 
                    
                    RectangleBox[{3.7, 0.6938917279327484}, {
                    3.7190000000000003`, 0.7057964898375104}], 
                    
                    RectangleBox[{3.8, 0.6938917279327484}, {3.819, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{3.82, 0.6938917279327484}, {3.839, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{3.9, 0.6938917279327484}, {3.919, 
                    0.7057964898375104}], 
                    
                    RectangleBox[{2.46, 0.7057964898375104}, {2.479, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{2.48, 0.7057964898375104}, {2.499, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{2.56, 0.7057964898375104}, {2.579, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{2.66, 0.7057964898375104}, {
                    2.6790000000000003`, 0.7177012517422724}], 
                    
                    RectangleBox[{2.68, 0.7057964898375104}, {
                    2.6990000000000003`, 0.7177012517422724}], 
                    
                    RectangleBox[{2.76, 0.7057964898375104}, {2.779, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{2.84, 0.7057964898375104}, {2.859, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{2.92, 0.7057964898375104}, {2.939, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{2.94, 0.7057964898375104}, {2.959, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{3.04, 0.7057964898375104}, {3.059, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{3.12, 0.7057964898375104}, {
                    3.1390000000000002`, 0.7177012517422724}], 
                    
                    RectangleBox[{3.14, 0.7057964898375104}, {
                    3.1590000000000003`, 0.7177012517422724}], 
                    
                    RectangleBox[{3.24, 0.7057964898375104}, {3.259, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{3.32, 0.7057964898375104}, {3.339, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{3.34, 0.7057964898375104}, {3.359, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{3.4, 0.7057964898375104}, {3.419, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{3.42, 0.7057964898375104}, {3.439, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{3.5, 0.7057964898375104}, {3.519, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{3.66, 0.7057964898375104}, {
                    3.6790000000000003`, 0.7177012517422724}], 
                    
                    RectangleBox[{3.68, 0.7057964898375104}, {
                    3.6990000000000003`, 0.7177012517422724}], 
                    
                    RectangleBox[{3.76, 0.7057964898375104}, {3.779, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{3.86, 0.7057964898375104}, {3.879, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{3.88, 0.7057964898375104}, {3.899, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{3.96, 0.7057964898375104}, {3.979, 
                    0.7177012517422724}], 
                    
                    RectangleBox[{2.44, 0.7177012517422724}, {2.459, 
                    0.7296060136470344}], 
                    
                    RectangleBox[{2.52, 0.7177012517422724}, {2.539, 
                    0.7296060136470344}], 
                    
                    RectangleBox[{2.54, 0.7177012517422724}, {2.559, 
                    0.7296060136470344}], 
                    
                    RectangleBox[{2.64, 0.7177012517422724}, {
                    2.6590000000000003`, 0.7296060136470344}], 
                    RectangleBox[{2.72, 0.7177012517422724}, {
                    2.7390000000000003`, 0.7296060136470344}], 
                    
                    RectangleBox[{2.74, 0.7177012517422724}, {2.759, 
                    0.7296060136470344}], 
                    
                    RectangleBox[{3.46, 0.7177012517422724}, {3.479, 
                    0.7296060136470344}], 
                    
                    RectangleBox[{3.48, 0.7177012517422724}, {3.499, 
                    0.7296060136470344}], 
                    
                    RectangleBox[{3.56, 0.7177012517422724}, {3.579, 
                    0.7296060136470344}], 
                    
                    RectangleBox[{3.64, 0.7177012517422724}, {
                    3.6590000000000003`, 0.7296060136470344}], 
                    
                    RectangleBox[{3.72, 0.7177012517422724}, {
                    3.7390000000000003`, 0.7296060136470344}], 
                    
                    RectangleBox[{3.74, 0.7177012517422724}, {3.759, 
                    0.7296060136470344}], 
                    
                    RectangleBox[{3.84, 0.7177012517422724}, {3.859, 
                    0.7296060136470344}], 
                    
                    RectangleBox[{3.92, 0.7177012517422724}, {3.939, 
                    0.7296060136470344}], 
                    
                    RectangleBox[{3.94, 0.7177012517422724}, {3.959, 
                    0.7296060136470344}], 
                    
                    RectangleBox[{3.44, 0.7296060136470344}, {3.459, 
                    0.7415107755517961}], 
                    
                    RectangleBox[{3.52, 0.7296060136470344}, {3.539, 
                    0.7415107755517961}], 
                    
                    RectangleBox[{3.54, 0.7296060136470344}, {3.559, 
                    0.7415107755517961}]}}, Antialiasing -> True]}}}}}}, 
               AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None, 
               PlotRange -> {All, All}, 
               PlotRangePadding -> {Automatic, 0.05}], {180, -35}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]], 
             InsetBox[
              GraphicsBox[{{{{
                   GrayLevel[0.6], 
                   RectangleBox[{0, 0}]}}}}, AspectRatio -> NCache[
                 Rational[6, 35], 0.17142857142857143`], Background -> 
               GrayLevel[1], Frame -> True, FrameStyle -> GrayLevel[0.75], 
               FrameTicks -> False, ImagePadding -> None], {180, -100}, 
              ImageScaled[{0.5, 0.5}], {350, 60}, Background -> 
              RGBColor[
               0.9568627450980393, 0.9568627450980393, 0.9568627450980393]]}, 
            
            InsetBox[
             StyleBox[
              TemplateBox[{
                StyleBox[
                "\" | \"", FontSize -> 14, FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False], 
                StyleBox[
                 TemplateBox[{"\"4\"", "\" s\""}, "RowDefault"], FontColor -> 
                 RGBColor[
                  0.5098039215686274, 0.5098039215686274, 0.5098039215686274],
                  StripOnInput -> False]}, "RowWithSeparator"], "DialogStyle",
               StripOnInput -> False], {355, -159}, 
             Scaled[{1, 0}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{5, -158}, {29, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 PolygonBox[
                  NCache[{{11, -157}, {11, -136}, {20., 
                    Rational[-293, 2]}, {11, -157}}, {{11, -157}, {
                    11, -136}, {20., -146.5}, {11, -157}}]], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             EventHandlerTag[{"MouseClicked" :> If[True, Sound`EmitMIDI[
                   Sound`MIDISequence[{
                    Sound`MIDITrack[{
                    Sound`MIDIEvent[0, "SetTempo", "Tempo" -> 1000000], 
                    Sound`MIDIEvent[
                    0, "ProgramCommand", "Channel" -> 0, "Value" -> 0], 
                    Sound`MIDIEvent[
                    0, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    24, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    24, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    48, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    48, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    72, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    72, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    96, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    96, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    120, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    120, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    144, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    144, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    168, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    168, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    192, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    192, "NoteOn", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    216, "NoteOff", "Note" -> 64, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    216, "NoteOn", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    240, "NoteOff", "Note" -> 63, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    240, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    264, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    264, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    288, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    288, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    312, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    312, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    336, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    336, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    360, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    360, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    384, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    384, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    408, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    408, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    432, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    432, "NoteOn", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    456, "NoteOff", "Note" -> 68, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    456, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    480, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    480, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    504, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    504, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    528, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    528, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    552, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    552, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    576, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    576, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    600, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    600, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    624, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    624, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    648, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    648, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    672, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    672, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    696, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    696, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    720, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    720, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    744, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    744, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    768, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    768, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    792, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    792, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    816, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    816, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    840, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    840, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    864, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    864, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    888, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    888, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    912, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    912, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    936, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    936, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    960, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    960, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    984, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    984, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1008, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1008, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1032, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1032, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1056, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1056, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1080, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1080, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1104, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1104, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1128, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1128, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1152, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1152, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1176, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1176, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1200, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1200, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1224, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1224, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1248, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1248, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1272, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1272, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1296, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1296, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1320, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1320, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1344, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 0\
], 
                    Sound`MIDIEvent[
                    1344, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1368, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1368, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1392, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1392, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1416, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1416, "NoteOn", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1440, "NoteOff", "Note" -> 65, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1440, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1464, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1464, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1488, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1488, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1512, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1512, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1536, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1536, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1560, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1560, "NoteOn", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1584, "NoteOff", "Note" -> 69, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1584, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1608, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1608, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1632, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1632, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1656, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1656, "NoteOn", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1680, "NoteOff", "Note" -> 67, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1680, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1704, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1704, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1728, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1728, "NoteOn", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1752, "NoteOff", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1752, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1776, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1776, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1800, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1800, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1824, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1824, "NoteOn", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1848, "NoteOff", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1848, "NoteOn", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1872, "NoteOff", "Note" -> 73, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1872, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1896, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1896, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1920, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1920, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1944, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1944, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1968, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1968, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    1992, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    1992, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2016, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2016, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2040, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2040, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2064, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2064, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2088, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2088, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2112, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2112, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2136, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2136, "NoteOn", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2160, "NoteOff", "Note" -> 66, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2160, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2184, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2184, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2208, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2208, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2232, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2232, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2256, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2256, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2280, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2280, "NoteOn", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2304, "NoteOff", "Note" -> 70, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2304, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2328, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2328, "NoteOn", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2352, "NoteOff", "Note" -> 72, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2352, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2376, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2376, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2400, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2400, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2424, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2424, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2448, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2448, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2472, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2472, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2496, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2496, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2520, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2520, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2544, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2544, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2568, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2568, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2592, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2592, "NoteOn", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2616, "NoteOff", "Note" -> 76, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2616, "NoteOn", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2640, "NoteOff", "Note" -> 75, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2640, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2664, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 0\
], 
                    Sound`MIDIEvent[
                    2664, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2688, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2688, "NoteOn", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2712, "NoteOff", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2712, "NoteOn", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2736, "NoteOff", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2736, "NoteOn", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2760, "NoteOff", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2760, "NoteOn", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2784, "NoteOff", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2784, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2808, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2808, "NoteOn", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2832, "NoteOff", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2832, "NoteOn", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2856, "NoteOff", "Note" -> 80, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2856, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2880, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2880, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2904, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2904, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2928, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2928, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2952, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2952, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    2976, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    2976, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3000, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3000, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3024, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3024, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3048, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3048, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3072, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3072, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3096, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3096, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3120, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3120, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3144, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3144, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3168, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3168, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3192, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3192, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3216, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3216, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3240, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3240, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3264, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3264, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3288, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3288, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3312, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3312, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3336, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3336, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3360, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3360, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3384, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3384, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3408, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3408, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3432, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3432, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3456, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3456, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3480, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3480, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3504, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3504, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3528, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3528, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3552, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3552, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3576, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3576, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3600, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3600, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3624, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3624, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3648, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3648, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3672, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3672, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3696, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3696, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3720, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3720, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3744, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3744, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3768, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3768, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3792, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3792, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3816, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3816, "NoteOn", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3840, "NoteOff", "Note" -> 77, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3840, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3864, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3864, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3888, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3888, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3912, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3912, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3936, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3936, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3960, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    3960, "NoteOn", "Note" -> 81, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    3984, "NoteOff", "Note" -> 81, "Channel" -> 0, "Velocity" -> 0\
], 
                    Sound`MIDIEvent[
                    3984, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4008, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4008, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4032, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4032, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4056, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4056, "NoteOn", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4080, "NoteOff", "Note" -> 79, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4080, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4104, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4104, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4128, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4128, "NoteOn", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4152, "NoteOff", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4152, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4176, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4176, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4200, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4200, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4224, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4224, "NoteOn", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4248, "NoteOff", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4248, "NoteOn", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4272, "NoteOff", "Note" -> 85, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4272, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4296, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4296, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4320, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4320, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4344, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4344, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4368, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4368, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4392, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4392, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4416, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4416, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4440, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4440, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4464, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4464, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4488, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4488, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4512, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4512, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4536, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4536, "NoteOn", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4560, "NoteOff", "Note" -> 78, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4560, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4584, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4584, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4608, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4608, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4632, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4632, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4656, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4656, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4680, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4680, "NoteOn", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4704, "NoteOff", "Note" -> 82, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4704, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4728, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4728, "NoteOn", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4752, "NoteOff", "Note" -> 84, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4752, "NoteOn", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4776, "NoteOff", "Note" -> 83, "Channel" -> 0, "Velocity" -> 
                    0], 
                    Sound`MIDIEvent[
                    4776, "NoteOn", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    127], 
                    Sound`MIDIEvent[
                    4800, "NoteOff", "Note" -> 71, "Channel" -> 0, "Velocity" -> 
                    0]}]}, "DivisionType" -> "PPQ", "Resolution" -> 1200]]; 
                 Sound`EmitSampledSound[None, "Preemptive"], 
                 FEPrivate`EmitSound[None, "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], 
            TagBox[{{{{
                 RGBColor[
                 0.9568627450980393, 0.9568627450980393, 0.9568627450980393], 
                 
                 RectangleBox[{29, -158}, {53, -135}]}}}, {{{
                 RGBColor[
                 0.27450980392156865`, 0.5411764705882353, 0.796078431372549], 
                 RectangleBox[{35, -152}, {47, -141}], 
                 StyleBox[{}, Antialiasing -> False]}}}}, 
             
             EventHandlerTag[{
              "MouseClicked" :> If[True, Sound`EmitSampledSound[
                   SampledSoundList[{{0.}}, 8000], "Preemptive"]; 
                 Sound`StopMIDI[], 
                 FEPrivate`EmitSound[
                  SampledSoundList[{{0.}}, 8000], "Preemptive"]], Method -> 
               "Preemptive", PassEventsDown -> Automatic, PassEventsUp -> 
               True}]], {{{{
                GrayLevel[0.75], 
                StyleBox[
                 LineBox[{{0, 0}, {360, 0}, {360, -164}, {0, -164}, {0, 0}}], 
                 Antialiasing -> False]}}}}}, ContentSelectable -> False, 
           ImageSize -> {160.47265625, Automatic}, 
           PlotRange -> {{0, 360}, {-164, 0}}, PlotRangePadding -> 1], 
          Sound[{
            SoundNote[3, 0.02], 
            SoundNote[3, 0.02], 
            SoundNote[5, 0.02], 
            SoundNote[4, 0.02], 
            SoundNote[4, 0.02], 
            SoundNote[3, 0.02], 
            SoundNote[5, 0.02], 
            SoundNote[5, 0.02], 
            SoundNote[4, 0.02], 
            SoundNote[3, 0.02], 
            SoundNote[6, 0.02], 
            SoundNote[6, 0.02], 
            SoundNote[8, 0.02], 
            SoundNote[8, 0.02], 
            SoundNote[7, 0.02], 
            SoundNote[7, 0.02], 
            SoundNote[9, 0.02], 
            SoundNote[8, 0.02], 
            SoundNote[8, 0.02], 
            SoundNote[6, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[6, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[5, 0.02], 
            SoundNote[9, 0.02], 
            SoundNote[9, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[9, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[5, 0.02], 
            SoundNote[9, 0.02], 
            SoundNote[9, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[9, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[5, 0.02], 
            SoundNote[9, 0.02], 
            SoundNote[9, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[9, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[7, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[13, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[13, 0.02], 
            SoundNote[13, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[6, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[6, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[10, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[12, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[11, 0.02], 
            SoundNote[15, 0.02], 
            SoundNote[15, 0.02], 
            SoundNote[17, 0.02], 
            SoundNote[16, 0.02], 
            SoundNote[16, 0.02], 
            SoundNote[15, 0.02], 
            SoundNote[17, 0.02], 
            SoundNote[17, 0.02], 
            SoundNote[16, 0.02], 
            SoundNote[15, 0.02], 
            SoundNote[18, 0.02], 
            SoundNote[18, 0.02], 
            SoundNote[20, 0.02], 
            SoundNote[20, 0.02], 
            SoundNote[19, 0.02], 
            SoundNote[19, 0.02], 
            SoundNote[21, 0.02], 
            SoundNote[20, 0.02], 
            SoundNote[20, 0.02], 
            SoundNote[18, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[18, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[17, 0.02], 
            SoundNote[21, 0.02], 
            SoundNote[21, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[21, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[17, 0.02], 
            SoundNote[21, 0.02], 
            SoundNote[21, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[21, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[17, 0.02], 
            SoundNote[21, 0.02], 
            SoundNote[21, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[21, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[19, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[25, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[25, 0.02], 
            SoundNote[25, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[18, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[18, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[22, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[24, 0.02], 
            SoundNote[23, 0.02], 
            SoundNote[11, 0.02]}]]], "ExerciseOutput", Selectable -> False, 
        Copyable -> False, Editable -> False, ShowCellBracket -> False]]], " "}
   },
   GridBoxAlignment->{"Columns" -> {{Left}}, "Rows" -> {{Center}}},
   GridBoxBackground->{"Columns" -> {{
        RGBColor[1, 1, 1]}}},
   GridBoxSpacings->{"Columns" -> {{Automatic}}, "Rows" -> {{1}}}],
  FrameMargins->0]], "ExerciseOutput",
 Editable->False,
 Selectable->False,
 Deletable->False,
 CellFrame->{{1, 1}, {1, 1}},
 ShowCellBracket->False,
 CellFrameColor->RGBColor[0.85, 0.85, 0.85],
 LineBreakWithin->
  Automatic,ExpressionUUID->"e9fb3668-3fd2-4987-823c-5b7dcdc3d6e1"]
}, Closed]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Q&A", "QASection",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"0d8334d8-3c41-41a0-9a55-5018044c59fe"],

Cell["How do I know which instruments are available?", "Question",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"f7e66415-4cb2-4270-9c94-5c7700d98981"],

Cell[TextData[{
 "Look at the list under \[OpenCurlyDoubleQuote]Details and Options\
\[CloseCurlyDoubleQuote] in the ",
 ButtonBox["SoundNote",
  BaseStyle->"DetailsLink"],
 " reference page, or just start typing and see the completions you\
\[CloseCurlyQuote]re offered. You can also use instrument numbers, from 1 to \
128. All the standard MIDI instruments are there, including percussion."
}], "Answer",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:9",ExpressionUUID->"5087e67e-98f4-4a20-94a0-5aaeee2d5193"],

Cell["How do I play notes below middle C?", "Question",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"d68deac5-8656-4b63-a512-fafcd4761ca4"],

Cell[TextData[{
 "Just use negative numbers, like ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["SoundNote",
    BaseStyle->"CodeLink"], "[", 
   RowBox[{"-", "10"}], "]"}]], "InlineCode",ExpressionUUID->
  "43a8135d-a3fc-4063-bd98-b42532e30271"],
 "."
}], "Answer",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"7aabe44b-0cff-4e97-bc47-16cf0c8a5faa"],

Cell["What are sharp and flat notes called?", "Question",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"6932ec5e-e06b-4e02-bfcf-51bcfbe4e92c"],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{"E", 
   ButtonBox["#",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/character/Sharp"]}]], "InlineCode",ExpressionUUID->
  "b6add624-4640-4574-89a1-1ec8f291833e"],
 " (E sharp), ",
 Cell[BoxData[
  RowBox[{"A", 
   ButtonBox["b",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/character/Flat"]}]], "InlineCode",ExpressionUUID->
  "e956e414-7165-4a23-a17e-713155d63e7e"],
 " (A flat), etc. They also have numbers (e.g. E# is ",
 Cell[BoxData[
 "5"], "InlineCode",ExpressionUUID->"03928f3a-4270-4f67-adfa-d5ba975b4d89"],
 "). The # and b can be typed as ordinary keyboard characters (though special \
",
 Cell[BoxData[
  ButtonBox["\[Sharp]",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/character/Sharp"]], "InlineCode",ExpressionUUID->
  "bc4e5e57-0b20-47f0-8a48-cb232f512aad"],
 " and ",
 Cell[BoxData[
  ButtonBox["\[Flat]",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/character/Flat"]], "InlineCode",ExpressionUUID->
  "dc6eb841-e5c6-4978-91dd-d3e87fa8b9bf"],
 " characters are available too). "
}], "Answer",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:10",ExpressionUUID->"361d5627-847d-4920-9703-e8731667d376"],

Cell["How do I make a chord?", "Question",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:11",ExpressionUUID->"ca76e3e5-4940-46b5-a9d3-31d922b0d222"],

Cell[TextData[{
 "Put note names in a list, as in ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["SoundNote",
    BaseStyle->"CodeLink"], "[", 
   RowBox[{"{", 
    RowBox[{"\<\"C\"\>", ",", "\<\"G\"\>"}], "}"}], "]"}]], "InlineCode",
  ExpressionUUID->"09990417-6b79-4240-a5e1-3f79ce274829"],
 "."
}], "Answer",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"88dae048-e372-4731-bae7-7d5f66ce5e9f"],

Cell["How do I make a rest?", "Question",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:12",ExpressionUUID->"bc50606e-0ac1-4171-8418-c062379a4067"],

Cell[TextData[{
 "For a 0.2-second rest, use ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["SoundNote",
    BaseStyle->"CodeLink"], "[", 
   RowBox[{"None", ",", "0.2"}], "]"}]], "InlineCode",ExpressionUUID->
  "c9e99b0b-756d-4c97-b32e-d8c38eb8d716"],
 "."
}], "Answer",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"dd728f88-6a1a-4e69-8e0f-574c30cd413d"],

Cell["\<\
How do I get a sound to play immediately, without having to press the play \
button?\
\>", "Question",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:13",ExpressionUUID->"2c06cb48-21ae-45b0-a9b1-e16d521b2d7f"],

Cell[TextData[{
 "Use ",
 ButtonBox["EmitSound",
  BaseStyle->"Link"],
 ", as in ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["EmitSound",
    BaseStyle->"CodeLink"], "[", 
   RowBox[{
    ButtonBox["Sound",
     BaseStyle->"CodeLink"], "[", 
    RowBox[{
     ButtonBox["SoundNote",
      BaseStyle->"CodeLink"], "[", "\<\"C\"\>", "]"}], "]"}], "]"}]], 
  "InlineCode",ExpressionUUID->"bfda8c70-70ae-4e9b-b37f-8d3cbaf353ff"],
 ", etc."
}], "Answer",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"dcb84f4a-bcc0-4f72-bf7a-565caa81e302"],

Cell["Why do I need quotes in the name of a note like \
\[OpenCurlyDoubleQuote]C\[CloseCurlyDoubleQuote]?", "Question",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"c16af06d-f6c8-479c-9cbd-72f8b1b7156d"],

Cell[TextData[{
 "Because the name is a Wolfram Language string. If you typed just C, it \
would be interpreted as a function named ",
 Cell[BoxData[
  ButtonBox["C",
   BaseStyle->"CodeLink"]], "InlineCode",ExpressionUUID->
  "46cc8bc2-704c-4277-bf94-a9d60da31513"],
 ", which isn\[CloseCurlyQuote]t what you want."
}], "Answer",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"2d2fcc08-7ad5-46a2-8231-dfa7b2938608"],

Cell["Can I record audio and manipulate it?", "Question",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:14",ExpressionUUID->"71fa9082-8665-4169-a63a-0ae9241e62b5"],

Cell[TextData[{
 "Yes.  Use ",
 ButtonBox["AudioCapture",
  BaseStyle->"Link"],
 ", then use functions like ",
 ButtonBox["AudioPlot",
  BaseStyle->"Link"],
 ", ",
 ButtonBox["Spectrogram",
  BaseStyle->"Link"],
 ", ",
 ButtonBox["AudioPitchShift",
  BaseStyle->"Link"],
 ", etc. "
}], "Answer",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->
  "LastCellOfAPage",ExpressionUUID->"17dc1df5-635b-45a5-876a-b28c585f5dbe"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Tech Notes", "TechNoteSection",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"p:56",ExpressionUUID->"cc05ab61-f5f9-4767-afc3-b0485e076325"],

Cell[TextData[{
 ButtonBox["SoundNote",
  BaseStyle->"Link"],
 " corresponds to MIDI sound. The Wolfram Language also supports \
\[OpenCurlyDoubleQuote]sampled sound\[CloseCurlyDoubleQuote], for example \
using functions like ",
 Cell[BoxData[
  ButtonBox["ListPlay",
   BaseStyle->"CodeLink"]], "InlineCode",ExpressionUUID->
  "60c9f898-c235-4a42-8146-4f89f5ea33be"],
 ", as well as an ",
 ButtonBox["Audio",
  BaseStyle->"Link"],
 " construct that represents all aspects of an audio signal."
}], "TechNote",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:15",ExpressionUUID->"34e9d6a8-cf3f-4c24-970e-4e63f05818f5"],

Cell[TextData[{
 "To get spoken output, use ",
 ButtonBox["Speak",
  BaseStyle->"Link"],
 ". To make a beep, use ",
 ButtonBox["Beep",
  BaseStyle->"Link"],
 "."
}], "TechNote",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->"i:16",ExpressionUUID->"b6246dd6-03d9-41cf-9138-e6705e197729"]
}, Open  ]],

Cell[CellGroupData[{

Cell["More to Explore", "MoreExploreSection",
 Editable->False,
 Deletable->False,
 ShowCellBracket->
  False,ExpressionUUID->"00306873-588c-4a63-bcaf-ecb3d0b30037"],

Cell[TextData[ButtonBox["Guide to Sound Generation in the Wolfram Language\
\[NonBreakingSpace]\[RightGuillemet]",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SoundAndSonification"]], "MoreExplore",
 Editable->False,
 Deletable->False,
 ShowCellBracket->False,
 CellTags->
  "LastCellOfAPage",ExpressionUUID->"d1366c45-8cc3-47af-9d03-e957de205489"]
}, Open  ]]
},
Saveable->False,
WindowToolbars->{},
BlinkingCellInsertionPoint->False,
CellInsertionPointCell->None,
WindowSize->{830, 800},
WindowMargins->Automatic,
WindowPersistentStyles->{
 WindowFrame, WindowMargins, WindowSize, Magnification},
WindowTitle->"Sound: Elementary Introduction to the Wolfram Language",
DockedCells->{
  Cell[
   BoxData[
    TagBox[
     GridBox[{{
        TemplateBox[{
         "\[ThickSpace]\[ThickSpace]\[ThickSpace]\[ThickSpace]", 
          "\"\[ThickSpace]\[ThickSpace]\[ThickSpace]\[ThickSpace]\"", 
          GraphicsBox[
           TagBox[
            RasterBox[CompressedData["
1:eJztl01IVFEUxyctMFzUysjIyUDCTauiIJFM6ENpM4NEtJrBUaSYZMYPhCRm
rYuoICgiW0VEBVIUEbTNEMTAIqRsk6JGCVFBHx7//+4JLo/36DqvB4Ie+Mk7
7+t3733nHpzq9NlEW0ksFsuX4U8i1duQy6X6kpuRtGTzHe3ZTOuxbFemPZPb
ny7FyZ2gEqwHInICfJToguOIReywPVHHlYDzcY4hJHZ486giyBOXtfms5Pms
Bs9zUBDTow6DJOgBT8DP/+B5CepAPbgqZm+zT70Bt9VXCx6H8DwE28Gw5ufB
IT2eA1N6PKquC0V4xtUxqfkE6ARD4BHYAx5Y938Gu8GdZXq4VsNWngIfwDzY
Aa6Bm55nOKYq8NXRw29eb+Uz4JSVd4Nn4IbPs2fARUdPQcfLWAQJ8Mq6Tkdd
gGcEHHT0sHYn9Jg1dcnnnhf6vnee8xzXVkcP9wdr97eY7z0NnlrX74qpxffg
iOZ2VDp6kvqOe6AVXAb7xNQB462YNfsCfoBzoE1MzTEqHD096mgE98WsEec4
Jqamvvk8cx2kwWuw19HDXnIAbNAx03ka9IO8mO+3ScyebRZTi01i1rqguHjY
r3aJWaugOA5O6vEC+K6ebWDW0cNgv6oBnwKus87KwKDmv8BRMOBz77/6G/sV
e8lkwHWuTwm4pQ7WzGIRHgb7FXsJ9/mI5z2si42gXMw8/ByuHgb7FXsJ9yX3
IPfHFjF1xTl5v0exnrCx5gnn+fu7JC7h/z/0Y52+P+rfWX9iCXVYErg=
             "], {{0,
              34}, {26, 0}}, {0, 255}, ColorFunction -> RGBColor], 
            BoxForm`ImageTag[
            "Byte", ColorSpace -> "RGB", Interleaving -> True, Magnification -> 
             1], Selectable -> False], DefaultBaseStyle -> "ImageGraphics", 
           ImageSize -> Magnification[1], ImageSizeRaw -> {26, 34}, 
           PlotRange -> {{0, 26}, {0, 34}}], 
          GraphicsBox[
           TagBox[
            RasterBox[CompressedData["
1:eJztXUuS47gRLX8WXtYRaJ+gl17SS+8U4/CeM+H2hDdtRY8jHF6ZR9AReATt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             "], {{0, 34}, {383, 0}}, {0, 255}, 
             ColorFunction -> RGBColor], 
            BoxForm`ImageTag[
            "Byte", ColorSpace -> "RGB", Interleaving -> True, Magnification -> 
             1], Selectable -> False], DefaultBaseStyle -> "ImageGraphics", 
           ImageSize -> Magnification[1], ImageSizeRaw -> {383, 34}, 
           PlotRange -> {{0, 383}, {0, 34}}]}, "RowWithSeparators"], 
        PaneBox[
         StyleBox["\[VeryThinSpace]", ShowStringCharacters -> False], 
         Alignment -> Right, ImageSize -> {174, 47}], 
        PaneBox[
         PaneBox[
          TemplateBox[{"\[ThinSpace]", "\" \"", 
            TagBox[
             TooltipBox[
              PaneSelectorBox[{False -> ButtonBox[
                  PaneBox[
                   GraphicsBox[
                    TagBox[
                    RasterBox[CompressedData["
1:eJxTTMoPSmNiYGAo5gASQYnljkVFiZXBAkBOaF5xZnpeaopnXklqemqRRRIz
UJAPiDmBmIUBO/j//z8jEDcA8W8g3oBFnhmIZ/9HgE9o8mxAvBZJ/hsQuyPJ
8wPxfiT5N0BsjiQvCsTnkOSfALEWkrwSEN9Bkr8OxDJobniAJP8ZZCYWfxCj
RpGQXcS4mVi/ExuGxMYFmlp4nAIAG14fWQ==
                    "], {{0, 14}, {9, 
                    0}}, {0, 255}, ColorFunction -> RGBColor], 
                    BoxForm`ImageTag[
                    "Byte", ColorSpace -> "RGB", Interleaving -> True], 
                    Selectable -> False], DefaultBaseStyle -> "ImageGraphics",
                     ImageSizeRaw -> {9, 14}, PlotRange -> {{0, 9}, {0, 14}}, 
                    ImageSize -> Magnification[1]], ImageSize -> {35, 30}, 
                   Alignment -> Center], 
                  ButtonFunction :> (
                   NotebookLocate[{"EIWL2-11-strings-and-text.nb", ""}, 
                    FrontEnd`HistoryData :> {
                    EvaluationNotebook[]}]& ), Enabled -> True, Appearance -> 
                  None, Evaluator -> Automatic, Method -> "Preemptive"], True -> 
                ButtonBox[
                  PaneBox[
                   GraphicsBox[
                    TagBox[
                    RasterBox[CompressedData["
1:eJxTTMoPSmNiYGAo5gASQYnljkVFiZXBAkBOaF5xZnpeaopnXklqemqRRRIz
UJAPiDmBmIUBO/j//z8jEDcA8W8g3oBFnhmIZ/9HgE9o8mxAvBZJ/hsQuyPJ
8wPxfiT5N0BsjiQvCsTnkOSfALEWkrwSEN9Bkr8OxDJobniAJP8ZZCYWfxCj
RpGQXcS4mVi/ExuGxMYFmlp4nAIAG14fWQ==
                    "], {{0, 14}, {9, 
                    0}}, {0, 255}, ColorFunction -> RGBColor], 
                    BoxForm`ImageTag[
                    "Byte", ColorSpace -> "RGB", Interleaving -> True], 
                    Selectable -> False], DefaultBaseStyle -> "ImageGraphics",
                     ImageSizeRaw -> {9, 14}, PlotRange -> {{0, 9}, {0, 14}}, 
                    ImageSize -> Magnification[1]], ImageSize -> {35, 30}, 
                   Alignment -> Center], 
                  ButtonFunction :> (
                   NotebookLocate[{"EIWL2-11-strings-and-text.nb", ""}, 
                    FrontEnd`HistoryData :> {
                    EvaluationNotebook[]}]& ), Enabled -> True, Appearance -> 
                  None, Background -> RGBColor[0.835294, 0.364706, 0.0196078],
                   Evaluator -> Automatic, Method -> "Preemptive"]}, 
               Dynamic[
                CurrentValue["MouseOver"]], ImageSize -> Automatic, 
               FrameMargins -> 0], "\"Strings and Text\"", TooltipDelay -> 
              0.15, LabelStyle -> "TextStyling"], 
             Annotation[#, "Strings and Text", "Tooltip"]& ], 
            InterpretationBox[
             StyleBox[
              
              GraphicsBox[{}, ImageSize -> {5, 0}, BaselinePosition -> 
               Baseline], "CacheGraphics" -> False], 
             Spacer[5], Selectable -> False], 
            TagBox[
             TooltipBox[
              PaneSelectorBox[{False -> ButtonBox[
                  PaneBox[
                   GraphicsBox[
                    TagBox[
                    RasterBox[CompressedData["
1:eJxTTMoPSmNiYGAo5gASQYnljkVFiZXBAkBOaF5xZnpeaopnXklqemqRRRIz
UJAfiDmBmAWI////vwKIfwFxMQMOAJT7+R8B6nGoSf+PCjpwqMskUl0SEP9F
UtcHxIxY1EWjqZuOQ10Mmr2TcNj7BEnNP1LNIeQeIDsZn78IhQ8x4fwfNb4a
cPgTFu8l2ORBAAC+1hnm
                    "], {{0, 15}, {9, 0}}, {0, 255}, 
                    ColorFunction -> RGBColor], 
                    BoxForm`ImageTag[
                    "Byte", ColorSpace -> "RGB", Interleaving -> True], 
                    Selectable -> False], DefaultBaseStyle -> "ImageGraphics",
                     ImageSizeRaw -> {9, 15}, PlotRange -> {{0, 9}, {0, 15}}, 
                    ImageSize -> Magnification[1]], ImageSize -> {35, 30}, 
                   Alignment -> Center], 
                  ButtonFunction :> (
                   NotebookLocate[{
                    "EIWL2-13-arrays-or-lists-of-lists.nb", ""}, 
                    FrontEnd`HistoryData :> {
                    EvaluationNotebook[]}]& ), Appearance -> None, Evaluator -> 
                  Automatic, Method -> "Preemptive"], True -> ButtonBox[
                  PaneBox[
                   GraphicsBox[
                    TagBox[
                    RasterBox[CompressedData["
1:eJxTTMoPSmNiYGAo5gASQYnljkVFiZXBAkBOaF5xZnpeaopnXklqemqRRRIz
UJAfiDmBmAWI////vwKIfwFxMQMOAJT7+R8B6nGoSf+PCjpwqMskUl0SEP9F
UtcHxIxY1EWjqZuOQ10Mmr2TcNj7BEnNP1LNIeQeIDsZn78IhQ8x4fwfNb4a
cPgTFu8l2ORBAAC+1hnm
                    "], {{0, 15}, {9, 0}}, {0, 255}, 
                    ColorFunction -> RGBColor], 
                    BoxForm`ImageTag[
                    "Byte", ColorSpace -> "RGB", Interleaving -> True], 
                    Selectable -> False], DefaultBaseStyle -> "ImageGraphics",
                     ImageSizeRaw -> {9, 15}, PlotRange -> {{0, 9}, {0, 15}}, 
                    ImageSize -> Magnification[1]], ImageSize -> {35, 30}, 
                   Alignment -> Center], 
                  ButtonFunction :> (
                   NotebookLocate[{
                    "EIWL2-13-arrays-or-lists-of-lists.nb", ""}, 
                    FrontEnd`HistoryData :> {
                    EvaluationNotebook[]}]& ), Appearance -> None, Background -> 
                  RGBColor[0.835294, 0.364706, 0.0196078], Evaluator -> 
                  Automatic, Method -> "Preemptive"]}, 
               Dynamic[
                CurrentValue["MouseOver"]], ImageSize -> Automatic, 
               FrameMargins -> 0], "\"Arrays, or Lists of Lists\"", 
              TooltipDelay -> 0.15, LabelStyle -> "TextStyling"], 
             Annotation[#, "Arrays, or Lists of Lists", "Tooltip"]& ]}, 
           "RowWithSeparators"], Alignment -> Right], ImageSize -> {100, 30}, 
         FrameMargins -> {{0, 6}, {0, 0}}, Alignment -> Right], 
        ActionMenuBox[
         PaneSelectorBox[{True -> PanelBox[
             GraphicsBox[
              TagBox[
               RasterBox[CompressedData["
1:eJxTTMoPSmNiYGAo5gASQYnljkVFiZXBAkBOaF5xZnpeaopnXklqemqRRRIz
UBAkIQ3ELED8////Jf/pA0D2/KWTXSB7ZtPBPrA9DKOAYgAMR1Eg9gPiABpi
P6g912mcLmAAZM9XOtkFsicKiNcD8QYaYpD5UQOdXoYD+E/HMooO9iDbR7c6
BQDxtyEt
                "], {{0, 16}, {27, 0}}, {0, 255}, ColorFunction -> 
                RGBColor], 
               BoxForm`ImageTag[
               "Byte", ColorSpace -> "RGB", Interleaving -> True, 
                Magnification -> 1], Selectable -> False], DefaultBaseStyle -> 
              "ImageGraphics", ImageSize -> Magnification[1], 
              ImageSizeRaw -> {27, 16}, PlotRange -> {{0, 27}, {0, 16}}], 
             Appearance -> None, Background -> 
             RGBColor[0.835294, 0.364706, 0.0196078], ImageSize -> {35, 30}, 
             Alignment -> Center], False -> PanelBox[
             GraphicsBox[
              TagBox[
               RasterBox[CompressedData["
1:eJxTTMoPSmNiYGAo5gASQYnljkVFiZXBAkBOaF5xZnpeaopnXklqemqRRRIz
UBAkIQ3ELED8////Jf/pA0D2/KWTXSB7ZtPBPrA9DKOAYgAMR1Eg9gPiABpi
P6g912mcLmAAZM9XOtkFsicKiNcD8QYaYpD5UQOdXoYD+E/HMooO9iDbR7c6
BQDxtyEt
                "], {{0, 16}, {27, 0}}, {0, 255}, ColorFunction -> 
                RGBColor], 
               BoxForm`ImageTag[
               "Byte", ColorSpace -> "RGB", Interleaving -> True, 
                Magnification -> 1], Selectable -> False], DefaultBaseStyle -> 
              "ImageGraphics", ImageSize -> Magnification[1], 
              ImageSizeRaw -> {27, 16}, PlotRange -> {{0, 27}, {0, 16}}], 
             Appearance -> None, Background -> 
             RGBColor[0.952941, 0.509804, 0], ImageSize -> {35, 30}, 
             Alignment -> Center]}, 
          Dynamic[
           CurrentValue["MouseOver"]]], {
         "Preface" :> 
          NotebookLocate[{"EIWL2-0a-preface.nb", ""}, FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], "What Is the Wolfram Language?" :> 
          NotebookLocate[{"EIWL2-0b-what-is-the-wolfram-language.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "Practicalities of Using the Wolfram Language" :> 
          NotebookLocate[{
            "EIWL2-0c-practicalities-of-using-the-wolfram-language.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], "Other Resources" :> 
          NotebookLocate[{"EIWL2-0d-other-resources.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "\[ThickSpace]\[ThickSpace]\[ThinSpace]1\[ThickSpace]\[ThickSpace]|\
\[ThickSpace]\[ThickSpace]Starting Out: Elementary Arithmetic" :> 
          NotebookLocate[{
            "EIWL2-01-starting-out-elementary-arithmetic.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "\[ThickSpace]\[ThickSpace]\[ThinSpace]2\[ThickSpace]\[ThickSpace]|\
\[ThickSpace]\[ThickSpace]Introducing Functions" :> 
          NotebookLocate[{"EIWL2-02-introducing-functions.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "\[ThickSpace]\[ThickSpace]\[ThinSpace]3\[ThickSpace]\[ThickSpace]|\
\[ThickSpace]\[ThickSpace]First Look at Lists" :> 
          NotebookLocate[{"EIWL2-03-first-look-at-lists.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          
          "\[ThickSpace]\[ThickSpace]\[ThinSpace]4\[ThickSpace]\[ThickSpace]|\
\[ThickSpace]\[ThickSpace]Displaying Lists" :> 
          NotebookLocate[{"EIWL2-04-displaying-lists.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "\[ThickSpace]\[ThickSpace]\[ThinSpace]5\[ThickSpace]\[ThickSpace]|\
\[ThickSpace]\[ThickSpace]Operations on Lists" :> 
          NotebookLocate[{"EIWL2-05-operations-on-lists.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "\[ThickSpace]\[ThickSpace]\[ThinSpace]6\[ThickSpace]\[ThickSpace]|\
\[ThickSpace]\[ThickSpace]Making Tables" :> 
          NotebookLocate[{"EIWL2-06-making-tables.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "\[ThickSpace]\[ThickSpace]\[ThinSpace]7\[ThickSpace]\[ThickSpace]|\
\[ThickSpace]\[ThickSpace]Colors and Styles" :> 
          NotebookLocate[{"EIWL2-07-colors-and-styles.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "\[ThickSpace]\[ThickSpace]\[ThinSpace]8\[ThickSpace]\[ThickSpace]|\
\[ThickSpace]\[ThickSpace]Basic Graphics Objects" :> 
          NotebookLocate[{"EIWL2-08-basic-graphics-objects.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "\[ThickSpace]\[ThickSpace]\[ThinSpace]9\[ThickSpace]\[ThickSpace]|\
\[ThickSpace]\[ThickSpace]Interactive Manipulation" :> 
          NotebookLocate[{"EIWL2-09-interactive-manipulation.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "10\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Images" :> 
          NotebookLocate[{"EIWL2-10-images.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "11\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Strings and \
Text" :> NotebookLocate[{"EIWL2-11-strings-and-text.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "12\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Sound" :> 
          NotebookLocate[{"EIWL2-12-sound.nb", ""}, FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "13\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Arrays, or \
Lists of Lists" :> 
          NotebookLocate[{"EIWL2-13-arrays-or-lists-of-lists.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "14\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Coordinates \
and Graphics" :> 
          NotebookLocate[{"EIWL2-14-coordinates-and-graphics.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "15\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]The Scope \
of the Wolfram Language" :> 
          NotebookLocate[{
            "EIWL2-15-the-scope-of-the-wolfram-language.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "16\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Real\
\[Hyphen]World Data" :> 
          NotebookLocate[{"EIWL2-16-real-world-data.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "17\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Units" :> 
          NotebookLocate[{"EIWL2-17-units.nb", ""}, FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "18\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
Geocomputation" :> 
          NotebookLocate[{"EIWL2-18-geocomputation.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "19\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Dates and \
Times" :> NotebookLocate[{"EIWL2-19-dates-and-times.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "20\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Options" :> 
          NotebookLocate[{"EIWL2-20-options.nb", ""}, FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "21\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Graphs and \
Networks" :> 
          NotebookLocate[{"EIWL2-21-graphs-and-networks.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          
          "22\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Machine \
Learning" :> 
          NotebookLocate[{"EIWL2-22-machine-learning.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "23\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]More about \
Numbers" :> NotebookLocate[{"EIWL2-23-more-about-numbers.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "24\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]More Forms \
of Visualization" :> 
          NotebookLocate[{"EIWL2-24-more-forms-of-visualization.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "25\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Ways to \
Apply Functions" :> 
          NotebookLocate[{"EIWL2-25-ways-to-apply-functions.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "26\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Pure \
Anonymous Functions" :> 
          NotebookLocate[{"EIWL2-26-pure-anonymous-functions.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "27\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Applying \
Functions Repeatedly" :> 
          NotebookLocate[{"EIWL2-27-applying-functions-repeatedly.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "28\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Tests and \
Conditionals" :> 
          NotebookLocate[{"EIWL2-28-tests-and-conditionals.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "29\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]More about \
Pure Functions" :> 
          NotebookLocate[{"EIWL2-29-more-about-pure-functions.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "30\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Rearranging \
Lists" :> NotebookLocate[{"EIWL2-30-rearranging-lists.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          
          "31\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Parts of \
Lists" :> NotebookLocate[{"EIWL2-31-parts-of-lists.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "32\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Patterns" :> 
          NotebookLocate[{"EIWL2-32-patterns.nb", ""}, FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "33\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Expressions \
and Their Structure" :> 
          NotebookLocate[{"EIWL2-33-expressions-and-their-structure.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "34\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
Associations" :> 
          NotebookLocate[{"EIWL2-34-associations.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "35\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Natural \
Language Understanding" :> 
          NotebookLocate[{"EIWL2-35-natural-language-understanding.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "36\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Creating \
Websites and Apps" :> 
          NotebookLocate[{"EIWL2-36-creating-websites-and-apps.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "37\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Layout and \
Display" :> NotebookLocate[{"EIWL2-37-layout-and-display.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "38\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Assigning \
Names to Things" :> 
          NotebookLocate[{"EIWL2-38-assigning-names-to-things.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "39\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Immediate \
and Delayed Values" :> 
          NotebookLocate[{"EIWL2-39-immediate-and-delayed-values.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "40\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Defining \
Your Own Functions" :> 
          NotebookLocate[{"EIWL2-40-defining-your-own-functions.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "41\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]More about \
Patterns" :> 
          NotebookLocate[{"EIWL2-41-more-about-patterns.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "42\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]String \
Patterns and Templates " :> 
          NotebookLocate[{"EIWL2-42-string-patterns-and-templates.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "43\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Storing \
Things" :> NotebookLocate[{"EIWL2-43-storing-things.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "44\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Importing \
and Exporting" :> 
          NotebookLocate[{"EIWL2-44-importing-and-exporting.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "45\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Datasets" :> 
          NotebookLocate[{"EIWL2-45-datasets.nb", ""}, FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "46\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Writing \
Good Code" :> 
          NotebookLocate[{"EIWL2-46-writing-good-code.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "47\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]Debugging \
Your Code" :> 
          NotebookLocate[{"EIWL2-47-debugging-your-code.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], 
          "What We Haven\[CloseCurlyQuote]t Discussed" :> 
          NotebookLocate[{"EIWL2-what-we-havent-discussed.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], "Afterword: Being a Programmer" :> 
          NotebookLocate[{"EIWL2-being-a-programmer.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], "Answers to Exercises" :> 
          NotebookLocate[{"EIWL2-answers-to-exercises.nb", ""}, 
            FrontEnd`HistoryData :> {
              EvaluationNotebook[]}], "Index" :> 
          NotebookLocate[{"EIWL2-book-index.nb", ""}, FrontEnd`HistoryData :> {
              EvaluationNotebook[]}]}, Appearance -> None]}}, 
      GridBoxAlignment -> {"Columns" -> {{Left}}, "Rows" -> {{Center}}}, 
      GridBoxItemSize -> {
       "Columns" -> {{Automatic}}, "Rows" -> {{Automatic}}}], "Grid"]], 
   "Text", ShowCellBracket -> False, CellFrame -> {{0, 0}, {3, 0}}, 
   Selectable -> False, CellFrameColor -> 
   RGBColor[0.835294, 0.364706, 0.0196078], CellMargins -> {{0, 0}, {0, -1}}, 
   CellFrameMargins -> {{27, 0}, {1, 7}}, Background -> 
   RGBColor[0.952941, 0.509804, 0], CellTags -> "DockedCellLogoCell"], 
  Cell["SECOND EDITION", "Text", FontFamily -> "Source Sans Pro", 
   CellFrame -> {{0, 0}, {3, 0}}, ShowCellBracket -> False, 
   CellMargins -> {{0, 0}, {-1, -1}}, CellFrameMargins -> {{66, 0}, {2, 4}}, 
   CellFrameColor -> RGBColor[0.952941, 0.509804, 0], FontSize -> 10, 
   FontTracking -> "Extended", FontColor -> RGBColor[1, 0.929412, 0.721569], 
   Background -> RGBColor[0.862745, 0.309804, 0], CellTags -> 
   "DockedCellLogoCell"]},
PrintingStartingPageNumber->93,
PageHeaders->{{
   Cell[
    TextData[{
      StyleBox[
       CounterBox["Page"], "PageNumber"], 
      StyleBox[
      "    An Elementary Introduction to the Wolfram Language", "Header"]}], 
    "Header", CellMargins -> {{37, Inherited}, {Inherited, Inherited}}], None,
    None}, {None, None, 
   Cell[
    TextData[{
      StyleBox[
       CounterBox["Section"], "Header"], 
      StyleBox[" | Sound \[LongDash] wolfr.am/eiwl-", "Header"], 
      StyleBox[
       CounterBox["Section"], "Header"], 
      StyleBox["    ", "Header"], 
      StyleBox[
       CounterBox["Page"], "PageNumber"]}], "Header", 
    CellMargins -> {{Inherited, 17}, {Inherited, Inherited}}]}},
PrintingOptions->{"FirstPageHeader"->True},
PrivateNotebookOptions->{"PluginToolbarEnabled"->False},
ShowCellBracket->Automatic,
ShowAutoSpellCheck->False,
ShowPredictiveInterface->False,
ShowSyntaxStyles->False,
FrontEndVersion->"12.3 for Mac OS X x86 (64-bit) (July 30, 2021)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[StyleDefinitions -> "Default.nb"]], 
   Cell[
    CellGroupData[{
      Cell[
      "Style Environments", "Section", 
       CellChangeTimes -> {{3.64661165091824*^9, 3.646611654013329*^9}, 
         3.6485723794335403`*^9, 3.651347081728628*^9, 3.651347179462186*^9, 
         3.651347384135805*^9, 3.651348171662589*^9}], 
      Cell[
       StyleData[All, "Working"], CellInsertionPointCell -> 
       FEPrivate`FrontEndResource["FEExpressions", "CellInsertionMenu"], 
       CreateCellID -> True, $CellContext`GhostCellInEmptyNotebook -> False], 
      
      Cell[
       StyleData[All, "Printout"], 
       PrintingOptions -> {
        "PageSize" -> {504, 720}, "PrintRegistrationMarks" -> True}, 
       Magnification -> 0.6]}, Closed]], 
   Cell[
    CellGroupData[{
      Cell["Styles for Headers and Footers", "Section"], 
      Cell[
       StyleData["Header"], StyleMenuListing -> None, FontFamily -> "Verdana",
        FontSize -> 7, FontSlant -> "Plain", FontColor -> GrayLevel[0.6]], 
      Cell[
       StyleData["PageNumber"], CellMargins -> {{0, 0}, {4, 1}}, 
       StyleMenuListing -> None, FontFamily -> "Verdana", FontSize -> 7, 
       FontWeight -> "Bold", FontSlant -> "Plain", FontColor -> 
       GrayLevel[0]]}, Closed]], 
   Cell[
    CellGroupData[{
      Cell[
      "WPL Specific", "Section", 
       CellChangeTimes -> {{3.6556608433734913`*^9, 3.6556608489412746`*^9}}], 
      Cell[
       StyleData["Link"], FontFamily -> "Source Sans Pro", FontSize :> 
       If[CurrentValue[
           EvaluationCell[], "FontFamily"] === "Source Sans Pro", Inherited, 
         If[CurrentValue[
            EvaluationCell[], "FontFamily"] === "WolframCodeSans", Inherited, 
          Inherited 0.9]], FontColor -> GrayLevel[0]], 
      Cell[
       StyleData["CodeLink"], 
       ButtonBoxOptions -> {
        Appearance -> None, ButtonFunction :> (FEPrivate`FrontEndExecute[
           FrontEndToken["OpenHelpLink", {
             Part[#, 1], Null}]]& ), ButtonSource -> ButtonContents, 
         Evaluator -> None, Method -> "Preemptive"}]}, Closed]], 
   Cell[
    CellGroupData[{
      Cell[
      "Custom Styles", "Section", 
       CellChangeTimes -> {{3.646611660445352*^9, 3.646611669493331*^9}}], 
      Cell[
       StyleData["Notebook"], PageWidth :> 700, 
       PrintingOptions -> {
        "InnerOuterMargins" -> {35, 16}, "PaperOrientation" -> "Portrait", 
         "PaperSize" -> {504, 720}, 
         "PrintingMargins" -> {{20, 20}, {54, 54}}}, CellLabelAutoDelete -> 
       False, ShowSyntaxStyles -> True, 
       AutoStyleOptions -> {
        "FormattingErrorTooltips" -> False, "HighlightComments" -> False, 
         "HighlightEmphasizedSyntaxErrors" -> False, 
         "HighlightExcessArguments" -> False, "HighlightFormattingErrors" -> 
         False, "HighlightGlobalToLocalScopeConflicts" -> False, 
         "HighlightGraphicsCompatibilityProblems" -> False, 
         "HighlightLocalScopeConflicts" -> False, "HighlightMissingArguments" -> 
         False, "HighlightNoKernelPresent" -> False, 
         "HighlightOrderOfEvaluationConflicts" -> False, "HighlightStrings" -> 
         True, "HighlightSymbolContexts" -> False, "HighlightSymbolShadowing" -> 
         False, "HighlightSyntaxErrors" -> False, "HighlightUndefinedSymbols" -> 
         False, "HighlightUnknownOptions" -> False, 
         "HighlightUnwantedAssignments" -> False}, MenuSortingValue -> None, 
       GraphicsBoxOptions -> {ImageSize -> 240}, 
       Graphics3DBoxOptions -> {ImageSize -> 240}], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["Title"], CellMargins -> {{64, Inherited}, {50, 72}}, 
          MenuCommandKey :> None, FontFamily -> "Verdana", FontSize -> 42, 
          FontWeight -> "Bold", FontColor -> GrayLevel[0.4]], 
         Cell[
          StyleData["Title", "Printout"], 
          CellMargins -> {{64, Inherited}, {50, 72}}]}, Closed]], 
      Cell[
       StyleData["Chapter"], MenuCommandKey :> None], 
      Cell[
       StyleData["Subchapter"], MenuCommandKey :> None], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["Section"], CellFrame -> {{0, 0}, {0, 0}}, CellDingbat -> 
          Cell[
            TextData[
             CounterBox["Section"], 
             StyleBox[
             "|\[NegativeThinSpace]", FontColor -> 
              RGBColor[
               0.9019607843137255, 0.4117647058823529, 0.027450980392156862`],
               FontFamily -> "Courier", FontSize -> 36, FontWeight -> Plain]],
             "SectionNumber"], CellMargins -> {{96, Inherited}, {12, 44}}, 
          Evaluatable -> True, CellGroupingRules -> {"SectionGrouping", 60}, 
          PageBreakAbove -> True, ShowCellLabel -> False, 
          CellFrameLabelMargins -> {{2, Inherited}, {Inherited, Inherited}}, 
          CellChangeTimes -> {3.652728456208679*^9, 3.652728527108994*^9}, 
          CounterIncrements -> "Section", 
          CounterAssignments -> {{"Subsection", 0}, {"Subsubsection", 0}, {
            "Exercise", 0}}, FontFamily -> "Source Sans Pro Semibold", 
          FontSize -> 26, FontSlant -> "Plain", FontColor -> 
          RGBColor[
           0.9019607843137255, 0.4117647058823529, 0.027450980392156862`], 
          InterpretationBoxOptions -> {Editable -> True}], 
         Cell[
          StyleData["Section", "Printout"], 
          CellMargins -> {{96, Inherited}, {12, 2}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["SectionDoubleDigit"], CellFrame -> {{0, 0}, {0, 0}}, 
          CellDingbat -> Cell[
            TextData[
             CounterBox["Section"], 
             StyleBox[
             "|\[NegativeThinSpace]", FontColor -> 
              RGBColor[
               0.9019607843137255, 0.4117647058823529, 0.027450980392156862`],
               FontFamily -> "Courier", FontSize -> 36, FontWeight -> Plain]],
             "SectionNumber"], CellMargins -> {{112, Inherited}, {12, 44}}, 
          Evaluatable -> True, CellGroupingRules -> {"SectionGrouping", 60}, 
          PageBreakAbove -> True, ShowCellLabel -> False, 
          CellFrameLabelMargins -> {{2, Inherited}, {Inherited, Inherited}}, 
          CellChangeTimes -> {3.652728456208679*^9, 3.652728527108994*^9}, 
          CounterIncrements -> "Section", 
          CounterAssignments -> {{"Subsection", 0}, {"Subsubsection", 0}, {
            "Exercise", 0}}, FontFamily -> "Source Sans Pro Semibold", 
          FontSize -> 26, FontSlant -> "Plain", FontColor -> 
          RGBColor[
           0.9019607843137255, 0.4117647058823529, 0.027450980392156862`], 
          InterpretationBoxOptions -> {Editable -> True}], 
         Cell[
          StyleData["SectionDoubleDigit", "Printout"], 
          CellMargins -> {{112, Inherited}, {12, 2}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["SectionNumber"], CellBaseline -> Baseline, FontFamily -> 
          "Source Sans Pro Semibold", FontSize -> 26, FontColor -> 
          RGBColor[
           0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]], 
         Cell[
          StyleData["SectionNumber", "Printout"]]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["SectionInline"], FontFamily -> 
          "Source Sans Pro Semibold", FontColor -> 
          RGBColor[
           0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]], 
         Cell[
          StyleData["SectionInline", "Printout"]]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["SectionNoDingbat"], 
          CellMargins -> {{64, Inherited}, {12, 44}}, Evaluatable -> True, 
          CellGroupingRules -> {"SectionGrouping", 60}, PageBreakAbove -> 
          True, ShowCellLabel -> False, 
          CellFrameLabelMargins -> {{2, Inherited}, {Inherited, Inherited}}, 
          CellChangeTimes -> {3.652728456208679*^9, 3.652728527108994*^9}, 
          CounterIncrements -> "Section", 
          CounterAssignments -> {{"Subsection", 0}, {"Subsubsection", 0}, {
            "Exercise", 0}}, FontFamily -> "Source Sans Pro Semibold", 
          FontSize -> 26, FontSlant -> "Plain", FontColor -> 
          RGBColor[
           0.9019607843137255, 0.4117647058823529, 0.027450980392156862`], 
          InterpretationBoxOptions -> {Editable -> True}], 
         Cell[
          StyleData["SectionNoDingbat", "Printout"], 
          CellMargins -> {{64, Inherited}, {12, 2}}]}, Closed]], 
      Cell[
       StyleData["AnswerKeyNumberedSection"], 
       CellMargins -> {{64, Inherited}, {8, 28}}, Evaluatable -> True, 
       CellGroupingRules -> {"SectionGrouping", 60}, PageBreakAbove -> True, 
       ShowCellLabel -> False, 
       CellFrameLabelMargins -> {{2, Inherited}, {Inherited, Inherited}}, 
       CellChangeTimes -> {3.652728456208679*^9, 3.652728527108994*^9}, 
       CounterIncrements -> "Section", 
       CounterAssignments -> {{"Subsection", 0}, {"Subsubsection", 0}, {
         "Exercise", 0}}, FontFamily -> "Source Sans Pro Semibold", FontSize -> 
       18, FontSlant -> "Plain", FontColor -> 
       RGBColor[
        0.9019607843137255, 0.4117647058823529, 0.027450980392156862`], 
       InterpretationBoxOptions -> {Editable -> True}], 
      Cell[
       StyleData["AnswerKeySectionNumber"], CellBaseline -> Baseline, 
       FontFamily -> "Source Sans Pro Semibold", FontSize -> 18, FontColor -> 
       RGBColor[
        0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]], 
      Cell[
       StyleData["IndexLinkGuide"], 
       CellMargins -> {{64, Inherited}, {48, 28}}, Evaluatable -> True, 
       CellGroupingRules -> {"SectionGrouping", 60}, PageBreakAbove -> True, 
       ShowCellLabel -> False, 
       CellFrameLabelMargins -> {{2, Inherited}, {Inherited, Inherited}}, 
       CellChangeTimes -> {3.652728456208679*^9, 3.652728527108994*^9}, 
       CounterIncrements -> "Section", 
       CounterAssignments -> {{"Subsection", 0}, {"Subsubsection", 0}, {
         "Exercise", 0}}, FontFamily -> "Source Sans Pro", FontSize -> 14, 
       FontSlant -> "Plain", FontColor -> 
       RGBColor[
        0.9019607843137255, 0.4117647058823529, 0.027450980392156862`], 
       InterpretationBoxOptions -> {Editable -> True}], 
      Cell[
       StyleData["Index"], MenuSortingValue -> 10000, FontFamily -> 
       "Source Sans Pro", FontSize -> 11], 
      Cell[
       StyleData["IndexSubentry"], MenuSortingValue -> 10000, FontFamily -> 
       "Source Sans Pro", FontSize -> 11], 
      Cell[
       StyleData["FunctionName"], MenuSortingValue -> 10000, FontFamily -> 
       "Source Sans Pro Semibold", FontWeight -> "Bold"], 
      Cell[
       StyleData["IndexColumn"], 
       CellMargins -> {{66, 0}, {Inherited, Inherited}}, TabSpacings -> {2.}, 
       MenuSortingValue -> 10000, FontFamily -> "Source Sans Pro", 
       GridBoxOptions -> {
        GridBoxAlignment -> {"Columns" -> {{Left}}, "Rows" -> {{Top}}}, 
         GridBoxItemSize -> {"Columns" -> {{25}}}, 
         GridBoxSpacings -> {"Columns" -> {{2}}}}], 
      Cell[
       StyleData["IndexSubsection"], 
       CellMargins -> {{64, Inherited}, {9, 28}}, 
       CellGroupingRules -> {"SectionGrouping", 40}, MenuSortingValue -> 
       10000, FontFamily -> "Source Sans Pro", FontSize -> 24], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["Subsection"], CellMargins -> {{64, Inherited}, {12, 32}},
           FontFamily -> "Source Sans Pro Semibold", FontSize -> 18, 
          FontColor -> 
          RGBColor[
           0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]], 
         Cell[
          StyleData["Subsection", "Printout"], 
          CellMargins -> {{64, Inherited}, {12, 32}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["ResourcesSubsection"], 
          CellMargins -> {{64, Inherited}, {4, 16}}, FontFamily -> 
          "Source Sans Pro Semibold", FontSize -> 15, FontSlant -> "Plain", 
          FontColor -> GrayLevel[0]], 
         Cell[
          StyleData["ResourcesSubsection", "Printout"], 
          CellMargins -> {{64, Inherited}, {4, 16}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["Subsubsection"], 
          CellMargins -> {{64, Inherited}, {6, 12}}, FontFamily -> 
          "Source Sans Pro Semibold", FontSize -> 16, FontColor -> 
          RGBColor[
           0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]], 
         Cell[
          StyleData["Subsubsection", "Printout"], 
          CellMargins -> {{64, Inherited}, {6, 12}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["VocabularySection"], PageWidth :> 700, 
          CellFrame -> {{0, 0}, {0, 3}}, 
          CellMargins -> {{64, Inherited}, {8, 24}}, 
          CellGroupingRules -> {"SectionGrouping", 60}, PageBreakBelow -> 
          False, CellFrameMargins -> {{8, 8}, {4, 6}}, CellFrameColor -> 
          RGBColor[0.901961, 0.411765, 0.027451], FontFamily -> 
          "Source Sans Pro", FontSize -> 18, FontSlant -> "Italic", FontColor -> 
          RGBColor[
           0.9019607843137255, 0.4117647058823529, 0.027450980392156862`], 
          GridBoxOptions -> {GridBoxAlignment -> {}}], 
         Cell[
          StyleData["VocabularySection", "Printout"], 
          CellMargins -> {{64, Inherited}, {8, 24}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["VocabularyTable", StyleDefinitions -> StyleData["Text"]],
           SpanMaxSize -> 1, FontFamily -> "Source Sans Pro Light", FontColor -> 
          GrayLevel[0.8], 
          GridBoxOptions -> {
           GridBoxAlignment -> {"Columns" -> {{Left}}}, 
            GridBoxDividers -> {"Columns" -> {{False}}, "Rows" -> {False, {
                 Directive[
                  Thickness[0.5], "VocabDivColor"]}, False}}, 
            GridBoxItemStyle -> {
             "Columns" -> {"VocabularyCode", Inherited, Inherited}, 
              "Rows" -> {{Automatic}}}, 
            GridBoxSpacings -> {
             "Columns" -> {{0, 1, 2, 1, 2}}, "Rows" -> {{1}}}}], 
         Cell[
          StyleData["VocabularyTable", "Printout"]]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData[
          "VocabularyTable3Column", StyleDefinitions -> StyleData["Text"]], 
          SpanMaxSize -> 1, FontFamily -> "Source Sans Pro Light", FontColor -> 
          GrayLevel[0.8], 
          GridBoxOptions -> {
           GridBoxAlignment -> {"Columns" -> {{Left}}}, 
            GridBoxDividers -> {"Columns" -> {{False}}, "Rows" -> {False, {
                 Directive[
                  Thickness[0.5], "VocabDivColor"]}, False}}, 
            GridBoxItemStyle -> {
             "Columns" -> {
               "VocabularyCode", Inherited, "VocabularyCode", Inherited, 
                Inherited}, "Rows" -> {{Automatic}}}, 
            GridBoxSpacings -> {
             "Columns" -> {{0, 1, 2, 1, 2}}, "Rows" -> {{1}}}}], 
         Cell[
          StyleData["VocabularyTable3Column", "Printout"]]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["VocabularyText"], FontFamily -> "Source Sans Pro", 
          FontSize -> 14, FontWeight -> "Plain", FontSlant -> "Plain", 
          FontColor -> GrayLevel[0.25]], 
         Cell[
          StyleData["VocabularyText", "Printout"], 
          CellMargins -> {{66, 10}, {7, 7}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["VocabularyCode", StyleDefinitions -> StyleData["Text"]], 
          Deletable -> False, FontFamily -> "Source Sans Pro", FontSize -> 14,
           FontWeight -> "Bold", FontColor -> GrayLevel[0.25], 
          GridBoxOptions -> {
           GridBoxAlignment -> {"Columns" -> {{Left}}}, 
            GridBoxDividers -> {
             "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
              "Rows" -> {True, {True}, True}, "RowsIndexed" -> {}}}], 
         Cell[
          StyleData["VocabularyCode", "Printout"], FontColor -> 
          GrayLevel[0.25]]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["VocabDivColor"], FontColor -> GrayLevel[0.8]], 
         Cell[
          StyleData["VocabDivColor", "Printout"]]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["ExerciseSection"], PageWidth :> 700, 
          CellFrame -> {{0, 0}, {0, 3}}, 
          CellMargins -> {{64, Inherited}, {8, 30}}, Evaluatable -> True, 
          CellGroupingRules -> {"SectionGrouping", 60}, PageBreakBelow -> 
          False, ShowCellLabel -> False, CellFrameMargins -> {{8, 8}, {4, 6}},
           CellFrameColor -> RGBColor[0.901961, 0.411765, 0.027451], 
          CounterAssignments -> {{"Subsection", 0}, {"Subsubsection", 0}, {
            "Exercise", 0}}, FontFamily -> "Source Sans Pro", FontSize -> 18, 
          FontSlant -> "Italic", FontColor -> 
          RGBColor[
           0.9019607843137255, 0.4117647058823529, 0.027450980392156862`], 
          GridBoxOptions -> {GridBoxAlignment -> {}}], 
         Cell[
          StyleData["ExerciseSection", "Printout"], 
          CellMargins -> {{64, Inherited}, {8, 30}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["Exercise"], PageWidth :> 700, CellDingbat -> Cell[
            TextData[{"  ", 
              CounterBox["Section"], ".", 
              CounterBox["Exercise"]}], "ExerciseNumber", FontColor -> 
            RGBColor[
             0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]], 
          ShowGroupOpener -> True, CellMargins -> {{108, Inherited}, {4, 8}}, 
          CellGroupingRules -> {"SectionGrouping", 67}, CounterIncrements -> 
          "Exercise", MenuSortingValue -> 10020, MenuCommandKey :> "2", 
          FontFamily -> "Source Sans Pro", FontSize -> 14, FontWeight -> 
          "Plain", FontSlant -> "Plain", FontColor -> GrayLevel[0.25]], 
         Cell[
          StyleData["Exercise", "Printout"], ShowGroupOpener -> False, 
          CellMargins -> {{110, Inherited}, {4, 8}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["ExerciseNumber"], FontFamily -> "Source Sans Pro", 
          FontSize -> 14, FontWeight -> "Bold", FontColor -> GrayLevel[0.25]], 
         Cell[
          StyleData["ExerciseNumber", "Printout"]]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["ExerciseSubsection"], PageWidth :> 500, 
          CellFrame -> {{False, False}, {1, False}}, 
          CellMargins -> {{64, Inherited}, {8, 18}}, 
          CellGroupingRules -> {"SectionGrouping", 60}, CellFrameColor -> 
          RGBColor[0.65, 0.6, 0.6], FontFamily -> "Source Sans Pro", FontSize -> 
          18, FontSlant -> "Italic", FontColor -> RGBColor[0.65, 0.6, 0.6]], 
         Cell[
          StyleData["ExerciseSubsection", "Printout"], 
          CellFrame -> {{False, False}, {1, False}}, CellFrameColor -> 
          RGBColor[0.65, 0.6, 0.6]]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["ExerciseNote"], PageWidth :> 700, 
          CellMargins -> {{64, Inherited}, {4, 8}}, PageBreakBelow -> False, 
          MenuSortingValue -> 10050, FontFamily -> "Source Sans Pro", 
          FontSize -> 14, FontWeight -> "Plain", FontSlant -> "Plain", 
          FontColor -> GrayLevel[0.25], $CellContext`ReturnCreatesNewCell -> 
          True], 
         Cell[
          StyleData["ExerciseNote", "Printout"], 
          CellMargins -> {{64, Inherited}, {4, 8}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["QASection"], PageWidth :> 700, 
          CellFrame -> {{0, 0}, {0, 3}}, 
          CellMargins -> {{64, Inherited}, {8, 30}}, 
          CellGroupingRules -> {"SectionGrouping", 60}, PageBreakBelow -> 
          False, CellFrameMargins -> {{8, 8}, {4, 6}}, CellFrameColor -> 
          RGBColor[0.901961, 0.411765, 0.027451], 
          CounterAssignments -> {{"Subsection", 0}, {"Subsubsection", 0}, {
            "Question", 0}}, FontFamily -> "Source Sans Pro", FontSize -> 18, 
          FontSlant -> "Italic", FontColor -> 
          RGBColor[
           0.9019607843137255, 0.4117647058823529, 0.027450980392156862`], 
          GridBoxOptions -> {GridBoxAlignment -> {}}], 
         Cell[
          StyleData["QASection", "Printout"], 
          CellMargins -> {{64, Inherited}, {8, 30}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["Question"], PageWidth :> 700, 
          CellMargins -> {{64, Inherited}, {2, 12}}, PageBreakBelow -> False, 
          CounterIncrements -> "Question", MenuSortingValue -> 10030, 
          MenuCommandKey :> "3", FontFamily -> "Source Sans Pro", FontSize -> 
          14, FontWeight -> "Bold", FontSlant -> 
          "Plain", $CellContext`ReturnCreatesNewCell -> 
          True, $CellContext`DefaultReturnCreatedCellStyle -> "Answer"], 
         Cell[
          StyleData["Question", "Printout"], 
          CellMargins -> {{65, Inherited}, {2, 12}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["Answer"], PageWidth :> 700, 
          CellMargins -> {{66, 10}, {4, 1}}, LineSpacing -> {1.125, 1., 2.}, 
          MenuSortingValue -> 10032, FontFamily -> "Source Sans Pro", 
          FontSize -> 14, FontWeight -> "Plain", FontSlant -> "Plain", 
          FontColor -> GrayLevel[0.25], $CellContext`ReturnCreatesNewCell -> 
          True, $CellContext`DefaultReturnCreatedCellStyle -> "Question"], 
         Cell[
          StyleData["Answer", "Printout"], 
          CellMargins -> {{66, 10}, {4, 1}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["QuestionNumber"], FontFamily -> "Source Sans Pro", 
          FontSize -> 14, FontWeight -> "Bold", FontColor -> GrayLevel[0.25]], 
         Cell[
          StyleData["QuestionNumber", "Printout"]]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["TechNoteSection"], PageWidth :> 700, 
          CellFrame -> {{0, 0}, {0, 3}}, 
          CellMargins -> {{64, Inherited}, {8, 30}}, 
          CellGroupingRules -> {"SectionGrouping", 60}, PageBreakBelow -> 
          False, CellFrameMargins -> {{8, 8}, {4, 6}}, CellFrameColor -> 
          RGBColor[0.901961, 0.411765, 0.027451], FontFamily -> 
          "Source Sans Pro", FontSize -> 18, FontSlant -> "Italic", FontColor -> 
          RGBColor[
           0.9019607843137255, 0.4117647058823529, 0.027450980392156862`], 
          GridBoxOptions -> {GridBoxAlignment -> {}}], 
         Cell[
          StyleData["TechNoteSection", "Printout"], 
          CellMargins -> {{64, Inherited}, {8, 30}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["TechNote"], PageWidth :> 700, CellDingbat -> Cell[
            TextData[{"\[FilledSmallSquare]"}], FontSize -> 12, FontColor -> 
            GrayLevel[0.5]], CellMargins -> {{78, Inherited}, {4, 8}}, 
          CellChangeTimes -> {{3.651515120816633*^9, 3.651515125814807*^9}, 
            3.6515155174947033`*^9}, LineSpacing -> {1.25, 1., 2.}, 
          MenuSortingValue -> 10040, FontFamily -> "Source Sans Pro", 
          FontSize -> 14, FontWeight -> "Plain", FontSlant -> "Plain", 
          FontColor -> GrayLevel[0.25], $CellContext`ReturnCreatesNewCell -> 
          True], 
         Cell[
          StyleData["TechNote", "Printout"], 
          CellMargins -> {{82, Inherited}, {4, 8}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["MoreExploreSection"], PageWidth :> 700, 
          CellFrame -> {{0, 0}, {0, 3}}, 
          CellMargins -> {{64, Inherited}, {8, 30}}, 
          CellGroupingRules -> {"SectionGrouping", 60}, PageBreakBelow -> 
          False, CellFrameMargins -> {{8, 8}, {4, 6}}, CellFrameColor -> 
          RGBColor[0.901961, 0.411765, 0.027451], FontFamily -> 
          "Source Sans Pro", FontSize -> 18, FontSlant -> "Italic", FontColor -> 
          RGBColor[
           0.9019607843137255, 0.4117647058823529, 0.027450980392156862`], 
          GridBoxOptions -> {GridBoxAlignment -> {}}], 
         Cell[
          StyleData["MoreExploreSection", "Printout"], 
          CellMargins -> {{64, Inherited}, {8, 30}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["MoreExplore"], PageWidth :> 700, 
          CellMargins -> {{64, Inherited}, {4, 8}}, PageBreakBelow -> False, 
          MenuSortingValue -> 10050, FontFamily -> "Source Sans Pro", 
          FontSize -> 14, FontWeight -> "Plain", FontSlant -> "Plain", 
          FontColor -> GrayLevel[0], $CellContext`ReturnCreatesNewCell -> 
          True], 
         Cell[
          StyleData["MoreExplore", "Printout"], 
          CellMargins -> {{64, Inherited}, {4, 8}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["MoreExploreShortURL"], 
          CellMargins -> {{64, Inherited}, {4, 8}}, PageBreakBelow -> False, 
          FontFamily -> "Source Sans Pro Light", FontSize -> 14, FontSlant -> 
          "Plain", FontColor -> GrayLevel[0]], 
         Cell[
          StyleData["MoreExploreShortURL", "Printout"], 
          CellMargins -> {{64, Inherited}, {4, 8}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["Text"], CellMargins -> {{66, 10}, {4, 14}}, FontFamily -> 
          "Baskerville", FontSize -> 18, FontColor -> 
          GrayLevel[0], $CellContext`ReturnCreatesNewCell -> True], 
         Cell[
          StyleData["Text", "Printout"], CellMargins -> {{66, 10}, {7, 7}}]}, 
        Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["ResourcesText"], CellMargins -> {{66, 10}, {2, 0}}, 
          FontFamily -> "Source Sans Pro", FontSize -> 14, FontWeight -> 
          "Plain", FontSlant -> "Plain", FontColor -> 
          GrayLevel[0], $CellContext`ReturnCreatesNewCell -> True], 
         Cell[
          StyleData["ResourcesText", "Printout"], 
          CellMargins -> {{66, 10}, {2, 0}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["ResourcesLink"], CellMargins -> {{66, 10}, {2, 0}}, 
          FontFamily -> "Source Sans Pro", FontSize -> 14, FontWeight -> 
          "Plain", FontSlant -> "Italic", FontColor -> 
          GrayLevel[0], $CellContext`ReturnCreatesNewCell -> True], 
         Cell[
          StyleData["ResourcesLink", "Printout"], 
          CellMargins -> {{66, 10}, {2, 0}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["CodeCaption"], CellFrame -> {{0, 0}, {0, 0}}, 
          CellMargins -> {{66, 10}, {4, 12}}, PageBreakBelow -> False, 
          CellFrameColor -> GrayLevel[0.4], MenuSortingValue -> 10010, 
          MenuCommandKey :> "1", FontFamily -> "Source Sans Pro", FontSize -> 
          16, FontColor -> GrayLevel[0.25]], 
         Cell[
          StyleData["CodeCaption", "Printout"], 
          CellMargins -> {{66, 10}, {4, 12}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["Input"], CellMargins -> {{99, 10}, {3, 7}}, SpanMaxSize -> 
          1, FontSize -> 15], 
         Cell[
          StyleData["Input", "Printout"], CellMargins -> {{99, 10}, {3, 7}}]},
         Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["Output"], CellMargins -> {{99, 10}, {14, 7}}, FontSize -> 
          15], 
         Cell[
          StyleData["Output", "Printout"], 
          CellMargins -> {{99, 10}, {14, 7}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["InputOnly"], CellMargins -> {{99, 10}, {25, 10}}, 
          SpanMaxSize -> 1, MenuSortingValue -> 1500, FontSize -> 15], 
         Cell[
          StyleData["InputOnly", "Printout"], 
          CellMargins -> {{99, 10}, {15, 10}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["CellLabel"], 
          CellLabelMargins -> {{12, Inherited}, {Inherited, Inherited}}, 
          LanguageCategory -> None, StyleMenuListing -> None, FontFamily -> 
          "Helvetica Neue", FontSize -> 9, FontWeight -> "Plain", FontSlant -> 
          "Plain", FontColor -> GrayLevel[0.5]], 
         Cell[
          StyleData["CellLabel", "Printout"], FontSize -> 8, FontColor -> 
          GrayLevel[0.5]]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["Message"], Editable -> False, 
          CellMargins -> {{66, Inherited}, {Inherited, Inherited}}, 
          CellGroupingRules -> "OutputGrouping", TextClipboardType -> 
          "PlainText", PageBreakWithin -> False, GroupPageBreakWithin -> 
          False, GeneratedCell -> True, CellAutoOverwrite -> True, 
          ShowCellLabel -> False, DefaultFormatType -> 
          DefaultOutputFormatType, ContextMenu -> 
          FEPrivate`FrontEndResource["ContextMenus", "Output"], 
          ShowAutoStyles -> False, ShowSyntaxStyles -> False, 
          AutoStyleOptions -> {
           "StringStyle" -> None, "SyntaxErrorStyle" -> None}, 
          HyphenationOptions -> {"HyphenationCharacter" -> "\[Continuation]"},
           AutoItalicWords -> {}, AutoQuoteCharacters -> {}, 
          PasteAutoQuoteCharacters -> {}, LanguageCategory -> None, 
          FormatType -> InputForm, ShowStringCharacters -> False, 
          CounterIncrements -> "Message", MenuSortingValue -> 1600, 
          StyleMenuListing -> None, FontFamily :> 
          CurrentValue["PanelFontFamily"], FontSize -> 11, FontColor -> 
          RGBColor[0.5072, 0.16933, 0.15221], $CellContext`StripStyleOnPaste -> 
          True], 
         Cell[
          StyleData["Message", "Printout"], 
          CellMargins -> {{100, Inherited}, {Inherited, Inherited}}, 
          CellFrameLabels -> {{None, None}, {None, None}}, FontColor -> 
          GrayLevel[0]]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["ExerciseInput", StyleDefinitions -> StyleData["Input"]], 
          CellMargins -> {{88, 10}, {4, 7}}, ShowCellLabel -> False, 
          SpanMaxSize -> 1, FontSize -> 
          14, $CellContext`GeneratedCellStyles -> {
           "Output" -> "ExerciseOutput"}], 
         Cell[
          StyleData["ExerciseInput", "Printout"], 
          CellMargins -> {{88, 10}, {4, 7}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData[
          "ExerciseOutput", StyleDefinitions -> StyleData["Output"]], 
          CellMargins -> {{88, 10}, {10, 4}}, ShowCellLabel -> False, 
          FontSize -> 14], 
         Cell[
          StyleData["ExerciseOutput", "Printout"], 
          CellMargins -> {{88, 10}, {10, 4}}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["Picture"], CellMargins -> {{66, 10}, {10, 10}}, 
          CellGroupingRules -> "GraphicsGrouping", StyleMenuListing -> None], 
         
         Cell[
          StyleData["Picture", "Printout"], 
          CellMargins -> {{66, 10}, {8, 8}}]}, Closed]], 
      Cell[
       StyleData["ExerciseSummaryCell"], PageWidth :> 700, CellFrame -> 0, 
       CellMargins -> {{64, Inherited}, {8, 10}}, 
       CellGroupingRules -> {"SectionGrouping", 64}, PageBreakBelow -> False, 
       ShowCellLabel -> False, CellFrameMargins -> 0, ShowStringCharacters -> 
       False, FontFamily -> "Source Sans Pro", FontSize -> 18, FontColor -> 
       RGBColor[1, 1, 1], 
       GridBoxOptions -> {GridBoxBackground -> {"Columns" -> {{
              RGBColor[0.990997, 0.539528, 0.155779], 
              RGBColor[0.989487, 0.434699, 0.0320439]}}, "Rows" -> {{None}}}, 
         GridBoxItemSize -> {"Columns" -> {
             Scaled[0.7], 
             Scaled[0.3]}, "Rows" -> {{Automatic}}}}], 
      Cell[
       StyleData["ExerciseSummaryHeading"], ShowStringCharacters -> False, 
       FontFamily -> "Source Sans Pro", FontSize -> 18, FontColor -> 
       RGBColor[1, 1, 1]], 
      Cell[
       StyleData["ExerciseSummarySubheading"], ShowStringCharacters -> False, 
       FontFamily -> "Source Sans Pro", FontSize -> 15, FontColor -> 
       RGBColor[1., 0.9294117647058824, 0.7215686274509804]], 
      Cell[
       StyleData["ExercisePreviewButtonText"], ShowStringCharacters -> False, 
       FontFamily -> "Source Sans Pro", FontSize -> 14, FontSlant -> "Plain", 
       FontColor -> RGBColor[0.6, 0.6, 0.6]], 
      Cell[
       StyleData["ExercisePreviewButtonTextHighlight"], ShowStringCharacters -> 
       False, FontFamily -> "Source Sans Pro", FontSize -> 14, FontSlant -> 
       "Plain", FontColor -> 
       RGBColor[
        0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]]}, 
     Open]], 
   Cell[
    CellGroupData[{
      Cell[
      "Inline and special formatting", "Section", 
       CellChangeTimes -> {{3.646681732592888*^9, 3.646681749414035*^9}}], 
      Cell[
       StyleData["Graphics"], MenuSortingValue -> None], 
      Cell[
       StyleData["Graphics3D"], MenuSortingValue -> None], 
      Cell[
       StyleData["StandardForm"], 
       InputAutoReplacements -> {
        "->" -> "\[Rule]", ":>" -> "\[RuleDelayed]", "<=" -> "\[LessEqual]", 
         ">=" -> "\[GreaterEqual]", "!=" -> "\[NotEqual]", "==" -> "\[Equal]",
          "@>" -> "\\[InlinePart]", "<->" -> "\[UndirectedEdge]", ParentList},
        LineSpacing -> {1.25, 0, 2}, SingleLetterItalics -> False, 
       StyleMenuListing -> None, FontFamily -> "Source Sans Pro", 
       PrivateFontOptions -> {"OperatorSubstitution" -> True}], 
      Cell[
       StyleData["OutputForm"], PageWidth :> DirectedInfinity[1], 
       TextAlignment -> Left, LineSpacing -> {0.6, 1}, 
       AutoQuoteCharacters -> {}, PasteAutoQuoteCharacters -> {}, 
       StyleMenuListing -> None, FontFamily -> "Source Sans Pro", 
       PrivateFontOptions -> {"OperatorSubstitution" -> True}], 
      Cell[
       StyleData["InputForm"], PageWidth :> DirectedInfinity[1], 
       InputAutoReplacements -> {}, AllowInlineCells -> False, Hyphenation -> 
       False, AutoQuoteCharacters -> {}, PasteAutoQuoteCharacters -> {}, 
       AutoSpacing -> False, NumberMarks -> True, AutoNumberFormatting -> 
       False, StyleMenuListing -> None, FontFamily -> "Source Sans Pro", 
       PrivateFontOptions -> {"OperatorSubstitution" -> True}], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["DetailsLink"], FontFamily -> "Source Sans Pro", FontSize :> 
          If[CurrentValue[
              EvaluationCell[], "FontFamily"] === "Source Sans Pro", 
            Inherited, 
            If[CurrentValue[
               EvaluationCell[], "FontFamily"] === "WolframCodeSans", 
             Inherited, Inherited 0.9]], FontColor -> GrayLevel[0], 
          ButtonBoxOptions -> {
           ButtonFunction :> (
             Function[$CellContext`t, 
              NotebookFind[$CellContext`t, "Notes", All, CellStyle]; 
              NotebookFind[$CellContext`t, "NotesSection", All, CellStyle, 
                AutoScroll -> Top]][
              Documentation`HelpLookup[
               StringJoin["ref/", #]]]& ), ButtonSource -> ButtonContents, 
            Evaluator -> Automatic}], 
         Cell[
          StyleData["DetailsLink", "Printout"], FontFamily -> 
          "Source Sans Pro", FontSize :> If[CurrentValue[
              EvaluationCell[], "FontFamily"] === "Source Sans Pro", 
            Inherited, 
            If[CurrentValue[
               EvaluationCell[], "FontFamily"] === "WolframCodeSans", 
             Inherited, Inherited 0.88]], FontColor -> GrayLevel[0]]}, 
        Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["LinkAlt"], FontFamily -> "Source Sans Pro", FontSize :> 
          If[CurrentValue[
              EvaluationCell[], "FontFamily"] === "Source Sans Pro", 
            Inherited, 
            If[CurrentValue[
               EvaluationCell[], "FontFamily"] === "WolframCodeSans", 
             Inherited, Inherited 0.9]], FontColor -> GrayLevel[0]], 
         Cell[
          StyleData["LinkAlt", "Printout"], FontFamily -> "Source Sans Pro", 
          FontSize :> If[CurrentValue[
              EvaluationCell[], "FontFamily"] === "Source Sans Pro", 
            Inherited, 
            If[CurrentValue[
               EvaluationCell[], "FontFamily"] === "WolframCodeSans", 
             Inherited, Inherited 0.88]], FontColor -> GrayLevel[0]]}, 
        Closed]], 
      Cell[
       CellGroupData[{
         Cell[
          StyleData["Hyperlink"], MenuSortingValue -> None, FontColor -> 
          RGBColor[0.3, 0.35, 0.5]], 
         Cell[
          StyleData["Hyperlink", "Printout"]]}, Closed]], 
      Cell[
       StyleData[
       "IndexNavigationLink", StyleDefinitions -> StyleData["Hyperlink"]], 
       MenuSortingValue -> None, FontColor -> 
       RGBColor[
        0.9019607843137255, 0.4117647058823529, 0.027450980392156862`]], 
      Cell[
       StyleData["Crosslink"], 
       ButtonBoxOptions -> {ButtonFunction :> (FrontEndExecute[{
            
            NotebookLocate[#2, "OpenInNewWindow" -> 
             CurrentValue["HyperlinkModifierKey"]]}]& ), Evaluator -> None, 
         Method -> "Queued"}], 
      Cell[
       StyleData["VocabularyLink"], FontColor -> GrayLevel[0], 
       ButtonBoxOptions -> {ButtonFunction :> (FrontEndExecute[{
            
            NotebookLocate[#2, "OpenInNewWindow" -> 
             CurrentValue["HyperlinkModifierKey"]]}]& ), Evaluator -> None, 
         Method -> "Queued"}], 
      Cell[
       StyleData["InlineCode"], SpanMaxSize -> 1, FontColor -> GrayLevel[0]], 
      
      Cell[
       StyleData["InlineCodeText"], SpanMaxSize -> 1, FontFamily -> 
       "Source Sans Pro", FontSize -> 0.95 Inherited], 
      Cell[
       StyleData["NoLabel"], ShowCellLabel -> 
       False, $CellContext`GeneratedCellStyles -> {
        "Output" -> "ExerciseOutput"}], 
      Cell[
       StyleData["MenuName"], FontFamily -> "Verdana", FontWeight -> "Bold", 
       FontColor -> GrayLevel[0.286275]], 
      Cell[
       StyleData["KeyEvent"], CellFrame -> {{0.5, 0.5}, {0.5, 0.5}}, 
       CellFrameMargins -> {{2, 2}, {2, 2}}, CellFrameColor -> GrayLevel[0.7],
        FontFamily -> "Verdana", FontColor -> GrayLevel[0.4]], 
      Cell[
       StyleData["KeyEventDelimiter"], FontColor -> GrayLevel[0.5]], 
      Cell[
       CellGroupData[{
         Cell["Key Styles", "Subsubsection"], 
         Cell[
          StyleData["Key0"], 
          TemplateBoxOptions -> {DisplayFunction -> (FrameBox[
              TagBox[
               GridBox[{{
                  ItemBox[
                   
                   TemplateBox[{"\[ThickSpace]", #, "\[ThickSpace]"}, 
                    "RowDefault"], Background -> GrayLevel[1], StripOnInput -> 
                   False]}}, 
                GridBoxAlignment -> {
                 "Columns" -> {{Center}}, "Rows" -> {{Baseline}}}, AutoDelete -> 
                False, GridBoxDividers -> {"Rows" -> {{None}}}, 
                GridBoxItemSize -> {"Columns" -> {{All}}, "Rows" -> {{1.45}}},
                 GridBoxSpacings -> {"Columns" -> {{0}}}, 
                BaselinePosition -> {{1, 1}, Baseline}], "Grid"], 
              RoundingRadius -> 4, Background -> 
              GrayLevel[0.9333333333333333], 
              FrameMargins -> {{0, 0}, {-2, -2}}, FrameStyle -> {
                GrayLevel[0.8352941176470589], 
                AbsoluteThickness[1]}, BaselinePosition -> Baseline, 
              StripOnInput -> False]& )}], 
         Cell[
          StyleData["Key1"], 
          TemplateBoxOptions -> {DisplayFunction -> (FrameBox[
              TagBox[
               GridBox[{{
                  ItemBox[
                   
                   TemplateBox[{"\[ThickSpace]", #, "\[ThickSpace]"}, 
                    "RowDefault"], Background -> 
                   GrayLevel[0.9333333333333333], StripOnInput -> False], 
                  ItemBox[
                   
                   TemplateBox[{"\[ThickSpace]", #2, "\[ThickSpace]"}, 
                    "RowDefault"], Background -> GrayLevel[1], StripOnInput -> 
                   False]}}, 
                GridBoxAlignment -> {
                 "Columns" -> {{Center}}, "Rows" -> {{Baseline}}}, AutoDelete -> 
                False, GridBoxDividers -> {
                 "ColumnsIndexed" -> {2 -> GrayLevel[0.8352941176470589]}, 
                  "Rows" -> {{None}}}, 
                GridBoxItemSize -> {"Columns" -> {{All}}, "Rows" -> {{1.45}}},
                 GridBoxSpacings -> {"Columns" -> {{0}}}, 
                BaselinePosition -> {{1, 1}, Baseline}], "Grid"], 
              RoundingRadius -> 4, Background -> 
              GrayLevel[0.9333333333333333], 
              FrameMargins -> {{0, 0}, {-2, -2}}, FrameStyle -> {
                GrayLevel[0.8352941176470589], 
                AbsoluteThickness[1]}, BaselinePosition -> Baseline, 
              StripOnInput -> False]& )}], 
         Cell[
          StyleData["Key2"], 
          TemplateBoxOptions -> {DisplayFunction -> (FrameBox[
              TagBox[
               GridBox[{{
                  ItemBox[
                   
                   TemplateBox[{"\[ThickSpace]", #, "\[ThickSpace]"}, 
                    "RowDefault"], Background -> 
                   GrayLevel[0.9333333333333333], StripOnInput -> False], 
                  ItemBox[
                   
                   TemplateBox[{"\[ThickSpace]", #2, "\[ThickSpace]"}, 
                    "RowDefault"], Background -> 
                   GrayLevel[0.9333333333333333], StripOnInput -> False], 
                  ItemBox[
                   
                   TemplateBox[{"\[ThickSpace]", #3, "\[ThickSpace]"}, 
                    "RowDefault"], Background -> GrayLevel[1], StripOnInput -> 
                   False]}}, 
                GridBoxAlignment -> {
                 "Columns" -> {{Center}}, "Rows" -> {{Baseline}}}, AutoDelete -> 
                False, GridBoxDividers -> {
                 "ColumnsIndexed" -> {
                   2 -> GrayLevel[0.8352941176470589], 3 -> 
                    GrayLevel[0.8352941176470589]}, "Rows" -> {{None}}}, 
                GridBoxItemSize -> {"Columns" -> {{All}}, "Rows" -> {{1.45}}},
                 GridBoxSpacings -> {"Columns" -> {{0}}}, 
                BaselinePosition -> {{1, 1}, Baseline}], "Grid"], 
              RoundingRadius -> 4, Background -> 
              GrayLevel[0.9333333333333333], 
              FrameMargins -> {{0, 0}, {-2, -2}}, FrameStyle -> {
                GrayLevel[0.8352941176470589], 
                AbsoluteThickness[1]}, BaselinePosition -> Baseline, 
              StripOnInput -> False]& )}], 
         Cell[
          StyleData["Key3"], 
          TemplateBoxOptions -> {DisplayFunction -> (FrameBox[
              TagBox[
               GridBox[{{
                  ItemBox[
                   
                   TemplateBox[{"\[ThickSpace]", #, "\[ThickSpace]"}, 
                    "RowDefault"], Background -> 
                   GrayLevel[0.9333333333333333], StripOnInput -> False], 
                  ItemBox[
                   
                   TemplateBox[{"\[ThickSpace]", #2, "\[ThickSpace]"}, 
                    "RowDefault"], Background -> 
                   GrayLevel[0.9333333333333333], StripOnInput -> False], 
                  ItemBox[
                   
                   TemplateBox[{"\[ThickSpace]", #3, "\[ThickSpace]"}, 
                    "RowDefault"], Background -> 
                   GrayLevel[0.9333333333333333], StripOnInput -> False], 
                  ItemBox[
                   
                   TemplateBox[{"\[ThickSpace]", #4, "\[ThickSpace]"}, 
                    "RowDefault"], Background -> GrayLevel[1], StripOnInput -> 
                   False]}}, 
                GridBoxAlignment -> {
                 "Columns" -> {{Center}}, "Rows" -> {{Baseline}}}, AutoDelete -> 
                False, GridBoxDividers -> {
                 "ColumnsIndexed" -> {
                   2 -> GrayLevel[0.8352941176470589], 3 -> 
                    GrayLevel[0.8352941176470589], 4 -> 
                    GrayLevel[0.8352941176470589]}, "Rows" -> {{None}}}, 
                GridBoxItemSize -> {"Columns" -> {{All}}, "Rows" -> {{1.45}}},
                 GridBoxSpacings -> {"Columns" -> {{0}}}, 
                BaselinePosition -> {{1, 1}, Baseline}], "Grid"], 
              RoundingRadius -> 4, Background -> 
              GrayLevel[0.9333333333333333], 
              FrameMargins -> {{0, 0}, {-2, -2}}, FrameStyle -> {
                GrayLevel[0.8352941176470589], 
                AbsoluteThickness[1]}, BaselinePosition -> Baseline, 
              StripOnInput -> False]& )}]}, Closed]], 
      Cell[
       CellGroupData[{
         Cell["Entity styles", "Subsubsection"], 
         Cell[
          StyleData["Entity"], ShowStringCharacters -> False, 
          TemplateBoxOptions -> {DisplayFunction -> (TooltipBox[
              FrameBox[
               PaneSelectorBox[{True -> GridBox[{{
                    
                    StyleBox[#, FontWeight -> "Bold", FontFamily -> 
                    "Source Sans Pro", FontColor -> 
                    RGBColor[0.395437, 0.20595, 0.061158]], 
                    StyleBox[
                    RowBox[{"(", #4, ")"}], FontFamily -> "Source Sans Pro", 
                    FontWeight -> "Plain", FontColor -> GrayLevel[0.65], 
                    Selectable -> False]}}, 
                   GridBoxItemSize -> {
                    "Columns" -> {{All}}, "Rows" -> {{All}}}, 
                   GridBoxSpacings -> {"Columns" -> {{0.2}}, "Rows" -> {{0}}},
                    BaselinePosition -> {1, 1}], False -> 
                 PaneBox[#, 
                   BaseStyle -> {
                    FontColor -> RGBColor[0.395437, 0.20595, 0.061158]}, 
                   BaselinePosition -> Baseline]}, 
                Dynamic[
                 CurrentValue[Evaluatable]], ImageSize -> Automatic, 
                BaselinePosition -> Baseline, 
                BaseStyle -> {FontFamily -> "Helvetica"}], FrameStyle -> 
               RGBColor[1., 0.504768, 0.], Background -> 
               RGBColor[1., 0.980392, 0.921569], 
               FrameMargins -> {{5, 5}, {1.5, 1.5}}, 
               ImageMargins -> {{2, 2}, {2, 2}}, BaselinePosition -> Baseline,
                RoundingRadius -> 4], #3, 
              BaseStyle -> {
               FontWeight -> "Plain", ShowStringCharacters -> False, 
                LineIndent -> 0, 
                PrivateFontOptions -> {"OperatorSubstitution" -> False}}]& ), 
            Editable -> False, InterpretationFunction -> (#2& )}], 
         Cell[
          StyleData["EntityFrameless"], ShowStringCharacters -> False, 
          TemplateBoxOptions -> {DisplayFunction -> (TooltipBox[
              PaneSelectorBox[{True -> GridBox[{{
                    
                    StyleBox[#, FontWeight -> "Bold", FontFamily -> 
                    "Source Sans Pro", FontColor -> 
                    RGBColor[0.395437, 0.20595, 0.061158]], 
                    StyleBox[
                    RowBox[{"(", #4, ")"}], FontFamily -> "Source Sans Pro", 
                    FontWeight -> "Plain", FontColor -> GrayLevel[0.65], 
                    Selectable -> False]}}, 
                  GridBoxItemSize -> {
                   "Columns" -> {{All}}, "Rows" -> {{All}}}, 
                  GridBoxSpacings -> {"Columns" -> {{0.2}}, "Rows" -> {{0}}}, 
                  BaselinePosition -> {1, 1}], False -> 
                PaneBox[#, 
                  BaseStyle -> {
                   FontColor -> RGBColor[0.395437, 0.20595, 0.061158]}, 
                  BaselinePosition -> Baseline]}, 
               Dynamic[
                CurrentValue[Evaluatable]], ImageSize -> Automatic, 
               BaselinePosition -> Baseline, 
               BaseStyle -> {FontFamily -> "Helvetica"}], #3, 
              BaseStyle -> {
               ShowStringCharacters -> False, LineIndent -> 0, 
                PrivateFontOptions -> {
                 FontWeight -> "Plain", "OperatorSubstitution" -> False}}]& ),
             Editable -> False, InterpretationFunction -> (#2& )}], 
         Cell[
          StyleData["EntityFramelessHot"], ShowStringCharacters -> False, 
          TemplateBoxOptions -> {
           DisplayFunction -> (PaneSelectorBox[{True -> GridBox[{{
                   
                   StyleBox[#, FontWeight -> "Bold", FontFamily -> 
                    "Source Sans Pro", FontColor -> GrayLevel[0]], 
                   StyleBox[
                    RowBox[{"(", #4, ")"}], FontWeight -> "Plain", FontFamily -> 
                    "Source Sans Pro", FontColor -> GrayLevel[0], Selectable -> 
                    False]}}, 
                 GridBoxItemSize -> {"Columns" -> {{All}}, "Rows" -> {{All}}},
                  GridBoxSpacings -> {"Columns" -> {{0.2}}, "Rows" -> {{0}}}, 
                 BaselinePosition -> {1, 1}], False -> 
               PaneBox[#, BaseStyle -> {FontColor -> GrayLevel[0]}, 
                 BaselinePosition -> Baseline]}, 
              Dynamic[
               CurrentValue[Evaluatable]], ImageSize -> Automatic, 
              BaselinePosition -> Baseline, 
              BaseStyle -> {
               FontWeight -> "Plain", FontFamily -> "Helvetica", 
                ShowStringCharacters -> False, LineIndent -> 0, 
                PrivateFontOptions -> {"OperatorSubstitution" -> False}}]& ), 
            Editable -> False, InterpretationFunction -> (#2& )}], 
         Cell[
          StyleData["EntityClass"], ShowStringCharacters -> False, 
          TemplateBoxOptions -> {DisplayFunction -> (TooltipBox[
              FrameBox[
               PaneSelectorBox[{True -> GridBox[{{
                    PaneBox[
                    DynamicBox[
                    FEPrivate`ImportImage[
                    
                    FrontEnd`ToFileName[{"Typeset", "Entity"}, 
                    "EntityClass.png"]], ImageSizeCache -> {8., {2., 6.}}], 
                    BaselinePosition -> Bottom], 
                    
                    StyleBox[#, FontWeight -> "Bold", FontFamily -> 
                    "Source Sans Pro", FontColor -> 
                    RGBColor[0.395437, 0.20595, 0.061158]], 
                    StyleBox[
                    RowBox[{"(", #4, ")"}], FontWeight -> "Plain", FontFamily -> 
                    "Source Sans Pro", FontColor -> GrayLevel[0.65], 
                    Selectable -> False]}}, 
                   GridBoxItemSize -> {
                    "Columns" -> {{All}}, "Rows" -> {{All}}}, 
                   GridBoxSpacings -> {"Columns" -> {{0.2}}, "Rows" -> {{0}}},
                    BaselinePosition -> {1, 2}], False -> GridBox[{{
                    PaneBox[
                    DynamicBox[
                    FEPrivate`ImportImage[
                    
                    FrontEnd`ToFileName[{"Typeset", "Entity"}, 
                    "EntityClass.png"]], ImageSizeCache -> {8., {2., 6.}}], 
                    BaselinePosition -> Bottom], 
                    
                    StyleBox[#, FontColor -> 
                    RGBColor[0.395437, 0.20595, 0.061158]]}}, 
                   GridBoxItemSize -> {
                    "Columns" -> {{All}}, "Rows" -> {{All}}}, 
                   GridBoxSpacings -> {"Columns" -> {{0.2}}, "Rows" -> {{0}}},
                    BaselinePosition -> {1, 2}]}, 
                Dynamic[
                 CurrentValue[Evaluatable]], ImageSize -> Automatic, 
                BaselinePosition -> Baseline, 
                BaseStyle -> {FontFamily -> "Helvetica"}], FrameStyle -> 
               RGBColor[1., 0.504768, 0.], Background -> 
               RGBColor[1., 0.980392, 0.921569], 
               FrameMargins -> {{5, 5}, {1.5, 1.5}}, 
               ImageMargins -> {{2, 2}, {2, 2}}, BaselinePosition -> Baseline,
                RoundingRadius -> 4], #3, 
              BaseStyle -> {
               FontWeight -> "Plain", ShowStringCharacters -> False, 
                LineIndent -> 0, 
                PrivateFontOptions -> {"OperatorSubstitution" -> False}}]& ), 
            Editable -> False, InterpretationFunction -> (#2& )}], 
         Cell[
          StyleData["EntityClassFrameless"], ShowStringCharacters -> False, 
          TemplateBoxOptions -> {DisplayFunction -> (TooltipBox[
              PaneSelectorBox[{True -> GridBox[{{
                    PaneBox[
                    DynamicBox[
                    FEPrivate`ImportImage[
                    
                    FrontEnd`ToFileName[{"Typeset", "Entity"}, 
                    "EntityClass.png"]], ImageSizeCache -> {8., {2., 6.}}], 
                    BaselinePosition -> Bottom], 
                    
                    StyleBox[#, FontWeight -> "Bold", FontFamily -> 
                    "Source Sans Pro", FontColor -> 
                    RGBColor[0.395437, 0.20595, 0.061158]], 
                    StyleBox[
                    RowBox[{"(", #4, ")"}], FontWeight -> "Plain", FontFamily -> 
                    "Source Sans Pro", FontColor -> GrayLevel[0.65], 
                    Selectable -> False]}}, 
                  GridBoxItemSize -> {
                   "Columns" -> {{All}}, "Rows" -> {{All}}}, 
                  GridBoxSpacings -> {"Columns" -> {{0.2}}, "Rows" -> {{0}}}, 
                  BaselinePosition -> {1, 2}], False -> GridBox[{{
                    PaneBox[
                    DynamicBox[
                    FEPrivate`ImportImage[
                    
                    FrontEnd`ToFileName[{"Typeset", "Entity"}, 
                    "EntityClass.png"]], ImageSizeCache -> {8., {2., 6.}}], 
                    BaselinePosition -> Bottom], 
                    
                    StyleBox[#, FontColor -> 
                    RGBColor[0.395437, 0.20595, 0.061158]]}}, 
                  GridBoxItemSize -> {
                   "Columns" -> {{All}}, "Rows" -> {{All}}}, 
                  GridBoxSpacings -> {"Columns" -> {{0.2}}, "Rows" -> {{0}}}, 
                  BaselinePosition -> {1, 2}]}, 
               Dynamic[
                CurrentValue[Evaluatable]], ImageSize -> Automatic, 
               BaselinePosition -> Baseline, 
               BaseStyle -> {FontFamily -> "Helvetica"}], #3, 
              BaseStyle -> {
               FontWeight -> "Plain", ShowStringCharacters -> False, 
                LineIndent -> 0, 
                PrivateFontOptions -> {"OperatorSubstitution" -> False}}]& ), 
            Editable -> False, InterpretationFunction -> (#2& )}], 
         Cell[
          StyleData["EntityClassFramelessHot"], ShowStringCharacters -> False,
           TemplateBoxOptions -> {
           DisplayFunction -> (PaneSelectorBox[{True -> GridBox[{{
                   PaneBox[
                    DynamicBox[
                    FEPrivate`ImportImage[
                    
                    FrontEnd`ToFileName[{"Typeset", "Entity"}, 
                    "EntityClass.png"]], ImageSizeCache -> {8., {2., 6.}}], 
                    BaselinePosition -> Bottom], 
                   
                   StyleBox[#, FontWeight -> "Bold", FontFamily -> 
                    "Source Sans Pro", FontColor -> GrayLevel[0]], 
                   StyleBox[
                    RowBox[{"(", #4, ")"}], FontWeight -> "Plain", FontFamily -> 
                    "Source Sans Pro", FontColor -> GrayLevel[0], Selectable -> 
                    False]}}, 
                 GridBoxItemSize -> {"Columns" -> {{All}}, "Rows" -> {{All}}},
                  GridBoxSpacings -> {"Columns" -> {{0.2}}, "Rows" -> {{0}}}, 
                 BaselinePosition -> {1, 2}], False -> GridBox[{{
                   PaneBox[
                    DynamicBox[
                    FEPrivate`ImportImage[
                    
                    FrontEnd`ToFileName[{"Typeset", "Entity"}, 
                    "EntityClass.png"]], ImageSizeCache -> {8., {2., 6.}}], 
                    BaselinePosition -> Bottom], 
                   StyleBox[#, FontColor -> GrayLevel[0]]}}, 
                 GridBoxItemSize -> {"Columns" -> {{All}}, "Rows" -> {{All}}},
                  GridBoxSpacings -> {"Columns" -> {{0.2}}, "Rows" -> {{0}}}, 
                 BaselinePosition -> {1, 2}]}, 
              Dynamic[
               CurrentValue[Evaluatable]], ImageSize -> Automatic, 
              BaselinePosition -> Baseline, 
              BaseStyle -> {
               FontWeight -> "Plain", FontFamily -> "Helvetica", 
                ShowStringCharacters -> False, LineIndent -> 0, 
                PrivateFontOptions -> {"OperatorSubstitution" -> False}}]& ), 
            Editable -> False, InterpretationFunction -> (#2& )}], 
         Cell[
          StyleData["EntityProperty"], ShowStringCharacters -> False, 
          TemplateBoxOptions -> {DisplayFunction -> (TooltipBox[
              FrameBox[
               
               PaneBox[#, 
                BaseStyle -> {
                 FontSlant -> Italic, FontWeight -> Plain, FontFamily -> 
                  "Source Sans Pro", FontColor -> 
                  RGBColor[0.395437, 0.20595, 0.061158]}, BaselinePosition -> 
                Baseline], FrameStyle -> 
               RGBColor[0.94227, 0.703639, 0.033387], Background -> 
               RGBColor[1., 0.980392, 0.921569], 
               FrameMargins -> {{5, 5}, {1.5, 1.5}}, 
               ImageMargins -> {{2, 2}, {2, 2}}, BaselinePosition -> Baseline,
                RoundingRadius -> 4], #3, 
              BaseStyle -> {
               FontWeight -> "Plain", ShowStringCharacters -> False, 
                LineIndent -> 0, 
                PrivateFontOptions -> {"OperatorSubstitution" -> False}}]& ), 
            Editable -> False, InterpretationFunction -> (#2& )}], 
         Cell[
          StyleData["EntityPropertyClass"], ShowStringCharacters -> False, 
          TemplateBoxOptions -> {DisplayFunction -> (TooltipBox[
              FrameBox[
               PaneBox[
                GridBox[{{
                   PaneBox[
                    DynamicBox[
                    FEPrivate`ImportImage[
                    
                    FrontEnd`ToFileName[{"Typeset", "Entity"}, 
                    "EntityClass.png"]], ImageSizeCache -> {8., {2., 6.}}], 
                    BaselinePosition -> Bottom], #}}, 
                 GridBoxItemSize -> {"Columns" -> {{All}}, "Rows" -> {{All}}},
                  GridBoxSpacings -> {"Columns" -> {{0.2}}, "Rows" -> {{0}}}, 
                 BaselinePosition -> {1, 2}], 
                BaseStyle -> {
                 FontSlant -> Italic, FontWeight -> Plain, FontFamily -> 
                  "Source Sans Pro", FontColor -> 
                  RGBColor[0.395437, 0.20595, 0.061158]}, BaselinePosition -> 
                Baseline], FrameStyle -> 
               RGBColor[0.94227, 0.703639, 0.033387], Background -> 
               RGBColor[1., 0.980392, 0.921569], 
               FrameMargins -> {{5, 5}, {1.5, 1.5}}, 
               ImageMargins -> {{2, 2}, {2, 2}}, BaselinePosition -> Baseline,
                RoundingRadius -> 4], #3, 
              BaseStyle -> {
               FontWeight -> "Plain", ShowStringCharacters -> False, 
                LineIndent -> 0, 
                PrivateFontOptions -> {"OperatorSubstitution" -> False}}]& ), 
            Editable -> False, InterpretationFunction -> (#2& )}], 
         Cell[
          StyleData["EntityValue"], ShowStringCharacters -> False, 
          TemplateBoxOptions -> {DisplayFunction -> (FrameBox[
              
              GridBox[{{#, #2}}, GridBoxAlignment -> {"Rows" -> {{Baseline}}},
                GridBoxItemSize -> {"Columns" -> {{All}}, "Rows" -> {{All}}}, 
               GridBoxSpacings -> {"Columns" -> {{0}}}, 
               BaselinePosition -> {1, 1}], Background -> 
              RGBColor[{0.9921568627450981, 0.9529411764705882, 
                0.8509803921568627}], BaselinePosition -> Baseline, 
              FrameMargins -> {{1, 1}, {1, 1}}, ImageMargins -> 2, FrameStyle -> 
              RGBColor[{0.9333333333333333, 0.8470588235294118, 
                0.5529411764705883}], RoundingRadius -> 5]& )}]}, Closed]]}, 
     Closed]]}, Visible -> False, FrontEndVersion -> 
  "12.3 for Mac OS X x86 (64-bit) (July 30, 2021)", StyleDefinitions -> 
  "StylesheetFormatting.nb"],
ExpressionUUID->"8052fe23-d437-4374-8ca2-9caf90f20c2e"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "p:53"->{
  Cell[648, 23, 436, 11, 93, "SectionDoubleDigit",ExpressionUUID->"c145940b-2ec1-4b3e-a40c-983b843a8511",
   CounterAssignments->{{"Section", 11}, {"Subsection", 0}, {"Subsubsection", 0}, {"Exercise", 0}, {"Question", 0}},
   CellTags->{"p:53", "i:1"}]},
 "i:1"->{
  Cell[648, 23, 436, 11, 93, "SectionDoubleDigit",ExpressionUUID->"c145940b-2ec1-4b3e-a40c-983b843a8511",
   CounterAssignments->{{"Section", 11}, {"Subsection", 0}, {"Subsubsection", 0}, {"Exercise", 0}, {"Question", 0}},
   CellTags->{"p:53", "i:1"}]},
 "i:2"->{
  Cell[1087, 36, 1344, 29, 116, "Text",ExpressionUUID->"e672c068-a9d9-45c2-a534-2864267d6cda",
   CellTags->"i:2"]},
 "i:3"->{
  Cell[2434, 67, 183, 4, 37, "CodeCaption",ExpressionUUID->"f048db40-dfb3-45cf-aa18-4f087bdc7289",
   CellTags->"i:3"]},
 "i:4"->{
  Cell[12503, 316, 423, 9, 90, "Text",ExpressionUUID->"10ddda16-2e39-4973-abe0-5304ecbd4e73",
   CellTags->"i:4"]},
 "LastCellOfAPage"->{
  Cell[18826, 481, 5482, 125, 70, "Output",ExpressionUUID->"ba7c44f2-f2e6-4098-b4f1-5eb13fb5cbfb",
   Evaluatable->False,
   CellTags->"LastCellOfAPage"],
  Cell[49458, 1224, 1447, 40, 70, "VocabularyTable",ExpressionUUID->"08513e65-c161-4097-98ee-4c6cb9d4fa96",
   CellTags->"LastCellOfAPage"],
  Cell[497390, 11340, 445, 19, 70, "Answer",ExpressionUUID->"17dc1df5-635b-45a5-876a-b28c585f5dbe",
   CellTags->"LastCellOfAPage"],
  Cell[499214, 11413, 351, 8, 70, "MoreExplore",ExpressionUUID->"d1366c45-8cc3-47af-9d03-e957de205489",
   CellTags->"LastCellOfAPage"]},
 "p:54"->{
  Cell[24323, 609, 557, 16, 70, "Text",ExpressionUUID->"9e11890a-e096-4a6c-81ab-7f348d72ed00",
   CellTags->{"p:54", "i:5"}]},
 "i:5"->{
  Cell[24323, 609, 557, 16, 70, "Text",ExpressionUUID->"9e11890a-e096-4a6c-81ab-7f348d72ed00",
   CellTags->{"p:54", "i:5"}]},
 "i:6"->{
  Cell[30917, 776, 330, 10, 70, "Text",ExpressionUUID->"284ee8b9-b53b-4513-8d4c-a6c2c653a86f",
   CellTags->"i:6"]},
 "i:7"->{
  Cell[31250, 788, 191, 4, 70, "CodeCaption",ExpressionUUID->"3b13963b-ed9b-419e-88b0-4635b0c7e3fd",
   CellTags->"i:7"]},
 "i:8"->{
  Cell[37393, 937, 303, 7, 70, "Text",ExpressionUUID->"dcc3c782-19b6-43a7-a67f-312ef91b579c",
   CellTags->"i:8"]},
 "i:9"->{
  Cell[491999, 11160, 542, 12, 70, "Answer",ExpressionUUID->"5087e67e-98f4-4a20-94a0-5aaeee2d5193",
   CellTags->"i:9"]},
 "i:10"->{
  Cell[493287, 11201, 1208, 35, 70, "Answer",ExpressionUUID->"361d5627-847d-4920-9703-e8731667d376",
   CellTags->"i:10"]},
 "i:11"->{
  Cell[494498, 11238, 178, 4, 70, "Question",ExpressionUUID->"ca76e3e5-4940-46b5-a9d3-31d922b0d222",
   CellTags->"i:11"]},
 "i:12"->{
  Cell[495109, 11260, 177, 4, 70, "Question",ExpressionUUID->"bc50606e-0ac1-4171-8418-c062379a4067",
   CellTags->"i:12"]},
 "i:13"->{
  Cell[495678, 11281, 248, 7, 70, "Question",ExpressionUUID->"2c06cb48-21ae-45b0-a9b1-e16d521b2d7f",
   CellTags->"i:13"]},
 "i:14"->{
  Cell[497194, 11334, 193, 4, 70, "Question",ExpressionUUID->"71fa9082-8665-4169-a63a-0ae9241e62b5",
   CellTags->"i:14"]},
 "p:56"->{
  Cell[497872, 11364, 173, 4, 70, "TechNoteSection",ExpressionUUID->"cc05ab61-f5f9-4767-afc3-b0485e076325",
   CellTags->"p:56"]},
 "i:15"->{
  Cell[498048, 11370, 645, 18, 70, "TechNote",ExpressionUUID->"34e9d6a8-cf3f-4c24-970e-4e63f05818f5",
   CellTags->"i:15"]},
 "i:16"->{
  Cell[498696, 11390, 313, 12, 70, "TechNote",ExpressionUUID->"b6246dd6-03d9-41cf-9138-e6705e197729",
   CellTags->"i:16"]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"p:53", 590269, 13261},
 {"i:1", 590534, 13265},
 {"i:2", 590799, 13269},
 {"i:3", 590926, 13272},
 {"i:4", 591057, 13275},
 {"LastCellOfAPage", 591195, 13278},
 {"p:54", 591773, 13288},
 {"i:5", 591910, 13291},
 {"i:6", 592047, 13294},
 {"i:7", 592174, 13297},
 {"i:8", 592307, 13300},
 {"i:9", 592433, 13303},
 {"i:10", 592566, 13306},
 {"i:11", 592701, 13309},
 {"i:12", 592836, 13312},
 {"i:13", 592971, 13315},
 {"i:14", 593106, 13318},
 {"p:56", 593241, 13321},
 {"i:15", 593383, 13324},
 {"i:16", 593519, 13327}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[648, 23, 436, 11, 93, "SectionDoubleDigit",ExpressionUUID->"c145940b-2ec1-4b3e-a40c-983b843a8511",
 CounterAssignments->{{"Section", 11}, {"Subsection", 0}, {"Subsubsection", 0}, {"Exercise", 0}, {"Question", 0}},
 CellTags->{"p:53", "i:1"}],
Cell[1087, 36, 1344, 29, 116, "Text",ExpressionUUID->"e672c068-a9d9-45c2-a534-2864267d6cda",
 CellTags->"i:2"],
Cell[2434, 67, 183, 4, 37, "CodeCaption",ExpressionUUID->"f048db40-dfb3-45cf-aa18-4f087bdc7289",
 CellTags->"i:3"],
Cell[CellGroupData[{
Cell[2642, 75, 199, 4, 29, "Input",ExpressionUUID->"2c57b7ac-fcd7-42f3-acee-75140e007d5b"],
Cell[2844, 81, 4081, 93, 94, "Output",ExpressionUUID->"b8541725-c8f5-4f87-bbf1-5fc07afa3502",
 Evaluatable->False]
}, Open  ]],
Cell[6940, 177, 205, 6, 40, "Text",ExpressionUUID->"f2f97578-e0cb-45b2-83c0-de9954b5d791"],
Cell[7148, 185, 172, 4, 37, "CodeCaption",ExpressionUUID->"93bc168a-9a4e-41e7-be6d-7a19090313ea"],
Cell[CellGroupData[{
Cell[7345, 193, 350, 8, 29, "Input",ExpressionUUID->"6fbc38bc-ebd9-4e04-bbcd-9118d2c99edf"],
Cell[7698, 203, 4790, 110, 94, "Output",ExpressionUUID->"9d309fcd-bc83-44eb-a7a4-a908296f0aa9",
 Evaluatable->False]
}, Open  ]],
Cell[12503, 316, 423, 9, 90, "Text",ExpressionUUID->"10ddda16-2e39-4973-abe0-5304ecbd4e73",
 CellTags->"i:4"],
Cell[12929, 327, 172, 4, 37, "CodeCaption",ExpressionUUID->"8523b151-2a30-4218-b3cc-e0a3ba406b37"],
Cell[CellGroupData[{
Cell[13126, 335, 326, 8, 29, "Input",ExpressionUUID->"63e46507-3c3f-4b88-a448-026fa8344f5a"],
Cell[13455, 345, 4784, 110, 94, "Output",ExpressionUUID->"cf5ec508-55d1-4ee9-b282-aafe7de13bc8",
 Evaluatable->False]
}, Open  ]],
Cell[18254, 458, 244, 9, 70, "CodeCaption",ExpressionUUID->"066bfb18-a215-4fef-9f4b-9adbe65949d9"],
Cell[CellGroupData[{
Cell[18523, 471, 300, 8, 70, "Input",ExpressionUUID->"a92c2711-9be1-4ddf-8203-f621c01f12f8"],
Cell[18826, 481, 5482, 125, 70, "Output",ExpressionUUID->"ba7c44f2-f2e6-4098-b4f1-5eb13fb5cbfb",
 Evaluatable->False,
 CellTags->"LastCellOfAPage"]
}, Open  ]],
Cell[24323, 609, 557, 16, 70, "Text",ExpressionUUID->"9e11890a-e096-4a6c-81ab-7f348d72ed00",
 CellTags->{"p:54", "i:5"}],
Cell[24883, 627, 174, 4, 70, "CodeCaption",ExpressionUUID->"4dff7c30-a770-4933-b75e-459f8b834266"],
Cell[CellGroupData[{
Cell[25082, 635, 328, 9, 70, "Input",ExpressionUUID->"43f60c0a-2d47-4572-9b96-0cec892c08dd"],
Cell[25413, 646, 5489, 127, 70, "Output",ExpressionUUID->"7d724a34-5906-45f8-a44f-e2e0fe6a4f83",
 Evaluatable->False]
}, Open  ]],
Cell[30917, 776, 330, 10, 70, "Text",ExpressionUUID->"284ee8b9-b53b-4513-8d4c-a6c2c653a86f",
 CellTags->"i:6"],
Cell[31250, 788, 191, 4, 70, "CodeCaption",ExpressionUUID->"3b13963b-ed9b-419e-88b0-4635b0c7e3fd",
 CellTags->"i:7"],
Cell[CellGroupData[{
Cell[31466, 796, 351, 9, 70, "Input",ExpressionUUID->"876070c2-6596-489f-8397-a8a2cbe12cba"],
Cell[31820, 807, 5558, 127, 70, "Output",ExpressionUUID->"7ed6d35a-4950-4ec4-b7f1-cfc4280a065a",
 Evaluatable->False]
}, Open  ]],
Cell[37393, 937, 303, 7, 70, "Text",ExpressionUUID->"dcc3c782-19b6-43a7-a67f-312ef91b579c",
 CellTags->"i:8"],
Cell[37699, 946, 191, 4, 70, "CodeCaption",ExpressionUUID->"46bd4957-8382-433e-b179-fcc7b6d83550"],
Cell[CellGroupData[{
Cell[37915, 954, 353, 9, 70, "Input",ExpressionUUID->"4c8b700b-73c7-4680-9789-af3d67a5b947"],
Cell[38271, 965, 10976, 247, 70, "Output",ExpressionUUID->"81b4aeea-b2c5-4f28-8121-0ccd78111707",
 Evaluatable->False]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[49296, 1218, 159, 4, 70, "VocabularySection",ExpressionUUID->"c7224fc5-aa6c-478d-94ed-1d8884610388"],
Cell[49458, 1224, 1447, 40, 70, "VocabularyTable",ExpressionUUID->"08513e65-c161-4097-98ee-4c6cb9d4fa96",
 CellTags->"LastCellOfAPage"]
}, Open  ]],
Cell[CellGroupData[{
Cell[50942, 1269, 1780, 46, 70, "ExerciseSection",ExpressionUUID->"381f1676-631f-4d04-ba24-144a95c1a552"],
Cell[CellGroupData[{
Cell[52747, 1319, 1431, 44, 70, "ExerciseSummaryCell",ExpressionUUID->"8094785e-3642-4480-8539-26ee2f84007b"],
Cell[CellGroupData[{
Cell[54203, 1367, 720, 20, 70, "Exercise",ExpressionUUID->"e957386a-cc4f-4903-b83a-bec1aabc97f5"],
Cell[54926, 1389, 12484, 280, 70, "ExerciseOutput",ExpressionUUID->"a7d2c89b-4b08-434e-9ba4-72bd55d94d04"]
}, Closed]],
Cell[CellGroupData[{
Cell[67447, 1674, 715, 20, 70, "Exercise",ExpressionUUID->"252ce3c0-e584-433c-9da7-8e5e86586af7"],
Cell[68165, 1696, 10821, 240, 70, "ExerciseOutput",ExpressionUUID->"09162a7a-7f18-491f-b9fd-621c6b4ccb61"]
}, Closed]],
Cell[CellGroupData[{
Cell[79023, 1941, 817, 22, 70, "Exercise",ExpressionUUID->"a0edb41f-5270-4a29-a976-592a0a91d558"],
Cell[79843, 1965, 52486, 1153, 70, "ExerciseOutput",ExpressionUUID->"e1319dbd-1806-4aea-a3f3-f6d413d59f8c"]
}, Closed]],
Cell[CellGroupData[{
Cell[132366, 3123, 738, 20, 70, "Exercise",ExpressionUUID->"66a5c279-97fc-4402-8e4d-7e8ab08efc09"],
Cell[133107, 3145, 20915, 468, 70, "ExerciseOutput",ExpressionUUID->"03af47f8-c8d1-4fdf-be8e-eec6844623d2"]
}, Closed]],
Cell[CellGroupData[{
Cell[154059, 3618, 772, 21, 70, "Exercise",ExpressionUUID->"ef5aebe8-52ee-417a-86cc-b0a565b0da50"],
Cell[154834, 3641, 14176, 318, 70, "ExerciseOutput",ExpressionUUID->"87dc6a09-a089-4ed1-a893-2238968a3920"]
}, Closed]],
Cell[CellGroupData[{
Cell[169047, 3964, 770, 21, 70, "Exercise",ExpressionUUID->"25f5589c-f940-4535-87b8-3508ffa6d85e"],
Cell[169820, 3987, 18801, 414, 70, "ExerciseOutput",ExpressionUUID->"efdad4b1-7621-4b64-a190-7fddf763cb8b"]
}, Closed]],
Cell[CellGroupData[{
Cell[188658, 4406, 777, 21, 70, "Exercise",ExpressionUUID->"034286b6-c669-41d7-898c-6136d12b0e2d"],
Cell[189438, 4429, 19139, 432, 70, "ExerciseOutput",ExpressionUUID->"035603dc-4464-4be3-8cf6-60ff19437d04"]
}, Closed]],
Cell[CellGroupData[{
Cell[208614, 4866, 732, 20, 70, "Exercise",ExpressionUUID->"ca3efd76-19c3-4700-867b-e5c417fa53e3"],
Cell[209349, 4888, 18535, 413, 70, "ExerciseOutput",ExpressionUUID->"184b3ccd-317b-4b99-8dfa-378bdc480ce3"]
}, Closed]],
Cell[CellGroupData[{
Cell[227921, 5306, 763, 21, 70, "Exercise",ExpressionUUID->"1ebccbe9-b4ca-4a8a-b364-fb6d48578200"],
Cell[228687, 5329, 16198, 357, 70, "ExerciseOutput",ExpressionUUID->"bc646827-b439-47a7-b9a0-216c6db3af38"]
}, Closed]],
Cell[CellGroupData[{
Cell[244922, 5691, 811, 21, 70, "Exercise",ExpressionUUID->"3c71a8d6-cf58-43e7-8239-7dd5a46b5e8e"],
Cell[245736, 5714, 15957, 356, 70, "ExerciseOutput",ExpressionUUID->"84a11213-9c37-408f-a0bb-e7d75177aac0"]
}, Closed]],
Cell[CellGroupData[{
Cell[261730, 6075, 767, 21, 70, "Exercise",ExpressionUUID->"df9103fb-6189-453e-8aad-baa31af51787"],
Cell[262500, 6098, 13949, 301, 70, "ExerciseOutput",ExpressionUUID->"77cf0aea-9781-49ff-b3c1-1699e08e71f0"]
}, Closed]],
Cell[CellGroupData[{
Cell[276486, 6404, 747, 21, 70, "Exercise",ExpressionUUID->"855cfe32-1eb8-4e14-9a88-0b5192659ab6"],
Cell[277236, 6427, 14235, 319, 70, "ExerciseOutput",ExpressionUUID->"7a4816c5-b844-42a3-b44c-33e0ee1d3c48"]
}, Closed]],
Cell[CellGroupData[{
Cell[291508, 6751, 794, 21, 70, "Exercise",ExpressionUUID->"7530991b-48c9-4a9e-b63d-f7034b186dbc"],
Cell[292305, 6774, 14233, 319, 70, "ExerciseOutput",ExpressionUUID->"5091d2a1-38ad-4bdf-b4fd-6af18d6fccf8"]
}, Closed]],
Cell[CellGroupData[{
Cell[306575, 7098, 768, 21, 70, "Exercise",ExpressionUUID->"6a6712ab-a311-4013-a552-591d8fbf3c6e"],
Cell[307346, 7121, 184256, 4020, 70, "ExerciseOutput",ExpressionUUID->"e9fb3668-3fd2-4987-823c-5b7dcdc3d6e1"]
}, Closed]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[491663, 11148, 144, 4, 70, "QASection",ExpressionUUID->"0d8334d8-3c41-41a0-9a55-5018044c59fe"],
Cell[491810, 11154, 186, 4, 70, "Question",ExpressionUUID->"f7e66415-4cb2-4270-9c94-5c7700d98981"],
Cell[491999, 11160, 542, 12, 70, "Answer",ExpressionUUID->"5087e67e-98f4-4a20-94a0-5aaeee2d5193",
 CellTags->"i:9"],
Cell[492544, 11174, 175, 4, 70, "Question",ExpressionUUID->"d68deac5-8656-4b63-a512-fafcd4761ca4"],
Cell[492722, 11180, 382, 13, 70, "Answer",ExpressionUUID->"7aabe44b-0cff-4e97-bc47-16cf0c8a5faa"],
Cell[493107, 11195, 177, 4, 70, "Question",ExpressionUUID->"6932ec5e-e06b-4e02-bfcf-51bcfbe4e92c"],
Cell[493287, 11201, 1208, 35, 70, "Answer",ExpressionUUID->"361d5627-847d-4920-9703-e8731667d376",
 CellTags->"i:10"],
Cell[494498, 11238, 178, 4, 70, "Question",ExpressionUUID->"ca76e3e5-4940-46b5-a9d3-31d922b0d222",
 CellTags->"i:11"],
Cell[494679, 11244, 427, 14, 70, "Answer",ExpressionUUID->"88dae048-e372-4731-bae7-7d5f66ce5e9f"],
Cell[495109, 11260, 177, 4, 70, "Question",ExpressionUUID->"bc50606e-0ac1-4171-8418-c062379a4067",
 CellTags->"i:12"],
Cell[495289, 11266, 386, 13, 70, "Answer",ExpressionUUID->"dd728f88-6a1a-4e69-8e0f-574c30cd413d"],
Cell[495678, 11281, 248, 7, 70, "Question",ExpressionUUID->"2c06cb48-21ae-45b0-a9b1-e16d521b2d7f",
 CellTags->"i:13"],
Cell[495929, 11290, 567, 21, 70, "Answer",ExpressionUUID->"dcb84f4a-bcc0-4f72-bf7a-565caa81e302"],
Cell[496499, 11313, 239, 5, 70, "Question",ExpressionUUID->"c16af06d-f6c8-479c-9cbd-72f8b1b7156d"],
Cell[496741, 11320, 450, 12, 70, "Answer",ExpressionUUID->"2d2fcc08-7ad5-46a2-8231-dfa7b2938608"],
Cell[497194, 11334, 193, 4, 70, "Question",ExpressionUUID->"71fa9082-8665-4169-a63a-0ae9241e62b5",
 CellTags->"i:14"],
Cell[497390, 11340, 445, 19, 70, "Answer",ExpressionUUID->"17dc1df5-635b-45a5-876a-b28c585f5dbe",
 CellTags->"LastCellOfAPage"]
}, Open  ]],
Cell[CellGroupData[{
Cell[497872, 11364, 173, 4, 70, "TechNoteSection",ExpressionUUID->"cc05ab61-f5f9-4767-afc3-b0485e076325",
 CellTags->"p:56"],
Cell[498048, 11370, 645, 18, 70, "TechNote",ExpressionUUID->"34e9d6a8-cf3f-4c24-970e-4e63f05818f5",
 CellTags->"i:15"],
Cell[498696, 11390, 313, 12, 70, "TechNote",ExpressionUUID->"b6246dd6-03d9-41cf-9138-e6705e197729",
 CellTags->"i:16"]
}, Open  ]],
Cell[CellGroupData[{
Cell[499046, 11407, 165, 4, 70, "MoreExploreSection",ExpressionUUID->"00306873-588c-4a63-bcaf-ecb3d0b30037"],
Cell[499214, 11413, 351, 8, 70, "MoreExplore",ExpressionUUID->"d1366c45-8cc3-47af-9d03-e957de205489",
 CellTags->"LastCellOfAPage"]
}, Open  ]]
}
]
*)

(* NotebookSignature XK4OqpF3pdCGvK51VE1FJwYH *)
