(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 8.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       157,          7]
NotebookDataLength[     45241,       1024]
NotebookOptionsPosition[     43929,        974]
NotebookOutlinePosition[     44585,        999]
CellTagsIndexPosition[     44542,        996]
WindowTitle->Ray Diagrams for Spherical Mirrors - Source
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["Ray Diagrams for Spherical Mirrors", "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False,
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztnVuSozgWQDNiPqa3MJ+zpVlCbaB3WlGVNV9dvYH6TQRUph/ZGIHQmwvG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   "], {{0, 0}, {1800, 25}}, {0, 255},
   ColorFunction->RGBColor],
  ImageSize->{1800, 25},
  PlotRange->{{0, 1800}, {0, 25}}]], "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False],

Cell[BoxData[
 RowBox[{"Manipulate", "[", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "radius", ",", "objdistance", ",", "focusloc", ",", "imageloc", ",", 
       "radiusloc", ",", "mirrorloc"}], "}"}], ",", "\[IndentingNewLine]", 
     "\[IndentingNewLine]", 
     RowBox[{"(*", "Definitons", "*)"}], "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"imagedistance", "[", 
        RowBox[{"foclength_", ",", "objdistance_"}], "]"}], ":=", 
       RowBox[{"If", "[", 
        RowBox[{
         RowBox[{"mirrortype", " ", "\[Equal]", "\"\<concave\>\""}], ",", 
         RowBox[{"If", "[", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{"First", "[", "ptobj", "]"}], "\[Equal]", 
             RowBox[{"First", "[", "focusloc", "]"}]}], "||", 
            RowBox[{"objdistance", "==", "0"}], "||", 
            RowBox[{"foclength", "==", " ", "objdistance"}]}], ",", "0", ",", 
           
           RowBox[{
            RowBox[{"Solve", "[", 
             RowBox[{
              RowBox[{
               RowBox[{"1", "/", "foclength"}], "\[Equal]", 
               RowBox[{
                RowBox[{"1", "/", "objdistance"}], " ", "+", " ", 
                RowBox[{"1", "/", "q"}]}]}], ",", "q"}], "]"}], "[", 
            RowBox[{"[", 
             RowBox[{"1", ",", "1", ",", "2"}], "]"}], "]"}]}], "]"}], ",", 
         RowBox[{"If", "[", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{"First", "[", "ptobj", "]"}], "\[Equal]", 
             RowBox[{"First", "[", "focusloc", "]"}]}], "||", 
            RowBox[{"objdistance", "==", "0"}], "||", 
            RowBox[{"foclength", "==", " ", "objdistance"}]}], ",", "0", ",", 
           
           RowBox[{
            RowBox[{"Solve", "[", 
             RowBox[{
              RowBox[{
               RowBox[{"1", "/", 
                RowBox[{"-", "foclength"}]}], "\[Equal]", 
               RowBox[{
                RowBox[{"1", "/", "objdistance"}], " ", "+", " ", 
                RowBox[{"1", "/", "q"}]}]}], ",", "q"}], "]"}], "[", 
            RowBox[{"[", 
             RowBox[{"1", ",", "1", ",", "2"}], "]"}], "]"}]}], "]"}]}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"magnification", "[", 
        RowBox[{"objdistance_", ",", "imgdistance_"}], "]"}], ":=", 
       RowBox[{"If", "[", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"First", "[", "ptobj", "]"}], "\[Equal]", 
           RowBox[{"First", "[", "focusloc", "]"}]}], "||", 
          RowBox[{"objdistance", "==", "0"}], "||", 
          RowBox[{"imgdistance", "\[Equal]", "0"}]}], ",", "0", ",", 
         RowBox[{"imgdistance", "/", "objdistance"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
      RowBox[{"(*", "Calculations", "*)"}], "\[IndentingNewLine]", 
      RowBox[{"radius", " ", "=", " ", 
       RowBox[{"2", "focus"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"objdistance", " ", "=", " ", 
       RowBox[{
        RowBox[{"First", "[", "location", "]"}], "-", 
        RowBox[{"First", "[", "ptobj", "]"}]}]}], ";", "\[IndentingNewLine]", 
      
      RowBox[{"focusloc", " ", "=", " ", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"First", "[", "location", "]"}], "-", "radius"}], ")"}], 
          "+", "focus"}], ",", "0"}], "}"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"imageloc", " ", "=", " ", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{"First", "[", "location", "]"}], "-", 
          RowBox[{"(", 
           RowBox[{"imagedistance", "[", 
            RowBox[{"focus", ",", "objdistance"}], "]"}], ")"}]}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"magnification", "[", 
           RowBox[{"objdistance", ",", 
            RowBox[{
             RowBox[{"-", "1"}], "*", 
             RowBox[{"imagedistance", "[", 
              RowBox[{"focus", ",", "objdistance"}], "]"}]}]}], "]"}], "*", 
          RowBox[{"Last", "[", "ptobj", "]"}]}]}], "}"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"mirrorloc", " ", "=", " ", 
       RowBox[{"If", "[", 
        RowBox[{
         RowBox[{"mirrortype", "\[Equal]", "\"\<concave\>\""}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{"Sqrt", "[", 
             RowBox[{
              RowBox[{"radius", "^", "2"}], " ", "-", " ", 
              RowBox[{
               RowBox[{"(", 
                RowBox[{"Last", "[", "ptobj", "]"}], ")"}], "^", "2"}]}], 
             "]"}], "+", 
            RowBox[{"First", "[", "location", "]"}], "-", "radius"}], ",", 
           RowBox[{"Last", "[", "ptobj", "]"}]}], "}"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{"-", 
             RowBox[{"Sqrt", "[", 
              RowBox[{
               RowBox[{"radius", "^", "2"}], " ", "-", " ", 
               RowBox[{
                RowBox[{"(", 
                 RowBox[{"Last", "[", "ptobj", "]"}], ")"}], "^", "2"}]}], 
              "]"}]}], "+", 
            RowBox[{"First", "[", "location", "]"}], "+", "radius"}], ",", 
           RowBox[{"Last", "[", "ptobj", "]"}]}], "}"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", "\[IndentingNewLine]", 
      RowBox[{"(*", 
       RowBox[{"Begin", " ", "Graphics"}], "*)"}], "\[IndentingNewLine]", 
      RowBox[{"Graphics", "[", 
       RowBox[{
        RowBox[{"{", "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"If", "[", 
           RowBox[{
            RowBox[{"mirrortype", "\[Equal]", "\"\<concave\>\""}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{"Thick", ",", 
              RowBox[{"Circle", "[", 
               RowBox[{
                RowBox[{"{", 
                 RowBox[{
                  RowBox[{
                   RowBox[{"First", "[", "location", "]"}], "-", "radius"}], 
                  ",", "0"}], "}"}], ",", "radius", ",", 
                RowBox[{"{", 
                 RowBox[{
                  RowBox[{
                   RowBox[{"-", "Pi"}], "/", "3"}], ",", 
                  RowBox[{"Pi", "/", "3"}]}], "}"}]}], "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"Thick", ",", 
              RowBox[{"Circle", "[", 
               RowBox[{
                RowBox[{"{", 
                 RowBox[{
                  RowBox[{
                   RowBox[{"First", "[", "location", "]"}], "+", "radius"}], 
                  ",", "0"}], "}"}], ",", "radius", ",", 
                RowBox[{"{", 
                 RowBox[{
                  RowBox[{"2", 
                   RowBox[{"Pi", "/", "3"}]}], ",", 
                  RowBox[{"4", 
                   RowBox[{"Pi", "/", "3"}]}]}], "}"}]}], "]"}]}], "}"}]}], 
           "]"}], ",", "\[IndentingNewLine]", "\[IndentingNewLine]", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"Arrowheads", "[", "Small", "]"}], ",", 
            RowBox[{"ColorData", "[", 
             RowBox[{"\"\<HTML\>\"", ",", "\"\<SlateBlue\>\""}], "]"}], ",", 
            RowBox[{"Arrow", "[", 
             RowBox[{"{", 
              RowBox[{
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"First", "[", "ptobj", "]"}], ",", "0"}], "}"}], ",",
                "ptobj"}], "}"}], "]"}]}], "}"}], ",", "\[IndentingNewLine]", 
          
          RowBox[{"{", 
           RowBox[{"Thick", ",", 
            RowBox[{"Line", "[", 
             RowBox[{"{", 
              RowBox[{
               RowBox[{"{", 
                RowBox[{"0", ",", "0"}], "}"}], ",", 
               RowBox[{"{", 
                RowBox[{"20", ",", "0"}], "}"}]}], "}"}], "]"}]}], "}"}], ",",
           "\[IndentingNewLine]", "\[IndentingNewLine]", 
          RowBox[{"If", "[", 
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"First", "[", "ptobj", "]"}], "\[GreaterEqual]", 
              RowBox[{"First", "[", "location", "]"}]}], "||", 
             RowBox[{
              RowBox[{"First", "[", "ptobj", "]"}], "\[Equal]", 
              RowBox[{"First", "[", "focusloc", "]"}]}]}], ",", 
            RowBox[{"{", "}"}], ",", "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"{", 
               RowBox[{
                RowBox[{"RGBColor", "[", 
                 RowBox[{"1", ",", ".47", ",", "0"}], "]"}], ",", 
                RowBox[{"Line", "[", 
                 RowBox[{"{", 
                  RowBox[{"ptobj", ",", "mirrorloc"}], "}"}], "]"}]}], "}"}], 
              ",", "\[IndentingNewLine]", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"RGBColor", "[", 
                 RowBox[{".6", ",", ".73", ",", ".36"}], "]"}], ",", 
                RowBox[{"Line", "[", 
                 RowBox[{"{", 
                  RowBox[{"ptobj", ",", "location", ",", 
                   RowBox[{"{", 
                    RowBox[{
                    RowBox[{"First", "[", "ptobj", "]"}], ",", 
                    RowBox[{"-", 
                    RowBox[{"Last", "[", "ptobj", "]"}]}]}], "}"}]}], "}"}], 
                 "]"}]}], "}"}], ",", "\[IndentingNewLine]", 
              "\[IndentingNewLine]", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"Arrowheads", "[", "Small", "]"}], ",", 
                RowBox[{"ColorData", "[", 
                 RowBox[{"\"\<HTML\>\"", ",", "\"\<SlateBlue\>\""}], "]"}], 
                ",", "Dashed", ",", 
                RowBox[{"Arrow", "[", 
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{"{", 
                    RowBox[{
                    RowBox[{"First", "[", "imageloc", "]"}], ",", "0"}], 
                    "}"}], ",", "imageloc"}], "}"}], "]"}]}], "}"}], ",", 
              "\[IndentingNewLine]", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"RGBColor", "[", 
                 RowBox[{"1", ",", ".47", ",", "0"}], "]"}], ",", 
                RowBox[{"Line", "[", 
                 RowBox[{"{", 
                  RowBox[{"mirrorloc", ",", "imageloc"}], "}"}], "]"}]}], 
               "}"}], ",", "\[IndentingNewLine]", "\[IndentingNewLine]", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"RGBColor", "[", 
                 RowBox[{"1", ",", ".47", ",", "0"}], "]"}], ",", 
                RowBox[{"Line", "[", 
                 RowBox[{"{", 
                  RowBox[{"imageloc", ",", 
                   RowBox[{"{", 
                    RowBox[{
                    RowBox[{
                    RowBox[{"First", "[", "imageloc", "]"}], "-", 
                    RowBox[{"2", "*", 
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"First", "[", "mirrorloc", "]"}], "-", 
                    RowBox[{"First", "[", "imageloc", "]"}]}], ")"}]}]}], ",", 
                    RowBox[{
                    RowBox[{"Last", "[", "imageloc", "]"}], "-", 
                    RowBox[{"2", "*", 
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"Last", "[", "mirrorloc", "]"}], "-", 
                    RowBox[{"Last", "[", "imageloc", "]"}]}], ")"}]}]}]}], 
                    "}"}]}], "}"}], "]"}]}], "}"}], ",", 
              "\[IndentingNewLine]", "\[IndentingNewLine]", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"RGBColor", "[", 
                 RowBox[{".6", ",", ".73", ",", ".36"}], "]"}], ",", 
                RowBox[{"Line", "[", 
                 RowBox[{"{", 
                  RowBox[{"location", ",", "imageloc"}], "}"}], "]"}]}], 
               "}"}]}], "\[IndentingNewLine]", "\[IndentingNewLine]", "}"}]}],
            "]"}]}], "\[IndentingNewLine]", "}"}], ",", "\[IndentingNewLine]",
         "\[IndentingNewLine]", 
        RowBox[{"ImageSize", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"450", ",", "350"}], "}"}]}], ",", 
        RowBox[{"PlotRange", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"0", ",", "20"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"-", "6"}], ",", "6"}], "}"}]}], "}"}]}], ",", 
        RowBox[{"Axes", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"False", ",", "False"}], "}"}]}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"PlotLabel", "\[Rule]", "\[IndentingNewLine]", 
         RowBox[{"Column", "[", 
          RowBox[{"{", "\[IndentingNewLine]", 
           RowBox[{
            RowBox[{"If", "[", 
             RowBox[{
              RowBox[{
               RowBox[{
                RowBox[{"First", "[", "ptobj", "]"}], "\[GreaterEqual]", 
                RowBox[{"First", "[", "location", "]"}]}], "||", 
               RowBox[{
                RowBox[{"First", "[", "ptobj", "]"}], "\[Equal]", 
                RowBox[{"First", "[", "focusloc", "]"}]}]}], ",", "\"\<\>\"", 
              ",", 
              RowBox[{"Text", "@", 
               RowBox[{"Row", "[", 
                RowBox[{"{", 
                 RowBox[{"\"\<magnification = \>\"", ",", 
                  RowBox[{"NumberForm", "[", 
                   RowBox[{
                    RowBox[{"N", "@", 
                    RowBox[{"Abs", "[", 
                    RowBox[{"magnification", "[", 
                    RowBox[{"objdistance", ",", 
                    RowBox[{"imagedistance", "[", 
                    RowBox[{"focus", ",", "objdistance"}], "]"}]}], "]"}], 
                    "]"}]}], ",", 
                    RowBox[{"{", 
                    RowBox[{"4", ",", "2"}], "}"}]}], "]"}]}], "}"}], 
                "]"}]}]}], "]"}], ",", 
            RowBox[{"If", "[", 
             RowBox[{
              RowBox[{
               RowBox[{
                RowBox[{"First", "[", "ptobj", "]"}], "\[GreaterEqual]", 
                RowBox[{"First", "[", "location", "]"}]}], "||", 
               RowBox[{
                RowBox[{"First", "[", "ptobj", "]"}], "\[Equal]", 
                RowBox[{"First", "[", "focusloc", "]"}]}]}], ",", "\"\<\>\"", 
              ",", 
              RowBox[{"Text", "@", 
               RowBox[{"Row", "[", 
                RowBox[{"{", 
                 RowBox[{"\"\<This is a \>\"", ",", 
                  RowBox[{"If", "[", 
                   RowBox[{
                    RowBox[{
                    RowBox[{"magnification", "[", 
                    RowBox[{"objdistance", ",", 
                    RowBox[{"imagedistance", "[", 
                    RowBox[{"focus", ",", "objdistance"}], "]"}]}], "]"}], 
                    "<", "0"}], ",", "\"\<virtual, upright\>\"", ",", 
                    "\"\<real, inverted\>\""}], "]"}], ",", 
                  "\"\< image.\>\""}], "}"}], "]"}]}]}], "]"}]}], 
           "\[IndentingNewLine]", "}"}], "]"}]}]}], "\[IndentingNewLine]", 
       "\[IndentingNewLine]", "]"}]}]}], 
    RowBox[{"(*", 
     RowBox[{"End", " ", "Graphics"}], "*)"}], "\[IndentingNewLine]", 
    "\[IndentingNewLine]", "]"}], 
   RowBox[{"(*", 
    RowBox[{"End", " ", "Module"}], "*)"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{"(*", "Controls", "*)"}], "\[IndentingNewLine]", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "mirrortype", ",", "\"\<concave\>\"", ",", "\"\<mirror type\>\""}], 
      "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"\"\<concave\>\"", ",", "\"\<convex\>\""}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"focus", ",", "2", ",", "\"\<focal length\>\""}], "}"}], ",", 
     "2", ",", "3.5", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"ptobj", ",", 
       RowBox[{"{", 
        RowBox[{"5", ",", "2"}], "}"}], ",", "\"\<object\>\""}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", 
       RowBox[{"-", "3"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"20", ",", "3"}], "}"}], ",", "Locator"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"location", ",", 
       RowBox[{"{", 
        RowBox[{"15", ",", "0"}], "}"}], ",", "\"\<location\>\""}], "}"}], 
     ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"20", ",", "0"}], "}"}], ",", "Locator"}], "}"}], ",", 
   RowBox[{"Button", "[", 
    RowBox[{
     RowBox[{"Row", "[", 
      RowBox[{"{", "\"\<reset values\>\"", "}"}], "]"}], ",", 
     RowBox[{
      RowBox[{"mirrortype", " ", "=", " ", "\"\<concave\>\""}], ";", 
      RowBox[{"focus", " ", "=", " ", "2"}], ";", 
      RowBox[{"ptobj", " ", "=", " ", 
       RowBox[{"{", 
        RowBox[{"5", ",", "2"}], "}"}]}], ";", 
      RowBox[{"location", " ", "=", " ", 
       RowBox[{"{", 
        RowBox[{"15", ",", "0"}], "}"}]}], ";"}]}], "]"}], ",", 
   RowBox[{"TrackedSymbols", "\[RuleDelayed]", "Manipulate"}]}], 
  "\[IndentingNewLine]", "\[IndentingNewLine]", "]"}]], "Input",
 CellChangeTimes->{{3.4017583430707855`*^9, 3.4017586715330915`*^9}, {
   3.401758988889427*^9, 3.401759175627944*^9}, {3.4017592099673214`*^9, 
   3.4017593168510127`*^9}, {3.4017593939518785`*^9, 3.401759454549013*^9}, {
   3.401759557226656*^9, 3.4017595586186576`*^9}, {3.401759628779544*^9, 
   3.401759656849907*^9}, {3.401759709866141*^9, 3.4017597336403265`*^9}, {
   3.401759782170109*^9, 3.4017597834619665`*^9}, {3.401759816128939*^9, 
   3.401759939005627*^9}, {3.4017599805453587`*^9, 3.4017600085956926`*^9}, {
   3.401760043245517*^9, 3.401760085416155*^9}, {3.401760132303576*^9, 
   3.401760161405422*^9}, {3.4017601964658365`*^9, 3.401760261389192*^9}, {
   3.4017603177101774`*^9, 3.401760338660302*^9}, {3.4017605445563664`*^9, 
   3.40176060658556*^9}, {3.401760884545246*^9, 3.401760909941765*^9}, {
   3.4017611177105217`*^9, 3.401761171728195*^9}, {3.4017612121262846`*^9, 
   3.4017612456244526`*^9}, {3.4017612958166256`*^9, 
   3.4017613732980385`*^9}, {3.401761425593235*^9, 3.4017614315618176`*^9}, {
   3.401761474633752*^9, 3.40176147925039*^9}, {3.401761581006709*^9, 
   3.401761581056781*^9}, {3.401761615766691*^9, 3.4017617607351456`*^9}, {
   3.4018359209575987`*^9, 3.401835957630331*^9}, {3.401836004748083*^9, 
   3.4018360052588177`*^9}, {3.4018361328923454`*^9, 
   3.4018361508381505`*^9}, {3.401836427886526*^9, 3.4018364618653855`*^9}, {
   3.401836636997213*^9, 3.4018366379485807`*^9}, {3.4018367264558477`*^9, 
   3.4018367746651697`*^9}, {3.4018368210318413`*^9, 3.401836822163469*^9}, {
   3.4018370943448467`*^9, 3.401837095666747*^9}, {3.4018371563740396`*^9, 
   3.4018371711853375`*^9}, {3.4018372390829697`*^9, 
   3.4018372671633472`*^9}, {3.4018373020935745`*^9, 3.401837370171466*^9}, {
   3.4018374467315536`*^9, 3.40183745981036*^9}, {3.401837495962344*^9, 
   3.40183752290108*^9}, {3.4018391749265757`*^9, 3.401839245758427*^9}, {
   3.401839404757056*^9, 3.40183961315672*^9}, {3.401839711177667*^9, 
   3.401839760098011*^9}, {3.401839810109925*^9, 3.4018398664809823`*^9}, {
   3.4018399134485188`*^9, 3.401839973855379*^9}, {3.4018400242077827`*^9, 
   3.4018401948231153`*^9}, {3.4018402436833725`*^9, 
   3.4018402437334447`*^9}, {3.4018403252206173`*^9, 3.401840335124859*^9}, {
   3.4018404058765955`*^9, 3.4018404631088915`*^9}, {3.4018405430538464`*^9, 
   3.401840589791051*^9}, {3.401840626093251*^9, 3.40184073827456*^9}, {
   3.4018412393350496`*^9, 3.401841243030363*^9}, {3.401841361480686*^9, 
   3.4018413637239122`*^9}, {3.4018414443398323`*^9, 3.401841445942136*^9}, {
   3.401841477707813*^9, 3.401841479219987*^9}, {3.401841553516821*^9, 
   3.4018415548086786`*^9}, {3.4018430146378098`*^9, 
   3.4018430296794386`*^9}, {3.4018430798515825`*^9, 3.401843112638728*^9}, {
   3.40184314623704*^9, 3.401843298025301*^9}, {3.401843376488125*^9, 
   3.4018434508650737`*^9}, {3.401843496440608*^9, 3.4018435460319166`*^9}, {
   3.4018435860594735`*^9, 3.4018436460657587`*^9}, {3.401843693373784*^9, 
   3.4018437426145887`*^9}, {3.4018438076280737`*^9, 3.401844016708717*^9}, {
   3.401844153996126*^9, 3.4018442320583744`*^9}, {3.4018443209562035`*^9, 
   3.401844382294403*^9}, {3.4018444604367666`*^9, 3.4018444806458254`*^9}, {
   3.401844615800168*^9, 3.401844701623576*^9}, {3.401844734791269*^9, 
   3.4018447811679554`*^9}, {3.4018533467146063`*^9, 3.401853353634557*^9}, {
   3.401853401883936*^9, 3.401853402014123*^9}, {3.4018535197434096`*^9, 
   3.401853612937416*^9}, {3.401853644342574*^9, 3.401853709486246*^9}, {
   3.4018538816638255`*^9, 3.401853930504054*^9}, {3.4018539691095667`*^9, 
   3.4018540346738434`*^9}, {3.401854161456147*^9, 3.4018541735936003`*^9}, {
   3.4018542587560577`*^9, 3.4018543444793215`*^9}, {3.4018543815526304`*^9, 
   3.4018543948217106`*^9}, {3.4018547090835967`*^9, 
   3.4018547774519053`*^9}, {3.4018549040339217`*^9, 
   3.4018549114545918`*^9}, {3.401854976448048*^9, 3.401855071184272*^9}, {
   3.4018551829850335`*^9, 3.401855338138133*^9}, {3.4018554193749456`*^9, 
   3.4018554345968337`*^9}, {3.40185553104552*^9, 3.4018555319568305`*^9}, {
   3.401855711955656*^9, 3.4018557180043535`*^9}, {3.4018558133414416`*^9, 
   3.401855819880845*^9}, {3.401855900847269*^9, 3.4018559676132736`*^9}, {
   3.4018561035587535`*^9, 3.401856105822008*^9}, {3.4018562877435985`*^9, 
   3.40185628963632*^9}, {3.401856370122053*^9, 3.4018563880778723`*^9}, {
   3.4018564291969986`*^9, 3.4018564800701504`*^9}, {3.4018565182450433`*^9, 
   3.4018565201577935`*^9}, {3.401856654931589*^9, 3.4018567100608606`*^9}, {
   3.401856757088483*^9, 3.401856808692686*^9}, {3.4018598217143073`*^9, 
   3.4018598217643795`*^9}, {3.401860190795019*^9, 3.401860313711765*^9}, {
   3.401860366517696*^9, 3.4018603838626366`*^9}, {3.4018604543039265`*^9, 
   3.4018604551150928`*^9}, {3.401860512767994*^9, 3.4018605466867666`*^9}, {
   3.4018605917515664`*^9, 3.401860635384307*^9}, {3.4018606850156736`*^9, 
   3.401860743589899*^9}, {3.401860831516331*^9, 3.401860832097166*^9}, {
   3.4018802676394176`*^9, 3.4018804671462946`*^9}, {3.401894040841736*^9, 
   3.401894049424077*^9}, {3.401894090663376*^9, 3.40189416186576*^9}, {
   3.401894261859544*^9, 3.4018942971703186`*^9}, {3.4018943411936207`*^9, 
   3.401894357467021*^9}, 3.4018945371954575`*^9, 3.4018946946218257`*^9, {
   3.4018947513534017`*^9, 3.401894753997203*^9}, {3.40189516546887*^9, 
   3.4018951659195185`*^9}, 3.401895226977315*^9, {3.4018952767789264`*^9, 
   3.4018953621116285`*^9}, {3.4018954919383106`*^9, 
   3.4018955200988035`*^9}, {3.401895604169691*^9, 3.4018956689127874`*^9}, {
   3.40189577985231*^9, 3.4018958174363537`*^9}, {3.4018959689241824`*^9, 
   3.4018960071391325`*^9}, {3.401896152828624*^9, 3.401896154781432*^9}, {
   3.4018963777320194`*^9, 3.4018964338427024`*^9}, {3.4018965778297453`*^9, 
   3.401896692855144*^9}, {3.4018968450439806`*^9, 3.401896956944886*^9}, {
   3.4018970377711086`*^9, 3.4018970524321904`*^9}, {3.4018971166645517`*^9, 
   3.4018971437635183`*^9}, {3.401897761271451*^9, 3.4018978321333456`*^9}, {
   3.4018978621865597`*^9, 3.401897902254174*^9}, {3.4018979879774384`*^9, 
   3.4018980686734734`*^9}, {3.4018981076495185`*^9, 3.401898110723939*^9}, {
   3.4018981768990946`*^9, 3.4018982301857166`*^9}, {3.4018982667382765`*^9, 
   3.4018982952192307`*^9}, {3.4018983252323875`*^9, 
   3.4018983358075933`*^9}, {3.4018984251560707`*^9, 
   3.4018985650171804`*^9}, {3.4018986817650557`*^9, 
   3.4018987158741026`*^9}, {3.40189947409437*^9, 3.4018994786809645`*^9}, {
   3.4018995274210496`*^9, 3.401899531106349*^9}, {3.4018997086816893`*^9, 
   3.4018997884463854`*^9}, {3.401899878706173*^9, 3.401899968365096*^9}, {
   3.40190014738251*^9, 3.401900161012109*^9}, {3.4019002032828913`*^9, 
   3.401900217373152*^9}, 3.4019002742249007`*^9, {3.401900606332448*^9, 
   3.4019006293555536`*^9}, {3.401900670324464*^9, 3.4019007047639856`*^9}, {
   3.4019007384223843`*^9, 3.40190076176595*^9}, {3.4019008218723793`*^9, 
   3.4019008476895027`*^9}, {3.4019346534864817`*^9, 3.401935016638669*^9}, {
   3.4019350562756643`*^9, 3.4019350651984944`*^9}, {3.4019351756673408`*^9, 
   3.401935332402715*^9}, {3.4019672786797037`*^9, 3.4019673668164387`*^9}, {
   3.4019674333220687`*^9, 3.4019674917460785`*^9}, {3.401967833607651*^9, 
   3.401967902346493*^9}, {3.4019679336515074`*^9, 3.4019679514070387`*^9}, {
   3.401996103703176*^9, 3.401996135278579*^9}, {3.401996170268893*^9, 
   3.4019962041375933`*^9}, {3.4019962492224226`*^9, 
   3.4019962728363776`*^9}, {3.4019963033202114`*^9, 3.401996331350517*^9}, {
   3.401996370717123*^9, 3.40199637435235*^9}, {3.4019964173742127`*^9, 
   3.401996437603301*^9}, {3.4020514554193544`*^9, 3.4020514794818544`*^9}, {
   3.4020727786627946`*^9, 3.4020727825534196`*^9}, {3.4020728346627946`*^9, 
   3.4020728931471696`*^9}, {3.4020729694440446`*^9, 
   3.4020729763971696`*^9}, {3.4020730102096696`*^9, 3.4020730447096696`*^9},
    {3.402232421546875*^9, 3.40223244078125*^9}, {3.402232505765625*^9, 
   3.40223250596875*^9}},
 CellID->818468755]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`focus$$ = 2, $CellContext`location$$ = {15, 
    0}, $CellContext`mirrortype$$ = "concave", $CellContext`ptobj$$ = {5, 2}, 
    Typeset`show$$ = True, Typeset`bookmarkList$$ = {}, 
    Typeset`bookmarkMode$$ = "Menu", Typeset`animator$$, Typeset`animvar$$ = 
    1, Typeset`name$$ = "\"untitled\"", Typeset`specs$$ = {{{
       Hold[$CellContext`mirrortype$$], "concave", "mirror type"}, {
      "concave", "convex"}}, {{
       Hold[$CellContext`focus$$], 2, "focal length"}, 2, 3.5}, {{
       Hold[$CellContext`ptobj$$], {5, 2}, "object"}, {0, -3}, {20, 3}}, {{
       Hold[$CellContext`location$$], {15, 0}, "location"}, {0, 0}, {20, 0}}, {
      Hold[
       Button[
        Row[{"reset values"}], $CellContext`mirrortype$$ = 
         "concave"; $CellContext`focus$$ = 
         2; $CellContext`ptobj$$ = {5, 2}; $CellContext`location$$ = {15, 0}; 
        Null]], Manipulate`Dump`ThisIsNotAControl}}, Typeset`size$$ = {
    450., {173., 177.}}, Typeset`update$$ = 0, Typeset`initDone$$, 
    Typeset`skipInitDone$$ = True, $CellContext`mirrortype$126214$$ = 
    False, $CellContext`focus$126215$$ = 0, $CellContext`ptobj$126216$$ = {0, 
    0}, $CellContext`location$126217$$ = {0, 0}}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, 
      "Variables" :> {$CellContext`focus$$ = 
        2, $CellContext`location$$ = {15, 0}, $CellContext`mirrortype$$ = 
        "concave", $CellContext`ptobj$$ = {5, 2}}, "ControllerVariables" :> {
        Hold[$CellContext`mirrortype$$, $CellContext`mirrortype$126214$$, 
         False], 
        Hold[$CellContext`focus$$, $CellContext`focus$126215$$, 0], 
        Hold[$CellContext`ptobj$$, $CellContext`ptobj$126216$$, {0, 0}], 
        Hold[$CellContext`location$$, $CellContext`location$126217$$, {0, 
         0}]}, "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, "Body" :> 
      Module[{$CellContext`radius$, $CellContext`objdistance$, \
$CellContext`focusloc$, $CellContext`imageloc$, $CellContext`radiusloc$, \
$CellContext`mirrorloc$}, $CellContext`imagedistance[
           Pattern[$CellContext`foclength$, 
            Blank[]], 
           Pattern[$CellContext`objdistance$, 
            Blank[]]] := If[$CellContext`mirrortype$$ == "concave", 
           If[
            Or[
            First[$CellContext`ptobj$$] == 
             First[$CellContext`focusloc$], $CellContext`objdistance$ == 
             0, $CellContext`foclength$ == $CellContext`objdistance$], 0, 
            Part[
             Solve[
             1/$CellContext`foclength$ == 
              1/$CellContext`objdistance$ + 1/$CellContext`q, $CellContext`q],
              1, 1, 2]], 
           If[
            Or[
            First[$CellContext`ptobj$$] == 
             First[$CellContext`focusloc$], $CellContext`objdistance$ == 
             0, $CellContext`foclength$ == $CellContext`objdistance$], 0, 
            Part[
             Solve[
             1/(-$CellContext`foclength$) == 
              1/$CellContext`objdistance$ + 1/$CellContext`q, $CellContext`q],
              1, 1, 2]]]; $CellContext`magnification[
           Pattern[$CellContext`objdistance$, 
            Blank[]], 
           Pattern[$CellContext`imgdistance$, 
            Blank[]]] := If[
           Or[
           First[$CellContext`ptobj$$] == 
            First[$CellContext`focusloc$], $CellContext`objdistance$ == 
            0, $CellContext`imgdistance$ == 0], 
           0, $CellContext`imgdistance$/$CellContext`objdistance$]; \
$CellContext`radius$ = 2 $CellContext`focus$$; $CellContext`objdistance$ = 
         First[$CellContext`location$$] - 
          First[$CellContext`ptobj$$]; $CellContext`focusloc$ = {(
            First[$CellContext`location$$] - $CellContext`radius$) + \
$CellContext`focus$$, 0}; $CellContext`imageloc$ = {
          First[$CellContext`location$$] - \
$CellContext`imagedistance[$CellContext`focus$$, $CellContext`objdistance$], \
$CellContext`magnification[$CellContext`objdistance$, \
-$CellContext`imagedistance[$CellContext`focus$$, $CellContext`objdistance$]] 
           Last[$CellContext`ptobj$$]}; $CellContext`mirrorloc$ = 
         If[$CellContext`mirrortype$$ == "concave", {
           Sqrt[$CellContext`radius$^2 - Last[$CellContext`ptobj$$]^2] + 
            First[$CellContext`location$$] - $CellContext`radius$, 
            
            Last[$CellContext`ptobj$$]}, {-
             Sqrt[$CellContext`radius$^2 - Last[$CellContext`ptobj$$]^2] + 
            First[$CellContext`location$$] + $CellContext`radius$, 
            Last[$CellContext`ptobj$$]}]; Graphics[{
           If[$CellContext`mirrortype$$ == "concave", {Thick, 
             
             Circle[{First[$CellContext`location$$] - $CellContext`radius$, 
               0}, $CellContext`radius$, {(-Pi)/3, Pi/3}]}, {Thick, 
             
             Circle[{First[$CellContext`location$$] + $CellContext`radius$, 
               0}, $CellContext`radius$, {2 (Pi/3), 4 (Pi/3)}]}], {
            Arrowheads[Small], 
            ColorData["HTML", "SlateBlue"], 
            Arrow[{{
               First[$CellContext`ptobj$$], 0}, $CellContext`ptobj$$}]}, {
           Thick, 
            Line[{{0, 0}, {20, 0}}]}, 
           If[
            Or[
            First[$CellContext`ptobj$$] >= First[$CellContext`location$$], 
             First[$CellContext`ptobj$$] == 
             First[$CellContext`focusloc$]], {}, {{
              RGBColor[1, 0.47, 0], 
              Line[{$CellContext`ptobj$$, $CellContext`mirrorloc$}]}, {
              RGBColor[0.6, 0.73, 0.36], 
              Line[{$CellContext`ptobj$$, $CellContext`location$$, {
                 First[$CellContext`ptobj$$], -
                 Last[$CellContext`ptobj$$]}}]}, {
              Arrowheads[Small], 
              ColorData["HTML", "SlateBlue"], Dashed, 
              Arrow[{{
                 First[$CellContext`imageloc$], 
                 0}, $CellContext`imageloc$}]}, {
              RGBColor[1, 0.47, 0], 
              Line[{$CellContext`mirrorloc$, $CellContext`imageloc$}]}, {
              RGBColor[1, 0.47, 0], 
              
              Line[{$CellContext`imageloc$, {
                First[$CellContext`imageloc$] - 
                 2 (First[$CellContext`mirrorloc$] - 
                  First[$CellContext`imageloc$]), 
                 Last[$CellContext`imageloc$] - 
                 2 (Last[$CellContext`mirrorloc$] - 
                  Last[$CellContext`imageloc$])}}]}, {
              RGBColor[0.6, 0.73, 0.36], 
              Line[{$CellContext`location$$, $CellContext`imageloc$}]}}]}, 
          ImageSize -> {450, 350}, PlotRange -> {{0, 20}, {-6, 6}}, 
          Axes -> {False, False}, PlotLabel -> Column[{
             If[
              Or[
              First[$CellContext`ptobj$$] >= First[$CellContext`location$$], 
               First[$CellContext`ptobj$$] == First[$CellContext`focusloc$]], 
              "", 
              Text[
               Row[{"magnification = ", 
                 NumberForm[
                  N[
                   Abs[
                    $CellContext`magnification[$CellContext`objdistance$, 
                    $CellContext`imagedistance[$CellContext`focus$$, \
$CellContext`objdistance$]]]], {4, 2}]}]]], 
             If[
              Or[
              First[$CellContext`ptobj$$] >= First[$CellContext`location$$], 
               First[$CellContext`ptobj$$] == First[$CellContext`focusloc$]], 
              "", 
              Text[
               Row[{"This is a ", 
                 If[$CellContext`magnification[$CellContext`objdistance$, 
                    $CellContext`imagedistance[$CellContext`focus$$, \
$CellContext`objdistance$]] < 0, "virtual, upright", "real, inverted"], 
                 " image."}]]]}]]], 
      "Specifications" :> {{{$CellContext`mirrortype$$, "concave", 
          "mirror type"}, {
         "concave", "convex"}}, {{$CellContext`focus$$, 2, "focal length"}, 2,
          3.5, Appearance -> 
         "Labeled"}, {{$CellContext`ptobj$$, {5, 2}, "object"}, {0, -3}, {20, 
         3}, ControlType -> 
         Locator}, {{$CellContext`location$$, {15, 0}, "location"}, {0, 0}, {
         20, 0}, ControlType -> Locator}, 
        Button[
         Row[{"reset values"}], $CellContext`mirrortype$$ = 
          "concave"; $CellContext`focus$$ = 
          2; $CellContext`ptobj$$ = {5, 2}; $CellContext`location$$ = {15, 0}; 
         Null]}, "Options" :> {TrackedSymbols :> Manipulate}, 
      "DefaultOptions" :> {ControllerLinking -> True}],
     ImageSizeCache->{501., {245., 250.}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 CellID->299115617],

Cell[CellGroupData[{

Cell["CAPTION", "Section",
 CellFrame->{{0, 0}, {1, 0}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->12,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell["\<\
This Demonstration lets you visualize the ray diagrams for concave and convex \
spherical mirrors. By manipulating the object and mirror locations, you can \
create real or virtual images. The ray parallel to the principal axis and the \
ray that hits the center of the mirror are drawn.\
\>", "Text"]
}, Close]]
}, Open  ]],

Cell[CellGroupData[{

Cell["DETAILS", "Section",
 CellFrame->{{0, 0}, {1, 0}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->12,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell["\<\
Locators allow you to drag both the object and the mirror. You can change the \
focal length using a slider. The degree of magnification and nature (real or \
virtual) of the image are displayed.\
\>", "Text"]
}, Close]],

Cell[CellGroupData[{

Cell["THIS NOTEBOOK IS THE SOURCE CODE FROM", "Text",
 CellFrame->{{0, 0}, {0, 0}},
 CellMargins->{{48, 10}, {4, 28}},
 CellGroupingRules->{"SectionGrouping", 25},
 CellFrameMargins->{{48, 48}, {6, 5}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->10,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[TextData[{
 "\"",
 ButtonBox["Ray Diagrams for Spherical Mirrors",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/RayDiagramsForSphericalMirrors/"], 
    None},
  ButtonNote->
   "http://demonstrations.wolfram.com/RayDiagramsForSphericalMirrors/"],
 "\"",
 " from ",
 ButtonBox["the Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
 "\[ParagraphSeparator]\[NonBreakingSpace]",
 ButtonBox["http://demonstrations.wolfram.com/RayDiagramsForSphericalMirrors/",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/RayDiagramsForSphericalMirrors/"], 
    None},
  ButtonNote->
   "http://demonstrations.wolfram.com/RayDiagramsForSphericalMirrors/"]
}], "Text",
 CellMargins->{{48, Inherited}, {0, Inherited}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.5]],

Cell[TextData[{
 "Contributed by: ",
 ButtonBox["Ernest Lee",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/author.html?author=Ernest+Lee"], 
    None},
  ButtonNote->
   "http://demonstrations.wolfram.com/author.html?author=Ernest+Lee"]
}], "Text",
 CellDingbat->"\[FilledSmallSquare]",
 CellMargins->{{66, 48}, {2, 4}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.6]],

Cell[CellGroupData[{

Cell[TextData[{
 "A full-function Wolfram ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " system (Version 6 or higher) is required to edit this notebook.\n",
 StyleBox[ButtonBox["GET WOLFRAM MATHEMATICA \[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolfram.com/products/mathematica/"], None},
  ButtonNote->"http://www.wolfram.com/products/mathematica/"],
  FontFamily->"Helvetica",
  FontWeight->"Bold",
  FontSlant->"Italic",
  FontColor->RGBColor[1, 0.42, 0]]
}], "Text",
 CellFrame->True,
 CellMargins->{{48, 68}, {8, 28}},
 CellFrameMargins->12,
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 CellChangeTimes->{3.3750111182355957`*^9},
 ParagraphSpacing->{1., 1.},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.411765],
 Background->RGBColor[1, 1, 1]],

Cell[TextData[{
 "\[Copyright] ",
 StyleBox[ButtonBox["Wolfram Demonstrations Project & Contributors",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Terms of Use",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/termsofuse.html"], None},
  ButtonNote->"http://demonstrations.wolfram.com/termsofuse.html"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Make a new version of this Demonstration \
\[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/participate/upload.jsp?id=\
RayDiagramsForSphericalMirrors"], None},
  ButtonNote->None],
  FontColor->GrayLevel[0.6]]
}], "Text",
 CellFrame->{{0, 0}, {0, 0.5}},
 CellMargins->{{48, 10}, {20, 50}},
 CellFrameMargins->{{6, 0}, {6, 6}},
 CellFrameColor->GrayLevel[0.6],
 FontFamily->"Verdana",
 FontSize->9,
 FontColor->GrayLevel[0.6]]
}, Open  ]]
}, Open  ]]
},
Editable->True,
Saveable->False,
ScreenStyleEnvironment->"Working",
CellInsertionPointCell->None,
WindowSize->{710, 650},
WindowMargins->{{Inherited, Inherited}, {Inherited, 0}},
WindowElements->{
 "StatusArea", "MemoryMonitor", "MagnificationPopUp", "VerticalScrollBar", 
  "MenuBar"},
WindowTitle->"Ray Diagrams for Spherical Mirrors - Source",
DockedCells->{},
CellContext->Notebook,
FrontEndVersion->"8.0 for Microsoft Windows (32-bit) (November 7, 2010)",
StyleDefinitions->"Default.nb"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[636, 23, 154, 3, 70, "Section"],
Cell[793, 28, 3700, 64, 70, "Section"],
Cell[4496, 94, 25340, 529, 70, "Input",
 CellID->818468755]
}, Open  ]],
Cell[CellGroupData[{
Cell[29873, 628, 9153, 184, 70, "Output",
 CellID->299115617],
Cell[CellGroupData[{
Cell[39051, 816, 209, 6, 70, "Section"],
Cell[39263, 824, 311, 5, 70, "Text"]
}, Close]]
}, Open  ]],
Cell[CellGroupData[{
Cell[39622, 835, 209, 6, 70, "Section"],
Cell[39834, 843, 219, 4, 70, "Text"]
}, Close]],
Cell[CellGroupData[{
Cell[40089, 852, 355, 9, 70, "Text",
 CellGroupingRules->{"SectionGrouping", 25}],
Cell[40447, 863, 979, 28, 70, "Text"],
Cell[41429, 893, 427, 14, 70, "Text"],
Cell[CellGroupData[{
Cell[41881, 911, 815, 24, 70, "Text"],
Cell[42699, 937, 1202, 33, 70, "Text"]
}, Open  ]]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
(* NotebookSignature vTzwa0nzq3ZbbC0D6qYKo7NE *)
