(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 8.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       157,          7]
NotebookDataLength[     53662,       1305]
NotebookOptionsPosition[     51764,       1237]
NotebookOutlinePosition[     52406,       1262]
CellTagsIndexPosition[     52363,       1259]
WindowTitle->Lensmaker's Equation - Source
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["Lensmaker's Equation", "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False,
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztnVuSozgWQDNiPqa3MJ+zpVlCbaB3WlGVNV9dvYH6TQRUph/ZGIHQmwvG
xoZzQu3Aku7VfUj4Nk53//fLn//78q+Xl5f/NP/88e+Xl8v154Xj5+fp89y0
4+X63L7VnQu02XqOt9FzjRK5qtRMoYZFpk11/Ji+WDye92+r7+fFfbnGkuu9
eIQ4LBVJ+YRo5DPnRdITDp0TLRg6m/62nY/n08f5+H461G37fWqum57z4Xy2
jby0prMZOh4/jodLOx2bt6fzRSEA3B/KMKGSBy/DSKVcyYOn8krHqajvqed6
G6iol4qkfAIVNQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAADsizPA/lj72AEA
AMCmWLu0AXh01j6jAAAAcdb+hNwUJ5e1zZnPDEeey/e1jx0AAABsirVLm03x
XFVlhs1X1JNY+4wCAAB02B9PJ4BdsvYpBAAAgOeGihpADo9GAUDIr+Lnr7ef
a9vw16/i73VtMPx6+2v1gGwMijeAE89FAQA2DRW1BxX14lBRA0wi+lx0baMA
AAAAAAAAAAAAbkj0uegR4L4cDoe1TYA1YQMAbIaDy9rm3JU9+w4AjwB3np3D
BgDYDHuuKu/su3k0ynNRACGLHMy93dm2ijCPpHvD7LliAQAAmA0VNRioqIGK
GlYk9Vz0AAAAAAAAAAAAALBR9KPR8Kf0a9sFAAAAAAAAAAAAcCt4LgoAAAAA
AAAAAAB7I/Vc9ONheH9/X9uEzUJsZ0PoAGBZ3gNuLQgggR21GUjl7SC2syF0
ALAsVNQgxMv14XAI/xOja9sIAAAAT0lYWM5jbT/msGffAQAAAGAp9lxV3sF3
b1r0uehSZgAAAAAAAAAAAAA8Ar9b6rpuXpu3+tf0PBcFWJcFvxYBAAAAANgh
VNQAADCK5LnobwAAAAAAAAAAAIANUbcPRav+wehH7LmoUqpsqKqmNTSXRVE0
by593mtRXlrfo7mImTlVN6ebUOhR07q3jmxVWXqquqxVOUjZq1eDVNe662ro
Nxe1Xr5o2lt5cUc7VWr3BpGindNL6TkDl5nVIGWtbuaXWbwJo/OVKjSX68oL
shMTO4ylHf+yy5GbhU5EubJJPeWgx5ayBfvOIR2qD5QJXWmFtJs2pLsLvu40
4oOesh/tJ3TKW/3DomZONgVdVPtNXvVXYWydyBRl/NWNWNW62L06G344TcNW
L/SVpaTqcmeOWGWOm3W+TFPWsbL8dE1yklUal73ghDtBH+Hu1dE/HBM7/t7h
sjNoN29Calrl2GadWfsMeoc0elcR584clrm5qzO5a+5mQe4imy2Tu1qcu+g2
7od7mkvVtqJtajgInQuqzUrRtWFCzIAkzQdK0/RR1U33JEhtgOhosnmLKtX0
DArVhaJHtZOlmgPbVN+mWZhRGzNvEc13MD61kIRcEPS2WciRp/C97a0ow569
DCOVRurZU0lFnUoBFbWkKiv7I0xFLc4dFXUAFfVUtVTUYRD2V1EnfTd7u920
v+va/NXoSXM86qXVwFtzgNvS7vJG3730Ga76Wkq/6lFz3dlWOa+69XpUOSix
ixQtrrVV1pxBleO+uesMsv6QjenXF+bViFjGO5rtObaUp7bVoENe21Kqy1Un
1M8xOaxNftysOh61q5joKStQpbkIQ2p54HRWXcqsOsbVU7oZNBr8Fbv+yotk
GBkvbuFbu9/LkY6DF3ZvTvqtOVXdBi0dlN6WZekEsHfQ2flmjpnQ+25vfn//
2yH1eswnokp/tNkpsJNrTqJruX4tzAkKBJXJVdUbbJmtrwtrUfuoGm2Ru5S3
AcrEGQyPld3jzbFTH+oP+is3d+XVuev2wI1zN9w8rZM1I3dFf203xxKvuUyT
TevJkd8JSzF7iVHBzJa+D0vFZJLapawNh6xiQ91ixaf03R+lDHvaMqwPCKl8
+lRGgkZF3W3Q0oGKmoq685yKmoo6L5jZ0vdhqZhMUruUteEQFfUk37VU98C8
qRvq+uPjQz8UPZ/P+snoZYIqVAwtaFY0r+E0+8l8VIM3Gs70JkelUoLR0XBm
2X5dldcmWS5lQ2r1jDue+DBaRvrzvidVpc2WOKJicRvVZu+HaFqjgpngC/MS
Fek2rRvSjKCy9rnxfepWFFKokVMzWaEsU9GDbEYnbfuoYOpGkdcwql/Sv+fc
CVdJcY1a+Yo3WuUaZsfE+9S+j3mTRuFZoAxL9eTtFK6ecccTH0bnlmGkMtXz
dKnMLCTpifZTUUfnyNOR4QGrstQuHTXGFqSiXiV3wlVSXKNWvuKNVrmG2TGh
ooZb0+T0rUWf/br9e1HvuWjR/z1/+JOUIujKPxYu+l9YCEX0kGVARDxjjHxa
ygZv6ejF6KiE0cfp0eUk6JMrmVamQx1qkPTIRzNSo4LC0OnbV7S/tD7WRw32
ElFYP1/y7EkZn09lPhHedej7vDiHSkbzq79YsWcWfUUUzYgXf7mdqa04eiK0
ebYxe86d9yE+T3n+OzuVqHbMx01Up/kkCud7Q0ZJ5uvCRQhNGvVCoueayK9O
mKP8qDDd9szol7b5AM5LSup0P7LvxXBX8dWGvlCGTTJm1IBwianK3QySyo2k
clR/KtShBkmPfDQjNSpIRS0xexJRy8NjS0UtMfVBcmegoh6FijrkEarKPVfU
Gd89w+rwd/Tt34uq/kuT6C2xsG5WhXvLsu+KUb+8/kxsvQMryUIxdp8PLc/c
FiblXaJEcgvKuODFf54N8yYXwYejZ2fUbO8i5b7kXwRSPfJtFu1JWZ7BtlZu
eRg9XXJEbcuriorIU2xkjYjQi2hJFjXeFvFmhjeNUWu9HhO3/LQoe8tdSKqC
ys+f+tE5aZVoBSU0T87U+sr2emq4Ur7k9eQrhydiaqiF3LqSX4QFfacMk684
yaT7l2GkUr7iJJOoqKMrUlGPritxZ9RyT9aIUFHn133e3IVQUQvVUlHPhor6
1r6r/k9G9X1A/47+dDrZ/9+lb9++vbr8+PHDe7X7w077rafHU+JNTo1GF/Wu
U52hhtCjqGEZJVGdoZsp5ZmZoRe2kc3Fd0tnKGKvJQ9UKr8pRh15dY0MZTOO
e/pTa43OSUmFpn5Pbzzv7agveTNCJRnZ70FCX4OU5ZWn9KfiELrpLZ3aIZn0
RYmGN+NRPsIpd7y19pa7qIhQZ15coicVmamMLjpb1SLmLUgmzlOtzau6RRAm
bYxFTJoh9Vy+U4aNKonqDN1MKc/MDL2wjXydWIaRylElz5LKVExSS2Scigqm
HHl1jQxlM457+lNrjc5JSYWmUlF7jqQ2RmqHZNIXJRrejEf5CKfc8dbaW+6i
IkKdeXGJnlRkpjK66GxVi5i3IJk4T7U2r+oWQZi0MRYxaYbUZnyPLvH/lte2
FP/69esnAAAAAAAAAAAAwM74B4thROM=
   "], {{0, 0}, {1800, 25}}, {0, 255},
   ColorFunction->RGBColor],
  ImageSize->{1800, 25},
  PlotRange->{{0, 1800}, {0, 25}}]], "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", 
   RowBox[{"n_", ",", "d_", ",", "s1_", ",", "s2_"}], "]"}], ":=", 
  FractionBox["1", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"n", "-", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"s1", "-", "s2", "+", 
      RowBox[{
       FractionBox[
        RowBox[{"n", "-", "1"}], "n"], "d", " ", "s1", " ", "s2"}]}], 
     ")"}]}]]}]], "Input",
 CellChangeTimes->{{3.4180677771256723`*^9, 3.418067856685467*^9}, {
  3.418068154757093*^9, 3.418068240847741*^9}, {3.4180690203860197`*^9, 
  3.418069025653666*^9}, {3.4202343920920753`*^9, 3.420234400065445*^9}},
 CellID->879060982],

Cell[BoxData[
 RowBox[{
  RowBox[{"color", "[", 
   RowBox[{"n_", ",", "d_", ",", "s1_", ",", "s2_"}], "]"}], ":=", 
  RowBox[{"If", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"f", "[", 
       RowBox[{"n", ",", "d", ",", "s1", ",", "s2"}], "]"}], "<", "0"}], "&&",
      " ", 
     RowBox[{
      RowBox[{"Abs", "[", "s1", "]"}], ">", ".05"}], " ", "&&", " ", 
     RowBox[{
      RowBox[{"Abs", "[", "s1", "]"}], ">", ".05"}]}], ",", "Gray", ",", 
    "White"}], "]"}]}]], "Input",
 CellChangeTimes->{
  3.35696210375764*^9, {3.4180798820902863`*^9, 3.418079976458024*^9}, {
   3.418086320196332*^9, 3.418086375625718*^9}, {3.418086426640218*^9, 
   3.418086428508378*^9}, {3.418086470993368*^9, 3.41808651144881*^9}, {
   3.418086728619928*^9, 3.418086732296266*^9}},
 CellID->1798036416],

Cell[BoxData[
 RowBox[{
  RowBox[{"type", "[", 
   RowBox[{"n_", ",", "d_", ",", "s1_", ",", "s2_"}], "]"}], ":=", 
  RowBox[{"If", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"s2", "<", 
      RowBox[{"-", ".05"}]}], " ", "&&", " ", 
     RowBox[{"s1", ">", ".05"}]}], ",", "\"\<convex\>\"", ",", 
    RowBox[{"If", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"s2", ">", ".05"}], " ", "&&", " ", 
       RowBox[{"s1", "<", 
        RowBox[{"-", ".05"}]}]}], ",", "\"\<concave\>\"", ",", 
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"s2", "<", 
           RowBox[{"-", ".05"}]}], " ", "&&", " ", 
          RowBox[{"s1", "<", 
           RowBox[{"-", ".05"}]}]}], "||", 
         RowBox[{
          RowBox[{"s2", ">", ".05"}], " ", "&&", " ", 
          RowBox[{"s1", ">", ".05"}]}]}], ",", "\"\<convexo-concave\>\"", ",", 
        RowBox[{"If", "[", 
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"s2", "<", 
             RowBox[{"-", ".05"}]}], " ", "&&", " ", 
            RowBox[{
             RowBox[{"Abs", "[", "s1", "]"}], "<", ".05"}]}], "||", 
           RowBox[{
            RowBox[{
             RowBox[{"Abs", "[", "s2", "]"}], "<", ".05"}], " ", "&&", " ", 
            RowBox[{"s1", ">", ".05"}]}]}], ",", "\"\<plano-convex\>\"", ",", 
          
          RowBox[{"If", "[", 
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"s2", ">", ".05"}], " ", "&&", " ", 
              RowBox[{
               RowBox[{"Abs", "[", "s1", "]"}], "<", ".05"}]}], "||", 
             RowBox[{
              RowBox[{
               RowBox[{"Abs", "[", "s2", "]"}], "<", ".05"}], " ", "&&", " ", 
              
              RowBox[{"s1", "<", 
               RowBox[{"-", ".05"}]}]}]}], ",", "\"\<plano-concave\>\"", ",", 
            
            RowBox[{"If", "[", 
             RowBox[{
              RowBox[{
               RowBox[{
                RowBox[{"Abs", "[", "s2", "]"}], "<", ".05"}], " ", "&&", " ", 
               RowBox[{
                RowBox[{"Abs", "[", "s1", "]"}], "<", ".05"}]}], ",", 
              "\"\<flat\>\"", ",", "\"\<\>\""}], "]"}]}], "]"}]}], "]"}]}], 
       "]"}]}], "]"}]}], "]"}]}]], "Input",
 CellChangeTimes->{
  3.35696210375764*^9, {3.4180848908941*^9, 3.418084924689885*^9}, {
   3.418085153898864*^9, 3.418085180390108*^9}, {3.418085272795938*^9, 
   3.4180853550861*^9}, {3.4180854733657913`*^9, 3.418085498443776*^9}, {
   3.418085657276918*^9, 3.418085735865505*^9}, {3.418165454515748*^9, 
   3.418165501464137*^9}, {3.418165538367725*^9, 3.418165576618239*^9}, {
   3.418165696779222*^9, 3.41816584267439*^9}, {3.418165872787356*^9, 
   3.418165967778542*^9}, {3.418658061308446*^9, 3.418658086928513*^9}, 
   3.4186586264885893`*^9, {3.419893878953536*^9, 3.419894076732689*^9}, {
   3.41989413575697*^9, 3.419894164865429*^9}, {3.4198942026947193`*^9, 
   3.419894228768649*^9}, {3.4198942825106487`*^9, 3.419894300341861*^9}, {
   3.4211819389565687`*^9, 3.421182039354789*^9}},
 CellID->728566442],

Cell[BoxData[
 RowBox[{
  RowBox[{"R1", "[", "s1_", "]"}], ":=", 
  RowBox[{"If", "[", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"Abs", "[", "s1", "]"}], "<", ".05"}], ",", 
    "\"\<\[Infinity]\>\"", ",", 
    RowBox[{"PaddedForm", "[", 
     RowBox[{
      RowBox[{"N", "[", 
       RowBox[{"1", "/", "s1"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"3", ",", "1"}], "}"}]}], "]"}]}], "]"}]}]], "Input",
 CellChangeTimes->{
  3.35696210375764*^9, {3.418170736024839*^9, 3.4181708995573874`*^9}, {
   3.41817093307876*^9, 3.418170977728436*^9}, {3.418202706574357*^9, 
   3.418202716253804*^9}, 3.4182050023506193`*^9, {3.41865165413167*^9, 
   3.4186516611120043`*^9}, 3.418652744940783*^9, {3.418657156144565*^9, 
   3.418657261910139*^9}, {3.418659082871817*^9, 3.418659096125736*^9}, {
   3.4212292056488037`*^9, 3.42122921229354*^9}},
 CellID->652727127],

Cell[BoxData[
 RowBox[{
  RowBox[{"R2", "[", "s2_", "]"}], ":=", 
  RowBox[{"If", "[", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"Abs", "[", "s2", "]"}], "<", ".05"}], ",", 
    "\"\<\[Infinity]\>\"", ",", 
    RowBox[{"PaddedForm", "[", 
     RowBox[{
      RowBox[{"N", "[", 
       RowBox[{"1", "/", "s2"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"3", ",", "1"}], "}"}]}], "]"}]}], "]"}]}]], "Input",
 CellChangeTimes->{
  3.35696210375764*^9, {3.418170998444695*^9, 3.4181710274888887`*^9}, {
   3.418202724564427*^9, 3.4182027340800123`*^9}, 3.4182050051074944`*^9, {
   3.4186516694563026`*^9, 3.418651676745051*^9}, {3.418652751025551*^9, 
   3.418652756457204*^9}, {3.418657275045384*^9, 3.418657312997362*^9}, {
   3.4186591043347893`*^9, 3.418659112942197*^9}, {3.421229209174014*^9, 
   3.421229217013564*^9}},
 CellID->1038354613],

Cell[BoxData[
 RowBox[{
  RowBox[{"F", "[", 
   RowBox[{"n_", ",", "d_", ",", "s1_", ",", "s2_"}], "]"}], ":=", 
  RowBox[{"If", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"f", "[", 
      RowBox[{"n", ",", "d", ",", "s1", ",", "s2"}], "]"}], ">", "20"}], ",", 
    "\"\<\[Infinity]\>\"", ",", 
    RowBox[{"If", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"f", "[", 
        RowBox[{"n", ",", "d", ",", "s1", ",", "s2"}], "]"}], "<", 
       RowBox[{"-", "20"}]}], ",", "\"\<-\[Infinity]\>\"", ",", 
      RowBox[{"PaddedForm", "[", 
       RowBox[{
        RowBox[{"N", "[", 
         RowBox[{"f", "[", 
          RowBox[{"n", ",", "d", ",", "s1", ",", "s2"}], "]"}], "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"3", ",", "1"}], "}"}]}], "]"}]}], "]"}]}], "]"}]}]], "Input",\

 CellChangeTimes->{
  3.35696210375764*^9, {3.4181687508268833`*^9, 3.4181689702878857`*^9}, {
   3.418169216946224*^9, 3.4181692185177307`*^9}, {3.418169402834545*^9, 
   3.418169426009761*^9}, {3.418170174968566*^9, 3.418170180825172*^9}, {
   3.421229219759326*^9, 3.421229225854889*^9}},
 CellID->995012816],

Cell[BoxData[
 RowBox[{
  RowBox[{"caption", "[", 
   RowBox[{"n_", ",", "d_", ",", "s1_", ",", "s2_"}], "]"}], ":=", 
  RowBox[{"Style", "[", 
   RowBox[{
    RowBox[{"Row", "[", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"type", "[", 
        RowBox[{"n", ",", "d", ",", "s1", ",", "s2"}], "]"}], ",", 
       "\"\< lens: \>\"", ",", 
       RowBox[{"Subscript", "[", 
        RowBox[{
         RowBox[{"Style", "[", 
          RowBox[{"\"\<R\>\"", ",", "Italic"}], "]"}], ",", "1"}], "]"}], 
       ",", "\"\< = \>\"", " ", ",", 
       RowBox[{"R1", "[", "s1", "]"}], ",", "\"\<  \>\"", ",", 
       RowBox[{"Subscript", "[", 
        RowBox[{
         RowBox[{"Style", "[", 
          RowBox[{"\"\<R\>\"", ",", "Italic"}], "]"}], ",", "2"}], "]"}], 
       ",", "\"\< = \>\"", ",", 
       RowBox[{"R2", "[", "s2", "]"}], ",", "\"\<  \>\"", ",", 
       RowBox[{"Style", "[", 
        RowBox[{"\"\<f\>\"", ",", "Italic"}], "]"}], ",", "\"\< = \>\"", ",", 
       
       RowBox[{"F", "[", 
        RowBox[{"n", ",", "d", ",", "s1", ",", "s2"}], "]"}]}], "}"}], "]"}], 
    ",", "24", ",", "Bold", ",", "Black"}], "]"}]}]], "Input",
 CellChangeTimes->{
  3.35696210375764*^9, {3.418166017361451*^9, 3.418166317025556*^9}, {
   3.418166471660833*^9, 3.4181665106025877`*^9}, {3.418166542801285*^9, 
   3.418166559698118*^9}, {3.418166603458662*^9, 3.418166613779097*^9}, {
   3.418166735508973*^9, 3.418166736393373*^9}, {3.418166923999042*^9, 
   3.4181669351761503`*^9}, {3.418166971080418*^9, 3.418167009840201*^9}, {
   3.418167370621045*^9, 3.418167531781351*^9}, {3.418168985213716*^9, 
   3.418169047568441*^9}, 3.41817031548841*^9, 3.418170450009679*^9, {
   3.418170492517233*^9, 3.4181705141900463`*^9}, {3.418171042428417*^9, 
   3.418171064108149*^9}, 3.418171162419948*^9, {3.4181713210956173`*^9, 
   3.418171341448579*^9}, 3.418171801852685*^9, 3.418171993752228*^9, {
   3.4183917549030704`*^9, 3.418391755918696*^9}, {3.418595881296875*^9, 
   3.4185958831875*^9}, {3.41860656254593*^9, 3.418606567359833*^9}, {
   3.418770749890625*^9, 3.41877074996875*^9}, {3.41877313465625*^9, 
   3.4187731933125*^9}, {3.419892897593762*^9, 3.419892901765891*^9}},
 CellID->1145946428],

Cell[BoxData[
 RowBox[{
  RowBox[{"g", "[", 
   RowBox[{"n_", ",", "d_", ",", "s1_", ",", "s2_"}], "]"}], ":=", 
  RowBox[{"Graphics", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"color", "[", 
           RowBox[{"n", ",", "d", ",", "s1", ",", "s2"}], "]"}], ",", 
          RowBox[{"Line", "[", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"{", 
              RowBox[{"0", ",", "\[Epsilon]"}], "}"}], ",", 
             RowBox[{"{", 
              RowBox[{
               RowBox[{"f", "[", 
                RowBox[{"n", ",", "d", ",", "s1", ",", "s2"}], "]"}], ",", 
               "0"}], "}"}]}], "}"}], "]"}], ",", "Thick", ",", "Yellow", ",", 
          RowBox[{"Line", "[", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"{", 
              RowBox[{
               RowBox[{"-", "4"}], ",", "\[Epsilon]"}], "}"}], ",", 
             RowBox[{"{", 
              RowBox[{"0", ",", "\[Epsilon]"}], "}"}], ",", 
             RowBox[{"{", 
              RowBox[{"4", ",", 
               RowBox[{
                FractionBox[
                 RowBox[{
                  RowBox[{"f", "[", 
                   RowBox[{"n", ",", "d", ",", "s1", ",", "s2"}], "]"}], "-", 
                  "4"}], 
                 RowBox[{"f", "[", 
                  RowBox[{"n", ",", "d", ",", "s1", ",", "s2"}], "]"}]], 
                "\[Epsilon]"}]}], "}"}]}], "}"}], "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"\[Epsilon]", ",", 
          RowBox[{"-", ".6"}], ",", ".6", ",", ".1"}], "}"}]}], "]"}], ",", 
      RowBox[{"Thickness", "[", ".008", "]"}], ",", 
      RowBox[{"Lighter", "[", "Gray", "]"}], ",", 
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"Line", "[", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{
             FractionBox[
              RowBox[{"\[IndentingNewLine]", 
               RowBox[{"2", "-", 
                RowBox[{"d", " ", "s1"}], "-", 
                RowBox[{"2", " ", 
                 SqrtBox[
                  RowBox[{"1", "-", 
                   RowBox[{
                    SuperscriptBox["s1", "2"], " ", 
                    SuperscriptBox["\[Epsilon]", "2"]}]}]]}]}]}], 
              RowBox[{"2", " ", "s1"}]], ",", "\[Epsilon]"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{
             FractionBox[
              RowBox[{"2", "+", 
               RowBox[{"d", " ", "s2"}], "-", 
               RowBox[{"2", " ", 
                SqrtBox[
                 RowBox[{"1", "-", 
                  RowBox[{
                   SuperscriptBox["s2", "2"], " ", 
                   SuperscriptBox["\[Epsilon]", "2"]}]}]]}]}], 
              RowBox[{"2", " ", "s2"}]], ",", "\[Epsilon]"}], "}"}]}], "}"}], 
         "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"\[Epsilon]", ",", 
          RowBox[{"-", "1"}], ",", "1", ",", ".05"}], "}"}]}], "]"}]}], "}"}],
     ",", 
    RowBox[{"PlotRange", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "4"}], ",", "4"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1.5"}], ",", "1.5"}], "}"}]}], "}"}]}], ",", 
    RowBox[{"ImageSize", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"600", ",", "300"}], "}"}]}], ",", 
    RowBox[{"Epilog", "\[Rule]", 
     RowBox[{"Inset", "[", 
      RowBox[{
       RowBox[{"caption", "[", 
        RowBox[{"n", ",", "d", ",", "s1", ",", "s2"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"0", ",", 
         RowBox[{"-", "1.35"}]}], "}"}]}], "]"}]}]}], "]"}]}]], "Input",
 CellChangeTimes->{
  3.35696210375764*^9, {3.418075249305657*^9, 3.4180753452445517`*^9}, {
   3.418075405715784*^9, 3.418075417826034*^9}, {3.4180757791557693`*^9, 
   3.4180757930490923`*^9}, {3.418075893119368*^9, 3.418075945954172*^9}, {
   3.418076063337467*^9, 3.4180760640276737`*^9}, {3.4180761070597982`*^9, 
   3.418076203450293*^9}, {3.4180762468020563`*^9, 3.418076276707671*^9}, {
   3.418076655960197*^9, 3.418076851484838*^9}, {3.4180768860084457`*^9, 
   3.418076893340652*^9}, 3.418076937907786*^9, {3.418076986093257*^9, 
   3.418077032834291*^9}, {3.4180771150396013`*^9, 3.418077167885743*^9}, {
   3.4180773121016407`*^9, 3.418077334617972*^9}, 3.418077427266523*^9, {
   3.418077606316411*^9, 3.41807763560078*^9}, {3.418077774193081*^9, 
   3.418077777700556*^9}, {3.418077891584276*^9, 3.418077919323176*^9}, {
   3.418078558780699*^9, 3.41807858455376*^9}, {3.418078626142837*^9, 
   3.4180786670320387`*^9}, 3.418078708074594*^9, 3.418078743751081*^9, 
   3.418078803828834*^9, {3.418078864837406*^9, 3.4180788746832867`*^9}, 
   3.418078939406727*^9, {3.4180790116818457`*^9, 3.418079017116849*^9}, {
   3.418079079037241*^9, 3.418079083315537*^9}, {3.41807912086064*^9, 
   3.418079128312476*^9}, 3.418079185698028*^9, {3.418079769938895*^9, 
   3.418079773158338*^9}, {3.4180798090453243`*^9, 3.418079831208728*^9}, {
   3.418080587551588*^9, 3.4180806083264227`*^9}, {3.418080689239768*^9, 
   3.41808069492139*^9}, {3.418080747882346*^9, 3.418080752848077*^9}, {
   3.418080852313641*^9, 3.4180808608528833`*^9}, {3.418084680483959*^9, 
   3.4180846841039057`*^9}, 3.418086554958386*^9, {3.418086596283731*^9, 
   3.418086657494194*^9}, 3.418159255050995*^9, {3.418159362257082*^9, 
   3.4181593644288816`*^9}, {3.418159466874556*^9, 3.41815947740909*^9}, 
   3.4181595974817553`*^9, {3.418160867111857*^9, 3.4181608904905577`*^9}, 
   3.418161240820723*^9, {3.418161367920556*^9, 3.418161388883923*^9}, 
   3.41816143203128*^9, 3.41816295409356*^9, {3.418163015038414*^9, 
   3.418163058717478*^9}, 3.418163154436796*^9, {3.4181632747899113`*^9, 
   3.418163300438483*^9}, 3.4181633989342527`*^9, {3.418163735200264*^9, 
   3.418163740830267*^9}, {3.41816384690655*^9, 3.418163854175425*^9}, 
   3.418163976575383*^9, {3.418164020348226*^9, 3.418164028046363*^9}, {
   3.41816407720228*^9, 3.418164082077252*^9}, 3.4181641748050947`*^9, {
   3.418164217474227*^9, 3.4181642237011833`*^9}, {3.418164278681508*^9, 
   3.418164284446534*^9}, {3.418164327044635*^9, 3.418164357430141*^9}, {
   3.418164464178502*^9, 3.418164494395433*^9}, {3.418164542588826*^9, 
   3.418164546750703*^9}, {3.418164580472588*^9, 3.418164605938572*^9}, {
   3.4181646553573713`*^9, 3.418164669648744*^9}, 3.418164703688821*^9, 
   3.418164744063737*^9, 3.418164835104268*^9, 3.418164901999707*^9, {
   3.418171189272768*^9, 3.4181712481538267`*^9}, {3.418171483441313*^9, 
   3.418171555044299*^9}, {3.418171592797895*^9, 3.4181716572834797`*^9}, {
   3.418171704760054*^9, 3.418171717074648*^9}, {3.4181717637373753`*^9, 
   3.4181717724263687`*^9}, {3.41817184023002*^9, 3.4181718577156897`*^9}, {
   3.418171896673716*^9, 3.418171924456727*^9}, 3.4181721046084967`*^9, {
   3.418202811374279*^9, 3.4182028126649723`*^9}, {3.4183923664105873`*^9, 
   3.418392373895817*^9}, {3.4183924763479424`*^9, 3.41839247713974*^9}, {
   3.4183926666968718`*^9, 3.418392672783039*^9}, {3.418595821265625*^9, 
   3.4185958215625*^9}, {3.418595866515625*^9, 3.41859587309375*^9}, {
   3.418606383873624*^9, 3.4186064357931623`*^9}, {3.41877075259375*^9, 
   3.418770753921875*^9}, {3.419892344641584*^9, 3.4198923488584423`*^9}, {
   3.419892580561172*^9, 3.419892593628316*^9}, {3.4198926432785*^9, 
   3.419892645164299*^9}, {3.419892696679639*^9, 3.419892847794042*^9}, 
   3.4198929513754377`*^9, 3.421182439155263*^9, 3.4211824931688557`*^9, {
   3.421182750417871*^9, 3.42118276230342*^9}, 3.421187445302856*^9, 
   3.421187760340027*^9, {3.4211879571386623`*^9, 3.421187965745809*^9}, {
   3.421188067209043*^9, 3.421188081391617*^9}, {3.4211881633772907`*^9, 
   3.4211881796661043`*^9}, 3.421188262176498*^9, 3.421188315630548*^9, {
   3.4211883689098663`*^9, 3.421188420783215*^9}, 3.4211884635286818`*^9, 
   3.421188521169799*^9, 3.4211885731436663`*^9, {3.4211886084480124`*^9, 
   3.421188620944539*^9}, 3.421188652954227*^9, {3.4211887450642653`*^9, 
   3.421188772274876*^9}, 3.421188814840152*^9, 3.421188870937858*^9, 
   3.421188975534292*^9, 3.421189017009808*^9, 3.421189109521081*^9, {
   3.421189159369404*^9, 3.4211891889848022`*^9}, 3.421189226008747*^9, {
   3.421189277008815*^9, 3.421189278485111*^9}, {3.4211893331528797`*^9, 
   3.42118936100884*^9}, 3.421189391080886*^9, {3.421189441745386*^9, 
   3.4211894650729637`*^9}, {3.421229494231509*^9, 3.421229517811763*^9}, {
   3.4212296519417467`*^9, 3.42122968477987*^9}},
 CellID->1837916761],

Cell[BoxData[
 RowBox[{"Manipulate", "[", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"g", "[", 
    RowBox[{"n", ",", "d", ",", "s1", ",", "s2"}], "]"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"n", ",", "1.65", ",", "\"\<refractive index n\>\""}], "}"}], 
     ",", "1.0008", ",", "2.5", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"d", ",", ".5", ",", "\"\<thickness of lens d\>\""}], "}"}], 
     ",", ".45", ",", ".7", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"s1", ",", ".3", ",", 
       RowBox[{"Row", "[", 
        RowBox[{"{", 
         RowBox[{"\"\<curvature \>\"", ",", 
          RowBox[{"1", "/", 
           RowBox[{"Subscript", "[", 
            RowBox[{"R", ",", "1"}], "]"}]}]}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"-", ".3999"}], ",", ".4001", ",", ".001", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"s2", ",", 
       RowBox[{"-", ".3"}], ",", 
       RowBox[{"Row", "[", 
        RowBox[{"{", 
         RowBox[{"\"\<curvature \>\"", ",", 
          RowBox[{"1", "/", 
           RowBox[{"Subscript", "[", 
            RowBox[{"R", ",", "2"}], "]"}]}]}], "}"}], "]"}]}], "}"}], ",", 
     RowBox[{"-", ".3999"}], ",", ".4001", ",", ".001", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}]}], "}"}], ",", 
   RowBox[{"SaveDefinitions", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellChangeTimes->{
  3.35696210375764*^9, 3.418767664297329*^9, {3.4198950595*^9, 
   3.419895060984375*^9}, {3.41989516209375*^9, 3.419895163375*^9}, {
   3.4198955845625*^9, 3.419895585015625*^9}, {3.419895860421875*^9, 
   3.4198958624375*^9}, {3.420234421933832*^9, 3.4202344484450808`*^9}, 
   3.420235455736022*^9, {3.421182690499831*^9, 3.4211826942739077`*^9}, 
   3.4214247870780272`*^9},
 CellID->1009555335]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`d$$ = 0.5, $CellContext`n$$ = 
    1.65, $CellContext`s1$$ = 0.3, $CellContext`s2$$ = -0.3, Typeset`show$$ = 
    True, Typeset`bookmarkList$$ = {}, Typeset`bookmarkMode$$ = "Menu", 
    Typeset`animator$$, Typeset`animvar$$ = 1, Typeset`name$$ = 
    "\"untitled\"", Typeset`specs$$ = {{{
       Hold[$CellContext`n$$], 1.65, "refractive index n"}, 1.0008, 2.5}, {{
       Hold[$CellContext`d$$], 0.5, "thickness of lens d"}, 0.45, 0.7}, {{
       Hold[$CellContext`s1$$], 0.3, 
       Row[{"curvature ", Subscript[$CellContext`R, 1]^(-1)}]}, -0.3999, 
      0.4001, 0.001}, {{
       Hold[$CellContext`s2$$], -0.3, 
       Row[{"curvature ", Subscript[$CellContext`R, 2]^(-1)}]}, -0.3999, 
      0.4001, 0.001}}, Typeset`size$$ = {600., {148., 152.}}, 
    Typeset`update$$ = 0, Typeset`initDone$$, Typeset`skipInitDone$$ = 
    False, $CellContext`n$1162$$ = 0, $CellContext`d$1163$$ = 
    0, $CellContext`s1$1164$$ = 0, $CellContext`s2$1165$$ = 0}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, 
      "Variables" :> {$CellContext`d$$ = 0.5, $CellContext`n$$ = 
        1.65, $CellContext`s1$$ = 0.3, $CellContext`s2$$ = -0.3}, 
      "ControllerVariables" :> {
        Hold[$CellContext`n$$, $CellContext`n$1162$$, 0], 
        Hold[$CellContext`d$$, $CellContext`d$1163$$, 0], 
        Hold[$CellContext`s1$$, $CellContext`s1$1164$$, 0], 
        Hold[$CellContext`s2$$, $CellContext`s2$1165$$, 0]}, 
      "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, 
      "Body" :> $CellContext`g[$CellContext`n$$, $CellContext`d$$, \
$CellContext`s1$$, $CellContext`s2$$], 
      "Specifications" :> {{{$CellContext`n$$, 1.65, "refractive index n"}, 
         1.0008, 2.5, Appearance -> 
         "Labeled"}, {{$CellContext`d$$, 0.5, "thickness of lens d"}, 0.45, 
         0.7, Appearance -> "Labeled"}, {{$CellContext`s1$$, 0.3, 
          Row[{"curvature ", Subscript[$CellContext`R, 1]^(-1)}]}, -0.3999, 
         0.4001, 0.001, Appearance -> "Labeled"}, {{$CellContext`s2$$, -0.3, 
          Row[{"curvature ", Subscript[$CellContext`R, 2]^(-1)}]}, -0.3999, 
         0.4001, 0.001, Appearance -> "Labeled"}}, "Options" :> {}, 
      "DefaultOptions" :> {ControllerLinking -> True}],
     ImageSizeCache->{645., {237., 242.}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    Initialization:>(({$CellContext`g[
          Pattern[$CellContext`n, 
           Blank[]], 
          Pattern[$CellContext`d, 
           Blank[]], 
          Pattern[$CellContext`s1, 
           Blank[]], 
          Pattern[$CellContext`s2, 
           Blank[]]] := Graphics[{
           Table[{
             $CellContext`color[$CellContext`n, $CellContext`d, \
$CellContext`s1, $CellContext`s2], 
             Line[{{0, $CellContext`\[Epsilon]}, {
                $CellContext`f[$CellContext`n, $CellContext`d, \
$CellContext`s1, $CellContext`s2], 0}}], Thick, Yellow, 
             
             Line[{{-4, $CellContext`\[Epsilon]}, {
               0, $CellContext`\[Epsilon]}, {
               4, (($CellContext`f[$CellContext`n, $CellContext`d, \
$CellContext`s1, $CellContext`s2] - 
                  4)/$CellContext`f[$CellContext`n, $CellContext`d, \
$CellContext`s1, $CellContext`s2]) $CellContext`\[Epsilon]}}]}, \
{$CellContext`\[Epsilon], -0.6, 0.6, 0.1}], 
           Thickness[0.008], 
           Lighter[Gray], 
           Table[
            
            Line[{{(2 - $CellContext`d $CellContext`s1 - 2 
                Sqrt[1 - $CellContext`s1^2 $CellContext`\[Epsilon]^2])/(
               2 $CellContext`s1), $CellContext`\[Epsilon]}, {(
                2 + $CellContext`d $CellContext`s2 - 2 
                Sqrt[1 - $CellContext`s2^2 $CellContext`\[Epsilon]^2])/(
               2 $CellContext`s2), $CellContext`\[Epsilon]}}], {$CellContext`\
\[Epsilon], -1, 1, 0.05}]}, PlotRange -> {{-4, 4}, {-1.5, 1.5}}, 
          ImageSize -> {600, 300}, Epilog -> Inset[
            $CellContext`caption[$CellContext`n, $CellContext`d, \
$CellContext`s1, $CellContext`s2], {0, -1.35}]], $CellContext`color[
          Pattern[$CellContext`n, 
           Blank[]], 
          Pattern[$CellContext`d, 
           Blank[]], 
          Pattern[$CellContext`s1, 
           Blank[]], 
          Pattern[$CellContext`s2, 
           Blank[]]] := If[
          
          And[$CellContext`f[$CellContext`n, $CellContext`d, $CellContext`s1, \
$CellContext`s2] < 0, Abs[$CellContext`s1] > 0.05, Abs[$CellContext`s1] > 
           0.05], Gray, White], $CellContext`f[
          Pattern[$CellContext`n, 
           Blank[]], 
          Pattern[$CellContext`d, 
           Blank[]], 
          Pattern[$CellContext`s1, 
           Blank[]], 
          Pattern[$CellContext`s2, 
           Blank[]]] := 
        1/(($CellContext`n - 
          1) ($CellContext`s1 - $CellContext`s2 + (($CellContext`n - 
             1)/$CellContext`n) $CellContext`d $CellContext`s1 \
$CellContext`s2)), 
        Attributes[PlotRange] = {ReadProtected}, $CellContext`caption[
          Pattern[$CellContext`n, 
           Blank[]], 
          Pattern[$CellContext`d, 
           Blank[]], 
          Pattern[$CellContext`s1, 
           Blank[]], 
          Pattern[$CellContext`s2, 
           Blank[]]] := Style[
          Row[{
            $CellContext`type[$CellContext`n, $CellContext`d, \
$CellContext`s1, $CellContext`s2], " lens: ", 
            Subscript[
             Style["R", Italic], 1], " = ", 
            $CellContext`R1[$CellContext`s1], "  ", 
            Subscript[
             Style["R", Italic], 2], " = ", 
            $CellContext`R2[$CellContext`s2], "  ", 
            Style["f", Italic], " = ", 
            $CellContext`F[$CellContext`n, $CellContext`d, $CellContext`s1, \
$CellContext`s2]}], 24, Bold, Black], $CellContext`type[
          Pattern[$CellContext`n, 
           Blank[]], 
          Pattern[$CellContext`d, 
           Blank[]], 
          Pattern[$CellContext`s1, 
           Blank[]], 
          Pattern[$CellContext`s2, 
           Blank[]]] := If[
          And[$CellContext`s2 < -0.05, $CellContext`s1 > 0.05], "convex", 
          If[
           And[$CellContext`s2 > 0.05, $CellContext`s1 < -0.05], "concave", 
           If[
            Or[
             And[$CellContext`s2 < -0.05, $CellContext`s1 < -0.05], 
             And[$CellContext`s2 > 0.05, $CellContext`s1 > 0.05]], 
            "convexo-concave", 
            If[
             Or[
              And[$CellContext`s2 < -0.05, Abs[$CellContext`s1] < 0.05], 
              And[Abs[$CellContext`s2] < 0.05, $CellContext`s1 > 0.05]], 
             "plano-convex", 
             If[
              Or[
               And[$CellContext`s2 > 0.05, Abs[$CellContext`s1] < 0.05], 
               And[Abs[$CellContext`s2] < 0.05, $CellContext`s1 < -0.05]], 
              "plano-concave", 
              If[
               And[Abs[$CellContext`s2] < 0.05, Abs[$CellContext`s1] < 0.05], 
               "flat", ""]]]]]], Attributes[Subscript] = {NHoldRest}, 
        Subscript[$CellContext`\[Lambda], $CellContext`e] = 0, 
        Subscript[$CellContext`\[Lambda], $CellContext`i] = 
        0.04363323129985824, 
        Subscript[$CellContext`\[Lambda], $CellContext`max] = 15 Degree, 
        Subscript[$CellContext`\[Lambda], $CellContext`min] = (-15) Degree, 
        Subscript[$CellContext`\[Lambda], $CellContext`e] = 0, 
        Subscript[$CellContext`\[Lambda], $CellContext`i] = 
        0.04363323129985824, 
        Subscript[$CellContext`\[Lambda], $CellContext`max] = 15 Degree, 
        Subscript[$CellContext`\[Lambda], $CellContext`min] = (-15) Degree, 
        Subscript[$CellContext`\[Lambda], $CellContext`e] = 0, 
        Subscript[$CellContext`\[Lambda], $CellContext`i] = 
        0.04363323129985824, 
        Subscript[$CellContext`\[Lambda], $CellContext`max] = 15 Degree, 
        Subscript[$CellContext`\[Lambda], $CellContext`min] = (-15) Degree, 
        Subscript[$CellContext`\[Lambda], $CellContext`e] = 0, 
        Subscript[$CellContext`\[Lambda], $CellContext`i] = 
        0.04363323129985824, 
        Subscript[$CellContext`\[Lambda], $CellContext`max] = 15 Degree, 
        Subscript[$CellContext`\[Lambda], $CellContext`min] = (-15) Degree, 
        Subscript[$CellContext`\[Lambda], $CellContext`e] = 0, 
        Subscript[$CellContext`\[Lambda], $CellContext`i] = 
        0.04363323129985824, 
        Subscript[$CellContext`\[Lambda], $CellContext`max] = 15 Degree, 
        Subscript[$CellContext`\[Lambda], $CellContext`min] = (-15) Degree, 
        Subscript[$CellContext`\[Lambda], $CellContext`e] = 0, 
        Subscript[$CellContext`\[Lambda], $CellContext`i] = 
        0.04363323129985824, 
        Subscript[$CellContext`\[Lambda], $CellContext`max] = 15 Degree, 
        Subscript[$CellContext`\[Lambda], $CellContext`min] = (-15) Degree, 
        Subscript[$CellContext`\[Lambda], $CellContext`e] = 0, 
        Subscript[$CellContext`\[Lambda], $CellContext`i] = 
        0.04363323129985824, 
        Subscript[$CellContext`\[Lambda], $CellContext`max] = 15 Degree, 
        Subscript[$CellContext`\[Lambda], $CellContext`min] = (-15) Degree, 
        Subscript[$CellContext`\[Phi], 1] = (-10) Degree, 
        Subscript[$CellContext`\[Phi], 2] = (-20) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`i] = (-25) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`max] = 5 Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`min] = (-35) Degree, 
        Subscript[$CellContext`\[Phi], 1] = (-10) Degree, 
        Subscript[$CellContext`\[Phi], 2] = (-20) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`i] = (-25) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`max] = 5 Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`min] = (-35) Degree, 
        Subscript[$CellContext`\[Phi], 1] = (-10) Degree, 
        Subscript[$CellContext`\[Phi], 2] = (-20) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`i] = (-25) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`max] = 5 Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`min] = (-35) Degree, 
        Subscript[$CellContext`\[Phi], 1] = (-10) Degree, 
        Subscript[$CellContext`\[Phi], 2] = (-20) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`i] = (-25) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`max] = 5 Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`min] = (-35) Degree, 
        Subscript[$CellContext`\[Phi], 1] = (-10) Degree, 
        Subscript[$CellContext`\[Phi], 2] = (-20) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`i] = (-25) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`max] = 5 Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`min] = (-35) Degree, 
        Subscript[$CellContext`\[Phi], 1] = (-10) Degree, 
        Subscript[$CellContext`\[Phi], 2] = (-20) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`i] = (-25) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`max] = 5 Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`min] = (-35) Degree, 
        Subscript[$CellContext`\[Phi], 1] = (-10) Degree, 
        Subscript[$CellContext`\[Phi], 2] = (-20) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`i] = (-25) Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`max] = 5 Degree, 
        Subscript[$CellContext`\[Phi], $CellContext`min] = (-35) 
         Degree, $CellContext`R1[
          Pattern[$CellContext`s1, 
           Blank[]]] := If[Abs[$CellContext`s1] < 0.05, "\[Infinity]", 
          PaddedForm[
           N[1/$CellContext`s1], {3, 1}]], $CellContext`R2[
          Pattern[$CellContext`s2, 
           Blank[]]] := If[Abs[$CellContext`s2] < 0.05, "\[Infinity]", 
          PaddedForm[
           N[1/$CellContext`s2], {3, 1}]], $CellContext`F[
          Pattern[$CellContext`n, 
           Blank[]], 
          Pattern[$CellContext`d, 
           Blank[]], 
          Pattern[$CellContext`s1, 
           Blank[]], 
          Pattern[$CellContext`s2, 
           Blank[]]] := 
        If[$CellContext`f[$CellContext`n, $CellContext`d, $CellContext`s1, \
$CellContext`s2] > 20, "\[Infinity]", 
          
          If[$CellContext`f[$CellContext`n, $CellContext`d, $CellContext`s1, \
$CellContext`s2] < -20, "-\[Infinity]", 
           PaddedForm[
            N[
             $CellContext`f[$CellContext`n, $CellContext`d, $CellContext`s1, \
$CellContext`s2]], {3, 1}]]]}; Typeset`initDone$$ = True); ReleaseHold[
       HoldComplete[{$CellContext`f[
           Pattern[$CellContext`n, 
            Blank[]], 
           Pattern[$CellContext`d, 
            Blank[]], 
           Pattern[$CellContext`s1, 
            Blank[]], 
           Pattern[$CellContext`s2, 
            Blank[]]] := 
         1/(($CellContext`n - 
           1) ($CellContext`s1 - $CellContext`s2 + (($CellContext`n - 
              1)/$CellContext`n) $CellContext`d $CellContext`s1 \
$CellContext`s2)), $CellContext`color[
           Pattern[$CellContext`n, 
            Blank[]], 
           Pattern[$CellContext`d, 
            Blank[]], 
           Pattern[$CellContext`s1, 
            Blank[]], 
           Pattern[$CellContext`s2, 
            Blank[]]] := If[
           
           And[$CellContext`f[$CellContext`n, $CellContext`d, \
$CellContext`s1, $CellContext`s2] < 0, Abs[$CellContext`s1] > 0.05, 
            Abs[$CellContext`s1] > 0.05], Gray, White], $CellContext`type[
           Pattern[$CellContext`n, 
            Blank[]], 
           Pattern[$CellContext`d, 
            Blank[]], 
           Pattern[$CellContext`s1, 
            Blank[]], 
           Pattern[$CellContext`s2, 
            Blank[]]] := If[
           And[$CellContext`s2 < -0.05, $CellContext`s1 > 0.05], "convex", 
           If[
            And[$CellContext`s2 > 0.05, $CellContext`s1 < -0.05], "concave", 
            If[
             Or[
              And[$CellContext`s2 < -0.05, $CellContext`s1 < -0.05], 
              And[$CellContext`s2 > 0.05, $CellContext`s1 > 0.05]], 
             "convexo-concave", 
             If[
              Or[
               And[$CellContext`s2 < -0.05, Abs[$CellContext`s1] < 0.05], 
               And[Abs[$CellContext`s2] < 0.05, $CellContext`s1 > 0.05]], 
              "plano-convex", 
              If[
               Or[
                And[$CellContext`s2 > 0.05, Abs[$CellContext`s1] < 0.05], 
                And[Abs[$CellContext`s2] < 0.05, $CellContext`s1 < -0.05]], 
               "plano-concave", 
               If[
                And[Abs[$CellContext`s2] < 0.05, Abs[$CellContext`s1] < 0.05],
                 "flat", ""]]]]]], $CellContext`R1[
           Pattern[$CellContext`s1, 
            Blank[]]] := If[Abs[$CellContext`s1] < 0.05, "\[Infinity]", 
           PaddedForm[
            N[1/$CellContext`s1], {3, 1}]], $CellContext`R2[
           Pattern[$CellContext`s2, 
            Blank[]]] := If[Abs[$CellContext`s2] < 0.05, "\[Infinity]", 
           PaddedForm[
            N[1/$CellContext`s2], {3, 1}]], $CellContext`F[
           Pattern[$CellContext`n, 
            Blank[]], 
           Pattern[$CellContext`d, 
            Blank[]], 
           Pattern[$CellContext`s1, 
            Blank[]], 
           Pattern[$CellContext`s2, 
            Blank[]]] := 
         If[$CellContext`f[$CellContext`n, $CellContext`d, $CellContext`s1, \
$CellContext`s2] > 20, "\[Infinity]", 
           
           If[$CellContext`f[$CellContext`n, $CellContext`d, $CellContext`s1, \
$CellContext`s2] < -20, "-\[Infinity]", 
            PaddedForm[
             N[
              $CellContext`f[$CellContext`n, $CellContext`d, $CellContext`s1, \
$CellContext`s2]], {3, 1}]]], $CellContext`caption[
           Pattern[$CellContext`n, 
            Blank[]], 
           Pattern[$CellContext`d, 
            Blank[]], 
           Pattern[$CellContext`s1, 
            Blank[]], 
           Pattern[$CellContext`s2, 
            Blank[]]] := Style[
           Row[{
             $CellContext`type[$CellContext`n, $CellContext`d, \
$CellContext`s1, $CellContext`s2], " lens: ", 
             Subscript[
              Style["R", Italic], 1], " = ", 
             $CellContext`R1[$CellContext`s1], "  ", 
             Subscript[
              Style["R", Italic], 2], " = ", 
             $CellContext`R2[$CellContext`s2], "  ", 
             Style["f", Italic], " = ", 
             $CellContext`F[$CellContext`n, $CellContext`d, $CellContext`s1, \
$CellContext`s2]}], 24, Bold, Black], $CellContext`g[
           Pattern[$CellContext`n, 
            Blank[]], 
           Pattern[$CellContext`d, 
            Blank[]], 
           Pattern[$CellContext`s1, 
            Blank[]], 
           Pattern[$CellContext`s2, 
            Blank[]]] := Graphics[{
            Table[{
              $CellContext`color[$CellContext`n, $CellContext`d, \
$CellContext`s1, $CellContext`s2], 
              Line[{{0, $CellContext`\[Epsilon]}, {
                 $CellContext`f[$CellContext`n, $CellContext`d, \
$CellContext`s1, $CellContext`s2], 0}}], Thick, Yellow, 
              
              Line[{{-4, $CellContext`\[Epsilon]}, {
                0, $CellContext`\[Epsilon]}, {
                4, (($CellContext`f[$CellContext`n, $CellContext`d, \
$CellContext`s1, $CellContext`s2] - 
                   4)/$CellContext`f[$CellContext`n, $CellContext`d, \
$CellContext`s1, $CellContext`s2]) $CellContext`\[Epsilon]}}]}, \
{$CellContext`\[Epsilon], -0.6, 0.6, 0.1}], 
            Thickness[0.008], 
            Lighter[Gray], 
            Table[
             
             Line[{{(2 - $CellContext`d $CellContext`s1 - 
                 2 (1 - $CellContext`s1^2 $CellContext`\[Epsilon]^2)^
                  Rational[1, 2])/(
                2 $CellContext`s1), $CellContext`\[Epsilon]}, {(
                 2 + $CellContext`d $CellContext`s2 - 
                 2 (1 - $CellContext`s2^2 $CellContext`\[Epsilon]^2)^
                  Rational[1, 2])/(
                2 $CellContext`s2), $CellContext`\[Epsilon]}}], \
{$CellContext`\[Epsilon], -1, 1, 0.05}]}, PlotRange -> {{-4, 4}, {-1.5, 1.5}},
            ImageSize -> {600, 300}, Epilog -> Inset[
             $CellContext`caption[$CellContext`n, $CellContext`d, \
$CellContext`s1, $CellContext`s2], {0, -1.35}]]}]]; Typeset`initDone$$ = True),
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 CellID->109723962],

Cell[CellGroupData[{

Cell["CAPTION", "Section",
 CellFrame->{{0, 0}, {1, 0}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->12,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[TextData[{
 "The lensmaker's equation relates the focal length of a simple lens with the \
spherical curvature of its two faces: ",
 "\n",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox["1", "f"], "=", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"n", "-", "1"}], ")"}], 
     RowBox[{"(", 
      RowBox[{
       FractionBox["1", 
        SubscriptBox["R", "1"]], "-", 
       FractionBox["1", 
        SubscriptBox["R", "2"]], "+", 
       FractionBox[
        RowBox[{
         RowBox[{"(", 
          RowBox[{"n", "-", "1"}], ")"}], "d"}], 
        RowBox[{"n", " ", 
         SubscriptBox["R", "1"], 
         SubscriptBox["R", "2"]}]]}], ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 ",\nwhere ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["R", "1"], TraditionalForm]], "InlineMath"],
 " and",
 Cell[BoxData[
  FormBox[" ", TraditionalForm]], "InlineMath"],
 Cell[BoxData[
  FormBox[
   SubscriptBox["R", "2"], TraditionalForm]], "InlineMath"],
 " represent the radii of curvature of the lens surfaces closest to the light \
source (on the left) and the object (on the right). The sign of ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["R", "i"], TraditionalForm]], "InlineMath"],
 " is determined by the location of the center of curvature along the optic \
axis, with the origin at the center of the lens.",
 " ",
 "Thus for a doubly convex lens, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["R", "1"], TraditionalForm]], "InlineMath"],
 " is positive while ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["R", "2"], TraditionalForm]], "InlineMath"],
 " is negative. "
}], "Text"]
}, Close]]
}, Open  ]],

Cell[TextData[{
 "The focal length ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " is positive for a converging lens but negative for a diverging lens, ",
 StyleBox["giving a virtual focus, indicated by a cone of gray rays. ",
  FontSize->12]
}], "Text"],

Cell[TextData[{
 "The lens index of refraction is given by ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 ". Optical-quality glass has ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " in the vicinity of 2.65. The top slider enables you to vary ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " between 1.0008, its value for air, and 3.42, the refractive index of \
diamond. "
}], "Text"],

Cell[TextData[{
 "The width ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " represents the distance between the faces of the lens along the optical \
axis. The value of ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["R", "2"], TraditionalForm]], "InlineMath"],
 " is restrained by the slider so that the lens faces never intersect \
anywhere. "
}], "Text"],

Cell[TextData[{
 "The parameters ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["R", "1"], ",", " ", 
    SubscriptBox["R", "2"]}], TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " are to be expressed in the same length units, often cm. The reciprocal ",
 Cell[BoxData[
  FormBox[
   RowBox[{"1", "/", "f"}], TraditionalForm]], "InlineMath"],
 " is known as the optical power of the lens, expressed in diopters ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    SuperscriptBox["m", 
     RowBox[{"-", "1"}]], ")"}], TraditionalForm]], "InlineMath"],
 ". A converging lens, as shown in the thumbnail, can serve as a simple \
magnifying glass."
}], "Text"],

Cell[TextData[{
 "In the thin-lens approximation, the lens width ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " is small compared to the other lengths and the lensmaker's equation can be \
simplified to ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox["1", "f"], "=", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"n", "-", "1"}], ")"}], 
     RowBox[{"(", 
      RowBox[{
       FractionBox["1", 
        SubscriptBox["R", "1"]], "-", 
       FractionBox["1", 
        SubscriptBox["R", "2"]]}], ")"}]}]}], TraditionalForm]], "InlineMath"],
 ". "
}], "Text"],

Cell[CellGroupData[{

Cell["THIS NOTEBOOK IS THE SOURCE CODE FROM", "Text",
 CellFrame->{{0, 0}, {0, 0}},
 CellMargins->{{48, 10}, {4, 28}},
 CellGroupingRules->{"SectionGrouping", 25},
 CellFrameMargins->{{48, 48}, {6, 5}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->10,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[TextData[{
 "\"",
 ButtonBox["Lensmaker's Equation",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/LensmakersEquation/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/LensmakersEquation/"],
 "\"",
 " from ",
 ButtonBox["the Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
 "\[ParagraphSeparator]\[NonBreakingSpace]",
 ButtonBox["http://demonstrations.wolfram.com/LensmakersEquation/",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/LensmakersEquation/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/LensmakersEquation/"]
}], "Text",
 CellMargins->{{48, Inherited}, {0, Inherited}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.5]],

Cell[TextData[{
 "Contributed by: ",
 ButtonBox["S. M. Blinder",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/author.html?author=S.+M.+Blinder"],
     None},
  ButtonNote->
   "http://demonstrations.wolfram.com/author.html?author=S.+M.+Blinder"]
}], "Text",
 CellDingbat->"\[FilledSmallSquare]",
 CellMargins->{{66, 48}, {2, 4}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.6],
 CellID->418556502],

Cell[CellGroupData[{

Cell[TextData[{
 "A full-function Wolfram ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " system (Version 6 or higher) is required to edit this notebook.\n",
 StyleBox[ButtonBox["GET WOLFRAM MATHEMATICA \[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolfram.com/products/mathematica/"], None},
  ButtonNote->"http://www.wolfram.com/products/mathematica/"],
  FontFamily->"Helvetica",
  FontWeight->"Bold",
  FontSlant->"Italic",
  FontColor->RGBColor[1, 0.42, 0]]
}], "Text",
 CellFrame->True,
 CellMargins->{{48, 68}, {8, 28}},
 CellFrameMargins->12,
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 CellChangeTimes->{3.3750111182355957`*^9},
 ParagraphSpacing->{1., 1.},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.411765],
 Background->RGBColor[1, 1, 1]],

Cell[TextData[{
 "\[Copyright] ",
 StyleBox[ButtonBox["Wolfram Demonstrations Project & Contributors",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Terms of Use",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/termsofuse.html"], None},
  ButtonNote->"http://demonstrations.wolfram.com/termsofuse.html"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Make a new version of this Demonstration \
\[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/participate/upload.jsp?id=\
LensmakersEquation"], None},
  ButtonNote->None],
  FontColor->GrayLevel[0.6]]
}], "Text",
 CellFrame->{{0, 0}, {0, 0.5}},
 CellMargins->{{48, 10}, {20, 50}},
 CellFrameMargins->{{6, 0}, {6, 6}},
 CellFrameColor->GrayLevel[0.6],
 FontFamily->"Verdana",
 FontSize->9,
 FontColor->GrayLevel[0.6]]
}, Open  ]]
}, Open  ]]
},
Editable->True,
Saveable->False,
ScreenStyleEnvironment->"Working",
CellInsertionPointCell->None,
WindowSize->{780, 650},
WindowMargins->{{Inherited, Inherited}, {Inherited, 0}},
WindowElements->{
 "StatusArea", "MemoryMonitor", "MagnificationPopUp", "VerticalScrollBar", 
  "MenuBar"},
WindowTitle->"Lensmaker's Equation - Source",
DockedCells->{},
CellContext->Notebook,
FrontEndVersion->"8.0 for Microsoft Windows (32-bit) (November 7, 2010)",
StyleDefinitions->"Default.nb"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[622, 23, 140, 3, 70, "Section"],
Cell[765, 28, 3700, 64, 70, "Section"],
Cell[4468, 94, 623, 17, 70, "Input",
 CellID->879060982],
Cell[5094, 113, 807, 21, 70, "Input",
 CellID->1798036416],
Cell[5904, 136, 3077, 75, 70, "Input",
 CellID->728566442],
Cell[8984, 213, 869, 21, 70, "Input",
 CellID->652727127],
Cell[9856, 236, 853, 21, 70, "Input",
 CellID->1038354613],
Cell[10712, 259, 1105, 29, 70, "Input",
 CellID->995012816],
Cell[11820, 290, 2200, 45, 70, "Input",
 CellID->1145946428],
Cell[14023, 337, 8592, 170, 70, "Input",
 CellID->1837916761],
Cell[22618, 509, 2095, 51, 70, "Input",
 CellID->1009555335]
}, Open  ]],
Cell[CellGroupData[{
Cell[24750, 565, 18861, 398, 70, "Output",
 CellID->109723962],
Cell[CellGroupData[{
Cell[43636, 967, 209, 6, 70, "Section"],
Cell[43848, 975, 1598, 52, 70, "Text"]
}, Close]]
}, Open  ]],
Cell[45472, 1031, 276, 7, 70, "Text"],
Cell[45751, 1040, 449, 12, 70, "Text"],
Cell[46203, 1054, 376, 11, 70, "Text"],
Cell[46582, 1067, 798, 25, 70, "Text"],
Cell[47383, 1094, 591, 20, 70, "Text"],
Cell[CellGroupData[{
Cell[47999, 1118, 355, 9, 70, "Text",
 CellGroupingRules->{"SectionGrouping", 25}],
Cell[48357, 1129, 887, 24, 70, "Text"],
Cell[49247, 1155, 456, 15, 70, "Text",
 CellID->418556502],
Cell[CellGroupData[{
Cell[49728, 1174, 815, 24, 70, "Text"],
Cell[50546, 1200, 1190, 33, 70, "Text"]
}, Open  ]]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
(* NotebookSignature lSNoui8JQI@OsADm2Jxvryh@ *)
