(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 8.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       157,          7]
NotebookDataLength[     33150,        867]
NotebookOptionsPosition[     31785,        816]
NotebookOutlinePosition[     32438,        841]
CellTagsIndexPosition[     32395,        838]
WindowTitle->Single-Slit Diffraction Pattern - Source
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["Single-Slit Diffraction Pattern", "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False,
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztnVuSozgWQDNiPqa3MJ+zpVlCbaB3WlGVNV9dvYH6TQRUph/ZGIHQmwvG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   "], {{0, 0}, {1800, 25}}, {0, 255},
   ColorFunction->RGBColor],
  ImageSize->{1800, 25},
  PlotRange->{{0, 1800}, {0, 25}}]], "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False],

Cell[BoxData[
 RowBox[{"Manipulate", "[", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"Grid", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", " ", 
        RowBox[{
         RowBox[{"ListLinePlot", "[", 
          RowBox[{
           RowBox[{"IntensityData", "[", "b", "]"}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{"InterpolationOrder", "\[Rule]", "3"}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{"AxesLabel", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"None", ",", "\"\<I\>\""}], "}"}]}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{"PlotRange", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"0", ",", "All"}], "}"}]}], ",", "\[IndentingNewLine]", 
           RowBox[{"Ticks", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"Automatic", ",", "None"}], "}"}]}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{"Filling", "\[Rule]", "Axis"}], ",", "\[IndentingNewLine]", 
           RowBox[{"Epilog", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"{", 
               RowBox[{
                RowBox[{"Thickness", "[", ".01", "]"}], ",", 
                RowBox[{"{", 
                 RowBox[{
                  RowBox[{"Line", "[", 
                   RowBox[{"{", 
                    RowBox[{
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"-", "4"}], ",", "0"}], "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{
                    RowBox[{"-", "b"}], "/", "2"}], ",", "0"}], "}"}]}], 
                    "}"}], "]"}], ",", 
                  RowBox[{"Line", "[", 
                   RowBox[{"{", 
                    RowBox[{
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"b", "/", "2"}], ",", "0"}], "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{"4", ",", "0"}], "}"}]}], "}"}], "]"}]}], "}"}]}],
                "}"}], ",", 
              RowBox[{"Line", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"{", 
                  RowBox[{"p", ",", "0"}], "}"}], ",", 
                 RowBox[{
                  RowBox[{"IntensityData", "[", "b", "]"}], 
                  "\[LeftDoubleBracket]", 
                  RowBox[{"IntegerPart", "[", 
                   RowBox[{
                    RowBox[{"10", 
                    RowBox[{"(", 
                    RowBox[{"4", "+", "p"}], ")"}]}], "+", "1"}], "]"}], 
                  "\[RightDoubleBracket]"}]}], "}"}], "]"}]}], "}"}]}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{"ImageSize", "\[Rule]", "350"}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"ListLinePlot", "[", 
          RowBox[{
           RowBox[{"SpiralData", "[", 
            RowBox[{"p", ",", "b"}], "]"}], ",", "\[IndentingNewLine]", 
           RowBox[{"Axes", "\[Rule]", "False"}], ",", "\[IndentingNewLine]", 
           RowBox[{"InterpolationOrder", "\[Rule]", "3"}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{"AspectRatio", "\[Rule]", "Automatic"}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{"Epilog", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"Red", ",", 
              RowBox[{"Line", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{
                  RowBox[{"SpiralData", "[", 
                   RowBox[{"p", ",", "b"}], "]"}], "\[LeftDoubleBracket]", 
                  "1", "\[RightDoubleBracket]"}], ",", 
                 RowBox[{
                  RowBox[{"SpiralData", "[", 
                   RowBox[{"p", ",", "b"}], "]"}], "\[LeftDoubleBracket]", 
                  RowBox[{"-", "1"}], "\[RightDoubleBracket]"}]}], "}"}], 
               "]"}]}], "}"}]}], ",", "\[IndentingNewLine]", 
           RowBox[{"PlotRange", "\[Rule]", "Full"}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{"ImageSize", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"250", ",", "300"}], "}"}]}]}], "]"}]}], "}"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"ListDensityPlot", "[", 
          RowBox[{
           RowBox[{"DensityData", "[", "b", "]"}], ",", "\[IndentingNewLine]", 
           RowBox[{"FrameTicks", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"Automatic", ",", "None"}], "}"}]}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{"InterpolationOrder", "\[Rule]", "3"}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{"PlotRange", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"0", ",", "Full"}], "}"}]}], ",", "\[IndentingNewLine]", 
           
           RowBox[{"ColorFunction", "\[Rule]", "GrayLevel"}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{"AspectRatio", "\[Rule]", ".2"}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{"ImageSize", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"350", ",", "100"}], "}"}]}]}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"Text", "@", 
          RowBox[{"Grid", "[", 
           RowBox[{"{", 
            RowBox[{"{", 
             RowBox[{"\"\<relative intensity at position x = \>\"", ",", 
              RowBox[{"TraditionalForm", "[", "p", "]"}], ",", "\"\<: \>\"", 
              ",", 
              RowBox[{"TraditionalForm", "[", 
               RowBox[{"int", "[", 
                RowBox[{"p", ",", "b"}], "]"}], "]"}]}], "}"}], "}"}], 
           "]"}]}]}], "}"}]}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"Spacings", "\[Rule]", "0"}]}], "]"}], ",", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"b", ",", "0.25", ",", "\"\<slit width (mm)\>\""}], "}"}], ",", 
     "0.05", ",", "6", ",", "0.01", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"p", ",", "0", ",", "\"\<position\>\""}], "}"}], ",", 
     RowBox[{"-", "4"}], ",", "4", ",", ".1", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"TrackedSymbols", "\[RuleDelayed]", 
    RowBox[{"{", 
     RowBox[{"b", ",", "p", ",", "se", ",", "spe"}], "}"}]}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"Initialization", "\[RuleDelayed]", 
    RowBox[{"(", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"e", "=", "1000"}], ";", "\n", 
      RowBox[{"(*", 
       RowBox[{
        RowBox[{"distance", " ", "slit"}], "-", 
        RowBox[{"screen", " ", "in", " ", "mm"}]}], "*)"}], 
      "\[IndentingNewLine]", 
      RowBox[{"\[Lambda]", "=", 
       RowBox[{"630", " ", 
        SuperscriptBox["10", 
         RowBox[{"-", "6"}]]}]}], ";", 
      RowBox[{"(*", 
       RowBox[{
        RowBox[{"wavelength", " ", "in", " ", "mm"}], ",", " ", 
        RowBox[{
         RowBox[{"1", "/", "\[Lambda]"}], "=", "1587.3"}]}], "*)"}], 
      "\[IndentingNewLine]", 
      RowBox[{"n", "=", "200"}], ";", 
      RowBox[{"(*", 
       RowBox[{"number", " ", "of", " ", "paths"}], "*)"}], 
      "\[IndentingNewLine]", "\[IndentingNewLine]", 
      RowBox[{"CalculateVector", "=", 
       RowBox[{"Compile", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"x", ",", "_Real"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"b", ",", "_Real"}], "}"}]}], "}"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{
          UnderoverscriptBox["\[Sum]", 
           RowBox[{"j", "=", "1"}], "200"], 
          RowBox[{"Exp", "[", 
           RowBox[{"2", "\[Pi]", " ", "\[ImaginaryI]", " ", 
            SqrtBox[
             RowBox[{
              SuperscriptBox["1000", "2"], "+", 
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"x", " ", "-", " ", 
                 RowBox[{"b", 
                  RowBox[{"(", 
                   RowBox[{
                    FractionBox["1", "2"], " ", "-", 
                    FractionBox["j", "201"]}], ")"}]}]}], ")"}], "2"]}]], 
            "1587.3"}], "]"}]}]}], "]"}]}], ";", "\[IndentingNewLine]", 
      "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"VectorData", "[", 
        RowBox[{"pos_", ",", "SlitWidth_"}], "]"}], ":=", 
       RowBox[{"FoldList", "[", 
        RowBox[{"Plus", ",", 
         RowBox[{"0", "+", 
          RowBox[{"0", "\[ImaginaryI]"}]}], ",", 
         RowBox[{"Table", "[", 
          RowBox[{
           RowBox[{".1", 
            RowBox[{"Exp", "[", 
             RowBox[{"2", "\[Pi]", " ", "\[ImaginaryI]", " ", 
              SqrtBox[
               RowBox[{
                SuperscriptBox["1000", "2"], "+", 
                SuperscriptBox[
                 RowBox[{"(", 
                  RowBox[{"pos", "-", " ", 
                   RowBox[{"SlitWidth", 
                    RowBox[{"(", 
                    RowBox[{
                    FractionBox["1", "2"], " ", "-", 
                    FractionBox["j", "201"]}], ")"}]}]}], ")"}], "2"]}]], 
              "1587.3"}], "]"}]}], ",", 
           RowBox[{"{", 
            RowBox[{"j", ",", "1", ",", "200"}], "}"}]}], "]"}]}], "]"}]}], 
      ";", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"IntensityData", "[", "SlitWidth_", "]"}], ":=", 
       RowBox[{"Module", "[", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"LocationOnScreen", ",", 
           RowBox[{"vec0", "=", 
            RowBox[{"CalculateVector", "[", 
             RowBox[{"0", ",", "SlitWidth"}], "]"}]}]}], "}"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"Table", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"LocationOnScreen", ",", 
             SuperscriptBox[
              RowBox[{"(", 
               RowBox[{"Abs", "[", 
                RowBox[{
                 FractionBox["1", "vec0"], 
                 RowBox[{"CalculateVector", "[", 
                  RowBox[{"LocationOnScreen", ",", "SlitWidth"}], "]"}]}], 
                "]"}], ")"}], "2"]}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"LocationOnScreen", ",", 
             RowBox[{"-", "4"}], ",", "4", ",", ".1"}], "}"}]}], "]"}]}], 
        "]"}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"DensityData", "[", "SlitWidth_", "]"}], ":=", 
       "\[IndentingNewLine]", 
       RowBox[{"Flatten", "[", 
        RowBox[{
         RowBox[{"Table", "[", 
          RowBox[{
           RowBox[{"Map", "[", 
            RowBox[{
             RowBox[{
              RowBox[{"{", 
               RowBox[{
                RowBox[{
                "#", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}], 
                ",", "i", ",", 
                RowBox[{
                "#", "\[LeftDoubleBracket]", "2", "\[RightDoubleBracket]"}]}],
                "}"}], "&"}], ",", 
             RowBox[{"IntensityData", "[", "SlitWidth", "]"}]}], "]"}], ",", 
           RowBox[{"{", 
            RowBox[{"i", ",", "0", ",", "1"}], "}"}]}], "]"}], ",", "1"}], 
        "]"}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"SpiralData", "[", 
        RowBox[{"pos_", ",", "SlitWidth_"}], "]"}], ":=", 
       RowBox[{"Map", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"Re", "[", "#", "]"}], ",", 
            RowBox[{"Im", "[", "#", "]"}]}], "}"}], "&"}], ",", 
         RowBox[{"VectorData", "[", 
          RowBox[{"pos", ",", "SlitWidth"}], "]"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"int", "[", 
        RowBox[{"pos_", ",", "SlitWidth_"}], "]"}], ":=", 
       RowBox[{"NumberForm", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Norm", "[", 
           RowBox[{"Last", "[", 
            RowBox[{"SpiralData", "[", 
             RowBox[{"pos", ",", "SlitWidth"}], "]"}], "]"}], "]"}], "/", 
          "20"}], ",", 
         RowBox[{"{", 
          RowBox[{"1", ",", "2"}], "}"}]}], "]"}]}], ";"}], 
     "\[IndentingNewLine]", ")"}]}]}], "]"}]], "Input",
 CellChangeTimes->{
  3.35696210375764*^9, 3.449148791686205*^9, {3.449149619505871*^9, 
   3.44914970621647*^9}, {3.4491504526928663`*^9, 3.449150465706765*^9}, {
   3.4493218682169313`*^9, 3.449321883133423*^9}, {3.449321931017211*^9, 
   3.449322257526859*^9}, {3.449322291017207*^9, 3.4493223354996552`*^9}, {
   3.4493249635734367`*^9, 3.449325000441123*^9}, {3.449425049077967*^9, 
   3.449425049637023*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`b$$ = 0.25, $CellContext`p$$ = 0, 
    Typeset`show$$ = True, Typeset`bookmarkList$$ = {}, 
    Typeset`bookmarkMode$$ = "Menu", Typeset`animator$$, Typeset`animvar$$ = 
    1, Typeset`name$$ = "\"untitled\"", Typeset`specs$$ = {{{
       Hold[$CellContext`b$$], 0.25, "slit width (mm)"}, 0.05, 6, 0.01}, {{
       Hold[$CellContext`p$$], 0, "position"}, -4, 4, 0.1}}, 
    Typeset`size$$ = {600., {200., 205.}}, Typeset`update$$ = 0, 
    Typeset`initDone$$, Typeset`skipInitDone$$ = 
    False, $CellContext`b$28838$$ = 0, $CellContext`p$28839$$ = 0}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, 
      "Variables" :> {$CellContext`b$$ = 0.25, $CellContext`p$$ = 0}, 
      "ControllerVariables" :> {
        Hold[$CellContext`b$$, $CellContext`b$28838$$, 0], 
        Hold[$CellContext`p$$, $CellContext`p$28839$$, 0]}, 
      "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, "Body" :> Grid[{{
          ListLinePlot[
           $CellContext`IntensityData[$CellContext`b$$], InterpolationOrder -> 
           3, AxesLabel -> {None, "I"}, PlotRange -> {0, All}, 
           Ticks -> {Automatic, None}, Filling -> Axis, Epilog -> {{
              Thickness[0.01], {
               Line[{{-4, 0}, {(-$CellContext`b$$)/2, 0}}], 
               Line[{{$CellContext`b$$/2, 0}, {4, 0}}]}}, 
             Line[{{$CellContext`p$$, 0}, 
               Part[
                $CellContext`IntensityData[$CellContext`b$$], 
                IntegerPart[10 (4 + $CellContext`p$$) + 1]]}]}, ImageSize -> 
           350], 
          ListLinePlot[
           $CellContext`SpiralData[$CellContext`p$$, $CellContext`b$$], Axes -> 
           False, InterpolationOrder -> 3, AspectRatio -> Automatic, 
           Epilog -> {Red, 
             Line[{
               Part[
                $CellContext`SpiralData[$CellContext`p$$, $CellContext`b$$], 
                1], 
               Part[
                $CellContext`SpiralData[$CellContext`p$$, $CellContext`b$$], \
-1]}]}, PlotRange -> Full, ImageSize -> {250, 300}]}, {
          ListDensityPlot[
           $CellContext`DensityData[$CellContext`b$$], 
           FrameTicks -> {Automatic, None}, InterpolationOrder -> 3, 
           PlotRange -> {0, Full}, ColorFunction -> GrayLevel, AspectRatio -> 
           0.2, ImageSize -> {350, 100}], 
          Text[
           Grid[{{"relative intensity at position x = ", 
              TraditionalForm[$CellContext`p$$], ": ", 
              TraditionalForm[
               $CellContext`int[$CellContext`p$$, $CellContext`b$$]]}}]]}}, 
        Spacings -> 0], 
      "Specifications" :> {{{$CellContext`b$$, 0.25, "slit width (mm)"}, 0.05,
          6, 0.01, Appearance -> 
         "Labeled"}, {{$CellContext`p$$, 0, "position"}, -4, 4, 0.1, 
         Appearance -> "Labeled"}}, 
      "Options" :> {
       TrackedSymbols :> {$CellContext`b$$, $CellContext`p$$, \
$CellContext`se, $CellContext`spe}}, 
      "DefaultOptions" :> {ControllerLinking -> True}],
     ImageSizeCache->{645., {258., 263.}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    Initialization:>(($CellContext`e = 1000; $CellContext`\[Lambda] = 
       630 10^(-6); $CellContext`n = 200; $CellContext`CalculateVector = 
       Compile[{{$CellContext`x, 
           Blank[Real]}, {$CellContext`b$$, 
           Blank[Real]}}, 
         Sum[
          Exp[
          2 Pi I (1000^2 + ($CellContext`x - $CellContext`b$$ (
                1/2 - $CellContext`j/201))^2)^Rational[1, 2] 
           1587.3], {$CellContext`j, 1, 200}]]; $CellContext`VectorData[
         Pattern[$CellContext`pos, 
          Blank[]], 
         Pattern[$CellContext`SlitWidth, 
          Blank[]]] := FoldList[Plus, 0 + 0 I, 
         Table[
         0.1 Exp[2 Pi 
            I (1000^2 + ($CellContext`pos - $CellContext`SlitWidth (
                 1/2 - $CellContext`j/201))^2)^Rational[1, 2] 
            1587.3], {$CellContext`j, 1, 200}]]; $CellContext`IntensityData[
         Pattern[$CellContext`SlitWidth, 
          Blank[]]] := 
       Module[{$CellContext`LocationOnScreen, $CellContext`vec0 = \
$CellContext`CalculateVector[0, $CellContext`SlitWidth]}, 
         Table[{$CellContext`LocationOnScreen, 
           Abs[(1/$CellContext`vec0) \
$CellContext`CalculateVector[$CellContext`LocationOnScreen, \
$CellContext`SlitWidth]]^2}, {$CellContext`LocationOnScreen, -4, 4, 
           0.1}]]; $CellContext`DensityData[
         Pattern[$CellContext`SlitWidth, 
          Blank[]]] := Flatten[
         Table[
          Map[{
            Part[#, 1], $CellContext`i, 
            Part[#, 2]}& , 
           $CellContext`IntensityData[$CellContext`SlitWidth]], \
{$CellContext`i, 0, 1}], 1]; $CellContext`SpiralData[
         Pattern[$CellContext`pos, 
          Blank[]], 
         Pattern[$CellContext`SlitWidth, 
          Blank[]]] := Map[{
          Re[#], 
          Im[#]}& , 
         $CellContext`VectorData[$CellContext`pos, $CellContext`SlitWidth]]; \
$CellContext`int[
         Pattern[$CellContext`pos, 
          Blank[]], 
         Pattern[$CellContext`SlitWidth, 
          Blank[]]] := NumberForm[Norm[
           Last[
            $CellContext`SpiralData[$CellContext`pos, \
$CellContext`SlitWidth]]]/20, {1, 2}]; Null); Typeset`initDone$$ = True),
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 CellID->7746025],

Cell[CellGroupData[{

Cell["CAPTION", "Section",
 CellFrame->{{0, 0}, {1, 0}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->12,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell["\<\
This Demonstration shows the intensity distribution of one-slit diffraction \
over a wide range of slit widths so that both Fraunhofer and Fresnel \
diffraction are covered. You can produce an interference pattern as it would \
be seen on a screen and the vector sum for any position on the screen. The \
results are derived from Feynman's method of \"integrating over paths\" and \
can be proven experimentally if the light source is a He-Ne laser and the \
screen is 1 m distance from the slit.\
\>", "Text"]
}, Close]]
}, Open  ]],

Cell[CellGroupData[{

Cell["DETAILS", "Section",
 CellFrame->{{0, 0}, {1, 0}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->12,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[TextData[{
 "The aperture of the slit is divided into ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " points, each of which can be considered the origin of a wave. The wave \
vectors are summed up at a point on the screen. At a given point each wave \
has a different distance to travel, which results in a different phase angle ",
 Cell[BoxData[
  FormBox["\[CurlyPhi]", TraditionalForm]], "InlineMath"],
 ". Let ",
 Cell[BoxData[
  FormBox["e", TraditionalForm]], "InlineMath"],
 " be the distance between the slit and the screen, ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 " the distance of the point on the screen relative to the optic axis and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["b", "i"], TraditionalForm]], "InlineMath"],
 " the space between one of the points in the slit to the optic axis. The \
length of a path is then ",
 Cell[BoxData[
  FormBox[
   RowBox[{"l", "=", 
    SqrtBox[
     RowBox[{
      SuperscriptBox["e", "2"], "+", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"a", "\[PlusMinus]", 
         SubscriptBox["b", "i"]}], ")"}], "2"]}]]}], TraditionalForm]], 
  "InlineMath"],
 " . Because ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{"a", "\[PlusMinus]", 
      SubscriptBox["b", "i"]}], ")"}], "\[LessLess]", "e"}], 
   TraditionalForm]], "InlineMath"],
 " it is approximately ",
 Cell[BoxData[
  FormBox[
   RowBox[{"l", "\[TildeTilde]", 
    RowBox[{"e", "+", 
     FractionBox[
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"a", "\[PlusMinus]", 
         SubscriptBox["b", "i"]}], ")"}], "2"], 
      RowBox[{"2", "e"}]]}]}], TraditionalForm]], "InlineMath"],
 ". In the case of Fraunhofer diffraction where the slit width ",
 Cell[BoxData[
  FormBox[
   RowBox[{"b", "\[LessLess]", "a"}], TraditionalForm]], "InlineMath"],
 " it can be further approximated to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"l", "\[TildeTilde]", 
    RowBox[{"e", "+", 
     RowBox[{
      FractionBox[
       SuperscriptBox["a", "2"], 
       RowBox[{"2", "e"}]], "\[PlusMinus]", 
      FractionBox[
       RowBox[{"a", " ", "b"}], "e"]}]}]}], TraditionalForm]], "InlineMath"],
 ", which means that ",
 Cell[BoxData[
  FormBox[
   StyleBox["l",
    FontSlant->"Italic"], TraditionalForm]], "InlineMath"],
 " and hence ",
 Cell[BoxData[
  FormBox["\[CurlyPhi]", TraditionalForm]], "InlineMath"],
 " depend linearly on ",
 Cell[BoxData[
  FormBox["b", TraditionalForm]], "InlineMath"],
 ". The sum of the wave vectors gives a line of constant curvature, a circle. \
It is completed for the first time in the first minima. When ",
 Cell[BoxData[
  FormBox["b", TraditionalForm]], "InlineMath"],
 " becomes greater, ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["b", "2"], TraditionalForm]], "InlineMath"],
 " cannot be neglected and the phase angle between two neighboring wave \
vectors is no longer constant. The result of the vector sum is now the \
so-called ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["Cornu", "1"], TraditionalForm]]],
 " spiral, which is an important tool for investigating Fresnel diffraction."
}], "Text"],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[InvisibleSpace]", "1"], TraditionalForm]]],
 "After the French physicist Alfred Cornu (1841-1902)."
}], "Text"],

Cell[TextData[{
 "R. P. Feynman, ",
 StyleBox["QED: The Strange Theory of Light and Matter",
  FontSlant->"Italic"],
 ", Princeton: Princeton University Press, 1985."
}], "Text"]
}, Close]],

Cell[CellGroupData[{

Cell["THIS NOTEBOOK IS THE SOURCE CODE FROM", "Text",
 CellFrame->{{0, 0}, {0, 0}},
 CellMargins->{{48, 10}, {4, 28}},
 CellGroupingRules->{"SectionGrouping", 25},
 CellFrameMargins->{{48, 48}, {6, 5}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->10,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[TextData[{
 "\"",
 ButtonBox["Single-Slit Diffraction Pattern",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/SingleSlitDiffractionPattern/"], 
    None},
  ButtonNote->
   "http://demonstrations.wolfram.com/SingleSlitDiffractionPattern/"],
 "\"",
 " from ",
 ButtonBox["the Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
 "\[ParagraphSeparator]\[NonBreakingSpace]",
 ButtonBox["http://demonstrations.wolfram.com/SingleSlitDiffractionPattern/",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/SingleSlitDiffractionPattern/"], 
    None},
  ButtonNote->
   "http://demonstrations.wolfram.com/SingleSlitDiffractionPattern/"]
}], "Text",
 CellMargins->{{48, Inherited}, {0, Inherited}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.5]],

Cell[TextData[{
 "Contributed by: ",
 ButtonBox["Hans-Joachim Domke",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/author.html?author=Hans-Joachim+\
Domke"], None},
  ButtonNote->
   "http://demonstrations.wolfram.com/author.html?author=Hans-Joachim+Domke"],\

 " and ",
 ButtonBox["Martin Domke",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/author.html?author=Martin+Domke"], 
    None},
  ButtonNote->
   "http://demonstrations.wolfram.com/author.html?author=Martin+Domke"]
}], "Text",
 CellDingbat->"\[FilledSmallSquare]",
 CellMargins->{{66, 48}, {2, 4}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.6]],

Cell[CellGroupData[{

Cell[TextData[{
 "A full-function Wolfram ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " system (Version 6 or higher) is required to edit this notebook.\n",
 StyleBox[ButtonBox["GET WOLFRAM MATHEMATICA \[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolfram.com/products/mathematica/"], None},
  ButtonNote->"http://www.wolfram.com/products/mathematica/"],
  FontFamily->"Helvetica",
  FontWeight->"Bold",
  FontSlant->"Italic",
  FontColor->RGBColor[1, 0.42, 0]]
}], "Text",
 CellFrame->True,
 CellMargins->{{48, 68}, {8, 28}},
 CellFrameMargins->12,
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 CellChangeTimes->{3.3750111182355957`*^9},
 ParagraphSpacing->{1., 1.},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.411765],
 Background->RGBColor[1, 1, 1]],

Cell[TextData[{
 "\[Copyright] ",
 StyleBox[ButtonBox["Wolfram Demonstrations Project & Contributors",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Terms of Use",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/termsofuse.html"], None},
  ButtonNote->"http://demonstrations.wolfram.com/termsofuse.html"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Make a new version of this Demonstration \
\[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/participate/upload.jsp?id=\
SingleSlitDiffractionPattern"], None},
  ButtonNote->None],
  FontColor->GrayLevel[0.6]]
}], "Text",
 CellFrame->{{0, 0}, {0, 0.5}},
 CellMargins->{{48, 10}, {20, 50}},
 CellFrameMargins->{{6, 0}, {6, 6}},
 CellFrameColor->GrayLevel[0.6],
 FontFamily->"Verdana",
 FontSize->9,
 FontColor->GrayLevel[0.6]]
}, Open  ]]
}, Open  ]]
},
Editable->True,
Saveable->False,
ScreenStyleEnvironment->"Working",
CellInsertionPointCell->None,
WindowSize->{780, 650},
WindowMargins->{{Inherited, Inherited}, {Inherited, 0}},
WindowElements->{
 "StatusArea", "MemoryMonitor", "MagnificationPopUp", "VerticalScrollBar", 
  "MenuBar"},
WindowTitle->"Single-Slit Diffraction Pattern - Source",
DockedCells->{},
CellContext->Notebook,
FrontEndVersion->"8.0 for Microsoft Windows (32-bit) (November 7, 2010)",
StyleDefinitions->"Default.nb"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[633, 23, 151, 3, 70, "Section"],
Cell[787, 28, 3700, 64, 70, "Section"],
Cell[4490, 94, 12773, 315, 70, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[17300, 414, 5812, 126, 70, "Output",
 CellID->7746025],
Cell[CellGroupData[{
Cell[23137, 544, 209, 6, 70, "Section"],
Cell[23349, 552, 520, 8, 70, "Text"]
}, Close]]
}, Open  ]],
Cell[CellGroupData[{
Cell[23917, 566, 209, 6, 70, "Section"],
Cell[24129, 574, 3157, 92, 70, "Text"],
Cell[27289, 668, 174, 5, 70, "Text"],
Cell[27466, 675, 178, 5, 70, "Text"]
}, Close]],
Cell[CellGroupData[{
Cell[27680, 685, 355, 9, 70, "Text",
 CellGroupingRules->{"SectionGrouping", 25}],
Cell[28038, 696, 966, 28, 70, "Text"],
Cell[29007, 726, 707, 23, 70, "Text"],
Cell[CellGroupData[{
Cell[29739, 753, 815, 24, 70, "Text"],
Cell[30557, 779, 1200, 33, 70, "Text"]
}, Open  ]]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
(* NotebookSignature EQ4bcATV@6kcYCDxJzqSo2yc *)
