(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 8.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       157,          7]
NotebookDataLength[     34920,        934]
NotebookOptionsPosition[     33093,        870]
NotebookOutlinePosition[     33777,        896]
CellTagsIndexPosition[     33734,        893]
WindowTitle->Propagation of Reflected and Refracted Waves at an Interface - \
Source
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["Propagation of Reflected and Refracted Waves at an Interface", "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False,
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztnVuSozgWQDNiPqa3MJ+zpVlCbaB3WlGVNV9dvYH6TQRUph/ZGIHQmwvG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   "], {{0, 0}, {1800, 25}}, {0, 255},
   ColorFunction->RGBColor],
  ImageSize->{1800, 25},
  PlotRange->{{0, 1800}, {0, 25}}]], "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False],

Cell[BoxData[
 RowBox[{"Manipulate", "[", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"\[Theta]1", "=", 
     RowBox[{"\[Theta]", " ", "Degree"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"sn1", "=", 
     RowBox[{"Sin", "[", "\[Theta]1", "]"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"sn2", "=", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"v2", "/", "v1"}], ")"}], "sn1"}]}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{"\[Theta]2", "=", 
     RowBox[{"ArcSin", "[", "sn2", "]"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"kx1", "=", 
     RowBox[{"sn1", "*", 
      RowBox[{"f", "/", "v1"}]}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"ky1", "=", 
     RowBox[{"cs1", "*", 
      RowBox[{"f", "/", "v1"}]}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"kx2", "=", 
     RowBox[{"sn2", "*", 
      RowBox[{"f", "/", "v2"}]}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"ky2", "=", 
     RowBox[{"cs2", "*", 
      RowBox[{"f", "/", "v2"}]}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"If", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Im", "[", "\[Theta]2", "]"}], " ", "\[NotEqual]", "0"}], ",", 
      " ", 
      RowBox[{
       RowBox[{"RC", " ", "=", " ", "1.0"}], ";", " ", 
       RowBox[{"TC", " ", "=", " ", "0.0"}]}], ",", 
      RowBox[{
       RowBox[{"cs1", "=", 
        RowBox[{"Sqrt", "[", 
         RowBox[{"1", "-", 
          RowBox[{"sn1", "*", "sn1"}]}], "]"}]}], ";", 
       RowBox[{"cs2", "=", 
        RowBox[{"Sqrt", "[", 
         RowBox[{"1", "-", 
          RowBox[{"sn2", "*", "sn2"}]}], "]"}]}], ";", 
       RowBox[{"del", " ", "=", " ", 
        RowBox[{
         RowBox[{"v1", "*", "cs1"}], " ", "+", " ", 
         RowBox[{"v2", "*", "cs2"}]}]}], ";", 
       RowBox[{"RC", " ", "=", " ", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"v2", "*", "cs2"}], " ", "-", " ", 
           RowBox[{"v1", "*", "cs1"}]}], ")"}], "/", "del"}]}], ";", 
       RowBox[{"TC", " ", "=", " ", 
        RowBox[{"2", "v2", "*", 
         RowBox[{"cs2", "/", "del"}]}]}]}]}], "]"}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{"p1", "=", 
     RowBox[{"DensityPlot", "[", 
      RowBox[{
       RowBox[{"0.5", "+", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"Cos", "[", 
            RowBox[{
             RowBox[{"kx1", "*", "x"}], " ", "-", " ", 
             RowBox[{"ky1", "*", "y"}], " ", "-", " ", 
             RowBox[{"2", "\[Pi]", "*", "f", "*", "t"}]}], "]"}], "+", 
           RowBox[{"RC", "*", 
            RowBox[{"Cos", "[", 
             RowBox[{
              RowBox[{"kx1", "*", "x"}], " ", "+", " ", 
              RowBox[{"ky1", "*", "y"}], " ", "-", " ", 
              RowBox[{"2", "\[Pi]", "*", "f", "*", "t"}]}], "]"}]}]}], ")"}], 
         "/", "4"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "50000"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", "0", ",", "25000"}], "}"}], ",", 
       RowBox[{"PlotPoints", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"101", ",", "51"}], "}"}]}], ",", 
       RowBox[{"Frame", "\[Rule]", "False"}], ",", 
       RowBox[{"ColorFunction", "\[Rule]", "\"\<GrayTones\>\""}], ",", 
       RowBox[{"ColorFunctionScaling", "\[Rule]", "False"}]}], "]"}]}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{"p2", "=", 
     RowBox[{"DensityPlot", "[", 
      RowBox[{
       RowBox[{"0.5", " ", "+", " ", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"TC", "*", 
           RowBox[{"Cos", "[", 
            RowBox[{
             RowBox[{"kx2", "*", "x"}], " ", "-", " ", 
             RowBox[{"ky2", "*", "y"}], " ", "-", " ", 
             RowBox[{"2", "\[Pi]", "*", "f", "*", "t"}]}], "]"}]}], ")"}], 
         "/", "4"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "50000"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", 
         RowBox[{"-", "25000"}], ",", "0"}], "}"}], ",", 
       RowBox[{"PlotPoints", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"101", ",", "51"}], "}"}]}], ",", 
       RowBox[{"Frame", "\[Rule]", "False"}], ",", 
       RowBox[{"ColorFunction", "\[Rule]", "\"\<GrayTones\>\""}], ",", 
       RowBox[{"ColorFunctionScaling", "\[Rule]", "False"}]}], "]"}]}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{"p3", " ", "=", " ", 
     RowBox[{"Graphics", "[", 
      RowBox[{"Line", "[", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0", ",", "0"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"50000", ",", "0"}], "}"}]}], "}"}], "]"}], "]"}]}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{"Show", "[", 
     RowBox[{"p1", ",", "p2", ",", "p3", ",", 
      RowBox[{"PlotRange", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0", ",", "50000"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "25000"}], ",", "25000"}], "}"}]}], "}"}]}], ",", 
      RowBox[{"AspectRatio", "\[Rule]", "1"}], ",", 
      RowBox[{"ImageSize", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"400", ",", "400"}], "}"}]}]}], "]"}]}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", "\"\<time\>\""}], "}"}], ",", "0", ",", 
     "10", ",", "1", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}], ",", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\[Theta]", ",", "30.", ",", "\"\<incident angle\>\""}], "}"}], 
     ",", "0", ",", "90.", ",", ".01", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}], ",", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"v1", ",", "300", ",", "\"\<upper velocity\>\""}], "}"}], ",", 
     "50", ",", "1000", ",", "50", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}], ",", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"v2", ",", "500", ",", "\"\<lower velocity\>\""}], "}"}], ",", 
     "50", ",", "1000", ",", "50", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}], ",", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"f", ",", "0.3", ",", "\"\<frequency\>\""}], "}"}], ",", "0.1", 
     ",", "1.0", ",", "0.1", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}], ",", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"ControlPlacement", "\[Rule]", "Left"}], ",", 
   RowBox[{"TrackedSymbols", "\[Rule]", "True"}]}], "\[IndentingNewLine]", 
  "]"}]], "Input",
 CellChangeTimes->{{3.467334713970749*^9, 3.467334715129867*^9}, {
   3.467334756731056*^9, 3.4673348780046663`*^9}, {3.467334946436989*^9, 
   3.4673349506842422`*^9}, {3.4673350135747137`*^9, 
   3.4673350247346697`*^9}, {3.467335157238212*^9, 3.467335172244628*^9}, {
   3.467335249375614*^9, 3.467335317623757*^9}, {3.467335365776032*^9, 
   3.4673355097775106`*^9}, {3.467335585723627*^9, 3.467335657150317*^9}, {
   3.467335697851609*^9, 3.467335702300552*^9}, {3.467335739398863*^9, 
   3.467335740922421*^9}, {3.467335774789909*^9, 3.467335831287167*^9}, {
   3.467336125851647*^9, 3.467336236504758*^9}, 3.4674763047909927`*^9, {
   3.467476422816022*^9, 3.467476442344054*^9}, {3.467476494477336*^9, 
   3.467476600489105*^9}, {3.4674768962299767`*^9, 3.467476905624865*^9}, {
   3.46757220352001*^9, 3.467572218548266*^9}, {3.467572249480641*^9, 
   3.467572280753858*^9}, {3.4675723122798967`*^9, 3.4675724232934647`*^9}, {
   3.4675726494932213`*^9, 3.46757275362976*^9}, {3.4675728303085403`*^9, 
   3.467572842382476*^9}, {3.467572899861629*^9, 3.467572948182743*^9}, {
   3.467574197552953*^9, 3.467574298425812*^9}, {3.4675744365408688`*^9, 
   3.4675744498627167`*^9}, {3.467574492039424*^9, 3.46757449280202*^9}, {
   3.467574618259288*^9, 3.4675746252823343`*^9}, {3.467574728939949*^9, 
   3.467574729196691*^9}, {3.4676544823985662`*^9, 3.4676544826628304`*^9}, {
   3.467654741775206*^9, 3.467654757160859*^9}, {3.467654822201096*^9, 
   3.467654958858766*^9}, {3.4676550049969473`*^9, 3.467655032859355*^9}, {
   3.467655166601034*^9, 3.46765517418888*^9}, {3.467655262285894*^9, 
   3.467655274232442*^9}, {3.4676553194096127`*^9, 3.467655319671101*^9}, {
   3.4676555003971233`*^9, 3.467655697133409*^9}, {3.4676558365958357`*^9, 
   3.4676558372825623`*^9}, {3.4676566284312773`*^9, 
   3.4676566440792227`*^9}, {3.467656674466927*^9, 3.467656703717373*^9}, {
   3.4676575997653*^9, 3.467657631976699*^9}, {3.467659955192553*^9, 
   3.4676600879061327`*^9}, {3.4676606073674583`*^9, 3.46766062366389*^9}, {
   3.467660856352748*^9, 3.467660863874345*^9}, {3.4676609359222393`*^9, 
   3.467661018239966*^9}, {3.4676610694447927`*^9, 3.467661073142624*^9}, {
   3.467661485963498*^9, 3.467661489487808*^9}, {3.467662440635914*^9, 
   3.4676624782315617`*^9}, {3.467733133809226*^9, 3.467733216523369*^9}, {
   3.46773330543491*^9, 3.467733353144759*^9}, 3.467733386609363*^9, {
   3.4677366545353317`*^9, 3.4677366611440973`*^9}, {3.467736701119681*^9, 
   3.467736724813698*^9}, {3.467736758719717*^9, 3.467736803130644*^9}, {
   3.4677368630796556`*^9, 3.467736869968235*^9}, {3.4677369106443377`*^9, 
   3.467736919692808*^9}, 3.46773704886799*^9, {3.4677373700994043`*^9, 
   3.467737370339958*^9}, {3.467739964642385*^9, 3.467739978427424*^9}, 
   3.4677400449535303`*^9, {3.4677400807178183`*^9, 3.467740088802642*^9}, {
   3.46774234862097*^9, 3.4677423561413507`*^9}, 3.4678220901294127`*^9, {
   3.467822147555612*^9, 3.467822151240446*^9}, {3.467825032025984*^9, 
   3.467825079883603*^9}, {3.467825133093853*^9, 3.467825168928506*^9}, 
   3.4678253292045717`*^9, {3.467825396705426*^9, 3.4678253986758966`*^9}, {
   3.467825444226083*^9, 3.46782544767301*^9}, 3.4678255111323557`*^9, {
   3.4679863400698*^9, 3.467986356467268*^9}, {3.467986471104648*^9, 
   3.467986482593014*^9}, {3.468073366338584*^9, 3.468073428044754*^9}, {
   3.468688159511445*^9, 3.468688162668538*^9}},
 CellID->45291526]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`f$$ = 0.3, $CellContext`t$$ = 
    0, $CellContext`v1$$ = 300, $CellContext`v2$$ = 
    500, $CellContext`\[Theta]$$ = 30., Typeset`show$$ = True, 
    Typeset`bookmarkList$$ = {}, Typeset`bookmarkMode$$ = "Menu", 
    Typeset`animator$$, Typeset`animvar$$ = 1, Typeset`name$$ = 
    "\"untitled\"", Typeset`specs$$ = {{{
       Hold[$CellContext`t$$], 0, "time"}, 0, 10, 1}, {{
       Hold[$CellContext`\[Theta]$$], 30., "incident angle"}, 0, 90., 0.01}, {{
       Hold[$CellContext`v1$$], 300, "upper velocity"}, 50, 1000, 50}, {{
       Hold[$CellContext`v2$$], 500, "lower velocity"}, 50, 1000, 50}, {{
       Hold[$CellContext`f$$], 0.3, "frequency"}, 0.1, 1., 0.1}}, 
    Typeset`size$$ = {400., {198., 202.}}, Typeset`update$$ = 0, 
    Typeset`initDone$$, Typeset`skipInitDone$$ = 
    True, $CellContext`t$394538$$ = 0, $CellContext`\[Theta]$394539$$ = 
    0, $CellContext`v1$394540$$ = 0, $CellContext`v2$394541$$ = 
    0, $CellContext`f$394542$$ = 0}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, 
      "Variables" :> {$CellContext`f$$ = 0.3, $CellContext`t$$ = 
        0, $CellContext`v1$$ = 300, $CellContext`v2$$ = 
        500, $CellContext`\[Theta]$$ = 30.}, "ControllerVariables" :> {
        Hold[$CellContext`t$$, $CellContext`t$394538$$, 0], 
        Hold[$CellContext`\[Theta]$$, $CellContext`\[Theta]$394539$$, 0], 
        Hold[$CellContext`v1$$, $CellContext`v1$394540$$, 0], 
        Hold[$CellContext`v2$$, $CellContext`v2$394541$$, 0], 
        Hold[$CellContext`f$$, $CellContext`f$394542$$, 0]}, 
      "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, 
      "Body" :> ($CellContext`\[Theta]1 = $CellContext`\[Theta]$$ 
         Degree; $CellContext`sn1 = 
        Sin[$CellContext`\[Theta]1]; $CellContext`sn2 = \
($CellContext`v2$$/$CellContext`v1$$) $CellContext`sn1; \
$CellContext`\[Theta]2 = 
        ArcSin[$CellContext`sn2]; $CellContext`kx1 = $CellContext`sn1 \
($CellContext`f$$/$CellContext`v1$$); $CellContext`ky1 = $CellContext`cs1 \
($CellContext`f$$/$CellContext`v1$$); $CellContext`kx2 = $CellContext`sn2 \
($CellContext`f$$/$CellContext`v2$$); $CellContext`ky2 = $CellContext`cs2 \
($CellContext`f$$/$CellContext`v2$$); 
       If[Im[$CellContext`\[Theta]2] != 
         0, $CellContext`RC = 1.; $CellContext`TC = 0., $CellContext`cs1 = 
          Sqrt[1 - $CellContext`sn1 $CellContext`sn1]; $CellContext`cs2 = 
          Sqrt[1 - $CellContext`sn2 $CellContext`sn2]; $CellContext`del = \
$CellContext`v1$$ $CellContext`cs1 + $CellContext`v2$$ $CellContext`cs2; \
$CellContext`RC = ($CellContext`v2$$ $CellContext`cs2 - $CellContext`v1$$ \
$CellContext`cs1)/$CellContext`del; $CellContext`TC = 
          2 $CellContext`v2$$ ($CellContext`cs2/$CellContext`del)]; \
$CellContext`p1 = 
        DensityPlot[
         0.5 + (Cos[$CellContext`kx1 $CellContext`x - $CellContext`ky1 \
$CellContext`y - 2 Pi $CellContext`f$$ $CellContext`t$$] + $CellContext`RC 
             Cos[$CellContext`kx1 $CellContext`x + $CellContext`ky1 \
$CellContext`y - 2 Pi $CellContext`f$$ $CellContext`t$$])/4, {$CellContext`x, 
           0, 50000}, {$CellContext`y, 0, 25000}, PlotPoints -> {101, 51}, 
          Frame -> False, ColorFunction -> "GrayTones", ColorFunctionScaling -> 
          False]; $CellContext`p2 = 
        DensityPlot[
         0.5 + ($CellContext`TC 
            Cos[$CellContext`kx2 $CellContext`x - $CellContext`ky2 \
$CellContext`y - 2 Pi $CellContext`f$$ $CellContext`t$$])/4, {$CellContext`x, 
           0, 50000}, {$CellContext`y, -25000, 0}, PlotPoints -> {101, 51}, 
          Frame -> False, ColorFunction -> "GrayTones", ColorFunctionScaling -> 
          False]; $CellContext`p3 = Graphics[
          Line[{{0, 0}, {50000, 0}}]]; 
       Show[$CellContext`p1, $CellContext`p2, $CellContext`p3, 
         PlotRange -> {{0, 50000}, {-25000, 25000}}, AspectRatio -> 1, 
         ImageSize -> {400, 400}]), 
      "Specifications" :> {{{$CellContext`t$$, 0, "time"}, 0, 10, 1, 
         Appearance -> "Labeled", ImageSize -> 
         Tiny}, {{$CellContext`\[Theta]$$, 30., "incident angle"}, 0, 90., 
         0.01, Appearance -> "Labeled", ImageSize -> 
         Tiny}, {{$CellContext`v1$$, 300, "upper velocity"}, 50, 1000, 50, 
         Appearance -> "Labeled", ImageSize -> 
         Tiny}, {{$CellContext`v2$$, 500, "lower velocity"}, 50, 1000, 50, 
         Appearance -> "Labeled", ImageSize -> 
         Tiny}, {{$CellContext`f$$, 0.3, "frequency"}, 0.1, 1., 0.1, 
         Appearance -> "Labeled", ImageSize -> Tiny}}, 
      "Options" :> {ControlPlacement -> Left, TrackedSymbols -> True}, 
      "DefaultOptions" :> {ControllerLinking -> True}],
     ImageSizeCache->{639., {226., 231.}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 CellID->298488656],

Cell[CellGroupData[{

Cell["CAPTION", "Section",
 CellFrame->{{0, 0}, {1, 0}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->12,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell["\<\
A propagating wave of given frequency (in Hz) is incident on an interface \
between two acoustic media, with varying propagation velocities. The incident \
angle is measured from the vertical, running from 0 to 90 degrees. The \
amplitudes of the reflected and refracted waves are computed for the \
specified incident angle. The reflected wave is added to the incident wave in \
the upper medium and the refracted wave is shown in the lower medium. At a \
certain incident angle that depends on the relative velocities, the incident \
wave will be totally reflected and the bottom half of the plot will become \
solid gray, indicating zero amplitude.\
\>", "Text"]
}, Close]]
}, Open  ]],

Cell[CellGroupData[{

Cell["DETAILS", "Section",
 CellFrame->{{0, 0}, {1, 0}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->12,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[TextData[{
 "This Demonstration shows how a propagating plane wave interacts with an \
interface between two acoustic media. The incident wave impinges on the \
boundary from the top and causes a reflected wave that interferes with the \
incident wave. The incident wave is transmitted into the bottom layer as a \
refracted wave with its own propagation velocity. The incident and refracted \
wave angles ",
 Cell[BoxData[
  FormBox["\[Theta]", TraditionalForm]], "InlineMath"],
 " (angle of the wave normal relative to vertical) are related by the \
well-known equation"
}], "Text"],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox[
     RowBox[{"sin", "(", 
      SubscriptBox["\[Theta]", "1"], ")"}], 
     RowBox[{"sin", "(", 
      SubscriptBox["\[Theta]", "2"], ")"}]], "=", " ", 
    FormBox[
     FractionBox[
      SubscriptBox["v", "1"], 
      SubscriptBox["v", "2"]],
     TraditionalForm]}], TraditionalForm]], "InlineMath"],
 ","
}], "Text"],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox["v", TraditionalForm]], "InlineMath"],
 " is velocity and the subscripts 1 and 2 denote the top and bottom media. \
The incident wave is taken to have amplitude 1. If ",
 Cell[BoxData[
  FormBox["RC", TraditionalForm]], "InlineMath"],
 " denotes the reflection coefficient for the scattered wave amplitudes, then \
the wave motion in the top layer is described by (Towne, 1988)"
}], "Text"],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"cos", "(", 
     RowBox[{
      RowBox[{
       FormBox[
        SubscriptBox["k", 
         SubscriptBox["x", "1"]],
        TraditionalForm], "x"}], "-", 
      RowBox[{
       FormBox[
        RowBox[{" ", 
         SubscriptBox["k", 
          SubscriptBox["y", "1"]]}],
        TraditionalForm], "y"}], "-", "\[Omega]t"}], ")"}], "+", 
    RowBox[{"RC", " ", 
     RowBox[{"cos", "(", 
      RowBox[{
       RowBox[{
        FormBox[
         SubscriptBox["k", 
          SubscriptBox["x", "1"]],
         TraditionalForm], "x"}], "+", 
       RowBox[{
        FormBox[
         RowBox[{" ", 
          SubscriptBox["k", 
           SubscriptBox["y", "1"]]}],
         TraditionalForm], "y"}], "-", 
       RowBox[{"\[Omega]", " ", "t"}]}], ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 ","
}], "Text"],

Cell[TextData[{
 "while, with ",
 Cell[BoxData[
  FormBox["TC", TraditionalForm]], "InlineMath"],
 " denoting the transmission coefficient, the motion in the bottom layer is \
described by"
}], "Text"],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{"TC", " ", 
    RowBox[{"cos", "[", 
     RowBox[{
      RowBox[{
       FormBox[
        SubscriptBox["k", 
         SubscriptBox["x", "2"]],
        TraditionalForm], "x"}], "-", 
      RowBox[{
       FormBox[
        RowBox[{" ", 
         SubscriptBox["k", 
          SubscriptBox["y", "2"]]}],
        TraditionalForm], "y"}], "-", 
      RowBox[{"\[Omega]", " ", "t"}]}], "]"}]}], TraditionalForm]], 
  "InlineMath"],
 ","
}], "Text"],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["k", "x"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["k", "y"], TraditionalForm]], "InlineMath"],
 " are the horizontal and vertical wavenumbers, with the subscripts 1 and 2 \
for the top and bottom layers, respectively, and ",
 Cell[BoxData[
  FormBox["\[Omega]", TraditionalForm]], "InlineMath"],
 " is the frequency in radians. Here all lengths are in meters."
}], "Text"],

Cell[TextData[{
 "The coefficients ",
 Cell[BoxData[
  FormBox["RC", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["TC", TraditionalForm]], "InlineMath"],
 " are computed for variable angles of incidence and velocities with"
}], "Text"],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{"RC", "=", 
    FormBox[
     FractionBox[
      StyleBox[
       RowBox[{
        RowBox[{
         SubscriptBox["v", "2"], 
         RowBox[{"cos", "(", 
          SubscriptBox["\[Theta]", "2"], ")"}]}], " ", "-", " ", 
        RowBox[{
         SubscriptBox["v", "1"], 
         RowBox[{"cos", "(", 
          SubscriptBox["\[Theta]", "1"], ")"}], " "}]}],
       FontSize->12], 
      RowBox[{
       StyleBox[
        RowBox[{
         SubscriptBox["v", "1"], 
         RowBox[{"cos", "(", 
          SubscriptBox["\[Theta]", "1"], ")"}]}],
        FontSize->12], 
       StyleBox[" ",
        FontSize->12], 
       StyleBox["+",
        FontSize->12], 
       StyleBox[" ",
        FontSize->12], 
       RowBox[{
        StyleBox[
         SubscriptBox["v", "2"],
         FontSize->12], 
        RowBox[{
         StyleBox["cos",
          FontSize->12], 
         StyleBox["(",
          FontSize->12], 
         StyleBox[
          SubscriptBox["\[Theta]", "2"],
          FontSize->12], ")"}]}]}]],
     TraditionalForm], " "}], TraditionalForm]], "InlineMath"],
 ","
}], "Text"],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{"TC", "=", 
    FormBox[
     StyleBox[
      FractionBox[
       RowBox[{
        StyleBox["2",
         FontSize->12], 
        StyleBox[
         SubscriptBox["v", "2"],
         FontSize->12], 
        RowBox[{
         StyleBox["cos",
          FontSize->12], 
         StyleBox["(",
          FontSize->12], 
         StyleBox[
          SubscriptBox["\[Theta]", "2"],
          FontSize->12], 
         StyleBox[")",
          FontSize->9]}]}], 
       RowBox[{
        StyleBox[
         RowBox[{
          SubscriptBox["v", "1"], 
          RowBox[{"cos", "(", 
           SubscriptBox["\[Theta]", "1"], ")"}]}],
         FontSize->12], 
        StyleBox[" ",
         FontSize->12], 
        StyleBox["+",
         FontSize->12], 
        StyleBox[" ",
         FontSize->12], 
        RowBox[{
         StyleBox[
          SubscriptBox["v", "2"],
          FontSize->12], 
         RowBox[{
          StyleBox["cos",
           FontSize->12], 
          StyleBox["(",
           FontSize->12], 
          StyleBox[
           SubscriptBox["\[Theta]", "2"],
           FontSize->12], ")"}]}]}]],
      FontSize->14],
     TraditionalForm]}], TraditionalForm]], "InlineMath"],
 ","
}], "Text"],

Cell[TextData[{
 "assuming, with no loss of generality, that the densities of the two media \
are equal. This definition puts ",
 Cell[BoxData[
  FormBox["RC", TraditionalForm]], "InlineMath"],
 " in the range ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{
     RowBox[{"-", "1"}], ",", 
     RowBox[{"+", "1"}]}], "]"}], TraditionalForm]], "InlineMath"],
 ", while ",
 Cell[BoxData[
  FormBox["TC", TraditionalForm]], "InlineMath"],
 " has the range ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "2"}], "]"}], TraditionalForm]], "InlineMath"],
 ". Because of wave interference, the top media can have amplitudes in the \
range ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{
     RowBox[{"-", "2"}], ",", 
     RowBox[{"+", "2"}]}], "]"}], TraditionalForm]], "InlineMath"],
 ". The overall amplitudes are scaled in the plots such that black is -2, \
white is +2, and medium gray is 0. Note that amplitudes appear to be \
continuous across the boundary. This physical fact is the basis for the \
equations for computation of ",
 Cell[BoxData[
  FormBox["RC", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["TC", TraditionalForm]], "InlineMath"],
 "."
}], "Text"],

Cell["Reference:", "Text"],

Cell[TextData[{
 "D. H. Towne, ",
 StyleBox["Wave Phenomena",
  FontSlant->"Italic"],
 ", New York: Dover Publications, 1988."
}], "Text"]
}, Close]],

Cell[CellGroupData[{

Cell["THIS NOTEBOOK IS THE SOURCE CODE FROM", "Text",
 CellFrame->{{0, 0}, {0, 0}},
 CellMargins->{{48, 10}, {4, 28}},
 CellGroupingRules->{"SectionGrouping", 25},
 CellFrameMargins->{{48, 48}, {6, 5}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->10,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[TextData[{
 "\"",
 ButtonBox["Propagation of Reflected and Refracted Waves at an Interface",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/\
PropagationOfReflectedAndRefractedWavesAtAnInterface/"], None},
  ButtonNote->
   "http://demonstrations.wolfram.com/\
PropagationOfReflectedAndRefractedWavesAtAnInterface/"],
 "\"",
 " from ",
 ButtonBox["the Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
 "\[ParagraphSeparator]\[NonBreakingSpace]",
 ButtonBox["http://demonstrations.wolfram.com/\
PropagationOfReflectedAndRefractedWavesAtAnInterface/",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/\
PropagationOfReflectedAndRefractedWavesAtAnInterface/"], None},
  ButtonNote->
   "http://demonstrations.wolfram.com/\
PropagationOfReflectedAndRefractedWavesAtAnInterface/"]
}], "Text",
 CellMargins->{{48, Inherited}, {0, Inherited}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.5]],

Cell[TextData[{
 "Contributed by: ",
 ButtonBox["David von Seggern",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/author.html?author=David+von+\
Seggern"], None},
  ButtonNote->
   "http://demonstrations.wolfram.com/author.html?author=David+von+Seggern"],
 " (University of Nevada)"
}], "Text",
 CellDingbat->"\[FilledSmallSquare]",
 CellMargins->{{66, 48}, {2, 4}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.6],
 CellID->588875106],

Cell[CellGroupData[{

Cell[TextData[{
 "A full-function Wolfram ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " system (Version 6 or higher) is required to edit this notebook.\n",
 StyleBox[ButtonBox["GET WOLFRAM MATHEMATICA \[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolfram.com/products/mathematica/"], None},
  ButtonNote->"http://www.wolfram.com/products/mathematica/"],
  FontFamily->"Helvetica",
  FontWeight->"Bold",
  FontSlant->"Italic",
  FontColor->RGBColor[1, 0.42, 0]]
}], "Text",
 CellFrame->True,
 CellMargins->{{48, 68}, {8, 28}},
 CellFrameMargins->12,
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 CellChangeTimes->{3.3750111182355957`*^9},
 ParagraphSpacing->{1., 1.},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.411765],
 Background->RGBColor[1, 1, 1]],

Cell[TextData[{
 "\[Copyright] ",
 StyleBox[ButtonBox["Wolfram Demonstrations Project & Contributors",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Terms of Use",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/termsofuse.html"], None},
  ButtonNote->"http://demonstrations.wolfram.com/termsofuse.html"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Make a new version of this Demonstration \
\[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/participate/upload.jsp?id=\
PropagationOfReflectedAndRefractedWavesAtAnInterface"], None},
  ButtonNote->None],
  FontColor->GrayLevel[0.6]]
}], "Text",
 CellFrame->{{0, 0}, {0, 0.5}},
 CellMargins->{{48, 10}, {20, 50}},
 CellFrameMargins->{{6, 0}, {6, 6}},
 CellFrameColor->GrayLevel[0.6],
 FontFamily->"Verdana",
 FontSize->9,
 FontColor->GrayLevel[0.6]]
}, Open  ]]
}, Open  ]]
},
Editable->True,
Saveable->False,
ScreenStyleEnvironment->"Working",
CellInsertionPointCell->None,
WindowSize->{780, 650},
WindowMargins->{{Inherited, Inherited}, {Inherited, 0}},
WindowElements->{
 "StatusArea", "MemoryMonitor", "MagnificationPopUp", "VerticalScrollBar", 
  "MenuBar"},
WindowTitle->"Propagation of Reflected and Refracted Waves at an Interface - \
Source",
DockedCells->{},
CellContext->Notebook,
FrontEndVersion->"8.0 for Microsoft Windows (32-bit) (November 7, 2010)",
StyleDefinitions->"Default.nb"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[664, 24, 180, 3, 70, "Section"],
Cell[847, 29, 3700, 64, 70, "Section"],
Cell[4550, 95, 10356, 228, 70, "Input",
 CellID->45291526]
}, Open  ]],
Cell[CellGroupData[{
Cell[14943, 328, 5308, 94, 70, "Output",
 CellID->298488656],
Cell[CellGroupData[{
Cell[20276, 426, 209, 6, 70, "Section"],
Cell[20488, 434, 675, 10, 70, "Text"]
}, Close]]
}, Open  ]],
Cell[CellGroupData[{
Cell[21211, 450, 209, 6, 70, "Section"],
Cell[21423, 458, 589, 11, 70, "Text"],
Cell[22015, 471, 392, 15, 70, "Text"],
Cell[22410, 488, 442, 10, 70, "Text"],
Cell[22855, 500, 877, 34, 70, "Text"],
Cell[23735, 536, 201, 6, 70, "Text"],
Cell[23939, 544, 494, 20, 70, "Text"],
Cell[24436, 566, 488, 14, 70, "Text"],
Cell[24927, 582, 259, 8, 70, "Text"],
Cell[25189, 592, 1144, 44, 70, "Text"],
Cell[26336, 638, 1255, 51, 70, "Text"],
Cell[27594, 691, 1229, 38, 70, "Text"],
Cell[28826, 731, 26, 0, 70, "Text"],
Cell[28855, 733, 138, 5, 70, "Text"]
}, Close]],
Cell[CellGroupData[{
Cell[29029, 743, 355, 9, 70, "Text",
 CellGroupingRules->{"SectionGrouping", 25}],
Cell[29387, 754, 1115, 31, 70, "Text"],
Cell[30505, 787, 493, 16, 70, "Text",
 CellID->588875106],
Cell[CellGroupData[{
Cell[31023, 807, 815, 24, 70, "Text"],
Cell[31841, 833, 1224, 33, 70, "Text"]
}, Open  ]]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
(* NotebookSignature KQzvtAxsE2MJLCDHT9CC0AFE *)
