(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 8.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       157,          7]
NotebookDataLength[    174402,       4161]
NotebookOptionsPosition[    172632,       4100]
NotebookOutlinePosition[    173327,       4126]
CellTagsIndexPosition[    173284,       4123]
WindowTitle->Dynamic Approximation of Static Quantities (Visual Depth \
Perception 14) - Source
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["\<\
Dynamic Approximation of Static Quantities (Visual Depth Perception 14)\
\>", "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False,
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztnVuSozgWQDNiPqa3MJ+zpVlCbaB3WlGVNV9dvYH6TQRUph/ZGIHQmwvG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   "], {{0, 0}, {1800, 25}}, {0, 255},
   ColorFunction->RGBColor],
  ImageSize->{1800, 25},
  PlotRange->{{0, 1800}, {0, 25}}]], "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False],

Cell[BoxData[
 RowBox[{"Manipulate", "[", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"nORM", "[", 
      RowBox[{"vector_", "?", "VectorQ"}], "]"}], ":=", 
     SqrtBox[
      RowBox[{"vector", ".", "vector"}]]}], ";", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"degrees", "[", "q_", "]"}], ":=", 
     RowBox[{"q", "*", 
      RowBox[{"180", "/", "\[Pi]"}]}]}], ";", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"arcMinutes", "[", "q_", "]"}], ":=", 
     RowBox[{"q", "*", "60", "*", 
      RowBox[{"180", "/", "\[Pi]"}]}]}], ";", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"cIRCLE", "[", 
      RowBox[{
       RowBox[{"L_", "?", "VectorQ"}], ",", 
       RowBox[{"R_", "?", "VectorQ"}], ",", 
       RowBox[{"F_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"Module", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"a", ",", "bX", ",", "bY", ",", "c", ",", "r"}], "}"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"a", "=", 
         RowBox[{"Det", "[", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{
              RowBox[{"L", "[", 
               RowBox[{"[", "1", "]"}], "]"}], ",", 
              RowBox[{"R", "[", 
               RowBox[{"[", "1", "]"}], "]"}], ",", 
              RowBox[{"F", "[", 
               RowBox[{"[", "1", "]"}], "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"L", "[", 
               RowBox[{"[", "2", "]"}], "]"}], ",", 
              RowBox[{"R", "[", 
               RowBox[{"[", "2", "]"}], "]"}], ",", 
              RowBox[{"F", "[", 
               RowBox[{"[", "2", "]"}], "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "}"}], "]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"bX", "=", 
         RowBox[{"Det", "[", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{
              RowBox[{"L", ".", "L"}], ",", 
              RowBox[{"R", ".", "R"}], ",", 
              RowBox[{"F", ".", "F"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"L", "[", 
               RowBox[{"[", "2", "]"}], "]"}], ",", 
              RowBox[{"R", "[", 
               RowBox[{"[", "2", "]"}], "]"}], ",", 
              RowBox[{"F", "[", 
               RowBox[{"[", "2", "]"}], "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "}"}], "]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"bY", "=", 
         RowBox[{"Det", "[", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{
              RowBox[{"L", "[", 
               RowBox[{"[", "1", "]"}], "]"}], ",", 
              RowBox[{"R", "[", 
               RowBox[{"[", "1", "]"}], "]"}], ",", 
              RowBox[{"F", "[", 
               RowBox[{"[", "1", "]"}], "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"L", ".", "L"}], ",", 
              RowBox[{"R", ".", "R"}], ",", 
              RowBox[{"F", ".", "F"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "}"}], "]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"c", "=", 
         RowBox[{"-", 
          RowBox[{"Det", "[", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"{", 
              RowBox[{
               RowBox[{"L", ".", "L"}], ",", 
               RowBox[{"R", ".", "R"}], ",", 
               RowBox[{"F", ".", "F"}]}], "}"}], ",", 
             RowBox[{"{", 
              RowBox[{
               RowBox[{"L", "[", 
                RowBox[{"[", "1", "]"}], "]"}], ",", 
               RowBox[{"R", "[", 
                RowBox[{"[", "1", "]"}], "]"}], ",", 
               RowBox[{"F", "[", 
                RowBox[{"[", "1", "]"}], "]"}]}], "}"}], ",", 
             RowBox[{"{", 
              RowBox[{
               RowBox[{"L", "[", 
                RowBox[{"[", "2", "]"}], "]"}], ",", 
               RowBox[{"R", "[", 
                RowBox[{"[", "2", "]"}], "]"}], ",", 
               RowBox[{"F", "[", 
                RowBox[{"[", "2", "]"}], "]"}]}], "}"}]}], "}"}], "]"}]}]}], 
        ";", "\[IndentingNewLine]", 
        RowBox[{"Circle", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"{", 
            RowBox[{"bX", ",", "bY"}], "}"}], "/", 
           RowBox[{"(", 
            RowBox[{"2", " ", "a"}], ")"}]}], ",", 
          FractionBox[
           SqrtBox[
            RowBox[{
             SuperscriptBox["bX", "2"], "+", 
             SuperscriptBox["bY", "2"], "-", 
             RowBox[{"4", " ", "a", " ", "c"}]}]], 
           RowBox[{"2", 
            RowBox[{"Abs", "[", "a", "]"}]}]], ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"\[Pi]", "/", "6"}], ",", 
            RowBox[{"\[Pi]", 
             RowBox[{"(", 
              RowBox[{"5", "/", "6"}], ")"}]}]}], "}"}]}], "]"}]}]}], "]"}]}],
     ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"aim\[Psi]", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"ArcTan", "[", 
      RowBox[{
       RowBox[{"headAimVector", "[", 
        RowBox[{"[", "1", "]"}], "]"}], ",", 
       RowBox[{"headAimVector", "[", 
        RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], ";", "\[IndentingNewLine]",
     "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"midPoint", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"base", "-", 
      RowBox[{"aimOffset", "*", "interOcular", "*", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"headAimVector", "[", 
          RowBox[{"[", "2", "]"}], "]"}], ",", 
         RowBox[{"-", 
          RowBox[{"headAimVector", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}]}], "}"}]}]}]}], ";", 
    "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"rightEyeCenter", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"base", "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         FractionBox["1", "2"], "-", "aimOffset"}], ")"}], "*", "interOcular",
        "*", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"headAimVector", "[", 
          RowBox[{"[", "2", "]"}], "]"}], ",", 
         RowBox[{"-", 
          RowBox[{"headAimVector", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}]}], "}"}]}]}]}], ";", 
    "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"leftEyeCenter", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"base", "-", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         FractionBox["1", "2"], "+", "aimOffset"}], ")"}], "*", "interOcular",
        "*", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"headAimVector", "[", 
          RowBox[{"[", "2", "]"}], "]"}], ",", 
         RowBox[{"-", 
          RowBox[{"headAimVector", "[", 
           RowBox[{"[", "1", "]"}], "]"}]}]}], "}"}]}]}]}], ";", 
    "\[IndentingNewLine]", "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"rightViewVector", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"fixate", "-", 
      RowBox[{"rightEyeCenter", "[", 
       RowBox[{"headAimVector", ",", " ", "base"}], "]"}]}]}], ";", 
    "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"right\[CurlyPhi]", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"ArcTan", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"rightViewVector", "[", 
         RowBox[{"headAimVector", ",", " ", "fixate", ",", " ", "base"}], 
         "]"}], "[", 
        RowBox[{"[", "1", "]"}], "]"}], ",", 
       RowBox[{
        RowBox[{"rightViewVector", "[", 
         RowBox[{"headAimVector", ",", " ", "fixate", ",", " ", "base"}], 
         "]"}], "[", 
        RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], ";", "\[IndentingNewLine]",
     "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"rightEyeNode", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{
      RowBox[{"rightEyeCenter", "[", 
       RowBox[{"headAimVector", ",", " ", "base"}], "]"}], "+", 
      RowBox[{"nodePercent", "*", "eyeRadius", "*", 
       RowBox[{
        RowBox[{"rightViewVector", "[", 
         RowBox[{"headAimVector", ",", "fixate", ",", " ", "base"}], "]"}], 
        "/", 
        RowBox[{"nORM", "[", 
         RowBox[{"rightViewVector", "[", 
          RowBox[{"headAimVector", ",", " ", "fixate", ",", " ", "base"}], 
          "]"}], "]"}]}]}]}]}], ";", "\[IndentingNewLine]", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"leftViewVector", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"fixate", "-", 
      RowBox[{"leftEyeCenter", "[", 
       RowBox[{"headAimVector", ",", " ", "base"}], "]"}]}]}], ";", 
    "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"left\[CurlyPhi]", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"ArcTan", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"leftViewVector", "[", 
         RowBox[{"headAimVector", ",", " ", "fixate", ",", " ", "base"}], 
         "]"}], "[", 
        RowBox[{"[", "1", "]"}], "]"}], ",", 
       RowBox[{
        RowBox[{"leftViewVector", "[", 
         RowBox[{"headAimVector", ",", " ", "fixate", ",", " ", "base"}], 
         "]"}], "[", 
        RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], ";", "\[IndentingNewLine]",
     "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"leftEyeNode", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{
      RowBox[{"leftEyeCenter", "[", 
       RowBox[{"headAimVector", ",", " ", "base"}], "]"}], "+", 
      RowBox[{"nodePercent", "*", "eyeRadius", "*", 
       RowBox[{
        RowBox[{"leftViewVector", "[", 
         RowBox[{"headAimVector", ",", " ", "fixate", ",", " ", "base"}], 
         "]"}], "/", 
        RowBox[{"nORM", "[", 
         RowBox[{"leftViewVector", "[", 
          RowBox[{"headAimVector", ",", " ", "fixate", ",", " ", "base"}], 
          "]"}], "]"}]}]}]}]}], ";", "\[IndentingNewLine]", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"fixateConvergence", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"right\[CurlyPhi]", "[", 
        RowBox[{"headAimVector", ",", "fixate", ",", "base"}], "]"}], "-", 
       RowBox[{"left\[CurlyPhi]", "[", 
        RowBox[{"headAimVector", ",", "fixate", ",", "base"}], "]"}]}], 
      ")"}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"left\[Alpha]", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{
      RowBox[{"left\[CurlyPhi]", "[", 
       RowBox[{"headAimVector", ",", "fixate", ",", "base"}], "]"}], "-", 
      RowBox[{"aim\[Psi]", "[", 
       RowBox[{"headAimVector", ",", "base"}], "]"}]}]}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"right\[Alpha]", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{
      RowBox[{"right\[CurlyPhi]", "[", 
       RowBox[{"headAimVector", ",", "fixate", ",", "base"}], "]"}], "-", 
      RowBox[{"aim\[Psi]", "[", 
       RowBox[{"headAimVector", ",", "base"}], "]"}]}]}], ";", 
    "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"rightDistractVector", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"distraction_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"distraction", "-", 
      RowBox[{"rightEyeNode", "[", 
       RowBox[{"headAimVector", ",", "fixate", ",", " ", "base"}], "]"}]}]}], 
    ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"right\[Delta]", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"distraction_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"ArcTan", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"rightDistractVector", "[", 
         RowBox[{
         "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
         "]"}], "[", 
        RowBox[{"[", "1", "]"}], "]"}], ",", 
       RowBox[{
        RowBox[{"rightDistractVector", "[", 
         RowBox[{
         "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
         "]"}], "[", 
        RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], ";", "\[IndentingNewLine]",
     "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"leftDistractVector", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"distraction_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"distraction", "-", 
      RowBox[{"leftEyeNode", "[", 
       RowBox[{"headAimVector", ",", "fixate", ",", " ", "base"}], "]"}]}]}], 
    ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"left\[Delta]", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"distraction_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"ArcTan", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"leftDistractVector", "[", 
         RowBox[{
         "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
         "]"}], "[", 
        RowBox[{"[", "1", "]"}], "]"}], ",", 
       RowBox[{
        RowBox[{"leftDistractVector", "[", 
         RowBox[{
         "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
         "]"}], "[", 
        RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], ";", "\[IndentingNewLine]",
     "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"convergenceDistraction", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"distraction_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{"Or", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Abs", "[", 
           RowBox[{
            RowBox[{"left\[CurlyPhi]", "[", 
             RowBox[{"headAimVector", ",", "fixate", ",", " ", "base"}], 
             "]"}], "-", 
            RowBox[{"left\[Delta]", "[", 
             RowBox[{
             "headAimVector", ",", "fixate", ",", " ", "distraction", ",", 
              "base"}], "]"}]}], "]"}], ">", 
          RowBox[{"\[Pi]", "/", "3."}]}], ",", 
         RowBox[{
          RowBox[{"Abs", "[", 
           RowBox[{
            RowBox[{"right\[CurlyPhi]", "[", 
             RowBox[{"headAimVector", ",", "fixate", ",", " ", "base"}], 
             "]"}], "-", 
            RowBox[{"right\[Delta]", "[", 
             RowBox[{
             "headAimVector", ",", "fixate", ",", " ", "distraction", ",", 
              "base"}], "]"}]}], "]"}], ">", 
          RowBox[{"\[Pi]", "/", "3."}]}]}], "]"}], ",", "0.0", ",", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"right\[Delta]", "[", 
          RowBox[{
          "headAimVector", ",", "fixate", ",", " ", "distraction", ",", 
           "base"}], "]"}], "-", 
         RowBox[{"left\[Delta]", "[", 
          RowBox[{
          "headAimVector", ",", "fixate", ",", " ", "distraction", ",", 
           "base"}], "]"}]}], ")"}]}], "]"}]}], ";", "\[IndentingNewLine]", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"right\[Theta]", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"distraction_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"right\[CurlyPhi]", "[", 
        RowBox[{"headAimVector", ",", "fixate", ",", " ", "base"}], "]"}], 
       "-", 
       RowBox[{"right\[Delta]", "[", 
        RowBox[{
        "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
        "]"}]}], ")"}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"left\[Theta]", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"distraction_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"left\[CurlyPhi]", "[", 
        RowBox[{"headAimVector", ",", "fixate", ",", " ", "base"}], "]"}], 
       "-", 
       RowBox[{"left\[Delta]", "[", 
        RowBox[{
        "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
        "]"}]}], ")"}]}], ";", "\[IndentingNewLine]", "\n", 
    RowBox[{
     RowBox[{"binocularDisparity", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"distraction_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{"Or", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"left\[Theta]", "[", 
           RowBox[{
           "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
           "]"}], "<", 
          RowBox[{
           RowBox[{"-", "\[Pi]"}], "/", "2."}]}], ",", 
         RowBox[{
          RowBox[{"left\[Theta]", "[", 
           RowBox[{
           "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
           "]"}], ">", 
          RowBox[{"\[Pi]", "/", "3."}]}], ",", 
         RowBox[{
          RowBox[{"right\[Theta]", "[", 
           RowBox[{
           "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
           "]"}], "<", 
          RowBox[{
           RowBox[{"-", "\[Pi]"}], "/", "3."}]}], ",", 
         RowBox[{
          RowBox[{"right\[Theta]", "[", 
           RowBox[{
           "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
           "]"}], ">", 
          RowBox[{"\[Pi]", "/", "2."}]}]}], "]"}], ",", "0", ",", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"right\[Theta]", "[", 
          RowBox[{
          "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
          "]"}], "-", 
         RowBox[{"left\[Theta]", "[", 
          RowBox[{
          "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
          "]"}]}], ")"}]}], "]"}]}], ";", "\[IndentingNewLine]", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"right\[Theta]inside", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"distraction_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{
      RowBox[{"right\[Theta]", "[", 
       RowBox[{
       "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
       "]"}], "+", 
      RowBox[{"ArcSin", "[", 
       RowBox[{"nodePercent", "*", 
        RowBox[{"Sin", "[", 
         RowBox[{"right\[Theta]", "[", 
          RowBox[{
          "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
          "]"}], "]"}]}], "]"}]}]}], ";", "\[IndentingNewLine]", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"left\[Theta]inside", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"distraction_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{
      RowBox[{"left\[Theta]", "[", 
       RowBox[{
       "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
       "]"}], "+", 
      RowBox[{"ArcSin", "[", 
       RowBox[{"nodePercent", "*", 
        RowBox[{"Sin", "[", 
         RowBox[{"left\[Theta]", "[", 
          RowBox[{
          "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
          "]"}], "]"}]}], "]"}]}]}], ";", "\[IndentingNewLine]", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"retinalDisparity", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"distraction_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{"Or", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"left\[Theta]", "[", 
           RowBox[{
           "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
           "]"}], "<", 
          RowBox[{
           RowBox[{"-", "\[Pi]"}], "/", "2."}]}], ",", 
         RowBox[{
          RowBox[{"left\[Theta]", "[", 
           RowBox[{
           "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
           "]"}], ">", 
          RowBox[{"\[Pi]", "/", "3."}]}], ",", 
         RowBox[{
          RowBox[{"right\[Theta]", "[", 
           RowBox[{
           "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
           "]"}], "<", 
          RowBox[{
           RowBox[{"-", "\[Pi]"}], "/", "3."}]}], ",", 
         RowBox[{
          RowBox[{"right\[Theta]", "[", 
           RowBox[{
           "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
           "]"}], ">", 
          RowBox[{"\[Pi]", "/", "2."}]}]}], "]"}], ",", "0", ",", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"right\[Theta]inside", "[", 
          RowBox[{
          "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
          "]"}], "-", 
         RowBox[{"left\[Theta]inside", "[", 
          RowBox[{
          "headAimVector", ",", "fixate", ",", "distraction", ",", "base"}], 
          "]"}]}], ")"}]}], "]"}]}], ";", "\[IndentingNewLine]", 
    "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"eyePict", "[", 
      RowBox[{
       RowBox[{"eyeCenter_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"Module", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"\[Lambda]", ",", "viewVector"}], "}"}], ",", 
       RowBox[{
        RowBox[{"viewVector", "=", 
         RowBox[{"fixate", "-", "eyeCenter"}]}], ";", "\[IndentingNewLine]", 
        RowBox[{"\[Lambda]", "=", 
         RowBox[{"ArcTan", "[", 
          RowBox[{
           RowBox[{"viewVector", "[", 
            RowBox[{"[", "1", "]"}], "]"}], ",", 
           RowBox[{"viewVector", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"viewVector", "=", 
         RowBox[{"viewVector", "/", 
          RowBox[{"nORM", "[", "viewVector", "]"}]}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"Show", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"Graphics", "[", 
             RowBox[{"{", 
              RowBox[{
               RowBox[{"AbsoluteThickness", "[", "2", "]"}], ",", "Gray", ",", 
               RowBox[{"Circle", "[", 
                RowBox[{"eyeCenter", ",", "eyeRadius", ",", 
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{"\[Lambda]", "+", ".4"}], ",", 
                   RowBox[{
                    RowBox[{"2", "\[Pi]"}], "+", "\[Lambda]", "-", ".4"}]}], 
                  "}"}]}], "]"}]}], "}"}], "]"}], ",", 
            RowBox[{"Graphics", "[", 
             RowBox[{"{", 
              RowBox[{"Gray", ",", "\[IndentingNewLine]", 
               RowBox[{"Circle", "[", 
                RowBox[{
                 RowBox[{"eyeCenter", "+", " ", 
                  RowBox[{".8", "eyeRadius", " ", "viewVector"}]}], ",", 
                 RowBox[{".5", "eyeRadius"}], ",", 
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{"\[Lambda]", "-", "1.5"}], ",", 
                   RowBox[{"\[Lambda]", "+", "1.5"}]}], "}"}]}], "]"}]}], 
              "}"}], "]"}], ",", "\[IndentingNewLine]", 
            RowBox[{"Graphics", "[", 
             RowBox[{"{", 
              RowBox[{"Black", ",", "\[IndentingNewLine]", 
               RowBox[{"Disk", "[", 
                RowBox[{
                 RowBox[{"eyeCenter", "+", 
                  RowBox[{
                  "eyeRadius", "*", "nodePercent", "*", "viewVector"}]}], ",", 
                 RowBox[{"eyeRadius", "/", "10"}]}], "]"}]}], 
              "\[IndentingNewLine]", "}"}], "]"}]}], "\[IndentingNewLine]", 
           "}"}], ",", 
          RowBox[{"AspectRatio", "\[Rule]", "Automatic"}]}], "]"}]}]}], 
      "]"}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"movingEyePict", "[", 
      RowBox[{
       RowBox[{"eyeCenter_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"Module", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"\[Lambda]", ",", "viewVector"}], "}"}], ",", 
       RowBox[{
        RowBox[{"viewVector", "=", 
         RowBox[{"fixate", "-", "eyeCenter"}]}], ";", "\[IndentingNewLine]", 
        RowBox[{"\[Lambda]", "=", 
         RowBox[{"ArcTan", "[", 
          RowBox[{
           RowBox[{"viewVector", "[", 
            RowBox[{"[", "1", "]"}], "]"}], ",", 
           RowBox[{"viewVector", "[", 
            RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"viewVector", "=", 
         RowBox[{"viewVector", "/", 
          RowBox[{"nORM", "[", "viewVector", "]"}]}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"Show", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"Graphics", "[", 
             RowBox[{"{", 
              RowBox[{
               RowBox[{"AbsoluteThickness", "[", "2", "]"}], ",", 
               RowBox[{"Dashing", "[", ".01", "]"}], ",", "Gray", ",", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"Line", "[", 
                  RowBox[{"{", 
                   RowBox[{
                    RowBox[{"eyeCenter", "-", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"1.", "eyeRadius"}], ",", 
                    RowBox[{"1.5", "eyeRadius"}]}], "}"}]}], ",", 
                    RowBox[{"eyeCenter", "+", 
                    RowBox[{"{", " ", 
                    RowBox[{
                    RowBox[{"1.", "eyeRadius"}], ",", 
                    RowBox[{
                    RowBox[{"-", "1.5"}], "eyeRadius"}]}], "}"}]}], ",", 
                    RowBox[{"eyeCenter", "+", 
                    RowBox[{"{", " ", 
                    RowBox[{
                    RowBox[{".7", "eyeRadius"}], ",", 
                    RowBox[{
                    RowBox[{"-", "1.6"}], "eyeRadius"}]}], "}"}]}]}], "}"}], 
                  "]"}], ",", 
                 RowBox[{"Circle", "[", 
                  RowBox[{"eyeCenter", ",", "eyeRadius", ",", 
                   RowBox[{"{", 
                    RowBox[{
                    RowBox[{"\[Lambda]", "+", ".4"}], ",", 
                    RowBox[{
                    RowBox[{"2", "\[Pi]"}], "+", "\[Lambda]", "-", ".4"}]}], 
                    "}"}]}], "]"}]}], "}"}]}], "}"}], "]"}], ",", 
            RowBox[{"Graphics", "[", 
             RowBox[{"{", 
              RowBox[{"Gray", ",", "\[IndentingNewLine]", 
               RowBox[{"Circle", "[", 
                RowBox[{
                 RowBox[{"eyeCenter", "+", " ", 
                  RowBox[{".8", "eyeRadius", " ", "viewVector"}]}], ",", 
                 RowBox[{".5", "eyeRadius"}], ",", 
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{"\[Lambda]", "-", "1.5"}], ",", 
                   RowBox[{"\[Lambda]", "+", "1.5"}]}], "}"}]}], "]"}]}], 
              "}"}], "]"}], ",", "\[IndentingNewLine]", 
            RowBox[{"Graphics", "[", 
             RowBox[{"{", 
              RowBox[{"Black", ",", "\[IndentingNewLine]", 
               RowBox[{"Disk", "[", 
                RowBox[{
                 RowBox[{"eyeCenter", "+", 
                  RowBox[{
                  "eyeRadius", "*", "nodePercent", "*", "viewVector"}]}], ",", 
                 RowBox[{"eyeRadius", "/", "10"}]}], "]"}]}], 
              "\[IndentingNewLine]", "}"}], "]"}]}], "\[IndentingNewLine]", 
           "}"}], ",", 
          RowBox[{"AspectRatio", "\[Rule]", "Automatic"}]}], "]"}]}]}], 
      "]"}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"eyeEdge", "[", "\[Theta]_", "]"}], ":=", 
     RowBox[{"eyeRadius", "*", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Cos", "[", "\[Theta]", "]"}], ",", 
        RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "}"}]}]}], ";", 
    "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"eyeEdge", "[", "\[Theta]_", "]"}], ":=", 
     RowBox[{"eyeRadius", "*", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Cos", "[", "\[Theta]", "]"}], ",", 
        RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "}"}]}]}], ";", 
    "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"inRays", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"distraction_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"Show", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"eyePict", "[", 
          RowBox[{
           RowBox[{"leftEyeCenter", "[", 
            RowBox[{"headAimVector", ",", " ", "base"}], "]"}], ",", 
           "fixate"}], "]"}], ",", 
         RowBox[{"eyePict", "[", 
          RowBox[{
           RowBox[{"rightEyeCenter", "[", 
            RowBox[{"headAimVector", ",", "base"}], "]"}], ",", "fixate"}], 
          "]"}], ",", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Red", ",", 
            RowBox[{"{", 
             RowBox[{"Line", "[", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"rightEyeNode", "[", 
                 RowBox[{"headAimVector", ",", "fixate", ",", "base"}], "]"}],
                 ",", 
                RowBox[{
                 RowBox[{"rightEyeCenter", "[", 
                  RowBox[{"headAimVector", ",", "base"}], "]"}], "-", 
                 RowBox[{"eyeRadius", "*", 
                  RowBox[{
                   RowBox[{"rightViewVector", "[", 
                    RowBox[{"headAimVector", ",", "fixate", ",", "base"}], 
                    "]"}], "/", 
                   RowBox[{"nORM", "[", 
                    RowBox[{"rightViewVector", "[", 
                    RowBox[{"headAimVector", ",", "fixate", ",", "base"}], 
                    "]"}], "]"}]}]}]}]}], "}"}], "]"}], "}"}]}], "}"}], "]"}],
          ",", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Red", ",", 
            RowBox[{"{", 
             RowBox[{"Line", "[", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"leftEyeNode", "[", 
                 RowBox[{"headAimVector", ",", "fixate", ",", "base"}], "]"}],
                 ",", 
                RowBox[{
                 RowBox[{"leftEyeCenter", "[", 
                  RowBox[{"headAimVector", ",", " ", "base"}], "]"}], "-", 
                 RowBox[{"eyeRadius", "*", 
                  RowBox[{
                   RowBox[{"leftViewVector", "[", 
                    RowBox[{"headAimVector", ",", "fixate", ",", "base"}], 
                    "]"}], "/", 
                   RowBox[{"nORM", "[", 
                    RowBox[{"leftViewVector", "[", 
                    RowBox[{"headAimVector", ",", "fixate", ",", "base"}], 
                    "]"}], "]"}]}]}]}]}], "}"}], "]"}], "}"}]}], "}"}], "]"}],
          ",", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Blue", ",", 
            RowBox[{"{", 
             RowBox[{"Line", "[", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"rightEyeNode", "[", 
                 RowBox[{"headAimVector", ",", "fixate", ",", "base"}], "]"}],
                 ",", 
                RowBox[{
                 RowBox[{"rightEyeCenter", "[", 
                  RowBox[{"headAimVector", ",", "base"}], "]"}], "+", 
                 RowBox[{"eyeEdge", "[", 
                  RowBox[{
                   RowBox[{"right\[CurlyPhi]", "[", 
                    RowBox[{"headAimVector", ",", "fixate", ",", "base"}], 
                    "]"}], "-", "\[Pi]", "-", 
                   RowBox[{"right\[Theta]inside", "[", 
                    RowBox[{
                    "headAimVector", ",", "fixate", ",", "distraction", ",", 
                    "base"}], "]"}]}], "]"}]}]}], "}"}], "]"}], "}"}]}], 
           "}"}], "]"}], ",", "\[IndentingNewLine]", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Blue", ",", 
            RowBox[{"{", 
             RowBox[{"Line", "[", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"leftEyeNode", "[", 
                 RowBox[{"headAimVector", ",", "fixate", ",", "base"}], "]"}],
                 ",", 
                RowBox[{
                 RowBox[{"leftEyeCenter", "[", 
                  RowBox[{"headAimVector", ",", "base"}], "]"}], "+", 
                 RowBox[{"eyeEdge", "[", 
                  RowBox[{
                   RowBox[{"left\[CurlyPhi]", "[", 
                    RowBox[{"headAimVector", ",", "fixate", ",", "base"}], 
                    "]"}], "-", "\[Pi]", "-", 
                   RowBox[{"left\[Theta]inside", "[", 
                    RowBox[{
                    "headAimVector", ",", "fixate", ",", "distraction", ",", 
                    "base"}], "]"}]}], "]"}]}]}], "}"}], "]"}], "}"}]}], 
           "}"}], "]"}], ",", "\[IndentingNewLine]", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Purple", ",", 
            RowBox[{"{", 
             RowBox[{"Line", "[", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"rightEyeCenter", "[", 
                 RowBox[{"headAimVector", ",", "base"}], "]"}], ",", 
                RowBox[{
                 RowBox[{"rightEyeCenter", "[", 
                  RowBox[{"headAimVector", ",", "base"}], "]"}], "+", 
                 RowBox[{"eyeEdge", "[", 
                  RowBox[{
                   RowBox[{"right\[CurlyPhi]", "[", 
                    RowBox[{"headAimVector", ",", "fixate", ",", "base"}], 
                    "]"}], "-", "\[Pi]", "-", 
                   RowBox[{"right\[Theta]inside", "[", 
                    RowBox[{
                    "headAimVector", ",", "fixate", ",", "distraction", ",", 
                    "base"}], "]"}]}], "]"}]}]}], "}"}], "]"}], "}"}]}], 
           "}"}], "]"}], ",", "\[IndentingNewLine]", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Purple", ",", 
            RowBox[{"{", 
             RowBox[{"Line", "[", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{"leftEyeCenter", "[", 
                 RowBox[{"headAimVector", ",", "base"}], "]"}], ",", 
                RowBox[{
                 RowBox[{"leftEyeCenter", "[", 
                  RowBox[{"headAimVector", ",", "base"}], "]"}], "+", 
                 RowBox[{"eyeEdge", "[", 
                  RowBox[{
                   RowBox[{"left\[CurlyPhi]", "[", 
                    RowBox[{"headAimVector", ",", "fixate", ",", "base"}], 
                    "]"}], "-", "\[Pi]", "-", 
                   RowBox[{"left\[Theta]inside", "[", 
                    RowBox[{
                    "headAimVector", ",", "fixate", ",", "distraction", ",", 
                    "base"}], "]"}]}], "]"}]}]}], "}"}], "]"}], "}"}]}], 
           "}"}], "]"}], ",", "\[IndentingNewLine]", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Green", ",", 
            RowBox[{"{", 
             RowBox[{"Line", "[", 
              RowBox[{"N", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"leftEyeCenter", "[", 
                  RowBox[{"headAimVector", ",", " ", "base"}], "]"}], ",", 
                 RowBox[{"rightEyeCenter", "[", 
                  RowBox[{"headAimVector", ",", " ", "base"}], "]"}]}], "}"}],
                "]"}], "]"}], "}"}]}], "}"}], "]"}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"min", "=", 
           RowBox[{"Min", "[", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{
               RowBox[{"left\[CurlyPhi]", "[", 
                RowBox[{"headAimVector", ",", "fixate", ",", "base"}], "]"}], 
               "-", "\[Pi]", "-", 
               RowBox[{"left\[Theta]inside", "[", 
                RowBox[{
                "headAimVector", ",", "fixate", ",", "distraction", ",", 
                 "base"}], "]"}], "-", 
               RowBox[{"retinalDisparity", "[", 
                RowBox[{
                "headAimVector", ",", "fixate", ",", "distraction", ",", 
                 "base"}], "]"}]}], ",", 
              RowBox[{
               RowBox[{"left\[CurlyPhi]", "[", 
                RowBox[{"headAimVector", ",", "fixate", ",", "base"}], "]"}], 
               "-", "\[Pi]", "-", 
               RowBox[{"left\[Theta]inside", "[", 
                RowBox[{
                "headAimVector", ",", "fixate", ",", "distraction", ",", 
                 "base"}], "]"}]}]}], "}"}], "]"}]}], ";", 
          "\[IndentingNewLine]", 
          RowBox[{"MAX", "=", 
           RowBox[{"Max", "[", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{
               RowBox[{"left\[CurlyPhi]", "[", 
                RowBox[{"headAimVector", ",", "fixate", ",", "base"}], "]"}], 
               "-", "\[Pi]", "-", 
               RowBox[{"left\[Theta]inside", "[", 
                RowBox[{
                "headAimVector", ",", "fixate", ",", "distraction", ",", 
                 "base"}], "]"}], "-", 
               RowBox[{"retinalDisparity", "[", 
                RowBox[{
                "headAimVector", ",", "fixate", ",", "distraction", ",", 
                 "base"}], "]"}]}], ",", 
              RowBox[{
               RowBox[{"left\[CurlyPhi]", "[", 
                RowBox[{"headAimVector", ",", "fixate", ",", "base"}], "]"}], 
               "-", "\[Pi]", "-", 
               RowBox[{"left\[Theta]inside", "[", 
                RowBox[{
                "headAimVector", ",", "fixate", ",", "distraction", ",", 
                 "base"}], "]"}]}]}], "}"}], "]"}]}], ";", 
          "\[IndentingNewLine]", 
          RowBox[{"Graphics", "[", 
           RowBox[{"{", 
            RowBox[{"Brown", ",", 
             RowBox[{"AbsoluteThickness", "[", "3", "]"}], ",", 
             RowBox[{"{", 
              RowBox[{"Circle", "[", 
               RowBox[{
                RowBox[{"leftEyeCenter", "[", 
                 RowBox[{"headAimVector", ",", " ", "base"}], "]"}], ",", 
                "eyeRadius", ",", 
                RowBox[{"{", 
                 RowBox[{"min", ",", "MAX"}], "}"}]}], "]"}], "}"}]}], "}"}], 
           "]"}]}]}], "\[IndentingNewLine]", "}"}], ",", 
       RowBox[{"AspectRatio", "\[Rule]", "Automatic"}], ",", 
       RowBox[{"PlotRange", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"midPoint", "[", 
               RowBox[{"headAimVector", ",", "base"}], "]"}], "[", 
              RowBox[{"[", "1", "]"}], "]"}], "-", "interOcular"}], ",", 
            RowBox[{
             RowBox[{
              RowBox[{"midPoint", "[", 
               RowBox[{"headAimVector", ",", "base"}], "]"}], "[", 
              RowBox[{"[", "1", "]"}], "]"}], "+", "interOcular"}]}], "}"}], 
          ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"midPoint", "[", 
               RowBox[{"headAimVector", ",", "base"}], "]"}], "[", 
              RowBox[{"[", "2", "]"}], "]"}], "-", "interOcular"}], ",", 
            RowBox[{
             RowBox[{
              RowBox[{"midPoint", "[", 
               RowBox[{"headAimVector", ",", "base"}], "]"}], "[", 
              RowBox[{"[", "2", "]"}], "]"}], "+", "interOcular"}]}], "}"}]}],
          "}"}]}]}], "]"}]}], ";", "\[IndentingNewLine]", 
    "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"outRays", "[", 
      RowBox[{
       RowBox[{"headAimVector_", "?", "VectorQ"}], ",", 
       RowBox[{"fixate_", "?", "VectorQ"}], ",", 
       RowBox[{"distraction_", "?", "VectorQ"}], ",", 
       RowBox[{"base_", "?", "VectorQ"}]}], "]"}], ":=", 
     RowBox[{"Show", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Black", ",", 
            RowBox[{"AbsolutePointSize", "[", "4", "]"}], ",", 
            RowBox[{"{", 
             RowBox[{"Point", "[", 
              RowBox[{"N", "[", "fixate", "]"}], "]"}], "}"}]}], "}"}], "]"}],
          ",", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Black", ",", 
            RowBox[{"AbsolutePointSize", "[", "4", "]"}], ",", 
            RowBox[{"{", 
             RowBox[{"Point", "[", 
              RowBox[{"N", "[", "distraction", "]"}], "]"}], "}"}]}], "}"}], 
          "]"}], ",", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Black", ",", 
            RowBox[{"{", 
             RowBox[{"Line", "[", 
              RowBox[{"N", "[", 
               RowBox[{"{", 
                RowBox[{"fixate", ",", 
                 RowBox[{"rightEyeNode", "[", 
                  RowBox[{"headAimVector", ",", "fixate", ",", " ", "base"}], 
                  "]"}]}], "}"}], "]"}], "]"}], "}"}]}], "}"}], "]"}], ",", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Black", ",", 
            RowBox[{"{", 
             RowBox[{"Line", "[", 
              RowBox[{"N", "[", 
               RowBox[{"{", 
                RowBox[{"fixate", ",", 
                 RowBox[{"leftEyeNode", "[", 
                  RowBox[{"headAimVector", ",", "fixate", ",", " ", "base"}], 
                  "]"}]}], "}"}], "]"}], "]"}], "}"}]}], "}"}], "]"}], ",", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Black", ",", 
            RowBox[{"{", 
             RowBox[{"Line", "[", 
              RowBox[{"N", "[", 
               RowBox[{"{", 
                RowBox[{"distraction", ",", 
                 RowBox[{"rightEyeNode", "[", 
                  RowBox[{"headAimVector", ",", "fixate", ",", " ", "base"}], 
                  "]"}]}], "}"}], "]"}], "]"}], "}"}]}], "}"}], "]"}], ",", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Black", ",", 
            RowBox[{"{", 
             RowBox[{"Line", "[", 
              RowBox[{"N", "[", 
               RowBox[{"{", 
                RowBox[{"distraction", ",", 
                 RowBox[{"leftEyeNode", "[", 
                  RowBox[{"headAimVector", ",", "fixate", ",", " ", "base"}], 
                  "]"}]}], "}"}], "]"}], "]"}], "}"}]}], "}"}], "]"}]}], 
        "\[IndentingNewLine]", "}"}], ",", 
       RowBox[{"AspectRatio", "\[Rule]", "Automatic"}]}], "]"}]}], ";", 
    "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{"(*", " ", 
     RowBox[{"real", " ", "program", " ", "starts", " ", "here"}], " ", 
     "*)"}], "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"right\[Beta]Ratio", "[", 
      RowBox[{"f_", ",", "d1_", ",", "d2_", ",", "tt_"}], "]"}], "=", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         SuperscriptBox["d1", "2"], " ", 
         RowBox[{"(", 
          RowBox[{"f", "+", 
           RowBox[{"n", " ", "r"}]}], ")"}], " ", 
         SqrtBox[
          RowBox[{
           SuperscriptBox[
            RowBox[{"(", 
             RowBox[{"f", "+", 
              RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
           RowBox[{
            SuperscriptBox["i", "2"], " ", 
            SuperscriptBox["tt", "2"]}]}]]}], "+", 
        RowBox[{"d1", " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{
            RowBox[{"-", "n"}], " ", "r", " ", 
            SuperscriptBox[
             RowBox[{"(", 
              RowBox[{"f", "+", 
               RowBox[{"n", " ", "r"}]}], ")"}], "2"]}], "+", 
           RowBox[{
            SuperscriptBox["f", "2"], " ", 
            SqrtBox[
             RowBox[{
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"f", "+", 
                 RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
              RowBox[{
               SuperscriptBox["i", "2"], " ", 
               SuperscriptBox["tt", "2"]}]}]]}], "+", 
           RowBox[{"2", " ", "f", " ", "n", " ", "r", " ", 
            SqrtBox[
             RowBox[{
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"f", "+", 
                 RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
              RowBox[{
               SuperscriptBox["i", "2"], " ", 
               SuperscriptBox["tt", "2"]}]}]]}], "+", 
           RowBox[{
            SuperscriptBox["n", "2"], " ", 
            SuperscriptBox["r", "2"], " ", 
            SqrtBox[
             RowBox[{
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"f", "+", 
                 RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
              RowBox[{
               SuperscriptBox["i", "2"], " ", 
               SuperscriptBox["tt", "2"]}]}]]}], "-", 
           RowBox[{
            SuperscriptBox["i", "2"], " ", 
            SuperscriptBox["tt", "2"], " ", 
            SqrtBox[
             RowBox[{
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"f", "+", 
                 RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
              RowBox[{
               SuperscriptBox["i", "2"], " ", 
               SuperscriptBox["tt", "2"]}]}]]}]}], ")"}]}], "+", 
        RowBox[{"d2", " ", 
         RowBox[{"(", 
          RowBox[{"f", "+", 
           RowBox[{"n", " ", "r"}]}], ")"}], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"d2", " ", 
            SqrtBox[
             RowBox[{
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"f", "+", 
                 RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
              RowBox[{
               SuperscriptBox["i", "2"], " ", 
               SuperscriptBox["tt", "2"]}]}]]}], "+", 
           RowBox[{"i", " ", "tt", " ", 
            RowBox[{"(", 
             RowBox[{
              RowBox[{
               RowBox[{"-", "n"}], " ", "r"}], "+", 
              RowBox[{"2", " ", 
               SqrtBox[
                RowBox[{
                 SuperscriptBox[
                  RowBox[{"(", 
                   RowBox[{"f", "+", 
                    RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
                 RowBox[{
                  SuperscriptBox["i", "2"], " ", 
                  SuperscriptBox["tt", "2"]}]}]]}]}], ")"}]}]}], ")"}]}]}], 
       ")"}], "/", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"d1", " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{
            RowBox[{"-", "n"}], " ", "r", " ", 
            SuperscriptBox[
             RowBox[{"(", 
              RowBox[{"f", "+", 
               RowBox[{"n", " ", "r"}]}], ")"}], "2"]}], "+", 
           RowBox[{
            SuperscriptBox["f", "2"], " ", 
            SqrtBox[
             RowBox[{
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"f", "+", 
                 RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
              RowBox[{
               SuperscriptBox["i", "2"], " ", 
               SuperscriptBox["tt", "2"]}]}]]}], "+", 
           RowBox[{"2", " ", "f", " ", "n", " ", "r", " ", 
            SqrtBox[
             RowBox[{
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"f", "+", 
                 RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
              RowBox[{
               SuperscriptBox["i", "2"], " ", 
               SuperscriptBox["tt", "2"]}]}]]}], "+", 
           RowBox[{
            SuperscriptBox["n", "2"], " ", 
            SuperscriptBox["r", "2"], " ", 
            SqrtBox[
             RowBox[{
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"f", "+", 
                 RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
              RowBox[{
               SuperscriptBox["i", "2"], " ", 
               SuperscriptBox["tt", "2"]}]}]]}], "+", 
           RowBox[{
            SuperscriptBox["i", "2"], " ", 
            SuperscriptBox["tt", "2"], " ", 
            SqrtBox[
             RowBox[{
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"f", "+", 
                 RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
              RowBox[{
               SuperscriptBox["i", "2"], " ", 
               SuperscriptBox["tt", "2"]}]}]]}]}], ")"}]}], "+", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"f", "+", 
           RowBox[{"n", " ", "r"}]}], ")"}], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{
            RowBox[{"-", "2"}], " ", 
            SuperscriptBox["n", "3"], " ", 
            SuperscriptBox["r", "3"]}], "-", 
           RowBox[{"i", " ", "n", " ", "r", " ", "tt", " ", 
            RowBox[{"(", 
             RowBox[{"d2", "+", 
              RowBox[{"2", " ", "i", " ", "tt"}]}], ")"}]}], "+", 
           RowBox[{"2", " ", 
            SuperscriptBox["n", "2"], " ", 
            SuperscriptBox["r", "2"], " ", 
            SqrtBox[
             RowBox[{
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"f", "+", 
                 RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
              RowBox[{
               SuperscriptBox["i", "2"], " ", 
               SuperscriptBox["tt", "2"]}]}]]}], "+", 
           RowBox[{
            SuperscriptBox["i", "2"], " ", 
            SuperscriptBox["tt", "2"], " ", 
            SqrtBox[
             RowBox[{
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"f", "+", 
                 RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
              RowBox[{
               SuperscriptBox["i", "2"], " ", 
               SuperscriptBox["tt", "2"]}]}]]}], "+", 
           RowBox[{
            SuperscriptBox["f", "2"], " ", 
            RowBox[{"(", 
             RowBox[{
              RowBox[{
               RowBox[{"-", "2"}], " ", "n", " ", "r"}], "+", 
              SqrtBox[
               RowBox[{
                SuperscriptBox[
                 RowBox[{"(", 
                  RowBox[{"f", "+", 
                   RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
                RowBox[{
                 SuperscriptBox["i", "2"], " ", 
                 SuperscriptBox["tt", "2"]}]}]]}], ")"}]}], "+", 
           RowBox[{"2", " ", "f", " ", "n", " ", "r", " ", 
            RowBox[{"(", 
             RowBox[{
              RowBox[{
               RowBox[{"-", "2"}], " ", "n", " ", "r"}], "+", 
              SqrtBox[
               RowBox[{
                SuperscriptBox[
                 RowBox[{"(", 
                  RowBox[{"f", "+", 
                   RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
                RowBox[{
                 SuperscriptBox["i", "2"], " ", 
                 SuperscriptBox["tt", "2"]}]}]]}], ")"}]}]}], ")"}]}]}], 
       ")"}]}]}], ";", "\n", 
    RowBox[{
     RowBox[{"t1", "[", 
      RowBox[{"f_", ",", "d1_", ",", "d2_"}], "]"}], ":=", 
     FractionBox[
      RowBox[{
       RowBox[{"-", 
        SuperscriptBox["d1", "2"]}], " ", "-", 
       SuperscriptBox["d2", "2"], " ", "-", 
       RowBox[{"2", " ", "d1", " ", "f"}], " ", "-", 
       RowBox[{
        SqrtBox[
         RowBox[{
          SuperscriptBox["d1", "2"], "+", 
          SuperscriptBox["d2", "2"]}]], " ", 
        SqrtBox[
         RowBox[{
          SuperscriptBox["d1", "2"], "+", 
          SuperscriptBox["d2", "2"], "+", 
          RowBox[{"4", " ", "d1", " ", "f"}], "+", 
          RowBox[{"4", " ", 
           SuperscriptBox["f", "2"]}]}]], " "}]}], 
      RowBox[{"2", " ", "d2", " ", "i"}]]}], ";", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"t2", "[", 
      RowBox[{"f_", ",", "d1_", ",", "d2_"}], "]"}], ":=", 
     FractionBox[
      RowBox[{
       RowBox[{"-", 
        SuperscriptBox["d1", "2"]}], " ", "-", 
       SuperscriptBox["d2", "2"], " ", "-", 
       RowBox[{"2", " ", "d1", " ", "f"}], " ", "+", 
       RowBox[{
        SqrtBox[
         RowBox[{
          SuperscriptBox["d1", "2"], "+", 
          SuperscriptBox["d2", "2"]}]], " ", 
        SqrtBox[
         RowBox[{
          SuperscriptBox["d1", "2"], "+", 
          SuperscriptBox["d2", "2"], "+", 
          RowBox[{"4", " ", "d1", " ", "f"}], "+", 
          RowBox[{"4", " ", 
           SuperscriptBox["f", "2"]}]}]], " "}]}], 
      RowBox[{"2", " ", "d2", " ", "i"}]]}], ";", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"tC", "[", 
      RowBox[{"f_", ",", "d1_", ",", "d2_"}], "]"}], ":=", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Abs", "[", 
         RowBox[{"N", "[", "d2", "]"}], "]"}], ">", ".01"}], ",", 
       RowBox[{"If", "[", 
        RowBox[{
         RowBox[{"d1", "\[GreaterEqual]", "0"}], ",", 
         RowBox[{"t2", "[", 
          RowBox[{"f", ",", "d1", ",", "d2"}], "]"}], ",", 
         RowBox[{"t1", "[", 
          RowBox[{"f", ",", "d1", ",", "d2"}], "]"}]}], "]"}], ",", "0"}], 
      "]"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"rightPeak\[Beta]Ratio", "[", 
      RowBox[{"f_", ",", "d1_", ",", "d2_"}], "]"}], ":=", 
     RowBox[{"right\[Beta]Ratio", "[", 
      RowBox[{"f", ",", "d1", ",", "d2", ",", 
       RowBox[{"tC", "[", 
        RowBox[{"f", ",", "d1", ",", "d2"}], "]"}]}], "]"}]}], ";", 
    "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"right\[Theta]Derivative", "[", 
      RowBox[{"f_", ",", "d1_", ",", "d2_", ",", "tt_"}], "]"}], "=", 
     RowBox[{
      RowBox[{"-", 
       RowBox[{"(", 
        RowBox[{"i", " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{
            RowBox[{"-", 
             SuperscriptBox["d1", "2"]}], " ", 
            RowBox[{"(", 
             RowBox[{"f", "+", 
              RowBox[{"n", " ", "r"}]}], ")"}], " ", 
            SqrtBox[
             RowBox[{
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"f", "+", 
                 RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
              RowBox[{
               SuperscriptBox["i", "2"], " ", 
               SuperscriptBox["tt", "2"]}]}]]}], "+", 
           RowBox[{"d1", " ", 
            RowBox[{"(", 
             RowBox[{
              RowBox[{"n", " ", "r", " ", 
               SuperscriptBox[
                RowBox[{"(", 
                 RowBox[{"f", "+", 
                  RowBox[{"n", " ", "r"}]}], ")"}], "2"]}], "-", 
              RowBox[{
               SuperscriptBox["f", "2"], " ", 
               SqrtBox[
                RowBox[{
                 SuperscriptBox[
                  RowBox[{"(", 
                   RowBox[{"f", "+", 
                    RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
                 RowBox[{
                  SuperscriptBox["i", "2"], " ", 
                  SuperscriptBox["tt", "2"]}]}]]}], "-", 
              RowBox[{"2", " ", "f", " ", "n", " ", "r", " ", 
               SqrtBox[
                RowBox[{
                 SuperscriptBox[
                  RowBox[{"(", 
                   RowBox[{"f", "+", 
                    RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
                 RowBox[{
                  SuperscriptBox["i", "2"], " ", 
                  SuperscriptBox["tt", "2"]}]}]]}], "-", 
              RowBox[{
               SuperscriptBox["n", "2"], " ", 
               SuperscriptBox["r", "2"], " ", 
               SqrtBox[
                RowBox[{
                 SuperscriptBox[
                  RowBox[{"(", 
                   RowBox[{"f", "+", 
                    RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
                 RowBox[{
                  SuperscriptBox["i", "2"], " ", 
                  SuperscriptBox["tt", "2"]}]}]]}], "+", 
              RowBox[{
               SuperscriptBox["i", "2"], " ", 
               SuperscriptBox["tt", "2"], " ", 
               SqrtBox[
                RowBox[{
                 SuperscriptBox[
                  RowBox[{"(", 
                   RowBox[{"f", "+", 
                    RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
                 RowBox[{
                  SuperscriptBox["i", "2"], " ", 
                  SuperscriptBox["tt", "2"]}]}]]}]}], ")"}]}], "-", 
           RowBox[{"d2", " ", 
            RowBox[{"(", 
             RowBox[{"f", "+", 
              RowBox[{"n", " ", "r"}]}], ")"}], " ", 
            RowBox[{"(", 
             RowBox[{
              RowBox[{"d2", " ", 
               SqrtBox[
                RowBox[{
                 SuperscriptBox[
                  RowBox[{"(", 
                   RowBox[{"f", "+", 
                    RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
                 RowBox[{
                  SuperscriptBox["i", "2"], " ", 
                  SuperscriptBox["tt", "2"]}]}]]}], "+", 
              RowBox[{"i", " ", "tt", " ", 
               RowBox[{"(", 
                RowBox[{
                 RowBox[{
                  RowBox[{"-", "n"}], " ", "r"}], "+", 
                 RowBox[{"2", " ", 
                  SqrtBox[
                   RowBox[{
                    SuperscriptBox[
                    RowBox[{"(", 
                    RowBox[{"f", "+", 
                    RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
                    RowBox[{
                    SuperscriptBox["i", "2"], " ", 
                    SuperscriptBox["tt", "2"]}]}]]}]}], ")"}]}]}], ")"}]}]}], 
          ")"}]}], ")"}]}], "/", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"f", "+", 
             RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
          RowBox[{
           SuperscriptBox["i", "2"], " ", 
           SuperscriptBox["tt", "2"]}]}], ")"}], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           RowBox[{"-", "2"}], " ", 
           SuperscriptBox["n", "3"], " ", 
           SuperscriptBox["r", "3"]}], "-", 
          RowBox[{"2", " ", "i", " ", "n", " ", "r", " ", "tt", " ", 
           RowBox[{"(", 
            RowBox[{"d2", "+", 
             RowBox[{"i", " ", "tt"}]}], ")"}]}], "+", 
          RowBox[{
           SuperscriptBox["d1", "2"], " ", 
           SqrtBox[
            RowBox[{
             SuperscriptBox[
              RowBox[{"(", 
               RowBox[{"f", "+", 
                RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
             RowBox[{
              SuperscriptBox["i", "2"], " ", 
              SuperscriptBox["tt", "2"]}]}]]}], "+", 
          RowBox[{"2", " ", 
           SuperscriptBox["n", "2"], " ", 
           SuperscriptBox["r", "2"], " ", 
           SqrtBox[
            RowBox[{
             SuperscriptBox[
              RowBox[{"(", 
               RowBox[{"f", "+", 
                RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
             RowBox[{
              SuperscriptBox["i", "2"], " ", 
              SuperscriptBox["tt", "2"]}]}]]}], "+", 
          RowBox[{
           SuperscriptBox[
            RowBox[{"(", 
             RowBox[{"d2", "+", 
              RowBox[{"i", " ", "tt"}]}], ")"}], "2"], " ", 
           SqrtBox[
            RowBox[{
             SuperscriptBox[
              RowBox[{"(", 
               RowBox[{"f", "+", 
                RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
             RowBox[{
              SuperscriptBox["i", "2"], " ", 
              SuperscriptBox["tt", "2"]}]}]]}], "+", 
          RowBox[{
           SuperscriptBox["f", "2"], " ", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{
              RowBox[{"-", "2"}], " ", "n", " ", "r"}], "+", 
             SqrtBox[
              RowBox[{
               SuperscriptBox[
                RowBox[{"(", 
                 RowBox[{"f", "+", 
                  RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
               RowBox[{
                SuperscriptBox["i", "2"], " ", 
                SuperscriptBox["tt", "2"]}]}]]}], ")"}]}], "+", 
          RowBox[{"2", " ", "f", " ", "n", " ", "r", " ", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{
              RowBox[{"-", "2"}], " ", "n", " ", "r"}], "+", 
             SqrtBox[
              RowBox[{
               SuperscriptBox[
                RowBox[{"(", 
                 RowBox[{"f", "+", 
                  RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
               RowBox[{
                SuperscriptBox["i", "2"], " ", 
                SuperscriptBox["tt", "2"]}]}]]}], ")"}]}], "+", 
          RowBox[{"2", " ", "d1", " ", 
           RowBox[{"(", 
            RowBox[{"f", "+", 
             RowBox[{"n", " ", "r"}]}], ")"}], " ", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{
              RowBox[{"-", "n"}], " ", "r"}], "+", 
             SqrtBox[
              RowBox[{
               SuperscriptBox[
                RowBox[{"(", 
                 RowBox[{"f", "+", 
                  RowBox[{"n", " ", "r"}]}], ")"}], "2"], "+", 
               RowBox[{
                SuperscriptBox["i", "2"], " ", 
                SuperscriptBox["tt", "2"]}]}]]}], ")"}]}]}], ")"}]}], 
       ")"}]}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{"\[Phi]", "=", 
     RowBox[{"\[Phi]D", " ", 
      RowBox[{"\[Pi]", "/", "180"}]}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"distractor", ":=", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{"f", "+", "d"}], ")"}], "/", "2"}], 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"Sin", "[", "\[Phi]", "]"}], ",", 
         RowBox[{"Cos", "[", "\[Phi]", "]"}]}], "}"}]}], "+", 
      RowBox[{"{", 
       RowBox[{"0", ",", 
        RowBox[{
         RowBox[{"n", " ", "r"}], " ", "+", 
         RowBox[{
          RowBox[{"(", " ", 
           RowBox[{"f", "+", "d"}], ")"}], "/", "2"}]}]}], "}"}]}]}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{"r", "=", 
     RowBox[{"eyeRadius", "=", "1.3"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"nodePercent", "=", "n"}], ";", "\[IndentingNewLine]", 
    RowBox[{"i", "=", 
     RowBox[{"interOcular", "=", "6.5"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"aimOffset", "=", "0"}], ";", "\[IndentingNewLine]", 
    RowBox[{"d1", ":=", 
     RowBox[{
      RowBox[{"distractor", "[", 
       RowBox[{"[", "2", "]"}], "]"}], "-", 
      RowBox[{"(", 
       RowBox[{"f", "+", 
        RowBox[{"n", " ", "r"}]}], ")"}]}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"d2", ":=", 
     RowBox[{"distractor", "[", 
      RowBox[{"[", "1", "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"fixate", "=", 
     RowBox[{"{", 
      RowBox[{"0", ",", 
       RowBox[{"f", "+", 
        RowBox[{"n", " ", "r"}]}]}], "}"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"headAimVector", "=", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"base", "[", "tt_", "]"}], "=", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"interOcular", "*", 
        RowBox[{"(", "tt", ")"}]}], ",", "0"}], "}"}]}], ";", 
    "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{"t", "=", "0"}], ";", "\[IndentingNewLine]", 
    RowBox[{"Quiet", "@", 
     RowBox[{"Show", "[", 
      RowBox[{
       RowBox[{"{", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Green", ",", 
            RowBox[{"AbsolutePointSize", "[", "5", "]"}], ",", 
            "\[IndentingNewLine]", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"Line", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"base", "[", "t", "]"}], ",", 
                 RowBox[{
                  RowBox[{"base", "[", "t", "]"}], "+", 
                  RowBox[{"scale", 
                   RowBox[{"{", 
                    RowBox[{".4", ",", "0"}], "}"}]}]}], ",", 
                 RowBox[{
                  RowBox[{"base", "[", "t", "]"}], "+", 
                  RowBox[{"scale", 
                   RowBox[{"{", 
                    RowBox[{".3", ",", 
                    RowBox[{"-", ".015"}]}], "}"}]}]}]}], "}"}], "]"}], 
              "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
              RowBox[{"Text", "[", 
               RowBox[{"\"\<B\>\"", ",", 
                RowBox[{
                 RowBox[{"base", "[", "t", "]"}], "-", 
                 RowBox[{"scale", "*", 
                  RowBox[{"{", 
                   RowBox[{"0", ",", "0.05"}], "}"}]}]}]}], "]"}]}], "}"}]}], 
           "}"}], "]"}], "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", "\[IndentingNewLine]", 
           RowBox[{
            RowBox[{"Text", "[", 
             RowBox[{
              RowBox[{"Row", "[", 
               RowBox[{"{", 
                RowBox[{"\[Delta]", ",", "\"\< = \>\"", ",", 
                 RowBox[{"Round", "[", 
                  RowBox[{
                   RowBox[{"N", "[", 
                    RowBox[{"arcMinutes", "[", 
                    RowBox[{"binocularDisparity", "[", 
                    RowBox[{
                    "headAimVector", ",", "fixate", ",", "distractor", ",", 
                    RowBox[{"base", "[", "t", "]"}]}], "]"}], "]"}], "]"}], 
                   ",", ".001"}], "]"}], ",", "\"\< min\>\""}], "}"}], "]"}], 
              ",", 
              RowBox[{"scale", "*", 
               RowBox[{"{", 
                RowBox[{".35", ",", ".1"}], "}"}]}]}], "]"}], 
            "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
            RowBox[{"Text", "[", 
             RowBox[{
              RowBox[{"Row", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"HoldForm", "[", 
                  FractionBox[
                   RowBox[{
                    RowBox[{"Style", "[", 
                    RowBox[{"\"\<d\>\"", ",", "Italic"}], "]"}], 
                    "\[InvisibleSpace]", "\[Theta]"}], 
                   RowBox[{
                    RowBox[{"Style", "[", 
                    RowBox[{"\"\<d\>\"", ",", "Italic"}], "]"}], 
                    "\[InvisibleSpace]", "t"}]], "]"}], ",", "\"\< = \>\"", 
                 ",", 
                 RowBox[{"Round", "[", 
                  RowBox[{
                   RowBox[{"arcMinutes", "[", 
                    RowBox[{"right\[Theta]Derivative", "[", 
                    RowBox[{"f", ",", "d1", ",", "d2", ",", "t"}], "]"}], 
                    "]"}], ",", ".01"}], "]"}], ",", "\"\< min/sec\>\""}], 
                "}"}], "]"}], ",", 
              RowBox[{"scale", "*", 
               RowBox[{"{", 
                RowBox[{".4", ",", ".4"}], "}"}]}]}], "]"}], 
            "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
            RowBox[{"Text", "[", 
             RowBox[{
              RowBox[{"Row", "[", 
               RowBox[{"{", 
                RowBox[{
                "\"\<\!\(\*SubscriptBox[\(\[Kappa]\), \(F\)]\) = \>\"", ",", 
                 RowBox[{"Round", "[", 
                  RowBox[{
                   RowBox[{"arcMinutes", "[", 
                    RowBox[{"fixateConvergence", "[", 
                    RowBox[{"headAimVector", ",", "fixate", ",", 
                    RowBox[{"base", "[", "t", "]"}]}], "]"}], "]"}], ",", 
                   ".01"}], "]"}], ",", "\"\< min\>\""}], "}"}], "]"}], ",", 
              RowBox[{"scale", "*", 
               RowBox[{"{", 
                RowBox[{".35", ",", ".9"}], "}"}]}]}], "]"}], 
            "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
            RowBox[{"Text", "[", 
             RowBox[{
              RowBox[{"Row", "[", 
               RowBox[{"{", 
                RowBox[{
                "\"\<\!\(\*SubscriptBox[\(\[Kappa]\), \(F\)]\)/\>\"", ",", 
                 RowBox[{"HoldForm", "[", 
                  FractionBox[
                   RowBox[{
                    RowBox[{"Style", "[", 
                    RowBox[{"\"\<d\>\"", ",", "Italic"}], "]"}], 
                    "\[InvisibleSpace]", "\[Alpha]"}], 
                   RowBox[{
                    RowBox[{"Style", "[", 
                    RowBox[{"\"\<d\>\"", ",", "Italic"}], "]"}], 
                    "\[InvisibleSpace]", "t"}]], "]"}], ",", "\"\< = \>\"", 
                 ",", 
                 RowBox[{"Round", "[", 
                  RowBox[{
                   RowBox[{
                    RowBox[{"(", 
                    RowBox[{"f", "+", 
                    RowBox[{"n", " ", "r"}]}], ")"}], " ", 
                    RowBox[{
                    RowBox[{"fixateConvergence", "[", 
                    RowBox[{"headAimVector", ",", "fixate", ",", 
                    RowBox[{"base", "[", "t", "]"}]}], "]"}], "/", "i"}]}], 
                   ",", ".001"}], "]"}]}], "}"}], "]"}], ",", 
              RowBox[{"scale", "*", 
               RowBox[{"{", 
                RowBox[{".35", ",", ".75"}], "}"}]}]}], "]"}], 
            "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
            RowBox[{"Text", "[", 
             RowBox[{
              RowBox[{"Row", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"HoldForm", "[", 
                  FractionBox[
                   RowBox[{
                    RowBox[{"Style", "[", 
                    RowBox[{"\"\<d\>\"", ",", "Italic"}], "]"}], 
                    "\[InvisibleSpace]", "\[Alpha]"}], 
                   RowBox[{
                    RowBox[{"Style", "[", 
                    RowBox[{"\"\<d\>\"", ",", "Italic"}], "]"}], 
                    "\[InvisibleSpace]", "t"}]], "]"}], ",", "\"\< = \>\"", 
                 ",", 
                 RowBox[{"Round", "[", 
                  RowBox[{
                   RowBox[{"arcMinutes", "[", 
                    RowBox[{"i", "/", 
                    RowBox[{"(", 
                    RowBox[{"f", "+", 
                    RowBox[{"n", " ", "r"}]}], ")"}]}], "]"}], ",", ".01"}], 
                  "]"}], ",", "\"\< min/sec\>\""}], "}"}], "]"}], ",", 
              RowBox[{"scale", "*", 
               RowBox[{"{", 
                RowBox[{".4", ",", ".6"}], "}"}]}]}], "]"}], 
            "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
            RowBox[{"Text", "[", 
             RowBox[{
              RowBox[{"Row", "[", 
               RowBox[{"{", 
                RowBox[{"\"\<\[Delta]/\>\"", ",", 
                 RowBox[{"HoldForm", "[", 
                  FractionBox[
                   RowBox[{
                    RowBox[{"Style", "[", 
                    RowBox[{"\"\<d\>\"", ",", "Italic"}], "]"}], 
                    "\[InvisibleSpace]", "\[Theta]"}], 
                   RowBox[{
                    RowBox[{"Style", "[", 
                    RowBox[{"\"\<d\>\"", ",", "Italic"}], "]"}], 
                    "\[InvisibleSpace]", "t"}]], "]"}], ",", "\"\< = \>\"", 
                 ",", 
                 RowBox[{"Round", "[", 
                  RowBox[{
                   RowBox[{
                    RowBox[{"binocularDisparity", "[", 
                    RowBox[{
                    "headAimVector", ",", "fixate", ",", "distractor", ",", 
                    RowBox[{"base", "[", "t", "]"}]}], "]"}], "/", 
                    RowBox[{"right\[Theta]Derivative", "[", 
                    RowBox[{"f", ",", "d1", ",", "d2", ",", "t"}], "]"}]}], 
                   ",", ".001"}], "]"}]}], "}"}], "]"}], ",", 
              RowBox[{"scale", "*", 
               RowBox[{"{", 
                RowBox[{".35", ",", ".25"}], "}"}]}]}], "]"}], 
            "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
            RowBox[{"Text", "[", 
             RowBox[{
              RowBox[{"Row", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"HoldForm", "[", 
                  FractionBox["\"\<|D-B|-|F-B|\>\"", 
                   RowBox[{"Style", "[", 
                    RowBox[{"\"\<f\>\"", ",", "Italic"}], "]"}]], "]"}], ",", 
                 "\"\< = \>\"", ",", 
                 RowBox[{"Round", "[", 
                  RowBox[{
                   RowBox[{
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"nORM", "[", 
                    RowBox[{"distractor", "-", 
                    RowBox[{"base", "[", "0", "]"}]}], "]"}], "-", 
                    RowBox[{"nORM", "[", 
                    RowBox[{
                    RowBox[{"base", "[", "0", "]"}], "-", "fixate"}], "]"}]}],
                     ")"}], "/", "f"}], ",", ".01"}], "]"}]}], "}"}], "]"}], 
              ",", 
              RowBox[{"scale", "*", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"-", ".45"}], ",", ".55"}], "}"}]}]}], "]"}], 
            "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
            RowBox[{"Text", "[", 
             RowBox[{
              RowBox[{"Row", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"HoldForm", "[", 
                  FractionBox["\"\<|D-F|\>\"", 
                   RowBox[{"Style", "[", 
                    RowBox[{"\"\<f\>\"", ",", "Italic"}], "]"}]], "]"}], ",", 
                 "\"\< = \>\"", ",", 
                 RowBox[{"Round", "[", 
                  RowBox[{
                   RowBox[{
                    RowBox[{"nORM", "[", 
                    RowBox[{"distractor", "-", "fixate"}], "]"}], "/", "f"}], 
                   ",", ".01"}], "]"}]}], "}"}], "]"}], ",", 
              RowBox[{"scale", "*", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"-", ".5"}], ",", ".4"}], "}"}]}]}], "]"}], 
            "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
            RowBox[{"Text", "[", 
             RowBox[{
              RowBox[{"Row", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"HoldForm", "[", 
                  FractionBox[
                   RowBox[{
                    RowBox[{"Style", "[", 
                    RowBox[{"\"\<d\>\"", ",", "Italic"}], "]"}], 
                    "\[InvisibleSpace]", "\[Theta]"}], 
                   RowBox[{
                    RowBox[{"Style", "[", 
                    RowBox[{"\"\<d\>\"", ",", "Italic"}], "]"}], 
                    "\[InvisibleSpace]", "\[Beta]"}]], "]"}], ",", 
                 "\"\<(\!\(\*SubscriptBox[\(t\), \(c\)]\)) = \>\"", ",", 
                 RowBox[{"Round", "[", 
                  RowBox[{
                   RowBox[{"rightPeak\[Beta]Ratio", "[", 
                    RowBox[{"f", ",", "d1", ",", "d2"}], "]"}], ",", ".01"}], 
                  "]"}]}], "}"}], "]"}], ",", 
              RowBox[{"scale", "*", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"-", ".5"}], ",", ".25"}], "}"}]}]}], "]"}], 
            "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
            RowBox[{"Text", "[", 
             RowBox[{
              RowBox[{"Row", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"HoldForm", "[", 
                  FractionBox[
                   RowBox[{
                    RowBox[{"Style", "[", 
                    RowBox[{"\"\<d\>\"", ",", "Italic"}], "]"}], 
                    "\[InvisibleSpace]", "\[Theta]"}], 
                   RowBox[{
                    RowBox[{"Style", "[", 
                    RowBox[{"\"\<d\>\"", ",", "Italic"}], "]"}], 
                    "\[InvisibleSpace]", "\[Beta]"}]], "]"}], ",", 
                 "\"\<(0) = \>\"", ",", 
                 RowBox[{"Round", "[", 
                  RowBox[{
                   RowBox[{"right\[Beta]Ratio", "[", 
                    RowBox[{"f", ",", "d1", ",", "d2", ",", "0"}], "]"}], ",",
                    ".01"}], "]"}]}], "}"}], "]"}], ",", 
              RowBox[{"scale", "*", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"-", ".5"}], ",", ".1"}], "}"}]}]}], "]"}], 
            "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
            RowBox[{"Text", "[", 
             RowBox[{
              RowBox[{"StringForm", "[", 
               RowBox[{
               "\"\<\!\(\*FractionBox[\(Sin[\*FractionBox[\(\[Delta]\), \
\(2\)]]\), \(Cos[\*FractionBox[\(\[Kappa]\), \(2\)]]\\\ Sin[\*FractionBox[\(\
\[Kappa] - \[Delta]\), \(2\)]]\)]\) = `1`\>\"", ",", 
                RowBox[{"Round", "[", 
                 RowBox[{
                  RowBox[{
                   RowBox[{"Sin", "[", 
                    RowBox[{
                    FractionBox["1", "2"], 
                    RowBox[{"binocularDisparity", "[", 
                    RowBox[{
                    "headAimVector", ",", "fixate", ",", "distractor", ",", 
                    RowBox[{"base", "[", "t", "]"}]}], "]"}]}], "]"}], "/", 
                   RowBox[{"(", 
                    RowBox[{
                    RowBox[{"Cos", "[", 
                    RowBox[{
                    FractionBox["1", "2"], 
                    RowBox[{"fixateConvergence", "[", 
                    RowBox[{"headAimVector", ",", "fixate", ",", 
                    RowBox[{"base", "[", "t", "]"}]}], "]"}]}], "]"}], " ", 
                    RowBox[{"Sin", "[", 
                    RowBox[{
                    FractionBox["1", "2"], 
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"fixateConvergence", "[", 
                    RowBox[{"headAimVector", ",", "fixate", ",", 
                    RowBox[{"base", "[", "t", "]"}]}], "]"}], "-", 
                    RowBox[{"binocularDisparity", "[", 
                    RowBox[{
                    "headAimVector", ",", "fixate", ",", "distractor", ",", 
                    RowBox[{"base", "[", "t", "]"}]}], "]"}]}], ")"}]}], 
                    "]"}]}], ")"}]}], ",", ".01"}], "]"}]}], "]"}], ",", 
              RowBox[{"scale", "*", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"-", ".3"}], ",", 
                 RowBox[{"-", ".1"}]}], "}"}]}]}], "]"}]}], 
           "\[IndentingNewLine]", "}"}], "]"}], ",", "\[IndentingNewLine]", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Brown", ",", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"Line", "[", 
               RowBox[{"{", 
                RowBox[{"distractor", ",", "fixate"}], "}"}], "]"}], ",", 
              RowBox[{"Text", "[", 
               RowBox[{
                RowBox[{"Row", "[", 
                 RowBox[{"{", 
                  RowBox[{"\"\<|D-F| = \>\"", ",", 
                   RowBox[{"Round", "[", 
                    RowBox[{
                    RowBox[{"nORM", "[", 
                    RowBox[{"distractor", "-", "fixate"}], "]"}], ",", ".1"}],
                     "]"}], ",", "\"\< cm\>\""}], "}"}], "]"}], ",", 
                RowBox[{"scale", 
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{"-", ".5"}], ",", ".9"}], "}"}]}]}], "]"}]}], 
             "}"}]}], "}"}], "]"}], "\[IndentingNewLine]", ",", 
         "\[IndentingNewLine]", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Red", ",", 
            RowBox[{"AbsolutePointSize", "[", "4", "]"}], ",", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"Point", "[", "fixate", "]"}], ",", 
              RowBox[{"Text", "[", 
               RowBox[{"\"\<F\>\"", ",", 
                RowBox[{
                 RowBox[{"1.02", "fixate"}], "-", 
                 RowBox[{"scale", 
                  RowBox[{"{", 
                   RowBox[{".05", ",", "0"}], "}"}]}]}]}], "]"}], ",", 
              RowBox[{"Line", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"leftEyeNode", "[", 
                  RowBox[{"headAimVector", ",", "fixate", ",", 
                   RowBox[{"base", "[", "t", "]"}]}], "]"}], ",", "fixate"}], 
                "}"}], "]"}], ",", 
              RowBox[{"Line", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"rightEyeNode", "[", 
                  RowBox[{"headAimVector", ",", "fixate", ",", 
                   RowBox[{"base", "[", "t", "]"}]}], "]"}], ",", "fixate"}], 
                "}"}], "]"}], ",", 
              RowBox[{"Text", "[", 
               RowBox[{
                RowBox[{"Row", "[", 
                 RowBox[{"{", 
                  RowBox[{"\"\<|F-B| = \>\"", ",", 
                   RowBox[{"Round", "[", 
                    RowBox[{
                    RowBox[{"N", "[", 
                    RowBox[{"nORM", "[", 
                    RowBox[{"fixate", "-", 
                    RowBox[{"base", "[", "t", "]"}]}], "]"}], "]"}], ",", 
                    ".1"}], "]"}], ",", "\"\< cm\>\""}], "}"}], "]"}], ",", 
                RowBox[{"scale", 
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{"-", ".5"}], ",", ".8"}], "}"}]}]}], "]"}]}], 
             "}"}]}], "}"}], "]"}], "\[IndentingNewLine]", ",", 
         "\[IndentingNewLine]", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Blue", ",", 
            RowBox[{"AbsolutePointSize", "[", "5", "]"}], ",", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"Point", "[", "distractor", "]"}], ",", 
              RowBox[{"Text", "[", 
               RowBox[{"\"\<D\>\"", ",", 
                RowBox[{
                 RowBox[{"1.02", "distractor"}], "+", 
                 RowBox[{"scale", 
                  RowBox[{"{", 
                   RowBox[{".05", ",", "0"}], "}"}]}]}]}], "]"}], ",", 
              RowBox[{"Line", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"leftEyeNode", "[", 
                  RowBox[{"headAimVector", ",", "fixate", ",", 
                   RowBox[{"base", "[", "t", "]"}]}], "]"}], ",", 
                 "distractor"}], "}"}], "]"}], ",", 
              RowBox[{"Line", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"rightEyeNode", "[", 
                  RowBox[{"headAimVector", ",", "fixate", ",", 
                   RowBox[{"base", "[", "t", "]"}]}], "]"}], ",", 
                 "distractor"}], "}"}], "]"}], ",", 
              RowBox[{"Text", "[", 
               RowBox[{
                RowBox[{"Row", "[", 
                 RowBox[{"{", 
                  RowBox[{"\"\<|D-B| = \>\"", ",", 
                   RowBox[{"Round", "[", 
                    RowBox[{
                    RowBox[{"N", "[", 
                    RowBox[{"nORM", "[", 
                    RowBox[{"distractor", "-", 
                    RowBox[{"base", "[", "t", "]"}]}], "]"}], "]"}], ",", 
                    ".1"}], "]"}], ",", "\"\< cm\>\""}], "}"}], "]"}], ",", 
                RowBox[{"scale", 
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{"-", ".5"}], ",", "1."}], "}"}]}]}], "]"}]}], 
             "}"}]}], "}"}], "]"}], "\[IndentingNewLine]", ",", 
         "\[IndentingNewLine]", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{"Blue", ",", 
            RowBox[{"{", 
             RowBox[{"cIRCLE", "[", 
              RowBox[{
               RowBox[{"leftEyeNode", "[", 
                RowBox[{"headAimVector", ",", "fixate", ",", 
                 RowBox[{"base", "[", "t", "]"}]}], "]"}], ",", 
               RowBox[{"rightEyeNode", "[", 
                RowBox[{"headAimVector", ",", "fixate", ",", 
                 RowBox[{"base", "[", "t", "]"}]}], "]"}], ",", 
               "distractor"}], "]"}], "}"}]}], "}"}], "]"}], 
         "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
         RowBox[{"Graphics", "[", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"Dashing", "[", ".01", "]"}], ",", 
            RowBox[{"{", 
             RowBox[{"Circle", "[", 
              RowBox[{
               RowBox[{"{", 
                RowBox[{"0", ",", 
                 RowBox[{
                  FractionBox[
                   RowBox[{"f", "+", "d"}], "2"], "+", 
                  RowBox[{"n", " ", "r"}]}]}], "}"}], ",", 
               FractionBox[
                RowBox[{"f", "+", "d"}], "2"], ",", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"\[Pi]", "/", "6"}], ",", 
                 RowBox[{"\[Pi]", 
                  RowBox[{"(", 
                   RowBox[{"5", "/", "6"}], ")"}]}]}], "}"}]}], "]"}], 
             "}"}]}], "}"}], "]"}]}], "\[IndentingNewLine]", "}"}], ",", 
       RowBox[{"PlotRange", "\[Rule]", 
        RowBox[{"scale", "*", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{
             RowBox[{"-", ".75"}], ",", ".75"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"-", ".2"}], ",", "1.2"}], "}"}]}], "}"}]}]}], ",", 
       RowBox[{"ImageSize", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"350", ",", "350"}], "}"}]}]}], "]"}]}]}], 
   "\[IndentingNewLine]", ",", "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"scale", ",", "75"}], "}"}], ",", "20", ",", "210", ",", ".01", 
     ",", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}], ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"n", ",", ".5", ",", "\"\<node percent\>\""}], "}"}], ",", "0", 
     ",", "1", ",", ".01", ",", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}], ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\[Phi]D", ",", "10", ",", "\"\<\[Phi] degrees\>\""}], "}"}], 
     ",", 
     RowBox[{"-", "60"}], ",", "60", ",", ".01", ",", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}], ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"d", ",", "10", ",", "\"\<symmetric distance d\>\""}], "}"}], 
     ",", 
     RowBox[{"-", "50"}], ",", "100", ",", ".01", ",", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}], ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"f", ",", "75", ",", "\"\<fixate distance f\>\""}], "}"}], ",", 
     "55", ",", "200", ",", ".01", ",", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}], ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}]}], "}"}], ",", 
   RowBox[{"TrackedSymbols", "\[Rule]", "Manipulate"}], ",", 
   RowBox[{"ControlPlacement", "\[Rule]", "Left"}]}], "]"}]], "Input",
 CellChangeTimes->{{3.41573264828045*^9, 3.415732663297811*^9}, {
   3.4157326941409283`*^9, 3.415732720473076*^9}, {3.4157328243877153`*^9, 
   3.415732824964424*^9}, {3.415732865522566*^9, 3.41573286608815*^9}, {
   3.415733012158147*^9, 3.415733012828665*^9}, {3.415733114065093*^9, 
   3.4157331675419617`*^9}, {3.4157332472565203`*^9, 
   3.4157332608900003`*^9}, {3.415733336088482*^9, 3.415733406047418*^9}, {
   3.415733441107559*^9, 3.415733442165854*^9}, 3.415733509419024*^9, {
   3.415733577271695*^9, 3.415733584678537*^9}, {3.415733621665522*^9, 
   3.415733645906082*^9}, {3.4157337145361967`*^9, 3.41573372069619*^9}, {
   3.415733755411544*^9, 3.415733793147501*^9}, {3.415733832983678*^9, 
   3.4157338598785048`*^9}, {3.415736926420271*^9, 3.415736953777897*^9}, 
   3.415737067138328*^9, 3.415737125549876*^9, {3.4157373453934593`*^9, 
   3.4157376127013903`*^9}, {3.415737655701983*^9, 3.415737671401142*^9}, {
   3.4157377377401953`*^9, 3.4157378403476267`*^9}, 3.415738027536368*^9, {
   3.4157380950443907`*^9, 3.415738139990448*^9}, {3.415738454137229*^9, 
   3.4157384749435463`*^9}, {3.4157385119989843`*^9, 3.415738533157938*^9}, {
   3.4157385698463306`*^9, 3.4157386260108147`*^9}, {3.4157387074694*^9, 
   3.415738711153388*^9}, {3.4157388878006773`*^9, 3.4157388993657293`*^9}, {
   3.415739006570376*^9, 3.415739017893659*^9}, {3.415739087842042*^9, 
   3.415739132289524*^9}, {3.415739427644112*^9, 3.415739427931178*^9}, {
   3.415739466338339*^9, 3.415739466696035*^9}, 3.4157397689064293`*^9, {
   3.41574001257264*^9, 3.41574003379578*^9}, {3.4157401791100597`*^9, 
   3.415740234772656*^9}, {3.4157406828138933`*^9, 3.4157407160926313`*^9}, {
   3.415740805226531*^9, 3.415740807263657*^9}, {3.416055153574029*^9, 
   3.416055155102697*^9}, {3.416055230752762*^9, 3.416055259690935*^9}, {
   3.416055326194036*^9, 3.416055327278791*^9}, {3.4160553829985943`*^9, 
   3.416055400483132*^9}, {3.4160554552544937`*^9, 3.416055466441257*^9}, {
   3.416055527731104*^9, 3.416055539272952*^9}, {3.41605563998598*^9, 
   3.416055641394607*^9}, {3.416055681343926*^9, 3.4160556879118633`*^9}, {
   3.416055835685843*^9, 3.4160558888137827`*^9}, {3.4166702048262777`*^9, 
   3.4166702052848387`*^9}, {3.418314051249695*^9, 3.418314113427012*^9}, {
   3.418321416501067*^9, 3.4183215428451*^9}, {3.418321590510632*^9, 
   3.41832169057261*^9}, {3.418321760706936*^9, 3.4183218001282377`*^9}, {
   3.418321834074183*^9, 3.4183219144574413`*^9}, {3.418322405751123*^9, 
   3.4183224302124*^9}, {3.418322463050568*^9, 3.4183225199619102`*^9}, {
   3.4183244050987797`*^9, 3.418324480952291*^9}, {3.418324610098054*^9, 
   3.418324632672805*^9}, {3.418324665464651*^9, 3.418324694260735*^9}, {
   3.418324766276655*^9, 3.4183247914886312`*^9}, {3.41833733707206*^9, 
   3.418337344827713*^9}, {3.418386221699806*^9, 3.4183862525644407`*^9}, {
   3.418386292639154*^9, 3.418386312731148*^9}, {3.418386612567225*^9, 
   3.418386681528775*^9}, 3.424777720540947*^9, {3.4247778089405127`*^9, 
   3.424777869907885*^9}, {3.424777914009974*^9, 3.424777919885515*^9}, {
   3.424777996415601*^9, 3.424778001109645*^9}, {3.4247780407587347`*^9, 
   3.4247780435650253`*^9}, {3.424778078018956*^9, 3.424778079649662*^9}, {
   3.424778170606295*^9, 3.424778186697206*^9}, {3.424778260953587*^9, 
   3.424778262878087*^9}, {3.4247784734030123`*^9, 3.424778503365807*^9}, {
   3.42477857238556*^9, 3.4247786485132933`*^9}, {3.424778697480385*^9, 
   3.424778739971427*^9}, {3.42477878136969*^9, 3.4247788565708513`*^9}, {
   3.424778894911014*^9, 3.424778917785376*^9}, 3.4247789574485893`*^9, {
   3.42477904992148*^9, 3.424779120043117*^9}, {3.424779155681419*^9, 
   3.4247791909561243`*^9}, 3.424779237048615*^9, {3.42477938182802*^9, 
   3.4247794244202547`*^9}, {3.4247794619645653`*^9, 
   3.4247794640686283`*^9}, {3.424779535039043*^9, 3.4247795771539383`*^9}, {
   3.424779657151449*^9, 3.424779727949729*^9}, 3.4247797645880537`*^9, {
   3.424779840007519*^9, 3.42477991536265*^9}, {3.42477999187486*^9, 
   3.424780045196884*^9}, {3.424780159457184*^9, 3.4247802593951883`*^9}, 
   3.424780300376895*^9, {3.424780358101946*^9, 3.4247803717573547`*^9}, {
   3.424780418976314*^9, 3.424780445662821*^9}, {3.424780494201126*^9, 
   3.4247806294717627`*^9}, {3.424780662917844*^9, 3.424780719628998*^9}, {
   3.4247807556075068`*^9, 3.424780826448084*^9}, {3.424780945982049*^9, 
   3.424780946343569*^9}, {3.424781051087682*^9, 3.424781073792123*^9}, {
   3.424781123106785*^9, 3.4247811520639267`*^9}, 3.424781216038103*^9, {
   3.424781335487718*^9, 3.424781339333867*^9}, {3.424781972212229*^9, 
   3.4247819859878693`*^9}, 3.4247826568669767`*^9, 3.424782708198242*^9, 
   3.4247879408166437`*^9, {3.424787977421166*^9, 3.424788010053637*^9}, {
   3.4247933373711777`*^9, 3.424793347174033*^9}, {3.424793433367702*^9, 
   3.4247934597463093`*^9}, {3.4247935092135267`*^9, 3.424793549504335*^9}, {
   3.4247936266239147`*^9, 3.424793633969736*^9}, {3.4247937810465107`*^9, 
   3.424793791780013*^9}, {3.424793838591022*^9, 3.4247938439165993`*^9}, {
   3.424793903706249*^9, 3.424793923525511*^9}, {3.4247940000110617`*^9, 
   3.4247940090664167`*^9}, 3.4247940550449553`*^9, {3.424805634413597*^9, 
   3.424805638585621*^9}, {3.4248061138444643`*^9, 3.424806163192486*^9}, {
   3.424806219883102*^9, 3.424806250941197*^9}, {3.424806369029105*^9, 
   3.424806441357008*^9}, {3.424806576010338*^9, 3.4248066500930443`*^9}, 
   3.4248066868439713`*^9, 3.424814333083622*^9, {3.424814424526122*^9, 
   3.42481444110086*^9}, {3.424814517400052*^9, 3.4248145317632647`*^9}, {
   3.424814582959087*^9, 3.424814589012623*^9}, {3.424814638649612*^9, 
   3.424814787807247*^9}, {3.4248148178837557`*^9, 3.424814825190977*^9}, {
   3.424814886341198*^9, 3.424815104349188*^9}, {3.424815137120809*^9, 
   3.42481518634131*^9}, {3.424815553900392*^9, 3.424815695152609*^9}, {
   3.424815730774639*^9, 3.424815731015217*^9}, {3.424815883403761*^9, 
   3.424815890556801*^9}, {3.424816227689104*^9, 3.4248162474067383`*^9}, {
   3.4248162838159733`*^9, 3.424816296072957*^9}, {3.425217158742362*^9, 
   3.425217222775681*^9}, {3.425217328105283*^9, 3.425217385772232*^9}, {
   3.425217419642167*^9, 3.425217430344409*^9}, 3.425217464053289*^9, {
   3.4252176338928633`*^9, 3.425217639126997*^9}, {3.425217680059767*^9, 
   3.425217699420208*^9}, {3.425218103735376*^9, 3.425218104267652*^9}, 
   3.425223172720374*^9, {3.42524080115734*^9, 3.425240844809227*^9}, 
   3.42524092217918*^9, {3.425240962581938*^9, 3.425240991499153*^9}, {
   3.425241219916551*^9, 3.4252412220902*^9}, {3.425246002380082*^9, 
   3.425246035134251*^9}, {3.425246107854253*^9, 3.42524615663881*^9}, {
   3.425246608112568*^9, 3.425246614095883*^9}, {3.4264413419387207`*^9, 
   3.426441514894015*^9}, {3.4271347098581395`*^9, 3.427134715295709*^9}, {
   3.427134782452819*^9, 3.4271348454379997`*^9}, {3.4271348969074087`*^9, 
   3.427135122472796*^9}, {3.4274596167735033`*^9, 3.427459618451886*^9}, {
   3.427461648722515*^9, 3.4274616573528347`*^9}, {3.4274929707047653`*^9, 
   3.427493030906386*^9}, {3.427493067709016*^9, 3.4274931078139343`*^9}, {
   3.4274931410676193`*^9, 3.4274931454422007`*^9}, {3.427493254291329*^9, 
   3.427493297735796*^9}, {3.427493331868307*^9, 3.427493335183069*^9}, {
   3.427493452222808*^9, 3.42749351211193*^9}, 3.427493608239159*^9, {
   3.4274936623685627`*^9, 3.427493715661745*^9}, {3.4274937920503473`*^9, 
   3.4274938225061903`*^9}, 3.4274938626268787`*^9, {3.427493986554912*^9, 
   3.427494020023137*^9}, {3.42749405883908*^9, 3.427494083751637*^9}, 
   3.4274941293917923`*^9, {3.4274942071146393`*^9, 3.427494223650837*^9}, {
   3.427494260306336*^9, 3.4274942798103857`*^9}, 3.427494339580312*^9, {
   3.427494523810478*^9, 3.427494534140443*^9}, 3.4274945814187527`*^9, 
   3.4274946602800007`*^9, {3.427494714643152*^9, 3.427494735060342*^9}, {
   3.427494769884571*^9, 3.42749479433286*^9}, {3.4274948458078117`*^9, 
   3.427494851214518*^9}, {3.427494945147579*^9, 3.4274949618092127`*^9}, 
   3.42749502909678*^9, {3.427531587243821*^9, 3.427531589664893*^9}, {
   3.427531622181613*^9, 3.4275316283932943`*^9}, {3.427531715454647*^9, 
   3.4275317474979477`*^9}, 3.427531814226468*^9, {3.427531856647848*^9, 
   3.427531860789976*^9}, {3.427531925349272*^9, 3.42753197627922*^9}, 
   3.427532012682157*^9, {3.427532083729274*^9, 3.4275321138456087`*^9}, 
   3.4275321780884047`*^9, 3.427532288778102*^9, 3.4275323978804293`*^9, 
   3.4275324980047894`*^9, 3.427532577984984*^9, 3.4275327814836473`*^9, 
   3.427532863006768*^9, 3.427533024929261*^9, {3.427533103267379*^9, 
   3.427533141351046*^9}, 3.427533208888294*^9, 3.427533567273754*^9, {
   3.427534166166935*^9, 3.427534167633409*^9}, {3.4304981936712723`*^9, 
   3.430498195922497*^9}, {3.430499359413807*^9, 3.430499360745247*^9}},
 CellID->1024101491]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`d$$ = 10, $CellContext`f$$ = 75, 
    DemonstrationsTools`n$$ = 0.5, $CellContext`scale$$ = 
    75, $CellContext`\[Phi]D$$ = 10, Typeset`show$$ = True, 
    Typeset`bookmarkList$$ = {}, Typeset`bookmarkMode$$ = "Menu", 
    Typeset`animator$$, Typeset`animvar$$ = 1, Typeset`name$$ = 
    "\"untitled\"", Typeset`specs$$ = {{{
       Hold[$CellContext`scale$$], 75}, 20, 210, 0.01}, {{
       Hold[DemonstrationsTools`n$$], 0.5, "node percent"}, 0, 1, 0.01}, {{
       Hold[$CellContext`\[Phi]D$$], 10, "\[Phi] degrees"}, -60, 60, 0.01}, {{
      
       Hold[$CellContext`d$$], 10, "symmetric distance d"}, -50, 100, 0.01}, {{
       Hold[$CellContext`f$$], 75, "fixate distance f"}, 55, 200, 0.01}}, 
    Typeset`size$$ = {350., {173., 177.}}, Typeset`update$$ = 0, 
    Typeset`initDone$$, Typeset`skipInitDone$$ = 
    True, $CellContext`scale$3499$$ = 0, DemonstrationsTools`n$3500$$ = 
    0, $CellContext`\[Phi]D$3501$$ = 0, $CellContext`d$3502$$ = 
    0, $CellContext`f$3503$$ = 0}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, 
      "Variables" :> {$CellContext`d$$ = 10, $CellContext`f$$ = 75, 
        DemonstrationsTools`n$$ = 0.5, $CellContext`scale$$ = 
        75, $CellContext`\[Phi]D$$ = 10}, "ControllerVariables" :> {
        Hold[$CellContext`scale$$, $CellContext`scale$3499$$, 0], 
        Hold[DemonstrationsTools`n$$, DemonstrationsTools`n$3500$$, 0], 
        Hold[$CellContext`\[Phi]D$$, $CellContext`\[Phi]D$3501$$, 0], 
        Hold[$CellContext`d$$, $CellContext`d$3502$$, 0], 
        Hold[$CellContext`f$$, $CellContext`f$3503$$, 0]}, 
      "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, "Body" :> ($CellContext`nORM[
          PatternTest[
           Pattern[$CellContext`vector, 
            Blank[]], VectorQ]] := 
        Dot[$CellContext`vector, $CellContext`vector]^
         Rational[1, 2]; $CellContext`degrees[
          Pattern[$CellContext`q, 
           Blank[]]] := $CellContext`q (180/Pi); $CellContext`arcMinutes[
          Pattern[$CellContext`q, 
           Blank[]]] := ($CellContext`q 60) (180/Pi); $CellContext`cIRCLE[
          PatternTest[
           Pattern[$CellContext`L, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`R, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`F, 
            Blank[]], VectorQ]] := 
        Module[{$CellContext`a, $CellContext`bX, $CellContext`bY, \
$CellContext`c, $CellContext`r}, $CellContext`a = Det[{{
               Part[$CellContext`L, 1], 
               Part[$CellContext`R, 1], 
               Part[$CellContext`F, 1]}, {
               Part[$CellContext`L, 2], 
               Part[$CellContext`R, 2], 
               Part[$CellContext`F, 2]}, {1, 1, 1}}]; $CellContext`bX = 
           Det[{{
               Dot[$CellContext`L, $CellContext`L], 
               Dot[$CellContext`R, $CellContext`R], 
               Dot[$CellContext`F, $CellContext`F]}, {
               Part[$CellContext`L, 2], 
               Part[$CellContext`R, 2], 
               Part[$CellContext`F, 2]}, {1, 1, 1}}]; $CellContext`bY = 
           Det[{{
               Part[$CellContext`L, 1], 
               Part[$CellContext`R, 1], 
               Part[$CellContext`F, 1]}, {
               Dot[$CellContext`L, $CellContext`L], 
               Dot[$CellContext`R, $CellContext`R], 
               Dot[$CellContext`F, $CellContext`F]}, {1, 1, 
              1}}]; $CellContext`c = -Det[{{
                Dot[$CellContext`L, $CellContext`L], 
                Dot[$CellContext`R, $CellContext`R], 
                Dot[$CellContext`F, $CellContext`F]}, {
                Part[$CellContext`L, 1], 
                Part[$CellContext`R, 1], 
                Part[$CellContext`F, 1]}, {
                Part[$CellContext`L, 2], 
                Part[$CellContext`R, 2], 
                Part[$CellContext`F, 2]}}]; 
          Circle[{$CellContext`bX, $CellContext`bY}/(
            2 $CellContext`a), ($CellContext`bX^2 + $CellContext`bY^2 - (
               4 $CellContext`a) $CellContext`c)^Rational[1, 2]/(2 
            Abs[$CellContext`a]), {Pi/6, Pi (5/6)}]]; $CellContext`aim\[Psi][
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], VectorQ]] := ArcTan[
          Part[$CellContext`headAimVector, 1], 
          Part[$CellContext`headAimVector, 2]]; $CellContext`midPoint[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := $CellContext`base - ($CellContext`aimOffset \
$CellContext`interOcular) {
           Part[$CellContext`headAimVector, 2], -
           Part[$CellContext`headAimVector, 1]}; $CellContext`rightEyeCenter[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := $CellContext`base + ((
            1/2 - $CellContext`aimOffset) $CellContext`interOcular) {
            Part[$CellContext`headAimVector, 2], -
            Part[$CellContext`headAimVector, 1]}; $CellContext`leftEyeCenter[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], VectorQ]] := $CellContext`base - ((
           1/2 + $CellContext`aimOffset) $CellContext`interOcular) {
           Part[$CellContext`headAimVector, 2], -
           Part[$CellContext`headAimVector, 1]}; $CellContext`rightViewVector[
         
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := $CellContext`fixate - \
$CellContext`rightEyeCenter[$CellContext`headAimVector, $CellContext`base]; \
$CellContext`right\[CurlyPhi][
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], VectorQ]] := ArcTan[
          Part[
           $CellContext`rightViewVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base], 1], 
          Part[
           $CellContext`rightViewVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base], 2]]; $CellContext`rightEyeNode[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := \
$CellContext`rightEyeCenter[$CellContext`headAimVector, $CellContext`base] + \
($CellContext`nodePercent $CellContext`eyeRadius) \
($CellContext`rightViewVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base]/$CellContext`nORM[
            $CellContext`rightViewVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base]]); $CellContext`leftViewVector[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := $CellContext`fixate - \
$CellContext`leftEyeCenter[$CellContext`headAimVector, $CellContext`base]; \
$CellContext`left\[CurlyPhi][
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], VectorQ]] := ArcTan[
          Part[
           $CellContext`leftViewVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base], 1], 
          Part[
           $CellContext`leftViewVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base], 2]]; $CellContext`leftEyeNode[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := \
$CellContext`leftEyeCenter[$CellContext`headAimVector, $CellContext`base] + \
($CellContext`nodePercent $CellContext`eyeRadius) \
($CellContext`leftViewVector[$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`base]/$CellContext`nORM[
            $CellContext`leftViewVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base]]); $CellContext`fixateConvergence[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := \
$CellContext`right\[CurlyPhi][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base] - \
$CellContext`left\[CurlyPhi][$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`base]; $CellContext`left\[Alpha][
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := \
$CellContext`left\[CurlyPhi][$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`base] - $CellContext`aim\[Psi][$CellContext`headAimVector, \
$CellContext`base]; $CellContext`right\[Alpha][
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := \
$CellContext`right\[CurlyPhi][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base] - \
$CellContext`aim\[Psi][$CellContext`headAimVector, $CellContext`base]; \
$CellContext`rightDistractVector[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`distraction, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := $CellContext`distraction - \
$CellContext`rightEyeNode[$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`base]; $CellContext`right\[Delta][
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`distraction, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], VectorQ]] := ArcTan[
          Part[
           $CellContext`rightDistractVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base], 1], 
          Part[
           $CellContext`rightDistractVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base], 
           2]]; $CellContext`leftDistractVector[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`distraction, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := $CellContext`distraction - \
$CellContext`leftEyeNode[$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`base]; $CellContext`left\[Delta][
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`distraction, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], VectorQ]] := ArcTan[
          Part[
           $CellContext`leftDistractVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base], 1], 
          Part[
           $CellContext`leftDistractVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base], 
           2]]; $CellContext`convergenceDistraction[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`distraction, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], VectorQ]] := If[
          Or[
          Abs[$CellContext`left\[CurlyPhi][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base] - \
$CellContext`left\[Delta][$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`distraction, $CellContext`base]] > Pi/3., 
           Abs[$CellContext`right\[CurlyPhi][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base] - \
$CellContext`right\[Delta][$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`distraction, $CellContext`base]] > Pi/3.], 
          0., $CellContext`right\[Delta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base] - \
$CellContext`left\[Delta][$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`distraction, $CellContext`base]]; $CellContext`right\[Theta][
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`distraction, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := \
$CellContext`right\[CurlyPhi][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base] - \
$CellContext`right\[Delta][$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`distraction, $CellContext`base]; $CellContext`left\[Theta][
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`distraction, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := \
$CellContext`left\[CurlyPhi][$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`base] - $CellContext`left\[Delta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base]; \
$CellContext`binocularDisparity[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`distraction, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], VectorQ]] := If[
          
          Or[$CellContext`left\[Theta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base] < (-Pi)/
            2., $CellContext`left\[Theta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base] > 
           Pi/3., $CellContext`right\[Theta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base] < (-Pi)/
            3., $CellContext`right\[Theta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base] > Pi/2.], 
          0, $CellContext`right\[Theta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base] - \
$CellContext`left\[Theta][$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`distraction, $CellContext`base]]; \
$CellContext`right\[Theta]inside[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`distraction, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := \
$CellContext`right\[Theta][$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`distraction, $CellContext`base] + 
         ArcSin[$CellContext`nodePercent Sin[
             $CellContext`right\[Theta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base]]]; \
$CellContext`left\[Theta]inside[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`distraction, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], 
           VectorQ]] := $CellContext`left\[Theta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base] + 
         ArcSin[$CellContext`nodePercent Sin[
             $CellContext`left\[Theta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base]]]; \
$CellContext`retinalDisparity[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`distraction, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], VectorQ]] := If[
          
          Or[$CellContext`left\[Theta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base] < (-Pi)/
            2., $CellContext`left\[Theta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base] > 
           Pi/3., $CellContext`right\[Theta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base] < (-Pi)/
            3., $CellContext`right\[Theta][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base] > Pi/2.], 
          0, $CellContext`right\[Theta]inside[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base] - \
$CellContext`left\[Theta]inside[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base]]; \
$CellContext`eyePict[
          PatternTest[
           Pattern[$CellContext`eyeCenter, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ]] := 
        Module[{$CellContext`\[Lambda], $CellContext`viewVector}, \
$CellContext`viewVector = $CellContext`fixate - $CellContext`eyeCenter; \
$CellContext`\[Lambda] = ArcTan[
             Part[$CellContext`viewVector, 1], 
             
             Part[$CellContext`viewVector, 
              2]]; $CellContext`viewVector = \
$CellContext`viewVector/$CellContext`nORM[$CellContext`viewVector]; Show[{
             Graphics[{
               AbsoluteThickness[2], Gray, 
               
               Circle[$CellContext`eyeCenter, $CellContext`eyeRadius, \
{$CellContext`\[Lambda] + 0.4, 2 Pi + $CellContext`\[Lambda] - 0.4}]}], 
             Graphics[{Gray, 
               
               Circle[$CellContext`eyeCenter + (
                  0.8 $CellContext`eyeRadius) $CellContext`viewVector, 
                0.5 $CellContext`eyeRadius, {$CellContext`\[Lambda] - 
                 1.5, $CellContext`\[Lambda] + 1.5}]}], 
             Graphics[{Black, 
               
               Disk[$CellContext`eyeCenter + ($CellContext`eyeRadius \
$CellContext`nodePercent) $CellContext`viewVector, $CellContext`eyeRadius/
                10]}]}, AspectRatio -> Automatic]]; $CellContext`movingEyePict[
          PatternTest[
           Pattern[$CellContext`eyeCenter, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ]] := 
        Module[{$CellContext`\[Lambda], $CellContext`viewVector}, \
$CellContext`viewVector = $CellContext`fixate - $CellContext`eyeCenter; \
$CellContext`\[Lambda] = ArcTan[
             Part[$CellContext`viewVector, 1], 
             
             Part[$CellContext`viewVector, 
              2]]; $CellContext`viewVector = \
$CellContext`viewVector/$CellContext`nORM[$CellContext`viewVector]; Show[{
             Graphics[{
               AbsoluteThickness[2], 
               Dashing[0.01], Gray, {
                
                Line[{$CellContext`eyeCenter - {
                  1. $CellContext`eyeRadius, 
                   1.5 $CellContext`eyeRadius}, $CellContext`eyeCenter + {
                   1. $CellContext`eyeRadius, (-1.5) $CellContext`eyeRadius}, \
$CellContext`eyeCenter + {
                   0.7 $CellContext`eyeRadius, (-1.6) \
$CellContext`eyeRadius}}], 
                
                Circle[$CellContext`eyeCenter, $CellContext`eyeRadius, \
{$CellContext`\[Lambda] + 0.4, 2 Pi + $CellContext`\[Lambda] - 0.4}]}}], 
             Graphics[{Gray, 
               
               Circle[$CellContext`eyeCenter + (
                  0.8 $CellContext`eyeRadius) $CellContext`viewVector, 
                0.5 $CellContext`eyeRadius, {$CellContext`\[Lambda] - 
                 1.5, $CellContext`\[Lambda] + 1.5}]}], 
             Graphics[{Black, 
               
               Disk[$CellContext`eyeCenter + ($CellContext`eyeRadius \
$CellContext`nodePercent) $CellContext`viewVector, $CellContext`eyeRadius/
                10]}]}, AspectRatio -> Automatic]]; $CellContext`eyeEdge[
          Pattern[$CellContext`\[Theta], 
           Blank[]]] := $CellContext`eyeRadius {
           Cos[$CellContext`\[Theta]], 
           Sin[$CellContext`\[Theta]]}; $CellContext`eyeEdge[
          Pattern[$CellContext`\[Theta], 
           Blank[]]] := $CellContext`eyeRadius {
           Cos[$CellContext`\[Theta]], 
           Sin[$CellContext`\[Theta]]}; $CellContext`inRays[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`distraction, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], VectorQ]] := Show[{
           $CellContext`eyePict[
            $CellContext`leftEyeCenter[$CellContext`headAimVector, \
$CellContext`base], $CellContext`fixate], 
           $CellContext`eyePict[
            $CellContext`rightEyeCenter[$CellContext`headAimVector, \
$CellContext`base], $CellContext`fixate], 
           Graphics[{Red, {
              Line[{
                $CellContext`rightEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base], \
$CellContext`rightEyeCenter[$CellContext`headAimVector, $CellContext`base] - \
$CellContext`eyeRadius \
($CellContext`rightViewVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base]/$CellContext`nORM[
                  $CellContext`rightViewVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base]])}]}}], 
           Graphics[{Red, {
              Line[{
                $CellContext`leftEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base], \
$CellContext`leftEyeCenter[$CellContext`headAimVector, $CellContext`base] - \
$CellContext`eyeRadius \
($CellContext`leftViewVector[$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`base]/$CellContext`nORM[
                  $CellContext`leftViewVector[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base]])}]}}], 
           Graphics[{Blue, {
              Line[{
                $CellContext`rightEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base], \
$CellContext`rightEyeCenter[$CellContext`headAimVector, $CellContext`base] + \
$CellContext`eyeEdge[$CellContext`right\[CurlyPhi][$CellContext`headAimVector,\
 $CellContext`fixate, $CellContext`base] - 
                  Pi - $CellContext`right\[Theta]inside[$CellContext`\
headAimVector, $CellContext`fixate, $CellContext`distraction, \
$CellContext`base]]}]}}], 
           Graphics[{Blue, {
              Line[{
                $CellContext`leftEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base], \
$CellContext`leftEyeCenter[$CellContext`headAimVector, $CellContext`base] + \
$CellContext`eyeEdge[$CellContext`left\[CurlyPhi][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base] - 
                  Pi - $CellContext`left\[Theta]inside[$CellContext`\
headAimVector, $CellContext`fixate, $CellContext`distraction, \
$CellContext`base]]}]}}], 
           Graphics[{Purple, {
              Line[{
                $CellContext`rightEyeCenter[$CellContext`headAimVector, \
$CellContext`base], $CellContext`rightEyeCenter[$CellContext`headAimVector, \
$CellContext`base] + \
$CellContext`eyeEdge[$CellContext`right\[CurlyPhi][$CellContext`headAimVector,\
 $CellContext`fixate, $CellContext`base] - 
                  Pi - $CellContext`right\[Theta]inside[$CellContext`\
headAimVector, $CellContext`fixate, $CellContext`distraction, \
$CellContext`base]]}]}}], 
           Graphics[{Purple, {
              Line[{
                $CellContext`leftEyeCenter[$CellContext`headAimVector, \
$CellContext`base], $CellContext`leftEyeCenter[$CellContext`headAimVector, \
$CellContext`base] + \
$CellContext`eyeEdge[$CellContext`left\[CurlyPhi][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base] - 
                  Pi - $CellContext`left\[Theta]inside[$CellContext`\
headAimVector, $CellContext`fixate, $CellContext`distraction, \
$CellContext`base]]}]}}], 
           Graphics[{Green, {
              Line[
               N[{
                 $CellContext`leftEyeCenter[$CellContext`headAimVector, \
$CellContext`base], 
                 $CellContext`rightEyeCenter[$CellContext`headAimVector, \
$CellContext`base]}]]}}], $CellContext`min = 
            Min[{$CellContext`left\[CurlyPhi][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base] - 
               Pi - $CellContext`left\[Theta]inside[$CellContext`\
headAimVector, $CellContext`fixate, $CellContext`distraction, \
$CellContext`base] - \
$CellContext`retinalDisparity[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base], \
$CellContext`left\[CurlyPhi][$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`base] - 
               Pi - $CellContext`left\[Theta]inside[$CellContext`\
headAimVector, $CellContext`fixate, $CellContext`distraction, \
$CellContext`base]}]; $CellContext`MAX = 
            Max[{$CellContext`left\[CurlyPhi][$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base] - 
               Pi - $CellContext`left\[Theta]inside[$CellContext`\
headAimVector, $CellContext`fixate, $CellContext`distraction, \
$CellContext`base] - \
$CellContext`retinalDisparity[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distraction, $CellContext`base], \
$CellContext`left\[CurlyPhi][$CellContext`headAimVector, $CellContext`fixate, \
$CellContext`base] - 
               Pi - $CellContext`left\[Theta]inside[$CellContext`\
headAimVector, $CellContext`fixate, $CellContext`distraction, \
$CellContext`base]}]; Graphics[{Brown, 
              AbsoluteThickness[3], {
               Circle[
                $CellContext`leftEyeCenter[$CellContext`headAimVector, \
$CellContext`base], $CellContext`eyeRadius, {$CellContext`min, \
$CellContext`MAX}]}}]}, AspectRatio -> Automatic, PlotRange -> {{Part[
               $CellContext`midPoint[$CellContext`headAimVector, \
$CellContext`base], 1] - $CellContext`interOcular, Part[
               $CellContext`midPoint[$CellContext`headAimVector, \
$CellContext`base], 1] + $CellContext`interOcular}, {Part[
               $CellContext`midPoint[$CellContext`headAimVector, \
$CellContext`base], 2] - $CellContext`interOcular, Part[
               $CellContext`midPoint[$CellContext`headAimVector, \
$CellContext`base], 2] + $CellContext`interOcular}}]; $CellContext`outRays[
          PatternTest[
           Pattern[$CellContext`headAimVector, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`fixate, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`distraction, 
            Blank[]], VectorQ], 
          PatternTest[
           Pattern[$CellContext`base, 
            Blank[]], VectorQ]] := Show[{
           Graphics[{Black, 
             AbsolutePointSize[4], {
              Point[
               N[$CellContext`fixate]]}}], 
           Graphics[{Black, 
             AbsolutePointSize[4], {
              Point[
               N[$CellContext`distraction]]}}], 
           Graphics[{Black, {
              Line[
               N[{$CellContext`fixate, 
                 $CellContext`rightEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base]}]]}}], 
           Graphics[{Black, {
              Line[
               N[{$CellContext`fixate, 
                 $CellContext`leftEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base]}]]}}], 
           Graphics[{Black, {
              Line[
               N[{$CellContext`distraction, 
                 $CellContext`rightEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base]}]]}}], 
           Graphics[{Black, {
              Line[
               N[{$CellContext`distraction, 
                 $CellContext`leftEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`base]}]]}}]}, AspectRatio -> 
          Automatic]; $CellContext`right\[Beta]Ratio[
          Pattern[$CellContext`f$, 
           Blank[]], 
          Pattern[$CellContext`d1$, 
           Blank[]], 
          Pattern[$CellContext`d2$, 
           Blank[]], 
          Pattern[$CellContext`tt$, 
           
           Blank[]]] = (($CellContext`d1$^2 ($CellContext`f$ + 
             DemonstrationsTools`n$$ $CellContext`r)) (($CellContext`f$ + 
               DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^
            Rational[
             1, 2] + $CellContext`d1$ (((-
               DemonstrationsTools`n$$) $CellContext`r) ($CellContext`f$ + 
               DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`f$^2 \
(($CellContext`f$ + 
                 DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^
              Rational[1, 2] + (((2 $CellContext`f$) 
               DemonstrationsTools`n$$) $CellContext`r) (($CellContext`f$ + 
                 DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^Rational[1, 2] + (
              DemonstrationsTools`n$$^2 $CellContext`r^2) (($CellContext`f$ + 
                 DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^
              Rational[
               1, 2] - ($CellContext`i^2 $CellContext`tt$^2) \
(($CellContext`f$ + 
                DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^Rational[1, 2]) + ($CellContext`d2$ ($CellContext`f$ + 
             DemonstrationsTools`n$$ $CellContext`r)) ($CellContext`d2$ \
(($CellContext`f$ + 
                 DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^
              Rational[
               1, 2] + ($CellContext`i $CellContext`tt$) ((-
                DemonstrationsTools`n$$) $CellContext`r + 
              2 (($CellContext`f$ + 
                   DemonstrationsTools`n$$ $CellContext`r)^2 + \
$CellContext`i^2 $CellContext`tt$^2)^Rational[1, 2])))/($CellContext`d1$ (((-
              DemonstrationsTools`n$$) $CellContext`r) ($CellContext`f$ + 
              DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`f$^2 \
(($CellContext`f$ + 
                DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^
             Rational[1, 2] + (((2 $CellContext`f$) 
              DemonstrationsTools`n$$) $CellContext`r) (($CellContext`f$ + 
                DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^Rational[1, 2] + (
             DemonstrationsTools`n$$^2 $CellContext`r^2) (($CellContext`f$ + 
                DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^
             Rational[
              1, 2] + ($CellContext`i^2 $CellContext`tt$^2) (($CellContext`f$ + 
                DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^Rational[1, 2]) + ($CellContext`f$ + 
           DemonstrationsTools`n$$ $CellContext`r) (((-2) 
             DemonstrationsTools`n$$^3) $CellContext`r^3 - ((($CellContext`i 
              DemonstrationsTools`n$$) $CellContext`r) $CellContext`tt$) \
($CellContext`d2$ + (2 $CellContext`i) $CellContext`tt$) + ((2 
              DemonstrationsTools`n$$^2) $CellContext`r^2) (($CellContext`f$ + 
                DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^
             Rational[
              1, 2] + ($CellContext`i^2 $CellContext`tt$^2) (($CellContext`f$ + 
                DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^
             Rational[
              1, 2] + $CellContext`f$^2 (((-2) 
               DemonstrationsTools`n$$) $CellContext`r + (($CellContext`f$ + 
                 DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^Rational[1, 2]) + (((2 $CellContext`f$) 
              DemonstrationsTools`n$$) $CellContext`r) (((-2) 
               DemonstrationsTools`n$$) $CellContext`r + (($CellContext`f$ + 
                 DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^Rational[1, 2]))); $CellContext`t1[
          Pattern[$CellContext`f, 
           Blank[]], 
          Pattern[$CellContext`d1, 
           Blank[]], 
          Pattern[$CellContext`d2, 
           Blank[]]] := (-$CellContext`d1^2 - $CellContext`d2^2 - (
           2 $CellContext`d1) $CellContext`f - ($CellContext`d1^2 + \
$CellContext`d2^2)^
           Rational[
            1, 2] ($CellContext`d1^2 + $CellContext`d2^2 + (
              4 $CellContext`d1) $CellContext`f + 4 $CellContext`f^2)^
           Rational[1, 2])/((
          2 $CellContext`d2) $CellContext`i); $CellContext`t2[
          Pattern[$CellContext`f, 
           Blank[]], 
          Pattern[$CellContext`d1, 
           Blank[]], 
          Pattern[$CellContext`d2, 
           Blank[]]] := (-$CellContext`d1^2 - $CellContext`d2^2 - (
           2 $CellContext`d1) $CellContext`f + ($CellContext`d1^2 + \
$CellContext`d2^2)^
            Rational[
             1, 2] ($CellContext`d1^2 + $CellContext`d2^2 + (
               4 $CellContext`d1) $CellContext`f + 4 $CellContext`f^2)^
            Rational[1, 2])/((
          2 $CellContext`d2) $CellContext`i); $CellContext`tC[
          Pattern[$CellContext`f, 
           Blank[]], 
          Pattern[$CellContext`d1, 
           Blank[]], 
          Pattern[$CellContext`d2, 
           Blank[]]] := If[Abs[
            N[$CellContext`d2]] > 0.01, 
          If[$CellContext`d1 >= 0, 
           $CellContext`t2[$CellContext`f, $CellContext`d1, $CellContext`d2], \

           $CellContext`t1[$CellContext`f, $CellContext`d1, $CellContext`d2]],
           0]; $CellContext`rightPeak\[Beta]Ratio[
          Pattern[$CellContext`f, 
           Blank[]], 
          Pattern[$CellContext`d1, 
           Blank[]], 
          Pattern[$CellContext`d2, 
           
           Blank[]]] := $CellContext`right\[Beta]Ratio[$CellContext`f, \
$CellContext`d1, $CellContext`d2, 
          $CellContext`tC[$CellContext`f, $CellContext`d1, $CellContext`d2]]; \
$CellContext`right\[Theta]Derivative[
          Pattern[$CellContext`f$, 
           Blank[]], 
          Pattern[$CellContext`d1$, 
           Blank[]], 
          Pattern[$CellContext`d2$, 
           Blank[]], 
          Pattern[$CellContext`tt$, 
           
           Blank[]]] = (-($CellContext`i (((-$CellContext`d1$^2) \
($CellContext`f$ + 
               DemonstrationsTools`n$$ $CellContext`r)) (($CellContext`f$ + 
                 DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^
              Rational[
               1, 2] + $CellContext`d1$ ((
                DemonstrationsTools`n$$ $CellContext`r) ($CellContext`f$ + 
                 DemonstrationsTools`n$$ $CellContext`r)^2 - \
$CellContext`f$^2 (($CellContext`f$ + 
                  DemonstrationsTools`n$$ $CellContext`r)^2 + \
$CellContext`i^2 $CellContext`tt$^2)^
               Rational[1, 2] - (((2 $CellContext`f$) 
                DemonstrationsTools`n$$) $CellContext`r) (($CellContext`f$ + 
                  DemonstrationsTools`n$$ $CellContext`r)^2 + \
$CellContext`i^2 $CellContext`tt$^2)^Rational[1, 2] - (
               DemonstrationsTools`n$$^2 $CellContext`r^2) (($CellContext`f$ + 
                  DemonstrationsTools`n$$ $CellContext`r)^2 + \
$CellContext`i^2 $CellContext`tt$^2)^
               Rational[
                1, 2] + ($CellContext`i^2 $CellContext`tt$^2) \
(($CellContext`f$ + 
                   DemonstrationsTools`n$$ $CellContext`r)^2 + \
$CellContext`i^2 $CellContext`tt$^2)^
                Rational[1, 2]) - ($CellContext`d2$ ($CellContext`f$ + 
              DemonstrationsTools`n$$ $CellContext`r)) ($CellContext`d2$ \
(($CellContext`f$ + 
                  DemonstrationsTools`n$$ $CellContext`r)^2 + \
$CellContext`i^2 $CellContext`tt$^2)^
               Rational[
                1, 2] + ($CellContext`i $CellContext`tt$) ((-
                 DemonstrationsTools`n$$) $CellContext`r + 
               2 (($CellContext`f$ + 
                    DemonstrationsTools`n$$ $CellContext`r)^2 + \
$CellContext`i^2 $CellContext`tt$^2)^Rational[1, 2])))))/((($CellContext`f$ + 
            DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2) (((-2) 
            DemonstrationsTools`n$$^3) $CellContext`r^3 - ((((
              2 $CellContext`i) 
             DemonstrationsTools`n$$) $CellContext`r) $CellContext`tt$) \
($CellContext`d2$ + $CellContext`i $CellContext`tt$) + $CellContext`d1$^2 \
(($CellContext`f$ + 
               DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^
            Rational[1, 2] + ((2 
             DemonstrationsTools`n$$^2) $CellContext`r^2) (($CellContext`f$ + 
               DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^
            Rational[
             1, 2] + ($CellContext`d2$ + $CellContext`i $CellContext`tt$)^2 \
(($CellContext`f$ + 
               DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^
            Rational[
             1, 2] + $CellContext`f$^2 (((-2) 
              DemonstrationsTools`n$$) $CellContext`r + (($CellContext`f$ + 
                DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^Rational[1, 2]) + (((2 $CellContext`f$) 
             DemonstrationsTools`n$$) $CellContext`r) (((-2) 
              DemonstrationsTools`n$$) $CellContext`r + (($CellContext`f$ + 
                DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^Rational[1, 2]) + ((2 $CellContext`d1$) ($CellContext`f$ + 
             DemonstrationsTools`n$$ $CellContext`r)) ((-
              DemonstrationsTools`n$$) $CellContext`r + (($CellContext`f$ + 
                DemonstrationsTools`n$$ $CellContext`r)^2 + $CellContext`i^2 \
$CellContext`tt$^2)^
             Rational[1, 2]))); $CellContext`\[Phi] = $CellContext`\[Phi]D$$ (
          Pi/180); $CellContext`distractor := (($CellContext`f$$ + \
$CellContext`d$$)/2) {
            Sin[$CellContext`\[Phi]], 
            Cos[$CellContext`\[Phi]]} + {
          0, DemonstrationsTools`n$$ $CellContext`r + ($CellContext`f$$ + \
$CellContext`d$$)/2}; $CellContext`r = ($CellContext`eyeRadius = 
         1.3); $CellContext`nodePercent = 
        DemonstrationsTools`n$$; $CellContext`i = ($CellContext`interOcular = 
         6.5); $CellContext`aimOffset = 0; $CellContext`d1 := 
        Part[$CellContext`distractor, 2] - ($CellContext`f$$ + 
         DemonstrationsTools`n$$ $CellContext`r); $CellContext`d2 := 
        Part[$CellContext`distractor, 1]; $CellContext`fixate = {
         0, $CellContext`f$$ + 
          DemonstrationsTools`n$$ $CellContext`r}; $CellContext`headAimVector = \
{0, 1}; $CellContext`base[
          Pattern[$CellContext`tt, 
           Blank[]]] = {$CellContext`interOcular $CellContext`tt, 
          0}; $CellContext`t = 0; Quiet[
         Show[{
           Graphics[{Green, 
             AbsolutePointSize[5], {
              Line[{
                $CellContext`base[$CellContext`t], \
$CellContext`base[$CellContext`t] + $CellContext`scale$$ {
                  0.4, 0}, $CellContext`base[$CellContext`t] + \
$CellContext`scale$$ {0.3, -0.015}}], 
              Text[
              "B", $CellContext`base[$CellContext`t] - $CellContext`scale$$ {
                0, 0.05}]}}], 
           Graphics[{
             Text[
              Row[{$CellContext`\[Delta], " = ", 
                Round[
                 N[
                  $CellContext`arcMinutes[
                   $CellContext`binocularDisparity[$CellContext`headAimVector,\
 $CellContext`fixate, $CellContext`distractor, 
                    $CellContext`base[$CellContext`t]]]], 0.001], 
                " min"}], $CellContext`scale$$ {0.35, 0.1}], 
             Text[
              Row[{
                HoldForm[
                Style["d", Italic] ($CellContext`\[Theta]/(
                  Style["d", Italic] $CellContext`t))], " = ", 
                Round[
                 $CellContext`arcMinutes[
                  $CellContext`right\[Theta]Derivative[$CellContext`f$$, \
$CellContext`d1, $CellContext`d2, $CellContext`t]], 0.01], 
                " min/sec"}], $CellContext`scale$$ {0.4, 0.4}], 
             Text[
              Row[{"\!\(\*SubscriptBox[\(\[Kappa]\), \(F\)]\) = ", 
                Round[
                 $CellContext`arcMinutes[
                  $CellContext`fixateConvergence[$CellContext`headAimVector, \
$CellContext`fixate, 
                   $CellContext`base[$CellContext`t]]], 0.01], 
                " min"}], $CellContext`scale$$ {0.35, 0.9}], 
             Text[
              Row[{"\!\(\*SubscriptBox[\(\[Kappa]\), \(F\)]\)/", 
                HoldForm[
                Style["d", Italic] ($CellContext`\[Alpha]/(
                  Style["d", Italic] $CellContext`t))], " = ", 
                Round[($CellContext`f$$ + 
                  DemonstrationsTools`n$$ $CellContext`r) \
($CellContext`fixateConvergence[$CellContext`headAimVector, \
$CellContext`fixate, 
                    $CellContext`base[$CellContext`t]]/$CellContext`i), 
                 0.001]}], $CellContext`scale$$ {0.35, 0.75}], 
             Text[
              Row[{
                HoldForm[
                Style["d", Italic] ($CellContext`\[Alpha]/(
                  Style["d", Italic] $CellContext`t))], " = ", 
                Round[
                 $CellContext`arcMinutes[$CellContext`i/($CellContext`f$$ + 
                  DemonstrationsTools`n$$ $CellContext`r)], 0.01], 
                " min/sec"}], $CellContext`scale$$ {0.4, 0.6}], 
             Text[
              Row[{"\[Delta]/", 
                HoldForm[
                Style["d", Italic] ($CellContext`\[Theta]/(
                  Style["d", Italic] $CellContext`t))], " = ", 
                
                Round[$CellContext`binocularDisparity[$CellContext`\
headAimVector, $CellContext`fixate, $CellContext`distractor, 
                   $CellContext`base[$CellContext`t]]/$CellContext`right\
\[Theta]Derivative[$CellContext`f$$, $CellContext`d1, $CellContext`d2, \
$CellContext`t], 0.001]}], $CellContext`scale$$ {0.35, 0.25}], 
             Text[
              Row[{
                HoldForm["|D-B|-|F-B|"/Style["f", Italic]], " = ", 
                
                Round[($CellContext`nORM[$CellContext`distractor - \
$CellContext`base[0]] - $CellContext`nORM[$CellContext`base[
                    0] - $CellContext`fixate])/$CellContext`f$$, 
                 0.01]}], $CellContext`scale$$ {-0.45, 0.55}], 
             Text[
              Row[{
                HoldForm["|D-F|"/Style["f", Italic]], " = ", 
                
                Round[$CellContext`nORM[$CellContext`distractor - \
$CellContext`fixate]/$CellContext`f$$, 0.01]}], $CellContext`scale$$ {-0.5, 
               0.4}], 
             Text[
              Row[{
                HoldForm[
                Style["d", Italic] ($CellContext`\[Theta]/(
                  Style["d", Italic] $CellContext`\[Beta]))], 
                "(\!\(\*SubscriptBox[\(t\), \(c\)]\)) = ", 
                Round[
                 $CellContext`rightPeak\[Beta]Ratio[$CellContext`f$$, \
$CellContext`d1, $CellContext`d2], 0.01]}], $CellContext`scale$$ {-0.5, 
               0.25}], 
             Text[
              Row[{
                HoldForm[
                Style["d", Italic] ($CellContext`\[Theta]/(
                  Style["d", Italic] $CellContext`\[Beta]))], "(0) = ", 
                Round[
                 $CellContext`right\[Beta]Ratio[$CellContext`f$$, \
$CellContext`d1, $CellContext`d2, 0], 0.01]}], $CellContext`scale$$ {-0.5, 
               0.1}], 
             Text[
              StringForm[
              "\!\(\*FractionBox[\(Sin[\*FractionBox[\(\[Delta]\), \(2\)]]\), \
\(Cos[\*FractionBox[\(\[Kappa]\), \(2\)]] Sin[\*FractionBox[\(\[Kappa] - \
\[Delta]\), \(2\)]]\)]\) = `1`", 
               Round[
               Sin[(1/2) \
$CellContext`binocularDisparity[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distractor, 
                    $CellContext`base[$CellContext`t]]]/(
                Cos[(1/2) \
$CellContext`fixateConvergence[$CellContext`headAimVector, \
$CellContext`fixate, 
                    $CellContext`base[$CellContext`t]]] 
                Sin[(1/2) \
($CellContext`fixateConvergence[$CellContext`headAimVector, \
$CellContext`fixate, 
                    $CellContext`base[$CellContext`t]] - \
$CellContext`binocularDisparity[$CellContext`headAimVector, \
$CellContext`fixate, $CellContext`distractor, 
                    $CellContext`base[$CellContext`t]])]), 
                0.01]], $CellContext`scale$$ {-0.3, -0.1}]}], 
           Graphics[{Brown, {
              Line[{$CellContext`distractor, $CellContext`fixate}], 
              Text[
               Row[{"|D-F| = ", 
                 Round[
                  $CellContext`nORM[$CellContext`distractor - \
$CellContext`fixate], 0.1], " cm"}], $CellContext`scale$$ {-0.5, 0.9}]}}], 
           Graphics[{Red, 
             AbsolutePointSize[4], {
              Point[$CellContext`fixate], 
              Text[
              "F", 1.02 $CellContext`fixate - $CellContext`scale$$ {0.05, 0}], 
              Line[{
                $CellContext`leftEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, 
                 $CellContext`base[$CellContext`t]], $CellContext`fixate}], 
              Line[{
                $CellContext`rightEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, 
                 $CellContext`base[$CellContext`t]], $CellContext`fixate}], 
              Text[
               Row[{"|F-B| = ", 
                 Round[
                  N[
                   $CellContext`nORM[$CellContext`fixate - \
$CellContext`base[$CellContext`t]]], 0.1], 
                 " cm"}], $CellContext`scale$$ {-0.5, 0.8}]}}], 
           Graphics[{Blue, 
             AbsolutePointSize[5], {
              Point[$CellContext`distractor], 
              Text[
              "D", 1.02 $CellContext`distractor + $CellContext`scale$$ {
                 0.05, 0}], 
              Line[{
                $CellContext`leftEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, 
                 $CellContext`base[$CellContext`t]], \
$CellContext`distractor}], 
              Line[{
                $CellContext`rightEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, 
                 $CellContext`base[$CellContext`t]], \
$CellContext`distractor}], 
              Text[
               Row[{"|D-B| = ", 
                 Round[
                  N[
                   $CellContext`nORM[$CellContext`distractor - \
$CellContext`base[$CellContext`t]]], 0.1], 
                 " cm"}], $CellContext`scale$$ {-0.5, 1.}]}}], 
           Graphics[{Blue, {
              $CellContext`cIRCLE[
               $CellContext`leftEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, 
                $CellContext`base[$CellContext`t]], 
               $CellContext`rightEyeNode[$CellContext`headAimVector, \
$CellContext`fixate, 
                $CellContext`base[$CellContext`t]], \
$CellContext`distractor]}}], 
           Graphics[{
             Dashing[0.01], {
              
              Circle[{0, ($CellContext`f$$ + $CellContext`d$$)/2 + 
                DemonstrationsTools`n$$ $CellContext`r}, ($CellContext`f$$ + \
$CellContext`d$$)/2, {Pi/6, Pi (5/6)}]}}]}, 
          PlotRange -> $CellContext`scale$$ {{-0.75, 0.75}, {-0.2, 1.2}}, 
          ImageSize -> {350, 350}]]), 
      "Specifications" :> {{{$CellContext`scale$$, 75}, 20, 210, 0.01, 
         ImageSize -> Tiny, Appearance -> "Labeled"}, {{
         DemonstrationsTools`n$$, 0.5, "node percent"}, 0, 1, 0.01, ImageSize -> 
         Tiny, Appearance -> 
         "Labeled"}, {{$CellContext`\[Phi]D$$, 10, "\[Phi] degrees"}, -60, 60,
          0.01, ImageSize -> Tiny, Appearance -> 
         "Labeled"}, {{$CellContext`d$$, 10, "symmetric distance d"}, -50, 
         100, 0.01, ImageSize -> Tiny, Appearance -> 
         "Labeled"}, {{$CellContext`f$$, 75, "fixate distance f"}, 55, 200, 
         0.01, ImageSize -> Tiny, Appearance -> "Labeled"}}, 
      "Options" :> {TrackedSymbols -> Manipulate, ControlPlacement -> Left}, 
      "DefaultOptions" :> {ControllerLinking -> True}],
     ImageSizeCache->{617., {201., 206.}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 CellID->691466289],

Cell[CellGroupData[{

Cell["CAPTION", "Section",
 CellFrame->{{0, 0}, {1, 0}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->12,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[TextData[{
 "This Demonstration shows the asymptotic relation between ",
 Cell[BoxData[
  FormBox[
   RowBox[{"d", "\[VeryThinSpace]", 
    RowBox[{"\[Theta]", "/", "d"}], "\[VeryThinSpace]", "t"}], 
   TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["\[Delta]", TraditionalForm]], "InlineMath"],
 " (binocular disparity) and between ",
 Cell[BoxData[
  FormBox[
   RowBox[{"d", "\[VeryThinSpace]", 
    RowBox[{"\[Alpha]", "/", "d"}], "\[VeryThinSpace]", "t"}], 
   TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Kappa]", "F"], TraditionalForm]], "InlineMath"],
 " (fixate convergence angle). It also shows the motion/pursuit law at the \
critical time ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["t", "c"], TraditionalForm]], "InlineMath"],
 " where it is a better two-dimensional measure of depth than the time zero \
M/PL or the static formula (based on the symmetric case). The angle slider \
moves the distractor around a circle where retinal motion is constant. The \
\"node percent\" ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " is the fraction of the eye radius where the node point lies with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "0"}], TraditionalForm]], "InlineMath"],
 " at the center of the eye."
}], "Text"]
}, Close]]
}, Open  ]],

Cell[CellGroupData[{

Cell["DETAILS", "Section",
 CellFrame->{{0, 0}, {1, 0}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->12,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[TextData[{
 "For translation perpendicular to the line of sight at 6.5 cm/sec \
(interocular distance per unit time), retinal motion, the derivative of ",
 Cell[BoxData[
  FormBox["\[Theta]", TraditionalForm]], "InlineMath"],
 " at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", "=", "0"}], TraditionalForm]], "InlineMath"],
 ", approximates binocular disparity provided the fixate is far from the eyes \
and the distraction is not near the eyes. (Specifically, they are asymptotic \
(retinal motion/binocular disparity \[TildeTilde]1) provided the distractor \
is not more than 45\[Degree] nasal angle and you are not near the \
Vieth\[Dash]M\[UDoubleDot]ller circle.) If the speed is changed by a factor \
",
 Cell[BoxData[
  FormBox["s", TraditionalForm]], "InlineMath"],
 ", then ",
 Cell[BoxData[
  FormBox[
   RowBox[{"d", "\[VeryThinSpace]", 
    RowBox[{"\[Theta]", "/", "d"}], "\[VeryThinSpace]", "t"}], 
   TraditionalForm]], "InlineMath"],
 " is also multiplied by ",
 Cell[BoxData[
  FormBox["s", TraditionalForm]], "InlineMath"],
 ", so"
}], "Text"],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      FractionBox[
       RowBox[{"d", "\[InvisibleSpace]", "\[Theta]"}], 
       RowBox[{"d", "\[InvisibleSpace]", "t"}]], "~", 
      FractionBox["speed", 
       RowBox[{"interocular", " ", "distance"}]]}], "\[Cross]", "binocular"}],
     " ", "disparity"}], TraditionalForm]], "InlineMath"],
 ", when ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox["i", "f"], "\[TildeTilde]", "0"}], TraditionalForm]], 
  "InlineMath"],
 " (the approximation is good as a percentage of ",
 Cell[BoxData[
  FormBox["\[Delta]", TraditionalForm]], "InlineMath"],
 " = B.D.)."
}], "Text"],

Cell[TextData[{
 "The multiplier effect means that at high speed, the ",
 Cell[BoxData[
  FormBox["\[Theta]", TraditionalForm]], "InlineMath"],
 " derivative is much larger than binocular disparity, so it can be detected \
more easily, even though, by itself, it is not sufficient to measure depth. \
(The rate of change ",
 Cell[BoxData[
  FormBox[
   RowBox[{"d", "\[InvisibleSpace]", 
    RowBox[{"\[Alpha]", "/", "d"}], "\[InvisibleSpace]", "t"}], 
   TraditionalForm]], "InlineMath"],
 " would also be magnified, but ",
 Cell[BoxData[
  FormBox[
   RowBox[{"d", "\[InvisibleSpace]", 
    RowBox[{"\[Theta]", "/", "d"}], "\[InvisibleSpace]", "\[Alpha]"}], 
   TraditionalForm]], "InlineMath"],
 " and the motion/pursuit formula would be unchanged.) This means people can \
detect depth from motion when they cannot detect the depth of the same \
objects when at rest. For example, the retinal disparity of an object 20 m \
beyond a fixate at 100 m is ",
 Cell[BoxData[
  FormBox["0.372", TraditionalForm]]],
 " min of arc, below what people can detect, while at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"100", " ", 
    RowBox[{"km", "/", "hr"}]}], TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"d", "\[InvisibleSpace]", 
    RowBox[{"\[Theta]", "/", "d"}], "\[InvisibleSpace]", "t"}], 
   TraditionalForm]], "InlineMath"],
 " = 159 min/sec."
}], "Text"],

Cell[TextData[{
 "The retinal motion rate ",
 Cell[BoxData[
  FormBox[
   RowBox[{"d", "\[InvisibleSpace]", 
    RowBox[{"\[Theta]", "/", "d"}], "\[InvisibleSpace]", "t"}], 
   TraditionalForm]], "InlineMath"],
 " and motion/pursuit ratio ",
 Cell[BoxData[
  FormBox[
   RowBox[{"d", "\[InvisibleSpace]", 
    RowBox[{"\[Theta]", "/", "d"}], "\[InvisibleSpace]", "\[Alpha]"}], 
   TraditionalForm]], "InlineMath"],
 " at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", "=", "0"}], TraditionalForm]], "InlineMath"],
 " are constant on circles passing through the eye node and distractor ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"f", "+", 
      SubscriptBox["d", "1"]}], ",", 
     SubscriptBox["d", "2"]}], ")"}], TraditionalForm]], "InlineMath"],
 " with diameter on the ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["d", "1"], TraditionalForm]], "InlineMath"],
 " axis. (These circles are similar to the ",
 ButtonBox["invariant circles",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/\
ViethMullerCirclesVisualDepthPerception8/"], None}],
 " for binocular disparity, but slightly different.)"
}], "Text"],

Cell[TextData[Cell[BoxData[
 FormBox[
  GraphicsBox[{{{
      {GrayLevel[0.5], AbsoluteThickness[2], 
       CircleBox[{-3.25, 0}, 1.3, {1.7063079442456641`, 7.18949325142525}]}, 
      {GrayLevel[0.5], 
       CircleBox[{-2.978127905242549, 1.0038354267967415`}, 
        0.65, {-0.193692055754336, 2.806307944245664}]}, 
      {GrayLevel[0], DiskBox[{-3.25, 0}, 0.13]}}, {
      {GrayLevel[0.5], AbsoluteThickness[2], 
       CircleBox[{3.25, 0}, 1.3, {2.2352847093441293`, 7.718470016523716}]}, 
      {GrayLevel[0.5], 
       CircleBox[{2.978127905242549, 1.0038354267967415`}, 0.65, {
        0.33528470934412913`, 3.335284709344129}]}, 
      {GrayLevel[0], DiskBox[{3.25, 0}, 0.13]}}, 
     {RGBColor[1, 0, 0], 
      LineBox[{{3.25, 0}, {3.5898401184468134`, -1.2547942834959267`}}]}, 
     {RGBColor[1, 0, 0], 
      LineBox[{{-3.25, 0}, {-3.5898401184468134`, -1.2547942834959267`}}]}, 
     {RGBColor[0, 0, 1], 
      LineBox[{{3.25, 0}, {4.301356909594145, -0.7646232069775603}}]}, 
     {RGBColor[0, 0, 1], LineBox[{{-3.25, 0}, {-2.886, -1.248}}]}, 
     {RGBColor[0.5, 0, 0.5], 
      LineBox[{{3.25, 0}, {4.301356909594145, -0.7646232069775603}}]}, 
     {RGBColor[0.5, 0, 0.5], LineBox[{{-3.25, 0}, {-2.886, -1.248}}]}, 
     {RGBColor[0, 1, 0], LineBox[{{-3.25, 0.}, {3.25, 0.}}]}, 
     {RGBColor[0.6, 0.4, 0.2], AbsoluteThickness[3], 
      CircleBox[{-3.25, 0}, 
       1.3, {-1.2870022175865687`, -1.1577730515138982`}]}}, {
     {GrayLevel[0], AbsolutePointSize[4], PointBox[{0., 12.}]}, 
     {GrayLevel[0], AbsolutePointSize[4], PointBox[{-5., 6.}]}, 
     {GrayLevel[0], LineBox[{{0., 12.}, {3.25, 0.}}]}, 
     {GrayLevel[0], LineBox[{{0., 12.}, {-3.25, 0.}}]}, 
     {GrayLevel[0], LineBox[{{-5., 6.}, {3.25, 0.}}]}, 
     {GrayLevel[0], LineBox[{{-5., 6.}, {-3.25, 0.}}]}}, 
    {AbsolutePointSize[4], PointBox[{0, 0}]}, 
    {GrayLevel[0], AbsolutePointSize[4], PointBox[{0, 0}]}, 
    {RGBColor[1, 0, 0], AbsolutePointSize[4], PointBox[{0, 12}], 
     InsetBox["\<\"F\"\>", {0, 12.600000000000001`}], 
     LineBox[{{-3.25, 0}, {0, 12}}], LineBox[{{3.25, 0}, {0, 12}}], 
     CircleBox[{8.358897582593104*^-17, 5.559895833333333}, 
      6.440104166666666]}, 
    {RGBColor[0, 0, 1], AbsolutePointSize[5], 
     PointBox[{0, 9.516199417474784}], PointBox[{-5, 6}], 
     InsetBox["\<\"D\"\>", {-5.25, 6.300000000000001}], 
     LineBox[{{-3.25, 0}, {-5, 6}}], LineBox[{{3.25, 0}, {-5, 6}}], 
     CircleBox[{0., 4.203124999999999}, 5.313074417474781]}, 
    {GrayLevel[0], LineBox[{{0, 0}, {0, 12}}]}, 
    {GrayLevel[0], AbsolutePointSize[5], Dashing[0.01], 
     LineBox[{{0, 0}, {-5, 6}}], 
     PointBox[NCache[{0, Rational[61, 6]}, {0, 10.166666666666666`}]], 
     CircleBox[NCache[{0, Rational[61, 12]}, {0, 5.083333333333333}], NCache[
      Rational[61, 12], 5.083333333333333]]}, {
     {GrayLevel[0.5], AbsoluteThickness[2], Dashing[0.01], 
      LineBox[{{-1.3, -1.9500000000000002`}, {1.3, -1.9500000000000002`}, {
       0.9099999999999999, -2.08}}], 
      CircleBox[{0, 0}, 1.3, {1.9707963267948965`, 7.453981633974482}]}, 
     {GrayLevel[0.5], 
      CircleBox[{0, 1.04}, 0.65, {0.07079632679489656, 3.0707963267948966`}]}, 
     {GrayLevel[0], DiskBox[{0, 0}, 0.13]}}},
   AspectRatio->Automatic,
   ImageSize->{258., Automatic},
   PlotRange->{{-8, 8}, {-2.5, 13}}], 
  TraditionalForm]], "InlineMath"]], "Text"],

Cell[TextData[{
 "The pursuit rate satisfies ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"d", "\[InvisibleSpace]", 
     RowBox[{"\[Alpha]", "/", "d"}], "\[InvisibleSpace]", 
     RowBox[{"t", "(", "0", ")"}]}], " ", "=", 
    RowBox[{"i", "/", "f"}]}], TraditionalForm]], "InlineMath"],
 " and the convergence of the eyes on the fixate satisfies ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"tan", "(", 
     RowBox[{
      SubscriptBox["\[Kappa]", "F"], "/", "2"}], ")"}], "=", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"i", "/", "2"}], ")"}], "/", "f"}]}], TraditionalForm]], 
  "InlineMath"],
 " (when the eyes and fixate are symmetric). Sine and tangent are asymptotic \
to their argument for small angles, ",
 Cell[BoxData[
  FormBox[Cell[TextData[Cell[BoxData[
    FormBox[
     RowBox[{
      RowBox[{
       RowBox[{"tan", "(", 
        RowBox[{
         SubscriptBox["\[Kappa]", "F"], "/", "2"}], ")"}], "/", 
       RowBox[{"(", 
        RowBox[{
         SubscriptBox["\[Kappa]", "F"], "/", "2"}], ")"}]}], "\[TildeTilde]", 
      "1"}], TraditionalForm]], "InlineMath"]]], TraditionalForm]]],
 "when ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"i", "/", "f"}], "\[TildeTilde]", "0"}], TraditionalForm]], 
  "InlineMath"],
 "; for large fixate distances we may replace sine or tangent with the angle \
(and cosine with 1)."
}], "Text"],

Cell[TextData[{
 "By simple geometry, the static relative depth for fixate and distractor on \
the ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["d", "1"], TraditionalForm]], "InlineMath"],
 " axis is"
}], "Text"],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox["d", "f"], "=", 
    FractionBox[
     RowBox[{"sin", "(", 
      FractionBox["\[Delta]", "2"], ")"}], 
     RowBox[{
      RowBox[{"cos", "(", 
       FractionBox["\[Kappa]", "2"], ")"}], " ", 
      RowBox[{"sin", "(", 
       FractionBox[
        RowBox[{"\[Kappa]", "-", "\[Delta]"}], "2"], ")"}]}]]}], 
   TraditionalForm]], "InlineMath"],
 "."
}], "Text"],

Cell[TextData[{
 "The two asymptotic relations above then show that the static relative depth \
formula is asymptotic to the motion/pursuit law (at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", "=", "0"}], TraditionalForm]], "InlineMath"],
 "):"
}], "Text"],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox["d", "f"], "=", 
    RowBox[{
     FractionBox[
      RowBox[{"sin", "(", 
       FractionBox["\[Delta]", "2"], ")"}], 
      RowBox[{
       RowBox[{"cos", "(", 
        FractionBox["\[Kappa]", "2"], ")"}], " ", 
       RowBox[{"sin", "(", 
        FractionBox[
         RowBox[{"\[Kappa]", "-", "\[Delta]"}], "2"], ")"}]}]], "~", 
     FractionBox["\[Delta]", 
      RowBox[{"\[Kappa]", "-", "\[Delta]"}]], "~", 
     FractionBox[
      RowBox[{"d", "\[InvisibleSpace]", "\[Theta]"}], 
      RowBox[{
       RowBox[{"d", "\[InvisibleSpace]", "\[Alpha]"}], "-", 
       RowBox[{"d", "\[InvisibleSpace]", "\[Theta]"}]}]]}]}], 
   TraditionalForm]], "InlineMath"],
 "."
}], "Text"]
}, Close]],

Cell[CellGroupData[{

Cell["THIS NOTEBOOK IS THE SOURCE CODE FROM", "Text",
 CellFrame->{{0, 0}, {0, 0}},
 CellMargins->{{48, 10}, {4, 28}},
 CellGroupingRules->{"SectionGrouping", 25},
 CellFrameMargins->{{48, 48}, {6, 5}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->10,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[TextData[{
 "\"",
 ButtonBox["Dynamic Approximation of Static Quantities (Visual Depth \
Perception 14)",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/\
DynamicApproximationOfStaticQuantitiesVisualDepthPerception1/"], None},
  ButtonNote->
   "http://demonstrations.wolfram.com/\
DynamicApproximationOfStaticQuantitiesVisualDepthPerception1/"],
 "\"",
 " from ",
 ButtonBox["the Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
 "\[ParagraphSeparator]\[NonBreakingSpace]",
 ButtonBox["http://demonstrations.wolfram.com/\
DynamicApproximationOfStaticQuantitiesVisualDepthPerception1/",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/\
DynamicApproximationOfStaticQuantitiesVisualDepthPerception1/"], None},
  ButtonNote->
   "http://demonstrations.wolfram.com/\
DynamicApproximationOfStaticQuantitiesVisualDepthPerception1/"]
}], "Text",
 CellMargins->{{48, Inherited}, {0, Inherited}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.5]],

Cell[TextData[{
 "Contributed by: ",
 ButtonBox["Keith Stroyan",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/author.html?author=Keith+Stroyan"],
     None},
  ButtonNote->
   "http://demonstrations.wolfram.com/author.html?author=Keith+Stroyan"]
}], "Text",
 CellDingbat->"\[FilledSmallSquare]",
 CellMargins->{{66, 48}, {2, 4}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.6],
 CellID->95413255],

Cell[CellGroupData[{

Cell[TextData[{
 "A full-function Wolfram ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " system (Version 6 or higher) is required to edit this notebook.\n",
 StyleBox[ButtonBox["GET WOLFRAM MATHEMATICA \[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolfram.com/products/mathematica/"], None},
  ButtonNote->"http://www.wolfram.com/products/mathematica/"],
  FontFamily->"Helvetica",
  FontWeight->"Bold",
  FontSlant->"Italic",
  FontColor->RGBColor[1, 0.42, 0]]
}], "Text",
 CellFrame->True,
 CellMargins->{{48, 68}, {8, 28}},
 CellFrameMargins->12,
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 CellChangeTimes->{3.3750111182355957`*^9},
 ParagraphSpacing->{1., 1.},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.411765],
 Background->RGBColor[1, 1, 1]],

Cell[TextData[{
 "\[Copyright] ",
 StyleBox[ButtonBox["Wolfram Demonstrations Project & Contributors",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Terms of Use",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/termsofuse.html"], None},
  ButtonNote->"http://demonstrations.wolfram.com/termsofuse.html"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Make a new version of this Demonstration \
\[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/participate/upload.jsp?id=\
DynamicApproximationOfStaticQuantitiesVisualDepthPerception1"], None},
  ButtonNote->None],
  FontColor->GrayLevel[0.6]]
}], "Text",
 CellFrame->{{0, 0}, {0, 0.5}},
 CellMargins->{{48, 10}, {20, 50}},
 CellFrameMargins->{{6, 0}, {6, 6}},
 CellFrameColor->GrayLevel[0.6],
 FontFamily->"Verdana",
 FontSize->9,
 FontColor->GrayLevel[0.6]]
}, Open  ]]
}, Open  ]]
},
Editable->True,
Saveable->False,
ScreenStyleEnvironment->"Working",
CellInsertionPointCell->None,
WindowSize->{780, 650},
WindowMargins->{{Inherited, Inherited}, {Inherited, 0}},
WindowElements->{
 "StatusArea", "MemoryMonitor", "MagnificationPopUp", "VerticalScrollBar", 
  "MenuBar"},
WindowTitle->"Dynamic Approximation of Static Quantities (Visual Depth \
Perception 14) - Source",
DockedCells->{},
CellContext->Notebook,
FrontEndVersion->"8.0 for Microsoft Windows (32-bit) (November 7, 2010)",
StyleDefinitions->"Default.nb"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[675, 24, 199, 5, 70, "Section"],
Cell[877, 31, 3700, 64, 70, "Section"],
Cell[4580, 97, 98688, 2363, 70, "Input",
 CellID->1024101491]
}, Open  ]],
Cell[CellGroupData[{
Cell[103305, 2465, 52688, 1143, 70, "Output",
 CellID->691466289],
Cell[CellGroupData[{
Cell[156018, 3612, 209, 6, 70, "Section"],
Cell[156230, 3620, 1328, 36, 70, "Text"]
}, Close]]
}, Open  ]],
Cell[CellGroupData[{
Cell[157606, 3662, 209, 6, 70, "Section"],
Cell[157818, 3670, 1065, 27, 70, "Text"],
Cell[158886, 3699, 662, 22, 70, "Text"],
Cell[159551, 3723, 1375, 36, 70, "Text"],
Cell[160929, 3761, 1173, 36, 70, "Text"],
Cell[162105, 3799, 3362, 63, 70, "Text"],
Cell[165470, 3864, 1378, 42, 70, "Text"],
Cell[166851, 3908, 212, 7, 70, "Text"],
Cell[167066, 3917, 431, 16, 70, "Text"],
Cell[167500, 3935, 256, 7, 70, "Text"],
Cell[167759, 3944, 750, 24, 70, "Text"]
}, Close]],
Cell[CellGroupData[{
Cell[168545, 3973, 355, 9, 70, "Text",
 CellGroupingRules->{"SectionGrouping", 25}],
Cell[168903, 3984, 1168, 32, 70, "Text"],
Cell[170074, 4018, 455, 15, 70, "Text",
 CellID->95413255],
Cell[CellGroupData[{
Cell[170554, 4037, 815, 24, 70, "Text"],
Cell[171372, 4063, 1232, 33, 70, "Text"]
}, Open  ]]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
(* NotebookSignature 0S4hrC4eJEuqGD0zGi#fzsNv *)
