(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 9.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       157,          7]
NotebookDataLength[     41605,       1061]
NotebookOptionsPosition[     39690,        991]
NotebookOutlinePosition[     40326,       1016]
CellTagsIndexPosition[     40283,       1013]
WindowTitle->Fern Fractals - Source
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["Fern Fractals", "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False,
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztnVuSozgWQDNiPqa3MJ+zpVlCbaB3WlGVNV9dvYH6TQRUph/ZGIHQmwvG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   "], {{0, 0}, {1800, 25}}, {0, 255},
   ColorFunction->RGBColor],
  ImageSize->{1800, 25},
  PlotRange->{{0, 1800}, {0, 25}}]], "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", 
   RowBox[{"Barnsley", " ", "fern"}], " ", "*)"}], "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"p1", " ", "=", " ", "0.01"}], ";"}], " ", 
   RowBox[{"(*", 
    RowBox[{"probability", " ", "for", " ", "stem"}], "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"p2", " ", "=", " ", "0.85"}], ";"}], " ", 
   RowBox[{"(*", 
    RowBox[{"successively", " ", "smaller", " ", "leaflets"}], "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"p3", " ", "=", " ", "0.07"}], ";"}], " ", 
   RowBox[{"(*", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"largest", " ", "left"}], "-", 
       RowBox[{"hand", " ", "leaflet"}]}], ";", " ", 
      RowBox[{"p4", " ", "=", " ", "0.07"}]}], ",", " ", 
     RowBox[{
      RowBox[{"largest", " ", "right"}], "-", 
      RowBox[{"hand", " ", "leaflet"}]}]}], "*)"}]}]}]], "Input",
 InitializationCell->True,
 CellChangeTimes->{3.63155211141936*^9},
 CellID->140126495],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", 
   RowBox[{
    RowBox[{"multiplies", " ", "matrix"}], " ", "+", " ", 
    RowBox[{"point", " ", 
     RowBox[{"{", 
      RowBox[{"x", ",", "y"}], "}"}], " ", "to", " ", "successively", " ", 
     "generate", " ", "points"}]}], "*)"}], "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"f1", " ", "=", " ", 
     RowBox[{"Compile", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"point", ",", "_Real", ",", "1"}], "}"}], "}"}], ",", 
       RowBox[{"Dot", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"0.", ",", "0."}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"0.", ",", ".16"}], "}"}]}], "}"}], ",", "point"}], 
        "]"}]}], "]"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"f2", "=", " ", 
     RowBox[{"Compile", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"point", ",", "_Real", ",", "1"}], "}"}], "}"}], ",", 
       RowBox[{
        RowBox[{"Dot", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{".85", ",", ".04"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"-", ".04"}], ",", ".85"}], "}"}]}], "}"}], ",", 
          "point"}], "]"}], " ", "+", " ", 
        RowBox[{"{", 
         RowBox[{"0.", ",", "1.6"}], "}"}]}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"f3", "=", " ", 
     RowBox[{"Compile", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"point", ",", "_Real", ",", "1"}], "}"}], "}"}], ",", 
       RowBox[{
        RowBox[{"Dot", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{".2", ",", 
              RowBox[{"-", ".26"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{".23", ",", ".22"}], "}"}]}], "}"}], ",", "point"}], 
         "]"}], " ", "+", " ", 
        RowBox[{"{", 
         RowBox[{"0.", ",", "1.6"}], "}"}]}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"f4", "=", " ", 
     RowBox[{"Compile", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"point", ",", "_Real", ",", "1"}], "}"}], "}"}], ",", 
       RowBox[{
        RowBox[{"Dot", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{
              RowBox[{"-", ".15"}], ",", ".28"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{".26", ",", ".24"}], "}"}]}], "}"}], ",", "point"}], 
         "]"}], " ", "+", " ", 
        RowBox[{"{", 
         RowBox[{"0.", ",", ".44"}], "}"}]}]}], "]"}]}], ";"}]}]}]], "Input",
 InitializationCell->True,
 CellChangeTimes->{{3.63155211141936*^9, 3.631552120441043*^9}},
 CellID->1493634199],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", 
   RowBox[{"Thelypteridaceae", " ", "fern"}], " ", "*)"}], 
  "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"p1t", " ", "=", " ", "0.02"}], ";"}], " ", 
   RowBox[{"(*", 
    RowBox[{"probability", " ", "for", " ", "stem"}], "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"p2t", " ", "=", " ", "0.84"}], ";"}], " ", 
   RowBox[{"(*", 
    RowBox[{"successively", " ", "smaller", " ", "leaflets"}], 
    "*)"}]}]}]], "Input",
 InitializationCell->True,
 CellChangeTimes->{{3.63155211141936*^9, 3.6315521274796333`*^9}},
 CellID->894911208],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", 
   RowBox[{
    RowBox[{"multiplies", " ", "matrix"}], " ", "+", " ", 
    RowBox[{"point", " ", 
     RowBox[{"{", 
      RowBox[{"x", ",", "y"}], "}"}], " ", "to", " ", "successively", " ", 
     "generate", " ", "points"}]}], "*)"}], "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"f1t", "=", " ", 
     RowBox[{"Compile", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"point", ",", "_Real", ",", "1"}], "}"}], "}"}], ",", 
       RowBox[{
        RowBox[{"Dot", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{"0.", ",", "0."}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"0.", ",", ".25"}], "}"}]}], "}"}], ",", "point"}], 
         "]"}], " ", "+", " ", 
        RowBox[{"{", 
         RowBox[{"0.", ",", 
          RowBox[{"-", ".4"}]}], "}"}]}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"f2t", "=", " ", 
     RowBox[{"Compile", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"point", ",", "_Real", ",", "1"}], "}"}], "}"}], ",", 
       RowBox[{
        RowBox[{"Dot", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{".95", ",", ".005"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"-", ".005"}], ",", ".93"}], "}"}]}], "}"}], ",", 
          "point"}], "]"}], " ", "+", " ", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", ".002"}], ",", ".5"}], "}"}]}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"f3t", "=", " ", 
     RowBox[{"Compile", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"point", ",", "_Real", ",", "1"}], "}"}], "}"}], ",", 
       RowBox[{
        RowBox[{"Dot", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{".035", ",", 
              RowBox[{"-", ".2"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{".16", ",", ".04"}], "}"}]}], "}"}], ",", "point"}], 
         "]"}], " ", "+", " ", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", ".09"}], ",", ".02"}], "}"}]}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"f4t", "=", " ", 
     RowBox[{"Compile", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"point", ",", "_Real", ",", "1"}], "}"}], "}"}], ",", 
       RowBox[{
        RowBox[{"Dot", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{
              RowBox[{"-", ".04"}], ",", ".2"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{".16", ",", ".04"}], "}"}]}], "}"}], ",", "point"}], 
         "]"}], " ", "+", " ", 
        RowBox[{"{", 
         RowBox[{".083", ",", ".12"}], "}"}]}]}], "]"}]}], ";"}]}]}]], "Input",\

 InitializationCell->True,
 CellChangeTimes->{{3.63155211141936*^9, 3.631552141265061*^9}},
 CellID->32957756],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", 
   RowBox[{"Leptosporangiate", " ", "fern"}], "*)"}], "\[IndentingNewLine]", 
  RowBox[{"(*", 
   RowBox[{
    RowBox[{"multiplies", " ", "matrix"}], " ", "+", " ", 
    RowBox[{"point", " ", 
     RowBox[{"{", 
      RowBox[{"x", ",", "y"}], "}"}], " ", "to", " ", "successively", " ", 
     "generate", " ", "points"}]}], "*)"}], "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"f1l", "=", " ", 
     RowBox[{"Compile", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"point", ",", "_Real", ",", "1"}], "}"}], "}"}], ",", 
       RowBox[{
        RowBox[{"Dot", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{"0.", ",", "0."}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"0.", ",", ".25"}], "}"}]}], "}"}], ",", "point"}], 
         "]"}], " ", "+", " ", 
        RowBox[{"{", 
         RowBox[{"0.", ",", 
          RowBox[{"-", ".14"}]}], "}"}]}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"f2l", "=", " ", 
     RowBox[{"Compile", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"point", ",", "_Real", ",", "1"}], "}"}], "}"}], ",", 
       RowBox[{
        RowBox[{"Dot", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{".85", ",", ".02"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"-", ".02"}], ",", ".83"}], "}"}]}], "}"}], ",", 
          "point"}], "]"}], " ", "+", " ", 
        RowBox[{"{", 
         RowBox[{"0.", ",", "1."}], "}"}]}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"f3l", "=", " ", 
     RowBox[{"Compile", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"point", ",", "_Real", ",", "1"}], "}"}], "}"}], ",", 
       RowBox[{
        RowBox[{"Dot", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{".09", ",", 
              RowBox[{"-", ".28"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{".3", ",", ".11"}], "}"}]}], "}"}], ",", "point"}], 
         "]"}], " ", "+", " ", 
        RowBox[{"{", 
         RowBox[{"0.", ",", ".6"}], "}"}]}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"f4l", "=", " ", 
     RowBox[{"Compile", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"point", ",", "_Real", ",", "1"}], "}"}], "}"}], ",", 
       RowBox[{
        RowBox[{"Dot", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{
              RowBox[{"-", ".09"}], ",", ".28"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{".3", ",", ".09"}], "}"}]}], "}"}], ",", "point"}], 
         "]"}], " ", "+", " ", 
        RowBox[{"{", 
         RowBox[{"0.", ",", ".7"}], "}"}]}]}], "]"}]}], ";"}]}]}]], "Input",
 InitializationCell->True,
 CellChangeTimes->{{3.63155211141936*^9, 3.63155215235957*^9}},
 CellID->1321613716],

Cell[BoxData[
 RowBox[{
  RowBox[{"point", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"0.", ",", "0."}], "}"}]}], ";"}]], "Input",
 InitializationCell->True,
 CellChangeTimes->{{3.63155211141936*^9, 3.631552155583618*^9}},
 CellID->1493693289],

Cell[BoxData[
 RowBox[{
  RowBox[{"generatePointB", "[", "point_", "]"}], " ", ":=", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"r", "=", 
     RowBox[{"RandomReal", "[", "]"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"Which", "[", 
     RowBox[{
      RowBox[{"p1", " ", "<", " ", "r", " ", "<", " ", 
       RowBox[{"p1", "+", "p2"}]}], ",", 
      RowBox[{"f2", "[", "point", "]"}], ",", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"p1", "+", "p2"}], " ", "<", " ", "r", " ", "<", " ", 
       RowBox[{"p1", "+", "p2", "+", "p3"}]}], ",", 
      RowBox[{"f3", "[", "point", "]"}], ",", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"p1", "+", "p2", "+", "p3"}], "<", "r"}], ",", 
      RowBox[{"f4", "[", "point", "]"}], ",", "\[IndentingNewLine]", 
      RowBox[{"r", "<", "p1"}], ",", 
      RowBox[{"f1", "[", "point", "]"}]}], "]"}]}], ")"}]}]], "Input",
 InitializationCell->True,
 CellChangeTimes->{{3.63155211141936*^9, 3.631552169215313*^9}},
 CellID->857600144],

Cell[BoxData[
 RowBox[{
  RowBox[{"generatePointT", "[", "point_", "]"}], " ", ":=", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"r", "=", 
     RowBox[{"RandomReal", "[", "]"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"Which", "[", 
     RowBox[{
      RowBox[{"p1t", " ", "<", " ", "r", " ", "<", " ", 
       RowBox[{"p1t", "+", "p2t"}]}], ",", 
      RowBox[{"f2t", "[", "point", "]"}], ",", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"p1t", "+", "p2t"}], " ", "<", " ", "r", " ", "<", " ", 
       RowBox[{"p1t", "+", "p2t", "+", "p3"}]}], ",", 
      RowBox[{"f3t", "[", "point", "]"}], ",", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"p1t", "+", "p2t", "+", "p3"}], "<", "r"}], ",", 
      RowBox[{"f4t", "[", "point", "]"}], ",", "\[IndentingNewLine]", 
      RowBox[{"r", "<", "p1t"}], ",", 
      RowBox[{"f1", "[", "point", "]"}]}], "]"}]}], ")"}]}]], "Input",
 InitializationCell->True,
 CellChangeTimes->{{3.63155211141936*^9, 3.631552167383029*^9}},
 CellID->1922865191],

Cell[BoxData[
 RowBox[{
  RowBox[{"generatePointL", "[", "point_", "]"}], " ", ":=", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"r", "=", 
     RowBox[{"RandomReal", "[", "]"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"Which", "[", 
     RowBox[{
      RowBox[{"p1t", " ", "<", " ", "r", " ", "<", " ", 
       RowBox[{"p1t", "+", "p2t"}]}], ",", 
      RowBox[{"f2l", "[", "point", "]"}], ",", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"p1t", "+", "p2t"}], " ", "<", " ", "r", " ", "<", " ", 
       RowBox[{"p1t", "+", "p2t", "+", "p3"}]}], ",", 
      RowBox[{"f3l", "[", "point", "]"}], ",", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"p1t", "+", "p2t", "+", "p3"}], "<", "r"}], ",", 
      RowBox[{"f4l", "[", "point", "]"}], ",", "\[IndentingNewLine]", 
      RowBox[{"r", "<", "p1t"}], ",", 
      RowBox[{"f1l", "[", "point", "]"}]}], "]"}]}], ")"}]}]], "Input",
 InitializationCell->True,
 CellChangeTimes->{{3.63155211141936*^9, 3.63155216615882*^9}},
 CellID->113861434],

Cell[BoxData[
 RowBox[{"Manipulate", "[", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"ListPlot", "[", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"NestList", "[", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"Which", "[", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"type", "===", "\"\<Barnsley\>\""}], ",", "generatePointB", 
         ",", "\[IndentingNewLine]", 
         RowBox[{"type", " ", "===", " ", "\"\<Thelypteridaceae\>\""}], ",", 
         "generatePointT", ",", "\[IndentingNewLine]", 
         RowBox[{"type", " ", "===", " ", "\"\<Leptosporangiate\>\""}], ",", 
         "generatePointL"}], "\[IndentingNewLine]", "]"}], ",", "point", ",", 
       "iterations"}], "\[IndentingNewLine]", "]"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"PlotStyle", "\[Rule]", "Green"}], ",", 
     RowBox[{"Background", " ", "\[Rule]", " ", "Black"}], ",", 
     RowBox[{"Axes", "\[Rule]", "False"}], ",", 
     RowBox[{"AspectRatio", "\[Rule]", "1"}], ",", 
     RowBox[{"ImageSize", "\[Rule]", 
      RowBox[{"1.1", 
       RowBox[{"{", 
        RowBox[{"400", ",", "400"}], "}"}]}]}]}], "]"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"iterations", ",", "100"}], "}"}], ",", "0", ",", "20000", ",", 
     "100", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"type", ",", "\"\<Barnsley\>\"", ",", "\"\<fern type\>\""}], 
      "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
      "\"\<Barnsley\>\"", ",", "\"\<Thelypteridaceae\>\"", ",", 
       "\"\<Leptosporangiate\>\""}], "}"}], ",", 
     RowBox[{"ControlType", "\[Rule]", "RadioButtonBar"}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"SaveDefinitions", " ", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellChangeTimes->{
  3.631292597078434*^9, {3.631293280547967*^9, 3.631293298179347*^9}, {
   3.631552187769333*^9, 3.6315523427393312`*^9}, {3.631645384967783*^9, 
   3.631645386270564*^9}, {3.631730146636921*^9, 3.63173014700478*^9}, {
   3.6317302480340643`*^9, 3.631730421051914*^9}, {3.63173059163305*^9, 
   3.631730709019433*^9}}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`iterations$$ = 100, $CellContext`type$$ = 
    "Barnsley", Typeset`show$$ = True, Typeset`bookmarkList$$ = {}, 
    Typeset`bookmarkMode$$ = "Menu", Typeset`animator$$, Typeset`animvar$$ = 
    1, Typeset`name$$ = "\"untitled\"", Typeset`specs$$ = {{{
       Hold[$CellContext`iterations$$], 100}, 0, 20000, 100}, {{
       Hold[$CellContext`type$$], "Barnsley", "fern type"}, {
      "Barnsley", "Thelypteridaceae", "Leptosporangiate"}}}, 
    Typeset`size$$ = {440., {218., 222.}}, Typeset`update$$ = 0, 
    Typeset`initDone$$, Typeset`skipInitDone$$ = 
    False, $CellContext`iterations$181663$$ = 0, $CellContext`type$181664$$ = 
    0}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, 
      "Variables" :> {$CellContext`iterations$$ = 100, $CellContext`type$$ = 
        "Barnsley"}, "ControllerVariables" :> {
        Hold[$CellContext`iterations$$, $CellContext`iterations$181663$$, 0], 
        
        Hold[$CellContext`type$$, $CellContext`type$181664$$, 0]}, 
      "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, "Body" :> ListPlot[
        NestList[
         Which[$CellContext`type$$ === 
          "Barnsley", $CellContext`generatePointB, $CellContext`type$$ === 
          "Thelypteridaceae", $CellContext`generatePointT, \
$CellContext`type$$ === 
          "Leptosporangiate", $CellContext`generatePointL], \
$CellContext`point, $CellContext`iterations$$], PlotStyle -> Green, 
        Background -> Black, Axes -> False, AspectRatio -> 1, ImageSize -> 
        1.1 {400, 400}], 
      "Specifications" :> {{{$CellContext`iterations$$, 100}, 0, 20000, 100, 
         Appearance -> 
         "Labeled"}, {{$CellContext`type$$, "Barnsley", "fern type"}, {
         "Barnsley", "Thelypteridaceae", "Leptosporangiate"}, ControlType -> 
         RadioButtonBar}}, "Options" :> {}, 
      "DefaultOptions" :> {ControllerLinking -> True}],
     ImageSizeCache->{487., {271., 276.}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    Initialization:>(({$CellContext`generatePointB[
          Pattern[$CellContext`point, 
           Blank[]]] := ($CellContext`r = RandomReal[]; 
         Which[$CellContext`p1 < $CellContext`r < $CellContext`p1 + \
$CellContext`p2, 
           $CellContext`f2[$CellContext`point], $CellContext`p1 + \
$CellContext`p2 < $CellContext`r < $CellContext`p1 + $CellContext`p2 + \
$CellContext`p3, 
           $CellContext`f3[$CellContext`point], $CellContext`p1 + \
$CellContext`p2 + $CellContext`p3 < $CellContext`r, 
           $CellContext`f4[$CellContext`point], $CellContext`r < \
$CellContext`p1, 
           $CellContext`f1[$CellContext`point]]), $CellContext`point = {0., 
         0.}, $CellContext`r = 0.03341643137241013, $CellContext`p1 = 
        0.01, $CellContext`p2 = 0.85, $CellContext`f2 = 
        CompiledFunction[{8, 9., 5468}, {{
            Blank[Real], 1}}, {{3, 1, 0}, {3, 1, 
          2}}, {{{0., 1.6}, {3, 1, 3}}, {{{0.85, 0.04}, {-0.04, 0.85}}, {3, 2,
             1}}, {4, {2, 0, 0}}}, {0, 1, 0, 0, 
          4}, {{42, "Dot", 3, 2, 1, 3, 1, 0, 2, 0, 0, 3, 1, 2}, {44, 2, 3, 
           2}, {1}}, 
          
          Function[{$CellContext`point}, 
           Dot[{{0.85, 0.04}, {-0.04, 0.85}}, $CellContext`point] + {0., 
            1.6}], Evaluate], $CellContext`p3 = 0.07, $CellContext`f3 = 
        CompiledFunction[{8, 9., 5468}, {{
            Blank[Real], 1}}, {{3, 1, 0}, {3, 1, 
          2}}, {{{0., 1.6}, {3, 1, 3}}, {
           4, {2, 0, 0}}, {{{0.2, -0.26}, {0.23, 0.22}}, {3, 2, 1}}}, {0, 1, 
          0, 0, 4}, {{42, "Dot", 3, 2, 1, 3, 1, 0, 2, 0, 0, 3, 1, 2}, {44, 2, 
           3, 2}, {1}}, 
          
          Function[{$CellContext`point}, 
           Dot[{{0.2, -0.26}, {0.23, 0.22}}, $CellContext`point] + {0., 1.6}],
           Evaluate], $CellContext`f4 = CompiledFunction[{8, 9., 5468}, {{
            Blank[Real], 1}}, {{3, 1, 0}, {3, 1, 
          2}}, {{
           4, {2, 0, 0}}, {{{-0.15, 0.28}, {0.26, 0.24}}, {3, 2, 1}}, {{0., 
            0.44}, {3, 1, 3}}}, {0, 1, 0, 0, 
          4}, {{42, "Dot", 3, 2, 1, 3, 1, 0, 2, 0, 0, 3, 1, 2}, {44, 2, 3, 
           2}, {1}}, 
          
          Function[{$CellContext`point}, 
           Dot[{{-0.15, 0.28}, {0.26, 0.24}}, $CellContext`point] + {0., 
            0.44}], Evaluate], $CellContext`f1 = 
        CompiledFunction[{8, 9., 5468}, {{
            Blank[Real], 1}}, {{3, 1, 0}, {3, 1, 
          2}}, {{{{0., 0.}, {0., 0.16}}, {3, 2, 1}}, {4, {2, 0, 0}}}, {0, 1, 
          0, 0, 3}, {{42, "Dot", 3, 2, 1, 3, 1, 0, 2, 0, 0, 3, 1, 2}, {1}}, 
          Function[{$CellContext`point}, 
           Dot[{{0., 0.}, {0., 0.16}}, $CellContext`point]], 
          Evaluate], $CellContext`generatePointT[
          Pattern[$CellContext`point, 
           Blank[]]] := ($CellContext`r = RandomReal[]; 
         Which[$CellContext`p1t < $CellContext`r < $CellContext`p1t + \
$CellContext`p2t, 
           $CellContext`f2t[$CellContext`point], $CellContext`p1t + \
$CellContext`p2t < $CellContext`r < $CellContext`p1t + $CellContext`p2t + \
$CellContext`p3, 
           $CellContext`f3t[$CellContext`point], $CellContext`p1t + \
$CellContext`p2t + $CellContext`p3 < $CellContext`r, 
           $CellContext`f4t[$CellContext`point], $CellContext`r < \
$CellContext`p1t, 
           $CellContext`f1[$CellContext`point]]), $CellContext`p1t = 
        0.02, $CellContext`p2t = 0.84, $CellContext`f2t = 
        CompiledFunction[{8, 9., 5468}, {{
            Blank[Real], 1}}, {{3, 1, 0}, {3, 1, 
          2}}, {{{-0.002, 0.5}, {3, 1, 3}}, {{{0.95, 0.005}, {-0.005, 
            0.93}}, {3, 2, 1}}, {4, {2, 0, 0}}}, {0, 1, 0, 0, 
          4}, {{42, "Dot", 3, 2, 1, 3, 1, 0, 2, 0, 0, 3, 1, 2}, {44, 2, 3, 
           2}, {1}}, 
          
          Function[{$CellContext`point}, 
           Dot[{{0.95, 0.005}, {-0.005, 0.93}}, $CellContext`point] + {-0.002,
             0.5}], Evaluate], $CellContext`f3t = 
        CompiledFunction[{8, 9., 5468}, {{
            Blank[Real], 1}}, {{3, 1, 0}, {3, 1, 
          2}}, {{{{0.035, -0.2}, {0.16, 0.04}}, {3, 2, 1}}, {{-0.09, 0.02}, {
            3, 1, 3}}, {4, {2, 0, 0}}}, {0, 1, 0, 0, 
          4}, {{42, "Dot", 3, 2, 1, 3, 1, 0, 2, 0, 0, 3, 1, 2}, {44, 2, 3, 
           2}, {1}}, 
          
          Function[{$CellContext`point}, 
           Dot[{{0.035, -0.2}, {0.16, 0.04}}, $CellContext`point] + {-0.09, 
            0.02}], Evaluate], $CellContext`f4t = 
        CompiledFunction[{8, 9., 5468}, {{
            Blank[Real], 1}}, {{3, 1, 0}, {3, 1, 
          2}}, {{{{-0.04, 0.2}, {0.16, 0.04}}, {3, 2, 1}}, {{0.083, 0.12}, {3,
             1, 3}}, {4, {2, 0, 0}}}, {0, 1, 0, 0, 
          4}, {{42, "Dot", 3, 2, 1, 3, 1, 0, 2, 0, 0, 3, 1, 2}, {44, 2, 3, 
           2}, {1}}, 
          
          Function[{$CellContext`point}, 
           Dot[{{-0.04, 0.2}, {0.16, 0.04}}, $CellContext`point] + {0.083, 
            0.12}], Evaluate], $CellContext`generatePointL[
          Pattern[$CellContext`point, 
           Blank[]]] := ($CellContext`r = RandomReal[]; 
         Which[$CellContext`p1t < $CellContext`r < $CellContext`p1t + \
$CellContext`p2t, 
           $CellContext`f2l[$CellContext`point], $CellContext`p1t + \
$CellContext`p2t < $CellContext`r < $CellContext`p1t + $CellContext`p2t + \
$CellContext`p3, 
           $CellContext`f3l[$CellContext`point], $CellContext`p1t + \
$CellContext`p2t + $CellContext`p3 < $CellContext`r, 
           $CellContext`f4l[$CellContext`point], $CellContext`r < \
$CellContext`p1t, 
           $CellContext`f1l[$CellContext`point]]), $CellContext`f2l = 
        CompiledFunction[{8, 9., 5468}, {{
            Blank[Real], 1}}, {{3, 1, 0}, {3, 1, 
          2}}, {{{{0.85, 0.02}, {-0.02, 0.83}}, {3, 2, 1}}, {{0., 1.}, {3, 1, 
            3}}, {4, {2, 0, 0}}}, {0, 1, 0, 0, 
          4}, {{42, "Dot", 3, 2, 1, 3, 1, 0, 2, 0, 0, 3, 1, 2}, {44, 2, 3, 
           2}, {1}}, 
          
          Function[{$CellContext`point}, 
           Dot[{{0.85, 0.02}, {-0.02, 0.83}}, $CellContext`point] + {0., 1.}],
           Evaluate], $CellContext`f3l = CompiledFunction[{8, 9., 5468}, {{
            Blank[Real], 1}}, {{3, 1, 0}, {3, 1, 
          2}}, {{{0., 0.6}, {3, 1, 3}}, {{{0.09, -0.28}, {0.3, 0.11}}, {3, 2, 
            1}}, {4, {2, 0, 0}}}, {0, 1, 0, 0, 
          4}, {{42, "Dot", 3, 2, 1, 3, 1, 0, 2, 0, 0, 3, 1, 2}, {44, 2, 3, 
           2}, {1}}, 
          
          Function[{$CellContext`point}, 
           Dot[{{0.09, -0.28}, {0.3, 0.11}}, $CellContext`point] + {0., 0.6}],
           Evaluate], $CellContext`f4l = CompiledFunction[{8, 9., 5468}, {{
            Blank[Real], 1}}, {{3, 1, 0}, {3, 1, 
          2}}, {{4, {2, 0, 0}}, {{{-0.09, 0.28}, {0.3, 0.09}}, {3, 2, 1}}, {{
            0., 0.7}, {3, 1, 3}}}, {0, 1, 0, 0, 
          4}, {{42, "Dot", 3, 2, 1, 3, 1, 0, 2, 0, 0, 3, 1, 2}, {44, 2, 3, 
           2}, {1}}, 
          
          Function[{$CellContext`point}, 
           Dot[{{-0.09, 0.28}, {0.3, 0.09}}, $CellContext`point] + {0., 0.7}],
           Evaluate], $CellContext`f1l = CompiledFunction[{8, 9., 5468}, {{
            Blank[Real], 1}}, {{3, 1, 0}, {3, 1, 
          2}}, {{{0., -0.14}, {3, 1, 3}}, {{{0., 0.}, {0., 0.25}}, {3, 2, 
            1}}, {4, {2, 0, 0}}}, {0, 1, 0, 0, 
          4}, {{42, "Dot", 3, 2, 1, 3, 1, 0, 2, 0, 0, 3, 1, 2}, {44, 2, 3, 
           2}, {1}}, 
          
          Function[{$CellContext`point}, 
           Dot[{{0., 0.}, {0., 0.25}}, $CellContext`point] + {0., -0.14}], 
          Evaluate]}; Typeset`initDone$$ = True); ReleaseHold[
       HoldComplete[{($CellContext`p1 = 0.01; Null) ($CellContext`p2 = 0.85; 
          Null) ($CellContext`p3 = 0.07; 
          Null), ($CellContext`f1 = Compile[{{$CellContext`point, 
               Blank[Real], 1}}, 
             Dot[{{0., 0.}, {0., 0.16}}, $CellContext`point]]; 
          Null) ($CellContext`f2 = Compile[{{$CellContext`point, 
               Blank[Real], 1}}, 
             Dot[{{0.85, 0.04}, {-0.04, 0.85}}, $CellContext`point] + {0., 
              1.6}]; Null) ($CellContext`f3 = Compile[{{$CellContext`point, 
               Blank[Real], 1}}, 
             Dot[{{0.2, -0.26}, {0.23, 0.22}}, $CellContext`point] + {0., 
              1.6}]; Null) ($CellContext`f4 = Compile[{{$CellContext`point, 
               Blank[Real], 1}}, 
             Dot[{{-0.15, 0.28}, {0.26, 0.24}}, $CellContext`point] + {0., 
              0.44}]; Null), ($CellContext`p1t = 0.02; 
          Null) ($CellContext`p2t = 0.84; 
          Null), ($CellContext`f1t = Compile[{{$CellContext`point, 
               Blank[Real], 1}}, 
             Dot[{{0., 0.}, {0., 0.25}}, $CellContext`point] + {0., -0.4}]; 
          Null) ($CellContext`f2t = Compile[{{$CellContext`point, 
               Blank[Real], 1}}, 
             Dot[{{0.95, 0.005}, {-0.005, 
               0.93}}, $CellContext`point] + {-0.002, 0.5}]; 
          Null) ($CellContext`f3t = Compile[{{$CellContext`point, 
               Blank[Real], 1}}, 
             Dot[{{0.035, -0.2}, {0.16, 0.04}}, $CellContext`point] + {-0.09, 
              0.02}]; Null) ($CellContext`f4t = 
           Compile[{{$CellContext`point, 
               Blank[Real], 1}}, 
             Dot[{{-0.04, 0.2}, {0.16, 0.04}}, $CellContext`point] + {0.083, 
              0.12}]; Null), ($CellContext`f1l = Compile[{{$CellContext`point, 
               Blank[Real], 1}}, 
             Dot[{{0., 0.}, {0., 0.25}}, $CellContext`point] + {0., -0.14}]; 
          Null) ($CellContext`f2l = Compile[{{$CellContext`point, 
               Blank[Real], 1}}, 
             Dot[{{0.85, 0.02}, {-0.02, 0.83}}, $CellContext`point] + {0., 
              1.}]; Null) ($CellContext`f3l = Compile[{{$CellContext`point, 
               Blank[Real], 1}}, 
             Dot[{{0.09, -0.28}, {0.3, 0.11}}, $CellContext`point] + {0., 
              0.6}]; Null) ($CellContext`f4l = Compile[{{$CellContext`point, 
               Blank[Real], 1}}, 
             Dot[{{-0.09, 0.28}, {0.3, 0.09}}, $CellContext`point] + {0., 
              0.7}]; Null), $CellContext`point = {0., 0.}; 
         Null, $CellContext`generatePointB[
           Pattern[$CellContext`point, 
            Blank[]]] := ($CellContext`r = RandomReal[]; 
          Which[$CellContext`p1 < $CellContext`r < $CellContext`p1 + \
$CellContext`p2, 
            $CellContext`f2[$CellContext`point], $CellContext`p1 + \
$CellContext`p2 < $CellContext`r < $CellContext`p1 + $CellContext`p2 + \
$CellContext`p3, 
            $CellContext`f3[$CellContext`point], $CellContext`p1 + \
$CellContext`p2 + $CellContext`p3 < $CellContext`r, 
            $CellContext`f4[$CellContext`point], $CellContext`r < \
$CellContext`p1, 
            $CellContext`f1[$CellContext`point]]), $CellContext`generatePointT[
           Pattern[$CellContext`point, 
            Blank[]]] := ($CellContext`r = RandomReal[]; 
          Which[$CellContext`p1t < $CellContext`r < $CellContext`p1t + \
$CellContext`p2t, 
            $CellContext`f2t[$CellContext`point], $CellContext`p1t + \
$CellContext`p2t < $CellContext`r < $CellContext`p1t + $CellContext`p2t + \
$CellContext`p3, 
            $CellContext`f3t[$CellContext`point], $CellContext`p1t + \
$CellContext`p2t + $CellContext`p3 < $CellContext`r, 
            $CellContext`f4t[$CellContext`point], $CellContext`r < \
$CellContext`p1t, 
            $CellContext`f1[$CellContext`point]]), $CellContext`generatePointL[
           Pattern[$CellContext`point, 
            Blank[]]] := ($CellContext`r = RandomReal[]; 
          Which[$CellContext`p1t < $CellContext`r < $CellContext`p1t + \
$CellContext`p2t, 
            $CellContext`f2l[$CellContext`point], $CellContext`p1t + \
$CellContext`p2t < $CellContext`r < $CellContext`p1t + $CellContext`p2t + \
$CellContext`p3, 
            $CellContext`f3l[$CellContext`point], $CellContext`p1t + \
$CellContext`p2t + $CellContext`p3 < $CellContext`r, 
            $CellContext`f4l[$CellContext`point], $CellContext`r < \
$CellContext`p1t, 
            $CellContext`f1l[$CellContext`point]])}]]; 
     Typeset`initDone$$ = True),
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 CellID->1546394690],

Cell[CellGroupData[{

Cell["CAPTION", "Section",
 CellFrame->{{0, 0}, {1, 0}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->12,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell["\<\
This Demonstration plots three types of ferns: Barnsley, Thelypteridaceae, \
and Leptosporangiate. The ferns are obtained by repeatedly applying one of \
four affine transformations chosen randomly, starting at the origin. By \
adjusting matrix coefficients and probabilities, we can generate different \
types of ferns.\
\>", "Text"]
}, Close]]
}, Open  ]],

Cell["THIS NOTEBOOK IS THE SOURCE CODE FROM", "Text",
 CellFrame->{{0, 0}, {0, 0}},
 CellMargins->{{48, 10}, {4, 28}},
 CellGroupingRules->{"SectionGrouping", 25},
 CellFrameMargins->{{48, 48}, {6, 5}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->10,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[TextData[{
 "\[NonBreakingSpace]",
 ButtonBox["Elaine Lin",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/author.html?author=Elaine+Lin"], 
    None},
  ButtonNote->
   "http://demonstrations.wolfram.com/author.html?author=Elaine+Lin"]
}], "Text",
 CellMargins->{{48, Inherited}, {0, Inherited}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.6]],

Cell[TextData[{
 "\"",
 ButtonBox["Fern Fractals",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/FernFractals/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/FernFractals/"],
 "\"",
 "\[ParagraphSeparator]\[NonBreakingSpace]",
 ButtonBox["http://demonstrations.wolfram.com/FernFractals/",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/FernFractals/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/FernFractals/"],
 "\[ParagraphSeparator]\[NonBreakingSpace]",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
 "\[ParagraphSeparator]\[NonBreakingSpace]",
 "Published: ",
 "February 12, 2015"
}], "Text",
 CellMargins->{{48, Inherited}, {0, Inherited}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.5]],

Cell[CellGroupData[{

Cell[TextData[{
 "A full-function Wolfram ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " system is required to edit this notebook.\n",
 StyleBox[ButtonBox["GET WOLFRAM MATHEMATICA \[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolfram.com/products/mathematica/"], None},
  ButtonNote->"http://www.wolfram.com/products/mathematica/"],
  FontFamily->"Helvetica",
  FontWeight->"Bold",
  FontSlant->"Italic",
  FontColor->RGBColor[1, 0.42, 0]]
}], "Text",
 CellFrame->True,
 CellMargins->{{48, 68}, {8, 28}},
 CellFrameMargins->12,
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 CellChangeTimes->{3.3750111182355957`*^9},
 ParagraphSpacing->{1., 1.},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.411765],
 Background->RGBColor[1, 1, 1]],

Cell[TextData[{
 "\[Copyright] ",
 StyleBox[ButtonBox["Wolfram Demonstrations Project & Contributors",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Terms of Use",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/termsofuse.html"], None},
  ButtonNote->"http://demonstrations.wolfram.com/termsofuse.html"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Make a new version of this Demonstration \
\[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/participate/upload.jsp?id=\
FernFractals"], None},
  ButtonNote->None],
  FontColor->GrayLevel[0.6]]
}], "Text",
 CellFrame->{{0, 0}, {0, 0.5}},
 CellMargins->{{48, 10}, {20, 50}},
 CellFrameMargins->{{6, 0}, {6, 6}},
 CellFrameColor->GrayLevel[0.6],
 FontFamily->"Verdana",
 FontSize->9,
 FontColor->GrayLevel[0.6]]
}, Open  ]]
},
Editable->True,
Saveable->False,
ScreenStyleEnvironment->"Working",
CellInsertionPointCell->None,
WindowSize->{710, 650},
WindowMargins->{{Inherited, Inherited}, {Inherited, 0}},
WindowElements->{
 "StatusArea", "MemoryMonitor", "MagnificationPopUp", "VerticalScrollBar", 
  "MenuBar"},
WindowTitle->"Fern Fractals - Source",
DockedCells->{},
CellContext->Notebook,
FrontEndVersion->"9.0 for Microsoft Windows (32-bit) (November 20, 2012)",
StyleDefinitions->"Default.nb"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[615, 23, 133, 3, 70, "Section"],
Cell[751, 28, 3700, 64, 70, "Section"],
Cell[4454, 94, 989, 29, 70, "Input",
 InitializationCell->True,
 CellID->140126495],
Cell[5446, 125, 2934, 90, 70, "Input",
 InitializationCell->True,
 CellID->1493634199],
Cell[8383, 217, 605, 18, 70, "Input",
 InitializationCell->True,
 CellID->894911208],
Cell[8991, 237, 3129, 98, 70, "Input",
 InitializationCell->True,
 CellID->32957756],
Cell[12123, 337, 3162, 97, 70, "Input",
 InitializationCell->True,
 CellID->1321613716],
Cell[15288, 436, 245, 7, 70, "Input",
 InitializationCell->True,
 CellID->1493693289],
Cell[15536, 445, 992, 23, 70, "Input",
 InitializationCell->True,
 CellID->857600144],
Cell[16531, 470, 1006, 23, 70, "Input",
 InitializationCell->True,
 CellID->1922865191],
Cell[17540, 495, 1005, 23, 70, "Input",
 InitializationCell->True,
 CellID->113861434],
Cell[18548, 520, 2194, 49, 70, "Input"]
}, Open  ]],
Cell[CellGroupData[{
Cell[20779, 574, 14552, 276, 70, "Output",
 CellID->1546394690],
Cell[CellGroupData[{
Cell[35356, 854, 209, 6, 70, "Section"],
Cell[35568, 862, 344, 6, 70, "Text"]
}, Close]]
}, Open  ]],
Cell[35938, 872, 355, 9, 70, "Text",
 CellGroupingRules->{"SectionGrouping", 25}],
Cell[36296, 883, 407, 13, 70, "Text"],
Cell[36706, 898, 963, 27, 70, "Text"],
Cell[CellGroupData[{
Cell[37694, 929, 793, 24, 70, "Text"],
Cell[38490, 955, 1184, 33, 70, "Text"]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
(* NotebookSignature @Qj#vVKCx8KbsA0LFmhdIE4O *)
