(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 3.0,
MathReader 3.0, or any compatible application. The data for the notebook 
starts with the line of stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     77415,       2866]*)
(*NotebookOutlinePosition[     87620,       3185]*)
(*  CellTagsIndexPosition[     87044,       3159]*)
(*WindowFrame->Normal*)



Notebook[{
Cell[TextData[{
  "Numerics in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " 3.0"
}], "Title"],

Cell[CellGroupData[{

Cell["by Mark Sofroniou", "Author"],

Cell[TextData[{
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " 3.0 has several new and important features. They include a new input \
syntax for floating-point numbers, enhancements to the function ",
  StyleBox["N", "InlineInput"],
  ", and a mechanism for extending the functionality of symbolic constants. \
This article summarizes the precision model used by ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ", and explains these new features. The caching mechanism for numerical \
quantities is also described."
}], "Text"],

Cell[TextData[{
  "Version 3.0 of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " has several important new numerical features. They include a new input \
syntax for floating-point numbers, enhancements to the range arithmetic used \
to track the propagation of errors through numerical functions, efficient new \
techniques for handling exact numeric quantities, a caching mechanism for \
numerical expressions, and enhancements to the numericization function ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  ". "
}], "Text"],

Cell[TextData[{
  "This article introduces these new numerical features. Much of the \
functionality discussed here began with the pioneering work of Jerry B. \
Keiper. Parts of this article are based upon his technical report ",
  ButtonBox["[Keiper 1994]",
    ButtonData:>"R6",
    ButtonStyle->"Hyperlink"],
  ". We will not cover the significant enhancements to the numerical \
compiler, the new and enhanced standard numerical packages, and several new \
numerical functions in Version 3.0."
}], "Text"],

Cell[TextData[{
  "To set the new features in context, we begin with a summary of the \
arithmetic model used in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell["Range Arithmetic", "Section"],

Cell[TextData[{
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "'s bigfloat arithmetic model is a variant of interval arithmetic where a \
single floating point number is used to represent the error. This is formally \
known as ",
  StyleBox["significance",
    FontSlant->"Italic"],
  " or ",
  StyleBox["range",
    FontSlant->"Italic"],
  " arithmetic ",
  ButtonBox["[Keiper 1995]",
    ButtonData:>"R7",
    ButtonStyle->"Hyperlink"],
  ". When the number of significant digits is not too small, this model \
accurately parallels interval arithmetic whilst being considerably more \
efficient. The model can be used to reflect the conditioning of an algorithm \
and therefore gives a good indication of how many digits in a result can be \
trusted (see ",
  ButtonBox["[Sofroniou 1996]",
    ButtonData:>"R10",
    ButtonStyle->"Hyperlink"],
  " for an example). This mechanism has important implications, such as the \
accurate determination of branch cuts in numerical functions."
}], "Text"],

Cell[TextData[{
  "The ",
  StyleBox["scale",
    FontSlant->"Italic"],
  " of a number is the number of digits to the left of the decimal point. We \
can define the decimal scale as follows."
}], "Text"],

Cell[BoxData[
    \(\(Scale[x_] := \ Log[10, Abs[x]]; \)\)], "InputOnly"],

Cell[TextData[{
  "Then the model used in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "'s bigfloat arithmetic is"
}], "Text"],

Cell[BoxData[
    \(Precision[x]\  == \ Scale[x]\  + \ Accuracy[x]\)], "InlineInput",
  TextAlignment->Center,
  FontSize->18],

Cell[TextData[{
  "For reasons of backward compatibility, ",
  StyleBox["Precision", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["Accuracy", "InlineInput",
    FontWeight->"Bold"],
  " return integer results even though these values are stored internally as \
floats. The equality above holds only for the real-valued extensions of these \
functions. Here is a visual interpretation of this relation."
}], "Text"],

Cell[BoxData[
    \(\(\(\ x\_1\ x\_2\ \[CenterEllipsis]\ x\_s\)\+\(\[UnderBrace]\+Scale\).
        \(x\_\(s + 1\)\ x\_\(s + 2\)\ \[CenterEllipsis]\ 
            x\_\(s + a\)\)\+\(\[UnderBrace]\+Accuracy
              \)\)\&\(\[OverBrace]\&Precision\)\)], "InlineInput",
  Evaluatable->False,
  TextAlignment->Center,
  FontSize->24],

Cell[TextData[{
  "The ",
  Cell[BoxData[
      StyleBox[\(x\_i\),
        "InlineInput"]]],
  " denote (decimal) digits of a number. The ",
  StyleBox["Accuracy", "InlineInput",
    FontWeight->"Bold"],
  " is the number of digits to the right of the decimal point and the ",
  StyleBox["Precision", "InlineInput",
    FontWeight->"Bold"],
  " is the total number of significant digits. Of course, the scale of a \
number can be negative, as can the precision and accuracy."
}], "Text"],

Cell["\<\
No attempt is made to keep track of the precision or accuracy of \
machine floats. The assumption is that if you are using machine numbers, you \
are primarily concerned with efficiency. Furthermore, machine arithmetic is \
contagious. The result of an operation that involves a machine number should, \
if possible, be a machine number.\
\>", "Text"],

Cell[TextData[{
  "We need a mechanism to enable errors to be propagated automatically \
through computations. For simple functions, the propagated error can be easy \
to determine using the ",
  StyleBox["condition number",
    FontSlant->"Italic"],
  " ",
  ButtonBox["[Skeel and Keiper 1993]",
    ButtonData:>"R9",
    ButtonStyle->"Hyperlink"],
  ". However, as expressions become very large, it can become tedious or \
impractical to undertake a rigorous numerical stability analysis."
}], "Text"],

Cell[TextData[{
  "Our goal is to find a way to represent a number so that we can \
independently certify its accuracy or precision, and hence determine its \
error, regardless of the means by which it was computed. Given this \
information, we need to equip our numerical functions with the ability to \
propagate errors. ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "'s numerical functions are designed to reflect the correct amount of \
uncertainty for a given input. The elementary arithmetic functions ",
  StyleBox["Plus", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["Times", "InlineInput",
    FontWeight->"Bold"],
  " then propagate errors between functions assuming that errors are ",
  StyleBox["independent",
    FontSlant->"Italic"],
  "."
}], "Text"],

Cell[TextData[{
  "The way that ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " measures errors is to associate with a number self-contained and readily \
accessible information about the degree of uncertainty. Each bigfloat is \
effectively a pseudo-interval with error ",
  Cell[BoxData[
      RowBox[{
        StyleBox[\(10\^\(-a\)\),
          "InlineInput"], ","}]]],
  "where ",
  StyleBox["a", "InlineInput"],
  " is the ",
  StyleBox["Accuracy", "InlineInput",
    FontWeight->"Bold"],
  ". Thus, the ",
  StyleBox["Accuracy", "InlineInput",
    FontWeight->"Bold"],
  " is the negative of the scale of the error. The ",
  StyleBox["Accuracy", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["Precision", "InlineInput",
    FontWeight->"Bold"],
  " represent the scales of the absolute and relative errors. The absolute \
error is ",
  StyleBox["Log[10,Abs[", "InlineInput",
    FontWeight->"Bold"],
  StyleBox["error", "InlineInput",
    FontWeight->"Bold",
    FontSlant->"Italic"],
  StyleBox["]]", "InlineInput",
    FontWeight->"Bold"],
  " and the relative error is ",
  StyleBox["Log[10,Abs[", "InlineInput",
    FontWeight->"Bold"],
  StyleBox["error", "InlineInput",
    FontWeight->"Bold",
    FontSlant->"Italic"],
  StyleBox["/x]]", "InlineInput",
    FontWeight->"Bold"],
  ". Since the ",
  StyleBox["Accuracy", "InlineInput",
    FontWeight->"Bold"],
  " and the ",
  StyleBox["Precision ", "InlineInput",
    FontWeight->"Bold"],
  "are actually fields in the bignum structure, the scale can be immediately \
ascertained as the difference of these two quantities."
}], "Text"],

Cell[TextData[{
  "The amount of precision that you should expect in a result varies \
according to each function. Certain functions can increase the amount of \
precision. The way that errors are propagated can be illustrated using the ",
  StyleBox["differential quotient",
    FontSlant->"Italic"],
  ", which we can write as"
}], "Text"],

Cell[BoxData[
    \(TraditionalForm
    \`\(\[CapitalDelta]f(z)\  \[TildeTilde] \ f' \((z)\)\ \[CapitalDelta]z, 
    \)\)], "InlineInput",
  TextAlignment->Center,
  TextJustification->0,
  FontSize->24],

Cell[TextData[{
  "where \[CapitalDelta] represents the difference operator. The differential \
quotient is a linearized approximation estimating the variation of a \
function. As an example, the error function ",
  StyleBox["Erf", "InlineInput",
    FontWeight->"Bold"],
  " can give a result that is much more precise than the input."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(z\  = \ N[20, 20]; \)\)], "InputOnly",
  CellGroupingRules->"InputGrouping"],

Cell[BoxData[
    \(Erf[z]\)], "Input",
  CellGroupingRules->"OutputGrouping"],

Cell[BoxData[
    \(0.9999999999999999999999999999999999999999999999999999999999999999999999\
999999999999999999999999999999999999999999999999999999999999999999999999999999\
99999999999999999999999999946041343883920990710650008445`192.4815\)], "Output"]
}, Open  ]],

Cell["The scale of the error is", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(-Accuracy[%]\)\)], "Input"],

Cell[BoxData[
    \(\(-192\)\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The following function projects an error interval for an arbitrary \
differentiable function ",
  StyleBox["f", "InlineInput",
    FontWeight->"Bold"],
  " at a point ",
  StyleBox["z", "InlineInput",
    FontWeight->"Bold"],
  " using the differential quotient."
}], "Text"],

Cell[BoxData[
    \(\(PropagateError[f_, z_] := \ \(f'\)[z]*10^\(-Accuracy[z]\); \)\)], 
  "InputOnly"],

Cell[TextData[{
  "We use this function to propagate the error for ",
  StyleBox["Erf", "InlineInput",
    FontWeight->"Bold"],
  " at the value ",
  StyleBox["z", "InlineInput",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(PropagateError[Erf, z]\)], "Input"],

Cell[BoxData[
    \(2.1610374743867984610095633243`17.0969*^-193\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Recompute the scale of the error and compare it to the ",
  StyleBox["Accuracy", "InlineInput",
    FontWeight->"Bold"],
  " of the result returned by ",
  StyleBox["Erf", "InlineInput",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Log[10, %]\)], "Input"],

Cell[BoxData[
    \(\(-192.66533770198381679241346199`17.4552\)\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "To illustrate the correspondence between range arithmetic and interval \
arithmetic, we adapt an example from ",
  ButtonBox["[Jacobson 1992]",
    ButtonData:>"R5",
    ButtonStyle->"Hyperlink"],
  ". Given a floating point number, double it and subtract the original \
number. Repeating this process we might expect to continue to lose the least \
significant bit from our internal binary representation. The result obtained \
using ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " can be more pessimistic."
}], "Text"],

Cell["Begin with a number with 30 decimal digits.", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(x\_1 = N[1/9, 30]\)], "Input"],

Cell[BoxData[
    \(0.111111111111111111111111111111111111109805`30\)], "Output"]
}, Open  ]],

Cell["\<\
Using fixed precision we might expect to lose 25 binary digits in \
25 iterations, which is roughly 7.5 decimal digits.\
\>", "Text"],

Cell[BoxData[
    \(Do[x\_1 = 2\ x\_1 - x\_1, {25}]\)], "InputOnly"],

Cell["\<\
The result is actually worse, in that we have lost around 12 \
significant decimal digits.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \({Precision[x\_1], x\_1}\)], "Input"],

Cell[BoxData[
    \({18, 0.111111111111111111111111111111`18.072}\)], "Output"]
}, Open  ]],

Cell["\<\
The scale of the error is 19 digits to the right of the decimal \
point.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Accuracy[x\_1]\)], "Input"],

Cell[BoxData[
    \(19\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Now, let us repeat the example using ",
  StyleBox["exact",
    FontSlant->"Italic"],
  " interval arithmetic. Using an exact interval avoids the outward rounding \
of inexact endpoints that would otherwise occur at each step of the \
computation. The midpoint of the interval is"
}], "Text"],

Cell[BoxData[
    \(\(midpt = 111111111111111111111111111111/10^30; \)\)], "InputOnly"],

Cell["and the offset (half the width of the interval) is", "Text"],

Cell[BoxData[
    \(\(offset = 10^\(-31\)/2; \)\)], "InputOnly"],

Cell[TextData[{
  "Here is an interval ",
  Cell[BoxData[
      StyleBox[\(x\_2\),
        "InlineInput",
        FontWeight->"Bold"]]],
  " with the same uncertainty as our original number ",
  Cell[BoxData[
      \(x\_1\)], "InlineInput",
    FontWeight->"Bold"],
  ". "
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(x\_2 = Interval[{midpt - offset, midpt + offset}]\)], "Input"],

Cell[BoxData[
    \(Interval[{
        2222222222222222222222222222219/20000000000000000000000000000000, 
        2222222222222222222222222222221/20000000000000000000000000000000}]\)],
   "Output"]
}, Open  ]],

Cell["We repeat the previous iteration using the exact interval.", "Text"],

Cell[BoxData[
    \(Do[x\_2 = 2\ x\_2 - x\_2, {25}]\)], "InputOnly"],

Cell[TextData[{
  "Next, we determine the scale of the width of the interval and find that it \
is very close indeed to the accuracy of the value that was obtained using ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "'s range arithmetic."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(IntervalWidth[Interval[{l_, u_}]] := \ u - l; \)\)], "InputOnly"],

Cell[BoxData[
    \(\(-Log[10., IntervalWidth[x\_2]\ ]\)\)], "Input",
  CellGroupingRules->"OutputGrouping"],

Cell[BoxData[
    \(19.0719686320084349`\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "It is well known that interval arithmetic can sometimes give overly \
pessimistic results, a phenomenon referred to as the ",
  StyleBox["dependency problem",
    FontSlant->"Italic"],
  ". As we have seen, the assumption of independence of errors in range \
arithmetic can have a similar effect. Some heuristics for rearranging \
expressions to improve the bounds given by interval arithmetic are discussed \
in ",
  ButtonBox["[Hansen 1992]",
    ButtonData:>"R3",
    ButtonStyle->"Hyperlink"],
  ". Iterative algorithms must generally be recast for interval computations. \
A user who has information about how errors are propagated can manually \
specify how errors combine, using the functions ",
  StyleBox["SetPrecision", "InlineInput",
    FontWeight->"Bold"],
  " or ",
  StyleBox["SetAccuracy", "InlineInput",
    FontWeight->"Bold"],
  ". A user also has the choice of working with fixed-precision arithmetic by \
setting values for ",
  StyleBox["$MinPrecision", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["$MaxPrecision", "InlineInput",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[TextData[{
  "The primary goal of range arithmetic is to give a good estimate of the \
number of correct digits in a result, whilst being efficient and accurate \
most of the time. The arithmetic model works well provided that the length of \
the interval representing the error is short relative to the magnitude of the \
numbers represented. However, some problems require a rigorous error \
analysis. For such problems, ",
  StyleBox["bona fide",
    FontSlant->"Italic"],
  " interval arithmetic can be used."
}], "Text"],

Cell[TextData[{
  "Here are some more examples to illustrate various aspects of the \
arithmetic model. The values of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "'s accuracy and precision are machine floats, although the functions ",
  StyleBox["Accuracy", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["Precision", "InlineInput",
    FontWeight->"Bold"],
  " round results to the nearest integer value."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(num\  = \ SetPrecision[23/10, 21.2]; \)\)], "InputOnly"],

Cell[BoxData[
    \(Precision[num]\)], "Input",
  CellGroupingRules->"OutputGrouping"],

Cell[BoxData[
    \(21\)], "Output"]
}, Open  ]],

Cell["A trap-door mechanism can be used to switch off rounding.", "Text"],

Cell[BoxData[
    \(\(SetPrecision[Round, False]; \)\)], "InputOnly"],

Cell["Now, the precision and accuracy are real-valued functions.", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Precision[num]\)], "Input"],

Cell[BoxData[
    \(21.2000016164808746`\)], "Output"]
}, Open  ]],

Cell["This switches rounding back on again.", "Text"],

Cell[BoxData[
    \(\(SetPrecision[Round, True]; \)\)], "InputOnly"],

Cell[TextData[{
  "The precision of a bigfloat can be negative (although negative precision \
can be a little confusing). The default value of ",
  StyleBox["$MinPrecision", "InlineInput",
    FontWeight->"Bold"],
  " bounds numbers with negative precision. Here is the default behavior when \
the set precision is negative."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(SetPrecision[1, \(-10\)]\)], "Input"],

Cell[BoxData[
    \(SetPrecision::"precsm" \( : \ \) 
      "Requested precision \!\(-10\) is smaller than $MinPrecision. Using \
$MinPrecision instead."\)], "Message"],

Cell[BoxData[
    \(1.`0\)], "Output"]
}, Open  ]],

Cell["\<\
We can change the default to allow numbers to attain negative \
precision.\
\>", "Text"],

Cell[BoxData[
    \(\($MinPrecision = \ \(-20\); \)\)], "InputOnly"],

Cell["\<\
Numbers with zero or negative precision are printed as zero, since \
none of the digits are known to be correct.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(SetPrecision[1, \(-10\)]\)], "Input"],

Cell[BoxData[
    \(1.`-10\)], "Output"]
}, Open  ]],

Cell["This restores the system default value.", "Text"],

Cell[BoxData[
    \(\($MinPrecision = 0; \)\)], "InputOnly"],

Cell[TextData[{
  "The precisions of the real and imaginary parts of a complex number are \
independent. The scale of a complex number is given by the absolute value ",
  StyleBox["Abs", "InlineInput",
    FontWeight->"Bold"],
  ", the ",
  StyleBox["Accuracy", "InlineInput",
    FontWeight->"Bold"],
  " is the minimum of the accuracies of the real and imaginary parts, and the \
",
  StyleBox["Precision", "InlineInput",
    FontWeight->"Bold"],
  " is calculated as the sum of the accuracy and the scale. To illustrate, \
let us define a complex number whose real and imaginary parts have different \
scales and precisions."
}], "Text"],

Cell[BoxData[
    \(\(z\  = \ 
      SetPrecision[10^20, 200]\  + \ I\ SetPrecision[10^\(-50\), 100]; \)\)], 
  "InputOnly"],

Cell[TextData[{
  "Computing the decimal ",
  StyleBox["Scale", "InlineInput",
    FontWeight->"Bold"],
  ", the ",
  StyleBox["Accuracy", "InlineInput",
    FontWeight->"Bold"],
  ", and the ",
  StyleBox["Precision", "InlineInput",
    FontWeight->"Bold"],
  " shows the connection between these quantities."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \({Floor[Log[10, Abs[z]]], \ Accuracy[z], \ Precision[z]}\)], "Input"],

Cell[BoxData[
    \({20, 150, 170}\)], "Output"]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["New Syntax of Floating-Point Numbers", "Section"],

Cell[TextData[{
  "The changes to ",
  StyleBox["InputForm", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["FullForm", "InlineInput",
    FontWeight->"Bold"],
  " in Version 3.0 address several problems. In the past, the ",
  StyleBox["FullForm", "InlineInput",
    FontWeight->"Bold"],
  " of a very large or small number was given in \"scientific\" notation with \
a raised exponent. This raised exponent made the character string unparsable \
and ",
  StyleBox["FullForm", "InlineInput",
    FontWeight->"Bold"],
  " could not be used as input. In Version 3.0, ",
  StyleBox["FullForm", "InlineInput",
    FontWeight->"Bold"],
  " of floating-point numbers has been changed to be the same as ",
  StyleBox["InputForm", "InlineInput",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[TextData[{
  "Another limitation was that in scientific notation, numbers became \
multiplication problems. For example, to evaluate ",
  StyleBox["1.234*10^56789", "InlineInput",
    FontWeight->"Bold"],
  ", ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " would multiply the two numbers ",
  StyleBox["1.234", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["10^56789", "InlineInput",
    FontWeight->"Bold"],
  ". The multiplication of the inexact and exact numbers was not so much of a \
problem as the computation and representation of the exponent ",
  Cell[BoxData[
      StyleBox[\(10\^56789\),
        "InlineInput"]]],
  ", which requires ",
  Cell[BoxData[
      FormBox[
        StyleBox["56790",
          "InlineInput",
          FontWeight->"Bold"], TraditionalForm]], "InlineFormula"],
  " decimal digits. This was wasteful of time and memory."
}], "Text"],

Cell[TextData[{
  "A further problem with the earlier ",
  StyleBox["InputForm", "InlineInput",
    FontWeight->"Bold"],
  " was that the specification of certain types of numbers was ambiguous. ",
  StyleBox["InputForm", "InlineInput",
    FontWeight->"Bold"],
  " specified the precision of arbitrary-precision numbers by counting digits \
and did not display hidden digits. By merely counting digits, \
arbitrary-precision numbers with low precision were given in a form that \
could potentially be mistaken for machine numbers."
}], "Text"],

Cell[TextData[{
  "The new ",
  StyleBox["InputForm", "InlineInput",
    FontWeight->"Bold"],
  " of a floating-point number consists of three fields: the mantissa, the \
precision (or accuracy), and the exponent. The mantissa and precision are \
separated by a backquote symbol (",
  StyleBox["`", "InlineInput"],
  ") and the exponent is preceded by ",
  StyleBox["*^", "InlineInput",
    FontWeight->"Bold"],
  ". For example, the input syntax for the bigfloat 1.1",
  Cell[BoxData[
      \(TraditionalForm\`\((\[Cross]10)\)\^\(-5\)\)]],
  " with 6.25 digits of precision is"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(1.1`6.25*^-5\)], "Input"],

Cell[BoxData[
    \(0.000011`6.25\)], "Output"]
}, Open  ]],

Cell["\<\
An empty exponent field indicates that the exponent is zero; an \
empty precision field indicates a machine number. The machine number 1.25 is \
\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(1.25`\)], "Input"],

Cell[BoxData[
    \(1.25`\)], "Output"]
}, Open  ]],

Cell["and the bigfloat 3.0 with 7.7124 digits of precision is", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(3.`7.7124\)], "Input"],

Cell[BoxData[
    \(3.`7.7124\)], "Output"]
}, Open  ]],

Cell["\<\
Not all decimal numbers are exactly representable as (finite) \
binary numbers. This gives the machine number closest to the decimal number \
0.1.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(1.*^-1\)], "Input"],

Cell[BoxData[
    \(0.1`\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The representation error is not apparent because it is smaller than \
machine precision and is masked by the output formatting code. In contrast, \
the bigfloat ",
  StyleBox["1`100*^-1", "InlineInput",
    FontWeight->"Bold"],
  " is guaranteed to have 100 decimal digits. Using bigfloats, representation \
errors can thus be made small enough for all practical purposes. In addition, \
we could use rational arithmetic for exact representation."
}], "Text"],

Cell[TextData[{
  "Unfortunately, the concept of precision has a singularity at zero. We need \
a way to specify zeros to various accuracies. The syntax for specifying the \
accuracy of a number is the same as that for specifying the precision, except \
that ",
  StyleBox["`", "InlineInput",
    FontWeight->"Bold"],
  " is replaced by ",
  StyleBox["``", "InlineInput",
    FontWeight->"Bold"],
  ". ",
  StyleBox["InputForm", "InlineInput",
    FontWeight->"Bold"],
  " only uses the accuracy notation to represent the number zero, but the \
parser will accept it as a specification of accuracy for other numbers as \
well. Here is an inexact zero with accuracy 23.2 digits."
}], "Text",
  CellTags->"S0.0.1"],

Cell[CellGroupData[{

Cell[BoxData[
    \(0``23.2\)], "Input"],

Cell[BoxData[
    \(0``23.2\)], "Output"]
}, Open  ]],

Cell["\<\
This bigfloat has accuracy 7.7124 digits. Notice how significant \
trailing zeros are displayed.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(3``7.7124\)], "Input"],

Cell[BoxData[
    \(3.`8.1895\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "This bigfloat has scale ",
  Cell[BoxData[
      \(TraditionalForm\`10\^\(-5\)\)]],
  " and accuracy 5.6 digits. Significant leading zeros are displayed."
}], "Text",
  CellTags->"S0.0.1"],

Cell[CellGroupData[{

Cell[BoxData[
    \(1.``5.6*^-5\)], "Input"],

Cell[BoxData[
    \(0.00001`0.6\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Because there is no attempt to keep track of significant digits with \
machine arithmetic, the machine numbers ",
  StyleBox["3`", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["3``", "InlineInput",
    FontWeight->"Bold"],
  " are exactly the same: each gives a machine number without regard to \
either the precision or the accuracy."
}], "Text"],

Cell[TextData[{
  "To maintain compatibility both with earlier versions of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " and with other programs, Version 3.0 has a global variable, ",
  StyleBox["$NumberMarks", "InlineInput",
    FontWeight->"Bold"],
  ", which can have one of three possible values: ",
  StyleBox["True", "InlineInput",
    FontWeight->"Bold"],
  ", ",
  StyleBox["False", "InlineInput",
    FontWeight->"Bold"],
  ", or ",
  StyleBox["Automatic", "InlineInput",
    FontWeight->"Bold"],
  ". When the value is ",
  StyleBox["True", "InlineInput",
    FontWeight->"Bold"],
  ", the syntax used involves ` and *^ for the ",
  StyleBox["InputForm", "InlineInput",
    FontWeight->"Bold"],
  " of all floating point numbers. The value ",
  StyleBox["False", "InlineInput",
    FontWeight->"Bold"],
  " indicates that the syntax is not to be used at all. The default value is \
",
  StyleBox["Automatic", "InlineInput",
    FontWeight->"Bold"],
  ", which indicates that the syntax is used for bigfloats but not for \
machine numbers. "
}], "Text",
  CellTags->"S0.0.1"],

Cell[CellGroupData[{

Cell[BoxData[
    \(InputForm[\ N[Pi, 20]\ ]\)], "Input"],

Cell["\<\
3.141592653589793238462643383\\
2795`20\
\>", "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(InputForm[\ N[Pi]\ ]\)], "Input"],

Cell["3.141592653589793", "Output"]
}, Open  ]],

Cell["\<\
You can turn off the syntax for both machine and \
arbitrary-precision numbers.\
\>", "Text",
  CellTags->"S0.0.2"],

Cell[BoxData[
    \(\($NumberMarks = False; \)\)], "InputOnly",
  CellTags->"S0.0.2"],

Cell["\<\
Now, neither machine numbers nor arbitrary-precision numbers are \
shown with the new syntax.\
\>", "Text",
  CellTags->"S0.0.2"],

Cell[CellGroupData[{

Cell[BoxData[
    \(InputForm[\ 1.11`22.3\ ]\)], "Input"],

Cell["1.11", "Output"]
}, Open  ]],

Cell["This resets the default.", "Text",
  CellTags->"S0.0.2"],

Cell[BoxData[
    \(\($NumberMarks = Automatic; \)\)], "InputOnly"],

Cell[TextData[{
  "Various functions like ",
  StyleBox["OpenWrite", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["OpenAppend", "InlineInput",
    FontWeight->"Bold"],
  " have the option ",
  StyleBox["NumberMarks", "InlineInput",
    FontWeight->"Bold"],
  ", which can take the value ",
  StyleBox["$NumberMarks", "InlineInput",
    FontWeight->"Bold"],
  " or any of its three valid values. Likewise, all streams have the option ",
  
  StyleBox["NumberMarks", "InlineInput",
    FontWeight->"Bold"],
  " with the same four valid values."
}], "Text",
  CellTags->"S0.0.2"]
}, Closed]],

Cell[CellGroupData[{

Cell["The N Function", "Section",
  CellTags->"S0.0.3"],

Cell[TextData[{
  "A frustration with ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " in previous versions has been that it would sometimes give a result with \
less precision than was requested. For example, if you asked for ",
  StyleBox["N[\[Pi] - q, 20]", "InlineInput",
    FontWeight->"Bold"],
  " where ",
  StyleBox["q", "InlineInput",
    FontWeight->"Bold"],
  " is a rational number very near ",
  Cell[BoxData[
      FormBox[
        StyleBox["\[Pi]",
          "InlineInput",
          FontWeight->"Bold"], TraditionalForm]], "InlineFormula",
    CellTags->"S0.0.7"],
  ", you would get a result with much less precision than 20 digits. This is \
because ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " was applied to each of ",
  StyleBox["\[Pi]", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["q", "InlineInput",
    FontWeight->"Bold"],
  ", and the result was the difference, which lost digits due to \
cancellation. In Version 3.0, if digits are lost due to cancellation, the \
working precision is increased in an effort to get the requested precision in \
the result."
}], "Text",
  CellTags->"S0.0.7"],

Cell[TextData[{
  "To achieve the desired precision or accuracy, the new algorithm for ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " attempts to anticipate the required precision with which to call a \
fixed-precision numericization routine. This is done by keeping track of the \
deficiency in the resulting precision or accuracy as a function of the \
precision used. Normally, but not always, this function will be linear with \
slope one."
}], "Text"],

Cell[TextData[{
  "For example, this rational number is close to ",
  Cell[BoxData[
      FormBox[
        StyleBox["\[Pi]",
          "InlineInput",
          FontWeight->"Bold"], TraditionalForm]], "InlineFormula",
    CellTags->"S0.0.7"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(q = 6134899525417045/1952799169684491; \)\)], "InputOnly"],

Cell[TextData[{
  "Here is a 20-digit approximation to the difference ",
  StyleBox["\[Pi]-q", "InlineInput",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    StyleBox[\(N[\[Pi] - q, 20]\),
      "InlineInput",
      FontWeight->"Bold"]], "Input"],

Cell[BoxData[
    \(4.862714977556358872222591476946`20*^-32\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Precision[%]\)], "Input"],

Cell[BoxData[
    \(20\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "In trying to achieve a requested precision, ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " stops increasing the working precision when the extra precision used \
reaches ",
  StyleBox["$MaxExtraPrecision", "InlineInput",
    FontWeight->"Bold"],
  ". The default value of this global parameter is 50 digits, but you can set \
it to higher or lower values. For example, this constructs a rational number \
that is even closer to \[Pi]."
}], "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(q2 = Rationalize[\[Pi]\ , 1*^-100]\)], "Input"],

Cell[BoxData[
    \(394372834342725903069943709807632345074473102456264/
      125532772013612015195543173729505082616186012726141\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[\[Pi] - q2, 20]\)], "Input"],

Cell[BoxData[
    \($MaxExtraPrecision::"meprec" \( : \ \) 
      "$MaxExtraPrecision = \!\(50.`\) reached while evaluating \
\!\(\(-\(\(39437283434272590306 \(\\[LeftSkeleton] 10 \\[RightSkeleton]\) \
632345074473102456264\)\/\(12553277201361201519 \(\\[LeftSkeleton] 10 \
\\[RightSkeleton]\) 505082616186012726141\)\)\) + \\[Pi]\). Increasing the \
value of $MaxExtraPrecision may help resolve the uncertainty."\)], "Message"],

Cell[BoxData[
    \(1.155`-8.1894*^-91\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Set ",
  StyleBox["$MaxExtraPrecision", "InlineInput",
    FontWeight->"Bold"],
  " to 200 digits."
}], "Text",
  CellTags->"S0.0.7"],

Cell[BoxData[
    \(\($MaxExtraPrecision = 200; \)\)], "InputOnly",
  CellTags->"S0.0.7"],

Cell["\<\
The difference can now be computed to 20 digits of precision.\
\>", 
  "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[\[Pi] - q2, 20]\)], "Input"],

Cell[BoxData[
    \(5.752043867880545649130287813495`20*^-102\)], "Output"]
}, Open  ]],

Cell["This resets the default value.", "Text",
  CellTags->"S0.0.7"],

Cell[BoxData[
    \(\($MaxExtraPrecision = 50; \)\)], "InputOnly",
  CellTags->"S0.0.7"],

Cell["\<\
Of course, this approach could lead to difficulties: the two \
numeric quantities could have exactly the same value and no amount of \
increased precision would give a satisfactory result. This expression is \
equal to zero, but inexact heuristics cannot be used to determine that \
fact.\
\>", "Text",
  CellTags->"S0.0.7"],

Cell[BoxData[
    \(\(z = \@2 + \@3 - \@\(5 + 2\ \@6\); \)\)], "InputOnly",
  CellTags->"S0.0.7"],

Cell[TextData[{
  "You should realize that if we compute a numerical approximation using \
machine numbers, there is no ",
  StyleBox["guarantee",
    FontSlant->"Italic"],
  " that any of the digits in the result are correct. The result can depend \
upon the number of bits of the mantissa, which effectively determines the \
precision of a machine's floating-point arithmetic."
}], "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[z]\)], "Input",
  CellTags->"S0.0.7"],

Cell[BoxData[
    \(4.44089209850062616`*^-16\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Here, we explicitly request 20 significant digits in the result. ",
  StyleBox["Mathematica", "SO",
    FontSlant->"Italic"],
  " gives up when the working precision used exceeds 70 digits. In contrast \
to the results of machine arithmetic, this result is consistent across \
different computer platforms because the arbitrary-precision arithmetic is \
implemented in software."
}], "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[z, 20]\)], "Input",
  CellTags->"S0.0.7"],

Cell[BoxData[
    \($MaxExtraPrecision::"meprec" \( : \ \) 
      "$MaxExtraPrecision = \!\(50.`\) reached while evaluating \!\(\@2 + \@3 \
- \@\(5 + \(2\\ \@6\)\)\). Increasing the value of $MaxExtraPrecision may \
help resolve the uncertainty."\)], "Message"],

Cell[BoxData[
    \(0``83.0486\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "In fact, using algebraic transformations on nested radicals, ",
  StyleBox["Mathematica", "SO",
    FontSlant->"Italic"],
  " can now show that the expression is indeed equal to zero."
}], "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(RootReduce[z]\)], "Input",
  CellTags->"S0.0.7"],

Cell[BoxData[
    \(0\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "There are other situations in which ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " may be unable to achieve a requested precision, for example when an \
expression contains low-precision inexact values. Here is a product with an \
inexact imaginary part."
}], "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((\[Pi] + E\ I)\)\ \((\[Pi] + 2.3\ I)\)\)], "Input",
  CellTags->"S0.0.7"],

Cell[BoxData[
    RowBox[{
      RowBox[{"(", 
        RowBox[{
          RowBox[{
            StyleBox["2.29999999999999982`",
              StyleBoxAutoDelete->True,
              PrintPrecision->2], " ", "I"}], "+", "\[Pi]"}], ")"}], " ", 
      \((I\ E + \[Pi])\)}]], "Output"]
}, Open  ]],

Cell["\<\
Here, we ask for 100 digits of precision, but the result only has \
machine precision.\
\>", "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[%, 100]\)], "Input",
  CellTags->"S0.0.7"],

Cell[BoxData[
    RowBox[{
      StyleBox["3.61755619563355512`",
        StyleBoxAutoDelete->True,
        PrintPrecision->16], "+", 
      RowBox[{
        StyleBox["15.7653973259300927`",
          StyleBoxAutoDelete->True,
          PrintPrecision->16], " ", "I"}]}]], "Output"]
}, Open  ]],

Cell[TextData[{
  "Sometimes, you may want to avoid using ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "'s new dynamic precision control mechanism. For example, you may already \
know how well or how badly conditioned your algorithm is or you may wish to \
use fixed-precision arithmetic. In such cases, you can set ",
  StyleBox["$MaxExtraPrecision", "InlineInput",
    FontWeight->"Bold"],
  " to zero to indicate that ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " should not increment precision. This setting gives the same results as \
you would have obtained in previous versions of ",
  StyleBox["Mathematica.",
    FontSlant->"Italic"]
}], "Text",
  CellTags->"S0.0.7"],

Cell[BoxData[
    \(\($MaxExtraPrecision = 0; \)\)], "InputOnly"],

Cell[TextData[{
  "Here is the evaluation of the difference ",
  StyleBox["\[Pi]-q", "InlineInput",
    FontWeight->"Bold"],
  " that we considered above using fixed precision arithmetic. There are no \
significant digits in the result, so it displays as zero. In order to see \
this, we first have to clear any numerical value that ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " may have stored (see the section on numerical caching)."
}], "Text",
  CellTags->"S0.0.7"],

Cell[BoxData[
    \(\(N[NumericQ, Clear]; \)\)], "InputOnly"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[\[Pi] - q, 20]\)], "Input"],

Cell[BoxData[
    \(2.2`-9.4617*^-29\)], "Output"]
}, Open  ]],

Cell[BoxData[
    \(\($MaxExtraPrecision = 50; \)\)], "InputOnly"],

Cell["\<\
A variant of this technology is used in many instances in order to \
adopt canonical forms for expressions. Canonical forms are essential because \
they assist in many types of expression simplification. The following radical \
is normalized by determining the sign of its argument. Since the argument is \
ascertained to be negative, the sign is extracted.\
\>", "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Sqrt[\(-\[Pi]\)]\)], "Input",
  CellTags->"S0.0.7"],

Cell[BoxData[
    \(I\ \@\[Pi]\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "If we were to encounter a hidden numeric zero as the argument, this \
canonicalization could become quite costly and futile. Therefore, a milder \
version of the mechanism used by ",
  StyleBox["N ", "InlineInput",
    FontWeight->"Bold"],
  "is adopted internally by many functions. Even though the argument here is \
negative, the sign is not extracted."
}], "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Sqrt[z - 1*^-50]\)], "Input"],

Cell[BoxData[
    \(\[Sqrt]\((
        \(-\(1/100000000000000000000000000000000000000000000000000\)\) + 
          \@2 + \@3 - \@\(5 + 2\ \@6\))\)\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "An N-value associated to a symbol is a number that specifies how ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " should evaluate expressions involving the symbol. For example, here is an \
N-value defined for the symbol ",
  StyleBox["x", "InlineInput",
    FontWeight->"Bold"],
  ". "
}], "Text"],

Cell[BoxData[
    \(\(N[x] = 3.0; \)\)], "InputOnly"],

Cell[TextData[{
  "This value is used in the numericization of an expression involving ",
  StyleBox["x", "InlineInput",
    FontWeight->"Bold"],
  "."
}], "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[Sin[x]]\)], "Input"],

Cell[BoxData[
    \(0.141120008059867103`\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "This gives the list of N-values of ",
  StyleBox["x", "InlineInput",
    FontWeight->"Bold"],
  "."
}], "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NValues[x]\)], "Input"],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(HoldPattern[N[x]]\), "\[RuleDelayed]", 
        StyleBox["3.`",
          StyleBoxAutoDelete->True,
          PrintPrecision->2]}], "}"}]], "Output"]
}, Open  ]],

Cell[TextData[{
  "Because the precision of 0 is not well defined, the values of ",
  StyleBox["N[0]", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["N[0, ", "InlineInput",
    FontWeight->"Bold"],
  StyleBox["prec",
    FontSlant->"Italic"],
  StyleBox["]", "InlineInput",
    FontWeight->"Bold"],
  " give the integer ",
  StyleBox["0", "InlineInput",
    FontWeight->"Bold"],
  " by default. (In fact, the precision of an exact zero is indeterminate \
while the precision of a floating-point zero is ",
  StyleBox["0.", "InlineInput",
    FontWeight->"Bold"],
  ") "
}], "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[0]\)], "Input"],

Cell[BoxData[
    \(0\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "However, one might sometimes prefer that ",
  StyleBox["N[0]", "InlineInput",
    FontWeight->"Bold"],
  " give a different result. What that value should be is open to question. \
Another new feature of the function ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " is that the values of ",
  StyleBox["N[0]", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["N[0, ", "InlineInput",
    FontWeight->"Bold"],
  StyleBox["prec",
    FontSlant->"Italic"],
  StyleBox["]", "InlineInput",
    FontWeight->"Bold"],
  " have been made user-definable. To set a value for ",
  StyleBox["N[0]", "InlineInput",
    FontWeight->"Bold"],
  ", first unprotect the system symbol ",
  StyleBox["Integer", "InlineInput",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(Unprotect[Integer]; \)\)], "InputOnly",
  CellTags->"S0.0.3"],

Cell[TextData[{
  "Assign a value to ",
  StyleBox["N[0]", "InlineInput",
    FontWeight->"Bold"],
  "."
}], "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[0] = 0.0\)], "Input",
  CellTags->"S0.0.3"],

Cell[BoxData[
    \(0.`\)], "Output"]
}, Open  ]],

Cell[TextData[{
  StyleBox["N[0]", "InlineInput",
    FontWeight->"Bold"],
  " is no longer the integer 0."
}], "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[0]\)], "Input",
  CellTags->"S0.0.3"],

Cell[BoxData[
    \(0.`\)], "Output"]
}, Open  ]],

Cell["This is how the definition is stored.", "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NValues[Integer]\)], "Input"],

Cell[BoxData[
    \({HoldPattern[N[0]] \[RuleDelayed] 0.`}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The two-argument form of ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " is unchanged."
}], "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[0, 23]\)], "Input",
  CellTags->"S0.0.3"],

Cell[BoxData[
    \(0\)], "Output"]
}, Open  ]],

Cell["We can define a rule for the two-argument form.", "Text",
  CellTags->"S0.0.3"],

Cell[BoxData[
    \(N[0, prec_] := SetAccuracy[0, prec]\)], "InputOnly",
  CellTags->"S0.0.3"],

Cell["Now the two-argument form works.", "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[0, 45]\)], "Input",
  CellTags->"S0.0.3"],

Cell[BoxData[
    \(0``45\)], "Output"]
}, Open  ]],

Cell[BoxData[
    \(\(Protect[Integer]; \)\)], "InputOnly",
  CellTags->"S0.0.3"],

Cell[TextData[{
  "Because precision and accuracy are stored internally as floating-point \
numbers, rather than being computed by simply counting digits, a rule such a \
",
  StyleBox["N[0, 23] = foo", "InlineInput",
    FontWeight->"Bold"],
  " will not work. You need to give the precision as a floating-point number, \
as in ",
  StyleBox["N[0, 23.] = foo", "InlineInput",
    FontWeight->"Bold"],
  ". "
}], "Text",
  CellTags->"S0.0.3"],

Cell[TextData[{
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " works by recursively going inside an expression and then numericizing \
from the leaves upward through the expression tree. It is sometimes useful to \
be able to control the evaluation of ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " on specific parts of an expression. The attributes ",
  StyleBox["NHoldFirst", "InlineInput",
    FontWeight->"Bold"],
  ", ",
  StyleBox["NHoldRest", "InlineInput",
    FontWeight->"Bold"],
  ", and ",
  StyleBox["NHoldAll", "InlineInput",
    FontWeight->"Bold"],
  " can be used for this purpose. For example, since arrays and functions \
have the same syntax in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ", we may wish to specify that certain expressions are indexed variables \
and that the indices should be treated differently. We can use the ",
  StyleBox["NHoldAll", "InlineInput",
    FontWeight->"Bold"],
  " attribute to protect the indices of an array from numericization."
}], "Text",
  CellTags->"S0.0.7"],

Cell[BoxData[
    \(\(SetAttributes[x, NHoldAll]; \)\)], "InputOnly"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[\[Pi]\ Sin[x[1]]]\)], "Input"],

Cell[BoxData[
    \(3.14159265358979311`\ Sin[x[1]]\)], "Output"]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Numeric Quantities", "Section",
  CellTags->"S0.0.4"],

Cell[TextData[{
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " has several kinds of expressions: numbers (either exact or inexact), \
expressions that represent numeric quantities but aren't numbers according to \
",
  StyleBox["NumberQ", "InlineInput",
    FontWeight->"Bold"],
  " (such as ",
  StyleBox["\[Pi]", "InlineInput",
    FontWeight->"Bold"],
  " or ",
  StyleBox["Sin[2]", "InlineInput",
    FontWeight->"Bold"],
  "), and expressions that do not represent numbers even after passing \
through ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " (such as ",
  StyleBox["Sin[x + y]", "InlineInput",
    FontWeight->"Bold"],
  "). When combining numbers with any of the four arithmetic operations, the \
result is always a number (except when division by zero, overflow, or \
underflow occurs). If the numbers are all exact, the result will likewise be \
exact. If inexact numbers are involved, the result will be inexact, unless \
all of the inexactness is lost due to multiplication by an exact zero. For \
example:"
}], "Text",
  CellTags->"S0.0.4"],

Cell[CellGroupData[{

Cell[BoxData[
    \(23\/11 + 1\/3\ \((2\/3 - 0.43)\) - 13\/7\)], "Input",
  CellTags->"S0.0.4"],

Cell[BoxData[
    \(0.312655122655122408`\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "In previous versions, arithmetic expressions involving non-number numeric \
quantities, such as ",
  StyleBox["0.5 + Sin[2]", "InlineInput",
    FontWeight->"Bold"],
  ", were returned unevaluated. Version 3.0 can evaluate such expressions \
efficiently to get an inexact number whose precision is determined using \
machine arithmetic or by the accuracy of the arguments."
}], "Text",
  CellTags->"S0.0.4"],

Cell[CellGroupData[{

Cell[BoxData[
    \(0.5\  + \ Sin[2]\)], "Input"],

Cell[BoxData[
    \(1.40929742682568175`\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "To perform this addition, the exact numeric quantity ",
  StyleBox["Sin[2]", "InlineInput",
    FontWeight->"Bold"],
  " must be \"coerced\" into an inexact number with a certain precision. Here \
is another example."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(1.`50*^-10\  + \ \[Pi]^100\)], "Input"],

Cell[BoxData[
    \(5.1878483143196131920862615246303013562686760680405784990071942358082188\
14439170444889750694577840234220453514273665432055`109.715*^49\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The scale of the numeric quantity ",
  StyleBox["\[Pi]^100", "InlineInput",
    FontWeight->"Bold"],
  " is very large. The amount of precision required to coerce it is \
determined by the accuracy of the inexact number."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Accuracy[1.`50*^-10]\)], "Input"],

Cell[BoxData[
    \(60\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Accuracy[%%]\)], "Input"],

Cell[BoxData[
    \(60\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "In order to distinguish numeric quantities from other expressions, ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " needs to be able to tell quickly whether an expression would become a \
number if it were encountered by the function ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  ". This is done by marking numeric quantities during evaluation. The new \
predicate ",
  StyleBox["NumericQ", "InlineInput",
    FontWeight->"Bold"],
  " checks if an expression is marked as numeric."
}], "Text",
  CellTags->"S0.0.4"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Map[\ NumericQ, \ {23, \[Pi] - E, \[Pi] - a}]\)], "Input",
  CellTags->"S0.0.5"],

Cell[BoxData[
    \({True, True, False}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "How does ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " decide whether an expression is numeric? ",
  StyleBox["NumericQ ", "InlineInput",
    FontWeight->"Bold"],
  "acts on expressions recursively. All numbers, as well as the constants ",
  StyleBox["Catalan", "InlineInput",
    FontWeight->"Bold"],
  ", ",
  StyleBox["Degree", "InlineInput",
    FontWeight->"Bold"],
  ", ",
  StyleBox["E", "InlineInput",
    FontWeight->"Bold"],
  ", ",
  StyleBox["EulerGamma", "InlineInput",
    FontWeight->"Bold"],
  ", ",
  StyleBox["GoldenRatio", "InlineInput",
    FontWeight->"Bold"],
  ", and ",
  StyleBox["\[Pi]", "InlineInput",
    FontWeight->"Bold"],
  ", are numeric quantities. Any expression built up from numeric quantities \
using functions with the new attribute ",
  StyleBox["NumericFunction", "InlineInput",
    FontWeight->"Bold"],
  " is also a numeric quantity. For example, ",
  StyleBox["Sin", "InlineInput",
    FontWeight->"Bold"],
  " is a ",
  StyleBox["NumericFunction", "InlineInput",
    FontWeight->"Bold"],
  ", so ",
  StyleBox["Sin[2]", "InlineInput",
    FontWeight->"Bold"],
  " is a numeric quantity."
}], "Text",
  CellTags->"S0.0.5"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Attributes[Sin]\)], "Input",
  CellTags->"S0.0.5"],

Cell[BoxData[
    \({Listable, NumericFunction, Protected}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(NumericQ[Sin[2]]\)], "Input",
  CellTags->"S0.0.5"],

Cell[BoxData[
    \(True\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "On the other hand, ",
  StyleBox["f", "InlineInput",
    FontWeight->"Bold"],
  " is not a ",
  StyleBox["NumericFunction", "InlineInput",
    FontWeight->"Bold"],
  ", so ",
  StyleBox["f[2]", "InlineInput",
    FontWeight->"Bold"],
  " is not a numeric quantity."
}], "Text",
  CellTags->"S0.0.5"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Attributes[f]\)], "Input",
  CellTags->"S0.0.5"],

Cell[BoxData[
    \({}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(NumericQ[f[2]]\)], "Input",
  CellTags->"S0.0.5"],

Cell[BoxData[
    \(False\)], "Output"]
}, Open  ]],

Cell[TextData[{
  StyleBox["NumericQ", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["NumericFunction", "InlineInput",
    FontWeight->"Bold"],
  " provide an efficient mechanism for predicting whether an expression would \
become a number if it were encountered by ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  ". However, this mechanism is not foolproof. ",
  StyleBox["NumericQ[Sin[2, 3]]", "InlineInput",
    FontWeight->"Bold"],
  " gives ",
  StyleBox["True", "InlineInput",
    FontWeight->"Bold"],
  ", but ",
  StyleBox["N[Sin[2, 3]]", "InlineInput",
    FontWeight->"Bold"],
  " is simply ",
  StyleBox["Sin[2., 3.]", "InlineInput",
    FontWeight->"Bold"],
  ", not a number. Likewise, ",
  StyleBox["NumericQ", "InlineInput",
    FontWeight->"Bold"],
  " can give false negatives. For example, with this definition of ",
  StyleBox["N[f[x]]", "InlineInput",
    FontWeight->"Bold"],
  ","
}], "Text",
  CellTags->"S0.0.5"],

Cell[BoxData[
    \(N[f[x_]] := x\)], "InputOnly",
  CellTags->"S0.0.5"],

Cell[TextData[{
  "the expression ",
  StyleBox["f[2]", "InlineInput",
    FontWeight->"Bold"],
  " just evaluates to itself and is not a numeric quantity."
}], "Text",
  CellTags->"S0.0.5"],

Cell[CellGroupData[{

Cell[BoxData[
    \(f[2]\)], "Input",
  CellTags->"S0.0.5"],

Cell[BoxData[
    \(f[2]\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(NumericQ[%]\)], "Input",
  CellTags->"S0.0.5"],

Cell[BoxData[
    \(False\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Nevertheless, ",
  StyleBox["N[f[2]]", "InlineInput",
    FontWeight->"Bold"],
  " evaluates to a number."
}], "Text",
  CellTags->"S0.0.5"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[f[2]]\)], "Input",
  CellTags->"S0.0.5"],

Cell[BoxData[
    \(2.`\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "This clears the N-value associated with ",
  StyleBox["f", "InlineInput",
    FontWeight->"Bold"],
  "."
}], "Text",
  CellTags->"S0.0.5"],

Cell[BoxData[
    \(NValues[f] =. \)], "InputOnly"],

Cell[TextData[{
  "In writing programs, you should be aware that ",
  StyleBox["NumericQ", "InlineInput",
    FontWeight->"Bold"],
  " may not always give an accurate prediction of the effects of ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  ". If you must be certain of the effect of ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  ", you should actually use ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " and check the result."
}], "Text",
  CellTags->"S0.0.5"],

Cell[TextData[{
  "We can use the ",
  StyleBox["NumericFunction", "InlineInput",
    FontWeight->"Bold"],
  " attribute in conjunction with ",
  StyleBox["SetPrecision", "InlineInput",
    FontWeight->"Bold"],
  " to specify the conditioning of a function. The following function takes \
an inexact number as input and returns a number whose precision depends on \
the input precision."
}], "Text",
  CellTags->"S0.0.7"],

Cell[BoxData[
    \(\(SetAttributes[f, NumericFunction]; \)\)], "InputOnly"],

Cell[BoxData[
    \(\(f[x_?NumberQ] := \ \ 
      SetPrecision[x, Precision[x]/2]\  /; \ Precision[x] < \[Infinity]; 
    \)\)], "InputOnly"],

Cell[TextData[{
  "The requested 200 digits of precision cannot be attained, so ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " gives up when the result exceeds 125 digits."
}], "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Precision[N[f[2], 200]]\)], "Input"],

Cell[BoxData[
    \($MaxExtraPrecision::"meprec" \( : \ \) 
      "$MaxExtraPrecision = \!\(50.`\) reached while evaluating \!\(f[2]\). \
Increasing the value of $MaxExtraPrecision may help resolve the uncertainty."\
\)], "Message"],

Cell[BoxData[
    \(132\)], "Output"]
}, Open  ]],

Cell["\<\
If we raise the extra precision by a sufficient amount, we can \
obtain an answer to the desired precision.\
\>", "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\($MaxExtraPrecision = 200; \)\)], "InputOnly",
  CellGroupingRules->"InputGrouping"],

Cell[BoxData[
    \(Precision[N[f[2], 200]]\)], "Input",
  CellGroupingRules->"OutputGrouping"],

Cell[BoxData[
    \(200\)], "Output"]
}, Open  ]],

Cell["\<\
Of course, we can always define a function that is so badly \
conditioned that no amount of precision will give us a satisfactory result. \
Here is such a function.\
\>", "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(f[x_?NumberQ] := SetPrecision[x, 1] /; Precision[x] < \[Infinity]; 
    \)\)], "InputOnly"],

Cell[BoxData[
    \(N[f[5], 20]\)], "Input",
  CellGroupingRules->"OutputGrouping"],

Cell[BoxData[
    \($MaxExtraPrecision::"meprec" \( : \ \) 
      "$MaxExtraPrecision = \!\(200.`\) reached while evaluating \!\(f[5]\). \
Increasing the value of $MaxExtraPrecision may help resolve the uncertainty."\
\)], "Message"],

Cell[BoxData[
    \(5.`1\)], "Output"]
}, Open  ]],

Cell["Reset the system default value.", "Text",
  CellTags->"S0.0.7"],

Cell[BoxData[
    \(\($MaxExtraPrecision\  = \ 50; \)\)], "InputOnly"],

Cell[TextData[{
  "Many of the built-in functions, like ",
  StyleBox["Equal", "InlineInput",
    FontWeight->"Bold"],
  ", ",
  StyleBox["Less", "InlineInput",
    FontWeight->"Bold"],
  ", and ",
  StyleBox["Mod", "InlineInput",
    FontWeight->"Bold"],
  ", support the new numeric functionality. Numeric quantities with different \
numerical values can be tested for equality or inequality. For example, we \
can compare \[Pi] with the rational approximation defined above."
}], "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[Pi] < q\)], "Input"],

Cell[BoxData[
    \(True\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "In expressions that contain inexact numbers, exact numeric quantities are \
coerced appropriately. The amount of precision used in the coercion is \
determined by the ",
  StyleBox["Accuracy", "InlineInput",
    FontWeight->"Bold"],
  " of the inexact numbers present."
}], "Text"],

Cell["\<\
Not all expressions involving machine numbers can be resolved using \
machine arithmetic. The right-hand side of this equality is a machine number, \
but a large amount of precision is needed to verify the equality.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\@2 + \@3 - \@\(5 + 2\ \@6\)\  == \ 0.\)], "Input",
  CellTags->"S0.0.8"],

Cell[BoxData[
    \($MaxExtraPrecision::"meprec" \( : \ \) 
      "$MaxExtraPrecision = \!\(50.`\) reached while evaluating \!\(\@2 + \@3 \
- \@\(5 + \(2\\ \@6\)\)\). Increasing the value of $MaxExtraPrecision may \
help resolve the uncertainty."\)], "Message"],

Cell[BoxData[
    \(\@2 + \@3 - \@\(5 + 2\ \@6\) == 0.`\)], "Output"]
}, Open  ]],

Cell["\<\
In fact, the amount of precision required to ascertain if a numeric \
quantity is not equal to a machine zero is the negative of the scale of the \
minimum machine number.\
\>", "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(-Round[Log[10, $MinMachineNumber]]\)\)], "Input"],

Cell[BoxData[
    \(308\)], "Output"]
}, Open  ]],

Cell[TextData[{
  StyleBox["Accuracy", "InlineInput",
    FontWeight->"Bold"],
  " gives us the same information directly."
}], "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Accuracy[0.]\)], "Input"],

Cell[BoxData[
    \(308\)], "Output"]
}, Open  ]],

Cell["\<\
If an equality or inequality cannot be evaluated using maximum \
working precision, the input is returned unevaluated.\
\>", "Text",
  CellTags->"S0.0.8"],

Cell[TextData[{
  "The functions ",
  StyleBox["Mod", "InlineInput",
    FontWeight->"Bold"],
  ", ",
  StyleBox["Round", "InlineInput",
    FontWeight->"Bold"],
  ", ",
  StyleBox["Floor", "InlineInput",
    FontWeight->"Bold"],
  ", and ",
  StyleBox["Ceiling", "InlineInput",
    FontWeight->"Bold"],
  " also attempt to resolve numeric quantities numerically."
}], "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Floor[\[Pi]\^E]\)], "Input",
  CellTags->"S0.0.8"],

Cell[BoxData[
    \(22\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Mod[\[Pi], EulerGamma]\)], "Input",
  CellTags->"S0.0.8"],

Cell[BoxData[
    \(\(-5\)\ EulerGamma + \[Pi]\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The new functions ",
  StyleBox["IntegerPart", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["FractionalPart", "InlineInput",
    FontWeight->"Bold"],
  " are closely related to ",
  StyleBox["Floor", "InlineInput",
    FontWeight->"Bold"],
  " and ",
  StyleBox["Mod", "InlineInput",
    FontWeight->"Bold"],
  "."
}], "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{

Cell[BoxData[
    \(IntegerPart[\[Pi] + I\ E]\)], "Input"],

Cell[BoxData[
    \(3 + 2\ I\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(FractionalPart[\[Pi] + I\ E]\)], "Input"],

Cell[BoxData[
    \(\((\(-3\) - 2\ I)\) + I\ E + \[Pi]\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Expressions involving inverse trigonometric functions of trigonometric \
functions and logarithms of exponentials are often closely related to ",
  StyleBox["Mod", "InlineInput",
    FontWeight->"Bold"],
  ". When such expressions are numeric quantities, they can be evaluated \
numerically. "
}], "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Log[E\^\(I\ \[Pi]\^E\)]\)], "Input",
  CellTags->"S0.0.8"],

Cell[BoxData[
    \(\(-8\)\ I\ \[Pi] + I\ \[Pi]\^E\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(ArcSin[Cos[23 + 7\ I]]\)], "Input",
  CellTags->"S0.0.8"],

Cell[BoxData[
    \(\((23 + 7\ I)\) - \(15\ \[Pi]\)\/2\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "When ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " sorts a list of expressions, all of the numbers come before all of the \
non-numbers. Real numbers are ordered according to magnitude and complex \
numbers according to the magnitude of their real parts. Numbers are followed \
by symbols, which in turn are followed by normal expressions. However, \
numeric quantities that are not numbers (according to ",
  StyleBox["NumberQ", "InlineInput",
    FontWeight->"Bold"],
  ") are sorted ",
  StyleBox["lexicographically",
    FontSlant->"Italic"],
  ". Thus, they might be sandwiched between non-numeric quantities."
}], "Text",
  CellTags->"S0.0.6"],

Cell[CellGroupData[{

Cell[BoxData[
    \(s = Sort[{a, 2, E, x, 3\ I, EulerGamma, b, Sin[\@3]}]\)], "Input",
  CellTags->"S0.0.6"],

Cell[BoxData[
    \({3\ I, 2, a, b, E, EulerGamma, x, Sin[\@3]}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Map[\ NumberQ, \ s\ ]\)], "Input"],

Cell[BoxData[
    \({True, True, False, False, False, False, False, False}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Map[\ NumericQ, \ s\ ]\)], "Input"],

Cell[BoxData[
    \({True, True, False, False, True, True, False, True}\)], "Output"]
}, Open  ]],

Cell["\<\
The numbers themselves are sorted by value, but the numeric \
quantities are ordered lexicographically.\
\>", "Text",
  CellTags->"S0.0.6"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[s]\)], "Input",
  CellTags->"S0.0.6"],

Cell[BoxData[
    \({\(0.`\ \) + 3.`\ I, 2.`, a, b, 2.71828182845904509`, 
      0.577215664901532843`, 3.`, 0.987026644990353929`}\)], "Output"]
}, Open  ]],

Cell["\<\
By specifying an ordering function you can sort real-valued numeric \
quantities by magnitude, but this is slower than the canonical ordering.\
\>",
   "Text",
  CellTags->"S0.0.6"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Sort[{2, E, EulerGamma, Sin[\@3], \[Pi]}, Less]\)], "Input",
  CellTags->"S0.0.6"],

Cell[BoxData[
    \({EulerGamma, Sin[\@3], 2, E, \[Pi]}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[%]\)], "Input"],

Cell[BoxData[
    \({0.577215664901532843`, 0.987026644990353929`, 2.`, 
      2.71828182845904509`, 3.14159265358979311`}\)], "Output"]
}, Open  ]],

Cell["\<\
Because of the non-uniqueness of representation and the limitations \
of using numerical heuristics to distinguish expressions, it is not always \
possible to order numeric expressions uniquely according to magnitude.\
\>", 
  "Text",
  CellTags->"S0.0.6"],

Cell[TextData[{
  "The extension of predicates to handle numeric quantities is useful, for \
example, for the assumptions mechanism in ",
  StyleBox["Integrate", "InlineInput",
    FontWeight->"Bold"],
  ". Previous versions of ",
  StyleBox["Integrate", "InlineInput",
    FontWeight->"Bold"],
  " returned only generic solutions. Now, ",
  StyleBox["Integrate", "InlineInput",
    FontWeight->"Bold"],
  " returns multiple-valued solutions in the form of conditional statements. \
"
}], "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[Integral]\_0\%\[Infinity] Exp[\(-a\)\ x]\ \[DifferentialD]x\)], 
  "Input"],

Cell[BoxData[
    RowBox[{"If", "[", 
      RowBox[{\(Re[a] > 0\), ",", \(1\/a\), ",", 
        RowBox[{
          SubsuperscriptBox["\[Integral]", "0", 
            InterpretationBox["\[Infinity]",
              DirectedInfinity[ 1]]], 
          \(\(E\^\(\(-a\)\ x\)\) \[DifferentialD]x\)}]}], "]"}]], "Output"]
}, Open  ]],

Cell[TextData[{
  "If ",
  StyleBox["a", "InlineInput",
    FontWeight->"Bold"],
  " is \[Pi], the correct solution branch is generated because the assertion \
",
  StyleBox["Re[", "InlineInput",
    FontWeight->"Bold"],
  StyleBox["\[Pi]",
    FontFamily->"Times",
    FontSize->16,
    FontWeight->"Bold",
    FontSlant->"Plain",
    FontTracking->"Plain",
    PrivateFontOptions->{"FontPostScriptName"->Automatic}],
  StyleBox["] > 0", "InlineInput",
    FontWeight->"Bold"],
  " can be evaluated."
}], "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[Integral]\_0\%\[Infinity] Exp[\(-\[Pi]\)\ x]\ \[DifferentialD]x\)], 
  "Input"],

Cell[BoxData[
    \(1\/\[Pi]\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The fact that multiple solutions are represented as conditional statements \
means that you can ",
  StyleBox["programmatically",
    FontSlant->"Italic"],
  " obtain all solution branches."
}], "Text",
  CellTags->"S0.0.8"]
}, Closed]],

Cell[CellGroupData[{

Cell["Numerical Caching", "Section",
  CellTags->"S0.0.8"],

Cell[TextData[{
  "The new numeric technology gives us much more flexibility, but \
occasionally at considerable additional expense. In order to recover some of \
the arithmetic efficiency of Version 2.2, ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " 3.0 makes extensive use of a cache table to store results of numerical \
computations. Since common structures often appear at various levels of an \
expression and across computations, storing results helps to avoid \
duplication of effort."
}], "Text",
  CellTags->"S0.0.8"],

Cell["\<\
As you perform numerical computations involving numeric quantities, \
you will often see that expressions are faster to evaluate in subsequent \
calls. Here is a simple compound numeric expression.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(N[Sqrt[2]\  + \ Sqrt[3], 10000]; \) // Timing\)], "Input"],

Cell[BoxData[
    \({3.05000000000018189`\ Second, Null}\)], "Output"]
}, Open  ]],

Cell["\<\
If we evaluate the same expression with the same or lower \
precision, it is retrieved from the cache and returned with that precision.\
\
\>", "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(N[Sqrt[2]\  + \ Sqrt[3], 5000]; \) // Timing\)], "Input"],

Cell[BoxData[
    \({0.`\ Second, Null}\)], "Output"]
}, Open  ]],

Cell["\<\
Numeric constants have their own local caching mechanism, which \
stores the highest-precision result obtained in a session.\
\>", "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(N[Pi, 5000]; \) // Timing\)], "Input"],

Cell[BoxData[
    \({0.0833333333330301684`\ Second, Null}\)], "Output"]
}, Open  ]],

Cell["\<\
This is a trap-door mechanism that clears the entire cache table.\
\
\>", "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(N[NumericQ, Clear]; \)\)], "InputOnly"],

Cell[BoxData[
    \(\(N[Sqrt[2]\  + \ Sqrt[3], 10000]; \) // Timing\)], "Input",
  CellGroupingRules->"OutputGrouping"],

Cell[BoxData[
    \({3.06666666666660603`\ Second, Null}\)], "Output"]
}, Open  ]],

Cell["\<\
Locally cached values are not affected when the cache table is \
cleared.\
\>", "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(N[Pi, 5000]; \) // Timing\)], "Input"],

Cell[BoxData[
    \({0.`\ Second, Null}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The numerical cache uses hashing of numeric quantities to determine an \
index to a two-dimensional look-up table. Each entry in the cache is a \
structure that contains various fields of information, including a pointer to \
the numeric quantity and the approximate numerical value. The working \
precision used to obtain the entry and the precision of the entry are also \
stored, in order to be able to ascertain any deficiency in precision. In \
addition, the value of ",
  StyleBox["$MaxExtraPrecision", "InlineInput",
    FontWeight->"Bold"],
  " used to obtain an entry is stored, as is a system time-stamp of the head \
of the expression."
}], "Text",
  CellTags->"S0.0.7"],

Cell["\<\
Because the numerical cache is finite in size, we need to decide \
which values to keep and which values to update once the cache is full. \
Therefore, the cache also stores the reference count of each entry and a \
numerical time stamp for the most recent reference. Items are selected for \
removal using a heuristic measure to determine the oldest, least referenced, \
lowest precision entry.\
\>", "Text",
  CellTags->"S0.0.7"],

Cell[TextData[{
  "To illustrate how these cached quantities are used, let us define a \
numeric function and set a value in the cache by numericizing with ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(SetAttributes[g, NumericFunction]; \)\)], "InputOnly"],

Cell[BoxData[
    \(\(g[x_?NumberQ] := \ x\  /; \ Precision[x] < \[Infinity]; \)\)], 
  "InputOnly",
  CellGroupingRules->"OutputGrouping"],

Cell[BoxData[
    \(N[\ g[3], 30\ ]\)], "Input",
  CellGroupingRules->"OutputGrouping"],

Cell[BoxData[
    \(3.`30\)], "Output"]
}, Open  ]],

Cell["\<\
Now, we numericize to higher precision. The cached value is flushed \
and a new value is set in its place.\
\>", "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[g[3], 40]\)], "Input"],

Cell[BoxData[
    \(3.`40\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Next, we redefine the function, but maintain the same pattern in the \
left-hand side of the definition. This replaces the definition in the list of \
",
  StyleBox["DownValues", "InlineInput",
    FontWeight->"Bold"],
  " rather than creating a new entry and also changes the time stamp of the \
symbol ",
  StyleBox["g", "InlineInput",
    FontWeight->"Bold"],
  "."
}], "Text",
  CellTags->"S0.0.7"],

Cell[BoxData[
    \(\(g[x_?NumberQ] := \ x + 1 /; \ Precision[x] < \[Infinity]; \)\)], 
  "InputOnly"],

Cell[TextData[{
  "When we now evaluate ",
  StyleBox["g[3]", "InlineInput",
    FontWeight->"Bold"],
  " numerically, the change in the time stamp causes the old value to be \
flushed, even though we are requesting lower precision."
}], "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[\ g[3], 20\ ]\)], "Input"],

Cell[BoxData[
    \(4.`20\)], "Output"]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Summary", "Section",
  CellTags->"S0.0.9"],

Cell[TextData[{
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " 3.0 contains considerable enhancements to its basic arithmetic. Many \
types of arithmetic are supported. A new syntax for numbers provides more \
efficient and flexible input. The function ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " now uses an adaptive-precision algorithm that attempts to return a result \
to the requested number of digits. Attributes can be used to restrict the \
application of ",
  StyleBox["N", "InlineInput",
    FontWeight->"Bold"],
  " to specific parts of expressions. Since there might be no way to \
distinguish two forms of an expression numerically, the system parameter ",
  StyleBox["$MaxExtraPrecision", "InlineInput",
    FontWeight->"Bold"],
  " has been introduced to bound the amount of extra precision used \
internally. The predicate ",
  StyleBox["NumericQ", "InlineInput",
    FontWeight->"Bold"],
  " detects numeric quantities and this information can be propagated through \
expressions using the ",
  StyleBox["NumericFunction", "InlineInput",
    FontWeight->"Bold"],
  " attribute. Numerical heuristics are used to determine relations involving \
numeric quantities."
}], "Text"],

Cell[TextData[{
  "For recent references on the nuances of machine-precision arithmetic and \
the accuracy and stability of numerical algorithms, the interested reader is \
encouraged to consult [",
  ButtonBox["Acton 1996",
    ButtonData:>"R1",
    ButtonStyle->"Hyperlink"],
  "; ",
  ButtonBox["Goldberg 1991",
    ButtonData:>"R2",
    ButtonStyle->"Hyperlink"],
  "; ",
  ButtonBox["Higham 1996",
    ButtonData:>"R4",
    ButtonStyle->"Hyperlink"],
  "; ",
  ButtonBox["Koren 1993",
    ButtonData:>"R8",
    ButtonStyle->"Hyperlink"],
  "] and the references therein."
}], "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell["Acknowledgments", "Section",
  CellTags->"S0.0.10"],

Cell["\<\
Many of the features and much of the functionality discussed here \
began with the pioneering work of Jerry B. Keiper. I deeply regret that he is \
not still with us to see many of his ideas come to fruition.\
\>", "Text",
  CellTags->"S0.0.8"],

Cell[TextData[{
  "I would like to thank Michael Trott for several insightful examples and \
Professor Benno Fuchssteiner for suggesting the inclusion of an example of \
the error propagation mechanism used by ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "'s numerical functions."
}], "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell["References", "Section",
  CellTags->"S0.0.11"],

Cell[TextData[{
  "Acton, F.S. 1996. ",
  StyleBox[
  "Real Computing Made Real: Preventing Errors in Scientific and Engineering \
Calculations",
    FontSlant->"Italic"],
  ". Princeton University Press."
}], "Reference",
  CellTags->"R1"],

Cell[TextData[{
  "Goldberg, D. 1991. What every computer scientist should know about \
floating point arithmetic. ",
  StyleBox["ACM Computing Surveys",
    FontSlant->"Italic"],
  " 23(1): 5--48."
}], "Reference",
  CellTags->"R2"],

Cell["\<\
Hansen, E. 1992. Global optimization using interval analysis. \
Marcel Dekker, Inc. New York.\
\>", "Reference",
  CellTags->"R3"],

Cell[TextData[{
  "Higham, N.J. 1996. ",
  StyleBox["Accuracy and Stability of Numerical Algorithms",
    FontSlant->"Italic"],
  ". SIAM, Philadelphia."
}], "Reference",
  CellTags->"R4"],

Cell[TextData[{
  "Jacobson, D. 1992. Floating point in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ". ",
  StyleBox["The Mathematica Journal",
    FontSlant->"Italic"],
  " 2(3): 42--46."
}], "Reference",
  CellTags->"R5"],

Cell[TextData[{
  "Keiper, J.B. 1994. New numerical features of ",
  StyleBox["Mathematica. ",
    FontSlant->"Italic"],
  "Wolfram Research Technical Report."
}], "Reference",
  CellTags->"R6"],

Cell[TextData[{
  "Keiper, J.B. 1995. Interval arithmetic in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ". ",
  StyleBox["The Mathematica Journal",
    FontSlant->"Italic"],
  " 5(2): 66--71. Reprinted from the proceedings of the International \
Conference on Numerical Analysis with Automatic Result Verification, ",
  StyleBox["Interval Computations",
    FontSlant->"Italic"],
  " 3, 1993."
}], "Reference",
  CellTags->"R7"],

Cell[TextData[{
  "Koren, I. 1993. ",
  StyleBox["Computer Arithmetic Algorithms.",
    FontSlant->"Italic"],
  " Prentice Hall. Englewood Cliffs, New Jersey."
}], "Reference",
  CellTags->"R8"],

Cell[TextData[{
  "Skeel, R.D., and J.B. Keiper. 1993. ",
  StyleBox["Elementary Numerical Computing with Mathematica.",
    FontSlant->"Italic"],
  " McGraw Hill, New York."
}], "Reference",
  CellTags->"R9"],

Cell[TextData[{
  "Sofroniou, M. 1996. Automatic precision control. ",
  StyleBox["The Mathematica Journal.",
    FontSlant->"Italic"],
  " 6(1): 29--30."
}], "Reference",
  CellTags->"R10"]
}, Closed]]
}, Open  ]]
},
FrontEndVersion->"X 3.0",
ScreenRectangle->{{0, 1024}, {0, 768}},
ScreenStyleEnvironment->"Working",
WindowToolbars->{},
WindowSize->{420, 360},
WindowMargins->{{189, Automatic}, {Automatic, 154}},
Magnification->1,
StyleDefinitions -> "Report.nb"
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "S0.0.1"->{
    Cell[25644, 861, 712, 17, 132, "Text",
      CellTags->"S0.0.1"],
    Cell[26728, 903, 207, 6, 36, "Text",
      CellTags->"S0.0.1"],
    Cell[27453, 932, 1097, 32, 148, "Text",
      CellTags->"S0.0.1"]},
  "S0.0.2"->{
    Cell[28843, 985, 125, 4, 36, "Text",
      CellTags->"S0.0.2"],
    Cell[28971, 991, 85, 2, 48, "InputOnly",
      CellTags->"S0.0.2"],
    Cell[29059, 995, 139, 4, 36, "Text",
      CellTags->"S0.0.2"],
    Cell[29320, 1009, 62, 1, 20, "Text",
      CellTags->"S0.0.2"],
    Cell[29455, 1015, 593, 19, 84, "Text",
      CellTags->"S0.0.2"]},
  "S0.0.3"->{
    Cell[30085, 1039, 55, 1, 41, "Section",
      CellTags->"S0.0.3"],
    Cell[41293, 1438, 185, 6, 36, "Text",
      CellTags->"S0.0.3"],
    Cell[41618, 1455, 152, 6, 20, "Text",
      CellTags->"S0.0.3"],
    Cell[42053, 1476, 619, 20, 72, "Text",
      CellTags->"S0.0.3"],
    Cell[43582, 1534, 83, 2, 48, "InputOnly",
      CellTags->"S0.0.3"],
    Cell[43668, 1538, 138, 6, 20, "Text",
      CellTags->"S0.0.3"],
    Cell[43831, 1548, 65, 2, 32, "Input",
      CellTags->"S0.0.3"],
    Cell[43951, 1556, 141, 5, 20, "Text",
      CellTags->"S0.0.3"],
    Cell[44117, 1565, 59, 2, 32, "Input",
      CellTags->"S0.0.3"],
    Cell[44231, 1573, 75, 1, 20, "Text",
      CellTags->"S0.0.3"],
    Cell[44470, 1585, 155, 6, 20, "Text",
      CellTags->"S0.0.3"],
    Cell[44650, 1595, 63, 2, 32, "Input",
      CellTags->"S0.0.3"],
    Cell[44766, 1603, 85, 1, 20, "Text",
      CellTags->"S0.0.3"],
    Cell[44854, 1606, 94, 2, 48, "InputOnly",
      CellTags->"S0.0.3"],
    Cell[44951, 1610, 70, 1, 20, "Text",
      CellTags->"S0.0.3"],
    Cell[45046, 1615, 63, 2, 32, "Input",
      CellTags->"S0.0.3"],
    Cell[45166, 1623, 81, 2, 52, "InputOnly",
      CellTags->"S0.0.3"],
    Cell[45250, 1627, 442, 12, 84, "Text",
      CellTags->"S0.0.3"]},
  "S0.0.7"->{
    Cell[30143, 1042, 1168, 33, 148, "Text",
      CellTags->"S0.0.7"],
    Cell[32630, 1129, 500, 12, 100, "Text",
      CellTags->"S0.0.7"],
    Cell[33957, 1170, 152, 6, 20, "Text",
      CellTags->"S0.0.7"],
    Cell[34112, 1178, 89, 2, 48, "InputOnly",
      CellTags->"S0.0.7"],
    Cell[34204, 1182, 110, 4, 20, "Text",
      CellTags->"S0.0.7"],
    Cell[34482, 1197, 68, 1, 20, "Text",
      CellTags->"S0.0.7"],
    Cell[34553, 1200, 88, 2, 48, "InputOnly",
      CellTags->"S0.0.7"],
    Cell[34644, 1204, 334, 7, 84, "Text",
      CellTags->"S0.0.7"],
    Cell[34981, 1213, 97, 2, 57, "InputOnly",
      CellTags->"S0.0.7"],
    Cell[35081, 1217, 413, 9, 84, "Text",
      CellTags->"S0.0.7"],
    Cell[35519, 1230, 59, 2, 32, "Input",
      CellTags->"S0.0.7"],
    Cell[35655, 1238, 432, 9, 84, "Text",
      CellTags->"S0.0.7"],
    Cell[36112, 1251, 63, 2, 32, "Input",
      CellTags->"S0.0.7"],
    Cell[36501, 1265, 237, 6, 70, "Text",
      CellTags->"S0.0.7"],
    Cell[36763, 1275, 68, 2, 70, "Input",
      CellTags->"S0.0.7"],
    Cell[36884, 1283, 324, 8, 70, "Text",
      CellTags->"S0.0.7"],
    Cell[37233, 1295, 95, 2, 70, "Input",
      CellTags->"S0.0.7"],
    Cell[37627, 1310, 132, 4, 70, "Text",
      CellTags->"S0.0.7"],
    Cell[37784, 1318, 64, 2, 70, "Input",
      CellTags->"S0.0.7"],
    Cell[38148, 1333, 705, 17, 70, "Text",
      CellTags->"S0.0.7"],
    Cell[38924, 1355, 482, 11, 70, "Text",
      CellTags->"S0.0.7"],
    Cell[39681, 1383, 403, 7, 70, "Text",
      CellTags->"S0.0.7"],
    Cell[40109, 1394, 71, 2, 32, "Input",
      CellTags->"S0.0.7"],
    Cell[40242, 1402, 409, 9, 84, "Text",
      CellTags->"S0.0.7"],
    Cell[45695, 1641, 1063, 27, 164, "Text",
      CellTags->"S0.0.7"],
    Cell[55402, 2024, 421, 11, 68, "Text",
      CellTags->"S0.0.7"],
    Cell[56049, 2045, 218, 6, 36, "Text",
      CellTags->"S0.0.7"],
    Cell[56638, 2068, 153, 4, 36, "Text",
      CellTags->"S0.0.7"],
    Cell[57074, 2088, 210, 5, 52, "Text",
      CellTags->"S0.0.7"],
    Cell[57800, 2115, 69, 1, 20, "Text",
      CellTags->"S0.0.7"],
    Cell[69475, 2586, 700, 13, 164, "Text",
      CellTags->"S0.0.7"],
    Cell[70178, 2601, 441, 8, 116, "Text",
      CellTags->"S0.0.7"],
    Cell[71244, 2637, 152, 4, 36, "Text",
      CellTags->"S0.0.7"],
    Cell[71522, 2652, 421, 12, 68, "Text",
      CellTags->"S0.0.7"],
    Cell[72051, 2670, 267, 7, 52, "Text",
      CellTags->"S0.0.7"]},
  "S0.0.4"->{
    Cell[47024, 1685, 59, 1, 41, "Section",
      CellTags->"S0.0.4"],
    Cell[47086, 1688, 1083, 28, 180, "Text",
      CellTags->"S0.0.4"],
    Cell[48194, 1720, 95, 2, 49, "Input",
      CellTags->"S0.0.4"],
    Cell[48362, 1728, 426, 9, 100, "Text",
      CellTags->"S0.0.4"],
    Cell[49959, 1792, 559, 14, 104, "Text",
      CellTags->"S0.0.4"]},
  "S0.0.5"->{
    Cell[50543, 1810, 100, 2, 32, "Input",
      CellTags->"S0.0.5"],
    Cell[50714, 1818, 1198, 40, 136, "Text",
      CellTags->"S0.0.5"],
    Cell[51937, 1862, 70, 2, 32, "Input",
      CellTags->"S0.0.5"],
    Cell[52119, 1872, 71, 2, 32, "Input",
      CellTags->"S0.0.5"],
    Cell[52246, 1880, 318, 12, 36, "Text",
      CellTags->"S0.0.5"],
    Cell[52589, 1896, 68, 2, 32, "Input",
      CellTags->"S0.0.5"],
    Cell[52733, 1906, 69, 2, 32, "Input",
      CellTags->"S0.0.5"],
    Cell[52859, 1914, 965, 30, 116, "Text",
      CellTags->"S0.0.5"],
    Cell[53827, 1946, 72, 2, 48, "InputOnly",
      CellTags->"S0.0.5"],
    Cell[53902, 1950, 190, 6, 36, "Text",
      CellTags->"S0.0.5"],
    Cell[54117, 1960, 59, 2, 32, "Input",
      CellTags->"S0.0.5"],
    Cell[54254, 1970, 66, 2, 32, "Input",
      CellTags->"S0.0.5"],
    Cell[54377, 1978, 159, 6, 20, "Text",
      CellTags->"S0.0.5"],
    Cell[54561, 1988, 62, 2, 32, "Input",
      CellTags->"S0.0.5"],
    Cell[54678, 1996, 157, 6, 20, "Text",
      CellTags->"S0.0.5"],
    Cell[54892, 2007, 507, 15, 68, "Text",
      CellTags->"S0.0.5"]},
  "S0.0.8"->{
    Cell[57945, 2121, 512, 14, 84, "Text",
      CellTags->"S0.0.8"],
    Cell[59147, 2163, 93, 2, 41, "Input",
      CellTags->"S0.0.8"],
    Cell[59591, 2177, 217, 5, 52, "Text",
      CellTags->"S0.0.8"],
    Cell[59959, 2193, 157, 5, 20, "Text",
      CellTags->"S0.0.8"],
    Cell[60241, 2209, 164, 4, 40, "Text",
      CellTags->"S0.0.8"],
    Cell[60408, 2215, 398, 15, 36, "Text",
      CellTags->"S0.0.8"],
    Cell[60831, 2234, 70, 2, 36, "Input",
      CellTags->"S0.0.8"],
    Cell[60977, 2244, 77, 2, 36, "Input",
      CellTags->"S0.0.8"],
    Cell[61132, 2252, 384, 15, 40, "Text",
      CellTags->"S0.0.8"],
    Cell[61828, 2287, 346, 8, 72, "Text",
      CellTags->"S0.0.8"],
    Cell[62199, 2299, 78, 2, 39, "Input",
      CellTags->"S0.0.8"],
    Cell[62381, 2309, 77, 2, 36, "Input",
      CellTags->"S0.0.8"],
    Cell[65098, 2415, 518, 14, 84, "Text",
      CellTags->"S0.0.8"],
    Cell[66069, 2447, 535, 19, 44, "Text",
      CellTags->"S0.0.8"],
    Cell[66790, 2478, 243, 7, 56, "Text",
      CellTags->"S0.0.8"],
    Cell[67070, 2490, 58, 1, 41, "Section",
      CellTags->"S0.0.8"],
    Cell[67131, 2493, 540, 11, 116, "Text",
      CellTags->"S0.0.8"],
    Cell[68088, 2521, 183, 5, 70, "Text",
      CellTags->"S0.0.8"],
    Cell[68446, 2537, 170, 4, 70, "Text",
      CellTags->"S0.0.8"],
    Cell[68791, 2552, 113, 4, 70, "Text",
      CellTags->"S0.0.8"],
    Cell[69200, 2571, 119, 4, 70, "Text",
      CellTags->"S0.0.8"],
    Cell[74441, 2751, 254, 5, 68, "Text",
      CellTags->"S0.0.8"]},
  "S0.0.6"->{
    Cell[62544, 2317, 681, 16, 136, "Text",
      CellTags->"S0.0.6"],
    Cell[63250, 2337, 108, 2, 57, "Input",
      CellTags->"S0.0.6"],
    Cell[63815, 2363, 149, 4, 40, "Text",
      CellTags->"S0.0.6"],
    Cell[63989, 2371, 59, 2, 32, "Input",
      CellTags->"S0.0.6"],
    Cell[64211, 2380, 191, 5, 56, "Text",
      CellTags->"S0.0.6"],
    Cell[64427, 2389, 102, 2, 38, "Input",
      CellTags->"S0.0.6"],
    Cell[64829, 2407, 266, 6, 72, "Text",
      CellTags->"S0.0.6"]},
  "S0.0.9"->{
    Cell[72482, 2691, 48, 1, 41, "Section",
      CellTags->"S0.0.9"]},
  "S0.0.10"->{
    Cell[74381, 2748, 57, 1, 41, "Section",
      CellTags->"S0.0.10"]},
  "S0.0.11"->{
    Cell[75039, 2770, 52, 1, 41, "Section",
      CellTags->"S0.0.11"]},
  "R1"->{
    Cell[75094, 2773, 240, 8, 46, "Reference",
      CellTags->"R1"]},
  "R2"->{
    Cell[75337, 2783, 233, 7, 46, "Reference",
      CellTags->"R2"]},
  "R3"->{
    Cell[75573, 2792, 140, 4, 32, "Reference",
      CellTags->"R3"]},
  "R4"->{
    Cell[75716, 2798, 188, 6, 32, "Reference",
      CellTags->"R4"]},
  "R5"->{
    Cell[75907, 2806, 236, 9, 32, "Reference",
      CellTags->"R5"]},
  "R6"->{
    Cell[76146, 2817, 194, 6, 32, "Reference",
      CellTags->"R6"]},
  "R7"->{
    Cell[76343, 2825, 442, 13, 74, "Reference",
      CellTags->"R7"]},
  "R8"->{
    Cell[76788, 2840, 194, 6, 32, "Reference",
      CellTags->"R8"]},
  "R9"->{
    Cell[76985, 2848, 209, 6, 32, "Reference",
      CellTags->"R9"]},
  "R10"->{
    Cell[77197, 2856, 190, 6, 32, "Reference",
      CellTags->"R10"]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"S0.0.1", 78105, 2886},
  {"S0.0.2", 78329, 2893},
  {"S0.0.3", 78690, 2904},
  {"S0.0.7", 79960, 2941},
  {"S0.0.4", 82415, 3012},
  {"S0.0.5", 82783, 3023},
  {"S0.0.8", 83914, 3056},
  {"S0.0.6", 85456, 3101},
  {"S0.0.9", 85958, 3116},
  {"S0.0.10", 86045, 3119},
  {"S0.0.11", 86133, 3122},
  {"R1", 86216, 3125},
  {"R2", 86297, 3128},
  {"R3", 86378, 3131},
  {"R4", 86459, 3134},
  {"R5", 86540, 3137},
  {"R6", 86621, 3140},
  {"R7", 86702, 3143},
  {"R8", 86784, 3146},
  {"R9", 86865, 3149},
  {"R10", 86947, 3152}
  }
*)

(*NotebookFileOutline
Notebook[{
Cell[1709, 49, 107, 5, 104, "Title"],

Cell[CellGroupData[{
Cell[1841, 58, 35, 0, 66, "Author"],
Cell[1879, 60, 545, 12, 116, "Text"],
Cell[2427, 74, 549, 12, 116, "Text"],
Cell[2979, 88, 509, 10, 116, "Text"],
Cell[3491, 100, 185, 6, 36, "Text"],

Cell[CellGroupData[{
Cell[3701, 110, 35, 0, 67, "Section"],
Cell[3739, 112, 1006, 25, 180, "Text"],
Cell[4748, 139, 204, 6, 36, "Text"],
Cell[4955, 147, 73, 1, 48, "InputOnly"],
Cell[5031, 150, 138, 5, 20, "Text"],
Cell[5172, 157, 126, 3, 51, "InlineInput"],
Cell[5301, 162, 431, 10, 84, "Text"],
Cell[5735, 174, 332, 7, 83, "InlineInput"],
Cell[6070, 183, 487, 13, 84, "Text"],
Cell[6560, 198, 361, 6, 100, "Text"],
Cell[6924, 206, 503, 12, 100, "Text"],
Cell[7430, 220, 793, 19, 148, "Text"],
Cell[8226, 241, 1625, 50, 182, "Text"],
Cell[9854, 293, 341, 7, 68, "Text"],
Cell[10198, 302, 203, 6, 36, "InlineInput"],
Cell[10404, 310, 348, 7, 68, "Text"],

Cell[CellGroupData[{
Cell[10777, 321, 98, 2, 48, "InputOnly",
  CellGroupingRules->"InputGrouping"],
Cell[10878, 325, 78, 2, 32, "Input",
  CellGroupingRules->"OutputGrouping"],
Cell[10959, 329, 253, 3, 111, "Output"]
}, Open  ]],
Cell[11227, 335, 41, 0, 20, "Text"],

Cell[CellGroupData[{
Cell[11293, 339, 49, 1, 32, "Input"],
Cell[11345, 342, 42, 1, 47, "Output"]
}, Open  ]],
Cell[11402, 346, 294, 9, 52, "Text"],
Cell[11699, 357, 103, 2, 64, "InputOnly"],
Cell[11805, 361, 221, 8, 36, "Text"],

Cell[CellGroupData[{
Cell[12051, 373, 55, 1, 32, "Input"],
Cell[12109, 376, 78, 1, 50, "Output"]
}, Open  ]],
Cell[12202, 380, 248, 8, 36, "Text"],

Cell[CellGroupData[{
Cell[12475, 392, 43, 1, 32, "Input"],
Cell[12521, 395, 77, 1, 47, "Output"]
}, Open  ]],
Cell[12613, 399, 550, 13, 116, "Text"],
Cell[13166, 414, 59, 0, 20, "Text"],

Cell[CellGroupData[{
Cell[13250, 418, 50, 1, 32, "Input"],
Cell[13303, 421, 81, 1, 47, "Output"]
}, Open  ]],
Cell[13399, 425, 143, 3, 36, "Text"],
Cell[13545, 430, 68, 1, 48, "InputOnly"],
Cell[13616, 433, 114, 3, 36, "Text"],

Cell[CellGroupData[{
Cell[13755, 440, 56, 1, 32, "Input"],
Cell[13814, 443, 79, 1, 47, "Output"]
}, Open  ]],
Cell[13908, 447, 96, 3, 20, "Text"],

Cell[CellGroupData[{
Cell[14029, 454, 47, 1, 32, "Input"],
Cell[14079, 457, 36, 1, 47, "Output"]
}, Open  ]],
Cell[14130, 461, 311, 7, 68, "Text"],
Cell[14444, 470, 87, 1, 64, "InputOnly"],
Cell[14534, 473, 66, 0, 20, "Text"],
Cell[14603, 475, 64, 1, 48, "InputOnly"],
Cell[14670, 478, 284, 11, 36, "Text"],

Cell[CellGroupData[{
Cell[14979, 493, 82, 1, 48, "Input"],
Cell[15064, 496, 197, 4, 95, "Output"]
}, Open  ]],
Cell[15276, 503, 74, 0, 20, "Text"],
Cell[15353, 505, 68, 1, 48, "InputOnly"],
Cell[15424, 508, 261, 6, 52, "Text"],

Cell[CellGroupData[{
Cell[15710, 518, 87, 1, 48, "InputOnly"],
Cell[15800, 521, 108, 2, 32, "Input",
  CellGroupingRules->"OutputGrouping"],
Cell[15911, 525, 54, 1, 47, "Output"]
}, Open  ]],
Cell[15980, 529, 1131, 28, 196, "Text"],
Cell[17114, 559, 530, 10, 116, "Text"],
Cell[17647, 571, 446, 12, 68, "Text"],

Cell[CellGroupData[{
Cell[18118, 587, 78, 1, 48, "InputOnly"],
Cell[18199, 590, 86, 2, 32, "Input",
  CellGroupingRules->"OutputGrouping"],
Cell[18288, 594, 36, 1, 47, "Output"]
}, Open  ]],
Cell[18339, 598, 73, 0, 20, "Text"],
Cell[18415, 600, 69, 1, 48, "InputOnly"],
Cell[18487, 603, 74, 0, 20, "Text"],

Cell[CellGroupData[{
Cell[18586, 607, 47, 1, 32, "Input"],
Cell[18636, 610, 54, 1, 47, "Output"]
}, Open  ]],
Cell[18705, 614, 53, 0, 20, "Text"],
Cell[18761, 616, 68, 1, 48, "InputOnly"],
Cell[18832, 619, 337, 7, 68, "Text"],

Cell[CellGroupData[{
Cell[19194, 630, 57, 1, 32, "Input"],
Cell[19254, 633, 168, 3, 60, "Message"],
Cell[19425, 638, 38, 1, 47, "Output"]
}, Open  ]],
Cell[19478, 642, 98, 3, 36, "Text"],
Cell[19579, 647, 68, 1, 48, "InputOnly"],
Cell[19650, 650, 136, 3, 36, "Text"],

Cell[CellGroupData[{
Cell[19811, 657, 57, 1, 32, "Input"],
Cell[19871, 660, 40, 1, 50, "Output"]
}, Open  ]],
Cell[19926, 664, 55, 0, 20, "Text"],
Cell[19984, 666, 60, 1, 48, "InputOnly"],
Cell[20047, 669, 640, 15, 116, "Text"],
Cell[20690, 686, 124, 3, 64, "InputOnly"],
Cell[20817, 691, 322, 11, 36, "Text"],

Cell[CellGroupData[{
Cell[21164, 706, 88, 1, 48, "Input"],
Cell[21255, 709, 48, 1, 47, "Output"]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{
Cell[21352, 716, 55, 0, 41, "Section"],
Cell[21410, 718, 795, 22, 116, "Text"],
Cell[22208, 742, 902, 26, 132, "Text"],
Cell[23113, 770, 544, 11, 116, "Text"],
Cell[23660, 783, 590, 15, 100, "Text"],

Cell[CellGroupData[{
Cell[24275, 802, 45, 1, 32, "Input"],
Cell[24323, 805, 47, 1, 47, "Output"]
}, Open  ]],
Cell[24385, 809, 169, 4, 52, "Text"],

Cell[CellGroupData[{
Cell[24579, 817, 38, 1, 32, "Input"],
Cell[24620, 820, 39, 1, 47, "Output"]
}, Open  ]],
Cell[24674, 824, 71, 0, 20, "Text"],

Cell[CellGroupData[{
Cell[24770, 828, 42, 1, 32, "Input"],
Cell[24815, 831, 43, 1, 47, "Output"]
}, Open  ]],
Cell[24873, 835, 170, 4, 52, "Text"],

Cell[CellGroupData[{
Cell[25068, 843, 39, 1, 32, "Input"],
Cell[25110, 846, 38, 1, 47, "Output"]
}, Open  ]],
Cell[25163, 850, 478, 9, 100, "Text"],
Cell[25644, 861, 712, 17, 132, "Text",
  CellTags->"S0.0.1"],

Cell[CellGroupData[{
Cell[26381, 882, 40, 1, 32, "Input"],
Cell[26424, 885, 41, 1, 50, "Output"]
}, Open  ]],
Cell[26480, 889, 120, 3, 36, "Text"],

Cell[CellGroupData[{
Cell[26625, 896, 42, 1, 32, "Input"],
Cell[26670, 899, 43, 1, 47, "Output"]
}, Open  ]],
Cell[26728, 903, 207, 6, 36, "Text",
  CellTags->"S0.0.1"],

Cell[CellGroupData[{
Cell[26960, 913, 44, 1, 32, "Input"],
Cell[27007, 916, 45, 1, 47, "Output"]
}, Open  ]],
Cell[27067, 920, 383, 10, 68, "Text"],
Cell[27453, 932, 1097, 32, 148, "Text",
  CellTags->"S0.0.1"],

Cell[CellGroupData[{
Cell[28575, 968, 57, 1, 32, "Input"],
Cell[28635, 971, 65, 3, 62, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[28737, 979, 53, 1, 32, "Input"],
Cell[28793, 982, 35, 0, 47, "Output"]
}, Open  ]],
Cell[28843, 985, 125, 4, 36, "Text",
  CellTags->"S0.0.2"],
Cell[28971, 991, 85, 2, 48, "InputOnly",
  CellTags->"S0.0.2"],
Cell[29059, 995, 139, 4, 36, "Text",
  CellTags->"S0.0.2"],

Cell[CellGroupData[{
Cell[29223, 1003, 57, 1, 32, "Input"],
Cell[29283, 1006, 22, 0, 47, "Output"]
}, Open  ]],
Cell[29320, 1009, 62, 1, 20, "Text",
  CellTags->"S0.0.2"],
Cell[29385, 1012, 67, 1, 48, "InputOnly"],
Cell[29455, 1015, 593, 19, 84, "Text",
  CellTags->"S0.0.2"]
}, Closed]],

Cell[CellGroupData[{
Cell[30085, 1039, 55, 1, 41, "Section",
  CellTags->"S0.0.3"],
Cell[30143, 1042, 1168, 33, 148, "Text",
  CellTags->"S0.0.7"],
Cell[31314, 1077, 472, 9, 100, "Text"],
Cell[31789, 1088, 259, 9, 20, "Text"],
Cell[32051, 1099, 80, 1, 48, "InputOnly"],
Cell[32134, 1102, 152, 5, 20, "Text"],

Cell[CellGroupData[{
Cell[32311, 1111, 106, 3, 32, "Input"],
Cell[32420, 1116, 74, 1, 50, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[32531, 1122, 45, 1, 32, "Input"],
Cell[32579, 1125, 36, 1, 47, "Output"]
}, Open  ]],
Cell[32630, 1129, 500, 12, 100, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[33155, 1145, 67, 1, 32, "Input"],
Cell[33225, 1148, 144, 2, 95, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[33406, 1155, 50, 1, 32, "Input"],
Cell[33459, 1158, 428, 6, 150, "Message"],
Cell[33890, 1166, 52, 1, 50, "Output"]
}, Open  ]],
Cell[33957, 1170, 152, 6, 20, "Text",
  CellTags->"S0.0.7"],
Cell[34112, 1178, 89, 2, 48, "InputOnly",
  CellTags->"S0.0.7"],
Cell[34204, 1182, 110, 4, 20, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[34339, 1190, 50, 1, 32, "Input"],
Cell[34392, 1193, 75, 1, 50, "Output"]
}, Open  ]],
Cell[34482, 1197, 68, 1, 20, "Text",
  CellTags->"S0.0.7"],
Cell[34553, 1200, 88, 2, 48, "InputOnly",
  CellTags->"S0.0.7"],
Cell[34644, 1204, 334, 7, 84, "Text",
  CellTags->"S0.0.7"],
Cell[34981, 1213, 97, 2, 57, "InputOnly",
  CellTags->"S0.0.7"],
Cell[35081, 1217, 413, 9, 84, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[35519, 1230, 59, 2, 32, "Input",
  CellTags->"S0.0.7"],
Cell[35581, 1234, 59, 1, 50, "Output"]
}, Open  ]],
Cell[35655, 1238, 432, 9, 84, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[36112, 1251, 63, 2, 32, "Input",
  CellTags->"S0.0.7"],
Cell[36178, 1255, 261, 4, 98, "Message"],
Cell[36442, 1261, 44, 1, 70, "Output"]
}, Open  ]],
Cell[36501, 1265, 237, 6, 70, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[36763, 1275, 68, 2, 70, "Input",
  CellTags->"S0.0.7"],
Cell[36834, 1279, 35, 1, 70, "Output"]
}, Open  ]],
Cell[36884, 1283, 324, 8, 70, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[37233, 1295, 95, 2, 70, "Input",
  CellTags->"S0.0.7"],
Cell[37331, 1299, 281, 8, 70, "Output"]
}, Open  ]],
Cell[37627, 1310, 132, 4, 70, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[37784, 1318, 64, 2, 70, "Input",
  CellTags->"S0.0.7"],
Cell[37851, 1322, 282, 8, 70, "Output"]
}, Open  ]],
Cell[38148, 1333, 705, 17, 70, "Text",
  CellTags->"S0.0.7"],
Cell[38856, 1352, 65, 1, 70, "InputOnly"],
Cell[38924, 1355, 482, 11, 70, "Text",
  CellTags->"S0.0.7"],
Cell[39409, 1368, 61, 1, 70, "InputOnly"],

Cell[CellGroupData[{
Cell[39495, 1373, 49, 1, 70, "Input"],
Cell[39547, 1376, 50, 1, 70, "Output"]
}, Open  ]],
Cell[39612, 1380, 66, 1, 70, "InputOnly"],
Cell[39681, 1383, 403, 7, 70, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[40109, 1394, 71, 2, 32, "Input",
  CellTags->"S0.0.7"],
Cell[40183, 1398, 44, 1, 53, "Output"]
}, Open  ]],
Cell[40242, 1402, 409, 9, 84, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[40676, 1415, 49, 1, 32, "Input"],
Cell[40728, 1418, 160, 3, 173, "Output"]
}, Open  ]],
Cell[40903, 1424, 331, 9, 52, "Text"],
Cell[41237, 1435, 53, 1, 48, "InputOnly"],
Cell[41293, 1438, 185, 6, 36, "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{
Cell[41503, 1448, 42, 1, 32, "Input"],
Cell[41548, 1451, 55, 1, 47, "Output"]
}, Open  ]],
Cell[41618, 1455, 152, 6, 20, "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{
Cell[41795, 1465, 43, 1, 32, "Input"],
Cell[41841, 1468, 197, 5, 47, "Output"]
}, Open  ]],
Cell[42053, 1476, 619, 20, 72, "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{
Cell[42697, 1500, 37, 1, 32, "Input"],
Cell[42737, 1503, 35, 1, 70, "Output"]
}, Open  ]],
Cell[42787, 1507, 792, 25, 88, "Text"],
Cell[43582, 1534, 83, 2, 48, "InputOnly",
  CellTags->"S0.0.3"],
Cell[43668, 1538, 138, 6, 20, "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{
Cell[43831, 1548, 65, 2, 32, "Input",
  CellTags->"S0.0.3"],
Cell[43899, 1552, 37, 1, 47, "Output"]
}, Open  ]],
Cell[43951, 1556, 141, 5, 20, "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{
Cell[44117, 1565, 59, 2, 32, "Input",
  CellTags->"S0.0.3"],
Cell[44179, 1569, 37, 1, 47, "Output"]
}, Open  ]],
Cell[44231, 1573, 75, 1, 20, "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{
Cell[44331, 1578, 49, 1, 32, "Input"],
Cell[44383, 1581, 72, 1, 47, "Output"]
}, Open  ]],
Cell[44470, 1585, 155, 6, 20, "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{
Cell[44650, 1595, 63, 2, 32, "Input",
  CellTags->"S0.0.3"],
Cell[44716, 1599, 35, 1, 47, "Output"]
}, Open  ]],
Cell[44766, 1603, 85, 1, 20, "Text",
  CellTags->"S0.0.3"],
Cell[44854, 1606, 94, 2, 48, "InputOnly",
  CellTags->"S0.0.3"],
Cell[44951, 1610, 70, 1, 20, "Text",
  CellTags->"S0.0.3"],

Cell[CellGroupData[{
Cell[45046, 1615, 63, 2, 32, "Input",
  CellTags->"S0.0.3"],
Cell[45112, 1619, 39, 1, 70, "Output"]
}, Open  ]],
Cell[45166, 1623, 81, 2, 52, "InputOnly",
  CellTags->"S0.0.3"],
Cell[45250, 1627, 442, 12, 84, "Text",
  CellTags->"S0.0.3"],
Cell[45695, 1641, 1063, 27, 164, "Text",
  CellTags->"S0.0.7"],
Cell[46761, 1670, 69, 1, 48, "InputOnly"],

Cell[CellGroupData[{
Cell[46855, 1675, 52, 1, 32, "Input"],
Cell[46910, 1678, 65, 1, 47, "Output"]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{
Cell[47024, 1685, 59, 1, 41, "Section",
  CellTags->"S0.0.4"],
Cell[47086, 1688, 1083, 28, 180, "Text",
  CellTags->"S0.0.4"],

Cell[CellGroupData[{
Cell[48194, 1720, 95, 2, 49, "Input",
  CellTags->"S0.0.4"],
Cell[48292, 1724, 55, 1, 47, "Output"]
}, Open  ]],
Cell[48362, 1728, 426, 9, 100, "Text",
  CellTags->"S0.0.4"],

Cell[CellGroupData[{
Cell[48813, 1741, 49, 1, 32, "Input"],
Cell[48865, 1744, 54, 1, 47, "Output"]
}, Open  ]],
Cell[48934, 1748, 248, 6, 52, "Text"],

Cell[CellGroupData[{
Cell[49207, 1758, 59, 1, 32, "Input"],
Cell[49269, 1761, 170, 2, 82, "Output"]
}, Open  ]],
Cell[49454, 1766, 252, 6, 56, "Text"],

Cell[CellGroupData[{
Cell[49731, 1776, 53, 1, 32, "Input"],
Cell[49787, 1779, 36, 1, 70, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[49860, 1785, 45, 1, 36, "Input"],
Cell[49908, 1788, 36, 1, 70, "Output"]
}, Open  ]],
Cell[49959, 1792, 559, 14, 104, "Text",
  CellTags->"S0.0.4"],

Cell[CellGroupData[{
Cell[50543, 1810, 100, 2, 32, "Input",
  CellTags->"S0.0.5"],
Cell[50646, 1814, 53, 1, 70, "Output"]
}, Open  ]],
Cell[50714, 1818, 1198, 40, 136, "Text",
  CellTags->"S0.0.5"],

Cell[CellGroupData[{
Cell[51937, 1862, 70, 2, 32, "Input",
  CellTags->"S0.0.5"],
Cell[52010, 1866, 72, 1, 70, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[52119, 1872, 71, 2, 32, "Input",
  CellTags->"S0.0.5"],
Cell[52193, 1876, 38, 1, 47, "Output"]
}, Open  ]],
Cell[52246, 1880, 318, 12, 36, "Text",
  CellTags->"S0.0.5"],

Cell[CellGroupData[{
Cell[52589, 1896, 68, 2, 32, "Input",
  CellTags->"S0.0.5"],
Cell[52660, 1900, 36, 1, 47, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[52733, 1906, 69, 2, 32, "Input",
  CellTags->"S0.0.5"],
Cell[52805, 1910, 39, 1, 47, "Output"]
}, Open  ]],
Cell[52859, 1914, 965, 30, 116, "Text",
  CellTags->"S0.0.5"],
Cell[53827, 1946, 72, 2, 48, "InputOnly",
  CellTags->"S0.0.5"],
Cell[53902, 1950, 190, 6, 36, "Text",
  CellTags->"S0.0.5"],

Cell[CellGroupData[{
Cell[54117, 1960, 59, 2, 32, "Input",
  CellTags->"S0.0.5"],
Cell[54179, 1964, 38, 1, 47, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[54254, 1970, 66, 2, 32, "Input",
  CellTags->"S0.0.5"],
Cell[54323, 1974, 39, 1, 47, "Output"]
}, Open  ]],
Cell[54377, 1978, 159, 6, 20, "Text",
  CellTags->"S0.0.5"],

Cell[CellGroupData[{
Cell[54561, 1988, 62, 2, 32, "Input",
  CellTags->"S0.0.5"],
Cell[54626, 1992, 37, 1, 47, "Output"]
}, Open  ]],
Cell[54678, 1996, 157, 6, 20, "Text",
  CellTags->"S0.0.5"],
Cell[54838, 2004, 51, 1, 48, "InputOnly"],
Cell[54892, 2007, 507, 15, 68, "Text",
  CellTags->"S0.0.5"],
Cell[55402, 2024, 421, 11, 68, "Text",
  CellTags->"S0.0.7"],
Cell[55826, 2037, 76, 1, 48, "InputOnly"],
Cell[55905, 2040, 141, 3, 80, "InputOnly"],
Cell[56049, 2045, 218, 6, 36, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[56292, 2055, 56, 1, 32, "Input"],
Cell[56351, 2058, 232, 4, 90, "Message"],
Cell[56586, 2064, 37, 1, 47, "Output"]
}, Open  ]],
Cell[56638, 2068, 153, 4, 36, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[56816, 2076, 105, 2, 48, "InputOnly",
  CellGroupingRules->"InputGrouping"],
Cell[56924, 2080, 95, 2, 32, "Input",
  CellGroupingRules->"OutputGrouping"],
Cell[57022, 2084, 37, 1, 47, "Output"]
}, Open  ]],
Cell[57074, 2088, 210, 5, 52, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[57309, 2097, 113, 2, 64, "InputOnly"],
Cell[57425, 2101, 83, 2, 32, "Input",
  CellGroupingRules->"OutputGrouping"],
Cell[57511, 2105, 233, 4, 90, "Message"],
Cell[57747, 2111, 38, 1, 47, "Output"]
}, Open  ]],
Cell[57800, 2115, 69, 1, 20, "Text",
  CellTags->"S0.0.7"],
Cell[57872, 2118, 70, 1, 48, "InputOnly"],
Cell[57945, 2121, 512, 14, 84, "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{
Cell[58482, 2139, 42, 1, 32, "Input"],
Cell[58527, 2142, 38, 1, 47, "Output"]
}, Open  ]],
Cell[58580, 2146, 300, 7, 68, "Text"],
Cell[58883, 2155, 239, 4, 68, "Text"],

Cell[CellGroupData[{
Cell[59147, 2163, 93, 2, 41, "Input",
  CellTags->"S0.0.8"],
Cell[59243, 2167, 261, 4, 98, "Message"],
Cell[59507, 2173, 69, 1, 69, "Output"]
}, Open  ]],
Cell[59591, 2177, 217, 5, 52, "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{
Cell[59833, 2186, 71, 1, 32, "Input"],
Cell[59907, 2189, 37, 1, 47, "Output"]
}, Open  ]],
Cell[59959, 2193, 157, 5, 20, "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{
Cell[60141, 2202, 45, 1, 32, "Input"],
Cell[60189, 2205, 37, 1, 47, "Output"]
}, Open  ]],
Cell[60241, 2209, 164, 4, 40, "Text",
  CellTags->"S0.0.8"],
Cell[60408, 2215, 398, 15, 36, "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{
Cell[60831, 2234, 70, 2, 36, "Input",
  CellTags->"S0.0.8"],
Cell[60904, 2238, 36, 1, 70, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[60977, 2244, 77, 2, 36, "Input",
  CellTags->"S0.0.8"],
Cell[61057, 2248, 60, 1, 70, "Output"]
}, Open  ]],
Cell[61132, 2252, 384, 15, 40, "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{
Cell[61541, 2271, 58, 1, 32, "Input"],
Cell[61602, 2274, 42, 1, 70, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[61681, 2280, 61, 1, 36, "Input"],
Cell[61745, 2283, 68, 1, 70, "Output"]
}, Open  ]],
Cell[61828, 2287, 346, 8, 72, "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{
Cell[62199, 2299, 78, 2, 39, "Input",
  CellTags->"S0.0.8"],
Cell[62280, 2303, 64, 1, 70, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[62381, 2309, 77, 2, 36, "Input",
  CellTags->"S0.0.8"],
Cell[62461, 2313, 68, 1, 70, "Output"]
}, Open  ]],
Cell[62544, 2317, 681, 16, 136, "Text",
  CellTags->"S0.0.6"],

Cell[CellGroupData[{
Cell[63250, 2337, 108, 2, 57, "Input",
  CellTags->"S0.0.6"],
Cell[63361, 2341, 77, 1, 70, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[63475, 2347, 54, 1, 36, "Input"],
Cell[63532, 2350, 88, 1, 70, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[63657, 2356, 55, 1, 36, "Input"],
Cell[63715, 2359, 85, 1, 70, "Output"]
}, Open  ]],
Cell[63815, 2363, 149, 4, 40, "Text",
  CellTags->"S0.0.6"],

Cell[CellGroupData[{
Cell[63989, 2371, 59, 2, 32, "Input",
  CellTags->"S0.0.6"],
Cell[64051, 2375, 145, 2, 70, "Output"]
}, Open  ]],
Cell[64211, 2380, 191, 5, 56, "Text",
  CellTags->"S0.0.6"],

Cell[CellGroupData[{
Cell[64427, 2389, 102, 2, 38, "Input",
  CellTags->"S0.0.6"],
Cell[64532, 2393, 69, 1, 70, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[64638, 2399, 37, 1, 36, "Input"],
Cell[64678, 2402, 136, 2, 70, "Output"]
}, Open  ]],
Cell[64829, 2407, 266, 6, 72, "Text",
  CellTags->"S0.0.6"],
Cell[65098, 2415, 518, 14, 84, "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{
Cell[65641, 2433, 97, 2, 47, "Input"],
Cell[65741, 2437, 313, 7, 70, "Output"]
}, Open  ]],
Cell[66069, 2447, 535, 19, 44, "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{
Cell[66629, 2470, 101, 2, 47, "Input"],
Cell[66733, 2474, 42, 1, 70, "Output"]
}, Open  ]],
Cell[66790, 2478, 243, 7, 56, "Text",
  CellTags->"S0.0.8"]
}, Closed]],

Cell[CellGroupData[{
Cell[67070, 2490, 58, 1, 41, "Section",
  CellTags->"S0.0.8"],
Cell[67131, 2493, 540, 11, 116, "Text",
  CellTags->"S0.0.8"],
Cell[67674, 2506, 221, 4, 70, "Text"],

Cell[CellGroupData[{
Cell[67920, 2514, 80, 1, 70, "Input"],
Cell[68003, 2517, 70, 1, 70, "Output"]
}, Open  ]],
Cell[68088, 2521, 183, 5, 70, "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{
Cell[68296, 2530, 79, 1, 70, "Input"],
Cell[68378, 2533, 53, 1, 70, "Output"]
}, Open  ]],
Cell[68446, 2537, 170, 4, 70, "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{
Cell[68641, 2545, 60, 1, 70, "Input"],
Cell[68704, 2548, 72, 1, 70, "Output"]
}, Open  ]],
Cell[68791, 2552, 113, 4, 70, "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{
Cell[68929, 2560, 61, 1, 70, "InputOnly"],
Cell[68993, 2563, 119, 2, 70, "Input",
  CellGroupingRules->"OutputGrouping"],
Cell[69115, 2567, 70, 1, 70, "Output"]
}, Open  ]],
Cell[69200, 2571, 119, 4, 70, "Text",
  CellTags->"S0.0.8"],

Cell[CellGroupData[{
Cell[69344, 2579, 60, 1, 70, "Input"],
Cell[69407, 2582, 53, 1, 70, "Output"]
}, Open  ]],
Cell[69475, 2586, 700, 13, 164, "Text",
  CellTags->"S0.0.7"],
Cell[70178, 2601, 441, 8, 116, "Text",
  CellTags->"S0.0.7"],
Cell[70622, 2611, 232, 6, 52, "Text"],

Cell[CellGroupData[{
Cell[70879, 2621, 76, 1, 48, "InputOnly"],
Cell[70958, 2624, 139, 3, 48, "InputOnly",
  CellGroupingRules->"OutputGrouping"],
Cell[71100, 2629, 87, 2, 32, "Input",
  CellGroupingRules->"OutputGrouping"],
Cell[71190, 2633, 39, 1, 47, "Output"]
}, Open  ]],
Cell[71244, 2637, 152, 4, 36, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[71421, 2645, 44, 1, 32, "Input"],
Cell[71468, 2648, 39, 1, 47, "Output"]
}, Open  ]],
Cell[71522, 2652, 421, 12, 68, "Text",
  CellTags->"S0.0.7"],
Cell[71946, 2666, 102, 2, 48, "InputOnly"],
Cell[72051, 2670, 267, 7, 52, "Text",
  CellTags->"S0.0.7"],

Cell[CellGroupData[{
Cell[72343, 2681, 48, 1, 32, "Input"],
Cell[72394, 2684, 39, 1, 70, "Output"]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{
Cell[72482, 2691, 48, 1, 41, "Section",
  CellTags->"S0.0.9"],
Cell[72533, 2694, 1220, 27, 228, "Text"],
Cell[73756, 2723, 588, 20, 84, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[74381, 2748, 57, 1, 41, "Section",
  CellTags->"S0.0.10"],
Cell[74441, 2751, 254, 5, 68, "Text",
  CellTags->"S0.0.8"],
Cell[74698, 2758, 304, 7, 68, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[75039, 2770, 52, 1, 41, "Section",
  CellTags->"S0.0.11"],
Cell[75094, 2773, 240, 8, 46, "Reference",
  CellTags->"R1"],
Cell[75337, 2783, 233, 7, 46, "Reference",
  CellTags->"R2"],
Cell[75573, 2792, 140, 4, 32, "Reference",
  CellTags->"R3"],
Cell[75716, 2798, 188, 6, 32, "Reference",
  CellTags->"R4"],
Cell[75907, 2806, 236, 9, 32, "Reference",
  CellTags->"R5"],
Cell[76146, 2817, 194, 6, 32, "Reference",
  CellTags->"R6"],
Cell[76343, 2825, 442, 13, 74, "Reference",
  CellTags->"R7"],
Cell[76788, 2840, 194, 6, 32, "Reference",
  CellTags->"R8"],
Cell[76985, 2848, 209, 6, 32, "Reference",
  CellTags->"R9"],
Cell[77197, 2856, 190, 6, 32, "Reference",
  CellTags->"R10"]
}, Closed]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

