(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 3.0,
MathReader 3.0, or any compatible application. The data for the notebook 
starts with the line of stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    628166,      14421]*)
(*NotebookOutlinePosition[    633489,      14578]*)
(*  CellTagsIndexPosition[    632982,      14558]*)
(*WindowFrame->Normal*)



Notebook[{
Cell[TextData[{
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " Graphics"
}], "Title"],

Cell[CellGroupData[{

Cell["1. Overview of Basic Graphics Commands", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Two-Dimensional Graphics", "Subsection"],

Cell[CellGroupData[{

Cell["Plot", "Subsubsection",
  CellTags->"Plot"],

Cell[TextData[{
  "The simplest example of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "\[CloseCurlyQuote]s graphing capabilities is a graph of a function of one \
variable created with ",
  StyleBox["Plot", "Input"],
  ". ",
  StyleBox["Plot", "Input"],
  " takes a function to be graphed and a domain for the variable, and \
generates a two-dimensional graph."
}], "Text",
  CellTags->"Plot"],

Cell["Plot[Sin[x]/x, {x, -10, 10}]", "Input",
  CellTags->"Plot"],

Cell[TextData[{
  StyleBox["Plot", "Input"],
  " also accepts a list of functions to plot on the same set of axes."
}], "Text",
  CellTags->"Plot"],

Cell[BoxData[
    \(\(Plot[{Sin[x], Cos[x]}, {x, \(-\[Pi]\), \[Pi]}]; \)\)], "Input",
  CellTags->"Plot"]
}, Closed]],

Cell[CellGroupData[{

Cell["Options", "Subsubsection",
  CellTags->"Options"],

Cell[TextData[{
  "There are dozens of options we can use to control almost every aspect of a \
graph. ",
  StyleBox["Options[", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["FunctionName", "Text",
    FontSlant->"Italic"],
  StyleBox["]", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " returns a list of the options available for a function, along with their \
default values."
}], "Text",
  CellTags->"Options"],

Cell[BoxData[
    \(Options[Plot]\)], "Input",
  CellTags->"Options"],

Cell[TextData[{
  "For example, by default ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " uses an algorithm to choose the most \[OpenCurlyDoubleQuote]interesting\
\[CloseCurlyDoubleQuote] ",
  StyleBox["y",
    FontSlant->"Italic"],
  "-range for a graph. In the above list, we see that the default value for ",
  
  StyleBox["PlotRange", "Input"],
  " is ",
  StyleBox["Automatic", "Input"],
  "."
}], "Text",
  CellTags->"PlotRange"],

Cell[BoxData[
    \(\(Plot[Sin[x\^2]/x\^2, {x, \(-10\), 10}]; \)\)], "Input"],

Cell[TextData[{
  "We can override the default setting by giving a different value to the ",
  StyleBox["PlotRange", "Input"],
  " option."
}], "Text",
  CellTags->"PlotRange"],

Cell[BoxData[
    \(\(Plot[Sin[x\^2]/x\^2, {x, \(-10\), 10}, \ 
      PlotRange -> {\(-0.25\), 1.05}]; \)\)], "Input",
  CellTags->"PlotRange"],

Cell[TextData[{
  "In every case, options are added after the required arguments to the \
function. We set an option by typing the name of the option, an arrow made by \
the two characters ",
  StyleBox["-", "Input"],
  " and ",
  StyleBox[">", "Input"],
  " (or the special character ",
  StyleBox["\[Rule]", "Input"],
  " made by typing \[EscapeKey]",
  StyleBox["->", "Input"],
  "\[EscapeKey]), and the new value of the option. Note that most plotting \
commands have a similar set of options."
}], "Text"],

Cell[BoxData[
    \(\(Plot[Sin[x]/x, \ {x, \(-10\), 10}, \ Frame -> True, \ 
      PlotLabel -> "sinc function", \ GridLines -> Automatic, \ 
      PlotRange -> {{\(-11\), 11}, {\(-0.5\), 1.15}}, \n\tAspectRatio -> 1]; 
    \)\)], "Input",
  CellTags->{"PlotRange", "GridLines", "AspectRatio", "Frame", "PlotLabel"}],

Cell[TextData[
"In the following sections we will change many of a graph\[CloseCurlyQuote]s \
default option settings."], "Text"],

Cell[TextData[{
  "See also ",
  StyleBox["SetOptions", "Input"],
  ", ",
  StyleBox["FullOptions", "Input"],
  ", ",
  StyleBox["FullGraphics", "Input"],
  " "
}], "SeeAlso"]
}, Closed]],

Cell[CellGroupData[{

Cell["ParametricPlot", "Subsubsection",
  CellTags->"ParametricPlot"],

Cell[TextData[{
  StyleBox["ParametricPlot", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " plots a two-dimensional curve described by two functions of the same \
parameter, one that describes movement in the ",
  StyleBox["x",
    FontSlant->"Italic"],
  " direction and one for the ",
  StyleBox["y",
    FontSlant->"Italic"],
  " direction. This allows us to plot curves that are not functions, in the \
mathematical sense. Here is a parametric plot of a circle."
}], "Text",
  CellTags->"ParametricPlot"],

Cell[BoxData[
    \(\(ParametricPlot[{Sin[t], Cos[t]}, {t, 0, 2\ \[Pi]}]; \)\)], "Input",
  CellTags->"ParametricPlot"],

Cell[TextData[{
  "The option ",
  StyleBox["AspectRatio", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " controls the relative sizes of units on the two axes; the setting ",
  StyleBox["Automatic", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " makes them equal (that is, makes one unit on the vertical axis equal to \
one unit on the horizontal axis)."
}], "Text",
  CellTags->"AspectRatio"],

Cell[BoxData[
    \(\(ParametricPlot[{Sin[t], Cos[t]}, {t, 0, 2\ \[Pi]}, 
      AspectRatio \[Rule] Automatic]; \)\)], "Input",
  CellTags->{"AspectRatio", "ParametricPlot"}],

Cell[TextData[{
  StyleBox["ParametricPlot", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " can be used to plot graphs of complicated curves that cannot be expressed \
as a function of two variables."
}], "Text",
  CellTags->"ParametricPlot"],

Cell[BoxData[
    \(\(ParametricPlot[{4\ Cos[\(-5\)\ t/4] + 7\ Cos[t], 
        4\ Sin[\(-5\)\ t/4] + 7\ Sin[t]}, {t, 0, 8\ \[Pi]}, 
      AspectRatio \[Rule] Automatic]; \)\)], "Input",
  CellTags->{"ParametricPlot", "AspectRatio"}],

Cell[TextData[{
  StyleBox["ParametricPlot", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " takes many of the same options as ",
  StyleBox["Plot", "Input"],
  "."
}], "Text",
  CellTags->"ParametricPlot"],

Cell[BoxData[
    \(\(ParametricPlot[{4\ Cos[\(-11\)\ t/4] + 7\ Cos[t], 
        4\ Sin[\(-11\)\ t/4] + 7\ Sin[t]}, {t, 0, 8\ \[Pi]}, 
      AspectRatio \[Rule] Automatic, \ Axes -> False, Frame -> True, 
      FrameLabel -> {"x", "y"}]; \)\)], "Input",
  CellTags->{"AspectRatio", "Axes", "Frame", "FrameLabel"}]
}, Closed]],

Cell[CellGroupData[{

Cell["ImplicitPlot", "Subsubsection"],

Cell[TextData[{
  StyleBox["ImplicitPlot", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows us to plot implicit relations, rather than functions. It is \
defined in one of the standard packages, so we must load it first with the ",
  StyleBox["Needs", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " command."
}], "Text"],

Cell[BoxData[
    \(Needs["Graphics`ImplicitPlot`"]\)], "Input"],

Cell[TextData[{
  StyleBox["ImplicitPlot[", "Inpute",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["eqn",
    FontSlant->"Italic"],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["{", "Inpute",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["x",
    FontSlant->"Italic"],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["a",
    FontSlant->"Italic"],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["b",
    FontSlant->"Italic"],
  StyleBox["}]", "Inpute",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " draws a graph of the set of points that satisfy ",
  StyleBox["eqn",
    FontSlant->"Italic"],
  ". The variable ",
  StyleBox["x",
    FontSlant->"Italic"],
  " is associated with the horizontal axis and ranges from ",
  StyleBox["a",
    FontSlant->"Italic"],
  " to ",
  StyleBox["b",
    FontSlant->"Italic"],
  ". The remaining variable in the equation is associated with the vertical \
axis. We can also specify a vertical range for the graph using the form ",
  StyleBox["ImplicitPlot[", "Inpute",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["eqn",
    FontSlant->"Italic"],
  StyleBox[",", "Inpute",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox[" ",
    FontSize->11],
  StyleBox["{", "Inpute",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["x",
    FontSlant->"Italic"],
  StyleBox[",", "Inpute",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox[" ",
    FontSize->11],
  StyleBox["a",
    FontSlant->"Italic"],
  StyleBox[",", "Inpute",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox[" ",
    FontSize->11],
  StyleBox["b",
    FontSlant->"Italic"],
  StyleBox["},", "Inpute",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox[" ",
    FontSize->11],
  StyleBox["{", "Inpute",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["y",
    FontSlant->"Italic"],
  StyleBox[",", "Inpute",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox[" ",
    FontSize->11],
  StyleBox["c",
    FontSlant->"Italic"],
  StyleBox[",", "Inpute",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox[" ",
    FontSize->11],
  StyleBox["d",
    FontSlant->"Italic"],
  StyleBox["}]", "Inpute",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(ImplicitPlot[3\ x\^2 + 3\ x\ y + 12\ y\^2 == 12, {x, \(-2.5\), 2.5}, 
      AxesOrigin \[Rule] {0, 0}]; \)\)], "Input"],

Cell[TextData[{
  "Like most graphing functions, ",
  StyleBox["ImplicitPlot", "Input"],
  " accepts a list of functions to plot on the same set of axes."
}], "Text"],

Cell[BoxData[
    \(\(ImplicitPlot[{3\ x\^2 + 3\ x\ y + 12\ y\^2 == 12, 
        12\ x\^2 + 3\ x\ y + 3\ y\^2 == 12, 
        3\ x\^2 + 12\ x\ y + 3\ y\^2 == 1}, {x, \(-2.5\), 2.5}, {y, \(-2.5\), 
        2.5}, AxesOrigin \[Rule] {0, 0}]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Graphics Directives and Plot Styles", "Subsubsection"],

Cell[TextData[{
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " contains several objects called ",
  StyleBox["graphics directives",
    FontSlant->"Italic"],
  ", which specify the style in which a graph should be drawn. Graphics \
directives control the color, thickness, point size, and dashing of lines, \
points, and other objects."
}], "Text"],

Cell[TextData[{
  "For example, to specify that lines should be drawn with a specified \
thickness, we use the directive ",
  StyleBox["Thickness[", "Input"],
  StyleBox["t",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  ", where ",
  StyleBox["t",
    FontSlant->"Italic"],
  " is given as a percentage of the width of a graph."
}], "Text"],

Cell[TextData[{
  "All two-dimensional graphing functions have an option called ",
  StyleBox["PlotStyle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", which allows us to specify a list of graphics directives that control \
how the actual curve (as opposed to the surrounding axes, grid lines, etc.) \
is drawn. To draw a sine wave so that the curve is drawn with a thickness 2% \
of the width of the graph, we set the option ",
  StyleBox["PlotStyle -> Thickness[0.02]", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(Plot[Sin[x], {x, \(-3\), 3}, PlotStyle -> {Thickness[0.02]}]; \)\)], 
  "Input"],

Cell[TextData[{
  "To draw a curve with a dashed line, we use the directive ",
  StyleBox["Dashing", "Input"],
  ". ",
  StyleBox["Dashing[{", "Input"],
  StyleBox["d",
    FontSlant->"Italic"],
  StyleBox["}]", "Input"],
  " draws a line so that it alternates between line segments ",
  StyleBox["d",
    FontSlant->"Italic"],
  " percent of the width of the graph and gaps ",
  StyleBox["d",
    FontSlant->"Italic"],
  " percent of the width of the graph. ",
  StyleBox["Dashing[{", "Input"],
  Cell[BoxData[
      \(TraditionalForm\`d\_1\)]],
  StyleBox[",", "Input"],
  Cell[BoxData[
      \(TraditionalForm\`d\_2\)]],
  StyleBox["}]", "Input"],
  " alternates between line segments ",
  Cell[BoxData[
      \(TraditionalForm\`d\_1\)]],
  " long and gaps ",
  Cell[BoxData[
      \(TraditionalForm\`d\_2\)]],
  " long, and ",
  StyleBox["Dashing[{", "Input"],
  Cell[BoxData[
      \(TraditionalForm\`d\_1\)]],
  StyleBox[",", "Input"],
  Cell[BoxData[
      \(TraditionalForm\`d\_2\)]],
  StyleBox[",", "Input"],
  "\[Ellipsis]",
  StyleBox["}]", "Input"],
  " applies the successive widths cyclically. The following graph uses line \
segments twice as long as the gaps."
}], "Text"],

Cell[BoxData[
    \(\(Plot[Sin[x], {x, \(-3\), 3}, PlotStyle -> {Dashing[{0.04, 0.02}]}]; 
    \)\)], "Input"],

Cell[TextData[{
  "There are several ways to specify colors using graphics directives. ",
  StyleBox["RGBColor[", "Input"],
  StyleBox["r",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  StyleBox["g",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  StyleBox["b",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " describes a color made up of ",
  StyleBox["r",
    FontSlant->"Italic"],
  ", ",
  StyleBox["g",
    FontSlant->"Italic"],
  ", and ",
  StyleBox["b",
    FontSlant->"Italic"],
  " percent of red, green, and blue, respectively. Thus, ",
  StyleBox["RGBColor[", "Input"],
  "1",
  StyleBox[",", "Input"],
  "0",
  StyleBox[",", "Input"],
  "0",
  StyleBox["]", "Input"],
  " is red, and ",
  StyleBox["RGBColor[", "Input"],
  "1",
  StyleBox[",", "Input"],
  "0",
  StyleBox[",", "Input"],
  "1",
  StyleBox["]", "Input"],
  " is purple. (The parameters ",
  StyleBox["r",
    FontSlant->"Italic"],
  ", ",
  StyleBox["g",
    FontSlant->"Italic"],
  ", and ",
  StyleBox["b",
    FontSlant->"Italic"],
  " must all be between 0 and 1.)"
}], "Text"],

Cell[TextData[{
  "Other color functions are ",
  StyleBox["Hue[", "Input"],
  StyleBox["h",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  ", which represents the spectrum of colors going through red, orange, \
yellow, green, blue, purple, and back to red; and ",
  StyleBox["GrayLevel[", "Input"],
  StyleBox["g",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " (",
  Cell[BoxData[
      \(TraditionalForm\`0 \[LessEqual] g \[LessEqual] 1\)]],
  "), where ",
  StyleBox["GrayLevel[0]", "Input"],
  " is black and ",
  StyleBox["GrayLevel[1]", "Input"],
  " is white."
}], "Text"],

Cell[TextData[{
  "To change the style of each graph in a list given to ",
  StyleBox["Plot", "Input"],
  ", we give a list containing as many graphics directives as there are \
functions being plotted. Here we draw the first curve in the list using a \
thick line (",
  StyleBox["Thickness[0.02]", "Input"],
  "), and the second curve using a green line (",
  StyleBox["RGBColor[0,1,0]", "Input"],
  ")."
}], "Text"],

Cell[BoxData[
    \(\(Plot[{Sin[x], Cos[x]}, {x, \(-3\), 3}, 
      PlotStyle -> {Thickness[0.02], RGBColor[0, 1, 0]}]; \)\)], "Input"],

Cell["\<\
To apply multiple styles to each function in a list, we surround \
the styles that apply to each function inside a set of list brackets.\
\>", 
  "Text"],

Cell[BoxData[
    \(\(Plot[{Sin[x], Cos[x]}, {x, \(-3\), 3}, 
      PlotStyle -> {{Thickness[0.02], Hue[0.1]}, {Dashing[{0.04, 0.02}], 
            GrayLevel[0.5]}}]; \)\)], "Input"],

Cell[TextData[{
  "One tricky case to be aware of is that in order to specify more than one \
graphics directive in the plot style of a single function, we must surround \
the graphics directives with the double list brackets ",
  StyleBox["{{", "Input"],
  " and ",
  StyleBox["}}", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(Plot[Sin[x]/x, {x, \(-10\), 10}, \n\t
      PlotStyle \[Rule] {{RGBColor[0, 0, 1], Thickness[0.015]}}]; \)\)], 
  "Input"],

Cell[TextData[{
  "See also ",
  StyleBox["PointSize", "Input"],
  ", ",
  StyleBox["CMYKColor", "Input"],
  ", ",
  StyleBox["AbsoluteThickness", "Input"],
  ", ",
  StyleBox["AbsolutePointSize", "Input"],
  ", ",
  StyleBox["Graphics`Colors`", "Input"],
  ", ",
  StyleBox["Graphics`ArgColor`", "Input"]
}], "SeeAlso"]
}, Closed]],

Cell[CellGroupData[{

Cell["Combining Graphs", "Subsubsection"],

Cell[TextData[{
  StyleBox["Show", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows us to display a previously computed graph without having to \
recompute any of the points that make up the curve(s)."
}], "Text"],

Cell[TextData[{
  StyleBox["Show[", "Input"],
  StyleBox["graphics",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["options",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " displays two- and three-dimensional graphics using the new option \
settings specified. ",
  StyleBox["Show", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " accepts options that affect the way a graph or its surrounding elements \
(axes, frame, etc.) is drawn, without requiring any points of the graph to be \
recomputed."
}], "Text"],

Cell[BoxData[
    \(\(p1\  = \ Plot[\ Sin[x]/x, \ {x, \ \(-10\), 10}]; \)\)], "Input"],

Cell[TextData[{
  "None of the options given below require points on the graph to be \
recomputed, so we can use them inside ",
  StyleBox["Show", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(Show[p1, Frame \[Rule] True, AxesStyle \[Rule] Hue[0], 
      PlotRange \[Rule] {\(-0.1\), 0.8}, PlotLabel \[Rule] "p1"]; \)\)], 
  "Input"],

Cell[TextData[{
  "When given a list of graphics, ",
  StyleBox["Show", "Input"],
  " combines them onto the same set of axes. Here is another two-dimensional \
plot."
}], "Text"],

Cell[BoxData[
    \(\(p2 = 
      Plot[BesselJ[2, x], {x, \(-10\), 10}, 
        PlotStyle \[Rule] {GrayLevel[0.5]}]; \)\)], "Input"],

Cell[TextData[{
  "We combine ",
  StyleBox["p1", "Input"],
  " and ",
  StyleBox["p2", "Input"],
  " by placing their names in a list and giving the list to ",
  StyleBox["Show", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(Show[{p1, p2}]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["Show", "Input"],
  " also combines three-dimensional graphics. Here are two three-dimensional \
graphs."
}], "Text"],

Cell[BoxData[
    \(p3 = Plot3D[1 - x, {x, 0, 1}, {y, 0, 1}, ColorFunction \[Rule] Hue]; \n
    p4 = Plot3D[y, {x, 0, 1}, {y, 0, 1}, ColorFunction \[Rule] GrayLevel]; 
    \)], "Input"],

Cell[TextData[{
  "We put them in the same box using ",
  StyleBox["Show", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(Show[{p3, p4}]; \)\)], "Input"],

Cell[TextData[{
  "The object ",
  StyleBox["GraphicsArray", "Input"],
  " takes an array of graphs, and when used with ",
  StyleBox["Show", "Input"],
  ",",
  " displays the graphs in an array. Here is an array containing the four \
previous graphs."
}], "Text"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{{p1, p3}, {p4, p2}}]]; \)\)], "Input"],

Cell[TextData[{
  "See also ",
  StyleBox["DisplayFunction", "Input"],
  ", ",
  StyleBox["$DisplayFunction", "Input"],
  ", ",
  StyleBox["Identity", "Input"],
  " "
}], "SeeAlso"]
}, Closed]],

Cell[CellGroupData[{

Cell["Exercises: Two-Dimensional Graphics", "Subsubsection",
  PageBreakAbove->True],

Cell[TextData[{
  "Make a simple two-dimensional plot of ",
  Cell[BoxData[
      \(TraditionalForm\`x\^2 - 20 \( cos(x\^2)\)\)]],
  " between ",
  Cell[BoxData[
      \(TraditionalForm\`\(-10\)\)]],
  " and 10. (If there are obvious flaws in the graphic, plot it again using \
more plot points.)"
}], "Question"],

Cell[BoxData[
    \(\(Plot[x\^2 - 20\ Cos[x\^2], {x, \(-10\), 10}]; \)\)], "Input"],

Cell[TextData[{
  "Use ",
  StyleBox["Options[Plot]",
    FontFamily->"Courier"],
  StyleBox[" ",
    FontWeight->"Plain"],
  "or the on-line help to find a list of all the options that ",
  StyleBox["Plot",
    FontFamily->"Courier"],
  " accepts. Plot the same function as above, this time changing at least \
five of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "\[CloseCurlyQuote]s default options."
}], "Question"],

Cell[BoxData[
    \(\(Plot[x\^2 - 20\ Cos[x\^2], {x, \(-10\), 10}, PlotPoints \[Rule] 75, 
      Frame \[Rule] True, GridLines \[Rule] Automatic, 
      PlotLabel \[Rule] "Exercise One", AspectRatio \[Rule] 0.5]; \)\)], 
  "Input"],

Cell[" ", "EndExercises"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Three-Dimensional Graphics", "Subsection"],

Cell[CellGroupData[{

Cell["Plot3D", "Subsubsection"],

Cell[TextData[{
  StyleBox["Plot3D",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is the three-dimensional analog of the ",
  StyleBox["Plot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " command. Given a function of two variables and a domain for each \
variable, ",
  StyleBox["Plot3D",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " produces a surface plot."
}], "Text"],

Cell[BoxData[
    \(Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}]\)], "Input"],

Cell["\<\
Applying options to three-dimensional graphics works the same as \
with two-dimensional graphics; in fact, many of the options are the \
same.\
\>", "Text"],

Cell[TextData[{
  "One of the differences between two- and three-dimensional plotting in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " is point sampling. Instead of adaptive sampling, three-dimensional plots \
rely on a fixed grid of points at which to evaluate the function. By default, \
a 15 \[Times] 15 grid is used, resulting in ",
  Cell[BoxData[
      \(TraditionalForm\`15\^2\)]],
  " = 225 points plotted; raising this number results in a smoother graph, \
but takes more time and memory to generate."
}], "Text"],

Cell["Here is a smoother graph of the same function as above.", "Text"],

Cell[BoxData[
    \(\(Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
      Axes \[Rule] False, FaceGrids \[Rule] All, PlotPoints \[Rule] 25]; 
    \)\)], "Input"],

Cell[TextData[{
  "See also ",
  StyleBox["HiddenSurface", "Input"],
  ", ",
  StyleBox["RenderAll", "Input"],
  ", ",
  StyleBox["Lighting", "Input"],
  ", ",
  StyleBox["ColorFunction", "Input"],
  " "
}], "SeeAlso"]
}, Closed]],

Cell[CellGroupData[{

Cell["Changing the Viewpoint", "Subsubsection"],

Cell[TextData[{
  "One important option to three-dimensional plotting functions is the \
viewpoint, the point in space from which the observer looks at the object. ",
  StyleBox["ViewPoint", "Input"],
  " is an option to all three-dimensional graphics functions. Its default \
value is {1.3, \[Dash]2.4, 2.0}, which can be changed by entering a new value \
directly as an option."
}], "Text"],

Cell["Show[%, ViewPoint->{0,3,2}];", "Input"],

Cell[TextData[{
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " provides an easier way to do this using the 3D ViewPoint Selector. To use \
this front-end feature we pull down the Input menu and choose ",
  StyleBox["3D ViewPoint Selector", "MenuItem"],
  ". Rotating the box with the mouse will have ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " compute the point from which to view the object. The ",
  StyleBox["Paste", "MenuItem"],
  " button enters the viewpoint at the current text insertion point."
}], "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell["ParametricPlot3D", "Subsubsection"],

Cell[TextData[{
  StyleBox["ParametricPlot3D",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is the three-dimensional analog of ",
  StyleBox["ParametricPlot", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". Depending on the input, ",
  StyleBox["ParametricPlot3D", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " produces a space curve or a surface."
}], "Text"],

Cell[TextData[{
  "When we give ",
  StyleBox["ParametricPlot3D", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " a list of three parametric functions in one parameter, the result is a \
space curve."
}], "Text"],

Cell[BoxData[
    \(\(ParametricPlot3D[{Sin[t], Cos[t], t/3}, {t, 0, 6\ \[Pi]}, 
      Axes \[Rule] False]; \)\)], "Input"],

Cell["\<\
A list of three parametric functions in two parameters results in a \
surface.\
\>", "Text"],

Cell[BoxData[
    \(\(ParametricPlot3D[{Sin[v]\ Cos[u], Sin[v]\ Sin[u], Cos[v]}, {u, 0, 
        1.5 \[Pi]}, {v, 0, \[Pi]}]; \)\)], "Input"],

Cell[TextData[{
  "Like most graphing functions, ",
  StyleBox["ParametricPlot3D", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " accepts a list of sets of parametric equations and plots the surfaces \
together."
}], "Text"],

Cell[BoxData[
    \(\(ParametricPlot3D[{{Sin[v]\ Cos[u], Sin[v]\ Sin[u], Cos[v]}, {\ 
          Sin[v]\ Cos[4 u/3]/2, \ Sin[v]\ Sin[4 u/3]/2, Cos[v]/2}}, {u, 0, 
        1.5 \[Pi]}, {v, 0, \[Pi]}]; \)\)], "Input"],

Cell[TextData[{
  "Options are given to ",
  StyleBox["ParametricPlot3D", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " the same way as for ",
  StyleBox["Plot3D", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". Most of the options are the same."
}], "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell["Exercises: Three-Dimensional Graphics", "Subsubsection",
  PageBreakAbove->True],

Cell[TextData[{
  "Make a three-dimensional plot of the function sin(",
  StyleBox["x",
    FontSlant->"Italic"],
  "+sin(",
  StyleBox["y",
    FontSlant->"Italic"],
  ")) between ",
  Cell[BoxData[
      \(TraditionalForm\`\(-3\)\)]],
  " and 3 on both axes."
}], "Question"],

Cell[BoxData[
    \(\(Plot3D[Sin[x + Sin[y]], {x, \(-3\), 3}, {y, \(-3\), 3}]; \)\)], 
  "Input"],

Cell[TextData[{
  "Enter ",
  StyleBox["Options[Plot3D]",
    FontFamily->"Courier"],
  " or use the on-line help to find a list of all the options that ",
  StyleBox["Plot3D",
    FontFamily->"Courier"],
  " accepts. Plot the same function again, this time changing at least four \
of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "\[CloseCurlyQuote]s default options, including the options that control \
the smoothness of the plot and the color."
}], "Question"],

Cell[BoxData[
    \(\(changedplot3d = 
      Plot3D[Sin[x + Sin[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
        PlotPoints \[Rule] {15, 45}, Mesh \[Rule] False, 
        ColorFunction \[Rule] Hue, FaceGrids \[Rule] All]; \)\)], "Input"],

Cell[" ", "EndExercises"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Contour and Density Graphics", "Subsection"],

Cell[CellGroupData[{

Cell["ContourPlot and DensityPlot", "Subsubsection"],

Cell[TextData[{
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " plots contour and density plots of functions of two or three variables. \
These plotting functions work much like ",
  StyleBox["Plot", "Input"],
  " and ",
  StyleBox["Plot3D", "Input"],
  " with the exception of special options that apply only to contour and \
density plots."
}], "Text"],

Cell[TextData[{
  StyleBox["ContourPlot", "Input"],
  " displays a graphic of a function of two variables, where regions of \
different intensities of gray have (nearly) the same function value."
}], "Text"],

Cell[BoxData[
    \(\(ContourPlot[Exp[x]Sin[y], {x, \(-3\), 3}, {y, \(-3\), 3}]; \)\)], 
  "Input"],

Cell[TextData[{
  StyleBox["DensityPlot", "Input"],
  " by default generates a grid of gray levels, where the lighter gray areas \
have greater function values than the darker gray areas."
}], "Text"],

Cell[BoxData[
    \(\(DensityPlot[Exp[x]\ Sin[y], {x, \(-3\), 3}, {y, \(-3\), 3}]; \)\)], 
  "Input"],

Cell[TextData[{
  "See also ",
  StyleBox["ColorFunction", "Input"],
  ", ",
  StyleBox["Mesh", "Input"],
  ", ",
  StyleBox["Contours", "Input"],
  ", ",
  StyleBox["ContourLines", "Input"],
  ", ",
  StyleBox["ContourStyle", "Input"],
  " "
}], "SeeAlso"]
}, Closed]],

Cell[CellGroupData[{

Cell["ContourPlot3D", "Subsubsection"],

Cell[TextData[{
  "The function ",
  StyleBox["ContourPlot3D", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " provides a way to plot surfaces showing particular values of a function \
of three variables. This function is defined in one of the standard add-on \
packages, so we must load the package before using the function."
}], "Text"],

Cell["Needs[\"Graphics`ContourPlot3D`\"]", "Input"],

Cell[TextData[{
  StyleBox["ContourPlot3D[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["fun",
    FontSlant->"Italic"],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["{",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["x",
    FontSlant->"Italic"],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  Cell[BoxData[
      \(TraditionalForm\`x\_0\)]],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  Cell[BoxData[
      \(TraditionalForm\`x\_1\)]],
  StyleBox["},",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["{",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["y",
    FontSlant->"Italic"],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  Cell[BoxData[
      \(TraditionalForm\`y\_0\)]],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  Cell[BoxData[
      \(TraditionalForm\`y\_1\)]],
  StyleBox["},",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["{",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["z",
    FontSlant->"Italic"],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  Cell[BoxData[
      \(TraditionalForm\`z\_0\)]],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  Cell[BoxData[
      \(TraditionalForm\`z\_1\)]],
  StyleBox["}]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " plots the surface implicitly defined by ",
  StyleBox["fun[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["x",
    FontSlant->"Italic"],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["y",
    FontSlant->"Italic"],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["z",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["==",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["0",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". Setting the option ",
  StyleBox["Contours",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " to ",
  StyleBox["{",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["val",
            FontSlant->"Italic"], "1"], TraditionalForm]]],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["val",
            FontSlant->"Italic"], "2"], TraditionalForm]]],
  StyleBox[",",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " \[Ellipsis] ",
  StyleBox["}",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " plots the level surfaces corresponding to the values ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["val",
            FontSlant->"Italic"], "1"], TraditionalForm]]],
  ", ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["val",
            FontSlant->"Italic"], "2"], TraditionalForm]]],
  ", \[Ellipsis]."
}], "Text"],

Cell[BoxData[
    \(\(ContourPlot3D[
      Sqrt[x\^2 + y\^2 + z\^2], {x, \(-1\), 1}, {y, 0, 1}, {z, 0, 1}, 
      Contours \[Rule] {0.25, 0.5, 0.75}]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Exercises: Contour and Density Graphics", "Subsubsection",
  PageBreakAbove->True],

Cell[TextData[{
  "Create a density plot of the function sin(",
  StyleBox["x ",
    FontSlant->"Italic"],
  Cell[BoxData[
      \(TraditionalForm\`-\)]],
  " sin(",
  StyleBox["y",
    FontSlant->"Italic"],
  ")) over any range that includes the origin. Render the graphic with twice \
as many ",
  StyleBox["PlotPoints",
    FontFamily->"Courier"],
  ". Experiment with other options."
}], "Question"],

Cell[BoxData[
    \(\(DensityPlot[Sin[x - Sin[y]], {x, \(-10\), 10}, {y, \(-10\), 10}]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(DensityPlot[Sin[x - Sin[y]], {x, \(-10\), 10}, {y, \(-10\), 10}, 
      PlotPoints \[Rule] 30, Mesh -> False, FrameLabel -> {"x", "y"}]; \)\)], 
  "Input"],

Cell[TextData[{
  "Repeat the above exercise using ",
  StyleBox["ContourPlot", "Input",
    FontFamily->"Courier"],
  " instead of ",
  StyleBox["DensityPlot", "Input",
    FontFamily->"Courier"],
  ". Experiment with the options to ",
  StyleBox["ContourPlot", "Input"],
  " that do not apply to ",
  StyleBox["DensityPlot", "Input",
    FontFamily->"Courier"],
  "."
}], "Question"],

Cell[BoxData[
    \(\(ContourPlot[Sin[x - Sin[y]], {x, \(-10\), 10}, {y, \(-10\), 10}]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(ContourPlot[Sin[x - Sin[y]], {x, \(-10\), 10}, {y, \(-10\), 10}, 
      PlotPoints \[Rule] 30, Contours \[Rule] 30, 
      ContourLines \[Rule] False]; \)\)], "Input"],

Cell[" ", "EndExercises"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Plotting Data", "Subsection"],

Cell["\<\
There are many occasions when we want to work with data rather than \
functions. There are several functions designed to visualize data in two or \
three dimensions. For these examples, we need data to work with. In practice, \
we would most likely read these data from a file or use the output of other \
calculations. For this demonstration we will create a list of ordered pairs \
to use as data.\
\>", "Text"],

Cell[BoxData[
    \(\(exampleData = 
      N[Table[{n, n + Sin[n] + Random[]}, {n, 0, 5\ \[Pi], 0.2 \[Pi]}]]; 
    \)\)], "Input"],

Cell[TextData[{
  StyleBox["ListPlot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " takes an array of data and plots it in two dimensions. Given a \
one-dimensional set of data such as ",
  StyleBox["{10, 20, 30, 40}", "Input"],
  ", ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " plots the ordered pairs ",
  Cell["{{1, 10}, {2, 20}, {3, 30}, {4, 40}}", "Input"],
  ". In this case we supply a list of ordered pairs and ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " plots the points using our explicit ",
  StyleBox["x",
    FontSlant->"Italic"],
  " values. (The graphics directive ",
  StyleBox["PointSize[", "Input"],
  StyleBox["p",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " specifies that points should be drawn so they are ",
  StyleBox["p",
    FontSlant->"Italic"],
  " percent of the width of the graph.)"
}], "Text"],

Cell[BoxData[
    \(\(pointplot = ListPlot[exampleData, PlotStyle \[Rule] PointSize[0.02]]; 
    \)\)], "Input"],

Cell[TextData[{
  "Options to ",
  StyleBox["ListPlot", "Input"],
  " include nearly all of those applicable to ",
  StyleBox["Plot", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". One exception is the option ",
  StyleBox["PlotJoined", "Input"],
  ", which when set to ",
  StyleBox["True", "Input"],
  " draws a line connecting each of the points."
}], "Text"],

Cell[BoxData[
    \(\(joinedplot = 
      ListPlot[exampleData, PlotJoined \[Rule] True, 
        PlotStyle \[Rule] RGBColor[0, 0, 1]]; \)\)], "Input"],

Cell[TextData[{
  "At this point in our analysis we can easily find a good least-squares fit \
to these data. The function ",
  StyleBox["Fit", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " takes as arguments a set of data, a set of basis functions for the \
best-fit polynomial, and a list of variables to be used. Here we include only \
constant, linear, and quadratic terms for the best-fit function."
}], "Text"],

Cell[BoxData[
    \(exampleFit = Fit[exampleData, {1, x, x\^2}, x]\)], "Input"],

Cell["Here is a plot of the best-fit quadratic polynomial.", "Text"],

Cell[BoxData[
    \(\(fitplot = 
      Plot[exampleFit, {x, 0, 5\ \[Pi]}, PlotStyle -> Dashing[{0.01}]]; \)\)],
   "Input"],

Cell["Here we combine the previous three graphs.", "Text"],

Cell[BoxData[
    \(\(Show[{pointplot, joinedplot, fitplot}]; \)\)], "Input"],

Cell[TextData[{
  "When working with three-dimensional data, we use analogs to ",
  StyleBox["Plot3D", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["DensityPlot", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", and ",
  StyleBox["ContourPlot", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". ",
  StyleBox["ListPlot3D", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " plots a three-dimensional surface from a rectangular array of height \
values."
}], "Text"],

Cell[BoxData[
    \(\(examplearray = 
      Table[n + Sin[n] + 3\ Random[], {i, 1, 5\ \[Pi], 0.3 \[Pi]}, {n, 1, 
          5\ \[Pi], 0.3 \[Pi]}]; \)\)], "Input"],

Cell[BoxData[
    \(\(ListPlot3D[examplearray]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["ListContourPlot", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " and ",
  StyleBox["ListDensityPlot", "Input",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " create density and contour plots from rectangular arrays of data."
}], "Text"],

Cell[BoxData[
    \(\(ListDensityPlot[examplearray]; \)\)], "Input"],

Cell[CellGroupData[{

Cell["Exercises: Plotting Data", "Subsubsection",
  PageBreakAbove->True],

Cell[TextData[{
  "Create tabular data from the curve ",
  Cell[BoxData[
      \(TraditionalForm\`x\ \(cos(x)\) - sin(x\^2)\)]],
  " between ",
  Cell[BoxData[
      \(TraditionalForm\`x = \(-2\)\[Pi]\)]],
  StyleBox[" ",
    FontSlant->"Italic"],
  "and ",
  Cell[BoxData[
      \(TraditionalForm\`x = 2 \[Pi]\)]],
  StyleBox[" ",
    FontSlant->"Italic"],
  "and display the data using ",
  StyleBox["ListPlot", "Input",
    FontFamily->"Courier"],
  "."
}], "Question"],

Cell[BoxData[
    \(\(mydata = 
      Table[N[{x, x\ Cos[x] - Sin[x\^2]}], {x, \(-2\)\ \[Pi], 2\ \[Pi], 
          0.05 \[Pi]}]; \)\)], "Input"],

Cell[BoxData[
    \(\(ListPlot[mydata, PlotStyle \[Rule] PointSize[0.02]]; \)\)], "Input"],

Cell[TextData[{
  "Create a list of the first 20 prime numbers. (Hint: use ",
  StyleBox["Table", "Input",
    FontFamily->"Courier"],
  " and ",
  StyleBox["Prime[", "Input",
    FontFamily->"Courier"],
  StyleBox["n", "Text",
    FontSlant->"Italic"],
  StyleBox["]", "Input",
    FontFamily->"Courier"],
  ", which gives the ",
  Cell[BoxData[
      \(TraditionalForm\`n\^th\)]],
  " prime number.) Plot the list using ",
  StyleBox["ListPlot", "Input",
    FontFamily->"Courier"],
  ", then fit the data to a quadratic polynomial, and plot the data and the \
best-fitting curve on the same set of axes. Use options to change the color \
and other aspects of the graphic."
}], "Question"],

Cell[BoxData[
    \(primes = Table[Prime[n], {n, 1, 20}]\)], "Input"],

Cell[BoxData[
    \(\(pointplot = 
      ListPlot[primes, PlotStyle \[Rule] {Red, PointSize[0.02]}]; \)\)], 
  "Input"],

Cell[BoxData[
    \(fitline = Fit[primes, {1, x, x\^2}, x]\)], "Input"],

Cell[BoxData[
    \(\(fitplot = Plot[fitline, {x, 0, 20}, PlotStyle \[Rule] {Blue}]; \)\)], 
  "Input"],

Cell[BoxData[
    \(\(Show[{pointplot, fitplot}]; \)\)], "Input"],

Cell[" ", "EndExercises"],

Cell[BoxData[""], "Input",
  PageBreakAbove->True,
  PageBreakBelow->True]
}, Closed]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["2. Using Packages", "Section"],

Cell[TextData[{
  "A number of tools that we use to create and customize ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " graphics are defined in the standard packages that ship with ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ". Generally, these packages either define new functions or graphics \
primitives. We may browse these packages using the help browser."
}], "Text"],

Cell[TextData[{
  "As an example, The ",
  StyleBox["Graphics`Arrow`",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " package defines graphics primitives and options for drawing arrows. To \
load these functions into the ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " kernel, we execute the following ",
  StyleBox["Needs", "Input"],
  " command."
}], "Text"],

Cell[BoxData[
    \(\(?Arrow\)\)], "Input"],

Cell["Needs[\"Graphics`Arrow`\"]", "Input"],

Cell[BoxData[
    \(\(?Arrow\)\)], "Input"],

Cell[TextData[{
  "It is important to load the package defining a function ",
  StyleBox["before",
    FontSlant->"Italic"],
  " we first invoke that function, for if we don\[CloseCurlyQuote]t, we may \
shadow the package definition and prevent it from being correctly defined \
when we do load the package."
}], "Text"],

Cell[TextData[{
  "Each set of related packages also contains a ",
  StyleBox["master",
    FontSlant->"Italic"],
  " package that defines the names of all ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " symbols defined in those packages.  If we load this package first, using \
any function defined in any of the packages will cause that package to load \
automatically."
}], "Text"],

Cell["Needs[\"Graphics`Master`\"]", "Input"],

Cell[TextData[{
  "If we use certain package functions often, it is a good idea to load them \
automatically by adding the appropriate lines to the ",
  StyleBox["init.m",
    FontFamily->"Courier"],
  " file."
}], "Text"],

Cell[TextData[{
  StyleBox["init.m",
    FontFamily->"Courier"],
  " is a user-configurable package that is loaded last in the startup \
process. ",
  StyleBox["init.m",
    FontFamily->"Courier"],
  " can contain any code that we would like to be evaluated automatically \
each time we launch the ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " kernel. Here are some examples of code that might be found in an ",
  StyleBox["init.m",
    FontFamily->"Courier"],
  " file."
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[TextData[{
  "We could add a new directory to the search path contained in ",
  StyleBox["$Path",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[BoxData[
    RowBox[{"AppendTo", "[", 
      RowBox[{"$Path", ",", 
        RowBox[{\(\*"\"\<Hard Drive\>:"\), 
          RowBox[{
            StyleBox["Mathematica",
              FontSlant->"Plain"], ":", 
            RowBox[{
              StyleBox["Mathematica",
                FontSlant->"Plain"], " ", "Applications", 
              \(\*"\"\<\>]"\)}]}]}]}]}]], "Input"],

Cell["\<\
The following line changes the default viewpoint for \
three-dimesional graphics.\
\>", "Text"],

Cell[BoxData[
    \(\(SetOptions[{Plot3D, ListPlot3D, Graphics3D, SurfaceGraphics}, \ 
      ViewPoint \[Rule] {\(-2.581\), \(-1.357\), 1.717}]; \)\)], "Input"],

Cell[TextData[{
  "Placing the following code in the ",
  StyleBox["init.m",
    FontFamily->"Courier"],
  " file will automatically load the ",
  StyleBox["Graphics`Colors`",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " package each time the kernel is launched."
}], "Text"],

Cell[BoxData[
    \(\(Needs["\<Graphics`Colors`\>"]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["3. Selected Graphics Packages", "Section"],

Cell[CellGroupData[{

Cell["Shapes", "Subsection"],

Cell[TextData[{
  "The standard package ",
  StyleBox["Graphics`Shapes`",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " provides ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " definitions of common three-dimensional shapes."
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["Needs[\"Graphics`Shapes`\"]", "Input",
  AspectRatioFixed->True],

Cell[TextData[{
  "Once a package is loaded, we can get a list of all the objects that it \
defines by using ",
  StyleBox["?",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " with the full package name and ",
  StyleBox["*",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " (to denote all objects)."
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["?Graphics`Shapes`*", "Input",
  AspectRatioFixed->True],

Cell[TextData[{
  StyleBox["Torus[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  Cell[BoxData[
      \(TraditionalForm\`r\_1\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`r\_2\)]],
  ", ",
  StyleBox["n",
    FontSlant->"Italic"],
  ", ",
  StyleBox["m",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " creates a list of ",
  StyleBox["n",
    FontSlant->"Italic"],
  " ",
  StyleBox["m",
    FontSlant->"Italic"],
  " polygons approximating a torus centered around the ",
  StyleBox["z ",
    FontSlant->"Italic"],
  "axis with radii ",
  Cell[BoxData[
      \(TraditionalForm\`r\_1\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`r\_2\)]],
  ". Default values of ",
  Cell[BoxData[
      \(TraditionalForm\`r\_1\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`r\_2\)]],
  ", ",
  StyleBox["n",
    FontSlant->"Italic"],
  ", and ",
  StyleBox["m",
    FontSlant->"Italic"],
  " are 1, 0.5, 20 ",
  Cell[BoxData[
      \(TraditionalForm\`r\_1\)]],
  ", and 20 ",
  Cell[BoxData[
      \(TraditionalForm\`r\_2\)]],
  ", respectively. The other shapes are defined similarly."
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[TextData[{
  "We use the ",
  StyleBox["Show",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " command to render the object in our notebook. "
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[BoxData[
    \(\(Show[Graphics3D[Torus[1, 0.75, 25, 25]]]; \)\)], "Input",
  AspectRatioFixed->True],

Cell[TextData[{
  "Transformations on these ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " objects, such as ",
  StyleBox["WireFrame",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " and ",
  StyleBox["RotateShape",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " are wrapped around the shape, using the following syntax. "
}], "Text"],

Cell[BoxData[
    \(\(Show[WireFrame[Graphics3D[Torus[1, 0.75, 25, 25]]]]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Polyhedra", "Subsection"],

Cell[TextData[{
  "\n",
  StyleBox["Graphics`Polyhedra`", "Input"],
  " defines the polyhedra ",
  StyleBox["Tetrahedron", "Input"],
  ", ",
  StyleBox["Cube", "Input"],
  ", ",
  StyleBox["Hexahedron", "Input"],
  ", ",
  StyleBox["Octahedron", "Input"],
  ", ",
  StyleBox["Dodecahedron", "Input"],
  ", ",
  StyleBox["SmallStellatedDodecahedron", "Input"],
  ", ",
  StyleBox["GreatDodecahedron", "Input"],
  ", ",
  StyleBox["GreatStellatedDodecahedron", "Input"],
  ", ",
  StyleBox["Icosahedron", "Input"],
  ", and ",
  StyleBox["GreatIcosahedron", "Input"],
  ", as well as functions to operate on them: ",
  StyleBox["Faces", "Input"],
  ", ",
  StyleBox["Geodesate", "Input"],
  ", ",
  StyleBox["OpenTruncate", "Input"],
  ", ",
  StyleBox["Stellate", "Input"],
  ", ",
  StyleBox["Truncate", "Input"],
  ", and ",
  StyleBox["Vertices", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`Polyhedra`\>"]\)], "Input"],

Cell[TextData[{
  StyleBox["Polyhedron[", "Input"],
  StyleBox["name",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " gives a ",
  StyleBox["Graphics3D", "Input"],
  " object representing the specified solid centered at the origin and with \
unit distance to the midpoints of the edges. The possible names are in the \
list ",
  StyleBox["Polyhedra", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(Polyhedra\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n
        \t\t{Show[Polyhedron[Tetrahedron], Boxed \[Rule] False], \n\t\t
          Show[Polyhedron[Cube], Boxed \[Rule] False], \n\t\t
          Show[Polyhedron[Octahedron], Boxed \[Rule] False], \n\t\t
          Show[Polyhedron[Dodecahedron], Boxed \[Rule] False], \n\t\t
          Show[Polyhedron[Icosahedron], Boxed \[Rule] False]}, \n
        \t\t{Show[Polyhedron[Hexahedron], Boxed \[Rule] False], \n\t\t
          Show[Polyhedron[GreatDodecahedron], Boxed \[Rule] False], \n\t\t
          Show[Polyhedron[SmallStellatedDodecahedron], Boxed \[Rule] False], 
          \n\t\tShow[Polyhedron[GreatStellatedDodecahedron], 
            Boxed \[Rule] False], \n\t\t
          Show[Polyhedron[GreatIcosahedron], Boxed \[Rule] False]}}]; \)\)], 
  "Input"],

Cell[TextData[{
  StyleBox["Polyhedron[", "Input"],
  StyleBox["name",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["center",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["size",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " uses the given ",
  StyleBox["center",
    FontSlant->"Italic"],
  " and ",
  StyleBox["size",
    FontSlant->"Italic"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(Show[\n
      \ \ \ \ \ {\ Polyhedron[Dodecahedron, {4, 1, 0}, 2], \n\t\t
        Polyhedron[Icosahedron, {4, 2, 5}, .5], \n\t\t
        Stellate[Polyhedron[Icosahedron]]\n\ }\ ]; \)\)], "Input"],

Cell[TextData[{
  "Alternatively, the polyhedra can be used as functions in their own right, \
each of which can take ",
  StyleBox["center",
    FontSlant->"Italic"],
  " and ",
  StyleBox["size",
    FontSlant->"Italic"],
  " options. This allows the easier  inclusion of other primitives and \
directives. "
}], "Text"],

Cell[BoxData[
    \(\(?Dodecahedron\)\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{
        Show[Graphics3D[Stellate[Dodecahedron[{0, 0, 0}, 10]]], 
          Axes -> True], \n\t\t
        Show[Graphics3D[Stellate[Dodecahedron[{10, 10, 10}, 10]]], 
          Axes -> True]}]; \)\)], "Input"],

Cell[TextData[{
  "The package also includes functions for operating on polyhedra, such as ",
  StyleBox["Stellate", "Input"],
  ", ",
  StyleBox["Geodesate", "Input"],
  ", ",
  StyleBox["Truncate", "Input"],
  ", and ",
  StyleBox["OpenTruncate", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t\t
        Show[Stellate[Polyhedron[Icosahedron]], Boxed \[Rule] False], \n\t\t
        Show[Geodesate[Polyhedron[Dodecahedron], 3], Boxed \[Rule] False], \n
        \t\tShow[Truncate[Polyhedron[Dodecahedron]], Boxed \[Rule] False], \n
        \t\tShow[OpenTruncate[Polyhedron[Dodecahedron]], 
          Boxed \[Rule] False]}]; \)\)], "Input"],

Cell[TextData[{
  "Notice in our examples that complex graphics objects are made up of \
several polygons.  ",
  StyleBox["EdgeForm", "Input"],
  " allows us to specify directions for the drawing of the edges of these \
polygons. ",
  StyleBox["EdgeForm[ ]", "Input"],
  " draws no lines at all on the edges of the polygons, while ",
  StyleBox["EdgeForm[", "Input"],
  StyleBox["g",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  ", where ",
  StyleBox["g",
    FontSlant->"Italic"],
  " is a list of graphics directives, draws polygon edges in the styles \
defined by ",
  StyleBox["g",
    FontSlant->"Italic"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(Show[\ 
      Graphics3D[\n
        \ \ \ \ \ \ \ \ \ \ \ \ \ \ {EdgeForm[], \n
          \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ Dodecahedron[{2, 2, 2}, \ 1.2], \n
          \ \ \ \ \ \ \ \ \ \ \t\ \ \ 
          EdgeForm[{GrayLevel[.4], \ Thickness[0.01]}], \n
          \ \ \ \ \ \ \ \ \ \ \t\ \ \ 
          Stellate[Icosahedron[{0, 0, 0}, \ .5]]\ }\n
        \ \ \ \ \ \ \ \ \t\ \ \ ]]; \)\)], "Input"],

Cell[TextData[{
  "\n",
  StyleBox["Lighting", "Input"],
  " is an option to ",
  StyleBox["Show", "Input"],
  " that can be ",
  StyleBox["True", "Input"],
  " or ",
  StyleBox["False", "Input"],
  ". In the default case,  ",
  StyleBox["Lighting", "Input"],
  " is ",
  StyleBox["True", "Input"],
  StyleBox[",", "Input",
    FontWeight->"Plain"],
  " and we control the lighting of the surface just as we do for  \
three-dimensional plotting, by giving options such as ",
  StyleBox["LightSources", "Input"],
  " to ",
  StyleBox["Show", "Input"],
  ". Turning this option off allows us to control the shading of a surface \
via its intrinsic qualities. ",
  StyleBox["FaceForm[", "Input"],
  StyleBox["gfront",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["gback",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " allows the use of graphics directives for the front and back faces of  \
each polygon. To see the back face of a polygon, we first remove some of the \
pieces of an object. The internal form of",
  StyleBox[" Dodecahedron[", "Input"],
  " ",
  StyleBox["]", "Input"],
  " is a list of polygon specifications. We choose part of this list just as \
we would take part of a list of numbers."
}], "Text"],

Cell[BoxData[
    \(\(partialdodec\  = \ Drop[\ Dodecahedron[\ ], \ {4, 6}\ ]; \)\)], 
  "Input"],

Cell[BoxData[
    \(\(Show[Graphics3D[{FaceForm[SlateGray, \ Maroon], partialdodec}\ ], \ \n
      \ \ \tLighting\  -> \ False, \ Boxed\  -> \ False]; \)\)], "Input"],

Cell[TextData[{
  "The ",
  StyleBox["Polyhedra", "Input"],
  " package also provides functions for accessing the coordinates of the \
vertices and the vertex numbers of the faces for the various polyhedra."
}], "Text"],

Cell[BoxData[
    \(Vertices[Dodecahedron] // First\)], "Input"],

Cell[BoxData[
    \(Faces[Icosahedron] // First\)], "Input"],

Cell[BoxData[
    \(Polyhedron[Octahedron] // First\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Arrow", "Subsection"],

Cell[TextData[{
  StyleBox["Arrow[", "Input"],
  StyleBox["start",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["finish",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " is a graphics primitive representing an arrow starting at ",
  StyleBox["start",
    FontSlant->"Italic"],
  " and ending at ",
  StyleBox["finish",
    FontSlant->"Italic"],
  ". Various options control the shape of the head of the arrow. We can \
control the proportions of the head of the arrow with ",
  StyleBox["HeadLength", "Input"],
  ", ",
  StyleBox["HeadWidth", "Input"],
  ", and ",
  StyleBox["HeadCenter", "Input"],
  ", or we can specify an arbitrary graphic to use as the head of the arrow \
with ",
  StyleBox["HeadShape", "Input"],
  ". ",
  StyleBox["HeadScaling", "Input"],
  " controls the size of the head of the arrow, with default setting ",
  StyleBox["Automatic", "Input"],
  " the head will be proportional to the length of the arrow. The setting ",
  StyleBox["Absolute", "Input"],
  " will keep all arrowheads the same size."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`Arrow`\>"]\)], "Input"],

Cell[BoxData[
    \(\(Show[Graphics[Arrow[{0, 0}, \ {1, \ 1}]]]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[
      Graphics[\n{
          Table[{Hue[Random[]], Arrow[{0, 0}, \ t\ {Cos[t], \ Sin[t]}]}, \n
            \t\t{t, \ 0, \ 2  Pi, \ 0.25}]}\ ], \ 
      AspectRatio\  -> \ Automatic, PlotRange\  -> \ All]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[
      Graphics[\n{
          Table[Arrow[{t, 0}, \ {t, \ 1}, \ \n\t\tHeadCenter\  -> \ t, 
              HeadLength\  -> \ .2], \n\t\t{t, \ 0, \ 1, \ .2}]}], \n\t\t
      AspectRatio\  -> \ Automatic, \n\t\t
      PlotRange\  -> \ {{\(-.25\), \ 1.25}, \ All}]; \)\)], "Input"],

Cell[TextData[{
  "Since ",
  StyleBox["Arrow", "Input"],
  " is a graphics primitive, it can be used with ",
  StyleBox["Epilog", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(Plot[Sin[x], {x, 0, 2\ \[Pi]}, 
      Epilog \[Rule] {Arrow[{4, .5}, {\[Pi], 0}], 
          Text["\<(\[Pi],0)\>", {4.1, .6}]}]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Legend", "Subsection"],

Cell[TextData[{
  "The package ",
  StyleBox["Graphics`Legend`",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " provides the ability to create a legend and place it in any \
two-dimensional graphic. There are two ways to use legends. First, as an \
option to ",
  StyleBox["Plot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " and, second, as a graphics object that can be combined with any \
two-dimensional graphics object."
}], "Text"],

Cell["Needs[\"Graphics`Legend`\"]", "Input"],

Cell[CellGroupData[{

Cell["Legends as a Plot Option", "Subsubsection"],

Cell[TextData[{
  "Specifying the option ",
  StyleBox["PlotLegend",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["->",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["{", "Input"],
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["text",
            FontSlant->"Italic"], "1"], TraditionalForm]]],
  StyleBox[",", "Input"],
  " ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["text",
            FontSlant->"Italic"], "2"], TraditionalForm]]],
  StyleBox[",", "Input"],
  " \[Ellipsis] ",
  StyleBox["}", "Input"],
  " in any plot command will produce a basic legend, with text for each curve \
plotted. If there are more curves than text, the text will be repeated \
cyclically."
}], "Text"],

Cell[BoxData[
    \(\(Plot[{Sin[x]/x, 1/x}, {x, \(-10\), 10}, \n\t
      PlotLegend\  -> \ {"\<Sin[x]/x\>", "\<1/x\>"}]; \)\)], "Input"],

Cell["The legend will incorporate styles assigned to the curves.", "Text"],

Cell[BoxData[
    \(\(Plot[\ {Sin[x]/x, \ 1/x}, \ {x, \ \(-10\), 10}, \n\t\t
      PlotStyle\  -> \ {GrayLevel[0.6], \ Dashing[{0.02}]}, \n\t\t
      PlotLegend\  -> \ {Sin[x]/x, 1/x}]; \)\)], "Input"],

Cell[TextData[{
  "The form of each ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["text",
            FontSlant->"Italic"], "i"], TraditionalForm]]],
  " can be altered by using ",
  StyleBox["StyleForm",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". "
}], "Text"],

Cell[BoxData[
    \(\(Plot[\ {Sin[x]/x, \ 1/x}, \ {x, \ \(-10\), \ 10}, \n\t
      PlotStyle\  -> \ {GrayLevel[0.6], \ Dashing[{0.02}]}, \n\t
      PlotLegend\  -> \n
        \t\t{StyleForm[Sin[x]/x, \n\t\t\t\tFontFamily\  -> \ "\<Times\>", \n
            \t\t\t\tFontSlant -> "\<Italic\>", \n\t\t\t\tFontSize -> 14], \n
          \t\t\ \ StyleForm[1/x, \n\t\t\t\tFontFamily\  -> \ "\<Helvetica\>", 
            \n\t\t\t\tFontWeight\  -> \ "\<Bold\>", \n\t\t\t\t
            FontSize\  -> \ 14]}]; \)\)], "Input"],

Cell[TextData[{
  "There are numerous options that affect the way the legend is drawn. As the \
examples show, ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " will do a good job creating, sizing, and placing legends on its own, but \
there are many aspects that may be controlled. ",
  StyleBox["LegendPosition",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["LegendSize",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["LegendShadow",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["LegendOrientation",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["LegendLabel",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["LegendTextDirection",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["LegendTextOffset",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["LegendSpacing",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["LegendTextSpace",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["LegendLabelSpace",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["LegendBorderSpace",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["LegendBorder",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", and ",
  StyleBox["LegendBackground",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allow us to determine the look and placement of the legend. Following is \
an example with many of these options changed."
}], "Text"],

Cell[BoxData[
    \(\(Plot[\ {Sin[x]/x, \ 1/x}, \ {x, \ \(-10\), \ 10}, \n\t\t
      PlotStyle\  -> \ {{RGBColor[1, 0, 0]}, \ {Dashing[{0.02}]}}, \n\t\t
      PlotLegend\  -> \ \n
        \t\t\ \ \ {
          StyleForm[Sin[x]/x, \n\t\t\t\tFontFamily\  -> \ "\<Times\>", \n
            \t\t\t\tFontSlant -> "\<Italic\>", FontSize\  -> \ 14], \ \n\t\t\t
          StyleForm[1/x, \n\t\t\t\tFontFamily\  -> \ "\<Helvetica\>", \n
            \t\t\t\tFontWeight\  -> \ "\<Bold\>", \ FontSize\  -> \ 12]}, \n
      \t\tLegendPosition\  -> \ {\(-1\), \ 0.15}, \ \n\t\t
      LegendSize\  -> \ {0.75, 0.8}, \n\t\t
      LegendShadow\  -> \ {0.075, \ 0.1}, \ \n\t\t
      LegendLabel\  -> \ \n\t\t\t
        StyleForm["\<Example\>", \ FontFamily\  -> \ "\<Times\>", \n\t\t\t
          FontWeight\  -> \ "\<Bold\>", \ FontSize\  -> \ 18], \n\t\t
      LegendBorder\  -> \ \n\t\t\t{RGBColor[0.4, 1, 0.7], \ Thickness[0.02]}, 
      \n\t\tLegendBackground\  -> \ Yellow, \n\t\t
      ShadowBackground\  -> \ SlateGray\t\t\t\t\ \ \ \n\t]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Legends in Other Types of Graphics", "Subsubsection"],

Cell[TextData[{
  "The form ",
  StyleBox["ShowLegend[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["graphic",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["legend",
            FontSlant->"Italic"], "1"], TraditionalForm]]],
  StyleBox[",", "Input"],
  " ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["legend",
            FontSlant->"Italic"], "2"], TraditionalForm]]],
  ", \[Ellipsis]",
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " will place the legends specified in the graphic, where ",
  StyleBox["graphic",
    FontSlant->"Italic"],
  " is any graphic. Each ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["legend",
            FontSlant->"Italic"], "i"], TraditionalForm]]],
  " has the form ",
  StyleBox["{{{", "Input"],
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["box",
            FontSlant->"Italic"], "1"], TraditionalForm]]],
  StyleBox[",", "Input"],
  " ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["text",
            FontSlant->"Italic"], "1"], TraditionalForm]]],
  StyleBox["},", "Input"],
  " \[Ellipsis] ",
  StyleBox["},", "Input"],
  " ",
  StyleBox["opts",
    FontSlant->"Italic"],
  StyleBox["}", "Input"],
  " where each ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["box",
            FontSlant->"Italic"], "i"], TraditionalForm]]],
  " is a color primitive or graphic and each ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["text",
            FontSlant->"Italic"], "i"], TraditionalForm]]],
  " is an expression. A special form for a legend with ",
  StyleBox["n",
    FontSlant->"Italic"],
  " boxes, each colored according to the function ",
  StyleBox["colorfunction",
    FontSlant->"Italic"],
  " is ",
  StyleBox["{", "Input"],
  StyleBox["colorfunction",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["n",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["minstring",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["maxstring",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["opts",
    FontSlant->"Italic"],
  StyleBox["}", "Input"],
  ", where ",
  StyleBox["minstring",
    FontSlant->"Italic"],
  " and ",
  StyleBox["maxstring",
    FontSlant->"Italic"],
  " are optional text strings to be placed with the end boxes."
}], "Text"],

Cell[TextData[{
  "The next example uses this form. Since we know that ",
  StyleBox["ContourPlot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " shades each contour by mapping the function value onto ",
  StyleBox["GrayLevel",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", we can easily mimic this in the legend."
}], "Text"],

Cell["\<\
ShowLegend[ ContourPlot[ Sin[x y], {x, 0, Pi}, 
\t\t\t\t\t{y, 0, Pi}, ContourLines -> False,
\t\t\t\t\tContours -> 35, 
\t\t\t\t\tDisplayFunction -> Identity],
\t\t\t
\t\t\t{GrayLevel[1-#]&, 10, \" 1\", \"-1\",
\t\t\t LegendPosition -> {1.1, -.4}}];\
\>", "Input"],

Cell[BoxData[
    \(\(ShowLegend[\n\t
      DensityPlot[Sin[x\^2 - Cos[y\^2]], {x, \(-3\), 3}, \ {y, \(-3\), 3}, \n
        \t\t\tMesh -> False, PlotPoints -> 30, DisplayFunction -> Identity], 
      \n{GrayLevel[1 - #]&, 10, "\< 1\>", "\<-1\>", \n\t\t\t
        LegendPosition -> {1.1, \(-0.4\)}}]; \)\)], "Input"],

Cell["Needs[\"Graphics`Colors`\"]", "Input"],

Cell["func[a_,b_,c_] := c/(b x)^(a Log[b x]);", "Input"],

Cell["\<\
ShowLegend[ 
 Plot[{func[2,2,1], func[2,4,1], func[4,4,1]}, {x, 0.001, 1},
\tPlotStyle -> {Red, Green, Blue},
\tPlotLabel -> StyleForm[c/(b x)^(a Log[b x]),
\t\t\t\t FontFamily -> \"Helvetica\", 
\t\t\t\t FontWeight -> \"Bold\", FontSize -> 12],
\t\t\t\t DisplayFunction -> Identity],
\t\t\t
{{{Red, \"2,2,1\"}, {Green, \"2,4,1\"}, {Blue, \"4,4,1\"}},
LegendPosition -> {1, -.4}, LegendSize -> {1,1},
LegendLabel -> StyleForm[\"Values of a, b, c\",
\t\t\t   FontFamily -> \"Helvetica\", FontWeight -> \"Bold\",
\t\t\t   FontSize -> 10]}];\
\>", "Input"],

Cell[TextData[{
  "A legend may be created as a standalone graphics object by using the \
function ",
  StyleBox["Legend[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["legendargs",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["opts",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", where ",
  StyleBox["legendargs",
    FontSlant->"Italic"],
  " is of the form ",
  StyleBox["{{", "Input"],
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["box",
            FontSlant->"Italic"], "1"], TraditionalForm]]],
  StyleBox[",", "Input"],
  " ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["text",
            FontSlant->"Italic"], "1"], TraditionalForm]]],
  StyleBox["},", "Input"],
  " \[Ellipsis] ",
  StyleBox["}", "Input"],
  ". We may also generate just the graphics primitives describing the legend\
\[CloseCurlyQuote]s box and drop shadow with the command ",
  StyleBox["ShadowBox[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["pos",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["size",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["opts",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". Options for ",
  StyleBox["ShadowBox",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " are ",
  StyleBox["ShadowOffset",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["ShadowBackground",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["ShadowForeground",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", and ",
  StyleBox["ShadowBorder",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". Note that these options can also be included in calls to ",
  StyleBox["ShowLegend",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["PlotLegend",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", or ",
  StyleBox["Legend",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  "."
}], "Text"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Exercises: Selected Graphics Packages", "Subsection",
  PageBreakAbove->True],

Cell[TextData[{
  "Load the package ",
  StyleBox["Graphics`Master`",
    FontFamily->"Courier"],
  ". This loads the name of every function defined in a graphics package into \
memory, and tells ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to load the appropriate package when a package function is first used."
}], "Question"],

Cell["Needs[\"Graphics`Master`\"]", "Input"],

Cell[TextData[{
  "Have ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " draw the default ",
  StyleBox["Cylinder",
    FontFamily->"Courier"],
  ". Draw another cylinder whose length is twice its diameter and use twice \
the number of polygons as ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " would by default. Use ",
  StyleBox["RotateShape",
    FontFamily->"Courier"],
  " to rotate this ",
  StyleBox["Cylinder",
    FontFamily->"Courier"],
  " off the vertical. Use ",
  StyleBox["AffineShape",
    FontFamily->"Courier"],
  " to deform the cylinder in any interesting way."
}], "Question"],

Cell["?Cylinder", "Input"],

Cell[BoxData[
    \(\(Show[Graphics3D[Cylinder[\ ]], Boxed \[Rule] False]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[Graphics3D[Cylinder[1, 2, 40]], Boxed \[Rule] False]; \)\)], 
  "Input"],

Cell["?RotateShape", "Input"],

Cell[BoxData[
    \(\(Show[
      Graphics3D[\n\t\t
        RotateShape[Cylinder[1, 2, 40], 0, \[Pi]\/2, \[Pi]\/2]], \n\t\t
      Boxed \[Rule] False]; \)\)], "Input"],

Cell["?AffineShape", "Input"],

Cell[BoxData[
    \(\(Show[
      Graphics3D[\n\t\tAffineShape[Cylinder[1, 2, 40], {0.2, 0.5, 0.02}]], \n
      \t\tBoxed \[Rule] False]; \)\)], "Input"],

Cell["Do the same with any other shape.", "Question"],

Cell["\<\
Create a two-dimensional plot with two or more functions and attach \
a legend to it.\
\>", "Question"],

Cell[BoxData[
    \(\(Plot[{x, x\^2, x\^3}, \ {x, 0, 10}, \n\t
      PlotStyle -> {GrayLevel[0.3], \ Dashing[{0.02}], \ Hue[0]}, \n\t
      PlotLegend -> {x, x^2, x^3}]; \)\)], "Input"],

Cell[TextData[{
  "Modify the legend in the previous exercise using the various ",
  StyleBox["Legend", "Input"],
  " options."
}], "Question"],

Cell[BoxData[
    \(\(Plot[{x, x\^2, x\^3}, \ {x, 0, 10}, \n\t
      PlotStyle -> {GrayLevel[0.3], \ Dashing[{0.02}], \ Hue[0]}, \n\t
      PlotLegend -> {x, x^2, x^3}, \n\tLegendPosition -> {1.1, \ \(-0.4\)}, \n
      \tLegendSize -> {0.9, \ 1}, \n\t
      LegendLabel -> 
        StyleForm["\<Exponents\>", FontFamily -> "\<Times\>", 
          FontWeight -> "\<Bold\>", FontSize -> 18], \n\t
      ShadowBackground -> GrayLevel[0.5]]; \)\)], "Input"],

Cell[" ", "Special3"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["4. Standard Add-on Packages", "Section"],

Cell[TextData[{
  "In addition to the graphics functions that are built into ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ", many of the functions defined in  the standard add-on packages increase \
",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "\[CloseCurlyQuote]s graphical capabilities. These functions allow us to \
create additional types of graphics, simplify some plotting tasks, or enhance \
functions already defined in the ",
  StyleBox["Mathematica ",
    FontSlant->"Italic"],
  "kernel",
  StyleBox[".",
    FontSlant->"Italic"]
}], "Text"],

Cell[TextData[{
  "This section will provide an overview of all the graphics functions \
contained in the standard add-on packages. For more in-depth coverage of \
these functions, consult the Help Browser or the text ",
  StyleBox["Mathematica 3.0 Standard Add-on Packages",
    FontSlant->"Italic"],
  ", which is included with most versions of ",
  StyleBox["Mathematica ",
    FontSlant->"Italic"],
  "3.0."
}], "Text"],

Cell[CellGroupData[{

Cell["Graphics", "Subsection"],

Cell[TextData[{
  "This group of packages contains most of the graphics functions found in \
the standard add-on packages. It is important to note that we could load each \
package indivdually as we needed to use the functions in that package, but \
there is a more convenient way to do this automatically. This is through the \
use of ",
  StyleBox["master",
    FontSlant->"Italic"],
  " packages. "
}], "Text"],

Cell[CellGroupData[{

Cell["Animation", "Subsubsection"],

Cell[TextData[{
  "This package contains various animation functions for regular ",
  StyleBox["x",
    FontSlant->"Italic"],
  "-",
  StyleBox["y",
    FontSlant->"Italic"],
  ", density, contour, and parametric curve plots.  Animation of \
three-dimensional plots and rotation of two-dimensional plots is also \
supported."
}], "Text"],

Cell[TextData[{
  "Evaluate the following code to see examples of animation functions \
available in the ",
  StyleBox["Animation", "Input"],
  " package."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`Animation`\>"]\)], "Input"],

Cell[BoxData[
    \(Animate[
      Plot[Sin[x\ t], {x, \(-3\), 3}, PlotRange \[Rule] {\(-1\), 1}], {t, 0, 
        1}]\)], "Input"],

Cell[BoxData[
    \(\(MoviePlot[Sin[x\ t], {x, \(-5\), 5}, {t, 0, 1}, 
      PlotRange \[Rule] {\(-1\), 1}\)\)], "Input"],

Cell[BoxData[
    \(MoviePlot3D[Sin[x\ y\ t], {x, \(-2\), 2}, {y, \(-2\), 2}, {t, 0, 1}, 
      PlotRange \[Rule] {\(-1\), 1}, Frames \[Rule] 24]\)], "Input"],

Cell[BoxData[
    \(MovieParametricPlot[{Sin[x\ t], Cos[x\ t]}, {x, \(-\[Pi]\), \[Pi]}, {t, 
        0, 1, 1\/11}, \n\tPlotRange \[Rule] {{\(-1\), 1}, {\(-1\), 1}}, 
      AspectRatio \[Rule] 1]\)], "Input"],

Cell[BoxData[
    \(\(SpinShow[
      Plot3D[Sin[x\ y], {x, \(-2\), 2}, {y, \(-2\), 2}, Axes \[Rule] None, 
        Boxed \[Rule] False]]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["ArgColors", "Subsubsection"],

Cell["\<\
This package provides functions for specifying the color or gray \
level of graphics elements using the argument of a complex number.\
\>", 
  "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`ArgColors`\>"]\)], "Input"],

Cell[BoxData[
    \(func[z_] := Exp[\(2\ \[Pi]\ \[ImaginaryI]\ z\)\/10]\)], "Input"],

Cell[BoxData[
    \(\(circ1\  = \ \n\t
      Graphics[Table[\n
          \t\t\t{ArgShade[func[z]], Disk[{Re[func[z]], Im[func[z]]}, 0.2]}, \n
          \t{z, 10}], AspectRatio \[Rule] Automatic]; \)\)], "Input"],

Cell[BoxData[
    \(\(circ2\  = \ \n\t
      Graphics[Table[\n
          \t\t\t{ArgColor[func[z]], Disk[{Re[func[z]], Im[func[z]]}, 0.2]}, \n
          \t{z, 10}], AspectRatio \[Rule] Automatic]; \)\)], "Input"],

Cell[BoxData[
    \(\(circ3\  = \ \n\t
      Graphics[Table[\n
          \t\t\t{ColorCircle[func[z]], 
            Disk[{Re[func[z]], Im[func[z]]}, 0.2]}, \n\t{z, 10}], 
        AspectRatio \[Rule] Automatic]; \)\)], "Input"],

Cell["Show[GraphicsArray[ {circ1, circ2, circ3} ]];", "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Colors", "Subsubsection"],

Cell[TextData[{
  "This package provides functions that convert a color expressed in one of \
several color systems to an ",
  StyleBox["RGBColor", "Input"],
  ". It also allows you to find the appropriate ",
  StyleBox["RGBColor", "Input"],
  " for a particular color name (such as ",
  StyleBox["Blue", "Input"],
  ")."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`Colors`\>"]\)], "Input"],

Cell[TextData[{
  "The following example code will display a vertical listing of the name and \
a swatch of each color in the ",
  StyleBox["Colors", "Input"],
  " package."
}], "Text"],

Cell[TextData[{
  "Use this definition for ",
  StyleBox["colors", "Input"],
  " for ten random colors."
}], "Text"],

Cell[BoxData[
    \(\(colors\  = \ 
      Sort[Table[
          AllColors[\([Random[Integer, \ {1, Length[AllColors]}]]\)], {10}]]; 
    \)\)], "Input"],

Cell[TextData[{
  "Use this initial value for ",
  StyleBox["colors", "Input"],
  " if you want an alphabetical listing of the all colors available."
}], "Text"],

Cell[BoxData[
    \(\(colors\  = \ AllColors; \)\)], "Input"],

Cell[TextData[{
  "Use these two lines to create a listing of all colors available sorted by \
their ",
  StyleBox["CMYKColor", "Input"],
  " value."
}], "Text"],

Cell[BoxData[
    \(colors = \(({#1, ToColor[ToExpression[#1], CMYKColor]}&)\)/@AllColors; 
    colors = First[
        Transpose[
          Sort[colors, 
            OrderedQ[{#1\[LeftDoubleBracket]2\[RightDoubleBracket], #2
                    \[LeftDoubleBracket]2\[RightDoubleBracket]}]&]]]; \)], 
  "Input"],

Cell[TextData[{
  "The rest of this code can be used by either value of ",
  StyleBox["colors", "Input"],
  " above to display the color names and swatches."
}], "Text"],

Cell[BoxData[
    \(colors = 
      Table[{ToExpression[colors\[LeftDoubleBracket]i\[RightDoubleBracket]], 
          \n\t\t\tRectangle[{0, \(-10\)\ i}, {15, \(-10\)\ i - 10}], 
          RGBColor[0, 0, 0], 
          Text[colors\[LeftDoubleBracket]i\[RightDoubleBracket], {20, 
              \(-10\)\ i - 5}, {\(-1\), 0}]}, {i, Length[colors]}]; \n
    Show[Graphics[colors], \n\tAspectRatio \[Rule] Automatic, 
      PlotRange \[Rule] {{0, 80}, {0, \(-\((Length[colors]\ 10 + 10)\)\)}}]; 
    \)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["ComplexMap", "Subsubsection"],

Cell[TextData[{
  "This package plots the images of Cartesian coordinate lines and polar \
coordinate lines under a user-supplied function ",
  StyleBox["f",
    FontSlant->"Italic"],
  "."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`ComplexMap`\>"]\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t
        CartesianMap[Cos, {0.2, \[Pi] - 0.2}, {\(-2\), 2}], \n\t
        CartesianMap[Zeta, {0.1, 0.9}, {0, 20}], \n\t
        CartesianMap[
          1\/Conjugate[#1]&, {\(-2\), 2, 4\/19}, {\(-2\), 2, 4\/19}, \n\t\t
          PlotRange \[Rule] All]}]; \)\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t
        PolarMap[Log, {0.1, 10, 0.5}, {\(-3\), 3, 0.15}], \n\t
        PolarMap[
          1\/Conjugate[#1]&, {0.1, 5.1, 0.5}, {\(-\[Pi]\), \[Pi], 
            \[Pi]\/12}], \n\t
        PolarMap[Sqrt, {1}, {\(-\[Pi]\) - 0.0001, \[Pi]}], \n\t
        PolarMap[Identity, {1, 2}, {\(-\[Pi]\), \[Pi], \[Pi]\/12}, 
          Frame \[Rule] True]}]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["ContourPlot3D", "Subsubsection"],

Cell[TextData[{
  "This package introduces ",
  StyleBox["ContourPlot3D", "Input"],
  " and ",
  StyleBox["ListContourPlot3D", "Input"],
  ", the three-dimensional analogs of the built-in functions ",
  StyleBox["ContourPlot", "Input"],
  " and ",
  StyleBox["ListContourPlot", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`ContourPlot3D`\>"]\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t\t
        ContourPlot3D[
          x\ y\ z, {x, \(-1\), 1}, {y, \(-1\), 1}, {z, \(-1\), 1}, \n\t\t\t
          Contours \[Rule] {.1}], \n\t\t
        ContourPlot3D[
          \@\(x\^2 + y\^2 + z\^2\), {x, \(-2\), 2}, {y, \(-2\), 2}, {z, 
            \(-2\), 2}, \n\t\t\tContours \[Rule] {1.}], \n\t\t
        ContourPlot3D[
          Sin[\@\(x\^2 + y\^2 + z\^2\)], {x, \(-2\), 2}, {y, 0, 2}, {z, 
            \(-2\), 2}, \n\t\t\tContours \[Rule] {0.75}]}]; \)\)], "Input"],

Cell[BoxData[
    \(data = 
      Table[x\^2 + 2\ y\^2 + 3\ z\^2, \n
        \t\t{z, \(-1\), 1, 0.25}, {y, \(-1\), 1, 0.25}, {x, \(-1\), 1, 
          0.25}]; \n
    ListContourPlot3D[data, \n\t
      MeshRange \[Rule] {{\(-1\), 1}, {\(-1\), 1}, {\(-1\), 1}}, 
      Contours \[Rule] {3.}]; \)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["FilledPlot", "Subsubsection"],

Cell["\<\
This package allows us to fill the space between a plotted function \
and the x-axis or between a pair of plotted functions with a color.\
\>", 
  "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`FilledPlot`\>"]\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t\tFilledPlot[Sin[x]\/x, {x, \(-10\), 10}], 
        FilledPlot[{Sin[x]\/x, Cos[x]\/x, 1\/x}, {x, \(-10\), 10}], 
        FilledPlot[{x\^2\/18, Cos[x], Sin[x]}, {x, 0, 2\ \[Pi]}, 
          Fills \[Rule] {{{1, Axis}, GrayLevel[0.8]}, {{2, 3}, 
                GrayLevel[0.3]}}, Curves \[Rule] Front]}]; \)\)], "Input"],

Cell["FilledListPlot[{2, 3, 1, 5, 4}]; ", "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t\t
        Show[Graphics[\n
            \t\t\t\t{GrayLevel[0.6], Polygon[{{0, 0}, {1, 0}, {1, 1}}]}, 
            AxesFront \[Rule] False, Axes \[Rule] True, 
            AxesOrigin \[Rule] {0.5, 0.5}]], \n\t\t
        Show[Graphics[\n
            \t\t\t\t{GrayLevel[0.6], Polygon[{{0, 0}, {1, 0}, {1, 1}}]}, 
            AxesFront \[Rule] True, Axes \[Rule] True, 
            AxesOrigin \[Rule] {0.5, 0.5}]]}]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Graphics", "Subsubsection"],

Cell["\<\
This package provides special functions for plotting in two \
dimensions.  Special formats include bar charts, pie charts, log plots, polar \
plots, and error bar plots.\
\>", "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`Graphics`\>"]\)], "Input"],

Cell[BoxData[
    \(LinearScale[1, 2]\)], "Input"],

Cell[BoxData[
    \(Short[\ \ LogScale[1, 10], 10\ \ ]\)], "Input"],

Cell[BoxData[
    \(UnitScale[2, 10, 0.7]\)], "Input"],

Cell[BoxData[
    \(PiScale[0, 10]\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t
        TextListPlot[\n\t\t{{1.5, 2.5}, {1.6, 2.6}, {1.7, 2.7}, {1.8, 2.8}}, 
          \n\t\t\tTextStyle -> {FontSize -> 6}], \n\t
        TextListPlot[{{1.5, 2.5, 4}, {1.6, 2.6, 3}, {1.7, 2.7, 2}, {1.8, 2.8, 
              1}}, \n\t\t\tTextStyle -> {FontSize -> 6}], \n\t
        LabeledListPlot[{{1.5, 2.5, 1}, {1.6, 2.6, 2}, {1.7, 2.7, 3}, {1.8, 
              2.8, 4}}, \n\t\t\tTextStyle -> {FontSize -> 6}]}]; \)\)], 
  "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n
        \t\t{LogPlot[Sin[x], {x, 0.1, 3.1}], \n\t\ \ \ \ \ 
          LogPlot[Exp[4\ x], {x, 1, 5}, Frame \[Rule] True]}, \n
        \t\ \ {LogPlot[Exp[2\ x], {x, 1, 5}, \n\t\t\tFrame \[Rule] True, 
            GridLines \[Rule] {Automatic, LogGridMajor}], \n\t\ \ \ \ 
          LogPlot[Exp[4\ x], {x, 1, 3}, \n\t\t\tFrame \[Rule] True, 
            GridLines \[Rule] {Automatic, LogGridMinor}]}}]; \)\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t\t
        LogListPlot[Table[{i\/2, i\^2}, {i, 20}]], \n\t\t
        LogLogPlot[Sin[x], {x, 0.1, 3.1}], \n\t\t
        LogLogListPlot[Table[{i\^2, i\^3}, {i, 10}]]}]; \)\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t\t
        PolarPlot[{Cos[t], Sin[2\ t]}, {t, 0, 2\ \[Pi]}], \n\t\t
        PolarListPlot[Table[{t\/2, Cos[t]}, {t, 0, 2\ \[Pi], 0.1}]]}]; \)\)], 
  "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t\tErrorListPlot[Table[{i, \@i}, {i, 10}]], 
        \n\t\tErrorListPlot[Table[{Sin[t], Cos[t], t}, {t, 10}]]}]; \)\)], 
  "Input"],

Cell[BoxData[{
    \(\(data = 
      Table[{n\/15, \((n\/15)\)\^2 + 2 + Random[Real, {\(-0.3\), 0.3}]}, {n, 
          15}]; \)\), 
    \(\(datafit = Fit[data, {1, x, x\^2}, x]; \)\), 
    \(\(ListAndCurvePlot[data, datafit, {x, 0, 1}]; \)\)}], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t\tBarChart[Table[i, {i, 1, 10}]], \n\t\t
        BarChart[Table[{Sin[t], Sin[t]}, {t, 0.6`3, 3`3, 0.6`3}]]}]; \)\)], 
  "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t\tPieChart[Table[i, \ {i, \ 5}]], \n\t\t
        PieChart[Table[{i, \ A[i]}, \ {i, \ 7}]]}]; \)\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t\tPieChart[{.2, .3, .1}], \n\t\t
        PieChart[{.2, .3, .1}, PieExploded \[Rule] All], \n\t\t
        PieChart[{.2, .3, .1}, PieExploded \[Rule] {3, .2}]}]; \)\)], "Input"],

Cell[BoxData[
    \(PlotStyle[Plot[Sin[x], {x, 0, \[Pi]}]]\)], "Input"],

Cell[BoxData[
    \(PlotStyle[
      Plot[Sin[x], {x, 0, \[Pi]}, 
        PlotStyle \[Rule] {{Dashing[{.02, .02}], Thickness[.007]}}]]\)], 
  "Input"],

Cell[BoxData[
    \(\(?TransformGraphics\)\)], "Input"],

Cell[BoxData[
    \(g1 = 
      Plot[Sin[t], {t, 0, 2  \[Pi]}, \ DisplayFunction \[Rule] Identity]; \n
    g2 = TransformGraphics[g1, \((Sin[#] + Cos[#])\)&]; \n
    Show[GraphicsArray[{g1, \ g2}]]; \)], "Input"],

Cell[BoxData[
    \(g1 = Plot[Sin[t], {t, 0, \[Pi]}, DisplayFunction \[Rule] Identity]; \n
    g2 = SkewGraphics[g1, {{1, 2}, {0, 1}}]; \n
    Show[GraphicsArray[{g1, \ g2}]]; \)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Graphics3D", "Subsubsection"],

Cell["\<\
This package provides special functions for plotting in three \
dimensions.  Special formats include bar charts, scatter plots, surface \
plots, shadow plots, and projections.\
\>", "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`Graphics3D`\>"]\)], "Input"],

Cell[BoxData[{
    \(Needs["\<Graphics`ParametricPlot3D`\>"]\), 
    \(g1 = 
      CylindricalPlot3D[r\^2, {r, 0, 1}, {\[Phi], 0, 2\ \[Pi]}, \n\t\t
        DisplayFunction \[Rule] Identity]; \n
    g2 = TransformGraphics3D[g1, Cos[#1]&]; \n
    Show[GraphicsArray[{g1, \ g2}]]; \)}], "Input"],

Cell[BoxData[
    \(\(Show[Graphics3D[Plot[Sin[t], {t, 0, \[Pi]}]]]; \)\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[\n
      \t{CylindricalPlot3D[r\^2, {r, 0, 1}, {\[Phi], 0, 2\ \[Pi]}, \n\t\t\t\ 
          Axes -> False], \n\t\t
        Show[SkewGraphics3D[g1, {{1, 2, 0}, {0, 1, 0}, {0, 0, 1}}], \n\t\t\t
          Axes -> False]}]; \)\)], "Input"],

Cell[BoxData[
    \(g1 = 
      Table[Plot[x\^n, {x, 0, 5}, TextStyle -> {FontSize -> 6}, 
          DisplayFunction \[Rule] Identity], {n, 4}]; \n
    DisplayTogetherArray[{\n\t\tShow[GraphicsArray[Partition[g1, \ 2]]], \n
        \t\tShow[StackGraphics[g1]]}]; \)], "Input"],

Cell[BoxData[
    \(\(BarChart3D[{{1, 2, 3}, {4, 5, 6}}]; \)\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t
        ScatterPlot3D[Table[{t, Sin[t], Cos[t]}, {t, 0, 10, 0.1}], \n\t\t\t
          Axes \[Rule] False], \n\t
        ScatterPlot3D[Table[{t, Sin[t], Cos[t]}, {t, 0, 10, 0.1}], \n\t\t\t
          PlotJoined \[Rule] True, \ Axes \[Rule] False]}]; \)\)], "Input"],

Cell[BoxData[
    \(\(ListSurfacePlot3D[
      Table[{i, j, Sin[i - Cos[j]]}, {i, 1, 10, .5}, {j, 1, 10, .5}]]; \)\)], 
  "Input"],

Cell[BoxData[
    \(g1 = 
      ParametricPlot3D[{x, Cos[t]\ Sin[x], Sin[t]\ Sin[x]}, \n
        \t\t{x, \(-\[Pi]\), \[Pi]}, {t, 0, 2\ \[Pi]}, \n\t\t
        Axes \[Rule] False, DisplayFunction \[Rule] Identity]; 
    g2 = Shadow[g1, ZShadow \[Rule] False, DisplayFunction \[Rule] Identity]; 
    g3 = Show[Project[g1, {0, 1, 0}], DisplayFunction \[Rule] Identity]; 
    Show[GraphicsArray[{g1, g2, g3}], 
      DisplayFunction \[Rule] $DisplayFunction]; \)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["ImplicitPlot", "Subsubsection"],

Cell[TextData[{
  "The built-in function ",
  StyleBox["Plot", "Input"],
  " requires one to specify an explicit function. Many simple graphs (",
  StyleBox["e.g.",
    FontSlant->"Italic"],
  ", circles, ellipses, etc.) are not functions. ",
  StyleBox["ImplicitPlot", "Input"],
  " allows one to plot figures defined by equations."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`ImplicitPlot`\>"]\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n
        \t\t{ImplicitPlot[x\^2 + y\^2 == x\ y + 3, {x, \(-3\), 3}], \n\t\t
          ImplicitPlot[\((x\^2 + y\^2)\)\^2 == x\^2 - y\^2, {x, \(-2\), 2}], 
          \n\t\tImplicitPlot[x\^3 + y\^3 == 3\ x\ y, {x, \(-3\), 3}], \n\t\t
          ImplicitPlot[x\^2 + y\^2 == 3\ x\ y + 3, {x, \(-10\), 10}, 
            PlotRange \[Rule] {{\(-10\), 10}, {\(-10\), 10}}]}, \n
        \t\t{ImplicitPlot[{\((x\^2 + y\^2)\)\^2 == x\^2 - y\^2, 
              \((x\^2 + z\^2)\)\^2 == 2\ x\ z}, {x, \(-2\), 2}, \n\t\t\t
            PlotStyle \[Rule] {GrayLevel[0], Dashing[{.01}]}], \n\t\t
          ImplicitPlot[x\^2 + y\^2 == \[Pi], {x, \(-2\), 2}], \n\t\t
          ImplicitPlot[
            Sin[2\ x] + Cos[3\ y] == 1, {x, \(-2\)\ \[Pi], 2\ \[Pi]}, {y, 
              \(-2\)\ \[Pi], 2\ \[Pi]}], \n\t\t
          ImplicitPlot[
            x\^3 + x\ y + y\^2 == 1, {x, \(-2\)\ \[Pi], 2\ \[Pi]}, {y, 
              \(-2\)\ \[Pi], 2\ \[Pi]}]}}]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["MultipleListPlot", "Subsubsection"],

Cell["\<\
This package contains a function for plotting multiple sets of data \
in a graph and related utilities.\
\>", "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`MultipleListPlot`\>"]\)], "Input"],

Cell[BoxData[
    \(list1 = Table[{x, Sin[x]}, {x, 0, 2  \[Pi], \[Pi]\/6}]; \n
    list2 = Table[{x, Cos[x]}, {x, 0, 2  \[Pi], \[Pi]\/6}]; \)], "Input"],

Cell[BoxData[
    \(\(MultipleListPlot[list1, list2, PlotJoined -> True, ]; \)\)], "Input"],

Cell[BoxData[
    \(\(MultipleListPlot[\n
      \t{2, \n\t{1.5, 3.2}, \n\t{2.5, ErrorBar[0.3]}, \n
        \t{{4.4, 5.2}, ErrorBar[{\(-0.5\), 0.3}]}, \n
        \t{{5.5, 2.1}, ErrorBar[{\(-0.4\), 0.3}, {\(-0.2\), 0.5}]}}, \n
      Frame -> True]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[Graphics[RegularPolygon[10, 1, {0, 0}, 0, 3]], \n\t
      AspectRatio -> Automatic]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[
      Graphics[MapIndexed[\n
          \t\t\t{#1, Line[{{0, \(-First[#2]\)}, {1, \(-First[#2]\)}}]}&, \n
          \t\t{Dashing[{Dot}], \n\t\t\ \ Dashing[{Dot, Dash}], \n\t\t\ \ 
            Dashing[{Dot, Dash, Dot, LongDash}], \n\t\t\ \ 
            Dashing[{Dot, Dot, Dash}]}]]]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["ParametricPlot3D", "Subsubsection"],

Cell[TextData[{
  "This package allows us to specify the step size of the grid used by ",
  StyleBox["ParametricPlot3D", "Input"],
  ".  In contrast, the built-in function ",
  StyleBox["ParametricPlot3D", "Input"],
  " requires that the grid be specified using the option ",
  StyleBox["PlotPoints", "Input"],
  ".  The package also introduces the functions ",
  StyleBox["PointParametricPlot3D", "Input"],
  ", ",
  StyleBox["SphericalPlot3D", "Input"],
  ", and ",
  StyleBox["CylindricalPlot3D", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`ParametricPlot3D`\>"]\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n
        \t\t{ParametricPlot3D[{x, y, x\ y}, {x, \(-1\), 1, 0.2}, {y, \(-1\), 
              1}, Axes \[Rule] False], \n\t\t
          ParametricPlot3D[{Cos[t], Sin[t], t\/6}, {t, 0, 4\ \[Pi], 
              \[Pi]\/12}, Axes \[Rule] False], \n\t\t
          CylindricalPlot3D[
            r\^2, {r, 0, 1}, {\[Phi], \(-\(\[Pi]\/4\)\), \(5\ \[Pi]\)\/4}, 
            Axes \[Rule] False, ViewPoint \[Rule] {1.3, \(-2.4\), 1.6}]}, \n
        \t\t{SphericalPlot3D[
            Sin[\[Theta]]\^2, {\[Theta], 0, \[Pi]}, {\[Phi], 0, 
              \(3\ \[Pi]\)\/2}, Axes \[Rule] False], \n\t\t
          CylindricalPlot3D[
            1.5\ \@\(1 + r\^2\), {r, 0, 2}, {\[Phi], 0, 2\ \[Pi]}, 
            Axes \[Rule] False], \n\t\t
          ParametricPlot3D[{Cosh[z]\ Cos[\[Phi]], Cosh[z]\ Sin[\[Phi]], z}, {
              z, \(-2\), 2}, {\[Phi], 0, 2\ \[Pi]}, Axes \[Rule] False]}}]; 
    \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["PlotField", "Subsubsection"],

Cell[TextData[{
  "This package does plots of vector fields in the plane. ",
  StyleBox["PlotVectorField", "Input"],
  " allows us to specify the functions describing the two components of the \
field.  ",
  StyleBox["PlotGradientField", "Input"],
  " and ",
  StyleBox["PlotHamiltonianField", "Input"],
  " plot the respective vector fields associated with a scalar function. ",
  StyleBox["PlotPolyaField", "Input"],
  " plots the field associated with a complex-valued function.  ",
  StyleBox["ListPlotVectorField", "Input"],
  " plots a rectangular array of vectors."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`PlotField`\>"]\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t\t
        PlotVectorField[{Sin[x\ y], Cos[x\ y]}, {x, 0, \[Pi]}, {y, 0, 
            \[Pi]}], \n\t\t
        PlotGradientField[x\^3 + y\^4, {x, 0, 10}, {y, 0, 10}], \n\t\t
        PlotPolyaField[\((x + I\ y)\)\^4, {x, 5, 10}, {y, 5, 10}]}]; \)\)], 
  "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["PlotField3D", "Subsubsection"],

Cell[TextData[{
  "This package does plots of vector fields in three dimensions. ",
  StyleBox["PlotVectorField3D", "Input"],
  " allows us to specify the functions describing the three components of the \
field.  ",
  StyleBox["PlotGradientField3D", "Input"],
  " plots the gradient vector field associated with a scalar function. ",
  StyleBox["ListPlotVectorField3D", "Input"],
  " plots a three-dimensional array of vectors."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`PlotField3D`\>"]\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n\t\t
        PlotVectorField3D[{y, \(-x\), 0}\/z, {x, \(-1\), 1}, {y, \(-1\), 1}, {
            z, 1, 3}], \n\t\t
        PlotGradientField3D[
          x\ y\ z, {x, \(-1\), 1}, {y, \(-1\), 1}, {z, \(-1\), 1}], \n\t\t
        ListPlotVectorField3D[\n\t\t\t
          Flatten[Table[{{i, j, k}, \n
                \t\t\t\t\t\t{Random[Real, {\(-1\), 1}], 
                  Random[Real, {\(-1\), 1}], Random[Real, {\(-1\), 1}]}}, {i, 
                7}, {j, 7}, {k, 7}], 2]]}]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Shapes", "Subsubsection"],

Cell[TextData[{
  "This package provides lists of polygons for some common three-dimensional \
shapes.  Functions for translating, rotating, and affine transformations of ",
  
  StyleBox["Graphics3D", "Input"],
  " objects and converting a ",
  StyleBox["Graphics3D", "Input"],
  " object to its wire frame representation are also included."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`Shapes`\>"]\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n
        \t\t{Show[Graphics3D[Cylinder[0.5, 0.5]], Boxed \[Rule] False], \n\t\t
          Show[Graphics3D[Torus[2, 0.7, 15, 14]], Boxed \[Rule] False], \n\t\t
          Show[Graphics3D[Sphere[]], Boxed \[Rule] False], \n\t\t
          Show[Graphics3D[MoebiusStrip[2, 1, 80]], Boxed \[Rule] False], \n
          \t\tShow[Graphics3D[Helix[]], Boxed \[Rule] False]}, \n
        \t\t{Show[Graphics3D[DoubleHelix[]], Boxed \[Rule] False], \n\t\t
          Show[RotateShape[Graphics3D[MoebiusStrip[]], \[Pi]\/4, \[Pi]\/3, 
              \[Pi]\/2], Boxed \[Rule] False], \n\t\t
          Show[TranslateShape[
              RotateShape[Graphics3D[MoebiusStrip[]], \[Pi]\/4, \[Pi]\/3, 
                \[Pi]\/2], {1, 2, 3}], \n\t\t\t\t\t
            RotateShape[Graphics3D[MoebiusStrip[]], \[Pi]\/4, \[Pi]\/3, 
              \[Pi]\/2], Boxed \[Rule] False], \n\t\t
          Show[AffineShape[Graphics3D[Cone[]], {1, 2, 3}], 
            Boxed \[Rule] False], \n\t\t
          Show[WireFrame[Graphics3D[Cone[]]], Boxed \[Rule] False]}}]; \)\)], 
  "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Spline", "Subsubsection"],

Cell[TextData[{
  "This package introduces a ",
  StyleBox["Spline", "Input"],
  " graphics primitive and provides utilities for rendering splines."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`Spline`\>"]\)], "Input"],

Cell[BoxData[
    \(pts\  = \ {{0, 0}, {1, 2}, {\(-1\), 3}, {0, 1}, {3, 0}}; \n
    DisplayTogetherArray[{\n\t\t
        Show[Graphics[{Hue[0], \ Line[pts], \ GrayLevel[0], \ 
              Spline[pts, \ Cubic]}], PlotRange\  \[Rule] All], \n\t\t
        Show[Graphics[{Hue[0], \ Line[pts], \ GrayLevel[0], \ 
              Spline[pts, \ Bezier]}], PlotRange\  \[Rule] All], \n\t\t
        Show[Graphics[
            Spline[pts, \ CompositeBezier, \ 
              SplineDots \[Rule] Automatic]]]}]; \)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["SurfaceOfRevolution", "Subsubsection"],

Cell["\<\
This package provides functions for plotting surfaces generated by \
revolving a curve about an axis.  The curve may be expressed as a function, \
parametrically, or as a list of points.  Rotation can be about an arbitrary \
axis.\
\>", "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`SurfaceOfRevolution`\>"]\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n
        \t{SurfaceOfRevolution[Sin[x], {x, 0, 2  \[Pi]}], \n\t\t
          SurfaceOfRevolution[{1.1\ Sin[u], u\^2}, {u, 0, \(3  \[Pi]\)\/2}, 
            BoxRatios -> {1, 1, 2}]}, \n
        \t{SurfaceOfRevolution[x\^2, {x, 0, 1}, RevolutionAxis -> {1, 1, 1}], 
          \n\t\tListSurfaceOfRevolution[
            Table[{n, n\^3}, {n, 0, 1, 0.1}], {t, 0, \[Pi]\/2}, \n\t\t\t
            RevolutionAxis -> {1, \(-1\), 1}, PlotRange -> All]}}]; \)\)], 
  "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["ThreeScript", "Subsubsection"],

Cell[TextData[{
  "This package provides a utility for converting three-dimensional graphical \
objects from ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to the 3-Script file format and writing them into files (or suitable \
operating system pipes)."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`ThreeScript`\>"]\)], "Input"],

Cell[BoxData[
    \(obj = Graphics3D[Polygon[{{0, 0, 0}, {0, 1, 0}, {0, 1, 1}}]]\)], "Input"],

Cell[BoxData[
    \(ThreeScript["\<object.ts\>", obj]\)], "Input"],

Cell[BoxData[
    \(\(! \(! object.ts\)\)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["DiscreteMath", "Subsection"],

Cell[CellGroupData[{

Cell["Combinatorica", "Subsubsection"],

Cell["\<\
This package defines over 230 combinatoric and graph theory \
functions. It includes functions for constructing graphs and other \
combinatorial objects, computing invariants of these objects, and finally \
displaying them.\
\>", "Text"],

Cell[TextData[{
  "The package contains all the programs from the book, ",
  StyleBox[
  "Implementing Discrete Mathematics: Combinatorics and Graph Theory with \
Mathematica, ",
    FontSlant->"Italic"],
  "by Steven S. Skiena, Addison-Wesley Publishing Co., Advanced Book Program, \
350 Bridge Parkway, Redwood City CA 94065.  ISBN 0-201-50943-1. For ordering \
information, call 1-800-447-2226.\n\nThe programs from the book, ",
  StyleBox[
  "Implementing Discrete Mathematics: Combinatorics and Graph Theory with \
Mathematica",
    FontSlant->"Italic"],
  ", are available by anonymous ",
  StyleBox["ftp.cs.sunysb.edu",
    FontFamily->"Courier"],
  " in the ",
  StyleBox["pub/Combinatorica",
    FontFamily->"Courier"],
  " directory.  \n\nAny comments, bug reports, or requests to get on the \
Combinatorica mailing list should be forwarded to:\n\nSteven Skiena\n\
Department of Computer Science\nState University of New York\nStony Brook, NY \
11794\n\nskiena@sbcs.sunysb.edu\n\n(516) 632-9026 / 8470"
}], "Text"],

Cell[BoxData[
    \(Needs["\<DiscreteMath`Combinatorica`\>"]\)], "Input"],

Cell[BoxData[
    \(\(c1 = FerrersDiagram[RandomPartition[100]]; \)\)], "Input"],

Cell[BoxData[
    \(\(c2 = ShowGraph[CompleteGraph[5]]; \)\)], "Input"],

Cell[BoxData[
    \(\(c3 = ShowGraph[RankedEmbedding[GridGraph[5, 5], {13}]]; \)\)], "Input"],

Cell[BoxData[
    \(\(c4 = ShowGraph[LineGraph[CompleteGraph[5]]]; \)\)], "Input"],

Cell[BoxData[
    \(\(c5 = ShowGraph[CirculantGraph[21, RandomKSubset[Range[10], 3]]]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(c6 = ShowGraph[OrientGraph[Wheel[10]], Directed]; \)\)], "Input"],

Cell[BoxData[
    \(\(c7 = ShowLabeledGraph[MakeGraph[Range[8], \((Mod[#1, #2] == 0)\)&]]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(c8 = ShowGraph[MinimumSpanningTree[CompleteGraph[6, 6, 6]]]; \)\)], 
  "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{{c1, c2, c3, c4}, {c5, c6, c7, c8}}]]; \)\)], 
  "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["ComputationalGeometry", "Subsubsection"],

Cell["\<\
This package implements selected functions from computational \
geometry relevent to nearest neighbor point location, including \
triangulation.\
\>", "Text"],

Cell[BoxData[
    \(Needs["\<DiscreteMath`ComputationalGeometry`\>"]\)], "Input"],

Cell[BoxData[
    \(input2D
       = {{4.4, 14}, {6.7, 15.25}, {6.9, 12.8}, {2.1, 11.1}, \n
        \t\t\t\t\t\t\t\t\t\t{9.5, 14.9}, {13.2, 11.9}, {10.3, 12.3}, {6.8, 
          9.5}, \n\t\t\t\t\t\t\t\t\t\t{3.3, 7.7}, {.6, 5.1}, {5.3, 2.4}, {
          8.45, 4.7}, \n
        \t\t\t\t\t\t\t\t\t\t{11.5, 9.6}, {13.8, 7.3}, {12.9, 3.1}, {11, 
          1.1}}; \ninput3D = 
      Map[{#1\[LeftDoubleBracket]1\[RightDoubleBracket], #1
              \[LeftDoubleBracket]2\[RightDoubleBracket], 
            \@\(64 - 
                \((\((#1\[LeftDoubleBracket]1\[RightDoubleBracket])\) - 8)
                    \)\^2 - 
                \((\((#1\[LeftDoubleBracket]2\[RightDoubleBracket])\) - 8)
                    \)\^2\)}&, input2D]; \)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n
        \t\t{PlanarGraphPlot[input2D, ConvexHull[input2D]], \n\t\t\t
          PlanarGraphPlot[input2D]}, \n
        \t\t{DiagramPlot[input2D], \n\t\t\tTriangularSurfacePlot[input3D]}}]; 
    \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Tree", "Subsubsection"],

Cell["\<\
This package introduces functions for creating, searching and \
displaying trees represented as nested lists.\
\>", "Text"],

Cell[BoxData[
    \(Needs["\<DiscreteMath`Tree`\>"]\)], "Input"],

Cell[BoxData[
    \(\(TreePlot[MakeTree[Range[15]]]; \)\)], "Input"],

Cell[BoxData[
    \(\(ExprPlot[f[g[a, b, c, d], g[x, y, h[x, y]], g[x, y]]]; \)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Geometry", "Subsection"],

Cell[CellGroupData[{

Cell["Polytopes", "Subsubsection"],

Cell["\<\
This package contains functions that give geometric characteristics \
of regular polygons and polyhedra.\
\>", "Text"],

Cell[BoxData[
    \(Needs["\<Geometry`Polytopes`\>"]\)], "Input"],

Cell[BoxData[
    \(\(?Geometry`Polytopes`*\)\)], "Input"],

Cell[BoxData[
    \(Area[Octagon]\)], "Input"],

Cell[BoxData[
    \(vert\  = \ Vertices[Octagon]\)], "Input"],

Cell[BoxData[
    \(\(Show[Graphics[{PointSize[.03], Point/@vert}, AspectRatio \[Rule] 1]]; 
    \)\)], "Input"],

Cell[BoxData[
    \(Volume[Tetrahedron]\)], "Input"],

Cell[BoxData[
    \(vert\  = \ Vertices[Tetrahedron]\)], "Input"],

Cell[BoxData[
    \(\(Show[
      Graphics3D[{PointSize[.03], Point/@vert}, AspectRatio \[Rule] 1]]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(TriangularSurfacePlot[vert]; \)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Miscellaneous", "Subsection"],

Cell[CellGroupData[{

Cell["WorldPlot", "Subsubsection"],

Cell[TextData[{
  "This package allows us to plot maps as graphic objects, where positions \
are expressed in terms of latitude and longitude.  The standard map \
projections ",
  StyleBox["Albers", "Input"],
  ", ",
  StyleBox["Equirectangular", "Input"],
  ", ",
  StyleBox["LambertAzimuthal", "Input"],
  ", ",
  StyleBox["LambertCylindrical", "Input"],
  ", ",
  StyleBox["Mercator", "Input"],
  ", ",
  StyleBox["Mollweide", "Input"],
  ", and ",
  StyleBox["Sinusoidal", "Input"],
  " are included in the package. In addition, ",
  StyleBox["WorldPlot", "Input"],
  " also loads data describing the names and outlines of countries of the \
world."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Miscellaneous`WorldPlot`\>"]\)], "Input"],

Cell[BoxData[
    \(\(w1 = WorldPlot[NorthAmerica]; \)\)], "Input"],

Cell[BoxData[
    \(\(w2 = WorldPlot[{World, RandomGrays}]; \)\)], "Input"],

Cell[BoxData[
    \(\(w3 = 
      WorldPlot[{World, RandomGrays}, \n\tWorldRotation -> {90, 0, 0}, 
        WorldRange -> {{0, 90}, {\(-180\), 180}}, \n\t
        WorldProjection -> LambertAzimuthal]; \)\)], "Input"],

Cell[BoxData[
    \(\(w4 = WorldPlot[{Africa, RandomGrays}, WorldProjection -> Sinusoidal]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(w5 = 
      WorldPlot[Africa, \n\t\tWorldBackground -> GrayLevel[0.7], \n\t\t
        WorldGrid -> None, \n\t\tWorldFrame -> Thickness[0.012], \n\t\t
        WorldProjection -> LambertCylindrical]; \)\)], "Input"],

Cell[BoxData[
    \({capetown, cairo} = \n\t
      ToMinutes[{{{\(-33\), \(-56\)}, {18, 22}}, {{30, 3}, {31, 15}}}]; \n
    w6 = Show[{w5, \n\t\t\t
          WorldGraphics[\n
            \t\t\t\t{Dashing[{0.05, 0.03}], Thickness[0.015], 
              Line[{capetown, cairo}]}]}]; \)], "Input"],

Cell[BoxData[
    \(\(Show[
      GraphicsArray[Partition[Map[Graphics, {w1, w2, w3, w4, w5, w6}], 3]]]; 
    \)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["NumericalMath", "Subsection"],

Cell[CellGroupData[{

Cell["Butcher", "Subsubsection"],

Cell[TextData[{
  "This package gives the order conditions that a Runge-Kutta method must \
satisfy to be of a particular order. It works for both implicit and explicit \
methods. The package also calculates and plots Butcher trees and implements \
the functions of trees defined in John Butcher\[CloseCurlyQuote]s book, ",
  StyleBox[
  "The Numerical Analysis of Ordinary Differential Equations: Runge-Kutta and \
General Linear Methods",
    FontSlant->"Italic"],
  ". Butcher\[CloseCurlyQuote]s row and column simplifying conditions assist \
in the derivation of high-order methods. A more compact and efficient \
stage-independent tensor notation has also been implemented."
}], "Text"],

Cell[BoxData[
    \(Needs["\<NumericalMath`Butcher`\>"]\)], "Input"],

Cell[BoxData[
    \(\(\(ButcherPlot[#, \n\tButcherPlotLabel -> \((InputForm\ /@\ #)\)]&\)\ @
      \ Flatten[ButcherTrees[4]]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Microscope", "Subsubsection"],

Cell[TextData[{
  "This package allows functions to be plotted on a microscopic scale to \
exhibit the granularity of machine arithmetic.  Alternatively the actual \
error (measured in ulps (",
  StyleBox["u",
    FontSlant->"Italic"],
  "nit in the ",
  StyleBox["l",
    FontSlant->"Italic"],
  "ast ",
  StyleBox["p",
    FontSlant->"Italic"],
  "lace, ",
  StyleBox["i.e.",
    FontSlant->"Italic"],
  ", one digit in the last place)) can be plotted."
}], "Text"],

Cell[BoxData[
    \(Needs["\<NumericalMath`Microscope`\>"]\)], "Input"],

Cell["Microscope[Sqrt[x], {x, 5, 20}, PlotJoined -> Real];", "Input"],

Cell["MicroscopicError[Sqrt[x], {x, 5, 20}, PlotJoined -> Real];", "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["OrderStar", "Subsubsection"],

Cell["\<\
This package plots the order star of an approximating function, to \
an essentially analytic function.  It is common to consider rational \
approximants to functions such as Pad\[EAcute] approximants. Various \
information about a numerical scheme (such as order and stability) may be \
ascertained from its order star. For example, Runge-Kutta methods may be \
considered as rational approximants to the exponential, where relative and \
absolute stability regions are considered in terms of the linear scalar test \
problem of Dahlquist. The zeros, poles, and interpolation points convey \
important additional information and may also be displayed.\
\>", "Text"],

Cell[BoxData[
    \(Needs["\<NumericalMath`OrderStar`\>"]\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[{\n
        \t\t{OrderStar[Pade[Exp[z], {z, 0, 1, 0}], E\^z], \n\t\t\t
          OrderStar[Pade[Exp[z], {z, 0, 1, 0}], 1], \n\t\t\t
          OrderStar[Pade[Exp[Cos[z] + I\ Cosh[z]], {z, 0, 2, 3}], \n\t\t\t\t
            Exp[Cos[z] + I\ Cosh[z]], \n\t\t\t\t
            PlotRange -> {{\(-5\), 5}, {\(-5\), 5}}, PlotPoints -> 40]}, \n
        \t\t{OrderStar[Pade[Log[1 + z], {z, 1, 3, 2}], Log[1 + z], \n\t\t\t\t
            OrderStarKind -> Second, \n\t\t\t\tPlotPoints -> 50, 
            PlotRange -> {{\(-0.5\), 2}, {\(-1\), 1}}, \n\t\t\t\t
            OrderStarSubPlots -> \n
              \t\t\t\t\t{{PlotRange -> {{0.9, 1.1}, {\(-0.1\), 0.1}}, 
                  PlotPoints -> {20, 20}}}], \n\t\t\t
          OrderStar[Pade[Exp[z], {z, 0, 2, 3}], Exp[z], \n\t\t\t\t
            OrderStarPoles -> {True, False}, \n\t\t\t\t
            OrderStarZeros -> {True, False}], \n\t\t\t
          OrderStar[Pade[Sinh[z - 1], {z, 0, 3, 3}], Sinh[z - 1], \n\t\t\t\t\t
            OrderStarLegend -> {{0.6, 0.6}, {0.98, 0.98}}, \n\t\t\t\t\t
            TextStyle -> {FontFamily -> "\<Times\>", FontSize -> 4}, \n
            \t\t\t\t\tOrderStarSymbolSize -> 0.02]}}]; \)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Statistics", "Subsection"],

Cell[CellGroupData[{

Cell["LinearRegression", "Subsubsection"],

Cell[TextData[{
  "This package provides least squares or weighted least squares linear \
regression for data whose errors are assumed to be normally and independently \
distributed.  It supports a number of commonly used statistics such as ",
  StyleBox["RSquared", "Input"],
  ", ",
  StyleBox["EstimatedVariance", "Input"],
  ", and an ",
  StyleBox["ANOVATable", "Input"],
  ".  It also provides diagnostics for leverage, influence, collinearity, and \
correlation."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Statistics`LinearRegression`\>"]\)], "Input"],

Cell[TextData[{
  "This example demonstrates how to generate a ",
  StyleBox["ParameterConfidenceRegion", "Input"],
  " of some data."
}], "Text"],

Cell[BoxData[
    \(\(data
       = {{0.055, 90}, {0.091, 97}, {0.138, 107}, \n
        \t\t\t\t\t\t\ \ {0.167, 124}, {0.182, 142}, {0.211, 150}, \n
        \t\t\t\t\t\t\ \ {0.232, 172}, {0.248, 189}, {0.284, 209}, \n
        \t\t\t\t\t\t\ \ {0.351, 253}}; \)\)], "Input"],

Cell[BoxData[
    \(regressdata = 
      Regress[data, {1, x\^2}, x, \n\t\t
        RegressionReport -> ParameterConfidenceRegion]\)], "Input"],

Cell[BoxData[
    \(\(Show[Graphics[ParameterConfidenceRegion /. regressdata], \n\t
      Axes -> True, AxesLabel -> {"\<Constant\>", \*"\"\<\!\(x\^2\)\>\""}]; 
    \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["MultiDescriptiveStatistics", "Subsubsection"],

Cell["\<\
This package computes descriptive statistics (location, dispersion, \
shape, and association statistics) for a sample represented as a data matrix. \
The data matrix is a list of independent identically distributed \
vector-valued or multivariate observations.\
\>", "Text"],

Cell[BoxData[
    \(Needs["\<Statistics`MultiDescriptiveStatistics`\>"]\)], "Input"],

Cell[TextData[{
  StyleBox["MultiDescriptiveStatistics", "Input"],
  " defines two new graphics functions, ",
  StyleBox["Ellipsoid", "Input"],
  " and ",
  StyleBox["Polytope", "Input"],
  ". These functions are used in other statistics functions, including ",
  StyleBox["EllipsoidQuartiles", "Input"],
  " and ",
  StyleBox["PolytopeQuartiles", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(data = 
      N[{{1232, 4175}, {1115, 6652}, {2205, 7612}, {1897, 10914}, {1932, 
            10850}, {1612, 7627}, {1598, 6954}, {1804, 8365}, {1752, 9469}, {
            2067, 6410}, {2365, 10327}, {1646, 7320}, {1579, 8196}, {1880, 
            9709}, {1773, 10370}, {1712, 7749}, {1932, 6818}, {1820, 9307}, {
            1900, 6457}, {2426, 10102}, {1558, 7414}, {1470, 7556}, {1858, 
            7833}, {1587, 8309}, {2208, 9559}, {1487, 6255}, {2206, 10723}, {
            2332, 5430}, {2540, 12090}, {2322, 10072}}]; \)\)], "Input",
  CellTags->"S6.4.1"],

Cell[BoxData[
    \({stiffness, strength} = Transpose[data]; 
    \n{{minx, maxx}, {miny, maxy}} = 
      Map[{Min[#1], Max[#1]}&, {stiffness, strength}]; \)], "Input",
  CellTags->"S6.4.1"],

Cell[BoxData[
    \(q1 = 
      Graphics[EllipsoidQuartiles[data], \n\t\tFrame \[Rule] True, 
        AspectRatio \[Rule] 1, \n\t\t
        PlotRange \[Rule] {{minx, maxx}, {miny, maxy}}, 
        FrameTicks \[Rule] {{1200, 1600, 2000, 2400}, Automatic}]; \n
    q2 = Graphics[PolytopeQuartiles[data], \n\t\tFrame \[Rule] True, 
        AspectRatio \[Rule] 1, \n\t\t
        PlotRange \[Rule] {{minx, maxx}, {miny, maxy}}, 
        FrameTicks \[Rule] {{1200, 1600, 2000, 2400}, Automatic}]; \n
    Show[GraphicsArray[{q1, q2}]]; \)], "Input",
  CellTags->"S6.4.1"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Utilities", "Subsection"],

Cell[CellGroupData[{

Cell["DXF", "Subsubsection"],

Cell[TextData[{
  "This package allows the export of ",
  StyleBox["Graphics3D", "Input"],
  " objects to a file in the AutoCAD DXF format."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Utilities`DXF`\>"]\)], "Input"],

Cell[BoxData[
    \(\(gr = 
      Plot3D[BesselJ[0, \@\(x\^2 + y\^2\)], {x, \(-6\), 6}, {y, \(-6\), 6}, 
        PlotPoints \[Rule] 20]; \)\)], "Input"],

Cell[BoxData[
    \(\(gr = Graphics3D[gr]; \)\)], "Input"],

Cell[BoxData[
    \(WriteDXF["\<bessel.dxf\>", gr]\)], "Input"],

Cell[TextData[{
  "Some of the graphics primitives that can be transferred are ",
  StyleBox["Cuboid", "Input"],
  ", ",
  StyleBox["Polygon", "Input"],
  ", ",
  StyleBox["Line", "Input"],
  ", and ",
  StyleBox["Point", "Input"],
  "."
}], "Text"]
}, Closed]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["5. Colors", "Section"],

Cell[CellGroupData[{

Cell["Built-in Color Functions", "Subsection"],

Cell[TextData[{
  "Colors can be specified in any of several color systems. ",
  StyleBox["RGBColor[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["r",
    FontSlant->"Italic"],
  ", ",
  StyleBox["g",
    FontSlant->"Italic"],
  ", ",
  StyleBox["b",
    FontSlant->"Italic"],
  StyleBox["],",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " where 0 ",
  StyleBox["\[LessEqual]",
    FontFamily->"Symbol"],
  " ",
  StyleBox["r, g, b",
    FontSlant->"Italic"],
  " ",
  StyleBox["\[LessEqual]",
    FontFamily->"Symbol"],
  " 1 specifies a color with the specified proportions of red, green, and \
blue. ",
  StyleBox["GrayLevel[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["level",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", where 0 ",
  StyleBox["\[LessEqual]",
    FontFamily->"Symbol"],
  " ",
  StyleBox["level",
    FontSlant->"Italic"],
  " ",
  StyleBox["\[LessEqual]",
    FontFamily->"Symbol"],
  " 1, specifies a shade of gray. ",
  StyleBox["CMYKColor[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["cyan",
    FontSlant->"Italic"],
  ", ",
  StyleBox["magenta",
    FontSlant->"Italic"],
  ", ",
  StyleBox["yellow",
    FontSlant->"Italic"],
  ", ",
  StyleBox["black",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " accepts arguments that are each between zero specifies colors in the CMYK \
color system. ",
  StyleBox["Hue[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["hue",
    FontSlant->"Italic"],
  ", ",
  StyleBox["saturation",
    FontSlant->"Italic"],
  ", ",
  StyleBox["brightness",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows color specification by hue, saturation and brightness levels, all \
of which must be between 0 and 1. The simplified form ",
  StyleBox["Hue[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["hue",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", which corresponds to ",
  StyleBox["Hue[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["h",
    FontSlant->"Italic"],
  ", 1, 1",
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", allows color specification with one argument. As ",
  StyleBox["h",
    FontSlant->"Italic"],
  " varies from 0 to 1, ",
  StyleBox["Hue[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["hue",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " runs through red, yellow, green, cyan, blue, magenta, and back to red \
again. Values of ",
  StyleBox["h",
    FontSlant->"Italic"],
  " outside this range are taken mod 1. "
}], "Text"],

Cell["\<\
Show[ Graphics[ 
{PointSize[.15], 
\tRGBColor[0,0,1],Point[{0,0}], 
\tCMYKColor[.2,.9,.6,.3], Point[{1,0}], 
\tGrayLevel[.7], Point[{2,0}],
\tHue[.5], Point[{3,0}],
\tHue[.5, .4, .6], Point[{4,0}]}],
\tPlotRange -> {{-1,5},Automatic}];\
\>", "Input"],

Cell[TextData[{
  "A color wheel can be created by mapping ",
  StyleBox["Hue",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " over the sectors of a disk."
}], "Text",
  CellTags->"Hue"],

Cell[BoxData[
    \(ColorWheel[n_] := 
      Show[Graphics[
          Map[{Hue[#/\((2  \[Pi] - n)\)], Disk[{0, 0}, 1, {#, # + n}]}&, 
            Range[0, 2  \[Pi] - n, n]], \ AspectRatio -> Automatic]]\)], 
  "Input",
  CellTags->"Hue"],

Cell[BoxData[
    \(\(wheel\  = \ ColorWheel[\[Pi]/128]; \)\)], "Input",
  CellTags->"Hue"],

Cell[TextData[{
  "Another way of specifying colors in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " is by using the Color Selector built into the ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " front end. The menu item can be found, on the Macintosh, in the ",
  StyleBox["Input", "MenuItem"],
  " menu. Manipulate the controls until the shade you want appears in the New \
box. Clicking the OK button pastes the correct ",
  StyleBox["RGBColor",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " specification at the current insertion point in the current notebook."
}], "Text"],

Cell[GraphicsData["PICT", "\<\
SCH000000Bl1W@0A0_l<0?ooool000000000006M0001;`000000000N01[ooooo
ool00@0:0000004_0Id0P@0:000000000000X02201X000000000VH6N0000004_
0Id00000000000180000B0000000200100P000000>LMO000000000ShP000_`00
oooooooo007>c^SXc/h00TU9BDU9B@03i^Kbl_;b00CBd];Bk>`01EfY^fnR>@06
GJZkKefY00L004A7XZ802000^[ZQ`P09GEfkD?oo00Zj^P00GMT02ooo@d=MO00<
00000jZZ00fj^P0000003^k^0000000?A4@00000013noD?l00404@01A03no@0B
OgmoOgmo01>RXZ:QGUh055iNGUnRXP0ERXX0000001HG5aLG5aL05j:RXYl0000H
XZ800::N01VR3oooGEd06UiK002RXP0KGEgooj8<01`00::RGUl07OooXPaMG@0N
GEeMH?oo01mMH?ooGEd08>7QhN7Qh@0QXj>SXj>S02800OkmXOh08`00000J6P0T
6QX0000002FR0_km00409UeMooooo`0Wo_fQoP0102Sno@01XO`0:J7h007no@0Z
oomMGOok02_oooonXJ40;6i^0000000]007noEgm02iMo@01o_d0;eh1o_d00@0`
ooooo5eM03400Y2@0000<P03/[80000c00?De00003@01?Kf0000=@01C4`0000f
009^KP0003L00003e=@0>00000CfmP0i00000X2003[no@01GO/0>i6ATI6AT@0l
:b/[:b/[03e^KVi^KVh0?[fm_Kfm_@0oG5aLG5aL04020Z^[j^X0@@01ZPCno@12
oomEEJZW04>ZZEEEool0A:VYZJ[oo`15T91aL3Xj04IV09SoIOl0AlC4a<C4a018
ZOmDojWm04VHV6Fh0000B/_n<`0c001;oonYZJVY04c<c000VIL0CFIVc<c<c01>
IP;;oS<004oDe=CDe=@0D?ooook;b`1A000c<lc<05;ooc<c<c<0Do;b20R4PP1D
7_Vfd1?j05EEEZZZ0000EYVI<c>IU`1GV>o<MYS^05RIVIVG<c<0FFIU<c>IV@1J
VI[<c6IV05/bfP00V9P0G9VLc<`0001M<c?ooiVG05iUIOooc7L0G`00V9PbgP1P
VILc<ooo066IVLc=ool0H_ooIFG<LP1SoonIUc<c06AVHc<cool0I@00c<aVI01V
IVS<c00006MVJooo<c<0J?ooc7AUI@1Y<c<c<c<c06[no@01<_/0J`00IKjHV01/
002HV9RH06dc<iVIVIT0KVGoV?nHo`1_VIUVIC<c072HofGoIOl0LOkmIOl00@1b
<`;;oS<007>IU000c<`0M9Sl<`3;oP1e<c<c=IVI07Hc<fIWVIT0MiT2bohc001h
VI_ooc<c07W<c6IVc<X0N/_nV?`c001kc<bIV6IV07c<M/aeV>h0OFGl<`3;oP1n
IVIVJ<c<07lbgFFkIK/0P3<cVIUVI@21007noC;m088c<000c<`0Pc;m007no@24
<cS<c00008Dc=_oo<c<0QS<coomVJ027ooo<c3<c08S<0Okm0040ROkmbod00@2:
o_gnn@0108^HV000<]T0S9RH001U]`2=001VJ<c<08h009VLc<`0S`01IP?no@2@
006Hookm094c09T2boh0TS<cVI_oo`2CIVJIV/c<09AUIIS`ool0UC<0bok;oP2F
<c?oolc<09Lc<oooool0V<afV>k<M@2IoonIVLc:09[;oVGl<`00Vlc<IVIVIP2L
oomVHc<c09goofEUIFD0WP00c<c<c02O007noL_m0:000L`1o_d0X@01o_gno@2R
IK/bgC;M0:=U^c;MIKX0Y9VI<c<c<`2UVITc<fIT0:Hc03<2boh0Yc<0IP;;oP2X
<c<c=_oo0:Tc<fIVool0Z/_nbo`c002[c<c<bVIV0:c<N_ooIFD0[C;PV9P0002^
<]iU^c;M0:lc=IVI<c<0/6FnV9P0002aIK]U^S;M0;9VIiVI<c<0/l_n<`3;nP2d
boL00Okm0;Gno@01bo/0]_km007nm`2gc7JHkYS^0;SoofEUV>T0^ISoIOnHoP2j
ZOmDoeCo0;_;oS<0IO/0_:ZZEE@0002mc<`c<0000;hbgC;NIK/0_`0000000000
0001;`6M0000004_0Id0000:0000004_0Id02X6oPKn1_nBo0001XWnoAdm7Adm7
Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7
Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7
Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Agm?AdM?AdM?AdM?AdM?AdM?AdM?AdM?
AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?
AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?
AdM?AdM?AdM?Agm7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7
CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7
CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7Cae7Adm7
Adm7Adm7Adm7Adm7Adm7Adm7Adm7AdlB_`01XWnoAchnAchnAchnAchnAchnAchn
AchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchn
AchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchn
AchnAchnAchnAchnAgln?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn
?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn
?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?Wm7
?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7
?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7
?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?QdnAchnAchnAchnAchnAchn
AchnAchnAchnAchB_`004@:oCcj14X4BPA;Z4P<n4[l001<3_dLn4X6oPKn1_nbo
14Ln4[l001D4_dln4[n10840P@3^00FoAchB_`005@BoAchB_h40P@210>h01Km7
?Q:o000E1;m7?Q:oP@210840kP05_dln4[l001D4_dLn4[n10840P@3^00FoAchB
_`005@BoCchB_h40P@210>h01Km7?Q:o000E1;m7?Q:oP@210840kP05_dLn4[l0
01D4_dln4[n10840P@3^00FoCchB_`005@BoAchB_h40P@210>h01Km7?Q:o000E
1;m7?Q:oP@210840kP05_dLn4[l001D4_dLn4[n10840P@3^00FoCchB_`005`Bo
CchB_oL0PKn1_h:om`05_dLn4[l001/4_dLn4[og002oP@2108@00;og00FoAchB
_`006`BoAchB_oL00;n10840Q000_oL01Km??Q:o000K1;m7?Q:om`00_h40P@24
002om`05_dLn4[l001/4_dln4[og002oP@2108@00;og00FoAchB_`006`BoAchB
_oL00;n10840Q000_oL01Km??Q:o000K1;m7?Q:om`00_h40P@24002om`05_dLn
4[l001/4_dln4[og002oP@2108@00;og00FoAchB_`006`BoAchB_oL00;n10840
Q000_oL01Km??Q:o000K1;m7?Q:om`00_h40P@24002om`05_dLn4[l001/4_dln
4[og002oP@2108@00;og00FoAchB_`007`BoAchB_oL00;n10840f`2d_oL00;og
00FoCchB_`008`BoAchB_oL00;n10840f`00_kHB0;og002om`05_dLn4[l002<4
_dln4[og002oP@210=/00;nf4P2om`00_oL01Km7?Q:o000Z1;m7?Q:om`00_md0
oKof006o_h40T000_kHB0;og002om`05_dln4[l002l4_dLn4[og002ogP01_kon
002om`01_kn109000;nf4P2om`00_oL01Km7?Q:o000k1;m??Q:om`00_mh00Kno
nP3m_oh00KnooP3m_oh01;no0;noP@2O002o]Q80_oL00;og00FoAchB_`00?`Bo
AchB_oL00;oN006o_o/04kno002o_`00_kl00;no002o_`00o[n109d00;nf4P2o
m`00_oL01Km??Q:o000o1;m7?Q:om`00_mh00Knon`0E_kl00;no002o_`00_kl0
0;no002o_h40W000_kHB0;og002om`05_dLn4[l003l4_dLn4[og002ogP01_kok
01Fo_`00_kl00;no002o_`00_kl00;noP@2L002o]Q80_oL00;og00FoAchB_`00
?`BoCchB_oL00;oN006o_o/05Kno002o_`00_kl00;no002o_`00_kn109`00;nf
4P2om`00_oL01Km??Q:o00121;m7?Q:om`00_mh00KnooP0H_`00_kl00;no002o
_`00_kl00;no002o_h40W000_kHB0;og002om`05_dLn4[l003L4_dLn4[og002o
g@3m_od0oKon006o_oh0oKon006o_h40W000_kHB0;og002om`05_dLn4[lB02<4
_dln4[og002oP@210=/00;nf4P2om`00_oL01Km??Q:o000S1;m7?Q:om`00_h40
P@3K002o]Q80_oL00;og00FoAchB_`00=0BoAchB_oL00;n109@0oKoh006o_oL0
0Knok`01_kog002o]Q80_oL00;og00FoAchB_a80;`BoCchB_oL00;n109D01Kno
002o_mT00Knom`00_kHB0;og002om`05_dln4[l004h4_dLn4[og002oP@2E00no
_`00_kl00;no0;no0;nooP3l_`D00;no003l_od0oKon00Fo_`00_kok002o]Q80
_oL00;og00FoAchB_`00N`BoAchB_oL00;n109D01kno002o_`00o[on01Ro_`00
_kl00;no002o_`00_kl00;no002ooP09_kl00;no002o_o/00;oj4P0ko180>o`B
03_l4P0ko180>o`B03_l4P0ko180>o`B03_^4P2om`00_oL01Km7?Q:o001H1;m?
?Q:om`00_h40U@09_kl00;no002o_od05Kno002o_`00_kl00;no002o_`00_kon
0?bo0`00_kog002o`a80>o`B03_k4P2om`00_oL01Km??Q:o001C1;m7?Q:om`00
_h40U@09_kl00;no002o_od08Kno002o_`00_kl00;no002o_`00_kl00;no002o
_`00_kog002o]Q80_oL00;og00FoAchB_`00EPBoAchB_oL00;nB006o_hH02Kno
002o_`00_kom026o_`00_kl00;no002o_`00_kl00;no002o_`00_kl00;nom`00
_kHB0;og002om`05_dLn4[l006@4_dln4[og002oWP3n_oh0o[l5002o002oQ`09
_kl00;no002o_od09Kno002o_`00_kl00;no002o_`00_kl00;no002o_`00_kl0
0;non`00_kHB0;og002om`05_dln4[lB05h4_dLn4[og002oW`00_oh00[l0_oh0
1Kl0_`00_hH0oKon006o_od00KnooP3l_`/00;no002o_`00_kon0?bo1`00_kl0
0;non`00_kHB0;og002om`05_dLn4[l00604_dLn4[og002oW`00_oh00[l0_oh0
1;l00;noP@3`006o_ml00;ol4P0ko180>o`B03_l4P0ko180>o`B03_l4P0ko180
>o`B03_h4P0kmQ80_oL00;og00FoAchB_`00?@BoAchB_oL00;nN0?fo0@2ooP00
_h40l000_oh00Knog`00_lLB03_f4P0koA80_oL00;og00FoCchB_`00;`BoCchB
_oL00;nK00:o0;on002oP@3_0?fogP00_kHB0;og002om`05_dLn4[l002`4_dLn
4[og002oW003_`00_oh00;n10<P00;nf4P2om`00_oL01Km7?Q:o000Z1;m??Q:o
m`00_ih00Knoo@3n_h40a`00_kHB0;og002om`05_dln4[l002<4_dLn4[og002o
P@210=/00;nf4P2om`00_oL01Km7?Q:o000O1;m7?Q:om`00_h40P@3K0;Bom`00
_oL01Km7?Q:o000S1;m7?Q:om`00_h40P@3K002o]Q80_oL00;og00FoCchB_a80
9@BoCchB_oL00;nO0?2oP@3>002o]Q80_oL00;og00FoAchB_a80=PBoAchB_oL0
0;nU0?^oo2l09Oh_1bD_9BlU9BlUnkn10=@00;nf4P2om`00_oL01Km7?Q:o000g
1;m7?Q:om`00_jX0o;oe;`Z8;gL_;bF8;bDU;o/Uo;n10=T00;nf4P2om`00_oL0
1Km??Q:o001:1;m??Q:om`00_jd0oKl6;bld;bmWEO/_1BDU;bmg;ohU1blU;bFZ
9BF8oRD3;hXU9OfoP@3L002o]Q80_oL00;og00FoAchB_`80D0BoAchB_oL00;n`
0?jo12ld;bldoBl0<_`_0RD_Moh_2RD_MblU;hP_9BEgoRD0R?`U0hXUMh[n_h40
g`00_kHB0;og002om`05_dLn4[l005P4_dLn4[og002o/P07_kld;c<_;cCn;`1D
oBl1<gOl;`ag;fLU9BlU;gLU;bEgoRD?R2Eg9BF89HXU9HXU9HZo_h40h@00_kHB
0;og002om`05_dln4[l005h4_dln4[og002o]@3n_`Xd;c@_=2l_=2l_<_h_0FN]
oBl@Ibl_Mbl_MblU;jX_9HQg;hSn9@ig9HR:;hYg9HXURRDURX[n_h40i000_kHB
0;og002om`05_dLn4[l206@4_dLn4[og002o]`0B_kld=2ld<blc;bld;c8_;cAW
EOh_0CAgoBlDMbl_MbD_;gLU;bEg9BlU9GLU9HYgoRD<RREgRRF:9HZ:9HZo_h40
iP00_kHB0;og002om`05_dLn4[l206T4_dLn4[og002o^@0__kld=2ld=2ld;c@_
<fLc;bmD;bmW[Bl_Ibl_EFL_;gL_;hQg;bn8MbD_R7LUR2EgoRDA9bDUMhXURRF:
MhZ:9HZ:9KnoP@3X002o]Q80_oL00;og00FoCchB_`00L`BoAchB_oL00;nk00:o
_bon=1Dc;c@cIc<_=2mD;blcIbn];fL_;eEWoRl?Mbl_Mbmg;hP_MbD_MbDUMohU
4JXU9HYg9BF:MhXU9bF:9HZ:9Oj:0KnoT000_o`00;oR002o]Q80_oL00;og00Fo
AchB_`00M0BoCchB_oL00;nl006o_odd;Bld=2ld=6Lc;e@_=6N];bmW<bl_EFL_
IbmEIbD_Mbmg;gL_9GL_R7LU9GLU9GOn9@f:MbF:9GN:RRF:9F>:9Of:0KnoT@01
_kom002ohP00_kHB0;og002om`05_dLn4[lB08X4_dLn4[og002o_P01_kol=01W
oS@/;cAD;cAW=2mD;cAW<Rl_EFL_IeD_;gLU;gL_;gL_MbD_ZRD_MbDUZREg9HZZ
oRD<HbDURWN:9HZ:MhZ:9Ob:0KnoT`3n_oh00;on0?fooP0=_kl00;no002o_`00
_koj002o]Q80_oL00;og00FoCchB_kl0T@BoCchB_oL00;o0006o_oXd@fLd<blc
=6Lc;e@dIc@_Ie@_;fMD;bmg<bmg;bmg;gL_;jX_MbD_MbEg9GLUMbDUMhX_RWLU
RWN:9HYgRXXURRN:9HX`oXX29knoU@3m_aX00;l00;no002o_`00_kl00;no002o
_`00_koj002o]Q80_oL00;og00FoAchB_kl0T`BoAchB_oL00;o100:o_bgm=01W
oSA0<bld=6Lc;cB_;c<_[blc7bmDIbmEIbmWEBmg;bnZ;gL_Mbmg9Bmg9JXUMbDU
MhYg9BEgRWN:9HXUHbF:9HYSRRGmRP@WRXZo_iH00Kl0o[lE0;l00;no002o_`00
_kl00;no002o_oH00;nf4P2om`00_oL01Km7?Q:o0P2A1;m7?Q:om`00_l800kld
=2gm=48]=3<_=3=W=5@d;eAW<bmDIc<_Ie@_;gMD;fMEIbmg;gL_Mbmg;gL_R7Mg
9GLUMhPUMbDUHbDUMhXUMhZ:MhZ:9HYgoHX79hXURV>:R[nG00:o003m_`400?^o
2`00_kl00;no002o_oH00;nf4P2om`00_oL01Km??Q:o>`2G1;m7?Q:om`00_l@0
2[no=3@]=3AW<c@]oSA5E3AW<bldE6Lc;e@cIbmDIblb;fMEIfME;gL_;gMWEGL_
9GL_MbEg;gLUMbEgRWLU9GN:MbF:MhZ:MhXUHhXURV>:9HZ:9oj:0bN:_knI002o
oP3n_`<00;non`09_kl00;no002o_oH00;nf4P2om`00_oL01Km7?Q:oJ@2P1;m?
?Q:om`00_lD0Bkno=2dd=2dd<cAW<c@dIcAD=3AW=:ld;jl_=6MD;fMD;fMEIbmg
E2mWMbmg;eEgIbFZ;gLUMbmg9GLG9GN:MbEgRREgRWN:RWN:RWOnRPegRXXWRXYg
RRN:RRNo_iX00;om00Fo_`00_kon00^o002o_`00_kl00;on006o_oX00;nf4P2o
m`00_oL01Km7?Q:o4P2F1;m7?Q:om`00_lH0G[lcIc@]=3@]=3@]E3@cIc@dE3AW
=3ADIc<_[bmDIc=WEBlOEBlOEBmgIeD_Mbmg;gMW5fMg9GN8MbEgZRDUMbES9BEg
RWN:MhZ:MhZ:MhZ:MhZ:9hYgRXXWRRN:9hZoV`00_o`00;on0?fooP3i_od00Kno
nP00_kHB0;og002om`05_dln4[lB08D4_dLn4[og002oa`1P_c@]=2dc;C<d;CAW
<cAD;C@cIc=D=6LcE2mWE3AWE6N]IbmO;fMEIbmg[bmg;gL_MfMg;aMgIbEgMbEg
9GLGRWLUMhYg9F<URWN:MhZ:MhZ:MhZ:HhXU9hZ:HhXWRRN:_h40mP00_kHB0;og
002om`05_dLn4[lo08L4_dln4[og002ob01R_bdd;C@]=2dd[c<]=3@b;C=W=3AD
=6LdE6Ld[bmDIc=WE6MEIblbMbl6;gMWEFMg9EEg;gMWMbEg9A<UMbEgMbEgRWN:
MbDG<2DW9C2:5hX`9BN:RWN:9hYgRRN:9hXWRRNoP@3g002o]Q80_oL00;og00Fo
AchB_cl0R@BoAchB_oL00;o906Bo;C@]=2dc;C<]<S@];bda=6LcE3AW<jldIe@d
Ie@dIeDOE2mW[bmWEFN]IfMgEFMg;gMg;gMg;jYg9JYg9GMg9BMWMhYgRWN:MhYg
RWN:9bD`RRLUHhZ:HbF:9hYSRRLGRRN:_h40n000_kHB0;og002om`05_dln4[lg
08/4_dLn4[og002obP1V_c@];C<]<bdd;C8]=1ld<beD<ald=:ld;a`c;e@O<eAW
;jl_1S=gIeD_1RlOEGL_IgMEMbmgIgMg9GMg9GMg9GMg9F=g9GN:MhYgRQL`9BL`
9F>:RWN:9hYg9hZ:HaN:9hXW9hXW_h40n@00_kHB0;og002om`05_dLn4[l208d4
_dln4[og002ob`0e_bdd;Bdd;Bdb;C@]E2dcIcAD;C@d7c4dIe@d;EEW<g8_[fMD
;jmWE6L_1Rmg[bmgIeEWMgL_oWL0;oig:bEgMbDG[1MgRWMgRWN:MhYg<2LU<2LU
RV>:MhXWRWLWRV>:5hXW9hYS9hZoP@3j002o]Q80_oL00;og00FoAchB_`00SPBo
AchB_oL00;o<03Ro;C<];bd]=1`]<ald;C4];bdb;CAE;C@aIc@b;Bn_=6N_;eAW
<QmE;`H_1RlOEFMg;jmgMbmJ5bonM`0_oGLYRWMW9gLUMc1g9BMgRQTWRQTWRQTW
RWLWRQTW9hYg9hXWMbN:9bN:9bNoP@3k002o]Q80_oL00;og00FoCchB_`00S`Bo
AchB_oL00;o=00>o;BddoRdb<Rd]<bf_;C<]<VLc;HAW<bdb=6MD=:mW<jmW<jmW
[bmWE6N]7eEW1Rmg[fMg;gMgEJYWoGL3;gMgR_eg7XYgMhXG6BLIRWLW9C0W9F>:
MhYSRWLWRV<U9bN:HbGn9`2:oRL0_h40o000_kHB0;og002om`05_dLn4[l20904
_dLn4[og002ocP03_bd]<oh]<jl];C@][bdc;H@]IfDcIe@]=5@O=:ld[bl]E2n_
E6MDIeD]Mbl6;`H_MjmWMgMEIgMgIeGnM`0Uo7L0R_ig8BES9GL`5c0UHbES9F>:
MhYS9F>:9gN:HbLUHhXW9gLW9h[m9`2oP@3m002o]Q80_oL00;og00FoAchB_`00
TPBoCchB_oL00;o?002oo2d0<oh]<jd];H@]=1l]Q2f];BlL=6MD;E@dIRd_E2eE
7eAW[FN_Ijl_7jdG[fMgIeEWM`Ig;gMgEAWmMbHGMbEJ9F=WHbmS9ATW9F<U<1N:
Mf>:MhYS9F>:HbDWHhXGHhXW9gOn9`29oRL0_h40oP00_kHB0;og002om`05_dln
4[l00904_dLn4[og002od005_bd]<Rf1oRd0[Oh]<c8]Ibf_;C@L7cB_;CAW73=W
[c>_;a`cIjl_[fMD1Rn0;`H_1VMEM`H_M`H_4bmgMeYEMbGnM`<IMgLWoWLKRUZ:
MaL`5c0UHbESRQ>:9gN:7AN:9gMSRRLWMohW07Oj9`2oP@01002d_oL00;og00Fo
AchB_`80SPBoCchB_oL00;oA006o_o`]<h4]71l]<bdLIbdb;FLL=1lL=:l];a`d
7eADIc<L;jmW<`IWE:mWIeF0IeDOEEYW/5YEMfOmMc2_6JYgMbEg5eXU4bMWHaTU
HbEgHhYg<1N:7BESRQ<WRQ<WRV<GRV<G9hN:9aN79bN9oRL1_kn10:T00;og00Fo
AchB_`80R`BoAchB_oL00;oA00No;B8];H4][od]<ED];A`O<bdL7cB_;C8]Ia`d
Ia`dIjd]<VN_;a`_[bn_7eDO/1mEM`H_1WLOEGLC;eYEoWL1FRGmM`4G<?ig82Mg
Mc0GRQUS9F=gRQdU9a>:9a>:HaN:HbMg9bMg9bMS5oXW0;n10:T00;og00FoCchB
_`00S@BoAchB_oL00;oB002ooRd88Rdb;B8c;Bd6oRd^1Rd]<ReW72dbN2db[bdb
Ic<LIeB_=1n_IeB_IjmWEAn_IeEJIeFbMfMg1Rm_Ia?iMbXU4gMgRQ=g9aUgRQ>:
4bLU7BDM9F<WMhYSMbMS9F<W9gPW9f<G9bN79bMLoBL0_h40ZP00_oL01Km7?Q:o
002@1;m7?Q:om`00_m<0?kl]8Rf1Ib8];A`]7h4][bf1;K8];H@];a`]EBeD;FMD
1RdbIc8]E6ME77L]EFN_7c9gIjmg[gN6EA<_/Q<_Mk;mM`4C9Oig01gnMb<G<7LU
Ha<U7BDMRWLG<2MgHbESHbES9gLWHaN:HbLGHbLW5f?j9`2oP@2[002om`05_dLn
4[lB0944_dln4[og002oe004_kl];B;n;C@LPIHc;AlL;BdO72dO72f_Ibf_;JmW
;C8]EBeE;K9WE1`_[an_IeAgE1l6[Amg[gMg[fMg1_ig0JlInWLU5a=gMbMg6BMg
6BMg<1N:4hYSMbLU7AMSRQ<W9gLWQgLW9f<GRF?n9`4GHo`W0KnoP@2/002om`05
_dln4[lk09L4_dLn4[og002oe002_bdRoRdg8X4][b8];A`OPBdLIbf_PAaW;Jl]
[AaW;@Hd[beDIaaDI`Hd1S9W[h0_1QmE1VN_MgN_IeYEM`KnM`5JEOig:7XI5gLI
Mf<I4bDMMaL`MaL`9gLM9F=g9a>:HgLW9gQS9gLWHaMS9aMSoRL1Ma3n9`=a9bNo
P@2/002om`05_dLn4[l009L4_dLn4[og002oe@11_bdR;H4R;BdL6c@]K6L]PA`O
;A`O;A`O;A`O<Rfb;JmW;Jn_;BlLEBeE7jn_IaaWEJmW1UDOM`IEFZmWMgL6MgL6
oGL04oig0a>ZMbOnMb4`5aeg<1MgHhYJ9gMSMbMgHhYSMbLG7BMS5f<G9f<W9aMS
oRL1Hkgm9`2oP@2]002om`05_dLn4[lB09H4_dln4[og002oe@08_bdR;BdR;C:F
oRdh21l];AaW;AaW;AaW76L][bd6;Jd6QC8O1S@LIjlb7jlb7`IEIaag[Rl6Mjig
1VMg[gMH1WMg4gL6o7L07Oeg9QdUAC0G<7LM9F=gHgLWMadUHaLM5f<WMhLGHaMS
9bM59hY:9hZmW?hW0a0W9kn10:d00;og00FoCchB_`80VPBoAchB_oL00;oF04:o
;B8]8Rd]8QaW8RdL7h4R71l]Gal]71lL;AlLIa`O76LLE1lb;K9D7jn_Ia`_77N_
IjlO[gN0EAnbMgL6MjmgFWL6oGLbAGMg6DEgMc0G4gL`5a<W6BMgMf<`5gMS9gMS
MbLM5f<GHf<GQbLGHbMS9aMS9a0G9f<@oRL14;n10:h00;og00FoAchB_`00W@Bo
AchB_oL00;oG04Jo_b8]8QlR72d]URdL;B8O72d]1Bd]1ReO7bf_;A`O[bdO[bf_
7jl]1S5W76N_[fN_7jmg77N_I`Ig[gN0M`IgMk9gM`Ig4oig1@HW6GMgHoig;F<I
Mf=g7AL`5aeg9gMSMf<GJ2MgHaMS9flWHbMh46<G9f<WLAMS9a0W9a0W_kn10:l0
0;og00FoAchB_`00WPBoCchB_oL00;oG07no8RdR;BdR;FaW;B8L7b8]GalR71n1
GalL7el]7a`OGalL1Rd6[fLLIab_Ic8LI`J_IjlOE@H6MjlO4eDC1WL6M`IgMjmg
4gMg4gMg4gLCMbMgMf<IMf<U7GLWMf=gHgLW4bLC9f=g9f=59gLWHbM_9bMS_2MS
9bNmHbLW42Nm9`2oP@2_002om`05_dln4[lk0:04_dLn4[og002of01o_b8]8Rd]
UQ`R;BfF72dR7a`]8PH];@D]71l]1Rd6;Jl]1Rfb;Jnb;@Hb7jn_N1aW[jlO1UFb
P6N_Mjl6M`Ig[gN0M`IgMeYgAGMg4eYgMaegMf=J5j`W4aL`Mf=gHaLCJ1LMG1dG
RA=S5f=S5f>lHf<G9f<@5f<@9bM_9a0W42L242NoP@2`002om`05_dLn4[l20:04
_dLn4[og002of01o_b8]8RdR;F`]URdLUX4]1R8]1R:11Re/IbeO7faW76LL7a`6
;@J_7emW77LL[fLb/ZlO1ZmW1WMOMdIg1PIg1WL6MeP6KaUgAGLI5a=g4gLWFWLM
MgLMMf=gHgLC9aTW4bLG7ALM5bLMABLGHf<GHdXW5f<WBRLWHkdW9a1S_BL29a2o
P@2`002om`05_dLn4[lB0:84_dLn4[og002of@1o_b8R728]URdR;AbF;B8L6h4L
8QlL8QmO7f`]7jlR1Rd678DL7jdOG`HO7:mW76N_IGM6<Al6EJl6I`H6MjdCP7Nb
M`IL4`Ig4gLCF:]g4bDCMaegMbMg4gMSMf=gHeXW4f<G4f>9MdDM9a<WAF<WHaMS
_F<WKbLWBRLW_6<@9a049a0W4;n10;400;og00FoCchB_`80X@BoCchB_oL00;oI
03>o8RdR8RdRK2fFPAbF;Bd5;B86PB86;@PO;@H]1Bfb71lL7a`67elO[`Db7aaL
7:lO[`IWoZm=7`IgP0IEFPHC1WN0M`HCMa<UAQ=hKaUSAWLMMgLMMa=SMgMS4gMS
Mf<G]bLG7DDW5adG9ae59f<GBRMS5f>mHbL@KbLW46>m9a0W476oP@2a002om`05
_dLn4[l20:@4_dLn4[og002ofP01_bgn8Wl]UQ`R;B8R1R8]8X0R;K8R;@D]1Rd8
Ia`OK1lLI`PO7:lO71n_1Q`O1ZlO[aag77LLMaagP;9W4Zd64`Ig/WLmMeP6Ma<C
Ma=g4gLC4gPMMa=gHgLCHeYL7A<GHa=SMdDW4bLC9dDM9aMSBQMSHdTW2aLWHkeS
42M9Ha1a9a1a9`4@_h40/P00_oL01Km7?Q:o002S1;m7?Q:om`00_mX00klR8Rgn
8WlL8QlR22fF;FaW8Qb68Q`571m/;@HR1Rd6Q@QW70H]1VeD7el67el61ZlO1ZlO
[`Ig[`H6Mk86M`J_FZlCMa9gM`IjMc/GFQ=g4a=g7GLCN6<CMbLC5afZ5kLWMdES
5ae5HbLC9dES_GMS9dDW46=9Hf>l9dXW42M:9a0W474@_h40/P00_oL01Km??Q:o
002T1;m??Q:om`00_mX00;on8Wl]UPP]8RfFK2dR7`P]6a`R1R8]1Bd68[8]KBfb
;@DL7jlKG`H6;DJ_1Rf_[an_[an_1[86[gM6[`Ig1X1g1UY_1WLC1Q<CMa<64gLM
MjXCMa<WMa<MMflI9a=JHaLCHdESMf=5HbM5HdDWHdDWHdD@HaMS_BM:9f>m42M:
9a1a9`8@LKn10;800;og00FoAchB_cL0YPBoAchB_oL00;oK00Bo8R:O;OhRO`PO
8R8LURdR1R8R1R8L6bfb8PHR71l5;@HL6el67a`5Iel61RfbP0H]4ZlO[`J_1VN_
?al64jlC1WL64UDC4`IH4gLBMa<U4dDI4f<6MadC9gLCH`ISMflM5g`WAF=gBWMS
9dESABMSABM:HbNlHdDW44XWHkdWBRL@LBNm475S0;n10;<00;og00FoAchB_fT0
Y@BoAchB_oL00;oK04^o8R:F;B8RSRfF;IHL8R8OSR8OK2d5;A/L8X0R1Rd5/Re^
;@J_UQag280OG`H67`HL[an_1Zl61X06[gL6/PHC1Zmg1Q<64a<6Ma8UoQ<5Ma<C
5chGoQ<`7A=gHeY_He`B9ae_Mf=5HdES5f=:HgNmHflWHdXWAA1S_2M:9f>m411S
4;dWLA0@_h40/`00_oL01Km??Q:o002X1;m??Q:om`00_m`00klR8Yon8Wl]UV`]
UR8LURe/URd5;B868PHR1R868PH]TA`K[am/1QmO1JlO1Vag70HO[`F]AQ`C[`J_
1X0C[eX6/Q<6McmL1Q=J4WM_FWLC4dDINWLM4jXG]aLC7FlCMf=_J6mS5kLGBV<G
7DES_6=_Ha15Ha159g5S_6=:9keSLA1aBRL@40=aLA2oP@2d002om`05_dLn4[l0
0:D4_dLn4[og002og001_`Sm8SFF;B88URfFK2dR1BdRKBfF71/L6abF70H571^_
7`PO[iJ]T`H]4Rd61Jn_7jl6/PH677Nb1[;n1P`C[`HC[`HC4`IJKa<6oA<j9KXI
AELWMa9S1RLC4f=gAAe5M`/G4f=5HkXG7DXWAF=59dYSAG559f>mHkeS_BM:474W
BQ0@LA1a4;n10;@00;og00FoAchB_`80Y`BoAchB_oL00;oL00:o8Yom8Wl8URdR
21/R;@PK8QbF1R8R1R868PHR1V`O25X]1ElO1A`66jn_Tbdo7jl5[al6[`H6[h06
P;9J1Smg1PHC?eX64fmJ1Q<CZQ<6Na=S1Qeg>aNg5a<CQa8W4dDCHdDWAA=SAF=5
9dXC9kXW^RM:HaM:9dYS4;aS44XW_A1S_G5a40=a44ZoP@2d002om`05_dln4[l0
0:T4_dln4[og002og@0:_iHR8PPR8YlR8Y?n8Rd68YHL8Q/8;I<R7f`K71/LKBd5
;@D68TJ0K1l6KA`O[`DLFQaJ7;:01[86P0JboPH0AOh6BEXmFVl6Ma9JKe`kFTDC
4eXG]`ISFTDCH`HW4a=_9a>jHgM5HdFM5kYSAAflHdES_6=:HkaS_F=:9dT;LKeS
4;dWBQ0@LKd@LA2oP@2e002om`05_dLn4[l00:T4_dLn4[og002og@1C_b:O8R:Q
8V`RURd8URfFK28O22:CPIHL6`POK1/L1Am/7f`66a`51Qm/1PJF[`Jb1JlO4Q`C
70J_1PJ0/UX21UZ_1Q<64WLoFQ<64a<BMc/G4c]goQ<e9c/G4V<CAA=S4WYS4RM5
4idGVbM5HdD@4g6jHkaSBV>lBRM:HkdWBQ1S_G5:9`hW_A1aJRNoP@2e002om`05
_dLn4[lB0:T4_dLn4[og002og@00_ohR<ilRUR:>8R:F79HRUQbF79JF79HL6abF
21l58PJ>7`DL6aaG76eWK805[al5718O[`D6/X3n1U6bP186[eY6?gLm1UP64a9g
>gLB4gLB9eY_ZQ=J^Q<CHa8G>bLCAF=54dESAF=57DXCBF>j9dD;_6=:9dE:Ha15
4;eSBQ2mQa2m44XW3RL>LA2oP@2e002om`05_dln4[lB0:`4_dln4[og002ogP04
_klR8Z7n8Wl8UR8RTb8R21/]UPPR1Bd58Y4]1B:08PEO6h0R1H1/7`H575LL1An^
1@J01Jm][eXL4jm]1Q>_1GLo1PHC4ZlC4U`B4`HCAATC>a=5FQd6Ha8C9ceg4`]5
4bLB5kYSAF<B9kY5HkbgLDES_0^lHdYS_4YS_F>mBRNmH`iS_G6m40H@BQ0@3[no
P@2f002om`05_dLn4[l00:/4_dLn4[og002ogP0;_b8RXB8R29LR8Yl5oR8XTb:F
22:C;A]/6`QWSQ]/1R:C;@D68PD61BdB8TH6K@J_UZl6KX061PGn1QDo7cl6/X0B
1PI_1S/64a8CCQ<C4Q=5oQ<j>aNgKa<I4Qe_KRLCVaLkHdFjHdESAKYSAF>j_6>j
BF=:Hk`;_6=:_F<@B[eS44X@_BMZLKd@LG4>4;n10;H00;og00FoAchB_`00[0Bo
AchB_oL00;oN07no8Z4R8YHRSR:F8XhRUR858R:C8V`OUPPRAYH81YHL6`DL6fdL
6jm]KRdk20HO52dBTjl51Zm61Jl6[dH64Zl6FSnbAVjb4`HkKeXBMc/C4c]gVUXC
4kXC4fmj4kXCVae54dES^Q=9]kdCHdE:HdE:Hfl@AF=:BRM:AG5:9dX@BPhWBW5a
BQ2m46XW3Q0@LKn10;H00;og00FoCchB_cL0ZPBoCchB_oL00;oN00^o8Z4R8YJ>
8Z4R8PGn8T9]WiH88PDR6abF29JbSYJbUQ`9[b9JK0H57`PB;Ffb4R861Jl6KPJ0
4PJ01@H6?alB/PHm4`HB1Q<m4a9JAGXk4gLBoA<hAA8M4U`B7DDC?f>jHfm^9dFj
HdFjHdD@4dY:HdEaA@]aAA2j44XWBW5:LDX@_F>m3XL>4;ea3Q2oP@2f002om`05
_dLn4[l00:`4_dLn4[og002ogP1o_b:Q8R:Q8PRF8YJ>UR:>UR:C8R:C8PDRSQ/L
UPFF[iJ_UQ@R1Y<R1Ve6224L1EXL1H2C1ZnCCX1^1PHm[ceJ73^0A@HBFPHm4ceJ
4WLB4c]jAEXCVQ<kAA<C^Q>j4bLBAF>jAA>MKdES^TESBTESBTD@^V=:BF=:BRM:
BRM:_A0>2keaBQ2m44Ya3Q1a3W6oP@2f002om`05_dLn4[l00:h4_dLn4[og002o
g`1o_b:Q8T4R8Z4RUXjF8YH88R858YH86`PRTb858PDK79JC21l88F`6U@H8FPQJ
28051Vd61@I6K@H61K8B1Jl6FSl64PHm1UXoFSeH4Q<B4a<BF[XCF[YJVQ8G>jFZ
4dFj4kYSAKYSKa9SBTFjHdE:HkY9HdY:AF=:_F=:44E:4;d@WKd@BW5:40hWJ[ea
3Q0@3[n10;L00;og00FoCchB_c/0[@BoCchB_oL00;oO02:oXB8RXB:Q29HRUPRF
8YL8UR:>UXiMSYJF71^>1IH6UV`5;Oh5JRd5KDH88A`5Ea`51Vj_1@H6KDHBP0D6
XeZ_>jm^4Vl6?EXoFQ86]`Ig^UXB4fmJ4V=o4j]_>flW4TDCBQ>j9ceS^TE:4dYS
^TEa^VmSBTYS3F=:B@]92keaBTYaBQ0@BW4>BQ1a3Q2m3Q2oP@2g002om`05_dLn
4[l20:h4_dLn4[og002og`1o_j4RXB:Q8YI/XB:FSYHRUXjFK9HR1B858PDR6hj0
8PDKSVd5;I451Bd54R851PD5[Vd6T`I]1PD64[:A/S^01Q:_4PHk[eXBAA864Q<B
KeXB^UXCAC^jF[XC^Q=_>kYS4TFj7@9S^Q=:AKXWBQ>m2`9S_KYSBTZlWA15BW5:
44X@44Y:LDYa3Q0@JQ2mL@ia3[n10;L00;og00FoAchB_`00[PBoAchB_oL00;oO
07noXB:Q8Z4RUXjF8Z5/UR:FK9H8UYJ>6f`KSYH58PF>G@J>6i<L6ffCK5Z>1VdK
4Xh61Jl51PDo>h051Q861K8o1[ToFPHD1Q8C?A9J4TEJ4Q<B4i/kAA<BAKM5AKM_
7Cm5W1=5^TD;AKYS^TFM_;YSBTE:BV>l2dZlBV=:_DYaBQ1aBQ1Z_A1:40i:L@jm
40h>_h40]`00_oL01Km??Q:o002^1;m7?Q:om`00_ml0Okn>XB:G8Z4R@IHRUT4R
UXjF8YH86fbFUPPKK94R1B851B862F`6UPD575M/1H2C1Jl51Y=]APF_1PFi1QB0
4PF_FSdB[a8m1Q<B4UY5>a8C>iXC4Q<BVQ=5AKM5AB6aVQ>j4dZj4kY:4dY5^V=:
BQ=:B@^jBW55BQ2jLDZm2ke:411:LDX>LDX>LDX>4;d>L@joP@2g002om`05_dLn
4[lB0:d4_dln4[og002og`18_j4RXB:QUXjQ8YJ>UR91UR:FSYH58YH8TiH8UPFN
1F`Q8PD8EfbC7hh51VeNL0P6Tfe61@F0>i>b1@Hm1Q@6^@HD1Q964Q864ohBA4EJ
4S]54a:j>a=5>kXC^S]54e9^A@]54dZJFTZjA@]52dE:AG6jWKa:2dU:BV<=2ke:
LDY:44X@3TX@3TYa3[d>40iZ_h40]`00_oL01Km7?Q:o002_1;m7?Q:om`00_n00
BKnQUZ4RXJ4RUT4RXIJ>UYJ>UXjF8YH8UY>FK1/86fbAUPD86i<8K@JF1F`BUPH5
1Fd6KPF01@I]4PF04PD6?`DB1Q864PHBK`E5oQ95FQ:J4a8C>dDBAKM5>kZj4dDk
BA8;AA:LADD;KkZMADY5WDE:^[a:HdX:9dY:_@]:BW5:_Kd;40i:46X@40i:3W4>
3Q0>_h40^000_oL01Km??Q:o002_1;m??Q:om`00_n00M[nQ8Z6QUZ48XIJ>UZ4R
SYJFSYHRUhjF1IH8UY>>UPDRT`DRT@DRTfd5SUX81C/8APE]1@H51J=G21<5?i6b
4X0kKA8o1Q864Q95>a9J4Q95>c]5>dDC4[XBAKM5^Q>JYKXCKi^UA@]5^TXCBVm:
AF=:B[Y:BTD@oTXHLKY:LDX>BW5:BPiaBPjmJQ0>_A1Z3W4>_h40^000_oL01Km7
?Q:o=`2^1;m7?Q:om`00_n00A;nQUZ518Z6FXII18Z6F2:6FSYJFSQ^@6fbFUXh5
UXh5UVdR59H6UPE/E`QGSPFCK@H5Tfe6KA851Y<o1G061C/B1SdB1_hB003m4TEJ
^Q8BF[XB4YXC4Y/B_;LmVa>j^YXCBQ9:4dY_A@^jVdT;ADY:^[d;^PeSBTX=HfY:
_6X@B[e:3TYaJ[d>LFZm3Pia3PjoP@2h002om`05_dLn4[l20:l4_dLn4[og002o
h013_j6Q8Z6QUT2Q8YN>XB898T6F8YJ>UPTR1B:@6fe06f`91Z45KF`K1I5^25M/
1@H91PE]APDD1H1^KA:0>fdB1Q@B1@Kn4PL60;lBAA8B4ohB8[YJ^Q8B^Q=_4[Zj
4kZj4TXC^VnjVdXCBVnjBVl;BTVj2dY9oTX12kgnBQFmBQ1ZB[e:3Q1Z_@i:40h>
BPh>40joP@2h002om`05_dln4[l00:h4_dLn4[og002oh014_j50XJ4RXJ6F@:4R
9PRQUT2FUXhR2HjFUXjF1B85X@H9SQ]/TiH5SSh8Ed4B;Fd5KI<k2;T51PDB2JmG
?A:D[c^i1Q8moQ8H0;lB4TDk4S]54S^j?a=F4kXkKkXB^[Y_4ofj3DD;^TE:^[Y:
BTD;^TY9oTX2WDZmoTXB_@^mBPi:_FX@BPi:3Q0>BPh>4?h>0;n10;P00;og00Fo
AchB_`00]`BoCchB_oL00;oU006o_oh00;onXD0RXJ6F@:6FXHjFXIJ>UZ45X@RF
UXjFTd0KUXh5UPFF59H5STd6UQBF4PRCE`PBT`D5P0EI1@H54PDB4PDo4PF0L?hB
5P2o1Q8B^S/BAC/B^S]54[XB^[XC^[XBo;XZA@]_^[ZlBTRm^TY52dY:_4Y:2dX=
L10jBDYZBQ1Z40fM3[eZ_@h>BPh>4?h>0;oh006o_h40``00_oL01Km7?Q:o0P30
1;m7?Q:om`00_oH00;om0?jooP3n_`X00;l00;l00;nQU_jQ?HjQUZ50XII0UZ6F
@289UY2FSYJF20FF2HjF1HjFTA^>Ed2i8QBF1@DDUPD61Fd5L9454V`B1KTB1A86
4Q85oA810;ol4Q154S^j4Q9_4TFj4[XmB[NjAObj2fnK^TZjADY:^TZLAOQ:0;gn
BPl>B[eZ_6X@BPi:3PjmJPjmo@h0_oh0o[l5002o002oP@34002om`05_dln4[l0
0<l4_dLn4[og002om`01_kon002ooP02_`2ooP08_`2o002o002oo:5HUT2QXII0
XII0UZ489YI0UXjFUY05UY0K@0T6Uf`5UPE]@EL81@T6K@D5Tfd51I454Y45AQ@5
>`Dm1A:0^A8D>a854[lBKa8k4Q:j4Q:j4Y/B^Q:j^Q9:4TGk^P1:o[X@B[Y:^TY:
^TY:_0/:_DY:W0gnBPd@JPd;_FZmJQ0>BPh>B_h>0;gn3P>o002ooP05_`2o002o
V`3>_md00;og00FoAchB_`00``BoAchB_oL00;of002ooP00_oh00[l0_oh01;l0
0;nooP02_j50oJ4jUZ50XIJQ@9JG@9I0UZ45XIJ>HFfF@1/8TQ/82@F>UY<5UQBF
59D5Tfd5KXhkKY45L945>fdBTA:A4Vkl0?hBo00<4[XB4[XB^Q:j^Q:j?OFj1TZj
B[Y:B[[nBQVj_H^M_6Zj5Id=J[e:J[e:3TX>BPiZ_@h>B_/>0kl00;on00Bo002o
_iX00;o@002og@00_oL01Km7?Q:o=P341;m??Q:om`00_oH00;om0?jo0P00_oh0
0;oj002oo:4l@9JQXD2FXD2GUT2FXHjF2@RGK2H8UY@8UY<920FFSSi/UPFASR6>
E`F>E`D5M@D5T`DBKC/51ClkKQ8k4_booQ;l_`nj4Q:j4[XB^Q:j^UI_4P/ooKX>
2kZjBSe:B[Y:^TY:E[d;oDXE2[e:BVY:_DYZ_DYZ_FX>BPh>BPh>B_`>0kl00;on
002oUP00_m000;o`0?Zom@00_oL01Km??Q:o4P3>1;m7?Q:om`00_oH00;on002o
oP02_`2ooP00_oX00;olXCe1XIJQ@:6F@9L8X@VFSZ45XII02I>FSYI0?XhKK9<9
SPFF59H51ABFKI<5KA@5T`DDK@DB1ET61A@B1C/B5?dB003m4Q8k^Q@B4[XB^Q:j
4Z@B^[YF4`XBn[XW2fl:^[X@ETZjBTT:BP]:_Ha:BP/=JTY:3TY:3TYZ_FX>BPh>
BPh>J_h>0kl00;on002oUP00_m000;oa002onP00_oH00;og00FoAchB_`00f@Bo
CchB_oL00;of002ooP00_oh00[l0_oh00;oj002ooZ50@:6QUd2QUT2QUT2GSYI0
UfbGUXjD8PU/UPF@2@D8UPD5T95^TA^V1IHD1FfC1@FA50D5^@DB23/m50DB53/5
4QCn4Q80_a8B5198>dDB^VhB^Q:jB;XBo[X1^Fom^S5FB[Zg3KZj2`ZjB[Y:BVm:
BSZj_DY:JTY:JTY:J[e:J[eZJ[eZ3TX>3VZm3PiZ_`00_oh00;nF002od000_o80
0;om002oo@00_oL00;og00FoAchB_`00e0BoAchB_oL00;of002oo@3n_oh0o[oi
00>oXJ50oZ5F@IJQT9JQSYN>Ud2F@9J>2@PV20V>UPF@1IJ>1@V>1Fh91Hh5KI<8
V0PD1I45?I=]50DDTcfC4Q854S/74Q8D4Q80_a8k4S/B^Q984[Vj4TRK4[Y_4[Y8
o[X24TYFoKX;B[Y:B4Y:^[YZA@ZMoDXJ2TZM5FY:BVY:BVZmJTX>JTX>J[d>JPiZ
3PjooP3n_iD00;o@002ooP01_kog002ooP3n_oh00;og002om`05_dln4[lg0>@4
_dLn4[og002oh01R_j6Q@J6Q@9NQSZ6FT9I1UY2F@9J@UXh929J>2@P920D9K0FF
59HDUQB@1@E]T`D5S[TD1I4mT`DD1A8D1A9]518D>a@B>a8D0;lD4Q8D4[XkEQ96
^S]F^Q:jB6njB;Zj4TY8oKX@BTQ:^[Y:ETY:2dZj2TY:2`[nBP1ZoTXCJTYZBPi:
JTX>JTX>JPiZ_@iZ3[oF00Fo_`00_koZ0?fokP01_ko`002ok03m_od0oKom0?fo
n@06_`00_`00_oP00[l00?bo0P00_oL00;og00FoAchB_`80m@BoAchB_oL00;oP
06foXJ50XJ50XIN>XD2G@II0UT10UPV>UXjG1I2FT`P9K0T5@9=]K0VCK@F@4Xh5
KI=]50D550D550DD51884QAG?AhB?A8B?A80_c/DVa984Q9F>a:j4TRj4[Y8^Q:j
^TQ_?CXB^P/mBUH;^[Y:E_i:0EH;oTXL2TZl2dY:JPY:BVY:JTX>BVY:3VYZ_FX>
JPiZ3[oF00Fo_`00_ko[00Fo_`00_ko_006o_o000;o]002ooP0A_kl00;no002o
_`00_kl00;nonP06_`00_`00_oP00Kl0n[l10;og002om`05_dLn4[l00@l4_dln
4[og002oh01:_j51XJ50XD2GXHjG@:489T2FT8jF@0T82FbF7R:>2@F>1D052I=/
1@FB4Xh51A@5T`EeT`Ee1A@550DB50DB51@B1ClD4Q@B5;mFoQ8SB18BEQ:j4UHk
^Q9F^Q8;5;Y8YKZjEP]_^UI:^[Y:B4Zj>VnjoTXMDfm:>VmZ^VY:BVY:JTX:3TYZ
JTX>JVZmJVX>JPjoeP0=_kl00;no002o_`00_kon0?fonP07_kl00;no003l_od0
o;l3002o_oh0oKon006o_o`00;oY016o_`00_kl00;no002o_`00_koj002ooP01
_kog002ooP3n_oh00;og002om`05_dln4[l20AP4_dLn4[og002oh01;_d2QXD2Q
@9NQ@46F@9N>UhjG@8jF2HjFT0F@1IJ>1IB>KIJ>50T5SY<8V8hBSY=]5945?I4D
1@DD50DD4PDD?A@B518D4Q@k?C/BE_hB>4PBES/BVdPk^UHk^[YF4[ZjB;ZjE[ZM
E[ZjBTQ:B4Y:^PZM^X/;BTX:2`Y:BVX;BPYZBVY:JTYZ_OiZ1dX>JPiZJPjoeP0E
_kl00;no002o_`00_kl00;no002o_o/09Kno002o_`00_kl00;no002o_`00_kl0
0;no002o_`00_kl00;noo000_nT04Kno002o_`00_kl00;no002o_oX00;ob002o
oP3n_oh00;og002om`05_dLn4[l20?h4_dLn4[og002og`0h_j51XD2Q@9M0XD2F
@IJ@@9J@1I2FSPV>UXh91HjF542C@6h5SY@BSPD5Tai]218559<5594D1ET5oQ@V
1ETB518D53/m4Q@B4TPBKQ9F4S]F4[XkEQ98^Q:jB;YF4P]FKi]8o[XZEP^j2R4:
B[Y:EP]:>[X:2dZ;2dY:2Pi:WFX:BVYZBVY:3VYZBVX>JVX>9=D0nkl?002o_`00
_kl00;no002o_o/0nklK002o_`00_kl00;no002o_`00_kl00;no002o_oP00;oZ
006o_od0oKom0?fon@00_o800;oh002om`00_oL01Km7?Q:o0P461;m??Q:om`00
_ml0?Km1XD2Q@:50UhjQ@0V>XI05XHh929B>XABF@I<8U8h5K@V>TF`57VfCK6dN
>hh5^HhDKA@5OFdQ1cdD1A@koA@64Q@D4Q@BE_hB?TPBB1984UHk^TPBB;Xm@Q9F
^Vm8^ZEF^T:jK`ZgS9X;ETY:B4ZjDdY:2Yb<BYa:S3Y:BPh;2VY:JVY:JVY:3_iZ
0PiZ_mD03kno002o_`00_kl00;no003k_o/06kno002o_`00_kl00;no002o_`00
_kl00;no003k_oP00;o[006o_od03Kno002o_`00_kl00;nonP00_o800;oh002o
m`00_oL01Km??Q:o0P4<1;m7?Q:om`00_ml0OknQ@J50XD09@:50UY10UY101I2F
T0F@UQi/UQi/UXhDU0P91HhD@3^>4Y0kSQ:D50Q821@D1KT551@mTA@B518D?A@B
518D>aA84S]F4TPBEQ984TRj>eJjB;XkE[Y8^UJL>hbg2[Y8B[Y2RkY:B4Y:DkY:
BUHj2`Y:W@Y:JTYC1dX>BSY:JVY:oVX5BVYZ3VZoe@0A_kl00;no002o_`00_kl0
0;nom`0M_kl00;no002o_`00_kl00;no002o_`00_kl00;nom000_n`00Knoo00=
_kl00;no002o_`00_koj002olP00_oh0o[on002om`00_oL01Km7?Q:o0P4G1;m7
?Q:om`00_ml0>;nQ@42QT:50Ud2@UT2@UT2?1J6>2@RD@0E1SVd9SPP956bB1F`5
7PF>1AB>4YA^SWFC56hkJg@8>ohD0PDD?OhDAa9n4Q@DB1@BB1@BB1:jB7TBES]F
>kY8^TQ8^TRj^TRjETQ:^UH;4U>jBUH;ETZj>W0:WH/;S9`:BP]ZR`/:JTXjBVXj
BVYZB_iZ0`iZJ[oE016o_`00_kl00;no002o_`00_kon002on`0M_kl00;no002o
_`00_kl00;no002o_`00_kl00;nooP04_`00_kol002ok@01_kok00fo_`00_kl0
0;no002o_oX00;ob002ooP3n_oh00;og002om`05_dLn4[lk0?X4_dln4[og002o
g`0l_j51@J50XI2FT410Uhi02IjPSPT8U0Q0TPE0Thi01@T72C^>SS^@50F>4Y0B
7PDBSSfA51@551BA51874_`DAQ8m51Ad4Q@BB;TBB1A84[TBE[MF^TPBE[L:4UH;
KTY8B;YF2eI_BTP;ETYFBYeFBU=_J[Y:DdYCKfY::4XjBPZMJVY:JSY:oFX3BVYZ
_mD01Kno002o_oh0o;on0?fonP0=_kl00;no002o_`00_kon0?bo0`00_kon0?fo
oP01_kol002ok@3k_oh0oKom0?fon@00_o800Kl0n[l10;og002om`05_dln4[l0
0<44_dLn4[og002og`0l_j6@XD6Q@HjG@HjG4II0@IHAUY05@905T0F@1I05SPTD
SPF@>d0DSY4DSQ@94XhD1AhB1i<DKA@B51@5^OhDBA8D?A@DB1@BB1@BES/DB198
B198B;YF]eHBB4Q:ES^jEP]F^TQF2kY82TZg>UJj^SY:B0X;S9`:2`Xj^SY:>[dj
BSY:>TXjJVY:nfX0_jD00Knok000_m000;ob00:o003l_`800;og002om`05_dLn
4[lB0<@4_dLn4[og002ogP0k_d6Q@D2Q@449@42Q1I10@0T8@0V>SYA/U0RD27e0
1I05I0F>1AB>KQh558hD1AB>4Qh553d7>ahm50M^oAA9^AA^54PD4Q@kEQ9854PB
B4PBETPB@Q9FB;Y8^TQF]`Y8^UH;^KY:M9eFBUJM2[U:DkXj^P/:BT:;>TYCBU=:
>TXjBSX>BSYZBS[mJP2oZ000_oh00Knok000_m000;ob002ooP3n_oh00;og002o
m`05_dLn4[lk0;T4_dLn4[og002ogP0h_d6Q@92Q@409@46>Uhi02I08U42@1D10
2AB@1I05K1j>1HhDU0L52AA/7Q:>FI4D58ie59<D?JH5maA64TPD55Hk54PBES]8
EQ98B3/H>dRjB4RjB4RjB5H;E[YF@[YFBUH;^@X;E[XjEP/:B[R<WH/;2SZj>VZM
2SY:JU=:JSY:JS[mJP2oY`3m_n/00;o@002olP00_od00;om002om`00_oL01Km?
?Q:o002h1;m??Q:om`00_mh02Km1XI10XI10@0Wn@2jF4IHAUQ6F4@F@KT6>1HlD
@@FX29<N26eT21BA7V`D50DDSQ@D2AAf50Lk51BAFOhD05Km506i^OhDADPDB3]8
ES]8B5HkB5I8B;Y8B;Y8B4Y8^KYF@[YFBTQCKeH;ESZj@PY:EP/j^U=:BRR<W@X;
>X]Z2cY:>VXjBVXjJTXjJ[n=002od000_o400;oj002omP00_oL01Km7?Q:o002`
1;m7?Q:om`00_mh01KnQ@D2@XOi0;650@16F@8h9@9082A45SY05T0DA1HjD540D
T7d8TaB>50TD58hDKAj_7QAm1OXD16h7EPEFoQ@;B1@B54Q^B19FB3]FoDPi>aPB
@UHBB5H;B5JjET9F2eI8BUH;B6mCB4YCB4Z;@TXjETYCBU>mC4XjW@YZDdYZ>SY:
JSYZ>VYZ_hd00;o@002ol03j_oD00;og00FoAchB_cL0[PBoCchB_oL00;oN00Zo
_d51XI10@449X?i0:`V>@8h9SXh9SPDAU0R>50V>SQA04Qj>1Ai/50EeSQ@N4PL5
5:PBJa@DTG@7oQ@0OOdD2eHD4QAd4QA8FDQ85?i801;lB3MFB4RjB4Q:@kYF@[YF
EYa8BWAC^T8:KeIC2eJM2U>jDdYCW@YC2h/jBU<jBSXjBSYZ>VY:>VZo_hd00;o@
002og@00_oL01Km??Q:oJ@2T1;m7?Q:om`00_md0=Kn@Ud51@450@0U0@Hj>2I2>
U8j>U8j@1A45T1B>U1B>1A45SQhmT1@D@Gh7KAhBDA@550MdToTD418JKQ@m54PD
54PDB6i8B5HBo4Ph549F549FB;Y3BWBj@UI_B2PBDdRj>TP:@[Z<^0X;64ZhS5<;
2U>L2SX;RcYZ2`Xj>TXjJSYZ>VZoS03>_md00;og00FoAchB_`00[PBoAchB_oL0
0;oM002ooT4b@:6@@D102Hi0T8jDSY05T1i/@I@7@AA12@L57Xh51ai]SQ@N58hD
1@LD7QBA50Md27@7oQ@0Y_dD@DPD5:IF^DPD4W@BLa9d>gA84WA8B5JgET9FETQ8
^TQF@UH;EUI2ETY8DkYFC;Lj2TPjKbQ_:5=_D`/jS5>M2SZMS?hj04[n>PA:>VXj
_h`00;o@002og@00_oL01Km7?Q:oC`2W1;m7?Q:om`00_md0=;m1@449@450@I2>
U92>SPV@SXh94@E04@F>7Xh54I4A1HdN58hDSQhDSQ@DSQ@N4V/D7Q@7n1@0M?dD
34PD54PDM19d>eId>eKjB2dH]aPB@WBjET98B0Y2ETQ:@UIF2e=F2T:jDha2^SX:
@VYFDdYCDdXjDdYCJSY:oCX3JSYZ_h`00;o@002ol03j_oD00;og00FoCchB_`80
/`BoCchB_oL00;oL002ooT4h@450@I2>2I2>T9:>SQj>@9@7@0DA2@L54@DA1Ai/
58iT58hD58dN51@751hD1fhN50LDYQAd50M8oA@8EQAd519d54PDoTP254QdoDP3
ETQ2E_i89UI2E[Y8DkUF@[XH@[XHWEIC^CZj:;YCESX;ESYCBU=::4YCDcY:Dohj
1DYCJSYZ_h/00;o@002ol@00_oX00;of002om`05_dLn4[l00;44_dLn4[og002o
g006_i09@42@@?j@8XjD@4472D2@1`TA1D472F`NT1B@590DSQ@A1A@A50D77Q8A
oQ@84AAmKE/N0QjVna@@M1@DEQ@DB4PDB1A8M198B7CjB2iF@UI8ET9FET9FET8:
V0Y2B0YC]e=dBQP;60/jEU>M64XXBRQ:>U=CBSXjDdXjJ_dj0;n;002od000_o80
0;om002oo@00_oL00;og00FoAchB_`00]0BoAchB_oL00;oL036o@D6@@D2@T8j@
2Hj@T1i/T8h94@DASPDASQh54@DA58hDT1B>M8hDYXl551@A51A[7_dD6``B7QBV
57@7M1@DM1@DB1A854Q8M1A8B5Id>gCnB2ed@UI8B5I2ET9FET9FET:<@UIF2aQ2
2T8:@PY22T8jESY::5Hj2e=C2SXjWE<:nSX0_h/00;o@002olP00_oh0o[on002o
m`00_oL01Km??Q:o>`3H1;m??Q:om`00_m/0=Km1T46>T910@149SY104@F@SQj>
SQj@58j@590DT1B>7QB>514D1A4D50NXKPLN51@7M0Md1o`D07Cn50ed51A851Ad
M1A8M3]FM?e8<7A8B5I2M4Q8ET9F@UI8B5=8EU=2M4YF@U=dBUHXW5HjB3YCERQ:
:4XXBRQ:>SYCDd[k>P2oe@3n_oD00Knoi`01_kon006o_nP00;o/0?fon@3m_od0
oKom002ooP02_kl0o[ok00:o003l_`800;og002om`05_dLn4[lk0>T4_dLn4[og
002of`01_d7mT02>oY0^U0M1T1i/T1i/4D0DT1j>590DT1B@58dN1aB>7Q@D4A@7
4ZQ[51hD1gB=51@JYQkn51BVM1@DEQ@DM1A8M1A8M1A8M3/HB7CnB2id@TQ8M49F
ET8HB4Q26;Y265I2B[@;EU=FDd:<@PY<@PYC2aPjDieC>XaC>SX;Dohj0PXj_mH0
1;no002omP01_koW006o_n<00;o]00Fo_`00_kok00fo_`00_kl00;no002o_oh0
2Kl00;l00;l00;ok006o0?Zo0@2om`00_oL01Km7?Q:o004<1;m7?Q:om`00_mX0
F;n@T4507Y2@SY107Xi04HhNSXhNSXhN4@DA1A4D4@DA50L9SG>>514D50LN50LN
6Q@D1a@NYaXD5:HDM1@DYW@DM1Ad54PDB4Qd54Q8M4QF@WA8M49dM3/XoTPT:4Q8
DdQFDd9FB5=8:6mCEU=22T9CERQ:De=8:4XXBSYC:4XjDo`j0;oE006o_o/0oKon
0?fo4000_kl00;no002o_`2o_`00oKon0?fo0`00_kon0?fooP3l_oh00Knoo000
_nd01Kno002o_o/03Kno002o_`00_kl00;nooP08_`00_`00_`2onP00_oh0o[on
002om`00_oL01Km??Q:o004I1;m??Q:om`00_mX00;olT7Z>4D6@2@N@4D091a49
1i0DT1j=1`T77PL91ahD1aj>51@A51@A51@7M0LN50`DZ08N51XD51hm6SeP51Ad
EQAd4V1F4V18B7AFB7@n64Qd@WAF@WA2EQQ2M4QCB48HBT=CET8XEU=2S48:@U<H
2e=CRd9C>U<:De<j2e?l>P2oe@3n_od00;on006o_oh00KnooP07_kl00;no003n
_oh00;on006o_oh00KnooP0E_kl00;no002o_`00_kl00;no002o_o`00;o]00Fo
_`00_kok00fo_`00_kl00;no002o_oh00;on00Bo_`00_oX00;on0?jooP00_oL0
0;og00FoAchB_fT120BoAchB_oL00;oJ006o4Oj@Ohi07Y114D107XhASQj>4@T7
T1B?2@LA58hN590D4A@77PLD7PLN50L<7Xed1g@7^AXNYQ@D6ZPDEQ@DM1Ad57@D
H19dEWA8B7A8M4Q8M49d69Q8649dB48HET98:5I8Dd9d2T9CB5=8>QQ2^`Y2De=F
:4YCDbQ:DcYC>U<jDcYC0CZoe03n_od0o;on006o_oh02Kno002o_`00_kol0?bo
oP01_kon016o_`00_kl00;no002o_`00_koh002ok@05_kl00;non`0=_kl00;no
002o_`00_kon002on`00_oT00;oh002om`00_oL01Km7?Q:o004@1;m7?Q:om`00
_mT0Okn@T150T90A@1j>T8hN@@M17XhA590A1A4D4AB@514DSQh7M0LN1a@A51@N
31@D4DP7M0LJ7SfWM0Md51AT57@DEQAd57A857A8B7A8M4QdB7A2M4PXM9PHB48H
B48HB2Q8@QQF@U<H@UICEU=2DeICDdZf>UICDd8jDcXHJU=C13YC>U>odP3n_`L0
0;no002o_oh00KnooP09_kl00;no002o_od01Kno002o_oh00KnooP0A_kl00;no
002o_`00_kl00;non000_nd01Kno002o_o/03Kno002o_`00_kl00;nooP00_o/0
0;oi002on000_oL00;og00FoCchB_`015`BoAchB_oL00;oI07>oT4504D0A@D0A
T1A14HhNSY0A590A59077PLN21h74G@77V/N1a@NSA@N1aiK1g@D52j=57@7M0Md
560D57@DM1Ad57@DM5IdB7A8M4QdB7A2M7A2M5Id@WA2649dB2QF@QQ2M0Y2648H
@PY2DaQC@U<H2oiC3EIC>U=C:3YC>U<jDcZod@09_kl00;no002o_oh00KnooP09
_kl00;no002o_od01Kno002o_oh00KnooP0A_kl00;no002o_`00_kl00;non000
_nd01Kno002o_o/03Kno002o_`00_kl00;nooP00_o`01;l00;noo000_oh0o[on
002om`00_oL01Km7?Q:o0P4P1;m??Q:om`00_mP0NKn@4J2@4Hi14I0NSQ6>7XhA
SQh7@Ah77XhA514D4AA0;PP^1a@N1a@A51XA57@76Q4DM0MdSG@D6Q@NFAAc57Ae
M1Ad57A8M5IdB7A8M4QdB2YFM7A269PXH5I2]4PXB48H@QQ265IC@QQ2DaQC@PXX
@U<::5=2>QPjoU<5>U<j:3Zod`0<_`00_kl00;no002o_oh00KnooP09_kl00;no
002o_od01Kno002o_oh00KnooP0E_kl00;no002o_`00_kl00;no002o_o`00;o]
01Vo_`00_kl00;no002o_`00_kl00;no002o_oh00;ol00Fo0;l00;om002ooP3n
_oh00;og002om`05_dLn4[l20?H4_dLn4[og002of01d_i114@V?SQ514HhNSQ6>
7T6?1aj>4A@ASQh77PLN1ah77PLN1ah<51h77P`D1bh7M1@^SG@7M1@D6ZPDM1Ad
54Qd57Ad54QdEWA8M7A8M7A2M4PH:TQ8]49d648HV1Q2648H@QQ2649CERQFDd9C
65=2De=FmU<2>U>odP3n_od0o;om006o_oh0o;l3002o_o`0o;om00Fo_`00_kon
0?fooP09_kl00;no002o_o`00;o/0?fooP01_kon0?foo@3m_od00;om00Jo002o
002oo@01_`3j_`40_oL00;og00FoCchB_`00/@BoCchB_oL00;oG07Bo@A6@4D6>
4D4NSQ6?SQj>4Ah7ShhN1ahASAh77V/N1ah77XeK4A@A6Qh76Qi[6ZLN53TN51XD
7QXDH1@DH7@DM1AdEW@DM4QdM4QdM4QdM49dM4:dB7A2]4Y3:4Qd:49d:4PXB2Qd
Dd8HDd8HDbQ22RPXDeKgD`4j_hH00;o@002oo@00_oh00Knoo002_`00o;l2002o
m`00_oL01Km7?Q:o002V1;m7?Q:om`00_mL0@[no4I114I2?SQ4N@@LAT1i[4HhN
S`LN4F/N1ah77PN?M6/AM0LN1bj=M6/^1g@D>7DN6W@752hDH1@J57AeM7@DOACl
M2a8M7A8M7AFM49367A2@bQd@UJd@W@H@V0X@W@XM2Q265=2ERPX649C65=2DbSg
D`6o_hH00;o@002olP00_oh0o[on002om`00_oL01Km7?Q:o>`2Z1;m7?Q:om`00
_mH0?;n?T1514HhA@Aj>4HlN1i0A514A@1h77PN?7P`N1ah77V/^Jai[6Q4D;V/J
7PLJ7Q@J51hJ5:QFM0MPM6_nM3<DLdRXEWAdEWA8M7A8]4Qd@WAd:QQ8]TQd@QQ2
]5HX@W@XB2QdDgAC@QPX@QQC:5IC:5=2oU<0:?]C0;n5002od000_o800;om002o
o@00_oL00;og00FoCchB_`00Y0BoAchB_oL00;oE04>oSa6>@A6?4HhN@A474Aj>
Sa4DSa4DS`LN4A@h1ah77P`N1ah<7Q@<7P`N50`N6QhD37@DYbhDM1@^57@DM5HN
EWAd5?ad04;nM2YFH49F]49d@bQ8]49F]49d:7@XHW@XB2PH@RQF:2Q8:5=CLd8X
De<XDe<XoE<0_h@00;o@002ol@00_oX00;of002om`05_dLn4[l00:44_dln4[og
002oe@0o_a514D4A4HhA7Q6>Sa4D4A6?516?515[7Q47;T0^Jah7;PLN31@h514J
53PD7QZW;QAd1bhDM1@^57@J561d5?id01ClM018oW@Z:7A3B2Q362YF:QQ2]5HX
:UHXM2Q267@XHQPX@W<X@QPXDbQ262QC:5=C:?iC0BRoQ000_m000;o`0?Zom@00
_oL01Km7?Q:o0P2I1;m7?Q:om`00_m@0?[lA@A6?SQj?4A6?SQj?1hlN4F/AM8l7
7Q6=;V/AM147;Xd^1ah<7QX7;PMd4AXD;PMd52hDM1B357AF7QYd5?id04SjM2Y8
:D9dM5HZ64<X@aQ2]48H:W@XM2Q264:d:4PX:4PX:1QR:2QF:5<XDbQCoRP2De>o
P`00_m000;oM002om`05_dln4[l209@4_dLn4[og002oe010_klASi0AShj?4AhA
1hn?7PN?4A@A4E/A7V/A6XlN31h7>1@A6Q4J7P`D;PL^51h<M0MPFaiF7QXDM7Ae
H1Ym563nM01FnG@N@RU8]4PZ67@X:W@X@bQd:2XH@QPX:QPXM2Q262PXM?hX2d8H
:2QC:5<XDbRo_h<0c[oM002om`05_dLn4[l008l4_dLn4[og002od`0n_hn@S`N?
4A6?SQj?4A477Q4h1aj?Jahh1ah7;Q47;V/^1bi[;PL^1ah<M1h<7U]d52h7OAhJ
M1@^EQiFM1hJo7@0B?ed8APZM7A8]UI3:7Bd@V12]1Q2]W@XB;@X@W@XDkAR:7Be
:5Kn:01FnbP3DbQC_h40Z`00_oL01Km7?Q:o002>1;m??Q:om`00_m80@KlA4Hn@
Sa4AJahAShl77SQ0>1h74Ah<Sah<SgB?31h<7P`N31h<7P`N6QiK7QXN51XN6QiI
M0LY57@^B7@D:A@H@o]d91Q369Rd64Rd@V1d:2XH]4:d@RUF:2XH:7@X:U<H:2YC
62PXH5?m:012oBP0_h40ZP00_oL01Km??Q:o4P281;m7?Q:om`00_m40>[n?Sa4A
Jaj?4A6?1ahA>6/N4Hl<ZA4<7Q4<7PL^4E/A6SPN31h<7QXD>1@^52hDZE]d7QYd
52hDM4<<mW@QH2QdM4RfM7Bd@W@Y@RU2]4:d62XX62Ze64:fM2PXM2PXH?hX0f8H
:;Gm:05C_h40Z@00_oL01Km7?Q:o002=1;m??Q:om`00_m40=[no4HlA4HlASf/N
4CQ0>1h74CPN1cPN1cPN38lJ4CPD>1@h6Q4J7P`N6QiK7P`N6QiF;QAd7U_nM04A
6?ed0AY3o7@R]5Id]4Qd]TRd@RUd:RQd]W@X]4:d@[ACM2RfERQd:2RdDd;m:0Id
:2Q<:;noP@2Y002om`05_dLn4[l008l4_dLn4[og002od00l_a6?4Hn?1ahA4CPA
JahA>6/N>15[;Xl<7SP7;V/h7P`N32h7;Q@hMBhD>7@D>7AK7QB3M1XNM5H^57@Y
5?id02WnM2Y8]7Ad62YdM;A8]UHY@[@Z6;A2]QPZ:7Bf:4Rf:1PZ:5>d:5<Z:1Re
:2ReoRP0_h40Z000_oL01Km??Q:o0P2:1;m7?Q:om`00_ll0=klASa4AScQ0>8lN
4@`ASbh7ScPN1cPN314^315K4Bh<7P`NFahh6Qj27U/N6QhJH1@^M:L^M1@^oG@1
6V3lM00YoW@N:61d]4RfB;Ad]T:dM;I2]2Qd]T:d:7@X]T:dDbRd@_hX0KACoRP2
]BRoP@2W002om`05_dLn4[lk08X4_dLn4[og002ocP0k_hl7ShlA>16?4@L^Sa4h
1bj?4@`N>0Lh;V/h7P`h7U/h7Pad>1@^52h7;Q@^M2h7M60JM1hHZAYdH5H^oG@Z
H1QPB;AF]4<H@bQd]4RfH2Q3:;I3:;A2]QRf@[AC]4:d:2Q2]U>d:5>f]OhX0;n1
0:H00;og00FoAchB_kl0R@BoCchB_oL00;o=03JoSa4hSa6?>6/A7SR?38lN>0Lh
7SP<Sbh7>2i[;P`N>5/N>7@<7QXN;QB352iK7QYP52idM1XNoW@2;QQ3oW@/:EHY
M7BdB;AdM;Id:4>fM;Id:2Zd:;Ad:2XX]7@X]RQd:;HX:;AC:2YC:2RoP@2U002o
m`05_dln4[l008H4_dLn4[og002oc00e_a6?4CPNSa4h1cPA7SP7>8l^1cPh7P`h
7P`h7P`^4AXh52hN32hDPa@hM7@^52i8>7Ad;UH^oG@`;WAdH5JdM60HH1PZM;AF
:BXH:BXXM;Id][A2]2Rd:RQd]RRd:4:d:2RfDkIC]RPX_h40Y000_oL01Km7?Q:o
>0251;m7?Q:om`00_l/0?KlA>8lA>8n?314^4CR=>1hhSCP^4@`h7P`h7P`^4AX^
4AX^58<A6W@^M3PD;W@J7WB3M1Y3M2iFH7@JH7BdoW@W]7@YM7Bf62Yd]1PZ]2QP
]WBf:;@Z:;Ad:;Id:;HX]48X]RPX]2PX8h40X`00_oL01Km7?Q:o>`231;m??Q:o
m`00_lX0>[lA>8lA>16?>6/h7SPh1cPN>0`A;SP<7SQK4Bh<7Rh<7Rh<H0`NH0ad
;ZTJM8<D;V0JH5H^OG@^M7@YoW@X]7Ad]7BdB;Ad]UJd]9Rd]QPZ]2Rd:RRd:2Zd
:;@X:RPX]2Rf:2Rf:;n10:800;og00FoCchB_`00P@BoAchB_oL00;o906Bo4CR?
>14h4@b?;Q4hSCP^4@`h;Q4<;SP7;SQd>7@h53P^M0`N;QXN6V0^52im;WAd;Q@Y
M1YPM2UdM2id]4PYM;Ad]7@X@kAd][A2:BRd:[@X]2XX];HX]4:f:;@X]RRf:7@X
8h40X@00_oL01Km7?Q:o4P1o1;m7?Q:om`00_lP0H[n?4Bj?>8lh>8l<4Bhh4@`h
;Xl<>2i[;SP^1cQd>60<;QiKPa@^M3QdPa@^H1Yd;WAd;WAd;W@YM7@YM7@YM;Ad
]4Rd6;@ZM;Bd62Zd:;@Z]2RfM;Jd:;HZ:;@X]RRf:;HX_h40X000_oL01Km7?Q:o
=`1m1;m??Q:om`00_lL0H;lh4CPA>14h>6/h;Q4h33PN>0`^4CPJ>1h<;Rh7;Rid
>7@h58<^52id;U]P7QXYM8=dM2idM2id]7Ad:G@YM;Ad]1Rd@bRd@k@X]4<X][@Z
:;Bf]W@X][HX]2PZ]RRf:;n109l00;og00FoCchB_`00N`BoAchB_oL00;o605jo
ScR?>3PA>3R?FcPhScP<;SR?6SPh7P`^>1hJ>60<;QhJ;QiK;W@^7QYd;Wd^H1YP
H1Yd]7@^H1Qd:GBfM;Ad]7BfM;Id]WBd]WBf]WBf];HX:[Bf:;Jd:;HX]RRf_h40
WP00_oL01Km7?Q:o001i1;m7?Q:om`00_lD01[no>8lh>17n>5:=>2hh>0L^>3QK
7SPh6Qhh;QXA;QX^ZAX^7U/^H5/N;W@^M2hHH2idM2id:G@YM2Ud:GBdM;@H]4>d
:2Vd:QRd]WBf][@Z:;Jd]RRf]RRf]2Ro_h40W@00_oL01Km7?Q:o0P1g1;m7?Q:o
m`00_l@02[no>8lh>14h32hAoSQ0FcPA;SP<;Q4^6SPN;P`^ZBiK7RiK;W@^7QX^
M2id;W@YOBiP62id:K<YM;Ad]7BdM;BdM;Id];IP:;BfM;Bf]WCn]P6d:?jf0bRf
_kn109`00;og00FoCchB_`80LPBoCchB_oL00;o200>o>3R?oCPAS``h;SPhM3Ph
;V/^>2id>2h7oRha58<^6Qh^OBh^M2iKH7@^M2hYM2id]7@YM2Ud:GBdM;@YM;Jd
M;Jd]7Bf]2Rd]RZd]RSn]P@X];HX_h40VP00_oL01Km7?Q:o001a1;m7?Q:om`00
_l401Kno>0`h4Ohh@bhh>5^?>2hh6SPN>2hJ>:T^Fh<N;U/^I2hJ7Rid;RTJ7RUd
;W@Y;QQP:G@YM2Ud];Ad]7Bd]1Rd:BZd:BRd][Id][Jdo[H4:;Jf_kn109T00;og
00FoAchB_`80J`BoCchB_oL00;o0006o_oXhBA4^33P^>3Qd>2hhM3P^360h;W@h
M2h^6Qj3;W@^M2id;RUd;QQP:G@^]7@Y/bTYM;Bd:GBf]7Bf];Ad][Bf:;Bf];HZ
][HX][noP@2H002om`05_dln4[mY06L4_dLn4[og002o_P01_kok>39K>3P^>14^
32h^>1hJ>2h^7U/^;W@hOBh^M2j3H1XY;W@Y;W@YM2hYM2Ud]60Y6;AP]WCn]0XH
];@Z];@X];Jf]?ff0KnoP@2F002om`05_dLn4[no06H4_dLn4[og002o_001_kom
>3T^>3P^33P^4Bh^32h^4BhJPbhN;QX^;W@^;W@^;Wd^H2TJH2T^M2TYM2TYM2Vd
:GBd]1Rd]62f];@Xo[@2:[Jdo[H1_kn109@00;og00FoAchB_`00I@BoAchB_oL0
0;nk006o_ohh<P`^>8l^>2hh;P`^;Q4^;P`^;Qh^Fbh^M2j3M2h^M2hY;W@Y;WBd
;RUd]2Ud:KAd:KBdM?jd3KId];Bf];Jd][Bf][noP@2C002om`05_dln4[lB0644
_dln4[og002o^@0^_klh>2hh>2hh;SP^32h^Sbh^6Rj37Rh<;V0^;QYPPbid;RT^
M2T^]7@^]7@Y:GCn:A:fM;Bd:KBdM;Jd][Bd][Bf];noP@2A002om`05_dLn4[lk
05T4_dLn4[og002o]`0?_kl^>3P^32hh;SP^7SR36_h^0CQdoBhKOBh^:Bid;RT^
M2hYM2T^:BUd]2TYM;HYH;Bd6?^d1KJd];Jo_h40S`00_oL01Km7?Q:o001G1;m7
?Q:om`00_kD0o[lT>2hh;SP^;SP^;U/^PbhN;RhJPbiP;Rid;RhY;W@Y;RUd:Bjd
M?hY3[Ad:BVd:KBd:KAd][Bf]?joP@2=002om`05_dln4[l204/4_dln4[og002o
/P07_klh;Rhh;SSl;P<hM2j3oBh@M8<^:H=d;RT^:G@^:K@^:GCm:@>dM;@Yn[@1
_kn108X00;og00FoAchB_c/0C0BoAchB_oL00;n`0?jo1Rhh;Rhh;X?m;P<JPbiP
oRhJ:AX^:F0^:BhYM2T^]2T^];@H:KAP]2Vf]2Vdo[n108P00;og00FoAchB_`00
@0BoAchB_oL00;n]0?fo0BhhoBh0Fod^42T^;RT^;W@^:BT^:BhYM;@^nbT2]2Vd
oKn108D00;og00FoCchB_`00?@BoCchB_oL00;nZ0?booRh4Pbh^PbWn;PLY;W@^
:Bjd;_hY1bjdM2TY]2TToKn108800;og00FoAchB_`80=@BoAchB_oL00;nU0?^o
oRh@:H<^:BhY;RT^:BT^:BVo_bCn_h40P@3m002om`05_dLn4[lg0284_dLn4[og
002oW`3b_`4T_h40P@3g002om`05_dln4[l001/4_dLn4[og002oP@2108@00;og
00FoAchB_`006`BoCchB_oL00;n10840Q000_oL01Km7?Q:o=P0K1;m7?Q:om`00
_h40P@24002om`05_dln4[l00204_dln4[og002oR`01_kn10840o000_oL01Km7
?Q:o000`1;m7?Q:om`00_id0o[l1003l_`400?jo1@00_`00_h40P@3m002om`05
_dLn4[l20384_dLn4[og002oWP00_oh00;ol00:o0;on00Fo0;l00;n10840o@00
_oL01Km??Q:oJ@0^1;m7?Q:om`00_iX00;om00>o002ooP04_`00_kn10840o000
_oL01Km7?Q:o000Z1;m??Q:om`00_i/00;ol00>o002ooP00_h40P@3h002om`05
_dLn4[l002/4_dLn4[og002oW000_o`00;on002ooP00_h40P@3h002om`05_dln
4[lB02/4_dLn4[og002oW@00_o/00;on002ooP00_h40P@3h002om`05_dLn4[l0
02L4_dLn4[og002oWP3l_oh00;om0?joP@210?L00;og00FoAchB_`006`BoCchB
_oL00;n10840Q000_oL01Km??Q:o=P0K1;m7?Q:om`00_h40P@24002om`05_dLn
4[l201/4_dLn4[og002oP@2108@00;og00FoAchB_`006`BoCchB_oL00;n10840
Q000_oL01Km??Q:o000K1;m7?Q:om`00_h40P@24002om`05_dLn4[mY01l4_dLn
4[og002oP@210=/0c[oM002om`05_dLn4[l002D4_dln4[og002oV`3j_h40b000
_m000;oM002om`05_dln4[l002d4_dLn4[og002oW000_oX00;n10<T00;o@002o
l03j_oD00;og00FoAchB_`80=@BoAchB_oL00;nL00:o003n_`800;n10<T00;o@
002ol@00_oX00;of002om`05_dLn4[lk03T4_dLn4[og002oW002_`2ooP02_`2o
P@39002od000_o800;om002oo@00_oL00;og00FoCchB_`00=`BoCchB_oL00;oK
0<6o003l_`00_kn:002od000_o800;on0?jooP00_oL00;og00FoAchB_`00a`Bo
AchB_oL00;oO0?Zo0AJooQH0_o/F33`F?3`F?3aY?6TlJ@;nJ@<2J@9YnP85?`8o
0Sl2oCl=?Clm?cdo4SdB?a:o0;on4P:o0;ol4P4k4_/k124k8B4koB44>chQ?S_k
?P57?_]71Dm7Adm?Aoe?0B1?oR0:Cb0084l08000_koB006o_o`00Knom`01_koj
006o_mT00;oe0?^ooP3m_oT0oKom0?foo@00_o800[l00?bo0P00_oL00;og00Fo
AchB_cH0fPBoAchB_oL00;oO0?^o0aJo5[om5QVo5S`F5S`l5S`l5VTlJCaY?6Tl
J@9Y0VToJO`21Sl2?`8o0Sgl?`0moSl84SlB?a8B_`2ooQ88_`2o4S/B4S/BoC/5
8C/Q4R4koB48?R4n>dLQ?ShknckkAoi?0TM?8?i?nR0:04l0800P04l00;oC006o
_o000KnonP01_koI002om@01_kok00Fo_`00_kok00fo_`00_kl00;no002o_oh0
0;on00:o_`3n_o/00Kl0n[l10;og002om`05_dln4[l20?h4_dln4[og002og`3m
_a@F_aJo5QJo5QJoJKll5QHl5S`F?6Wn??iY2C`2?6UY0VT2?fWl0Qlo0Sl2?@82
?cdo?Clo4Sdo4SlB?a:o0;lm4Q:o0;lk4_Tk0Q8Q>oXQ2c/n8DLk?ShkAchn8Ohn
oDL3CdM?Ao]?0B1?nb0;C`0P04l0800P002oe001_kol006o_oh0o;l1003l_oh0
oKl1003l_od0oKom0?joo@3n_oh00Knoo000_oD00Knon`05_kl00;non`0=_kl0
0;no002o_`00_kon00Vo002o002o002on`00_oh0o[on002om`00_oL01Km7?Q:o
004G1;m7?Q:om`00_n00oKl55[lF_aJonaHI?;lF?1Hl5SaY?1HlJC`25VT2J@9Y
JCmYJCol0QLo0Sl2?`8m?`8m?clm0Q8o?a8o4Sno0;on4P:o0;on4P4k4_Xk124Q
>b4ko246>b57?S]?>o/noTL8CdM?Adm7CdlPoTon811?020P04L082008000C`00
_mD00Knoo00A_kl00;no002o_`00_kl00;nooP01_kon01no_`00_kl00;no002o
_`00_kl00;l00;no002o002o_o`00;oe0?booP05_kl00;non`0=_kl00;no002o
_`00_kon00Ro002o002o0;oj002ooP3n_oh00;og002om`05_dLn4[l20@T4_dLn
4[og002oh03k_b<F_aHF_aJo5QJo?1Hl_c`F?3`F?6TlJCaYJ@8F0VUY0VT20VWm
0Q8o0Sl2?@8m0Slm?clm?a824SdooQ8<_`2o4SdB_`2o4Q8k4_/k0R4k>o`Q1c/n
8ChQ?R4knSh1AckmA`9?AdOmC`=7821?oB0:Cb00C`0P0200023m002oe@01_kol
016o_`00_kl00;no002o_`00_kon006o_oh03kno002o_`00_kl00;no003n_od0
o[of002ol@09_kl00;no002o_o/03Kno002o_`00_kl00;nooP00_oh01;no002o
nP00_oP00;og002om`05_dln4[l20@D4_dln4[og002oh03l_`<F_aJonaH3?;ll
5_hl4QHl?1HlJCaY?6TlJFT2J@9Y?fWm0PTo0P8o0P8o0Sd2oCl;4P8B?clB?a8o
_`2ooQ85_`2o4S/BoS/64S/k8C/k8OhkoB44>b4n8B7n?P<k?ShQoSklA`]?CdM?
AdlPCdLPCdom80M?04lP800PCoh00R00_mD00Knoo00A_kl00;no002o_`00_kl0
0;nooP01_kon00No_`00_kl00?^ooP3n_od0o[og002ol@09_kl00;no002o_o/0
3Kno002o_`00_kl00;nooP00_o/00;oi002on000_oL00;og00FoAchB_`00o`Bo
AchB_oL00;oP0?^o8AJo5[lF_aHF_aHl5QHl_c`F?6Tl5SaY?6T2JFT2J@9Y?fWm
0P8oJCon0_do03gn?``m?cdo4Q8o4SlB_`2ooQ87_`2o>a8B>a;k>`4Q>o/Q2C]7
8C/n?R4n>dOm?_Q70Dm7oTl2821?o209021?020080008?d00;oE006o_o`04Kno
002o_`00_kl00;no002o_oh00KnooP09_kl00;no002o_oT0o[om0?jon000_o40
2Kno002o_`00_kok00fo_`00_kl00;no002o_oh00;ok002on@00_oh0o[on002o
m`00_oL01Km7?Q:o004P1;m7?Q:om`00_n00o;l35[lF_o`F1Cbo5S`l_ohl2QHF
JCaY?6T25P8loVT40VUY?fWn0RDo0Sl2?@8o?`82?Clm?clB?clm4SlB4[l0_a8m
4[l0_a8k>a8k4_dk124k8B4koB4?Ac/Q8Chk?R578ChQ?Si7?_i70Dm7o4l1Ab3n
CoPP2P0PC`0P001?002oe@01_kol016o_`00_kl00;no002o_`00_kon006o_oh0
2Kno002o_`00_kon01:o002o002o_`00_`00_kl00;noo000_oD00;on01fo_`00
_kl00;no002o_`00_kl00;no002o_`00_kon002oo004_`00_kol002ooP3n_oh0
0;og002om`05_dln4[mY0?D4_dLn4[og002og`3n_a/F_klF5[no5QJo5QHl_c`l
5S`F?6TFJCaY?1H2oFT40VT2?fWj0QHo0Sl2?clm?cdo?cdo?a8o?a8o4[l0_ohB
1Kl0_c/B4_Pk0B4BnR44Ab4kAc_j?_Y714m7Adm7oDl184on80i?800PCb0004l0
80008;oD0?bo0`00_kon0?bo1`00_kl00;noo@09_kl00;no002o_oh0oKom0?jo
o@3n_oh00Knoo000_o@0oKom0?fooP01_kon0?foo@3m_od00;ol00Fo0;l00;om
006o0?Zo0@2om`00_oL01Km7?Q:oJ@2l1;m??Q:om`00_ml0o;l45[lF5[om5PHl
_aHl5S`FoS`?5SaY?3aYJ@8lJ@9Y0VUY?o/25cl2?`8m?`8o0Slm?clm4Slo4SlB
?kl0_ohB2;l0_a8B>a8k4_`k1B4k8C/Q>ohQ13hk8DLQoSh18B7k?_i70Dm7nTlC
84m?821?0217020084l0801?0;o4006o_l`00;o@002oo@06_`00_`00_od00[l0
0?bo0P00_oL00;og00FoAchB_`00`0BoAchB_oL00;oO0?^o0AJooQHL_aHF_c`F
5Sbo?3`F?6Tl?6UY?08F0SaY0VT20VWl0PDo0Sl2?`;n?`0moSlC?Clo4SlB4SlB
_`2o4SdB_`2o4Q;m>`0BoS/38C/Q>o/Q1S]78ChQ?S_k?P97?SknA`A?AdM?Aoi?
0B1?o20<Cb00C`0P0200022o_lL00;on006o_l`00;o@002oo@00_oh00Knoo000
_oh0o[on002om`00_oL01Km??Q:o0P0k1;m7?Q:om`00_m/0`Kl00?bo002o_l@0
oKo;002od000_o800;om002oo@00_oL00;og00FoAchB_c/0=@BoCchB_oL00;nL
00:o0;on00:o0;n10<T00;o@002ol@00_oX00;of002om`05_dLn4[l00344_dLn
4[og002oW002_`00o[l2002oP@39002od000_o00n[oe002om`05_dln4[l002T4
_dLn4[og002oW000_oX00;n10<T00;o@002og@00_oL01Km7?Q:o0P0Q1;m??Q:o
m`00_i/0n[n10<P0c[oM002om`05_dLn4[mY01/4_dLn4[og002oP@2108@00;og
00FoCchB_`006`BoAchB_oL00;n10840Q000_oL01Km7?Q:o8@0K1;m7?Q:om`00
_h40P@24002om`05_dLn4[l201/4_dln4[og002oP@2108@00;og00FoCchB_`00
6`BoAchB_oL00;n10840Q000_oL01Km7?Q:o000K1;m7?Q:om`00_h40P@24002o
m`05_dLn4[lk01/4_dln4[og002oP@2108@00;og00FoCchB_c/06`BoAchB_oL0
0;n10840Q000_oL01Km7?Q:o000K1;m7?Q:om`00_h40P@24002om`05_dLn4[l0
01/4_dln4[og002oP@2108@00;og00FoCchB_a805`BoAchB_oL0PKn1_h:om`05
_dLn4[l001D4_dLn4[n10840P@3^00FoAchB_`005@BoAchB_h40P@210>h01Km?
?Q:o>`0E1;m??Q:oP@210840kP05_dLn4[l201D4_dLn4[n10840P@3^00FoAchB
_`005@BoAchB_h40P@210>h01Km??Q:o000E1;m??Q:oP@210840kP05_dLn4[le
01D4_dLn4[n10840P@3^00FoAchB_`005`BoAchB_h40P@310;VomP05_dln4[l0
01L4_dln4[n10840``2e_oP01Km7?Q:o>`0G1;m7?Q:oP@210<@0/koi00FoAchB
_a806`BoAchB_h40P@350?bo^`3l_oX01Km??Q:o000S1;m??Q:om02K_hT0_Kod
0?jooP2m_oh0o[oj00FoAchB_`00<`BoAchB_oH00KnoV`01_kn=006o_kd00Kno
m`3m_`80_knm00:o_`3m_o/01Km7?Q:o>`0_1;m7?Q:omP00_iT00;n=002o^`00
_oL0o[l2002o^`02_`00o[ok00FoCchB_`00;@BoCchB_oL00;nG002oS`00_kT0
0;oh0?jo0@2o^@01_`3n_o/01Km7?Q:o000]1;m7?Q:om`00_iL00;n?002o^@00
_oP0o[l10;ni006o0?jon`05_dLn4[lk05X4_dLn4[og002om`00_oT00;oT0?fo
oP01_koc006o_n400;n?002ol@3m_ml00Knol@00_oP0o[l10;oT0?fooP01_kon
006o_n@00Kl0o[ok00FoCchB_fT0H`BoAchB_oL00;og006o_o/00Knoi@01_kon
00Bo002o_m400;n?002olP01_kon002oh001_koa002on03n_`40_nD03Kno002o
_`00_kl00;noh`01_`3n_o/01Km7?Q:o002=1;m??Q:om`00_oL0o[om0?jooP3m
_oh01[no0;no003m_oX00Knon`3l_od0oKon006o_oh0o[om0?foo@3n_oH00;n?
002olP01_koj0?fooP3l_od0o[om0?fooP01_koa002on03n_`40_nD03;no002o
_`00_kl0_koR006o0?jon`05_dLn4[l00;84_dLn4[og002om`3m_`400?fo2@00
_kl00;no003n_oh01Kno002o_o/00Knon`0W_kl00;no002o_`00_kl00;no002o
_`00_`00_kl00;no002o_`00_oL00;n?002olP01_kok002ooP0L_kl00;no002o
_`00_kl00;l00;no002o_`00_koa002on03n_`40_nD01kno002o_`00oKoQ006o
0?jon`05_dln4[lg0:P4_dLn4[og002om`01_`3l_`h0_kl00;no002o_`00_kom
00Fo_`00_kok006o_o/05Kno002o_`00_kl00;no002o_`00_kol00No_`00_kl0
0?jom@00_hl00;ob006o_oX0o;l;002o_`00_kl00;noo009_kl00;no002o_o40
0;oh0?jo0@2oi@07_kl00;no003n_n000Kl0o[ok00FoAchB_`80X0BoCchB_oL0
0;og00:o003n_`l00;no002o_`00_kl00;noo@3k_o/00Knon`0E_kl00;no002o
_`00_kl00;no002o_o`0nkon0?jomP00_hl00;ob006o_o/04Kno002o_`00_kl0
0;no002o_o`0nkl3002o_o400;oh0?jo0@2oi@07_kl00;no003m_n400Kl0o[ok
00FoAchB_`00YPBoAchB_oL00;og002ooP00_oh03Kno002o_`00_kl00;noo@01
_kog006o_o/05Kno002o_`00_kl00;no002o_`00_kol006o_oT0o[og002oS`00
_o800Knon`0A_kl00;no002o_`00_kl00;noo001_kok006o_o400;oh0?jo0@2o
i@0<_kl00;no002o_`2o_n800Kl0o[ok00FoCchB_fT0^`BoAchB_oL00;og002o
nP0=_kl00;no002o_`00_kom006o_oh00;ok006o_oh07kl00;no002o_`00_kl0
0;no002o_`00_kl00;l00;nooP07_`00_`00_kog002oS`00_o800KnooP0K_`00
_kl00;no002o_`00_kl00;no002o002o_oh01;l00;nol@00_oP0o[l10;oU00fo
_`00_kl00;no002o_n<00Kl0o[ok00FoAchB_c/0T@BoCchB_oL00;og002onP01
_kon0?fooP01_kol0?fon@3m_oh01Kno002o_oh0oKon006o_oh0o[om0?foo@3n
_oH00;n?002ol@3m_od0o;l7002o_`00_kon0?joo@3m_oh00Knol@00_oP0o[l1
0;oT0?fooP01_kon006o_n@00Kl0o[ok00FoAchB_`00;@BoAchB_oL00;nG002o
S`00_kT00;oh0?jo0@2o^@01_`3n_o/01Km??Q:o000]1;m7?Q:om`00_iL00;n?
002o^@00_oP0o[l10;ni006o0?jon`05_dLn4[l202d4_dLn4[og002oU`00_hl0
0;ni002on03n_`40_kT00Kl0o[ok00FoAchB_`80;`BoCchB_oH00;nI002oS@00
_k/00;og0?jo0P00_k/00[l00?jon`05_dln4[l003<4_dLn4[of006o_i/00Kno
S@01_knm006o_oL0oKl20;no_@02_kl0oKok00FoAchB_`808`BoAchB_o@0Vkn9
0;fom03n_oh0_Kon0?jonP05_dLn4[lB01/4_dln4[n10840a@3l_k/0o;oj00Fo
CchB_kl05`BoAchB_h40P@340;>on@05_dLn4[l001L4_dLn4[n10840``2e_oP0
1Km7?Q:o0P0G1;m??Q:oP@210<40^Kof00FoCchB_a805@BoAchB_h40P@210>h0
1Km7?Q:o000E1;m7?Q:oP@210840kP05_dLn4[lP01D4_dLn4[n10840P@3^00Fo
CchB_`805@BoCchB_h40P@210>h01Km7?Q:o000E1;m7?Q:oP@210840kP05_dLn
4[l001D4_dM74[n10840P@3^00FoCchB_c/05@BoAchB_h40P@210>h01Km7?Q:o
0P0E1;m??Q:oP@210840kP05_dLn4[l001D4_dLn4[n10840P@3^00FoCchB_`00
5@BoAchB_h40P@210>h01Km7?Q:o>`0C0km7?Q:1_h6oPKo/_`A7?Q:o006ROkm?
?Q97Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7
Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7
Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7OdM?AdM?AdM?AdM?AdM?AdM?
AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?
AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?AdM?
AdM?AdM?AdM?AdM?AdM?OdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7
CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7
CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7CdM7
7Dm7Adm7Adm7Adm7Adm7Adm7Adm7Adm7CdM??Q:o006R0Km7oSioAchnAchnAchn
AchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchn
AchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchnAchn
AchnAchnAchnAchnAchnAchnAchnAcio?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn
?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn
?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn?TLn
?TLn?TLn?TMo?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7
?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7
?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Si7?Sh?AchnAchn
AchnAchnAchnAohn1DLn?Q:o4P0>0;n14X4BPA;W4P8m_`802X6oPKn1_nBo0000
0:00P`3o\>"], "Graphics",
  ImageSize->{413, 303},
  ImageMargins->{{35, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}}]
}, Closed]],

Cell[CellGroupData[{

Cell["External Packages", "Subsection"],

Cell[TextData[{
  "Loading the package ",
  StyleBox["Graphics`Colors`",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " provides definitions for three alternate color designation functions and \
194 English names for colors and shades of gray. Once loaded, we can use \
these color names in place of the function calls defined earlier. ",
  StyleBox["AllColors",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is a list of all the defined colors. "
}], "Text"],

Cell["Needs[\"Graphics`Colors`\"]", "Input"],

Cell[BoxData[
    \(Length[AllColors]\)], "Input"],

Cell["Short[AllColors, 10]", "Input"],

Cell[TextData[{
  "Here is the ",
  StyleBox["RGBColor", "Input"],
  " value of the color ",
  StyleBox["VanDykeBrown", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(VanDykeBrown\)], "Input"],

Cell[TextData[{
  StyleBox["CMYColor[", "Input"],
  StyleBox["c",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["m",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["y",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " represents a color in the CMY (cyan-magenta-yellow) system, ",
  StyleBox["HLSColor[", "Input"],
  StyleBox["h",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["l",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["s",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " represents a color in the HLS  (hue-lightness-saturation) system, and ",
  StyleBox["YIQColor[", "Input"],
  StyleBox["y",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["i",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["q",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " represents a color in the YIQ (NTSC video form) system. These functions \
simply return the matching ",
  StyleBox["RGBColor", "Input"],
  " specification."
}], "Text"],

Cell[BoxData[
    \(\(Show[\ 
      Graphics[\n
        \t\t{PointSize[0.15], \n\t\t\tHLSColor[0.5, 0.7, 0.23], 
          Point[{0, 0}], \n\t\t\tCMYColor[0.2, 0.3, 0.6], Point[{1, 0}], \n
          \t\t\tYIQColor[0.5, \(-0.1\), \(-0.05\)], Point[{2, 0}]}], \n\t\t\t
      PlotRange -> {{\(-1\), 3}, Automatic}]; \)\)], "Input"],

Cell[TextData[{
  "The package ",
  StyleBox["Graphics`ArgColors`", "Input"],
  " defines three more color utilities; ",
  StyleBox["ArgColor", "Input"],
  ", ",
  StyleBox["ArgShade", "Input"],
  ", and  ",
  StyleBox["ColorCircle", "Input"],
  ". ",
  StyleBox["ArgColor[", "Input"],
  StyleBox["z",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " gives a color value whose hue is proportional to the argument of the \
complex number ",
  StyleBox["z",
    FontSlant->"Italic"],
  ", ",
  StyleBox["ArgShade[", "Input"],
  StyleBox["z",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " gives a gray level proportional to the argument of the complex number ",
  StyleBox["z",
    FontSlant->"Italic"],
  ", and ",
  StyleBox["ColorCircle[", "Input"],
  StyleBox["r",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " (",
  StyleBox["light",
    FontSlant->"Italic"],
  ":1)",
  StyleBox["]", "Input"],
  " gives a color value whose hue is proportional to ",
  StyleBox["r",
    FontSlant->"Italic"],
  " (mod 2 \[Pi]) with lightness ",
  StyleBox["light",
    FontSlant->"Italic"],
  "."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`ArgColors`\>"]\)], "Input"],

Cell[BoxData[
    \(ArgColor[4\  + \ 12  I]\)], "Input"],

Cell[TextData[{
  "Remember the definition of the color wheel from above. ",
  StyleBox["ColorCircle", "Input"],
  " performs the same tranformation and allows a simpler definition."
}], "Text"],

Cell[BoxData[
    \(ColorWheel[n_] := 
      Show[Graphics[
          \(({Hue[#1\/\(2\ \[Pi] - n\)], Disk[{0, 0}, 1, {#1, #1 + n}]}&)\)/@
            Range[0, 2\ \[Pi] - n, n], AspectRatio \[Rule] Automatic]]\)], 
  "Input",
  CellTags->"Hue"],

Cell[BoxData[
    \(ColorWheel2[n_] := 
      Show[Graphics[
          \(({ColorCircle[#1], Disk[{0, 0}, 1, {#1, #1 + n}]}&)\)/@
            Range[0, 2\ \[Pi] - n, n], AspectRatio \[Rule] Automatic]]\)], 
  "Input",
  CellTags->"Hue"],

Cell[BoxData[
    \(\(wheel2\  = \ ColorWheel2[\[Pi]/128]; \)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["6. Graphics Primitives and Directives", "Section"],

Cell[CellGroupData[{

Cell["Overview", "Subsection"],

Cell[TextData[{
  "All ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " graphics are built from a small number of graphics primitives and \
directives. The user may build arbitrary graphics in two and three dimensions \
from these basic building blocks of points, lines, circles, and so on. \
Graphics primitives define shapes, while graphics directives control the way \
the shape is drawn."
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
The following syntaxes are used to render a series of two- and \
thee-dimensional graphics, respectively.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[BoxData[
    RowBox[{
      RowBox[{
        StyleBox["Show",
          "Input",
          FontFamily->"Courier",
          FontWeight->"Bold"], 
        StyleBox["[",
          "Input",
          FontFamily->"Courier",
          FontWeight->"Bold"], 
        StyleBox[" ",
          "Input",
          FontFamily->"Courier",
          FontWeight->"Bold"], 
        RowBox[{
          StyleBox["Graphics",
            "Input",
            FontFamily->"Courier",
            FontWeight->"Bold"], 
          StyleBox["[",
            "Input",
            FontFamily->"Courier",
            FontWeight->"Bold"], 
          StyleBox[" ",
            "Input",
            FontFamily->"Courier",
            FontWeight->"Bold"], 
          RowBox[{
            StyleBox["{",
              "Input",
              FontFamily->"Courier",
              FontWeight->"Bold"], 
            StyleBox[" ",
              "Input",
              FontFamily->"Courier",
              FontWeight->"Bold"], 
            StyleBox[\(series\ of\ graphics\ primitives\ and\ directives\),
              "Text",
              FontFamily->"Times",
              FontWeight->"Plain",
              FontSlant->"Italic"], " ", 
            StyleBox["}",
              "Text"]}], 
          StyleBox[" ",
            "Input",
            FontFamily->"Courier",
            FontWeight->"Bold"], 
          StyleBox["]",
            "Input",
            FontFamily->"Courier",
            FontWeight->"Bold"]}], 
        StyleBox[" ",
          "Input",
          FontFamily->"Courier",
          FontWeight->"Bold"], 
        StyleBox["]",
          "Input",
          FontFamily->"Courier",
          FontWeight->"Bold"]}], 
      StyleBox[" ",
        FontFamily->"Courier",
        FontWeight->"Bold"]}]], "Input",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[BoxData[
    RowBox[{
      RowBox[{
        StyleBox["Show",
          "Input",
          FontFamily->"Courier",
          FontWeight->"Bold"], 
        StyleBox["[",
          "Input",
          FontFamily->"Courier",
          FontWeight->"Bold"], 
        StyleBox[" ",
          "Input",
          FontFamily->"Courier",
          FontWeight->"Bold"], 
        RowBox[{
          StyleBox["Graphics3D",
            "Input",
            FontFamily->"Courier",
            FontWeight->"Bold"], 
          StyleBox["[",
            "Input",
            FontFamily->"Courier",
            FontWeight->"Bold"], 
          StyleBox[" ",
            "Input",
            FontFamily->"Courier",
            FontWeight->"Bold"], 
          RowBox[{
            StyleBox["{",
              "Input",
              FontFamily->"Courier",
              FontWeight->"Bold"], " ", 
            StyleBox[\(series\ of\ graphics\ primitives\ and\ directives\),
              "Text",
              FontFamily->"Times",
              FontWeight->"Plain",
              FontSlant->"Italic"], 
            StyleBox[" ",
              "Text",
              FontFamily->"Times",
              FontWeight->"Plain"], 
            StyleBox["}",
              "Text"]}], 
          StyleBox[" ",
            "Input",
            FontFamily->"Courier",
            FontWeight->"Bold"], 
          StyleBox["]",
            "Input",
            FontFamily->"Courier",
            FontWeight->"Bold"]}], 
        StyleBox[" ",
          "Input",
          FontFamily->"Courier",
          FontWeight->"Bold"], 
        StyleBox["]",
          "Input",
          FontFamily->"Courier",
          FontWeight->"Bold"]}], 
      StyleBox[" ",
        FontFamily->"Courier",
        FontWeight->"Bold"]}]], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Two-Dimensional Graphics Primitives", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[TextData[{
  "The primitives ",
  StyleBox["Point",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["Line",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["Rectangle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["Polygon",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["Circle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["Disk",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["Raster", "Input"],
  ", ",
  StyleBox["RasterArray", "Input"],
  ", and ",
  StyleBox["Text",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " form the basis for all two-dimensional graphics. "
}], "Text"],

Cell[TextData[{
  StyleBox["Circle[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["{", "Input"],
  StyleBox["x",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["y",
    FontSlant->"Italic"],
  StyleBox["},", "Input"],
  " ",
  StyleBox["r",
    FontSlant->"Italic"],
  " ",
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is a two-dimensional graphics primitive that represents a circle of \
radius ",
  StyleBox["r",
    FontSlant->"Italic"],
  " centered at the point ",
  StyleBox["{", "Input"],
  StyleBox["x",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["y",
    FontSlant->"Italic"],
  StyleBox["}", "Input"],
  ". ",
  StyleBox["Circle[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["{", "Input"],
  StyleBox["x",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["y",
    FontSlant->"Italic"],
  StyleBox["},", "Input"],
  " ",
  StyleBox["{", "Input"],
  StyleBox["rx",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ry",
    FontSlant->"Italic"],
  StyleBox["}", "Input"],
  StyleBox["]",
    FontWeight->"Bold"],
  " yields an ellipse with semi-axes ",
  StyleBox["rx",
    FontSlant->"Italic"],
  " and ",
  StyleBox["ry",
    FontSlant->"Italic"],
  ". ",
  StyleBox["Circle[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["{", "Input"],
  StyleBox["x",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["y",
    FontSlant->"Italic"],
  StyleBox["},", "Input"],
  " ",
  StyleBox["r",
    FontSlant->"Italic"],
  ", ",
  StyleBox["{", "Input"],
  StyleBox["theta1",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["theta2",
    FontSlant->"Italic"],
  StyleBox["}]", "Input"],
  " represents a circular arc. ",
  StyleBox["Line[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["{", "Input"],
  StyleBox["pt1",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["pt2",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " \[Ellipsis]",
  StyleBox["}", "Input"],
  " ",
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is a graphics primitive that represents a line joining a sequence of \
points. ",
  StyleBox["Text[", "Input"],
  StyleBox["expr",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["coords",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " is a graphics primitive that represents text corresponding to the printed \
form of ",
  StyleBox["expr",
    FontSlant->"Italic"],
  ", centered at the point specified by ",
  StyleBox["coords",
    FontSlant->"Italic"],
  ". Other primitives behave similarly."
}], "Text"],

Cell[BoxData[
    \(\(Show[
      Graphics[{\n\t\t\t
          Polygon[{{0, 0}, {1, 1}, {0, 2}, {\(-1\), 1}, {0, 0}}], \n\t\t\t
          Line[{{0, 0}, {1, 1}, {0, 2}, {\(-1\), 1}, {0, 0}}], \n\t\t\t
          Line[{{0, 0}, {0, 2}}], \n\t\t\tCircle[{0, 0}, 1], 
          Circle[{0, 0}, \@2], \n\t\t\tLine[{{\(-2\), \(-2\)}, {2, 2}}], 
          Line[{{\(-2\), 2}, {2, \(-2\)}}], \n\t\t\t
          Line[{{\(-2\), 0}, {2, 4}}], Line[{{\(-2\), 4}, {2, 0}}], \n\t\t\t
          Line[{{0, 0}, {1, 0}, {1, 1}, {0, 1}, {0, 0}}], \n\t\t\t
          Text["\<Duplicating\>", {\(-1\), \ 1.85}], \n\t\t\t
          Text["\<the Square\>", {\(-1\), 1.7}]}], \n\tAxes -> True, 
      AxesOrigin -> {0, 0}, AspectRatio -> Automatic, 
      PlotRange -> {Automatic, {\(-0.2\), 2.2}}]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["Raster", "Input"],
  " and ",
  StyleBox["RasterArray", "Input"],
  " provide a mechanism for representing a rectangular array of gray or \
colored cells, respectively."
}], "Text"],

Cell[BoxData[
    \(\(rasterarrayexample = 
      Table[Hue[0.1\ Mod[i, j]], {i, 2, 10}, {j, 2, 10}]; \)\)], "Input",
  CellTags->"RasterArray"],

Cell[BoxData[
    \(\(Show[Graphics[RasterArray[rasterarrayexample]]]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Graphics Directives", "Subsection"],

Cell[TextData[{
  "The directives ",
  StyleBox["Thickness", "Input"],
  ", ",
  StyleBox["PointSize",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox[",",
    FontFamily->"Courier"],
  " and ",
  StyleBox["Dashing",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", and any valid color specification are the basic means of changing the \
way the graphics primitives are drawn. Interspersing these with the graphics \
primitives allows us to create any configuration necessary."
}], "Text"],

Cell[TextData[{
  StyleBox["Thickness[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["r",
    FontSlant->"Italic"],
  " ",
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is a graphics directive that specifies that lines that follow are to be \
drawn with a thickness ",
  StyleBox["r",
    FontSlant->"Italic"],
  ", where ",
  StyleBox["r",
    FontSlant->"Italic"],
  " is given as a fraction of the total width of the graph. ",
  StyleBox["Dashing[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["{", "Input"],
  Cell[BoxData[
      \(TraditionalForm\`r\_1\)]],
  StyleBox[",", "Input"],
  " ",
  Cell[BoxData[
      \(TraditionalForm\`r\_2\)]],
  StyleBox[",", "Input"],
  " \[Ellipsis]",
  StyleBox["} ]", "Input"],
  " is a two-dimensional graphics directive which specifies that lines which \
follow are to be drawn dashed, with successive segments of lengths ",
  Cell[BoxData[
      \(TraditionalForm\`r\_1\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`r\_2\)]],
  ", \[Ellipsis] (repeated cyclically). The ",
  Cell[BoxData[
      \(TraditionalForm\`r\_i\)]],
  " are given as a fraction of the total width of the graph. For each \
directive that scales sizes relative to the size of the graphic, there is a \
corresponding directive, for example, ",
  StyleBox["AbsoluteThickness", "Input"],
  ", allowing us to exactly specify the size or thickness in printer\
\[CloseCurlyQuote]s points, approximately equal to ",
  Cell[BoxData[
      \(TraditionalForm\`1\/72\ \ \(inch.\)\)]]
}], "Text"],

Cell[BoxData[
    \(\(Show[
      Graphics[{\n
          \t\t\t{GrayLevel[0.5], \n\t\t\t\t
            Polygon[{{0, 0}, {1, 1}, {0, 2}, {\(-1\), 1}, {0, 0}}]}, \n
          \t\t\t{Hue[0], Thickness[0.01], \n\t\t\t\t
            Line[{{0, 0}, {1, 1}, {0, 2}, {\(-1\), 1}, {0, 0}}]}, \n\t\t\t\t
          Line[{{0, 0}, {0, 2}}], \n
          \t\t\t{Dashing[{0.01}], \n\t\t\t\tCircle[{0, 0}, 1], 
            Circle[{0, 0}, \@2], \n\t\t\t\tLine[{{\(-2\), \(-2\)}, {2, 2}}], 
            Line[{{\(-2\), 2}, {2, \(-2\)}}], \n\t\t\t\t
            Line[{{\(-2\), 0}, {2, 4}}], Line[{{\(-2\), 4}, {2, 0}}]}, \n
          \t\t\t{Thickness[0.01], \n\t\t\t\t
            Line[{{0, 0}, {1, 0}, {1, 1}, {0, 1}, {0, 0}}]}, \n
          \t\t\t{Hue[0.65], \n\t\t\t\t
            Text["\<Duplicating\>", {\(-1\), \ 1.85}], \n\t\t\t\t
            Text["\<the Square\>", {\(-1\), 1.7}]}}], \n\tAxes -> True, 
      AxesOrigin -> {0, 0}, AspectRatio -> Automatic, 
      PlotRange -> {Automatic, {\(-0.2\), 2.2}}]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Additional Primitives for Three Dimensions", "Subsection"],

Cell[TextData[{
  "When creating three-dimensional graphics most of the same primitives apply \
providing we specify points in the appropriate way. ",
  StyleBox["Cuboid[{", "Input"],
  StyleBox["x",
    FontSlant->"Italic"],
  Cell[BoxData[
      FormBox[
        StyleBox["min",
          FontSlant->"Italic"], TraditionalForm]]],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ymin",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["zmin",
    FontSlant->"Italic"],
  StyleBox["}]", "Input"],
  " is a three-dimensional graphics primitive that represents a unit cuboid, \
oriented parallel to the axes. ",
  StyleBox["Cuboid[{", "Input"],
  StyleBox["xmin",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ymin",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["zmin",
    FontSlant->"Italic"],
  StyleBox["},", "Input"],
  " ",
  StyleBox["{", "Input"],
  StyleBox["xmax",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ymax",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["zmax",
    FontSlant->"Italic"],
  StyleBox["}", "Input"],
  StyleBox["]", "Input"],
  " specifies a cuboid by giving the coordinates of opposite corners."
}], "Text"],

Cell[BoxData[
    \(\(Show[
      Graphics3D[{\n\t\t\tCuboid[{0, 0, 0}], Cuboid[{5, 0, 0}], 
          Cuboid[{0, 5, 0}], Cuboid[{0, 0, 5}], Cuboid[{5, 0, 5}], 
          Cuboid[{5, 5, 0}], Cuboid[{0, 5, 5}], Cuboid[{5, 5, 5}], 
          Line[{{0.5, 0.5, .5}, {5.5, 5.5, 5.5}}], 
          Line[{{5.5, 0.5, .5}, {.5, 5.5, 5.5}}], 
          Line[{{0.5, 0.5, 5.5}, {5.5, 5.5, 0.5}}], 
          Line[{{5.5, 0.5, 5.5}, {.5, 5.5, 0.5}}], 
          Polygon[{{1, 1, 0}, {5, 1, 0}, {5, 5, 0}, {1, 5, 0}, {1, 1, 0}}], {
            PointSize[0.05], Point[{3, 3, 3}]}}], \nBoxed -> False]; \)\)], 
  "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Additional Directives for Three Dimensions", "Subsection"],

Cell[TextData[{
  StyleBox["SurfaceColor[", "Input"],
  StyleBox["col",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " is a three-dimensional graphics directive specifying that the polygons \
that follow should act as diffuse reflectors of light with a color given by ",
  
  StyleBox["col",
    FontSlant->"Italic"],
  ". "
}], "Text"],

Cell[BoxData[
    \(\(Show[
      Graphics3D[\n\t\t
        Table[{SurfaceColor[Hue[Random[]]], \n\t\t\t
            Cuboid[\n\t\t\t\t{Random[Integer, {1, 15}], \n\t\t\t\t\t
                Random[Integer, {1, 15}], \n\t\t\t\t\t
                Random[Integer, {1, 15}]}]}, \n\t\t{50}]]]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["EdgeForm[", "Input"],
  StyleBox["g",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " is a three-dimensional graphics directive that specifies that edges of \
polygons are to be drawn using the graphics directive or list of graphics \
directives ",
  StyleBox["g",
    FontSlant->"Italic"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(Show[\n\t
      Graphics3D[{\n\t\t\tEdgeForm[Thickness[0.02]], 
          Polygon[{{0, 0, 0}, {0, 1, 1}, {1, 1, 1}}], 
          EdgeForm[{Thickness[0.01], Hue[0]}], 
          Polygon[{{0, 0, 0}, {1, 1, 0}, {1, 1, 1}}]}]]; \)\)], "Input",
  CellTags->"EdgeForm"],

Cell[TextData[{
  StyleBox["FaceForm[", "Input"],
  StyleBox["gf",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["gb",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " is a three-dimensional graphics directive that specifies that the front \
faces of all polygons are to be drawn with the graphics primitive ",
  StyleBox["gf",
    FontSlant->"Italic"],
  ", and the back faces with ",
  StyleBox["gb",
    FontSlant->"Italic"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(view1\  = \ 
      Graphics3D[{\n\t\t\tFaceForm[Hue[0], RGBColor[0, 1, 0]], \n\t\t\t
          Polygon[{{0, 0, 0}, \ {0, 1, 1}, \ {1, 0, 1}}]}, \n\t\t\t
        Lighting -> False, \n\t\t\tAxes\  -> \ True, \ Ticks\  -> \ None, \ \n
        \t\t\tAxesLabel\  -> \ {"\<x\>", \ "\<y\>", \ "\<z\>"}, \n\t\t\t
        ViewPoint -> {1.300, \ \(-2.400\), \ 2.000}, ]; \)\)], "Input"],

Cell[BoxData[
    \(\(view2\  = \ 
      Graphics3D[{\n\t\t\tFaceForm[Hue[0], RGBColor[0, 1, 0]], \n\t\t\t
          Polygon[{{0, 0, 0}, \ {0, 1, 1}, \ {1, 0, 1}}]}, \n\t\t\t
        Lighting -> False, \n\t\t\tAxes\  -> \ True, \ Ticks\  -> \ None, \ \n
        \t\t\tAxesLabel\  -> \ {"\<x\>", \ "\<y\>", \ "\<z\>"}, \n\t\t\t
        ViewPoint -> {1.300, \ 2.400, \ 2.000}]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{view1, \ view2}]]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["More about Text", "Subsection"],

Cell[TextData[{
  "The form ",
  StyleBox["Text[", "Input"],
  StyleBox["expr",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["coords",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " will produce text corresponding to the printed form of ",
  StyleBox["expr",
    FontSlant->"Italic"],
  ", centered at the point specified by ",
  StyleBox["coords",
    FontSlant->"Italic"],
  ", but there are several more options available. Third and fourth arguments \
to ",
  StyleBox["Text", "Input"],
  " control the orientation alignment and orientation of the text with the \
form ",
  StyleBox["Text[", "Input"],
  StyleBox["expr",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["coords",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["align",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["orient",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  ". ",
  StyleBox["Align",
    FontSlant->"Italic"],
  " is of the form ",
  StyleBox["{", "Input"],
  StyleBox["a",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["b",
    FontSlant->"Italic"],
  StyleBox["}", "Input"],
  " where ",
  StyleBox["a",
    FontSlant->"Italic"],
  " controls the left",
  Cell[BoxData[
      \(TraditionalForm\`-\)]],
  "right alignment and ",
  StyleBox["b",
    FontSlant->"Italic"],
  " controls the above",
  Cell[BoxData[
      \(TraditionalForm\`-\)]],
  "below alignment. ",
  StyleBox["orient",
    FontSlant->"Italic"],
  " is of the form ",
  StyleBox["{", "Input"],
  StyleBox["a",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["b",
    FontSlant->"Italic"],
  StyleBox["}", "Input"],
  " where ",
  StyleBox["a",
    FontSlant->"Italic"],
  " controls the rightside-up and upside-down alignment and ",
  StyleBox["b",
    FontSlant->"Italic"],
  " controls the horizontal-vertical alignment. Note that rotated text does \
not display correctly but does print correctly."
}], "Text"],

Cell[BoxData[GridBox[{
        {
          StyleBox["offset",
            FontFamily->"Times",
            FontWeight->"Bold",
            FontSlant->"Italic"], 
          StyleBox[
            RowBox[{\(Relation\ to\), "\n", "\t", 
              StyleBox["coords",
                FontSlant->"Italic"]}],
            FontFamily->"Times",
            FontWeight->"Bold"], 
          StyleBox["orient",
            FontFamily->"Times",
            FontWeight->"Bold",
            FontSlant->"Italic"], 
          StyleBox[\(Orientation\ of\ text\),
            FontFamily->"Times",
            FontWeight->"Bold"]},
        {
          StyleBox[\({0, 0}\),
            FontWeight->"Bold"], 
          StyleBox["Centered",
            FontFamily->"Times",
            FontWeight->"Plain"], 
          StyleBox[\({1, 0}\),
            FontWeight->"Bold"], 
          StyleBox[\(Horizontal \((rightside\[Dash]up)\)\),
            FontFamily->"Times",
            FontWeight->"Plain"]},
        {
          StyleBox[\({\(-1\), 0}\),
            FontWeight->"Bold"], 
          StyleBox[\(Left\ Justified\),
            FontFamily->"Times",
            FontWeight->"Plain"], 
          StyleBox[\({0, 1}\),
            FontWeight->"Bold"], 
          StyleBox[\(Vertical \((up)\)\),
            FontFamily->"Times",
            FontWeight->"Plain"]},
        {
          StyleBox[\({1, 0}\),
            FontWeight->"Bold"], 
          StyleBox[\(Right\ Justified\),
            FontFamily->"Times",
            FontWeight->"Plain"], 
          StyleBox[\({0, \(-1\)}\),
            FontWeight->"Bold"], 
          StyleBox[\(Vertical \((down)\)\),
            FontFamily->"Times",
            FontWeight->"Plain"]},
        {
          StyleBox[\({0, \(-1\)}\),
            FontWeight->"Bold"], 
          StyleBox[\(Centered\ Above\),
            FontFamily->"Times",
            FontWeight->"Plain"], 
          StyleBox[\({\(-1\), 0}\),
            FontWeight->"Bold"], 
          StyleBox[\(Horizontal \((upside\[Dash]down)\)\),
            FontFamily->"Times",
            FontWeight->"Plain"]},
        {
          StyleBox[\({0, 1}\),
            FontWeight->"Bold"], 
          StyleBox[\(Centered\ Below\),
            FontFamily->"Times",
            FontWeight->"Plain"], 
          StyleBox["",
            FontWeight->"Bold"], ""}
        },
      GridFrame->True,
      RowLines->True,
      ColumnLines->True]], "Input"],

Cell[BoxData[
    \(\(Show[
      Graphics[\n\t\t
        Table[Text["\<Text\>", \ {i, j}, \n
            \t\t\t\t{Random[Integer, {\(-1\), 1}], 
              Random[Integer, {\(-1\), 1}]}, \n
            \t\t\t\t{Random[Integer, {\(-1\), 1}], 
              Random[Integer, {\(-1\), 1}]}], \n\t\t{i, 1, 5}, {j, 1, 5}]], 
      PlotRange -> {{0, 7}, {\(-1\), 7}}]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["TextStyle", "Input"],
  " is an option for graphics functions and for ",
  StyleBox["Text", "Input"],
  " that specifies the default style and font options with which text should \
be rendered. Options such as ",
  StyleBox["FontSize", "Input"],
  ", ",
  StyleBox["FontSlant", "Input"],
  ", ",
  StyleBox["FontFamily", "Input"],
  ", ",
  StyleBox["FontColor", "Input"],
  StyleBox[",", "Input",
    FontWeight->"Plain"],
  " and ",
  StyleBox["Background", "Input"],
  " are recognized."
}], "Text"],

Cell[BoxData[
    \(\(Show[
      Graphics[Text["\<Hello World\>", {10, 10}, 
          TextStyle 
            \[Rule] {FontFamily \[Rule] "\<Times\>", 
              FontWeight \[Rule] "\<Bold\>", FontSize \[Rule] 20, 
              FontColor \[Rule] Hue[0.9], FontSlant \[Rule] "\<Italic\>", 
              Background \[Rule] Hue[0.4]}]]]; \)\)], "Input"],

Cell[TextData[{
  "The same effect can be achieved using ",
  StyleBox["StyleForm", "Input"],
  ". ",
  StyleBox["StyleForm[", "Input"],
  StyleBox["expr",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["options",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " prints using the specified style options. ",
  StyleBox["StyleForm[", "Input"],
  StyleBox["expr",
    FontSlant->"Italic"],
  StyleBox[", \"", "Input"],
  StyleBox["style",
    FontSlant->"Italic"],
  StyleBox["\"]", "Input"],
  " prints using the specified cell style in the current notebook."
}], "Text"],

Cell[BoxData[
    \(Table[
      StyleForm["\<text\>", FontFamily \[Rule] "\<Times\>", 
        FontSize \[Rule] s], {s, 10, 20}]\)], "Input",
  CellTags->"S2.8.9"],

Cell[BoxData[
    \(StyleForm[\ "\<Hello World\>", \t"\<Title\>"]\)], "Input"],

Cell[TextData[{
  "Among the aspects of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " we will not discuss in this course is its typesetting capabilities. ",
  StyleBox["Display", "Input"],
  " allows us to take typeset expressions and display them as graphics. ",
  StyleBox["stdout", "Input"],
  " directs the output to the front end."
}], "Text"],

Cell[BoxData[
    \(\(Display[$Output, ToBoxes[Nest[1 + 1/#&, x, 7]], "\<MPS\>"]; \)\)], 
  "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Exercises: Graphics Primitives and Directives", "Subsection",
  PageBreakAbove->True],

Cell[TextData[{
  "Below is the ",
  StyleBox["Mathematica ",
    FontSlant->"Italic"],
  "code to draw a face. Use other two-dimensional graphics commands (",
  StyleBox["e.g.",
    FontWeight->"Plain",
    FontSlant->"Italic"],
  ", ",
  StyleBox["Polygon",
    FontFamily->"Courier"],
  ") to add other features (nose, beard, hat, etc.) and make changes to the \
face (",
  StyleBox["e.g.",
    FontWeight->"Plain",
    FontSlant->"Italic"],
  ", eye color). Experiment and have fun."
}], "Question"],

Cell[BoxData[
    \(\(Show[
      Graphics[{Thickness[0.03], Circle[{0, 0}, 1], PointSize[0.04], 
          Point[{\(-.5\), .3}], Point[{.5, .3}], 
          Circle[{0, \(-0.1\)}, .5, {\[Pi] + \[Pi]\/4, 
              2\ \[Pi] - \[Pi]\/4}]}], AspectRatio \[Rule] Automatic]; \)\)], 
  "Input"],

Cell[BoxData[
    \(\(Show[
      Graphics[{Thickness[0.03], Circle[{0, 0}, 1], Thickness[0.009], Red, 
          Circle[{\(-0.5\), 0.3}, 0.04], Circle[{0.5, 0.3}, 0.04], Blue, 
          Circle[{0, \(-0.1\)}, 
            0.5, {\[Pi] + \[Pi]\/4, 2\ \[Pi] - \[Pi]\/4}], Pink, 
          Polygon[{{\(-0.25\), \(-0.9\)}, {0.25, \(-0.9\)}, {0, \(-1.25\)}}], 
          Green, Polygon[{{\(-0.6\), 0.9}, {0.6, 0.9}, {0.6, 1.1}, {\(-0.6\), 
                1.1}}], 
          Polygon[{{\(-0.4\), 1.1}, {0.4, 1.1}, {.4, 1.5}, {\(-0.4\), 
                1.5}}]}], AspectRatio \[Rule] Automatic]; \)\)], "Input"],

Cell[" ", "Special3"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["7. Displaying and Combining Graphics", "Section"],

Cell[CellGroupData[{

Cell["Show", "Subsection"],

Cell[TextData[{
  "Perhaps the most versatile tool ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " provides for working with graphics is the ",
  StyleBox["Show",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " command. ",
  StyleBox["Show",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", in its simplest form, allows us to rerender a graphic that has already \
been generated. ",
  StyleBox["Show",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " also allows us to change some of the options associated with a graphic \
while redrawing the graphic. In general, we can change those options that don\
\[CloseCurlyQuote]t affect the curve itself, such as the ",
  StyleBox["PlotRange",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(firstplot\  = \ Plot[x\ Sin[x]\^2, \ {x, \ 0, \ 3  \[Pi]}]; \)\)], 
  "Input"],

Cell[BoxData[
    \(\(Show[firstplot, \ 
      PlotRange\  \[Rule] \ {{\(-5\), \ 15}, {\(-5\), \ 10}}]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["Show",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " will render two or more graphics together on the same set of axes, \
scaling them appropriately."
}], "Text"],

Cell[BoxData[
    \(\(secondplot\  = \ 
      Plot[x\ Cos[x]\^2, \ {x, \ 0, \ 3\ \[Pi]}, \n\t\t
        PlotStyle\  -> \ {Dashing[{0.03}], \ Thickness[0.01]}]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[firstplot, secondplot]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["Show", "Input"],
  " works the same with three-dimensional graphics as it does with \
two-dimensional plots."
}], "Text"],

Cell[BoxData[
    \(\(first3d = 
      Plot3D[\((1 - Sin[x])\)\ \((2 - Cos[2\ y])\), {x, \(-2\), 2}, {y, 
          \(-2\), 2}]; \)\)], "Input"],

Cell[BoxData[
    \(\(second3d = 
      Plot3D[\((2 + Sin[x])\)\ \((1 + Cos[2\ y])\), {x, \(-2\), 2}, {y, 
          \(-2\), 2}]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[first3d, \ second3d]; \)\)], "Input"],

Cell[TextData[{
  "Sometimes when generating three-dimensional graphics with ",
  StyleBox["Show", "Input"],
  ", it may be easier to visualize the different plots by giving each plot a \
separate color instead of the the default shading scheme."
}], "Text"],

Cell[BoxData[
    \(\(Show[\n\t
      Plot3D[{\((1\  - \ Sin[x])\)\ \((2\  - \ Cos[2\ y])\), GrayLevel[0.4]}, 
        \n\t\t{x, \ \(-2\), \ 2}, \ {y, \ \(-2\), \ 2}, \ 
        DisplayFunction\  \[Rule] \ Identity], \n\t
      Plot3D[{\((2\  + \ Sin[x])\)\ \((1\  + \ Cos[2\ y])\), \ 
          GrayLevel[0.8]}, \ {x, \ \(-2\), \ 2}, \ {y, \ \(-2\), \ 2}, \ 
        DisplayFunction\  \[Rule] \ Identity], \n\t
      DisplayFunction\  \[Rule] \ $DisplayFunction]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["GraphicsArray", "Subsection"],

Cell[TextData[{
  "Another useful function is ",
  StyleBox["GraphicsArray",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", which allows us to create tables of graphics. It is important to know \
the difference between those functions that create graphics objects, those \
that render them, and those that do both. For example, ",
  StyleBox["Plot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " will both create and render a graphics object, ",
  StyleBox["Show",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " will render a previously computed graphics object, and ",
  StyleBox["GraphicsArray",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " will create a new graphics object from other graphics objects."
}], "Text"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{firstplot, secondplot}]]; \)\)], "Input"],

Cell[TextData[{
  "Using ",
  StyleBox["GraphicsArray",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox[",",
    FontFamily->"Courier"],
  " we are able to create a two-dimensional array of any shape and fill it \
with ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " graphics."
}], "Text"],

Cell[BoxData[
    \(\(Show[
      GraphicsArray[{{firstplot, secondplot}, \n
          \t\t\t{secondplot, firstplot, secondplot}}]]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Rectangle", "Subsection"],

Cell[TextData[{
  "You have seen how to constuct regular arrays of graphics using ",
  StyleBox["GraphicsArray", "Input"],
  ", but you can also combine and superimpose plots in any way with the ",
  StyleBox["Rectangle", "Input"],
  " function."
}], "Text"],

Cell[TextData[{
  StyleBox["Rectangle", "Input"],
  " allows you to render a graphics object within a specified rectangle and \
then to display that rectangle with other rectangles or individually in \
another plot."
}], "Text"],

Cell["First we generate a graphics object.", "Text"],

Cell[BoxData[
    \(\(rectplot\  = \ \n\t
      Plot3D[Sin[x]\ Exp[y], \ {x, \ \(-5\), \ 5}, \ {y, \ \(-2\), \ 2}, \n
        \t\tAxes\  -> \ False]; \)\)], "Input"],

Cell[TextData[{
  "Now we can use ",
  StyleBox["Rectangle", "Input"],
  " and the ",
  StyleBox["Show", "Input"],
  " command to place two copies of a graphic wherever we wish."
}], "Text"],

Cell[BoxData[
    \(\(Show[
      Graphics[{\n\t\t\tRectangle[{0, 0}, \ {1, 1}, \ rectplot], \n\t\t\t
          Rectangle[{0.8, 0.8}, \ {1.2, 1.4}, \ rectplot]}]]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["DisplayFunction", "Subsection"],

Cell[TextData[{
  StyleBox["DisplayFunction",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is an option for every graphics function in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ". With it, we specify the output medium for the image. By default, this \
option is set to the global variable ",
  StyleBox["$DisplayFunction",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox[",",
    FontFamily->"Courier"],
  " which in the notebook front end, will render the graphic inside the \
current notebook. To send the PostScript output to a file use the setting ",
  StyleBox["DisplayFunction",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["\[Rule]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["Display[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["\"", "Input"],
  StyleBox["file",
    FontSlant->"Italic"],
  StyleBox["\"", "Input"],
  StyleBox[", #]&",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[TextData[{
  "For our purposes the setting ",
  StyleBox["DisplayFunction",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["\[Rule]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["Identity",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " will be the most useful. This is because the graphic is computed, but not \
displayed. When you wish to display the graphic, you need to set ",
  StyleBox["DisplayFunction", "Input"],
  " back to the default value, ",
  StyleBox["$DisplayFunction", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(gr1 = 
      Plot[0.1\ x\^3 - 2\ x, {x, \(-10\), 10}, \n\t\t
        PlotStyle \[Rule] {{RGBColor[1, 0, 0], \ Thickness[0.01]}}, \n\t\t
        DisplayFunction \[Rule] Identity]; \n
    gr2 = Plot[x\ Sin[x]\^2, {x, \(-10\), 10}, \n\t\t
        PlotStyle \[Rule] {{GrayLevel[0.7]}}, \n\t\t
        DisplayFunction \[Rule] Identity]; \n
    Show[{gr1, gr2}, \n\t\tDisplayFunction \[Rule] $DisplayFunction]; \)], 
  "Input"],

Cell[TextData[{
  "When we are combining plots, usually we are only interested in the final \
graphic. If we had not set ",
  StyleBox["DisplayFunction", "Input"],
  " to ",
  StyleBox["Identity", "Input"],
  " in the previous example, ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " would have printed each component graphic also."
}], "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell["DisplayTogether and DisplayTogetherArray", "Subsection"],

Cell[TextData[{
  "The package ",
  StyleBox["Graphics`Graphics`", "Input"],
  " which defines the functions ",
  StyleBox["DisplayTogether", "Input"],
  " and ",
  StyleBox["DisplayTogetherArray", "Input"],
  ", allows us to combine graphs on the same set of axes or in an array \
without rendering each graph beforehand. To use the functions we first load \
the package."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`Graphics`\>"]\)], "Input"],

Cell[BoxData[
    \(\(DisplayTogether[\n\t
      Plot[Sin[x], {x, \(-3\), 3}, PlotStyle -> GrayLevel[0.5]], 
      Plot[Sin[3  x], {x, \(-3\), 3}, PlotStyle -> Dashing[{0.01}]]]; \)\)], 
  "Input"],

Cell[BoxData[
    \(\(DisplayTogetherArray[
      Plot[Sin[x], {x, \(-3\), 3}, PlotStyle -> GrayLevel[0.5]], 
      Plot[Sin[3  x], {x, \(-3\), 3}, PlotStyle -> Dashing[{0.01}]]]; \)\)], 
  "Input"],

Cell[TextData[{
  "It is important to note that ",
  StyleBox["DisplayTogether", "Input"],
  " and ",
  StyleBox["DisplayTogetherArray", "Input"],
  " do not work with graphics that we have named. If we have graphics that \
have already been assigned to a symbol name, we should use the ",
  StyleBox["Show", "Input"],
  " function."
}], "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell["Graphics Formats", "Subsection"],

Cell[TextData[{
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " generates graphics in PostScript form. In most ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " implementations you can see and edit the PostScript code directly. This \
form is transportable among all platforms and contains enough information to \
allow viewing and printing at any resolution or size."
}], "Text"],

Cell[TextData[{
  "We can save the PostScript to a file that can be read by many of the \
highest-quality graphics processors. For example, if we wanted to save a \
graphic as an Encapsulated PostScript file (EPS) for use in another graphics \
program, all we would have to do is select the graphic and choose the ",
  StyleBox["Save Selection As...", "MenuItem"],
  " command in the ",
  StyleBox["Edit", "MenuItem"],
  " menu of the front end and then choose which format we want from the \
submenu that appears."
}], "Text"],

Cell[TextData[{
  "The user has the ability to convert the graphic to any one of several \
local formats, depending on the platform that ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " runs on. Usually the results are a loss of generality but savings in \
speed (rendering) and space."
}], "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell["Image Conversions", "Subsection"],

Cell[TextData[{
  StyleBox["Display[", "Input"],
  StyleBox["channel",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["graphics",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  " writes graphics or sound to the specified output channel in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " PostScript format. ",
  StyleBox["Display[", "Input"],
  StyleBox["channel",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["graphics",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["\"", "Input"],
  StyleBox["format",
    FontSlant->"Italic"],
  StyleBox["\"]", "Input"],
  " writes graphics or sound in the specified format. ",
  StyleBox["Display[", "Input"],
  StyleBox["channel",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["expr",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["\"", "Input"],
  StyleBox["format",
    FontSlant->"Italic"],
  StyleBox["\"]", "Input"],
  " writes boxes, cells, or notebook expressions in the specified format. ",
  StyleBox["channel",
    FontSlant->"Italic"],
  " specifies where the image is to be written. Using a filename, such as ",
  StyleBox["graph1.eps",
    FontFamily->"Courier"],
  ", will create a new file. The following formats are available."
}], "Text"],

Cell[BoxData[
    StyleBox[GridBox[{
          {"EPS", 
            StyleBox[\(Encapsulated\ PostScript\),
              FontWeight->"Plain"]},
          {"GIF", 
            StyleBox["GIF",
              FontWeight->"Plain"]},
          {"Illustrator", 
            StyleBox[\(Adobe\ Illustrator\ format\),
              FontWeight->"Plain"]},
          {"Metafile", 
            StyleBox[\(Microsoft\ Windows\ metafile\),
              FontWeight->"Plain"]},
          {"MGF", 
            
            StyleBox[
              \(Mathematica\ system\[Dash]independent\ raster\ graphics\ 
                format\),
              FontWeight->"Plain"]},
          {"MPS", 
            StyleBox[\(Mathematica\ abbreviated\ PostScript\),
              FontWeight->"Plain"]},
          {"PCL", 
            StyleBox[\(Hewlett\[Dash]Packard\ printer\ control\ language\),
              FontWeight->"Plain"]},
          {"PDF", 
            StyleBox[\(Adobe\ Acrobat\ portable\ document\ format\),
              FontWeight->"Plain"]},
          {"PICT", 
            StyleBox[\(Macintosh\ PICT\),
              FontWeight->"Plain"]},
          {"PBM", 
            StyleBox[\(portable\ bitmap\),
              FontWeight->"Plain"]},
          {"PSImage", 
            StyleBox[\(PostScript\ image\ format\),
              FontWeight->"Plain"]},
          {"TIFF", 
            StyleBox["TIFF",
              FontWeight->"Plain"]},
          {"XBitmap", 
            StyleBox[\(X\ Windows\ bitmap\),
              FontWeight->"Plain"]}
          },
        GridFrame->True,
        RowLines->True,
        ColumnLines->True],
      FontFamily->"Times"]], "Input"],

Cell[TextData[{
  "Several different formats specific to other programs are also supported. \
In addition, on all platforms we can export a graphics as a DXF file, AutoCad\
\[CloseCurlyQuote]s standard. This functionality is in a ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " standard add-on package."
}], "Text"],

Cell[TextData[{
  "Enter ",
  StyleBox["Needs[\"Utilities`DXF`\"]", "Input"],
  " to load the package."
}], "Text"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["8. Using Options", "Section"],

Cell[CellGroupData[{

Cell["Plot Options", "Subsection"],

Cell[CellGroupData[{

Cell["Basics", "Subsubsection"],

Cell[TextData[{
  "Many aspects of every ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " graphic are controlled by the values of the options associated with it. \
Striking changes can be made by using these judiciously. ",
  StyleBox["Options[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["command",
    FontSlant->"Italic"],
  " ",
  StyleBox["]", "Input"],
  " lists all options for ",
  StyleBox["command",
    FontSlant->"Italic"],
  " and their defaults."
}], "Text"],

Cell[BoxData[
    \(Options[\ Plot] // TableForm\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["PlotRange and Domain", "Subsubsection"],

Cell[TextData[{
  "Besides the function we wish to plot, the only required parameter to the \
function ",
  StyleBox["Plot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is the domain over which to plot the function. Given that information, ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " will choose a range to display in the final graphic. The ",
  StyleBox["x",
    FontSlant->"Italic"],
  "-range displayed by default will be exactly the specified domain, while \
the ",
  StyleBox["y",
    FontSlant->"Italic"],
  "-range displayed will be one chosen to display what ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " considers to be the interesting parts of the graph. Sometimes, though, \
extreme values will be cut off so that the rest of the graphic may be scaled \
normally."
}], "Text"],

Cell[BoxData[
    \(\(Plot[x\ Sin[x]\^3, {x, \(-10\), 10}]; \)\)], "Input"],

Cell[TextData[{
  "The option ",
  StyleBox["PlotRange",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows us to specify the area of the graph to show. We may set this \
option to ",
  StyleBox["All",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " to ensure that the entire graphic is displayed, or we may choose \
arbitrary coordinates. We may also specify the ",
  StyleBox["x",
    FontSlant->"Italic"],
  " range to display by giving coordinates in the form ",
  StyleBox["{{", "Input"],
  StyleBox["xmin",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["xmax",
    FontSlant->"Italic"],
  StyleBox["}, {", "Input"],
  StyleBox["ymin",
    FontSlant->"Italic"],
  StyleBox[", ", "Input"],
  StyleBox["ymax",
    FontSlant->"Italic"],
  StyleBox["}}", "Input"],
  ". Note that the curve is computed only for the specified domain even if we \
display a larger one using ",
  StyleBox["PlotRange",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(range1 = 
      Plot[x\ Sin[x]\^3, {x, \(-10\), 10}, PlotRange \[Rule] All, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(range2 = 
      Plot[x\ Sin[x]\^3, {x, \(-10\), 10}, PlotRange \[Rule] {\(-10\), 10}, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(range3 = 
      Plot[x\ Sin[x]\^3, {x, \(-10\), 10}, 
        PlotRange \[Rule] {{\(-15\), 15}, {\(-15\), 15}}, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{range1, range2, range3}]]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Adaptive Sampling Algorithm, PlotDivision, and PlotPoints", 
  "Subsubsection"],

Cell[TextData[{
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " uses an adaptive sampling algorithm to choose the points sampled in a \
two-dimensional plot. Beginning with 25 equally spaced points dividing the \
domain to be plotted, ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " looks at each set of three consecutive points and computes the angle \
between the line segment joining the first and second points and the line \
segment joining the second and third points. If this angle is close to 180 \
degrees, then ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " connects the points with lines. If not, ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " subdivides that interval and tries again. This allows ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to sample more points in a \[OpenCurlyDoubleQuote]curvy\
\[CloseCurlyDoubleQuote] section of the function than in a flat section."
}], "Text"],

Cell[TextData[{
  "This process can be controlled by the options ",
  StyleBox["PlotDivision",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", which is the upper limit on the number of times an interval will be \
divided, and ",
  StyleBox["PlotPoints",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", which sets the initial number of points to be sampled."
}], "Text"],

Cell["\<\
Although this is a very robust algorithm, which produces accurate \
results in most cases, any scheme using a finite number of points is prone to \
miss sometimes. Here is an example of a function that is misplotted using the \
default number of plot points, but which can be accurately plotted by raising \
the initial number of plot points used.\
\>", "Text"],

Cell[BoxData[
    \(\(Plot[x + Sin[2\ \[Pi]\ x], {x, 0, 25}]; \)\)], "Input"],

Cell[BoxData[
    \(\(Plot[x + Sin[2\ \[Pi]\ x], {x, 0, 24}, PlotPoints \[Rule] 50]; \)\)], 
  "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["AspectRatio", "Subsubsection"],

Cell[TextData[{
  "The ratio of height to width of the plot is controlled with ",
  StyleBox["AspectRatio",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". The default value is the inverse of the golden ratio, or approximately \
0.618. We may set this to an arbitrary real number for explicit control, or \
use the keyword ",
  StyleBox["Automatic",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " to allow ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to compute a realistic ",
  StyleBox["AspectRatio",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " from the coordinate values in the plot. With this setting, circles and \
other geometric figures will appear proportionally correct. It is the default \
setting, historically the choice of artists and architects for their work, \
and ",
  StyleBox["Automatic",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " that will be most often used."
}], "Text"],

Cell[BoxData[
    \(\(aspect1 = 
      ParametricPlot[{Sin[t], Cos[t]}, {t, 0, 2\ \[Pi]}, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(aspect2 = 
      ParametricPlot[{Sin[t], Cos[t]}, {t, 0, 2\ \[Pi]}, 
        AspectRatio \[Rule] Automatic, DisplayFunction \[Rule] Identity]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{aspect1, aspect2}]]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["PlotStyle", "Subsubsection"],

Cell[TextData[{
  "This option allows the user control over the way the curve itself is \
drawn. Since ",
  StyleBox["Plot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows us to graph an arbitrary number of curves simultaneously, ",
  StyleBox["PlotStyle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " takes a list of lists of directives, pairing each list of directives with \
one curve. If there are more curves than lists of directives, the existing \
lists of directives are applied cyclically."
}], "Text"],

Cell[TextData[{
  "The color of the curve can be specified in any of the standard ways. \
Thickness of the line is controlled by ",
  StyleBox["Thickness[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["r",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", where ",
  StyleBox["r",
    FontSlant->"Italic"],
  " is given as a fraction of the graphic\[CloseCurlyQuote]s total width or \
by ",
  StyleBox["AbsoluteThickness[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["d",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", where ",
  StyleBox["d",
    FontSlant->"Italic"],
  " is specified in points (where 1 point is ",
  Cell[BoxData[
      \(TraditionalForm\`1\/72\)]],
  " inch). We can create a dashed line with the directive ",
  StyleBox["Dashing[{", "Input"],
  Cell[BoxData[
      \(TraditionalForm\`r\_1\)]],
  StyleBox[",", "Input"],
  " ",
  Cell[BoxData[
      \(TraditionalForm\`r\_2\)]],
  StyleBox[",", "Input"],
  " ",
  StyleBox["\[Ellipsis] ",
    FontSlant->"Italic"],
  StyleBox["}", "Input"],
  StyleBox["]", "Input",
    FontFamily->"Courier"],
  ", where ",
  StyleBox["{", "Input"],
  Cell[BoxData[
      \(TraditionalForm\`r\_1\)]],
  StyleBox[",", "Input"],
  StyleBox[" ",
    FontSlant->"Italic"],
  Cell[BoxData[
      \(TraditionalForm\`r\_2\)]],
  StyleBox[",", "Input"],
  StyleBox[" \[Ellipsis]",
    FontSlant->"Italic"],
  StyleBox["}", "Input"],
  " denote lengths of successive segments as a fraction of the total width of \
the graphic. The lengths are repeated cyclically. ",
  StyleBox["AbsoluteDashing[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["{", "Input"],
  Cell[BoxData[
      \(TraditionalForm\`d\_1\)]],
  StyleBox[",", "Input"],
  StyleBox[" ",
    FontSlant->"Italic"],
  Cell[BoxData[
      \(TraditionalForm\`d\_2\)]],
  StyleBox[",", "Input"],
  StyleBox[" \[Ellipsis] ",
    FontSlant->"Italic"],
  StyleBox["}", "Input"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " behaves identically but takes lengths in points. When plotting points \
with a function such as ",
  StyleBox["ListPlot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["PointSize[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["r",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " specifies the radius of individual points, where ",
  StyleBox["r",
    FontSlant->"Italic"],
  " is given as a fraction of the graphic\[CloseCurlyQuote]s total width. ",
  StyleBox["AbsolutePointSize[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["r",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows specification of the radius ",
  StyleBox["r",
    FontSlant->"Italic"],
  " in points."
}], "Text"],

Cell[BoxData[
    \(\(Plot[{2\ x\^2\ E\^\(-x\^2\), 2\ x\^2\ E\^\(\(-2\)\ x\^2\), 
        2\ x\^2\ E\^\(\(-3\)\ x\^2\)}, {x, \(-2\), 2}, \n\t
      PlotStyle \[Rule] \n
        \t\t{{Red}, {Dashing[{0.03, 0.02}], Magenta}, \ {Thickness[0.013]}}, 
      \n\tPlotRange \[Rule] {\(-1\), 1}, PlotPoints \[Rule] 50]; \)\)], 
  "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Frames, Axes, Labels, and Grids", "Subsubsection"],

Cell[TextData[{
  "By default, ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " will draw coordinate axes, but no frame and no labels, on a \
two-dimensional plot."
}], "Text"],

Cell[BoxData[
    \(\(Plot[x\ E\^x\ Sin[8\ \[Pi]\ x], {x, 0, 1}]; \)\)], "Input"],

Cell["\<\
If we ask for a frame, the ticks are automatically drawn on the \
frame rather than through the graphic, so often we will want to turn off the \
axes entirely.\
\>", "Text"],

Cell[BoxData[
    \(\(frame1 = 
      Plot[x\ E\^x\ Sin[8\ \[Pi]\ x], {x, 0, 1}, \n\t\tFrame \[Rule] True, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(frame2 = 
      Plot[x\ E\^x\ Sin[8\ \[Pi]\ x], {x, 0, 1}, \n\t\tFrame \[Rule] True, 
        Axes \[Rule] False, DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{frame1, frame2}]]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["Axes",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows is to control which, if any, axes are drawn, while ",
  StyleBox["AxesStyle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " controls how the axes are drawn. ",
  StyleBox["Axes",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " can bet set to ",
  StyleBox["True",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " or ",
  StyleBox["False",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", or to a list of values allowing us to control each axis independently. ",
  
  StyleBox["AxesStyle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " can take a list of lists of graphics directives and control the style of \
each axis independently. Analogous to ",
  StyleBox["AxesStyle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["FrameStyle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " gives us control over the way the frame is drawn."
}], "Text"],

Cell[BoxData[
    \(\(axes1 = 
      Plot[x\ E\^x\ Sin[8\ \[Pi]\ x], {x, 0, 1}, \n\t\t
        Axes \[Rule] {True, False}, DisplayFunction \[Rule] Identity]; \)\)], 
  "Input"],

Cell[BoxData[
    \(\(axes2 = 
      Plot[x\ E\^x\ Sin[8\ \[Pi]\ x], {x, 0, 1}, \n\t\t
        AxesStyle \[Rule] \n
          \t\t\t{{RGBColor[0, 1, 0], Thickness[0.02]}, {RGBColor[1, 0, 0], 
              Thickness[0.03]}}, DisplayFunction \[Rule] Identity]; \)\)], 
  "Input"],

Cell[BoxData[
    \(\(axes3 = 
      Plot[x\ E\^x\ Sin[8\ \[Pi]\ x], {x, 0, 1}, \n\t\tFrame \[Rule] True, \n
        \t\tAxes \[Rule] False, \n\t\t
        FrameStyle \[Rule] {Thickness[0.02], RGBColor[0, 0, 1]}, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{axes1, axes2, axes3}]]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["PlotLabel",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows us to specify a title for the entire graphic, while ",
  StyleBox["FrameLabel",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows labeling of each edge of the frame. (Note that while rotated text \
may not display correctly on the screen, it will print correctly.)"
}], "Text"],

Cell[BoxData[
    \(\(Plot[x\ E\^x\ Sin[8\ \[Pi]\ x], {x, 0, 1}, \n\tFrame \[Rule] True, \n
      \tAxes \[Rule] False, \n\tPlotLabel \[Rule] "\<Exponential & Sine\>", 
      FrameLabel 
        \[Rule] {"\<Time\>", "\<Response\>", None, "\<Right Label\>"}]; \)\)],
   "Input"],

Cell[TextData[{
  StyleBox["TextStyle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " will set the font for all text in the graphic. We may also use ",
  StyleBox["StyleForm",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " to choose fonts individually."
}], "Text"],

Cell[BoxData[
    \(\(font1 = 
      Plot[x\ E\^x\ Sin[8\ \[Pi]\ x], {x, 0, 1}, \n\t\tFrame \[Rule] True, \n
        \t\tAxes \[Rule] False, \n\t\tPlotLabel \[Rule] "\<Sin[x]/x\>", \n\t\t
        FrameLabel \[Rule] {"\<Time\>", "\<Response\>"}, \n\t\t
        TextStyle 
          \[Rule] {FontFamily \[Rule] "\<Helvetica\>", FontSize \[Rule] 10}, 
        \n\t\tDisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(font2 = 
      Plot[x\ E\^x\ Sin[8\ \[Pi]\ x], {x, 0, 1}, \n\t\tFrame \[Rule] True, \n
        \t\tAxes \[Rule] False, \n\t\t
        PlotLabel \[Rule] 
          StyleForm[Sin[x]/x, \n\t\t\t\tFontFamily \[Rule] "\<Times\>", 
            FontWeight \[Rule] "\<Bold\>", FontSize \[Rule] 14], \n\t\t
        FrameLabel \[Rule] {"\<Time\>", "\<Response\>"}, \n\t\t
        TextStyle 
          \[Rule] {FontFamily \[Rule] "\<Helvetica\>", FontSize \[Rule] 10}, 
        \n\t\tDisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{font1, font2}]]; \)\)], "Input"],

Cell[TextData[{
  "The ",
  StyleBox["RotateLabel",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " option defaults to ",
  StyleBox["True",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allowing labels on vertical axes to be written parallel to the axis."
}], "Text"],

Cell[TextData[{
  StyleBox["AxesOrigin",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows us to specify the point in a two-dimensional graphic where the \
coordinate axes meet. If set outside the range of the plot, the axes will not \
cross at all."
}], "Text"],

Cell[BoxData[
    \(\(origin1 = 
      Plot[x\ E\^x\ Sin[8\ \[Pi]\ x], {x, 0, 1}, \n\t\t
        AxesOrigin \[Rule] {0.5, 0.5}, \n\t\tTicks \[Rule] {Automatic, None}, 
        \n\t\tDisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(origin2 = 
      Plot[x\ E\^x\ Sin[8\ \[Pi]\ x], {x, 0, 1}, \n\t\t
        AxesOrigin \[Rule] {0.2, 1}, \n\t\tDisplayFunction \[Rule] Identity]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{origin1, origin2}]]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["GridLines",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows us to specify the placement of a grid under the graphic. The \
default setting is ",
  StyleBox["None",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". A setting of ",
  StyleBox["Automatic",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " will generate uniformly spaced grid lines in both directions, or we can \
specify the actual positions by giving a list of values. We may specify \
different grid lines in each direction with the format ",
  StyleBox["{", "Input"],
  StyleBox["xgrid",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ygrid",
    FontSlant->"Italic"],
  StyleBox["}", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(grid1 = 
      Plot[Sin[x]\/x, {x, \(-10\), 10}, GridLines \[Rule] Automatic, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(grid2 = 
      Plot[Sin[x]\/x, {x, \(-10\), 10}, 
        GridLines \[Rule] {{\(-10\), \(-5\), \(-2.5\), 2.5, 5, 10}, None}, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(grid3 = 
      Plot[Sin[x]\/x, {x, \(-10\), 10}, 
        GridLines \[Rule] {Range[\(-10\), 10], Automatic}, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{grid1, grid2, grid3}]]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Colors", "Subsubsection"],

Cell[TextData[{
  "With ",
  StyleBox["Background",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["DefaultColor",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", and ",
  StyleBox["PlotStyle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " we control the colors of various aspects of the graphic. Note that we \
explicitly load the ",
  StyleBox["Graphics`Colors`",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " package so that we may use names instead of RGB specifications. ",
  StyleBox["DefaultColor",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", the color to use for lines, points, tick marks, etc., can be set to any \
valid color specification. When set to ",
  StyleBox["Automatic",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", a color complementary to the color of the background, which can be set \
with ",
  StyleBox["Background",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", is used."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`Colors`\>"]\)], "Input"],

Cell[BoxData[
    \(\(Plot[2\ x\^5 - 3\ x\^4 + 10\ x\^2 - 2\ x - 1, {x, \(-2\), 2}, \n\t
      Background \[Rule] SkyBlue, \n\tDefaultColor \[Rule] RosyBrown, \n\t
      PlotStyle \[Rule] {{Thickness[0.02], Maroon}}]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Spacing", "Subsubsection"],

Cell[TextData[{
  StyleBox["PlotRegion",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows us to specify how much blank space to leave around the graphic. A \
setting of the form ",
  StyleBox["{{", "Input"],
  StyleBox["sxmin",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["sxmax",
    FontSlant->"Italic"],
  StyleBox["}", "Input"],
  ", ",
  StyleBox["{", "Input"],
  StyleBox["symin",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["symax",
    FontSlant->"Italic"],
  StyleBox["}}", "Input"],
  " specifies the region, in scaled coordinates, that the plot should fill in \
the final display area. When set to ",
  StyleBox["Automatic",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", the plot will fill the final display area."
}], "Text"],

Cell[BoxData[
    \(\(defaultregion = 
      Plot[2\ x\^5 - 3\ x\^4 + 10\ x\^2 - 2\ x - 1, {x, \(-2\), 2}, 
        Background \[Rule] SkyBlue, 
        PlotLabel \[Rule] 
          StyleForm["\<Quintic\>", FontFamily \[Rule] "\<Times\>", 
            FontWeight \[Rule] "\<Bold\>", FontSize \[Rule] 14], 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(largerregion = 
      Plot[2\ x\^5 - 3\ x\^4 + 10\ x\^2 - 2\ x - 1, {x, \(-2\), 2}, 
        Background \[Rule] SkyBlue, 
        PlotLabel \[Rule] 
          StyleForm["\<Quintic\>", FontFamily \[Rule] "\<Times\>", 
            FontWeight \[Rule] "\<Bold\>", FontSize \[Rule] 14], 
        PlotRegion \[Rule] {{0.1, 0.9}, {0.1, 0.9}}, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{defaultregion, largerregion}]]; \)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Plot3D Options", "Subsection"],

Cell[CellGroupData[{

Cell["Basics", "Subsubsection"],

Cell["\<\
Many of the options controlling three-dimensional graphics are \
identical to those used in the two-dimensional case.\
\>", "Text"],

Cell[TextData[{
  StyleBox["AspectRatio",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["Background",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ",  ",
  StyleBox["DefaultFont",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["PlotLabel",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["DefaultColor",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", and ",
  StyleBox["AxesStyle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " are options that are used identically with ",
  StyleBox["Plot3D",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " and ",
  StyleBox["Plot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", while ",
  StyleBox["AxesLabel",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["AxesStyle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["PlotRange",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", and ",
  StyleBox["PlotRegion",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " are called identically but extended to three dimensions via the addition \
of a third coordinate."
}], "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell["Point Sampling", "Subsubsection"],

Cell[TextData[{
  "Unlike ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "\[CloseCurlyQuote]s two-dimensional plotting algrorithm, three-dimensional \
plots have no built-in method for sampling more points where necessary to \
achieve an accurate plot. ",
  StyleBox["Plot3D",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " will choose uniformly distributed points along the ",
  StyleBox["x",
    FontSlant->"Italic"],
  "-range and the ",
  StyleBox["y",
    FontSlant->"Italic"],
  "-range and plot only these values. The option ",
  StyleBox["PlotPoints",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", with default value 15, sets the number of divisions to use. ",
  StyleBox["PlotPoints",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " can also be set to an ordered pair, denoting the number of points to use \
in each direction. "
}], "Text"],

Cell[BoxData[
    \(\(Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
      PlotPoints \[Rule] {10, 35}]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Lighting, Color, and Other Drawing Options", "Subsubsection"],

Cell["\<\
When plotting in three dimensions, several more color options are \
available.\
\>", "Text"],

Cell[TextData[{
  StyleBox["AmbientLight",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " and ",
  StyleBox["LightSources",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allow us to shade the three-dimensional graphic with external light. ",
  StyleBox["AmbientLight",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is by default white but can be set to any gray level or color \
specification. By setting a higher gray level, the graphics appears light and \
washed-out."
}], "Text"],

Cell[BoxData[
    \(\(Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
      AmbientLight \[Rule] SlateGray]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["LightSources",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows us to create an arbitrary number of lights in arbitrary colors and \
shine them from arbitrary locations. By default, one red, one green, and one \
blue light shine from  the points ",
  StyleBox["{1., 0., 1.}", "Input"],
  ",  ",
  StyleBox["{1., 1., 1.}", "Input"],
  ",  and ",
  StyleBox["{0., 1., 1.}", "Input"],
  ", respectively. Here we plot a surface with two light sources shining on \
it, one green and one blue."
}], "Text"],

Cell[BoxData[
    \(\(Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
      LightSources \[Rule] {{{\(-1\), 0, 1}, Green}, {{1, 0, 1}, Blue}}]; 
    \)\)], "Input"],

Cell[TextData[{
  "A more direct way of applying color to a surface is via the ",
  StyleBox["ColorFunction",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " option. ",
  StyleBox["ColorFunction",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " specifies a function to apply to function",
  StyleBox[" z ",
    FontSlant->"Italic"],
  "values to determine the color of a particular ",
  StyleBox["x",
    FontSlant->"Italic"],
  "-",
  StyleBox["y",
    FontSlant->"Italic"],
  " region. Since ",
  StyleBox["Hue",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", when called with only one argument, will run cyclically through the \
entire color wheel, it is suitable for use with ",
  StyleBox["ColorFunction",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(cf1 = 
      Plot3D[Sin[x\ y], {x, \(-3\), 3}, {y, \(-3\), 3}, 
        ColorFunction \[Rule] Hue, PlotPoints \[Rule] 20, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(mycolor[z_] := RGBColor[Random[], Random[], Random[]]; \)\)], "Input"],

Cell[BoxData[
    \(\(cf2 = 
      Plot3D[Sin[x\ y], {x, \(-3\), 3}, {y, \(-3\), 3}, 
        ColorFunction \[Rule] mycolor, PlotPoints \[Rule] 20, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell["We may also turn off the lighting altogether.", "Text"],

Cell[BoxData[
    \(\(cf3 = 
      Plot3D[Sin[x\ y], {x, \(-3\), 3}, {y, \(-3\), 3}, 
        Lighting \[Rule] False, PlotPoints \[Rule] 20, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{cf1, cf2, cf3}]]; \)\)], "Input"],

Cell[TextData[{
  "We may also specify a function to apply to each ",
  StyleBox["x",
    FontSlant->"Italic"],
  "-",
  StyleBox["y",
    FontSlant->"Italic"],
  " pair to indicate the color of each region. Use the format ",
  StyleBox["Plot3D[{",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["f",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  StyleBox[" s",
    FontSlant->"Italic"],
  StyleBox["},",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["{",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["x",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  StyleBox[" xmin",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  StyleBox[" xmax",
    FontSlant->"Italic"],
  StyleBox["},",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["{",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["y",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  StyleBox[" ymin",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  StyleBox[" ymax",
    FontSlant->"Italic"],
  StyleBox["}]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", where ",
  StyleBox["s",
    FontSlant->"Italic"],
  " is the shading function. ",
  StyleBox["s",
    FontSlant->"Italic"],
  " can be any expression that yields a valid color specification."
}], "Text"],

Cell[BoxData[
    \(\(shade1 = 
      Plot3D[{Sin[\@\(x\^2 + y\^2\)], Blue}, {x, \(-3\), 3}, {y, \(-3\), 3}, 
        PlotPoints \[Rule] 20, DisplayFunction \[Rule] Identity]; \)\)], 
  "Input"],

Cell[BoxData[
    \(\(shade2 = 
      Plot3D[{Sin[\@\(x\^2 + y\^2\)], If[Sin[x - Cos[y]] > 0.5, Blue, Red]}, {
          x, \(-3\), 3}, {y, \(-3\), 3}, PlotPoints \[Rule] 20, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(shade3 = 
      Plot3D[{Sin[\@\(x\^2 + y\^2\)], 
          RGBColor[Abs[x\/3], Abs[y\/3], Abs[\(x + y\)\/6]]}, {x, \(-3\), 
          3}, {y, \(-3\), 3}, PlotPoints \[Rule] 20, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{shade1, shade2, shade3}]]; \)\)], "Input"],

Cell[TextData[{
  "A related option is ",
  StyleBox["ClipFill",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", which determines how to render areas of the surface that extend beyond \
the bounding box. By default, these areas are drawn as the rest of the \
surface. Setting ",
  StyleBox["ClipFill",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " to ",
  StyleBox["None",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " will show holes in these areas. We may specify one color for all clipped \
region or a pair of colors, one for the top and one for the bottom."
}], "Text"],

Cell[BoxData[
    \(\(cf1 = 
      Plot3D[2\ Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
        PlotRange \[Rule] {\(-1\), 1}, DisplayFunction \[Rule] Identity]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(cf2 = 
      Plot3D[2\ Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
        ClipFill \[Rule] None, PlotRange \[Rule] {\(-1\), 1}, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(cf3 = 
      Plot3D[2\ Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
        ClipFill \[Rule] {GrayLevel[.5], RGBColor[0, 1, 0]}, 
        PlotRange \[Rule] {\(-1\), 1}, DisplayFunction \[Rule] Identity]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{cf1, cf2, cf3}]]; \)\)], "Input"],

Cell[TextData[{
  "By setting the option ",
  StyleBox["HiddenSurface",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " to ",
  StyleBox["False",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", we can see the graphic rendered with transparent polygons."
}], "Text"],

Cell[BoxData[
    \(\(Plot3D[2\ Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
      HiddenSurface \[Rule] False]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Box, Mesh, and Other Extrinsic Details", "Subsubsection"],

Cell[TextData[{
  StyleBox["Mesh",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " and ",
  StyleBox["Boxed",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " may be set only to ",
  StyleBox["True",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " or ",
  StyleBox["False",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", and do the obvious things. Since the mesh on the surface simply connects \
the points plotted by ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ", its fineness is controlled by the value of ",
  StyleBox["PlotPoints",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
      Boxed \[Rule] False, Mesh \[Rule] False]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["MeshStyle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " and ",
  StyleBox["BoxStyle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allow control of the way these lines are drawn. They take graphics \
directives as arguments."
}], "Text"],

Cell[BoxData[
    \(\(Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
      BoxStyle \[Rule] {Red, Thickness[0.02]}, 
      MeshStyle \[Rule] {Thickness[0.01]}]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["FaceGrids",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows us to specify whether to draw grids on the faces of the bounding \
box and to specify the arrangement of the grid. ",
  StyleBox["All",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " and ",
  StyleBox["None",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " are the simplest values for this option, but we may choose individual \
faces and arrange the grid lines. A face is specified by ",
  StyleBox["{", "Input"],
  Cell[BoxData[
      FormBox[
        StyleBox[
          SubscriptBox[
            StyleBox["dir",
              FontSlant->"Italic"], "x"],
          FontSlant->"Italic"], TraditionalForm]]],
  StyleBox[",", "Input"],
  " ",
  Cell[BoxData[
      FormBox[
        StyleBox[\(dir\_y\),
          FontSlant->"Italic"], TraditionalForm]]],
  StyleBox[",", "Input"],
  " ",
  Cell[BoxData[
      FormBox[
        StyleBox[\(dir\_z\),
          FontSlant->"Italic"], TraditionalForm]]],
  StyleBox["}", "Input"],
  ", where two of the elements are 0 and the third is either +1 or \[Dash]1. \
The form  ",
  StyleBox["{", "Input"],
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["face",
            FontSlant->"Italic"], "1"], TraditionalForm]]],
  StyleBox[",", "Input"],
  "  ",
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["face",
            FontSlant->"Italic"], "2"], TraditionalForm]]],
  StyleBox[",", "Input"],
  " \[Ellipsis] ",
  StyleBox["}", "Input"],
  " is used to specify individual faces, and the form ",
  Cell["{{", "Input"],
  Cell[BoxData[
      FormBox[
        SubscriptBox[
          StyleBox["face",
            FontSlant->"Italic"], "1"], TraditionalForm]]],
  StyleBox[", {", "Input"],
  StyleBox["xgrid",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ygrid",
    FontSlant->"Italic"],
  StyleBox["}},", "Input"],
  " \[Ellipsis] ",
  StyleBox["}", "Input"],
  " is used to specify the grid for individual faces. Grid specifications \
follow the same format as that of ",
  StyleBox["GridLines",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", so complex styled grids can be drawn."
}], "Text"],

Cell[BoxData[
    \(\(fg1 = 
      Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
        FaceGrids \[Rule] All, DisplayFunction \[Rule] Identity]; \)\)], 
  "Input"],

Cell[BoxData[
    \(\(fg2 = 
      Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
        FaceGrids \[Rule] {{0, \(-1\), 0}, {0, 0, 1}}, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(fg3 = 
      Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
        FaceGrids 
          \[Rule] {{{0, \(-1\), 0}, {
                Table[{i, RGBColor[1, 0, 0]}, {i, \(-3\), 3, .25}], 
                Table[{i, RGBColor[0, 0, 1]}, {i, \(-1\), 1, .4}]}}}, 
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{fg1, fg2, fg3}]]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["AxesEdge",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows control over which three edges of the bounding box the axes are \
drawn, with specifications given in the form ",
  StyleBox["{{", "Input"],
  StyleBox["xdir",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ydir",
    FontSlant->"Italic"],
  StyleBox["}, {", "Input"],
  StyleBox["xdir",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["zdir",
    FontSlant->"Italic"],
  StyleBox["}, {", "Input"],
  StyleBox["xdir",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ydir",
    FontSlant->"Italic"],
  StyleBox["}}", "Input"],
  ", with each element being either 1 or \[Dash]1, indicating whether to draw \
the axes on the edge of the box with the larger or smaller coordinate \
\[LongDash] the default being to draw toward the smaller coordinate. Any pair \
can be replaced by ",
  StyleBox["Automatic",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " or ",
  StyleBox["None",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
      AxesEdge \[Rule] {Automatic, {\(-1\), 1}, None}]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["ViewPoint and Related Options", "Subsubsection"],

Cell[TextData[{
  "The point in space from which a three-dimensional object is viewed, the ",
  StyleBox["ViewPoint",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", is by default {1.3, \[Dash]2.4, 2.0} but can be set to any triple. \
Notebook front ends to ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " all have a graphical viewpoint selector, which allows the user to select \
an orientation visually instead of numerically."
}], "Text"],

Cell[BoxData[
    \(\(Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
      ViewPoint \[Rule] {10, 2, 1}]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["ViewCenter",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows us to specify where the center of the bounding box is, while ",
  StyleBox["ViewVertical",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " specifies the direction in the image that should be vertical. In the \
following example, the positive ",
  StyleBox["y",
    FontSlant->"Italic"],
  " axis is vertical."
}], "Text"],

Cell[BoxData[
    \(\(v1 = 
      Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
        ViewCenter \[Rule] {1, 0, 0}, DisplayFunction \[Rule] Identity]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(v2 = 
      Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
        ViewVertical \[Rule] {0, 1, 0}, DisplayFunction \[Rule] Identity]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[{v1, v2}]]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["BoxRatios",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is similar to ",
  StyleBox["AspectRatio",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " but allows control over the ratios of the lengths of all three sides of \
the bounding box."
}], "Text"],

Cell[BoxData[
    \(\(Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
      BoxRatios \[Rule] {9, 4, 1}]; \)\)], "Input"],

Cell[TextData[{
  "By default, the option ",
  StyleBox["SphericalRegion",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is ",
  StyleBox["False",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", allowing the graphic to be drawn as large as possible. By setting it to \
",
  StyleBox["True",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", we require that the image be scaled so that a sphere drawn around the \
bounding box will still fit in the display area. This control is useful when \
creating animations. Since ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " will try to make each graphic as large as possible, multiple graphics \
generated to create a movie may be drawn to different scales, ruining the \
effect of the animation. Setting ",
  StyleBox["SphericalRegion",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " to ",
  StyleBox["True",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " guarantees that all frames will be drawn the same size."
}], "Text"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["ContourPlot and DensityPlot Options", "Subsection"],

Cell[TextData[{
  StyleBox["ContourPlot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " and ",
  StyleBox["DensityPlot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " generally take options available in ",
  StyleBox["Plot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " and ",
  StyleBox["Plot3D",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". The only exceptions are several options dealing specifically with ",
  StyleBox["ContourPlot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". ",
  StyleBox["ContourLines",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " controls the drawing of explicit contour lines. By default, this option \
is ",
  StyleBox["True",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". ",
  StyleBox["Contours",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " allows us to choose the specific contours plotted. This option may be set \
to an integer ",
  StyleBox["n",
    FontSlant->"Italic"],
  ", which will produce ",
  StyleBox["n",
    FontSlant->"Italic"],
  " equally spaced contours between the minimum and maximum ",
  StyleBox["z",
    FontSlant->"Italic"],
  " values, or to an explicit list of ",
  StyleBox["z",
    FontSlant->"Italic"],
  " values. The style in which contour lines are drawn is specified in ",
  StyleBox["ContourStyle", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(ContourPlot[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, 
      ContourLines \[Rule] False, Contours \[Rule] 20]; \)\)], "Input"],

Cell[TextData[{
  StyleBox["ContourShading",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " defaults to ",
  StyleBox["True",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". Setting it to ",
  StyleBox["False",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " will leave the areas between contour lines blank. ",
  StyleBox["ContourShading", "Input"],
  " also lets us specify the color between contour lines by using a list of \
color specifications."
}], "Text"],

Cell[BoxData[
    \(\(ContourPlot[\ Sin[x] - Cos[y], \ {x, \(-2\), 2}, {y, \(-2\), 2}, \n
      \ \ \ \ \ \ \ \ ContourShading\  -> \ {Hue[0.25], GrayLevel[0.25]}]; 
    \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["ParametricPlot and ParametricPlot3D Options", "Subsection"],

Cell[TextData[{
  StyleBox["ParametricPlot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " has the same list of options as ",
  StyleBox["Plot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(ParametricPlot[{t\ Cos[t], t\ Sin[t]}, {t, 0, 4\ \[Pi]}, 
      PlotStyle \[Rule] {{RGBColor[0, 1, 0], Dashing[{0.02}]}}, 
      Frame \[Rule] True, Axes \[Rule] False, AspectRatio \[Rule] Automatic, 
      FrameStyle \[Rule] {Thickness[0.02], RGBColor[0.4, 0.2, 0.1]}]; \)\)], 
  "Input"],

Cell[TextData[{
  StyleBox["ParametricPlot3D",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " has a few minor differences from ",
  StyleBox["Plot3D",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". ",
  StyleBox["ParametricPlot3D",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " lacks the options ",
  StyleBox["Mesh",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["MeshStyle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", and ",
  StyleBox["HiddenSurface",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", and has one new option, ",
  StyleBox["RenderAll", "Input"],
  StyleBox[".", "Input",
    FontWeight->"Plain"]
}], "Text"],

Cell[TextData[{
  "Large graphics can take a large amount of RAM to display\[LongDash] \
sometimes more memory than is available, especially if we are trying to \
create an animation in which each frame is a complicated ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " graphic."
}], "Text"],

Cell[TextData[{
  StyleBox["RenderAll", "Input"],
  ", an option for ",
  StyleBox["Graphics3D", "Input"],
  " and ",
  StyleBox["ParametricPlot3D", "Input"],
  ", which is ",
  StyleBox["True", "Input"],
  " by default, allows the option of generating PostScript only for those \
polygons that will be visible in the final image. The resulting image \
contains significantly less PostScript code and therefore takes less memory \
to display and renders faster, but requires considerably more time to \
create."
}], "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell["Ticks", "Subsection"],

Cell[TextData[{
  "Although ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " will usually supply useful tick marks on graphics, much customization is \
possible. ",
  StyleBox["Ticks",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is an option for all graphics functions. With ",
  StyleBox["Ticks",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["->",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["None",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", no tick marks are drawn on the axes. With ",
  StyleBox["Ticks",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["->",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["Automatic",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", tick marks are placed automatically. ",
  StyleBox["Ticks",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["->",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["{", "Input"],
  StyleBox["xticks",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["yticks",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " \[Ellipsis] ",
  StyleBox["}", "Input"],
  " specifies tick mark options separately for each axis."
}], "Text"],

Cell[BoxData[
    \(\(Plot[BesselJ[1, x], {x, 0, 10}, Ticks \[Rule] {None, Automatic}]; 
    \)\)], "Input"],

Cell[TextData[{
  "There are several levels of detail at which we may specify the properties \
of tick marks. The simplest is to supply a list of positions. Note that the \
list of positions can be generated by a ",
  StyleBox["Range",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " or ",
  StyleBox["Table",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " command, though this method may be limiting should we change the domain \
used in the plot."
}], "Text"],

Cell[BoxData[
    \(\(Plot[BesselJ[1, x], {x, 0, 10}, 
      Ticks \[Rule] {{0, 2, 4, 6, 7, 8, 9, 10}, Range[\(-0.2\), 0.5, 0.1]}]; 
    \)\)], "Input"],

Cell["\<\
We may also specify the label associated with each tick mark, the \
length of the mark (both how far above and below the axes it extends), and \
the style in which to draw the mark. The following is a list of allowable \
configurations for a tick mark. Note that you may vary among these throughout \
one list of tick marks.\
\>", "Text"],

Cell[BoxData[
    StyleBox[GridBox[{
          {
            StyleBox["position",
              FontWeight->"Plain",
              FontSlant->"Italic"]},
          {
            RowBox[{"{", 
              RowBox[{
                StyleBox["position",
                  FontWeight->"Plain",
                  FontSlant->"Italic"], ",", " ", 
                StyleBox["label",
                  FontWeight->"Plain",
                  FontSlant->"Italic"]}], "}"}]},
          {
            RowBox[{"{", 
              RowBox[{
                StyleBox["position",
                  FontWeight->"Plain",
                  FontSlant->"Italic"], ",", " ", 
                StyleBox["label",
                  FontWeight->"Plain",
                  FontSlant->"Italic"], ",", " ", 
                StyleBox["length",
                  FontWeight->"Plain",
                  FontSlant->"Italic"]}], "}"}]},
          {
            RowBox[{"{", 
              RowBox[{
                StyleBox["position",
                  FontWeight->"Plain",
                  FontSlant->"Italic"], ",", " ", 
                StyleBox["label",
                  FontWeight->"Plain",
                  FontSlant->"Italic"], ",", " ", 
                StyleBox["length",
                  FontWeight->"Plain",
                  FontSlant->"Italic"], ",", " ", 
                RowBox[{"{", 
                  StyleBox["style",
                    FontWeight->"Plain",
                    FontSlant->"Italic"], "}"}]}], "}"}]},
          {
            RowBox[{"{", 
              RowBox[{
                StyleBox["position",
                  FontWeight->"Plain",
                  FontSlant->"Italic"], ",", " ", 
                StyleBox["label",
                  FontWeight->"Plain",
                  FontSlant->"Italic"], ",", " ", 
                RowBox[{"{", 
                  RowBox[{
                    RowBox[{
                      StyleBox["positive",
                        FontWeight->"Plain",
                        FontSlant->"Italic"], " ", 
                      StyleBox["length",
                        FontWeight->"Plain",
                        FontSlant->"Italic"]}], ",", " ", 
                    RowBox[{
                      StyleBox["negative",
                        FontWeight->"Plain",
                        FontSlant->"Italic"], " ", 
                      StyleBox["length",
                        FontWeight->"Plain",
                        FontSlant->"Italic"]}]}], "}"}]}], "}"}]},
          {
            RowBox[{"{", 
              RowBox[{
                StyleBox["position",
                  FontWeight->"Plain",
                  FontSlant->"Italic"], ",", " ", 
                StyleBox["label",
                  FontWeight->"Plain",
                  FontSlant->"Italic"], ",", " ", 
                RowBox[{"{", 
                  RowBox[{
                    RowBox[{
                      StyleBox["positive",
                        FontWeight->"Plain",
                        FontSlant->"Italic"], " ", 
                      StyleBox["length",
                        FontWeight->"Plain",
                        FontSlant->"Italic"]}], ",", " ", 
                    RowBox[{
                      StyleBox["negative",
                        FontWeight->"Plain",
                        FontSlant->"Italic"], " ", 
                      StyleBox["length",
                        FontWeight->"Plain",
                        FontSlant->"Italic"]}]}], "}"}], ",", " ", 
                RowBox[{"{", 
                  StyleBox["style",
                    FontWeight->"Plain",
                    FontSlant->"Italic"], "}"}]}], "}"}]}
          },
        GridFrame->True,
        RowLines->True,
        ColumnLines->True],
      FontFamily->"Times"]], "Input"],

Cell[TextData[
"Plot[ BesselJ[1, x], {x, 0, 10}, \nTicks \[Rule]\n{{{1.8,\"max\", \
0.05,{Thickness[0.01],Red}},\n  {5.3,\"min\", 0.05,{Thickness[0.01],Red}},\n  \
{8.5,\"max\", 0.05,{Thickness[0.01],Red}}},\n {{-0.35,\"mark \
\",{0,0.02},{Thickness[0.01],Blue}},\n  {0.58,\"mark \
\",{0,0.02},{Thickness[0.01],Blue}},\n  {0.28,\"mark \
\",{0,0.02},{Thickness[0.01],Blue}} } }];"], "Input"],

Cell[TextData[{
  "In this way we can add explicitly known information to a graphic, but ",
  StyleBox["Ticks",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is also designed to work in a more general fashion. ",
  StyleBox["Ticks",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " can take as its arguments functions that take as arguments the minimum \
and maximum range for an axis and return a valid tick specification. Using \
this form, the specifications are not tied to explicit ranges. Here we define \
two tick specifications. Each function draws unlabeled blue ticks every unit \
and larger, labeled blue ticks every 5 or 10 units, respectively."
}], "Text"],

Cell["\<\
myticks5[min_, max_] :=
\tJoin[
\t\tTable[{x, x, 0.05, {RGBColor[0,0,1]}}, 
\t\t\t\t\t{x, 0, Ceiling[max],5}],
\t\tTable[{x, x, 0.05, {RGBColor[0,0,1]}}, 
\t\t\t\t\t{x,0, Floor[min],-5}],
\t\tTable[{x, \"\", 0.02, {RGBColor[0,0,1]}}, 
\t\t\t\t\t{x, Floor[min], Ceiling[max]}]
\t\t];\
\>", "Input"],

Cell["\<\
myticks10[min_, max_] :=
\tJoin[
\t\tTable[{x, x, 0.05, {RGBColor[0,0,1]}}, 
\t\t\t\t\t{x, 0, Ceiling[max],10}],
\t\tTable[{x, x, 0.05, {RGBColor[0,0,1]}}, 
\t\t\t\t\t{x,0, Floor[min],-10}],
\t\tTable[{x, \"\", 0.02, {RGBColor[0,0,1]}}, 
\t\t\t\t\t{x, Floor[min], Ceiling[max]}]
\t\t];\
\>", "Input"],

Cell[BoxData[
    \(\(Plot[x\ Sin[0.3\ \[Pi]\ x]\^2, {x, \(-20\), 20}, \n\t
      Ticks \[Rule] {myticks10, myticks5}, \n\tPlotPoints \[Rule] 75]; \)\)], 
  "Input"],

Cell["Show[%, PlotRange -> {{-30, 30}, {-25, 25}}];", "Input"],

Cell[TextData[{
  "When a frame is drawn around the plot, ",
  StyleBox["FrameTicks",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is used to specify ticks to be drawn on the frame. ",
  StyleBox["FrameTicks",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " takes the same arguments as ",
  StyleBox["Ticks",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(Plot[x\ Sin[0.3\ \[Pi]\ x]\^2, {x, \(-20\), 20}, \n\t
      Frame \[Rule] True, \n\t
      FrameTicks \[Rule] {myticks10, myticks5, None, None}]; \)\)], "Input"],

Cell["\<\
We use exactly the same specifications for three-dimensional plots, \
contour plots, and density plots.\
\>", "Text"],

Cell[BoxData[
    \(\(Plot3D[0.25\ Cos[0.03\ x\ y], {x, \(-30\), 30}, {y, \(-30\), 30}, 
      PlotPoints \[Rule] 20, 
      Ticks \[Rule] {myticks10, myticks10, Automatic}]; \)\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Prolog, Epilog, and Rectangle", "Subsection"],

Cell[TextData[{
  StyleBox["Epilog",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is an option to every graphics function that allows arbitrary graphics to \
be rendered after the main graphic has been rendered. ",
  StyleBox["Prolog",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " specifies arbitrary graphics primitives and directives to be rendered \
before the main graphics is rendered. For simplicity, we will look only at ",
  StyleBox["Epilog",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". "
}], "Text"],

Cell[TextData[{
  "Any two-dimensional graphic can be superimposed on any two-dimensional \
plot. ",
  StyleBox["Arrow",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", ",
  StyleBox["Text",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", and several graphics directives can be combined to add information to \
this plot. We can specify the font by using ",
  StyleBox["StyleForm",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " within a ",
  StyleBox["Text",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " primitive."
}], "Text"],

Cell[BoxData[
    \(\(Plot[4\ x\^2 + 12\ x - 10, {x, \(-10\), 10}, \n\t
      Epilog \[Rule] {\n\t\t\t
          Text[StyleForm["\<\[CapitalOmega]\>", FontSize \[Rule] 18, 
              FontWeight \[Rule] "\<Bold\>"], {0, 600}], 
          Text[StyleForm["\<\[CapitalDelta]\>", FontSize \[Rule] 18, 
              FontWeight \[Rule] "\<Bold\>"], {12, 0}]}, \n\t
      PlotRange \[Rule] All]; \)\)], "Input"],

Cell[BoxData[
    \(\(Plot[x\ Sin[0.9\ x]\^2, {x, \(-10\), 10}, 
      Epilog \[Rule] {\n\t\t\tRGBColor[1, 0.5, 0.6], \n\t\t\t
          Text[StyleForm["\<Minima\>", \n\t\t\t\t\t
              FontFamily \[Rule] "\<Helvetica\>", 
              FontWeight \[Rule] "\<Bold\>", FontSize \[Rule] 12], {6, 
              \(-8.5\)}], 
          Text[StyleForm["\<Maxima\>", \n\t\t\t\t\t
              FontFamily \[Rule] "\<Helvetica\>", 
              FontWeight \[Rule] "\<Bold\>", FontSize \[Rule] 12], {\(-6\), 
              8.5}], \n\t\t\tRGBColor[0, 0, 1], \n\t\t\t
          Arrow[{\(-6\), 7}, {\(-7\), 0}], Arrow[{\(-6\), 7}, {\(-3.5\), 0}], 
          Arrow[{\(-6\), 7}, {2, 2}], Arrow[{\(-6\), 7}, {5.25, 5.45}], 
          Arrow[{\(-6\), 7}, {8.75, 8.9}], 
          Arrow[{6, \(-7\)}, {\(-8.8\), \(-8.75\)}], 
          Arrow[{6, \(-7\)}, {\(-5.35\), \(-5.25\)}], 
          Arrow[{6, \(-7\)}, {\(-2.1\), \(-1.85\)}], 
          Arrow[{6, \(-7\)}, {3.5, 0}], Arrow[{6, \(-7\)}, {7, 0}]}]; \)\)], 
  "Input"],

Cell[TextData[{
  "More generally, we can include any graphic in any other graphic using ",
  StyleBox["Rectangle",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ". The form ",
  StyleBox["Rectangle[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["{", "Input"],
  StyleBox["xmin",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ymin",
    FontSlant->"Italic"],
  StyleBox["},", "Input"],
  " ",
  StyleBox["{", "Input"],
  StyleBox["xmax",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ymax",
    FontSlant->"Italic"],
  StyleBox["}", "Input"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " represents a filled rectangle oriented parallel to the axes. The more \
general form ",
  StyleBox["Rectangle[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["{", "Input"],
  StyleBox["xmin",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ymin",
    FontSlant->"Italic"],
  StyleBox["},", "Input"],
  " ",
  StyleBox["{", "Input"],
  StyleBox["xmax",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ymax",
    FontSlant->"Italic"],
  StyleBox["},", "Input"],
  " ",
  StyleBox["graphics",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " represents a rectangle filled with the arbitrary graphics."
}], "Text"],

Cell[BoxData[
    \(\(inset = 
      Plot[Sin[x]\/x, {x, \(-10\), 10}, \n\t\tFrame \[Rule] True, \n\t\t
        Ticks \[Rule] None, \n\t\tFrameTicks \[Rule] None, \n\t\t
        FrameStyle \[Rule] {GrayLevel[.5], Thickness[0.03]}, \n\t\t
        PlotStyle \[Rule] {{Dashing[{0.04}]}}, \n\t\t
        DisplayFunction \[Rule] Identity]; \)\)], "Input"],

Cell[BoxData[
    \(\(Plot[x\ Sin[x]\^2, {x, \(-10\), 10}, \n\t
      PlotStyle \[Rule] RGBColor[0, 0, 1], 
      Epilog \[Rule] Rectangle[{\(-9\), 3}, {\(-2\), 8}, inset]]; \)\)], 
  "Input"],

Cell[TextData[{
  "We can create functions which return graphics objects. Below is a function \
called ",
  StyleBox["CircledDigit",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox[",",
    FontFamily->"Courier"],
  " which creates labels we can place on any graphic."
}], "Text"],

Cell[BoxData[
    \(CircledDigit[d_String, {x_, y_}] := \n
      \t{Hue[0], Disk[{x, y}, 0.9], GrayLevel[1], Text[d, {x, y}]}\)], "Input"],

Cell[BoxData[
    \(\(Plot[x\ Sin[x]\^2, {x, \(-10\), 10}, \n\t
      Epilog \[Rule] {\n\t\t\tCircledDigit["\<1\>", {4.9, 5}], \n\t\t\t
          CircledDigit["\<2\>", {\(-9.4\), 0.5}]}, \n\t
      AspectRatio \[Rule] Automatic]; \)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["9. Animations", "Section"],

Cell[CellGroupData[{

Cell["Creating Animations in Notebooks", "Subsection"],

Cell[TextData[{
  "All versions of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " can create animations. Animation results when a series of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " graphics are displayed quickly in succession to create the illusion of \
smooth movement. "
}], "Text"],

Cell[TextData[{
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " provides many features to aid this process. Here is a simple example. The \
command ",
  StyleBox["Table",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " creates an array of results by iterating commands. Here we will create \
ten different plots of ",
  StyleBox["Sin[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["a  x",
    FontSlant->"Italic"],
  " ",
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  ", letting ",
  StyleBox["a",
    FontSlant->"Italic"],
  " vary. Notice that we explicitly set a ",
  StyleBox["PlotRange ",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  "because the default for ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " is to pick a new ",
  StyleBox["PlotRange",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " for each frame of the animation, which could ruin the animation effect. "
}], "Text"],

Cell[BoxData[
    \(\(Table[\n\t
      Plot[Sin[a\ x], {x, 0, 10}, PlotRange \[Rule] {{0, 10}, {\(-1\), 1}}], {
        a, 1, 5, 0.5}]; \)\)], "Input"],

Cell[TextData[{
  "(The cells of the animation have been deleted to save space. Enter the \
code into ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to see the animation.)"
}], "Text"],

Cell[TextData[{
  "If you have three-dimensional graphics with different viewpoints, you \
should similarly set ",
  StyleBox["SphericalRegion", "Input"],
  " to be ",
  StyleBox["True", "Input"],
  " to ensure that the scaling of different plots are the same."
}], "Text"],

Cell[BoxData[
    \(\(Table[
      Plot3D[\((2 + Sin[x])\)\ Cos[2\ y], {x, \(-2\), 2}, {y, \(-3\), 3}, 
        BoxRatios \[Rule] {1, 5, 1}, Boxed \[Rule] False, Axes \[Rule] False, 
        ViewPoint \[Rule] {a, 1, 1}], {a, 0, 1, 0.1}]; \)\)], "Input"],

Cell[TextData[{
  "(The cells of the animation have been deleted to save space. Enter the \
code into ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to see the animation.)"
}], "Text"],

Cell[TextData[{
  "Although all that is necessary to create an animation in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " is a series of graphics, a number of animation functions are defined in \
the package ",
  StyleBox["Graphics`Animation`", "Input"],
  "."
}], "Text"],

Cell["Needs[\"Graphics`Animation`\"]", "Input"],

Cell[BoxData[
    \(\(?Graphics`Animation`*\)\)], "Input"],

Cell[TextData[{
  StyleBox["MoviePlot[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " ",
  StyleBox["f[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["x",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["t",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox[", {", "Input"],
  StyleBox["x",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  Cell[BoxData[
      \(TraditionalForm\`x\_0\)]],
  StyleBox[",", "Input"],
  " ",
  Cell[BoxData[
      \(TraditionalForm\`x\_1\)]],
  StyleBox["}, {", "Input"],
  StyleBox["t",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  Cell[BoxData[
      \(TraditionalForm\`t\_0\)]],
  StyleBox[",", "Input"],
  " ",
  Cell[BoxData[
      \(TraditionalForm\`t\_1\)]],
  StyleBox["}", "Input"],
  " ",
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " will animate plots of ",
  StyleBox["f[",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  StyleBox["x",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["t",
    FontSlant->"Italic"],
  StyleBox["]",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " regarded as a function of ",
  StyleBox["x",
    FontSlant->"Italic"],
  ", with ",
  StyleBox["t",
    FontSlant->"Italic"],
  " serving as the animation (or time) variable."
}], "Text"],

Cell[BoxData[
    \(MoviePlot[Sin[a\ x], {x, 0, 10}, {a, 1, 5, 0.5}]\)], "Input"],

Cell[TextData[{
  "(The cells of the animation have been deleted to save space. Enter the \
code into ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to see the animation.)"
}], "Text"],

Cell[TextData[{
  "Notice that ",
  StyleBox["MoviePlot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " is essentially a shortcut for using the ",
  StyleBox["Table",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " command. One difference is that ",
  StyleBox["MoviePlot",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " automatically uses the same ",
  StyleBox["PlotRange",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " for each frame."
}], "Text"],

Cell[TextData[{
  "Another interesting animation results by varying the viewpoint, thereby \
creating a revolution or a flyby of an object. ",
  StyleBox["SpinShow",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " automates this process."
}], "Text"],

Cell[BoxData[
    \(\(SpinShow[\ Graphics3D[\ Stellate[\ Icosahedron[]\ ]\ ], \n\t
      Boxed \[Rule] False]; \)\)], "Input"],

Cell[TextData[{
  "(The cells of the animation have been deleted to save space. Enter the \
code into ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to see the animation.)"
}], "Text"],

Cell["\<\
Other effects can be achieved by varying colors, options, ranges, \
and so on. \
\>", "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["MathLive",
  FontSlant->"Italic"]], "Subsection"],

Cell[TextData[{
  StyleBox["MathLive",
    FontSlant->"Italic"],
  " is a real-time visualization environment that can be used with graphics \
generated within ",
  StyleBox["Mathematica.",
    FontSlant->"Italic"]
}], "Text"],

Cell[TextData[{
  "As we have seen, ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " animations are a collection of frames that are viewed in succession to \
give the impression of  motion. ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  " allows us to load a graphics object created in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " from a file or directly through a ",
  StyleBox["MathLink",
    FontSlant->"Italic"],
  " connection and manipulate the object in real time."
}], "Text"],

Cell[CellGroupData[{

Cell["Files", "Subsubsection"],

Cell[TextData[{
  "You can use ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  " with files in various ways. One is the 3-Script file interface to ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ".  Using the ",
  StyleBox["ThreeScript", "Input"],
  " package, you can write a representation of three-dimensional graphics to \
a standard text file format.  This may be read directly into ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  ".  However, because of the limitations of the 3-Script format, you will \
normally find that the use of ",
  StyleBox["MathLink",
    FontSlant->"Italic"],
  "  is preferable, since considerably more information about the surface can \
be passed directly via ",
  StyleBox["MathLink",
    FontSlant->"Italic"],
  ". Another type of file is the texture file, which allows you to add \
textures to your graphics. ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  " also has its own file type called an ASCII mesh file. ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  " scenes can be saved in this format and reloaded later in new sessions of \
",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  ".\n"
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`ThreeScript`\>"]\)], "Input"],

Cell[BoxData[
    \(\(bell = 
      Plot3D[Exp[\(-x\^2\) - y\^2], {x, \(-2\), 2}, {y, \(-2\), 2}, \n\t\t
        PlotPoints \[Rule] 25, \ Boxed\  \[Rule] \ False, \ 
        Axes\  \[Rule] \ False]; \)\)], "Input"],

Cell[BoxData[
    \(ThreeScript["\<bell.ts\>", \ bell]\)], "Input"],

Cell[TextData[{
  "Once we have a 3-Script file, we can launch ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  " and open the file."
}], "Text"],

Cell[GraphicsData["PICT", "\<\
=nP000000BL15@0A0_l<0?ooool000000000004E00019`000000000N01[ooooo
ool00@0:0000004W0AD0P@0:000000000000X02201X000000000VH4F0000004W
0AD00000000000180000B0000000200100P000000>LNj00000000175P000:`00
oooooooo006IU000c<`00YRH001oO@03IOnGoVCn00B2PP00PWl01Lc<002IU`06
c<bHV<c;00L000L75QH0261P001PGP09IF8009RH00Xc<P00IVH02eYJ0010?`0<
005HF00000fT5[E2Y1H03YRHV9W<c00?W9bLW9bL013<c9RHV9P04KRh^;Rh^00B
bBgLMlT]01=7AdM7AdL05?ooIFG<b@0El_9/K?;_01KM6oBem;D05l[:003:a@0H
d_DQom;a01TaZbbc;;<06RLXCdlW9`0Kk^kBd^3P01cool_;ooh07C4a4A4P800N
e=CDe>;R01o;bl_<ool08:BT002TX00Q^[X00;Zf02;Lg?3`g=`08oKfj>SDe00T
M7AdM7Ad02F4Q8B4Q8@09^g]WIg]j`0WoonIVOom02S<c4a<c<T0:McLB4SLf@0Z
BmiTdVCB02/00000000000000BL15@0000019`4E00002P0000019`4E00V1:h4[
kB/100003@0[PAn17o0O0`d[:`016Gl[7b<08a/08`0S6`0S02<K02<08a/08`0S
6`0S02<K02<08a/08`0S6`0S02<K02<08a/08`0S6`0S02<K02<08a/08`0S6`0S
02<K02<08a/08`0S6`0S02<K02<08a/08`0S6`0S02<K02<08a/08`0S6`0S02<K
02<08a/08gl08a/08`0S6`0S02<K02<08a/08`0S6`0S02<K02<08a/08`0S6`0S
02<K02<08a/08`0S6`0S02<K02<08a/08`0S6`0S02<K02<08a/08`0S6`0S02<K
02<08a/08`0S6`0S02<K02<08a/08`0S6`0S02<K02<08a/08`0S6aD08`0S6`0S
02<K02<08a/08a/=:b/00AUo:al08P0F01h07P0F6b805P0N01h05Q/R01H07P0N
01HK8P0F01h07P0F6b805P0N01h05Q/R01H07P0N01HK8P0F01h07P0F6b805P0N
01h05Q/R01H07P0N01HK8P0F01h07P0F6b805P0N01h05Q/R01H07P0N01HK8P0F
01io01h05Q/R01H07P0N01HK8P0F01h07P0F6b805P0N01h05Q/R01H07P0N01HK
8P0F01h07P0F6b805P0N01h05Q/R01H07P0N01HK8P0F01h07P0F6b805P0N01h0
5Q/R01H07P0N01HK8P0F01h07P0F6b805P0N01h05Q/R01HE01h07P0F6b805P0N
01h06aH03B/[003P0R/O:]PTOah08P0R028K01h08ah08P0R6`0N02<N02808Q/0
7P0S7P0R028K01h08ah08P0R6`0N02<N02808Q/07P0S7P0R028K01h08ah08P0R
6`0N02<N02808Q/07P0S7P0R028K01h08ah08P0R6`0N02<N02808Q/07P0S7P0R
028K?P0N02<N02808Q/07P0S7P0R028K01h08ah08P0R6`0N02<N02808Q/07P0S
7P0R028K01h08ah08P0R6bDT:^dT1a/J6AXI6AXCoQT36P0Z9OdT0`d[:`0157D[
7`0K01/S02808P007P0K8P0R028001h06b808P008P007P0S028F01/08P006`0K
8P0[8P007P0R01/[028001h001h06`0[:`0N02/[01/07P0K:b/07P0K02/[01h0
01h06`0R000N000N01/08P007P0KoR]o028001h001h06`0R000N02806b/08P00
7P006b/K028001h001/R02/[0280:b/N01/08P007P007P0K028001h001h06`0R
000N000K01h002/[7P0K:b/K028001h001h06`0R000N000N01/05Q/001h08`0R
000N01/R028:000N000N01TO7akj7`X06`0N02<03R/[003Z0B/On2@2:R@UnR@2
:R@Uo2@0:_<TOb8K02806a/[:a/F028K02/[6`0R8P0R6`0R01h[:a/08R/[6`0N
01h0:b/K02807P0K5P0R8`0R01h08R808Q/08P0N028N01h[:`0K5Q/08P0R8`0R
01h08R<07P0R02/[6aHS028N:b/07P0R8P0R6`0K02/[6`0N:b/08P0N>`0R8P0R
8`0R01h08R808Q/08P0N028R028R028S5Q/[:`0N01/05Q/08R808Q/08P0N028R
028S02808^XT300J7`d>40dJ3@dI7b?k90<=:b/00@8b:ahS01h05P0K01/06`0N
01/06`0K01/07P0K01/06`0N8`0N01h08`007P0K02806aH0oR/38`0K0?h[0QH0
6od[0P0N0?d[0Ah0o2/F8P00:b/F01/R02/[01/[:`0K:b/N01_m:`H07P0N8`00
oB/201/0oB/<7P0K:b/K:b/N01h06oh[0a/08P3n:`8N01_m:`807P3m:`4R0?`[
0`0N01_m:`8K7P3h:`<07P0KoB/27P00oB/6000[:`006oh[0a/F01_m:b8L8P0N
01/F6`0N02<001h06`0N028001/06b808aH06aTO4?h=3ad=3ATO01h001h06`d[
:`00h@4[7oHT00?h9003n2@09OLT1P0R01h06aKm:`4K0?d[A00R:`0K02/[0280
:b/0028[:`0K:b/S7R/[02806`0[:a/0:b/N02/[028[:`0K:b/08`0N0280:b/0
8P0[:a/0:b/N0?h[1P0K01/07P3m:`4N0?d[D@0N:`0K02/[01h0:b/06`0[:a/0
:b/08R/[01/[:`0S:b/06b/[01/[:`0R:`0K02/[6`0N:b/08Q/[:ah0:b/06b/0
8R/06`0[:a/002806`3Z90`06Ql?4@d>4a0>6AlSnb@33B/[004;=2/O028001/R
01H07P006`007P0F01/001/07P0F000N000K8P007P0K8P005Q/N02<F02/0o2/5
02/[7Q/0o2]88P0K:b/R01/[:ah0:b/N02/[8P0N01/[:`0K:b/06b/[8`0[:ah0
:b/N01h08P0K:b/N01/[:`0K:b/06b/[01h07P0R01/[0?`[1@0[:b800?`[9280
6b/[7P0N:b/06b/[6`0[:ah0:b/N02/[7P0[:b80:b/06`3l:a<08P0[:aHK02/[
01/[:b8001HK0?`[=00N8P0N01/R01h06b807P0R000K01h07P007P0K01/I7`d>
40dI3@dI7aH06aH06`0=:b/00>82:alZmb@0:_XT0@<ZkB@::P006aH08ah[6`3n
:bh07R/[000[:`0K:b/07P0[:a/F02/[028[:`0K:b/K000R02/[6`0[:a/0:b/0
7_/[800N01/08P0[:`0R02/[6`0[:a/0:b/002808Q/05R/K0?h[6`0R:b/S6b/[
01/[:`0R02/[02<0:b/K02/[01kk:c<07R/[028[:`0N:b/K02/[01/[:a/07R/[
000R:b/K02/[6aHS6`0[:`0K:b/06`0R01h0lR@00oXT300I7aTI6Q<I3Q0J7Q_k
90<=:b/00AUo:alS02806`0F8`0K02<001h08`0N02<06`008`0K8P006aH08P0K
01H06`0K8P0R01h0:`0N02/08`0[:`0K:b/N02/[8P0K:b/001/[:b<0:b/N02/[
01hR01h[:`0N:b/06b/[7P0[:`006`0S02808P0K:b/R01/[:`0K:b]o5P0[:ahS
7P0K0280:`0N02/K02<[:`00:b/N02/[7P0K:b/05Q/[:`0K:b/R02/[01h002<0
:b/S02/[6`0[:aH0:b/N02/[0280:b/S5Q/[:`0K:b/08`0F02/[7P0[:b807P0R
01/F8`0F01/001h06b805P0K01hS01H05Q/16Aon3A0>40d=6Al05P0F01/03B/[
003J0B/OebAo01/F6`008R/K028K01h0:b/K02/[028[:`0R02/[6aH0:b/07R/[
028[:`0K0280:b/K02/[6`0[01/0:b/K5P0[01hS5Q/N02/[01h0:b/K02/[01h[
:`006aH08Q/K:`008P0N6`0[:ah0:b/08R/[01h0:b/K01H[:a/0:b/k6`0[:a/0
8R/07R/[01h[:`0R:b/K5R/[028[:a/07R/[01/0:b/K02/[6aH[6aH[:`0R:b/K
01/N01h0jR@<01XO40h@3`d=3QTO8o/T0`d[:`012Cl[7`0K01/07P0K5P0N01/R
02805P0K01h08P0N01H06b805P0K01h08P0N02806`0R01/[5P0K01h06b/[0280
o2/?7P0R02/[01/[:ah0:b/06o`[1@0K:b/F0?d[11/R0280oB/=01h001/001/[
:`0S8P3m:aTK8P0[:a/R02807P00:b<F6b8001/[:`0R0?`[428001/[:b80:b/0
7R/[0280oB/?01h0:b/N02/[01/[:`0S0?`[300R000[:`0K:b/07P3n:`<K02<0
o2/d028F01/08P0N01/R01HS028002806`0N000K01h06aTO6QTI6Q<I4aTN7P0K
8`0N00d[:`00h04[7o4T02[X93DK5P0R6aH06`0R01h08P0K01h08Q/06`0K01hS
01/08P0K000R6`0R01/06`0K5P0K01/F01_n07lN6`0R6`006aH08Qh07P0F01/N
000N000K000K01H001h06aH08Q/07P007P0N01/07P0R8`0K01/08Qh001/08Q/0
7P0K01h08Q/06`0R02<06`0R6`0N01/07P0N01/06`0N028K01/F01/07P0R000K
01h07P0K01/07P0K00@F8`0Z9OTT02[f908301Wi7`8N7`3k90<=:b/00AUo:alR
02805P0R000K01H06`0K01/07P006`0N02<07P0K01/R02807P0S01H06`0N01/R
02807P0R01/F8aH06`0N01/F01/07P0R01h06aHK01h07P0N01h06b807P0K8aH0
7R8001HK01hS5P0K6`0R000K8P006b806b807R9o01HK8aHK8P0R01/07P0K8P0R
01h07P0K6`0N01h05Q/06b805Q/07P0K028001/07P0N02807P0F6`0K01h07P0R
02808P0R000K01h06b808P0R01/R01h07P0R01h07P0K01hS01h06b8001/07P0K
01/07P0K02805P0K01hE01/06`0K01/S01h06aH07P0S3R/[004IOb/O8a/08Q/0
6aHS5P0S7P0N01/08P0R6`0N02808P0R02806aHS000R5R<K5P0R01h06aHK000R
6`0F01h08P0N01h002807P0N02808P0S01h08`0N02807P0K5P0R01h08P0K7P0S
01h07P0R01HK028N01/F6`0N01HK000NO`0S01h001/F8ah07P0N01/07P008P0R
000N01h08R<07P0K5Q/07P0R01h08Qh06`0N02807P0R6`0N02808`0N01/K5Q/0
7Q/F02807P0K5Q/06aH06aH08P0S01h08P0R02807P008P0K5Q/08P0N01/08P0R
028K028K01h05A/08P0R02807P008P0K5P0R00d[:`00P`0[oPd03oX=00ob3@0?
oPd03o/=0Pl=3_h=00oi3@<?3@d?m`d03oP=00om3@0>kPd03_P=00of3@<?3@d?
oPd03oH=0`l=3@om3@0?o@d03^4=00of3@0?m`d03o`=00ok3@0?o@d03o`=00oe
3@0?n`d03oh=00om3@0?n@d2:b/000cn:`07PB^1:o0[00016Gl[6`0K01/06`0K
01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K
01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K
01/06`0K01/06`0K01/06`0K01/06gl06`0K01/06`006`0K01/06`0K01/06`0K
01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K
01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K01/06`0K
01/06`0K01DK01/06`0K01/06`0K01/06`0S:b/00AUo:`0N01h07P0N01h07P0N
01h07P0N01h07P0N01h07P0N01h07P0N02807P0N01h07P0N01h07P0N01h08P0R
02808P0R01h08P0N01h07P0N02807P0N01h07P0N02807P0R02808P0N01h07P0N
01h07P0N01h07P0N01h08P1o7P0N01h07P0R6`0N01h07P0N02807P0N01h08P0N
01h07P0N01h07P0R01h07P0N02807P0N01h07P0N01h07P0N01h07P0N01h08P0N
01h07P0N01h07P0N01h07P0N01h07P0N01h07P0N01h07P0R01h07P0N01h07P0N
01hE01h07P0N01h07P0R01h08P0N02/[004IOb/R02808P0R02808P0N01h08P0R
02808P0R02808P0N02807P0R02808P0N02808P0R02808P0R028001/F6aHK7P0N
6b805Q/R02807P0S5P0K8P0R02808P0R01h08aH06aHK7P0R02808P0R02808P0R
02808P0R02805Q/RO`0R02808P007P0N02808P0R01HK8P0R028001/R02807P0R
02808P006b808P0R01HK8P0R02807P0R01h08P0N02807P0R02807P0R02808P0N
02808P0R02808P0R02808P0R02808P0R02808P0N02808P0R01h08P0N02805Ah0
8P0N02808P0N02808P0N01/[:`016Gl[6`0N6`0R6`0N6`0R02808Q/K02807Q/0
7Q/08Q/0028K000R6a/07Q/0028K6`0R01hK01hK028K5R806`0K000R01/08Q/0
7P0N028K5P0R01hK028K01hK02807P0R7P0K01/N01hK01hK028K6`0R01hK01hK
028K02<F07lN6`0N6`0R8P0N028K01hK028K01hK028K7P0F6`0K028K028K6`0R
7P0K7P0R6`0S01hK028K000R6`008Q/0028K000R6`0N02<N01hK01h08Q/08Q/K
02807Q/06ah08Q/K028K6`0R6`0N6`0R6ah08Q/07P0R6`0R028K01D08Q/0028K
01h08ah07Q/00280:b/00AUo:`0R000K7P0R01H06b806ah001h07Q/08P005Q/0
8QHK5P0N6aH07P007R<N000N01hK02805P0K5P0K8P0N5P0N6b805Q/07P0R01HK
01h06ah001hK02805P0K8P0N6`0R028F000K02805P0N000N01hK01HK02807P0N
01io6`0R000K7P0K8P0K01h001HK02805P0N000N8`0R01HK01hK01H07P0K5P0K
6`0R01h001h06b807P0K8Qh07R<R01HK7P0R02<F6`0K02805Q/06ah001h07Q/0
8QH001/05P0N01H07P0N000F6`007Q/08P0K8aH07P0K01hE8P0K8P0N01hS01h0
7P006a806b/[004IOb/K028N000N6`0R6`0K5P008Qh08P008Q/N02<F6`0S028K
000R6`0R6`0F028N02<05Q/08Q/F02<N01/F02<N000N02808P0N8a/05R<08P0K
5R<05Q/K028K01/F000R6`0N028R028K6`0R6aHK0280028K028K000R0280O`0R
6ah05P0R01/F028K028K01h08Q/08ah001hK028K01h05R<K5P0R01/F01HN6aH0
8Q/08P0N8`0N01/07P0N028K000N6aHK01HR028K02<F01H08Qh08`0F6`0S7Qh0
8Q/F028K5P0R6aHS01HR6`0F6`0R01HK02<N01h05A/F01hS01h05R808Q/08P0K
5P0[:`016Gl[01/N01/R01H06aHK5P0N8ah001/07Q/F02<F6`0F6aHK5R<F6`0F
6aH06aHK01/F6b806ah08`0N01/F02<F01/R01HK7P0S01h08P0F6aHS5P0N01/F
01HK01hF8`0N6`0R01HK01hK01H07P0S02808aHK5P0N02<F6ah08glF6aH06b80
6ah08P0N02<N02807R<05Qh06b8F01/N02808P0N02806b8F01/R000S01h05Q/F
6`0R02808P0R000K01hS5P0K01HK01/N02806b8R01/07P0N8aH07P0R01h001/K
000N8`007R806`0N8`0F6aHK5P0N02808QD06aHK02805Q/06a/08P0N5P0N:b/0
0AUo:ah002807R<K5P008P0R01h08R8K000F6aH07P0R6`0K01h08P0R6`0R7P0S
5Q/06`0K5P008Qh002808Q/F6`0F6`0S01hK5Q/07P0R6`0K5P0R02808ah05R<0
6aHS01HK6`0S5Q/05R<K5P0R01hK7P0S01h06aHN000S7P1o7P0K5P0S7P008Q/N
01h0028K000N02<07P0K0280028K01/F028K7P0K02806ah07R808Q/06`0N6`0N
01/F6aHR02807P0R028K01h001hK5P0K01h08P0R01/08P0R8a/08R807P0R6`0R
6`0N01HK01h08`0N02<N028K01hE01HR028K02<F6`0F6`0N02<N02/[004IOb/0
8P0K8P0N000N8P0K6b808P0K01HN8`0K8P0F6`0N028001/F6`0N8`005Q/08P0N
02807Q/08QHK5Q/06`0N8P0K5R805P007P0K6`0N01h06a/R01h07R<07P0K01HK
5P0R5Q/08P0N01/R01/N000K5R808P0K02807P0NOb<05R<07P0N8ah002<08QHK
01hR01h08P0N8P0N01hK01hF8`0N000S5P0K8QH08P0K01/07P0R000F8b8F01/0
8ah08P0K7P0N02805Q/R000N01/R5Q/06ah08P0K7P007P0K8P0N8P007P0N01hS
028001hR000N000K01h05B806`0N01/N01H07P0K8P0N01/[:`016Gl[6`0R01/0
028S01h05P0K6`0R6aHS01h05Q/08aH05R8K5R806`0R01hK5P0F8aHK028K5P0F
8a/05R<07P0R01/F02<07P0N8P0R01HS02808P0N02808P0N01HR028K01/F6`0F
8`0R01h06aH0028N01/08Q/F028K5P0N07lF8`0N000R01h08Q/F6`0S5Q/06aH0
7Q/001hR02805Q/06aHS028N01/F01/08Q/F02808P0R6ah06`0K02807P0K7P0R
000N028K02806aHS01h06`0F8P0K5Q/F01/F6`0R01/06`0R7P0R02<07P0N6aH0
6b8K028N02807QDK01HS01h0028S7P0R01/F02<0:b/00AUo:`0F6aHK8QHK6b80
5Q/R01H07P0K01h07R<08Qh07P0N000K8P0F6b805P0K8P0K5P0N01/R01h07R<0
7P0R01h08aHK7P0N8`0S01hS01h06a/R02806a/R02806aHK5P0F8`0F6`0N6b80
8P0K8QHK01h06ah06aHK01/R001o6`0R5P0N6b8001/07P0K5Q/07P0S4P006b80
8b807R<08P0N01/N02<F01/R01h06b807Q/R01h08P0R01HK8P0K5P0S01hS8P0K
01hS7P0R02808P0N8`0N01/F01/F8`0F6aHK8P0R01/06`0K5Q/R000K8P0K01hR
000K5Q/E02<R01h08aHK7P006b806b807R/[004IOb/K01HK01/05P0N6`007R<K
5P0R02808P0N01/08Q/08ah05Q/08P0K5R<N01/F02<N01h06`008P0N01h08ah0
7P0K000R01hK5Q/07P0F8aH06ah07P006ah08P0S028K01h08aHK000N6`0R01/0
5R<07P008P0S5Q/07R8KOaHK02<N000N6aH08P0R01h08P0N01/F8P0K5P0K7P0N
01/F0280028F6aH06aH08P0K7P007Q/07Q/F6aHK01h05Q/F6`007P0R01h001/F
6ah08Q/001h08Q/08ah001h08P0K01h07Q/08QHR01/07P0R01h05R<06ah05R<0
5Ah06aH08QHK000R7P0K5P0K5P0[:`016Gl[01hS02805Q/R000S8P0N02806aHK
8P0N02<F01h08aH07R<08P0K5P0K02808P0N02<08P0R6ah06b8001h08P0S5P0N
6b805P007P0R01h08P008P0K8QH001/N01HK5R<F01h001hR000N02806b808P0N
8P0K5Q/07P0S8Wl05P0N01/R000R01h06b8001/R01/F01/07P0N000N02806b80
6ah06`0K8P0R7P0S5P0K5Q/08P007P0S02806b806`0R5Q/R01/R01h08aH001/0
7R805Q/07QH06b805Q/R02805Q/08P0K01h08P0N8P0K8P0F6aH07R<08QD08P0K
8P0K02807P0K5P0S5P0K:b/00AUo:a/07P0N02807Q/F6`0R028K7P0K01/06aHK
02808P0N01/F6`0R6`0R028K5Q/F028F6`0R6`008P0N8ah07P0R028N000S7P0R
6`0R6`0K7P0R6`0R01/08R806aHK01/F028K5R<07Q/F01hK5P0K7P0R01/F01h0
8P0N01]o028N02807R<N028K5P0N6aH06aH08ah08P0R6`0R028K5P0K5P0R6`0R
01/F8`0N01/F02<07P0R6`0N02808P0K5P0R6`007Q/06aH08P0S7Qh07P0S7P00
8P0S7P0N02<07Q/08Q/08Q/F028K5Q/06`0R01hS01HS7P0N6`0E8Q/F01hK028K
5P0R028N02<N02/[004IOb/07P0R01hK8P007P0R01/N000R01h08P0N01/R01/N
02<F000N6aH07P0N01/08aHK01h06aH07Q/R01HK02<R01hK02807P0K8P0F6`0R
5P0K7P0N02806ah06aHS01/F028K01/F8P006b8001/R5P006b806b8001/R01h0
Ob806`0K8P0N01HK01/R000K8P0R01h06b806`0N8P0K01/R02807P0R02806aH0
8b8F01/F8P0K7P0R01h07Q/R02<F6`0N8P007P0K8P0N7P006b807P007R<07P0R
01/N8P005Q/F01h08P0N01/08P0F6aHK02806`0R01H05Ah06b805P0N02806b80
6`0N01/[:`016@@[6`0R6oh0OahS7P0R6`0F028K02808Q/F02806ah001h08R8K
01HK8P0R8aH07P0S5Q/08P0K5P007P0K01h06ah0028K7P0N01/F6`0N02<N000R
6ah07P008P0N01h08ah05Q/06`0R7P0S7P0F6`0R7P0K5P0R6aH06`007Q/F0280
6`00Ob8K01h06aHR01/F6aH08P0K7P0R01/N01h06aHK01/F6ah08P0K7P0K0280
6`0R000R6`0F8`0F6aH08P0N01/F8`0R01hK000R6`008Q/08Q/F6`0N028001hK
5R<08Q/07P0N02808Q/F6`0S5Qh08P0R6`0R6`0N01/08R<03AhS7P0K5P0R0280
:b/00AUo:aH07P0K8R807P006`0R01/N02806ah06b807QH08aHK8P0K5P0K01h0
6`0K8P0N01/F01/R5P0S5Q/R02806aH001/N000R01/F8`0K8P0N000K7P006b80
7Q/R02806aH07R<08P0N8P007P0N8`0N02<R02807P0K8P0K8Qio02<07R<F01hK
02808P0R01/F8`0K8P0S5P0R01h002806aHS02806aH06a/R000N01HK8P0F6aHK
7P0R01h06aH06a/R02<F01/F6b8002<F6aH07Q/08Qh06`0R02806b805R<05Q/0
8P0N8`0K01h08aH07P006a/F6`0N6`0E8P0R02806aHK000N01/F6b806b/[004I
Ob/K000R01/K01HS7P0N02806aHK5P008P0K02<08QHK01hK01/F02808P0N01/F
02808ah06`0R6`007P0N01HS7P0F028K02806aH05Q/05R<N000R6`0K5P007Q/0
8P0K000N01h06`0R8`0N01HS01h06aHK000R01h05Q/0Ob807P0N02805R<F6`0R
6`0F6`0N01/N01h08ah08Q/08P0K5Q/08P0K5P0F6aHS028K01h08aH002<N01h0
8P0K5P0F6`0N02807P0N02808P0S7P0F6`008P0R01hK01h08Q/07P0F8`0R01h0
8Qh07P0S7P0R6`0F6`0N01HK5@0R6ah08P0N028K02808aH07P0[:`016Gl[028F
6b8F02807P0N01/K8P0N02<07Q/R01h04P0K028002<F028N01/N01/F8`0N01h0
6aH07P0N8P0F6b807P0F6b8K01h08aH07R<08P0N02807P0N02<R7P005Q/R01hR
02808aH06aHK8P0N02807P0S5P0N6b807R<F07lK8P0N02<R01/F6`0N6`0N8P0N
02<F01h08aHK01h05Q/R01/07QHS5P0N8`007P0N01HS01h06b8F01/S8P0N02<F
6`0R5P0K8P0R01h07Q/F6aH08aH07Q/R01hK02807Q/08P0F6`0N6b805Q/08P00
7P007P0F6`0R02806aDR01h001/N000K5P0N01h08P0K:b/00AUo:a/08`0N028K
000R02805Q/0028N000F6`0R6`0N028K5Qh08ah0028002807P0R02808P0S7P0R
01/S01HK028K000K028K5Q/08P0N028K028K7P0R01h06`0R6`0F6aH08ah07P0R
7P0K01/F01hK01HK028R000N000N029o01h08ah06ah06`0R01HK01/F02807P0R
6`0N028S7P007P0F8`0K5P0R01h08Q/06aHK02808Q/07P0F01/N02807P0R6`0R
6`0N02<08P008aHK5P0S7P007Q/0028K000N01hS6`0N000N6`0F8a/F8P0R8`0R
6`0N028K01HE6`008R806aHR02<N02807Q/05R/[004IOb/08P0N01hK028F6aHK
5Q/07R806`0N8`0F6`0N01HK028001h001/R01hK8P0R01/K8P0N01/R02<F01hS
01HK02<R5P0N01/05Q/R01hK5P0K5P006b806aH06b807P0S01h07P0F6`0N01h0
6b8001hS01HK01/R01hR000KOah08aH06aH08P0F6aHK02808P0K7P0F6`0R000N
000K5Q/R01HK5P0K6b807Q/07QHS02806ah08P0K8P0N000K8P0F6`0K7P0R000K
7P0K8P0N02<07P0K8P008aHK01hR02805P0N02<F6`0R01/F01/06aHK7P0R01HK
01HK5@0R7P0K8P0K8P0N000N01/05Q/[:`016Gl[6`0N02<05Q/08`0R02<F02<N
01HR01h08P0R01h07P0K8Qh08R8K02806`0R6ah001hK01h06ah06aH07P0R0280
7P0R02808Q/06`005R<N02<N01HK5P0S7P0K5P0R6aH08R<K01HS01h08P0S7P00
7QHK02806`0S7P0R07lK5Q/07P0S7P0R6`0K028K01h05R<K01HR6ah06aHK0280
7Q/05R8K000N02808P0K5Q/08P0K5Q/05Q/08Q/F01/08P0N000R6ah05P0N01h0
028N6`0R01hK5P0R01h06`0N8a/F02807P0R6aH08ah08P0K000R6`0R6`0R01DR
6`0R01h05Q/K028R8ah08Q/0:b/00AUo:`0N8`0F6b806aHR01/F6`0N02806b80
01/N01/S8P0R01/08P0K01/K8P0F6aH06aHK02806aH07P0K8P0N8aHK5R<R01/N
01h07P0R8P0N000N028N02<07P0R02806aHS01/F01hS01/R01h07P0S7P0R000N
01h07P0N6b9o01/F8aH07P007P0S5P0N02808P0N02808aH08aHS01HK8P008P0K
01hR01HK01/R01h08P0N01/07R<05Q/06b807Q/R01hR01/05Q/R01hS01hK01H0
6b805P0K7P0K8P0R01h06b806b807P0S01h05Q/R02807P0N6`0F6aHE6aH06ah0
6b8001HK6`005Q/06b/[004HOb/K01h07P0K5P0K6aH06`0R01hK01h07R800280
5Q/K02808Q/N01h05Q/08P0K5P0R028K5P0S7P0R01/001h06`0F6a/0028K000R
6a/0028R02806aH08P0R6aHK5P0K5Q/06aH07P0F028S7P0N028K028N02808P0S
01HK@1H06aH08P0R7P0N028K028K6`0R028K6`0K5P0K5Q/001hS7P0R02806a/0
5R806`008Q/N01h08P0N02808P0NoP1A7P007Q/08Q/06aH07P0R028K5P0K5R<N
000R01/N02807P0K5P0K5P0R6aH08Q/07Q/08Q/001HS6`0K01h05P0R01/F8Q/0
5Qh08aHK02/[004IOb/07P0R01h06b8F01/R01HK5Q/08P0R01/07Q/R01h08P0K
7P007P0F6`0R01/F8`0N01/F02<F01h06aHS7P0R02807P0R01h05Q/N000R5Q/K
02<N01/R01/N01/06b806`0N01/R02807Q/002805Q/07R<06b806aHK5Q/0Oa/R
02806ah08b8001/F01/F01/N01/F01h06b8F000N8P0N000K7P0S5P0N8`0N028F
6aH08aH06b805Q/F6b807R<N02<R01H07P0R01/R028K01h06b<F01h08aHK5Q/0
8P0N02<F01/F8`0N02<F6`0N000K7P0R8P0N02805Ah08b806b8F01/05Q/S01h0
01h[:`016Gl[6`0R6`008P0K02807P0R02<08Q/K02808P0K02807Q/F000R6`0R
6`0N02806aHS02806ah06`008P0N000N6`0N6`0R6ah08Q/001hK02807P0F8aH0
7Q/002808P0N01HR02806aHK02807Q/08Q/07P0N01HK6`0K5P0F6gl07P0K8P0K
7P0K5R808ah06aH0028K01h07P0K028R01/05R<N000N02<N01h06aHK01HK01h0
8P0K5R<06`0N000N028K6`0R8`0R6aH06`0N02807P0N02807P0S5P0F8`0N0280
7P0R6`0R01h08aHK028N000R6`0R02808QD08QHS7P0K02808Q/05R807P00:b/0
0AUo:`0F6`0R5Q/R02806b806aHK5Q/F02807Q/R01h06aH08`0N6aH07P0S5P0K
7P0K5R806aH08QH07Q/R000N8`0R01H06aH001/05Q/R02807P0N6`0K8P006aHK
5Q/R02807P0K8P0K01hS5Q/05Q/07P0S01hS5P0R000N8`1o7P0S5P0K5P0K5P0K
01h07P0K5R<N02808P0R01HK6`0K8P0N01/R01/N02808P0S01hS02808P0K5R<0
7P0R01HK8P0K5P0R01h07P0K8P0R000K8P0R000K8P0N01/F6`0N8`0K7P0F6`0F
6aHK01h05P0N01/F6`0N6b807Q/E7P0K000R01HK5Q/07P0K8P0N8b/[004IOb/K
01HK02<F6ah07P0K5P0S5P008Q/08P0K01h08P0S7P0R000R8`0N028N000R01/0
7P0S7P0S7P008QhK01h08Q/08P0R8ah08Q/06aHK000R01HK01h08P0R01/07Q/0
8P0R01h05R<002808Q/07P0R01h06`0R6aHS01h0OaHS5Q/F02<F028S5Q/08P0R
02<F01hK01h08ah05P0R01h002807P0F028K01h07P007P0N6`0R01h07P0R6`0S
5P0R01/N028K000R01hK028R01/N028R01/08Q/08aHK01h0028S6`0N000R028K
028S01h08aH001h07P005A/N01hK028K02<F01/F01/F01h[:`016Gl[01/R0280
7P007P0S5P0K5Q/07Q/05Q/N01/R01h07P0F6aHK7P0R01HK01/R01h07R807P00
7P0F6ah002807Q/05Q/R01h001h08P0N01/R6aHK02805Q/F6b805Q/08P0K6aHS
8P0N01hK5Q/08P0R01h08aH07P0R01h07WlS01h06b806aHK01h05Q/07P0N01h0
02806ah001/F6b808QhK8P0F6b807P0K8P0R8P0N000F6ah06b806aH07P0K6b80
01hK028F6aHK02806ah001h06b805Q/F01h001h08R807P0R02<F6aHK01HK5R80
01h07R808`0N8aDF02805P0K5P0F6b806b806b80:b/00AQo:ah06aHK5P0R6`0R
5Q/F000R7P008Q/002807P0S7P0R6`0K000N6`0R02807Q/08P0K5P0R8P0R6`00
7R<08P0K5R<07Q/08R808Q/002807P0S5Q/08Q/06`0N02<F6`0R01h001hK0280
02808Q/K028K5P0S7P0R01h08P0e7P008P0N01HS01h08Q/08P0N028S7P0N01H0
8R808`0N01/0028K000R6`0R01/N01/F8R<NoP1L8P0K5P0S7P0R01h08Q/05Q/0
8aHK028K7P007R807P0K5R<07P0F8Q/07P0K01HK028F6`0S5Q/001hK5R<07P0K
5Q/001h08Q/K02808ah001h05Q/05Q/0:b/00AUo:`0R01h08`0K5P0K8P006b8K
02807P0K5Q/R01HK5P0K02808P0N02806a/R01H06b8001hK01hS01HK8P0N01/F
8`0R01H07P0K6`0R5Q/R000N000N6`0R02805Qh001HK7P0K5Q/08P0K8P0N6aH0
5Q/F01/F6`0N02<R01io02<F6b806b806b807P0K8P0K8P0K02807R<F6a/R01h0
8P0S5Q/F8b807P0N01h08P0S01h08P0N6b808P0N02807P0K7P0F6`0K5Q/05Q/F
01/R02<R01/F8`0R02806`0R02807R<07Q/R000N01HS5Q/08P0R01/F8`0F6b8E
01h001hR01h06b807R<07R<06b/[004HH2/K02808P0N02<N01/F8P0K028K7P0R
01h06`0R02<N01h08Q/N028K5P0F6`0R7P0K5R<05R<07P0S5Q/F02807P0N6`0S
7P0R01h06`0N028R8ah001HR6aHK02<08Qkn07lR02<F6`0R01hK000K5Q/06aH0
8aHK01h06`0N01h001h05Q/F01/F02806ah06aH08Q/001h001h08Q/07Q/F01/0
7Q/0028K000R6ah06`0R6aH001hK02807R<K01h0028K02807P0R6`0K5P0N6`0K
01HK01hK028S5Q/07S4K01HK5P006ah0028K02807R<F6`0F6`0R6`0K5P0R6`0K
7P0N01h001h05P0R02/[004IOb/F01/N01hR01h06aH06b805Q/F000K8P0R01HK
01h08P0F6aH001/06aHK01hK02<F01/F6`0N01/F6`0R01/N000K5P0N02807P0R
02805Q/K000N8P0K01/06ah07P0K8R806aHK01HK5Q/001hF8`0R02805Q/08P0F
6b80Ob806b807R<06b8001h08aH07P0K5Q/07R806b806ah08P006b808P008aHK
01hR01/08P0F6`0K8aHK02806b<F02805Q/R01HK8P006`0N01/R000N01hS01HK
01HK5Q/07P005Q/06b807P0R8Q/05Q/R000K01hS5P0N02805B806aHS5P0R0280
8b806b8N01h[:`016Gl[6aH001h06aH08P0S7P0K5R808ah05Q/07Q/08Q/08Q/0
8aHK8Q/07P0S5P005R<F6aH08aHK02806`0N02808ahF02<N028K6`0R6ah08Q/0
5Qh001hS5P0F8P007P0R01/K5P0N028K01/08Qh001h07P0S7P0F8`0N000R6gl0
8P0N000N01HK8R<07P0R6`0R01/07P0K01HK01H08Q/N01HK5Q/F7P0S5Q/07P0F
6`0R6`0N01h05R<F6`0F6aHK02<F6`0R01hK5P0R02807Q/F02807P0R028K01/0
8P0N8Q/07P0K5P0R6`0F8a/07P0N01h0028N028K5QDS7P007P0R6`0N01h08P0K
0280:b/00AUo:`0K8P0K8P0S01h07P0K5P0K01h07R<08P0R01h05Q/07QHK01h0
8P0K5P0K8Q/F6`0R01HK01HK8P0R01/N01h08P0K5P0K000R01h001/05Q/08ah0
01/R01/R8P0N02<F01/R000N02805Q/08Qh08P0K7P0R01/F6b<F6`1o7Q/F8b8R
01/R01/07P0F6aH06b808P0R028R02806ah001HK01/06`0K5Q/06b807P0K7P0R
02808P0F6`0F6`0N02<F6`0F6b80028F6ah08P0F8`0K8P0F6`0N02805Q/R01/0
8P0R02806`0R01h07P0N028001hS000N01/E000N8aH06`0N02806a/R02806b/[
004IOb/K01hR01/F6aH08P0R028K02808P0N6`0R6aH08Q/07P005R<07P0R028K
000F6`0R6ah0028K01h07P008ah08R<N02<N028N02808Qh08Q/08P0R7P0F6`0F
6`0R6ah07P0F6b8K028K5Q/08R<08Q/05P0R6aH06`0N01HKO`007P0K02806aH0
8Q/08P0S7P0K5Q/06aHK028K01H07R8S01h08P0R01h07P0K5P0R000R6ah08Q/K
01HR6`0N01h06`0R6`0K5R<06`007Q/06aHN01/F8`0R6`0N02807P0F8`0N6aHK
5P0R6a/07P0S028K7P0K5R<N01h05B8K01h08Q/08ah07P0F6`0N6`0[:`016Gl[
01HK02807P0K8P0K7P0K5P0N01/R000F6`0R01/06b807R<08P0F6aHK5P0N8aH0
7P008P0N01h08P0S8QH06`0N000N01HK01h06a/05Q/08P0N6`0N8`0F6`0N8`00
7P0N8`0K5P0N02<08P0N01/07R<F6`0K8P0N01/R07lK8P0N02806aHS5Q/07P0N
01h05R<07QHS01hK01HK8P0K02805Q/F6b808P0R02806b<F6aH08`0N02806`0R
02808P0F6`0N01/F6aHS5Q/08P0K02806`0K7P0R01HK8P0N8`0R5P0S000K7P00
7P0K5R807P0R000N028001DN028001/F01h08P0R01/R000K:b/00AUo:a/0028K
000R01/F000R02<N02806ah08P0R6ah07P0K5P0N6`0R6`0K028N000S7P0R6`0R
02807Q/F6`0N01HS028N028K01h08P0N02<F6`0N000R01h08P0R01h08P0R01h0
028K01h08P0N02807P0K01HK01h05R<F6aIo01/06aHK5P0K5P0F8`0R6`008Q/0
7P0K01h0028K01/F028K02806`0N01/F6`0R7P0N000R6aHK028K5P0R6`0K5Q/F
01HS01h001HK02808Q/F028K5P0R000R6a/001h06aHK02<N028R01/F8`0R01/K
000R6aHS01h08QhE028K7P0R6aH08Q/K01h07P0R02/[004IOb/08QHK028F6b80
6b8R01/F6`0N01h08P0K7P008P0R01/R000F6`0N01hS000K7P007P0K8P0K7P00
6`0R01/R01/N000K028001/R02807P0R02<F6ah08P0K7P0K8P0K8b808QHK0280
8P0N6`0K8P0R01/R02808P0N000KOb808P0N01/S5P0N8`0N6`0R01h07P0R01/R
01/N02<F01/N01HK8P0N01/F8`0F6`0N02<F6`0K01h05R<07P0K5R<08P0K5R<F
01hS01h06ah08P0N000F6aHK7P0F01hS8P0N01/N01h06b806aHR01/F028F6aH0
6aHK01/S5@0N000F6`0K8`0N02808P0N01h[:`016Gl[6`0R01hS01h08P0K6aH0
8P0R8ah08Q/05P0N8`0R01h08ah08P0R02807P0R5P0R8`0R6`0R01/F8P0N0280
6aH001hR028K5R<07Q/08P0R6aHK000R6`0R01/F01hK01HK6`0R028K7P0N000R
01hK01h06aHK028K028N07lK5Q/F02807P0R01h0028N01h08P0R6aH00280028N
02806aHK01/F02806aHK01HS01h07P0N02807P0N6`0N01/F6`0R01h08ah07P00
8P0K5Q/08ah08P0S5P0R6aH06`0R01h002807P0K5P0K02808Q/08aHK5P0S5Qh0
7QD08ah08P0N01h0028K6`0R6`00:b/00AUo:aH06aHK01h06ah07P0R01/R01h0
01/07R806`0N01/K8P0K8P0K7P0K5Q/07R<06ah07P0K5R<F01/R01hK8P0K8R80
6`0N01/F8P007Q/R01/05R<F6`0N6b806b805P0N000N01/N000N02<F6ah001/R
01HS01hK01HK01]o5R<08P0K8P006b806b8N000K8P0K7P007R806ahR000K8P0S
01h08P0K8P0K01hS02808P0S5P0K8P0N000R02806`0F6ah001h08P0R8P0N02<F
01h08Q/07P0K8P006aH06b806aHK7P0S5P0F6b807Q/08P0N02<05Q/08P0E5Q/F
01/R01h08aHK5P0F6`0N8b/[004IOb/K5P0S01h08P008P0R6ah06`008Qh08P0K
7P0R6aH05Q/F01h05P0R02<F8P0N01H08Q/F01/F6`0F6`005Q/F01/08P0N0280
6`0N000N01h08Q/07P0N000N01HK02<N028K02806aHK02808P0K5Qh06aH07P00
8Q/05Q/0Oah07P0R01hK5P0K5P0K028R01/F000N8`0N6`007Q/F01h07P008Q/F
01h05R<07Q/08Q/F6`0R01/K028N028K5P0R6`0R6aH08Q/06a/F02808ah08Q/0
8P0R01/N028K5P0K5P0S01HS5Q/08P0N6`0F7Q/0028N6`0F8`0R5A/08ah06aH0
8P0N01/F01HS01h[:`016Gl[01/R01h08`0N8P0K7P007P0N7P0S8P0K8P007P0K
8P006b807Q/R01/F6`0K01hS01h06b806`0N8`0R5Q/06b807P0K8P0K5Q/R01/R
02805Q/07P0R01/R01hS01HK5P0K01HK8P0R5P0K7P0K5R<08aH07P0R8P0F6b80
8Wl08P006ah001/R01/R01HK01/F8`0N8`0F6`0R8P006b808P0K8Qh06b805Q/F
8P0F01h06aH07P0N02808b806`0F6aH06aH06a/08QH06b806ah05Q/07P0K7P0N
028001/R01/R01HK5Q/07Q/R000N8`008QHK01H07P0K7QD06aH07P0K8P0N6b80
01hS01h0:b/00AUo:a/07P0F8P0N01/N000R6`0R01/N01/F01/F8`0N01hK5P0K
02806aH06`0R02807Q/07P0N01HS01h06`0F8`0N028N01/F02<F6aH08Q/K0280
7P0S02806`0F6aHS01/F028K01/F8`0R01/F01/F6aH08R<K01hK01HK01io01hS
7P008Qh06aH08ah05R<F6`0R01h08P0R6a/F8`0N6`0R01/07P0K02806`0R6ah0
7P0S7P0R028K5P0K7P0R6`0S7P0S7P0F6`0K5P0N01H08Q/07P0R01/F028K7P0F
6aH07P0S01h07P007Q/001h08Q/05R<08P0R00078P0R6`0N01_n00XR8Q/07P00
8R/[004H7R/07P0N8`0N6b8002806aH06b8001h06b8001hR02;n07lN01hS8P0K
8P0R01hK8P008P0N01/R01h07R806aHR01HK02806b807P006ah001HK8P0R01hK
8P0F6aHS02808P0N01h06`0R01h06b<F01H06b<N028001hS02808P0N01hS7P0R
02807P006aHK01HK8P0K7P0K5P0K5R8007<F6b808P0R01HK8P0K7P008P007P00
7P0K5P0K7P006aH07P007P0R01/R02806b806a/08P0R02<F01/R000F6`0R01/R
01h08P0K8P007R806ah08P0K5Q/K5R<F6aHK02808P0N8R806`0R028F6b/[004I
1B/K02807_h0Ob8N6`0R6`0R01h08P0N01h08P0K7P0R8Q/08P007P0F6aHK000F
6ah06`0R01HK000R01/F01/08Q/08Q/F01/F028S5P0R6`007Q/07P007Q/00280
7Q/F6aH08P0N01hK01HR01h08Q/05P0K5Q/08P0N01hK02807P0N02<0OahK5P0R
8P0K028K01h05P0R01/F01/08Qh001hK028K02806aH001hS6aHS028R8ah06aH0
028R02<N028N028K01HK5Q/F02<N01HK028K7P0K5P0S7P0R028K5P0N000R6`0F
6aHS01h05P0N6`0F8`0F6`0N02<F6`0R6`0K300N6`0R6`0K01H[:`016Gl[01HK
01/R01hK01H07P0F6b807Q/R02806b807P006a/05Q/07P0K8P0S01hR8P008P0R
01hS028F6b806b807Q/05Q/06b808P0K5Q/06aHS5Q/08P0K8P0F8b807Q/08P0S
01/R01HK02806b805Q/08P0N01/07Q/R01h007lN02<08P006aHK02806a/R01HK
01h08b806b806b806ah08aHK01HK01hK8aH07R<R01H06aHK6`006aH08aHK7P0N
01/R000K01HK01h08`0N01hS01HK5P0K8P0N02806aHK01/S01hR6`0F6aHS0280
8P0N8`0R01h07P0K8QD05Q/06ah08P0R01H07P0R5P0K:b/00AUo:a/05R806b80
01hS7P0R6`0K000F6`0N028K6`0R7P0F8a/08Q/05Q/F6`006a/F8ah07P0K5Q/0
8`0N01HK000R7P008P0N6`0R01/08P0K5Q/07P0R01hK01h08P0F6`0R7P0K028K
01h08P0K5R<08Q/F02807P006aHR8aio01h07R<N01HK028K5P007P0R01h07P0K
7P0N01HK5P008Q/05R<K01h001h08P0N028K5P0R01HN01HS7P0K000R02806ah0
5R8K02808P0N02807P0S01h05P0N01h07P0S5Q/07P008P0R6`0K5QhS01hS01h0
8Q/08P0N01/E5P0F7P008Q/N028K5P0N8a/F02/[004IOb/08P0K01h08b807P00
6`0N01hR01/R01HK01H07P0K8P0F01h07R<07P0R01h001h001/R01/08P0N01hS
028F6`0S5Q/R000F6b805Q/R01/07P0K6b8001h06ah06`0N6`0R5P0N02806a/F
6`0N01h06b806b806aHS01/0Ob805Q/001hS01HK5P0K8P0N6b808P0R000N01hS
028F6ah08P0N01h06b806a/R01HK01/F6`0K8P0K5P0K5R<F6ah08aH05Q/S5P0N
01/F6b807P0K5R808P0N02<07R<07P007P0R8P0K7P0R01/001hK028001/F01/R
01h05A/S8P006ah001/N02<06`0F01/[:`016Gl[7P0R02807P0N028S7P0R0280
6aH08ah0028K5P0R01/N02807P0F6`0R6aH08R<08Qh06aH08Q/F02807P0S7P0N
01h06aHS6`0N02807P0F8`0R01HK5R<07P008P0R01HK02<N028K01H08aHK0280
7P0K5P0K5P0K5Q/08PhK6`0F8Q/07P0S01h08Q_n07lN6`0R6aHS01h06`0K01H0
8Q/05R<08P0N01H06a/05Q/08P0R01/F02<N000N000R7P0S7P007P0R01h08`0K
5P0R01/07Q/08ah08P0N028N000R6`0N000R6aH08Q/05Q/08Qh08ah06`0F8aH0
6aHR01/F6`008Qh05R<K5P<0:b/00AUo:`0R01/N01/F8b807P006b806aHS01h0
7R<N02<06b8F000K8P0K8aH07P0R01h07P0K8P0R01h08P0K5Q/07P0R02806aH0
7P0N01/N01/R01h07P0N02<08P0R01/F6aHK01/N000K01hR01h05P0K8P0R01/S
5P0N8P005Q]o5P0R01/07P0K5Q/R01h06b8R000N6`0R02808P0N5R<R01/07R80
5Q/R01HK8aH07R<F01/F6b808P0N000N028R01/07P0K8P0K8ah08P0N5R<06ah0
5Q/08Qh05Q/R01HK028F6`0N8`0N6`0R01h05Q/07Q/R01h07P0R01hE01h08P0K
5R<05Q/N02806aH06b/[004IOb/K01h002806`0K01HR6`0K5P0S5Q/08P0N028K
5P0K028R01/F01h08R<F8ah08Q/F01/F6aH07R<N01/08P0R01hK01HS6`0N0280
02807P008Q/F028F6a/F6aH08`0R02805R8K01h06b808Q/F01/N02807P0R01/F
8`0FOa/08Q/08P0R01h06aH08P0K6aHS01HK01hK01/F01/07R<F6`0K5Q/07P0S
5P0N000N02806`0N02<N028R6`0K6aH08P0R01h05P0K01h001h08P0S7P008Q/0
8Q/06`0R028K01HK01HR01/F6`0K5P0R000F8ah06`0R6`005B8K5R<N000N0280
6`0R7P008P0[:`016Gl[01/R01hK8P0F6b806`0N01/R01/F01/R000N01/F6b80
6ah06b8001/06`0K01h001/F8`0R01/002805Q/N01/K02807P0R01HK01hK8P0N
8ah06b806aH06`0S5Q/F6aHK8P0K02808P0K5Q/06b<F01/F8aH06b808P0N8gl0
5Q/F01h06b807P0S01/R01H05Q/R01HK02806b808P0N000K5R<08P007P0N028R
000K8P0R01HK5P0K6`0R5P0K8P0K7P0K8P0N02805Q/R01h07P0S5Q/F01/08P0N
8P0K01hS01h06b806`0R028N01HK7P008aH07P0R5QDK01h001/R01HK8P0N02<0
7P0N:b/00AUo:ah06aH001hK01HK5P0R02807P0F8ah06aHS01h08`0N01H08P0N
6aH07P0N02808Qh06aHK02807Q/08Q/002805R<K5P0R6`0S5Qh001h001h08P0K
7P0K5P0R02806`0K01h07P0N01/F02<F01HK02806aHK5P0K7P0R01io02<08Qh0
8P0K01HR01h07Q/08Q/06aHK028K5P0K5Q/0028N01/F6`0R7P0R8a/07R<N01h0
7Q/05R<N01HK01/F01h05P0R01/R01hK02<F8ah08P0R02806aH08Q/06`0R6`00
7P0R6`0N01HS6aH08ah0028K5P0R8`0R6`0E5R<07P0F6`0R01/F01h08Q/08R/[
004IOb/07P0K8b805P0N02<F6aHK5Q/R01/07P0R02808P0N02<R01/R000K8P0R
02806ah08aH001/N8aH06b8001hK8P0N02806`0F7P006b808b808P0K5P0S5P0F
6ah07P0R01HK8P0R02<F8`0N01/N02807P0K01/R000K7P0KOb807R8001/N0280
6aHK8P0F01h06aHS01HK5P0K5R<08QHK01h06aH07P0K8aH08P0N02807P008P0K
01HK02808P0K8P0K8P0N01/R028N01/001h06ah08aH06ah08P0K7P0N8P0F6`0N
01hR01HS01h08aHK02806aHK7P0K5B808P0N8`0N6b808P0K5Q/07R<[:`016Gl[
6`0R01hK028S01h08`0N01/07P0F8`0N6ah08P0N02807Q/07P0R01hK028K01H0
8P0R7P0F01h08P0N02807P0R01h08Q/F8`0R6`0N6aHK01h05R<N02<N000S7P0R
6aH0028K6aHK01h00280028K7P0R02807Q/05P0R07lN6`0K5R<05R8K01HK01/0
8R<07P007P0S01h05P0N02<N02808P0S7P0R01HK6`0N028K02<N028K028K028K
5Q/F02<N01/F02806`0K02808Qh08P0K7P0S7P008Q/F000R01/F8`0N02<06ah0
7P008Q/05R<07P0K000R01DK7P0R01h001HK6`0R02808P00:b/00APG:`0F6b80
01/N02808P0N02808P0F6aHRoP1o7P0K6b807Q/08P0N6b805P0N01hS01/N01/0
6b806a/R01HK7P0K6b806b8001h07P0R000K01hS8P0N000N02807P007P0K8aHK
5P0N02808QHK8QHK5P006aHK8P008R806b8001h06aHR01/07R<06aHS7P0R01hR
01h08P1j7R<F6`0N01HK01h07P0R01h06`0R000K5P0N01HK5P0N01HK01h08`0N
01/F8`0K8P0R01HK5Q/07P0S5P007P0F6ah001hR01/F8`0R02808P007P0S5Q/F
01/F8P0N01hR01hF01/K8P0N8b8001hR01/R01/N:b/00AUo:a/05Q/F6`007Q/0
7P0K01h06`0R01/K5R<N01h001hK000R6`007P0R6`0K5P0N01H08P0R01/F01HK
02<002805Q/07P0R6`0R6aHK5Q/07P007Q/08Qh08Q/F028S01h07P008ah07P0R
6`008Q/08ah05Q/07Q/F6`0R01io6`008P0K02807P0N01H07P0N8`0K028K5Q/0
01h08Q/08Q/08P0N028S01h05Qh08R808ah08Q/08ah08Q/07P0N02808P0K5Qh0
6aHK5Q/06`0R01hK028N028K000N8`0N6`0N01hK01hK028K028K01HS7P0K0280
8P0K6`0E6aH001h001h08Q/06ah06aH002/[004IOb/08P0F8`0F7R<05Q/R0280
8P0N6b805P0N02<07R8002807P0R8P0N6`0R01/R01hS7P0N6b806b806b807R80
7P0K8P006aH07P0K02808P007P0F01/07R<001/N01h08P0R01h08P0K6`0F6ah0
01h07R<08P008P006aHKO`0R5Q/R01hK8P0R01HK8P006`0R5P0N02<05Q/R01/0
5Q/05Q/R01/K02807R<07Q/06ah001/08P0F01/08P0R01HK7P0R01/08P0N02<0
8P0F6ah001hR000K01hR01/R000R01HK028001HK5P0N02807P0R01h07Q/07P0R
5@007R808b806ah08P007P0K5Q/[:`016Gl[6`0R6`0R6`0F6`0F6aHK6`0R000N
6`0R01h08P0K7P0N6`0N01/08P0R6aH06`007P0S01HK5P0K5P0N6`0K5P0R01hK
5P0S7P0N028K5Q/F8`0R6aH08P0N02806aH08Q/K02808P0N01HS6`008R807P0N
01hS7P0R7P0S07lR6`0F6`007P0R6`0S01hS7P0N8`0N02808Q/002808Q/08Q/0
6aH07Q/07P0F6`0F8aH08R8K01hK02808P0N8a/001hK02808Q/0028N01h08`00
7R806ah08Q/07P0F6aHS028K01hK5R<K01h06aHK000R6`0K5P008P0R6aDF8`0N
6aHK01H08Q/F8`0N02<F:b/00AUo:`0F6aH07P0K8P0F6`0S5P0R01/R01H06b80
5Q/07P0R01H06b805Q/07P0K8P0R8P006aHK01/R01/R000R02806aHK02807P0S
01HK028001h06b807P0S01hK8P0R01/F01/N02<N02807P0F7P0K8P0R02807P0R
01/07P1o7P0F6`0N8P0N6`0N02807P0R01H06b806ah08QHK6b805Q/07P0K8P00
7P0N8`0R01HK01/N028001HK8P0N6`007R8001HK5Q/F01hK02808P0N8P0K8P00
5Q/08P0K8P0K01hK028F000N01h08aHS01HK7P0R01/S8P0K7P0E8P0R000K01hS
5Q/06aHR000N02/[004IOb/K000R7P0R01/F8`0N02<N01h08Q/08P0K5R<08P0R
6`0R6`0K5R<08P0N01/F6`0R8P0N01h06aH07R<N028K7P0K028K5P0R6aHK01h0
8R<04P0K028K5Q/001hK028K01h001h06`0R6`0R6`0R01/K028K000R6ah08P0R
Ob<K01HS01/001HS01807P0R01hK5P0K5P0K5Q/05P0K5R808P0R01h08P0S01h0
6ah002806aHK5R<K01hK01HN028K028K01/06b805R<07Q/F02<N01/F8a/07P0R
01/F02807P0K028R01h08P0N028K000R6aH07Q/F01/F5A/N028N01h002<F6`0K
028R028[:`016Gl[8P0N6`0N02808P0R01HK5P0K8P0K8P0K5R<08P0N6`0R6`0R
5P0K5Q/R01/F8`0F6a/R02807P0K8P0F01/N000R01HK02806aHS02806ah07P0K
5P0N01/07R8001HK01h08aHK8P0N6`0F6`0F6a/F01HK01/R01h001h06glF01/R
02805Q/R01h06`0N01/K01/R01/R02<F01/R5P0K01/N02807Q/R01h08aH06b80
7P0S5P0N028001hS01HK01HK5P0F8aH05Q/07Q/08`0N02<F01h08QH06b806aHK
8P0R01HK6b8001/R01/N02807P0K8P006b8001DN000K01h06b807P0N01/N01/K
:b/00AUo:a/F000R8ah07R<F6`0R01/F01hK01hK01h07Q/001HS01h06`0R01/0
7P0F01h08aH06`0N028K01/08Q/001hS028K01h08P0K5Q/08P0K5P0R02<N0280
8P0N8ah05Q/08QHK01/F000R6`0R6`0F6`0R01h06aH08R<07P1o6aH06ah08Q/0
7P0N01h08Q/05P0F6aH07P0N02806`0R02806ah08`0F6aH08QHK01HK5P0R02<N
01h08Q/07P0S028K028N01h08Q/07P0F7P0R01h08ah08Q/08P0N01HS01hK028K
000N8ah06aH0028K7P0N01h08P0N028E028N02808P0N000R02806aH002/[004I
Ob/06b<R01/05Q/06`0F6b806b80028002808P007Q/R01h08P0F6b808P006b80
6ah05Q/R01HK02805Q/07R805Q/N028001/N01/08P0N01/R01/N02807P0S01H0
7R<08P0K01/F01/R8P0F6`0N8`0F6b808aH06ah08P0NO`0K8QH06ah08P0R01/R
01/08P0N8`0K8P0R000K8P0R01hK8P007P0N8`007P0N02<R01/R01/N02805Q/0
7P0R7P0N01h08b8001/08P0K8P006aHK01h001/F01/N01/R028001HK01hR01H0
6aH08aHK5P0S02805Q/R01hS5AhS01HK01/F6`0N6b807QHS01h[:`016Gl[6`0N
02807P008P0N01/08P0N6aHK5R8K6`0R6`0F6aH07Q/08P0N6`0R8P0K5P008Q/0
7P0S5Q/08Q/07P0K01h0028K7P0F02808Q/N00@P0R0P8@8Q7P0N6`0R6`0N01h0
5R808ah06`0R6`0N01h08P0N6aH08P0K7P0R07lR01/08P0K5Q/08Q/002808Q/F
01h0028K6aHR01/N02806`0R6`0N01h07P0S01h06aH06aH0028K02808P0S01/F
02807P0K7R808Q/F01/R7P0S5Qh08Q/06aH0028K01hK5R<K01h06aHK5P0R6aHK
01/F6aHK02<07P006aD07P0S5Q/08aHS01HK000R01h0:b/00A=?:`0N5P0K7P0S
5Q/F6b805Q/R000S5P0K01H07P0K8P007R<05Q/R000N01/R01/S5Q/07P0R01/F
01h08P0R02805Q/N000N8b8N01h001_k800Qo21o5Q/08P0F6b806`0N01h06ah0
8P0R000K8P0F01/K8P006a/R01h08P0K5R<05Q/07R<N01h06b8001hK01H06ah0
01h08aH07P0R01h08b806b807P0K8P0R01hK01hS7P0K5R808P0K8P0K5P0K01h0
6b806`0R028001/08QHi01/R7P0F6`007P0R01h08`0R5Q/06`0R01h06`0R01h0
6b807P0S5Q/07QH06aHK028F6b806b/[004@Eb/K01/F000N02806`0N02807Q/F
02<N028K5P0R01/N000N02807Q/F01h07P0F6`0F8`0R6aH08ah08Q/07Q/08Q/0
028K000K028K5Q/0120P124P827f87lK5P0R02808P008Q/N028N01/08Q/05Q/F
6`0F6a/08Q/F01/F6`0F6`0N02807P0K8Qh0028K01HR6`0S7P0N6`0R6`008QHK
5P0N028K01hK02805Q/001h07P0K7P0R01/F02<N02807P0K5P0R6ah08R8K5Q/0
8P0K028K038R7P0N02808P0N02<F7P0R6`008P0R6ah07P0N000R5Q/08P0K5P0S
5Q/08`0N01H[:`014eD[5R<F01/R8P0K7P0R01HK5Q/08P0N01HK01/R01hF02<R
01hS7P006b808P0N8P0R01h06aHS01h05Q/07P005Q/06b8N01/R5P0N01/07P0P
12448O`P0R4P8OdP027l87lQ81/F6ah001h08b805Q/07R<08P0F6`0F01h06b80
5P0K8P0R000K8P0R01/05R<N02808aH07P0R000F6`0N8P0K01/R01HK02800280
7R<06b807P0R000K7P0K8P0N000K8P0R02807P0K02<N000F01/R01HK01HK0280
02dK8P0N6b806`0F6`0R8P0K7P008P0R01hR01/05Q/R02<F5Q/07P0R01HK:b/0
0@m@:`0N02806`0R01/F6`0S01h08Q/F028K01h06`0R6`0N02806`0R7P0K6`0R
01/F6`0F8P0K5R<08Q/08P0N8Q/08P0K02806`0R02808P0No2028B0Qo2008OdP
027n8?dQ023l8GlK01h08P0R01h06aHR028K5P0R01h08ah06aHK5R806ah08P0R
6`007Q/07P0R6`0N8aHK01h06ah07P0K02808Q/N01hK01h05Q/08ahK5Q/00280
6b808Qh06a/08Q/F01h07P008Qh08P0N02<F6`0F8a/F8P0K6`007P0F8b@07P0N
6aH001hK028K01/08Q/07P0K5P0S01HS01hK02<0:b/00@eC:b8001/N02806aHS
01HK8P0F6`0R01/08P0R01HK02806a/R01HK02805P0K8P0S01hS01/F01/F6`0F
6b806`0F6b808P0R01/N01/F6b808204oR058@8Q820QoB008O`PoR408?`Q0aLQ
8AOj8GPR01/07P0S01/R01HK5P0F8aH06`0N000K5Q/08QHK02806b805Q/08P0R
02<F02808P0N6b807P0R000N01hS5P0N000K028N01/F01hK02805P0N01/S0280
7R806b<R01/R01h06b805R<06aH06b807R<07P0RoP0N7R<R01HK02806ah08P0K
5P0K5R807P0N02807R/[004>Db/K5R<07P0N6`0F6aHS01h08aHK01h08Q/07Q/0
5R<F6`0F6`0R028K5R<N01/F6aH07P0F8ah06`0R6`0N028K01HK5Q/N01h0028K
01HK01hP8OdP027m80<Q820QoR048B0P8B3n8@0PnB465b4Q5b4Q5o/QOah06`0R
02<N01h08Q/F028R02<F6`0R028K01HK02807Q/07P0N028K01h001/08P0R6a/F
02<07P0R028N028K02808ah0028K5Q/F02807P0K01/08P0N01h06aH08P0K7P0N
6`0N01HK000N028K6aHR6`0N028K01h08P0K4aHK01HS7P0K6aH08P0N6aH0:b/0
0AIG:`007R807P007R806`0N01/F6`0N8`0N028001hS01h05Q/08P0K7P006`0N
01h06b808P0N01/F01h08P0R01H06aH06`0R01HS7P0F6`0R024P824P8OhP027n
800QoR048B0P8B3n8@0PoB4:5b4G8ALQ8ALQ8AOm8@0GoR5o824P800N028001/F
8`0N01/F6`0F6`0N01hS01HK5Q/08P0R000K01hS8R808P0K6`0F02<F8P006a/R
000N000N01h08aHK01h08`0K8P0N02805Q/N02806aHS01h07P0R000R01HK028R
01HK5P007P0S5P0K02805Q/R01/B01hS5P0K5P0R01/F6`006b/[004CDb/K000N
02<N000K5P0R02806`0N01h06aHK5R<07P008Q/08Q/F01/F8aH002807P0K028K
01h08ah08Q/K028K5P0S7P0N028K000R6`0R6aH28OdP027n800QoB088B4P8B0P
8B4PoR408?hQ01On8@/G8B4G8B4G8ALQ8AOk8Gl28@8Q0R006aHS02807P0R6`0R
01h07P0S01h08Q/K028R01h001/K5Q/F01HS6aHK01h08P0N028S028N02807P0R
01h08P0N01/F01hK5Q/001hK01H08R<08P0R6ah07P0S5Q/07R<K028N000R5Q/F
028K02807P0F8`0N00hR02<N028K01HR6`0[:`01358[028R028N028R028F6aHK
8P0R02808P0K5R<07P0R5Q/F01h06b<F01HK01hK8P0R01hK028001h001/F01/N
02<07P0N01hK01HK7P0R01h08OhP027n800Qo2018B3j8@8G8B7l5ohQ1QLQ8ALQ
8AOm8@0GoB4:824P8B010P4Q0R7m87lR01HK01HK8P0R01h08P0R01h08P0K7P00
6aH07P0K8P0N01/08P0N6b806a/F8P0K01/R01HK8P006b807P0K8b808QH06b80
5Q/R01/F6ah06`0R02<F6`0R01H05Q/08aHK6`0S5Q/07Q/R01HK5R8001h07P0K
7P0R01/06`8[:`013U@[6`0K6`008Q/08R<06`0F6aHK01h08Q/06aHN028K000R
7P0N01h08Q/08P007Q/08P0N028R7P0R7P0K5P008Qh002808`0F8a/0028K5P0R
820QoB048@8Q827n80@Q8B0Q8?dQ0ALQnQOm60/Q8ALQ8ALQ5b4Q5b3m8@8P827n
80<Q820QoB1o0R0401/08Q/08ah07R<0028K7P008R8K01h07P0K5P0R028K000F
6`0F01/K5P0R6`0R6`0N6aH07P0R6`0N01/08ah06`0R01hK01/002807Q/07P0F
8`0N6`0R028K01H08P0R01h001/N000S5Q/N02807P008Q/F0280:b/00044Db/0
8P0F8aHK01/N02<F01h06`0R01hK02808P006`007Q/08`0R01HK02806b805P0K
8P0F6a/001/08P0K5R<N02806aHK8P0N01HK7P0R01/P8OhP0b4P827m80<Q824P
oR45824Q5b4QnQOg6?hGnb42824QoR008O`POb4P82448@8P120P100N02<F01hK
5P0K5Q/N02808P0K5R<07P0N01/R8P0F6b<F01/F6`0N6`0R000K8P0F6`0N02<F
01h08P0F6b8001hF01hK8P008P0K8P0N6`0R01/N01HK8P0K6b807R<F02<R01/F
02<07P0S5Q/06b800ah[:`010ED[7P0R6`0N01h001h07R808P0R6`007P0N01/R
7P0R8P0F8aHK028K028K7P0K028N01/08P0N028K01h05R<06aHK8P0K01h0028K
000N6aH07R0P8OdP007k8?hQ220Q8B0Q8ALQ8O/GmQSl5o`Q0@8QoR028B0QoR1o
8B020@810R44820N01h06ah001h05R<0028K01/F01/F8`0R01h06`0R6`0N0280
8aHK01HK028N01/08P0R01h08ah08Q/08P0K7P008Qh001hK5Q/F01/F01HS7P0F
028K01h05P0K000N02807Q/07P0N02<N01/F01HK0002:b/00?EB:`0N6`0N0280
01h05Q/08`0N6`0R8P0R02806`0F6`0N8`0N01HK01HK5P0R028K02805Q/R028R
02807R<07P0K01/R01h06b8N01hR000K8P3l800QoR078B0Q820Q8B3i8OTGmaSk
5`4Q5o/Qn21o0@8P0P42804201h06aH06b807R<05Q/N02806b807P0N6b808aH0
7P0N02806aHK01HK01hK02807R<07P0N01h001/07P0S5P0S5Q/06b805Q/06b80
6b807P0K8P0K01h07Q/R028F6b807Q/08P0R028F01/F01hS01/R:b/0003nDR/K
000R028K7R808Q/07P0R01h06a/F6`0R7P0R6`0R01h06aHS028K01/F6aHK028K
02806a/06aHK000N5P0R02806aH08P0K02806ah001h0oR008OXPoR4281LGo244
5b4G8B7k5oHHo1L48B4G8AOk8@HP8B0Q820QoR1l0P42820K02806aH06`007P0R
01/F6aH06aH08P0K01h08Q/08P0R6aH07P0R6`0N01HS028K028K000R01hR028K
7P0K5P0R02807P0S5P0F6aH06`008P0K5P0R02<05Q/K01/06aH0028K6`0R6`0R
02<N01h05Q/[:`0105D[02807P0K5P006a/08P0F6ah08aH07P0S7P006aH06ah0
8aH06aHK01h07P0S01HK01h06aH07P0K01hR02<F6b807P0K8P0S5P0K8P007R80
6b0P8OdP124P824PnR435b4G8OdHoBT25ADVmaSj5odQ120Q824Qo21o8@8Q0R0P
0R0P0QH06b806b808QHK8P0N01/06b806b807P0R01HK5P0K5Q/06b8001/F0280
6`0N6aH07P0R5Q/R01/07P006aH06ah06b808ah05Q/06b808QHK8QH06b807P0K
5P0R01h06aHK7P0F01h06ah07P0N01hS02/1:`01358[6`0R6aH08ah05P0R6`0R
000R5Q/0028F8ah05R<N000R5Q/05P0F8`0R01h08Q/08P0N02<N01h08P0K5Q/0
5Q/05R806ah07P0F6aHS01h05_dP027n80@Q820Q8?hQ01On8@8G8B7m6?`Y20HE
5BTV5BTV5OdYo1L28B4Go244824P827n808Q807n87X282025P0N01HK5Q/0028K
02807P0K5P0K5P0R6ah0028K01/F01HK5R808ah08P0R000R6`0R6`0N02808Q/F
6`0R6`008P0N6`008Q/08P0K5Q/0028K7P0K5P0R01/F6aH08P0S5P0R6aH08P00
8P0R02<07P0[:`012UD[01/N02<07P0R01/N02806ah06`0S5Q/N000R01h001h0
8`0F6b806aHK8P0F6`0N01hS01h08P0F6b806`0R01/R01/07P0R02806b808P0K
8b0P8OhP027l80/Q8B0Q8ALQ5b4Q5b7n6?`Y2ADV:BHE1RTV:BKn:OhHoAOl8@@P
8B4P8OhP027n804Q0_`PMP0N01/R02<08aHK01HK8P0N01/R01/R01/08P0K5Q/F
01hS5P0K01h08P0K7P0N6`0F6`0R02806ah08`0F6`0R5Q/N01H07R805Q/R0280
01hK000R02<F6b808aH06b807P0K8P006b<N8P0K6b807P0K:b/00@IB:ah0028N
01/F6`0R01hK01H07P0R5Q/001hK02808Qh07Q/08P0S7P0K01/F6`0R02807P00
8Q/08P0K5P0R6aH07P0R6`0R6a/08P0K7P0N003i80@Q820Q8?`Q1ALQ8ALQ5odH
o2T<9RTV:ADV:BHY9QPY:OhH11LG8B4Go24382428OXPO24480@P81h08P0K5Qh0
7P0F8a/07P008P0N01HK5P0R6aH08P0S7P0K02807P0R6aH002807P0S5Q/K01h0
5P0R5R<N028K000S7P0R8`0N000N6`0R8P0N028K5Q/05Q/F02<N01/F02806ah0
5P0K02805Q/F0280:b/00A9N:`0F6ah08aHS01HK8P007R806b806`0N8aH06a/R
01/R000N8`0N01/F01HS01h06a/R028N6`0F6b808P0N01/R01HK5P0K5P0F6b80
02806aHK8B0P8B0Q820Q8B3n8@0PoB405odQoQSm:@dV:BHY5BHY9RTV:BHY:OhH
oQMo8B4G8B0Q82418B0P8B0Q824P82448B04000K8P0K02808P0N01h08aHK8P0N
8`0K5Q/08P0K01h08P0R01hS01h08`0N6b808P0K5P0R01hS5Q/07P007P0R01h0
5Q/07P0S5Q/08P0K6b807P0K01HK02807P0S5P0N02<F01H>6b805Q/F6`0K8P0N
:b/00@MH:a/05P0R01h08Q/06aHS01/F01hK01h07P0R000N6`0K7P0F6aH08P0R
7P0N000R01HK01/05R<K01h06ah07P0K02<08Qh06aHK01hK01h05R<0820Q0R7n
8?/Q0ALQoAOm60<V5BTVoBT99RTV:BTV:BHY9_dYoQSn5`8Q8AOn8@4P8OTPOB4P
1244820N01HK5P0R6a/08P0N02808P0K000N01HS028K7P0R01h08Q/K01h06aHK
5P007Q/08P0N01/001h05R<08Qh08Q/N02<K01h08P0N028K7P006aH08P0R6`0R
6ah07P0S7P0N02<N000N02<F6`0N01/002/[0045Db/07R<06ah05Q/07P0R0280
6b8001h08`0R01/B000S5P0F6`0R01/N02<R02<F6aHK02808P0N01h08aH08P0S
5P0N8P0K01HS028002807R<07_hP1b4P824P8B4PoB7l5ohH11LV:BHYmbH5:ADY
62TYo1Sm5odQ0R0Q8OhP0b4P827n800QoR1g0P0N8`0R01/F01HK8P0R01hK7P0R
8P0R01/F6aH001/R01/K5P0R02807P0S8Qh001HK8P0F6b8R02<N02806`0K5P00
7P0N02807R<07P006b808P0K5Q/F01h0028001h08P0K7P0S8P0S5Q/08P0N01hS
:b/00@YD:a/05R806`0R02807R<F6aH08Q/F028F6a/F01/F7P0S7P0F8`0N000N
01/F6`0F8`0N6aHK000R02807Q/08P0R6`0N01HS01h07Q/F6`007P007_hP027n
804Q8?hQ0QLQ8OdGoAP45BTV9RWi9P<Y9RTVoBT161Sn5ohQ01Ol8@DP8B0Q0R7m
808Q80Cn80d201HK028K028K01/F6oh0IAh07Q/08Q/07P0S7P0F6`0F02<N028K
000R01/08R8K01/08P0K6aH06ah08P0R02<N000R01hK01/F6`0R7P0N02<N02<0
6aH08Q/07R808Q/05P0R01hK5P0F6`0R02807R/[0045Eb/08P0K8P0R7P0K5Q/0
6`0K5Q/06b806`0F01/F8`0R02806aHR01/R01h06`0N8`0N5P0S02807P0K7P0F
01/N01/R02806aHK8P0F8`0R8P0N8P0Q827m8?hQ1B0G8B4G8OdGoQP35bHY:OLV
0bTV:BKm:@4H6?hGnR4:824Q8201824P827m87P20@@07P0K7P0F6`0N01/06b8R
01h001/N028001h05Q/07R<07P0N6`0R5Q/N02806`0N02806aH06b8002806ah0
7P0K8Q/F6`0R5P005Q/08P007P0K5R806b806b806`0K01HK8P0N6`007R<F01h0
6b80:b/00@=K:a/08P0K5R<08P0K01h07P0S5P0F6ah08Q/05P0N6`0N02806a/F
01h002808P0N000S7P0N6aH08P0K8P0R01/F01/F6aH08`0N01/F6`0N028K008Q
820Q823k8@4G8O`GoAP59RHY5BTKnBH25BTVo2T161Sm5`HQ8B0Q824QnR028@8Q
oB1g0R4N0280028K02808P0R01hS028K5P008Q/F8`0R6`0R01h002807P0K01H0
7Q/F02807Q/F028K01h08Q/N000R02807P0S01h08ah08P0R6ah002806`0R01hK
01h05R8K02806ah0028K01h08R<N01/0:b/00@aI:`0F6ah06aH06b808P0R01h0
6b8R000F6`0F6b805P0N01/S5P0K8P0S5Q/F6b805Q/N02805P0K6b806`0K7P0K
8P0K01/R01h08P0K02805Q/08P0P824PoR44824P8AOn8OdGoAP55bTV1RTWnbH5
9a/Y9RTVoBWn6?hGoR408?hQ120Q820QoR1m0@8180@P80@001/R5Q/05Q/F6b80
7P0N01/F8`0N6aH06aHK01h08P0R8P0N02806b806`0R01/R000K5Q/07P0S7P00
6b806a/R02808P0R01/R01/N000R7Q/R01HK5Q/08P0R01/05Q/07P0R01h07P0K
7P007P0N:b/00A=E:a/05P0R01/N01h07Q/08P0R01/K5R<05R<K01h08ah08P0N
01h07P0N000R01/F8`008Q/F8ah05Q/F028002807P0F8`0N01h08Qh08Q/K01HK
01kn808Q8B3m8@DP8ALQ5b7m60PV5BTV5BTV9RWn9PXY9RTY9QPG61LH6?dG027n
5`LQ8ALQ8B0Q8OdP0b4P827n87TQ120P7P0F6`0F8`0F6`0N6`0K02806aHS01HS
7P0K02807Q/06a/06aHK5P0S7P0R6ah06aHR02<F01HS5P0N01HK7P007Q/07Q/0
7P0F6aH06aHS000N028K01/07P0N6aH08Q/08P0R6aH08P0S000N000R02/[004<
F2/06b806b805P0S8P0F01/N02805P0K8P0N01HS8R8001/N02808P0R02<N01/F
8`0F6ah001h001/F01/R01hK8P0F6aHR02807Q/001h001hS028F8B0PoR49824Q
824G8ALQ8OdH2BTY9RHY9RTV:AGn9P<Y9PHEnQSm5`dQ8B0Q820Q0R0P8B0P8OXP
NR42804281hS02806aHK028002808P0N01/F8P0K02805Q/F6`0S5P0N5R<08P0N
000N000S5P0K01h07R<07P0K8R8001hR000F6`0R01hS01/R01/F6b806ah08P0F
6b8002<R7P0R01/N02<06b807R807R806b/[004:FB/N01/F01hK02807Q/08P0K
7P0R6`0F6`008Q/06`0R7P007Q/06ah08Q/08P0N028K000R8`0R8P0S7P0K5P00
8Q/0028K6`0R01/F8`0R7P0N01/00R4P8?hQ023k8@4G8OdH42HV:BTV:BHY9RTV
9QDY5ADXnASm5odQ1B0P8@8Q8OhP1B4P8B0P8OhPN08P80805Q/08P0R028K7P0N
01hK02806`0R01h08Q/08aHN01/F000N02<N028R8aHK5Q/F02808P0N01/F01/0
8Q/07Q/F8`0N02807P0F6`0F01h08P008Q/F000R7P0N000R6aH0028F6`0N01/0
8R<07P0[:`013UH[02806b805P0K5Q/05Q/R000N6`0F6`0B01h07P0K7P0S5Q/0
8aH06ah05Q/R01HK01hR01/F6a/07P0R02807Q/05Q/N000N02<F01/F6`0N0280
7R7n80`Q824Q824Q81LQ8ALQoQP@:2TE9RTV:ADV:BHV:AD65BKh6?`G0B4QoR09
0@8P824P82428OhPNR420@8P81h06b807Q/R01/08P0F6`0R01h08P0N8P0K5P0N
000N02<R7P0R01/R01h06`0R000K7P0S7P0R01/R01/N02805R<08P0N01h07R<0
5Q/R000K5Q/N01/S7P0S8P0S5Q/06aHK7P0K8P0R01hK02806b/[004=FR/K01h0
6`0R8P0K028K01h08P008Q/08P0N02807P008P0R01h08Q/0028K01h08aHK01hK
01h05R<07P0R6aH05R<K000R7P0R028K01h08P0R6aH00R4P8B3m8@0PoB425b4Q
oQPA5bHY9RTV:BTV9RTV9RTE:BHGn1Sm5`dQ5b4Q824Q12448@8P8OhPNb4P8081
0R007P0K5P007Q/07P0R6`0R6aH07Q/001/F02<N028K01h06`0R6aH06aH07P0R
01hR01/F01/F6aH06aH0028K01/F6`0R6`008P0N02<08Q/N01/001h07P0N01/F
6`0F6`0S000R01hK02807Q/08P0[:`0145H[02806aHS7P0S5P0N02807P0S5Q/R
01h06b807P0K8P0N6b805Q/08QHK01h08P0N0280028001h07P0K7P0S8P0N01HN
02<R01/N02808P0K6`007P3m80/Q8B0Q8B0Q8ALQ8AOm610Y9RTV:BHV:BTE:2TV
:BHQ8OhH02Sj8@<P8242oB0110;m87l4820Q820Q0P42820R01h06b<F6`0R01/R
000N01/R000N8P0K8Qh001h08P0K5P0K5P0N01/S01HK8P0K8P007P0K01/R028F
6ah08P0K01hR000N6b805Q/F6aH08P0N8P0N02807P0K01/R02807QHK7P005Q/R
000N01h[:`000A9K:a/08P0K5P0N02<N028K028K5Q/06aH08P0K6`0R01hK000N
6`0F8`0N02807R<F028K7P0N02808P0R01/F6`0R028K01h06ah0028K01h05P0R
8`0N8B0P8B3m8@8G8B7l5odY41DY9RTY5@HV:APH9QDY8ALQoQPE8B0Q824P8B4P
824P808P0B0480@48OhPO24P82420@8P81h07P0N000R6aH06ah002806b8K01h0
01/08R8S5Q/05R<N01/F02807P0S01h0028R6`0R01h06aHS000K5Q/F02806ahR
000S7P008P0S7P0R01/08Q/08P0N01h06aHK5P0S01HS7P007P0R6`00:b/00@eM
:`0R01/F6`0K5Q/F01/05Q/06`0N02806b<F01HK8P006b805P0K5Q/R01h06`0K
8P0K02808ah06ah07P0K02807Q/08P0N000F6ah05Q/R01/N01h00R4P824Q8?dQ
naOm:A@X9RTV:BPE:BHH:2TY9R4Q5b4H623l8@0PoR7n80X1820282048B048OhP
NR420@8P80808P0R02807P0K8P008QhK8P0K02808R808P0K01/08P0N000F8`0K
7P0R01h08ah06`0F6b808P0K01/R01h06b806aH06`0K7P0S8P0K7P006b805Q/F
01/R02808P0K01/R02807P0K8P0R6`0F6b/[0046Eb/N02806`0R01/08Qh08Q/0
7P0R028K7P0N028K01h08P0N6`0R01/06aH08P0N01/F028K5Q/08P0K5P0R028K
6`0F8`0N028K01H08Q/07P0R01h07R7n8?hQ0B0GoR7l5o`Y01Gm:@8E9QOn604V
:OhQ1ALQ624Q8?dQ0b0Q827h8084127l800QoR1e124N028K01hS01h07P0R6`00
7P0F8Q/K01hK02807P0N02807R<07QH001hK5P0R5Q/F028K01h07P0F7P0N0280
5Q/F02<N0280028F6a/0028N01/F8`0K5P0K7P0R6aH08P0K7P0R02806ah0028K
02/[004:G2/07Q/R01HK8P0N8`007P0R02806ah0028001/07P0N6b8001HK8P0F
5R<06b807P0R01h06`0F6b806b806ah001HK01hS01h06b806`0R01HK01/R020P
8B0Q8?hQ0aLQ5b7l5ohY0QPX9_`YoAP45bHG8AOj8@4P8OhP027m80`2804210@P
80@48B0QoB1f1244000K01h001h08P0R6`0K5Q/R01/0028001HK7P0R01/K8P0N
8P007R<R000N01/08aHK01h08P0N8`0R02807R<08P0N000N01hK01h08P0N01/F
8`0R02807P007P0K8P0K5Q/08ah06aH08aHK01h[:`013eH[6`007P0R01h001hK
5P0R6aHK5P008QhK5R808P0R01/08R8K01h08a/F6`0N000N6aH08P0R6`0N01HK
01H08Qh0028K02808P0K8P0R6`0R02806aKm80LQ8B0Q824G8O/GoRT35APV6?hY
0BHGoAP15B7m5`4Q0_dQ0R028OhP027n80d20P4P0P@P80@P80@P8OhPM@@P7Q/0
8Q/F8P0R6`0N01h08`0N6aHN01hS7P007P0N01H06`007R<N01h08R<K5P0N02<F
6`0R6`007P0S7R<06aHK02808R8S7P0F6`0N01h08P0K5Q/08Q/N028S01h07P0F
01HR01/F028K5Q/08P0[:`013EP[01/R01HK8P0N8P0F8`0K5R<001hS7P006b<0
7P0K8P0F6a/08P007P0K02807R<F02<06aHK01h07R<08P0N02<R01h07P0K5R<F
01hK02807Q/R02808OhPoB405odQoAL0:OdH01Om:@0Go1P08O`G0B01oR455`81
80@QoR1o8B0Q820282441042120P120P124P0P0F6aHS01/K02808P0K5R8001/0
8P0N01/R02808b806b8R01h08`0R01h06b806aHK01HK01hR01/N000N01/08P0N
6`0N000K02806b806b806`0F6aH001/07P0R01HK8P0K7P0K5Q/801/08P0K:b/0
0@]N:a/07P0S01/06`0R6`0R01/F7P007P0R6`0N01/F01/F6`0F8aHK8P0N0280
8R<07P0N01HS01h08`0N01hS01h06b807P0R6`0N02807P0K6`007Q/08R0P8B0P
8B3n8@<G8ALXmaSn:@4G5odH0B4QoAOn8?hQ1B0282048OhP027n80410_hPOP@P
1204808P80@Q81HS000N01H07Q/07P0F6`0R7P0R6`008P0N6`0R5Q/F01/08P0N
01hS01h05Q/F01/08Q/07P0K5P008Qh07P0R6a/05Q/08Qh08Q/F01/F01h05R8K
02<N5P0R6`0N02<07P006aH08aHN01h08P0[:`012eL[02807P0N028F01h08P0N
01/08R8001/07P0R5P0K5R<05Q/F6`0K8P0N01hK01HS8P0N8P0R01/F8P0F6`0R
01h06b8001h08P0R01h08aH07R8001h0oB018B3l8@8G8AOg6?hYoQL961PQ8B0Q
5aLP8OhP6b4Q0R0112420@8Q108P80@P12040P@P80@P10Cn87<46`0N8P0N8aHK
02807R<05Q/05Q/08QHK8P007P0K01/R01hK8P0F6`0R01HK01/R5P0N02808P0K
8QHK01/R01h05P0N01/R000N01/R01/R02806aH07P006ah08P0K7R808b8F01/F
6`0R02806b/[0046FR/N028K5P0N8`0N028K7P0N028K6aHR02808Q/08P0K5P0R
01/07P0K5P0R01h06`0N01/07P0R01/K5R<07Q/002807R<N01hS01/N01h08ah0
6aHS01h0823m8@0PoB405oDH02Wl5`8H8B3n8@0Gnb0I8B4P824P8@@P124P808P
120480@P10@P123n1?hPLAH06aH07P008Q/001h08Q/08aHK01/07Q/F02807P0K
5P006ah001hK02808P0K02808Q/07P008Q/07P0K5P0R6`0R6`0N02807P0K5P0K
7P0R02<N028S5P0R6`0F01/N01/08P0N028K6`0N02/[0048F2/05Q/08aHK0280
6ah0028001/F000K5Q/R01h06b806aHK8P0N02<F01h08`0N01HS8P0R01/K8QH0
6`0R01H07Q/R01h08P006ah08P007P0N02808P0KoB018B3n8@<G8B4GmQP0:O`G
0APPoR4181Ok8@4P0_dP<04212048081120480@P120P120480@P10@P6b806b80
01/N028R01HK01hF6`0K5R;n05LN6b808aH06b8F02<R01H06ah08P0R01hK0280
8QHK02808P007R<08P007P0N6b808P0K8P007R<07P007P0N01/05Q/S5P0R01HK
8P006aH08P0K:b/00@QO:a/05Q/08aHK02806aHS018K01/F8`0N01/0028K01h0
8`0N01HR02<N01h08Q/F6`0K5Q/05P0K02807Q/08R<05Q/0028K5Q/0028K7P00
8P0N6ah07P0P8B0P8B4PoR425b4QmASn8@0Gnb445`8P8B3n8@440O`P5P8P1204
808P120480@P10@P120480@PoPA`01h06b8R01/F6a/08P0R01/07P0S5R8R6`00
7P0N02806`0R01hK02806ah08Q/K01HK6`0K01h07P0R6aH07Q/07P0R6`007Q/0
8P0N8Q/07P0S7P0R6`0R028K01h08Q/F8`0N02806aHK5P0[:`012eP[01hS01HF
6`0N02<F01/F6`0N02808P0R01HK7P0R02807P0N8`0N02<R01HK01/07P0K01hR
01h07P0N02807P0N02<F6`0N02807P008Q/R01/001h06o`PoB445b4Q5b7g5`<Q
5b4GoB405ohQ0R0P0_hQ120Q0R02oB1o120480@P10@P120480@P80@P120480@P
0P@48P0K01/F8`007R806b808P0K5Q/06`0K8P0N000K5Q/R01hK01HK8P007Q/0
5P0K5P0F6b808P0N6`0K8P006b8F6`0K8P007P0R01/07P0R5P0N8aH06aHK01h0
5Q/06`0R01h96b806aH06b/[0044FB/K01h08`0F7P0R01h05Q/07P0R6ah08Q/0
8P0K5Q/F6aH08`0N6`0N01/N01HK01h08P0R01/F028K5P0R6a/08P0N02808P0R
6ah08Q/05Q/F028K02806oXQlQOn8@0GoB4@824G8B4P12448B4212420@;m87l4
8@@Q1042120480@P10@1120480@P120480@402807P0R6`0K5P0K5Q/F01/08P0R
01h08ahR01/07Q/0028K01h08P0N028S01h08P0K5Q/08P0R01/N01HS01HS5Q/F
8`0N6`0F8`0R6`0R000R7P0S5Q/08Q/07P0R01h000PN01HS7P0[:`010UT[0280
8P0N8P006b807R<08P0R01h001/07Q/R01/08P006aHR000N01h001hS028F01/R
01h08b8N02<06aH05Q/R01/N02<N01h001h06b808`0N02806aKl8@8G8B7b5`4Q
5oLQ2P@Q120P8@@Q0P42oR0L10@Q0P420@8P120480@P12020P@P120480@P10;n
16XR000N01h06b807P0K8P0F6ah07R<07P0K8P0R000R01h08P0N02<07R<F8P0R
01/F8`0N02808aH07R<08P0N01H06aHK02806aHK02806ah06`0F6`0F6`0R01HK
7P0K8P0N8aH06b/[004;H2/K028K7P0K5R806`0F6aHK01h001hR01h001h05R<0
7R<F6`0R7P0R028K01h06`0R01hK01h06`0N02808ah001h05P0N01/0028R0280
6aHK5P0R6aH08R4P8B0G8B7c5`LQ8ALQ8ALQ8?`QO`8Q124482010P4282021042
82020@82104480@P120480@P120480@P120480@P10@R02808P0K5P0R01h08P00
8P0N000R01hK028N6aH08Q/08Q/F6`007Q/08Q/07P0R6`0N6aHK000N6`0R01h0
8P0R028K5P0K01hK01H06B808Q/08Q/07P0R6`008P0K000N0280:b/00@]J:`0K
7P007P0K8ah07R806`0R02<R01/R01/R01hR01h05Q/07Q/08P0K7P0R02805Q/F
6`0R01h08P0R01/N000N8`0K8P0N028N02<N01h07P0S5Q/06b808?`Q0QLQ8OHG
124G8B4GoB4>824P8B4112448B040P42oR0N120480@20@8P12020P@P120480@1
12048@@Q12420OdP6`@27P0K6b808P0K8P0R01/N5R<F01hK8P006akn04XN8`0F
6`0K5P0N000N6`0R01HK01h001/08R8001hK8`0R6aHK01h06b808P007R806aHK
5P0N02805Q/06aHK5Q/R8P0R01/[:`012e/[7P008Q/07P006aH06aH08Q/N01h0
01h05Q/08R<F6`0K01h0028K01H08Q/K028K01/08Q/0028K6`0R01HR6`0F8P0K
02<N01/F01/08Q/07P0S5Q/06aHQ8?`Q0ALQmQL98B4G8B4G8B0P8OdP2P8Q1244
80@20P42oR0O10840@828@@P120480@P12040P8P12048@@Q104280;m86L2100F
6ah08P0K7P0R01/07P0R01/08Qh001hS7P0N02808P0S7P0R6`0F6`0R6`0R028N
028K028N000N6`008P0R01h07P0N6`008Q/08`0N000R6`0R02808`0N01/K01hK
02/[004AHb/001/N028001h06b808P0N000N02<R01hS01HK01/08P0R000N6`0R
01/K01H07P0R5P0N02<F6aH05Q/R01/F01hS01h07P0R01/R01HK02806aHK01h0
6b4Q824Q5b4Q5b7l5`8Q8AOj8@8P8B7n82/Q80@Q12448@@20@@2820412048084
80@P12040@@P120480@P12040PTP0_hPK@@Q12020@81108F000K6aH06a/R0280
5Q/S5P0N01/R01h08`0R01/N01h001h06b805Q/08P0N6`0F6`0K7P0K8P008P0N
01/N02805Q/08R806b808P0R5Q/07Q/R01/F8P0K5P0R000N:b/00A1J:ah08P0R
6aHR02806ah08Q/F6`0R5Q/001h08aHK028K5Q/F8`0F7P0R01HS6aH08Q/08P0N
02806aHK01h0028K01h08Q/07P0R01/F8`0N02807P0N02807_TQ2ALQ8ALQ8ALQ
8AOn8@0PoR41827k81848@@Q12040R04820280@1120PoPAo80@P0R0480@P1044
80@P80840@@P0R448@@Q0P42820211h05Q/05P0K7P0R6`0N028K01HK01HS5Q/K
5P008Q/F8`0N01/F8`0N02807P0S5P0F02807Q/F6b808P0K01h08aHK01hK01hK
028K01HS01HK7P0K02807P0N8ah302/[004DLR/08P0K7P0S01/N01h001h08`0R
01h08ah08P0K5P0N01h08P0N8`0K6b807P0K8P0K01h06b806aHS01h06b8K0280
5Q/08P006ah08P0R01HK8P0R000K8P0Q8B0Q8ALQ5b4Q5b4Q5b4Q5b4Q5b0Q8B3n
8@4P8OhP4`8Q1244120480@P10@P2B0280@2o@@H80@2120480@P0R040R041204
8084824P8OhPoAMR820N8`0R02<F01/K028001h06b807R<07P0F01HK7P0R0280
8P0R01HK8P0K8P0N01hR01/R01HS01/06ah08P0S5R<05Q/08P007P0K8P0K5Q/0
02805Q/R02807P0K:b/00AAO:a/08P0K5Q/F000R028K02807Q/06aH08Q/08P0N
000R028K5P0N01H07P008P0N6`0R6aH07Q/001h08P0N01HS01h05R8K7P0F01hS
5Q/K01HK028N01HK020Q8B0GoR4/5b4Q5b4Q5b4Q5b4Q5b4P8B4P8B4P808180@Q
1204820480@P0P@P120480T2o@@H80@P120480@P1204124480@20@848B0P8OhP
oAMQ82008Q/N01h05P0R6ah002806aH07P008Q/R000S7P0N6`0N6`0R6`0K5P0K
5P0S01/N01/07P0R02806aHK5Q/07P0R028K5Qh08P0K5P0R028K028K01/F6`00
8P0[:`014ed[02806aH05R<07Q/R01h06aHK02806aHK01HK7P0S7R806`0K8P00
6b808QhK8P0F01h06b8001hR01HK8P006aHK8P0K8aH08b806`0N000N8`0F6`0N
8`0R8B4PoR465b4Q5b4Q5ohQ1ALQ8ALQ8?dQ2b0Q82040P@P12048Oh43B0480@P
120480@210@PoP@40@@P123l114P80@P108P108P2B0Q0R4P827m5f0P11/F0280
8P0K7P008QHK8P0K8P0R8P0K02<F7P006`0R000F6`0N01/R02<07R8001h08P0N
01hK8P0R01/08P0F6ah05R<08P0N02<F6`0N01/N01/F8`0N8P0K:b/00@DI:a/0
8P0R6`0N02806ah08P0N028K5P0S5Qkn06hR6`0K7P0N01/F8P0K01/05Q/08Q/0
7P0K5R<07P0S01hS7P0K01/F01h08P0K7P0N028K01h08P0R01h06b4P8B4P5b4G
8B4G8B4G8ALQ8ALQ5b4280@P808P104480@P1204820480@P1202123j1002m0@;
80@P10@P12021241o23m5elP800R6a/08P0K5R<K01HK5P0K7P0K7P0R01/08R80
7Q/F8a/07P0N01h07P0K028K028K000R01/07P0R028K02806aHK01h07P0R028K
5P0R000R01h07P0K5P0[:`011e/[01hK8P0K8P0R01/R000K8P006aH06aHK028R
01h06aH07P0S5P0K8P0R5P0K01hK02808aH07P0K8P0N01h08P0N01/R02806ah0
8P006`0N01/N02<N02808?XQ01On8@0Go2408?hQnB0G120480@P1204120480@1
108110@P10@Po0@00_h40b04123n10XP1204820480@P1?hP0b4P827l5ed46`0F
01hS5P0N01HK01/S5P007P0R01hR01h06b8001h08P0R000K8P0R01hR01HK5P0N
6b805Q/R01/N01HK8P0K01h06b<R01hK01/R01hR01h07P0S5P0K:b/00@YM:a/0
5Q/05Q/07P0F6aHS5Q/F8P0K5P0K028K6aH08P0S7P0R028K01hK02<N02805R<0
8QHK000R01hK02808P0R02806ah07P008Q/N028N02806aH06`0N01/Q8?`Q023m
8@0Go24P824Q120P0R0480@280@280@P120P12040P8P108P10T2oP@08?`4023h
17<11204124480@P0R0P0P4282HE9QDV800R6aH07P0R028K01h07P0R8`0N6`0K
028K5P0K5R<07P0R6aHR01/N02806aHS01/R000N02<07P0F028K01/F02808P0F
6`005Q/06aH06ah08`0R028K02/[004?G2/07P0N01h08P0K8P0S01h001/R01/R
01HK01HS01h07P0R01h06b805P0N02806b807P0K028F6aHK02807P0K6aHK5Q/0
8P0S5Q/08P0N02<07P0K8P0F6b808?hQ0R0Q8?hQ1QLQ8ALQ8B3n8AHP0B020@@Q
124480@P120480@480@20O/40P898?`40b040PWn100PoP@60P@480@P8?/4J202
0@8P80@E:2HY808N01/R02<07Q/08P0R02806aHK028F01h06b806aHR01/N0280
8aH001/N02<07QH06aHK8P0N01hR01/07P0R01h06aHK01/N8P0R02806`0R0280
7Q/07R/[0042Gb/K02808P0N6`0F6aHK5P0R6`0N01HK5R807P0N02<N01h07P0R
01/F8ah08Q/07P0R6ah08Q/002808Q/06aH05Q/06aHK028F6`0R6aH08P0N0280
6aHS01h08B0Q8?hQ7R0Q824G8B4G8B4P8B0Q8@810R448@@P120480@P123j100P
o@@00_P400;l100PmPA[80@482028@@P82HY5BH40@0R01/F6`005R8K01hK0280
6`0N02<N01h06aH06`0R01/F6a/F6aHS000R5Q/08ah06`0N000R8`0N02808Q/F
02806`0N000K5Q/F6aH08Q/N01h00280:b/00@IL:`0R01/R01/08R8001HS01h0
7P0K8P008b808P0S5P0N02<07P0S5P0N000K02806ah001HK01/N8`0N02808P0N
02<F000K6b8001h08`0K5Q/R01/F6`0N02<FoB47824Q824G8AOn8@0PoB481@42
8@@Q120Pn@@580@210@Po0@00_d412040PTPoP@08?`400Wj100Po@@380@Q1?hP
H@@W:1DE808K8P0N028R01/08P007Q/R028001h08`0N01/R02806aHS000N000K
5Qh06aH07P0R02807R807P0K8P0K7P006a/R02807R808P0K01/R01/08P0K8P0K
:b/00?1N:a/08P0K7P0R01/N02<F8`0R02806ah06`0N6`0N02808Q/08ah06aH0
8R807Q/05P0R6`0F6`0F6aH07Q/07R808P0R7P0F01/N000R7P008P0K5P008Qh0
7P0S8B3g8A<G8ALQ824Q824280@P120412090_@4023m100Po0@20PT2j`AW8@@Q
80@P81DV:2H20@0K5P0R6a/F01hK5Q/05Q/F6aHS01h08P0R01hK01h001hF8`0R
7P0S028K02807P0R6`0K02807P0K5P008R807P0R6`007Q/08P0R01hK01h08P0N
6`0[:`011e`[01/N01h08P0K7P0R01/F6ah07P0N02805Q/08P0N8`0K5P0N01h0
6b806ah001hR01/07R<07R8001hS01HK02<07Q/05Q/S5P0F6ah08ah07P0K8QHK
028001kn8@PP8B4P8B0Q8AOm8@hP8@8Q824480@P108110;m100Po@@380@48?`4
0@89m`@02?X41@P40P@48?h40R041?hPI0@Q0R44:1DD0@825R<06aH06b8002<F
01/08P007P0N01h06aHK02807R8001/F6`0F6aHK5P0K8P0N6`0R5P0K7P0S5P0K
8Q/07P0N6`0R8P0F01/F6b8001h06a/R000K:b/00?QM:ah001h08Q/05P0R6aH0
6`008Q/0028K7P008Q/001hN01/F02808P0K7P008Q/06aH08P0N000S7P007P0R
01h08P0K5Q/07P0R8`008QHK000R01/05R<07R808?hQ023m8@0Po243824P0OhP
2P8480@P108P10@Po@@880@42@842@89l0@320@42?P4023i1009o@@02Od4FA/0
8P0K5P0R7P0N02807Q/F8`0R028K5P0S5Q/08P0N8ah06`0R6`0N01/F01/001h0
6`0R01/F028K01h08P0S01h06a/08Qh06`0N02808P0F6aHN02/[003fH2/05Q/R
01/05Q/S01h08aH07Q/08R806`0R01h08Qh002806b806a/R000N8ah08P0S5Q/R
02807P0N8aH06b807P0K5R<07P0R01/F7P0K028R01/F8ah08P0K5Q/Q8B3n8@0P
oR44824P8B7l80TQ120480@P80@Pn0@00_H420P420@420@42?d400S^10D210@2
10;n15X801HK8P0K5Q/07P0R01/R000K5Q/K8P006aHK01HK8P0N02805Q/07P0R
01/R01HK8P0N01hS5P0K7P0R000K8P0N02<F01hK02805Q/R01/N01h08`0K:b/0
0@AN:a/05Q/F028K01h001h08Qh0028K6aH08Q/N01h08ah06aH07P0F01hK000N
028K5Q/001hK02808P0N028K028K01H07P0R6`0R02<07P0R6`0N01/F01hK01H0
5R3n8@<P8B4PoR408?hQoB098B0Q120480@48?h41B0412090_/40`P410Sk1008
n`@02?d41@P410P42Od400Sk100Pn`@00_h4FA/05Q/F02<08Q/08Qh06aHR01/0
5Q/F8aH05R<K01/0028K5P0R6`0R6aH08Q/05P0R02807P0R01/F6aHR01/N01h0
6aH0028K02806aH0028K5P0R02/[003hHb/07R<08aHK01h07R805Q/06b807P0R
01h0028001h08P0K8P0N8aHK028R000N01/08QHK02806a/R01hS01HK01HK8P0F
6`0N01/F8P0F6`0N02806b8001hS01/Q8B0Q8B3n8@0PoR41827n800QoB071204
0P440P7n10822@;l1008n`@320@42?`400Si1008k@@08?d4FPP06b8001HK5Q/0
7Q/08aH06b805Q/07P0K8P0N01h08P0N02<F6`0F6`0K5Q/08P0K8P0K7P006b80
8aH06ah001h08aH07R<N02807P0S5Q/N01/R01/[:`01064[6`0N01h05R<08P0K
8Q/08P0N6`0N028S7P0S7Q/08Q/F01/001h0028K028R01h07P0S5Q/08P0N02<0
7P0R028K01/08P0R6aH06a/F6`0R6`0N01HK5R<07QH0824Q8?hQ023n8@0PoR41
827n80XQ80@P12040R040_d400;g1008n@@02?d40`P410Sj10H810@810@8n`@0
8?`4GB0410TK01/F8a/07P0R000R5Q/05Q/F8P0R01h06`0F6`0R6ah07P0S5P0R
01h08aHN028K02806aHR01h07P0R01/F8`0N028K01H07Q/002808P008P0N6`0[
:`0115l[01h08P0N8`0F6b806`0F6b8001h06ah001/N000F6`0K8P0R8P0K8Qh0
5Q/L8P0R01/F6`0F6ah07P0N02806ah08P0N8`0N01/S5P0S5P0N01/R01h08`0N
01/S8B3m8@0PoR408?hQ5b0Q0R0282048@@P120P120P10@P10898?/400Sl10<8
10@8o0@02?`400Sn1008n@@420@42B3m100Pn`AI028F01h07P0R01/N01/08R80
01/06b808P0N8`0F6aH08P007P0N6b808P0K01hK01h06b807P0K8P006b808P0R
01HK01/R01/08Qh07Q/07Q/R000N:b/00@MR:a/08Q/001hK01HK5P0R6`0N0280
8P0K5Q/05R<N01HK01h06`0R01/08Q/08Q/07P0F028K01H08`0R6`0N6`008Q/F
01h002807P0N02<N01h06aH08Q/F02807R4Q824PoR42824QoR008OdP2`@1808P
120412048?`400;m1008o0@02?T400Sm1008o0@02?d400Sn1008o@@820@810T8
10P4oPQJ6`0K02808P0R6`0F01h08Q/08Qh08P0K6`0R01h08P0K5Q/F8`0R000N
6`0R0280028K5P0K02<N01h07P0K5Q/N028K01h06aH08Q/08P0N000N028K02/[
0043GR/05Q/08R8001hS000F6`0N02<N01/R02808P0N01/R02808P0K5R<F01h0
5Q/08P0N8aHK02806aHK7P0F6`0R01h06b8001hK8P0R000N01/R01HS01h06aHK
8P00oR038B0Q8?hQnR0;1242804480@P80@Pn@@00_/400Sl10<810@8oP@02?`4
00Sl1008o@@02?h4J0P810P410P420@810P802805Q/R01hK01HK8P0F6`0N6`0R
01/F01HK8P0R01/F8`0R02806b805P0N01/R01h06b8F01h08P0R01/F8`005Q/0
8P0R02806aH06b808R8001h06b/[0047>2/K01HK6`0R7P0N018002808P0K5P0K
5Q/07P008P0K5Q/07P0N01h08P0R6`0N02<07P0F8ah05_h0928K028K7P0N01/R
7P007Q/08Qh07P0K5P0R8`0N01/07P0R827n8?hQ620Q824P8244804280828@T2
12048204123n1009o@@02?`400Sf1008o0@02?`40PP81?h8J@@420@820@810P4
10P801hK02806a/05R<K01hS01HK000R6ah06aHK01/F6`0F6`0R6ah07P0K6`0R
6aH06b808P0K02808Q/K02807P0R6`0N01h08Q/08P0S7P0K01hK028K5P0[:`01
0eh[5P0K5P0N6`0R028K01/N01/N01/R01/07R<R8P0K5R<05Q/R01h08P0K6`0R
01/F8P0S01h07R<R7R8K5P0N000R02806`0K8P006ah08P0S5P0K7P0R02808P0N
0?`P0R4Q8?hQ4@8Q820Q80810R010R0480@P8?0400Sm10D810@810Sl1008o@@6
20@820@42?h42PP810P410P420@4oPQK01/08P0K5P0R01h08P006b806b807P00
5R<07P0K01hS01h07P006b808P0N01/R01h06aHK8P0K7P005Q/R01/N02805Q/R
01/R01/N02805Q/08R8001/[:`011fX[6aH08aH05R<07P0F8P008P0K5P0N01HS
01HK6aH06aHK000R6`0R6ah05Q/08P0K028N000R01/001/08Q/08Q/08Q/F0280
7Q/F000R6`0N02806ah08Q/F6`0S7R4P824P8B0Q8B0P8O`P2P8P808P80@P10@9
n0@520@810@8o0@02?P46PP410P420P420@810P810P420@810P420@81?h8GP@8
01h08Q/N01/F6`0F8aHS7P0K5P0N6`0R8Q/07P0R02807P008P0N6`0N01hR01h0
6aH08P0K01h05P0N8a/07P0F028K02<07Q/07Q/0028K7P008Q/08Qh0:b/00@eP
:`0K8P0R01/F6a/R01/S5Q/N01/R01/R01HK000K8QH08P0N8`0F6`007R<05Q/R
01/N01/R01h06b805Q/F01h05Q/06aHK8P0F01hR01/R000K8P006ah06`0K5P02
8OhPoR42824QoR008OhP1@840@848?h40P840_h40PP42?P400Sk120810@810P4
20@810P420@810P410P420@810P420@810Sn15h8200K8P0N000F8`0R01h07P0S
5P0F6`0R01/06aH06aHK8P0K8Q/F6`0R028002<06aHS01/R02807R<R01H07P0R
01/N02<F8P007P007R806`0R01h06ah06b/[0047HB/K01hK02806`0F6`0F6`0F
02807P0F6aHK028N01/08Q/F01h08aHK000N5Q/07P0F02806a/07P0N02<08Q/0
8Q/07P0S01hK02806ah07P0N01h08P008P0R01h08@8QoR3m8OdP20@P80@P1084
0O840`P410Sm17l810@810P420@420@810P420@810P420@810P420P420@810P4
20@820@421h06aH08R8K01hK000N01h08R<K01h08P0N02<N01/07P0F6`0S5Q/0
8Q/N01h05P0N6`0N01h001hS6`0R028K01H07Q/07Q/F028N01/F028K7PP07P00
8P0[:`012V<[028001HK8P0R01H07P0K8P0K8P0K8P0S01HK02805Q/06b808QHK
01hR02<07P0K8P0K8QH08P0R01HK5Q/05Q/07P0R01h08P0K8P007P0R8P0K8P0K
5Q/F6aHK01hQ820QnB0?120Q120480@P0R040P@48?`40`P410Sk1008o0@Q20@8
10P810P420@810P420@810P420@810@810P420@81?d8GP/K01h06b806`0N5P0S
7P0R01h07P0R01/F8aH07P0R01h07R<07P0N02806`0R01/R7P008P0F6`0N8aH0
8P0K7P0S8P0K0280028F6`0R01/F6aH08`0K8P0K:b/00?eO:ah08Q/07P0R6a/0
8P0R01/F01/F01/F6aH07P0K5R<07P0K01/05R<07P0N000R01/N01/08Q/07R8S
01h08Q/08P0R6aH08Q/F02<N028K01/F028K02806`0R0280oB038B0P8O/P1`@Q
120480@Pl@@320@42?d400Sn11X810P420@810P420@810P420@820@810P420Cf
25d07P0R01/N01h08ah001h08Q/0028K5P0K5Q/F028K000R01h002808Q/N01h0
8P0K028N02<K028001hK02806aHK02808Q/N01/05R<07P0S01/F02807Q/0:b/0
0@5R:`0R01/R000K5P0F6aHK8P0K8P0K8P0N01/R01/F8`0N02808QHK8P0F6b80
8P0N8aH07P0K7P0R01/08P0F6`0F6aHK02806`0S5Q/F01/07P0S5Q/05Q/R01HK
8P0N820Qnb0A8@@P80@Q120480@P10@P10@Po0@320@42?T400Sm11@810P420@8
10P420@810P420@820Cc264420PF8P006b8002808P007R807Q/08P0N01/R01/0
8aHK028F6b808R806ah001HK01/R01hS01HK5P0N01hK02807P0N02806ah00280
7R<07P0R01hF01hK8P007R/[0047J2/K01h07P0N02<N000S5Q/F01h05Q/00280
6aH06aHK028K01h001hK000R6a/F01h08`0N000R6`0F6`0R6`0R6`0K01h08P0N
01/08Qh08P0R02808Q/06aHK01/F01/P824P8B0P8O`P20@Q120480@P8?`40R04
2O`40`P410Sj118810@810P420@810P420@810P4nPP01?h80`@820Cl25l;01hK
5P0N6`0N02<N000R8`0F8aHN01h07P0F8aHK01h08`0N6`0N6`008Q/05R806aH0
7P0S5Q/F0280028K7P0N028K7P007Q/08P0N6`0R6`008Q/05Q/F8`0[:`012FH[
01HK8P0S8P0N01/R5Q/06b807R<08QHK8QH06b805P0N02806b805R<F6aH06b80
01hR01HK7P0R01h06aH07P0R01h06a/F6b805R<R01h06ah07Q/F01HS01h06b<F
8B0P8@8Qo2001?hP2@@P120480@22B3m1002oP@02?X40`P410Sm114810P420@8
10P420@810P420Cn20@420P;1?P8H0/07P006b8001HK8P0K8P0K7P0K5Q/08`0R
02805Q/08P0N028001HK02807P0N8`0N01/R02<F8`007R<07Qh001/R01h001HK
8P0F6b8001HK01HN8`0F6`0K5Q/[:`00n6D[6`0F6aHK02808P0K01HK01/05Q/0
8Q/06`0R01/F8ah07P0R01/08P0R01/F02<N000R6`0F01hK028K02<N01h08P0R
01HK01/F6`0K000R01/001HS7P0N000R01h00R4P807h80/48@@P120480@22@;g
1008m`@@20P420@810P420@810P420Cd20<420P4o@QO5Q/F6`0N8Q/001h05Q/F
000R01/08QHK6`0R6`0F6`0R6ah07P0S5Q/08Q/05Q/002806ah07P0R8`0N0280
8Q/06`007R8S01/F6`0K5R8K028K01h08aH001h0:b/00@iQ:`0F6`0K5P0K7P0S
5P0N01h07R<05Q/08P0F6b8F01/08P006aHK8P0N6b808P0N028N6`0R01hS0280
01hF01/R01h06aHK01h06`0N028F6ah08R807P0K01hR01h07R7j800QoB0>1244
80@P120480@210@2oP@?20@420@420@420@420@42?h45@P420@810P420@810P4
20@820@820Cn2004nPQQ2`0N02<07P0S5R<R01HK5R<07Q/R01HK6`0F01h06b80
6ah002808ah05P0N01/R5P0N6b8001h06ah08P0F6`0K01h08R806`0N01h08aH0
6`0F6`0N02806b8R01/[:`00oVL[6`0F8P0K5P004P0N8P0R02807P0R01hK0280
6`0R028K7P0F8`0N000K028K7P0K000N02<N01h08Q/F02<N01/F02808aH08Q/0
7P0R6`008ah06`008P0R01/N02<07R4P8B0P8OdP2P@P8@@Q120480@Po@@280T2
m0@02?d440P810P420@810P420@810P4l`P01?d8H@XS01h08P0N01/F6`0R01/F
8`0F6aH05P0R6aH08P0K5P008Q/07P007R8K02806`0R01h08Q/07P007Q/08P0R
6aH08Q/08P0N02808P0R6`0R6`0R028K5P0K0280:b/00@iQ:`0K8P0S5P0K8P0N
01/07P0K8P0F6ah001HK5Q/R01/N000N8`0N01hR01/N000R01hR01/N028001/F
8`0N01h08`0K8P0K5Q/08P0F6`0S5Qh001/R5Q/N02<F02807P3m800QoR0B8@8P
8@@P80@P120480@P10890_d40`P410Sn1008o0@A20@810P420@810P420@810P4
n`P01?h800Cm200;o@QQ8`0N01HK8P0N01/07P0S5P0F6b8001hS01h08ah08aH0
7R806b807R808aH06b805Q/R01/N02808aHK01hK5Q/F01HK01hK8P006b807Q/F
01h07P0K7P007P0R01/[:`012V<[7P0K5Q/08P0N6`0R028S7P0K02808ah08`0K
01h05P0R6`0N000R8`0N000R6`0R01/F000R6ah05Q/08P0R028F7P0K5P0S5Qh0
8Q/07P0S028N01/05P0N6`0R6aH08R0QoB0?8B0P0@848B0480@P12048?h400;g
1008oP@I20@420@810P420@810P420@810P420@820Ci26T420P:20/801HK028K
01/002808Q/N01/F6`0K5R<07Q/08P0K7P0S7P0K7P0K5P0K7P0K5P0K5R<07P0F
028K5P0R028002<08Q/05Q/05Q/F8P0K6`008Q/08P0R01/F8`0N6aH0:b/00@QQ
:`0K5R<05Q/F6`0R01/N000R01HK01h08aHR02808P0K7P0R02<F6`0N01hR01h0
6b808aHK8P007R<08P0K6b806`0N01/R028001/07P0R01h06b808P0N8`0F6`0K
8P3l800QoB0=124P124480@P1204123i10<810@8n`@A20@810P420@810P420@8
10P4oPP01?X8K0@82@/82`P;20PN6aH06`0N01hS7P0N000K5R<06aH07R8001h0
8QH001h08P007P0K8aH08aH06aHS01h06b806`0K5Q/N02<F6aHK01hS5P0N02<0
6aH06aHK02807P0S5P0K8P0F01/[:`0136D[6`007P0R02<07P0R01/N01/F8`0R
01h07P0N6`0R6`008Q/F6`0F6`0R01/N02806aHK01HK5R<07P0N6`0F6a/08P0R
01hK028R02808Q/N01h06aHK000N02807P0K7R4P827n81@Q808Q12010R0480@P
12091084123h11H810@810P420@810P420@810P420@81?/80`@820Cl200:oPP0
2_h0H28N01h07P0F028K5Q/001h05R<K01h08Q/K02<N5P0R6aHS01h07P0R028K
01h07P0R01/N01h06`007P0R01h08P0K02808P0N028N02<F7P0N01h08Q/F6`0R
6aH[:`011F4[01hR01HK8P0F6b806aH08aH07P0N01/R01h001/N02807P0K01hS
5P0N01/08P0N01/F01h08`0F6b8001HK01H07P0K8P007P0K8P0K7P007P0K5R<0
8R805Q/R02800?XP1P@P8@@P80;n80D480@P10;i10<810@8o@@C20@810P420@8
10P420@810P420Cl2004n0QY2`P82`PS8P0N6`0R02806b806aH001hR01hS5P0N
01/R01H07P006ah08`0N02808P0N6`0R02808P0S5P0R02806b807Q/R000K8P0K
8P0K7P0F6`0N01/07R<06b807P005Q/F02/[0046I2/K01h08`0N6`0F6`0F8ah0
8ah08Q/F01/08R8K000N6aH08P0N000R6`0R028K6`0R02<N02807P008Qh08P0R
6aH08P0N6aH07P0K7P008Q/08P0K5Q/07Q/001hK028K020Qnb0<12440B0280@P
12048?h4023h11@810@810P420@810P420@810P420Cm2004n`P01?X8J0/820X0
6`0R028K01/F01/R02<N000R6`0F6`008P0K5R<N028S000R5Q/08Q/07Q/05Q/0
8Q/K01h08Q/08Q/F01h001h08Qh06aH07Q/001hR02<07P0R01h001hK028K000R
6b/[0044HR/08P0K7R8001hS02807P007P0F6`0K8P0R01/05QhS02806b808aHK
5P0K5Q/05P0N01h08P0N8`0N6`0F6aHK02806ah001/R02<F000K7P0R01h06`0F
6`0S8P0F01h06ohP4b4282048@@Q0P8Q120480@P10@Pn`@320@42?H430P420@8
10P420@820Cj2004nPP02oh8IPXS01/R01hK02806b806`0N028N6`0F6`0R7P0B
000N02806aHK7P0N01h08P006b808P0N02805Q/06ah001hS8R8001/07P0K8P0F
01hS01/F8P0R02<05QhS000N028F6`0[:`0137L[6`0R000N6aH07P0R6`0R7P0S
01HK01hK02808Q/05R8K7P0K5Q/0028R01/08Q/F02807R<07P0R01hR8`0K028K
5P0S7P0F6aHK028R01/F6`0F8P0R6`0N01hS01h08P0P824P80@Q80@Q12440B04
80@P123m100Pn0@B20@420@420P420@810P420@81?d80`@820Cl2004o@P12P_n
26D07P0K000N01h05Q/F02808R<05R<K028K02807R<07P0R01h002808R<F6ah0
5Q/F6`008Q/N01HK000R6`006a/F8P0R02806aHS7P0F6`0K01hK7P0R6`0F8R<0
8Q/05R<[:`011B0[02806ah001/R01hK01HK02807R808P008P0K7P0R01_n03lF
6`0N8P0K8P0F6aH06b806`0N01HK8P0K01/F01h06b<F01/R01/08P0K8P0R0280
6`0N02808P0N01/R01hQnb0=12448@@Q0R0280@P123n10422Od41PP410P410Sm
10X810@810P420@81?8800_n200;o@QV6b807P0R8P0R01hS01/R01/K01h07P0F
6`0R01hK02807P0N01/R01/N000N000N02<05Q/N000N8`0F6ah08P0N02806a/R
01/F8`007R<08QH07P005Q/08P0N01/N01/R02/[0049I2/N02808Q/F01/001HR
6`0R01h06aHK5R8K7P008Q/N028R6`0F6`0N6`0N02808ah06aH08Q/06`0N000R
028N01h07P0R01/F028K7P0K5Q/08R8K5P0R6ah08P0R01/F008QoB0C12448@@Q
12440@@P1204120410;d10/810@820@810P420Cn20H420P420P4n`P22PP;oPQY
2`X201/001hK028K01HK5P0N01H07P0R028K01h08P007Q/08ah08P0K7P007Q/0
8R807P0R6`008Q/07P0R000N6aH08Q/05P0N6`0N028K01h06`0R028K01HS01h0
7P008Q/07R/[004?NB/001/F6`0K8P0R8P0K01/N02<R028001h001/R01/08P0N
01/R028001h07Q/07P0R02<F6`0R01h08aHK5Q/08P0K5P0K8P0R01h001h06`0N
01/001/N000F6`0K8P0K8R4P824P12448@@P1209808212048089o0@620@420@4
2?h44@P420P420@810P420@810@81?P8L@/820/820X;20PP7P0R8P007Q/05Q/0
6b808P0K8P0K01h08P0N02<N01H07P0S01hF02<F6`0R01/R01/N01HK7P0R0280
7R<F02<F6`0F6aHK028001h07P0R01HK8P0K8P0K5R<R02807P0R02/[004;IB/K
5P0S5QH06aHK02808P0K7P0K7P0R6`0R7P0N01HK02808P0K5Q/08P0R000N028K
5Q/05Q/08P0R02808aHK02808ah07Q/08Q/F6`0R028K5P0R8P0K5R<K02807P0F
6b0P1?`P3P@Q120480@P120480@40_d400Wj11<810P420@420@810P420@810P8
1?d800Ck200;o@Q[2`P:80@07Q/08Q/05R<K01h06aHK5P0K5P0R8aHK01/N000N
8Q/07P0R02<N01/08Q/K01/F000R6`008Q/F6`007P0R02<F6`0F8P0R6aHS0280
8Q/N000N6`0N01h06`0N6`0K5Q/0:b/00A1g:`0R01h06b<F01/R01hK8P005Q/0
8P0N01/07P0N8aH06ah06aHS01hK8P0K8R806`0N01/R01HK8P0N6ah08P007P0N
02805P0N02<F01h07P0K8P0K8P007P0R01h06b8000@Q120P120Q120480@P1204
80@Pn`@620@420@42?h44`P420@820@810P420@810P420P4nPP82`P82PP82PP8
oP9V8P006ah08P0N028F01HS01/R02807P0K028001hR01/08P0N01/N02805Q/F
01/F01hS8P0F6ah06`0R5Q/R01/F6`0R01/06`0K5Q/N01/F02<N000F6b808P0F
6`0R01/07R/[0049N2/K02808P0N02807Q/05Q/F6`0F6`0R02808P0S01h08P0K
5P0N028001hK01/08P0N01h06aHK01/F000K7P0R7P0R01hS6`0R01h08ah08P0N
01/N01/F8`0N01h08P0K7P0P12448B0480420P@P120480@P123c1008oP@<20@4
20@810P420@81?`800Cm20D:20P;20_n26X:0R0481h05P0R6`008Q/08Q/F028K
028K01h08Q/F8`0N000R6`0F8`008Q/F6`0K5P0S7P0N028K000R028K000R7P0K
028K5P0R01h001H08Q/08P0R6aH001hK028K028K5P0R02/[0048Hb/05Q/F6aHK
01/R000F6`0K5R<F01/K8P0K6b807P0R01h06b807P0S5Q/08P0F6b808P0R0280
6aH06b8001h08`0K5Q/001HK7P007P006b806aH07P0R01HK01h06b8001kl80h2
0B0280@P120480@P123m10T810@810@810@8o0@:20@820@810P420Cf20<;20P;
o0Q[2`@P120402<R01/07R807P0N01/F6`0F6aHS5P0N01/07P0S8Qh07R<07Qh0
5R<08P0R01h08P006`0N8P0K7P0S5Q/06aH07P0S5Q/R02<F6b<N000N01h06b<R
01H06aH07P0K8P0K:b/00@Af:a/05Q/06aHK01h08Q/07P007P0R01h08P0N01h0
8Q/K02806`0R7P0K028K5Q/07Q/08Q/07P0F8ah07P0R6ah08P0S5Qh08`0R8Q/0
8Qh06aH08aH08Q/N01HS02807Q/F00@P80@P808P80@1120480@P123b10h810@8
20@810@810P420Cm2004n0Qb2PP82`X82`P4120481h06ah08P0K7P0R02808aHK
000N02<N01h08P0R01/07P0N000S7P0N01h06aH08Q/N01HS01h05P0R5Q/08P0R
7P0N01/07P0N000N028R8ah07P0N01hS7P0K5P0R01h05R/[0049J2/07R<06b80
01h08ah08P0K8Q/06b807P0N01hS01h001HK8P0F6`0S5P0N01/08P007Q/08P0R
01/07P0S8P007P0K5Q/08aHR01/05Q/07P0K8P0K8P0K02807P0K7P0F01/P80@Q
1?`P2P8P108P120480@Pn@@320@42?d42PP410P420@810P4oPP01?`800Cj270:
20P;20T480@P100R5P0F6`0N000K8P0K5Q/08R8001h06b807P0K7P0R02808P0N
01h08P0R5R<07P006b808P0N8P0K6`0F6aHK02808P0F6b808R8001/K000K0280
8P0K02808P0K5Q/06b/[004:O2/K01h05P0R8P0R01/F6`0F01HR01/06aH08`0N
01/F8Q/07P0R01h08Qh07P0R6`0R01h07Q/08P0R02807Q/002807P0N01/08Q/F
028K02807P0F6`0F8a/F02806`0R6`0482448@@P80@P120P10@1120480@40P@2
n`@02?h42`P810P810P420@81?X800Ch274;20/820`4108P120N02<N01HS028N
01/F01/08Q/07R<07P0K5P0R01/F6`0K5Q/F02<08Q/K01/F8`0R7P0K7P0S01/F
01HS01HK028K01h08`0N6`0N02807P0R028K6`0R028K01hR02<F8P0[:`012FP[
02805Q/R01/N01/F8`0F6b806b808P0K8P0N02<F01/07P0N6b805Q/08`0F6`0N
6b808P006b807P0K7P0F01hK8P0R01h06ah08P0K5P0K8P0R01/R01h08P0K8P0R
6`0N02048@Ck80X4120P120212048?/400Sk114810P410P420@810P810P420Ck
2004oPP22`P;oPSm16PP108401h08P0K5Q/07P0K8P0R01h07P0N02808P0K8P0K
01hF8`0R01/F6aH06aH06aHK01/F01/F8aH07R<07R<05Q/05Q/R01h002807Q/R
01HK5Q/F01h06ah08P006aHK01/[:`0116@[6`0R6`0N6`008P0N028K01h07P0N
6aH06aH08Qh08P0R02<001hK01HS5Q/05Q/001hK028N01/N02806`0R8P0K028K
01HR000R6aH08ah06aHK01HK000R6ah06aH05R<07_`P00Cn80X480@480848204
8?P44`P410P810P420P420@810P410P4o0P01?L80PX;3?`4Ib4481h08a/F02<F
028K01h07Q/08P0R028K01hK01h05R<06aHK02806`0K5P0R6`0N02806aH07P0K
5P0N000N02808Q/07P0N028K000F6`0R01/06aH08P0K5Q/F8`0F0280:b/00@QW
:`0F6`0R01H07Q/R01HK02807P0R000K8P0K8P0K01h06b807R8001hS01h07R<0
7R8001/N02<F000K8P0N01/R01hK02806aHK7P0S01h07P0K01hS01HK7P006aH0
6b808P0P80Cl80X480@P80@20@@48?h4023l1008n@@:20@420@810P420Cm2004
nPP32`P:2oh8oPAZ0P@P2B04000N01/R01/F6`0R02805P0K8P0N6`0R000N01/R
01h06`0N6b808P0R01h08P006b806b808P0K8P0R5Q/R01hK02808P0F6aH07R<0
7Q/R02808P0K8P0R000K5Q/R01/[:`0127P[6`0F8`0N02806`0R6`0R6`0S7P0N
01HK5P0K5P0R8ah06aH06ah001h002807P008Qh05P0R028R6`0N02<N01/F01HK
02806`008Qh002808P0R01h08P008Qh08ah06ah08P81808P80@Q80@P120480@P
0P@1n@@30PT42?d42PP420P420@810P4n@P01?`81@/820X83?`4J082120R6`0N
01h05R<07P0N6`0S7P0K000N01hS7P0R01/F02808P0N02807Q/0028K7QH07P0F
6`0N000R6a/0028K000R6a/08Q/08ah07P005Q/F6`0N6`0N01/N01H06`0R02/[
0048NB/R01h07R<06b805Q/05Q/F01h07P0S8P006aHS01h001h06b<F02<R5P0N
6b8001hK02<R01/N01/07P0S5P0N02806`0N6b808P0N01/N8P0K5Q/N02807R<N
000N028001h082020B048@@P80@210440R048082oP@00_L42@P810P410P420Cl
2004n@P42`X820_m26`40PTP120P11/07P0R01hS01HK8P0F01h07P0N8P0R01h0
01/R01/F6b806`0N01hK028F6aH06b807R<07R<07Q/F02<F6`0K5Q/F01/R000N
02806b8N02<08P007P0S5P0F6b805Q/N:b/00@MX:a/F02807Q/07P0R028K01/F
02<08QHK5R<F01hK028K01h07P0R01h08P0K028R01h07P0R01/F02808QHK000R
6aH08P007Q/08Q/05P0K5P0S000N6`007P0R8P0R6aHS01h28202oB01107n80P2
80@40@@P2@;m100PoP@>20@420@420@810P420@;nPP01?X800Cl2?/4HR0480@0
7P0R6`007P0S01hS02808P0R6`0R6`008R807P0F8`0N01HS0280028K01HK000R
6`0F6`0F8P008ah07P0R02806aH07R<N01h08P0K01h08Q/N01HS5Q/08P0S7_h0
0R/[0043IR/08`0K7P005R<R01hK02808P0K5Q/N02<06b805P0N01h08P006b80
6a/R01h06b808P0K6b806b807Q/06b807P0K8P0S7P0F01h05Q/R01/R01HN8`0R
8P0F6`0K6`0N02800@;l81048@8P80@Q120P12040P@48?L42`P810P420@810P8
1?h800Ck2004m`Sn16X210@P1244000K01hR02807P0K5Q/K8P0K01HK01hS7P0N
01hS01h07R807P0S5Q/07R<08QHK01HK02807P0K7P0R000K8P0N01/R01h08P00
6b808P0K7P005Q/F6`0N8`0N02<F6b/[004BJB/N02806aHR6`0K000N01h08ah0
6`008QHN01/08R<07P0R6aHR01h05P0S7P0R01/K01h05Q/F01/F01HS01hS01h0
7Q/05R<K5P0R6`0K5P0R6`007P0N6`0S5Q/05Q/08Q/K00@P80Cn80d20@8P8084
80@P12090_/400Sn10h810P420@810P420@820Cm20<420P4o@P01?h810@820@8
oPAY80@480@Q8R807P0K7P0R01h06`0F6aH08Q/07P0N02<07P0N000R8`0N01hK
01HK01h08a/08Q/08Q/06aH0028K7P0F6`008P0K000R6ah07P0N02806aHS000R
02807Q/07P0R02/[0040IB/06aHS5P0K01h07R807R<07P0R5P0N01/08P0R6`0R
02807P0S7P0N8`0N000N01h08P0F6`0K8P0K8P0K5Q/08P0R000R01/F8`0N02<F
01hK028R01HK02807P0F8aH07P007_/P3@810R020@81120480@2m`@420@810Sl
10L810P410P81?T800Cj20442?`4J20P12427P0R02<F01/K8P0R01HK02<F6`0R
02808P0R028001hK01h08aH06b808P007P0F6aH07P0R01/R5Q/F01hS028F6ah0
8R806`0R01HK01/R01/F8aHK5Q/N000R02806b/[0049Nb/K01/F02808P0R8`0N
01HR028K028K5P0N8aHK01hK01/001h06`008Qh08R<07P0R6`0R01h06aH06aH0
7P0N6`0N6aHK5P0K5R<07P0S7P007Q/08Q/08Q/F028K028K028N008P10428202
8B0P0P420P@P1204123m1008o@@02?d42@P410P420@820Cl2004n@P01?h8n`AY
80@480@Q01hK01h08P0F6`0N02808P0N028K01hK01hK01hR6`0F6`0R01h05Q/F
6aHR028K01/F028K5P0K01HK000N6`0S5P0R6`0N028K02<F7P0K5P0K5P008`0F
8ah08Q/N02/[0040N2/07QH06a/R01hK01h08P0K8P0K01h06aHK01/08P007P0R
8P0N01hR01/R01/N01HK01HK8P0N01/R01/R01HK028001h06b806aHS5P0N01/R
000N8`0F6`0R01/05Q/05Q/07R448@8P0@@Q120282040P440R3g1008n0@520@8
10P4o@P01?T800Cm2?`4J`TP108P10425Q/08P006aHK02808Qh06ah05Q/07P00
5Q/08P0S5P0K5P0K7P0K8P0K01/S5Q/07P0S5Q/06b807R<08R805P0N01hK01HK
8P0N01h001h06b806b<N02807P0K7P006b/[0049JR/K02<N01HK000F6`0R6a/0
6aH08P0R02<F6`0N028N028K02808R<07P0K5P008Q/08Q/06aH08P0N01HK02<F
6`0R7P0R01/F01/F6aH08P0N8ah07P0S5Q/08Q/F6`0R6`0R00@Q1202oB0:1244
82040R04123n108P2@;k10/810@810P810P420Ci20<420P4n@Q_10@8108480@P
0R0P0P0N6aHR02<F6a/06`0R01/08P0R6`0R6`0N02808ah08ah002806aH08P0N
000R02808aHK01h001h06a/F8Q/08P007P007P008Q/F8`0N01h05P0R7P0R01h0
028N02/[004;N2/F01h001/F01hS5P0N000N01/R01/N01h001h06ah001/07P0K
6`0N01h08P0N8`0F6aH05R<06b807R<05Q/F01/N02<07P0K8P0N02<06b805P0N
01/F6`0N6`0R5P0N01h08`4282010R0P1242804282040P7l1002o0@02?h42@P4
10P410P420Cm2004oPP210P4o@Sn1008n@AX80@P120P6aH08`0N01/F028F01/R
01hS01h08ah05Q/R01hK5P0N000N6b806aHK8P007Q/R01hF6`0N028R02805P0K
02806aHK8b807R<N02<08P0R01HK8P0K02<N02807P0K:b/00@YY:a/F028R02<N
01/08Q/F6`0N01h05P0R6aHK02806b8N01h08P0F6`0R028K7P0N02<07P0R01h0
6`007R80028N000R6aH08P0K5P0R6aH06ah08`0N000F8`0F6`0S7P0R01h08b0P
0_hP3@@Q104P0P440P440P@PnP@02?/44PP810P810P420@420@810@820Ck20<4
20@8n`AY80@P12040@0S7P0R01h06aHS6aH06aH07Q/07P008Q/06`005R<N028N
000N01HS01hK5P007Q/06`0N02<07Q/08Q/05R<07P0R01h08P0N01h07Q/07P0S
5Q/F01h06`0R6`0N02/[004?N2/06b806`0N02<F01h06`0R02808P0K6`0R01HK
8P0K01/R01/F6`0R01/N000R01h08P0F6aHK8P0R5Q/08aHK02807P0K8P0K5R<0
7P0K8P007P0R01/R7P0N8aH07P006b806aHP0P420@@Q12420R420R040R3n100P
n@@620@420@42?/44@P810P420P410P420P420@82?h4KPP41084120480@P0P0N
01HK8P0R01/F01/R01/R000R028F6`0N01hR01/05Q/06b807R<08P0F02<F6`0R
02808P0N6`0K7P0R8P0R5P0K5Q/08b808P007P0R02<F6`0K8P0N01hK02806b/[
004:N2/K01hK02807P0R6`0R028K5Q/08R<F028K5P0F6aH08P0K5P0S5Q/08P0K
5R<08Q/N028K01/07P0S7P0R6`0N02<N01h06aH06aHS01h07P0R6aHK01HK01h0
028K028N01/F01/48@8P0R448@@Q120P10@P123j1008n`@02?d400Sn10/820@4
20@810P420Cm20@420@42?X4JB0480@P808N01h06`0R6`0F6`0F6aH08Q/F6`0K
01HS01h07P0F8`0F8aHK000N6`0R6aHK01HS01h08Q/F000R01/F8`0N6`0R01/0
7P0K6`0R8P0R6aHK01H07P0R02807P0N01H[:`010WL[028001HK8P0F6aH07P0N
01/F01/N01HK01/R6`0K8P0N01/R01h05Q/R01h06ah001/K02808P007P007P0R
000N02<07P0K8P0N01h08P0R6`0K01hS028001hS01HK01h06b8000@Q12448@81
0R01120P0P7_1008o@@;20P410P420@810P4n`P110Sn1008o0AY80@P1204800S
02805Q/07R<07R8001HK01h08aHK8P0R02808P0K8P0N000K8QH06ah08`0K8P0N
01/N01/S5Q/R01/07P0N01/R01h06aH05Q/N01/06aH06b807Q/06b806b806b/[
0048Ob/K028N000N6`008R<08Q/05R<N000R6`0N01/07P0K5P0R01h0028K01/F
02806aHK5P0R6aHK5R807R<07Q/F8P0R5Q/08P0N000R028K01h08P0N01h08ah0
7P0R6`0R02807Q/080@Q12440@81808Q2@8P120P0P4280@400;e11@810@820@8
10P420@810P810P810Sg16TP120P124481HK028K01h05Q/06aHS6`0R01h05P0K
7P0R6a/08P0N6`0R7P0K0280028F7P0K5P0R000N01h05Q/F028S000N01h08Q/F
028K000R01h06aH08ah002806aH06aH0:b/00@Ym:`0K7P0S7P008aHK5Q/N0280
7P007Q/08P0N02808P0R01/R01/R6`0N01/R01h08aH05Q/08P006b805Q/R000K
7P0K02807P0S8P0N6`0R01HK8P0R01/F6b8001/F01h06b805P048@@Q12428202
808P80@212040P@Pn@@C20@810@810@820@420@810P420Cj20@420@42?`40b04
80Cl864N01/N028001h08aH07P0F6b805Q/S5P0K7P007Q/R000F6`0R02807QHK
6`0R01/R01hR028001/F8`0N01/F6b808P0F01/N01/F6ah08P0R01h08aHK7P0K
8aH06b/[004>Ob/N000R6`0R7P0S01/0028K000R8P0F8`0R028K5Q/07Q/07P0F
6`0R01h06aH08P0S7P0N01/N01/08Q/06aHS000N028K7P0N01hK01HK6`0S5Q/0
7P0F000R8Qh08ah08P0K7P0N80810P4282020@8180@P12040@8P10@P0P@48?P4
40P420@820@810P420@810@8oP@320@42?X4JR0480@P80@Q100N000R6aHS0280
8R<08Q/06aHK01h08P0K5Q/05Q/N02<F6`0R6a/06`0F6`0R01/N01/F6aHR01/F
6`0N02<07Q/08Q/05P0R02<05R<07P0R01h08aH07P0N0280:b/00@Mm:`0R01h0
5Q/07P0N5P0N6`0R5Q/N01/07Q/N01HS028001h07R<05Q/R02808P0K7P0K8P00
7P0R01HK01hF01/F7P0F6aH001/R000K8P007QHK02805Q/S5Q/K000N01/R01h0
01/080810R0P1242820280@P120480@PnP@02?h410P420@8n@@920@810P810@8
2?L4J`840@@20@@Q0R0N02<F6aH06aHK7P0N01h06aHS028001/R028F01/F02<F
6`0F6aH08P0N8P0F6b8001h08`0R02<F000N01/F8P007Q/08R806aHK5Q/F8P0K
6b807P0K8P0R02806b/[004>Jb/K028K5R808P0R02<N01HK02<002807P006ah0
7Q/08Qh08`0N02<07Q/08Q/F000R01hS7P0R6aHS01h08Q/08`0R6`0S7P0F6aHS
5Q/F6`005R<07P0F6`0F028R01h06aH08R808R0P0_dP30@Q0R0P120480@2123n
100Pn`@02?/400Sn1008o@@920@810P420@82?h400Sm16`P2B020R028@@1800R
02<07P0F6`007P0R01h06aHK028N01/06`0R02<N01h08Q/06aHK000K5Q/07P0R
02807Q/K5P0N8P0R01/08Q/05R<06ah06`0N01/08P0N02807P0K5Q/08Q/F:b/0
0@]Z:`0K7P008b806ah07P0K8aH07P0N6b806b80028001hK02807P0R01h05P0N
02<07Q/R01h001/08P0R000K01HK5Q/F01h08b8001HK02<05Q/R01h07R<08P0N
01/R02806b806ah080;m81448@@Q120P1204808112040PWk1002o@@02?h400Sl
10P810P420@810Sn1008n@AZ0P@P10448@8P0Q/N02807R<F01hS01/R01h07P0F
6`0N02806b807P0R01HK01h06`0N8aHS01h06b806ah05P0K8P0K01/R01HK5P0K
5Q/F02808P0R01hK8P0N02806aHS02807P0[:`012Gl[6`008Qh06`0F02807P0N
02808P0N02807P0R6aHK01HS01h08Q/F028K01/F7P005Q/0018K028K7P0R8P0R
6`0R02<N01h06ah06aH07P0R01/N02807P0R6`0R01/K02806ah001h20@8P124P
12448@@P120480@280@10P428?043@P420@810P420@810P8o@@320@48?h40B04
oR1T12448AH07Q/001h08P0N01h08`0R028K02808Q/F01hK028K02808P0R0280
7P0N000R6`0R01/F6`0F6`0R7P0K5R<07P0F6`0S7P0R6`0K5P007P0K6`0N01/F
6`0R028[:`014FX[5P0N6`0N01hR01/R01h08ah06ah001/R02<N02<05Q/07R<0
7P007Q/08P0K01hS8P0S5Q/F01h001HK6`0N01HK01h07P0R5P0R01/R01hS01h0
01/S01hK01HK6aH05Q/R000R7P010_dP4@@Q12448@@Q0P4P120480848?d4023l
1008oP@320@42?h42@P420@810P420Sm10488?d4Jb0P108P808Q120P6b8001hR
01/R01/R000N01h06aH07P0N01/R01H06aH07P0K5Q/R01h08P0K8Qh08P0K5R<0
7R808P006aHS01h08b8F01h06ah08aH06b808aH08P0R5P006ah06b/[004<Ob/K
5P008P0R8`0N01/002806aH08Q/F01hK5P0N028K01h07P0S7P008Q/08P0R01h0
7P0S028N028K01H08Q/08P0R02808`0N6`0R01h001hR028N01h0028K01H08ah0
01hK01/08P810R0P12448@@Q1042808P80@P120P0@@Po0@08?`400Sk10H810P4
20@8n`@02?`4KP8480@P104282048B04800N028S01h08P0N8ah08P0N02<N0280
7P0K028N02<N02806`0N6`0R6`0F6`0N6aH06`0R01/N028K01h07P0R01/08P0N
000R5Q/07P0R028K01h08ahF01/F02/[00405b/F02<F6aHK01/K8P0R8P0N02<F
6`0R7_h0FQhS01h08P0R028F01/R7P0F6aHK8P006b807P0K01/08P0K7P0F6b80
6a/R01hK01HK5R<R8P0N01/08P0R01h06b807P0S5Q/08P0F6`0282048@@Q1244
8@;m80X480@480@480@40_D400Sf1008n`@08?`4Jb0480@P808180@Q0Qh05Q/0
7P0K6b807P0R01hS01h001/R02807R<001h001/R028001HK01hS028001/R01HK
8P006ah07P0K5P0K5Q/R01/R01h06`0R01h07Q/08P0K5Q/08aH06b/[004<JB/K
5Q/002807P007Q/07Q/08P0R02806aHR6`0F8R<07Q/08Q/08P0K028K01h06ah0
5Q/F02808Q/F6`0F028K01h05P0K5P008Q/06`0K02808Q/07Q/08P0R01/F028F
6`0R6aHS01kk80H48@@Q1042oR0;120P10@10P@P2@@2n`@02?/41PP410P410Sm
1008oP@02?h42b04120480@20@810_dPG`0R6`0N028001h0028K6`0F8P0R8P0K
6`0R01hK5P0R8P0N6`0R7P008P0N01h08P0N02806aHS5P0R02808ah06`0N01HS
7P0R6aHK02<N000N6`0F01HS5Q/05R/[0045Ob/06aHS8P0N01hR000N000F6aHS
7P0S5P0K01h06`0N000F6`0K6b805Q/08P0R000N8aH06b807P0K028R01/08P0F
6b<F01/R7P0N01h08P0K7P0R000F6aHK8P0K8P0K01HK5R<07P0P80@P80@Q1244
8@8P80@P80@4820210@P10@PoP@08>h400Sn10<810@Pn`@880@280810R02oR1O
11/08P0R01/R01hS7P008P0N01/K01h05Q/R000K8P0K7P006ah08aHK7P0R01hK
8P0N02<F01h07R<07P0N01h05R<R01h001h06`0R5P0K5Q/08R806aHK01HK:b/0
0@YY:a/001hK000R01/N028S7P007P0K7P0S7P0R02807P0R7P0F6`0F6aHK0280
7Q/F6`0F6`0R6aH08P0R6`0N028K02807P0R01/08P0R01hK01H08Q/N02<07P0N
01HK7P0R6`0K5Q/08_/P5@@Q12448@@Q80@P120480@P1202107k1002o0@620@4
20@42?T400Sn10hP2@8480@P0@8P0R420@;n85h08Q/07Q/07P007P0R6`0N0280
7P008Q/07P0R01hK000R7P008P0S000R6`007P0R6a/F6`008P0N6`0R02808Q/0
6`008R807P0R6`0R02808P0K8P0K028K02/[004?Jb/F01h0028F6b<F02807P0K
8P0K5P007P007P0K7P0F6`0K8aH07P0S01h06a/08QH06b807P006a/R01/R01HK
01hK8P0F6b<F01h06aHK01HK8ah001HK8P0N02<R000F6`0N01h08P0P1?hP5`@Q
12448@@P80@Q12448@@480@P12040_h4023n100Po0@02?/400Sn1008n`A_80@P
12040R020@810R0P120K7P0R000N02<R000N01hS01/R01/R7P0N01hK8P0F8aHK
01hR01/N5Q/N02<R01HK000N02<F6b8001/R01/F6`0N028N02<N02807P0K8P0N
01/N02<F01h06b/[0047K2/K5R8K01h001h08Q/002807P0F8ah07R8K02806`0R
02807P0R01h08P0R01HK02<N01/F028N01HK01HK02807P006aHK01h08P0R02<F
6`0F01/F8a/07P008QHK028K01h08P0N6`048@Cl81H48@@Q120Q12448@@10P@P
120480@40_X400;l1008o0@02?X41R040P4480Cm80820@;m85l4000N6aHS01h0
7R<N01h05R<F6`0F8`0N02<05Q/06aHK02806aH002806aHK028K028N01h08`0N
6aH06aH08aHK01h06ah06`0K5P0R01/N028001h08R808P0[:`013fT[000K0280
7P0N01/07R807R<R01h08P0K5P0K8P0R01hK8P006b805Q/R01hS5P0N02808P0K
01hS01HK01hK8P006aHS028001/R01/F6`0F6b8001h07P0S5Q/K01hR028001/R
000NoB0G120P12448@@Q80@Q12448@84820480@Pn`@780@40PT410Sj10490_h4
M@8410890R0480@P120P12420R44808Q7R8001/F8P0R01h08P006aHK01HK0280
8P0F6`0R5P007P0S5P0K7P0R01h05Q/05Q/07P0K5R805P0S5P0K5Q/08P0R000N
02<F8`0N01h08`0N8P0N6`0R01/[:`013Wl[8Q/08Q/08b808Q/001hK000N6`0R
6`008Qh06aHK000N028N01/08P0K7P008ah08P0N8ah08P0N02808P0K028R01/F
6`0R7P0K7P005R8K01hR6`0R02808P0F6`0K5Q/F8`0N6`007P810R4480420@81
0R0P12448@@P80L4120480@48?/40R090_L400;n10`P2@84108P108P1204oR1V
124P12440@8P01h05Q/07Q/0028K5R<07P0R6`0N6`0N6`0F8a/08Qh08P0N6`00
7Q/0028K028K01h08P0K01h08Qh08P0K01h07Q/F8`0N01h08R<N01h08P0K5P0K
7P0R02/[004COb/K02806b807P0N028N000K8P0F01/08QHK01/F8`0R8P0N6`0R
01hK01h001/N000K7P006`0S8P0F6`0K8P0N01/R01h08P0K8P007R806`0R01/F
01/R01hK5Q/08P0S01/F8P007R8080810P4P0R0282020@@Q12448@@P1@@P1204
8?`4023n1002oP@08?/400;n10D22@84123n16hP12040P42820112448@8P8280
7R<06b8001hK02808P0F6`0F6`0R000R01h08ah001/N000N8Q/08QHK5P0N0280
01/R01HK8P0K01/R02807P008`0R01h06ah001/07Q/R01/R000N01/[:`013fl[
02807P0N01/F028K028N01/08R807R<05R<F01hK01/0028K02805R<08Qh0028R
01/F7P0N01/08Q/F01/F01h06aH08Q/N01/F8`0N028K02808ah06`005Q/08Q/F
02806a/F8`0N008P0R02oB0H0R0P0R448@@Q120480@P12040@8P80T40_T40P84
8?h4023n1002o0A`0@8480@P80810R028@@Q1244800R01h05Q/F8`0F6a/F6`0S
5Q/05R<N028K01HN000R8P0K8Q/06`0R6`008R<08Q/F8`0N6`008Qh08P0N6`0R
6aHR01hK01HS000B01HK000N01HK5R<08P0[:`013FX[8P0K7P0K8P0R01/05Q/0
8P0N01/F6`0F6`0N01/08P0R5Q/05Q/R01/F6`0N8P0K7P0S02807P0F6`0K8P0K
8P0K5R<07P007P0R02805Q/05Q/07P0N028F6b807P007QHS5P0N02807_TP2`81
80@Q12448@@Q1?`P1P@P0P@P10;m10<210@PoP@70PT40P@42B3m10@P12021?dP
JP810R420@@Q1244820K6b807R<08P0F6aH06`0R5Q/08P0N01/N01hS01/F6a/R
01h06aH07P0S5Q/N01h06aHR000N8P0K01/R000N8P006ah001hS01hF6`0N01/R
01/R02<07P0N:b/00AMo:a/F000R01hK028K5R<08Q/F028K000R6`0R02808Q/N
02<F6`0F6aH08aH06`0R000R5Q/08Q/F6`0N01h05Q/05Q/08Q/08P0N8aHK6`0F
8a/08P0N028K000K5P0R6`0K5Q/001hK000482048204820282448@@Q12448@@9
820480@P12048?h41R0910842@;n1002oP@<0P@4820480@P120P2OdPJ@8P1242
0@810R44800F6`007P0K7P008Q/07P0K028K000R000R01h05R<001h002808Qh0
7P0S000R6`0F6`0R6`0N6`0R01hK000N8Q/05R8K01h002807P0R01h05Q/F6aH0
7P0[:`013f`[01/S5Q/R000K8P007P0N000K7P0S5Q/F01h06ah06`0R01h06b80
01/F6b806ah07Q/07P0N02<08P0N01hS01hS5P0N02806b<F01/08P0N01HK8P00
6`0N8P0K8ah08P0K028R000N8`42n20E12448@@Q124482048@@P120480@48?h4
72041084120410420@@280@P120480@P10828204o21T0R020@@Q0QHK01/N028F
02<F6`0R02805Q/F01hR01hK8P0N8`0N8P0S5Q/F6`0R01/F8P0N01HK01/N0280
01HK7P007R806`0R01/07P0N8P0N8`0K8P0N8`0K01/R01/[:`015G8[6`0N01HK
5R<F6aHS028K5R<0028F6`0S7P0R000R01hK000R01hK5P005Q/F000R01h0028K
5P0R6`008P0N000N028S5Q/N000N6`0R6`0F8a/07P0R02806aH07P0R6aH08Q/K
5Q/07P0282048204oB0712448@@Q127n80X480@P120480T40_d4520480@P1204
80@P1204104480@P1?hPK0@1808P80@Q8202000N01/07P0R6`0S7P0R028K7P0R
6`0S7P0K5P007P007P007Q/F000R028K01HK01h08Q/08P0K5Q/F8`0F028K01h0
5R<07P0R02806a/05R806`007P0F8P0K01H[:`015G0[5R8001h08`0N000N01h0
6`0N8P0K5P0N02806aHK7P008QHK8P0F02<F6aHS01hK8P0N8b8001/N02<F6b80
7R<R01/07P006b805P0N02807P0F6b807Q/R01/R000N02<07P008P0R02<P0@@Q
1?`P=0@Q12448@@P80@P124480@P12040R04820480@P12040@@2120480@P8082
80@P12041242oR1W7P0R01h001h07P0R01HK01HK5P0K5Q/F000K5P0N02<F01/R
02<R028F6`008QhK5Q/07R806b806`0F6b808P007R806b808P0N8`0N8P0K7P0S
5P0F6`0N01hR02806b805Q/[:`015gl[6`0R8P0R01h08Q/08P0R02807Q/08ah0
8Q/F02806aHS01h06`0R8Q/002808P0K02807P0R7P008QHK01/001h08P0N028K
5P0N028K028K01h0028K000N01HK5R<07P0R6aHK028K5Q/P0R448@@P80@P0@8P
12448B048OhPE`@P120482040R04104280@P108280@P120480@480@40@@P120P
5P0N01H07P0R01hS02808Q/F028K01HS7P0K01hR6`0S7P0N02<N01hK5Q/06`0F
8Q_n03LS5Q/06aH06b808Q/07P0K5R<07Q/06aHK000N01/08P0K5Q/08Q/08P0R
6`0R6aH06aHS02/[004GKR/05Q/K01/N01hS01HK8P0K7P0K02805P0K01/F6b80
8P0N02807P0N01/R01hK8P0R01h06a/07R806`0N01/R5P0K5Q/07P007R<07P0K
7P0R02807P0K8P0F6aHS01h06ah08`0F6`0N01HQ1?dPO`81808P824482048@@P
80@Q12420@@480@P120480@P10811081108P120480@P80@06`0N01/R01/R01/0
6aHK01h06ah08aHK02806`0R01h08aH07P0R000N028001/08P0N8`0N01hS5Q/0
7P0K8QH05Q/07P0R01h08P008R806`0R8P0R01HK8P0K5P0N01HK5Q/06ah06b80
8P0K:b/00AM_:a/05P0R01/001h08P0K7P008P0R6ah08P0N02<07Q/08Q/07P0S
7P0N01/001h08Q/K01h05Q/06aH08P0R01h08P0S5Q/08Q/07P008P0K5Q/08Q/0
7P0N02<06aHN01h0028F6`0F6`0R6`@Q1?dP00;n87l4824480428@@1808Q1042
0R040@@P108480@P108P1081108110007P0N02<F6`0N01/F01/F02808aHN0280
01h08aHK02808Q/002806aH08Q/F8P0R6aHK01h08P0N028K01h002808P0K028K
02808Q/08ah07Qh08P0R6ad07Q/08P0N01HS7P0R6`0K01h002807Q/07P0[:`01
5F`[02805Q/N028R01hK01h001hK8P0N000K5Q/R028001HK01hS01h08`0R01hR
01/N000N01hS02808P0K5Q/R01/K8P0K5P0N028001hK8P0K01/N02808P0S7P0F
6`0S01hR01/07R<F01h0nb001?hPO`8P0@@P808Q12420R44820480@P0P@P120P
12020@@P100S01HS01/N028001h001h07P0K8P0K8P0K5Q/08P0K8P0N01h08P0N
6`0R5Q/N02807P006aHK01/07R806ah001/08P0R01h06b806ah05Q/R01h07P0N
8`006aHK7P0N01H07P0S8P0N000K01h08P0K5Q/F6`0R01/[:`016Gl[6`0N000S
7P0K5P0F8`0R7P007Q/08Qh06`0N6`0R7P008P0N000R7P0N000R6`008Qh08`0N
01hK02808`0N01H07P0F8ah08Q/F8`0F6aH08P0K5Q/07P0N028K01h08P0K7P0N
000R8`0N80@P80@1808P12420R4P12448Gl48@@1808P0P@1108P120480@480@0
6`0N01h07P007P0N8ah08Q/08P0R01h05Q/F01/08Q/F01h08Q/07P0R01h06`0F
01hS7P0R6`0K01h08P0K5P0R8ah08Q/07R<08P0N01H08Q/06`008P0N01HR8P0K
028K028S01h06aD08R<N02808Q/F02<07P0N0280:b/00AM]:`0F6b807P0R01/R
01h07P0K8P005Q/08P0R000K7P007Q/R01hS7P0R02<F6aH07Q/08P0N028001HK
7P0F6b807R<R01h001h001h07P0K8P0K5R<05Q/R01HK01h08P0K5P0N02<F6aHK
5P3m87l280420B020B048@@P0P420@828244808480@P100N000K01h07P0N01h0
6b808P0N000N02807P0R01/R01/R01HK01/R01/S5P0N02<N028F01/R01/001h0
8P0R01HK8P0K8P0N000N01HK01HK7P0S8P0K01/F01hK8P006b<06b805Q/07Q/0
7P0F6b807P0F6`0N01/R01h08P0R01/[:`015Fl[6`008R807R<F02<N02808P0N
8ah0028K5Q/F7P007R<05Q/001h08Q/K5Q/08ah05R<07P0R6ah08aH07P007P00
6`0R028S01hS02808P0N01H07P0S01h08aHK028K5P0S7P0R01/06`0R6b04oR06
0R0P0R010_/P1`810R4P1204oP1o7P0K01h07P0F6`0N02<08P0R6`0R6`007R<0
8Q/K02807P0K5P0R7P0F6`0N01H08ah08P0K01/08P0S7P0R8`0N6aHK02806aH0
6`0N8Q/08Q/08Q/0028F6ah08P0R8P0K028R6aH06aHS01h05R<07P008Q/08Q/0
8P0N01/:5P0R6a/07P0[:`015Vd[028R01/N000K01h001h06b807P0S5Q/07P0S
000N8`0F6`0R5Q/08aH08P007P0R01/F6b806`0R01/R01/R02<R01h06ah08P0N
01h06b806b8N02807P0R7P0F01/F8`0N000K7P0R02805_dP0P810_dP2`8P80@Q
120P0R427_h0OaHK7P0N01h06`0N8`0R000N01h06`0N6`0R8P0N01/F02806a/R
028F6`0N8`0R02806ah08P0K5P0N8P0K7P007P0R01HS02807P0K8P0K8P0F01/F
01h05Q/N01/08P0N01/R01HK7P0K8P0R01h06b807P0S5Q/05Q/05Q/R3@0R0280
6aH08P0K:b/00AUo:a/07Q/0028N02807R<08P0K000R7P0F8`0N028R01h08P0R
6`0F7P0K5Q/F8`0N02806`0N01HS5Qh06aH07Q/F6aH08P0K7P0R01h08P0K5P0K
028K5P0S01/F8ah06aHN028N000R6aHK02820@8P80810R028042820F01io000N
000N7Q/05P0R01h08P0R01h08R<N028S7P0N000N8`0R6aH06aHK5P007P0K000R
01h07Q/08P008Q/N01/F02<N000R7P0N6`0K5Q/08P0R01h05R<K02808ah08Q/0
02808Q/K01h06aHK000N01/N01h08P0K7P0R6`0E5R<K028K01/N028K7P0K5Q/F
02/[004GKb/08P007R806`0K8P0N01/N028F6`0N8`0R01HK7P0K8P0N6`0F6`0K
5R<001hR000K7P0R01HK5Q/08P007R<001H06a/R000N8`0K8P0N01/R000N01/S
5R806aH06aH001/R000K7P0K01hS0@;m87l20@810R0N01/07P0K8P0N8P0K0280
7P0S02807P0R01h07P007P007P0K5Q/07Q/06b807P0K8aHK8P007R<N02805P0K
7R807P005R<07P0R01hS01/08P0N02806a/R02807P0F6aHK5P0K01/F6aHK5P0R
01h06`0K8P0R02408P0R02<F000N02806aH07P0K5P0K7P008P0K5P0K:b/00AUo
:a/K5Q/07P0F8P0K01HR01/F6`0F6`0N6`0R6`008P0K000F8a/08P0K5Q/0028N
6`008Q/08`0N028K5R<07QHS6aH05Q/F8`0N01HK6`0R01hS7P0N01h06aH08ah0
8ah05Q/F8P008P0N00028204800K01/05P008P0R6`1o5Q/001h05R<07P0N6`0K
7P0R6`0S7P0R6`0N000R02807P007P0K5P0R01h0028R8`0F8a/08Q/F000K7P0R
8Q/08P0N6aH07P0F8aHK028K7P0F6a/08P0R6`0K028N01h08`0N01/F6`008QHK
01h07Q/07Q/07P0R8ah07P0E8P0N000R01/F000N6`0R02<N02/[004IOb/F028F
01/R01/F6b806b806`0N8`0R01H06aH07Q/R01hR01h08P0N01h08R806`0R01h0
6ahR000K02807P0S01HS01h08`0N02<R000F6b807P0S01h07P0R01h001h08b80
01/S5Q/F6b80100N000N02808P0S8P0K7P00Oah08P0K8P0F6a/R01H07P006`0R
01/07P0R01/R8P0K8P007R805R<06b8001hK01h06aH05Q/06b807P0R01/05Q/N
000K8`0N8P0K5P0K5P006aH06ah06`0R01HK02808P0N01h07P0S8QHK02806b80
5Q/08P0N01h08`0N5A/R028R01hF8`0K8QH07P0N01/[:`016Gl[6`0K02807P00
8P0N6`0N01HS01h07P0R7P0S7P007P0S6aH08Q/F02808Q/07P0R6ah07P007P0R
6aHK5P0R01h07R808P0N01h06aHR6`0K01HK5P0R6`0R6`0N02807P0K5R807P00
8P0K7P0N01h06aHK5Q/N01h05P0R8WlS5Q/F02<N01H08ah08P0R7P0R01h08Q/F
6`0F6`0N01/F8`0K5Q/08P0N6aH001h08P0K5Q/08P0K8P0S01h08Q/0028N01h0
02<N02<N01/F8P0K5P008Q/F6aH001hK01h08Q/08P0N01/05R8K01HS7P008Q/0
8Q/07P0N01D07Q/06ah001hR01/08P0R0280:b/00AUo:`0R5P0K8P0R7Q/R000N
01/R01h08P0F6`0K5P0K8P0N01H06ah08P0K01h06b807P008P0K8P0F6aHS000N
01hR01/08P0K7P0K8QH06`0N01hS02806`0N01/R01HK8P0N5R<07P0K8P0K5P0K
02808aHS028001h07Q/R01]o01/F8`0N01hS01h06a/R01/06ah05Q/06`0N8`0N
02806aHK5R<05Q/R01H08aHK8P0K5R<05Q/R01h06ah05Q/07R<N02808P0N01/F
01/F01/R01/F8ah06`0S8QhS02805Q/001/R02808P0K01hR000K8Qh05Q/07P0K
8P0E7P008aH06b806b808P0K6b806b/[004IOb/K02<N01/06`007R<F02806aH0
8Q/K01HS01h05Q/08ah08P0K01hS5Q/07P0K5P0R6`0R01hK01H07Q/08ah06aH0
7R<F000R01/08P0R01h07P0K5P0R02806aHS01/002807P0R01hK01/F028K5Q/0
7P0K5R<08P007Q/0Oah06aHK01h07P008P0N02808P0K5R<07P0R01h08ah08P0K
01HK5Q/07P0S7P0S01hK01h08Q/07P008P0N02<F6`0N028K028N01h07P0F8Q/0
6aH08aH08P0N01/05R<K5R<08Qh06aHK5Q/05R<06ah05R<08Q/07P0R01/F5B8K
5Q/08P0K5P0K5Q/F000N6`0[:`016Gl[5P0N02808P0K8P0N01h08aH06ah001hS
02807R<05Q/F01/R02805Q/07P0S5P0K8P0F6aHK02807P008QH06aH06b<F02<F
6aHK8P0N6b808P0R02806b806aHS01h08Qh07R<06ah001HF8`0N01/08P0N0280
8P0K8P007Wl06aH001h08`0R7Q/R000K5Q/N02808P0F6b8001h001/R01HK5R<0
5Q/07P0N5R80028001h08P007Q/R02<F6`0R000K01hK02<08P0R01/07P0K8P0K
7P0R01/F01h05R<05Q/07P0N02<05Q/F8P007R<05Q/08P0R02<F01DK01h08P0S
5P0K5R<001hR000N:b/00AUo:a/F028K7P0R01h001h07P0R6`008Qh07P0N000R
6`0F8ah06aHK01/08P0R5Q/08Q/08P0S028K6`0R7P0S7P0K5P0N01h08`0N01/F
000K7P0R6ah08P0K7P0K5Q/08Q/08P0N01HR6`0R6`0R01h07P0R6`0N6`0R6`0R
7P1o8P0S7P0R6aH06`0F6aHR02<05R<07Q/08P0N6b808Qh06aH002808Q/08P0R
8`0N6aHK5R<07P0R8P0K01h07P0N028N01h0028F6`0R6aH08P0N01h05P0N6aH0
8ah07P0N6`0F8`0R02807P008aHN000N6`0F6`0R6ah06aHE02808Q/N01h05Q/0
8R<07Q/002/[004IOb/08`0N000N01hS01hS8P0F6`0K8Q/08P0R02<F6aHS8P00
7P0N02808P0N02<05Q/05Q/F6aHK5P0F6`0K7P0R01/R000K8P0N01/F01/R5P0S
01h001/K5P0N01/05Q/F02<N02806`0R6`0F6ah08P0F6`0R000K7P0F6b80Oah0
7P0F6`0K8P0K8P006aHK5Q/07Q/05Q/R000K5Q/07P0K8b806ah08P0K6`0N01HS
02808P0N01/R02808P0R01hK02806b806`0N6`0R01/R02806b806`0R01h001/R
000K8P0R01h06b8F6ah08b8R000K8P0K7P007P0K5B806ah001h07R<06ah08P0R
01h[:`016Gl[7P0R8aHK02807P007P0N8P0R01h08Q/K02808P0K5Q/F8`0R028K
7P0R6ah08Q/08Q/06`0N02<N01HS000R6aH07P0R01hK02808ah06`0R5Q/08Qh0
5Q/002808Q/06aH06aHK01HS01HS01H08Q/08P0R6ah05P0R6`0K07lR6`0R6`0N
01h05Q/R6`0K01h08P0F8a/07P0R02<F6`0N01h08P0K5Q/F01HK028K01hK7P0R
8aH06ah07Q/06`005R<08P0N6`0R01h06aH06aHK01HK8P0R6aH08R806aHN01/N
028S5P0K01H07P0K6aHN01/F000R6`0R01DK5P008Q/08P0N01H07Q/06aH0:b/0
0AUo:`007P0K02807P0S7P0R01/06b805Q/001h07Q/N01/001h06b807P006aH0
01h05Q/07P0R01/F01hS01hF6`0K8P0N6b805P0N01h08P0F6a/05Q/05R<06b8R
01h06aH06aHS02806aHK01hK8P0K8P0K7P007R806`0R01Io6`0F6`0R01h07R<0
6`0R02808P0K8P0F01/R01/F6`0R01/F01/N028001hS01HK01h001/08aH06ah0
01/08QH07R807Q/N000F6b806b806aHS01hR01/07P0S01h06aHS028001hK01/0
7P0K8P0N000N02<F01hS7P0R01/E5R<07Q/F01/R01hS5Q/08P0K8b/[004H6R/K
5P0R01hK000R6`0R6`0F8P0K02808Qh07_h0Ob8F8ah08P0K7P0R7P0S7P0K5R80
8P0R6`0F8a/07P008P0N01/001h08Q/08Q/08Q/F6`0F8a/08Q/07P0K02808P0R
6`0N028K5P0R01h001/N01/F01/F8`0K7P0R6aHS028K02808P0S01h05R8K5Q/0
7Q/06aHK5P0N01HKM`0R6aH08ah0028K5R<07P0S5Q/08Qh07P0K7P008Qh08Q/0
8Q/07P008Q/F6`0K5P0K5P0K5P0K02808P0N01h08P0K5Q/08Q/05Qh08P0R01/K
028N02806aH07P0R6aH06aHS01HS7P0K000N01HS5Qh07R/[004IOb/08P0K7P00
5Q/N02806b806aHK8P0N6`0R01/R01h07P006aHK02806`0N02808`0R01/N0280
7P0N02<F6b808P0R8P0N8`0F6aH07P0K02806aH07P0R02807P0K7P0N028001h0
6aHK8P0K8P007P0K8P0K5R<F000N000NO`0N01h07Q/R01h08P0K01/08P007P0N
01/R01hS5P0N02807P0R01h08`0N01hF6`0F6`0F6`0N000N8`0F6`0F6`0R01/F
6aHS01h06b806b806b8F01/F6b807R<06b806aH07P0K8P006aHK8aH05Q/08P0R
01/R000N01/S5AH05Q/R000N01hR01/R01/07P0[:`016Gl[6`0R02<N02806aHK
01HK01HK01h001h06aH08Q/08P0R7P0K01h08P0R01hK5Q/K01h0028K000R0280
8P0K7P0R6`0R01h08`0R6`0N028K5P0S7P0N028K5P0R01/F028K7P008P0S01h0
5Q/F8`0N01h05Q/08R8K028K07lR02808P0K01h08Q/F02808Q/N000R02806`00
7P0R028K5P0R6ah07P0N02806`0R6`0R6`0R028N01h08aHK01HS02806`007P0F
8aH08P0N000S6aH08aH08P0N01h07P0S7P0N01/F8P0S5P0R6`0F8`0N6aH08ah0
02807QD08Q/06aHK02806aH06aH08P0S:b/00AUo:`0K7P0N02806aHS01HK01hS
02806b808P0K8P0K8P0N6`0R01h06a/R01hS000F01h06b807P0N6b807Q/07P00
6`0N02<05Q/F6`0R02805R<07P0R01/N000N6b808P0K02807Q/R01h07R<08P0R
02807R<07P0K5P0K5P1o6b806a/R01HK01h06aHK8P0K028R01/K8P0R8P0R01/N
000K7P008P0R02808aH07P0F6`0F6ah08b808P0N01/R01h07R807P0R01h07Q/R
02<F7P0R01h06ah06b808P007P0R02<F01/07P0N6b807P0K7P0S01h001hK8P0E
5Q/07P0S5P0K8P0K8P007R<N02/[004IOb/N000N028K5P0K5Q/05R807P0N028K
01h05Q/05Q/001h06aH08P0F6a/05R8K02808P0N6`0R000N6`0F8`0R7P0N01hK
02806`0R6`0N6`0N000R6`0F028S01h08Q/N01h08P007Q/07P0N01/N028K000N
000R028N02<NO`0K5P0F6`0R02808P0K01hF028K02805Q/K01hK02806aHS000N
6`0R6a/07P0R6`0R6`0R6`007P0K6`0R02806ah08P0K5R8K02808`0F6aHK000R
6aH08P0K5P0K7P0R8P0R6aHK02807P0K000N02<N000R5R8K5R8001hK5@0F8`0N
028K01h05Q/F8`0F01h[:`016Gl[01hS8P0K5P0K8P005Q/S8P0R01HK02807R<0
5Q/08R808P0K8P0K8P007P0K8P0N01/F6`0R02<F6`0R01h07P0K8P0K01HK8QH0
7P0F6`0R01HK7P0R01/N01h06ah0028001/S5Q/08P0R000N000K01hR028F6ahS
000N07lN01/F6`0F6b806a/R01h06aHK01hK5Q/F028001h08aH06aHN8`0R01h0
8P0F6aH07P0F6`0N8P0K5P0R01/N01h08P0S5P0K01hK01h07P0K02807P0K8P0N
01/S5P006a/F6`0K01HK7P0R01hR000N01HK7P0K02<06b8001DK8P0R01HK01h0
8b8001h06b80:b/00AUo:a/07P0R02<N01HR8a/05Q/K028K028K000N8Q/08Q/0
7Q/F01/F01/F8`0R01/F02808aHK01h08P0R6`0K5P0R01/F028K01/08Q/08P0R
6`0R01/F6`0R02807P007R<K5R807P008Q/08R8K028N01h06`0S6`007Q/F029o
02808aHK01HK5P0F6aH08P0S5Q/002806aHK5R<07P0R6`0K01h07Q/07Q/08`0R
6`0R6`0N01/F028N02806aH08Q/K5Q/05R<05R<08P0R028K7P0N01/07P0F028S
7P006aH05R<05P0R6`007R<N028K000N01h08Q/08QhE01/N028K01h07P0K8Q/0
8P0K7R/[004IOb/07P0K7P0N01HK01/F01h001/N000N028R01/05Q/07P006b80
6b808P0F6ah08Qh06aHK02807P0K7P0R02806b808P0N01/F01h08P0K7P0R01/F
8`0N02806b806b805P0K8P0R01h07Q/06b808b808P0F7P0N8P008P0KOb806aHS
01HK01/F6`0S5Q/07P0F01h08P0K02808P0N6`0R02805Q/08P005Q/F6`0F6`0N
01h08aHK01h06aHS01/F01/06b808P0K5Q/N02806`0R000K8P0R01h07P0N8P0K
8P0K5Q/S6`0R8P0N000K01hR02<F01h06ah05B<F000N02808P0R01h05Q/N000[
:`016Gl[6`0N000R6`0R02807P0K5Qh0028N028K02808Q/07P0R6`0N01HK7P0R
6`008R<07P0K5Q/08Q/05P0N6aHK01h08Q/F02808ah08Q/F000N6aH06aH08ah0
7P0K5P0K5Q/07P0N6aH08P0N01HK7P0K02<N000R01hK5Q/F07lK5P0N028K0280
8aH07P008Q/08Q/07Q/F028K7P0R01HN028K02808Q/F8a/F01HS01HS02808P0K
028K01H07P0F6aH08P0K7P0R01h06ah07P0R6aHR01/F6b808P0K01h05Q/F0280
7P0N8`0R028N01h06aHK028K5P008QDF6aHS028K01hK02808Q/0028K:b/00AUo
:`0K8P0K8P007P0K7P0S5R<06b806`0K01/N01h06b807P0R01/R000K7P007Q/0
8P0S5P0F01h05Q/R000K02806ah001/R01h001h06b<F6`0R01HS01h07P0R01/R
02808P0N000S01/R01/R000N02806b806b805R<06b9o02<F6`0N02807P0K8P0K
8Q/F01h06b8001/N000N01hS01HK01hS01h001h06b807R807P0N01/N01HK0280
7P0R8`005Q/R01H06b808P005Q/07P0S01/F8`006a/R02807R<06b807P0R000N
01hK02808P0S5P0N000N8P0E6`0K5Q/N02805P0N01/05R<N02/[004IOb/K01/F
01hS7P0R000R5Q/07P0R7P0R01h0028K01H08R<F6`0F6aHS000R7P007Q/F6`0R
6aH08Q/06aHR028K01H07R<07Q/08R807P0N000R6ah07P0R6`0R6`0F6aHS7P0R
6aHS5Qh06`0F6aHK01h07P0K7P0K01h08P0NO`0N000R6`0N01HN01/F01/08ah0
8P0K5R<F028K02806`0R01h07P0K5R<07P0K000N8`0R02806aHR028K6`0R6aHS
7P007Q/08P0N6`0R6`0F8P0N02807P0R7P007Q/07P0F6`0N02<N028K02806aHK
01h07P0R6aHS01hK5@0R01/001hK028S01h08Q/0028[:`016Gl[01h06b807P0R
01/F6a/08P0R01/06b805Q/N02806ah06`0N8`0R02807P0K8QH06`0N8`0R01/0
6aHS01hK01HK8P0K8P0F01h08`0N01hS7P008P0R01H06`0N8P007P006`0K5Q/0
8aH07P0S02808P0S5P0N02807P0R07lR01/R7P0R01hS01h06b805Qh001/R0280
8P0K02807P0N6b808P0K8P0F6aH06b8R01h06ah07P0K8aHK01H07P007P0S8P0F
01/N01H07P0K8P0S01hK8P0F6`0K5Q/08P0K8P0R02<F01/N02806aHS02807P00
7P0S01HK01DR02807R8001h06b805Q/07QHK:b/00AUo:a/08P0K000R6aH08P0F
6`0R6aH08P0N02<F028K5P008P0N01h06ah07P0R01/08P0N01h08ah08P0N0280
5R<K01/F01hK028K6aH002807P0R6`0R6a/08P0R01/N01/F8P0R01/07P0R7P0R
01hK01hK5P0R028K01/N01io6`0F6`0R6`007P0R01HK5R808Qh06aHK7P0R6`0N
6`0N000N6`0R01/F01HS7P0K02808P0K5P0R000N028K8P0R6`0N01hS02806`0S
7P0R01/N01h001hK01HS5P008Q/05Q/F6`0N02806aHK01HK028K028S7P0R01h0
5R8E6a/07P0K8Q/08P0N8`0F6`0S02/[004HOb/05Q/R01HK7P0S5Q/06aH07P0K
5Q/N02<F8`0N02<07Q/R01h08aH07R<06b807Q/R02806ah05Q/R000K8P0N028F
01/R01HK01HK01hK8P006`0N6`0F01/F6aHK028001h06b807P0N8`006aHK0280
5P0K8ah06b8002805P0N8aH06`0R8P0N02<R000K01/07P0RoP1k7P0R000R02<F
6`0R01h06b<R000N01hK8P0K5R<07P0N8`0N000K7P0R01HK01h07R807P0K5R<F
01h08aHK02807P0S5Q/F01h08`0F6b807P0K01hS5P0N02807P0R01hK8P0K5P0R
02807P0F6aHK01hS01h07R/[004IOb/K01HK5R<002808Q/F02<N01h08`0F6aHK
02807P0R000N028K5Q/001h05Q/001HK01h05P0R6`0N01h06`008Q/08P0K5R80
8Q/08P007R<N02805R<K5P0S01/08Q/F8`0R01h002807Q/F02<F6`0R6aH07P0N
01h07Q/0Ob806`0R01hK028K01h08R808Q/0028K7P0R6`0N6aHK01h0028K5P0R
01h07P008P007Q/001hS02807P0S7P0F028K02808Q/001hR02806aHK01HS5Q/0
8Q/05R<F6`0S7P0N028K01/F02807P0K028K01hK02807P008ah05A/F6`0K5P0R
6`0K01h07P008P0[:`016Gl[5P0N02<07R807Q/08P0N02808P0F6`0K5P0N01/R
01/R01hS5P007R807R<08QHK02807R<F6`0R01HK8P0F6ah05Q/S5P0K01/08P0K
8P0N01HK8P0N01/S5R808P0N01h06aHK01hK8P008P0N000N8P007P0S02807R<0
5WlK8P0F6aHK01HK02807Q/06ah07R<N000N6`0R01HS02805Q/N02806ah08`0N
8P0K5Q/08P0N01HK7P007P0R01/K01HK5Q/08R806`0K8P0K01hS01/F01h06b80
6aH07P0R01HK01h08P0K8P0S5P0N01h001HK02<R01h06aDF01/R02806`0R0280
8P0R8P0K:b/00AUo:a/F02807P0K7P008Q/F028S6`0R6`0N01/F02807Q/07P00
7P0R6`0K5P0N01/08Q/08P0K01HS01h0028S6`008Q/07P0R02807Q/F01/0028K
01/F02807P0N6`0R028K5P0S5R<05Q/N02<N028K01/F8`0N028K000N02<>01h0
8aHK028K01h08P0NoP1o8P0N028K01HS01h07P0R6`0F028K5P007Q/08ah05R<0
7P0R01h0028S7P0R6aH05R<K01/07Q/08P0R01h05R<07P0F6`008P0K5P0S7P0R
6`0R02807R<N01h07P0R028K028K01h06aH08P0S7P0N02<N028K5Q/F6`0N6aH3
02/[004IOb/08`0K7P0S5P006ah001/K5P0K7P0R02<F01/K8P007P0S8P006`0R
01/R02805Q/05Q/R01/R01h06b807P0R5Q/07P0R01hK8P006b808QHK01h08P0K
8P0N000F6`0N02<07P0F6b8001HK5P0K5R<F01/R01HK028R01h0Ob807P0R000N
028001/R01/R7P0S5P0K01HK01hK8P0N6`0S8P0K01/R8P0F01h06b808P0N8P0K
8b806aH06`0K8P0N02808P0F01/F6aHK01h07P0N8`0R5Q/R01/F6aH06`0N8aHK
5Q/001h08P006aHK02<N02807P0S5Q/03Qh07P0K7P0F6`0N02<07_h00a/[:`01
6Gl[7P0R01h07P0R8P0K5R<F01/F000R6aH08Q/05Q/F8`0R5Q/N028N02806ah0
8Q/08Q/06aH06ah08P0N6`0N02<F6`0R6`007P0R7P0K5Q/05R<08Q/N01/F028R
6`0R8aHK5P0R6`0K5R<K01h05P0N02807Q/05Q/07P0S7Wl08Q/N028N028K7P00
7P0R6`0R5Q/F028K01h001/001HR01/N01h06a/F8Q/07P0K7P0R8`0N01h07P0S
7P0N01/F028K5Q/08Qh06`0K02808P0N01h08a/07P0F8`0K5P0R000K01/08Q/0
8Q/N8P0S5Qh0028K5P0R028K5QD08P0R000R6`0N01h08Q/F8Q/0:b/00AUo:`0F
6b808P0K7P0S5P0N01h06b807P0K5Q/05Q/08P0F6a/001hK02806b8001/F01h0
6aH07R8001/F6`0R01/F6`0F6`0N8P0K7P0K5R<08P0N01h001h08P0K6`0F6`0N
02<F6`0K5P0N02805Q/R01/K8P005Q/07R<07P1o7Q/002806`0K000N8P0F6`0R
01/08b8N02<F01/R01hR01/N02808P0F01/07P0S5P006aH06b807P007P0R0280
8P0K5R<05Q/08P0R01hR01/R028001h07P0N8`0N01/R01/N028F01h08P0N000R
01/05Q/N01/R01hK01/E5Q/06b807P0N02<F01/F01/07R/[004IOb/K01HK028K
000N028K02808P0N6`0N02<F8Q/08Q/N01H08Q/05R<07P0N01808ah07P0S7P0K
5R808`0N02806`0R6`0N01/F000R01/F6`0R01h08Q/08Q/F01HS6`0N01h06`0N
02<N028K02<07P0F01hS7P0F8`0N000RO`0N028K5P0R028K02<N01HK01h07P0K
028F6aH07R806aH0028K01h08ah08P0R028N6`0K5P0K5P0R8P0R6a/08Q/N01/F
6`0F8`0N02806ah06`0R7Q/07P0R01h05R806aH06aHK028K01h001hS01h08Q/0
02806ah002805B<F6`0N8`0R02808ah08Q/08P0[:`016Gl[01HK01hK028N01HK
5P0N01/F6`0R000N01/F01h002806b805Q/07Q/R01HK01h08P0K5P0N02<07P0R
5P0K8QH07P0R028001hR01/F6`0N01/R01/05Q/06b807P0N02808P0F8P0N01HK
01HK8P0F6b805P0N8`0N028F6glR01HK01/R01hK5P0N01hS01HK8P0N01h06`0K
8P0S8P0S01hK02808Qh001/R01hK000N01/R01/R6`0N6`0F01h001h07P0S7P0R
01hK8P007P0N6`008P006b808P0K8P0K8P0R000N01HK01hS01h05Q/07R806aH0
6b806aDF6`0F6`0F6b806ah05Q/08P0K:b/00AUo:a/05R<05Q/08R<K02<N0280
6`0R6aHR02808ah08Q/08Q/08Q/07P007P0S02808Q/05R<K01h08Q/N02<N01/0
8P0R6`0R7P0S6aH0028N02806aH08Q/07P0K01HS01h08Q/08Q/0028K028K01h0
8`0N6`0R01h0028K01]o028K01h06aH0028S01h07P0S01h002808P0R01/F01/N
02807P0K6`008R806ah0028N02807P007P0R01/F6aH08Q/07P0R01/N02806`0R
6`0R01h08Q/F8P0K7P0R01/F01h08ahR028K02807P0R6`0F8`0N6`0N01HK5P0E
6aHR01HS6`0N01H08Q/08Q/F02/[004H>B/08P0N01h08P0K5P0N000K8P0F6aHS
01/N01h001/07Q/F01h08P0K8P0F6b807Q/08P0N02807R;n07lN02805Q/F6`0N
6`0N01H06b8R000K8aH06ah08P0R02807P0N01/07Q/08QHK5P0N02805Q/R000N
02<07R8K02807Q/08P0R000R7Q/08P0R02807P0N8b806ah08aH06aH05Q/R02<F
01hR01/K5P007Q/08P0N028R000K8UP08P006a/08P0R01/F01/K8P0F6aH06b80
8Q/F01/F6`0R01/F01hS01h06`0K01HK8P0F6`0F6`0R01H08P0N01/S5P0S8P0N
01h07R<F6`0F6aH06b/[004IOb/K02808P0R6`0F6aH08R807P0R01/F6`008Q/F
8P0R000R6`0R02806a/05Q/001h08Q/07P0R01/N028K01h08aHK01h0028K028K
01h08R807P0R000R6ah08Q/002808Q/001HS02808ah08Q/08aHK028K5Q/001h0
8Q/0O`0F8Q/08QhK000N6`0R6`0N028K01h08P0N6aHK01HS6`0F6aHK02807Q/0
5Q/F8`0F8`0N02<07R<N01/F6aHS5P0R6`0N6`0F6`007P0R028N01/07P0S7P0K
01h07P0S7P0F8P0R028K5Q/06aHS028K028K6`0R6`0R6`0N5@0N01/002807P0K
01HS01HK01H[:`016Gl[01HK8P0N6`0F6`0R5Q/K01/R01/F01H07R8001/R01/F
6aH06ah08aH07R<07R807Q/08P006b8001/N02808aHK028001hS01HK01HK02<N
01/R01hS7P005Q/07R806ah08QHK5Q/N01/F01/08P0K5P0N01/08R805Q/07WlR
01/07Q/001/R000N6`0R01HK01h06a/R000K02807P0F6`0K01HK8P007R<08P0F
6`0N6b807P0N01h08P007P0K7P0R000N8P0K5Q/R01/K02808P007P0R01h07R<0
7P006`0K7P0N01/08aH06aHK5P0N01H07P0R000N01DR01h06ahF8`0N01hS01hS
01HK:b/00AUo:a/001hK000R6`0R8`0F02807Q/08ah08R<07Q/07P0F8`0N01HK
7P0N000N000R6`0F8aHN8`0N6aH0028K5Q/05R<N02807P0R028K01h06aH06aH0
7P0R6`0F8`0N6`008Q/001h002808ah07Q/F02<N02808P0K6`0F8`1o7P0F8`0F
8Q/07P0R01h07R<K01h07P0F6aHN028K000R6`0R028K01/F6`0N02<N01HK000N
6`0R01h08Q/08P0R01/F6`0R01/F02<07P0F028K7P0R8`0N02808P0N028N028K
000R6aH08QHK01HS028K028K5P0N6aHS01hE02808P0K01h08P007P0F6aHR02/[
004IOb/08R805P0K7P0F6`0R01hK8P0F01h05Q/07P007P0N8`0R01hS000R01HK
8P0N6`0R01h06`0R000R5Q/N01/08P0N01hR01/R01/N01h07P0K8P0K8P007P0K
8P0N000R5Q/07R808aHK5Q/F01h001HK5P0K5Q/N02807P0RO`0R6aHR01/08P0F
6b808P0N02808P0F6`0S01HK028F6aH07P0N01h08P0R000N01/R01/F6`0R01/R
01/05Q/F6b806`0N6b806b805Q/R01/N000K7P0N01hK01/R01HK028K01HK7P0K
8P0K01hS01HK5P0N02<06`0R02805B806a/R02808P0N8aH07P0S01/[:`016Gl[
6`0N8`0R01/F6`0N02806`0R6aH08Q/07P0R6`0R01hK02807P0R6`0R01/R01HS
01h08P0N6`0R8`0F02807P0R01/06aH08Q/002808P0N01/F02<N028K01/F018K
02<F6`0K5Q/05Q/08ah08R<K01/F02<07P0N8`0N6gl06`0K028K5Q/08`0N6`0S
01h08Q/08aHN028K028K01HS7P0R02808Q/N028R02806aH08aHK02806aH08Q/0
6`0N01HS01HK01HK5Q/07P0R028K000R028001h06`0R028K028K000R01h05R<0
7P0R02<N028N028K01hK01DN6`0F6aHK6`0R6`0R8`0N0280:b/00AUo:`0R01/R
01hF8`0F6`0S7P0N6`0R01/06b806aH06b805P0K8P0N6`0F6b<F01HK01hS01/N
000N6`0N8`0K8P0K6b808P0K8P0K8P0N01/R02808P0N01HK01h08aHK01/F6`0R
01/07R<F01h08P0N01h06b808P0N000F6`1o8P0R5P0N01h06ahR000N02805Q/0
6aHK01HK01h06b807P006b806ah001/N01/K8P0R01HK02807P0S5Q/08P0N01/R
01HK01HK5R<08P0F6`0N01/F6ah08aHK8P0N8P0K5P0N01/F6b807R807P0N6`0N
000K01HK01HK028E000F6`0R000F6`0F8aHR01h06b/[004IOb/K01h06aH08aHK
01HN000R01/F6b808P0K5P0R7P0K028N01/0028N000N028K01h05R806aHK01HK
01h05Q/F000N6`0R01hK01h08ah06aHK01hS028K02808P0S5Q/05P0R6aH08P0K
02808Q/002808P0N6`0R6aHS6`0NO`0R6`0R6`008P007Q/07Q/08Q/07P005R<K
01h08P0K028K5P0K5P0K5Q/002807P0R6ah05R<F6`0N02<F6`0R02806ah0028K
01/F6`0R6`0R02808`0K7P0S01h06`0R01/F02808P0K01/08P0R01HS028R028K
028K01hK5AHS01HK028N000R6`0K028K5P0[:`016Gl[02807P0K8aHK02806b80
6b806`0K8P0N02<F6`0R01HK01h08QHK02<R01hK02808P0K8P0R01HK01h07R80
01hR000F6b80028001h06aHS01HK01HK01HK8P0N7P007R<07P0K8P0S5P0K5Q/0
7R807Q/R01H07P007P0N07lN6`0N6`0R5Q/R5Q/08P005Q/07P0K8R805R<F01/R
01HK01/R01/R02806aHK7P0N01/08P0F6`0R01/F6`0F6ah08aH06b8N02806`0F
6`0R01/K8P0N5P0N01h08P0N6b806b807R<N028F01h07P0K5Q/N01/K5P0N0280
5QDS01hR01h08`0N6`0N01HK5P0K:b/00AUo:a/08P0R01h05R<08P0N6`0N01HK
01h07P0N000N6`0R02808Q/05Q/F6`008Q/08P0N01/F6`0R02808`0K5R<07R<N
000N6aHK5R808P0K5Q/05R8K028K01/F000N5P0N000R01hK5P0R02<F6`0K5P00
7Q/08Q/08P0R029o000R01HK02<07P008Q/F8Q/08P0R01/08P0N02807Q/05R80
6aH06aHK5P0S000R01h08Q/K01HR6aH06aHK01H07P0R6`0K02808P0R6`0R6aH0
5Q/F02<N02808Q/001HK5P0K000N028K028S7P0R01/002805R<N01h08Q/E01h0
6aH07P0R01h06aHK000R02/[004IOb/05Q/N02805Q/07Q/R000N01/R02808P0K
8P0N8`0R01/K8P0K01h06`0R5Q/05Q/N01/F8`0N8`0N01/F6b808P0N01/B000S
5P0K8P0N01/05Q/S000N01/F01/R01/R000N6b805P0N01/F6`0N01/S7P008Qh0
8P0K6b80Oa/R01/R01/F8P007R<F01/F01h07P0N01/N01/K8P007R<07P0K8P0K
01/R01hR01hS01h001HK01h08aH08P0K8P0F6`0N02807Q/F6aH07P0R01h08`0N
02<07Q/08QHK01/R01hR000K01/N000R02<F01hK8P0N02<05Q/05Ah08P0K8P0N
6b808P0K8P0N01h[:`016Gl[6`008Q/08Q/07P007Q/F02806aHK01h05R<N01HK
02807P0R02808P0R6`0F8a/002806aH05Q/F02806`0K7P0K01h06aHN02<N01/F
02808Q/07QHS01h08ah07P0R6aHR01/08Q/08Q/07P0N02807P0R6`008Q/F01HK
7Wl07Q/06aH06a/F8P0N02808ah08P0R02806aH05Q/F8`0N000R01h05R<07P00
7Q/05R<08Qh002808P0R6`0R01/R6`0N028K000N000R6`0N02808P0N01h08P0K
5Q/05R<07P0S5Qh05R<F01hK01h08Qh001hK01h08Q/07QD08Q/05Q/001h08Q/F
01/F8Q/0:b/00AUo:`0R5Q/07Q/08P0N8P006b807QHS01h08b8001hS01h07P0N
01hK8P0K7P0R01/F01hF8aH07R<08P0K8P0N5P0R02807P0R000N01/F01/R01h0
7P0K5R<F01h08P0R6`006b805Q/07Q/08P0R000K8P007P0K8P0N01/R001o8P00
7P0K8P0F01/R02<K01h08P0K6b806aHS01h08`0N028F6b807R807P0R5Q/08P0N
01/R01hK8P0N6`0F6b806`0R01HK01hR02<F6`0R01/N01/N02807P0R01/F01/F
8P007P0S8P0N01/R01h05Q/06b8002805Q/08P0E01/07R<08R808P0N02<F01/0
6b/[004IOb/K02<F000N01h001hS7P0K000N028K01h07P007P0S7P0S01h001hK
000N6aH08ah06aH08Q/07P0N01h002<N028K000R6aHS01h08ah06aH08P0R01/F
6aH08Q/06`0R7P0K5Q/08P0F8Q/07R8K01h08P0R6`0R02806ah0Oa/F8`0N01h0
8ah06aH07P007Q/05P0N6`0F6aH08P0N028K01HK000S7P0R6`0F8`0N02806aH0
01/N000R6`0N01HS028K01h06aHK01HR6`0F02806aHK02<N01h08ah06`0N8P0N
01/F02806`0S7P008P0N6aHK02807Q/F5B807P0N01/K01/K01h08ah08P0[:`01
6Gl[5P0N01hR02808b807P0N01/B000N02807P0F6b807P0R02806b8001hS5P0S
01h06aHS01/F01h06b808P0N01hK01hS7P007R8001h08P0R01/R01h06`0R01/0
8QH07P0K5R<05Q/R01/08P0K01HK8P0K7P0R01/N01h08Wl06aHR01/R01h07P0K
8P0N8b805Q/R000F6`0K8P0N6b806`0N02<F6aH06`0R01/F6aHK8P006b8N000K
7P0R01HK5Q/N02808P0R02806`0R01/K8P0R000N01h001h08P0R01/R01/F8`0K
8P0F7P007Q/R01HS02806b8001DK8P0R000N028F028001h08P0K:b/00AUo:a/F
028S01hK01h08`0N02806aHS028K02<N02807P0R6ah08P0N6aH07P0R5Q/07P0K
5Q/08ah08P0K7P0R02<05Q/07P0R6`0N02808Q/K02806ah08P0R6ah07P0S7P0R
01/F6`007Q/07P0R028K01h05P0R6ah002808Q]o5P0K02806aH08P0N01/05Q/0
8Q/07R<N01HS01h001HK01HR028K01HK5P0R6`0F6`0K01h08P0K028R01/F6`0R
01/0028K6`0N028K7P0N02807P0R6ah0028K5R<08Q/N02806aH06aHN01/F8`0R
8P0K028K01hK5P0K5R<E01hK028R028K028K7P008Q/F02/[004IOb/08`0K01hK
02806aHR000K8P0N02807P0N02<07R<07P006a/R000K8P0R01h08P0S5P0F01h0
01/R000N6`0N01h08P006aH06b806a/F01/N01h001h07P005R<07P006b806`0R
8P008QH06aHK01h06a/R01/08P0K5Q/0Oa/R01HK8P0R01/R02807R<05Q/08P0F
01/R01h08aHK01hS01/N01HK02<F6`0N8`0R02806ahS5P0N01/F8`0F6b807QHK
5P0R01/N000N01/K8P0K7P008b8N02<07P0K02807P0K8P0K02<F01/F6`0N01/N
028001/R01/F5B805P0K7P0K01/N000R5Q/F01/[:`0167l[7P0R01h0028K5P0K
6aHR01/F01hK000R028N000N01/F6`0F6aHS01/N028S5Q/F02<N02808R807Q/0
028N02808Q/F8`0N01HK5P0F6aH001h08R<08Q/08Q/08Q/F8P0K5P0R6`0N8`0S
7P0K5P0R6`0F6`0F6`0R6`0F6b807P0R01hK02806ah08P0N02<F6`0R6`0F6aH0
8Q/05R<07_h0Kb8K02808P0N01h07P0N01H07P0R02806aHK000N028S01/F6`0F
028K02805Q/05P0R7P0K01h08P0R01hK000R01h05R<F6`0F8`0R02806aHK5Q/0
7P0F6`0R6aH001h08P007R<K01HK01H[:`016Gl[01HK7P0R01h06b805R<06aHS
01/07R807Q/08ah08P0R01HK01/07QH06a/06`007Qh001/N02<N02807R<001/F
6`0K5R807R<08aHK01/S8P0R7P0N6`0F6aH07P0S01h08P0N02807P0N01/R01HK
01HS5P0N01/N01hS07lN01hS7P0F01/N000N02808QHK01HK5P0N02<07P0K8P0N
01hS7R805Q/F8ah08P0R01h08P0K8P0K7P0S5P0F8b805Q/07P0K01hR01/F01/F
6`0K8P0K01h06b807P0K5Q/08QHK5Q/R01HK01hS01HK7P0N01/08P0R01hS01DF
6`0K8b806aHS7P0F01hS01HK:b/00AUo:a/0028K028K01HK5P0N01HK02808Q/0
8P0N01h08Q/06aH08Qh08P0K5P0F7P0R6`008R806ah0028K5P0N02808aHK01/0
8P0N01h05R<F01hK000R01HS6`0K5P0R01h08Q/N01hS6`0N01h001hR028K02<N
028002807P1o028001/08Qh0028K01hS01/05R<K02<N02807P0R01/F02807P0K
5Q/06`007Q/08Q/08Q/05R<N000N6`0R6`0N02<07P0R02806b808ah06`0R01/N
02808P0K5P0N02<07P0S01h06aHK01h07P0S000N02808Q/F6`007P0E8P0R01h0
8P0K5P0R6`007P0R02/[004IOb/F8aHK01/N01HK01/R01HK01hR01h06a/R0280
01/08P0S8Qh001/R000N8`0F6`0N8ah08aH06aHK01/R01h06aHK01h07Q/R0280
01h06b805P0K6b807P0R5P0K6b806ah002<05P0N02807R806`0K5P0N01/R01h0
7P0ROb806ah05Q/07R<R01HK01h08P0K5P0N02807P0N02808P0K8P0N02<08P0F
6b805Q/05Q/08R8001/R7P0F6`0R01h08QH06aHK8P0K01h08P0F6b805P0S7P0R
02806b805Q/R01h06aHS02808P0F6aHN8`0N01h06`0R8P0N5B<N01h07P0S5P0N
6`0R8P0R01/[:`016Gl[01h05Q/0028K02806aHS02806ah07P007Q/08Qh08Q/F
01/08Qh06aHS01h08aH001h07P0K5P0S5Q/06aHR6`0F02808P007Q/08R807P0K
028S5P0K000R6`0R01HK01H08Qh08Q/0028K000R6`0R02<N01h06aHR6`0R6gl0
8P0K5R<08P0N028K01h08Q/F028K028K6`0R6a/08Q/F01/0028F6`0R6`0K5R<0
8Q/08Q/07P006`0R6`0N028K6`0S7P0K01h05R808Q/F6`0N02<N01/N028K5P0N
6`007P008P0K5Q/08Q/08P0K01HR01h08P0R01/001DN02808Q/N01/F000N6`0N
6`0F:b/00AUo:b806b808P0N01h08aH06aHK8P008P0K5Q/08P0K01h06b805Q/0
8aH07R8001h08b8F01/R02807P0F8aH06`0F6aHK8P0K8P007Q/06b807P0N01/R
01HK7P0K8P0N02806a/05Q/07R806b806b807P0N02807P0K8aH06`1o7Q/R0280
8P0N01/K028001h06aHK5P0K5P0F6`0F01h06b808Qh07P0K7P0R000N01h05Q/0
7P0S8QH07P0R01HK01H07P0N01h08P0K01h08`0N01/N028001HK01/R01H07R<0
7Q/R01/05Q/08P0N01hS01hS01h06ah07R8E02806ah002<F01/B000K8P0F6b/[
004IOb/K01HK5Q/R02807P0R6`0K01hK01h05R<07Q/F02807P0K5Q/08P0N000R
7R<07P0N02807Q/08P0R6`0N8P0R6`0K01hK01hK000R6`0K5P0R02807P0R000R
01hK028K5P0F8a/07P0K5P0N8`0N01/F028K000R01/08P0NO`007Q/07Q/00280
7P0R7R<07P0S01h05R<N000R6aH08P0K7P0R6`0N000R6aHS028K02807P0R01h0
8R<K028K028K5P0R02807Q/F028K5P0R028001hK5R<K01h06`0R8P0N000F6aH0
8Q/08Q/K02807P0K02808P0K5P0K5A/08P0K5Qh08ah06aHR01/F8`0[:`00284[
PB_/:`0000X0084[PB_]:`000:00P`3o\>"], "Graphics",
  ImageSize->{277, 295},
  ImageMargins->{{35, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}}]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
  "Using ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  " with ",
  StyleBox["MathLink",
    FontSlant->"Italic"]
}], "Subsubsection"],

Cell[TextData[{
  StyleBox["MathLive",
    FontSlant->"Italic"],
  " has been developed to handle the ",
  StyleBox["MathLink",
    FontSlant->"Italic"],
  " communications standard for direct communication to ",
  StyleBox["Mathematica.",
    FontSlant->"Italic"],
  "  By using ",
  StyleBox["MathLink",
    FontSlant->"Italic"],
  ", graphics may be sent directly from ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to ",
  StyleBox["MathLive.",
    FontSlant->"Italic"],
  " This method is faster and easier than using 3-Script files."
}], "Text"],

Cell[TextData[{
  "First we need to tell ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " where ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  " resides."
}], "Text"],

Cell[BoxData[
    \($LivePath\  = \ 
      "\<Red:Mathematica Files:Application Libraries:MathLive:MathLive\>"\)], 
  "Input"],

Cell[TextData[{
  "Then we can open the ",
  StyleBox["MathLink",
    FontSlant->"Italic"],
  " connection in one of two ways. The first is to have ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " launch ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  " for us."
}], "Text"],

Cell[BoxData[
    \(\(live\  = \ LinkOpen[$LivePath, \ LinkMode\  \[Rule] \ Launch]; \)\)], 
  "Input"],

Cell[TextData[{
  "Or we can connect to ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  " directly, if we have already launched it manually."
}], "Text"],

Cell[BoxData[
    \(\(live\  = \ LinkOpen["\<MathLive\>", \ LinkMode\  \[Rule] \ Connect]; 
    \)\)], "Input"],

Cell[TextData[{
  "Once the ",
  StyleBox["MathLink",
    FontSlant->"Italic"],
  " connection is open, we can send three-dimensional graphics to ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  " to be manipulated."
}], "Text"],

Cell[BoxData[
    \(\(lumpy = 
      Plot3D[Sin[x\^2]\ Sin[y\^2], {x, \(-\[Pi]\), \[Pi]}, 
        \ {y, \(-\[Pi]\), \[Pi]}, \n\t\tPlotPoints \[Rule] 30, \ 
        Boxed\  \[Rule] \ False, \ Axes\  \[Rule] \ False]; \)\)], "Input"],

Cell[BoxData[
    \(\(LinkWrite[live, \ lumpy]; \)\)], "Input"],

Cell[TextData[{
  "Now the graphic we called ",
  StyleBox["lumpy", "Input"],
  " has been sent to ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  "."
}], "Text"],

Cell[GraphicsData["PICT", "\<\
>B0000000BL15@0A0_l<0?ooool000000000004E00019`000000000N01[ooooo
ool00@0:0000004W0AD0P@0:000000000000X02201X000000000VH4F0000004W
0AD00000000000180000B0000000200100P000000>LNc000000001;_P000>P00
oooooooo006HV9RIc<`00YbLW9bLW0039bQ?CbLW00CoofEUc<T01Okn003;aP06
VI@00<c<00OBmB7od_402?kn003nn009oolc<lc:00XaZbbc;;<02ng]WIg]j`0<
00071aHF00d00001E5@03UaLG5aLG00?V9P007mm013<c000VIL04D_NI=9TdP0B
IOnGoVCn01<00B8R00005001F5P0000Eg=a8B=cI01Kooc<cooX05gAdM7AdM00H
Y1Je@Z@F01W;bl_<ool06/c<V9S<b`0Kc<^IVOoo01c<c4a<c<T07Lc7CDgoo`0N
l_9/K?;_01oooiVIood08:BT002TX00Q>SX003Xi028c<P00IVH08lPfeFK8=P0T
AdM7AdM702FQWP00o_h09/_8003noP0WQ8B4Q8B402P00Xj>0000:@03][H0000Z
e=CDe>;R02_M6oBem;D0;6ER002HV00]PX80089o02k^k];Bh>00;ooobl_ooP0`
5aLS8aLG034a<A4A8200<UYJ0010?`0cH600061N03CLg?3`g=`0=OKfj>SDe00f
c<bHV9RH03Nh^;Rh^;P0>;Bd002d/00ia/H00<K103X00000000000000BL15@00
00019`4E00002P0000019`4E00R1>X4jk3X0000=03Z16H4Il1T363Xj004IOcXI
=@0e;P0e03D^03D0=Bh0=@0e;P0e03D^03D0=Bh0=@0e;P0e03D^03D0=Bh0=@0e
;P0e03D^03D0=Bh0=@0e;P0e03D^03D0=Bh0=@0e;P0e03D^03D0=Bh0=@0e;P0e
03D^03D0=Bh0=@0e;P0e03D^03D0=Bh0=@0e;P0eO`0e;P0e03D^03D0=Bh0=@0e
;P0e03D^03D0=Bh0=@0e;P0e03D^03D0=Bh0=@0e;P0e03D^03D0=Bh0=@0e;P0e
03D^03D0=Bh0=@0e;P0e03D^03D0=Bh0=@0e;P0e03D^03D0=Bh0=@0e;P0e03D^
03D0=Bh0=@0e;P0e03D^5@0e03D^03D0=Bh0=@0e;P0e;QPj>P016Glj6@0[;R/0
:P0Z02/^:bh[02X0:P0Z=00Z02/^:bh[02X0:P0Z=00Z02/^:bh[02X0:P0Z=00Z
02/^:bh[02X0:P0Z=00Z02/^:bh[02X0:P0Z=00Z02/^:bh[02X0:P0Z=00Z02/^
:bh[02X0:P0Z=00Z02/^:bh[02X0:P0Z=7l0:P0[;R/^:`0Z02X0:S@0:P0[;R/^
:`0Z02X0:S@0:P0[;R/^:`0Z02X0:S@0:P0[;R/^:`0Z02X0:S@0:P0[;R/^:`0Z
02X0:S@0:P0[;R/^:`0Z02X0:S@0:P0[;R/^:`0Z02X0:S@0:P0[;R/^:`0Z02X0
:S@0:P0[;R/^:aD0:P0Z02Xd02X0:bh[;R/0:P0H>SX00?<2>QT>naL14Q7k5`8B
4BOe5`8B4BOh5gl0=3@0=00d02hZ03@^03@0=3D0=00d02hZ03@^03@0=3D0=00d
02hZ03@^03@0=3D0=00d02hZ03@^03@0=3D0=00d02hZ03@^03@0=3D0=00d02hZ
03@^03@0=3D0=00d02hZ03@^03@0=3D0=00d02hZ03@^03@0=3D0=00d03`^:P0d
;P0d03@e03@0=00^:P0d;P0d03@e03@0=00^:P0d;P0d03@e03@0=00^:P0d;P0d
03@e03@0=00An1L24Q4WmaL<03032P<:0`X:0`X:0?/G0aPj>P014WDj6@0^03@0
:P0e02h0=@00:P0e02h0;S@0:cD0=000:P0e02h0;S@0=00d02/^02hZ02hj:`0[
;P0d02hj;R/^:P0e:`0[;P0j>R/^03Xj;Rhd03@0>SX[;R/^03Xj:P0e:`0[;P0d
02X0;R/^:P0e:`00:P0^oSYo;R/^:P0e:`00;R/0:P0^:c@0;SXe:`00;R/0;SX0
;R/^:P0e:`0d03Xj:P0^>SX^:P0e:`0[;P0d02X0;R/^:P0e:`00;R/0:P0^:c@0
;Rhd03Xj:P0^>SX0;R/^:P0e:`0[;P0d02X0;R/^:bh^03D0:cD0=000:P0e02h0
;S@703@0:cD0=2Ch6@/[;S@0:P0^01Pj>P00hP4j6MXGOa4G5bh0=00^:`0j>Rh0
:cD[;SXj02h002X^03@^03@j>Rh0:SXj000Z;P0d>SX003@0:P0^02X0=BX0:cD0
:P0d02X002X^03@d03@j>P0^02X002X^03@d03DZ03@0;P0S03Xj;R/d03DZ>SX^
02X002X^03@^03Xj03@0>SX0?bhZ03DZ02/e02X0=00Z000Z;P0d=00e:P0d02h0
8`00:P0j>P0d02h0=00Z000Z;P0d;P0d;P0Z02/^02h0:a7_5`0AoQL<;PXZ63HH
=`XH60XI0?/G0aPj>P010c8j:SD0:P0e02X0;P0Z03@0:P0^02X002X0;P0Z02hd
02h0:P0Z000Z03D0=@0d03D[02kn>P<d02X0oSX2:P0ZoCX2;P0^oCX1=03l>QH[
02hj>P0Z=BX0>SXd03Xj02hj>S@0;_dj1R/^:`0Z02km>P80:P3m>P`d02hj>P0j
>RX0:P0^oSX302hd0?hj0P0Z0?dj0R/^0?dj0@0^o3X302X0;_dj0S@0;_Pj0bX0
02kn>P<<;S@0oCX6:P0j>P0d0?hj0`0[;P3m>SD0:SD0=00[;P0^02h0:P0^=00^
02X0:P00:P0e000^00XI0AP160XH=`XI:bhd02X0=@4j>P00e@4j6MLG1RX0:P0d
;P3m>P4^0?djA00d>P0^03Xj02/^>SX0;R/j>Rh0>SXZ03Xj;P00:P0j>Rh0>SX^
03Xj;R/j>P0^>SX0=BX0=00^>SX0=00j>Rh0>SX^0?hj1Rh003@^02_m>P40;_dj
D@00>Rh0;SXj03@^>SX0=00j>P00>SX0=3Xj02hj>Rh0>SX^03Xj;P0j>RX0>P0^
03Xj02X0>SX0;R/j>Rh0>SX^03X^03X^000j>P0Z02X0=B_Z5``00aTf63H12SH2
91T^naL3=SXj004:=3XI02h0:P0d02/0:P00;S@0=00[02hd03@0=00[02hd02/0
;P0d02h0:P0^02X0:bh0:SX0o3X503Xj02h0o3Y8:SD0>SXZ02hj>P0^>SX0;SXj
02Xd02hj>P0Z>SX0;SXj000j>S@0>SX0:P00;S@0>SXZ02hj>P0^>SX0;SXj;P0d
02X0:bhj0?`j1@0j>RXd0?`j92/^03Xj:P0^>SX[;SXj02hj>S@0>SX0:SXj02hj
>P0^>SX0=03l>Q<d02hj>S@0;SXj02hj>P0[;P0d0?`j83@0=00d:P0^02X0=00[
02hd02/0;P0d02h0;P0Z02h:6OdH3`XH0@<I:`0^000Z01Pj>P00hP<j6ALBoAL0
4OPG017X5`T0=BX003@0>Rh0oSX^;P0j>RX0>SX0;SXj02/^>SX0=00j>Rh0>SX^
03Xj;P0^:`0j>Rh0>SXZ03Xj:P3k>R0^03@Z02h[>SX0=00j>Rh0>SXZ03Xj02X0
:P0d;P0j;P3n>Q/^03Xj02hj>P00>SX^02Xj>P0d03Xj;P0j>Rh0ncXc;P0j>Rh0
>SX^03Xj;P0j>P0^>SX0=00j>Rh0:cXj:P0j>Rh0:cD0>SX0;SXj;P0Z02h0=?@G
01;h5`d[2QT:2R@:2QPH<BX04O`G0aPj>P016Glj6B/^02h0;P0^03@0:P0^02h0
=C@002h0:P0e02X0;P0^=00d02X0:P00:P0Z03@d02hj03@0>Rh0:SXj02hj>S@0
>SX[;P0j>S@0;SXj02Xj>P0^>SX[=B/0;SXj02hj>P0^>SX0=CXj;P0^03@0:P0^
:`0j>S@0;SXj02Xj>Wl0;SXj=00^=00^02hj:`0^>P00:SXj=00j>R/^>SX0:P0j
>S@0;SXj02hj>P0^>SX0;P0Z02hj>P0Z>SX0:SXj02hj>S@0>SX0;Rhj>P0d03Xj
02hj>P0[;P0Z>SXd03Xj02Xe=00d02X002X0;SD0:P0e02X0;P0Z02hd02/0;QDT
6AP161Pg63L:6@0^03@0;P0H>SX00=T1>QWG5gld03@_:P0[>Rh0=00[;P0j>Rh0
>SX^03Xj02/e>SX^:`0j>Rh0>SXZ03Xj02X0=00j>Rh0>SX^03X^02Xj>P0d03X0
=00d03DZ>SX^:`0j>Rh0>SX^03Xj;P0[;R/0=00j;R/0;R/^03Xj;P0j>Rh0>SX^
03@j>RhZ03Xj;R/j>S/Z03Xj;P0Z>P0[>SX^03Xj;P0j>RX0>SX^03Xj:P0[>SX^
02Xj>Rh0>SX^03X^03Xj;P0j>Rh002X^03CZ5`<02QTfoAP4=aP:6CCk5`<H>SX0
0@To>QTe000Z03@002X0=@0[02X0:P00;S@0:P0Z03@0:P0d000Z03D0:`0^=00^
03Dd02/^>P0[;R/^02hj>P0d0?`j3c@0:P0j>P0^>SX0;SXj02kl>PD0;SXj02[m
>P@^03@00?dj3BX0;RXe02X0>SX0;S@0oCXI;P0Z>SX0:SD0;S@0;SX0;S@0=00^
>SX0=03l>Q00:bh0>SX0;SXj:`0j>P0d0?dj3`0d03Xj02hj>P0^>SX002kl>P`0
:P0^>SXd03Xj03@0oSX302/0;_`j=2/0:P00:P0^=00e02/0:P00:P0Z02h0:P0[
02hd02h:6@<:0c8:<@X:6Bh0;P0Z02hH>SX00=P1>QWG5gl[;P0^03@0;R/^03D[
02/^03@0:P0^02h0;R/0=2h0=00^02X0;P0Z02h0;P0^:`0Z03@0;P00:P0d;RX0
;P0^000d02h[;P0d;P0Z02h[;P00:P0Z02h0:P0[;P0Z02/^02/^02X0:bh0=2h^
000Z02h[=@0Z02h[02h0=BX0;S/0:Rh002X0=2h^02h[02h0=00Z03@Z000^02h0
=00d02h0:Rh0;R/^02X0;R/e:P00;P0^03@^03@Z03CZ5`400o/I02[n6@404O`G
0aPj>P016Glj:RX0;S@0:P0^=00Z02h[=@00:P0Z03@002h0;S@002hd02h0:P0^
=00[02h0:P0[;S@0=000:P0Z=00d02h^=00d03@0=00^02/^=00d03@0=00d03@0
=00^=000;S@0:S@002h[03@0=00Z=00Z02h0:bh0:P0d02h0;S@0=7l0=00d02hd
03@0:SD0:SD0=00e=000:P00:S@0;R/e02X0=B/0=B/0:P0d03@002Xd000^:`0d
02hd02/^=000;P0^=00d:`0^=00Z03@003@0;P0d03@0;R/0;S@0=00[;P0[;P0d
02h0:P0d02X0;S@0=00Z03@[03D[02X0:P`0;P0^;P0Z000^02XeoP05:P0H>SX0
0AUo>QT0=00^:`0d02h[03@0=B/d;P0d02X0=3@0=00^:P0[;R/0=00d02h[=BX0
=00d;P0^:bh[=@0d=@0Z;R/002X^03@^:P0d03@^02X0:Rh0=2h[;R/^:bh[03@^
:P0^000d=2h0;R/^:bh002Xe02X0=2h0:P0d;R/0=00Z;P1o=2h0:P0[;P0Z000Z
02/^:bh[;P0d:P0d;P0^:`0^:bh0=2/^:bh0=00Z;P0d;R/0=BX^02h[;R/0:P0d
02X^:`0d02X^03@Z02h^03@^:P0Z02/d;P0^:`0e:P0Z02X0=00d;P0d;P0[;P0d
;P00=00^:cD^03@e02X0=BX0=2hE;P0Z000d=B/^03@0:P0d;P0Z=cXj002U03[k
6002oAP40QP2607h6002oAP00_hH00;n6002oAP00_HH007l6002oAP00OhH0P8H
0OhH00;n6001lQP00_LH00;d6002n1P80QPH0QPH0QP2kQP00_`H00;m6002laP0
0_`H0P8H0_hH03Kg6002oQP30QPH0_/H208H0QPH0QPH0_`H0`8260;k60<161P2
maP00OhH00;n6002o1P2>SX000Z1>[/j00bc>P000AUo>P0^02h0;P0^02h0;P0^
02h0;P0^02h0;P0^02h0;P0^02h0;P0^02h0;P0^02h0;P0^02h0;P0^02h002h0
;P0^02h0;P0^02h0;P0^02h0;P0^000^02h0;P0^02h0;P0^02h0;P0^02h0;P0^
02h0;P0^02h0;P0^02h0;P1o;P0^02h0;P0^02h0;P0^02h0;P0^02h0;P0^02h0
;P0^02h0;P0^02h0;P0^02h0;P0^02h0;P0^02h0;P0^02h0;P0^02h0;P0^02h0
;P0^02h0;P0^02h002h0;P0^02h0;P0^000^02h0;P0^02h0;P00;P0^02h0;P0^
02hE02h0;P0^000^02h0;P0^02h0;SXj004IOcX^02X0=00d03@0=00d03@0:P0Z
03@0:P0Z02X0:P0Z02X0=00Z02X0:P0Z02X0:P0Z02X0=00d02X0:P0[;P0Z03@0
:P0Z03@0:P0Z02X0=00Z03@^02X0:P0Z02X0:P0Z02X0:P0Z02X0:P0Z02X0:P0Z
02X0:P0Z02X0:P0ZO`0Z02X0:P0Z03@0:P0Z02X0:P0Z02X0:P0Z02X0:P0Z02X0
:P0Z02X0:P0Z02X0=00Z02X0:P0Z03@0:P0Z02X0:P0Z02X0=00Z02X0:P0Z02X0
:P0Z02X0=2X0:P0Z02X0:P0Z02X0:bX0:P0Z02X0:P0d:P0Z02X0:P0Z02X05BX0
=00d02X^02X0:P0Z02X0:P0j>P016Glj03@0:bh[;RX0;RX0;RX0=00d02X0=00d
03@0:P0d03@002hd03@0:P0d02X0=00d03@002hZ03@0=00^=00d02/^=00d000^
=00d03@002h[;P0Z02X0=00d03@0=00d03@0:P0d03@0=00d03@0=00d03@0=00Z
02X0=00d07lZ03@0=00d000^=00d03@0:P0d03@0=00d03@0=00d02X0:P0Z03@0
=00Z03@0=00Z03@0=00d02X0=00d03@0=00d02X0=00Z03@0:P0d03@0=00d03@0
:bh0:P0d03@0=00d03@0:bh0=00Z03@0=00[;P0d03@0=00Z03@0=1D0:bh[;RX0
02X0=00d03@0=00^>SX00AUo>RX0=2h0;P00=00^:`0^:P0d;P0^:P0d;P0Z03@^
03@^:P0[;P0Z03@^03@0=2h^03@^:c@0;RX0:Rh0:bh[;P0d02hZ03@Z02h^03@^
:c@0;P0d03@0=2h^03@0:Rh0:P0^:`0d;Rh0=2h^03@^:P0d;P0Z;P0d03D0=2h0
:P1o=2h0=2h^03@Z02h[;P0^03@^03@^;P0d;Rh0=2h0:P00=00e03@^02X0=2h0
=2h^03@^02X0=BX0:Rh0:Rh0=2h0:P0d;P0d;P0Z000d;Rh0=2h^03@^03@0:P0d
;P0Z;P0d;Rh0=2h0=2h003@^02X^02/^03@^02h0=2h0=2hE03D0:P00:S@0=2h^
03@^02X0:cXj004IOcX002h0:P0Z=00Z02hd000Z;P0e:`0^;P0d02/^02hZ000d
:P0^=00^03@0;RX002/^:`0^=000=00[02X0=@0[;RX002/^02h[02/^:`0^=00[
;S@0;RX002X0:Rh0=00Z03@0;P0[02h[02X0:P00;P0[;P00:P0^:bh0:P00Obh0
:Rh0:`0Z02h[=@0d02/^:`0^:`0[;R/0;RX0=00Z=C@0;R/^02X002h0:Rh0:`0Z
03@0;RX002h0=000:Rh0=00[;P0^:P0d03D[;R/0;RX002/^02X^=000;P0d000^
:P00:SD0:bh0:SDZ03@002Xe02X^03@0:bh0;RX05C@0:P0^=00^02h[02/^02X0
02hj>P016Glj=2X0=00d02hZ03@0:P0d000d:bh0:`0Z;P0e:`0[03@^000d02h[
03@^02/0=2X003@^02X^02Xe03@^:`0d;P00=3D^03@0;R/^02h[02h[;P0^:`00
=2X0=@0[;P0d;P0d=2h[;R/0=BX0=00d:P0d;P0d=3DZ02h[03@e:Wl0:P00=2h[
03@0;R/^03@^03DZ02h[;P0^:`00:Rh002X^:`0d03@^:c@0=00[=Bh[03@^:`00
=3@0=2hZ000[=@0Z000d02h[;R/^000d;P00=2X003@002Xe:P0Z;R/^000d:P0Z
03@0:P0Z03@^:P00:P00:P0Z;P0[02/0:QD^:`0d03@Z03@0;R/^03@0:S@0>SX0
0AUo>Rh002h[;P0Z000Z02hd02h[;Rh0=00^=000:P0Z=00^:cD[;S@0=B/^02/^
:P0e02X^03@002/^02hd000^:P0^:bhZ03@0:R/e03@0;S@0=00^:`0Z=BX002h[
;P0Z=@0Z02h0:P0e02X0:P0Z=@0[;P0d02X002X0;S@0:P1o=000;RX0=00^=00Z
02hZ02/^:`0[=@0^:`0Z=000:S@003D[;P0Z02he02h[;P0[=@0Z02he:bh^=00^
03@0:P0Z;SD[;S@0:P0^03@[;P0[;RX0=BX0;S@0:P0e:`0e02/^:P0d000^=00d
000^:`0e=00d=00d03@0;S@^=00E02hd02X^03@0;R/e03@0:SD0;SXj004IOcX0
=3@0=B/e03@d03@0:Rh0:P0[=B/Z02h[=00e02hZ02h0:P0Z02X0:cD^000Z03@0
:P0d:P0d03@^03@Z000d03@003@^000d03@^:`0^:bh0:cDZ02X0=2X0;P0d02X0
=BX0:P0d02X0=@0Z02X0=B/^02X0=2h0:P0^:`0dObh[=00^03@^02X003@003@^
02h[02X0:c@^02h[=@0Z;R/0=B/^02X0:P0[;P0d02X^02X0:P0[02hZ03@^:`0d
000Z02/^000d03@^02/d;P00=2/^:`0^:`0d02X0:P0d000Z;R/d02hZ03@^02h[
;P0d;P0Z;P0Z02/^02h[5CD0:P00:P0Z;P0Z03@^:`0Z02/j>P016Glj:bh^02X0
;R/^;P0^:bh0=000;R/^02h[=@0^:cD[03@0=000;S@0=00Z02hd02X^=00[;P0Z
=@0Z02hZ03@0;R/^:bhZ02Xd02X^:`0^:cD0=00^03@002h0=00[;S@002X0=000
;R/^=00d03@0;R/^02Xe02X0=00e:`0^:Wl002hd02/^02X0:S@0:SDZ03@0=2X0
=00e:`0^=00d02/0;R/^03@0:P0Z=@0[;RX002X0=00[;SD[000Z03D0;Rhd02/^
02Xd02hZ03@0;P0^:bhZ000^:cD0;RX0=00Z=@0Z=@00;P0Z000Z02X0;P0Z;P0[
;P0d02Xe03@0;QD[=00^=00d000d000Z02hd02/^>SX00AUo>Rh0:cD0:P00:`0d
;P0[=BX0=00^03@0;R/^02X0;R/^:bh[=00^:P0d03@0:P00:P0d;P0d02X^02X0
03@^:P0e02h003@^02X003DZ02h[;P0d:bh0=3@^02X0=00Z;R/0=2h[=@0Z02hZ
03@e02X0:bh0:P0^:bh[03@^001o:SD0:P0d03@0=@0Z;P0[03@^02h0=2h^02h[
02hZ03@^02/^03@^000d02X0=@0[=BX0=2h0=B/0=2X^02X0=00[;P0d03@0;R/0
02X^:`0d03D003@d02X0=00^03@^02X0:bh[=00d03@^03@0=00Z02/e;P0d;P00
:P0Z02/E;P0d;P0d;RX0=2h[03@0;P0e03Xj004IOcX[02X0:P0Z=00^:P0d02X0
:bhd02/^=00Z03@0;R/e03@002hd000^:P0^=00Z=C@0;R/0;S@003@0:bhZ000d
02X0:R/^03@0=00Z02X0;R/d02h0:bh^03@0:bhd02/e02X0=@0Z03@002X^03@0
:bh0=00d03@002X^03@dO`0[;S@0;R/^;R/d000d02h^03@[02h[02X0;S@[000^
02Xe:`0Z03@[;S@0=B/d02X002h[02X0:P0^03@0=00Z=@0[;R/^=B/0;R<002h[
;S@0:S@0;RX0=00Z02hZ03@0=00[=@0^:P0^=00^:bhd03@0:P0[;P0Z=00d03Dd
5@00:P0d02h0:bh[=@0Z02hd02Xj>P016Glj;P00=00d02h[000d;P00=2h0:P0d
02h[03@^:`0^:bh0=2X0;R/e000Z02h002X0=00e:P0^:c@^03D[03@^03@0=00e
:bh0:SDZ02h0=00^03@0=2h0:bX0=2h0;P0d;R/0=2/^03@^:bh002X^03D[;P0d
;R/e:P00:cD0:Wle02/^:`0e02/^;R/^03@0:P0^03@0;R/0=00^03@Z03@0:P0d
03@^02h0:P0Z;P0[=BX0;R/0=2h0:cD0:Rh[02X0=@0Z02X0=00^:cD0:P0Z000Z
;P00:Rh0:c@0;R/^02Xe02X0:`0Z02h[03D[;P0^:`0d;P0Z02h[;R/^01DZ=@0Z
;R/0=2h0;R/e03@0:P00>SX00AUo>P0d=00^;S@0;S@^=00[;RX0=00d02h[=@0Z
02hd02h0:bh0;R/0;R/Z03@0:S@0:SD0:P0^:`0^:`0Z02/^02X^=00^:bh0:bh0
02X0=00d03@0;RX0:bh0;Rh0:P0^:P00:P0^:`0Z02h0;R/^:`0Z02/0:P0^000Z
=C@0:P1o;R/^02hd02/^02/e02/^=00[;S@0;S@0:bhe:`0Z=@0^=00d02h^03@0
=00[;P0d02X0=00d02h[02X08`00;S@0:bhd03@002hd03@0=00d03@Z000Z=000
:SD0;R/e02/^03@0:bhd03D[02h[;P0d03@0;P0Z02h[=@0^:`0E;R/d000^:bh0
:P0Z03@0:P0Z=CXj004IOcX^02X^02/^:`0^02h[=@00=2h^02X0;R/e02X0:P0[
=2h0=00d;P0e03@^:`0e:P0Z000d03@^03DZ03@^02X002X^02X0=2h0:P0d;P0Z
;R/^:`00=2h0=00[=@0d000Z=@0Z03DZ02X0=00e02Xd03@^:`0d02X0;R/^02X0
Ob/^03@0:Rh0:cD0:P0d02X0=B/^:`0^=2h0:P0d02X0:P0e02X0:bh^:bh0=00d
;Rh0=2h0=BX0=BX0:P0d;P0^:cD0:Rh0=2X0;R/^:P0e02Xe03@^02Xd;P0Z02/0
:P0d02X^03@0;RX0=2h0:P0Z;R/^:`0d03@0;R/0=2h[5@0^03@Z03D0=00d02X^
02h[02Xj>P016Glj02X002/^02hd02X[=@0[;RX0:bhd02/^02X0=00d02h0:bh[
;P0[:bhZ03D0:P0Z03D[;R/^02/^:`0^03@0;S@003@002X0=00[;P0[;P0d000Z
=BX0=00^=00[;R/^=00Z=@0Z03@0=00^:c@0;P0Z03D0;R/^=00^03@0:Wle02/^
=000=00^:bhd02X0;RX002hd02h0=000;RX0=000;RX0:bh[02h0:Rh0;R/0:bh0
:RX002X0=000;R/0:bh0=00[02X0;R/e000Z02X002hZ03@0;P0d02/^=00[;R/^
02/^=B/0:bh0=00d000^03D[;R/^=B/0:bh0;QDd02hZ000Z02X0:P0^03@0;S@0
>SX00AUo>Rh0=3@^02X0;R/0;R/^02/0=2h0;R/0:cD0=2h0:S@0=2h0;P0d;P0[
03@0:P0e02X0=00e:bh0:c@^03@^:`0Z=BX0=3@0:Rh0=00d;P0Z03@^02/0=2h[
02hZ000d02h[02X003@^02X^02X0=00d=B/^:`0e02X0:bh0=01o:P0e:bh[=BX0
:`00:Rh0=000=2X0:P0[=2h[=00^:bh[=00^:cD0:P0[=@0[=00e:P00=000=3@0
:SDZ02/e:P0[=@0Z02X0=00^:bh0=2X0=2h003@^02/e03@^02h0=00e:bh0:bh0
=2h0=2h0:P0d03@0=@0Z02X0=2h0:P0E;R/0=2h[03@^02h[03@^:`0^03Xj004I
OcX0=00^02X0=B/0=B/0:cDd02h0:P0^=00[;RX0=00e:bh0=00[;R/0;S@0:Rhd
02X0:Rhd02X0=00e:`0Z02hd02X0:P0e:P00:SD0:P0d02h^:`0^:bh0;S@[03D[
;S@0;SD0:S@^02/^03@0=00Z;P0Z02hd02X0:S@0=00ZO`0e:bh002X0=00^=00[
02h[;RX0=00d03D[02hd03D0=00e:`0^=00[;R/d02h0:P0^8`0e:bh^=00^03@0
;P0Z=@0Z02Xe02hd02X0=00^02X0:bhZ02/^:bhZ03@0=00^:P00:SD[02h[02X0
=00[;R/^:P0Z03@0:bh0=00d5@0d:bh0=C@Z03@0=00Z02hd02Xj>P016Glj:P0d
02X0=00Z;P0d;P0Z03@0=00d02hZ000^:bh0:P00=2h[03@^:`0^000[;P00=000
:Rh0=2h^02h[03@0;P00=2h^000d;P0Z02h[;R/0:cDZ03D[;P0^03@[;P0Z02/[
;P0^03@^03@^;P0Z000Z02X0;R/0=00^:P0d;Wl0:P0[=2h0:P0d;P0d;R/0=B/0
=2h^02X0;R/0;R/^:bhZ02h[02h0=00^03@^:`0Z000Z02X002h[03@^:`0d02X^
03@0:P0[;P00=2h[03@e:bh003@^000[02X^:bh0:`0d;P0[;S@0=BX0:Rh0=00e
02X0;P0Z03D[;P0d;QD0;P0[;P0^03@^:bh003@0;R/0>SX00AUo>P00;RX0=00Z
000[;P0d000^:P0^:cD[02hd02h0=000;RX0=B/^02hd02/Z=@0Z=00^=00[02h[
02X0=00^=00d:bh[02/^:P0d03@0;P0^=00Z000Z000Z03@0;P0Z03@0:P0e:`0Z
02/^:`0Z02hd03@0=00d02hd000Z;P1o=00d02h0=00^=00[;P0^=00^=00^03@0
:P0^=00^:cD002X0;S@0:SD0:P0Z02hd03@0;S@0;S@0;R/^03@0;S@0:`0^=00Z
=@0^=2X0;R/^02X0=B/^:`0^=2X003D0:SDd:P0Z=@0^02X0=000:bhd02X0=00[
;S@0:P0^:P0E=00^=00^:`0Z03D0:S@[=B/0;SXj004IOcXd;P00=2hZ03@d;P0d
;RX0:`0Z02h[;P0[;R/0=2hZ02/0=00e:bh0;R/e02X002X^02X0=BX0;R/0=2h[
02hZ03D0;R/0:`0d;RX0=00d02h[03@d02X0=2h0=00d03@^000d:P0d03@^02h[
03@0:Rh0=2h^03@0;RX002/eOb/^03@^:bh0:P0d;P0d02X0:bh0:bX0=00d02X0
:`0Z03@0=00Z000Z=00d02X0;R/^:`0Z02/^:`0e:bh0=00^:P0d02h0:bh[02h0
=00e:bh0:P0e03@Z02h0=2X0:P00;P0d02X0:c@0=2h^:cD0:P0^:bh[=00^:`0d
02h[5Bh[02h[03DZ02X0=3D0;R/^02/j>P016Glj;P0Z=00^03@0;R/0:P00:SDZ
03@0;P0Z=00^=00^03@d02hZ02X0;R/e02X0;S@^03@0=BX002/e02X0;SD[03@0
:P0d:bhe:P00:bhd02X0=BX0=BX0:bh0:Rhd02h^03@[;P0^=00^02X0=00^=00[
02h[02hZ02X0=00^:gl^02hZ02h0=000;P0[;S@0:SD0:bh0;SDZ02X0;S@0;S@0
;Rhd03@d02h0;S@0=00^02hd02/^02hd02X0;RX0:P00;S@0:bh0;S@0=00^:bh0
=00[;R/^03@0:bh0=00^=00d02X0;S@0;Rh0:cD0:P0d03@003Dd02hd02X0;QD0
;S@0=00Z03D0:bh0:P0^02/^>SX00AUo>P0d02X^03@^02/e:P0d;P00=BX0=00d
02h[02X^03@^03@0;R/0=00Z02X0=00Z03@^:`0Z03@^03@0:P0Z03@0:P0d=@00
:P0d;P0Z03@^:`0^:P00=3D0:P0[;R/0:P0e;R/0:P0d03@0=2h[02h^03@0;R/0
02X0=2h[02io02X003@0=2hZ:`0d;P0^02/^:cD0:P0[02X0=00^:`0Z02/0;Rh0
:P0[=00^:P0d03@0;P0d03@0:P00=00^:`0d=00Z;P0[=00Z02X0:`0[=@0d;P0Z
03@^03D[;P0d02X^03@0=00^:`0[;P0Z02h[;P0Z;P0Z02X0;R/0=00E:P0Z03@Z
02X0=2h0:P0d03@^03Xj004IOcX0;RX002/^03@0:P00:P0e:bX002hZ03@0;S@0
02/^:`0^:P0d02hd03@002hd000Z02hd02Xd02/^=00Z03@0:SD0;P0Z=000;P0d
02/^02hd000Z=BX0=00[;P0^=000:P0^=00^;R/^:bh0;S@0:`0Z=B/0:SDd02h0
;S@0Oc@0=2/^:bh[03D[;P0Z02Xd02h0=00Z02Xe02hd02hd02/^=00[02hd02h0
:P00;R/^=00[;R/^=00e:bhd02hd02h[;P0d02h0;S@0=C@0;R/^:P0d02/^02h[
;P0Z;S@002X0;Rhd:`0^=00d02h[=@0[;P0d03D0=00d02hd5@0d02X^03@002X0
=000:P0Z02hj>P016Glj:P00=3@^02X^000d=00Z03D0=2X002X^02/^:cD0:bh[
000d;R/0;R/^:c@0:Rh[02X0;P0^03@^02h002Xe02h[:P0Z02hZ03@^03@^02X0
;R/^02X0:Rh0=B/0:bh[=00Z02h[000d03D[;P0^:cDZ02X0=00Z03@0=00Z;Wl0
=2h0:bh0;R/0=B/^02X0;R/0=2h003@0:P0[;R/0=BX003@Z03@0:SD[;P0d:P0^
02X0=@0Z02h[;P0[;R/0;RX0;P0d;R/0=00Z;R/^02/^000d;P0e:`0[02/e02/^
03@0=00^02h[02h[;R/0;R/e02/d;R/0:Rh0=BX0;QD[;P00:cD^:c@e:bh[=@0d
03@0>SX00AUo>P0e:bh^03@003@[;Rh0;P0Z:bh0=2X002Xe:`0^:bh0;S@0:P0^
:bhd000^=000;S@0=00d02/^02X0=2/^02X0;P0d03@0=00Z02hZ03D[02/e02X0
=@0[02X0:S@002hd03@002Xd02hZ02/e:`0Z03@002hd03@0:Rhd02]o02X002X0
=B/0;R/^03@0=000:bh[02X^=00[;P0^=00d02X^02h^=00[=@0[;P0d03@0:bhd
02h[=@0[;R/0=B/0=B/0:P0e:bhd000^02Xe02/^:P0d02h[;SDd02/^02/^:P0Z
02X0;S@0;P0^=00^:c@0;P0^:P0[02X0:P0E;P0d=00^:`0^02h0;P0Z;S@0;SXj
004IOcX^:`0[03@^:P0e;P0[=00d02h[;P0d=BX003@^03@0=00Z=@0Z02h0;R/e
02X^:`0e:P0d;R/e02X0=2h0:bh0:P0d;Rh0=2h[03@003@[;R/^03@0:P0d;R/0
=@0^:P0Z03@^:P0d;P0[03@^02h[03D^:c@0;Rh0=00^03@^Ob/0=3@0=00d;P0Z
03@^03@Z=Bh0:S@002h^02/e02X^03@0:cD[02h^02X0=00Z;P0Z03D[;P0[02X0
=@0d;R/^02X0=00d02X0;R/Z02X0:cD0:`0d;R/0=00Z02X003@^000Z03@0=00Z
02/d02X003@^:`0d02hd=BX0:P0d5@0d;P0d02X0:P0[:P0d02h0:P0j>P016Glj
:`0^;S@0:P00:P0Z02hd02hd000^02X0;R/^02/^:bh[;P0d03@0;R/0:S@002hd
02h0;R/e03@0:bh0;S@0=00[;P00:bh0;S@0:RX0;P0e02/^=00^:P0e02X0:P0d
02hZ000[;P0^=00^02X0;S@0:P0e:P0[02hZ02X^07ld02h^=00^:P0d02/^02X^
02X0=000;S@0:cDd03@0:`0^=00Z02h[03@0:bh[;P0d03D[;P0[;S@0:bX0:P0^
03@0;S@0;RX0;R/e03@0;S@0=00^:P0^=00Z03@0;S@Z02Xe=00^:bhd02X0=C@0
=2/^02hd02h[02h0=000;QDd02X0;RX0=00Z=@0d02X0=00^>SX00AUo>P0d02/^
000d=00d03@0:Rh0:P0d03@0:P0e:bh003@0=B/^03@^:`0e:P0^:bX0;R/0=00^
:bh0=2h0:P0^:bh0=00d:P00=00^000d:P0Z03@^02h[000d:P0e:P0d;P0[03@e
;P0Z02h[03@0=00^:`0d02h[;R/003@0:P1o=BX002X^000d;P0d02X003@0=2h0
=00Z;P0Z02X^03@^02h[03@0=2h0=2h0=B/^03@[;P0d;P0^03@0=00d03@^:`0^
:`00=00^:bh0=00^:P0d000d02h^03@^02/e02X0:P0d03D0:P0e:P0^;P0d02X0
;R/0=BX0:P0d=00E;Rh0=00^03@^02X0;RX0=00d03Xj004IOcXd02Xe02Xe:P0^
:P0^:P00:P0Z;S@0:SD0:bh0=2/^02X002X^03@0:P0^:cD0:P0^=00Z02X0;RX0
=00d03@0=00^:P0e:bhd02Xd;P0d02/^02X0;S@[;P0Z02/0;P0[;SD0:P0Z03@0
;S@0:Rhd03@0;RX0;P0^=B/^=000ObX0:S@002/^:P0[;S@0=00^:bh0:bh[;P0d
02/^02/^02X[=@0^:bh0:bh[02/^02/^:P00:P0d02X^=00^:P0Z02hd03@0:Rhd
02h0:bh0:P0d02hZ02X[02/^02X^03@0=000;R/^:bh0:P0^:`0Z03@0=00d02X0
02hd02h^5B/0:bhd02/^02X0=B/002hZ02Xj>P016Glj;P00:P00:P0d02h[03DZ
000d02/^;P0[=2h0=2h0:c@0:P00:P0^:`0d02h0=00d02X^02X0:P00=2h[;P0Z
;P0d02h[;P0[;P00:P0d;P0d03@0:P0^02/e03@^:`0d;P0Z000d03@^:`0^:`00
:Rh0=00^:`0d02X002Xe:Wl0=@0Z;R/d000d;P0^:bh[03D[;P0[=@0d;P0d03@^
02X0;R/e02X0=2h0=BX0:cD0:`0d:P0d03@0;P0d02h[03@0:Rh0=000:P0[=2h0
=00Z;P0[02X0=BX003@0:bh^02X0=00e:`0[=00Z02h[02X^02X^03@0=2X0;R/0
:aD^:`0^03@^03@0=2/^03@0;R/0>SX00AUo>P0e:P0d=000;S@0=00Z03@d02hd
000d02X002X0=00^02hd=00d02hd03D[02X0;S@003@0=@0[;RX0=@0[;P0d02X0
;P0Z=@0[;S@0;P0[;R/^=00d03@0:bhZ02he:P0d02/^:P0Z02hd02he:bh0=00Z
02hd02X0=C@0:P1o:P0d02/e03D[;P0Z02h0;SD[;P0d02X0:P0[;R/^03@0=00^
:c@002h[02X0=00[;S@0;P0^:P0e:P0Z;S@0=00^=000:bh^=00[;SD[02X0;P0d
02Xe02X0=B/^=00Z03@0:bhZ02hd02hd03@002Xe02X002/^:bh0:P0^:bhE02he
:`0Z02/^:bh0:P0Z=B/0;SXj004IOcXZ02h[;Rh[=00^:bh002X^03@^03@Z02X0
=2h0=2hZ03@0;Rh[;R/0;RX0:P0e:P0Z;R/^:`0d;P00=2/^02/d;Rh0:P0Z02X0
=00^:`0d;P0^02X0;RX0=2h003@0:P0d;P0d02h[03@0:P0[03@0=2h^02X0:P0d
;R/^000dO`0Z;P0^:bh0=00d03@0=00Z02/e;P00=2h0=00e:bh0=2h0:bh0=3@0
=BX0=2h^02/^:`0d02hZ000d02h0:Rh[02hZ03D[02X0=B/0=BX0=00d;P0d03@0
=00d02h003@^;P0[03@0;RX0;P0d:P00:P0e:P0e:bh0=00d03D[5@0[;P0d;P0d
03D[;P0d02X0=00j>P016Glj02X0;P0[=@0^:cD0:S@002hZ02/^02X0=@0Z02X0
03@0;R/0;R/0=B/0;S@0:P0d000d000Z=@0[:P0^03@0;R/0=00d03@0:bhd000Z
;P0d02/e=00d02h0:R/^=000;P0d02h[=@0^=00[;R/e:P0Z03@0=00[;P0^03@0
;Wl[;P0d02h0:bhd02X^=00Z03@0:P0[;RX0=00^:bh0:bh0:SD0:bh^:bh[02h0
02Xe03@0;S@002hd02X0=000;S@0=00Z02Xe02X0;RX002h[;P0[;S@0;Rh[=BX0
:S@^:`0[;S@0;R/0=00[;P0Z=C@0:P0d:cD0:bh0:P0Z01DZ=B/0:P0d02X0;R/0
:P0e=00^>SX00AUo>Rh0=00d02X^02h[03DZ03@0;R/^03@e:P0d03@^:cD0:P0d
;P0^:bh0=00^:`0d;RX0;R/e02X0=@0S03@^02/^:bh0:P0e:P00:SDZ000d;R/^
02h[;R/0=00e02XZ02/d;Rh0:bh[02h0=00^000Z02X0=2h0=00Z03@^:`1o=B/^
03@0=2h0;R/002X^03@^000d;P00=2hZ02h0=2h0:P0Z03D0:bh0;R/0=2X0:P0^
:P0^:bh0:Rh0:P0d;P0Z02h[03@0:P00=00[=3D0:P0d;P0^:`0[;P00=00^03DZ
000d:P0^02X0=00d02X0=00d;P0Z03D[=@0d03@E02X0=3D^03@^:`0e:P0d02hZ
03Xj004IOcX0:bh[;RX0:`0[;P0Z02X^=00d000Z000Z02hZ03D0:P0[;P0d03@0
=00Z02hd02h0=00d03@0:bh[;P0Z02/^03D0=00^:P0e=00Z03@0:P0^02X0:P0e
02X0;R/^03@0;R/0:SD0;S@0:Rhd02Xd03D0:Rh0:bhe:`0Z02hdO`0^:`0Z02X0
;R/0;S@002/^02Xe:P0d:bh[02h[02X0:P0Z03@0:bh0:P0e:bh0=00d000Z03@0
=000=00^:P0d03@0=00^=00d:bhZ03D[=00[;P0Z02h[;P0^:P0e:`0Z03DZ03Dd
03@0:P0^=BX0=00^;P0Z03@0:P0Z02h^5C@0;RX0:`0Z03D0:P0d02h[02hj>P01
6Glj;P0[000e:P0d;P0d02X002X0=BX0;R/^03@003@[;P0d;P0Z03@^:bh[03@0
:Rh0:Rh0:Rh0=2h0:P0d=BX0:c@0:Rh0:`0d02X003@^:P0Z03@0:P0d;R/0=00Z
03@^02/^02/0=00^:`00:P0^03@[;P0[=Bh0:P0d03@0:Wl0:cDZ03@0:P0e:P0Z
=BX003@0:P0d;P00=2h0=2h0=00d03@^:P0[=@0Z03@0:SD0:SDZ03@0:Rh[;P0d
02h[;P0Z;P0d02h^03D0:Rh0;P0d;P0d02X0=B/0:`0Z;P0d03@0:Rh0:P0d;P0d
02/e02X^02/^03@^:`0d;R/001DZ000e:P0d02X0=2h0:Rh0:P0[>SX00AUo>P0Z
=C@0:P0[;P0[;S@0;S@0;RX0=00e:`0Z02X^02X0:P0^=00Z02h0=00^=000=000
=000;RX0=000:P00=00^:P00:S@0;RX0;S@0;P0d000^=000;R/0;Rhd000^02Xe
02/^=00^:cD0:SDd02X0;P0d02X0:P0d02h^:cEo=00Z000^=000:P0d02X002<0
=B/0;P0e:bh0:bh[02hd02X^000[;P0d:`0^:bh0=00Z03@0:Rh0=00d02h[=@0Z
000[;Rh[02h[=000:P0Z;P0[;S@0:P0^=00Z=@0[;R/^=000:P0Z;P0d02X^02X0
02/^:`0Z03D0:P00:S@E02hd:P00;RX0:bh0:bh0=000;SXj004IOcX^02X0=00d
;P0d;P0^:`0Z02/0=2h[;P0d03@0:P00=2h[02h[03@0=2hZ02hZ02Xe:P0d:P00
=2hZ02h[;P0d02h[=@0^:`0^:`0Z;P0d;R/d02hd;P0Z03@0:SDZ02X0:P0e:bh[
02h[;P00:P0d:P0d;P00=00d;R/002X0Obh0:c@e:bh[=00d;P00=2X0:P0^:`0d
:bh0=2h0:P0[;P00=2X0=B/^03DZ02h0:P0d02h0=000=2h[;R/0;R/0=3DZ000[
=BX0;P0d:P0Z000d;P0^:`0d02X002X0=00^02X0=00e02X0:Rh003@^:cD0:cDZ
02X0=3DZ000d5Bh0;P0d:P00=2h0=2h0=2h[=00j>P016Glj03@002hZ=@0[;P0Z
02hd02Xe:bh0:P0d02h^=00^=2X0;S@0;S@0;RX002h[03@0:`0^:P0^=00Z03@0
;P0Z;S@0=00d02hd03Dd02/0:P0e02X0:P0d03@0:P0Z03@0=00e:bh0;S@0;R/0
:P0[;P00;R/0:Rhd02X0;S@0:gl^=00^02h[02h0;P0Z=00^02hd02hd02h0:bh0
=00Z=@0e:bh0=00^:`0Z03D[02X0=BX0:Rhd02X0;P0^=00^=00Z02hd02X0=00[
;P0Z03D[;R/0:cD0;S@0:S@0:bhd03@0:bhZ03@0=000;R/^:`0^=00Z03D0:bh[
03D[;QD0:P0^=000:SD0:bh[02X0=00^>SX00AUo>Rh0=2X002X0=00d02X0;P00
:P00=00d;R/002X0:cD0:P0^:`0^:`00=2X0=BX0=2h0:`0d02hZ03DZ02/e000Z
;P0Z;R/0:P0Z;P0d=@0Z03@^000d;P0d;P0d02X0=2h[02X0:P0Z02/e:P0d=@0d
:P0e:P00:Rh0=00Z;P1o:bh[03@0=BX0=00Z02h[03@0:P0[;RX0=2h0:P0d02X0
:P0[=Bh[03DZ02X0=2h0:P00=00^02X0=00Z02X0:bh003@0=2h0:P0d;P0d03@0
=00e:P0Z02/^03D^:cD0:Rh0=2h002X^03@^:`0e02X0:bh003@Z03D0;R/^02/E
=@0Z02X^:`0Z03D0:P0^:bh0:cXj004IOcX[02X0;S@0:SD[;S@0=00[;S@0:Rhd
02X0;S@0:SD0:P0e:`0e:`0Z=BX002X0=00^=00Z03D[000Z02Xe02X[;P0d000^
=00d000[;P0Z02hZ03@0:P0^:P0^=B/^02h0;S@0=00Z02Xe03@0:P0^:P00:P0^
=00[02hd000^Oc@[02hZ02X0:bhd03@0;S@0:P0Z=B/002X0=000;S@0:bhd02X0
=00Z03@0:bh0:P0^=00Z02hd02hd02X0:SD0=2/^:bh0;S@0:P0[;P0^:P0^:P0d
02Xe:`0Z02h0=00[02h[02Xe02/^:`0^:bhd02hd02X^03@0:R/e02/^5B/d03@0
03@0;S@0:P0e:cD0:bhj>P016Glj;R/0=00^000Z000Z;P0d;P0^:`00=BX0=00Z
000Z000d:bh[;P0d02X0=3D0:Rh0:bh0=BX0=2h[03D0:P00=00d;RX0:bh0=2h[
=@0Z03@003@^:P0Z000d02/^03@0:P0^02X0=2h0:P0^000d02h[=@0Z02h0=2h0
:Rh[07l^03@0;R/0=2h0:P0^:`0e:P0e02X0=3D0=2hZ02X0=Bh0;R/0=2h[03@^
03@0=00d02X^02X0;R/0;P0d000Z;P0e:bh0:P0^:`0d;P0d02h[000d;P00:P0d
03@0:Rh0=2h0=BX0:P0e03@Z000[;R/0;S@0:bh^:`0^:bh001Dd;R/^:cD^:`0Z
03@^:`0Z03D0>SX00AUo>P0e02hd02Xd02hd02/0:P0d03@0;RX002hd03DZ03@[
;Rh0=B/0;S@002h[=000:SD0:bX0:bh0;S@0:P0Z=00^:P00:bh0:Rh[=@0d:`0^
:bhZ000d03D[;R/^02/^=00Z02Xe=000:P0d03@Z;S@0;R/d03@0:Rh0=00[03Eo
:`0^=00d02h0=00Z03@0:P0^:c@002X0:P0Z03@002X0:R/0;Rh0=00^02X^02X0
;R/^03@0:P0^=00d02Xd000^:bh0:P0e:`0^;P0d02X0;SD[;P0Z=00^=00^:P00
:SD[;R/0:P0d02Xe000Z=@0^=00Z02h[02he:`0[02XE;P0d000Z02hd02/^02hd
03@0:SXj004IOcXZ03@0;R/0;R/0;SD^:`0d;P0d=00^:c@0;RX002Xe02/0=00d
:P0^:P0^02X^02/^:cD0=2h0:P0^:`0d02h[000Z=000=000;R/^03DZ03D[03@^
03@0:P0^03@^02X003@002X^000d;P0^02/^:`0^03@^:`0[=2h0=BX0OcDZ02h[
;R/0=2h0=BX0=00d02h^:cD0:P0d03@^:P00=00e:P0[;P0Z03@0:cD0=00^03@^
02/d02X^03@e02X[02h0=00d:bh0=00d;R/0=00Z02/^02h[02h[03DZ000Z02/^
03@^02X0:P0d02X0:bh003@0=B/0:P0d;S@05@0Z;R/e02X0;R/e02X0;R/^000j
>P016Glj000^:P0^=00^=00Z03@0;P0Z;P0^:cD0;R/0;S@0:P0^=00Z;P0e:`0d
03@002/^:`0^:bh0=00d03D[;P0^=00Z=@0^=00Z=B/0:P0Z02h[02hZ02/^:P0d
02/^02X0;S@Z02X^03@d02hd02/^02hd02hZ000Z=@0Z02X0=7l0:P0[=@0^=00Z
02X0=00^;S@0:`0^:bhd02h^03@0:SD0:P0Z=B/0;S@0:SD0:Rhd:`0Z03@0;S@0
02X^03@0;S@0:bhd02hd02X0:P0^=00Z02Xe03@0;S@0;SD[02Xd02/e02/^03@0
=00Z;S@0:SD0=2/^=00^=00[;P0^01DZ=000;S@002/e02X0=000:P0Z>SX00AUo
>RX0:`0d02h[02h[03@^:`0d02/^02/^:`0e:P0^000d;P0Z000d02Xe;R/^:cD0
:c@^02h0:P0d;R/0=B/Z02h[02X0=2h^02X0=00d03@0=2h003@^000d;R/0:cD0
:P0^03@003@^02X0:P0d03@0:P0[03@e02X003@^02Yo000d;R/0:P0d03@^02X^
02/^:bh0:bh0;R/0:cD0:P0Z000d02X0=00^:`0Z000Z03DZ02X0=00^:P00:Rh0
=00Z03@0;R/0;R/0=00Z02h[03@0:P0Z02/^:`0Z03@0;P0e:bh0:cD[;P0d000Z
000Z03D^02X0:P0d;P0d03@E02h[=B/^:c@^02X0=2h[=@0d03Xj004IOcX0=C@0
;S@0;S@0=2/^02hd02Xe02/^03@0:P0d02/^:P0d000Z;S@0:P0^:`0^=00^03@0
=00[;R/0;R/^02X0;SDd02X002X0=00^=00^:bh0:bhd000Z;R/0;S@0=00e:`0Z
02hZ02/^=000;R/^=00d02hZ03@0:bhZ03@0Obhd=000:P0d02h^03@0:`0Z03D0
:SD0:P0^=00^:bhd000Z;S@002hd02hd03@d000Z03D0:Rh0:P0d=000;RX0;S@0
:P0^=00d02hd02h[=@0^=00d02Xe02hd000^=00Z03@0:bh[;P0Z02hd02hS000Z
02/^=000;P0[;R/^5C@0:P0^:`0^03@0:bh0=00d02hj>P016Glj;R/^:`0Z02/^
02h0:cD0:Rh0:P0d03@^:P0Z03@^000d;RX002X0=00Z03DZ02X003@^:bh0=00^
:`0^03@0=00Z03@0=2h0=2h^02X^02X0=2h0:P0d000e:P0^:P0d03@^03@0;R/^
02Xd;P0^02X0=2h002X^03D[03@^:gl0;Rh[=@0d;Rh0:bh0=2h[03@0=00Z000d
02X0:bh0:SDZ000Z;R/0:P00:Rh0:Rh[03@0=00Z000d;Rh[:P00=00Z;P0Z02h[
;P0Z03@0;R/^02h[;P00:P0[;R/d02hZ03@^03@0;P0d;R/0:P0[02Xe03@^02X0
8`0d;P0^01D^:`0d02h[03@^03@0:RhZ03@^>SX00AUo>P0^02hd02Xe03@[02h[
=000=00d02hZ000e02/^02Xd02h0=3@0;Rhd000Z03@0;S@Z02h0:Rhd02hd02/^
=00Z03@0;Rh[02h[03@003@0:bh0=00d;R/^:P0d000Z02X^02/^=00e02X0;P0d
02/^02h[=B/^02X[=@0Z02io=B/0:P0^:P00:S@0:bh0;S@0:Rhd03@[;S@0:SD0
=00[02hd000^=00e:bh0=000;S@0:Rhd03@0;R/0;P0e:bhZ000d02h[=@0d000^
=00^02X0;P0Z=00Z=B/0;RX002/^02X0=B/0:P00:SDd02Xe02/^:`0^=00[;P0Z
02XE02hd02hd02X^02/^:bh002/^03Xj004IOcXZ02X0;P00:P0e;R/0;Rh[;P0d
;R/003@Z03@^03@0:Rh0:P0^:`00:SDZ02X0=00^02X0=000:P0[;R/^02h[03@^
;P0[;R/0=BX0=2X0=2h0=2h0:P0e:`0d;R/e02X003@^02X0:P00=00d;R/0:c@0
;R/003@0;R/^02X0ObX0=00d02h[;P0^:cD0:P0^000Z03@e02X0=@0Z02h[=BX0
:P0d02X^:`00=2hZ000d;P00;RX0=00^:`0d03@0;R/e03@0:P0d;RX0:bh[03@0
=00d02h[02h0=00e:P0e:bh0=2/^:`0d;P00:Rh0:P0d02h[02h^02/e03@05C@0
;R/0;R/0:cD^02h0=2h0:bhj>P016Glj02X0;R/e=2/0:P0^=B/0=00^:P00:SDZ
02/^02/^=000:P0e:`0^=00Z03@002hd03@0=00^=00^=00e03@002X^02/0:P0e
02X0:Rh0;S@002X0=000:P0[;P0d03@0=00Z02X0=00d=00^:P00:SD0:P0^=B/^
=00^03@0=7l002hd02h[=@0Z03@0=00d02Xd02h^03@0:P0Z03D[02X002hd02Xe
000Z=00^03@[;P0Z=000:Rhd02hd02X^=00d02X0:P0^:P00:SD0=00Z02h[;S@0
;S@0;S@0:P0d02X0=00^02hd02X0:P00=00Z03D[03D[02hd02/^=1D0:P0^=00e
=00Z03@[02X0:SD0>SX00AUo>Rh0=00^:bh0=00Z02X0=00d02h[=@0Z03D^03@^
02X0=00d:bh0:bh003@^:P0Z03@^02X^02X0:bh[;R/^:cD0:cD0=00d:P00=00[
;P0^:c@e:bhZ03@^02X0;RX^03@^03@0=2h^02X0:`0d=@0Z;P0Z02X002X0:c@^
02Yo;R/0;R/0;R/0=BX0:Rh0:P0^02X0:bh0=2h[03@Z03D0=2X0;R/0:P0d=@0Z
03@^02/^02X0=000:P0[;P00:P0d;RX0=00d02h[=@0Z02hd03@0;P0Z02/^:`0^
:`0d;R/0=2hZ03@0;P0d=@0Z=3D0:Rh[02X0=2X0;R/^02XE000d02X0:P0d03@^
03@0=00Z03Xj004IOcX0=00d02X002hZ03@002X0;RX0;R/d000Z02/^03@0:bh[
;Rh0:SD0:SDZ000d02X^02<003@0;S@0=@0d000Z02h[=00^:P0e:bh[;P0d:`0^
02h003@^03@002X002X^03@0:Rh0:`0^=00^;R/d000d02X0=B/^=00^03@0O`0^
=00^=00d02X0:bh0=00d03@0:bh0:Rh0;S@0;P0^:bh0=00^=00Z;P0Z02/^02Xe
03@0:P0^=00Z=@0Z=00d02h002X0;S@0:P0d03@0:P0^=00Z02Xe02hd03D[;P0^
=00Z000^=00Z=B/d02h0=000;S@0;RX002/e02X05CD[;RX0:cDd02h^03@0;Rhd
02Xj>P016Glj;P0Z;P00=3@0;R/^:cD0:P00=00^;R/e03@^02X0=2h0;P0[;P0Z
000Z03@^03@002X0=2X0=00^:bh[;R/e03@0;Rh[000d:bh0=B/^03DZ02X0=2h0
:cD^:c@0=2h0:cD0:P00=2h[03@^02/^;R/^03@0=00d02X^03@^:Wl0:bh[02X0
;S@0=2h0:Rh0:P0d;P0[;P0[;P0Z02/d03@0=2h0:bh003@^03@^02X0:P0^03@^
02X002X0;P0d;RX0=2h0=00^:`0Z;P0Z000Z02X0=2h0:bh0:P0Z000d02h^03@Z
02h[000^;R/0:Rh[;P0Z02/0=2X0:P00=1DZ000e:P00:Rh0:bX0=000:P00>SX0
0AUo>P0[;P0d:bh^=00d000Z=00e:bhd02/0:P0Z03@0:Rh0=00Z=00d03@d000Z
02h^=00Z02h0:P0d02h0:P0e02X^=00[02Xd02h0:P0^:`0Z03@002X0=00Z02hd
02h0:bh0:Rhd02X0=B/^02/^:`0e02/^=00Z;S@002/^:`1o:SD0;S@0=00^:bh0
=000=00^:P0^=00d02/0;S@0;S@0:Rh0:bh0;S@Z02/^03@0=00d03@^02/e=3@0
=00d02X002h[02hZ03@0;P0d02hd;S@0:bh002X0=00d02hd:cD[02/^02h[=@0Z
:`0^=B/0=@0Z03@^=00^=@0d:bhE02hd:`0^:P00:SD0:P0Z=00Z=CXj004IOcX^
02/^03D[02h[;R/e02X^:`0Z03@^02/d03@0=00^:bh002h[;P0Z;R/e02X0:bh[
03@0=2h^02X0=00Z02X002X^03@0;RX0=00d03DZ03@^:P0^:bh003@0;RX0=2h0
:P00=2X0=00e:bh003@Z03@^02h[02h0=3@^02h0Ob/0=00^:P0d02h0=2h[;P0d
000d02h[;P0d:P0^:`0^;P0[=Bh0=00^03@^03@^02h[;R/^03@^02h0:Rh[;P00
=3@0=2h003@^:`0d;R/0;P0^:cD0=3@0=2h^03@0;Rh0:P0d03@0;R/^02h[02X0
:P0d03@^02h[02X0=Bh05BX0;P0d02h[;P0Z03DZ02h002Xj>P0162Lj02Xe:`0Z
02X0=00^03@0:`0e02X^03@0;P0^:P0^:cD0:S@0;P0SoP1o;S@0:bh0;S@0;RX0
03@0:bhd03@0;S@002/^=B/0:bh0:P0Z000^:`0d03@0;RXd02X002X0=000:P0e
=00^:bh0=2/^02/^02X0;S@0:bh^02X0=00^:cD[02X0=B/0:P0^02<0=BX0:R/e
03@^02X0;SD[03@0:P0[;S@[02iZ:`0Z03@0:P0e02/^02X0:P00=B/0:Rh0:Rh0
:bhZ03D[;R/0;R/e:`0Z02X0;Rh0:`0Z02X0=00[;S@0;R/0:cD[02X0;S@0:P0^
02X0;S@0;RX0=00d03@0;R/e02X0=B/0;P0Z=00j>P016Glj;P0[=BX0=00d;P0Z
;P0d;RX0:P00=2hZ03@0;R/0;R/003@Z02X0=3@^02X0=B/0:bh0:`0d:Rh0=00^
:P0d02Xe:P00:P0d;P0d03@^:c@0;R/^02X0:`0^:P0d=00d;RX0=B/0;RX0;P0d
;P0[=Bh0:P0Z02h[;P0[;P0d;Wl[02h[;P0d;R/^:`0d02X0:P0^:`0^:bh0:cD0
:P0Z03@^02/e02/^03@0=BX0:P0d03@0=2h0:P0d;R/^03DZ02/e02/d;P00=00e
02X0=00Z03@0=2h0:`0d;RX0=00Z;P0e02X^02X0=B/^03@0:bh[03@0=00d02h[
000d;QD[;R/^:`0Z03@0:P0^:`0d02h[>SX00AUo>P0d02h0:P0^:P0d000^:P00
=00^=00^03@0:P0^=000:S@^03@003DZ03@002X0:SD0:bhe:P00:bh0:P00;S@0
:P0^=00Z;P0d02hZ03D0;R/e03@0:SDd000d02hd02h0=00^:P00:R/0:P0d02X0
:P0d02h[=@0d02/0:P1o;S@0;P0[;P0^03@0;S@0:P0d02hd02h0:bh0=00d000Z
03@0;R/^02hd02X0:P0^;S@0;RX0=000:P0e02/Z000Z03@0;P0[:P0^:c@0:bhd
02hd02h0:SD[;R/002hd000d:c@003@0=00Z03@0:SD0;S@0:Rhd02X0;S@0:P0E
;P0^02hd02/^=00d02h[;S@0;SXj004HOcX^03@0=00d02h[;R/d000Z;P0d02X^
:bh0=2h[02X0=00^03@^:P0Z03@^:S@0=00Z03D[02h[=Bh0=00d:P0^000d02X0
=00d;R/003@[;P0^:bh0=00Z;R/e02X0;RX0=2h[02h[=@0e:P0d;P00=00d;P0[
02X0:SD0=00dB@0Z02/e02/Z03@^:`0^:`0d;R/0;R/0=2h0=2hZ03@Z02X0=00Z
03@0;RX0=@0Z02/^:P00=2h[=00Z03@^03@d=B/^02/e;P0]oR18;CD0:P0[;RX0
:P0^02/e:P0Z02X^02h0:RX0=2h003@^02/^:`0^;P00:Rh0=00Z;P0Z03@0=00^
03@0;RX0=00e02X003Xj004GOcX0:P0^:P0e:P0d000^:bhd02/^=00[000[;P0^
=00[;RX0=00^03@002h[02h0;S@0=B/^=000:P0[;R/^02X0:S@0:SD0;P0Z02he
:P0^03@[000Z;S@003D0:P0e:`00:P0^=00d02/^:`0^03@d02X^02/^=00[;P0d
02h^=S@0:S@0:SD0:bh0;S@0=00^02hd03@[;R/0:P00:Rh0=@0Z02hd02X0:P0d
03@0:bh003@04aCm:5XQ8@0d02h0;P0d02X0;PLP>20P;C<0:SD003@0=00d02X0
=000;P0d02Xd000^02Xe:P0[;R/0=B/0:S@002/^=000=00d02X0=00d;P0Z000^
=00[;P0Z>SX00AEo>Rh0=00^:`0^:bh[=@0Z02h[;P0^:cDZ03@0:P0^03D002<0
:P0d;RX003@^03@0;RX0;P0Z03@0=2h0;P0d03@0;RX0:c@0=3D[02X0=00Z03DZ
000[;R/^:bX0=00Z;R/0=00^:bh[;P0^:`0Z;P0d000d;P0^:cD0:Rh[02/g;P00
=BX0:P0e:`0Z03@^:P0Z02h0;P0[=BX0=2h0:cD[;P0d02hh>CPh83PP>2d?;Bd0
:Q@D4aCl:?hQ2P0Z03@^:`0d00L7oR00>Oh]@b/e:P0d;P0Z03DZ03@d03@^02h0
=3@0:P0Z03@^02h[;P0d02hZ03D[;R/d;P0Z;P0Z;R/^03@0=2X0:Rh0:cD0>SX0
0A=o>P0d02h[02X0;P0d03@0=00d03D[02h0:Rhd03@0:bhd:P0[;P0Z000d:bh0
=00^:`0Z02/^=00^:P0d03@0:P0e:`00;P0^;R/0;R/0;S@0;RX0=2/^02h[=@00
;RX002hd02X0:P0e02X0=B/^03@0:SDZ02/^=00d000^:bhY03@0:P0Z03@0;S@0
;RX003@0=00d03@0:P00;R/0;R/^03@0:QDi>CPioCP7>Bd?82dP3`on5?/X03;n
8E8b:cD0;RX01`LP>20i;C<];2h0;P0Z03D[02hZ02hZ03@0=00^:P0d000^:`0[
=@0[;P0Z03@0;R/0;P0d02/e:`0^02/^=00^=000=00[;SXj004FOcXZ03@0=BX0
=00d;RX0=@0Z03@[;P0[=@00:Rh0=2h0;P0d;P0d;R/e03@0:Rh[03DZ03@^02X0
:`0d;R/^03@^:bh[=00d02/^02/^:`0Z02/0=3D^02X0;R/e:P00=2X0;P00=00d
03@0=00d02X^000Z03@^02hZ03@Z03D[9Rh0:P0d03@^:`0^:`00:SD0:Rh[;R/e
;P0d=00^:`0^03@^01DE;Ohi023n>0HP>BdP;Bd6oRd251<Do2P9<R8Q8C8Q8@00
:_h7B20h83T];C<]:P0d02X0=BX003@0;R/^:bh0:`0Z=BX003@e;R/0=00d03@^
:`0e:P0Z;P0^:`0d03@^02X002X^:bh0=00j>P015Glj02hZ02X0=00Z;P00:P0Z
;S@0;P0Z=@0Z=000:bh0:R/0;R/0:P00:bh^:bh0=00Z000^02X0:SD0:P0^02X^
03@002h[;R/^02Xe02hd02Xe02h[02hd02/0:P0^=2h0=00Z=00^;R/^=00Z;S@0
02Xe:`0^02h[02hZ000Z02<0:S@0;P0^:`0^:cD0:SD0:P00=B/0;R/0:P0e:`0e
:`0Z00_n5@h?>3Ti3cTP>B0i82dP3`om;@@C52PX5?hX0b4b8B;n8@0boPL4820h
>CWl;Ch^=00^:P00:Rhd02X0=@0[;S@0;P0d=00Z000Z;S@0:Rh0=00Z03@003@0
=00^:bh0:P0d:bh0=00d02hj>P013G@j;P00=2h0;R/0:cDZ02X002X^02X0:P00
:SDZ000d03DZ03DZ03@^02/0=B/^03@0=2X0=00e02X^03@0=00[;P0d=00e02X0
:P0[;P0^02/^:`0^:`0^:c@^02X0;P0d;P0^03@0:P0^02X002X^02/P80oj83X^
:`0d=@0^:`0d02h[02X0=00Z03@Z02X0=00e:P0Z03@Z03@Z01XO;ADE;B0h>B0P
>B0i8206;B0]8?`]01Cj:08b8C;n8@LD<PL7820h>Oh]03?n;C`Z000d=00^02X0
=00Z;P0^:`0d;P0Z03@^000Z;P00=2hZ02X0:Rh0=2hZ03D[02h0;P0[=@0Z;R/0
>SX00@ed>P0[;RX0=00^=00Z03@0;S@003@0=00d=00Z02hd=00Z000Z000Z03@0
;R/^03@0:SD0:bh0:S@003@0=00Z=@0Z02hd02X0=00^=00d02Xe03@0:P0e:`0^
02X0=2/0:P0d02/^=00Z02X0;S@002X;8206nB0[3b0?82dP=B/^=00^=000;RX0
=00^=000;P0Z03D0:Rh0:bh02alK3aD77Rgn>@hP83Ph1ST?3bd6;@lc;C?n;OhX
01Cm:0dQ8R4b<0db1`Lh83Ti<o/]>c<d02hd03@002h[;P0d02h[;P0Z02/^02Xd
000^=00Z000e=00d02X002X0;S@0:R/e=00Z02/0;SXj004>FCX^02/0=2h[02h0
03@^02/^:cD0:Rh0;P0d03@0;Rh[03@d02Xe03@^:`0^:bh0:P0Z03D[;P0Z;R/^
;P0Z02X0=BX0;R/0=2h0:bh[;R/0:P0e:P0d03@^<O`X6b40=2h[;P0Z000d03@0
:Rh05@@/810P83TP83Sl800]oR0Y3b00=00Z=BX002X^02/^=2X0=00d:bh003@?
;Bd;2al]81DE;@li1R0Po3P880l?82dV<`lcoBd2:2TDoBP?8C8D538Q<PL783Pi
>Bd]<od]>Rh[02hZ03@d02h0=2h[03D0=2h0=B/^02X0=00Z;P0d:P0^;P0d;R/e
02X0;R/0=B/^;P0d;P0[>SX00@mI>P0^=2/^02hd03@[;P0Z=B/0;R/^03@0:Rhd
02h[03@0;Rhd02X0:P0^=00^03@0:P0e:bh0=000=000:P0d000Z02X0=00^02Xe
02h0;S@0;RX0=00Z;Q@`nRPD8@0e02X0:Rhd02h[;P0d2ah?3b06oR00>?hP0@L]
oR0/;Ah?1Rd?80l[02hd000Z=@0Z000^:P0^;C<c;BdP;Bd;2b0?;ADE82di>@lP
oSP683T]82dP3o/]1B4Q<RTX5?dX2a@C538R8@L783Pi>Oh]?C<];B/0;R/0;RX0
=B/0:P00:bh[;R/0:P0[=@0Z;S@0:`0^02X0:`0Z02h0=00Z02hd02X002Xe02/^
>SX00@aM>Rh0;P0[;P0Z;P0e:bh0:P0d02X0:P0d000Z;P0^:bh[000^:`0d=B/0
:P0[=2h003@[;P0d;R/e02X[02X^:cD0:P0d;RX0:P0[;P0Z02hh>CTh>Bd051<D
4bPX5?dX524Q;P0^02/^:`0e:bh070/N;B0P>?hP0SPP>?`P30H?1PlP1PlP82dP
3`ok;@0bo3?n;@0?oRdS2`/]820M5BdP;CTi80li83TP>BdP;@li;Bdc;Bdc;B4Q
<RTDoRP952PC8B4b8@LP8?hioBdk<cD^02X0:`0Z;P0d;R/e;P0Z03DZ03@^02X0
02Xe03@0=2h0=2X0=00Z03@0=00^:`0d:P0Z03D0>SX00@AG>P0d02hd03@002h[
;P0d000^=00d02X0=B/^02/[=@00:S@0=B/^02hd03@0;P0d:bh^02/^02X0=00^
=3D002/^=00Z000d000Z03@0=00i>CPi>@l];OhD01?i:2<R8C4^=00e02X002X7
5@/];CPP83PP83PP1PL]820?80@P1Pol80<?1R0?oRgk<o`]9R0?;Bd;7bd]3aDP
;@li>@lP83TP>B0i;B0]83T];C<];C<c;B8Q<_/X1b4b8B8b1b0hoSWm;CX[03@0
;S@0;P0Z=000:P0d000Z000^02X0;S@002X^:bh0:SD002hd02X0;R/e:`0^;P0d
03@0:SXj004AECXZ03@0;R/^:c@0;P0Z03@Z02h0:P0e:P0^03@^02X0=2h0=00^
02X0;R/^:`0d;P0[03@^02X0=2h0:bh0=3D0:cD^03@^03@d=B/^:bh[>3Ti>3Ti
oBd551@C:2PDnbP<<R8Q;R/d02X001`E7_h]0cPP83Sm804h1ohP1bdN2`lP82dE
oR003oh]:20c<bdc<bdc;Bdc;B0];0l]7`/?82d?5B0];CTi82dP>B0i820]82di
oBeF<bd]<c<]8C8Q:38Q<R8Q<R4Q<PLP>3Ti;C<];Bh[;P0[;R/0=00^:cD0=2h[
=@0d=00Z03@0:Rh[000d02X0:P0d02X^03@0;R/^02/0=2h^03@0>SX00@YB>P00
;S@0:P0e03@0=00[;P0Z02Xe02X0:R/0:P0d02Xe02/^:P0d02X0:P0e02X0;S@0
;R/e:`0^02Xe:`0Z02X^02/0:P0Z02h0;R/e000h>CSn>@LP;Bd?;BdD4oPX6R4b
8B40;S@[;STL5Ah?;B0h820h83PP80L6;OhP01Gm80DE80l63`Km;O`c0BdcoRd2
3b`?oRdG7bd]80lM;@l6;CT]820i83TP>BdP;@lioRdE<bdc<bd]<b4R<R4R8C8Q
8C8b3C878?hioBdi:P0^=00e:bh0:P0Z02X0;R/d02hd000^=00[03D[;S@0=@0Z
;S@0:`0^=00^03@d02h[03@0;SXj004<EcXZ02X0=00Z02X^:bh0=00d03@0:P00
=00e:P0d;P00:P0e:`0d;P00=00d:Rh0:P0^:`0^:`0d;R/0;P0d;R/0:cD^:`0d
;R/0:P0^:bh[>CPi>3Ti823n;@H?;BdD:2PDnRP:<R4Q<P0e:`L72akn;@TP>20P
>20P>207oR03;AlE3ohP12dP3b0?oRd1<bgm<o/]00om;APO;@H]80lM3`lP;BH?
82di83TP>Bd63cTioRd9<bd]<c<];B4Q<_hQ238R8C8b8C878?hioRdi03@0;R/0
=B/e03@0=00Z02h0:P0Z=BX0;P0d=B/^02X0:P00:P0d;R/0:P0[=@0Z;P0^:bh[
03Xj004>FSX^=000;S@0:P00;P0Z;S@0;Rhd03@d02hZ000^03@Z03@0;R/^02Xd
02hZ000d03@0;S@0=00^:`0e:`0Z02hd02X0=00^02hd03@002/e03Pi>CPi>20?
;23m;@0CoRP05?XX02;n8@@b1`Li2od]0B0hoR0I>20P1`L?820]2ah]80lE1R0]
82d]<bdc<bgm<`HR<S<c;Bd?oRd:80l?;B0E;B0]83Sm81Pi83TP>@l]83Tc;B0]
<bdc;C<]8B8Q<R8boR458R4b8R4hoCTk;C<[;S@0=00Z02X0:P0^:P0d02/^=00Z
03@0:bh[;P0Z03@0=C@0:SD002Xe02X0:Rh0=00^:`0^>SX00@A8>P0Z02X0;R/0
:S@0=000:Rh002X^02h[000d=00Z03DZ02X0=B/^02X0:`0Z;P0d;R/0;R/^:`0e
:P0Z=00^02h0:cD0:P0d0?hQ32h[;P0Z>CPi>CPi823n;@0?oRd0<_HXoACn1`LE
7Rd?;BdP>?hP13PP80L7oR087P/P;@HE3b0PoRgi<`Lb<c<R<S<R<_d]:B0]82dP
5BdP;@li;@H?83TP>B0i;B0i9S<i<bdc;C<];B4Q<R4b8B8Q<_hQ0c8Q<R3m>CX]
000Z;P0d;P00=2h[02h0=2h002X003@^:cD0;P0Z03@^:`0Z02h0=2X0:cD0:P00
=2h[03DZ03Xj004:FCXd02/^=00^=00^=00^:P00:R/^03D[02Xe:P0Z=@0Z000Z
02/^03@0=00^=B/0:P0^=00Z03D0:P0Z03Dd03@0:bh[;S@0:P0Q8C8Q8@0Z03Pi
>CPi>B0?8?`]20lR8C8Y52PX5?`XoQ@0<_h701om;ATh820h820h80L7820]2aD;
820E5B0P;Bdc;OTc>c8c<c8R<S8c8S8P;@H]82dP82d?1Rdi3b0P;CTh>0Hi;@li
53Ph<bdc;Bdc<S8Q8R4Q<S8Q8C8R8C8R8O`i>CDZ02/0:P0[;RX0;S@0:Rh0=3@0
:bhZ03D0:P0Z:`0^02h[=C@0:Rh0=00^=00d02X0=00Z02hj>P013E`j;P0e02X0
:cD^02X0:c@^000d02X0=BX0:P0e02X002Xe03@^02X0:Rh0:`0e:P0Z02h[03@^
:`0d02X0;R/^:cD0;P0Z02hZ<R8Q<R4Q<R/i>3Ti>3Ti820];B3n;@<b8B8QmbPV
51@Q<PL77Rd]3bd]>20P>20P>2071b0P;AhN3b0E5B0];C<]<c<]oC<7<S<c8S<b
<R;n<P0RoS89;B0];ADP;@lP;OdP0BdioSP182gn>@8h>3?m;BPQ8C8Q8S8Q8C8R
8B4b8B4b8CTV>CT0;P0d=@0d;P00=00^:`0[=@0Z;_h08S@[;P0d03DZ02X0;P0Z
000Z02h[02X^03@^02h[03@0>SX00@QF>P0[;S@0=00Z02X0=00^02/Z=@0Z02X0
=@0d02X0=3@0:P0Z03@0:bh0=C@0:P0d000[=@0^:`0Z03D0:P0Z03D0:P0d03@[
038Q8B8Q<R4Q>3Ti>3TioB3l;@DQ<R4b8C;n:00Do2P6538R8@L77_`]0cPP83Sm
80l71b0P6QdN820E3bdc;C<]nS<08_hc5C8R<S8R<S<R<R8b8S8P82dP82dP>Bgm
808h>3Wn800ioCP1;C?n;@HR8C8Q<PdboR53<R4b8B8Q<STi;Bd0:bh[;RX0=B/^
=00d:bh0:Rh0=C@[;Rh0:bh0:P0d02X0:P0Z=00d02hd02/0:P0d02hd02hj>P01
3E<j;P0[;P0d;P0Z03@^03@^000Z03@^02X0:P0Z03@^;P0[=@0d;P0d03@[;Rh0
=2hZ03D[:P0e:P0d03@0=00Z03@0=00Z=@0Z8C8b8C8=<S8Q>CSm>@0]oR03;BdP
;?h]124b8B8QnBP?8C8Q8@L;;BdP;BdP>20P>?hP0@L7oR092aD;;Ad]<bdc;O`c
0S8b8_db3R8c<R8b8S8b8S8c<R8b<_h]0@H]oR093`HP>3PP>B0?>Odh023n>A8]
8BPQ<R4b8S8Q<Pdb8C8Q<R8QoRdh=Bh0;P00=2/^02h0;P0[=@0[:P0^02/0=2h0
=00d03@0=00e02X^03@0;Rh0=2h0:P0[;R/0>SX00@]B>P0^=00[;P0^=00^02/^
03@d02/^03@0=000;S@0;P0d02h[;P0Z;S@0;P0[02X003@0;S@0:P00;RX0;Rhd
02/^:P0Z;P0Z038Q8R4Q<R4Q<R7n>@8P>CWn8?h]1@l]82d];?h]3CTX<S8Q<R4b
8B4R<R4Vnbd183Sn80lh82071b0?5ADN81Dc;C<Rnc<9<R8b8S8Q8S8Q8_hb228b
<R8b<S<c8_hc3c8?82d?1PlP3cPh83T680oi>A4Q8S8Q:38Q8C8Q<R4b8R4b8B7n
;CPZ02X0:S@0;P0Z03@0=00^:bh0=00Z=B/^:`0Z02hZ02he:P0d000[;S@0=00Z
03@0:SD[02hj>P0133lj:P0^:cD0:P00=2h[=@0d;Rh0=00d;RX0:SD0:P0[:P0d
02X0:P00:Rh0=2h[03@^;R/0;R/0=3@0;R/002X^oP0@:P00=00Z<S8Q<R8b8B8Q
<R7l>@<]8206oRd23cTcoRd:<bdi<R8b8S8R8C;n8@Db8Ah];@on;@<P>20hoR3n
1ad62`/E2`L]<c8b<c<b8S<c<S8R<R8Q8C8Q<R8b8C;m<`8R<c;l<`@R<c<P3odP
oSP483P/>CSk>Ddc8B4b8R4b:28b8B8Q8C8Q<S8c3bd0:P0Z02h[03@0:Rh[;R/0
=B/^03@0:P00=3D0=00^:`0[;RX0:S@e02X0:SD0:P0d02h0=00j>P014V@j03@0
=00Z03D[;R/0:P0Z000[;S@0:P00;R/d02/^02h^=00d02hd000Z=000;S@0:`0Z
02hd02hd000Z=000;S@d03D[;S@0028b8C8Q<R4Q8R4Q>CTP>CTP80lP3bd]83T]
<oh]3cTV<R8b<S4R8C8Q8R4b8B3m;@<?;B0hoR01>23n1`X?5@/E2c<]<c<R<_hc
528b8S8b8C8;7cTi40db8C8R<S<c<_hc02;n<`0boC?n80T?>B0i>0lc>CTho3T@
<R8Q<R4Q<R4X<R4b8B8Q8B;n;CPZ03@0=B/0:P0e:`0e03@0:P00:P0d02hd02h[
;RX0;R/^000d02h0:P0d000Z02h^=00d02Xj>P013UDj;P0d02X0=2/^03DZ03@^
:cD^02X003@d02h0=2h0=00[;P0Z02/^:cD0;RX0=2h0=2h0=00^:P0^:cD0:Rh[
02h^:bh0:bhb:38R<R8Q8C8b8C8Q8Odi02gn80D/3bd]>B3m;A4i>C8b8S8c8C8Q
<R4Q8R4];B3n;@0?oR00>?hPoPL680/E2bdc<_dc2c8c<S<c<R4Q<PL;2odi320i
<c8R8C<c<S<b<c;n<`0bnc<31Pli>?h]0c<i>CSm>D`b<R8b8B8Q<R4Q<RPb8C8Q
<Rd]3`0Z03@0=2h0:P0d:P0d;R/0=2h0:Rh0:bh[02h003@0=B/^02X0:bh0=2hZ
000d02/^02X^03Xj004;FCX0:P0^:P0^=000:P00:P00:P0Z02Xd02hZ02X^02/^
:bh0=00Z=@0^:cD[02/^02/^03@0;R/0;R/0;S@002hd:`0^:`0^52PD:38R<R4b
8R4b8B8b8CTP>O`P72d]3b0h;Bdc;CTi8S8c8S8b8S8=<R4b8C8b;B`PoRd6>20P
>20h8?h72PlE7Rdc<R8b<S<RoS<7<R8b>@H71aon>@<P>B0hoBd08Odc128c<S<R
nC<0>?d]0S<]>?`iC2Hb<R8b8C8Q<R8b8S8b:28Q;@00=000;RX0;S@0:bh0;RX0
02hd03@002Xe02hd02X[;S@0:P0d02Xe02X0;P0d:bhd02hd000^>SX00@9B>Rh0
=000=00^:P00=3@e:P00=00e02X^000d02/e02/^03@^000Z02/^02Xd000d;P0d
;R/0=BX0;R/0:Rh[02h0=00d:P0C52PD<S8R<R4b8B;n8@Db8STh>@om80@];CTh
>?h]oSWn<P0RoC878S8Q8R4R<c;l;@0ioB00>?d75@/];C8R<S8R<c<b<c<Q<R`;
>A`i;Qon>@<P>B0hnbd0:?Lc02gm<`0?oBd3<cTi>?di138R<S8RoR53<R4b3C<b
0240:Rh0=2X003@0:Rh0:bh003@Z02hZ03@^02/Z02h[03D0:P0^:bh002X0=2h0
:P0e02X0:bh[=B/j>P0144`j03@0:SDZ02X0=2/^:P00=B/^02Xd000^:bh[;P0Z
=@0[;P0Z=00Z=@0d000^=2h[02X0=00Z03@0;S@002hd02/^:bh0;Q@D<2Pb8_hb
1B8b8B4R<_hQ0CTioR033`H];Ohh4R0];CT5>B8b8S8R<R8b8B4b<B;m<P<c;B0]
oR00>?hPoPL37Rd]8_hb3S<c<S<c8C<c3ahh>@Li7odi1B0i82d]<od]238c:28c
<S<c;Odc12dc;Bd?oS<1;CSl>D/V<S8R<S8R<R4R8C8Q<S@0;RX0=00^02X^=000
:SD0:bhZ03D[02hd02/^03@0;S@0:P0^:cD0=3@0:bh0=00d02X0;S@002X0>SX0
0@=K>Rh0:P0Z03@0=Bh002h[;R/^03@0;RX0:cD0;R/^02X0=00d02h[02X0:SDZ
02h0=BX0:P0d03@0=00Z;R/0=BX003@0:P0C51@C:38R<R8b<R4b8B4b8R4b1Pon
80P?82d]>CPh;Bgn>@4b8_hb5R8b8S8R<S8Q<R8b8S8R<Rd]83PP>20PoPOn;@Db
8S8R<c;k<`HP5@/h1`LOo3T283TPnRd4<c8c<bSh<`L]82`];C<]<o/ioS978S8b
8B4b8C8R8C8Z000d;RX0=00^03@^02X0=00^:`0d;P0^:bh0=2h0:bh[03@0;R/^
03@^02/d02X^:`0d02hZ000d>SX00@eW>P0d03@002h0:P0^=2/0:bh0=00^:`0^
=00d02X0=000:P0^=00^=00d02/0;R/0:P0Z02he:P0Z;R/^03@0:P0e:bhd02hD
51<D:38c8S8b8S8b8B8b8B4b<BH]80lP1R0]83Ph82gk>@`V;@lP:38b8R4b8S8R
oS838S8R3odP0SPP1oh]0S<c<_hc02cl<`Pi;@/N>@Li71on>@<P>B0hoBd282d]
oS<0<_/c0Rdc<oh]20lc<bdc;C<i>?hiBRDb8S8R<S8R8C8Q8C8Z02Xd02X002hd
02hZ03@0:bhd03D[;P0Z:cD0:bh0:SD0=00^=00^02/^02Xe02X^03D0;S@[03D[
;SXj004:FcX^;R/^:c@0=00d02h0=2h0=2hZ02h[02hZ03@0=2hZ=00d02h[02h^
03@^02/d;P0d03@0:cD0;P0Z03@^:`0d03D0:P0Z51<D4a@X<S8R<S8R<S8Q8C8R
8C8QoR043`lP;CWn>00]ncTM82H?;B0680lQ8BPb<c8R<S8c<R8h83PP83P7;Bd?
oS<0<_/c3Rdc>BdN5@/71cT_>CTV>Ohh023j;@Tc<b8c<S<c<R8XoS?l;@Hc;C<]
>3ThoSU9;B4b<R8b<R8Q<R4R03D0:P00=2X0;R/003@^03@0:P0Z000d02h[03@0
:P0Z02h[02X0:c@^03@0:P00:bX0=00^03@Z000j>P010f@j:`0d000e:P0Z;S@0
:bh0:bh[03D[03D[02/^02h[02h0;RX0;SD[02X^03@0;P0[;P0^:bh0:P0d000Z
03D0;RX0:P0d03TD4a@D<38R<S8R<R8b8B8b8B4R8C8?1Pl6820i>?Hi4RdP3b0h
82d?8B8X8S8b8S8R<S;k;@D?;Bdc<b;m<`0]oC<5>@lN6QhioPOm>OdhoBd282d]
oS<0<_dc4C8Q<S8c8S8Q<S<]<bdc;CPi4?hiB2d/<R4R<S8c3C8Q=00Z02hd:P0Z
02hd:bh0:Rhd03@0=B/^=000:Rhd03@0=00^=00d02h[02h[;P0Z=@0[;R/^=00^
02hj>P012fXj;P0^:P0Z03@0;P0d;P0d;P0^:`0d:P0e:P0[=@0Z02/e02/Z02X0
=00Z03@^03@^03@0;P0Z03@^:cD0:P0d02X0;P0Z02hD51<C538b8S8b8S8R<R4Q
<S8Q<Pd]<bdc;C<]>CPi>3Ti9_/i30l]>20P;Bd?80lb:2;n<P`R<Rdc;Bdc;2d?
;C<bnC<7;Bdi>AhN70_n1`0Oo@Om>OhP02gn81db<c<R<S<R<c<Q8R4b8C8=<R4]
<bdc;CPi>3Ti;C?n;D<Q<R4c8C8^:`0d02h0=00Z02h0:cD002X^02X0=00^:cD0
02X^03@^:`0^:P0d03DZ02h0=00Z03D0:P0Z02/d03Xj0049DCX0=2/0=00e02X0
;RX0:bh0:P0^=00^02X0;S@0=00Z=@0Z=@0d000^=00[;R/0:P0[;S@0=00[;P0Z
03@0:bhd03@0:SDd03TC51@C51lO:2;n<Q4R<S8R8C8Q<S<]<bdc;C<h>CSn>@05
nSWm80`i;Bd?1Pl6<RPb8S8bnbd;82d]<bdc;C<c;C<coRd91STE5@/N1`LE2od7
1B0i>3Ti3o`P62d?3C8b8C8b8C8Q<R4b8S8Q<R4b<bdc;CSm>Oh]A2h0:P0^0240
;S@0=B/0;S@0=00d02/^:bh0=00Z03D[02X0:P00:bh0;S@002Xe02X0=00[;S@0
:P0d02X0;S@0;SXj004>I3X^03DZ02X0=2h[000d;P0Z03@0:P0[=00d02hZ03@0
:P00:P0Z;R/0=BX003@e03@^02X0:Rh0:cD0:Rh0=00^:P0Z000Z;STD4a@D4`/;
7alb:3<b8S8b8R4b8Bdc;C<]<bdcoST0>?Xi0RHi3odP0Bd]oR093bd]8S8b8Rd]
<o/]0b`c;C;n<`l];C<];CTh2`/E2aD;7QD?oPL683Ti>3Ti;OhPoRd682dR8B4R
<_hQ3B8Q8C8Q8R4b8C<]<cThoSU7;Bdc02X0:P0d02X0;RX0;R/0;Rh0=2hZ02/0
03@^02X^:bh0=3@^03@^02X0:P0d02X^03@^:bh0;P0d:P0e03@0;RX0>SX00@U>
>R/0:P0d02hZ000^=2X0;S@0:P0^=00^02h[;P0d02X0=2/^=000=00Z03DZ02X0
;R/0;S@002hd03@002X^02X0=00^=000>A<D51<D5?h;1C<c8BPb8_hb30db<bdc
;C<]<bdi>CSj>@/5>CTV820i820?1Pon8?h]0R8b<o/]0C<]oC<0;?dcoRd2>3PN
oST4;@/E2b3m1`@P>3T6>O`P00On;F0?;38X8B4R<R4Q<R4b8C8Q8S8Q;C<i>CPi
>Bd]:P0d02X0=00^:`0^:bhe:`0[;R/0=@0Z=00^=000;P0d02h0;RX0=00d02X^
=00[02X0=@0d02/^02X0:P0d:`0^>SX00@U>>Rh[03@^02/0=2X0;P0Z000d;R/0
;R/0=00e:bh0:P0e02X0;RX0=BX0:P0d;P0d02X0:bh[=B/^:P0Z000d:P0^03@0
:Rh003TC51@C5?h;1S8c<b4b8BSn<PhR<bdc;C<]<bdc>CTh>CSi>@0hoR06>2d?
1PlP3oT]1S<];C<]<bgm<`4]<o`]0SP];Ohi1B0E2aD]3oh72R0h>CTh>@lP;B0P
oBeO3b4b8C8Q8C8R8B4R8B4b8B4b8C8c>CTh>Bd]03@^02hZ03@0:P0[02/^03@^
02h[;P0e:P0[;R/d03@^:`0d000d;P0^000Z03@^000d:bh0=2h0=2h0:Rh0;R/0
>SX00@lg>P0d02h0:SD[;P0d02hd02Xd000^:cDd02h[;P0[;P0^:P0d03@0:P0Z
02he:`0^=00Z=@0d03Cn01<^=2/^02h0:Rhd02/0:STD51<D4oh;6S<R<c<b8S8Q
8RPQ;C<]<bd]<bdc>CPi>3Th2?/i0CPho20@3`H?1R0];@l];C<c;Bdc;Bgm<`<]
<c8coBd3<`l];Ohi1@l]2ah]8?h70cPP>3Sn>OhPI0H?;@l];C8=<R4Q8R4b8C8Q
<R4R8C8Q8B8Q;CPi>3Tc02X0=B/0;Rhd03@0:SD[02X0:P0e:`0Z02/^:`0^02X0
;R/^:bhZ03@0=3@0;Rh0=2X0:P0^=00[;P0d000e:`0^>SX00@1;>Rh0=00Z02h0
:cD0:P0^:`0^:cD[02X^:`0^03@^03@0;P0Z02hd03D0:P0[;R/0;P0[;R/^:P0d
:Rh0;P0[=B/002X0=2h003Ti4ohD0a<;2c;n<a0R<b4b8C8Q;C<]<c<];C<]<ohi
0SPi>?Xi1SPh83PP83Sh80@c;C<c;O<coRd:3bdV>B0]7P/];B3n1`XP>3Th>@HP
3b0?1oh]538R<R4b8R4b8B8Q<R4b8C8=<S8Q<_dcARd]=@0Z02X0:`0Z;P0d02h0
=00d03@0=3@0=Bh0=BX0=00Z03D0:P00:Rh[;P0d02/e02hZ02X0;R/e02X^:bX0
=BX0>SX00@eA>R/0;RX0=00^=00d02X0;S@0=00^=000;S@[02X0=00Z02Xe02X0
=00d02/^02hd02/^:cD002hZ000d03@Z02hd=00Z=@0[;STi51<D4a@;7ohc03;n
<`4=<_hQ2c8Q;Bdc52dc;CTi>?`i1Bdi>3Th>OdP0ahP83Sm800]oR033bdP3o/]
0S<c;OdcoRdI3b`?>CT]82dE7R0P;@L7>20P>3Ti80HP80On;A@Q<R4b8B4b8C8a
8R4b8B4R<R4b8B;m<dH]<`0Z02hd02hd02/0;S@0=00^;S@0:Rh0;RX0=00[02hZ
03@0=00d:P00;R/0;R/^02X[03D0=00d03@003@002X0;SXj004:ESX^:`00=2h[
02hZ03@0=00^:P0Z02h[;P0^03@0=2hZ03@0:P00:RhZ03@^02X0;R/e000Z03@0
;R/e03@0;R/0;P0Z02X0=00^>CTD4a@D2`/b8S8c<_hc0S8Q<_dQ1Rdc;C<]<c?n
>@0hncT0>?hioR0?1cPP83PP7`lP82dP;@H]8?d]03;j<o`]4Pl]>CTP;B0;7Rd]
80L7820h>CSn>FD?1R07;B8b8B8Q<R4Q8S8b8B4R<R4Q<R8Q<S<c;C<c;BdZ03@0
;R/0=BX0=2h0=2h[02/^000Z02/0=2hZ03@0;P0Z;P0d=@0d:P0e:P0e:bh0=BX0
:P0Z=B/^:P0e:R/0=00j>P011E4j03D[;RX0;SD[02hZ02h[;P0d03@0=00d02/^
=00^03@0=00d:SD002/^03@0=00^02Xd02<0=B/0;R/^:P0^=00d03D0:P0Z03Ti
4a@D4a@;o3<C<S<b<R8Q<R8b8B4b;C<]<bdh>CSl>@X]>3Th>CTP>20P7_dP0@/6
oR031R0?3od]nc<1;C?i;@`?9RdP;Bd;;B0]3`L7oR3n>AHh80lP1bdb8C8Q<Pdb
8C8Q8C8R<S8Q<_hQ03;l<dHL;@0d03@[02h[=@0^:`0Z02h[;P0Z=00[;S@0:P00
;S@0=000:bh0;RX002X002X0:`0Z03@0:bh002X002X0;S@0;SXj004@H3X^02X0
03@0:P0d000Z03D[;P0Z;P0^:bh[=@0Z;P0Z=@0Z;P00:P0d;P0d;P0Z02/e02h[
02X0=BX0;P00=00Z;P0d03@^03D[>CTD4a@C50/];B4R<R8Q<R8Q<R4Q8S;n8@L]
<bdc;CPi>?di00on>@8h>CWn80`E83PP80/;3b0P>BdPoBgm<`0]oS?n;@<c;BdP
oRd<>@lP;B0E82dP;@L7>?hPoCTH8207;B8b8C8Q<R8b8B8b8C8R<R8b<R8b8_hc
B2dc>Bd^02Xe03@0:P0d03DZ02X0=@0d02h0=00^:P0d=00^:P0Z=@0Z02/0=3@^
:cD0=2h[03@^;P0[=2h0=3@0:P0^:`0j>P012e8j02X0;S@0:P0d02Xe=00[;P0[
;P0d02X0=@0[;P0d000Z000^=00[;R/0:P0d02Xd02X0=00^02X0:P0Z:bhd000Z
02hZ000Z03Ti1a@C51<O8?d]4C8Q<R4Q8R4b8B4R8C8Q;C<]<ohi03Sm>@8]>CSn
>@TP>20E5B0h820;oR05>20]82d]oS<0;O/c03;h<aLP;B0]82dP;B0P1b0P>20h
>CPh>B071c;n8@4R<_hQ2S8Q8C8Q8S8b8S8RoC9870LN700Z000^;S@0:P0^:P0d
03@0:bhd03@^:bh0=00^02X002X0=00Z=00^;P00:P0Z03D0;R/0=00^03@0;S@0
=B/0;SXj004>C3X^03@0:Rh0=2h0;R/^:`0[=Bh0:P0d03@[;P0[:P0d:P0d;P0Z
03D0=2h0:P0d02h[02Xe:P0d03@0=00e02X^:`0d02h[=00^03T7oA@2>B0?o2d9
8R4b8C8Q<R4Q<_hQ0bdc;C?n>@0hoST6;Bdi>3Th>OhP3ADE83PP2`lP83Ti;BdP
oBgj<`8R<bgn<a</<bdc<bdc82dP;B0];B07820h8?hi1SP?1Q`]<R;n8@Xb8R4Q
8S8Q<R4b8_hb128b8S8boQE42`0Z:`0[;P00=00^02X^03@^02h[;P0^03@^:P0d
03@Z03@^000e:P0[=2h0=2h[;R/0=BX0:P0d;RX0;R/0=2h[>SX00A1E>P0[;S@0
02/^03@[000^=00Z03@[02hZ02h0:SD0:bh0;RX0=00[;R/^02X0;Rhd03D[;P00
:P0^;S@0;R/d000^:bhd02he02X0>@LD4a@C>B06;@om;A8b<R4R8B4R<R4Q8S8Q
<S<]>3ThoCT;;Bdi>CPi>B0P>0LNoR062`/63cPi8?`]oS<0;Ohc0C8]oC<0;Odc
0bdc;C?n;@HP;B0?1b0hoR07>CPi>B07;B;n<PLR8C8Q<R4Q8_hQ238b8S8c8S8b
8_hEA0/Z=@0Z=B/0:Rhd03@0:`0Z02X0;P0d02/^:`00;R/^=@0^02/^:P0d02hd
02X002/e02X0:SD0:P00:P0d02h[03Xj0044C3X^02/^:cD^02X0=BX003@^02X0
=BX002X0=00Z03D[;P00=2h0=00d03@0=000:Rh0;P0d:P0Z000Z;P0^02X0=2h0
:P0[;R/0:ST7oA@2>CPPoBdE3bdQ8C8Q<R4b8B8b8B4b3C8c;CPi>?di5bd]>CTh
>CTP81hL5PlP7`/?1STh>@lP;?h]l3<3;C<]<oh]8C<];B0]>20P>20h>CTh>0L7
<S8R8B4b8B8Q<R4Q<R8b8B;m<TXR<S8R5AD;2`0d02h0=00Z03@^;P0d03@0=00d
;R/^02h[=00^02/d03@^000d;P0[;P00=2X0:P00=00Z02h[;P0Z02X0=BXj>P01
4d`j02Xe03@0:`0Z02h[02X^03@002X0=B/^=00Z03D[;P0[;RX0:P0Z=B/^=00^
:bh0=2/0:P0e02Xd000d02X0:bh0=000:P0^=@0i1ohD1Q<i>B0h;B3n;BDR<R4b
3C8Q8C8Q8C8Q<R8b<cTh>CTh>CT?;CPi>CPi>B075ADP8?h;220h>CTP;Bdc;Odc
02cm<`<]<c<]oS<9;C<c;C<]<c<];?`]0B0hoR01>CSn>@87<R;n<PHR8C8b8B4b
oR5@<R4Q8S8R<S8R<QDE2`/[;S@0:bhd000^:`0[;S@0;RX0:P0^03@0=00e:`0Z
02hZ02/^:P0[;P0d:P0e=000:Rhd03@[=@0d02Xe02X0>SX00@]6>Rh0:P0^:P0d
03@0=BX0:cD^:c@0:P0e02X0;R/^02/d000e:P0e02/^02h[03@0:P0e:P0Z03@0
;RX0:P0d=Bh0:P0a8C;m8@4b1odD0cTi>23l;AXb8B8Q<R4b8R4b8R4b8B4b8S<i
>3Ti>3T/3bgn>@<h>B0PoQD4820O2`on>@H]82d]<c<]nc<1<R;m<`L]<c8c;C<]
<o/]23<];@l]80lP;OdE6`L]<R8Q<S8R8B4b8R4b8B8Q<R4b8C8R<S8c<QGn2d40
;R/e02X^:`0e:P00:Rh003@^:`0d;R/^:P0d;P0d000d;P00=2h0=2h0:P0^:S@0
02h^02h[;P0d02X003@j>P0134@j02X0:S@002hd02hZ03@0;R/0;P0^=00Z03@0
;P0d02hd02X0:bh[;P0Z03@0;R/^02X0=00[;P0Z03@0=00Z02X08C;n8@@R<R4R
<_h7oSWn80`]82d?8B8Q<R4b8R4boR468R4b8B4b>O`c00on;@4i>?hi023n5@@N
80/;1_hi10l];C<]nS<0<_hc02;n<P/c;3<c;C<];C<c;C?l;@@P;B0];OdE=PL7
<R4Q8R4b8C8Q8C8=<R4b8C8Q8S8b8S8R<QDE2`/^:`0^:bh0=00Z03DZ000Z=BX0
02/^03Cn02H^:`0^:bhZ03D[;R/0:P0d03D[000^=00[=B/0:`0^=00d02hj>P01
3Tdj;P0d02h[=00^:P00=2h[03@^03@0:P0Z03@Z03@^02/^=00d;P0^02X0=2h[
03D[;P0d;P0d03@0:Rh0=2h0:P0d8B8b<R4b8C8Q8C;n1`@V>CPh8?d]4S8b8B4b
8B4b8B8Q<R4b8S8Q8C;l<`T];@l]>CTh>CTPoQD;1b0;2`@i>BH?;C<]oS<3;C<c
<_hc0c8c<R;m<PDR;C<]<c?k;@0coRd382dP;O`E0`LR<S;n8A0b3C8b8C8R8B4R
8C8Q8C8b8_hb01gn2d00=00^03@^:`0d03@^:`0Z03@e;P0Z02Xe:P0e:P0e:`0d
:bh0=BX0:P0Z03@^:`0Z02h[02Xd;P0^:bh[03Xj004ADSX0=00e:`0^=B/0:bhZ
03D0:P0d02X0=00Z;P0[;P0Z=@0^02h0:P0d02/^02hd03@0:bh0:SDZ02Xe02/^
02X0;R4Q<R4Q8S8Q8R4b3C871cTioR00>?h]328Q<S8=<R4R8C8b8C;n8@@R<R4c
;Ohc023m;@Di>CPi>B3n5@<L7`@KoST5;Bdc;C<]nc<98S8c<b8b<R8b8_db1Rdc
;C<];C?n;@Hc;B0]82d]oAEO1c8c8C8Q<R8b<Pdb8C8Q<R4b8B4b3B4b<R8b70/O
6P0^=2/0:P0d02h[;P0^=00d02X0=00d02X0:bh[02/^02hd000Z03D0=00Z;P0d
02hd02hd02h0:P0e:`0^>SX00@UP>Rh0:P0e:P0Z03@^000d:bh0=2hZ03D^000[
=@0[;P0Z02/d03@0=2h0=00Z02hZ03@^02X002h002X0=00d03@08C8Q8S8b8S8b
8R4b8C871cTi>20P;@l]<R4R8C8Q<_hQ228b8B8Q<R4Q<_/c023n;Ohi0SPi8?dE
0P/;5Ohi220c;C<]<S<c;Ohb03?m<PdR<S8R<R8b8S8b8S<]<oT]023n;O`EGPLc
8S8b3C8b8C8Q8R4b8B8Q<R8Q<R4R<R4R<P/;7c<0;P0d;P0^:P0^02X0;P0Z;P0Z
02Xe;P0d;P0^:`0[=00^:P0[=B/^03@0:P0e:P0^:`0^:`0d03@0=BX0>SX00AA5
>P0d02hZ03@002h[02X0;S@0:P00:P0[=B/^:bhd03@0=00^=00^:P0[;S@0=B/0
02h0:P0^=00d:P0d02h^=00Z<R8Q<_hQ02;n<PHQ<R8Q1`DioR0=>Bd]<S<Q<R4b
8R4b8C;n8@@b8C8b8_hc12dc;B`coRd4>CPi>3Wn5@`N2aD;83Ti83<]<c</oS<:
<R8b8S8R8B8b<b;n<P<R<S8RoS80<od]03?m;@<P;@l]oAEN1c8c<b8b8B4R8C8Q
8B8Q<R4Q<R4b8C8Q8C8b2alc;@0Z02X0=B/0=00d03@0:bh0=00d02X0:bh0:P0^
=00^=000=2h0:P0^;S@002X0:P0^=00d02hd02hZ02hj>P014e@j:P0d000d;RX0
:cDZ03@0;RX0=2h0=00Z000e:bh[;P0Z02X0=00^:bh0:P0Z03@d03@0=00Z02h0
=2hZ000Z;P0Q8C8Q8R4b<R8b8S8b8C8R1cThoR0O;Bdc8S<b8R4Q<Pdb8C8R8C8R
8B4b<R8b<R`c<bdc;C?m>@dh5@/N70/N5B0i>Bd]<_hc2Bdc<b8b8S8R<S;n8@db
8S8c<R8b<R8b8S<]<oh]0c<];B3m;@0MoAD08_db5B4b8C8Q<S8Q<R4R<R4R8C8Q
8S8Q8P_n;Cld03@^:bh0=2h[;P0Z03@0=2h^03@0:P0[=@0d02X^02X^:`0[;P0Z
02/^:cD0:P0d02X0=BX0:P0[03@0>SX00@iI>P0^:P0Z;R/0=00^02X0=B/002h[
02h[;P0Z:bh0;P0d02X002hd03@[02hd000Z02hZ02hd02hd02X002Xd000Z<R4R
8C8Q<R4Q<S8R<R8b8C8i820h80H]oC<X<R4b8C8b8R4Q<R4b8C8Q8S8R<S8c;C<]
;C<]83PP>B0N5@/N5@/P>CWn<`0]o3<1<R;m<PdQ8B8b8C8b8S8b8S8c8_db0B8b
nbd382d]3odE2@Lc<R8Q8R4b8B;n8@8R8C;n8DXb8B8Q8C8Q8Bd]<`0^=000:`0^
:cD0=00[;R/^02/0;S@0;S@0:bhd000d02/0;S@0=00^=000;S@0=00Z02hZ03@0
:bhd02hj>P014T<j;P00=000=2h0=00d;RX0:S@^03@Z03D0=00^02X0=2h0=3DZ
02h0;P0d02Xe:P0d02h[02h[02hd03@^02h[;P0Q<_hQ9C8=<S8R<S8R<R8b<R8P
>20h82d?<c<R<c<R<R4Q<R4b8B8Q8B8QoS8G8S8]<bdc<bdc>CPh83TP2aDE2aDP
83T]nc<8;C<b<R8b8S8RoR4I<Pdb<R8b<R8b<R8b<R8b<R8b<R`]3b`?;Bgl5ATb
8S8b8B8Q<R4b8R4Q<R4R<R4Q<R4b8B8Q8Oh]?bX0;R/e;R/0;R/^03@e02h0=2h[
02h[02hZ000Z=BX0=2h0=2h0:P0[;R/e:`0Z=Bh0:P00=2h^02/^:`0j>P0134dj
02/^=00Z;P0Z=00Z000Z02h0:bh002h[;S@0=2/0:P0d02X0:P0d:`0^=00Z03D0
:P0^=00^=00^:bh0=00^02XQ8C8Q<R8b<R4Q<R;m<P8R<S;m82di;C<b<c<b<c<b
8R4b8B8Q<R4b8C8Q8S<b<bdc;Bdc;C<h>3TP>1D;2aDN2b0io3<2;C<bnc?n8@<b
<R8bo24;<S8R<S8R<S8R<S8RoS843b`P;Bgm5E`c<S8Q<R4b8C8Q8C8Q8R4b8B4b
8R4Q<R4b8Bd]<`0e:`0^:`0Z02h0:Rh0=2/0:P00:P0^=B/0;S@0:P0^=B/0:P0^
=00Z03D0:P0e:`0Z03D[;R/0;S@002hj>P0154Dj;P0[;P00:cD0;RX0=2h0=00d
;P0d:P0^02X0;P0d;P0Z;P0Z03@^03@0;P0[=B/^02X0;R/0:P0[=@0Z03@0=00Q
8R4boR4;8S8Q8C8R<R8b<R8boR023bd6oS8Z8S8b8S8b8B8Q8C8Q8R4b8B8a<R8]
<bdc;C<];B0h>3TP>1D;2aDN3bHc<_`c22`b<c8b8S8Q<_hQ5C8Q<R8Q<R8Q<S8R
<S8R<S8R<S8R<R;n<`4?;OdEG0Lc;24Q<R4R8C8Q<R4Q<R4R<R4Q<R8Q<R4b<`00
:P0e:P0^:`0d03@0:Rh0=BX0=3D0:P0Z03@0:P00=00Z03@0:P0^:`0d02X0:P0^
:`0d03@0;R/0;R/d03Xj004FOcX0:SD0:S@0:P0Z000^03@0;S@0:bh0:P0d03@0
;S@0:bh0=000;P0[;S@0:bh0:P0d02X0;S@0:bh0=00[;R/^:P0Q<R4R8C8Q8B8b
8C8R<S8R<S8P80l680l?8R8b<R8b8R4Q<R4b8C8Q8B8Q8C8Q<S8]<bdc;C<c;CPh
83TP1@/E7QD;8?`c02gn<`lR<S8R<R4b8R4b8R4b8R4boR4?8R4Q<R8b<R8b8S8b
8S8c;Ohc6Qd?82d]<b4Q<R8Q8C8=<R4R<R4b8B4R8C8a8_hQ@P00:Rh0:P0d03@0
;R/^=000:bh[02h[=00d000^=00e:bh[;P0^=00d02hd02X^=00Z03@0;R/e:P0^
=00d02hj>P0155lj;P0Z03@^000d03@d03@^:P0^:c@0=00d;Rh0=00^:cD0=2h[
=00d;P0Z03@0=00Z03@0=00^:cD0:Rh0=00^000Z<R4Q<R4R8C8Q8C8Q<R8b<R8b
<RdP3bdP;Bdb<R;n<PdR<R4R8B4R<R4Q<R4R<_hQ4Rdc;C<]<bdP>CPh>B0;5@/E
82gk<a/]<S8R<S8R<R4b8B4b8B4b8B4b<R4b8C8=<S8RoS838S8b8_hcHRdc<b0]
82dc8C8Q8C8Q<S8Q8C8Q8R4b8B4b8S8Q<P0[:Rh0:c@0=2h[;R/0=@0Z03@^02h[
02h^:bh[=00^:P0^02X0=00^:P0d02h002/^02h[;R/0;P00=00Z;P0d03Xj004D
OcX0=00[;P0d:P0^:P0^:P00;R/0=B/^02h[02/^=2/0:P0Z000^:bh0:P0Z=BX0
:SD0;RX0=B/0;R/^02/^=00Z=00Q<R8Q<R4b8B4b8R4R<S8R<R8b1R0P1RdP<c8/
<S8R<S8Q8C8Q<R4Q<S8R8C8Q8C8R8C8c;C<]<cTh83TP3SPE2aD;83<c;C<c;C<R
<_hc728b8S8R<R8Q<R8Q8C8R8C8Q8R4b<R4b8C8R<R8bnS<E;@l];38b8R4b3C8Q
8B8b8C8Q8B8Q<_hQAP0[;RX002X0=B/^02h0;S@0:P0^:P0Z02he:`0^:`0^02X0
=00d02hd02X002hd02/Z=@0^:cD0;S@0:R/^:bh0=00^>SX00A9K>Rh0=2h0=2h0
=00^:`00=2X0;R/0=B/Z02h[;P0^03@0=2h[=@0Z03@0;P00:P0^:bX002X0=2h0
:P0d;P0^:`0^:b4Q<R4R8C8Q<R4Q<R4Q<S8R<S8P3b0]82gl<blb8S8R8B4b8R4b
8B4b8B4R<R4Q<R4b<bdc;C<h>B0i>20N2aD?;C8R;C<b;C8c8S?k<PHR<S8Q<R4R
oR498R4Q<R4R8B4b3Ohb228b<bdc<bdc8?h]Fc<Q<R4b<R8b8C8Q;2h0;P0^000^
03@0:P0d;P0d02h0:P0d02X0:cD003@0=00Z03@0=BX0=00d;R/^02/^000d:P0^
:cD0:P0[02X003@^03D0:P0d;RX0>SX00@i4>P0[;P0[;P0[;S@002Xe:P0d000Z
02h0=00Z03D[02h[;P0^:c@002hd02Xd03D[000^=00Z;P0d000^:`0[=@0Z024b
oR4m<R4b3C8Q8R4b8B8b<R8b3`H?;B0];C<c;3<c8C8Q<R4Q<R4Q8R4b8C8Q8C8Q
8R4R<S<]<cTP>B0h>0/E2b3h<b0b<R8b8S8R8B4b8B8Q<R4b8C8Q<R4R<S4R<R4b
8B8Q<S;k<`0PoReK<c<b<c<R<R4b8C<c03@0=00[:P0[;RX002h[02hZ03@0:bhd
02Xe03@[;R/^=2/0;R/^:`0^:bh0;P0Z=@0Z=3@0;R/e03@0:bhd03@[;P0^:c@0
02X002hj>P014E`j;P0[=@0Z03D[;R/e02X0=BX0=2h[03@^000d:bh[03D[;P0^
03@Z02h[02h[;P0d;P0Z000d:P0d=00e:P0Z000Z<R4R<R4R<R4b8C8Q<Pdb<R4R
<R8P80lP;B3k<b8Q8C8=<S8Q8R4Q<Pdb8B8b8B4b8C8Q<S<]<cTP>B0i80/E;Odc
02gn<`4b8_hb7R8b8C8Q8R4Q<R4R8C8Q<R4b8B4b8B8b8C8Q8B8Q<R7m<oh]G3<c
8S<c8C8R8B8]<bdZ03@^03D0=2h003@d03DZ000d;R/^02h0:bhZ03D0;P0^03@0
;P0d:P0^02X0:P0Z02h0;R/0;R/^03@^02h0=2h0:P0^03@d02X0:cXj004GDcX0
=00Z=00d02h[02hd000Z02X^02h[;P0Z=00^02he:bh0:P0Z;P0Z02hd03D[02X0
=00e=2h0:bh^02X0:P0Z=C@R8C8Q<R4Q8R4b8R4Q<S8Q8_hb1PH?82dP;C;n<`lb
<b4b8S8Q8R4Q<S8Q<R4boR4D8R4b8B8Q8R4]83TP>B0i5Bd]<c<]oC<0<_hcOb8b
<R8b8S8b8C8Q<R4b3C8Q8R4Q<R8b8B4b8B4b8C8Q8S8b<c<P;C<c<S<b8B4b8C8Q
<bdb000^02X[;RX0;R/^;P0Z02/^:P0e02X0:S@002X0:P0d02/^=00[;P0d03@0
=00d03@0:P0^=00^02X^03@0:Rh0=00d02X0=BX012/^>SX00@m4>Rh0=00^:P0d
02h[02X0=00e02/e02X0=00^:P0Z02X0:`0d;P0[=@0Z02h[;P0d;R/^:bh0:cD^
02/e02X0=@0Z03DRoR4m<R4Q<R4Q<R8Q8C8Q8B8b<R0]82d63c<b<b8c<R4Q<R4b
8C8Q8S8Q8S8Q<R4b8B4b8C8Q<R4i83TP>B0i;Bgi<`LR<c8b8S8b8_hQ2c8=<R4b
8C8Q<R4b8_hQ3S8R8C8R8C8Q8C8=<S8R3ohc138R8C8Rnc=@02X^03@0;P00=00d
02/e03@^000Z03@0=00^:cD0:P0d;R/^02X0=00d;R/^02X0;P0d;P0d02X0:c@0
:bh0=00[;P0d;R/0:P0^03@0>SX00@YN>P0^:P0Z000^=00d:bhd02hZ02X0=00d
03D[03@0=00d03@^02/^03@0=00d02/^02X0;P0Z=@0[02X0=00d02X0:P0Q<R4b
8R4b8B8Q<R4Q8R4Q<R4b8RdP;B0?3b;n<b<b8R4b8B8Q<R8Q<R4Q<R4b8B8Q<R8Q
<R4Q8R4Q;C<]<bdc;Bgn<`0]oS<0<_hc8B8b8C8Q<R4b8S8Q<Pdb8C8=<R4Q<R4b
8B4b8B4b8B8b8C;n;O8cD00^02/^=00d:bh[;S@0:P0Z03@0:P0Z02hd02h[=B/0
;R/e03@002h[;P0^02X0=00Z;P0d02X0:SD0:SD0=00^=00^:P0d02hd03@0;SXj
0045ISXZ000d03@Z02h[=@00:Rh0:cD0:Rh0:Rh0:P0Z02h[;P0Z03@^03@^02X^
03@^03@0=00Z02X0=2h0:Rh0;R/0=@0Z<R4R8B4b8C8Q<Pdb<R4b8R4b8C86;B0]
82d]<c<b<c8Q8C;n8APb8B4R<R4Q8R4Q<R4Q8R4b<R4b8C<]<bdcoRgi<adb<c8b
8S8Q8B8b8B4b8R4b8C8R<R4R8B4b3C<b8Rge<`0]nc=F;C<c;C<c;P0d;P0^:bh0
03@0;RX0=00d;RX0=00Z02h[02h[;R/0;R/^03@Z03D[;P0d02X^000[=2h003@0
:P00:Rh0=00Z02/0=2hZ02h[;P0[>SX00@E3>P0[;Rh[;P0Z02h0=B/^03@0:P0Z
03@002/^=00e:`0^=000;R/0:P0d02/0;R/0:P0^=00d02/^02/^03@0;S@0:P3n
8A4b<Pdb8B8Q<S8Q8C8Q8B8Q<R7k<bH]<b`c8B8b8B4R<R4R8C8Q8C8Q<R4R8C8b
8B8Q8B8Q;C<]<bdc;Bgm<`<]<c<RoC<D<S<R<S8R<S8R<S8Q<R8Q8C8Q<S8RoS;m
<`0]nS<0;O/c02gj<e0[02X0:P0Z03D[;R/^03@0;Rh[000^=00Z02he:`00;S@0
;R/0:SD0:bh0:bh[;P0d=00^:`0Z;S@0:S@002/^:P0d02hZ000^:cD0:bhj>P01
2UXj;P0[03D[;P0d02X0:P0d;P0[=@0d;RX0=@0Z;R/^02/^=2X0=BX0:Rh0=2X0
=BX0=00^:P0d;P0d;P0Z03@0:P0^:c8Q<R4b<R4Q<R4Q8S8Q8C8b8B4R8C<]oC<2
;C<coS8>3C8b8B4b8C8Q8S8Q8C8boB4?<R4b8C8Q<bdc;C<];C<c;Odc5C8c<c8R
<c<b<b8b<R8b<R8b8B4b8C;m8Ohc0b`c<bgm<`0]nS<2;3<]oC?m;E4^03@0=00Z
03@0=00^03@0:P00=3DZ02h0=2h0:P0d:P0^:`0e:P0Z02h0=2h0=B/^02X0;P0d
02h0=3D0:RX003DZ03@0;R/^02h[=Bh0>SX00?iH>P0d03D[;P0d02X0;S@0:P0d
02X0;R/0;R/d000^03@0:cD002X0=000:SD002X002X0:P00;R/0:P0d02/^:cDd
:`0Q<R4R8B4R<R4R<R4Q8S8Q8R4b8C;m<bL]<c<]<c8R<S8R<R4b8B4R8B4b3C8Q
8R4b8C8R8B4b8B8]<bdc;C<]mC<;<S<c<S8R<S8R<S4Rn3<0;OTc02gj<`D]<c8c
<S?n;@0coReA:`0Z02hd03D0:Rhd02/^:P0^=00Z03@0:bh0=00[;P0Z03@0:P0d
03@0:bh[02X0:`0^=00[;S@0:bh0:P00;S@[02h^=00e02/[02X0;SXj003oF3X^
03@Z000d;R/0=00^:`0d;Rh0=00^:`0e:bX0:cDZ;P0Z03@0=2h^:`0Z03@0=3D0
=00d=00e:P0d;P0e02X0;P0d8B8Q8C8Q<R4Q<R4b8C8Q8C8Q8R4bn3<Q8S8R<c8R
<R8b<R4b8B4b8B4b8B8b8B4b8R4b8C8R8Bdc;O`c1Bdc<S<R<_hc02;i<`4]<_dc
02ck<`0]nC<6<S8R<S8R<_hc0B`co2eC<bh0=2h[02h[;P00:P0d;P00=00Z;P0Z
03@^02X0=2h0=00d;P00=2h0=2X003@^03@^:`0e:P00=2h0:bh0=2h0;P0Z02/^
:`0d;P0d02X0>SX00@=K>P0Z;P0d02X0;RX0=B/0;Rh[02/^=2/0;RX0=C@0:P00
=00d02X^02/e02hd02h^:bhZ03DZ02h[02h0=00Z02X0;R/b<R4b8C8Q8R4Q<R8Q
<Pdb8B4b<R4c<bgn<`8]<c?l<QlQ8S8Q8C8=<R4b8S8Q8C8Q8C8Q<R4Q<R4Q<R4b
8C<c;O`c03;h<`0]nC<0;Odc02gl<`4];Ohc328b8B8Q<R8b<R8b;C?l;E@c;@0[
;P0^=00^02Xd02/^02h[;R/^03@0:Rh0=2/0;R/0;R/^02/^:P0Z03D0:SD0:bh0
;S@0=00Z;P0[;S@0:P0d03@0:SD002X^:`0^=00^>SX00@EH>Rh0:bh0=2h0:`0Z
03@^02/^03@0;P0d02h[02hZ03@^03@^;P0[;P0Z02/^:`0[;P00:P0^:`0e:P0Z
;R/0=00Z02hR<S8R8B4b8C8Q8C8Q8C8R<R4Q8S;i<al]8B8b8B4b8B4b8R4b8C8Q
8B8b:b4R<R8b8S8Q538D8_Xc02gn<`0/o3<0;O`c02gn<`0]o3<2;C<coBdC<bdb
<c8b8B4b8C8R<S8R<bdc;C?l;E<?02h0:P0^:`0d02h0=00d03@0=@0Z03D0:cD0
=2X0=BX0=B/^02/0=2hZ03@0:P0d03@0:Rh0=00Z03D0:P0e:P0d;P0[;R/e02/e
:P0^:`0j>P010F0j02hd02hZ02/^=00[;P0Z=@0d02X0=00Z:cD0:P00:P0^;R/0
:bh0=00d=000:SD0:S@0:P0d02X0=00[=@0^=00Z038b8S8Q<R8Q8C8R8C8R<R4Q
8S8Q8B8b8B8b;C<]oS<K<R8c<b4b8R4b8C8Q8R4b8B4b8B8b<S<b<c<R;O`c02gb
<`0]nc?n;@4/3oX]KC<]<c<Q<b4R<R4Q8R4b8B4]<bdc;C<];C<]82X0=00d03@0
:P0^=00Z;S@0;R/d000Z02X0:P0e02X002X0:cDd02h[000^:P0d02X^=00[02hd
02X0=00d:`0^:P0[;P0Z03@0:P0^:`0^>SX00?e?>RX0;R/003@^03DZ000d02X0
=2h0=2h^000Z;P0d=@0Z02/^03@0=2h0;P0d=@0Z000d;P0d;R/0=2h0;R/Z02h0
:bXb8S8b8S8b8B8Q8C;n8@0boR48<R4b8C8Q<c<]oS<4<S<c<R;n<PdQ8S8Q8R4b
8B8b8C8b8_Tc02gl<`0]oC<3;C<c;OXc0BdcoRd0<oH]o3<98C8Q8S8b8C8Q8_/b
F2d]820?800d;R/^:bh[03@0:Rh002X^02X0:SD0=00Z02X0=00d=00Z02X0=00e
:R/002X^000[;Rh0=00^:c@^02Xe02X003@^03@0:Rh003@0=2h0>SX00@5N>P00
:cD0:SD[02X0:bhZ02hd02X0;R/0;S@0:`0Z02X0:SD0=00^:P0d02hZ02X0;S@0
:P0Z000^;P0d:`00:P0Z=@0R<S8R<S8R<S8Q<R4b8B8Q<R4b8S8Q8A@]<bgm<`TR
<c8R<S8Q8R4boR44<R4Q<R7h<`0]o3<0;O`c02gi<`L];C<]<bd]<o/]60l];@lc
;Bdc;C<b8S8Q8C8Q8C8Q8R4b<R;m<odPDbd0:P0^03@002h^=000;R/^03@0=00^
:bhd02hd02hZ03DZ03D0;RX0:P0^=C@002Xe=000:bhZ03D0:P0^:`0^=00Z02/^
=000=B/^:bh0:SXj003gDCX^:bh0=00Z03@0=2h003@0;RX0=00^:`0d;P0d;P0[
=@0Z03@^000d;R/003@0=2h0=00d=BX0:`0d;P0d;P0d02h0<S8R<R8b<R8b8R4Q
8R7n<oP]o3<8<S8R<R8b8C8QmC<0;Ohc02gn<`0]o3<0;O/c2bdc<bdc<bdc;C<]
<od]03?l;@4P;?`]33<Q<R4R<R4Q8R4b8C;m<`8/<c?l858Z02X0=2h[=B/0:Rh[
03D[;P0d;R/0;P0Z02/^:`0^:P00=2X003DZ02X0:P0Z=00Z03@^02/0=00d03@0
;RX0;R/0=2h0:Rh[03@0=B/003Xj003cACX[=@0[;RX0=00^=C@0:S@0:P0d03D[
02X0=00Z02Xe03@0:bh[02X^:`0Z=BX0:P0^;S@0;P0[;S@0:P0^:`0^=00d02;n
<`LR<c<Q8C8Q<_hcnbd2<bd]oC<0;O@c0Rdc;?dc02cn<`0]nc<1<Rgi<`0]nC<3
;C<]<oh]03?i;A@?820];@lQ8R4b8B4R8C8Q8C8Q8S;l<eLP82dP3`0d02/^02X0
;S@003@0:P0[02X0;S@0:P0Z=@0^=B/002X^03@0:P0e02X0;S@0;P0Z=@0[;S@0
;RX0;Rh[02X0;SD^03@0:`0Z;P0Z02Xe>SX00?i6>P0Z03D002X^02/0;R/0;RX0
=@0Z03@^02X0:P0d02X^03@^02h[000d;P0Z03@0:P00:Rh0=2h0:Rh0=00e:P0^
:bh0<S;n<`Tb8C8Q8S8c<bdcnBgn<`0]o3<0;O`c02ck<`<]<c<]oS?n;@@c;Bdc
;Ohc02gn<`0]o3<;;C<c;C<]<bdc;BdcnRd183Wm8?h]438b8B8Q<R4b8B8b8B4b
3C8boS?l858Z03D^02X0=00^:P0e:P0d;R/0=00^:`0e02X0:`0d;R/0:P0Z=@0Z
03@0=00^02/^02X0=@0Z;P00=00[;P0[=00Z03@^03DZ000d03@^03Xj003bA3X0
;S@0:bhd02/^:P0d02h0=00Z02X^03@0:SD0;S@0:`0Z03D[03D[;P0d000^=00Z
=000:bh0=000;S@0:P0Z02h0:_hc1R4b8C8=<R7j<`0]nc<0;O`c02gl<`0]oC<0
;O`c0Bdco2d5<bdc<bd]kc<3;C<]<o/]00on;O`PKcP]3bdR8C8Q8C8Q8R4b8C8R
8C8b8B8b<b0]820]000Z03@0=00e:`0[;R/0;P0^=00Z02hd02X0:S@0:P0^=@0d
02/^:`0Z02hd02Xd03@002Xd000Z=00Z=@0Z=@0e:`0^:`0Z03D[;R/^02Xj>P00
nDhj:P0^:cD0;R/e000Z;R/0=2hZ03@0:P0e:P0[=@0Z;P0d02X0=2X0=00d;RX0
:cD0;P0Z:bh0=2h[;P0Z000d03@0=00b8R4b8B4R<R4boC<0;Nlc02gl<`8]<c;g
;@dc;Bdc;C<c;C<c;C<c;Ohc0bdc<bgn<`@]<c<]<oX]2B0P;Bdi820h823n;@Hb
8B8Q<R4boR4:8R4Q<R4R8B4b8Pom>E8Z;P0Z03@^:P0^03@0=BX0=00^:P0d02h[
03@0;RX0=00Z03DZ02/e:P0d02h[02h[;R/d02hZ000Z;P0[;P0Z02X0=00e:P0Z
03D0:P0Z03Xj003o43X0:P0^:c@0;P0^=2/0=@0ZoP0e;S@0;RX002X0=00[02hd
02X0;S@0;RX002/^02X0=00e:`0Z02h0=00d:Rhd02X0;R4b8B4bnS<0;O`c0bdc
<bgn<`X]<c<]<c<]<c<]<oh]03?g;@Pc;Bdc<bdc;Bgi<`0]nS<4;C<];C?n;@Pc
;Bdc1Rd]82gl81P]82d]8C8Q<R4R8B4R<R4Q<b8c<S<c8S8coSUC8000=00^:P00
:P0d=00Z02X0=B/002hd03@0;SD[000^=00^:P0d02h0=00^:`0^:cD0;SD0:P0d
=000;S@0=00[;P0Z02X0=00Z03@0:P0^>SX00?93>Rh0=00^03@[;P0Z03@^:`0Z
=BX0;R/003@^02X^03@Z02h0=2h0:P0[03@e;P0Z03@^:P0e:P0Z03@^02h002X^
03Cj<`0]lC<5<R8Q<S<boC<1;C?j;@<c;Bdco2d4<bdc;C?n;@8c<bgi<`0]oS<3
;C<]<oP]0@lPoBdF>B0P>2d/3bdR8C8Q<R4b8B4b<R4b<R;m<ePb<cTP>CTZ02X0
:`0d;P0d;P0d03@0:P0d;R/0:Rh0=00Z03@d02h[000Z;P0[=@0Z03DZ02h[02/^
:`0e02X^:`0^:bh0=00d;P00=2h0:P0Z03@^03Xj0042@cX0:bhd03@0;P0d02hZ
03Dd02/0:P0^=3@0;S@002X^03@0:bh0:P0^=00^:P0d03@0:P00:P0d000Z03@0
:S@002X0mc<0;Ohc2Rdc<bdc8S8b8C8RoS81<b;m<`@]<bd]<oh]03?k;@0cnBd1
<bgl<`0]oS<0;O`c4Rdc;C<]<bd]<bd]<bd]3bd]82gm86l?>CT?8B8Q<R4Q8R4b
<Pdb8B4b<c8b<c<R<STi>3Td03Dd02h0:bh0:P0^;S@0:Rh0;S@002/^=00Z;P0Z
02h[;S@0:bh[;S@0:P0e:`0Z=@0d02hZ000^=00d03@0:SD0:bhZ03@0:SD0;P0^
>SX00?Q9>Rh0:bhZ02X0=2h0:`0d02hd;P00=00^;R/0:SD[000Z;P0d03@0=00^
:P00=2hZ03@0=2h0=2h[=@0Z;P00=2h[02Xc<bdc<bce<`Tb8B8Q<R8b<R8bnS?l
;@0coRd6<bd]<bd]3oh]00om;@8c<bgg<oh]0S<]<od]03?l;@4c1_d]6B0i820V
>CT]<R4Q8S8Q8C8Q8C8R<R4Q<R8boC?n>E8h;P0Z02X0=2h0:P0d000Z000[=00^
:P0e:`0d02h0:cD0=00^03@0;P0Z000d:bh0:bh[;R/0=2h[02X^03@^;P0Z03@0
;R/^000Z:`0d03Xj0041A3X[02h[03D[02X0;S@0;RX0;P0Z=00^:`0^=00Z02Xe
=00[;S@0;Rhd000[;RX002X^02h[02h[02h[;P0d02X0;S@0;Ohc0BdcnBgn<a0b
8C8Q<R8b<R8b8S<]<c<]<oh]03?g;@@P3`H]3od]0S<c;OXc0bdc<bgk<`@]<bdc
<oh]03?n;@0?oRd23b0PoCU]9RdQ<R4b8C8Q8R4b8S8b8C8R8C8Q<c<b>CPi>00d
02Xe02X0=00d02h[=C@d02hd03@0:P0^=C@0:S@0:bhd02/^=00Z03@d02h0:SD0
:P0e02X0;S@002X0:P0d02/^=00^03@d02hd02hj>P010TPj;R/0=BX0=BX0:P0^
:`00=00d;P0d02X0:bh003@0:P0d;P0Z02/0:Rh0=000=2h0:c@0=BX0;R/0=00d
;RX0=00^:S8R<S<cnBdD<c<]8B8Q8C8Q<R8Q<R4b8B4b8R4boBd5<bd]<bd]oB02
3b0?oBd7;2d]<bdc;Bgd<`H]<bdc;C<coRd0<oh]10lP;@H?ncT73b8Q<R4R8C;n
8F4b8B8b8S8b8S8Q<b8c>CPi:P0d02X0=BX0=2hZ02h0;P0d02X^03@0:P0[;P0^
03@^02h[=@0^:`0d;P0Z03@0:P00=2X003@0:SD[03@0:P0d;P0Z02/e02X0:bh[
03Xj004?@SX0=@0Z000Z03@0=00d02X^:bh0:bhd02Xe03@0;RX002h0:P0d02hd
02/^:bhZ03@0;P0Z02X0=B/^02X003@0;R_n<_hcoBd2<bd]oS<L8B4b8C8Q8R4Q
<R4b8C8Q8S8c<Rd]<bd]3bdP3bgn80D?80HP;B3l;@8?<bgm<`<]<c<]oS<<;C<c
;C<]<bdc;Bdc<od]6R`P;B0?;CTV>CT5>Bdc8B8Q8C8Q8C8R8B4b8_dbFB8b8C8b
>3Ti>00e:P0e:`0Z;P00:P0Z02/^:P0[02hZ02/^03@[02h[02/^02X0;S@0;P0Z
02hd02/^:P0e:bhd02X0:P0Z=@0^02X0;S@0=00Z=B/0;SXj003k@cXZ03@0=2X0
:P0d;P0Z=000=00d;P0Z000Z03@d02h[=B/^03@^:`0^:c@0=@00=2hZ03@0=00d
03@0:P0d;P0d02hRoS<4;C<];C?n;Ohc4bdQ<R4R8C8Q<R4R8B8b8S<b<c<Rnbd1
82gj800?mbd1<bgf<`P];C<]<bdc;C?l;@Lc;C<?82dP;O/i40lb<b4b8B8b8B4b
8B8Q<R4RoS9H8S8R<R8i>CP^000d:bh002/d;P0d03@^000d;R/0=2h0:P0^03@^
03@^02X0=00^:P0d03@0;P0d000d:bh0;P00=3D^02/d03@0=00^:P0d02h0=00j
>P0104<j03@0:Rh0=00^;P0[;P0e:bh[;P0Z03Dd02X0;S@0;R/^02/^02he:`0e
02X0:Rh[000Z02hd02hZ02hd02hd02h[;O/coBd><c<h<b4b3C8Q8R4b8B8QoS87
<c8c<c8c;C?m;@<P;B0]nb3m;@0PoBgm<`0]mc<;<S<c;C<]<bdc<bdcoRd63b`]
82d]3o`iK20c<c8R<R4Q<R8Q<R4b8B4b8S8R<R8b<S<i>B00:RX0;P0Z=00^03@0
;RX002X^03@0;P0d03@0:bh[02X0=00d02h[02X0;Rhd02X0=B/^;P0d02Xd;P0[
02X0;Rhd02X0;P0e:P0d02Xj>P013DDj;P0d02X0;R/0:cD^02X0=@0Z02X0:P0Z
03DZ02h[03@0=2h0:P0Z03@0:P0d000d=BX0=00^:P00=00^:P0Z02/Q8C8boS?m
;@0coR05<c8R<R4boR4;<R4b3C8R<S<c8S<boRd5<b`]3b0]oR031PlP8?h]0B0P
o2d1<bgm<`4]<_`c228c<S<b<c<]<od]03?m;@0?oBdG9ST5>BHP<c8c<R8b8B4b
8B8Q8C8R8C8RoS828S8boSU@;P00:P0d02hZ02X0=00^:c@003@^:P0d;P0Z=Bh0
;R/0=2h0:P0[=Bh0=00[;R/0:P0d02X0;R/0;P0d;R/0=00Z03@^:`0Z02h[;P00
>SX00?mU>P0Z;S@0=00^=00Z03@0:P0Z03@0=00[;P0Z02h[02he:bh0=00d000^
;S@0;SD[;RX002h^:`00:Rhd000Z02Xb8C8b3C8R<R0]3bdP83PP8S8Q8C8Q8C8b
3C8Q<S8Q8S8Q<c8cnRd582dP;B0?oR04;BH?1Pon;@8c<bgf<`0boC<0;?XcnRd3
82d]3o`i0R0Q8_hc0S8R<_hQH38b8R4b8C8b8S8R<S<i>CPh:S@0:P0e:`00;S@0
:P0e02Xe:P00:bh0:bh0:`0d02X^02/^=00Z02/^:bh0;S@0:P0Z03@0;SD[02X0
02h^=000;P0^=00Z02h0:SDj>P012Bhj;P00;RX0=00^:`0d;P0[=@0d;P0^:cD0
=00Z03DZ02/^02X0=2h[=00Z03@0:_h0<c@d02/e:P00:P0d;P0d038R<S<c<S<R
3b0]82dh83PQ8C8Q8S8Q8R4b8B8Q8C8Q8C8R<R;k;@<?;@H]oB053b0i>@lPoRd2
<bd]oS<0;OXc0Rdc<_`c02gm<`@]<c<]<o`]0R0/3odiK3Pb8B4b8C8Q8S8Q<R4Q
<R4Q8R4Q<R8b8S8c>CTP02h[02X^03@d02hZ03@0:P0Z03@e;P0d;P0d;R/^03@0
=2h0;R/0=2h0:P0d02h0=2h0=2h[02X0=2h[=B/0:Rh[02X0;R/0=00d02Xj>P01
044j02Xd000Z03D[02hZ03@0:P0Z02h[=@0[;S@002X0:bh0=00d02X0;S@002hd
000Z=BX0=00Z03@d02hZ03@08B7l<`Db<b0]82gm814b<Pdb8B4b8C8Q<R4b<R8b
<R;n<P<c;C<coRd482dP;@on1ohi0R0P<o`]lC<0;O`c02gn<`D]<bd]<bcm;AT?
>BHi>B0]8C8Q<Pdb8B4b8B8Q8B8Q<R4b8OhbE28b>CPh:P0^=C@[02X0;R/002h[
;S@0=@0Z02/^:`0Z03D0=00[;P0Z03D[;P0d000^=00d;P0[;P0^=00[;P0Z02hd
000^=00d03@0;R/^=00j>P012G4j;P0Z03@^:P0^:`00:Rh0;R/0=00Z03@^02X0
=00d;P0Z02X^000d02X^:`0Z;R/0:P0Z=Bh0:P0^:`0^:bhQ<R8Q<S8R<c<b;B0]
82dP80lR8C8Q<R8Q<R4R8C8Q8S8b8S8b8S8]<bd]<c<]820?80Kn1ohi0BdPoBd3
<c<];OTc128c<R8bmc<<;C<]<c<]<`l];@l];Oh75B0D;B`b8C8Q<R8Q<R4b8C8Q
8B8Q<Pgn<U<R>CPh52h0:P0^=00d03@^:`0e02X0:P00=2h0=BX0:P0d;Rh0:cD0
:P0e:bh0=2X0;R/^03@^02X0;R/e03@0:P0^:bX0;R/^02Xd02h0;R/j>P0154@j
02X0=BX003@0=00Z=@0d03@0;S@0:bh0:P0d;R/^03@0:P00=B/^:P00;S@003Dd
000^000Z03@0;S@0;R/b8B4b8B;l<bL]82dP;B0P8C8Q8R4b8B4b8B8Q<R4Q8S8b
8S8R<bdc<bd]<b0?80H?oPOn>@T?82`?;Bdc;C<]oS<<;C<R<c8c<b8b<R8b<_hc
32dc<bdc;C<];C<];23n;@0?oPM/>20]<bd/<R8Q8C8Q8R4b8B8b8C8Q<S8Q8S8i
>20P02X0=00^02h[;P0^=00Z03@0=B/^:`0Z03@002X0:`0Z03@0:P0^:`0Z02h[
=@0[;P0Z02h[=@0[;R/^02/e02h[=@0[;P0d02X0;SXj004@DcX^02X003@^03@^
:`0Z03@^;R/0;Rh0:cD0=2h0;P0d;P0d:P0Z000d;R/0=2X0:Rh[=00Z02h0:P0[
;P0Q<R4b<R8b8C8Q8S<c82dP;B0P;C8Q<_hQ2c8=<S8Q8B8Q<R4b8_hb2bdc;C<c
;C<680lP8?h7oST880lP83<];C<]o3<0<_hc1c8c<R8b<S<RoS<0<_hc1bdc;C<c
;C<?o2gm1`0PoRdE<bdQ<R4Q<R4R8C8Q8C8=<R4R8B4b<_hPCRh0:P0d03@^02/d
02h[03@^:P0e02X003@^:S@0=2h0=BX0=00d03@^03@0:P0d;P0Z03@0;R/^000d
03@^02X0;R/^000Z;P0[;P0j>P012Bhj03@0=00Z02hZ02hd02/^:`0^=B/0=00Z
02X0=B/0:P0[;P0Z03@0:P0e:bh0:_h02Bh[;S@0=00Z=@3m8APR<S8R<S8R<S8c
;B0]82dP824R<R4R<R8boR4A<R4b<R4R8C8b8S<]<bd]<bd?oR0B3b071cPh9Rdi
820];C<]<bdc;O`c138c8S8bo3<0<_/c12dc;Bdco2d182gm1`0PnbdB8C8R8C8Q
<R4Q<R8b8C8Q<R4R<_hPCP0d000^:bh[02X0=B/0;RX002X0=00d:bh[000^;P0d
:`0^:P0^:bh0:bhd000Z03@0=00e:`0[02Xe02X0:P0d02h0=3@002Xd02hj>P01
2e/j;P0d;RX0=000=00e:P00=2h0:P0d;P0[=@0Z03@Z03@^03@0:Rh[03@0=B/e
03@d;P0^02X0=2h0:R4b8S8Q8B8b<R8b<R8b<RdP;B0?;B0c8B4b8B4b8C8R<_hQ
3R8Q8C8R<S8/<bdc;C<]<o`P0@L7oST23cTPoRgi<`@b<R8b<_hc0R8c<_hc02;m
<`H]<bdc<bdcnbd03oh70b0];C?n;F@Q8C8Q8R4Q<R8Q8C8R8C8Q<R4R83PP;P0d
:P0e02Xd02X0=2h003@^03@^;P0e:bh0=00[=@0Z02/0=00e:bh0:bh[=@0Z03@^
:P0^=2h[02X0=2h0=2h[03@^;R/e02h[03Xj004CACX0:bh[02hd02X0:P0^=00^
:c@002X0:SD0:P0Z=@00;P0d02hd02hd02hZ02h[;RX0=00d02/^02X08B4b8B4b
8B8b<R;n<QPR<bdP;@l6;C<c<R4Q<R4R8C8Q8B8b8C8boR4A<S<]<bdc;C<]82dP
82d71cThoST682d]<bdc;Odc7S8b8S8b8R4R8C8c8S<c<S<c<S<c<S<b;Bdc;C<]
;B3n;Od703Sm;@Lc<R8Q8C8Q<_hQFS8Q8C8Q8R4b8CPP800[;P0Z02X0;SD0:bh0
:bhZ02/^:`0^:bh0=00^=00Z02Xe02hZ:cD0:bh0=00d02hZ000Z02h0;S@0:bh[
02h[02/^:`00:bh[02hj>P012e0j;P00=2h0;R/d;P0Z02h[02h0=2X0=00Z000d
02X^:`0d;P0[=Bh0:P0[03@0;P00=2h[;P0d03@08C8Q8B8b8C8Q8C8b8S8R<S8]
82dP;B3m<a4b3C8Q<R4R8B4b8B4R<R4R8C;n<`<]<bdcnb051cTi>3TioBgm<`H]
<c<R<S8RoS8C8C8Q<S8Q8C<c<S<R<c<b8S8c;C?l;@4P;?d703Wl;F@Q8C8R8C8=
<S8Q8R4b8B4b8B4b8B0P;P0[=@0d03@0:Rh0:cD0:`0d;P0^:`0Z03@^:P0^:`0d
02X002h0:P0d03@^:P0d02h[;P0d03@0;P0d03DZ03DZ000d=2h0:c@^03Xj004:
ISX[02X^03@[02h0=00d02hd03@0;P0^=00d:P0^:`0e:bh0:bh[02X0;S@0;S@0
:bhZ02h0:Rhd02hQ8B8b8B4R8B4R<R8b8S8R<R0]1RdP3c<c;3<c<S8Q8R4Q<S8Q
8R4b8B4b8C;l<`4]<ohP2RdP3`Li>0Li>B0]mc<><R8b<R8b8B8Q8B8Q<R8QoC<0
8_hc138c<Rdco2d03od7JCP];C<]:38Q8C8Q<R8Q<R4b8B4b8B8b3C8P800^=00[
;RX0=000:SD0:SD[;P0Z02hd000^:`0^:`0^=C@0:S@0=00d02X^000Z03@0=00[
;R/^=00Z=@0Z000Z03D[;Rh0=00^02Xj>P0134Hj;R/003@^03@0=2h0:P0[;RX0
=00d02h^02/e:bh[03D[;P00=2h0=00^:P0Z03@003@0=00^02X08C8Q8C8Q8C8Q
<R4b8Ohb1R8?82dP;B3k<`<Q<R4boR4:<R4b8B8b8B8Q<bgk<`dP80H?1Q`h1`Li
820];OTc4C8R<R8b<R4b8C8Q<R4Q<R4R8OhcNc8c<b8c<c8c;Bd?;@l/1`Li3cTh
;Bd;<R8b8B8X8B4b3C8Q8S8Q<R4b<R4P82/0;RX002hZ03@^02X002X0:cD0=00Z
=2h0;R/0=2h0:bh0=BX0:Rh0=00Z03@^;P0Z03@^02h0:P0[;R/0=2X0=00e:`0d
;P0[=@0j>P014d8j03@[;RX0=00Z;P0d02Xe:`00;R/^;R/0:SD0=000;R/^03@[
;P0Z03D[03@0:P0e:bh[;S@0:P0Z<R4Q8R4b8B8boR4=<R8b8S8P;@l]82dc<bgm
<`XQ<R4b8S8Q8B8Q<_hQ03;n<`0]oC<582d?;@l?oPL3>CTP;Odc1Bdc;3<b8_hb
428Q8R4b8B8Q<R8Q<R4b8C8boC=g<S<R<c<];C<]<bdh>B0i>Bd]5C<c8S8Q<R4b
8C8Q<R4Q8R4b8B4R800^=000:SD[02hZ03@0:bhd02h[=00d02h0=00d:bh0=00Z
02X0=000:Rhd000^:`0^=00^02X0:P0e=000:SD0:P0^:P0Z02hd02/^>SX00@mC
>Rh0=@00:Rh002X0:P00:P0d:P0e:`0^:`0Z;P0d:P0^:bh0=00d;R/0=2h0=2h[
;P0[;P0^:`0d024Q<S8Q8R4Q<R8b8B4b8S8b;@H?82dP;C8Ro3<18B;n8@Pb8C8Q
8C8R8C;l<`P]<c<P;B0P;B3m1`8i;Bgj<gl]<S8R<S<b8C8Q8R4Q<R4b8B8Q8C8Q
8S8c<S<c<S<c<R`c;C<]<cTi3cTi82d;7S8c8S8Q<Pdb8C8=<S8Q8B8Q<R4b:`0Z
;P00:P0d02h[;P0d02h[03D[;P0d03@0:P0e:bh0=00d02X^03@0;P0d:P0^:`0^
:`0d03@0=00Z;Q8[02X0=2h003@^:`0^:`0[>SX00@Xl>P0d03@[;P0d=00Z03Dd
02/^=@0Z03@0;S@0:`0Z02hd000Z=@0Z000Z;P0[;P0^02Xe02X0;S@0;R4b8_hQ
0C8boR4?<R8Q8C8R82dP;B0];C<c<_hc3S8R8C8Q<R4R8C8R<R4Q8_Tc1BdP82dP
;Od70ST];OTc1S8R<R8b8C;n8@<b:24RoR49<R4R8B4b8B8c8_`c1bdc;C<]>3TP
oSUV<ahN<c8c<c8R<R4b8S8b8Bh08C8Q<P0^:bh0=3@0=00^:cD0:bhe:`0^:bh0
:Rh[;RX0;R/^03@0;RX0;P0[;S@0;RX0=00^=00d02X0;S@0:Rh0=C@002h0=00Z
000^:cD0;SXj004BD3X^02X^02/e02X0=BX0:Rh0:`0d02X^:`0^:cDZ03@0;R/e
02X^03@^02/e;P0Z02X0:bh0=00^:`0Q8C8Q<R4R8C8Q8R4Q<R4Q<R8?;B0]8?h]
33<c<R8c8C8Q<Pdb8C;n8@Hb<R4b8S<]oS<6;C<];B0]8?d70SP];OXc:C8R<c<b
3C8R<R4b8C8Q<R4b8B4b8C8Q8R4b8C8c<R8c;C<]<bdc>CPP8?hi2AD;<S<c<S<b
8S;n8EPR<R/^02XR02X0;P0d;P0d;R/0;R/^02/^03@0;P0Z000e02/e02X0=2hZ
000d03@^02h[000Z03@0:Rh0=2h[02hZ000d02Xe:P0d;RX0=2X0;R/e03Xj004:
BcX0:P00:SD0:P0^:P0d03@0;R/^:P00;S@002X002hd02h[=00[02X0=00Z03@0
=00[;P0d02h[02hQ8R4b8R4b8B4b8B4b8C8R8C;g;Ohc4C8Q8B8b8C8Q8R4b8R4Q
8S8c<_`c1BdP82dP;Od70Rd/<_dc03;n<a/Q<S8E>ADE>@Hi83TP>2`X<R4b3C8Q
<R4Q<R4QoS<7;C<]<bdc>B3m>@`h7Ahc<b8c<S<Q8R4boR5E02X0=C@0=00[;P0Z
;P0^=00[02hd02/^=00d02h[;R/^:c@002h[03D[;R/^02X0;S@d02h^=00[02X0
;SD[02h[;RX0:P0[;R/002h0=00^02Xj>P0144Hj;P0d;P0Z02h[000d;P0^:`0e
02/e:P0^:cD0=2X0;R/0;P0d;R/0=2h0:bX0=2h0=B/Z03@0=2h0;S4b3C8b8B8b
8C8Q8_hQ1C8Q;B0/3oT]7C<Q<R8a<R8Q<R4b8B4b<R4b<R8b<bdc<b0];@H]8?d7
22d]<S<c8S<c<_hc;C8U2CPh5ADi>@HP>CP];B`];C8Q<R4R<R4R<R4b;C<]<bdc
;CPP53Pi>3T;5C;m<edb<b4b8C8Q8@00=00^:P0d;P0Z02/^02X0=BX0;R/e02X^
03@0=00^02h0=2X0;R/0=B/^03@0:P0^03@0:bh0=BX0:P0Z03@0;P00=00d;P0^
:c@0=2h[03@0>SX00A<m>R/0:P0Z03@0=2/^:P0d03D[;S@0:P0e:`0^:bh0:P0^
=00Z;P0d02h0:SD0:bh0:P0d000^:bh0:P0b8S;n8@db8B4R8C8Q<S8=<Rd]3o`]
023m;A@b8R4b8C8Q8C8Q8S8Q8B8b8S8b8S;n<`@]80l?;O`73BdR<c8c<S<c;3<b
83TEoST45CTh>CWn>7l]<bd]83TX8C8Q<R4Q8S8Q<S<]<bdc>B0c>CPi>B0;7S<b
<b8c<S<Q<Pdb8@0Z;R/^02/^03@0:bh0;S@0:P0d02X0=000:bhd02h[=B/0:P0^
:`0^:bh0:bhd02X0:Rhd02X0:P0e02X0=00e:`0Z=00^02X0:SD0:P0^=00^>P4j
004>C3X^:`0d03@^02h0:`0Z;R/0=B/^02/^:`0d03@0=00Z02h[02/e;R/0=00Z
03D[;P0d;RX0:cD0=00d8B4b8B8b8B8Q8C8Q8R4b<R8boRd08?`]80l];B0b<Pdb
8C8R8C8Q8C8Q<R8b8S8c<R8c<b0]82dP;Oh71Bdb<R8c<_dc0S8/3ohi1SPi5CTi
>23m>?h]73PP>B8b<b4b8C8Q8R4b8C8c;CTc;C<i>CPi7Q/bnS=I8C8Q8SD0:bh0
=2h0:P0d;P0Z02h[03@^000Z;R/d;P0Z02/0:P0d;P0[=2h0;P0d;P0^000d000Z
03@e02X0=00d;RX0=BX0;RX0=00d02X^02X0:P0[>SX00@lo>P0^=@0Z;P0d03@0
:SD002/^:`0^=000:Rhd02h^=00d02hd02h[02hZ02hd02h[02X002Xe02X0:S<b
8B8Q<_hQ2C8b8R4Q<R4b<B;k;@0ioRd93b`P<c8c8B8Q<_hQ0c8Q8R7n<Q8R<R8b
<S<]82dP;B071cP?8S8bnS<;>20i>ADi>3TE>3TioSP08?h]0R0i8?dcMR8b:38Q
<Pdb8B8b<bdc;CTh>CTP13P63b0R<c8R<b4b8@0Z=@0[;P0d02/^03@0:P0e:bh0
=2X0:`0^02X0:bhd02hZ03@0;P0d02/^02X0=3@0;S@0:Rh0=00[;S@0:P00:P0d
000[;P0^=000=00d02/^>SX00@a1>Rh0:P00:P0d;P0d02X0:P0[;P0[;R/e02/^
:`00:Rh0:P0^:`0e:P00=00Z02/e:P0d;P0Z02/e51@X<R4b8B8boB48<R4R<R4b
8Bd?nbdC82`?;B0c<c8c8C8Q8S8Q8R4Q<R;n<PDR<R8b<R;k;@PP>Bdb8S8R<c;m
<`Pi83Ti5CTi>1Gm>Odh7bd]>20i>38R<c<b<b8b:38Q<R4c;C<]<bdi>3Ti7P/h
oR1L3`0e<c<Q8Bh0:P0d03@^03@0=2h[03@0=B/^03DZ03@0=00d;P0^:`0d;P0d
03@^:`0[=@0Z;P0d02X002/Z;P0d02h^03@Z03@^:cD^03@0:Rh[;P0^03@0>SX0
0@e@>P0Z03@d02hZ02h^=00d02Xe02Xe03@0:bh0;SD[;P0d03@0=00Z03D[;R/e
=00Z000^03@0:bh04a@D<R4R8C8Q<R4b8B8Q<R4b3C8?;B`PoRd6>Bd?;BdP;Odc
0R8Q<_hQ338b8R4Q<R8b<c8b8S;m;@@c;CTP;OPc0Rdi8?di1SPi>CPi>3Wm>1Tc
;CPi83Pc<c8c<c8c8S8Q<R8Q;C<]<bdc>?hiHR0N>20P3b0Z000Z0240:P0d02X^
02/^:bh0;S@0:P0^:`0Z000^=00^;P0d03@0;P0d02hZ03@0;R/d000[;RX0=C@0
;P0d02h[02X0;P0^000Z02/^:P0[=@0[;S@0;SXj004>CcX^03@^03@003@0:P0Z
=Bh0:bh0:P0^:P0[02/^02/e02X^02X003@[;P0^02X0=3DZ03@^03@0:Q@D:3<c
8C8Q8R4b8R4b8B4R8C8Q8Pl?oBd0>Oh]1@l6838c<_hc0R4b8_dQ1S8R8C8Q<R;n
<P0/o2d683T]<S<c;Ohc0Rdc;O`i21hh>CPE>CTh>Odh2BdP>3TP>C8c<b;n<`/b
8R4Q<R4b<bdc;C?m>@8h2cSm84/Z02h[02X0=2h002X0=@0Z02X0;R/0=00d;P0d
=00Z;P0[;P0d;R/0=2h[000Z;R/0=B/^:bh002X^03@0=2hZ02h[03@0=2h[=@0Z
oP0;=2h0=2h0:P0[>SX00@iE>P0[;P0^:P0Z;S@002h0:`0Z03@0:S@002hd=00[
;P0Z;P0d02Xd02h0:P0[=C@0:P0[;P0Z;S@051<C:38c8R4Q<R4Q<R4R<R4b8C8Q
;Bd/82d]>CWn;@<P3c<RoS<J<S<X<R4b8C8Q<R4Q<R8b8S8c<bd]<bd]>3P/o3<3
;3<c;OTi03Sn>@8h>CWm>0H]>3PP>B06oB1c8C8Q<R4b8C8=<S<]<bdc>CPi>ADi
>B0P;@0e:`0Z000^02Xd02X0=00d03@0;S@0:Rh0=00^=00[02X0;RX0=@0Z02he
:P00;S@0;R/e02/^=00[;R/^:`0e:`0^=00Z02X0=00d02X^:`0Z03@0:bhj>P01
33dj;P0[;P00=00Z03@d02X0=00d;P00:S@^02h0=2h0:P0[=@0d;P0d03@0=2h0
;P00=2h0:P00:RhD51<X<ohb3B8Q<R4Q<R4Q<R4R8C8cnbd0>Oh]1R0P<c<b<cWn
80h6>B4b3C8Q<R8Q8C8b8S;m;@Dc;Bdh<c;l<oh]0CThnST9>3Th>CPi>CPi>Ohh
NcTP>B0h83PP;B4R8C8Q8S8b8S8Q;C<]>3Th>CTh>@l682X0=BX0=3@0:P0^03@Z
=@0Z;P0d02hZ000d;RX0;P0d:P0d000d:bh0:P0[03@Z02h[02h[;P0[;P0d02h0
;R/0=2h0:P0^:`0Z;P0d=@00=2h0=2h0=@0j>P0143`j:bhd03@[;S@0:bh^02hd
02hZ03DZ000[=B/0:P0d02/^02X^02X^=00Z;P0Z02/^:P0Z03D[;P0D4a@CoS<=
8R4b8B8b8B4b8B8Q<R7n;@Lc;Bdc>CT];OhP02gn>@8680on>@h/<R4R8B4b8B8Q
8C8];C?m;@<P;C<]oS<4;C<?;CSn>@0VnST8>3Ti>20i>3Tio3P=83PP83P];B4R
8C8Q8C;n8@8]<c?n>F0h2cTi3`lP:P0^=00e:P0d03@0;P0Z02/0;S@[000^:P00
:P0Z=@00;R/^:P0^=00d02Xe02X0;SD[000Z=@0d02h[=B/0:Rh0=00d02hd000Z
;P0Z=00Z02X0=00Z>SX00@Pi>P0[;P0Z02h0=2h0:c@0;RX003@[;P0d;P0Z03DZ
03@^02X003@0;P0Z000d03@^000d03@0:P0Z5OdDoS<=<R8b8B4b8R4b8C8Q8R7k
;A`i>Bd]820;;CTi>3T?1Q0i>3T]3b4b8B4b8C8R8O`]3c<]>3<]<c<]<c<?;Bdi
>CSh>@PP>3Ti>3Ti>3Wl>0@P83PP>?h]43<b8B8Q8C8R<R4b;3Ph>CThoSUK8200
:P0^:P00:Rh0=2h0:P0d;R/0;P0d=00^:bh0=00Z03@0=B/0=00^:bh0;R/0=2h[
03@e:P0Z02X0:bh0:cD0:c@^;P0d03DZ000Z02h0=00d02X^000j>P013D0j:SD0
=00e:`0^:`0Z03D[02hd02h0=2X0;S@0:P0[;P0d02hd02X0=000;RX0:P0^=2X0
;RX0=00E51<C538c<_hc2R4b8R4Q8R4Q<R4QoRd<<bd]83Ti82d47@li>?hi00om
>@X?;CT6<R8Q<R4Q<_dQoRd1>2gl<`0PnRdA>@Di>@li1CT]>CTh>B0h>CPioCP6
83PP83Ph<oh]0b`b8C;n8FHR8C8c>CTh>CPi>0H?=00^:`00:SD0:bh[=B/0;P0^
=00[;Rhd03D[02X0;S@0;R/^02hZ03@0=000:SD[02X0:P0d03@0:SD0:SD0=00^
:`0d02X0;S@d03@0=00^:P0^02/^>SX00@Xn>P0Z02X^03@Z03DZ03@0;R/0:Rh0
=00^:`0^:`0d;P0d;R/0:Rh0=2hZ03@0=3D[02h0=00^02h0:QDD51<XoS<<<S<c
8B4b<R4b8R4b8O/]1cTi1RdN6`@]oSTI>3TP83Pi>3T]>CTh;B4R<R4Q<R4b8B4b
8B7i<`0/o2dB>CTV>Bdi>BH?3cTi>0lP>CTh>Odh0B0PoSSm;@/c;BdQ<R4b8C8Q
<SWn<ehi>CPi80l[03DZ000Z03D0:P0e:P0Z02h[=@0[;R/0=BX0=00Z;P0[03@0
;R/^02Xd;P00;R/0:P0e02X^000Z02/^:P0d02h[;RX0:P0^03@^:P0d02h[03@^
03Xj004?AcXd03@002Xe000Z000^=00^=000:P0e:`0e:`0^=000:P0^=000=00Z
03@0:bh0;S@0:bhd03@0:P0E51<D4b8c<S<R<S<Q8_hQ538Q8C8]<bd]<bdi>3Th
2aD;7STi8?hi5@li>3Ti;CPi>CP]824R<R4R8C8R8Bgm<`0]oS<1;C?k;Ohi3Pl]
>CT]82dh>CTP>CPi>Odh0R0h8?hh0BdcoBd78R4b8R4Q<R;m<ehi>3Th820^02X0
=2X0=2/d000Z03@0;R/e02/^03@0:P00;S@002Xe:bhd02h0=00^02hd:`0^=00^
:P00:S@0:SD[000^:P0d000d03@0:bh[02h^=00e:bh0:SXj0047>cX^02X^:`0Z
03@0=2X0:P0[;R/e02X0=BX0=BX0;R/d02X0;R/e02X0=2h^02/^02X0=B/^:bh[
03DE4ohDnc<:<S<Q<R8b8B8b8B7m;@0coCT47QD;7R3n>@D6>B0i>CSn>@<h>CT]
oR008OhX1A@X<bd?;Odc1Bdc<bdc3od]oST1;B3n>@4?8?hi10lP>CPioCSn8?dh
oRd9<bdQ<R4b8B8Q8OdcoSUK>3TP:cD0:P0e:P0^02XZ03@^02/0:Rh0:bh0=2h[
=00Z;R/0;P0[;R/0=2h[03@0;P0d02h[02h[=@0^:`0^03@Z000Z=BX0;P0Z;P00
=2h0:bh[03D[;P0j>P013D0j02X002hd03@0:Rh0:P0Z=@0^:c@0;RX0:bX0=B/0
;S@0=B/0;R/^02h[02/^02X0;R/^03@002X071hC51<D8?/c03;n8@0boR418S;n
8@Pb<SPi>3P;71kn>A8h>CPi>3Ti>3Th>CTh;CTi>3TioSP6>Bd?;2dc;Odc0Bdc
nRd1>BKn;A4i>@H]3cPh1Pli>CPi>CPh>B3l>04]<oh]3S8Q8R4Q<R4b<c<]<cTi
>?hiF2d0=00Z000Z02hd000^02/^=B/0;S@0:bh[02hd000^=00[;P0d02X0;R/^
=00[;S@0;S@0;R/e:`0e:`0Z02hd02h0:bhd02/e:bh0:bh0;S@0:P0^>SX00@]1
>Rh0=2X0;P0Z=@0[=@0Z02X003@^:`00=2h0:P0e:P0^:`0d;R/0:cD0=BX0:cD0
=00^02X0:SD0:QDE51<D4b0Pnc<D<R8Q8C8Q<R4Q<R8Q8SPi>3Ti7P/LoCT0>?hi
023j>00PoST1>3Wn>0<P3bd?oC<8;C<]<bdc;BdcoRdB>CT?;Bdi>@lP;CTi>0H?
>CTh>Ohh0B0Po3Sn;@`c:38Q<R8Q<R4b8R4boSUK>3Th82X0=3DZ000d02Xd;P0d
;P0Z03@0;R/e02h[02hZ02/^:c@0=2h[03@0;P0Z03D[;R/0:P0[;P0d;R/^:`0d
02X0:cD^02X0;R/^03@^02X0;R/0=00j>P013f0j02/^03@0=00^:bh0=00e02Xd
02h0:bhZ03@002X0:P0^:bh0;S@0:P0Z02hd02X0=00d03@0;S@0;QDE51<D3b0c
<bdc;3<c8C8b3C8X8B8Q8C8Q<cTi>3Th5Ah;>CPioSP08?`h31Di80H4>CPi>B0i
83Wn;@4c;Ohc0C8RoC;n;@<c;BdioRd=3cTP;B0]>CT?820h>CWm>04P3odh023n
;@`Z02PQ8C8Q8S8Q8B8Qo3UH>00Z02X002X^=00^:`0Z02h[02hd03@0:P0^=B/0
=C@002h0:P0e02hd02X0=B/^02hd02Xe:`0Z02h0;S@0;SDd02X0:P0^=00[02X0
:P0Z02hd02hj>P013D8j;P0[=@0Z;P0[;P0d;R/^000d;P0d;P00=2hZ000d03@0
=B/^02X0=2h0:P0^:`0Z;R/^02X0:bh0:aDE51@C520inS<58R4b8C8QoC<153?n
>@Lh>@/E7Q`i1ohh03Wl>0LE83T@7PHi>Ohh13TP;Bd/o3<8<S8R<R8Q8C8QoRdB
>BdP;BdV;B0]82di>B0?>CTh>Ohh0b0P3cWn>?h]JP0Z02hX8B4b8C8Q8C8Q>CTh
>CTh03D0=3@0;P0d03DZ02X0=BX0:P0^:`0d02X0=00Z=BX0=3D[02X0;R/0=2/^
02X0;P00:P0d03@0=00^:`0[;P00=00d02h[=BX0:P0d02X0;R/0>SX00@/j>P0[
;R/e:`0^=2/0:P0^02Xd02X0:P0[;RX003@[;S@0;R/^03@0:bh0=00d03@Z000^
03@0:SD0:bkn5@@C51@P>Odc02gn<`Lb8C8Q8S8c;Odc3CTh>CTP2`/i1`L]83TP
ncP75CPh1P@i>3Wn>?h]oC<>;3<b8S8R<R4b8R4b8C8ioBd73cT]82d?>CWn80<i
>CPioSSn8?dhK3<]=00Z02h0:24R8C8b3C8Q>CTh>00Z02X0;S@0:bh0:P0Z03D[
02X0:P0d02X0:P0^=00Z000Z000Z;S@0;S@0;P0Z02X0=3@0=00^:P0Z02hd02X0
=B/^:P0e:`0^03@002hd03D[02hj>P014SXj;P0[;P0Z02/e03@0:P0d02hZ03@0
=2h002Xe;P0[;R/0;R/^03@^03@^02X0;P0d:P0d;P00:P0d0?hE1A@C4b0i>O`c
538Q8S8Q8C8Q<R8c;C<c>CTh>CTE7_h70bdi83Wk>00EoSP32cPi>Ohh12d?<c<]
oS<R;C8b8S8R<R4b8B8Q8C8b;0l];CTP;B0];CT]1PlP>CTh>CSn808?>B3n>6/]
;P0Z02X^02hX8B4b<R8Q8CTi;P0^:`0Z02h[;P0d03@0=00Z000d;P0e:P0d03@0
:P00:SD0:Rh002X0:bh[03@0=00d;P0Z=Bh003@^:`0^:`0d:P00=2X0;R/0=2h[
=@0Z02X0=00j>P014Shj02Xe03@0:bh0;Rhd02/^=000;S@0;P0Z=00Z02/^02hd
:`00:bh[02X0=00d02Xe02/^02Xd03DZ02hE5A?n510h>C<]<b4b8S8Q8B8b8B8Q
<_`c33Pi>CPi5@/71bd]>B3k>0/E5CPh5@/h>CPP>Bgn<`Hb<c<b8S8RoS8Q:28Q
<R4Q8S8=<S8];@l]82dP83Ti3`H?>CPi>B0i82dP;OhhK3Tc03@0=000:P0^02PQ
<R4Q<STZ03@0;S@0;R/e03@0:P0^;S@0;S@^02X[;P0Z02hZ02hd02/^:`0^=00Z
=@0^=00^=BX0:P0e:`0d02X0;S@0=B/^03D[;P0d03@0:P0e02X0=00d02Xj>P01
46Dj;P0Z02Xe02/Z000Z;P00=2X0:bh[03@0;P00=2h0:P0^03@d;P0e:P0Z;P0Z
000Z03D[;P0Z02X0=00L5ADC51<i>3<Q<R8Q8C8Q<R4Q<R4b8C8c<bdc>CPi>CP;
5@L7;Bdh>B3k>00EoSP67SPi>CPP;Odc0S8c<ohb6b8b8R4Q<RPb<R4Q<R8Q<R8]
82dP;BdP>@lP80om>@@P>00]8?`hJRh0:Rh[;P0Z02X0:bPb8B4003@^:`0^:`0^
:bh0=2h[02/^02/^03@0;P0d03@0;R/0=3D^02X0:bh002X0:bh0:`0^000Z02hZ
03@^02/^:bh0:bh[;P0Z;P0e:P0Z03@^02X^03@e>SX00A0k>P0d03@002Xe02Xd
000e:bh0:SD0=00^=00Z=BX0=00e:`0^:P00:P0d000Z=C@d03D[;P0d024b=00^
oQD251<DoSP?80H@8206<R4Q<R8Q8C8R<_dc2STh>CTh2`L73bd]nCP95ADh>1D;
>CTP>Ohc;C8c8S<c<R8002X0;P0Z02/Q<R4b8B4R<R4P;B0]80l]1PlP83Ti>3Ti
82h[00om>6X[=B/0=@0d02X0:P0Z0240=00Z02hd02hd:`00:S@002/^02Xe02/^
=00d02h^=00^=00Z02X0:SD0:S@0:SD0:bhd02Xd02h[02/^02Xd000[02X0;P0d
02/0:P0d02hZ03@002X^03Xj004?=SX^03@^:P00:P00:SDZ02X0=@0Z02hd02h[
02X0=2h[03@d02h[=@0Z03@^000^;R/Z02/^03Cm8@L05AD7:1@C8?hho2068CTQ
8C8Q8_hQ03;n<`Li>CPi>Ah71oh]03Wj>0/E5CPh2aDh>CTc<R;m<glb<b/^:bh0
=00e02X0:P00:RPb8B4b8C8=80l]83PP82dP>CTh>B00;R/0800^02h[;R/0=2h0
03@e02hZ02X^000d02X003@^:cD0;RX0=B/0:bh[000Z;P0d02/^02/^;P0Z03D0
:P00=2h0:P0e02X0;P0d03@^02/^02h[=AT^:`0d03@^03@Z03D0:P00=2hZ000[
;SXj004>@CX0;RX002Xd03Dd02X0=00^:c@002X0:P0^=000:P00:P0^=00d02X0
;Rh[02X[02h0:SD0:P0Q<R8b8@L7:1@C5?dh420P>20P>3P6;B4b8C8Q<R8boS<0
>?hi1R0;1bd?;Bgi>0LE5CPP7P/i8?dc82h0;R/0;P0^:`0^=00Z02X0;RX002X0
:24b8S8Q<R8?;O`P03Sn>G8Z02hZ02X0:P0Z000^:P0[;RX0;RX002h0=2/^:P0d
:bh0;R/d000d:P0Z=B/0=BX002/^:bh0:S@002X0=00d03@[;P0d02/^:bhd02/^
:bh0;S@0=B/0;R/e02h[;P0Z;P0Z03@0:S@0:P00:SD0>SX00A8g>RX003@^02X0
:P0d02X0=00^;R/d03@0=00^=2h0=3D0:P0^:cDZ03@003@^02h[03@0:P0^:S;n
8@Hb8PLD:2PCo3PI820h83Ph>B0]8C8=<R4b8C8c<cTi>3Ti5@Om;@4i8?/hOaDE
>3PN5CTh02X002/^02X0:cD0;R/0;RX0=@0Z000d;P0Z02/^8C8Q8S8Q<Rd63b0]
>CTh>A@^000d=@0[;P0d:P00=2h002X003@Z02X0=@00:SD0:cD0;P0S;P00=00^
03@Z03@e;P0Z03@0;R/e03@^:P0Z03D[;P0d;P0^:00Z03@0=B/Z02h[03@^02/^
:`0e:bh003@^:P0d02hZ03@d02X0>SX00A4j>P00;RX0=00d02h^=00^;S@0:`0^
=00^;S@0:P0d02h[;S@002X002hd:bh0=00^=00Z03D[024Q8S;n8@<71a@Do3PD
83PP83Ph83P];2db8B8Q<Pdb<c<hoCT61bd]3bd]>O/hoQEo>1XN2cPd000Z=00d
000Z03@0=00Z03@0=00[;RX0=000;R/0:c8Q8C8R<R4b8B0P>CTa;R/0:Rh0:SD[
02X0=B/^02Xd02Xd02h0=C@0:S@0:P0^:c@0:P00=00^=00[;P0d02X0=000;S@0
=00Z02h0=00^:bh0:bh0:P0d03@U02h[;P0d02h[;P0Z=@0d02X0:cD[;P00:Rh0
;P0d02hd02hj>P012T4j:P0d03@^;P0Z000Z02/0;R/^02/^02/0:P00=2hZ02h0
:P0d03@Z02h0:c@^:`0^000d:bh0<R8a<R8Q<R471aCk>08P>23l>?h]3B4Q<R8b
8C8c>CTP>CT7o2gj>?hEOcPh2aDP02Xe02h[;R/d=B/^:cDZ02h0:Rh0=00^:bh[
=@0Z02h[024X<R4R8C8Q8CTR000Z000[;P0Z03@0=00d03@0;P00=2X0:P0^:`0^
000Z02h0=2h[;P0d02X^02/e02X0=2h[=00^:P0d03@0:P0d02h0=2h0:P0d;Rh0
9C@0:P0d03@0=00Z02X0=BX0=2h0:P0d;P0[=00d;Rh0;R/0>SX00@To>P0d;P0^
:`0d02h[=C@0;R/e02Xe02X^=000:Rh[02h[02hd02h^02X0=00^02hd02Xd02h0
:P0b<R4b<R4Q<_h7ncSn8?`h2C<];CPQ8C8b3C;l<`0NoRd282d]nSSn5GlP7QDO
;Rh0:P0e:`0^02h0:P00=B/0:P0d02h[=@0d03@0=00Z02h0:P0X<R4R<RX0;S@0
:bhd03@002he02X^:P0e:`0Z;P0d02X0;S@0=2X0=00Z02/e02/^:P00:SD0:Rh0
;R/0;S@002X0;Rhd02X0=B/0:P0d:`0^:`0Z;R@d000^:P0^;S@0=00[:P00;P0d
02/^03@0:P0^:`0^:`0^>SX00@ho>Rh0:c@0;R/^:`0^02X0:bh0=00Z000[=BXd
000e:P0Z02/^:`0[=@0d;R/0:P0^:`0d;P0d02Xc8S8Q8S8Q8_h7ncP283PPo3Sm
;@LP8B4b<Pdc;Ohc2B4]82`]3bdi83Wm>?hEOcP;5B/0:c@0=00d;P0Z03@0:Rh0
=2h0=2h[02h[;P0Z;P0Z000d03@0:P0Z02P003@^03@^02h[;R/d02X^000^:P0d
=00^:bh003@0;RX0=BX0=2h^02X0=00^:cD0:P00:c@0;R/0;RX003@0:P0d;RX0
=2h0=2h0=00^:`0092X^:`00=00Z03@^03D0=2X0:P0d;P0d;P0^:`0^:`0d;R/j
>P012d0j03@0;S@0;P0^=00Z02Xe02/^=00d02h[03Dd02X0=00[;P0^=00Z02X0
;S@0=000:Rh0:bhd03<c8R4b8C8Q8Oh7ncP1823k>0T]<bdP<c8R8C8]oC<3<Pl/
3od]0CTPoCSm5FlP5@0Z=@0d02X^03@[02X0;P0[;R/0:P0^=00^02X002Xe=3@0
;RX0;S@0=00Z:bh0:Rh0=00^:`0^=000:SD[02X0;S@0;P0^=2X0:P00:P00;R/0
=00[;S@0=00d02/Z=@0Z02he:`0e:bhZ02hZoP0`;R/0:P0^:cD[03D[;P0d02X^
=000;P0^:bhZ03@0;S@002X0=00^=00d02h[03Xj004?@cX^03@0:P0[=00^:`0d
02X0=B/^03@Z02h0:P0d02X0=2h0=00^:`0d=B/0;RX0=3D0:c@^02h[;S<c<R8Q
8C8b8@L7ncP283P]o3Sn;@DP83<b8R7m<`D/8C8b;0on;@8h>B3m>?hEOcP;;P0Z
;P0Z000Z03DZ03@0=2h0=BX0:P0^:`0d;R/^000Z;R/003@0;Rh0=3D0:P00:c@^
:`0e:P0^:bh0:bh003@0:P0[=00^03@0=3@0:SD0:P0Z03D0:Rh0:Rh0=@0Z03D[
02X0=00d000d02h[=2X0=BX0:P0^:bh[;P0[8cD^:`0[;R/d03@0;P00:P0d;P0d
:P0d;R/0;R/^:`0e:SXj004AA3X[02hZ02Xe03@0=00^=00^:bh0:Rh0=00Z02h^
=00^;P0d03D[02hZ000^:`0^:P0d02h0:P0^038c<c8Q<R4R8C871o/h0B0Po3P4
82d]823n<a0X<S<c<R4Q8R4b;Bd?;BdP>OdhoQEo1`/^=000=00^=00^=000;R/^
:`0Z03@0=00d02X0=B/0:P00;S@Z02h[02X^03@0:S@0;P0e02X0:P0d02/^02Xe
:P0^=00^=00d02X^02Xe03@0=00[;R/^03@002X[=00d02Xe:`0^:P0^=00Z02he
000Z03@0=00^02h0;S@S02/e02X0=@0Z;S@0:S@0;RX0=00^02h0;S@0=00e02X0
>SX00A4j>Rh[000d;P0Z;P0Z02h0=00^02X003@^02/^02/^02/0=2hZ03DZ000Z
;P0d;P00:Rh[03@0:P0Z03;n<`HQ<R4R8@L7ncP283P]o3P5;BdP83<boS<?<S<R
<S8Q8B8b8Rd?;CTP>OhhoAEo4`0Z02X^02/^:`0Z=BX0=@0Z000d;P0Z03DZ02X0
=BX0=2h0;P0d02X002X^02X0;R/0=2/e02X0=2h0=00d02X0=00Z000Z;P0d02h0
02XZ03@^;P0[;P0Z;R/e02h0:Rh[02X0=00^:`0Z;P0d02X^:`0d=@0Z02/d03@0
;R/R;P0d03@0:P00:P00=2h003@^:P0d02X0;R/^:bh[03Dj>P014TDj02he:bh0
:bh0=00d02/^=00Z03D[;P0Z=@0[;P0d02hZ000[:P00;S@0:bh0=2/^03@0;S@0
=00^<b8c<R8Q<R4Q<PL7o3P2820]o3PC82d]82h[02h[02h0:P0X8C8Q8C;n;@4i
8?dhoQEo:`0^=000:S@0;S@0:`0Z03@0:S@^02/^02X0;S@0:P00:P0d02/^:P0d
=000=00d02hd02h[=000;R/0:P0^=00Z03D0:S@0:`0^=00Z=000;RX002Xe03@0
02X0=00Z000^=00d02hd02X002/^=00[02h^03@0:SD0:P0d02hR:`0Z02hZ02hd
03@[;P0Z=00Z000^=00Z02X0;P0^=00j>P013cXj;P0Z02/e02/Z;P0d;P0[;R/0
=00d02X0:P0d03@^:`0d;R/^03@d02h[=00Z03D[;P0d02X^02X0<_hc1c8R8C8Q
8C87nSP1;Bgl>1D];B00;R/0;R/0;P0Z000Q8C8R<R8]ncQo5AD0=BX0;R/e02h[
03DZ03@0:Rh002h0=2h0=00d02h[03@e03@^:`00=BX0;RX0;P0d;P0Z;P0^03@d
02hZ03@0;P0^:P0d=@0Z02X0;R/0=2h[000d:P0Z02Xd;P0d;R/0:P00:Rh0=00^
:c@e:P00:P0e:P0[;P0d02X^03@U;R/0=BX0=00^:`0^;P0d03@0;RX0=2X0;P00
=00d03@0:Rhj>P014THj02X0:SD0:S@002hZ02/^02hd02X^:P0d03@0;RX002Xe
:cD0=00^=00e03@0;R/^02/^:bh0=00Z03<b8S<b<R4Q<R8Q1o`h0R0]3o`h5R0]
82X0;S@0;S@0=00^:P0X8C8Q8C8]o3Qo5Bh0:P0Z02h[;R/e02X0;SDZ000Z=C@0
:P0[;R/^=00d02h[;RX0;SD[:P00:P0d02hZ03@002X0;RX0;S@002hZ03@[02/^
02Xe02hd000Z;P0d02hZ03D0=00^02/^02hd02Xd02/0;RX0=@0Z02hd03@Z02/^
02X^:P00=00U:P0[;R/0;S@0=00Z03@0:Rh0:P00;P0Z02Xd=00^:bhd000j>P01
4SXj;P0Z02X0;P0d;P00=2h0=00^:`00=BX0=2h[000d=@0[;P0Z02X0:Rh[;P0d
02h0=2h0;P0Z03@0:_`c1R8b8R4Q<POk>0<P;BdioCQo;B00:P0Z02/^:bh0:`0d
02/Q8C8Q<R0i83TP5@0d03@0=00^02/^:`0d02/e:P00:P0d03@^02h0:P0^:`0^
000Z02X003@Z03@^:`00=2h[;P0d02h[02h[=@00=2h0=2h0:P0Z02/^=2h003@^
:`00=2X0;R/0=2h0:P0^=010=BX0=00Z02X003@0;P0^03@^02X002h[=2h002Xd
03DZ02h[;P00=2h002Xd03@Z03@0=00^02X0;P0Z03@j>P014bLj03@0:P0e:`0Z
02/e:P0d02X0;S@0:P0[;P0e=2X0;R/^02h[02hdoP0L:P0^=00[;R/0:cD[02h[
;P0c<c8c<c8b8C8Q8@Ol>08P02gk>7lP=00d000Z02h0:S@0;RX0;P0X<Pdb8C8Q
8@0[;P0Z02hd02/^02hd02hZ000^=00d02X^03@0:P0d02hd02Xd02hd02Xe02hd
000[;RX0=00d02X0;S@0=00[;RX0:bh0:P0Z02Xe02h0:bhZ03Dd:Rh0=00e:bh0
=00d02h0:Sl0;Rhd03@0=00^:bhd02/^03@0:SD[02h0:S@0;P0Z02X0;P0Z=BX0
=3@002he02/^02h0:P0^=00d02/^>SX00AE9>Rh0=00d03@^03@^:`0d;R/0=00^
:`0d;P0Z02h0=00Z02X0=BX0:P0d;P0d02X0=B/^:`0Z03@0=B/Z<S<R<S<R<S8R
<R4b83Wm>080;Rgl>7l];P0Z=BX0=00d02h[02h[02X08C8R<R4R=Bh0:cD0=00Z
03@0=00^:`00:P0d;P0Z;P0[;P0d03@^:`0^:`0^:`0^:`0Z000Z=BX003DZ03@^
:`0d02X^03@^000d;P0Z03D0:P0[=00d;P00:P0^02/^03@0=00Z03@^:`0d03ld
02X0=2h^03@0;P0Z03@0=2h002X0=00d=@0d03@0=00d03@0:P0Z=@0Z02/^03@0
=2h[03@0:Rh0=2h0>SX00A@l>R/0:P0^:P0Z02X0;P0^:cD0;R/^=00^02X0:P0[
;S@0=000:P0d02/^:`0^:P0d:bh0=B/^02hd02h0:?/c1R8b8C8=<R3m>7lZ000P
;P0^02h0=00Z000Z02hd02hd02hd02X0;P0Z03D0:`0^:bhZ02hd02h[=B/0:S@0
;RX0=000:SD0:bh[;P0^=00^=00^=00^=00e=00Z03Dd:`0Z;P0d02hd02/0:P0[
;RX0=00d03@0=00^:bh0:S@0=00[;P0Z;RX0=4L0:Rh0=00^=00Z02X^000Z;S@0
=00[;R/^02/^=00^:bh0:Rhd02hZ02h[;S@002h0:P0Z=00d02hZ03@0;S@0:`0Z
02hj>P015DPj;P0e:P00=00d03@0=00^:bX0;P0^:`0d03@0=2h0:P0d=BX0=2h0
=00d:P00:SD0:bh0=00d;P0Z02X0;P0Z:3<b<c8R<S8RncQo;P00:bX0:P0Z03@0
=3D0=00Z02/^:`0^000d03@0:P0d;R/0;P00=00Z;P0^:bh0=2h0:`0d;RX002X0
=@0Z02X0;R/0;R/0:P0[;R/^02X0:P0^02X003@^:P0^03@e03@^000d;Rh0:Rh0
:P00=00d02hZ03@^02X002hZ03E602/^02X0;Rh0;P0[=2h0:bh[;P0e03@0=2h0
:P0[=@0Z000Z;P00=00Z02h[=00Z03D0;RX0=00^:bh[02h^03@0=00j>P016Glj
03@[02h[;Rhd02hZ02X0=C@0:cD[02Xe02X0;P0Z02X0:P0d02X0;R/^02hd000Z
=@0[;R/^02X[02hd02/^02h0:3<b8C8Q8C8Q<R4002X[;P0d03@0:bh0;RX0;RX0
:SD0;S@0=2/^:bhd02hZ02he:`0Z;S@002X0;P0[;Wl0=00^:P00:S@0=B/d03@0
:P0^=00^=00Z=@0^03@0=00d03@0;S@0;P0d02/^:bhZ03D[;R/0:bh0=00e:bh[
;P0Z000^03@0:SD[03@0:P0^=00e:`0d03@0;R/0:P0e02/^:bh[;P0Z02Xe02X0
;S@0:`0Z;S@0;R/0;S@0;QD[=B/0:bhd02h0;SD[02/^=00^>SX00AUo>Rh0:P0[
=@0[;R/0;R/0:P0^:`0Z03@0:Rh0=00d03D0:P0d02X0=00e02X0:Rh002X0=00e
:bh0=BX0;R/^02X0:P0[024X<R4R8B4b:P00=00d;P0Z;P0[=000=000=00Z02/^
03@^02X0;R/003@0:P0d02X0=2X0=00d;P1o=2h[000d;P0^:bX0;Rh0=00d02X0
:bh002X0:c@^;P0d;RX0=00Z;P0Z03@^02h003@[;P0^:`0[=BX0:P00=00Z03@d
03@^000Z03@^:P0d02X^:bh[;R/^02/e:P0Z03@^02h0:P0d03D0:P00=00Z;P0d
000Z03@^02/^02/E;P0Z03D0:P0[=B/0=2h0:P0d03Xj004IOcX0=00Z=@0[;P0^
=00^=00Z03@0;P0^=00[02h[;RX0:P0e02Xe02hZ03@0=000:S@[02X0;R/0;R/0
;R/e02X0:P0^;R/^02/^02h0:00^=2h[;P0Z000d02h[;RX0:Rhd02/^02X^03@0
;R/0=B/^=2/0;S@0;RX002h[;R/0ObX0;S@^=00d02h0=00[02X0;S@0:SD0=3@0
=00^000Z02X002h[;P0d02/^02X0:S@0;P0d02hd02X0=00e:P0^=00^02X^02Xd
02/^000Z03@0:`00;P0e03@0:P00;P0Z03@0=000;S@0:P0d:bh[;P0d02hd02/^
02/^02Xe5@0[;S@0:P0d02X0:Rh0=00Z02hj>P016Glj;P0d02X^02/^02X003@^
03@^:`0d;P0d;R/0=@00:P0d02X0;R/0=BX0:Rh0=00^03@^:`0e:P0Z02/0:P00
=00d02h0:bh0:cD0:P0d02h0=B/^03@d;P0[=@00=00^03@^03@0:bh0=00d:P0^
02h0=00Z000e:P0[=@0e:Wl0:P0^02h[;R/0=2h^03@^:`0^000Z02X^03@Z03@^
03@0=2X0;P0Z03@^03@0=00Z;P0Z03@0;RX0;Rh[03@^02h[03@003@^03@^03@^
02X^03@Z02/Z03@^000d=00d;R/^:bh[=00^:`0d;P00=00d;RX0;R/e03@^03@0
:QD0=@0Z;P0d;P00=00[=Bh0;R/0>SX00AUo>P0Z;S@002hd03@002Xe:`0Z000^
=00[;P0^=00Z:bhd02X^=00d02X0:bh0=00^=00Z03D0:P0d000^=000:SDZ02X0
:bh[02X0:bh0;S@[02/^02/^02hd02X0:Rhd02/^03@0:SD0:bh[;P0d02X0:S@0
;S@0:P0d02X0:P1o=@0d03@0:P0e:bh0:`0Z02hd03@d02/^03@0;P0Z03@0;Rh0
=00d02X^02/^02X0;P0d02X0=B/0;R/0;S@0:P0[=B/^02X^03@0;P0Z03@002/^
02Xe02/^03@[;RX0:P0^03@002hZ03@0;P0Z=00^:P00=000:P0Z02/^=00E;R/d
000[;P0d:bhd02X0=2/0;SXj004IOcX^000Z;R/0;R/^:c@0;P0d;R/e:bh0=00d
02h[03D0:Rh002h[;P00=2h0=2hZ02h[03@[;P0d;R/d03DZ;P00;R/0=2h0=BX0
=2h0=00^03@^03@^02X0;P00=00Z03@^03@^000Z03@0=B/^03@0=00^:P0^:`0d
;P00=00ZO`0d02X0=00d03@^:`0d;P0[;R/^03@^03@^:P0d;R/^:`0[=@0Z03@0
03@^03@0=00d;R/0:P0d;P0d;P0Z000d;P0d03@0:P0Z;P0d;R/^03@^03@0:P0e
:bh0=@00:P0d03@^:cD002X^:`0d=@0Z000d;P0S;P0d03@^02h[5@0^03@Z02X0
;P0Z03@0:P0e:P0j>P016Glj02Xd000^=00d000^=00[;P0^:bh0=00Z;S@0=00^
=000;S@0;P0Z=2h[02X002X0;S@0;R/0;R/e02hZ000Z=00d02h[02X002X0:bhd
02/^:`0Z03@0=00d:Rhd02/^02/^03@[;S@0;R/^02X0;Rhd000Z03D[;P0d:Rhd
07l[;RX0;Rhd02h^:`0^=000:P0^02h^:`0Z000[;P0^02h[;P0Z;S@0:SDZ03@0
;Rh[;R/e02h[;P0[;P0Z02Xe:P0d02X^=00d02/0:P0^03@^:`0^=00d02X0;R/e
:bhd02hZ000Z02<003D[;P0Z02Xd02hd02/0;R/^02X0;QDd02X^03@0:P0d02hd
02h0:P0^>SX00AUo>Rh0:P0d02X^03@d02hZ02/^02h0:P0d02h0=2hZ02X0=00Z
02/e02h0=BX0=2h0=00^:P0e:P0^:bh0:cDZ02h[;S@0=BX0=3D0=2h0:P0d03DZ
03@^02X0;P0[;P0e:bh0:bh0=@0Z;P0^03@0=00[;R/e02X0=B/^000[;P1o=00^
:`0[;R/003@^02X0=00d03@0:cDZ03@^02/^02X0=B/^000Z;P0Z03@^:`0[;P0^
:c@0=B/^02/d03D0:P0d02X002hZ03@^:`0d03@^03DZ02h[;P00=2/^02X0;R/0
02Xe03@0:Rh0:P0Z03D0:Rh0:Rh0=00e:bh0:P0E:P00=2h0=2h0:P0Z03@0=00d
03Xj004IOcX[=@0Z;S@002/^;R/^03@0:cD[02X0=BX0;RX002X0:Rhd02Xd03@0
:bX002X0=00e:`00:P0Z02h0=00Z02h[=@0^02X0=00Z02X0:bh0;R/^:`0^02X0
=00Z=@0d02X0;S@0;R/d000d02/^:bh[;P0Z03@0=00^:`0Z=@0ZO`0e:`0[;P0^
=2/^03@0:S@0;R/^=00Z000^02Xe:`0d02X002Xd000d000Z000Z=@0Z02h0;R/^
02Xe02h[=000=BX0=2X002/^03D0:P0Z02h[02X0;P0d=00^02X0=00e:bhd02X0
;RX0=00d000^:bh0=000=00^:bh0=00d5@0Z=2h0:bh0:bhd03D0:Rhd02hj>P01
6Glj02X002/^=2h0:`0e:bh0=2h0:P0d;P00:P00=2h0:P00:P0^02X^:`0d;R/0
=2hZ03@^:`0d02X0=2h003@0;R/0:cD0=2h^03@0=2h0=00e:bh[03@^02X002X0
:Rh0=00Z02X0:Rh[;R/003@0=B/^03@^:P0Z03DZ02X0=7l^:bhd000d02h0:cD^
03D0:P0e02X003@d03@0:P0d;R/0=2h0;R/e02Xe03@^02/^03@0=00Z03@0:P0[
;P0Z:P0^02h0=3D^03@Z03DZ02X0;R/0=00Z;P0d03@0=2h[03D[;R/0=000=2h0
:SDZ02h0=2h[=B/^02/0=2h0:QD003D[;P0[=Bh0:P0Z;P00:Rh0>SX00AUo>S@0
:SDd02h0:SD0:P00:Rh0=00[;P0Z=00[;RX0=000:SDd02/^03@0;P0^=00^02/^
03@0;S@002X0:S@0:P0^=00^:bh0:`0Z02h0:bhZ02X0=@0Z03@0=B/^=00[02h[
;S@0=000=000:S@0;R/^03@0;P0d000Z03@0:bio02X003D[;R/^=00Z03@0:Rhd
02X0;S@^02X^=000:P0e02X0:P0^:c@0;R/^:`0Z03@0:P0d03@0:P0Z=@0Z000Z
03@0=00^:`0Z000Z03@0=00d02hd000Z;S@0;RX002h[;P0d02h[;RX0=00Z03@[
02X0;R/^02Xe:bh0:SDE=00Z02/e=00[02hd000Z=000:SXj004IOcX^000Z03@0
=00Z02h[=@0[=@0d;P0Z02h[=000=2h0=000:P0e:bh0=00d02hZ03@^03@^:`0^
=2X0;P0Z03@^02X0:bh0=2h[02X0=2h003@0=2/e02X^02X003@e;R/0;P0Z02hZ
03DZ03D^:`0^:bh0:P0Z;R/e03@^03D[Obh0=2X0=@0Z02h[03@^:`00:P0[=B/^
02X002Xe:P0Z03@e:`0d02h0=00^03@^03@^:P0Z;P0^:`0e02X0=BX0=2h0:Rh[
03@^03@d03D0:Rh[;R/0:Rh002/^:`0^:cD[02/^03@0=B/0=2h003@^03@0:P0^
02X0;P0[;P005BXe03@^02X^03@0:Rh003@^:`0j>P016Glj03@[;P0^:bhd03@[
02X0;R/^:P0d02X0;SD0:P0^=00^:P0e:P0[02h[;S@[000^:`0Z02hd02h0=00d
02/^02X0:SD0:S@002hd:bh0:S@0;RX0;R/^03@0=C@0:P0^=00Z02hd02/^:`0Z
02hd02X0=00e:`00;R/^02X[;Wl0:bh0=00Z02X0=00^:`0^=00[;R/^03@0;S@0
:P0d000Z02hd02X0:Rhd02/^:`0Z000d000^:cD0:S@002X0:bh0:P00=B/^:`0^
:P0^:P00;P0e:bh0=00Z02hd000Z02Xe02hd02X0;Rh0=2/^02/^=00d03@0=00[
;R/08aD002hZ03@002/^=000=2/^:`0^>SX00AUo>Rh0=B/Z02h0;P0d;P0[=00^
000d;P00=00Z03DZ02X0:`0d02h[=BX0;P0^03@Z03DZ02X0;R/0=2h[;P0d03@0
=00Z000^=2h003D[;P0Z02/0=00Z03@^:`0Z;P0Z02h[03@0;P0d03DZ02X0;R/0
=2h[;P0d:P0^02X003Ao;P0[=@0Z03@0=2hZ02h[03@^000d02X0=00^000d;RX0
;R/0;R/0=000:P0d03DZ03@^02X0=2h0=00Z;S@0=2h0=00d:P0^03@^000d02X0
=00d03D[:P0d03@0:P0d03@0:P0[;RX0=00[;P0e:bh0:bh0:P0Z03@^02/e:P0E
:P0[03@^:P0e:bh[=Bh003@^03Xj004IOcX[02X002X0=00[;P0Z=@0e:`0Z;P0d
:bhd02/^:`0^=00Z;SD[02X0=00d02Xd000Z03D0=00d02X0=@0^=00^:P0e02Xd
02X0:SD[;P0d02Xe02hZ000Z02hZ000Z03@0=00^=00d=00Z03@002/e02X0;R/0
:P0d03@0;S@^O`0[;P0Z;S@0;RX002h[=@0Z03DZ03D[02hZ03@d02h0=00^=00^
=00^=00Z=@0Z000Z02hd:bh0:bhd000Z02h0=00^:P0e:`0^:`0Z;S@0:Rhd02hZ
000^;S@0:SD0;RX0:P0Z=B/002h0:P0^:bh0:SD0:bhd02/^02Xe000Z5@0d02hZ
000d:bh002X0=00Z02hj>P016Glj;R/0=2h0=2h[;P0d02X^:`0d02X0;P0[;P0d
03DZ02h002/Z03D0=2h[;P00:SDZ02X0=2h[;P0^:`0d02h[02h[;P00=2h0:P0Z
000d;P00:P0[=2h003@0;R/e03@^:cDZ02hZ02hZ03@^:P0d02X0=00e:P0d;R/^
02/0:gle;R/^000Z000e:P0[02Xe02X0:cD[;R/002X^03@0=2h[02h[02hZ02h[
02X^03@d03@0;P0[=@0Z02Xd03@0=2h[000d:bh[03DZ000Z000Z03@0:P0d02X0
=00Z02/e;P0Z02h0=3@0=00d02X0=00Z03D0:Rh0:cD0:P0d01DZ;P0[03@^02h[
;R/0=2hZ03@0>SX00APP>P0e02h[02h[=@0[;S@0:`0^;S@0=00Z=@0[;P0Z02X0
oSAo000Z02X0;P0Z=00Z03@002X0;P0d:`0Z;S@0;S@0=B/0:Rh0=00d03@0:P0e
:P0d02h0=2/^=2/0;R/^02X002h[000Z000^000d02Xe02X0:P00:P0e02/^=00^
:`00:S@0=C@[02Xe:bh0=00[;R/^03Dd:P00;RX0:P0^=01a;S@002X0;S@002X0
=BX0=B/0:P0d02X0=C@0;RX0;S@0;RX0=@0Z02hd03@d02h^=00[;R/^03DZ03@0
;P0d03@0=00^=00Z;S@002X0;P0Z=000:SD0:P0Z;S@002Xe02h[=B/002hd02h0
=00^>SX00AUo>RX0=00d;P0^:bh002X0=BX0:bh0:P0d02Xd03@0=00Z=@0^:S@0
=00Z03@0;P00=2hZ000d03@^03@0:bh[02X0:P0d02/e02Xe02X^02X0:cD^:`0d
;P00;P0d02X0:P0d;P0d;R/0=3@^:cD0:P0Z03DZ03@e02X0=2h0:P1o:cDZ000Z
;P0Z03D0:P008`0d;P0^02X0;P0d:P00:P0d02h[03@Z03@^02X0=2h[02hZ03DZ
03@^02h0:bh0:`0d02X^000d:P00=00Z;P0^:`0[;P0e02X0=00^02X0:cD0:Rh[
;RX0;R/002Xe:P0d03@0;RX002X0=2h002XE=BX0:P0[;P0d:P0[;R/0=2h[03Xj
004IOcX0=00Z=B/^:`0[02Xe02X0:bh0:SDd02X0=C@0;RX0;P0Z000^02hZ000^
=00Z=00Z000Z;S@0:P0^:bh0;S@0=B/0;R/^02X^02/^03@0=00Z03D[;P0Z=00[
;S@0;S@0:P0[;P0^=00Z02h0:Rhd000Z03@0;R/^02X0=00dO`0Z03@d000Z03@0
:P0^=00^;P0Z02X0:P0^=00d02hd02X0=B/^02/^03@0:S@002h[02h[;P0Z03@0
:bh0:S@0;S@002h[;P0d:bh[;P00:P0[;R/0:S@002hZ02/^=00^:bh0=000;R/0
;S@0:P00;R/^:bh0=3@0:bh0;S@05BX0=00Z=@0[;P0^=00e02X002hj>P016Glj
;P0Z000^:`0d;R/0:P0d;P0[000Z02X0=00^:P00=00Z03@^03@0;S@Z02h0=BX0
03@d02h0=00d02h0:P0Z02X0=00e:bX003@^03@^:bh0:bh0=00d=@0Z02h0=00^
03@^02/d02h^02X0:P00=BX003@^:P0^03@0=2h^07ld02Xe02X0=BX0:P0d02X0
:`0d03@0=00d02hZ03@0;R/0=00e:cD^02X0=@0^:c@0;R/0;P0d;P0Z03@0=3D0
:P0Z;R/0=00d;P00=00d=00Z=@0e:P0^:c@0;R/e02X0:bh0:P0d;P0d;P0Z000d
=00e02X0=2h0=2h0:P0^01D0=2h002X0=00d02h[;R/0=3@0>SX00AUo>P0d02Xe
:`0[;P0^=00[;P0d02Xd03D0:P0^:`0^=00Z=@0Z02/^=00^02h[02X0=B/^;S@0
:P0^;S@0=00d02X002Xe:bh0;R/^:`0Z02h0:SD0:bh[;P0Z;S@0:bhd02/^02hd
03D[03@0=00d02X0:SDZ000Z03@0;Rh[02io:P0^02hd02X0:P00;S@0;S@0:P0^
=00^:P0Z000^=00^=00^:P0Z03@002X0:P0^=00^=00[;P0d02X^:bh0;Rhd000^
=00Z;P0d=00^:P0e02X0:P0d03@0;R/e02X0;S@0;S@0:P0[;P0[=C@d02hZ03@0
:bh0;RX0=00d:`0E:Rh0=3@0:bh[;S@0=B/0;RX0;SXj004IOcX^03@0:P0d=@0d
02h[=@0d;P0e02X^03DZ02h[02hZ02X0=BX0:bh[03@0=BX0:P0e:`0^:`0d02X0
=2h[;P0e:S@0:P00=00e:bh[03@0=00Z03D0;P0Z000Z03D[;P0d03@0;RX0;R/^
03@^03@0=00Z03@^03@08`00:P0[O`0d02X0;R/0=2h[=@0Z02/^03DZ02h[000d
03@Z02h[02X^000d03@^:P00=00d02X0:bh[=00Z03D003@0:P00:P0d02X002X^
02X^000Z;P00=00d02Xe:`0^:`00=00^:`0^:`0d;P0d;P0^02X002X^03D[02/0
=2h0;P0d5@0Z02h^03@0;P0Z02X0=00^:`0j>P016Glj03@0;P0Z;P0d02hd02X0
:P0d02X^03@[02h[03D[03@002X002X0=@0Z02X0=000:P0^:`0^;S@002X0;P0d
:P00:P0^=00^:bh0=@0Z02hZ03@0:P0Z02hd03D[;P0[;S@0:R/0;R/e02/^02X0
;Rhd000^03@0:P0Z=00e=7l0;S@0:P0e:bh0=@0Z02Xe03@[02X0=00Z;S@0;S@0
;S@002Xd02hZ000Z=BX0;RX0=C@002h0;S@0:S@0:P0Z=00Z;S@0;S@003@002Xd
02/e:bhZ02X^02hd03@0:Rhd02hd000Z=@0[;P0Z03@0;S@[02h[;S@0;Rh0=00[
;QDd03D[02/^=00[;S@0:bhe:`0^>SX00AUo>RX0:P0d02hZ03@0;P00=2h0:P00
:bX0;R/0=BX0;R/^:c@0=3D0:P0d02X0=2hZ02/e:`0d02X0=2h0:P0d02h[=@0Z
02X0:`0[=00Z03@0;R/^:`0d03@0:P0Z02/e;P0^:`0e:P00:P0d03@^:`0[;R/d
02X^02h002X^:bio:`0^:`0d;P0[:P0d03@0:Rh0;R/0=2h0=00^:P0^:`0^:cD0
:Rh003@^02X0=00^:bh0=3@0=00^:P0^02h002X0=00^:`0Z=BX0=2h0;R/0:P00
=BX002X0;R/^:`0[;R/0=2Xe02X0=00Z02X^02/^03@0;P0Z02/0:Rh[=@0E:P0Z
03@^02h[000Z03D[03DZ03Xj004IOcX^02hd03D[02X0=00Z=BX0=00[:SD0=00d
02X0:R/e000^=00^:bhd02Xe02X003@0:P0^;S@0:P0^=00d02X0=@0d03@0:bhd
02h[;S@0;R/e03@0:P0^=00Z03@0:P0Z03@0;P0Z=00[;R/^:`0^=000;R/^03@0
:S@0:`00Obhd03D[;P0Z=@0Z02h^=000=00e:bh0:bhd000Z02hd02h[=000;R/^
:cD[02hd02X0=00^:bh[;P0d03@[02Xe=00d02hd02/0;RX0=00^=00Z=B/0;S@0
:P0d000Z=@0^=00^03@0:bhd03@002Xe:`0^=00Z03@d02X0=@0[5Bhd02/^03@0
;S@0;S@0:P0Z02hj>P016Glj03@0:Rh0;R/0:Rh[02X0=2h^000Z;P0d;R/0=@0^
:c@^02X^02X0;R/0:S@0=2h^000d02/^000d02X^03@^:bhZ03@^03D0:P00=00^
:`0^:bh0=2h[02h0=BX0=00Z03@^:`0d02hZ000e02X0:bh[=@0Z03@^:`0^;P0d
:Wl0;R/0=B/^02/^03@0:bhZ02X0:P00=2h0:P0d03@0:P00=2hZ02/e02X0;R/0
;R/0=2hZ02h0;R/^03@^03@0;P0Z;P0[=Bh0=00^:bh[02h002X0=00^:`0d;P0d
02X0:bh0:bh0=2h0;R/^:bh0:P0d02X0=2h0;S@0=00Z01D0=2h0:P0Z02/e:P0^
:`0d03@0>SX00AUo>P0^:bh0=00d02h0;S@0=00^03@d000[;P0^=00d:`0^03@0
02X0:P0^=00^02h[03@d02Xe:`0Z;S@002/^02h[000^02X^=00d:Rhd02hd02h[
02X0;SD[02X0;Rhd000^02h[;S@003@^:c@0=C@002hd02/^02hd03@0:P1o=B/0
;RX0:`0Z03@0=00^03@0=00^=2h0:P0e:P0^=000:Rh002Xe02X0:P0^=00^=@0Z
000[=B/002X^02/^=00d02/0:P0^=00Z02h0;S@0;S@Z02hd03@0;P0Z;S@0:SD0
:S@0:bh0:P0Z03D0=000;S@0:bh0=00Z02hZ02hE=00^=00d03@0:P0e:`0^;S@0
;SXj004IOcXZ02h0:P0d;R/0=00^02X^02X0:SDZ02/^02X^02h[03@^:cD0=00d
02h[03@^03@0:Rh0:P0d02X0=2h0:bh0:P0d;R/0:bh0;P00:P0[;R/0=BX0:P0Z
03@0:P00:SDZ02X0=@0Z;P0^02X0=00Z;R/0:P0e:bh0:P0Z=00dO`0Z02/0=2h0
=BX0:Rh[03@^;P0d02X0=00d02h[02Xe:P0[=BX0:bh0=00d02X0:bX003@Z02X0
=2h0:cD^02X^03@^:`0d02X^03@0:P0^:`0^03@0:Rh0=00d000^02/^000e:P0[
=@0Z03@0:Rh[=2h0=2h0:P0d03@0;R/05BX^02X^03@^;P0Z02X0:`0Z;P0j>P01
6Glj03@0=00^:P0e:bhd02X003@0:P0[02Xe03@0:cD[02hZ000Z02hd02X0=B/^
03@0:Rh0=000;S@0;RX0;S@0=00e:P0d02h0=00^=00^=00e02X0=00d02/^=00^
=00Z03@0=00[;P0d02/^02hZ000^=00d:P00:P0d02h0;WlZ03Dd02hd02X0:bh0
=00^:`0[;S@002X0;RX0;S@0:P0d02X002h[02X0;S@0:bh0=BX0=C@002X0=00Z
02X002hZ03D0;S@002/^=00d02hd:`0^:P00:Rhd02hd02Xe02Xd02hd02hd000^
:P0[02h0:bh0:P0^=BX0:P0e:aD^03@002hZ000Z03@0=2hd000Z>SX00APh>Rh0
=2h[000d02h0:P0d=BX0=3D^03@0:P0Z02h[03@0;R/e03@0;RX0:P0e:bh0=000
=2h[=@0ZoP1o=00Z02X0:P0Z03@0=2h[02X0:bh[;R/0=3D^03@^02h[02X003@^
02X^02/d;R/d03@0;R/d02h0;P0d:P0e03@0=00^:bh0=00^:`0d;P0Z03@0;R/^
02X0=2h[000d02h003@^02/d;P0d=@0d02X^02/e:bh^02X^:`0d;P1I03@0=2X0
03@Z;P0Z03@^02X0:P0[;P0d02h[;P0[;RX0:P00:P00=2X0;R/0:S@^000d=BX0
=2h0=00d02/e;P0Z03@0=2h[;P00=2X0=2h0;P0Z03@0>SX00AUo>P0Z02X0=B/^
:bhd03@0:P0d02X0:P0e02X0=00^=00^=00[;RX0:P0d02h[;P0Z03D[;R/0;R/^
02Xd;R/^02hd03@0;S@0:P0^=00e=000:cD0;R/0:bh0:P0^=00[;RX0=000:SD0
;P0e=00Z02he:bhd02/^03@0:RhZ02io:cD0:bhZ03D[;P0d03@0;R/e02X0=BX0
=@0Z;S@0:S@0;S@0;P0d02h[;RX002Xe02X002X0:cD0;P0Z=00^:P00;RX003@0
:bh0:P0d02Xe02/^=00^02Xe000Z03@d000Z;P0Z02h[;P0[=B/^:P00:P00:P0^
:bh[03@0:bhE=00[=@0Z=BX002h0=2/0;S@0;SXj004IOcX^03@0=00Z02h0:P0^
;P0Z03D0:P0d:bh0=2h[02hZ02h[02/0=00d;P0[02/e02X0=00^:`0d03@0;P0d
03@0;R/^02/^:P0Z02h[;P0d:P0Z02X0=2h0:P0d02h0=00^:bhZ000Z02/^02X^
02X002X0;R/e03@0:P00:cD[O`0Z03D0:bX0=B/^02X^:`0^:`0d;R/0=00d02/^
02h0=00Z;P0d;R/0;P0[=BX0:P00=2X0=@0Z=00d;P0d02h[=00^:cD0:Rh0=00d
;P00:P0e:bh[03@0:P0d02Xe02Xd000d03@0=@0d;P0Z000d=3DZ000d02h0;R/^
03@05BhZ02X002X0=3@0:SD0=00Z;P0j>P016Glj02X0;Rh[=C@0=000:P0d02X0
=00Z03D0:bh0;SD[03@0=00^=00^;P0[;S@0;R/^:bhZ03@0:P0^=00[;S@0;R/e
02hd000e02X0;P0Z03Dd03@0;Rh0=000;S@0=00^:cD002Xd02Xd02X003@0:S@0
;R/e02/^:P0^=00Z07lZ03@0:SD002X0;S@002he:`0[;P0^=00Z;R/^03@0:Rhd
02/0:P0^=00d02X0=00[;RX0=00d02h[;P0[;S@[02hd000Z:bh0=00^;P0d=00d
02X002/^=00^:P0^02hd02hd02hZ02X0:P0d000Z;P0Z000Z;S@0:P0Z03@0;QD[
02h0:S@0=00^:P00:Rhd000^>SX00AUo>Rh0=00Z000Z;P0d:P0d;P0^:P0d;RX0
=2h0:P0Z03@^03@^02X0:`0d;P0Z02/^02h002X^03@^02X0=B/^:`0^:`0Z03@Z
03@0=00d;R/0;Rh0=00[:P0d=00Z;P0d02h[=2h0;P00=BX0=2h0=00^:`0^:bh0
02X0:bh0;P1o=BX0:P0Z03@0=00Z;R/0:P0d;P0Z02h002/e03@^:`00:Rh0=00Z
02h[;P00=2h0=00^02X^02X0;P0d;P0^03@0:P0d03D[;P0d02X0;Rh[;P0d;R/e
02h[000d02X0;R/0=2h003@0=2h0:S@^02/e03@d02h0=00d03@^;P0E:P0[=@0Z
;P0d02h[;P0[;R/Z03Xj004IOcX0=00Z02hZ000Z:cD0;P0d:`00:P00;RX0=00^
:`0^02X^03@0:SD[;P0d02Xe02X0:SDd02/^03@0;R/^02hd02hd02Xe02/^02X0
:P0^02X0=00Z=@0Z02hZ000Z;S@[02h0=00Z=2X002X0=00Z02hd:`00:S@0:SD0
=2/0ObX0:P0d03@0:Rhd000^=00[;P0d02X0=3@0:P0Z03Dd:P00:bhd02X0;P0d
:Rh0:bhd02/^03@0=B/0:P0^:`0Z02hd02h[;P0[;RX0=00[=@0[;P0d02h[=@0Z
;S@0:P0e:bh0:S@0;RX0=00^02/^02X0;S@0=00^:P0^:`0Z5@0Z=@0[;P0d02X0
=B/0:P0e02hj>P016Glj;P0d02X003@^02h[:P0Z03DZ;P0d;P00=2h[03DZ02X0
03@^000Z02/e03@0:P00=00Z03@^03@0=00Z02X0:P0[;P00:Rh0:cD0=00Z03@0
:Rh0:bh0=BX003@^000Z03@0=2h002h0=3@0:Rh003@0;P0d:P0^000Z03D^:gl0
=3D^03@^;P0[;R/^02X0=B/^03@0:Rh0=2h[02X0;P0d=Bh0:P00=00d;P00=2h0
;R/e02X0=00d;P0Z03DZ03@0:Rh002/d;P00=2hZ02X0=00d;R/0:P0d02/^02/d
02h0=00^:P00=2h0:cD0:P0^:`0^:P0d02h[03DZ01De02X0=00d;Rh0:P0e:P0Z
03@0>SX00AUo>P0Z;S@0:SDZ03@0;P0[=@0Z02/0:P0[;RX0=00Z03D[02hZ03@d
02/^02X^=00e:bhd000^02X0:Rhd03@0=00[;P0d=000:SD0:Rhd02/^=000:bh0
=00[02hZ02h[=C@0:Rh0=3@0:bh0:SD002X^=00d;P0d03@d02hZ03Ao02h[02X^
02/0:P0e03@0=B/e:`0Z02h0:bh0;S@0:P0d02X0:P0^=3@0:P0e:bh0:P0d03@0
:P0Z;P0d02/^:`0^=000:S@0;P0Z=00^03@002h[;P0^=00[;R/^03@^03@0:R/e
:`00;RX0=00Z=00d02he:`0Z02h[=@0Z03@E03@0:bh[;P0[=B/0:P0e02X0;SXj
004IOcX^000Z000Z03@^:`0d;P0d03@^000d;P00=2h[03@0=2X002X^03@^02X0
02X^:bh0:Rh[03@0=00^02Xe02X0=B/^02X^:`0Z000Z03@^02X^:cD0=2hZ03@0
03@0:P0^000Z02X^03@0:P0Z03@0:P0d02X^03@^03@0:cD0ObX0;R/0:cD^:`0d
:bh0:P0^03@^03@0=2h0:P0^000d;Rh0:P0d02h0:P0Z03@0=00d;RX0=2h002X0
:P0e:`0d;P0d;P0Z02/e02X^02X0=2X0=B/^02h0=00e:bh0:bh^:`0^:bh[=000
:Rh002X^03@0:P0d03@0;R/^02X05BX^03D0:P0Z02X0=00Z03@0=00j>P016Glj
02hd02Xe:`0^02h[;P0d02X^03@d;P0Z:bh0;S@0:Rh0=C@002X^03@0;S@002h0
=000;S@0;Rhd02/^02X0=B/^03@0:`0e02Xd02h^03@0:cD0:bh[02hd02X^=00Z
03@d02/^02/^=00e02X^=000;RX002/^02X0:SD0:Wl0=00^=00Z02hd02h0:bhd
02/^02/^:bh0:P0e:`0Z;R/0=000;S@0:bhd02/^=00^:P00:bh0:S@0;S@0:P0^
:P0^:P0d03Dd03@003@0:bhe02X002X0=00^:bh0:SD[02hd03@002he:bh0=2/^
03@0:P0d02h^=00Z03@0:QD002<0=00d03@002hd02/^:P0Z>SX00AUo>RX0;P00
:P0d03@0=B/^03@0;R/^02/^03D[;P0Z000Z02X^03@0:cD0=00Z;R/0=2h[=00Z
02/0;R/^02X0=00Z03@^;P0d:P00=2h0:bh0=2h0:P0[;P0[;R/002X0;R/^03@^
03@^02X^02X003@Z;P00=3D^02X0=00Z;P1o=2h[02h0:bh0:Rh0=2h0:P0d03@^
02h0=00d03@Z000d:P0d:P0^:P00=BX002X^000d;P0[;P0Z02/e:P0d000d02h[
;R/^02X^:bh0=2h0:c@0:P0d:P0d02h0:P0^02X0;R/^:cD[03@0=2h0:cD[;P0Z
;P0[02X003@^03DE:P0^:P0d;Rh0=3@0;R/^000d=CXj004IOcX0=B/0:S@0;S@0
;R/^02X^=00^02Xe:`0[;P0d02Xd03@002/^=00^:bhZ02/0:bh[02h0;S@0;R/e
03@002hZ000^:`0^=000:Rh0:bh0:Rh0=00[;P0Z=@0^=B/^=00^02X^02/^03@0
02hd02X^03@0:P0Z02h[02hd000[Obh0;S@0:S@0=000:bh[02hd02hZ03@0:bhd
02X^03D[;P0[;P0^:`0d02X0;S@002/^:P0Z=@0d03@0;P0d03D[;S@0;P0^:`0Z
03D0:bh0:P0^=00[;P0^;S@0=00d03@0;R/e000Z03@0:Rh0=00Z02hd000d02hd
02X^03@[5@0^:`0^:P00:bhZ02hd000Z;P0j>P016Glj;R/0=3D0:P0Z02/0:bh0
02X0:cD0:P0d;P0Z03D0:Rh0=3@^02h[02h003@e:P00=2h0=00Z02/^02X0=3@0
;R/d02h[02Xe:P00=2h0=00[;P0d;P0d02X0:`0e02X0:cD0:cD^02X0=2X003@0
;P0Z=00d03@0=2h0:P0d;Wl0:P0^000e:P0d:P00=2X0;R/003@^:bh0:P00:P0Z
000d;P0d03DZ03@0=2h0=2X003@^02X0:Rh0=00d;RX0;P0Z02/e02X003@[=2h0
:P0d02hZ000d02/^:bh0:P0Z02/0:P0d;Rh0:P00=2h0:cD[;R/^03@0;R/0:cD0
:QD0:bh0:`0d;P00;R/0;R/e02/^>SX00AUo>R/0;RX0;S@0;S@0;S@0;S@0;S@0
=00[;P0d02X0=000:P0^02X0;S@0:S@^03@0:Rh0=00Z03Dd03@0;RX0=B/0;S@0
;S@0:P00:S@0:bh[;P0Z;P0[;S@0:bhd02X0;S@0=00Z02X0;RX002X^=00d000^
:P0^:bh0=00^:P1o:P0d03@0:P0^:P0e:bh0:P0e:bhZ03D0:P0Z=00d03D[;S@0
:P0Z02hZ02X0;RX0=B/^03@0:bh0=00^:P00:P0Z02Xd03@0=3@0;Rh0=000;RX0
02hd:P0Z02h0=00d02Xe=00[;R/0;S@0;RX0:bh0:P0e02X^=000:SD0:P0E:SD0
:SD0;R/^=2/0=B/0;R/^03Xj004IOcX0=00^:`0^:`0Z03@^:`0Z02/^:P0e;P0Z
03@^:P0d;P0d03@0:P0^:`0^03@^03@0=2hZ03@0;RX0=00^:P0^:`0^:`0e:P0d
=@0^:`00=00d02/e;P0^:bh0:P00=00^:P0d03@0=000:SD0:bh[;R/e000S02h[
;P0d000dO`0d;RX0=00d02h[;P0[=@0Z03D[02X0=00e02hZ03@^000Z=00d03@0
;R/0=000=2h0:cD0=2h0=2h[000d;P0d03@0;RX0=2h0:P0[;P0d=00^:cD0;P0d
03@0=2h^03@0;Rh0=00^:`0^:`00:S@0:P0Z02X0:bh[=@0Z;P0d5@0Z000Z02/0
=2h0=2X0=00d03@j>P01618j:P0Z02hd02hd02/^02hd03DdoP1o:P0d000Z000^
:P0d02X^=00e:`0e:`0Z03@0;RX002X0;RX0:bhd000Z02hd03@0:P00;R/e:`0^
:SD[;S@0:P0^:cD0=00d:bhd:`0^:P0^=00Z=C@0:bh0:P0e02Xd03D[02/0:P0Z
;S@0:P00;RX0:P0e02/^03@0:P0^=01o:bhd02X002/^02Xd02h0;S@0;R/e02h[
;RX0:bh0:Rh[02X0;R/^:P0[;S@0;R/0;Rh0:bh0:P0d02hd02h[=00d02h^=00Z
03@0;R/0:bhd02hd02hd=00e=00Z02hd02h[02hd02/0;S@0=3@0:SD[;P0^=000
:P0^:P0j>P016Glj03@0=00Z02/^03@0:P0^:bh0:S@^03@^:P0d;P0[03@^;P0[
;R/^03@[;R/0=2h0:`0d;P0d000d;P0Z03@0=00Z;P0d03@Z02h[;P0d02h0:bh0
03@0:P0d;Rh0=@0^03@0;R/0;RX0:P0d;P0d03@0:P0^:P0^=2h[03@0;Wl0=00d
:P00=2h[03@^03@^02/Z02h0=00^:P0d=Bh0:P0^02/e02X^000Z:`0e000d;P0Z
000e:P0Z03D003@^02h[03DZ02/^02/d02X^:`0^:`0^03@0:P0[;Rh0:Rh0:cDZ
000Z02/^:`0^;P0Z03D0:P00=00e:P0Z03DZ01D^02X^000Z000d02X^:`0d02hZ
>SX00AUo>Rhe:P0Z02Xe:`0^=00Z02X0=00^02/^:`0[;P0Z=00^:`0[;P0^02/^
=000;Rh0=00^:P0d02X[;P0Z03@0:Rhd02/0;R/^02X0;P0[;S@0:bh0=3@0:P0Z
02X0=00Z03@0:P0^=B/0=000;R/0:P0^:P0e:`0Z02h0=00^=01o:P0Z=@0[;S@0
02/^:`0Z02Xe03@0:Rh0:P0d02X0=00d02/^:c@002Xd02hd02/^:P0d03@0:P0d
000Z=00Z02h[=@0Z02Xe02/^=@0Z03D0:P0[;S@0;RX0;R/0=000=00Z02hd02Xe
:`0e:`0d02X0=00Z=00^:P0d000Z02XE03@002Xd03D[;R/^03@0;S@003Xj004I
OcX0:cD^02X0;P0d;P0d03@0=2h[03@^02h[=@0d02hZ02h[02/e:bh0:cDZ02/0
:Rh0:`0d;Rh0=B/0;R/^000Z03@^:`0e:bh0=00d;P0^:bh0=00Z;P0d03@0:Rh[
02X^02/^02X0=2hZ02/e:P0d02h[;P0d03@0=2hZ02h[O`0Z02X0=B/^:cD^03@^
02X0:P0^:`00=00d;Rh0:P0d;P0d02h0:P00=2h0:Rh0:`0d;R/^:`0d;P0d02Xe
02X0:P0d02X0:P0d02X003@Z000d;P0Z02/0=00^:bh[=B/^03@0;P00:P0d02X^
02X0=2h002X^000d;RXd03@05C@^;P00=2h[;P0^03@^:P0^:cDj>P016Glj02X0
03@0=00[;R/0;S@0;RX0=B/^:`0[=@0[;S@003@0;S@0:P00:SD002Xe:bh0=3@0
;P0[02X0=B/e03@d000^02hd02h[02X0:P0^:cD0=00Z;P0d02h^=000;S@002Xe
02X0:Rh003@0:P00;S@0:P0d02hZ02X002X0;Wld03D0:P0d000Z02/^03@0=00d
02he=00^:P00=00[;P0Z;S@0:P0d:Rh0=00[=C@0;R/e000Z=@0[;R/^03@0=00d
02X0=@0[;S@0=B/^02Xe:P0d02/^=00^:cD0:P0^:`0^:bhd=00d02X^03@002h0
=2X003@[;R/0;P0e=1D0:P0d:bh0:P0e:`0Z000d03@0>SX00AUo>Rh0:Rh0=2h0
=00^:`0^:`00=00e03@Z02X0=B/^:cD0=00e:P0d:P0Z;P00:P0[=Bh0=2h[03@^
:`0Z03@^;R/d03@0:P00=3D0=00Z000Z02X^000d;P0[02hZ02/^:cD0:P00=00[
=BX0=00d;P0Z000d;R/002X0=2h0=01o;RX0=2h^03@^03@^02X0=2h0:P0[;P0d
02h[:P0d;P0Z000^:`0d=@0[:P0d;P0Z02/^03@e02X^02/^02X^02X0;RX0=00Z
02h0=2X0:bh0:P0d;P0d02h[02h[000d03DZ02h0;Rh0:Rh003@^:P0Z02h0=3@^
000d;P0Z02hE:P0Z03D[;P0Z03@d=BX0;RX0:SXj004IOcX0=000:bh0:bhd02hd
02he:bhZ02X^03@0;R/^03@0:P0^:P0[;P0d000d=00d02h[02X0=B/^02hd02X^
02/0;S@0;SDd:bh0:Rhd02Xd02/^03@0:P0[;S@002Xe03@0=00d:bh[;P0^=00^
;P0Z02Xe:P00:SDd02h0=00^Oc@002X002/^:`0Z03@0:Rh0=00[;P0Z;S@0=000
;P0Z02Xe:`0Z;P0d02h0;P0Z02Xe02X^:c@002Xe03@002Xe02X002h[;S@0:bh0
:SD0=000;P0Z;S@0;SD[02Xe:P0Z02X0:R/0=000;S@0:P0d03@[02hZ03D[;P0Z
02h[5@0^02h[;P0d03@0;P00=000=00j>P016Glj:P0d:P00=2h0:P0[;R/0:P00
:P00:P0e:P0^02X0=2h[000d;P0d;R/^02X^03@0;R/0=00e:bh0:P00:P0e:P0Z
02/0;P0[;P00;P00:SD0:cD0:P0d;P0Z03@0:Rh0:Rh0;P0[=@0Z02h[02/e02X0
:`0Z=@0Z03@^:bh[07l^:c@0=2X003@e02X0=00^:bh0=00d02X0=2hZ02/e03@0
:P0e02X^03@0=00d03@0:P00:P0^:P00:P0d:P00:P0d=BX0;P0^:bh0:P0Z03@^
:`0d000Z02/^03@0:P0d03@0=00^:bh[=00Z;P0d;P0e;R/003@[;P0d03@0;QD[
03@0;P0d;R/^:P0Z;P0Z;P0d>SX00AUo>P0^;P0Z=BX0=00[;P0^=00^=00d=000
:P0d02X0:Rh0;S@0;R/0:P0e02X002/^=00^=00^:bh0:P0e=000:P0d02/^=00^
=00^:bhd=00^:bh0:S@0;P0Z02X^=000:bh0=00Z=@0d02h[02hd03@0:SD0:P0d
000Z02h0;SAo03@0;Rh0;S@0;R/^02hd02h0:Rhd02X0:P0^03@0;R/^02h0:S@0
02/^:bh[;RX0=@0Z=000;P0d=00Z;P0Z=00d02X0:P0^:cD0=00d000Z02hd02hd
02Xd02/^=00e02X^02hd03D[02hd000d02X0:R/0=B/^;P0[;R/^=00E;R/^=00d
02/e000[;P0d02/0;SXj004IOcX^02/e02X0=2h0=00d02X0:bh0;Rh[=00d03@0
=000=00Z=@0d;P0Z03@^:cD^02X0:bhZ02h0:P0Z02Xe:P0d02Xe02X0:bh[03@0
;P0d03@0=00^:P0d03D0:bh[=Bh0:P0d02X^03@0=2h0:Rh0=00Z;P0d;RX003@0
=00ZObh0=00[=B/^:`0e:bX0;R/0=000:Rh0=@0Z03@^:`0e:bX0=00^:P0e02X0
;P00:Rh003@d:P0d02h002/e02hZ03@0=00d02X0=2h0:SDZ03@0:Rh0;P00=BX0
02X^:bh0:cD0:P0Z03@0;RX0;P00=00e:P0Z02/e02/^02X05B/d02hZ02X0:P0d
;P0d;RX0=00j>P016Glj:`0Z03@002h0;Rhd02X0:SD0:P0[03DZ02h^:P0e:bh[
;P0[;R/e:`0Z000Z02X0;S@002X0:R/0;S@0:`0Z02h^02X0;S@0;S@0:P0^:P0Z
02hZ000d02X0:SD002X0:P0Z;S@002/^:bh0=00[02hd000d02h0=2/^:bhd07l[
02h[;R/^03@0:P00:P0^=00^=000;R/e:`0^:cD0:P00:SD0:P00:S@002/e:bh0
=2X0;P0Z02h^=3@0:P0Z03@0;RX0;S@002X0=00Z000^:bh0=00d02X0:S@002h0
:bh0:P0d02Xe02X0=00Z=BX0:P0Z03@0;R/^03@0=1D0;P0Z000^=00[;P0Z02h0
=00^>SX00AUo>Rh[02X^:c@0:P00:Rh0=@0Z03@^03@0;R/002hZ02h0:P0d;P0^
02X0=BX003@0=00^:cD0=00e:P0^03@^03@0:bh0:P0^:`0^:`0d02h[03@0;R/e
02X0=00Z03@0=00d000Z;R/0:cD0=2h0=2X0;R/e02X0=2h0:bh0;R]o=BX0;P0^
03@^:P0d;P0Z02X0:P0d:P0^:bh[02h[=00d;P0Z03@e:P0^:c@^02X0=2h0=00d
03@0:bh^000d03@^;P00=00Z;R/0=2h003@d02h0=2hZ03@0=3D0:R/0=2h0=2h0
:P0^03@0=2h[02X0=00d;P0Z02/e03@0:RhE02/e03@Z02h[=@0d03@0=2hZ03Xj
004IOcX002Xe000^=00Z=000;R/d02/^02/^;R/0;S@[03@0=00[;P0Z03@0=2/0
;S@Z02hd:`0^:bh0:P0d02/^:`0^:bh0=00d02hd03@0;S@[02h^=00d02Xe02hZ
03@0;Rhd02hd000^=C@0;RX0:bh0:P0d02X0:bh0:SD0;R/0Obh0=00d02/^:`00
;P0d03@0=00^:P0e:`00;S@0;P0Z;P0d03@0:P0^:`0^03@002h[02hd02hZ02X0
02Xe:P0^:`0^:bhZ02/0;Rh0=2/^;S@0:bh[02/^:bh0;Rhd02h[02X0:cDd02/^
:bh0;S@0;S@0;P0d02Xe02X0:SD05C@0:P0Z02h[02h[;S@0;RX002hj>P016Glj
;R/0:P0d02h[03DZ02/^03@^03@^02/^:`0^03@^:bh0=B/^02X0=2h0=00^03@0
;P0d03@0=00d;Rh0:cDZ02h0=2h[;R/0;R/^:`0^03@0:P0d;RX0:cD002X^:`0[
;Rh0:Rh[02X0:P00=2h0=00d;RX0=2h0:P0Z02/e:Wl0:P0d;R/^03DZ:`0d;P0Z
;P0d000d:bh[=@0Z02/e02/e02X^000d03@e:bh0=2X0;R/0;RX003@e:P00:P0d
02h[03D0:P0e:P0[;P0e:`0^:bh0;P0d02h0:P0[=BX0=BX0:P0^03@^02h0:P0^
02/^:P0Z03D0:P0Z02h0=1D^03@^03@0=BX0;P0Z02/0=2h0>SX00AUo>P0^=00d
02h[=@0Z02Xe:`0^:`0Z02/e02hd02/^02h[02X0:cD[02h0:Rhd02/^=00[;RX0
=00^;R/0=00Z03D[02X0=000;R/e02hd02/^=00Z;P00:P0Z02<002h[;P0[=000
;S@0:P0e:bh[02hd02X002h0=00d03@0:P1o=00[;P0d000Z02hd02hd000Z03D[
;Rh0=00d03@0;R/^02X^03D[;R/^02X0;RX0=00^=2/002hZ000d=000;RX0=00[
;S@0:P0Z=B/0:P0^:cD0=00[;S@0:P0Z02h0;R/0:P0e:`0Z02X0=00d02Xd000d
03@0:P0d03D[02XE02/^:`0Z02X0:P0Z02Xe:bh0:SXj004IOcX^02h[;R/0;R/e
02X0;P0d03DZ03@^02X0:P0d03@0=BX0=2h0;R/0=000:P0e:bh[;P00:Rh[000d
;P0d03@0=2X0:P0d;P0[;R/0=BX002X002/e:P0e02X0=2X0=@0d;P0d:P0^:`0d
03@^03@0;Rh0=2X0=2h0;R/^000dObh0=B/^03@Z03@0:Rh0:Rh[02X0=00Z02hZ
03@^:`0e:bX003@[;P0^02X0=000=2h[02h0=3@0;RX0;Rh[=000:Rh[02/0=00e
02X0=00d02h[;P0d;P0^:`0d03@0:P0e:P0d02X^03@0=2h0:P00=BX0;RX0=2h0
:P0Z03@^5@0d03DZ03@0=00d03@0;P0[=@0j>P016Glj02X0=B/0;R/^02X0=00d
02hZ000Z03@0=00Z=B/^:bh[02h0:P0^=00^:bhd02X0=@0d02X0=@0Z;P0[=BX0
:Rh0=@0^=00[;R/e02X0;S@0=C@0:P0d02X0;S@002X0:P0[;P0Z02hd02X^03@0
;R/0;S@002X0;R/e02Xe:Wl0=00Z02hd000^=000:P0[=@0^:cDZ03@0;P0^:cD0
:P00;S@0;P0d03@002X0:Rh0;S@0=BX0;R/0=B/0:SD0:SD002Xe02h^:c@002hZ
03@[02/0:P0Z03@0;S@0;S@0:P00;RX003@0;RX0:cDd=00^=000:Rh0:bhd02/^
01DZ=@0Z02hd02h0;RX0=00d02/^>SX00AUo>Rh0=00d;P0[03@^02X0=2X003@d
02X^02h002X003@0=BX0:P0d02X^02h0:P00=2/^;R/0=2/e02/^02X0:P00=2/^
02h[=000:P00=00Z;P0Z;P0Z03@0=00^:P00=2h0=00d03@0;RX003@^:P0^:`0^
:P00=00Z02X0:`1o:Rh003@^03@Z02h[=@0d;P0d02h003@^:`0d02h[:P0d;P0Z
:`0d;R/^:cD0=00[;P0Z02X0=00^:bh0=00Z03@0=3D0:P0[03D[;RX002Xe02X0
=00d03@^:`0^:`0^:`0d=@0Z03@^02/0=2h0;P0Z02X0=000=2h0:P0e:bhE02X0
03@0;RX0=00^;P0d;RX003Xj0008PCZ1>^`j00002P0cPCZ1>^dj00000:00P`3o\>"], "Graphics",
  ImageSize->{277, 295},
  ImageMargins->{{35, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}}]
}, Closed]],

Cell[CellGroupData[{

Cell["Capabilities", "Subsubsection"],

Cell[TextData[{
  "Once we have a graphic in ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  " there are a number of ways we can view and manipulate it. First, just by \
clicking on the graphic and moving the mouse, we can impart rotation in the \
direction that we moved the mouse with the speed of the rotation relative to \
how fast we moved the mouse."
}], "Text"],

Cell[TextData[{
  "We can also manipulate how the graphic is visualized and how we view the \
graphic with commands built into ",
  StyleBox["MathLive",
    FontSlant->"Italic"],
  ". For instance, we can change the surface lighting, make the surface solid \
or wire frame, or map a texture onto the graphic."
}], "Text"],

Cell[GraphicsData["PICT", "\<\
>IH000000BL15@0A0_l<0?ooool000000000004E00019`000000000N01[ooooo
ool00@0:0000004W0AD0P@0:000000000000X02201X000000000VH4F0000004W
0AD00000000000180000B0000000200100P000000>LN^000000001B4P000RP00
oooooooo004c<000c<`00VETC4c<c003^k^k^k^k00@008:2PX801Lg=FEUIF@06
c<bc/P0000NIVC<c<c<02;2`000000090036a/K600[oofEUc<T02c<c/kG;b`0<
<c?oolc<00g6aP00a/403P00V9QUIP0?003<c9VI011VJL_;<c<04M;e8OoBl@0B
<c?;bfIW01=VI/c<VIP05<_;/[4c<`0Eo_jb[`0001H003<fo_h05l_;IFDc<`0H
<c==C/_;01UVI4e=VIT06P00o_k;b00K003<cOkn01bIU000c<`07FER002HV00N
c<`009VG01onoP00blH08000@45JFP0Q8OoBmM;e028c<oooool08iRH<S80000T
c<`c<00002F9S=3@L7009YbMm_JLW00WoomVHc<c02SLg9>CB4P0:IVLc<`0000Z
M7AdM7Ad02^HV<c<c<`0;<T]g7O9;@0]oolc<fIQ02kblVa/K6`0;m;eNOPQo`0`
o_iUH`00037<c<c<c<`0<ShnF5S^kP0cIV<c<ooo03@c=L_;<c<0=@03][H0000f
d_GBlR7o03O<dokn0000>000HV9RHP0i001UJ9RH03X00EQH0000>`02SXh0000l
g=a8B=cI03gooc<cooX0?VGoUoiToP0oooo;boon040009bLW9`0@Lc<C4c<b@12
c<M=COoo04?;bl_<ool0A6QX<c8000158OlRom;e04Hc<c<fool0Ab<3d_DQo`18
<cKooc<c04W>nBWo:Ol0B_oo<c<c<`1;5aLS8aLG04`W:4m?9bL0CJBTY:40001>
a/K6``0004la<A4A8200D59B3P`0001AF5g^kShn059UIoooIFD0DdQ8g=cLg01D
LG7UiNGU05E7AdM7AdL0ES7M<MiT^@1GXIh00?kn05S;b000o_h0FHB4Q8B4Q01J
c<a<C6ES05/aZbbc;;<0G4=4QhM3@`1MgA_d]OBe05kGNKXG^QL0G`00[Zj^[P1P
002IW<c<065D0:OlD_h0HV9Ta/IRHP1S00000Y:B06@00003];@0I@0000?<c01V
00001>k^06O<c0000000J>k^0000001Y00?<c00006X01>k^0000JgMhR8P0001/
UYImO00006eJFP00@3l0KV1P001PGP1_PX91@000072NWTm>0000LGMgEEDk>`1b
UYIHF3hn07<000P8=3@0M000005DE01e00000W9b07KLg?3`g=`0MoKfj>SDe01h
M7@0000007V@T0000000N]CDe=CRhP1kk^kBd^3P07bTY000Y:00OKBd002d/01n
VIVIVIVI07nVYZJVYZH0P8:20022O`21V9P007mm08:d]5UF0000PlK6Hf400024
BmiTdVCB08ELG5aLG5`0QUYJFUT00027J6QXIP0008P@40N?0000RA=F1/@?M`2:
000000000000004W0AD000000BL15@0000X000000BL15@09PHZ1R^f:0@0001<0
RX53jT<0N_e307Z=@`=oRXX00AUoRT=g07Mk07L0Mg/0M`1gN`1g07Mk07L0Mg/0
M`1gN`1g07Mk07L0Mg/0M`1gN`1g07Mk07L0Mg/0M`1gN`1g07Mk07L0Mg/0M`1g
N`1g07Mk07L0Mg/0M`1gN`1g07Mk07L0Mg/0M`1gN`1g07Mk07L0Mg/0M`1gN`1g
07Mk07Mo07Mk07L0Mg/0M`1gN`1g07Mk07L0Mg/0NP1gN`1k07Mk07L0Mg/0M`1g
N`1g07Mk07L0Mg/0M`1gN`1g07Mk07L0Mg/0M`1gN`1g07Mk07L0Mg/0M`1gN`1g
07Mk07L0Mg/0M`1gN`1g07Mk07L0Mg/0M`1gN`1g07Mk07L0Mg/E07L0Mg/0M`1g
N`1g07Mk07L0OhZ:004IOhY307X0GGMMNgX0MP1j05egGG]j07H0NP1MMeekNP1f
07X0GGMMNgX0MP1j05egGG]j07H0NP1MMeekNP1f07X0GGMMNgX0MP1j05egGG]j
07H0NP1MMeekNP1f07X0GGMMNgX0MP1j05egGG]j07H0NP1MMeekNP1f07X0GGMM
Og]j07H0NP1MMeekNP1f07X0GGMMNgX0MP00NP1gG@1j07H0NP1MMeekNP1f07X0
GGMMNgX0MP1j05egGG]j07H0NP1MMeekNP1f07X0GGMMNgX0MP1j05egGG]j07H0
NP1MMeekNP1f07X0GGMMNgX0MP1j05egGG]j07H0NP1M5GMMNgX0MP1j05egGG]j
07H0NWn:RP00fP6:@mLZOgX0NP1M07IkG@1j07X0G@1fNed0NP1j05d0MW]M07X0
NP1M07IkG@1j07X0G@1fNed0NP1j05d0MW]M07X0NP1M07IkG@1j07X0G@1fNed0
NP1j05d0MW]M07X0NP1M07IkG@1j07X0G@1fNgX007IMNed0MW]M07X0NP1M07Ik
>ed0NP1j05d0MW]M07X0NP1M07IkG@1j07X0G@1fNed0NP1j05d0MW]M07X0NP1M
07IkG@1j07X0G@1fNnXZ301<FdaEFdaKFdaKC03k:P=oRXX00A5eRT=f07L0NP1M
001j07]f07X0G@00NP1kMP1j001M07]f05ek07X007H0GG/0MP1kMP1kRWH0NgH0
MP1kRW]f07]f07Yg07X0RXYkMP2:RP1j07]f08Z:NWL0NP2:RW]f07Yg07X0NgIk
MP00NgH0NgH0NgH0Noj:O`1kMW]f001kMP1kMP1kMP1f07^:NgH0NgH0MP2:07/0
MWH0N`1f07^:RWYg08Z:NgH0NgH0NgH0NWL0NP1kMW]f001kMP1kMP1kMP1jM`1j
07^:RWH008Z:NP1kMW]f001kMP1kMP1kMP1kMUek07X0NP1j05d007X0NgH0NP00
1ed0NgH0GG]Kmd<:NP00NP1j07n:RP00g`>:@bZ4g2YoQ2XZ07]M07]M08Z:MP1k
07IkRXX0Ned0N`00NWH0MXZ:07/0RXYg07X0MgZ:RP1j07Mj07/0N`00NWH0NP1k
07YfN`1j05ekG@1kGHZ:07]M07/0NWIk07X0GG]M07Mj07H0RXYkG@1k07J:RW/0
NW/0MP1jMgX0RXX0NP2:RP0nNed0NgH0N`00NWH0NP1k07YfN`1j05ekG@1k001j
MP1j08Z:NeegN`00MP1k07YfN`1j05ekG@1k05eg05egjRX205]3oGl6FgmoFd<0
QO`Z0gn:RP010C::NW/0N`00NgH0NgH0G@1k07H0NgH0MP00NP1f07]f07X0N`1f
001j07L0MP1j07MM07_nRP<0NgH0oXX2NP1koHX2001koHX107[lRQH0NP2:RWX0
Med0RXYj08Z:002:RWMM0?f:1P1MN`1kMP3mRP80MP3mRP`0MP2:RP2:RWX0MP1k
oXX3NgH0N_j:0P1j0?f:0W/00?f:0@1fo8X3MP1j0?f:0P1g0?R:0g/0M`3mRP80
M`3mRPIf08Z:07X0oXX3N`1f0?f:8GH0NWL0MP1j05d0N`1f07]f05d0NgH007Yg
07X0MP1kFd?mO`mKOgmE@gH0N`1f001oRXX00=T1RT?G:PIg07Ij07H0oHX107[m
RT@0MXX0N`2:RW/0NhZ:07H0RXX0NhZ:07N:RP1f07X0RXX0MXZ:Nef:RP1kRXX0
NhZ:N`1f07]M08Z:N`1kRXX0NhZ:N`3nRPIk001fNed0oHX107KmRU9k08X0N`2:
RWX0NhZ:07IgRXX0NhZ:N`2:RP1kRXX0MXZ:002:RP1kRXYMNhX0N`2:RP1fNhZ:
07]jRXYk08Z:002:N`2:N`00RXYk07Yk07H0Q>/Z0P1<@oio1`=K0gmKNP24oRX5
Q2YoRXX00@/dRT<0G@1kMP00NP1kMP1kG@1kMP00NP1jM`1j07]M07/0MP1j07]f
07X0N`1kGG/0MP1kRW_lRPD0RXYMN`3lRTP0NP2:RWX0NhZ:MP2:RWX0RXX0NgH0
NhZ:07^:RP1kRXYf08Z:MP2:RP1f07MM07^:RP1f08Z:MP2:RP1kRXYk07H0Ned0
NhX0o8X5NhZ:G@1ko8XT07H0RXYjN`2:RWH0RXX0NhZ:MP2:RWYkRXYMNhZ:MP2:
RWMM0?b:4gYk08Z:MP00RXX0NhZ:MP1j07H0o8Xd07H005ekN`1gMP1g07X0NgL0
NP1k05ek07H007X0Ne]3Ogl3Oe]oOe]3Ng/0N`1g07n:RP00f06:@mLZ2@1j07Ik
G@2:07_nRRh0MXZ:05f:RP1kRXYk07J:RP1f08Z:N`2:RP1fRXYk07X0GHZ:NP2:
RW/0RXYkGO^:87/0NP1fN`2:RW/0MXZ:N`2:RW/0RXX0NP1j07Mj08Yk0?j:6`00
RXYkGHZ:07^:RW/0MXZ:001jRXX0MXZ:NP3kRSD008Z:N`2:RW/0RXX0NhZ:07^:
RP1MNhZ:07X0RXYj08Z:Ned0MW^:RP1kRXYj07Ik05d0Q5W`:P24oRX204a3oE/5
EGmoC4<0nbX3OhZ:004IOhY3MP1kG@1kMP1M07/0G@1kG@1kMP00N`1f001MM`1M
07]f001j07L0G@1kMP1MN`1f08YkG@2:N`1kRXX0NhZ:MP2:RUd0NhZ:MP1kRXX0
NXZ:07^:RP1f001kRXX0NhZ:07^:RP1kRXX0N`1k07X0GG/0NXZ:05ekRXX0NXZ:
O`1jRXYf07Yg07X0NhX0MP2:NeekRXX008Z:MP2:RP1MNhZ:NgH0RXYkNhZ:07^:
RP1k07X0NhZ:07^:RP1jRXYf08Z:MP2:RP1k08Z:NP1gRXX0NhZ:07H0N`2:RWH0
RXX007]M07]f07/0NP1MN`1M07X0MP1f05d0N`1g001k5E]30gmo0gmo0e]3MP1M
07/0Ngn:RP00iPB:@bXZ?_HZ03kh:P24l2Yo07Ik07X0NXX0MgX0GGL0RXYk08Z:
N`2:RP1f08Z:N`1jRXX008Z:NP2:RWX0NWH0RXYk08Z:NP2:N`1MRXYkG@2:07Ik
05eg08Z:N`1MRXYk08Z:N`2:RW/007X007H0RW/0MP00MP2:RWX0RXYk08Z:N`1M
RXX0NP2:RP00RXXkN`2:RW/0MXX0GHZ:N`2:RW/0RXX0NXZ:N`2:RWX0MXZ:07H0
RXYk08Z:N`2:07Z:RW/0RXYgNP1gNP1klRX0Q?/Z0h@0Fd?jO`9K@g_k:P<3RXX0
0@ToRT=g07/0N`1j07]gG@1kMP1k07/0NWMM07/0N`1kMP1j07]MMed0NgH0NP1j
N`1MMgH0RP1j001k07Z:RUd0Nob:3g]j07^:RWH0RXYjMhZ:07_lRPD0NhZ:07_m
RPAk07Yg0?f:3G]f07]f05ekRXX0NWL0oHXINP1kRXX0NWH0MWH0NXX0GGMjMP1k
RXX0NP3lRQ00NWL0RXX0NhZ:07Z:RP1f0?f:3`1MNhZ:07Z:RUd0RXYjN`3lRP`0
GG/0RXYkNXZ:07H0oXX3Neek0?b:=7X007X0MP1M07]M07/0MP1f07/0NWL007]M
07X0N`1K@e]<Fe]EC5]K@`1f07X0NP1oRXX00>41RT?W:P0nlRYo05eg05ekN`1f
07Ij07X0N`1fNP00N`1k05d0MP1k07X0N`00NP1kG@1k001j07H0N`1j07/005d0
MP1j07/007X0Ned0MW/0GG/0MP1jN`00NP00MP1k07X0NeekN`00NP1k07/0MP1j
07L0NP1k07/0MP1j07/0MP1k07H0N`0AMP1j07/0Neeg07/0MP1k07MjoP0;GGL0
N`1k07Ik07IkoP0GNP1fNg/007X0N`1fN`1k001j07H0MW]Mo2X3?RXZQ?<Z0@1<
nT<3NWY3No/Z0gn:RP016Gn:@g]f07X0MP00G@1kMP00NP1j07X007]f05d0N`1j
07L007X007X007X0NgH0GG]j001MNeek07H0MP1MN`1kMP1f07]f07]f07X0MP1j
MP1kGGL0GGMMN`1kMP1kMP1MMeekMP1j07H0NWH0MP1f07]M07]M07]f001jMP1f
07mkMP1f07]MM`00MWH0MP1j05ekNP00NeekMP1f07X0MP1f05ekMP1MNgX0MP1k
NgH0Neeg07H0NP1f07]MN`1jMgH0NP1fG@1kG@1j001jMP1kNP00NWH0MP1MN`1j
07Yf07IkGG/0NgMf07H0N`1k07/0G@1k07L0NP1g07H0NQEk07/0N`1kN`00Ned0
M`1f001oRXX00AUoRT<0Ned0MW]j07Mj07]MM`1f07H0MWX0NeegNP1k07X0MW/0
MW/0MW/0MP1kNP00Neeg05eg07IkN`1fN`1j07X0NP1MNed0Ned0MW/007Ik07X0
MW/0MP1f07X0GG]M07X007Ik07Ik001jN`1fNed0MgX0MgX0Neek07/0MWYo07X0
NP1MN`1fNP1kN`1f07Ik001fMP1g07X0NP1gNP1jN`1f07X0M`1MMeej001j05ek
07X0MW]k07X005ef07X0MW]MN`1f07Mj07Ij07/0G@1fN`1kGG/0MP1j07H0N`1j
07MMN`1MN`1kG@1j07L0MW]M07X0MP1fGG/0MP0EN`1MM`1f07YkG@1kGG]MNeeg
OhZ:002>08[iO`5n0o=o0Gh3kGl0O_io07kmO`03ngl4OP=nOgknO`1no7l0O_=o
07kkO`1nngl0O_Qo0Gh3nWl0O_eo0Wh3O_=o07kkO`1nngl10gknO`1nn7l0O_eo
0Wio0oeo07kiO`43O_Qo07kiO`1nnWl0O_eo00?kO`1nkWl0O_]o00?nO`5n0oao
0XZ:0008PHZ1R^b:00015bf:07X007/0N`00N`1j001k07/0N`1k07X007/007X0
07/0NP00N`00NP00N`1foP0:NP00N`1k07/007[n00=j001koP1YN`00N`1j001k
07/007X007/0N`1k07/007/007/0NP00N`1k001j001k07X007/007X005d007/0
07/007/007X007/0NP00N`1j001k07/0N`1k001j001k07/0N`1k001k07/0NP00
Noh08WX007/0N`00NP00N`1k07/007X007/0N`00N`00NP00N`1foP0lNP00N`00
N`00NP00N`1j001k001j001k07/0NP00N`1k07/0N`1k07X007/0NP00N`1M001k
001k08Z:004IOhYk07Ij07H0MWX0MP1fNP1j07H0MP1f07Ij07Ik07Ij07H0MWX0
MW/0MWX0MW]MMgX0MWX0NP1f07Ik07IfN`1fNP1fMW/0MWX0MP1fNP1f07Ik07Ij
07X0MP1f07Ij07Ij07H0MWX0MP1fN`1fNP1f07Ij07Ik07IgNeef07IjO`1fNP1f
N`1fNP1f07Ik07H0MWX0NP1f07X0MW/0MWX0MP1j07H0MWX0NP1f07Ij07IfN`1f
NP1f07Ik07Ij07H0MP1jMP1fNed0GGIk07Ik07Ij07H0NWMj07IkGGIk07Ik07Ij
07H0MWX0MW/0MWX0MP1f07Ij07H0MP1f07H05GH0MWX0MP1fN`1fNeegGGX0GGN:
RP016Gn:05ek07H0NWH007X0N`1kMP1j07X0NP1k05ek05ek07H0NWL0GG/0NWL0
GG/007X007/0MP1f07]j07H0N`1MN`1f07]M07X0MP1kNP00NP1j05ek07H0MP1j
07]f05ek07H0NgX007]MNed0N`1f07]j001j07H0Ned0Neek07mMN`1jM`1MN`1f
07]j07H0NW/0MP1j07]f07X0GG/0MP1f07H0NgX0MP1f07]k07H0N`1kMP00NP1j
07]f001kGG]f07/0NP1kMP1kG@1j07H0N`1gNP1j001kG@1kG@1j05ek07H0NgX0
07X0NgH007]f07]j001kGG]MNeekGAEkMP1k07X0N`1jMP00NP1f07X0RXX00AUo
RW/0GGL0NP1kNP1fN`1j07]M07Ik07]M07Ik07Ik07Ik07/0MW/0NP1j07MMNP1f
MgX0MW/0NP1M07YkG@1fN`1fNgX0MgX0M`1j001fMWMj07Ik07X0MW]j07H0NgX0
07Ik05d0MWH0MeekG@1fN`1M02ak07IkG@1gNP1k07IoN`1f07X0Meek07H0Neek
001j07]M07H0Ned0MW/0MW]j07H0NP00MW]k07X0GGL0;01j07]MMP1fNed0NgX0
GG/0NP1MM`1f07Yg07Ij07IkG@1f07/007If07Mj07Mj07Ik07Ik05d0MWL0MP1j
Ned0NP1M07If07/0NP1k07/E07X0GGL0MP1f07]j001fMgX0NXZ:004IOhYM07]M
Ng]f001MN`1f07H0MP1k07Yf07Yg05ekG@1j07MM07X0MP1f07H0N`1f07X0GG/0
GG]f07Yg001jM`1MNed007X0Neef07]fN`1j001kG@1kMP1k07H0NUd0NgIk05ek
Mg/0Neeg001jN`1f07]gG@1k07H0NP1f05ekO`1f07X0Meek07X0MP1k07If07H0
NgH0NP1kMgH007X007X0NW]f07]M07H0NWL0NP1j07X0NgL0Ned0Med0NWL0MP1f
07]MNgH007Yg001k07]MNgX0MUekN`1j07H0G@1k05ek07H0Ng/0NW]f001kMP1j
M`1kNgH0NP1f07YM5@1kMP1kMP1jNgH007IkNed0M`2:RP016Gn:Ned0N`1MNeeg
N`1fNeek07IfN`00Ned0NP1g07Ik07X0NP1kGG/0NgX0MP1jMg/0MW/0MW/0Ned0
NP1f07X0M`1kGGL0NP1k07H0N`1f07Ik07Mj07]M07IkN`1k05ek07Ik05d0NP1j
07Ig05ekNed0N`1f07Yk001fNeeg07mj07Ik07X0GGL0NW]M07Ik07IkG@1gNP1j
07X0MWX0MW/0MP00NWL0NP1j001jN`1j07H0MP1jN`1kGG/0MP1j07Yk07H0N`1j
Ned0NP1f07H0N`00MW/0G@1fN`1jN`1f07Ik07X0MP1MMP00NW]M07/007Ij001j
07Ik07H0MaEj07]M07Yk001jNed0G@1kGG/0RXX00AUoRP1kMP1jMP00Ned0NP1k
07H0N`1jMP1gMP1MNeek07H0MP1fGGL0MUd007]f07X0GG/0GG/0NP1gMP1MNgX0
Meef07H0MP1j07YkMP1MN`1j07]M07X0MP1j001f07YgG@1j07H0NP1f07X0Neek
G@1kMP1MNeek07IMNeeg07Yo07Yk07H0MP1j07X0M`1j07]j07L0NP1f07H0NP1k
07/0MP1kMP00NP1f07]fG@1kMP1j07]f001j07MM07]f07H007YkMP1f001f07]f
07YkMP1MNgX0MP1kNP1f001f07]j07H0GG]kMP1kNP00NgH0MP1j07]f05ek05ek
07XE07X0NgH007MMN`1f07]f07/0NXZ:004IOhYk07]M07]MMed0MP1f07Yk05ek
07X0NP1fN`1j07H0NW/0N`1j07MkGGH0N`00MW/0MW/0NP1f07]k001fNeek07Yk
GG/007H0GG]MN`1f07H0NP00NW/0Neej07H0N`1fN`1jN`1fN`1MMed007Ik07X0
Meek07X0M`1j07X0OgH007IkN`1f07H0NP1f07X007IMN`1fN`1j07Mj07IkGG]k
07YkG@1fN`1MN`1f07/002`0NW]M07H0MWX0NgX0MW/007Yk07Ij07H0NW/007X0
MP1kGG]M001fNgX0MWX007Ik07H0GG/0GGIkG@1fNg]M07H0Ng/0GGMk07H05GH0
MP1jNed0MP1fNed0MgX0NP2:RP016Gn:07H0NgH007X0NgH0Neek05ek07H0MP1k
N`1f07H0NW/0MP1f05ekG@1kGG]f07X0GG/0NP1f07]j07H0NP1k07H007/0NgIj
07X0M`1f07]kMP1kMW/0MP1g07X0Med0NP1f07H0NP1f07X0Meek07H0NUek07H0
MP1j07X0MWl0MeekG@1j07]kMP1MNgH0NeekN`1MN`1kMP1j001j07/0GGH007MM
N`1jMed0NgH0MWH0NP00NgH0NW/0Med007X0NgH0G@1j07]kMP00NeekMP1kGGL0
NgIf07/0GG/0Meek07X0NWL0MP1k07H0Ned0MP1kMP1MMgH0G@1j01Ej07]f05d0
NgX0NP1g07X0MP1kRXX00AUoRWX0MP1kMW/0MP1jN`1f07Ik07Ik07X0G@1fN`1j
N`00MW]j07Ik07X007H0NWL0MW/0NP1fNed007IkG@1f07YkGG/0MP1k07H0MUek
07H0GG/0G@1MMgX0NP1gNP1jMP1fN`1kG@1jN`00;01j07Ik001gGG/0MW]k07X0
MW]oNeek07]M07H0NP1fN`1kG@1f05egNed0GGMj07If07X0MW/0MWX0Meek07X0
MP1jN`1kG@1fNP1kG@1MN`1fNgX0MP1jN`1f07H0NP1fNP1k07X0G@1j001fNgX0
MW/0NP1f07Ik001jN`1f07IkG@1gNP1j07X0NeekGGMj07HE07X0N`1fMed0MWMM
Ned0MW/0GHZ:004IOhX007]f07X0GG]f05d0NeekG@1j05ekMP1kNP1f001kMP1j
001k07X0MWH0NP1j07X0MP00NP1g07Yf07H0Neek07H0GG]fG@1kNP1k07X0NWL0
GG]f07X0MP00NP1f001kN`1f07H0N`1fMP1j05ek07YgGG/0GG]M07H007/0Oed0
07X0NeekMP00N`1j07H0NgH0NP1kMP1k07X0MgX007]M07X0Neek07H007]f001j
001jN`1f001jM`1MNgH007]f05d0NgX0NP1jN`1f07H0MP1kMP1jN`00GG/0MP1M
N`1j07]fG@1f07Yk07H0NP1kMgH0MUeg001j001k5@1kMP1MN`1j07/0NP1gGG/0
GG^:RP016Gn:Ned0NP00MW/0NW/0MP1g07Ik07Ik07]M001jNeek07X007Ij07H0
M`1jN`1f07IgGG]MM`1j07Ik07IkG@1gGG/0MW/0N`1f07]M07IkG@1j07H0NgX0
MW]j001jNeegG@1jNeekMP1fN`1j07Mk07X0NP1MMg/0NeekGGH0MWmkNP00MP1g
GG]MMP1f07IkG@1kG@1j07X0GG/0NP1kGGH0Ned0MP1k07X0MWX0NgX0MWMj001f
NgX007X0MP1kGGL0NW/0MP1k001f07]MN`1jN`1f07]f05egNP1MMg]MN`1f07H0
N`1fNgX007IkNP1j001jN`1kGG]M07IfNaEM07]MM`1j07H0NP1j07/0MW/0RXX0
0AUoRP1kMP1jMgX0MP00GG]MNeekG@1j07MM07]fNP1g07H0NP1jM`1kMP1jN`1M
N`1k07X007X0NP1j07]j07L0NP1M07X0NP1MNgH0MP1j07]f05ekNP00MP1k07If
001j07]f001k07]MN`1MN`1j07X0MP00Ned0Ned0GGL0NgYo001fMP1kGG/007X0
Neek07]f07]f07X0NWL0MP1kG@1kMP1kMP1j07H0NgX0Med0MP1j07MMNed0MgH0
GG]f001jMP00GG]f07If07]MM`1f001MNgH0NgH0NP1f07X0NgH0NP1kNP1MNed0
07]j001k07Yf001j07/0MP1kN`0EMP1kMP1fG@1kMP1j07MM07X0NhZ:004IOhYk
07/007X0MW/0MW/0N`1j07Mj07X0MW/0N`1fGGX0MW]M07X0GG/005egN`1f07X0
MWL0MP1fN`1M07H0NP1fN`00MP1fN`1j07]j07X0N`1f07Mj07]M07IkNeeg07X0
NP1f07H0N`1fN`1f07H0MW]MMP1j05egNP1j05d0OgIkN`1f07X0MW/0MP1k07X0
Ned0N`1f001j07Ik07Mj07]M07Mj07IkG@00NP1gNP1f07X0M`1j07Yk07H0NgX0
07IkGGL0NW/0NW/007Yk07IfN`1j05ek001jMg/0NP1kG@1f07]MM`1j07H0Neeg
G@1k07Ik07H0MW]M05ek5@1f07]j07Mj07/007IMNed0MP2:RP016Gn:07X0MWH0
07X0MgX0MP1f001j07H0GG/0MP1f07L0GG/0NgL0NWL0NWH0NP1MNgH0NgX0MP1j
N`1f07X0NWH0N`1jMW]MN`1j07IM07L0MP1jM`1j07H0NgH0N`1f07H0MP1MNeek
MP1MN`1MNgH0NgX007]f07H0NP1g05ekMgmj001jNgH0MP1j07]f07H0NP1kMP1f
07]f05ek07YM07X0MP1j001j07]MNgH0NP1kNgH0NP1f07H007X0Med0GG]kGGL0
GG/0MP00NgIM05ekN`1f05ek07Mj05d0NgH0G@1kNgH0MP1f05ekMP1j07]f05ek
G@1j07X0MP1kMQD0NP1j001j07X0MWH0N`1kMP1kRXX00AUoRW/0MW]kGGL0NP00
MW]MNeeg07Ik07H0MW]MNgX0MW/0NP1j001j07MkG@1fN`1j001kNP1f07X0Ng/0
MWL0NP1f07/0Med0MgX0MgX0NP1f07X007IkG@1j05ek07Yk07Yk001f07]MM`1f
N`1j05d0MWL0NW/0MP1f07X007Yo07Ik05ek07Yk07H0NW/0MP1f07Yk07Ik07Ik
07H0MgX0MW/0NP1f07H0MeekG@1j001kGG]j07YkGGL0NW/0MW/0GG/0MW/0NW]M
NP1k07Ik05ek07Ik07IMN`1fN`1kGGIk05ekGGMMNg/0GG/007H0NP1gGG]M07Ig
07Yk07XE07Mj07Ik07H0MW/0NP1f07]M08Z:004IOhYM07X0G@1kGG]f07X0Ned0
Neek07H0NgX0NP00NW/0MP1f07YgG@1j07MMN`1j07]fG@00NgX0Med0GG/0NgH0
NgH0NP1f07X007X0GG/0NgH0GG]j07]f07]f05ek07H007If07]M07]MN`1j07H0
NgX0NP00Ng]f07YkMWH0O`1j05ek07H0G@1kNP1M07X0NgH0G@1j05ekG@1j07X0
07/0MP1jNgH0Neek07]f05ekGGL007Yg001kMP1M07]M07]M07X0Ned0M`1f05ek
07YgG@1j07]MM`1MN`1jG@1k05ek07/0NP00NWL0MUekMP1gGG/0Meek07X007X0
5GIj001j07H0NgX0MP1kGGMM07^:RP016Gn:Ned0MW/0G@00MWX0MP1gNP1k07X0
MP00MP1fN`1MN`1fN`1MN`1f07X0Med007H0NP1fMP1kGG/0MW/0NP1kG@1kG@1j
Ned0MW/0MW/0MP1kNP1f07X0Ned0Neeg07Yk07IkNed0MW/0NP1f07Ik07H0NP1f
NP00NgX007]kGGmg07Ik07Ik07Ik001jMW/0MP1kNP1fMgX007Mj07H0MWX0MW/0
07X0MP1k07X0N`1f07]MN`00NUegGG/0MWH0MgX0MgX0NP1gNP1fNeeg07X0NP1f
07H0Neek05eg07]M07Ik07H0NP1fN`1j07X0N`1fGG/0NP1gGG/0MW/0MQEg07Ik
07Ik05d0NW]M07]MNed0RXX00AUoRP1f07/0MP1k;01g07/0NP1fG@1j07]MNgX0
N`1jM`1MN`1jM`1MNgH007X0NWIkGG/0Ng]f001M07X0NP1j07]gG@1jM`00NWL0
GG/0GG]MN`1f07H0Med0Neeg07IM07]j001kGG/0MP1j07]k05ekMP1MN`1kMUd0
NgH0G@1oNP1kG@1j05ek07Yf07MM07]MN`1f07X007]j07H0NWL0GG/0NWH0Ng]f
07H0MP1MNgH0N`1fMP00NP00NWMj07X007X007H0NP00NP1k07H0NP1f07]kMP1k
05ekGGMM07Ij07H0NWL0NP1f07H0MP1j07/0MP1f07X0NgH0GG/E07X0NP1j07H0
NP00NgH0Ned0NhZ:004IOhYk07IkGG]M07]j07H0MP1f07Mj07H0Med0MP1j07X0
MP1j07X0MP1jMgX0M`1k07H0MP1j07Ik07H0MP1f07X0NP1f07X0MW/0MW/0MW/0
N`1fNg/0NP1fN`1j07X0MgX007Ij07MMN`1f07H0NP1g07]j001f07/0MP1j07Mj
O`1f07Mj07Ik07H0Neek07H0NP1fMg/0MWH007YgN`1j07MMN`1kG@1MNeek07X0
MeekG@1fNg]MM`1fN`00Ned0MWH0NWL0MP1fNP1j07H0MWL0NW/0GG]M07Ik07]M
07Ig07X0MP1j07]MN`1jN`1fNed0MP1jN`00MP1j07MM5G/0MP1fMeekN`1fNP1k
G@1gNP2:RP016Gn:G@1j07/0NgH005ekGG]j07]j07H0Neek07]j07H0GG]f07H0
07]f07X0NP1fG@1j07]j07H0NgH0NgX0NgH0MP00NgH0NWL0GG]M07X0MP1j07X0
MP1MN`1f05ek07X0NWH0N`1MN`1j07]kMP1f07YM07MMNgH0GG]f07Mj07mf07]j
001k05ekMP1kG@1kMP1MN`1M07X0NgIj001j07]MN`1kMP1MN`1k07H0Meek07H0
NP1M07]MNed0NP1gGG]k07]MMP1kNP1g07H0NW/0NP00GG/0MP1j07]M07]gNP1k
07]j07H0M`1j05d0NP1kMP1jN`1fGG]j07MMNaD0NP1kNP1k05d0NP1gG@1gG@1k
RXX00AUoRW]M07H0MP1jN`1f07/007X007IkNP00GGH0N`1k07H0NgX0MWH0N`00
MP1jN`1f07H0N`1jN`1j07H0MP1kNP1fMP1k001j07H0MgX0NP1f07H0NW/0MP1f
N`1f07H0MP1kNP1fN`1j07H0GG/0NW/0MgX0N`1j07L0NW/007IoNed007IfN`1f
07]M07Mj07/0MW]MN`1kG@1k07Ik07X005ej07]MMP1f07IkNP1k07X0MW/0MW/0
NP1gNP1j07MM07H0N`1j001fGG]k001j07Mj07H0MW]M07H0MW/0G@1f07H0N`1j
07X0MgX0MP1f07]j001fN`1M07H0NP0ENP1f001j07IkMP1j07Mj07X0GHZ:004I
OhX0M`1kGG]f001MNgH0NWH0GG]j001jMP1kMP1f05ekGG/0MP1kNP1kNWH0N`1f
07]kMP1MN`1f07H0NgX0NP1f07]j07If05ek07X0M`1f07]j07/0MP1kNP1MNgH0
Ng/0NP00NP1f07H0NP1kMP1M07X0MP1f07X0NP00MP1jO`1kMeekNP1f07X0MP1j
07H0GG/0N`1f07]gG@1j07H007Yg07H007Mf07]j001j07X0NgH007X0MP00NP1g
G@1j05ekMP1jMP1j07]M07Yf001j05ek07X0M`1kGG/0NP1j07YkMP1f07X0MP1f
07]j07]M07]MN`1jM`1kNP1f5@1f07]f05ek07]f001j07]f07^:RP016Gn:NP1f
07X0NeefN`1kG@1gNP00MW/007Yg07Yk07Ik07/0NW/0G@1f001kNP1j07H0GG]M
M`1j07Yk05d0MP1fNg/007Yk07Yf07H0NP1fNed007H0MW]M001fN`1j05d0MP1f
MWL0NW/0MP1f07Yk07H0MP1jN`1fNP1fNP1j07mj07X007]MN`1kGG/007IkGGL0
NP1fNed0NP1fMeekGGH0NP1jNed0Ned007Ik07H0MP1kNP1fN`1fNP1j07Mj07Ik
07]M07]f07Ik07H0NWMj07Ik07H0NP1f07H0MP1fM`1k07X0MW/0MW]j07H0Ned0
MgX0Med007YM07/0NQEk07Ik07Ik07X0Neef07X0Ned0RXX00AUoRP1MNgH0Med0
N`1kGGL0NP00NWMM07YgGGH007]j07H0MP00GG]f07YgG@1f07]j07X0M`1j07]f
05ekMP1kN`1M07X007X007]kMP1kNP1g07YkGG]MM`1jN`1j07YgGG]MNg/0GG/0
MP1kNgH0G@1kMP1jN`1MN`1f07/0NgIo07H0NWH0N`1fGGL0MUekGGL0NUd0NP1k
MP1MN`1j001kMP1jM`00NP1kGG]j07H0NP1gG@1MN`1jN`1f05ekG@1k07X0NgH0
NP1kG@1kMP1j001kG@1j07]f07]kMP1jN`1j07X0NW/0MP1k07H0Neeg07X007X0
NWH0NgH0MP0EG@1j05ek07X0Med0NgH0MUd0NhZ:004IOhYk07X0MP1fN`1j001j
M`1fMP1k07H0N`1jNed007IkGG]MMg/0Ned0NP1fN`1M07IkG@1f07H0Neeg07]j
05egNeefMgX0MW/0GG/0G@1f07X007H0Neeg05eg07X007L0NP1MN`1MNP1f07X0
Mg]M07]k05eg07X0Ned0MP1kOeek001fNP1fN`1j07/0GG/0NP1gNP1j07]MM`1j
07Ik07/0G@1fNP1f07MM07IkNP1fGG/0MP1f05eg07Ik07]M07H0NP1k001fN`1f
07/0GGIg07Ik07H0NW/0GG/005ek07H0MP1j07Yk07X0MP1kGGH0MW/0MP1kG@1g
NW/05GMj07Ik07H0MP1fN`1k07/0MP2:RP016Gn:07H0NP1kNP1f07]fG@1j07]f
05ekNP1f05d0NWMj07H007X0NP1gGG/0N`1jMeek07]MNeekMP1kGGMM001j05eg
07X0NgH0GG/0NWH0NgX0MgH0NP1j05ek07X0Nb`0MP1jM`1MN`1kNgH007X0NgH0
G@1kGGH0MP1kMP1kGGmg07]fN`1MN`1f07H0NWL0MP1f001j07]MM`1f001j07X0
NWL0NWL0NgH0NgH0N`1f07X0MgH0Neek07Yk07X0MeekMP1f07YgNP1f07X0NWL0
Neek05ekMP00GG]M07YgG@1j07]f07H007X0Ng]f07/0NgX0MP1kGGL0NP00MQEj
001k05ekMP1jN`1MNgH0MP1jRXX00AUoRW/0MWMM07]MNed0N`1fNed0Neek001f
Ng/0MP1j07X0MWL0NP1j07MMMP1j001gGG/0N`1k07X007X0MWMj07X0NP00MP1k
G@1MM`1kG@1j07Yk001f07X0MW/0MP00NW/007X0MP1f07X0MWX0MP1j07If07]k
GG]M07Yk07Yo07X0N`1fN`1j07Ik05ekGG/0MWX0MW/0NeejN`1f07H0MP1jN`1M
MP1j05ekG@1fNg/0NP1kNP1gGG/0GG/0MP1j07/0NP00NP1fNg/0MP1j05ek07Ik
07X0MW/0Ned0NP1f07X0Ng/0MWX0MP1MNed0MP1kGG/0GG/0MW]MN`0E07If07Ik
07X007X0MeekGG/0MXZ:004IOhX0GG/0NgH0NP1kMP1kNP00Neeg07]f07X0MP1j
07L0NgX0NP1f07]MN`1f07]f07X0MP1f07H007YgMP1j07L0NP1gGG]f001jM`1j
07]f07MM001jNgH0GG/0MP1kGG/0NWH0NW]f07X0MP1k07]f07Yk07]M07/0NgH0
07H0OgH0MP1MN`1j07Yk05ek07/0NP1k07/0NP1k001jNgH0NW]f001MN`1j07]j
07L0NP00NP1fG@00NP00NWL0GG]f07X0NWMfMP00Ned0MP1j07YgG@1j07X0MgX0
MP1gMP00NgH0Ned0NWH007]MN`1kGG]f07/0NWMM07X005eg5EekNeek07X0MgH0
GG]M07/0NWN:RP016Gn:N`1MN`1j001f07X0G@1fMP1kG@1j07H0MW/0Ned0MP1k
G@1fN`1M05ek07H0NW/0MP1jNeekGGH007Yk07X0MP1f07/0NW]M07X007H0NW]M
Neeg05ekMWL0NP1f07H0NP1k001MNg/0MP1j07H0N`00MW/0NP1MM`1jNeek07mj
07Ik05eg07H007Ik07H0MW]M07H0MP1f07Ik001jN`00NWMj001fN`1M07H0NP1f
N`1fN`1fMP1j001j07H0N`00MP00NW]MMP1kGG/007H0NP1f07H0MP1kGG]MN`1f
NP1kG@1gNP1kGGH0Med0GGL0NP1MM`1j07IkGGL0NQD0G@1gGG/0NP1j07MMN`1M
N`00RXX00AUoRUd0NP1j07MMNgH0GG]fN`1j07]f07H0Ng/0MUd0NW]f001jN`1f
07Yg07YkMP1M07]j05d0GGL0Neek07H007X0NgX0MP1f07]f07YkMP00N`1f05ek
07/0NP1f07YkMP1f05ejM`1M07]j05ekGG]f07X0MP1MNeef05eg05eoN`1kGG]f
05ek07YkG@1j07X0MeekGG]MNeek07Yf001gGG/0G@1kNP1f07]gNWH0N`1MN`1j
07]f07Yf07H0NP1fMP1kGG/0M`1j07MM07YkMP1f07]j07]j07/0NP1kMP1kGGL0
NP1gG@1k07X0NWL0NP1kMP1f05ek07H0MP0EMP1kGG/0MP1j07Mj001kMUd0NXZ:
004IOhYkG@1f07H0M`1j07L0N`1f07X0NW/0NP1MN`1gNP00NWIk05egGGL0NP00
NP1fNed0MgX0MW/0NP1k07IkGGH0NP1kGG/0Ned0NgH0GG]MMeek07Ik07H0NP1f
Ng/007Yk07Ik07X0MW/0N`1f07/0NWH0MW/0M`1j07Ik07IkO`1f07/0NP1f07H0
07IfMgX0NP1g07X0M`1j07H0NgX0NP1MMg]M001fNed0GGL0NP1fN`1j07H0Ng/0
NW/0MW]MNg]M07H0NP1fNed0MgX007]k07H007H0NP1MM`1j07]M07MMNP1j07Mj
07H0MP1j001f07Yk07Ik07H0NW]k5@1f07X0NP1f07H0NP1f07/0MP2:RP016Gn:
07L0NgH0NW]MN`1jMP1MNgX0MP00MP1jMed0NP1kMP1gG@1kGG/0NP1gMP1MN`1k
Med0Ng/0MP1fG@1j07L0NWL0GGL0MUd0Med0NgH007X007]f05ekMP00Ned0NgH0
05ek07H0MWX0MP1MNgH0MP1k07]M07Yf001kG@1j07mMNgH0MP00NgX0MgX0Ned0
N`1j07H0MP1j07H0Med0MP1kMP1M07]fNP1kGG/0NWL0NP1f001kNed0NgH0GG/0
NP00Neek07X0NP1g07X0NgIM07]j07YkMP1kMP1j07MM07]MN`1f001j07H0Ng]f
07IMNeek05ek07X0Ng]M01EMNgH0NgH0NP1kNP1jNgH0MP1kRXX00AUoRWX0MP1k
NP00NP1f07]M001gNP1fN`1f07/0MP1j07]M07Mj07/0NP1j07X0Meek07X0MP1M
N`1j07Mj07X0MP1jN`1j07X0MW/0MP1kNP1fNP1MN`1f07YkG@1kG@1jMgX007Ik
07/0MW]MM`1j07X0GGL0NWH0NgX0GGMj07IoN`1j07IjM`1jNed0MP1kG@1f07Yk
07X0NeekNP1kGG]M07Mj07H0N`1f07/0NP1j07]MNgIj05ekG@1kGGL0NP1fN`1f
07H0MWMM07X0MP1k07H007H0NP1f07]M07IMN`1f05egNeeg07IkG@00NW/0M`1j
07Ik07H0MP00MW/E05ek05egNP1f001f07/0MW]j08Z:004IOhX0MP1kG@1kMP00
NgH0MWX0NP1MN`1jNgH0GG]f07]MM`1j07X0MP1f07H0Meek07YM07]MN`1jM`1j
07]M07]j001f07Yg07]MNeegG@1MN`1jM`1jNgH007]f07]f07X007]j07MM07X0
07X0NgH0NgH0NP1gMP00MP1j001kO`1j07X0Meef001k07]j07]f07X0G@1kMP1f
001kGGL0NgH0NP1kMP1f07H0MP1j07H0Med007]f001kGGL0MP00N`1MNgH0NgX0
07Yg07X0NP1j07YkMP1f07X0NgH0N`1MNgH0NP1j07Yk07H0NP00NP1f05ek07X0
Ng]fGG/05Eek07]f001f07]j07X0NgX007^:RP016Gn:N`1j07Mj07X0MW/0NP1g
NP1fN`1j001j07MMN`1M07Yk07H0MP1jN`1jN`1j07H0MgX0Ned007X007H0MgX0
07Ij07H0NP1M05ek07Mj001f07X005ekGGL0NP1MN`00;7/007Ij07IgNP00MP1j
05ekG@1f07]j07Ig07Ij07mf07Ik05ek07Ij07H007H0NgX0MW]M07]MNeek05ek
G@1kG@1fN`1jNg/0MP1fN`1j07X0MW/0Neef07]MNP1fMP1fN`1j05egGG/0NP1f
N`1f07H0N`1fNg/0NP1j05egN`1k001f07H0NP1kNP1fN`1fN`1f07H0MP1j07/0
MQEk07X0NgX0Ned0MW/0NP00MWX0RXX00AUoRP1kMP1j07X0NgX0NWL0NP00N`1f
07]f07YMN`1kMWX007H0NP1jM`1MN`1f05ek07X0NP1kMWH0MgYf07X0Meek07]k
MP1f07]gG@1j07MMNgH0NgIj07L0NP1jM`1fMP00NWH0N`1kG@1gGG]j07Mf001j
07]M07]j07]j001oNP1k07Yk07]f001j07YkMP1f07/0NgH0MP1g05ek07]f07H0
NeegG@00NP1jN`1f07H0NgX0MUd0NgMM001jN`1jN`1f05ek07X0MP1MN`1f07]k
MP1j07X0MP1f07H0NP1j07YgNP1kNP1f07H0Ned0NP1f07YkMP1j07MM07XE07H0
Med0MP1gGG/0MP1f07X0NhZ:004IOhYj07]M07H0MP00MP1j001fMW]MN`1MN`00
Med0GGL0MWYg07]j07X0MP1fN`1gGG/0MP1f07]kG@1k07H0NP1f07H007Yk07H0
NP1f07Ik001j05eg07X0MP1f07X0NW]MM`1jN`1f07]MN`1M07H0NgYf07X0MgX0
07H0NgIkO`1f07X0GG]MN`1fN`00GG/0Ned0MP1kGG]M07Ik07X0Neek05d0NP1f
NP1f07X0NW/0MP00MW/0MP1j07Ik05eg05ek07Ik07X0MW/0MP1fNed0GG/007X0
MW/0NW/0MP1f07H0NP1f001fN`1kG@1gNP1fN`00NP1f07X0MW/05GIkGEek07X0
NP00MW]j07Ik05f:RP016Gn:07H0MP1kMP1gNP1j07MMNgX0N`1jM`1f07X0GG/0
NgX007]M07/0NWMMNed0NP1M07]f07]M07]f05ekNP1MNgH0NgX005ekNP00Neek
07Yf05ek07X0NgX0NP1kMP00Neef001MNgH0Ned0NgH0NgX007]g07X0NeekMP1k
07mMNgH007X0N`1MNed0NWH0MP1gGG]f07H007]j07H0Neeg05ekMP1kMP00NgH0
MP00NgH0NW/0GG]f05ek07H0Neek07Yk07H007/0MP1kNP00NWL0GGYg07/0GG/0
MP1kNP1kMP00NP1kNP1f07]f07X0N`1fGG/0MgX0GG/0GAEkG@1kG@1kMP1kNWH0
07Yk07H0RXX00AT>RW/0MW]M07]j001j07IkoP1oMP1j07X0MWX0MW/0NP00MWX0
Ned0MP00N`1jMP1fNed0Ned0MWX0NP1g07X0M`1jN`1MMgX007IfM`1j07X0MgX0
07H0MP1kG@1f07]j001fNgX0MP1j05egNP1kG@00MWX0GGH0MP1gG@1f07Ik07Yf
N`1f07Ik07Mj07]MNeeo07L0NW/0MWX007IkG@1kGGMk07]M07]j07/0MW]j05ek
G@1MMg/0NP1gGG/0MP1j07H0NP1fNW/0NW]M001jM`1j07L0NUd0MW/0MW]M001j
07YkG@1f07]MNed0MgX0MP1jM`1MNP1k07H0MW/0Ned0MgX0Ned007]j001MMg/3
GHZ:004IOhX0NgX0M`1j001jM`1j07H0NW]f07H0NP1g07/0MP00NW/0MP1kMP1k
NP1f07/0N`1kMP1gGG/007]f07X0Meef001MNgH007YgNP1f07H0MP1f07YkMP1j
07MMN`1j07If07X007X0MgH0NP1f07H0NWL0GG/0NgX0NP1kNP1jO`1j07MM07]M
Ned0NP1f07MM07]MMP00NgX0MP1j07]f07]M07X0NgMM07H0NgX007Yg07]f07X0
NP1gGG/0NW]f001kMP1f07/0MP00MeekMP1j07MMMP1gGG]M07X0NgH0NgH0GGL0
NgH0N`1f07X007]MN`1j07Mf05ekGG]M5@1MM`1j07MM07YgG@1gMP1j07^:RP01
6Gn:N`00NP1f07IfN`1f07YkG@00NW/0MW]j07IkGG]MMP1j07X0GG/0G@1fN`1M
MP1f07X0NP1MMgX0NP1j07Yk07Ij001jMgX007]j07Yg07]j07H0N`1fNed0Meek
07Ik07X0MWH0NP1k001fNgX0MW/0MW/0NP00MP1f07]M07mf07X0MgX0M`1gNP1f
N`1j07Ik07YkG@00NW]M07H0NP1MNed0MP1j07IkG@00MW/0GG/0N`00MP1f07X0
MP00NWMj07]MNed0MW]MNP1gGG]M07H0NP1j07L0MgX0NP1kMP1k07IkG@1j05eg
Ned0MWH0N`1j07H0Neeg07/0MaEj07X007IMN`1fN`1fGG/007H0RXX00AUoRP1j
MP1MNeekN`1MNgH007]gMP1M07]M001j07]M07]g07H0NWL0MP1kNP1jM`1jNgH0
NP1f07]M07Mf07H007]j07Mf05d0NgH007X007]j001kMed0Ned0Neek07X0NP1g
MP1kN`1j07Ij07X007/0GG/0MP1kMP1kNgH0MP1oNgH007X007X0NP1MN`1f07]j
07H007IMNgH0NgH0NP1jM`1kMP1j07Yk07H0NW]M07X0MP1jMgX0NgX0MP1kMP1j
001MM`1kGG]MM`1j07X0M`1kGGMMN`1j07X0MP1f07]f05ek07H0NgH0NP1kMP1k
NP1f07H0NP1kGGMM07XE07H0MgIMM`1MN`1f07/0NWH0NhZ:004IOhYk07X0Meek
05ek05ekNP1MN`1fNed0MgX007H0MgX0NP1gNP1j07Ik001f07X007X0M`1jN`1M
Ned0NgX0MWX007H0NW/0MP1jN`1gNP1M07Ij07X0MP1kG@1gG@1fN`1j07]M05eg
07Ik07H0MWX0MW/0NP1f07Yk05ekGGMjO`1kNP00MWH0MP1fN`1j07X007IkNP1k
05eg07X0M`1f07X007Ik07H007IkG@00MWL0MW]M07X0MP1kGG/0GG/0GGIk07X0
07H0N`1j07H0NP1f07/0N`1f07H0MW/0NP1MNeeg07X0MP1j001f07Yk001jN`1j
N`1j07]M07Ik5@1j07X0N`1fN`1j07Ik07X0MP2:RP016Gn:07X0Meek07]f05ek
07H0NP1kNP1g07X0NWMj07X0MP00NP1f05ek07IMNgH0NgH0MP1j05d0NP1g07X0
MP1k07X0NW/0GG]f05d0NP1MNgH0NgH007]f07H0NP1MN`1j07X0NgH0NeekG@1k
NP1g07X0MP1MNgH005ekG@1k07mf001fGG]j07]MN`1j07H0NgH0N`1f07Yg07H0
NP1f07X0MeekG@1j07]j07MfNP1j07X0NgH007]f07/0NWH0MP1k07X0MeekMP1f
07H0N`1jNgH0NP1MNgX0Ng]M07]f05eg07X0Ng]f07IjNgH005ekMP1j001f07]M
07Mj01EMNgH0NP1MN`1j07]j07H007]jRXX00AUoRW/0NP1j07X0Neeg07H0MP1f
07]M07H0MP1j07H0MW]j001fN`1fNeeg07X0NP1k07Yk07IgNP1f07X0MW]k07Ik
001MMg/0NP1gNP1fN`1kG@1kGGH0NgX0MP1jM`1j07L0NP1kG@1g07X007IkGG/0
MW/0MP1jNed007Ik07Yo07YgN`00Ned0Meek07IkG@1jN`1j07H0NW/0MW]j07X0
MP1f07Mj07H0Ned0N`1fN`00MP1jNed0NP1MM`1kNP1f07H0MUek07]MN`1j05eg
05ekG@1fN`00MP1MNed0NgX0NP00MP1j07L007X0NWIg07]M07Ik07H0Neek07HE
N`1kG@1j05eg07H0NeekGGH008Z:004HOhX0MP1f07H005eg07X0NgX0NgH0NgH0
Ng]f07H0NgX007IjN`1MN`1k07X0MP1f05ek07H0NP1g07Yf07/0G@1j07If07X0
MP1MNed0N`1j07]f07]f07X007]j001j07]MMP1f07]MNeekMP1j07/0GG/0GG]M
N`00MWYk07H06gX0G@1kMUd0Neek05ek07]f001MN`1kMP00Ng[n07Ik07]j07]j
07H0Ned0Med0Ned0NW]f001kMP1MNeegG@00NeekNP1kG@1MN`1j07Mf05ek07]M
N`1jGG]MN`1kMed0M`1fMP1j07]MNgH0MP1k07X0NgH0NgH0Neeg05ek07]MM`1k
MP1kGG]f07H007]MNhZ:004IOhYk07IkN`1fNP1j05eg07]M07X0GG]M001jN`1f
07]MN`00GGMk07X0NP1g07YkGGL0MW/0NeekG@1kG@1fNed0MW/0MW/0NP1gG@1f
07H0MP1j05egNP1fM`1gNP1f05ek07IkG@1gG@00MWX0MP1fN`1fN`1k07Ig001j
07X0OgMj07H0NP1f07/0MW/0NP1kGGIk07H0NgX0G@1fMWX0MP1kG@00MW]j07]M
N`1f07Mj05ekGGL0NP1g07X0NWIk07L0GGL0MgX0GGL0MUekN`1MN`1gG@1fM`1j
07H0NP1f07Yk07Ik05ek07X0Ned0NP1f07Yk07X0GG/0NP1M5G/0NP1f07Mj07/0
NP1kGGL0N`2:RP016Gn:07X0NP1f07]f07]f07X0NgH0MWH007Yf05d0NgH0MP1k
MP1j07H0MP1f07Yg001j07X0MP1j07]f001jN`1f07/0NW/0MP1kNP1gNeekNP1f
07Yk001f07X0NP1f07Yg07]j07L0GG/0NW/007]MNed0NP1gG@1j07Yf07H0MWmj
07H0NgH0NgIM07X0NP1gG@1k05ekMP00NWH0N`1kMP1gG@1kMP1j001MM`1MNeek
G@1k07]MMP1MNgH0MP1k07H0GG/0NWX007]MMP1k05ekMP1f07]MN`1j05ekMP00
NgX007]j05ek07H0MP1kMed0NgH007X0NWH0MP1jMaD0NP1kMP1j07X0NP1gG@1j
MP1jRXX00AUoRW/0MP1jNed0Ned0Ned0MP1kN`1kNWH0Ng/0MP1j07]M07Ik07X0
MW]k07X0NP1f07Ik07X007H0NgIk05eg07X0NP00MW]M001fG@1g07]j07L0GGYk
07H0MP1fN`00NP1M07IMN`1MM`1MMWL0NP1gNP1j07Ij07Ig07Yk07]o07IkNP1k
G@1gNed0MP1fGG/0MW/0GG]j07IgNP1j07]j07]M07Ik07IfN`1fN`1gG@1f07H0
N`1fN`1j07]M07IkNP1MN`00MWH0N`1j07X0GG]MNed0M`1j07Ik07X0MP1j07H0
07Ik07IkN`1j07X0MP1jN`00MP1kGG/007XE07H0NP1f07H0MP1fGG/007X008Z:
004IOhX0GG]f001kMP1kMP1gGG]f05d0NP00Med0GG]j07H0MeekG@1j07]M05ek
MP1f07]j07H0MeekMP1kG@1kGGH0NgH0NgX007Yf07]kGGH007X0NWL0G@1kNP1k
NP1gNP1f07]j07]f07Yf07]MMP00NP1f07H0N`1k07Yk07H0Oeek001j07L0NP1f
07]f07]f07X0NgH005ek001f07]M07X0Meek07H0Ned0NP1j07H0NeekMP1jN`1j
07X0MP1j001jM`1jMP1kNP1MNgH0NWL005d0NeegG@1k07H0GG]j07H0NW]f05ek
G@1f07H007]MN`1fGG]f07/0MWH05GH0NP1f07]j07]j07/0MWH0NWN:RP016Gn:
N`1MNeek07X0GG]MN`1MNeegNP1fN`1fN`1M07YkG@1g07Ik07H0Neek07]MNed0
NeekGG/0GG]M07Mj07]kG@1kG@00MWL0NW/0GG/0MW/0NP1j07Mj001f07]M07Ik
07H0Ned0Ned0NgX0N`1fMP1jM`1jN`1f07X007IkGGmk07Ik07IMM`1j07H0NW/0
Ned0MP1kGGMk07Yk07H0MgX0MP1f07Yk05ekG@1f07YkG@1g07X005eg07Ik07Mj
07Ik05ek07]M07]MN`1k001jMWMkG@1j07Mj07X0MW/007IkN`1MN`1f07]MNeek
GGH0M`1fN`00NP1MM`1jNaD0MW/0NW/007H0Ned0NWL0NP1jRXX00ADcRUd0NP1g
07H0GG/0N`1jM`00NP00N`1MNed0NgH007]MNeek07H0Neeg07X0N`1kMP1k0?]T
O`1j07]M07]f001fNW]MMP00GG/0NgX0MP1g07Ij05ekGGMM07X0NgH0NP1kMP1k
MP00MP1MNg/0N`1j001MNgX0MP1j07L0GG]M07X0NP1j07]j001j07H0NgIM07X0
MP1M07X0NP00NP1kNP00NWL0M`1kNP00NgH0NP1jMgH0Feek05ekG@1kG@1j07H0
NgH0M`1j07If001j07]f001j07H0NW/0MeekG@1jM`1MNgH0N`1j07]g07X0NP1g
GG]f07]j001jN`1f001jMP1j07MMN`1j07L0RXX00A8cRW/0MW]MN`1fN`1f07X0
NP1f07IfNeek07Mj07Mj001f07X0MW]M07]M05ef07H0NP1MMoQTOfETO7d0MWL0
07Yg07X0MP1f001fNeek07/0MW/0Neek07Ik07Yk07X0Ned0MgX0Neek05ef07X0
MW/0G@1fNgX0NP1fN`1gNP1f07H0MP1kGGL0NP1f07/0MP1fNeegNP1f07IfMed0
Neeg07X0NP1MMW/0GG]M07H0NWMMN`1fF7/0MgX0MgX0MW]M07X0NP1fN`1jN`00
MP1jMgX0MW/005ef07L0NP00NP1g07X0GGL0MP1jNP1f07X0MP1j05d0NWH007Ik
NP00NW/0MP1f07X0MP1jRXX00A0cRP1MN`1k05ek05ekMP1g07Yf07]j07L0GG]M
07X0MeekMP00NP1kMed0GG]gGG]f07]f0?UT16ETIGemngaoI`QW26P8IeekNP1j
07H0GG/0MP1k05ek07H007H0MP1f07X0Meeg07H0NgH007]f07]f07/0MP00Ned0
NP1f07]j07]gG@1MNgH0NeekMP1kNP00NP00NgX0N`1kMP1f07X0MP1f07/0NWH0
M`1kMP00N`1MN`1MNP00NP00N`1>NgH0MP1MN`1MN`1fGG]f07X007/0NWH0NgX0
MP1jMP1f07X0NWH0GG]f001f07X0NW/0MgH0NWL0MeekG@1kMW/0MP1jNgH007]j
08Z:004=<hYk05eM07Ik07Ik07X0NP1k07H0Ned0MW/0MgX0NP1f07YgNP1j07X0
MW/007X0Ned0Ng[jI0IUI6ETIFAToGclO@88J0SkO7l8NP1MMeej07Ik07X0MWX0
MW/0NW/0NP1j07IkG@1jMgX0NP1MN`1MMg]MMP1gNP1j07H0Ned0NP1fN`1kG@1g
GG/0G@1jM`1fNP00NP1j07]MNed0MW]j07H0MP1kGG]M07]j05eg05egN`1fNP1f
MP1j07]MN`1f07Ik07Ik045MN`00MWH0NP1kG@1kNP1g07Yk07H0MWL0NP1j07Ik
07X0M`1MNP1kG@1jN`1j07/0G@1MM`1j001jMgX007J:RP012S>:07H0MeekG@1j
07X0MP1fG@1kNP1kGG]M07X007]f07X0NP1g07H0GG/0NgH0Ned0Med0nF@5IFAU
I6ETo7cnO@=lJ0QXoPP5PX:1N@P8n7aoPG`0Ng/0NP00MP1f001j07]f07X0NP1j
M`1f07]M07/0NP1j07]kMP1kMP1MN`1kG@1kGG/0MP1jM`1MN`1fNP1f07X0MP00
NgX007Yg07]f07/0MP1j07H0NWH0NP1MN`1f07]MN`1MM`1jM`1j05ek05ek07Yg
NP1f07H0NgLeG@1MNeek07H0Ng]M07]f07H0NP1kMP1MN`1j07]f001f07H0MP1k
MP1f07]f07X0NeekRXX00A0cRWX0NP1g07Ik07H0MW]k07Mj001f07L0NP1MMW/0
NW/0N`1j07Ik07MMNeek05egNP1kGOaT3FETIFAUI6ETI7amO7eloWd9I`QX20QW
289_L_h80WalOO]l01ckO7n107Yk07Yg07X0N`00MP1g07X0NP1MMgX0MP1g07X0
07X0GG/0MeekG@1f07X0MW/0Neek05eg07Mj07H0MW]MM`00MW/0NP1MNed0MW]M
07Ik07/0MP1fN`1fNed0Meek07X007Yg07Ik07Ik07X0NP1jM`1j05ek07Ik07/0
MRikNP00NW/0N`1jN`1f07]j001f07H0NgX0NW/0MW]M07]j07H0N`00MP1g08Z:
00457HYgMP1MNeek07H0NW/0G@1j07]MNgX0MP1f07X0M_h04UegG@1j07H0NP1k
07H0NP1gG@3kI0AUI6ETIOiTo7cnO@T8J0QX20Q`Kg1_oPP8O7elO7elOGamlg`1
787mP7l8Neef05ekMP1j07H0Neef07]f07Yg07Yf05d0Ng]f001kGGMM07H0NP1k
NP1f07]j07L0GG]j07H0GG/0Neek07H0N`1f07H0NgX0GG/0NgH0NP1f07If07X0
Ned0NP1f07H007/0MP1jM`1MN`1jG@1j001jMP00MP1j05d0Nb5f001fMP1kMP1j
07H0G@1j07]fNP00NgX0MUekMP1jRXX00A4dRP1kNP00MP1fN`00NP1fNP1f07MM
07Yk07Ig07IkNP1fN`1fN`1j07H0NP1fN`00MUek06?kI1QUI6ETIFESO84LO7`=
OGeX26L8K6][K6]/oPP@O7amO7amO7elOGamO7elO7gjO061O?j0O`Qi20QhNGU[
N`1fNgX0N`1f07X007X0M`1j07H0GG]MMP1kGG/0NWH0NP00MW]k001j07X0G@1f
N`1f07H0Meek07X0MW]j07H007Ik07X0Ned0MW/0N`1f07H0MgX0MW/0NW]M07Yk
001j07H0MP1gNP1fN`1jNeek07Ik07H08GYkGGL0NP1kG@1fN`1gNP1f07/0MWX0
07Yk05ekG@2:RP0153>:Ned0MeekGG/0NWH0GG/0MP1j07]f001jN`1j07/0MWH0
GG/0NP1jNgH0GG/0MUekNP1goV@2HfASoV@0HoeT087mO?im40QWIfa=K6a[K6]_
PX9mO7amoG`0OOil0gelO7gnO7mmO7elO1b0PH20N@QiN@QiNFa[Jg1_PWQiN0Qh
27Uh07H0NW]f07Yf07L0NP1k07H0MgH0Meek05d0NP0/07Mf07/0NW]f07]MN`1j
05ekG@1kNgIMN`1j07X0MP1k07X0GG]MNeekG@1k07H007]f001jMP1jM`1j07]M
07/0NR/0GGL0GG/0GG]j001k07X0Neeg07X0MgX007]f05ek07Yf001kMP00NhZ:
004B?hX0MgX0M`1j07X0MgX007Ik07X0GG]MM`1MN`1f07Yg07X0GGL0MP00Neeg
07X0M`00MWYTHfATHfASI6=TI6?mI05SOOil07gn20==JddYoTd7PX9`O7elO7gm
O01moW`3OGalOOil07gnO061O?j00PQi2?eioW3nJoi`OgT8N@P8N0Qi26][=CDk
K4e[L5d0MW/0GG/0NP1j07Ik07H0NP1kMP1j001jN`1M07Ik07L0NP1M07/0GGL0
MP1fNed0MW]MN`1k07X0MgX0NW]MMP1kNP00MW/0NP1gNP1gNP1j07L0NP1f07Ik
001fMP1j07H0Neeg07X0GGIk4@1j07MMN`1gNP1MNeeg08Z:004H=hX0NP1kGGH0
MP00NP1gMP1kMP1MN`1kGG]f05ekMP00Ng]f07]MN`1jGGL0GG]f07YMN`1SI6AS
oV@NHfATHfATHfASI7amO7elO0QWJddYK0H6CVMi27alOOil27elOGalOGalOOil
07gnO0YmO7b0O7T8P0Qi2?ii5W1_L6]/Jg1`K`Qi27UiL71_L3De>`@4oTeoL71_
Kg1_K`1j07H007]M07]j07X0NgH0NgH007YgGG/0;01f05ekMP1kMP1f07Yk07H0
NP1g07H0MP00NP1gG@00NgH0MP1jN`1f001j001j07H0MP1f07Yg07X0MeekNgH0
GG]gG@1j07]f07/0MP1j07H007X0NWH0M`1jRXX0004?>8Yk07X0N`1fNgX007H0
NW/0NP1gG@00MP1kGG/0NP1j001j05eg07Ig07Yk001j001f07YTHfATHoiT0f=T
I6?mI0mSO7amO7am20Q=K4e=CRT6n`P6OH5lP7b0POal17elOGamoW`4OH1lP0Sn
N@08oWTTL6m`Kde=K6m`L7T8N@QiL6m`KcDk=@@4Jfa=Jfa=K4e[CF][K?hkoF]_
K6][05ek05ekGGL0NP00MP1jN`1g07X0GG]j07H0Neek07]M07IkGG]MM`1j07Ij
001j07]f05egNP0/07Ij07Ik07Yk07X0NP1kGEek05ek07MM07Ik07H0NW/0MW/0
MW/0MWX0M`1kGG]M08Z:004?A8X0MP1f07X0N`1fMP1jN`1j07Ij07Yf07]MM`1j
07H0NWH0MP1j07X0NP00NWH0GG]kMP1TI6=TI6=THfATHfATHfASI?]loPP6CDe[
CPI>Io`83GalP7b0PAb1O7amO7amoW`2OGaloPP9O0P8N@Qi27P8N?i`2fa=K71`
Kg1_L6m`Koi`4C/e=@A0K2U/Jde/Jde/K6]/Jo`k1V]/Jfa[JfcnJfLk>`1j07]f
07Yk07H0NP1jMP00NP1f07/0MUd0NgX0N`1kGGH0GG/0MeekMP1kMP1j07H0NgH0
07/0GG/0MP1f07MM07]M07]M07X0NW/0GG]f05d0NP1MN`1jN`1f07X0Ned0NhZ:
004>>8Yk07IkG@1f07Ik07X007H0M`1k07MMN`1j07H0MW/007Yk07H0NP1j07Ik
07]MM`1MN`1TI6ETIOeT0f=TI6?mI0F1O85mOGclPPJ3K2U/K89`oPP8O81lP7b0
PH0=o7`0OOil47el27T827T827T8N@Qi26m`oTd4Kg1_L6onL0A_L6m`Koi/oSD;
K4e/CDe[Jde/Jfa[oS/==C]/Jfa[K6][K6]/=C_mJ`B7Qf]_Koe[F7L0NP1gNP1j
07Ik07H007H0MP1k07Ik07IMN`1kG@1k07]MN`1j07YgGG/0GGIkN`1j07Ik07Mj
07Mj07X007Ik07X0MgX0MW/0NP00MW]M07H0Ned0RXX00@lhRP1kNP1f07]MN`1j
07MMNgX0MP1jGG/0MP1j07Yk07If001kMP1kMP00NP1kGGL0GG/0NfAUI6EToVD8
I6AUI6ESI6=Tng`2L89`oX85:F`Y9722oPP0O?f00Gb0oG`5OGamO7amoW`220Qi
o0P9N@Qi27U`Kf`YK?i=2Fa`Kg1_L6]`L6_nC@le>cE/Jde[CFa=Jde/Jf`eoC/D
Jf]=K6]/K6]/Jc/kJfa[K6a[K6][oS/2Qf^7oWQCNGQM07]M07]fGG]MNgH0NgX0
MP1k07X0NgH0MP1k07H0NP00NP1f07MM07H0GG]f07X007X0MP00NgX0MP1MNed0
N`1j07MMNed0NeegG@1kRXX00@HhRWX007IkNP1k07X0MP1k000/07X007X0NP00
MP1kGG]kGGH0NgX0NgIk07H0Neej05egGFETIFAUoV@8IFETIFATHfASoV@4O7el
PW3mPPU/CFa/PW08N@Qlo80;O81lOGalOGalOGalo0PMN@P8NGT8N@Qi26m`CF`Y
K4e=Kg1`Kg1`K6][CF][oSD<>de[K4e[CF]=Jfa[Jo`k06cnJ`9/JcGl>`Q/Jfa[
Jfa[K6cn>oj70g1_L4CiN09^P6klP49k07X0MP1f07Yk07H0MW/0MgX007IkNP1k
GG/0MP1kG@1fMP1fNeef001fN`1gG@1f07H0MP1f07]M07MMNed0RXX00@<nRP00
NgX007H0MP1f07Yf07X0NP1gMP1f05ekNgH0Ned0N`1gG@1f07/0MP1f07/0MP1j
06AUI6ETIFAUIFAUnV@2HfAmoG`127Wm20A/CFaW2?iW0@P8oG`33GalOOil07gn
O?h807Wn20Ai20Qi2?ii5W1`K4e=JF]=L71_L6m`Jf]/Jde/>cDkoSD8CDe/Jde[
CFa=oCD7>cE[K6]/Jfcn>`/e>c][K6]=K6a[K6_n>`>7JhM`oT@0NOYh0GUhmh0n
NeekMP00MP1kNP1fG@1jMgX007]MM`1MNgH0MP1j07]j001kGG]j07H0NgH0NW]f
07YkMP1kMP1kG@1kRXX00?m<RWX0G@1fN`1fNg/0MP1kG@1j07H0NW/0MW/0GG]M
07Mj07X0NeekG@1fNgX0NP1fN`00MVAUI6ETIFAUI6ETIFESI6=THfATO7amO7cn
201ioPP3Jde=2?iW2PP8O7b1784LO85ln`P0NOT8:gUi27T8L6m`:Fa=>cDeJfa=
Jde[K4e/K4e==C/e=C/eJf]=K6a=Jde[>c/eoC/5=C]/Jf`eo3/0K?a[0Va=Johk
16a[K51_oT@9NGQiN7QiN7T8N?Z0nVh2KGQ@o7PaN`1M07Ik07X0MW/0Neek07Mj
001jM`1gG@1jNed0Ned007Yk001j05ekG@1gNP2:RP010cj:07H0NgX0NgX0G@1k
NP1f07]g07Yk07]f001j07L0NP1f07H0N`1kMP1k07H007X0MeekIFAUI6ETIFAU
I6GlI0]S785lO7elOGal27Wn219iCDe/I`QWI`P8O84LPH5lPAalo0PDLX:2L89`
27T8N@QiL7T8Kg1_CFa=oSDMK4e[CFa/CF]=CF]/Jde/>cDeK4e[CDe[CFa/=C/e
o3/3K6]/=O`k5F]=K6]/K3E[Jc/k=Fb7JhM_KdA_A4CmQ`A[NGUhN?^0n6h2N7Uh
oHLa>c]4A81^KX1^P81^KW/0MWX007]f07X0NgH007]f07]gNP00NgH0NgH0M`1j
07^:RP00nDN:N`1f001f07]MNed0NeekG@1j001MMeekGGL0NP1f07Yk07YM07H0
NW/0NW]MM`1j07H0I6ETIFAUI6ETIFAUI6ASI6B1O87mO0ImO7amO0QioPP4Ide=
K6Ok20AlPGalPO`83X9`L89`PWT827Ui26m`Koi`8f]=JcDk=Fa/CF]=K6]=Jfa=
CF]/CCDk=Fa/CF]/CF]=K3Dk1?`k0V]/Jo/koV/2K6]/nc/1JfcmJ`=/Jg1[o8L0
N?iingP4P6i^P6kkN?Z70Fb7oWPZP81^Hf<007Ik07Ij07]M07H0NgX0GG]M05d0
MWX0Ned0Neej001f08Z:004<=hX0NgH0NW]f07/0NgH0M`1gGG/0NWL0NP00NgH0
GG]MN`1f07MMNgH007H007H0MP1jN`1kI6ASoF@AIFAUI6ETI6=TOGalOGamO7am
oW`0NOh82VL8CF`YI`QW20QWm`P6NG22PW22L?h80WT8NOi`9Fm`Kfa/CC/e=De=
Jde[CDe/CF]=Jfa=Jc/e=F/YK6]=CFa=K3/eoPCm>`4e>oh423/k=Fa[K6]/Johk
2SDk>fa[Jfa[K6]/oS/4K6][K7WnN0AiN7QiN?e^1ff0N7UhN7UhoXLcN7R6K6][
N7R0P6j0MG]f07X007X0Med0NeegG@1MN`1f07]fN`1j07]f07/0MWH0NXZ:003]
AhYj07]M001j05ef07X0NP1j05eg07X007If07Yk05ek07X0MP1k07YkGG]MM`1j
N`00GGH0HfATHfASIFAUI6ETIFAUI6Em3Oil1GelO7elOO`816M=CFaWm`P0NO/8
0G1`nddRL6m`Kg1_L6m`CF]/=C/eJfa=Jde[K4e[CF]=CF]==C/k=C_n=@@k=C/e
>od4nc_n1?<k4F]/K6]/K6]/Jf/k>fa[K51hNOih0gUhN7WlN0E_A4B7N7SnQoQh
:fj0KWD007Yg07Ij07H0MW/0NP1gNP00MW]M07/0MP1f07X0GGL0NW/0RXX00?/g
RP1f07]gGG]f07/0NgH0MP1f07Yf07Ij07]f001jM`1gG@1kNP1f001j07/0NW]M
07Yf07MTI6?mI0iUI6ETIFAUI6ETOGelO7gmO05mO?/81Fa[CF]=K?X807Wm20Mi
27V2PTdY1_i=06onL2M_L6]/Jf]=K3De>cDeJde/K4e[CF]=K6]=K3/e=C/k=CDk
=C/k=C]Om@@2>c/jnS/0=Ohk3f][CFa/Jf]/Jf`k>cF7JgWnN01ioWP0NOih0WUh
Koi4oXOkN01ioWP]NGR0KX1SNgH0NP1j07L0NW/0MP00NP1f07X0NgMM07X0NgH0
MP1kNP00NhZ:0044A8Yk07X0NP00MWX0MP1jN`1fN`00NW/0MgX0Neek05ej07Mj
001jNeeg07H0NP00MgX0NgX0HfATHfASIFAUI6ETIFAUO?im17amO7amoW`=O@QW
20Qi24e/CFa=CGWl20=i20Qio@P2Jfa[o3DPJFa_L6m`L6]/Jde[CCDk=CDkCF]=
Jfa=K6a=Jde[CCDeoP@8=C/e>cDk>cE0n`@0>?H40c/k>0Cn>`5/K?e[16a=Jfa[
oC/0>_j70WQhNOih07WmN094KdCnQcQhNGQiN7V0KX1^P6j0KX1d05ek001fNeek
001j07IjN`1fNgX0NP1j07H0NP1kNP1f07]MMP2:RP011cN:07H0MP00NWH007]f
001MN`1jMP00Ned0NP1f07]f001j07MMN`1f05ekMP1f07X0MP00NfATHoeT1VET
IFAUI6GmO@0=oW`3OGalOOil485l26L8L6a`CF][20Qi20Qio0P5N@QiCFa=ncDO
:@I=Jde/CFa=Jde/>cDe10@k=CE/:Fa[CF]=Jde/>cGn108k>cGn1090@5ol10=S
I6ASnP@0>Ohk30@4>c/eJf]/Jfa[CC[k>`4j>oj707WnN01ioWP;NGQhK6]/QgQ_
N7QioWPbKX1^P6j0KX1^HgYg07H0NP1k07Yf07Yk05d0NP00MP1f001kMP1kG@1f
07YMM`1kRXX00?]2RW/0MW]jMP1kNP1MNg]MM`1j07Mj05ek001fNed0Neef07X0
MP1fNgX007Yk07H0NW]MMeeTHfATHfATIFAUI6ETIOam2gamO7amO7amO7`LPOh8
1H:2L6a=K?h807Wn201in`P2Jde[ncGmCAI[CF`YK6a=CCDk=@A0=C/eK6a=CF]=
K?de33]O10@e>c]010AO@43l1?iS1FASI6=SI?T4oS/>K6]/Jfa[Jc/j>cXkQfb7
o4@0KoUh1GU[Qf]hQoeh1H1^P6j0K_j0;6j0KP00NP1jM`1j07H0Ned0MW/0MWMj
07]MNeef07]M07MjM`1k07X0GHZ:0045@XYMNgX007Mf001jMed0MP1f001j07Yg
05ekNP1f07X0NgH0GG/0NP00Meek07H0NWL007X0HfAlPGb1O7emI6ETIFClO@9l
O7gnO01mo7cn20M`L89=Jdd8NOd807Wn20Ii27T8CTd6ncGmCAD6Jde/JbU/Jc/e
=@@4>cDeJde[CFa/oSDH>c]O15le>cEO@41OHfATHf=THf=TN@QSI?eSo0@U>@@h
13/kJfa[>cDk>cY[Qf^7JhM[A4A_A6m4N7UhNGQ[Qfb7KfomN0M^N81^P6j0K_b0
:fiSNgH0GG/0MP1MNgH0MP1k07Mj001f07L0NgH0NgH0NP00NP1f05ekRXX00?Pf
RP1M07Ik07]MM`1j07IkNP1fM`1j07X0MP00NWMj07H0NgX0GGL0MWX0N`1fNgX0
GGMj07YUI?il01cjO01moG`0OOX=0galOH7mO0T827U/CF]/20Qio@P0NOd80gT8
1Tgk=O]=3F]=K6a=JcDe>cDe>cDkoSD=>de[=C/e>cD4Gd0e=C_n1?YS1VAS20Qi
27WmO?iS0VASHoh41C/kK6]/=OdkoXL0K?b7oTA8KgQiNGQiQf^7Kfm4NGQiP6i^
P6j0KX1^P81^KX1eM@1k07H0NW/0M`1jN`1f07H0NgX0NW]M05egNP1kG@1fN`1f
N`1f08Z:0040@HYjM`1j07X0NP1f001j001kNP1f07H0MP1kMP1j001kMP1f07]M
NgX0MP1MNed0M`1j07/0I6ElO7elOH5mO7elOOiloWd63Gd=O@em3O]l087n20][
CFa[CDd827T827Wn20EiPX88C@Kk=@9=C@KmC@m[CFa[CCDk=CDk=C/e>cDkoSD1
>cGn>oiO0PAO=Oh4nf<3I6=T2?ii0`QlO87nH`=THf=0oP@1>0Ck>`<j>hM/nXL=
A6m4Kb=_JhM/QhM[N7WnN?a^=H1^P6j0P6j0KX1^MgH0GG]gG@1kGGH0G@1kGG]j
G@1f001kMP1k07X0NgH0NP1j07H0NhZ:003l=8X0NP1kG@1fMeekNP1kGG/0NgX0
MW]k07Ik07H0MW/0NP1kG@1k001jNed007IkGG/0NP1foWd03Oim0WalOOil27el
OGd=3Gd=OOh=0GemnW`4N@Q/CDgnJoX81gT82722L4e[oTgn=Oa=2`I[CF]=K6`e
=C/e=OhkoSDI>cDk=C/e>c]015mO15l4141SI6=TH`Qi27Wl:PQIO7aTHfASI6?l
1?Pk5F]/Jfa[K6][K6]/Jfa[Jfa[Jfb7QgWnN0MiKVj0KX1^POb0;Vj0KVh0Neeg
07X0MP1k07X0MP1g001fNg]MNP1kG@1f07H0NgX0MWMMNg/0RXX00?ldRWH0Meeg
07]M07/0MP00MP1M07]j001MNed0Neek07X0MP1kMP1jMP00MWX0NP1k07H0MP3j
O@5lOOil1WelO7em3Ggl3@Mm3GamO7b0O?/84GT8C@I[20Qi27T827T8PW22Joe=
oSD1C@KlC@U[CFa[CDe/K3/eoPCn>`de>cE0Gc/k=@AOG`AOGoh406CnH`UTN@Qi
22XZQ2XZoUT727T8Hf=THf?m100hnc/AJfa[Jfa[K6]/Jfa[K6][K6]/oTCkN0B0
KVj0K_b0<6j0P6j007X0N`1jG@1kGG]f07YkMP1jGG]M07L0NP1gGG/0NP1j001k
07X007Z:RP010SB:N`1j07X0GG/0GGMjNeekGGIk001fNP00MWX0N`1f07Ik05ek
001jMg/0MW/0NP1fN`1jNoYm1WalOGamO7cmO@L=O@em3Gd=3O]loPP4N@Qi20Sn
C@=[K8=`oH88L722PV]/1Te=oSGnCA86:De/K4e=Jf]/:CDe>`A0=C/eoS_l@0D4
G`@4I6ClH`MTO7b1CF`8N?dZ1hAnFEUi27UToF<510@h10@ioP@E>c]/Jfa/Jfa[
K6]/Jf`e>c^7Qfa4A?ah2WUhKVj0KX1^P81^oH0]KVh0Ned0MP1gNP1k07X007X0
MP1g07X0NP00MP1fNed0MP1fMP1j07Ik08Z:003l=8X0MP1f07Yg07H0NP00MP00
N`1f07]f07Yk07H0GG]MN`1jM`1MNWL0G@1k07H007X0NP00nGd8O7amO7am3@em
oPd2O@emoPd8O7amPAb120Qio0P?CBU=K4e`L89`L89`PX9`K?e=ncGnC@A/:Fa[
COhe2S/k=@@4>cDk=C/eoD061410141THo]T0gb1O6cnCOi018F5@41OoUT427T8
HfCnHoh463P413/kK6]/Jf]=K6]/Jfa[>cXkQhM4A6okN0A^KH1^K_V0;Vj007H0
Neek07X0MP1j07Mf001kNP1j07H0MWH0NgX007]f07]kGG/0NP1kRXX00@DdRW/0
NP1f07Yk07Ik07IkGGL0NP1j07]f001fN`1f07MMN`1j07L0NUekG@1fNeef07Ik
07KdOOil3gelO0d=O@em3GelOGalPGcm201ioPPCK6a[:Fa/PW22L722L722CF]=
1Tgn=@8k=C_mCC1[K4e[=C/e>cDk10@e>cDk=C]TI6<415m0Gd1TI6=THgelO@Ql
CBU/CD1OGh@ZQEm0oET2N@QToF?n108h10Cn>a1[K6]/Jf/e>c/e>cY/>cZ7Qoi4
3WUhNGQiKX1^KX20KX20K_b0;Fik07X0Meek07H0MW]MN`1fNP00MW/0NW/0NW/0
07IfN`1f001f07H0NP2:RP00mCF:05ek07]j001MN`1MNed0NgH0NP1f07/0NgX0
MP1kGG/0MP1f07X0M`1f07X0NgL0N`1kNP3dO@5lOOel0Pd=OOh=o7ci20EiCDe/
1TgmK0E[CFa[CF_lCO/e34e=1V]/CF]=>cDk=C_n108e=C_n=@ASI6@41?i00V=T
HoiT17`=OGUio5T6Ogm?Be/Z:_j52WiIFGT8HfASI6=SoP@9>S^7Qf^7Jf/k>_hk
2f]/JhN7JhM4KdAhNO]h0X20K_b0<Fj0P6ik07Yg05ek07X0NgX0N`1MNgH0NW/0
GG/0MP00NWH0N`1kNP1jN`1kMP1kRXX00?4eRW/0GGH0NeegNed0MP1gNP1j07Mj
07H0MP1kGG]M07/0NP1fNed0MP1jN`00MP1jNP1f001jmgd83GelOGalOGaloPd2
O@d=o7cm20Ii20Qi26][ndd5Jde[CFa/ndgi=AM=CF]/CF/e>cDk=@AO15m0G`@e
>fASI0Cm@?iT4f=TH`el3@QiFEUnFEUnOU];BeTZoXGnF@88NFCnH`1To@@D>3Pk
>cXk>S/j>cPh10A[K6^7QgR7oGP4NGQhNGWdP2d0NP1MM`1MMP1f001f07Ik07]M
001fN`1fNgX007Yk07X007H0NP1j07H0RXX00@@eRP1f07]f001j07]f07]j07H0
07X007]MNgH0N`1kMP1f07]j07]f07Yg07IMNgX007H0NWH0mWd13GgmO0MmO7d=
O@em3Oil0H1ln`P2N@P8oX85=FU=CBT6oTd11RWmC@86CDgj=@M=CFa=K4dk=Ohk
0el4G`Cn@044Goh40VATHoiT0F=ToWd@20R2L71QHEaQFGioBdmI:XCnQOiI1WT8
HfASI6?n1?hh0C/joS/@>S/k>0@4Jfa[Qf^6Kfn7N7WnN09iNGSnP021oX0cKX20
KX1kMP00NP1jM`1kNP1jNgH0NP1M07]fNed0NP00MUek07H007YkMP1f07H0NhZ:
0042=HYj07Ik07If07X0Ned007IkNWH0MWX0N`1jG@1f07X0NP1M07H0NgX0GG/0
MP1kGGL0NP1gN_Mm2PemO7elOGalO@emoPd1OGgmO0=i20Qio@P8NH9`PSDe1Td6
nTd11RWkC@8k=C_n=@<k=C/eoS/6=C/e>D04@?h43Ul4141THfATHfASI6=mOOh8
3X9`L5aQHEUIOgj4Q6=THoiK1EmO@0QiO?eS20@4I7ESHcXk>_hk5CXk>3P4K6]/
QhM4A7QhNGQhNGUhNH7mP021nh0805ekMW/0NP1joP0SNP00NW/0MW/0GG/0MgX0
MW/0Meej07If07/0MW]k07X0RXX00@8eRP1jN`1f07]f07H0MeekNP00Meek07H0
NP1g07X0NgH0MgH0N`1j001kG@1kMP1j07Yg07X0nWd73@em3@elO7gmO0Em3@em
3@glO08827Wn20Ui20Q`PW0e=De>mDd01_i=4SDk=C/e>cDk=C/k=CDk>cD4@43n
10aO15mTI6=TI6=TI7emoPSnPQMQG5aO@5mIFHF5LF=SMEEKEEm0@0QiO87mH`@4
16=THo`k0cXh>23n>aE[K6][QdA4KhN7Jg1_Kg1_P821P84MoH0aKP1kMP1k07H0
MP1kMWX0MgH0G@1j07MfG@1j001j07]MN`1f07]f05ekG@1j07^:RP011CF:N`1M
N`1j07]MNed0Med0MW/0N`1fNed0MWX0Ned0NW]MNed0MP1fMP1gNP1kG@1f07X0
07KiO@<=3Gd=oW`<OGamO0d=O@em3GamO?/807Wm20j2L89=CFTeJCE=1RU=C@Kl
C@86CDgn=@mO13Dk=C/e10@i13Dk@0A0oP@MGd10HfATHfATHfAmO@QW20R2L65L
G5m0:WiIBd^9oV<1LHGm@0UIFGT8I6=SI6=Sn`@G>cXh>0@k>S^7K8M[K6m`KhM[
QWQ4KdA4nh0fPH20KX20NP1kG@1fN`1j05ek07H0NWMkG@1f07/0MP1fMWMM07/0
MW]j07]MN`1kG@1f08Z:0042=hX0GG/0MP1j07]M07]j05ek07YM07X0Meek07H0
NgH007/0MeekNP1kMP1j05eg07]j07MMNfMWo7d43Gd=3Ggl3@=m3@emo@d1OGgn
Ioh807Wn20Ii271`PTe=oCGlC@0YnDgn=@m0GcDk>cDk>d04141O15l4oV@0HoeT
1F=T3G`=O?/80GinoUl9:RY;BeV9RD]SMOj5oT0:FEUi27USHfASI6?n10A010hi
1?hk2XN7K8N7JhM[K6][oS/8Jf]4A6m^P80MoX00K_j0<G/0MP1f07X0MP1jM`1f
07Yk05eg07]fG@1kGG]k07]fG@1j001j07L0NP1kMP1kRXX00@PhRW/0GGL0MP1f
07Mj001fN`1f07Mj07X0Meek07H0NW]M07H0M`1kG@1kG@1fN`1f07]MN`0026Ql
nWdA3@em3Gd=O@d=O@em3Gd=OGd8oVL0NOh81GT827V2P_i=oCGjC@06oDd01_he
2El4=C/e=C/eGd3nG`A0Gem0HoiT2f=THfATHg`=O0d8Iod817moGd10oRX1RD_n
H`AeHd^9BohZ35aQPW1`O7b1I6=THf?n10Ti3P@j>hM/QhM[oXL2K6]/oS/5Qf]/
A6m4oWP2P81^o80a07X0MW/0Neek001j07Ik05ek07X0NP1gNP1k05ek07/0MW]M
N`1j07H0MP1j05f:RP010CR:07]f07X0NgX0NP00NWMM07X0NP1j07]MN`1MNgH0
07]f07X0NP1kGG]f07/0MP1j07]M07YX20gjO@4=OOh=0gd=3Ggn3@Mm3@emOFL8
Io/81722PW22oTgn=@E0Gem=C@KlC@1YoCD2>emOoSD2Gd1OoVCn@11O16=TI6=T
I6=TI6=T3G`=OOh816L8OginoUonF@28oV<@MF=SRD];Q2XZHEa`L1b1O7clH`0h
o@@@>3/k>XM[QhM[Qfa[K6/k=C[n>oah0X1^K_^0<GH0NW/0MP1k07If07Yk05ek
07H0MP00NP1gG@1j07H0GG/0NP1gG@1j07]f05ekRXX00@LgRWX0NeekG@00MW]M
M`1j07H0MP1g07H0NP1fN`1j07H0NW/0MP1f07/0Ned0MW]j07H0MgX0JdgnO?]m
80em3Gd=3Gd=O@em3Gd=O0elI`QW27T8NH:2L8:2L4e=1_he0Ul4Goi=00KnC@06
oSD4>cDeGeon=@D4G`AUIFCm@0/4I6ASI6=TI6=TOGgm20AW27ioO_iQ0GmnoUD<
Hf=eHgESHhQ;RHTZFOiQ1X:2PAb1O6CmHo`4oS/3K8N7Joj71f^7JhLk>S/joGP0
NOahoH0cPH00N`1MN`1f07Ik07H0GGMk07IkNP1fN`1j07Ik07H0NP1MM`1j07Mj
07H0Ng/008Z:0042>XX0MP1f001jMgX0MP1f07H0NgX0NeekMP1MN`1j07YkMP00
GG]MNgH0NP1kMP1j001j07X0Nde=O7elmWd;O0el3G`=O0elI`QWoX89NH9=K89`
PX9[K?i=oSD215m0oTd0:Oe=o3D2>emOoSD5Gd1OIFAUoT03G`ATHoeT1f=TI7em
20QWoPP2OgiooV41OWknEOiS0gESHgGnH`MKEEEOG`UnO_e`1h5lHfASI6=SoP@0
>O`k4cXk>cXk>S^7Jfa[N7QiN7QiN7Qinfhb07]M07]MMP1kNP1kNgH0NP1j07/0
MP1kMP1MN`1f07]f07]MMed0NP1f07]M07]fRXX00?`iRW/0MW]MM`1j07X0NW/0
NgX007H0NP1k07H0MP1f001jMgX007H0NP1MMeek001fM`1f07H0JdelO>em0`d8
I`SmPP]`Jde[PW22K2U/CDgn=A5O15le=C/e=Dd6CCDk=C/e=@CnG`04oUon@01U
nf@8Hgel3Gel3@P8oTd2OginoV7nGaIoOXB5Q6AeHf=eHf=eHgEEFeEOGgmnOoi`
oW`0I?iS0FASo@@6>c/j>cXk>_dk13Y[Qf]/oWP7NGQhNGQhKX3mKS80NP1f07/0
MP00NP1MN`00MP1f07Yk05ek07H0MW]j07]M07]MNed0MW]j07]M07^:RP010cN:
07]j07L0MP1f001k07MM07H0NgX0Med0NP1kMP1kMP1j001kNP1gMP1MN`1MNgX0
NW]f07]=CO]WnGd2O7amoW`126Om2?f28X==K4f21TdY1Te>CDde=D1O13Dk=CE=
CBU=>cDk=CDkGel4oEon@1IOI6ETI6=THfAl3Gel3G`824d6CGinOoiQ1emOOgmn
:W4ZoV<3MF=SMOiS15]KEGmooV7nFOi`0gT8HfCmH`T413/j>cXk>S/koXL0Jof7
0WQhNOeh0WUhN?e^<`1j001kMP1MNgYf05ek07If07YkMP00NWH0MP1kNP00NP1k
MP1k07MMN`00MP1kMP2:RP0113N:N`00NUek07YkGGH0NP1fN`1f07]M07Ig07X0
NgX0Ned0MWH0NgX0NgX007Ik001f07X0NP1/CO]Wn7dJO7amO7eW26L8N@Q[K6/6
CFYZ=De>1Te=CTd6oTd1=C_m=@Pk=CDk=Em0G`CnG`H4IVATGd1OoV@>IFAUI6Al
3Gd=O7f2PWmnoF45GemoOWmno7lABe];I6=SI6=eHgESHe]<EGinoV7nFOi`10Qi
I6=ToV<110Ck>`Hj>fm[QhM/oXL4N7UhN7WnN0aiN6j0KWQk07If07]joP0YNWL0
GGL0NW/0GG]MN`1kGG]M001fN`1f07X0GGMMN`1fN`1f07X0RXX00?dgRP1jMP1k
07Yg001kMP1MN`1kNWMM07YkGGH0Ned0Neeg07X0Ned0Ned0MeekGGMMNgX0MP1M
CDgkIoUm0GamoG`EI`QWN@P8K4e/:@IY=FU>CBU=CPI=:Oi=03_m=@dk=CDk=CE0
G`AO@41OGoiToUolI0aUI6@=O0elOGf2L7iooF4@2EmnOginOgioOd]?BfASHfCl
H`ieHe]EFGinFEUQG65QG5cnF@@8N@QTHo`423P4>C/kQf^7K?f70GQ_nGPhA6m4
A7/0MP1kNed0NgIf05ekMP1j05d0NP1g07H0MP00NWMj05ekMP1kMP1j001j05ek
MP1kRXX00@D^RW/0NP1MN`1MMW/0NP1f07X007X0MP1k07X0GGMM07]MM`1f07]M
07Mj07/007[n00EjN`1fK4gnI`Q=CF`8I`QX26SmO0EmO7eW26On20EiCF]=JDgn
JOh60bU=1TknC@L6=C/eGd0e>ode0FEToEl1@5ohI2eUI@d=OG`=20QW24e=GemI
:RZ4:XAnOgiI:UV4:Tm;Ff=THf=THf=eHf>8EE]noET>HF5LG65LOUUiN@QSHf@h
o0@0>?hkoHL7JhN7A8MhN7WkN3Q_A6l0GG/0NP1MN`1MNg/0N`1jN`1fN`00MUek
07IkNP00NP1fN`1j05ek001fMW/0NP1j05f:RP0143N:07X0NgH0MP1k07H0NP1k
MP1j07]gMP1f07Ij07]gG@1kGG]f07H0NP1f05ej07H0NWL0MP1W2?iW3De/:FL8
I`QWIgalOGaloFL126On20A[CF/eJ_iY1TdY1Td6:@KmC@Le=C]O15le>ohe16AT
Gem0oUl2I6EUnf@7IF@=3G`=OGgn20e=1UmO:UTZ:XDZOginFOhZ4XE;Fd]SHfAS
HfAeHgESBhU;QHGn:_iI0emO:XCm:PIi26=TH`@4oS/5=CY/QhM/oXL8JdA4Kfm4
KdA4oXLhJf/0NP1kMP1MM`1jMed0MP1f001MN`1fGG]j07]j001jMed0N`1j07Yg
07IMNg/0MP1f001kRXX00@Q8RW/0MP1j07Yk07Ik07Mj07]M07H0GG]k07X007H0
NP1f07/0NP1kG@1fNeeg07IkNP1j07Yk26MW26M=CFaWI`QXI`QmO7amO?iW0@QW
o@P4K4e/JSGnJ@H6CDe>C@HYoDd5=C/e15l4o3D9IFEO@5mO@5mTIOUT06Gn3@9l
OGgn21Q=C@UOFBYIQBZ5OWmn:UV4LAU;BdmTHf=To6<5MD];RHF5oRX4FEUn@43l
:PMi27UTHf@41?`k18N7JhM[oXLRA4A_A6m4A6n7Qf^7N01fN`1j07H0MP1k07Ik
NP1fN`1MMg_n01ef07]MM`1j07H0MP1g07X0NP1M07IkN`1fMP2:RP010TR:G@1k
GGMf05d0NP1fG@1j07]f07]j001j07]f07]M07]f05ekMP1f07X0M`1f07/0MP1f
001W20QXO7aWIde=I`P8I`QWIgaloFL726L8I`PYK4gl=@5=1_Y=0PI=>o`e063m
Goi0o6@BIFATIFAUI6ETIFEXI`QW20R2P_iQ75UIHfASI6=;Be]SI6=TI6=SI6AS
HfASI6=THf=TnV?n:P5I:_j5oUD5FEUSHf@ho@A>>c][K6]/JhN7K8M[K6][Qfa[
QXN7N7P0GG/0MP1j07YkMP1MNed0NgX0MP1j07]fGG]fG@1kMP00NeekMP1j07H0
NgMMNed0NgX0NhZ:004<E8YkG@1k07Yk07Ik07/0MP1f07X0G@1fN`1f07Yk07Ij
07]M001jN`1fNeekNP1f07IkN`1fNP1lOAalO6MWCDd8IfL8I`QW26L8I`QWI`QW
26M/CDgl=@A=:De=1_i=10I=CCDkoCD1Gd3nGoi0nf@9IFETIFAUI6ETI?`80VN2
L?iQ0UUII?iS16A;Cd]ToV<8I6=THf=THf=ToV<9I6=SI7ESMF=eHohZ1UUIQHEE
FeGnF@9THf?m11@i>c]/Jfa[K6^7Qfa[JhM/Jf]/QhKnN3Mk05eg07Ik001j07Ik
07]M001jN`00MP1j001k07H0NWMj07/0NP1kGG/0GG/0GG]M001f08Z:0040>8X0
NgH0NP1f07X0MP1kNP1kGGMf07]j05ekMP00GG/0NP1f07Yg05ek07X007X0NgX0
07Yk07H=OOil66P8CFa=I`QW26L8I`QW26MW26MW26M=:Fcl=Oe=02WkC@8e=C_n
=OiO0D1OoT3lI0aUIFAUI6ETIfMX26P8oX;nHOiO0GiIoV<XI6=THfASHfASHfAS
HfASI6=SI6=THf=THf=eHgESMD^9Bf=SRD]KEE_nF@IlO6=TH`@4n3/5>S/k>V]/
oTCkN3Mf07X0NW/0NWH007/0NP1kMgH0MP1jNgH0MeekMP1j07X0MP1j07H0N`1j
M`1MN`1kMUekNhZ:0049EXYk07X007Ik07]MNed007Ik05ek05d0MW/0NeefN`1f
07IkN`1MMg/0NP1fN`1f001fNP00MWMmOGamOGal26L8CDe/CDeW26L8I`P8I`P8
I`R2PX==COTe0Te=1_de03_m=@A0@5mO@?iT56ETIFAUI6AU3@eX26MW26P8289`
L?iQ1el9GeUII6=ToV<8I6=THf=THf=ToF<9I6=SI6=THf=TMOaS2HQ;RF=eBhUK
FeGnF@48NOiS0@@4oS_n10Hh>3XkQhM[o4CnN3YiN6h0M`1j001fN`1fNP1f07X0
NP1fNed007Yk07X0Ned0MP1fN`1MMeej07H0NP1gG@1MM`1M08Z:0041=XX0NP1k
MW/0MP1f001jMgX0GG/0NWL0NP1gG@1k05ek07]M07H0NP1f001kG@1kGG]j001j
N`3lO@elO6L8Ide/:De/CFL8Io`80VL8Ioj20De=nCGnCOhe03_l=Oi00EmOoF@@
IFAUI6ETI@d=26MX26P8Ih;nO_iQ6WmnOgioHfASHfASHfASHfASHfASHfASI6=S
I?iS06ClH`9eHgGnH`1eoV<8EEEKFET8NF=So@@1>c_m10Hi>cDkJfahoT@0Koah
>Fik07]f07H0NP1f07/0NgH0MP1MN`1kMeek07H0NP1kMgX0N`1MNeeg001kMP00
NP1jM`1f07^:RP0123N:N`1f07X0MW]MNeeg07X0MW/0NP1jMP1j07Ik07Ik07H0
Neek05efNeeg07Ij07/007If07X0O?em3galI`QWCDe/:De/IfL826On209WI`Sn
P_a=oSD0>ode0S/e=O]O40AOI6AUI6EUI6ET3@eXIfMXoVL[28>3PX>2Ogl3HF5o
OWioOV=THfASHfASHfASHfASHfASHfASI6=SI6=THfCmHaETHf=eHgESMF=eFe]E
FGii26=SI6=ToPCl>`I[K6]/CDA_o7PlNGQ^001j07Yk07Ik07Yk07H0NW/0MW/0
NP1j05eg07H0MP1j07H0MW/0NeefNP1kGGH0MW/0NW]M08Z:0046>8X0GG]f001j
07/0Neef001k07H0NP1gMP00NgH0NP1MNgH0MP1f07]M07]MN`1f07YgNP1g07X=
3Oem00gkI`I=K4e=K8:3nX87CFa=1PI=1PKmJOde4VTe=@T915mO15m0IFETIFAT
IFGn3@<8IfL8oVL2Ph>2oX?nOamRHGinOgioI6=THf=THf=THf=THf=THf=THfAS
HfASI?eS0VASI?eS07GmH`j4Q2YLHGP8NFASHfAS10Cl>`HeK6]/JdA_oWPoNGQi
N6h0NP1kMP00NgX0MP00GG]f001MN`1f07H0GG/0NW]f07X0MP1kNP1kG@1k07H0
NgH0N`1f05d0NhZ:0049>8Yk05ekGGL0NP1f07/0MWX0MW/007H0NgX0GG]M07Ik
07Yk07Ik05egNP1k07H0MP1j07X0NP1m3Oem07ckI`A/CDe/COR2oTd4:@I=:Dgn
J@TeJCEY=FTe=@T9oEl915mTI6ETIFATIOh=0VP8J?aW0H:3oWl1HV;n2AYnOgj4
QD];Ff=THfASHfASHfASHfASHfASHfCnHaiTHfASHfASI7ESMF=eHgES:W64HF48
N@QSI6=S10A0oS/;Jfa/Jfa[K4A4N7QioWPlN@1k07H0NP1f07]MNeefN`1jNeek
07H0NW/0MP1j001jN`1j07H007H0MW/0MP1f07]j07X0MW/0GHZ:0049?HX0GG/0
NeegG@1kGG]f07/0NP1gGG]kG@1jM`1f07/0MP00NgX0MP1k07X0NP1jNgH007]f
07]m3Gel3G`=nfL0:Oi=06ciPPPYK0HY1Te>1PKnJOiO0CDeoVT4ADEV143mI0=U
I@d=oVL;J6MXIfN3PX>2Ph83oGlWHV892EmoOWhZQ5]?Bf=SI6=SI6=SI6=SI6=S
I6=SI6=THfASI6=SI?eS07GnH`aeHgDZ:UmO@7T8NFASo@Cn>da[K6^7Qf][A6mh
N7UhN7UhKP1j07]f07X0NP1gG@1k07X0GGL0MP1kMP1MNgH0Neek07H0NP1gGG]M
N`1j07]j07X0MP1jN`1MNhZ:004:?8Yk05ef07]M07H0N`1j05ef07X0MP1MN`00
NP1gNP1f07X0G@1jN`1MM`1f07H0N`1fNP1k05d=O@d=O@gmO?iWoPSnC@1/oTd>
K4e/1RT6C@HYC@I=JSEYoUl0=OeY6TE5I5m0@6ATIFAT3@eX26L8I`QWIh>3PX>2
Poio1P=oHF83OgknEOiS4FASI6=SI6=SI6=SI6=SI6=SI?eS06ClH`1eoV<3MF=S
MOeS3BXZ15l8NGUSH`@4H`A0oS/3K6^7Job70DA_oGPmN@1k07H0N`1fN`1f07Ik
07H0MW/0NP1j07/0MP1kG@1gGG/0MW]j07/0MP1f07H0Ned0MW/005egN`2:RP01
3CN:07]f07MM07]kMP1j07Yf07L0NP1jM`1fMP1MNed0NgX0NgIj001jMP1j07]k
MP1MN`1j07H0OOd=17`=OGalnfL5CDe/:De[o4d?1RU=1Te>CDheJFUO@5mY=OiY
1dE5I@AOI6EXoVL3CPIo0oio10=oOf9QoF86HF9RHF892Oio8E]EHfASI6=THf=T
Hf=THfASI6=THf=THfASI6=SI6=THgGnH`MeHf=eHgESMOj5o5T3HfASHoh41c/k
Jfa[QhM[oXM4A6m4N7UhN81j05ekMP1MN`1f07Yk05ekGG/0Ned0NgH0GG]f07H0
NP00GG/007X0MP1j07YkMP1f07/0MWH0NP1kRXX00@M0RWX0Neek07H0GG/0GGL0
NgX0MP1j07X0Ng]MM`1j05d0MP1k07Ik07]M07H0NP1j05eg07IkN`em3Gd=OGal
OGcnIaP826M=K2U/CFXeJTd6:Dd6CTe>CDi=JSEYoSD02O]O2T1UI6ETIFMWCPI>
nGl2HV9QnV;n2@L30gmEEF=SI?iS36ASHfASHfASI6=THfCmH`ETHfASHfCnH`Ee
HgESHgGnH`5eHoj5oET0I?iS06Cn14lk>fa[Qfb7Qfb7Jc/j>dA4NGP0MgX007Yk
05efN`00GGL0GG/0NP1gNP1kGG/0NeekMP1fMW/0MW/0MW/0Ned007Yk07Ik07Yg
N`1j08Z:004:@8X0MP1k07H0NWL0MP1j07X007]j07L0MP1MM`1f07H0Ng]f07Yg
07H0NeekMP00MgH0NP1kG@0=3Gd=3Gem3G`=oVL<26MW20Q=CFa=JCEY:Oh684e>
CDi=1SEYJCEY=@T9G`UOGd1OI6ETIFAWI`HYCWmnO_@95Gmo0gl3O`=oEEETHf=T
HfASHfASHfClH`YTHf=THfASI6=SI?iS8gESMF=SMF=eHgESHgESFe]EFEUlO6=T
H`@4>c]/Jfa[K6]/Jodk0F]/oT@m07X0Meek07H0N`1kNWH0NWL0MP00NP1j07MM
05eg07X0Ng/0GG]M07X0NP1gMWX007]j07H0G@1j07^:RP011F6:N`1f07YkG@1j
07Ik07H0MWX007IMNg]M07Yk07IkG@00NP00NP1j05eg07YkG@1kNP1f07]M07d=
O@d=O0el3GaWIfP826MW24e=K6YYJDd6:De>CDi=CTeZJFTeJCEO2OiO16EUI6AU
oPd9IfMnOUUnFGh96oh90Q/96_`9o7lC0`=oEE]SI6=SI6=THf=THf=THfCnH`=T
Hf=ToV<9I6=SI6=SI6=TMOeS2gESMF=SMEEKFEUnPOiSoPCd>`5[K?i4?X1g07X0
MP1fNed0MP00NW/0NP1fMW/0MP1f07Mj07X007H0GGL0GGMj07X0NP1k07Ij001f
Ng/0NP1f08Z:0044@HX0GG]f001kMP1jN`1f07]f001jN`1j001j001MN`1kMWH0
MgX0MP1jM`1MN`1f07X007]f07]f3Gd=O@d=O0el3OiW50QWI`QWK8:31Td6JFTe
JTe>CDi=C_he0fTeJ@WnGoiU0VATIOh=oVL0FOenoPT06_X90a/9Ogon0b=nFEUK
BfASI6=SI6=SI6=SI6=THfASI6=SI6=THfASHfASHfClH`YeHgESMF=SMF=KEOiI
0PQSHo/40CP4nC_nQd5hNGQhNWH0GG/0NP1kMP1jMP00NP1f07/0GG]f07]j07X0
MeekNgH0MP1j07L0MP1f05ek07]f07X0MP1f07Z:RP01346:N`1MNeek07/007X0
MW/0NW]M001j07Ig07YfN`1j07/0MUek07Ik07X0MP1jN`00MWH0NP1MN`d=O@em
3@el3GcnI`T8I`QW26b3PTe>o6TH1Te>CDi=JCEY=FTe2@UO@5mUIFAUI6D=J?io
10T9Bd]KoGl00oin6bYIOgl3Ogl30gmnOT];CfATHfASHfASHfASHfCnH`9THfCm
H`iTHfASHfASHfASI7ESHgGnH`aeHf=e:RYIFGalI6=TnPCm>`0eoS_nA45hN7T0
07]MM`1fNed0NgX0Neeg07IkNP1fN`1j05d0MP1f07H0GG]j07H0NP1fN`1j07MM
N`1j001fNg/0MP2:RP011cV:05ek07H0MP1jMP1jN`1f07H0MeekMP1jMP1k07X0
NP0/07L0GG/0NP1jNgH007Ij07Mj07Yg00emoPd4O@d=O0gmI`P8I`QWPh:3C@Kj
J@<eJCEYnel12EonA@IUIFAUI6EXoWl70`UKBdm;O`?nO`5nO_iIoGl50gl9Fd];
oF@:HfASHfASHfASHfCnHa5THfASHfASI6=THf=THf=THfCnH`1eoV<3MF=eHohZ
1eUIPGaSH`@4nS/8K6]/Jf]/JdA_oGPnNWMM07]MNed0Med0NgH0GG]j001MN`1j
05ekGG/0NgX0NgH007X0M`1jN`1f07Ij001j07MMNed0NP1kRXX00@e1RW/0GGIk
07X007Ik05ej07Ik07X007Mj07]M07H0MP1j07X0MW/0MP1g001jN`1j07H0N`00
NP1l3@em3@emO0eloVL426L8I`SnPP86:CGmJ@DeJCEY=CGlG`89G`WnAOiU14E5
FTT3oWl62@UKBgmoEOhZ14];Fd];oRXHOUT92D]?BfASI6=THfASHfASHfASHfAS
I?iS0fASHfCnH`ITHf=THf=ToF<2I6=eoF<1MHGnF@Il76=TH`@4nS];Jfa[K6m_
NGQhNGQi001j07H0N`1kGG/0MP1kG@1M07Ik05ek07Ik07MMN`00NP1kGGL0NP1f
07X0Ng/007Ij07H0M`1j001j08Z:004:@HX0MP1k07H0NgIk05ek07]k07X0MP1j
07]MM`1j07]f07/0NW/0GG]f07YMN`1f001kNP1jMP1j3@em3@em3@el3O]W1`QW
PX:3CDd6o6T5=FTe=FU0oUl32El92O]50dX5BPGnHPMQOWi;FeIEEOhZ14]?Bdm;
oRX7FGl92Dm;Ff?nI29SI6=SI6=SI6=SI6=THfASI6=THf=THfASHfASHfASHfAS
I?eS0WESMOj51EUnO85THoh4o3/9K6]/Jfa[Kg1hNOeh?g]f07]kMP1f07H0MP1j
07Mf07]j07]f07]f001MN`1jGG]f07X0MP1j07X0Med0NWMj001kNP1j07If07^:
RP013T6:NP1fNeekG@1k07Ik07H0GGL0MW]j07X0Neeg07H0Ned0MP1MMg/0Ned0
Meej07If07/007Ik07d=3Gd=3Gel3GcnI`@826MW2?j21di=:@H6CFXeoVT4=CEO
Gd3lGo]54PE91DXBHV9nOUU?Bd]EEH@Z:T_kI0U;Fd];Fd];LfASoV@:Hf=THfAS
HfASHfCnH`1ToV<LI6=SI6=THf=THf=THf=THf=TMF=SMF>4Q5UIN@SnH`9010Cn
>`Hj>f]/Jfa=oVm5N7QiN7P007]M05ekGG]MN`1jN`1j07]M07Ik07]M07]fNP00
MP1g07X0NgX0MW/0Neek07H0NP1fMP1kG@1jM`1f08Z:004=DhX0NgX0N`1kMUd0
NP1MNgH0NeekG@1g07H0N`1MNgX0MP1kMP1j07]MM`1j001j07Mj07IMN`1j3@em
3Gd=3Gem76L8I`QWIfb3K4e/:@H6CBT6=OiY13EY=Em0oEl1@5okA@I:1CHfHV9Q
oWon:P5THoUT66=TCd];Fd];Ce]TI6=THf=THfASHfASHfCnH`1ToV<AI6=THf=T
Hf=THf=THf=THf=ToV<LMF<ZLBYIF@QiHf=THf@413/k=F]/Jfa[Jfm`N7WnN440
NgH0MUek07X0N`1f001f07H0MP1k07X0NgH0N`1gGG]j07H0MP00NW/0MP1j07H0
Med0N`1j07]f07X0NgZ:RP014CR:N`00MP1j07/0MP1fN`1j05eg07]MNeekG@1f
N`00NW]M07/0GGX007X0MWMj07H0Neek05eg07cm3@am3Gem3@QW26L8Ih:2oDd6
1Te>C@IZ=OiY1SEYGem0GdGnIOi51UY:1@E:=SKnHQioO`<Z:XATI6=THfASI6=T
I6=TI4m;Cd];Ff=TI6=ToF<<I6=SI6=SI6=THfASI?iS6fASHfASHfASHfASHfAS
HfAeHf=eGem/CDelHfCmH`441?hk16a[K4e[oFm5A7QiN7L0NP1k05eg07H0MW]j
07Yk07IkG@1f07H0NP1MMeek001jN`1fNP00NP1fN`1j07X0MgX0MP1f07/007X0
08Z:004CE8X0NeekMP1f07H0NW/0MP00NP1kGG/0MP00GG/0NWH007]f07Yg07]M
NgH0NP1f07X0M`1MN`1j3@em3Gd=3GemO6L8I`QW28:2CDe/1RU>C@I==FWmG`<9
G`UOoTD>IFE5AAQ91@E91CHfH@T9oP<6Q2YTHfATHoiT0f=TI6?nI1I;Fd];Fd]T
HfATHf=THfASHfASHfASI?iS86ASI6=THf=THf=THf=THf=THf=TMF=eHf<4@4e=
K7aSHo`4oS]?K6]/Jf]/Kfm`A4A_A7/0NP1f07Yg05ekMP1j07H007Yk07H0Neek
GG]f05ek07Yf001jN`1gMP1MN`1kMP1j07X0NP1kNgH0NWH0GG^:RP013Cj:NP1k
07X0NW/0MP1MMeegNP1f001fNP1fMg/0NP1kGGL0NP00NP1MM`1j07/0NgX0NP1f
N`1f07d=3Gd=O@gnOA1WI`R3PX86CTdYCTd6CDi=C_heoUl32@UO2Oi56FEUFTYJ
BPE:=SHU9AX92@=o0bXZI6ASI6ASoV@6HfATHfATBoYT3f=THfASI6=THf=THf=T
HfCnHa1THfASI6=SI6=SI6=SI6=SI?eS2e]<Gel4L71lPFASHod4o3]=Jfa[Jfa[
A4A_A7P0NP1g07H0NW/0GG/007IkNP00GGMkG@1g07X0Neek07H0NgIk05ekGG/0
MP1f07]M07H0MP1f05ek07MkGGL0RXX00@TjRP1f05ekMP00NgH0Neek07]f07IM
N`1f07X0NP1f07H0NP1jMP1jM`1j07MM07X0MP1MN`1f07X=3Ggn3Oim0`d826On
PPe>CTd6CDi=CTe>CFTeJOeO0VAVIOa53TX5B@E:1CHf9A<96`Uo0ohZ3V=TI6=T
I6=THfATHfATHo]T0f=TI6?nI0MSI6=SI6=SI?iS0VASI?]S6fASHfASHfASHfAS
HfAeHe]E141OL71lO6=S10Cj>`9[K6_n>`5[K?j7@`1f07X0NW]f001MN`1jMP1k
07If07X0NgH0NP1j07MM07]f07X0GG/0Ned0NgH0NP1kMP1kNP1kGG/0MUd0N`1j
RXX00A8nRW/0MW/0Neef07]j07/0NP1kNP1g07Yg07]M07IkNP1f07L0NP00NP1M
MeekG@1gNeek07Ik001l3@em3GdAoPd2BDUWoX8=CTh6CDi=CTe>CDheJCGmG`9U
I6GlA@T5BBi91EXf=U8UoWloFEU?BdmTHfATHfATHfATHfATHfATHfASI6=THfAT
HfASI6ASI6=THf=THfASI6=THfASI6=SI6=SI6ASI6=SI?eS32XZHEb2L721I6=S
10Cn>`0joS/513Dk>cE/oHM4QW/0MP1j001jMgX007Ik07H0MW/0MP1j07]M07H0
MP1kG@1kG@1fN`1f07Mj07]M07H0NgX007Ik05ek07X0GGL0RXX00@PgRP1MN`1j
07H0Med0MP1j07]M001kGG/0NP1f07]M001kMP1j07Mf07Yg07X0M`1j07MM07X0
GG_j4Oh=4fQ920I>1VUY:Di=CTe>CDi=JCEYoEl2I6IUoDD3BEXWF_h_2bDUDe=o
O`=IFD]KBoiT36=TI6=TI6=TI6=TI6?nI01SoV@>HfATHfATHfASI6=SI6=To6<>
I6=THf=THf=THf=THf=ToF?n:P5LHOi`0WaSHoX4DC/k13][K6]/Jf^7JhMkG@1k
MP1kMP1j07H0NP1kNP1j07]kMP1j07]f07]kMUd0Neeg07]M07X0NP1j07]f07]f
001kNP1jMed0NgH0GG^:RP012SN:N`1MM`1fNgX0Neek05ef07MjG@1j07X0MW]M
07MjN`1j07]MN`1f07X0NeekG@1j07Mj07Ik0?XA4`d=F4U9Idi=CVUYC@HYCDi=
JCEYo0WlA@hl?5XWFTUJBBR3;bDCDe?nOaEIFE]TI6=THfATHfATHfATHfATHfAS
oV@0HoiT06?nI05SI?iS1fASHfASI6=ToV<MI6=THf=THf=THf=THf=THf>8OWiL
HEaiNGalI6=So@Cl>di/K6]/Jfm`N7P007Mj07]M07/007IkNP1f07]M07X007X0
MW/0NP1M07/0MP1j07H0MgX0MP1f07H0NW/0NP1f001f07X0MP1jM`1MRXX00@Pg
RP1f07X0NP00Neeg07H0N`1j07]f07H007X0M`1j001j07]MN`1jNgH0MP1j07]f
001j001k07[j4Oh=0TU92?i=nfT9=FTeJCD92El9Goa533`l@EY:5dX^PbDUDe?n
FA@ZFD]KBf=TI6=TI6=TI6=TI6=TI6?nI0ISI6=THfASoV@0HoiT16=THf=ToV<H
I6=THfASI6=SI6=SI6=SI6=SI6=TR4]nFOiQ1GT8PGaSHod4o3]?K6][Qf]_L7Qi
NGH0NP1j07]f07IMNed0M`1j07]f07]f05ek07H007]f05ekMP1kMP1j001kMP1k
NgH007X0MWH0NW]f001kMP1j07^:RP013cV:N`1f07X0MW/0NP1fNed0MP1f07]k
07If07X0MW]MM`1f07X0NP00NW/0MP1f07YgNP1fMP1f00dAo0d8;@E:BDUWK4e=
o6T6=FTeJCEYGod90DE5oS`1BPGn;a4f=ThV4bDRDeUI:UTZFd]?I6?nI0mSI6AS
I6ASI6ASI6ASI6ASoV@5HfATHfAToV<5I6=THf=ToV<2I6=ToV<LI6=SI6=SI6=S
I6=SI6=<EEUIHGT827`LHfAS10Ck>e1[K6a[K4A_KgUhM`1j07H0NP1kGG/007Ik
GG/0MP1kG@1jN`1MMW]j05ekGGL0NP1MNgX0MWX0Ned0GG]MM`1fN`1f07/0MWX0
N`00MP2:RP0133V:07X0NWL0NP1f001j07MMNgX0Med0NP1kMP1kNP1f05ekMP1f
07]f05d0NP1kMP1j001j07]j014Ao0d19`GnB@<8CBU/nfT1=FWnG`892EomAOhl
0@E:oRl>=SHD9E8UDe=I:U];Fd]?o6@?HfATHfATHfATHfATHfATHoiT56=TI6=T
I6=THfASI6=SI6=SI6=SI?iS76ASI6=THfASHfASHfASFeEIFHAQG89_O85SH`@4
o3]BK6]/Jf]/Kfm`NGQiG@1k07]f07]MM`1f07X0N`1MNgH0NgH007H0Ned0MgH0
M`1j07Yg07H0N`1kG@1kMP00NP1j07]kMP1MN`1j07]fGG^:RP013D2:N`1f07X0
NeekGGH0NP1g07YkG@1fNed0Ned007IkNP00NP1kG@1kN`1fN`1MN`00MWIg05d0
MQ4A3@eH3@dLJ?eW1h9`P`I=1Td6oVWmG`492Oe5oS`@FP<U9A<U9B9CDb5N0bYI
BdonB`=THfASoV@0HoeT46=TI6=TI6=TI6=THfATHfASoVCmH`ETHf=THfCnH`9T
HfCnHaITHfASI6=SI6=SI6=SQBYLHH9_O7aSHo`4EC/kJf]/Jfa[Jg1hN7UiN`1f
07H0Ned0Neek07X0MP1fN`1j05ekMW/0MP1kGG/0NP1f07H0NP1kG@1f07Ik07]M
M`1f07X0GG]MN`1f07H0N`1MRXX00A8hRUd0NgX0MP00NP1gMP1j07H0G@1kMP00
NP1gGG]j001kMP1f07]gG@1MN`1jM`1fGG]k07H0Ng/0n`dFO6L8I`QWK2A/CDi=
CTeYJCEO@5l92EonA@1JoRlD0b`U4bDU4bEC8E@3<BYICd]?Be]SoV@2HfASoV@?
HfATHfATHfATHfATHfATHoeT56=THfASI6=SI6=THf=THf=THfASI?iS4fASHfAS
I6=SMHEE@41OKh9lPFASo@AF>c/eK6a[K6]_L6mhNGUhG@1kNP1j07]gG@1MMgH0
Neek07X0NWL0N`1MNgH0N`1j07]j07Mj07H0MeekGG/0MUd0NeekMP1MN`1g07Yg
07X0NP1kRXX00A@kRW]M07Ik07Ig07X0N`1j07X0MgX0Neeg07X0Med0MW/0NW/0
MP1j07Ik07X0NP1k05d0NW]M05ek3@dOoPdJO6MW26L896`TCTe>CDheJ@UO2EmV
I6I5ADY9oRon0`4UD_hU1R5D8C5nFd_lI@5THoiT3f=TI6=TI6=TI6=TI6=TI6?n
I0ESI6ASI6?nI0ASI6ASI?iS16ASI6=ToV<2I6=ToV<CI6=SI6=THfAEED10L89_
O6ASI3Sn15Hk>f][K6]/K6m_L7QiN7L0MP1kG@1f07X0MW/0NP1MM`1j07L0NP1M
MW/0NP1MM`1f07]j001jNed0M`1f07Ik07H0NP1k07H0NP1f07Yk07H0MP2:RP01
4cZ:07IMN`1kNP1j07X0MP1jM`1j07H0Neef07]MM`1j07H005ekMP00NP1f07H0
NeegMP1j001jM`0=F?d=6W`826MW28>2CTe=CTd6JCD92El9I6ITADD51Oh_4@<3
<C430b/a<@=oOWm;BfETIOeT4F=THfATHfATHfATHfATHfATHoiT1F=TI6=THoiT
36=TI6=THfASI6=THfCnH`1ToV<DI6=THfASHfASHeESHbXZCFa=O6=SoPCm>eA/
K6][CF]=K6m`N7T0NP1gG@1kGGMM07/0NP1jM`1f07H0NP1f07/0MP1MNeef07YM
07]f001kMP1j07Yk05ekMP1f05ekMP1MNgH005ekMP1kRXX00@lhRW/0N`1f07]M
07Ik07X007X007IkG@1k07H0NP1fNeekGGMk07YkMP1fN`1j05ek07Yk07Ik07X0
o0dKFVP8I`QWIh:2CDi>CDhY=CEO2@UOIVAUADEN<OiC037n0`8a0c7n0`QnOgi?
BfEUI6GmI0]SI6ASI6=TI6=TI6?nI0ISI6ASI6ASoF@6HfASI6ASI?iS06ClH`AT
HfASI?iS06CnHa9THf=THf=TFBXZK4elPFASH`@4o3]CJfa/Jde[Kf]hNH1k07X0
MW/0NP1gNP1f07H0NW/0MW]M07IkG@1fN`1g07X0MP1kG@1jNed0NP1gNP00MW/0
Ng/0MW/0NP1gGG]MMg/0NP1MRXX00AE<RP1f07H0MP1kGG/0NP1gMP1fNgX0NgH0
NW]f05ek07H007X0MP00NP1k07H0NWL0MP1M07X0MP1k3@em3@e:26MW26L8PV`6
CTe=1TgnJOiO1T4l1DYJ0`?nDoh30S43QOiE48E;Cd]UIFATIFAUHfASI6ASoV@C
HfATHfATHfASI6ASI6ASI6ASI6?nI0iSI6ASI6=THfASI6=THfCnHaITHfASI6=T
Hf=THf=THfASHeUINGUSHoh403Sn>eA[K3/e>c^7JhMiN7/0MP1kNP1f001j001j
07]f001MNed0NgX007Yk07H0NP1kNgH0NgH007X0MgIM07]MN`1jG@1MN`1MMgH0
NP00NP1f05ekRXX00ADhRW/0MW]j07X0Med0MW]MN`1j001f07]M001jN`1MN`1f
N`1j07Ij07IkGG/007X0NWL0MW/0NP1MoPd@O@d5BTUWIdU9Ih:2CDi>C@KnJ@5O
Gohl2DX5BB`3De=EQEGn:RaEQEF5EE];CfETIFETIFATHfATHfATHfATHfATHfAT
HfATHfASI6ASI6ASI6?nI0ASI6ASI?iS0VASI?iS16ASI6=ToV<9I6=THf=THf=T
HoiI10P8I6=So@@4>`@k>cGm>di[Qg1_P@1j07H0NeekGGH0NWL0MP1jNeek07]M
07Ik001j07Ij07X007X0GG]fN`1f07/0MP1f07X0MgX0GGL0GG]k07Ik07X0MW/0
RXX00@hjRP1j07X0MP1kGGL0NP1k07H0NgH0Ned0NgH007Yg07]j07H0NgH007X0
N`1fGG]f001kNP1f07]g3@glB@aW26Q9B@R2PX==CDi=oVTOG`8HOgioOWmnOXEE
QEF5FETZQEF5:RY;Fd]TIFETI6GmI0]SI6ASI6=TI6=TI6?nI01SoF@3HfATHoiT
1V=THfATHfCnH`1ToV<4I6=THfCmHa9THf=THf=THf=THhEIFGiiI6=SnPAE>c]/
Jc/kNGQiN7P0NP1kNgH0MP00NgH0NeekGG/0M`1f07IMN`1fMP1jN`1f07]f07H0
N`1M07]j07X0NW]f001j05ek07Yk05d0NP1f05ek07^:RP0153^:N`1f07X0MP1j
07H0NP1fNed0NW/0MW/0NP1f07X0G@1fNed0NgIk07X0MW/007Ij05d0MW/0G@19
BFSmBA9WI`Q9J6N2PV`61VYYGelH<aPcoGh2OgiooUD4QEEEOWkn:PAI:Tm;I_iT
2FEUI6ESI6=TI6?nI0ISI6ASI6ASoV@8HfASI6ASI6ASoV@:HfATHfASI6=THfCn
H`1ToV<CI6=THf=THf=THf=THf>4:UUIPGcnHoh40cPk>fcnJe9/JfahNGQiPGX0
MP1MN`1jMgX0Ned0M`1j07X0MW/0N`1MM`1j001j07]M07Yk07H0MW]M001fN`00
GG]fN`1fN`1j05egNed0MW/0MP1f08Z:004C>8X0MP1j07]kMP00NgX007X0NgH0
05ek07H0NW]f07Yg07X0NgH0N`1f05ek07Yf07]f07]k07H0Noa91fQ9C@I=1RU=
oThC=FXeJEmP0QP267mn0c43De=I:UWn:PmnORYIFBYIBd]TI6ETIFAUo6@>HfAT
HfATHfATHfATHfASoV@@HfATHfATHfASI6ASI6ASHfCnHaUTHfASI6=THfASHfAS
HfASHfASI6<Z:WioO?iS0@@4oS]HJfa[Jfa/Jg1hNGQiNGX007]MN`1f05d0Ned0
Neef07H007X0MP1jM`1j07]f07H0NgH007]f07X0NgMMNed0NWL0N`1MN`1f07H0
NP1f07/0NW]f07^:RP015g>:N`1fM`1M07YkG@1fMgX0NP1kMWX0GGL0MP00NP00
NWL0NP1j05efN`1gG@1k07H0NP1f07IkNP1XP`I=1RT61Te>C@I/=`I>JSEYJEmO
ADEV60<a0`<aDe<ZFBYI:WinFETZ<@<a0c5;Fd]UI6ETIF=THoiT3F=THfATHfAT
HfATHfASoV@DHfATHfATHfATHfATHfASHfASI6=ToV<0I?iSLFASHfASHfASHfAS
I6=nOgilHf=THf@k>f]/Jfa/JhM[KgUh27P007If07/0MW]k07H0MW/0NP1fNgX0
Neek001jNP1f07]j07H0NWMM07]j07H0NP00MWH0NP1MM`1j07X0MgX0NP1fN`00
GG]j08Z:004DBXX0GG/0NW]f001kMP1j07H0MP1k07H0NeekNgH0NWH007X0MP1f
07/0MP1kMP1jNgH0GG]f07X008>3:@HY1Te=CTe>CDd6CTi=1_he0FUYoTD9GP<a
0`=CDbYI9OaC0@<aoP<?<@<aBdmTIFAKBfETIFETHoiT0f=TI6?nI01SoV@7HfAS
I6=TI6?nI0USI6ASI6=SI6=ToF=gI6=SI6=THf=THf=THfASHfASHgmnOWaSI6=T
Hc/kK6]/Jf^7JgQiN7UhP7]j07]f05ekG@1MNeek07H0MP1k07H0N`1fMP00M`1j
MP00Neek07H0NP1f07X0MgX0MgX0MP1jMP1f07]j07H007]M07Yf001kRXX00A8m
RW/0GG/0GG]MM`1j001jM`1j05egNed0NP00NP1gNeef07IkN`1f07YkG@1kG@00
NW/007Yk07IfPh==1TglP`86:@Kn=Q=>CVUY=FUYADD3ADEJBPD3<BDU4oaC0c43
0c7m0`dZBe];Cd]KI6ETI6ETHoiT0f=TI6?nI01SoV@0HoiT0f=TI6?nI3=SI6AS
I6=SI6=THfASHfASHfASI6=SI6=SI6=THd]EEGioOGaT15l4>c][K6]/K6m_NGUh
oWU=001gNP1kGG/0Neef07MMN`1jN`1MMeej07Ik07YMN`00MWMj07/0MW]j07L0
NP1f07]j001jN`00NW/0MP00NW/0MP1fN`1kGG/0RXX00A8hRP1jM`1MN`1f07X0
Meek05ekMP1j07]f07]f07X0GGL0NgX005ekNP00MP1j07MMN`1fMP00NP1goTWj
P`9=1Tgn=PI>CVTe=FUYoUl@ADE:5eX30bDC9E<f=P<a4bGnD`d30c5I:T]KBdm;
IFATIOiT7f=THfATHfATHfATHfASI6ASI6ASI6ASI6=TI6=THf=ToV<>I6=SI6=T
Hf=THf=THf=ToV=XCd]EEGm=I4041404>c]/Jfa[Jg1_NGQiNGP0NgIM07X0Med0
GGL0Neek05ek05ekGGL0GG/0MP1kGGMj07X0Med0NP00MP1j07X0MUd007X007MM
N`1f07Yf001jNgH0NP1MM`1jRXX00A8kRW/0NP1gGG/0MP1f07X0MW/0NP00MP1j
05ek07Mj07X0G@1fNP00MWX0Ned0MW/007IkNeeg07X0NdU9nX<2CTe9oX<CCTh6
CDiYJEm0GdE51DT5BPD5BAOn=P<30bDCoU<a<@<ZOWmnOT];IFAUIFAUHfATHfAT
HfATHfASI6ASI6ASI6=TI6=TI6=TI6=THfASHfCnHb9THf=THfASHfASHfASHfAS
I8@ZOWi=CF@41404>c]/Jfa[K?e_DWUhNH1k07/0MP1j07Mj07X0G@1MMg/0MW/0
Neek05ek07H0Ned0MP1j07Ik07Ik07Ik07Mj07MjN`1fNP1gGG/0NP1gNP00NP1f
07X0NP00RXX00A=2RUegG@1j05d0NgX0NWL0N`1j07If07X0NWMM07X0MP1f07]k
07Mf07]f07MMN`1MNgX007X0MP1j04UW<0E9BFQWJ?i9oX<7CTi=CTeYJD3mG`i5
A@451DX5FSHf<@<a;bonH_iC3GmnOginFfAUFfAUI6ASoV@3HfATHoiT14]?Be]S
oV@0HoiT0f=TI6?nI0USI6ASI6=THf=ToF<CI6=SI6=SI6=TQBZ4OWh826@4143n
15/k>f]/Jf^7L6miNGQj07H0GG/0NP1j07H007]f07]M07X0NP1k07H0GG]MMed0
NgMj05ekG@1j07]j07MM07]j001f07/0NP00NWL0NP1gMP1j07]gMP1jMhZ:004B
@XX0NP1f07X0MP1kG@1j05eg07Yg07X0M`1j07H0NWL0NW/0GGX0NP00NP1j05eg
N`00MWX0MW]k07I:FPE:BDUWI`SnB@Z3Pc1>=dd61VTe2OaO3DDHBDX5BSaJBPE9
1Bl_oV;nD`inOgha0ginFd]UI6ETI6?nI3MSI6ASI4]KBdm;I6ASI6ASI6=TI6=T
I6=TI6=THf=THfASHfASI6=SI6=SI6=SI6=IFEl;20QTHo`4G3Pk13][K3/kQgQi
N0P0MW]MM`1f07H0NW]MMP1gNP1j001f07H0MW]k05ek07X0G@1k07H0MgX0NP00
MUek05d0MWIkG@1f07X007X0Neek07Ik05d0N`1MNhZ:004E?8Yf001kMP1kNgH0
NgH0MP1j07MM07]f07X0MgX0N`1j001jM`1j07MMNgH0MP1j07YgNP00N`1M07]J
9`GnBB=W26Q9J4V3<6a>1PHYCCEYG`T9Gem5AFA6AD52?4X5B@E:;bonHT/a0c5C
D`<3<GioBd]TI6ESI6ASI6=TI6=TI4m;Fd];HfATHfATHfATHfATHfATHfATHfAS
I6=SI6=THf=THf=THf=THf=IG`UL20QSoPCn>oh4FC/k=C/kJhM[Kfl0NP1k07Yf
07]j001g07]j07H0MeekGG]MNed0NWL0MP1MNgH0NWL0NP1f07]MNg/0GG]f07/0
NeekNP1fNP1f07H0GG/0GG]f07Yg08Z:004B?hYkNP1j07H0GG]M07]j07H0NP1k
NP1kG@1f07]M07IkGG/0NP00MW/007IkN`00MP1j07If07IkG@0GBEX=3D[kB@j3
Pc1>CPH6:CDe2FUO2EonI@i5ACa1?0DWFTX51DXD;bon0`=C0c43oS4I:UUUIFAT
HfATHfATHe];Be];Bdm;IF=TI6?nI0YSI6ASI6ASI6ASI?iS66ASHfASI6=SI6=S
I6=SI6<ZHF48IfAS10Cm>`04oC]GQf]/Jfb7JgMj07X0MP1kNP00;7]MN`00MW]M
N`1MM`1k07]M07X0NP1g07X007Yk07Ik05ek05egN`1jN`1j07L007X0MgYkGG/0
MP1fN`1j001j08Z:004??hYM07Mf07]MN`1kMed0NgX0NgH007X0NgH0NP1gGG/0
MP1f001jNeegGG]M07IMNgH007/0NW]M07/=F1h=3@GlB@1XoX<;CThYC@HeJVUY
2El9oVD2ADDlo4D:1@E:1H<_;`<XHV;nD`Da<BYI:VGmI15;BfATHfATBdm;BdlZ
I6=TI6?nI0YSI6ASI6ASI6ASI?iS6FASI6=SI6=SI6=TI6=SI6<ZQGio20QTH`@4
nc/4Jfb7JfcnKe9[Med0M`1f07]M07MMN`1j07H0NP1k07Yg07X0NP1kMP1f05ek
GGMfMP00GG/0NWH0MP1j07H007H0MP1jMWL0NP00M`1MNeek07X0NWH0NXZ:004?
>HYk07X0MP1gG@1MN`1f07]M07X0MP1f07]j07X0Meek07IkGGH007X0N`1kGG/0
GG]MMP1j001gNP3n3@Em1DX5BFSnB@63Poi>2@I>CFXeJFT92Eon2OU52TT5FTX5
BUX_;f9RoU<A<@>5EHETIFESI5];I6=TI6=ooP<1:R[nI19SI6ASI6ASI6ASI6AS
I6ASI6=ToV<MI6=SI6=SI6=SI6=SI5EEOWl826AT10@kK6][K6]/o6mDNGV007]M
07YkG@1gNP1k07X0MW]j07X0MP1j001f07H0NgX0Mg/0NP1kNeefN`1j07/0NP00
MW]j07X0NP00NP1f07Ij07Ik07MM07Ik07/008Z:004<>HX0MP00NP1j07Yg05ek
MP1kMP1gNP1kNP00MP1kGG/0GG]MM`1gNP1f07H0MP00NP1g07YgMP1j07_k3@0N
o4T;Ph=>=aDYC@HeJCDeoUl22@UOn4DO64T5BPE9;blf52DU4`<a0hEEIFAT:RYK
EE/ZFBYoO`?mOgm;Ff=TI6=THfATHfATHfATHfASHfASI6=THf=THf=THf=THf=T
HeEnFEUlO6<413DkJf]/K6^2N7UhNGR0201j07H0NP1g07X0MP1f001j001f07H0
NP1gGG]MNgIM001j07X0NP1M07/0NP1f07]f07YkG@1MNgH0MeekMP1jNa00GG/0
NP1kGG/0MP1kMXZ:004:>HYk07Ij07H0MP1j07L0NP1MN`1j07H0Neeg07H0NP1f
N`1kGG/0GGL0NW]MNeeg07H0NP00NW/0MP3l3@em3DU9J4U9Ph=>CTPeJ_eY1TM7
G`UO2@WlGoa550E:1EX5BPDf=RDC9@<3EDlZ:UTZ:_iE0d]I:P?jO`A?BfATHoiT
8F=TI6=TI6=TI6=TI6=THfASI6=SI6=SI6=SI4];Fe]nPFCm168e=C]/K4e[L6mi
N0Qh2820N`1jMgX0NP1f07YkGG]MM`1fN`1fNg/0NP1f07L0N`1fNP1f07L0MW]M
07H0MW/0GG]M001gNP00NW]M001fN`1MMg/0NP1f07L0NP1f07^:RP013cV:G@1j
M`1k07]j07H0NP1jM`1f001kMP1j07YkMP1MN`1j07X0GG/0NW/0MP00NgH0NWL0
NP00NP1knPdDBTUJBDT6CE5A4U4BDE4BDDM72DM7o5l3@5m5AOiU64E5FQM:FRLf
=Q<U9GmnFd]IFBXZQ5EoFGkmO`83Ogon0`mEEFASI6ASI6ASI6ASI6ASoV@VHfAT
Hf=THf=THf=TBd^5C4mKFeEII6<413Dk=C]/Jfa[Jg08N7UhoH1B07X0G@1g05ek
GG/0MP00NP1j07]kG@1f05ek07X0NP1jM`1k07X0NP1gGG/0Ned0NP1gMP1j07]f
05eg07Yk07H0NP1j07H0Neef05ekMP2:RP015D>:Ned0NP1fNP00MW]M07H0NW/0
MWX0Ned0MP1k07Ik07X0MP1fN`1j05eg07YgG@1j07X0MgX007H03Ce14A4=3@D5
B_i920I=DA0BDC@B5Oi>oTL1Gd3nG`49Goi556EUADE:FQM9FTXG9A<UOWm;BbYI
:_eo0ginOgon2@Io0gl30eEKoV@;HfASI6ASI6ASI6ASoV@1HfCnH`YTHf=THd]?
CeEoFOhZJEUSI0@e10@k>f]/Jde[27QiN821PH1lNP1g07X0MW/0N`1j07Ig07Ik
G@00MW/0MW/0MP1f07H0NP1fNP1f07X0Meeg07]f07H0NWL0NP1k07H0NP00NP1f
07H0MW]M07/0MW/0NP2:RP0134::07]f07Mj001jN`1f07]f05d0NP1kGGL0NgH0
GG/0MP00N`1j07H007X0MP1j07X0NP1kNP1fMP1k4D4=4A4=3DX5oTT4I`I=DA;n
CP46=oi>oTOmGoh9oTD1IFGlAA85BPEJ9E8UOgj5EC43<@=oGS5Cn`T9Ogl3OglZ
:V=THoiT0f=TI6?nI0iSI6ASI6ASI6=THf=THfCmO`]I:RYo20QTHd04>CGn>`Q[
K6`eN7QiPH7nP5=j07H0NP1kNP1f07H007]MNgX0NgH0NP1kNP1MNeekNeeg07Ij
001kNP00NP1j07X0MP1jN`1f07H0GG]j07Mf07]j07]MN`1kMP1jN`1j07^:RP01
13Z:N`1kG@1kGGL0NP1kNP1k07IfM`1M07X0NP1fN`1j07IfN`1kGG]fN`1fN`00
MW/007H007Ik07X03OdA10dA3@dAoX<3=SHBDOU>oTOmG`89GdGlGoa52UY91DXD
=SH<DeF5o@<3<B5CDoh911]RHPT9oWl1:R[nI0M;BfATHfATHoiT3f=TI6=TI6=T
HfASBe]SOWkmG`UoOV@41404=CDkoSD2K6TeoPSlP5@007IkNP1j001fNeekGGH0
N`00NP1j001f001fN`1j05ek07X007If07]fN`1f07H0NWL0N`1j07IkGGL002`0
NP1f07H0Meek07X005eg07X0RXX00@/iRP1j07]f07H0NP1gG@1f07X0N`1j07X0
MP1kNP1f07X0N`1jGGL0Ned0NP1MNgX0NgIkGG]j07]j0?XAoPgnP`Hf=U5ACSLE
o4knA`=0GemYoTGnG`50Goe516DH5TX5oSH4De=EE@?n9@ACDeARHOiR165R2@T3
oGl0:_i;46ASI6ASI6ASI6=TI6=TI6=ToV<8Be]nOWmn2Em=oPPAI0@4>cDk=CDk
=C/e=GP8NH1loX1E7@1kNP00MP1f07X0Neeg07X0NWH0MP1gNP1jMW/0NP1jM`1M
NgH0Ng]f07/0NP1kNP1k07X0MP1kNP00NP1j05ek07]f07]MN`1j07MMNeef07^:
RP013cV:N`1j07X0Ned0MP1jN`1fNed0NP1fMg/0NP00MW/0MWX0NP00NP1MMgX0
MW/007H0N`1f001f001fnQ42F0dOoX<6=TU=CDh6=oe>1`IY=FTe=FTeoTD6Gd1O
G`U5AOiU0dD15e[n=PYnOe=C9A<U4bEC0oeR2V5RHV5R2@T3ObXZoV@0HoeT:V=T
I6=TI6=THfASHfAKBeUIOgmnHGi=1PQTI6=013E/K3DeK6`e>hM`NH7mO5J1Med0
Neek07IkNP1j07/0MP1f07]j07H0MgX0N`1f07L0NP1g07X0G@1kG@1f07H0Ned0
MW]MNed007Yg07Ik07H0MP1kNP00GGX0MP1g07X0MP2:RP011cV:GGMM07]f07H0
NgX0G@1j001jMgH0NP1f001kNP1MN`1f07]f07H0NP00NP1gGG]f05ekGG]MMeek
nQ7n3Oa90@H6nTh0COiY13EY=CE5o5l32Em5AOeU6aPl?4YJOWmnDb8U9B/3<@<a
HV46CTioOf9Q2@WmO`eSI6=TI6=THfATHfATHoiT46=THfA;C`<92F5QOWm=26AT
oT1XGcE[CFa[CF`kJfaiPH4LPAb1G@00NP1gG@1kG@1g07H0MP1kNgH007X0NgX0
MP1MNgH0NP1f07X0NW]f07H0NP1kMP1f07X0Ned0MgH0NP1j07H0NW]f001jMgH0
07]j07X0MP1kRXX00@hiRP1j07H0NgX0MP1gNP1fMgX007X0M`1jNeef001fN`1f
N`1MN`1fM`1fN`1j07H0NP1g07X0Neek0?XAoPd1BFSnB@5=1_j33DhE=de>JFTe
JCEYJEm0oUon2@55AOiO3AQV?3`5BWmn<@=N=SHaoP<2;`EWoTh50gmQ0f9RoWom
I0]SI6ASI4]TI6=TI6?nI0USI6AEE@<92F5=o@P2I6=OoT1WK4e[CFa==Fl8N@R0
785lPG`0NW/0NP1gNP1kGG/0NP1jN`1MNeeg07X007Ik07X0Ned0MW]M07L0GG]M
Ned0MP1jN`1f07X0Neek07H0MW/0NW/007Yk05ek07Ij001j07IkNP2:RP013SV:
MP1jM`1j07H0Neeg07X007]f07X0MP1M07]MNgH0GG/0NWL0NW/0NgX0MP1j07H0
NP1f07H0N`1jnQ423Al=oTT0J?i9oX<4=di>1TglJ@4e=OaO03GnGcH9@5m0AAPl
AC`l@Bl30c4f54iN1Bl_FVM>PVMW20=oHF9o2GinI6=TI6<ZFd]THfAKHfATHfAS
oV@7FeEoHTi=1PSnI6eSI40k>de[CFa=K6]_8gQhP7b1O7b007Yk07H007X0MP1k
07H0NP00GG/0MP1f05ekNP1f07H007Yk07H0Neek07H0M`1kNP1M07]j07H0MP1f
07Yk07H007Yf001jM`1MN`1kMP1kNP00NhZ:0049>hYk001j001fNgX0N`1fNeek
07X0NgX0MW/0GGL0Neek07X0GGL0GGX007Ik07]j07IkNP1fNed0MP113O`AoPgl
B@5XBOj30PI>DOUY0Tde=OiYoUl02OeO007nAOhl3Rl_0eY9;PE9;bm:5fN2PoiW
3gmo0gl92GiII6ASI2XZBdon:Pe;I6=TI6=THfAEEDaoG?h85WeTH`@4Gd0e>fa=
Jde[Kg1hN0QlO81loX1G07/0MW]MMP1f07H0MW/0MWIk05eg07Yk07H007Yk07Yk
001j07]M07H0MP1jNed0MgX0MP1k07X0NgX0MP1j07Ik001jNed0NP1f07H0NP1M
07Ik08Z:004=>XX0NWH0NWMj001kG@1j07/0MP1gG@00N`1jM`1jGGL0NP1kMP1M
N`1kMP1kMP00GG/007]j07L0NP0ona7n3@99BFSmBA63Pc1>CQ9YJCEZC@I=1PIY
=FWn=OaO0D1OoDDX?4E5?3`5BBi9FS`lBPEJIh>2PfMWPh=oO`UI:UUTHfAT:P=o
ObZ4EEGnI01SoV@GHeEEH5l620QmOF=T10A0>cE/=F]=K6^2oWT3N820O?f0EgX0
NP1j07X0NgX0NgX0MP1kG@1j07Yk07X0Meek07H007If07H0MP1kMP1jN`1f07X0
07]f05ekMP00NP1g07Yk07Ij001kMP1f07YkMP1gMP1k07Z:RP013c^:N`1kG@1j
07Ik07X007H0MW]MN`1fNP1f07X007]M07Ik07]MN`1f07Yk07X0MW/0MW/007X0
MW]M047l4Oh=0TU:1Oi90FQ9oX<?CTiZJ@I=1Te>:Dd6JSEYJ_eO2@UOGd1OGfEU
ADGn?2A:1D4l2S`l1DTl?5Y:1FN3PWn3Pf9R:UUSI5UI0gmo0gmNFeESoVCnH`MT
EED?CDd=OOeToT0<=@@e=Fa/CFm_NGP8POf0FX5l07]M07H0MWMj001j001jN`1M
MgX0NP00MW]MN`1MMW]j07Yk07Yk07H0Ng/0GG/0MP1fNP1kGGL0NP1fNgX0NP00
MW/0MWX0N`1jN`00NW]MNed0MP2:RP0143^:07X0NgH007X0MP1gNWH0NP1j07H0
N`1kMP1MNgH0GG/0NUeg05ekMP00NP1jMed0NP1jMP1jMP00N`gl4@MH3@e:FUY9
J?i9oX<D=dheJ@I=1RT6CDi==FUY=El9GfTeoTD7GemUIDE5IFGnA@4l??i5oS`3
1DXGF_iWoX<2HF89oWlNFET32Gmo0`=E:VASI6=THfASC7iQPX9mO6@415l4>od4
26a=Jfl827T8POf0Fgb007X0NgH0Ng/007]MNgH0NP1f07/0MP1gGG]MM`1MNgL0
05ek07H005ekNed0NWL0GG]MN`1kGGL0MP1kMP00MP1f07X0NWL0MP1f001kMP00
N`1kMP1kRXX00@djRW/0MP1kMW/0MW]M07/0NP1f07IkNP1f07]MM`1kGGL0NP1k
G@1f07YkG@1f07/0MP1f07/0N`1fNP3k4Oh=0@E:o4T0J?j32dhEJVU=CTd6CDi=
C_iY1EmO=FTeJOi5043nGoi506GnA@5UIOi5oScnA@19oVL8P`QWCTh9O`=Qo@T0
0oio3BXZI6ASI6=TEGioOeb2oVAbHfAS=CDk=C/e16a=Jg1_N@PMP7b177b1O7X0
NP1kG@1MMW/0MeekG@1fNed0MW/0NP00NP1fN`1j07Ig07IkNP1f05ek001j07H0
Meek07X0NP1fN`1fN`1jN`00MP1jN`1fNgX0GG]MNP1f07Yk08Z:00489hX0GG]f
07]M07X0NgIM07]f07]j001MNgH0N`1jGGL0MP1f001MNg[n00mjM`1j07X0NgX0
MUd0NP1knQ443Al=B@GlBAV3PdiQB0I=CTe>CDi=CTeY=CEOGfTe=FUO2OaOoTD5
IFE5AFEUn4D01OiW1@QXCPP6C_iRoPUoHPUo0gn4EFASI6ASEEEoGde=26AT10AT
13Dk=C/e>cE[Jfm`NGT8P85lP7b1P7X0MP1f07]f07/0NP1j07L0NP1gGG/0MP1g
MP00NP1j07]j07X0NP1f07Yg07Yf07H0Neek07H0MP1kNP1kNP1MN`1fGG]j001M
Ned0NgH0MP0807]f001kRXX00@XkRW/0GG]M07Mj07X0N`1f07]M001jMWL0NP1M
M`1kGG/0MW]j07L002ak001j07Ig07H0Neek07H0MP0=na4J3@d51DY9BFQ9BDh6
CDQRCDi=CTe>CDi=1VTeoUl3=FTe=Oh9o5l22EmVoTD1IFGgA@1UoVL83FMWCPIR
PX<3oV822GmooUD=:VASI6AE:PUOCGemI6?n170eK6a=Jf`e>cE=L7P8NGV1O81l
P80007Ik07X0GG]M07H0MP1j07Ik07X0NP1jNeekGGL0NP1f001fN`00MW/007X0
MW/0MW/0NP1fN`1j05d0MP1kGGL0MW/007IkGG/0Ned0NeekGGH0NgX0GHZ:0048
>XYM07X0M`1j07/0MP1f07X0NgMf07/0NP1kMP1j07X0NgX005ekMUek07If07Yg
GGH0Neeg07H0NW/0nQ463EM:BPE9J?i91PI=CQ9ACX?nCPI=CTd6JCEZncD<J@UO
=FUZ@5l93fITAOiUoTGmIOe546ETI@d=26P8Ph:3Pgm>CPIRoGlP:RYTI6<ZQ2X9
O`QmOF=T@0@eK4e[Jfa[K6]=K71=N@P8o81LH`1jN`1f05ekGGL0NP1kMP1jN`1f
07H0MP00NP00GG]f07X0NW/0MUek07Yf05ek07]j07H0GG/0MP1jM`1kMP00NP1j
07MMNeeg07H0NgH0MP1g07MM05ekRXX00@PjRW/0MW]MN`1f07IkNP1fNed0NP1f
07X0NP1kG@1f07X007IgN`1j001fN`1f001k07X0Neek07H0Negj4@@=7`D5B_a9
0Te>COf35Di>=dd6:@IYJCEY=FTeJFU=JFTeJCGnG`<9GemToVGnAOeUoDD>I6D=
3FD=3FQ/J0QWI`HYoF;nO`E;BdmT:UGnO`YXIFAT10@e>fa[K?e[JFa[K4e[JgQh
NH1lP01kG@1kGG/0M`1j07IkG@1k001MMeekN`1jNed0MWL0GGMk07H0GGL0MW/0
N`1fN`1j001j07Ik07X0MP1j07X0MW/0MP1j07/0Neek07H0NW/0NP1j07Ik002:
RP012SZ:07H0G@1M07X0NP00NgX007X0Ng]f07H0Med0NP1g07Yf07X0NP1f07X0
NW/0NW]MNgH0N`1MNgMM0?dAn`d1BE[nB@A=CDi=C_f31Ti>1Ti=1TgnJ@PeJCE>
C@H6=CGnJO]O0fD9GeonA@=U2FEUoDD4IFATIFGn3@5X2?aW9@I>HF892Gm;EE];
EEEoOfP8IF=O140k=De[K6][8g1`K6][CF]`oWUON01j07]gG@1j07Yf001j07]f
07If07X0NP1f05eg07X0NW/0G@1kGG/0NW/0MUd0NP1f07]f07Yk07H0NgX0NgH0
07]M07X0NgIM05ek05ekGG/0MP1MNeekMP1jRXX00@hjRW/0NWIk07IkG@1fN`00
MWH0NP00NW/0MP1fN`1fGG/0NgX0MP1fNgX0N`1MM`1MN`1j05efN`1j07Km4@l=
3Al=3@D5BDUXB@H6CDi=oH<91Te>1RT6CFXdDOe>40HY1VTe=FUYGel9GaP22El9
o5l22FEUoTD0IOeT0fEU3@gjIgm>I`=o2@UoEEE;CeUIOWmUG`AO@0@k=Fa[K6]=
8gUi27T8N7V7Q`00N`1j07X0NP1g07H0NgIk07X0Neek07X0M`1jN`1gGGL0NP1M
MgX0GGL0NP00MW/0MW/0NP1MN`1f07H0MP1kG@1fNWL0MW/0MP1k07Ik07Ik07X0
MW/0MPP0N`1kG@2:RP012CZ:07H0N`1MN`1kMP1j07X0MgH0NWH007X0NW/0GG]k
07H007X0NgX007MM07]MMP00NP1f07/0NP1kModAn`d6BRM9BDe>COi91H>3<8<Y
COi>0cAA4U7lCPA=1Tde=OiY1emY=CD2<V1Zo5l62FAOGdE5IOeT1VETI0eV3@gk
Ioio1F43OeE;O_iIN`QWI@AO@3D4=CE/CBU/PWUi27UiN7Uh07/0Neeg07H0MP00
NeekMP1j07H0MP1k07H0Neef001f07X0MP1f07X0NgH0MP1kMP1j07X0MP1jM`1f
07X0Ng]j07MMN`1kGG/0GG]f05ekG@1j07H007/0NW]f07]M07N:RP011dR:N`1f
07Ik07H0Ng/0MW]j07/007YgNP1f001fN`1MM`0/N`1f07]MMeekG@1kNeeg07Ik
G@1f07X000dA4@d=F0em3EYJP`H6oX?nBO^3oTh4DA8@4SKnCQ0g1Te>JCEYGele
JCEYAUmZ=OeO00WmG`I5AEmOI6AUoFCl3OeWoX<?HWmKBdmnFGmn26QTGcDk>ohe
0de=PX;m26UhNGT0M`1j07X0NP1kNP1fMP1k07X007Ik07X0GGMkG@1gGGYk07X0
NW/0Ned0MP1kNP1f07Yk07Ig07X007X0MW/0MP1M07H0N`1j07X0MW/0Neeg07Mj
07IkGGH0NP00NP1MMgX0RXX00@dmRP1f07Yg05ekMP1M07X007X0NWH0NP00N`1j
Ned0NP1j05d0NgH005ek07]gG@1f05ek07]MNgH007X04A7k3@A:Ph==:Oj30dUX
BH?nBBH`Ph=>CQ9ADCHf=aE>CTe>CCEY=EmOJCEY=EmO=FUO2El92EmOAD3mGoQT
OfH=3FL=IfN3PWl3Oh@Z2@UOOP=TI5mO>cE/=Dde=De/JgT827R0N`1j07X0MP00
NgH007Yk07H0MP1MNgX0MP1jMP1MM`1j07/0GG]f001jGGL0NgH005ekMP00GG/0
GGMfMP1jN`1f07]f07]j07H0MP00NP1kG@1kGGX007X03P1kMP1kMP1MNed0NhZ:
004BBXYjM`1kGG/0GG]MMgX0MW/0NP1k05egNP1f001fM`1fN`1gNP1j07X0GG/0
NP1fNgX0GG/0M`1jNed0MQ4A3Al=3@EJ5h>31PJ3J?iW06SmB@63PoeA3A9A=SH6
=di>1Te>JCEOoSD>JCEYGd1Y=EmO2@UOGdE5o5lDI4EVGfATIFETI6EU3@eW3FMo
O`T9oRX42@T=I6GnG`5/K?i=oSD0Joh8JgT0N`1j07L0NW]MMP1kGGL0NW/0MW/0
Med0NW/007]j07X0Ned0MW/0Neeg07]MNP1kNP1gGG]j07MMN`1MN`1f07X0NW/0
NP1M07IkGG]MM`1f07Ik07/0MWIgGG/0NP1MN`1g07X0GHZ:004>?HX007X0N`1j
M`00NP00N`1f07H0MP1j05ekMeekGG]j07]j07H0MP1kMP1f001j001kMP1kGGH0
07Mj014Ao@d8BPE:Ph=>C@H8oFLDBDUXBCHfDA9R=54BCTd6C@I>CDi=nfT6C@I=
=FTeJOiO3FUYG`UO2El9Gem5H0mOnf@5I@d=I@d=oVL12@WnOPiW3FAUI5leK6]=
Jde=>cGm26ak07H0NP1kGGMM07L0NP1f07H007]j07H0NgH007]fNP1f07IMM`1j
07]MM`1j07/0Med0MP1j001kGG/0NW/0Ng]f05ek07H0MP1gNP1f001kGG]MN`1j
07H0N`1k07X0NgH0MeekMP1kRXX00@dmRW]M07H0MP1j07H0MWH0MW]k07Ik001f
N`1j001k001f001fNg/0MP1kGG]MM`1fN`1j05ek07Ij001j4A7m3@L5BPL61Te>
COeWo4T@CSHBDE4BJFT6CRU=1Te>CDkmJ@Y=1PI=CSEY=FUOGoiY1CD92El92O]O
0D10o6AoIFET3FD=IfMnOUUnIfETIFA013]=K4e/CF/eK0P8N@1MN`1f07H0NP1k
GG/0NP1kNP1fNP00NW]M07]j05eg07X0N`1kGGL0NP1kGGX0MP1f07Yk07H0MW/0
G@1f05ek05ek07H0MP1jNeek07X0MWH0N`1j07H0MW]j07X0MP0<MP1kGG/0G@1f
08Z:004C@HYM07]j07X0NgH0NW/0NW/0G@1j07]fNP1j07]f07YMNeekNP00NW]f
07MM07X0NP1j07]f07]j07]f014A3@eJB_i91Ti=CTe=9e[nB@=XBDUWoPKmJB8e
:Fa=1RU>CDi=JEle=@HYCDi=1TeY=EmO=FTe=@TeJFUO2OaO0T10I?iO5FAUI6AU
I@eWI`d8J6AUI6AO10@e=F_nCGA[=CE/N@1gGG/0NP1kMgX0MP1j07H0Med0GG/0
NWH007]f001jM`1f001kMP1j07H0MP1j001kGG]MN`1f07]j05ekMP1jM`1MNed0
NgH0NW/0N`1f07X0NgH0MP00NeekG@1kG@1j07]f07X0MP1j07^:RP012d6:N`1M
07IkG@1kG@00NP00MW]M07H0N`1f07H0NP00MP1j001fNP00NP1j07Ik001f07H0
Ned007H0Ng/=3EP=1@GkB@85BU[iB@9=:@KnJB46CFa=1Td6CDiYJEmO=FT61RT6
CDiY=CEOGcDeJFTeJCDenUl1@6CnG`5TIOeTOfEXI@d=J0QW@5m0Gc/4=C/6CF][
K71[07]M07/0MP1f05eg07X007YkGG/0MW/0NP1kGGH0NP1f07Yk07Ij07]M07Yk
07X0MW]M07L0NP1fNed007Ik07]M07X0M`1fNP1kG@00MP1fN`1gNP1kGG]MMP1k
07]M07Mj07H0NP0027Ik07Ik08Z:004H>hX0GG]gMP00NgH0NgIf07]j07MMNgMM
07X0NeekMP1kMP1f07X0NeekMP1kN`1fGG]j07MM07Yf07]M0?h=0PE:B_j356Q9
BDX5B@E9BTU9J4U9C@Hf1Th6C_i=42U=CDi=JFUOGdd6CDd6CDi=oVT:@5mYJCDe
JCDeJCGnG`QYJCETGem0GfGnI7mOI5mTI@eTIFETGd1O=C/e=De=Jfa[K`1gMP00
NP1f07X0NW/0NP1jMed0M`1MN`1j07MM07]f07YkMP00NP1kMP1f07/0MP1f05d0
Neef001kG@1kMW/0NP1kMP1f05ek07X0NgMMNeekG@1j07X0Ned0NgH0MP1f07X0
MP1j00YjNgH0GG/0NXZ:004C@8Yk05ek07Ij07]M07/0MP00MP1k07X0MP1f07/0
NWL0NW/0MW/0G@00MW]M07Ik05d0NP1gNP1kG@1gNQ4=BX=/oH?nB@EJ7Pd=B@Gm
B@MXCTdfCRU>1_j21De/CTeZ=OiY40HY1RT6CDi=1VUYGcDeJFT6oVT3=EmO=OiY
1SEO@41O@6CnG`A0G`AUGoiT16EO@5lkoCEiCFa`K`1kN`1kGGL0NW/0MgX007X0
MP1j07H0MW/0MP1f07Mj07/005ekGGL0NP1kGG]M07IkN`1fN`1fN`1fNP1kG@1k
G@1fN`1kGG]j001f07H0NP00MP1gNP1k07H0MgX0NeekGG/007IkN`1g05ek07H0
MP2:RP0145F:05ek07Yk07H0NgH0NgH0NW]f07X0MP1kNgH0MP1MMP00GG/0MP1f
NW/0MP1kG@1kMP1kNP1f07MMN`0=3@D5Ph<`PdT5BTT=3Gd=1DX5BUY9BDe>Po^2
18=/:Dd6oVTB=FY=1Td6CDi=CTdeJCEY=FX6COiY2SDeGfTe=FTkGem0nel6@41O
Gd1T@?eOOcDeCCDeJFa_07Mf001j07]MNgH0GGX007]f07/0NgH0NgX0GG]f07]j
07]M07Yg07H0MP1j07X0M`1j05d0NgH0NP1kNP1f07]gG@1MN`1j07]M07MMNgH0
NP1gGG/0NP1gG@1j07X0NP1j07/0MgYf001j07Yg07YkMP1kRXX00@lnRW/0GGH0
GGL0NP1j05ekG@1k07H0MW]j001jN`1fN`1jMg/0NP1j001MM`1j07Mj07]M001f
Neek05ek3EY:oX<2BDUXoPD07_`=17d5BTU9nH<SPX:3CDi=JFT6C@HY1RT6CBT6
CFXe=FUY1Te=CPI=JCEO@3EYoSD7>cDeGd0e=Eon@?iO2410GcDk=CE=COheN@00
MP1kNP1MN`1k07]MM`1fNP1kG@1f07Yk001fN`1j05d0MP1gNP1j07]j07H0MP1f
Ned0MW]j07]M07H0N`1j05ek07Ig07X0MP1gNP1k07]M07Ik05ef07X0MW/0MP1f
07H0MP1j001jNed0MP1j001j07H0RXX00@dmRP1f07X0N`1j07H0NgH0NgH0MP1k
NP00NWH007Yk07H0NP1f07]gMP1kGG]f07X0MP1kMP1j07/0GG/03C3mPoi90EY:
o0d6O@d=B@E:1Of3oTT0Pof23di>=FY=CTd6:@I=CPI=JSGnJ@Le:@I>CBT6=OiO
2fUYC@Hk=CDkGele=Oe00em0Gc_m=Oi=OG/0N`1kNP1jG@1kMP1f07H007X0NP1g
G@1jNgH007IMN`1j07Yg07X0NP1f001j07H0NgX0Ned0NgH007H0MP1kMP1f07Yf
07]j07H007X0NP1kGGMM07]j07H0NgX007]M07]f07]j07]f07Ij001kMP1j07]f
07X0NhZ:004;@HYk07H0MW]M07Ik05ek05ekGG]M001fN`1j07H0NP1kG@1f07H0
NWMj07/0NP00MW]M07Ik07X0MW/0NTU9Ph<`PoD=017n3@E:1H>3BFSnBOj23Th6
CVXe1Te>CDd6:Dd61_aY8`I=C@H6CDeZ=EmO=FT6CFTe>cEOGc]0@3Dk=Em0=CDk
=C/eCOh0O7IkG@1fN`1kG@1kGG]MNeeg07H0MP1fN`00NW]MN`1MM`1j07X0MgX0
MW]j001fNg/007Ik07Ik07Ik07IkG@1kGG/007Mj001jNeeg07H0NP00NP1f07]M
N`1MMWMj07Mj07]M001j07]MN`1fNP1k07Ik07X0MgX0RXX00@LURP1jM`1j07]M
N`1jM`1jM`1j07]gGG]MM`1jNgH0MP1kMP1jNg[n01Ij07X0MeekGGL0NP1kMP1j
07H0BDV3PoP=05Sn3Oam0@D=nTT@28:2PcI>C@I=1RU=1PI=CTglJ@m=:@I>CBT6
1VUY=CEYJTe=oCD0>odeOc/e@40e>dde>cDe07MMNgX0MP1k07X0NgH0NP1gG@1j
07]j07Yk07Yf001k07]f07H0MP00NP1MN`00MWYk05egGG]M07X0NgX0NgX007]M
M`1f07X0NWH0G@1kGG]f07]fG@1kMP1k07Yg07X007X0NP1kMgH0Neeg05ek07H0
2eek07X0Med0NhZ:004B?hYk07X0Ned0Meek05ek05ek07X0NP00Ned007X0MW]M
07/005d0NW/007H0MWX007]MM`1j07]M07Ik04d6BDWn3@0OoPd07od=oWd73@em
3@eXBFSnBA9XBDV3PcAZC@HY1PI=CBT6CDh6oFTB1Td6CDi=:@IYJCEYC@H6CCE=
JO`e1dde=@A=C@I=oSD2N`1joP1kMW]M07H0MP1j001f07Mj07H0N`00GGL0NW]M
07H0NgX0MW]MM`1fN`1fN`00GGL0NP00MWL0NP00MP1kGGH0Neek07H0MP1kMW/0
GG/0Ned0N`1fN`1j07H0NP00MWH0NP1j07X0MP1kGG/0MW/0MW/0MP1f07X0GHZ:
004;?hX0NP1gGGL0NP1M07X0NgH0MP1j07YgG@1jMP00N`1kMP1jMP1kMP1jMW]M
N`1kMP1kGGH0MP1f07/0NPHYJ4Wk3@1Ho@d04Oh=07gm3@59B_Y92H>3DCD6CH:2
Pf_mC@/6JFU>1Ti=1RU>C@KkJ@T6CBU=1PIY=C/eoTd2=C]YoTeo07/0N`1j001j
MgX0M`1j07]f07If07]j07H0NP1jMP1kMP1M07MMMed0GG/0NeekMP00NP1MNgH0
MP1MNgX0MP1kGG]fG@1kMP1j07H0Ng]f07/0NWL0NP1kMP1kNP1f07H0NP1gNP1k
MP1f07H0GG]gG@1M07]M07X0GG]f07X5N`1kRXX00A11RW/0NP1j07H0NWIk07H0
Neek07H007X0MWL0NWH0NP1kG@1gNP1kG@1k07H0NP1j001jM`1j07Mj07H0C@I9
B@dOn`gm4@d=O@d=OGd=3@D5BTU9J?i978>34Ti=Jh>2PTf2CFa>C@IZ=Di>1Td6
CDi=JUmOoVT71PI=CTe>JCGlCOheOdd0NP1MMed0MWMj05d0MWYk07H0Ng/0NW/0
07Ik001fN`1f07Yk07Ij07X0MW/0NP1k07X0MW/0MP1kNP1fN`00NW]M07L0N`1f
07X007YkG@1MNed0MP1j001f07X0MP1jN`1j07Mj001j07]k07Ik07H0NP1fMgX0
MgX0MW/027]M05ef08Z:004>?HX0MP1f05ekMP1k07H0NUeg07H0NWMf07/0NW/0
NP1kGGL0NP1f07]f07X0MP1f07IMN`1f07]j07H0NTi=o0d3F0dO3O`A0gd=3Ggm
3@A:FTU9J?e91H<6=Ti/COf244e/:De>CCE>CTdYCTe>C@HeoUl81TdYCDi=CTeY
ndeo>cD0N`1kMP1j07X0NP1f07]k05d0NP1j07H005ejN`1fGG/0NW]f001MN`1j
07]k07H0MP1f07]f05ekMP00NW/0MWX007]f07X0GG]f07YgG@1gGG/0MP1kMP1f
GG]f05ekMP00NWL0NP1MNP1gG@1MN`1jNgH007]j001j000;N`1kG@1jM`1kRXX0
0@dnRW/0MW/0M`1j001fN`00Neek07H0NP1fN`00GGL0MP1kGGL0NP1f07]M07Yg
N`1fN`1MN`1f001jN`00CDi=o@gh4@X=3Gd=OGd=3@DN3Oi91fQ9BDd6=ThYnh8B
CFa>CFYYC@H6CDe>CDiYJCEYJOh6Ode>C@I=1Te=1Td6C@1kN`1j07]M07Ik001j
Ned0GGMkG@1f07IkNP1g05ekN`1MN`1MNeegN`1j07H0GG/0MW]MN`1MN`1f07Yk
001MM`1j07H0NP1j07L0NP00NP1f07H0MW]M07/0NP00NWL0GG]MN`1MMP1gN`1f
GG/0MP1j5`00NW]M001fMP1fMP1j07Mj07X0GHZ:0044?XX0NgX0NP1f07IMN`1j
MP1kG@1kGG/0N`1jMP1kGG]f07X0MP1j07MM07]gG@1MN`1jM`1MNgYf001jMP1=
C_d=lQ7m3@Q9J5Y9IdU91TglPQ4828:2CDi=JVUYCV]=CDi=CTgmJ@P6CDdYCTe>
CF[mCGl6NP1gG@1j07H0MP1k07]fMP00NgH0G@1kMgX0NP1f05ekG@1kMP1kMP00
NP1f001kMP1j07X0N`1jM`1MNeek05ekMP1j07Mj07]f07H0NP1gGG]f07]kGG/0
NgH0NP1gMP1j07X0M`1MN`1kNP1f07/0GG]f07]f001gGG]f00ikNP1gG@1j07H0
07^:RP011cj:NP00MW]j07YgN`1j07]M07Mj07/0MP1j07]M07MM07X0NW/0NgX0
MgX007Yk05ek07X0M`00NWMj07]M1DWm3O<Ao0d1FPGmB@IWCTf3PX88oVLIPX86
:FTeJFU/CDe/:Fa=CVUY1Te=CPI=1RWnJGm=C@1j001f07X0MW/0MW]j07H0N`1f
MP1kNP1f07X0M`1fNgX007Ij07]M07]MM`1fNeef07/0MW/0NP1j07X0M`1gGG/0
GG]M07H0Ned0NeekG@1f07H0NgX007H0NP1kG@1fN`1f07X0MW]M07Ik05ek07Ik
GG/0Ned0NW]M019gGG]k001jN`1f07IkGGH0RXX00@U2RP00NgX005ek001f07H0
MP1j07H0GG]f07MM07]MN`1kMP00Med0GG]M07X007Yg07X0MP1jMP1j07MM04X5
3@eH3O<Ao0d2BFMXoVL824i>PVL8I`QWoPP1CPKmJGm=CH9/K4e>CCEZC@HY1RU=
1TeYCTek07L0MUekNP1MN`1kNP00GG]f05ekN`1j001kMP00NP1kG@1gGG/0NP1k
MUd0NeekG@1kMP1MN`1f07H0MP1fNP1j05egMP00Ng]fG@1kGGL0MP1kNW/0NP00
NP1f07X0Meek07X0NWL0NRL0Meek07Yg05ekGGL0NP1kMP00NgH0Ned0MP1j07H0
NgX0M`1kRXX00@e2RWX0MP1fN`1MMW/0MW]j07H0MW]MN`1j07X0MP1j07H0Neej
07Mj05efN`1fN`1MN`1fNed0N`00MP1fN`D=3Al=3O`A00gh4@mH3@emJ0QW20QW
20I=20QWoTT4IfM>PX;nJOf208?nPPD6JSD6C@KmCGm>JG/0GG]MN`1g07/0MP1f
07]MM`1j07Ik05ef07If07YkG@1f07]M07H0MP1f07/0MP1j07Mj07]MM`1j07Ik
07/0N`1f07Ik07X0MP1MN`1f07X0MW]M001j07IkGG]k07H0MP1f07X0MP1j001f
07H0NP1j07H0N`1j07H0NQH0MP1j05ekGG]M07IkG@00NP1f08Z:0049?8X0GG]f
07]f07X0NgX007Yg07X0N`1j07MM07]f05ekMP1f001j07H0MgH0NP1MNed0NP1k
MP1jMP1jN`3k3@4A4Od=n1443@emO@SmI`A/CBU/Ioi92VQ926L8PX9YJDe/nh9o
PddeJDe>PVa/CDh0N`1f001j07]MMP1MNeekMP1f07H0NgX0MP1kMP1kGG/0MP1k
NP1f07]MN`1kMP1MNgH007X0NP1f07H0NW/0GG]f07H0NW/0MP1j07Yg05ekMP1f
07/0NgH0GG/0MP00NP1kNeegNP1kMgX0MgX0NW]f07HQ07H0NP1jG@1kMP1jNgH0
NWL0MP1g07]M07]f07X0NhZ:0048?8Yk001jN`1k001f07]MN`1j07]M07H0MWX0
MP1k07H0NgX0MWH0NWL007Yk07Ik07Mj07X0Ned0NgX007Ki3@=H3@dOn1423Al=
oG`726LYK4d626SmB@AW26N2K?e=0@QWo89oCVXePh:2Ph=j07X0MW]j001f07Yk
07H0N`1j07]j07X007Ik07X0NP1k07IkG@00NW]M07MMNP1kGGL0NWMj07/0MP1j
N`00GGMkGG]MN`00NP1j07Mj07X0M`1jN`1j07H0MW/0GGMkGGH0MP1j001kG@00
Ned0MgX007]j07HPNg/0MP1gNP1k001MN`00NP1kGGX0GGMj07]M07X0RXX00@e4
RP1fMP00MP1jMP1kGGL0NP1f07H0Neek07]kMP1MNgX007X0MgX007]j05d0NP1k
G@1j07]M07]f001j00d=7`d=F0dOoPgg4@4=3Oil2GelI`Q/CVM9IfOmB@X8IdV3
PbU/I`QW2?b20TheCOj2OgMj07H0GG]M07MMNgX005ekMP1f001j07H0Meek07X0
MP1gG@1j001jMgH007]MN`1kGGL0NP1j07H0MP1jN`1fMP1kG@00N`1jMP1f07MM
07X0Neef001j07H0NW/0MP1kG@1kMP1j07]jG@1jMP1f07X0MP1j001k001MN`1j
6P1f05ejM`1fMP1kGGL0MP1k07H0NgH0NhZ:0049?XYk07YkGG]M07Ij001j07Ik
07IkG@1gGG/0GG]MM`00MW/0NP1kGGX007IgNP1f07Mj07MM07H0NeegNg/=F?L=
n14:3GalOG`=1@I>CDkmBGmXIdU9J4U9Ph=W26MW26L8Ih:2CPI=Jh<0NP00MW/0
MP1kG@1f07]MMg/0NW/0MWX0M`1j07H0MW/0NgX0MgX0MW/0NP1fMP1k07H0Neek
07X0M`1jN`00GG/0NP1MMgX0GGL0NW/0MW/0MP1f07]kGGL0MW]k05egNP1MN`1M
Nc@007H0N`1f07]j07H0NWMj07Ij07Ij001f07Ik07H0NW/0NP1MN`1fNgX0MW]M
07Mj08Z:004??8X0MP1MM`1f07/0NgH0GG/0NgX0M`1j05d0NP1g05ekNP1f07X0
M`1f07]j001k07X0Neeg07]j07H0G@3m3@0Oo@d47`eH3@gj4@X=3Gal;TUJCTe>
COa9oVL1BDWnPgl8I`P8I`QW28:21RU=PP1j001jN`1f07X0NeekMed0NP1f05d0
NP1kGGH0NgX0Ned0NP00NP1MN`1f07H0NgH0MP1j07/0MP1f07X0G@1jMed0NgH0
NP1MNeef001kNP1f07]kMP1M07]MNed0MP1j07Yg07YgG@1jNgH0MP1gGB/0NgX0
N`00NP1g07X0MeekGG/0NW]f001j07YgG@1j001MN`1f07X0NhZ:004@?8Yk07Ik
07X0MW/0GG/0MP1j001fNP1kGGMj07H0NP1M07Ik07X0NP1fNP00MWH0MP1j07]M
MP1kGG/0NP3k3@1HnPd14@gm4@h=3GamOEXWBDe>CDi9J5[nIgl8IeY9J8>3J0QW
I`QW26N2PTh6PP1j07Ij001jN`00MP1gG@1gNP1j07IkG@1f07/0MP1kG@1gNP1f
MP1fN`1f07Yk07X0MW]M07H0NP1fNgX0MW]M07/0MP1j001fN`1k07Ij001jNed0
07Yk07H0N`1j07X007H0NP00NP1f039k07IkNP1kG@00MP1j07Mj07Ik07X007H0
NP00Neeg07H0N`1fNeeg07X0Ned0MP2:RP0143b:07Yk07H0GG/0NWH0GG]f05ek
NP1gG@1j07L0NW]f07Yk07H0MP1jN`1gGG/0NW]f07X0N`1kGGL0MP1koPd0F?P=
0UP=3O`A37`=O0dNBDU>CDi=BAcnIgmlJ0Q:BDV3B@P8I`QW26N2PX=j07]f05ek
07]f001jMP1kGG/0NP1j07]f001kNgH0NW]f07H0NP1f07/0N`1MNgX007X0NW/0
MP1kGG]f07X007/0NgH0GG]f07MjM`1f07Yk05ekMP00NWH005ekMP1f07H0GG]k
MP1fMP00Nc=f05ekG@1kGGL0NW]f07]MN`1j07H0NgYgMP1jMed0NP1kMP1kNP1f
07H0MP1kMP1kRXX00A<lRW/0GG/0MW/0NP1k07H0Neef001fN`1f07X0NP00MgX0
07Yk07X005egGG/0MP00NP1fN`1j001j07Ik0?`=1Al=3EP=7oh=0Al=o1433GaJ
B_i93Te>CFMlO6Ql77aWI`D=3OeW00SnIgl8PP1j07X0NWL0NP1MMgX0MgX0GG/0
MP1f07H0NgX0G@1kG@00NW/0MP1jNed0MW/0MP1kGGL0NP00MW]j07/0N`00MWH0
MP1kGGL0NW/0GG]MN`00GGMk07]MM`1jMgX0GG]MN`1fN`1M07]k07YgNP1kGG/0
Ned0Neeg05ekGBD005ef07X0MP00NP1k07X0MP1f07X0G@1fNgX0NW]M07]M08Z:
0043?8X0GG/0NW/0MP1f07H0Neeg07MMNed0NP1kMP1f07X0Meek07H0MeekGG/0
MP1kMP1MN`1f07Yf07Yk07_a3O`A0g`5BU[nB@>3PdUWng`5Igel3G`8o6Mo20P0
NgH0MP1j07]f07H0NP1fG@1kMP1MNeek07]j001MNgH0NgMMN`1f07Yg001kNP1f
07MM07]f001kNP00NP1fG@1jN`1jNgH0NP1f05d0NP1k07If07X0NP1f07H0NP1M
N`1k07Yk05ekMed0MP1j07]MM`1j07]f07X0GG/W07]fMP1kM`1j07]f07X0MP1f
07]j07]f07]k001j05d0Med0NhZ:004@?8Yk05ef05eg07Yk07Ik07X0NP1g07Yf
07H0NgX0MP1f07H0MW]MN`00NP1fNgX0N`1f07H0MWL0NP1MN`3n3@<O3DE5nPd3
F0dA4O`=0@E:oTT3Ph=9IoeloPeoO@el3GeWIgeW201kG@1jN`1f07H0NgX0M`1j
M`1j07]M001f07H0MgX0M`1j05d0MP1fNg/0GGIk001fNgX0Ned0NWIk001fN`1f
N`1f07X007X0MWMMNg/0MP1f07Ik07H0MP1fNgX0M`1fN`1f07H0Neeg07X0MW/0
07H0NP0aMP1f07]M07Ik07X0N`1f07X0MgX0N`1kGG/0MW/007H0NP1f07Ik07Ik
07X0MP2:RP014Sn:07H0N`1k02`007]j07H0MP1j07X0MgH0Ned0NgX0NgX0NgX0
N`1fMP00NP00NP1jNgH0NW/0NP00NP1k3EP=oDD13GgnA@0OoPgn4@0OoPgnB@MX
PdU9BVMmO?d=07gn3OemOgX0MgH0NgH0G@1kMP1jG@00Neek07X0MP1kMeekNgH0
NP1MNgH0GG]MNgX0NP1f07/0MUekG@1j07]f07]M07Yf05ek05ekMP1gMP1MN`1k
05d0NP1kMP1j07]j07H0NP00Neek05ekMP1kGGL0NP1MN`1fGG]f07X0MP1kGGL0
:WX0MP1f07]j07X0NP1f07YMM`1jNed0NW]f05ekGG]M07/0NP1f07Z:RP012Cf:
NP1f07IkG@1j05d0NW/0NP1kG@1j07]j07/0G@1f07]M001f07IkNeeg07Ik07X0
07]j001fNeeg07H03OI50`e53@gn4Od=oDT5CFQ91FMln`d0OOh=07cn07mf07]M
07/0MW/0N`1k07If07/0NP1fNed0NP00NP1f07Ik07X0M`1j001f07X0MW]MN`00
MW/0MP1kG@1gNP1kGGL0NP1k07H0NW/0GG/0MW]M07H0NW/0MP1kNP1j07Ij07/0
MW/0NP1M07X007Ik07X0M`1j07]j07H0NP1fMbUMN`1j05d0MP1j07Ik001gGG/0
07Ik05ek07L0NP1gNP1f07YgN`2:RP013Cb:001kGG]M07]kMWH0N`1f07H0NgH0
NP00MP1MNgH0Neeg07YkMP1j001MNgX0MP1jMed0NeekG@1kMP1km4DBAPd=4A4=
3EP=3DUW1Td6BDT=OOd=Oa4=O@d=OG/0NWMj07X0NgH0NgX0MP1f05ekNgH0MP1M
N`1kMP1gMP1f07Yk07H0GG]f07H0NgH0NW/0NP1gGG/0MP1kGGL0NP1kGGL0MP1f
05ekNP00NWMM07X0MP1kNP00NeegG@00N`1k07MM07X0MP1MNgH0GGYg05ekMP1j
=01gG@00NgX007X0N`1f07]fNP1jM`1k07H0NP1MMeek05ek07]jMP00NP1f07X0
G@1kRXX00@4lRWMj07MM07H007]kG@1fNg/0MP1k07Ij07Mj001fN`1MNed007Yk
07Ig05d0MW/007X0MP1g07X0GG]M0?=5nPgmB@D6CDi9Iggi3Gmm001MM`1j07H0
NP1kG@00NW]MNeef001jN`1fN`1j07X0NW/0NgX007Ik07H0NW/0MP1k001MN`1f
GG/0NP1f07X0MP1f07]MN`1jN`1f07]MM`1j07IgN`1f07]MN`1j07MjNP1f07IM
Ned0MW/0M`1jN`00NP1g07Yk07X0MS5gNP00MWX0NP1fN`1MM`1f07X0NP1fNP1f
N`1j07Ik07H007YgNP1fNg/0MW]M08Z:0049?8Yj001j07X0NgIM07H0Ned0GG]M
NgH0N`1j07]fNed0NP1kMgH0G@1kGGH0Ng/0MWH007]j07H0GG/0NgKcA@D=3Al=
FT[nB@IXBDe>26M9nPeoMP00NgYg07H007]f07MM07YgG@1gG@1k07X005ek07X0
NP1gG@1gG@1gGG/0NW]f001MNgX0NgYf07H0N`1f07YkMP00NP1kMP1k05ek05ek
MP1f07H007]M07YkMP1f07H0NP1f001kNP1k07H0N`1kNWH007MMNgH0NP00MP0c
GG/0NP1j07Mf001j07H0Neek07]f07H0N`1k07H0GG/0MP1jMP1j001kG@1kMP00
NhZ:004:?XX0MWH0MWMMN`1kGG]M07Mj001f07X0GGL0MP1k07Ik07H0NP1j07Ik
07X0G@1fN`1jMP1kGG/0MP1f07/0IOE5o0d91DY9B@H6CF`T1O`=Oa40N`1jMP00
NW/0MWX0Neek07H0NP1f07Mj07IkGGIk07H0MP1f07X0NP1f07H0NP00NWMj001M
Med0NeekG@1fN`00NP1fNP1f07X0GGIk07Ik07]MNgX0MWX0Ned007Yk07Ik07X0
Neef001j07IkNP1j001jNed0MP1j07X0=WIk07H0MP1fNgX0NgX0Neek05ek07H0
NW/0MP1f07Yk07H0MW/0N`1f07Ij07]M07]MMP2:RP013Cj:GG/0NW/0N`1f07H0
M`1j07H0NP1kMP1kGG]f05ek07H0NP1kMgH0NP1gMP1jN`1MN`1kGGL0GG]f07X0
NQScA@d=3Al5C@H6CBV3P`Q:F_h=O`00N`1jM`1jMP00GG/0NP1k05ekMP1f07]j
001kG@1k05ek07X0NgX0Med0NgX0MP1kMP1j07H0NP1kMP1g07]MN`1jMP1MNgL0
NP1kMP1j001j07]MM`00NW/0Med0NgH0G@1j07H0MP1f07]fMP1j001f07]f05d0
NgX0NgH0=gMj05ekNP1kMP00Ned0MP1k07Yg05ekMP00NW]MNeek07H0NgX0MP1f
07]k07H0NgH0MP1kRXX00@hoRW/0NP00NP1j07YkGG]M07IkNP1f07]M07MMNeek
07IkG@1f05ekN`1j07]M05egN`1f07X0MW/0Ned0MWMV6?A50@d=oX=oCDh6K0QW
B@D=7`d0NgX0NP00MWL0NWX0GGL0NP1fN`1kGG]M001fMP1gNP1fN`1f07X0Neek
07H0NgX0MP1k001fNed0MP1j07X0GGL0NP1k07H0NP1kG@1gNP1fMP1j07]MN`00
GGMMN`1MN`1fM`1jN`1jN`1f07/0NP1fN`13MP1jMP1f001f07X0NP1fN`00NP1k
GGL0NP1fNP1f07X0Meek07H007H0N`1fNed0NeekGG]M05eg07H0NW/0MP2:RP01
2Sf:07H0NgH0GG]f001k07]f07/0MP1gG@1jGG/0Ned0NP1kMP1kNP00NP1j07]f
07X0GG]f05ek07X0MP1k0?=5o@d7Ph==1RU>Ie[m3Gmk07X0MP00NW/0NWL0MP1M
NgH0NP1j07/0NgMMNgX0NP1MN`1MNgX0MP1j07H0Ned0Ng]f05ekNP1g07]f07H0
NWL0MP1f07H0NP1gGGL0NP1j07Mf07X0N`1fGG]MN`1f07]f07]j001j001MNgH0
NgH0N`1MNgX0MgH0NW]f03UkMP00N`1kGG]f07]MMP1kMP00NgH0Neek07]j07Yk
MUd0NP1kMP1k07X0Meek07YkGG/0MP1kRXX00@`nRW/0MP1j07L0NW]M07H0Ned0
MW]j07/007H0NP1gNP1j07]M001fNP1f07H0N`1MMg/0NP1gGG/0MW]j07XHmDGl
3Gm9BDi=CTi=M`1k05d0MP1fN`1f07X0Neek07X007X0MW/0MP1f07X007Mj07Ik
07Ik001fN`00MW]j07]M001jN`1M07IkNP1k07X007Yk07IkNP1j07H0NP1f07L0
NP1k05ef07Ik001j07IkG@1j05d0NWH0MWMk07X0GG]M04EfN`00MP1kN`1MNed0
MWX0GGMM001j05ek07H0NgX0GG/0G@1MN`00MP1k07Ik07X0NP1MM`1j07MMN`00
MP1fNgX0RXX00@/mRUd0NP1kMP1j07H0GG]f07]f07/0MP1fMP1j07MM07X0Med0
NWMj001k07]MNgH0NP1f001j05d0NP00NP3dAOd=oTUo1Te/07X0NP1f07]j07/0
NW]f07X0N`1f07Mf05ek05ekMP1j07MMNP00N`1kMP00NW/0MWH0NP1f001jMP1M
MgH0Ned0NP1f07]fG@1kNP00NWL0NgX007]j07H0NP1f07]MN`1fMP00NP1kMP1k
NgH0MgH0NP1j07Yf07MMN`13NWH0NgX0GG/0NgH0NgH0NP1gMP1jM`1j07MM07Yg
07H0NWMM07YkMP1MN`1f07X0NgH0MP1MNed0Neek07X007^:RP0133f:Ned0MP1j
N`1fN`1f07X0GG/0GG/0MW/0NP1f07/007H0MW/0NP1fMP1f07/0NP00MW]jMP1f
N`1gNP00N_A5o@eoBFQ>J`00MP1fM`1jN`00NP1f001j001f07IkNP1jMWL0NP1M
Ned0MW/007If07H0NP1f07X0Ng]M07IkGGL0Ned0NP1f07/0GGIk07X0MgX007Ik
05ef07]MMP1kNP1fN`1j001f07Yg07Yg07X0Ned007Ik07]j07X0N`1kG@16Meek
07]M001fN`1f07]j07]M07H0NP00NP1gNP1kG@1j07Ik05egNP00NP1gGG/0MP1f
07]j07Ik07]M07H0NP1fN`2:RP013Cf:001kNgH005ek07H0NP1jM`1MN`1jN`1j
M`1kGG]fMP1jN`1j07H0NeekNP1f07IMNed0N`1k07Ij001jMo=5O`d=F4X5C@1k
MP1kN`1j001kGG]f07]MMgIf07]j001j07/0MP1MN`1gGG/0NWH0NgX0MP1jNgH0
07X0MP1k07H0MP1kMP1kNgH0NWL0NgH007X0NeekG@1j07MM07]f001jN`1j07MM
NgX007]j07H0MP1f07Yk05ek07H005egBEd0Neek07X0NgH0NP1f07]M07]f07X0
NWMMNgH0NP1j07]f07Yk07H0NP1f07]f07X0GG/0NP1j001k07X0MUekMP1MN`1j
RXX00@dnRWX0G@1jMgX007IkN`1f07X0MP1f05ek05ef07/0NW/005eg07Ik07X0
07Mj07YgN`1kG@1f07Ik07If001UmDGm3Gl5M`1j07]M05ek07Ij07L0NP1j001k
Ned007Ik07H0NW/0Meek07X0MP1kNP00NW/007X0MWX0MW]j07X0NW]M07]M001j
001j05ekGGL0NP1g07X0MgX0Ned0NP1f001fNeek001fNP1M07Yk07X0MgX0GGMk
GG/0MWIk07X0MTH0GGL0NP1jMP1fNed0MgX0N`1gNP00MP1kG@1f07H0N`00NP1g
NP1g02`0Ned0MW/0MP1f07Ij07H0MWL0N`1fN`1j08Z:004>?XYf07]f07X007]j
05d0NgH007]MNeek07X0NgH0MP00NWH0Neek07H0MP1j05ek001j07]f07Yk05ek
07]j6?E5oPeo07/0NP1f07]f07X0NP1f05ekMP00NgX0MP1jNed0Neek07H0NP1f
07X0Med0Meek07If05ekM`1k001f07X0G@1gG@1kMP1jMP1MN`1kGGH0MP1j07MM
07X0NgH0NW]f07X0N`1jMP1gMP1kMP1MN`1j07H0Ned005ekN`1j07I707]f07H0
MP00Ng/0MP1j07]M07X0NeegNP1f07X0NgH0GG]f07X0NP1j07]M07]j07H0Ng]f
07/0Neek07X0NgX0MP1kRXX00@lmRP1f07/0GGIf07]MMgX0NgYf07L0N`1f07H0
NW/0MW/0NP1MN`1jN`1fN`1fN`1fNP1j07]M05egN`1j07_eAGl=3@1k001fM`1j
05ek07Ig07IkNP00MWX0G@1fNed007If07/0NP1f07H0MW]MN`1f07H0MW]k07MM
07IkGG/0MWMk07Ik07H0MW/0MgX0GGL0N`1fNed0MW/0MP1f07/007X0MW/0GGL0
NW/0NP1MMg/0GGH0MW/0GG]MM`1M04]f07IkNP1kNP1fNeek05ek07H0NP1gNP1j
07/007Yk07Mj07X0MP1kG@1g07Ik05ek07H0MW]M07X0MP1f07/0NP1j001fNed0
RXX00A4nRW]kMP1jM`1kMP1kG@1kG@00NP1fG@1j07]j001kNP1f07Mf07H005ek
05ek05ek07H0MP1kMP1j07X0MP00mdEo3G/0N`1MNgX0MP1jM`1f07X0Ned0Meek
07Yg07X0NgMf07]f07X0Ng]f07Yk07/0GG]j07Yk05d0NP1kNP1gG@1kG@1kNP1f
07YkG@1j07Yg07X0NW]M07]j07H0NgX0NgIf05ek05ekGGH005egMP1j07H0Neek
07]fG@1kGG]<MP1kNP00Ned0NgX0M`1MN`1jNgH007X0MP1f05ekMUd0NP1f05ek
Med0NeekNP1f07H0NP1k07]f07H0NW]f07H0MP1f07]M07^:RP014d2:05ekG@1j
07/0MP1kG@1gNP1gNP1gNP1f07]MNP00MW]MN`1fNgX0MP1fN`1fN`1j07Ik05ek
001f07YkGG]UIOU5O`1MNeeg05egGG/0Neek07H0NP1kG@1f07H0NP1kG@1j07X0
Ned0MP1MNg/0GG/0MW/007H0NP1gNP1j001f07Ij07]M001fNg/007Mj07H0NP1M
M`1MN`1M07IkNP00MP1kN`1gGG/007IkNP1k07X007Ik05eg07X0N`1f07L0C7X0
G@1fN`1kG@00MUegN`1j001fNWL0NP1fN`1f07/0MP1fN`1f07X0MP1k001fNgX0
MP1f07X0N`1jN`1MNeek07Yk07H0MgX0RXX00AI0RUek07]f07H0GG]f07]f07H0
NP00NP1f07]MM`1f07X0Ned0Ned0MP1kNP1MN`1j07]k05ekMP1jNgH007/0IFGl
AGl0N`1jM`1M07]j07/0MP1k05ekNP1gG@1kNP1gNP1f07]f07H0NP1kMP1kMP00
GGL0GG]M07YkMP00NP1f07Yf07Yk07/0MUekNP1M07YMN`1gMP1jM`1MN`1f07]j
001jMP1kG@1fGG/0NWH0N`1f07X0NP1j07Yg07X0MP1MNdmf05ekMP1kN`1MMP1M
NgX0NP1f07]f07/0GG/0N`1jNgH0NgH0N`1jNgH007]f05ekNP00NgH0NgX0MP1M
N`00NP1g07H007]j07X0NhZ:004G?XX0GG/0NeekGG/0NP1MNgX0MP1jM`1j07H0
Neek07Ik05ekG@1gNP1f07]MMP1f07H0GGMk07]M001fNP1MMoeUOdEk07X0NP1j
07Mj001j07X0MP1fN`00MUek05d0NP1kGG]M07]MN`00MP1jN`1kGGIk07Ik07]M
001jMgX0MWL0N`1j001f07Ig05d0MW/007H0NP1k05ekGG/0MW]M001fN`1j05eg
NP1k07H0Ned0MW/007Ik07Ik07X0GGMkDUek07Yk001jMP1MMg/0MW/007H0NP1f
07Yk07Ik07H0NP00NP1MNgX0NP00NWMj07X0MP00MW/0Ned0MgX0MW/0MWH0NP1j
Neef001j07H0RXX00AUoRWYg07H0N`1g07X0NWMM001kGG/0GG]f07]MN`1MN`1j
MP1g07X0Ng]f001k07X0NgH0NP1f001fNW/0NgH0IFEk07/0MP1f07H007X0NWH0
MP1kNP1j05ekGGL0GG]f07]MM`1kMP1k07MMNgH007YM07/0NP1MMed0NgH0Neeo
07/0NP1f07]f07Yk07H0Ng/0MUekMP1j05ekGGL0GG]MM`1jNed0NgH0N`1f05ek
MP1kGG/0GGYg05ek07X0NWL0GGL0MP00NWH0MgH0NP1MN`1f07]j07YkMP1M07X0
GG]f07]f07Mf001f07]f07X0MP1MNeekNP1j07]gG@0ENgX0MP1kMP1MN`1f07MM
NgH0NhZ:004IOhX0NP1j05ef07H0M`1j07If07MMNP00NW/0G@1fN`1j07]MNed0
MP1MNeeg07IkG@1kG@1fNeeg001j07IkG@1k07X0NP1fNg/0MWH0MP1kNP1f07]M
07Ik07]MMg/0Ned007X0GG]M07Ij001jMgX0MgX0NP1fN`1j07Ik05ekOed0NP1f
Ned0NP00NW/0MP1MMg/007X0MgX007]M07H0Neeg001fN`1j05eg07Ik07X0GGL0
MW/0NP1f07H0MP1jN`1j07X0MP1kG@1kG@1fN`1fNed007H007Yk07H0MW/0NP1M
Neek07Ik07H0N`00MW/0M`1f001f07H0NP1j5@00MW]j07/0MP1fNgX0M`1j05f:
RP016Gn:MP1f07H0NeekNP1f05ek07]MN`1kMP00NWL0NP1f07L0NP1kMP1kMP00
NP1j001kGGL0NP1kGG]f05ek07]f07IM07X0Ned0GG/0NW/0N`1f07]MM`1j07MM
07X0NP1kMWH0NWL0MP1j07Mf07X007X0MP1MN`1f05ek07Yg07mkMP1MN`1kMP1j
MP00GG]f07X0NWH0Med0NgH0NgH0Ned0NgH0NP1MM`1j07Yk07X0MP1j07X0MP1j
M`1j07]f001j07H0NgH0NP1kGGL0Ned0NP1kMeekMeek07H0NW/0MP00NP1k07]j
05ekMP1kNWH0GG]MN`1jNgH0NP1kMQD0NgX007X0MP1kNP00NP1f05ekRXX00AUo
RW/0NW/0MP1k001jN`1gGG/0G@1f07YkG@1jMP1fNed0MP1j07]M07]fN`1f07If
07]MN`1j07/0NWH007H0Neek07H0MP1gNP1MM`1MMP1fNed0Neeg07X0NP00MP1f
07/0MP1j07]M07H0NP00MWL0NP1fN`1j07Ik07X0GG]o07X0Meek07/0MgX0MW/0
NP00MW/0NP1j05ekG@1fN`1fN`1j07Mj07X0MW/0GG/0MW/007X0MW/007Yk07H0
NW]M07Ik07X0MP1f07]MMP1gNP1j07X0GG/0GGMkG@1MMeegNP1f07X007Ik07X0
G@1kNP1MMP1f05ekG@1f07/EG@00MW/0MW]M001j07Mj07Ik08Z:004IOhX0MP00
NW]f07H0NP1gGG/0NWMMNgH007]f001kNP00NW]f07H0NgH0NP1f07]j07H0N`1f
07H0MP00NgH0Neeg07H0NgX0NP1kNP1f07]MN`1kMP1j07H0MP1gGG]f07X0NW]f
07X0MP1kNP1jMgX0NP1kN`1f05ek07H007X0OgH0Meek07IM07X0GG/0NP1gGG]M
07]f07Yg07H0Ned0NP1f001j07H0GG/0GGL0GG/0NWMf07/0NWH0G@1kNP00Neek
07H0MP1kNgIM07/0NP1f07H0GG/0NWL0GGL0NP1kNP00NgH0Meek07X0NWMf001f
07/0NgH0N`1j07MM5@1jMgX0MP1k07YgMP1j05ek07^:RP016Gn:N`1j001j07Ik
NP1j07X0NP1k05ekGGL0NP1f07]MM`00NW/0NP1kG@1fNgX0NgX0MP1fN`1j07Ik
NP1j07X0NP1gNP00MP1f07]j07H0N`1j07]M07YkGG/0NP1k07H0NP00Ned0MW]k
001f07X0MP1f05eg07X0MW/0MWH0MWmkGG/0GG/0MWL0MW/0MP1f07X0Ned0N`1M
NeekG@1gNP1fNgYf07Ik07H0MW]MN`1MN`1j07H0NP1gNP1f07]MM`1f07Yk07Yk
001j07IkG@1fN`1j07MM001jNP1j07Ik001fNP1j07X0GGL0NP1k07YgNed0MW/0
MP1fNgX0NaEM07X0MW]M07H0NP1f07Ik07X0RXX00AUoRP1MNgIf07]j001f07H0
MP1j07Yg07H0MP1jNgH007X0NP00MP1kGGL0NP00MP00NP1kNP1f07YkG@00NgH0
MP00NP1gGG]j07MM001kMP1j07H0NgH007/0MP1f07YkMP1jMed0NWL0G@1jNgH0
MP1kNP1j07Yg07X0NgX0NgYo07/0MP00NP1kGG]M07X0NgX0MP1kMP1jM`1j07L0
NP00Ned0N`1k07YkMP1j001jM`1f07H0NgH0NP00NgH0MP1MNeek07H007Yf07Yk
07H0N`1j07H0NgIj001f001k07YMN`1j07H0NWL0NP1gG@1j05eg07X0MP1kNP00
NP0EMgH007/0MP1kNP1f07Yk07X0NhZ:004IOhYk001jM`1M07Yg07Yk07/007H0
NW/0NP1f001jMgX0MW]MMeek07X0MW]MM`1fN`1f07]MN`00GGMj05ek07IjN`1j
07L0NW]M07Ij07]M07IkG@1gNP1MMW]k07X007/007X0MWL0NP1f07/0NP1f07]M
07Ik07H0MP1k05d0OgH0MW]j07Ik07/0MgX0MP1k07X0GG]M07X007Ij07Ij05ek
05ef07X007]j07Ik07X0NW/0MP1kG@1fNP1kGG]MN`1k07IkNP00N`00GGL0NP1f
N`1j001j07Ik07If07H0Meek07Ik001j05egGG]M07Ik07IkGG]M001fN`1f5@1j
MgX0NP1f07/0NgX007H0MP2:RP016Gn:GGMMN`1MNgH0Neek07H0MUekMP00MP1j
07]f07X0GG/0Neek07H0GG/0NeekNP1f07X0NP1gMP1j07H0NgH0N`1f07YMMP00
Neek07]MM`1j07]f07]f07]M07H0NeekMWH0GG/0NP1kNgH0MP1j07]MM`1kG@1j
07]f07Yg07mkGG]M05ek07X0GG]M07]f07H0NWL0NgH007Yg05ek07Yg07H0NgH0
NWMM001j07H0MP00NW]f07]f07]f07H0N`1gG@1j001jMP1jMgH0NP1MN`1jMgIf
07Yg05ekNgX0MP1kG@1k07Yf07Yg07X0M`1kG@1j07/007YgMP1MNaEMN`1M07]f
07]MMed0NeekMP1kRXX00AUoRP1j001fN`1kG@1gGG/0MW/0NP1fNg/0MgX0N`00
MW/0NP1k07Ik07H0NP00G@1fNg/0NP1f07Yg07IkG@1jN`1fN`00N`1fNP1gGG/0
NP1f07X0NP1MN`1MNeek05d007YgN`1MN`1f05ek07X0MgX007Yf07]M07H0NP00
NUeo07L0NP1gG@1fN`1M07H0NW/0MP1j05egNWH0NP1g07H0NP1fN`1k001j07Ig
07Yk07Yk001MNed0NgX0Neek05ef07Ij07Ik07]M07X0NP1fN`1j001k07H0NW/0
G@1kNP1f07Mj07X0Ned0NP00MWX0GGMj07X0MW/0NP1fN`0ENP1fNed0Ned0Neek
05eg07X0GHZ:004IOhYf05ekNP1f07H0NP00NgX0MP1kNed0MUd0NP1kMWX0MP1g
G@1j05ekMP1kMP1kNed0MP1f07Yk05ek07]f001MN`1fGG]f07/0GG/0MP00NWL0
NP1jM`1jMP1g07H0MgX0G@1kMP1f07Yg05ek07X0NgH0NgH0MP1kNP1fMP1fOgX0
MP1MNgH0NP1f07MkGG/0GG]f001kG@1gMP1MNeekMP1MN`1j07IMNgH0Neek07H0
07IMN`1kMP00NP1k07H0NgH0N`1j07MM07]gMP1jM`1f001j07]kMP1M07]f001k
NP1j07H0Med0M`1jNP1gMP1j07L0GG/0NP1jN`1f5@00NP1kMP1kMP1k07Yg07H0
GG^:RP016Gn:N`1gG@1fNeekG@1fNP00NW/0G@1gNP1k05ef07/0MP1fGG]M07Ik
07]M07Ik05ekGG/0Ned007Ik07X0NeegNeek07]MNed0MW/0MW]MM`1j001f07X0
07]MNeekNP1fN`1f07]j07X0NP1gGGL0NP1j001jN`1f07/0Ng/0Ngl0MW/0MP1k
NP1fNeek07/0MW/0NP1f07]M07]k05ek07/0MP1f07Ik001fN`1j07IkNP1gNed0
GG]MM`1f07IkG@1jN`1f07H0NP1M07/0Neek07IfNed007Yk07H0NP1f001j07Ik
GG/0MWX007Mj07/007Ij07MM07Ik05ek01EfMP1j07]M07X0GGL0NW/0MW/0RXX0
0AUoRP1jGGL0NP1k07MMN`1jMP00NgH0NP1f07H0NgH0NP1jNg/0MP1k07X0MP1j
05ek07/0MP1kMP1j07X0Med0NP1k07IM001gGG]M07X0NP1j07Mf07X0MgH0NP1f
001j07]f07]M07H0NP1gGG/0NP1j07Mf05d0NWMM07X0MUeo07X0NP1gG@00N`1j
07H0GG/0MP1jMP1j07]M07Yg07X0NW]f07]j07MMNeef001kG@00NUd0MgH007X0
NP1j07]f05d0NP1kNP1MNeekMP1j05ekN`00NWH005ekMP1jMP1jM`1j07/0GG/0
MP1j07X0GG]j07H0Neek001j07/ENP1g07X0NgH0NWH0MP1M07X0NhZ:004IOhYk
07]MN`1j07H0Meek07]MMP1kG@1fNgX0MP1kG@1f001j07Mj07H0MW]k07Ik07H0
MW]M07Mj07H0MP1fN`1j07X0MgX0NP00MgX0NP1f07X0NW/0NP1j07Ik07Ik07X0
NP1j07Ik001fGG/0NP1f07H0NW/0MWMMNed0MW/0OgIgGG]j07IfN`1j07IkGG/0
NP1g07]j07H0MgX0GG/0MP00NW/007H0NP1k07If07Mj001fNP1kNWH0MP1j07H0
Ng/0MP1f07]f001gGG/007Ik05efN`1jMW/007Ik07]M07X007H0MW/0MW]j07X0
MW/007IkG@1gGGIk07H05G]j07H0MP1k001jN`1fNed0MP2:RP016Gn:07H0N`1f
07H0Neek07]MM`1f07]f07/0MP1kGGL0Ng]f001j05ekGG/0G@1j07H0NgX0MP1j
001kMP1jN`1f07H0NP1kMP1f07X0MP00NgX0GG/0MP1f05ek07]j07H0NP1kMP1k
07If07/0MP1f07X0NWMM07X0N`1f07/0Ngmj07/007X0N`1f07H0GGL0MP1j07H0
MP1k07X0GG/0MP1kMP1M07YkMP1j07]j07]j07IMN`1kG@1gMP1kMP1k07X0MP1j
NgMM07MMNP00NgIj07H0N`1f07/0Meek07H0NgH0MgIkGG]M07X007H007]M07Yk
07H0NP1kG@1kMQEM001j07]MNgIf05d0NP1kMP1kRXX00AUoRW/0NP1fN`1jN`1j
07X0Neek07X0Ned0MW]j07]MN`1MNeef07Ik07MMN`1f07IkG@00NW]M07If07]j
001fNeek001f07Yk07Ik07YfN`00MW/0MW]k07Ik07X007Ik001f07]M07Yk07X0
MW/0NW/0MP00MgX0NP1j07H0MP1o07H0MW/0MP1jN`1jN`1j07X0MgX0NP1fNed0
MW/0MW]M07Yk07H0N`1f07H0Ned007Ik05egGG/0GG]k07]M07H0MW/007X0Neek
000/N`1k07IkG@1fN`1MMP1f07X0MP1kGG/0GG/0MgX0MW/0MWH0MgX0GG/0MP1f
07Mj07/E07IfMed0N`1kN`1fN`1j07]M08Z:004IOhYMMed0NP1f001j07H0MP1k
07X0MP1f07X007]M05d0NP1kMP1k07X0NP1kMP1j07]gGG/0MP1kN`1j07H0NP1k
G@1jNgH007]j07H0N`1kMW]M07X007Yk07H007Yk07IMNgH0NgH007Yg07X0GG/0
MP1kNP1kMP00NgH0NgX0OgYkGG]M07X0NP1f001j07]f07X0NWL0Ned0NgH007X0
Neek07H0NP1kNP1kMP1f07Yk07Yg07H007X007X0MeekGG/0NWH0NP1kGGMMN`1j
G@1kG@1kNP1f07]MNgX0Ng]f07/0NWMM07X007X0GG]k07X0NWL0MP1kNP1j07X0
5GH0N`1kMP1jG@1MN`1f07MM07^:RP016Gn:07X0MW/0NW]M07Ik07X0GGH0NW/0
NgX0MW/0MW/0MP1f07]j07H0MP1f07]j07X0NP00MW]M07X007IkNP1j07Mj05ek
GGMM07Ik05ej07/0MgX0MW/0GGMMMW/0GGL0MP1j05egNP00NP1kGGL0MW/0G@1f
07]MMP1kNP00MWl007H0MgX0MP1fNgX007H0Ned0MP1j05egNP1kGGH0NP1g07Ik
G@1f001j07X0Ned0GGL0NW/0MWIgGGX0MP1f07MMN`1kG@1f07]M05ek07Mj07]M
001fNed0N`00MP1MNed0NP00MgX0MWL0MW/0GGL0NP1j07Yk001f07H0MQEkNP1j
07]M07Mj05eg07IMNed0RXX00AUoRWH0NgX0MP00Meek07H0GG]g07/0MUd007]M
07X0NgH0NgX005ek07X0Neek07H0MP1f07X0NgH0MWH0NP00MP1j001j07L0NP1k
05ek07MM07X007X0MP1j07/0GG/0NW/0NgH0NP1gNP1f001j07X0MP1jNgH0MP1k
G@1gGG]oNgH0NgX0MP1kNP00NWMMMed0NgMj07H0NP1gG@1kMP1f07H0N`1j07MM
NeegMP1f07]MMP00NP1k07/0MP1j07]MN`1f07]f07MM07]f07]M07X0NgH0NP1k
MP1kGG]MNed0Med0NgX0MP1j07]M07X0MP1f07H005ekMP1kNP0EN`1f001MM`1j
05ek07YkMP00NhZ:004IOhYk05d0MW]j07X007Yk07H0NUd0MW/0MWH0MgX0MP1j
05efMgX007Ik05ek07Ik07Yk07H0MP1kN`1kNP1fM`1f07Ig07X0NP1f07Ik07IM
Ned0MWL0NP1fN`1MMW/0NP1MMP1kNP1j05egNP1f07IgGG/007X0NWMM07Mj07/0
OeekG@00MW/0G@1fN`1j07X0MP1MM`1jN`1j07Mj07]k07YkG@1fNeek001j07]j
07H0Ng]MNP1f07X0MW]k07H0N`1j07H0NW]M07H0Ned0MW/0MP1jM`1j07]M07H0
M`1j07X0MP00NW/0NP1MMgX0MW]k07IkNP1f07]j001f5@1jNeeg07X0MW/0MW/0
07YkG@2:RP016Gn:05ekM`1j07H007Yg07H0NP1gGG/0GG]k07X007]f07Yg07/0
NWMj07Yg05ek07H005ekNP1gG@1kG@1g07X0NgX0NeekMP1f07]j05ekMP00MgX0
MP1jN`1f07/0MP1jM`1kG@1k07Yg07]f07]j07/0NWH0GG/0N`1j07X0NWmf001j
MP1kMP1kN`1f07H007]MN`1jN`1f001j07]M07H007H0NP1k07If001j001kMed0
MP00Neek07X007X0MP1f05ekMP1M07]kMP1gGG/0GG]f001j07H0MUekMP1j07Mf
07]MNWL0MP1f07X007]M07]j001MNgMM07]MNaEMN`00NWH0GG/0MP1MNgH0G@1k
RXX00AUoRW/0GG]M07IkNWH0NP1fN`1f07MMN`1M07IkGGL0NP00NP1MM`1M07H0
NP1f07Yk07Ik000/07X0GGMj07X0MP1kG@1k07X0Ned007Ik07]MMP1kNP1f05ek
07IkGG/007X0GGMj07H0NP1MNed007X0MWL0NP1MN`1f07H0MP1oNgX0M`1j07X0
MP1fNg/0NWH0N`1j001fNeef07H0NeekNP1kMP1f07Yk07Ij07If07X0MW]MMP1k
GGL0MWX0NW]MN`1f07]k07H0GG]MN`1MM`1j07Ij07Ik07/007X0NP1j07H0M`1j
07IkN`00MWH0Ned007Yg05ek07H0MedEN`1fN`1kGGL0MW/0NP1kGGMj08Z:004I
OhYM07X0NgL0NP00M`1jNed0NgX0NeegMP1j07]MMP1MNgH0NP1kMP1kMP1MNeek
05ekG@1jN`1kMP1j05ek07]f07]f07H0MUd0NgIj07]M07]j001j07]f07]j07/0
NWH0MP1j001kMP1MN`1kMgH0GG/0MP1kMP1f07X0Ng]fO`00MP1jNgH0GG]MNed0
MP1kMP1f07]f07X0MgX0MP1gG@1f07X0NgH005ekM`1k07X0NgX0M`1gG@00NW/0
MP00N`1f07]M05ekGG/0NP1kMP1f07H0N`1k07H0NWH0NgH0NP1kMP1j07H0NP1f
MP1kNed0NWMf05ekG@1j07X05Ed0NP1f001j07X0MP1j07MM07^:RP016Gn:NgH0
MP1j001fNP1j001fNP00MP1kGG]M07H0N`1fN`1kG@1f07]j07/0MP1gGGMk07Mj
05ej07/007Ik07H0NP1MNeek07/0MP1k07H0MW/007Ik07H0Ned007H0MP1kNP1f
07Ij07/0MP1f07Yk07H0NP1fN`1fNgX0NP00NWmgNP1j001k07H0N`1kGG]M07]M
N`1MN`00NP1kNP1j07Mj07H0MP1jNeek05ef07X0MP1kGG]j07Ij001j07IkG@1f
NgX0MgX007H0MP1j07]j07Yk05ef07Yk001fN`00MP1f07Yk07IkN`1jN`1f05eg
NP1j07Ik07Mj07H0NQD0MW]MNeeg07H0NW/007H0Ned0RXX00AUoRP1kMP1j07]j
M`1j07MMN`1kMP1j07/0NgH0NP1jN`1f07H0NgH007X0NWL0NP1j07H0G@1k07H0
NWIk05ekGG/0NP1g07H0NW]fG@1kGG/0NWMj07H0NP1kMUekGG]MN`1f07]k07H0
GG]j07H007H0NP1kNed0NP00MP1kMP1oNP1f07]f05ekMP1f07X0NgH0N`1jM`1f
GG]f001j001j07]j07]j05eg07X0NeegMP1j07]M05ek07Yf07]j07MMNed0Ned0
NgIj07]f07H0G@1k07H0NgX007IMN`1jMgX0NgX0G@1kG@1f001jNgH0N`1f001k
G@1j001kMP0EMW/0N`1j07H0NWL007YkMUd0NhZ:004IOhYk07]M07H007X0MgX0
N`1f07Yk07X0NP1k07Ik05eg07YkG@1jN`00MP1jN`1j07IkNP1f07Ik001j07Ik
07X0MP1f07Yk001MN`1f07MM001j07IkN`1j07/0GGL0N`1fNgX0GGL0NP00MW/0
NW/0NP1f001fN`1fN`1f07X0O`1fN`1MNeeg07X0NP00MP1j05ef07X0N`1MNeeg
07Ij07X007H0N`1gGG/0MP1k07]M07H0N`1f07Ig07X007H0MP1kG@1fN`1k07H0
NgX0MW]M07IkG@1fN`1j001f07X0MP1k07Ij07]MN`1f07/0MP1jN`1fN`1f07Yf
07]k5@1MN`1f07YkN`1jGGL0GG/0MP2:RP016Gn:07X0MP1kMP1j07X0MP1MNeek
07H0MP1f05ek05ek02`007]f001fGG]j001f001kG@00Neek05ekMP1j07H007]f
07Yk05ejM`1kMP1j07]fG@1kG@1f07H0NWL0MUd0NP00NWL0NP1gGG/0;000NWMf
07Yf07X0NP1jNgH0MgmMN`1jMP00NP1kMP1gGG]f07Yg07H0MP1jM`1f07H0NgH0
07YkMP1f07X0MP1j07YMM`1kMP1jMeek07X007YkGGMj07MMN`1jG@1kNP00GG/0
Meek07H0N`1kMP1jNgH007]f05ek07H0N`1f07X0GG]f001kNP1f07Yg07]M01Ej
M`1MNeek001j07H0MP1kMP1kRXX00AUoRW/0MW]M07If07H0MW]M001f07IkGG/0
MW/0GGMk07X0MWX0NgX0Mg/007Ij07If07Mj07Ik07Ik07]M07IkGGH0NgX007Ik
07X0GG]M07Ik07Mj07]MN`1j001jN`1gNP1fN`1j07X0NP1MN`1fNP00Ng/0N`00
MP1f001jNeeoN`1j07]fN`1f07YkGG/0N`1MNg/0NW/0GG]MNgX0MP1kMW/007Yk
07X0MW]k07X0N`1f07X007X007IkGGL0GG/007H0MP1f07Mj001fMg/0NP1f07Yk
G@1f07]M001jMgX0Neeg07IkG@1fNg/0NP1j07Ij001fNgX0GG/0MW/E05efN`1g
GGX007IkNP1j07]M08Z:004IOhYM07X007Yk07]j07]M07]fN`1j07/0MP1kMP1j
07X0NW/0MP1f07X0NWMj05ekN`1j05ekG@1j07MM07]f001kGG/0MP1j07H0NWL0
MP1kG@1j07X0N`1jMgIf001MNed0N`1j07H0MP1jM`1kNP00NP1MMed0NW]f07]j
001kO`1f07H0N`1j07H007/0MP1MNed0GG/0MP1MM`00NW/0NP1MMeek07H007]M
07H0MP1MNgX0NWH0MeekG@1kGG/0NWH0NW]MN`1j07]MNgX0MP1MMgX007IMNgH0
NgMf07X0Med0NP1j07]f07X0MP1f07Yk05ekNP00NWL0GG]M5@1kMP00NP00NWMj
001j07MM07^:RP016Gn:N`00NWL0GG/007X0MgX0N`1f07X0MW]j07]M07H0MP00
NW/0Ned0MP1j07Mk05ef07Ik07Mj07X0MgX0NgX007H0MW]M07Ik001j07Mj07]M
07H0MP1j001kNeefN`1kG@1f07IkN`1f07X0G@1fN`1fNed0MP1k07H0NP1f07mf
N`1j05eg07YkGG/0MW]MM`1kG@1fN`1j07X0MP00MP1fN`1j07IkNP1MN`1j07]M
001k001jN`1f07Mj07/0NP1kG@00Meek07H0MP00MW]j001kGGL0N`1j05ek07H0
NP1gNP1f07X0Ned0MW/0N`00MW/0G@1fN`1j07L0NQD0GG]MM`1fN`1M07Ik07H0
MW]MRXX00AUoRP1jMgH0GG/0NWMf07X0Neef07]f07X007X0NgH0NP1gGG/0MP1f
07]f001j07X0Ng]f001j07H007X0MP00MWH0NgX007]k07IMNgH0NP1MM`1j07]f
07]f05d0N`1f07H0NP1kG@1kNgH0MP1jM`1MNed0Ng]f07YkMP1MNeeoN`1j07H0
NP1j07MM07X0N`1j07MMN`1jMgH0NP1gNP1jN`1f001j001jM`1f07H0NgIf07]f
N`1MN`1j07H0MP1kG@1kMP1j07H0MP1jMP1j001jGGL0MP1j07Yf07X0Med0NP00
NgH0MP1kGG/0GG]f07X0MgH0N`1j07MMMP0ENWL007X0NP1kMP1kG@1kNP1j08Z:
004IOhYk07X0MW/0NP1j07H0MP1k07H0Ned0MW/0MP1kNP1f07H0MW]MNed0Neeg
07H0MP1j07Ij07IkGGH0MW]j07]kG@00MWH0GG/0M`1j001fN`1fNed0NP1MNeek
G@1fNeek07Mj07]M05ek07Yk07X0M`1j05d0Ned007YfM`1jO`1f07Yk001f07H0
MWX0NP1f07H0Med007X0MW/0NP1f07]MNeef07Ik07X0NW/0MP1k07IMN`1fN`1f
07H0NP1f07Mj07X0NeekNP1k07]j07Ik07]MNP1f07L0Ned0NP1gNP1fNP1k07X0
07H0MW/0NP00MWX0MP1f07X0N`1f5@1jNed0MW]M07]j07Mj001f07J:RP016Gn:
07H0GG/0MP1f001j07]f05ekNP1f07]M07]MN`1f07Yk07X0N`1kMed0NP1j07]j
07H0N`1k001kMP1k07IM001jMgX0NgH0MP1j07YgMP1MN`1kMP1jM`1f07L0NP1g
07IM07X0NgH0NgH007X0Meef05ekMed0NgH0N`1j07mf07]f07IjNgH0NW/0MP00
NgH0NP1kMWX007/0;000NgH0NP1g07]j07H0MP1M07]j07H0Ned0Ned0NP1kNP1k
NgH0NP1gMP1f001kGGMM001k07H0N`1kNgH0NP1kMP1kNP1MN`1j07H0MeekGG/0
NP1fNW/0MP1kNP1kMP1MNaEf05eg07]M07MM07H0G@1kNP1kRXX00AUoRW/0MW/0
MW/0NWMj07H0NP1f07]MNed0MgX0N`1fNg/0GGL0NP1f07X0MP1j07H007YgNP1f
07Yg07Yk07X0MgX007X0MP1kNP1j07H0NP1fN`1j07/007X0NeekG@1f07Yk07H0
MP1j05ek07H0MP1jN`1f07X0MP1j001fN`1oNP1k07H007X0MP1k07IjN`1jN`1j
07/0MWX0NP1fNP1kG@1f07X007IkN`1fNed0NeekG@1gNP1kG@1f07]M05ek001f
07Yk07Ik07X0MW]M07IkG@1f001jN`1j07X0G@1fN`1f07YkGG/007H0MP1fM`1M
Ng]M001f07Yk000ENP1gGGH0Neek07IkNP1f07]M08Z:004IOhX0GG]M07X0MP1j
001kMP1j07X0MP1g07X0MP1MN`1M07X0MP1f07X0Ng]f05ekMeek001kMP1j07X0
07X0MP1kMWH007X0NP1f07H0N`1jN`1f07H0MWH0MP1j07]f07YkG@1kMP1j07Yg
07YkMP1jN`1f07MM07]f07MMNed0Og]f07H0NgH0Ng]f07H0N`1f001f07X0MP1k
MP00N`1j07]f07Yg07YkG@1MN`1kMP1kGGL0NP1gG@1kNgH0MUek07IMNgX0G@1j
07H0GG/0MP1j07]f07YMN`1f07H0MgH0Ned0NgH007MM07YkMP1jN`1MN`1MM`1j
NgH007Yf5@1kNP1kMP1gG@1kG@00NgMM07^:RP016Gn:N`00MW/0MW/007If07]k
07H0MW]MNed0MW]MN`1fNed0NW/0NP1fN`1MNgIg05ek07X0GG]M07IkGG/0MW]M
07]kGGL0NP1fNg/0MP1j001fNgX0NW/0MW]M07H0Ng/007Mj07]M07L0NP00GG]k
05ek07X0MWX0NgX0M`1j07l0MWX0NP1kG@00NW/0MP1fNgX0NP1fN`1f07YkG@1f
07H0NgX0GGL007Ik05ef07]M07]MNP1j07Ik05ek07/0GGL0NP00NWMj07Ik07H0
NW/007H0Ned0Meek07IkGG/0MP1gNP1kGGX0MgX007Yk001j05eg07X007X0MW/0
NQD0G@1f07Yk07Yf07MjG@1j07H0RXX00AUoRUd0NW/0GG/0NWH0NgX0G@1kGG/0
Neeg07X0M`1MN`1f07/0MP1MN`1jM`00NP1j07H0NWH0M`1j07MM07]M07]M07]M
NgH0N`00GG]f07]f07/0MP00NgX0NgH0Med0MP1j07X0NgH0NP1jMgH007Yg05ek
GG/0Ned0NP1f07IoGG/0NWL0NP1gGG/0GG]MNed0MP1MN`1jNgH007MMN`1kMP00
NgH0GG]j07H0NgH0NgH0N`1f05ek05ek07H0NgH0MP1kMP1j001k05ekNP00MgYM
Med0Neek05ek07X0NgH0NP1kGGL0Ned0NgH007]f07Yg07X0MWH0NgX0MP0EMgH0
NgX007X0N`1j07]f07H0NhZ:004IOhYkMP1MMg/0NP1kG@1kGGIk07MMN`1kGGL0
NP1fN`1fNed0MW]k05ek07X0MP1fMeek001kGG]M07H0MW/0Ned0MW/0N`1kG@1f
NP1k07H0NW/0NP1fN`00MP1jNeekGG]M07H0MP1kG@1f07X0MW/0GGIk07/0MP1g
NP1jN`1kO`1MM`1j001fNP00MW/0N`1kGG/0Med007X0MWX0Meej07]MN`1kGGL0
07IkNP1kG@1j05egN`1f07Ik07IkG@1kNP1f07/007If07Ik001fNP1k07X0MP1k
07Ik07X0NP1kG@1f07]M05egNP1j07H0NP1kGG/0MW/0MP1kGG/05GX0MP1kGGL0
MP1f07H0NgX0MP2:RP016Gn:07]f07X0NP1f07]gG@1gG@1MN`1j07X0MP1MNed0
NP1gGG]M07]f07X0MP1kNP1k07Yf07X0NgH0NW/0MP1fGG/0MP1j07]f07]f05ek
MP00NP1kNed0NW]f001k07/0Meek07]j07H0NgH0NgX0NP1k07MM07X0NP1f001f
07mjM`1j07YgNP1f07X0MP1j07]f07X0NWH0NW/0MP1k07X0MP1f07H0GG]j001j
07]f07H0NP1MNeekG@1j07]gG@1kNgH0MWYg07Yk07Yf07]f07H0GG]gG@1j07X0
M`1j07]f07X0NgH0N`1f05ekGG]f07/0GG/0NW]f07/0MQD0Ng]f07H0GG]MNeek
07X0MP1kRXX00AUoRW/0Ned0MP1fNed0NP1f07Mj05ek07H0NW/0MP1gNP1j07L0
NP1MN`1fN`1j001j07H0Ned0MP1k001j07X0Mg/0NP1f07H0NW/0Neek07X0MP1f
05ekG@00NW]M07H0NP1gGGX007IkNP1k05d0MP1f07X0MW/0MP1fNgX0NP1o07X0
07H0NP1gNW/0MP1f07H0Ned0M`1k001j07X0GGL0MW]MNeek07H0Neeg07X0N`1f
N`1fN`1k07Ij07X0NP1j001jN`00NW/0GG/0NP1MNeek07H0NP1f07H0MUek07H0
Ned0MP1jN`1j07Mk07/0NP1MMW/0NP00NP1MMW/EG@1MN`1kNP00M`1j07H0MW]j
08Z:004IOhYM05eg07]MN`1kMP1f07]j07]fG@1j07/0GG/0NP1g07X0MP1jM`1M
N`1f07]f07H0Neeg07X0NgIf05ek07X0MP1MNeekMP00NP1k07H0GG]MNeek07]g
MP1M05ekMUd0NP00NWMj001f05ekMP1kNP1j07X0MP1kNP00MP1kOb`007YkMP00
NP00NP1jNgH0Neeg07Yf05ekMP1kMP1j07X0N`1k07YkMP1f07H0MP1jN`1MN`1f
05ek07L0NP1f07Yf001jMP00NWL0MP1MN`1k07YkMP1f07]kMP1k07H0NP1gGG]f
001f07]j07H0MP1f07/0NP1kGG]f07/05GH0Ned0NP1f07Yf001kGG/007^:RP01
6Gn:Ned0NP1MM`1j07]MN`1M07H0N`1f07H0MW/0MP1j07IkGG/007X0MP1fNed0
NW/0MP1kGGL0MP1jN`1f07H0MW]MN`1k07X0MP1f07Ik07H0M`1j07H0NP1jMg/0
GG/0MP1j001j07Ik07Ik07]M001f07IgGG]M07]MM`1f07mkGGH007YgNP1fN`1f
07/0NP00NWL0NP1f07]M07Mj07H0NP1f07H007Yk07Yk07X0MP1fN`1fNed0GGMM
N`1fN`00NW]M07Mj001j07Mj07H0GGL007Yk07H0GG]j07IkG@1f07/0NW]MNed0
07IkGG/0MW/0MP1f07/0NP1j01EkG@1gNP1fNgX0Neef07MMNP1MRXX000R1RX6:
k8X0000:0021RX6:kHX0002P08<0o`\>"], "Graphics",
  ImageSize->{277, 295},
  ImageMargins->{{35, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}}],

Cell["\<\
Some of the different animations available include rotating or \
tilting the camera, flying past the graphic or doing corkscrews around the \
graphic, and varying the lighting around the graphic.\
\>", "Text"],

Cell["Animations can also be exported into QuickTime files.", "Text"],

Cell["Exercises: Animation", "Exercises",
  PageBreakAbove->True],

Cell[TextData[{
  "Using ",
  StyleBox["Table",
    FontFamily->"Courier"],
  ", ",
  StyleBox["Do",
    FontFamily->"Courier"],
  ", or ",
  StyleBox["MoviePlot",
    FontFamily->"Courier"],
  ", create a two-dimensional animation of a function that changes over time. \
Be sure that the domain and range remain the same throughout the animation."
}], "Question"],

Cell[BoxData[
    \(\(myanim = 
      Table[Plot[Sin[k\ x], {x, 0, 3\ \[Pi]}, 
          PlotRange \[Rule] {{0, 10}, {\(-1\), 1}}, PlotPoints \[Rule] 50], {
          k, 1, 12}]; \)\)], "Input"],

Cell[TextData[{
  "The line above will generate the animation, but for better viewing on the \
printed page, we use ",
  StyleBox["GraphicsArray",
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " to view all the frames at once."
}], "Text"],

Cell[BoxData[
    \(\(Show[GraphicsArray[Partition[myanim, 3]]]; \)\)], "Input"],

Cell[TextData[{
  "Choose your favorite three-dimensional graphic from this set of exercises \
(or make a new one) and create an animation using ",
  StyleBox["SpinShow", "Input",
    FontFamily->"Courier"],
  ". (Remember to load the ",
  StyleBox["Graphics`Animation`", "Input",
    FontFamily->"Courier"],
  " package first!)"
}], "Question"],

Cell[BoxData[
    \(Needs["\<Graphics`Animation`\>"]\)], "Input"],

Cell[BoxData[
    \(\(simpleplot3d = 
      Plot3D[Sin[x - Cos[y]], {x, \(-3\), 3}, {y, \(-3\), 3}, \n\t\t
        Boxed -> False, Axes -> False]; \)\)], "Input"],

Cell[BoxData[
    \(my3danim = SpinShow[simpleplot3d, Frames \[Rule] 12]\)], "Input"],

Cell[TextData[{
  "(The cells of the animation have been deleted to save space. Enter the \
code into ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to see the animation.)"
}], "Text"],

Cell["", "EndExercises"],

Cell[BoxData[""], "Input",
  PageBreakAbove->True,
  PageBreakBelow->True]
}, Closed]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["10. Creating Graphics Functions", "Section"],

Cell[TextData[{
  "There are times when we will want to create a specific type of plot that ",
  
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " does not directly support. We can eventually make up any kind of graphics \
we want within ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ". Sometimes we\[CloseCurlyQuote]ll be pleasantly surprised to find out \
that what we want is just a variant of a type of plot ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " already knows, other times we will have to start from the basic building \
blocks of points and lines."
}], "Text"],

Cell[CellGroupData[{

Cell["InequalityPlot", "Subsection"],

Cell[CellGroupData[{

Cell["Basis", "Subsubsection"],

Cell[TextData[{
  "As an exercise, we will use ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to create a flexible function that will draw inequality plots; that is, \
plots where the area above or below the curve are shaded."
}], "Text"],

Cell[TextData[{
  "To create an inequality plot seems dificult until we realize that we can \
already achieve the desired results fairly easily using ",
  StyleBox["ContourPlot", "Input"],
  ". ",
  StyleBox["ContourPlot[", "Input"],
  StyleBox["f",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["{", "Input"],
  StyleBox["x",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["xmin",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["xmax",
    FontSlant->"Italic"],
  StyleBox["},", "Input"],
  " ",
  StyleBox["{", "Input"],
  StyleBox["y",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ymin",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  " ",
  StyleBox["ymax",
    FontSlant->"Italic"],
  StyleBox["}]", "Input"],
  "generates a contour plot of ",
  StyleBox["f",
    FontSlant->"Italic"],
  " as a function of ",
  StyleBox["x",
    FontSlant->"Italic"],
  " and ",
  StyleBox["y",
    FontSlant->"Italic"],
  ". Several options allow us to change ",
  StyleBox["ContourPlot", "Input"],
  " into what we need. ",
  StyleBox["Contours", "Input"],
  " is an option that lists the specific contours to plot. For example, the \
following example specifically draws contours that are the solutions to ",
  Cell[BoxData[
      \(TraditionalForm\`4  x\^2 - y = 1\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`4  x\^2 - y = 2\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`4  x\^2 - y = 3\)]],
  ". Note that by default ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " shades the areas between the contour lines."
}], "Text"],

Cell[BoxData[
    \(\(ContourPlot[4\ x\^2 - y, {x, \(-3\), 3}, {y, \(-3\), 3}, \n\t
      Contours \[Rule] {1, 2, 3}]; \)\)], "Input"],

Cell[TextData[{
  "When we tell ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to draw only the contour at 0, the resulting graph represents ",
  Cell[BoxData[
      \(TraditionalForm\`4  x\^2 - y < 0\)]],
  ", which can be rewritten ",
  Cell[BoxData[
      \(TraditionalForm\`4  x\^2 < y\)]],
  "."
}], "Text"],

Cell[BoxData[
    \(\(ContourPlot[4\ x\^2 - y, {x, \(-3\), 3}, {y, \(-3\), 3}, \n\t
      Contours \[Rule] {0.}, PlotPoints \[Rule] 50]; \)\)], "Input"],

Cell[TextData[{
  "What remains is to define a new function that accepts inequalities as \
input and translates that format into a standard ",
  StyleBox["ContourPlot", "Input"],
  " command, and then to add appropriate options to our new function. We'll \
call our new function ",
  StyleBox["InequalityPlot", "Input"],
  ", and we will want it to transform something that looks like ",
  StyleBox["a",
    FontSlant->"Italic"],
  " < ",
  StyleBox["b",
    FontSlant->"Italic"],
  " into ",
  StyleBox["a",
    FontSlant->"Italic"],
  " - ",
  StyleBox["b",
    FontSlant->"Italic"],
  ". The prefix form of ",
  StyleBox["a",
    FontSlant->"Italic"],
  " < ",
  StyleBox["b",
    FontSlant->"Italic"],
  " is ",
  StyleBox["Less[", "Input",
    FontWeight->"Bold"],
  StyleBox["a",
    FontSlant->"Italic"],
  StyleBox[",", "Input"],
  StyleBox["b",
    FontSlant->"Italic"],
  StyleBox["]", "Input"],
  ", so we tell ",
  StyleBox["InequalityPlot", "Input"],
  " to expect input in this form. We also require the user to give an ",
  StyleBox["x",
    FontSlant->"Italic"],
  "- and ",
  StyleBox["y",
    FontSlant->"Italic"],
  "-range to the function, which is passed on to ",
  StyleBox["ContourPlot", "Input"],
  ". To give the user control over all of the ",
  StyleBox["ContourPlot", "Input"],
  " options, we include an argument for arbitrary options."
}], "Text"],

Cell[BoxData[
    \(\(InequalityPlot[a_ < b_, xrange_List, yrange_List, opts___] := 
      ContourPlot[a - b, xrange, yrange, \n\t\tPlotPoints \[Rule] 45, 
        Contours \[Rule] {0.}, opts]; \)\)], "Input"],

Cell[BoxData[
    \(\(InequalityPlot[x\^2 < y, {x, \(-3\), 3}, {y, \(-3\), 3}]; \)\)], 
  "Input"],

Cell[TextData[
"Creating our own coloring function that will return gray inside the area \
bounded by the curve and white outside will make this start to look like a \
\[OpenCurlyDoubleQuote]finished\[CloseCurlyDoubleQuote] graphic."], "Text"],

Cell[BoxData[
    \(Clear[InequalityPlot, mycolorfunction]; 
    mycolorfunction[x_] := If[x == 0, GrayLevel[0.5], GrayLevel[1]]; \n\n
    InequalityPlot[a_ < b_, xrange_List, yrange_List, opts___] := 
      ContourPlot[a - b, xrange, yrange, \n\t\tPlotPoints \[Rule] 45, 
        Contours \[Rule] {0.}, ColorFunction \[Rule] mycolorfunction, opts]; 
    \)], "Input"],

Cell[BoxData[
    \(\(InequalityPlot[x\^2 < y, {x, \(-3\), 3}, {y, \(-3\), 3}]; \)\)], 
  "Input"],

Cell["\<\
To make this look more like the inequalities we graphed in school, \
we should move the axes to the center of the graph and remove the frame. \
\
\>", "Text"],

Cell[BoxData[
    \(Clear[InequalityPlot, mycolorfunction]; 
    mycolorfunction[x_] := If[x == 0, GrayLevel[0.5], GrayLevel[1]]; \n\n
    InequalityPlot[a_ < b_, xrange_List, yrange_List, opts___] := \n\t
      ContourPlot[a - b, xrange, yrange, \n\t\tPlotPoints \[Rule] 45, \ 
        Contours \[Rule] {0.}, \n\t\tColorFunction \[Rule] mycolorfunction, 
        Frame \[Rule] False, \n\t\tAxes \[Rule] True, 
        AxesOrigin \[Rule] {0, 0}, opts]; \)], "Input"],

Cell[BoxData[
    \(\(InequalityPlot[x\^2 < y, {x, \(-3\), 3}, {y, \(-3\), 3}]; \)\)], 
  "Input"],

Cell[TextData[{
  "This didn\[CloseCurlyQuote]t quite work as expected, since the axes are \
now hidden behind the graph. Recall that ",
  StyleBox["FilledPlot", "Input"],
  " has an option ",
  StyleBox["AxesFront", "Input"],
  " that takes care of this problem. Looking at the package we see that what \
the author did was define the option ",
  StyleBox["AxesFront", "Input"],
  " for ",
  StyleBox["Graphics", "Input"],
  ". If we load this package, we can make use of this solution for our \
function. Since ",
  StyleBox["AxesFront", "Input"],
  " is an option for graphics functions and not plotting functions, we must \
rearrange our function to create a graphics object and then use ",
  StyleBox["Show", "Input"],
  " to display it with the ",
  StyleBox["AxesFront", "Input"],
  " option set to ",
  StyleBox["True", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`FilledPlot`\>"]\)], "Input"],

Cell[BoxData[
    \(Clear[InequalityPlot, mycolorfunction]; 
    mycolorfunction[x_] := If[x == 0, GrayLevel[0.5], GrayLevel[1]]; \n\n
    InequalityPlot[a_ < b_, xrange_List, yrange_List, opts___] := 
      Show[Graphics[
          ContourPlot[a - b, xrange, yrange, \n\t\tPlotPoints \[Rule] 45, 
            Contours \[Rule] {0.}, \n\t\t
            ColorFunction \[Rule] mycolorfunction, Frame \[Rule] False, \n\t\t
            Axes \[Rule] True, AxesOrigin \[Rule] {0, 0}, \n\t\t
            DisplayFunction \[Rule] Identity, opts]], \n\t
        AxesFront \[Rule] True, DisplayFunction \[Rule] $DisplayFunction]; 
    \)], "Input"],

Cell[BoxData[
    \(\(InequalityPlot[x\^2 < y, {x, \(-3\), 3}, {y, \(-3\), 3}]; \)\)], 
  "Input"],

Cell[TextData[{
  "Note that we haven\[CloseCurlyQuote]t given the function a way to accept \
other forms, such as ",
  Cell[BoxData[
      \(TraditionalForm\`a > b\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`a \[LessEqual] b\)]],
  ", etc. Of course, the user can insert a minus sign in the appropriate \
place, but we can easily automate this feature by providing several cases in \
our definition. Using patterns so that ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " treats various inputs differently is one of the most powerful techniques \
available within ",
  StyleBox["Mathematica.",
    FontSlant->"Italic"],
  " We also customize the way the curve is drawn using ",
  StyleBox["ContourStyle", "Input"],
  ", drawing a dashed line when \[OpenCurlyDoubleQuote]<\
\[CloseCurlyDoubleQuote] is used and solid line when \[OpenCurlyDoubleQuote]\
\[LessEqual]\[CloseCurlyDoubleQuote] is used. "
}], "Text"]
}, Closed]],

Cell[CellGroupData[{

Cell["Final InequalityPlot Code", "Subsubsection"],

Cell[BoxData[
    \(Needs["\<Graphics`Master`\>"]\)], "Input"],

Cell[BoxData[
    \(Clear[InequalityPlot, mycolorfunction]; 
    mycolorfunction[x_] := If[x == 0, GrayLevel[0.5], GrayLevel[1]]; \n\n
    SetOptions[ContourPlot, \n\t\t\ PlotPoints \[Rule] 45, \n\t\t\ 
      Contours \[Rule] {0.}, \n\t\t\ Frame \[Rule] False, \n\t\t\ 
      Axes \[Rule] True, \n\t\t\ AxesOrigin \[Rule] {0, 0}, \n\t\t\ 
      DisplayFunction \[Rule] Identity]; \n\n
    InequalityPlot[a_ < b_, xrange_List, yrange_List, opts___] := 
      Show[Graphics[
          ContourPlot[a - b, xrange, yrange, opts, \n\t\t
            ColorFunction \[Rule] mycolorfunction, \n\t\t
            ContourStyle 
              \[Rule] {Thickness[0.02], RGBColor[1, 0, 0], 
                Dashing[{0.04}]}]], opts, AxesFront \[Rule] True, 
        DisplayFunction \[Rule] $DisplayFunction]; \n\n
    InequalityPlot[a_ > b_, xrange_List, yrange_List, opts___] := 
      Show[Graphics[
          ContourPlot[\(-\((a - b)\)\), xrange, yrange, opts, \n\t\t
            ColorFunction \[Rule] mycolorfunction, \n\t\t
            ContourStyle 
              \[Rule] {Thickness[0.02], RGBColor[1, 0, 0], 
                Dashing[{0.04}]}]], opts, AxesFront \[Rule] True, 
        DisplayFunction \[Rule] $DisplayFunction]; \n\n
    InequalityPlot[a_ \[GreaterEqual] b_, xrange_List, yrange_List, opts___]
         := \n\tShow[
        Graphics[
          ContourPlot[\(-\((a - b)\)\), xrange, yrange, opts, \n\t\t
            ColorFunction \[Rule] mycolorfunction, \n\t\t
            ContourStyle \[Rule] {Thickness[0.02], RGBColor[1, 0, 0]}]], 
        opts, AxesFront \[Rule] True, 
        DisplayFunction \[Rule] $DisplayFunction]; \n\n
    InequalityPlot[a_ \[LessEqual] b_, xrange_List, yrange_List, opts___] := 
      \n\tShow[Graphics[
          ContourPlot[a - b, xrange, yrange, opts, \n\t\t
            ColorFunction \[Rule] mycolorfunction, \n\t\t
            ContourStyle \[Rule] {Thickness[0.02], RGBColor[1, 0, 0]}]], 
        opts, AxesFront \[Rule] True, 
        DisplayFunction \[Rule] $DisplayFunction]; \n\n
    InequalityPlot[a_ == b_, xrange_List, yrange_List, opts___] := 
      Show[Graphics[
          ContourPlot[a - b, xrange, yrange, opts, \n\t\t
            ContourStyle \[Rule] {Thickness[0.02], RGBColor[1, 0, 0]}]], 
        opts, AxesFront \[Rule] True, 
        DisplayFunction \[Rule] $DisplayFunction]; \n\n
    equalcolorfunction[x_] := GrayLevel[1]; \n\n
    InequalityPlot[a_ == b_, xrange_List, yrange_List, opts___] := 
      Show[Graphics[
          ContourPlot[a - b, xrange, yrange, opts, \n\t\t
            ContourStyle \[Rule] {Thickness[0.02], RGBColor[1, 0, 0]}, \n\t\t
            ColorFunction \[Rule] equalcolorfunction]], opts, 
        AxesFront \[Rule] True, DisplayFunction \[Rule] $DisplayFunction]; \n
    \nnotequalcolorfunction[x_] := GrayLevel[.5]; \n\n
    InequalityPlot[a_ \[NotEqual] b_, xrange_List, yrange_List, opts___] := \n
      \tShow[Graphics[
          ContourPlot[a - b, xrange, yrange, opts, \n\t\t
            ColorFunction \[Rule] notequalcolorfunction, \n\t\t
            ContourStyle \[Rule] {Thickness[0.02], GrayLevel[1]}]], opts, 
        AxesFront \[Rule] True, DisplayFunction \[Rule] $DisplayFunction]; 
    \)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Examples", "Subsubsection"],

Cell[BoxData[
    \(\(InequalityPlot[x \[LessEqual] 2\ y, {x, \(-3\), 3}, {y, \(-3\), 3}]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(InequalityPlot[
      x\^2 \[LessEqual] \(-y\^2\) + 25, {x, \(-10\), 10}, {y, \(-10\), 10}]; 
    \)\)], "Input"],

Cell[BoxData[
    \(\(variations = \n
      \t{{InequalityPlot[
            x\^2 \[LessEqual] y\^2 - 20, {x, \(-10\), 10}, {y, \(-10\), 10}, 
            \n\t\t\t\tDisplayFunction \[Rule] Identity], \n\t\t\t
          InequalityPlot[
            x\^2 < y\^2 - 20, {x, \(-10\), 10}, {y, \(-10\), 10}, \n\t\t\t\t
            DisplayFunction \[Rule] Identity], \n\t\t\t
          InequalityPlot[
            x\^2 > y\^2 - 20, {x, \(-10\), 10}, {y, \(-10\), 10}, \n\t\t\t\t
            DisplayFunction \[Rule] Identity]}, \n
        \t\t{InequalityPlot[
            x\^2 \[GreaterEqual] y\^2 - 20, {x, \(-10\), 10}, {y, \(-10\), 
              10}, \n\t\t\t\tDisplayFunction \[Rule] Identity], \n\t\t\t
          InequalityPlot[
            x\^2 \[NotEqual] y\^2 - 20, {x, \(-10\), 10}, {y, \(-10\), 10}, \n
            \t\t\t\tDisplayFunction \[Rule] Identity], \n\t\t\t
          InequalityPlot[
            x\^2 == y\^2 - 20, {x, \(-10\), 10}, {y, \(-10\), 10}, \n\t\t\t\t
            DisplayFunction \[Rule] Identity]}}; \)\)], "Input"],

Cell[BoxData[
    \(\(Show[GraphicsArray[variations]]; \)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["RiemannPlot", "Subsection"],

Cell[CellGroupData[{

Cell["Code", "Subsubsection"],

Cell[BoxData[
    \(Needs["\<Utilities`FilterOptions`\>"]\)], "Input"],

Cell[BoxData[
    \(\(RiemannPlot::usage\  = \ 
      "\<RiemannPlot[f, {x, xmin, xmax}] generates a plot of f as a function \
of x from xmin to xmax, with the area under the curve approximated by \
rectangles. \>"; \)\)], "Input"],

Cell[BoxData[
    \(\(Options[RiemannPlot]\  = \n
      \t{Subdivisions\  -> \ 16, \n\t\tShowCurve\  -> \ True, \n\t\t
          FillColor\  -> GrayLevel[0.7], Method -> Left, \n\t\t
          BorderStyle -> \ GrayLevel[0]}~Join~Options[Plot]; \)\)], "Input"],

Cell[BoxData[
    \(\(SetOptions[RiemannPlot, PlotRange \[Rule] All, \n\t
      TextStyle 
        \[Rule] {FontFamily \[Rule] "\<Helvetica\>", 
          FontSlant \[Rule] "\<Oblique\>", FontSize \[Rule] 10}]; \)\)], 
  "Input"],

Cell[BoxData[
    \(Clear[RiemannPlot]; \ 
    RiemannPlot[func_, \ {var_, \ min_, \ max_}, \n\topts___?OptionQ]\  := \n
      \tModule[{rects, \ inc, \ subs, \ sc, fillc, outlines, n, meth, \ disp, 
          \ area = 0}, \n
        \t\t{subs, \ sc, fillc, \ outlines, meth, disp} = 
          \({Subdivisions, \ ShowCurve, FillColor, \ BorderStyle, Method, 
                DisplayFunction} /. Flatten[{opts}]\) /. 
            Options[RiemannPlot]; \n\t\tinc\  = \ \((max\  - \ min)\)/subs; \n
        \ \ \ \ \ \ \ 
        adj = Switch[meth, Left, 0, Right, inc, Midpoint, inc/2]; \n\t\t\t
        rects\  = \ 
          Flatten[Table[\n\t\t\t\t
              \(\((area = area + #\ inc; {fillc, 
                      Rectangle[{n, 0}, {n + inc, #}], outlines, 
                      Line[{{n, 0}, {n + inc, 0}, {n + inc, #}, {n, #}, {n, 
                            0}}]})\)&\)[func /. var \[Rule] \((n + adj)\)], {
                n, min, max - \ inc, inc}]\ \ \ ]; \n\t\t\n\t\t\t\t
        If[TrueQ[sc], \n\t\t\t
          Show[{Graphics[rects], 
              Plot[func, \ {var, min, max}, DisplayFunction\  -> \ Identity, 
                Evaluate[
                  \(FilterOptions[Plot, \ ##]&\)\ @@\ 
                    Flatten[{opts, Options[RiemannPlot]}]]]}, \n\t\t\t\t
            \(FilterOptions[Graphics, ##]&\)@@
              Flatten[{DisplayFunction\  \[Rule] \ disp, opts, \n\t\t\t\t\t
                  PlotLabel\  \[Rule] \ 
                    "\<Approximate Area = \>"<>ToString[area]<>
                      "\<\n Calculated Area = \>"<>
                      ToString[NIntegrate[func, \ {var, \ min, \ max}]], 
                  Options[RiemannPlot]}]\n\t\t\t], \n\t\t\t\t
          Show[Graphics[rects], 
            \(FilterOptions[Graphics, ##]&\)@@
              Flatten[{opts, \n\t\t\t\t\t\t\t
                  PlotLabel\  \[Rule] \ 
                    "\<Approximate Area = \>"<>ToString[area]<>
                      "\<\n Calculated Area = \>"<>
                      ToString[NIntegrate[func, \ {var, \ min, \ max}]], 
                  Options[RiemannPlot]}]\n\t\t\t\t]\n\t\t]\n]\)], "Input"]
}, Closed]],

Cell[CellGroupData[{

Cell["Examples", "Subsubsection"],

Cell[BoxData[
    \(Needs["\<Graphics`Master`\>"]\)], "Input"],

Cell[BoxData[
    \(\(?RiemannPlot\)\)], "Input"],

Cell[BoxData[
    \(Options[\ RiemannPlot]\)], "Input"],

Cell[TextData[{
  StyleBox["Method", "Input"],
  " can be ",
  StyleBox["Midpoint", "Input"],
  ", ",
  StyleBox["Left", "Input"],
  ", or ",
  StyleBox["Right", "Input"],
  ". It defaults to ",
  StyleBox["Left", "Input"],
  "."
}], "Text"],

Cell[BoxData[
    \(\(RiemannPlot[Sin[x]\/x, {x, 0.001, \[Pi]}, \n\tFillColor -> Red, \n\t
      BorderStyle -> {Blue}, \n\tPlotStyle -> {Thickness[0.01], Magenta}, \n\t
      Method -> Right, \n\t\ Subdivisions -> 40]; \)\)], "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell["Building Primitives", "Subsection"],

Cell[TextData[{
  "Using similar techniques we can easily build our own graphics primitives. \
As a simple example, we can create a function that returns an ",
  StyleBox["n",
    FontSlant->"Italic"],
  "-sided polygon."
}], "Text"],

Cell[BoxData[
    \(poly[n_Integer] := \n\t
      Polygon[Table[{Cos[\(i\ 2\ \[Pi]\)\/n], \ Sin[\(i\ 2\ \[Pi]\)\/n]}, {i, 
            1, n}]]\)], "Input"],

Cell[BoxData[
    \(\(Show[\ Graphics[\ poly[5]\ ], \ AspectRatio\  -> Automatic\ ]; \)\)], 
  "Input"],

Cell[TextData[{
  "We can easily rewrite ",
  StyleBox["poly", "Input"],
  " to accept the position of the center of the polygon."
}], "Text"],

Cell[BoxData[
    \(poly[{x_, y_}, n_Integer] := \n\t
      Polygon[Table[{Cos[\(i\ 2\ \[Pi]\)\/n] + x, \ 
            Sin[\(i\ 2\ \[Pi]\)\/n] + y}, {i, 1, n}]]\)], "Input"],

Cell[BoxData[
    \(\(Show[\ 
      Graphics[\ \n
        \t{Hue[0], poly[{\(-1\), 1}, 3], Hue[0.5], \ poly[{2, 3}, 5]}\ ], \ \n
      \tAspectRatio\  -> Automatic\ ]; \)\)], "Input"]
}, Closed]]
}, Closed]]
},
FrontEndVersion->"Microsoft Windows 3.0",
ScreenRectangle->{{0, 800}, {0, 540}},
ScreenStyleEnvironment->"Working",
WindowToolbars->{},
WindowSize->{550, 437},
WindowMargins->{{2, Automatic}, {Automatic, 5}},
PrintingCopies->1,
PrintingPageRange->{168, Automatic},
PageHeaders->{{Cell[ 
        TextData[ {
          CounterBox[ "Page"]}], "PageNumber"], Inherited, Cell[ 
      "Mathematica Graphics", "Header"]}, {
    Cell[ "Mathematica Graphics", "Header"], Inherited, Cell[ 
        TextData[ {
          CounterBox[ "Page"]}], "PageNumber"]}},
PrintingOptions->{"FirstPageHeader"->False,
"FirstPageFooter"->False,
"FacingPages"->True},
StyleDefinitions -> "Classroom.nb"
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "Plot"->{
    Cell[2029, 67, 49, 1, 37, "Subsubsection",
      CellTags->"Plot"],
    Cell[2081, 70, 407, 12, 68, "Text",
      CellTags->"Plot"],
    Cell[2491, 84, 65, 1, 50, "Input",
      CellTags->"Plot"],
    Cell[2559, 87, 147, 4, 30, "Text",
      CellTags->"Plot"],
    Cell[2709, 93, 105, 2, 50, "Input",
      CellTags->"Plot"]},
  "Options"->{
    Cell[2851, 100, 55, 1, 29, "Subsubsection",
      CellTags->"Options"],
    Cell[2909, 103, 458, 14, 53, "Text",
      CellTags->"Options"],
    Cell[3370, 119, 69, 2, 26, "Input",
      CellTags->"Options"]},
  "PlotRange"->{
    Cell[3442, 123, 447, 15, 30, "Text",
      CellTags->"PlotRange"],
    Cell[3972, 143, 176, 5, 20, "Text",
      CellTags->"PlotRange"],
    Cell[4151, 150, 143, 3, 29, "Input",
      CellTags->"PlotRange"],
    Cell[4810, 170, 316, 5, 63, "Input",
      CellTags->{
      "PlotRange", "GridLines", "AspectRatio", "Frame", "PlotLabel"}]},
  "GridLines"->{
    Cell[4810, 170, 316, 5, 63, "Input",
      CellTags->{
      "PlotRange", "GridLines", "AspectRatio", "Frame", "PlotLabel"}]},
  "AspectRatio"->{
    Cell[4810, 170, 316, 5, 63, "Input",
      CellTags->{
      "PlotRange", "GridLines", "AspectRatio", "Frame", "PlotLabel"}],
    Cell[6196, 217, 429, 12, 30, "Text",
      CellTags->"AspectRatio"],
    Cell[6628, 231, 174, 3, 26, "Input",
      CellTags->{"AspectRatio", "ParametricPlot"}],
    Cell[7068, 245, 233, 4, 38, "Input",
      CellTags->{"ParametricPlot", "AspectRatio"}],
    Cell[7530, 261, 313, 5, 51, "Input",
      CellTags->{"AspectRatio", "Axes", "Frame", "FrameLabel"}]},
  "Frame"->{
    Cell[4810, 170, 316, 5, 63, "Input",
      CellTags->{
      "PlotRange", "GridLines", "AspectRatio", "Frame", "PlotLabel"}],
    Cell[7530, 261, 313, 5, 51, "Input",
      CellTags->{"AspectRatio", "Axes", "Frame", "FrameLabel"}]},
  "PlotLabel"->{
    Cell[4810, 170, 316, 5, 63, "Input",
      CellTags->{
      "PlotRange", "GridLines", "AspectRatio", "Frame", "PlotLabel"}]},
  "ParametricPlot"->{
    Cell[5473, 194, 69, 1, 29, "Subsubsection",
      CellTags->"ParametricPlot"],
    Cell[5545, 197, 526, 14, 41, "Text",
      CellTags->"ParametricPlot"],
    Cell[6074, 213, 119, 2, 26, "Input",
      CellTags->"ParametricPlot"],
    Cell[6628, 231, 174, 3, 26, "Input",
      CellTags->{"AspectRatio", "ParametricPlot"}],
    Cell[6805, 236, 260, 7, 30, "Text",
      CellTags->"ParametricPlot"],
    Cell[7068, 245, 233, 4, 38, "Input",
      CellTags->{"ParametricPlot", "AspectRatio"}],
    Cell[7304, 251, 223, 8, 20, "Text",
      CellTags->"ParametricPlot"]},
  "Axes"->{
    Cell[7530, 261, 313, 5, 51, "Input",
      CellTags->{"AspectRatio", "Axes", "Frame", "FrameLabel"}]},
  "FrameLabel"->{
    Cell[7530, 261, 313, 5, 51, "Input",
      CellTags->{"AspectRatio", "Axes", "Frame", "FrameLabel"}]},
  "S6.4.1"->{
    Cell[110967, 3814, 584, 9, 110, "Input",
      CellTags->"S6.4.1"],
    Cell[111554, 3825, 190, 4, 50, "Input",
      CellTags->"S6.4.1"],
    Cell[111747, 3831, 564, 11, 150, "Input",
      CellTags->"S6.4.1"]},
  "Hue"->{
    Cell[116467, 4025, 195, 7, 23, "Text",
      CellTags->"Hue"],
    Cell[116665, 4034, 237, 6, 71, "Input",
      CellTags->"Hue"],
    Cell[116905, 4042, 91, 2, 23, "Input",
      CellTags->"Hue"],
    Cell[170474, 4969, 243, 6, 105, "Input",
      CellTags->"Hue"],
    Cell[170720, 4977, 234, 6, 90, "Input",
      CellTags->"Hue"]},
  "RasterArray"->{
    Cell[180141, 5341, 144, 3, 39, "Input",
      CellTags->"RasterArray"]},
  "EdgeForm"->{
    Cell[186642, 5552, 285, 6, 103, "Input",
      CellTags->"EdgeForm"]},
  "S2.8.9"->{
    Cell[194695, 5824, 164, 4, 50, "Input",
      CellTags->"S2.8.9"]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"Plot", 629283, 14452},
  {"Options", 629629, 14463},
  {"PlotRange", 629860, 14470},
  {"GridLines", 630220, 14480},
  {"AspectRatio", 630370, 14484},
  {"Frame", 630879, 14496},
  {"PlotLabel", 631133, 14502},
  {"ParametricPlot", 631286, 14506},
  {"Axes", 631870, 14521},
  {"FrameLabel", 631995, 14524},
  {"S6.4.1", 632116, 14527},
  {"Hue", 632344, 14534},
  {"RasterArray", 632703, 14545},
  {"EdgeForm", 632796, 14548},
  {"S2.8.9", 632885, 14551}
  }
*)

(*NotebookFileOutline
Notebook[{
Cell[1709, 49, 94, 4, 56, "Title"],

Cell[CellGroupData[{
Cell[1828, 57, 105, 2, 56, "Section",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[1958, 63, 46, 0, 46, "Subsection"],

Cell[CellGroupData[{
Cell[2029, 67, 49, 1, 37, "Subsubsection",
  CellTags->"Plot"],
Cell[2081, 70, 407, 12, 68, "Text",
  CellTags->"Plot"],
Cell[2491, 84, 65, 1, 50, "Input",
  CellTags->"Plot"],
Cell[2559, 87, 147, 4, 30, "Text",
  CellTags->"Plot"],
Cell[2709, 93, 105, 2, 50, "Input",
  CellTags->"Plot"]
}, Closed]],

Cell[CellGroupData[{
Cell[2851, 100, 55, 1, 29, "Subsubsection",
  CellTags->"Options"],
Cell[2909, 103, 458, 14, 53, "Text",
  CellTags->"Options"],
Cell[3370, 119, 69, 2, 26, "Input",
  CellTags->"Options"],
Cell[3442, 123, 447, 15, 30, "Text",
  CellTags->"PlotRange"],
Cell[3892, 140, 77, 1, 29, "Input"],
Cell[3972, 143, 176, 5, 20, "Text",
  CellTags->"PlotRange"],
Cell[4151, 150, 143, 3, 29, "Input",
  CellTags->"PlotRange"],
Cell[4297, 155, 510, 13, 41, "Text"],
Cell[4810, 170, 316, 5, 63, "Input",
  CellTags->{"PlotRange", "GridLines", "AspectRatio", "Frame", "PlotLabel"}],
Cell[5129, 177, 129, 2, 20, "Text"],
Cell[5261, 181, 175, 8, 36, "SeeAlso"]
}, Closed]],

Cell[CellGroupData[{
Cell[5473, 194, 69, 1, 29, "Subsubsection",
  CellTags->"ParametricPlot"],
Cell[5545, 197, 526, 14, 41, "Text",
  CellTags->"ParametricPlot"],
Cell[6074, 213, 119, 2, 26, "Input",
  CellTags->"ParametricPlot"],
Cell[6196, 217, 429, 12, 30, "Text",
  CellTags->"AspectRatio"],
Cell[6628, 231, 174, 3, 26, "Input",
  CellTags->{"AspectRatio", "ParametricPlot"}],
Cell[6805, 236, 260, 7, 30, "Text",
  CellTags->"ParametricPlot"],
Cell[7068, 245, 233, 4, 38, "Input",
  CellTags->{"ParametricPlot", "AspectRatio"}],
Cell[7304, 251, 223, 8, 20, "Text",
  CellTags->"ParametricPlot"],
Cell[7530, 261, 313, 5, 51, "Input",
  CellTags->{"AspectRatio", "Axes", "Frame", "FrameLabel"}]
}, Closed]],

Cell[CellGroupData[{
Cell[7880, 271, 37, 0, 29, "Subsubsection"],
Cell[7920, 273, 361, 10, 30, "Text"],
Cell[8284, 285, 64, 1, 26, "Input"],
Cell[8351, 288, 2446, 101, 54, "Text"],
Cell[10800, 391, 141, 2, 46, "Input"],
Cell[10944, 395, 166, 4, 20, "Text"],
Cell[11113, 401, 254, 4, 62, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[11404, 410, 60, 0, 29, "Subsubsection"],
Cell[11467, 412, 357, 9, 41, "Text"],
Cell[11827, 423, 349, 11, 30, "Text"],
Cell[12179, 436, 511, 11, 52, "Text"],
Cell[12693, 449, 102, 2, 26, "Input"],
Cell[12798, 453, 1189, 40, 63, "Text"],
Cell[13990, 495, 110, 2, 26, "Input"],
Cell[14103, 499, 1081, 47, 41, "Text"],
Cell[15187, 548, 596, 20, 41, "Text"],
Cell[15786, 570, 417, 10, 41, "Text"],
Cell[16206, 582, 135, 2, 38, "Input"],
Cell[16344, 586, 163, 4, 30, "Text"],
Cell[16510, 592, 182, 3, 51, "Input"],
Cell[16695, 597, 311, 8, 30, "Text"],
Cell[17009, 607, 144, 3, 38, "Input"],
Cell[17156, 612, 320, 13, 46, "SeeAlso"]
}, Closed]],

Cell[CellGroupData[{
Cell[17513, 630, 41, 0, 29, "Subsubsection"],
Cell[17557, 632, 236, 6, 53, "Text"],
Cell[17796, 640, 559, 17, 73, "Text"],
Cell[18358, 659, 86, 1, 30, "Input"],
Cell[18447, 662, 174, 5, 53, "Text"],
Cell[18624, 669, 162, 3, 50, "Input"],
Cell[18789, 674, 179, 5, 53, "Text"],
Cell[18971, 681, 133, 3, 30, "Input"],
Cell[19107, 686, 207, 8, 34, "Text"],
Cell[19317, 696, 53, 1, 30, "Input"],
Cell[19373, 699, 144, 4, 34, "Text"],
Cell[19520, 705, 185, 3, 50, "Input"],
Cell[19708, 710, 102, 4, 34, "Text"],
Cell[19813, 716, 53, 1, 30, "Input"],
Cell[19869, 719, 264, 8, 53, "Text"],
Cell[20136, 729, 80, 1, 30, "Input"],
Cell[20219, 732, 181, 8, 26, "SeeAlso"]
}, Closed]],

Cell[CellGroupData[{
Cell[20437, 745, 84, 1, 29, "Subsubsection",
  PageBreakAbove->True],
Cell[20524, 748, 313, 9, 42, "Question"],
Cell[20840, 759, 83, 1, 31, "Input"],
Cell[20926, 762, 431, 14, 42, "Question"],
Cell[21360, 778, 231, 4, 72, "Input"],
Cell[21594, 784, 25, 0, 25, "EndExercises"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[21668, 790, 48, 0, 30, "Subsection"],

Cell[CellGroupData[{
Cell[21741, 794, 31, 0, 43, "Subsubsection"],
Cell[21775, 796, 404, 14, 30, "Text"],
Cell[22182, 812, 88, 1, 26, "Input"],
Cell[22273, 815, 166, 4, 30, "Text"],
Cell[22442, 821, 535, 11, 53, "Text"],
Cell[22980, 834, 71, 0, 20, "Text"],
Cell[23054, 836, 172, 3, 38, "Input"],
Cell[23229, 841, 218, 10, 36, "SeeAlso"]
}, Closed]],

Cell[CellGroupData[{
Cell[23484, 856, 47, 0, 29, "Subsubsection"],
Cell[23534, 858, 392, 7, 41, "Text"],
Cell[23929, 867, 45, 0, 26, "Input"],
Cell[23977, 869, 538, 12, 52, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[24552, 886, 41, 0, 29, "Subsubsection"],
Cell[24596, 888, 408, 13, 54, "Text"],
Cell[25007, 903, 229, 7, 53, "Text"],
Cell[25239, 912, 123, 2, 30, "Input"],
Cell[25365, 916, 102, 3, 33, "Text"],
Cell[25470, 921, 140, 2, 50, "Input"],
Cell[25613, 925, 242, 7, 53, "Text"],
Cell[25858, 934, 213, 3, 70, "Input"],
Cell[26074, 939, 295, 10, 53, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[26406, 954, 86, 1, 29, "Subsubsection",
  PageBreakAbove->True],
Cell[26495, 957, 277, 11, 25, "Question"],
Cell[26775, 970, 97, 2, 30, "Input"],
Cell[26875, 974, 476, 13, 59, "Question"],
Cell[27354, 989, 234, 4, 70, "Input"],
Cell[27591, 995, 25, 0, 25, "EndExercises"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[27665, 1001, 50, 0, 30, "Subsection"],

Cell[CellGroupData[{
Cell[27740, 1005, 52, 0, 43, "Subsubsection"],
Cell[27795, 1007, 362, 10, 30, "Text"],
Cell[28160, 1019, 207, 4, 30, "Text"],
Cell[28370, 1025, 99, 2, 26, "Input"],
Cell[28472, 1029, 200, 4, 30, "Text"],
Cell[28675, 1035, 101, 2, 26, "Input"],
Cell[28779, 1039, 257, 12, 36, "SeeAlso"]
}, Closed]],

Cell[CellGroupData[{
Cell[29073, 1056, 38, 0, 29, "Subsubsection"],
Cell[29114, 1058, 356, 8, 72, "Text"],
Cell[29473, 1068, 51, 0, 30, "Input"],
Cell[29527, 1070, 3200, 143, 73, "Text"],
Cell[32730, 1215, 166, 3, 52, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[32933, 1223, 88, 1, 29, "Subsubsection",
  PageBreakAbove->True],
Cell[33024, 1226, 403, 14, 42, "Question"],
Cell[33430, 1242, 108, 2, 30, "Input"],
Cell[33541, 1246, 177, 3, 50, "Input"],
Cell[33721, 1251, 385, 13, 44, "Question"],
Cell[34109, 1266, 108, 2, 30, "Input"],
Cell[34220, 1270, 189, 3, 50, "Input"],
Cell[34412, 1275, 25, 0, 25, "EndExercises"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[34486, 1281, 35, 0, 30, "Subsection"],
Cell[34524, 1283, 423, 7, 90, "Text"],
Cell[34950, 1292, 130, 3, 30, "Input"],
Cell[35083, 1297, 877, 27, 113, "Text"],
Cell[35963, 1326, 112, 2, 30, "Input"],
Cell[36078, 1330, 381, 12, 54, "Text"],
Cell[36462, 1344, 151, 3, 50, "Input"],
Cell[36616, 1349, 435, 9, 72, "Text"],
Cell[37054, 1360, 79, 1, 31, "Input"],
Cell[37136, 1363, 68, 0, 33, "Text"],
Cell[37207, 1365, 123, 3, 30, "Input"],
Cell[37333, 1370, 58, 0, 33, "Text"],
Cell[37394, 1372, 77, 1, 30, "Input"],
Cell[37474, 1375, 547, 19, 73, "Text"],
Cell[38024, 1396, 161, 3, 50, "Input"],
Cell[38188, 1401, 63, 1, 30, "Input"],
Cell[38254, 1404, 291, 9, 53, "Text"],
Cell[38548, 1415, 68, 1, 30, "Input"],

Cell[CellGroupData[{
Cell[38641, 1420, 73, 1, 43, "Subsubsection",
  PageBreakAbove->True],
Cell[38717, 1423, 472, 18, 43, "Question"],
Cell[39192, 1443, 144, 3, 31, "Input"],
Cell[39339, 1448, 90, 1, 30, "Input"],
Cell[39432, 1451, 691, 20, 79, "Question"],
Cell[40126, 1473, 69, 1, 30, "Input"],
Cell[40198, 1476, 119, 3, 30, "Input"],
Cell[40320, 1481, 71, 1, 31, "Input"],
Cell[40394, 1484, 103, 2, 30, "Input"],
Cell[40500, 1488, 65, 1, 30, "Input"],
Cell[40568, 1491, 25, 0, 25, "EndExercises"],
Cell[40596, 1493, 74, 2, 30, "Input",
  PageBreakAbove->True,
  PageBreakBelow->True]
}, Closed]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[40731, 1502, 36, 0, 36, "Section"],
Cell[40770, 1504, 398, 9, 71, "Text"],
Cell[41171, 1515, 380, 12, 54, "Text"],
Cell[41554, 1529, 43, 1, 30, "Input"],
Cell[41600, 1532, 43, 0, 30, "Input"],
Cell[41646, 1534, 43, 1, 30, "Input"],
Cell[41692, 1537, 320, 7, 71, "Text"],
Cell[42015, 1546, 395, 10, 71, "Text"],
Cell[42413, 1558, 44, 0, 30, "Input"],
Cell[42460, 1560, 222, 6, 52, "Text"],
Cell[42685, 1568, 544, 17, 71, "Text",
  Evaluatable->False],
Cell[43232, 1587, 172, 6, 34, "Text"],
Cell[43407, 1595, 381, 10, 30, "Input"],
Cell[43791, 1607, 105, 3, 33, "Text"],
Cell[43899, 1612, 160, 2, 50, "Input"],
Cell[44062, 1616, 286, 9, 53, "Text"],
Cell[44351, 1627, 68, 1, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[44456, 1633, 48, 0, 36, "Section"],

Cell[CellGroupData[{
Cell[44529, 1637, 28, 0, 47, "Subsection"],
Cell[44560, 1639, 306, 11, 53, "Text",
  Evaluatable->False],
Cell[44869, 1652, 70, 1, 30, "Input"],
Cell[44942, 1655, 375, 13, 54, "Text",
  Evaluatable->False],
Cell[45320, 1670, 61, 1, 30, "Input"],
Cell[45384, 1673, 1217, 54, 72, "Text",
  Evaluatable->False],
Cell[46604, 1729, 215, 8, 34, "Text",
  Evaluatable->False],
Cell[46822, 1739, 105, 2, 30, "Input"],
Cell[46930, 1743, 363, 13, 53, "Text"],
Cell[47296, 1758, 90, 1, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[47423, 1764, 31, 0, 31, "Subsection"],
Cell[47457, 1766, 876, 36, 63, "Text"],
Cell[48336, 1804, 65, 1, 26, "Input"],
Cell[48404, 1807, 387, 12, 30, "Text"],
Cell[48794, 1821, 42, 1, 26, "Input"],
Cell[48839, 1824, 800, 13, 151, "Input"],
Cell[49642, 1839, 423, 20, 20, "Text"],
Cell[50068, 1861, 217, 4, 76, "Input"],
Cell[50288, 1867, 322, 10, 30, "Text"],
Cell[50613, 1879, 50, 1, 26, "Input"],
Cell[50666, 1882, 252, 5, 63, "Input"],
Cell[50921, 1889, 276, 10, 30, "Text"],
Cell[51200, 1901, 391, 6, 76, "Input"],
Cell[51594, 1909, 639, 20, 52, "Text"],
Cell[52236, 1931, 421, 9, 88, "Input"],
Cell[52660, 1942, 1250, 38, 96, "Text"],
Cell[53913, 1982, 97, 2, 26, "Input"],
Cell[54013, 1986, 166, 2, 38, "Input"],
Cell[54182, 1990, 219, 5, 30, "Text"],
Cell[54404, 1997, 64, 1, 26, "Input"],
Cell[54471, 2000, 60, 1, 26, "Input"],
Cell[54534, 2003, 64, 1, 26, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[54635, 2009, 27, 0, 31, "Subsection"],
Cell[54665, 2011, 1069, 32, 74, "Text"],
Cell[55737, 2045, 61, 1, 26, "Input"],
Cell[55801, 2048, 80, 1, 26, "Input"],
Cell[55884, 2051, 248, 5, 51, "Input"],
Cell[56135, 2058, 304, 6, 88, "Input"],
Cell[56442, 2066, 158, 6, 20, "Text"],
Cell[56603, 2074, 166, 3, 38, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[56806, 2082, 28, 0, 31, "Subsection"],
Cell[56837, 2084, 457, 13, 73, "Text"],
Cell[57297, 2099, 44, 0, 30, "Input"],

Cell[CellGroupData[{
Cell[57366, 2103, 49, 0, 43, "Subsubsection"],
Cell[57418, 2105, 791, 29, 30, "Text"],
Cell[58212, 2136, 136, 2, 38, "Input"],
Cell[58351, 2140, 74, 0, 20, "Text"],
Cell[58428, 2142, 201, 3, 51, "Input"],
Cell[58632, 2147, 304, 12, 20, "Text"],
Cell[58939, 2161, 513, 8, 151, "Input"],
Cell[59455, 2171, 1592, 60, 74, "Text"],
Cell[61050, 2233, 1041, 17, 263, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[62128, 2255, 59, 0, 29, "Subsubsection"],
Cell[62190, 2257, 2572, 101, 132, "Text"],
Cell[64765, 2360, 347, 10, 54, "Text"],
Cell[65115, 2372, 274, 8, 138, "Input"],
Cell[65392, 2382, 315, 5, 114, "Input"],
Cell[65710, 2389, 44, 0, 30, "Input"],
Cell[65757, 2391, 56, 0, 30, "Input"],
Cell[65816, 2393, 563, 14, 246, "Input"],
Cell[66382, 2409, 2141, 86, 134, "Text"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[68572, 2501, 83, 1, 31, "Subsection",
  PageBreakAbove->True],
Cell[68658, 2504, 342, 9, 59, "Question"],
Cell[69003, 2515, 44, 0, 30, "Input"],
Cell[69050, 2517, 619, 21, 59, "Question"],
Cell[69672, 2540, 26, 0, 30, "Input"],
Cell[69701, 2542, 90, 1, 30, "Input"],
Cell[69794, 2545, 99, 2, 30, "Input"],
Cell[69896, 2549, 29, 0, 30, "Input"],
Cell[69928, 2551, 167, 4, 84, "Input"],
Cell[70098, 2557, 29, 0, 30, "Input"],
Cell[70130, 2559, 153, 3, 70, "Input"],
Cell[70286, 2564, 53, 0, 25, "Question"],
Cell[70342, 2566, 113, 3, 25, "Question"],
Cell[70458, 2571, 185, 3, 72, "Input"],
Cell[70646, 2576, 143, 4, 26, "Question"],
Cell[70792, 2582, 453, 8, 172, "Input"],
Cell[71248, 2592, 21, 0, 25, "Special3"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[71318, 2598, 46, 0, 36, "Section"],
Cell[71367, 2600, 576, 16, 90, "Text"],
Cell[71946, 2618, 423, 10, 71, "Text"],

Cell[CellGroupData[{
Cell[72394, 2632, 30, 0, 47, "Subsection"],
Cell[72427, 2634, 413, 9, 90, "Text"],

Cell[CellGroupData[{
Cell[72865, 2647, 34, 0, 43, "Subsubsection"],
Cell[72902, 2649, 337, 10, 30, "Text"],
Cell[73242, 2661, 167, 5, 20, "Text"],
Cell[73412, 2668, 65, 1, 26, "Input"],
Cell[73480, 2671, 131, 3, 26, "Input"],
Cell[73614, 2676, 121, 2, 26, "Input"],
Cell[73738, 2680, 158, 2, 38, "Input"],
Cell[73899, 2684, 207, 3, 57, "Input"],
Cell[74109, 2689, 154, 3, 38, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[74300, 2697, 34, 0, 29, "Subsubsection"],
Cell[74337, 2699, 160, 4, 30, "Text"],
Cell[74500, 2705, 65, 1, 26, "Input"],
Cell[74568, 2708, 84, 1, 40, "Input"],
Cell[74655, 2711, 211, 4, 63, "Input"],
Cell[74869, 2717, 211, 4, 63, "Input"],
Cell[75083, 2723, 225, 5, 63, "Input"],
Cell[75311, 2730, 62, 0, 26, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[75410, 2735, 31, 0, 29, "Subsubsection"],
Cell[75444, 2737, 333, 9, 30, "Text"],
Cell[75780, 2748, 62, 1, 26, "Input"],
Cell[75845, 2751, 185, 5, 30, "Text"],
Cell[76033, 2758, 116, 4, 20, "Text"],
Cell[76152, 2764, 152, 4, 38, "Input"],
Cell[76307, 2770, 161, 4, 20, "Text"],
Cell[76471, 2776, 61, 1, 26, "Input"],
Cell[76535, 2779, 161, 5, 20, "Text"],
Cell[76699, 2786, 312, 7, 38, "Input"],
Cell[77014, 2795, 169, 4, 20, "Text"],
Cell[77186, 2801, 508, 9, 88, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[77731, 2815, 35, 0, 29, "Subsubsection"],
Cell[77769, 2817, 201, 6, 30, "Text"],
Cell[77973, 2825, 66, 1, 26, "Input"],
Cell[78042, 2828, 317, 6, 101, "Input"],
Cell[78362, 2836, 407, 8, 111, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[78806, 2849, 38, 0, 29, "Subsubsection"],
Cell[78847, 2851, 305, 10, 30, "Text"],
Cell[79155, 2863, 69, 1, 26, "Input"],
Cell[79227, 2866, 521, 10, 132, "Input"],
Cell[79751, 2878, 304, 7, 71, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[80092, 2890, 35, 0, 29, "Subsubsection"],
Cell[80130, 2892, 164, 4, 30, "Text"],
Cell[80297, 2898, 66, 1, 26, "Input"],
Cell[80366, 2901, 365, 5, 150, "Input"],
Cell[80734, 2908, 50, 0, 26, "Input"],
Cell[80787, 2910, 476, 9, 101, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[81300, 2924, 33, 0, 29, "Subsubsection"],
Cell[81336, 2926, 193, 4, 30, "Text"],
Cell[81532, 2932, 64, 1, 26, "Input"],
Cell[81599, 2935, 50, 1, 26, "Input"],
Cell[81652, 2938, 67, 1, 26, "Input"],
Cell[81722, 2941, 54, 1, 26, "Input"],
Cell[81779, 2944, 47, 1, 26, "Input"],
Cell[81829, 2947, 483, 8, 138, "Input"],
Cell[82315, 2957, 462, 7, 101, "Input"],
Cell[82780, 2966, 230, 4, 88, "Input"],
Cell[83013, 2972, 205, 4, 69, "Input"],
Cell[83221, 2978, 178, 3, 63, "Input"],
Cell[83402, 2983, 253, 5, 70, "Input"],
Cell[83658, 2990, 176, 3, 51, "Input"],
Cell[83837, 2995, 156, 2, 51, "Input"],
Cell[83996, 2999, 223, 3, 63, "Input"],
Cell[84222, 3004, 71, 1, 26, "Input"],
Cell[84296, 3007, 150, 4, 38, "Input"],
Cell[84449, 3013, 55, 1, 26, "Input"],
Cell[84507, 3016, 212, 4, 70, "Input"],
Cell[84722, 3022, 189, 3, 70, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[84948, 3030, 35, 0, 29, "Subsubsection"],
Cell[84986, 3032, 199, 4, 52, "Text"],
Cell[85188, 3038, 66, 1, 30, "Input"],
Cell[85257, 3041, 292, 6, 112, "Input"],
Cell[85552, 3049, 84, 1, 30, "Input"],
Cell[85639, 3052, 277, 5, 114, "Input"],
Cell[85919, 3059, 276, 5, 76, "Input"],
Cell[86198, 3066, 73, 1, 26, "Input"],
Cell[86274, 3069, 312, 5, 76, "Input"],
Cell[86589, 3076, 130, 3, 38, "Input"],
Cell[86722, 3081, 468, 8, 88, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[87227, 3094, 37, 0, 29, "Subsubsection"],
Cell[87267, 3096, 345, 9, 30, "Text"],
Cell[87615, 3107, 68, 1, 26, "Input"],
Cell[87686, 3110, 993, 16, 197, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[88716, 3131, 41, 0, 29, "Subsubsection"],
Cell[88760, 3133, 127, 3, 20, "Text"],
Cell[88890, 3138, 72, 1, 26, "Input"],
Cell[88965, 3141, 152, 2, 65, "Input"],
Cell[89120, 3145, 91, 1, 26, "Input"],
Cell[89214, 3148, 262, 5, 101, "Input"],
Cell[89479, 3155, 127, 2, 38, "Input"],
Cell[89609, 3159, 325, 6, 88, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[89971, 3170, 41, 0, 29, "Subsubsection"],
Cell[90015, 3172, 527, 14, 52, "Text"],
Cell[90545, 3188, 72, 1, 26, "Input"],
Cell[90620, 3191, 942, 17, 230, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[91599, 3213, 34, 0, 29, "Subsubsection"],
Cell[91636, 3215, 584, 13, 52, "Text"],
Cell[92223, 3230, 65, 1, 26, "Input"],
Cell[92291, 3233, 308, 6, 75, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[92636, 3244, 36, 0, 29, "Subsubsection"],
Cell[92675, 3246, 441, 9, 41, "Text"],
Cell[93119, 3257, 67, 1, 26, "Input"],
Cell[93189, 3260, 537, 10, 136, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[93763, 3275, 31, 0, 29, "Subsubsection"],
Cell[93797, 3277, 354, 8, 41, "Text"],
Cell[94154, 3287, 62, 1, 26, "Input"],
Cell[94219, 3290, 1092, 18, 265, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[95348, 3313, 31, 0, 29, "Subsubsection"],
Cell[95382, 3315, 160, 4, 20, "Text"],
Cell[95545, 3321, 62, 1, 26, "Input"],
Cell[95610, 3324, 513, 9, 113, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[96160, 3338, 44, 0, 29, "Subsubsection"],
Cell[96207, 3340, 254, 5, 30, "Text"],
Cell[96464, 3347, 75, 1, 26, "Input"],
Cell[96542, 3350, 509, 9, 150, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[97088, 3364, 36, 0, 29, "Subsubsection"],
Cell[97127, 3366, 276, 7, 52, "Text"],
Cell[97406, 3375, 67, 1, 30, "Input"],
Cell[97476, 3378, 93, 1, 30, "Input"],
Cell[97572, 3381, 66, 1, 30, "Input"],
Cell[97641, 3384, 54, 1, 30, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[97744, 3391, 34, 0, 31, "Subsection"],

Cell[CellGroupData[{
Cell[97803, 3395, 38, 0, 43, "Subsubsection"],
Cell[97844, 3397, 247, 5, 30, "Text"],
Cell[98094, 3404, 1024, 23, 206, "Text"],
Cell[99121, 3429, 73, 1, 26, "Input"],
Cell[99197, 3432, 80, 1, 26, "Input"],
Cell[99280, 3435, 71, 1, 26, "Input"],
Cell[99354, 3438, 93, 1, 26, "Input"],
Cell[99450, 3441, 82, 1, 26, "Input"],
Cell[99535, 3444, 107, 2, 26, "Input"],
Cell[99645, 3448, 87, 1, 26, "Input"],
Cell[99735, 3451, 111, 2, 26, "Input"],
Cell[99849, 3455, 101, 2, 26, "Input"],
Cell[99953, 3459, 99, 2, 26, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[100089, 3466, 46, 0, 29, "Subsubsection"],
Cell[100138, 3468, 168, 4, 30, "Text"],
Cell[100309, 3474, 81, 1, 26, "Input"],
Cell[100393, 3477, 745, 14, 98, "Input"],
Cell[101141, 3493, 254, 5, 76, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[101432, 3503, 29, 0, 29, "Subsubsection"],
Cell[101464, 3505, 133, 3, 33, "Text"],
Cell[101600, 3510, 64, 1, 30, "Input"],
Cell[101667, 3513, 68, 1, 30, "Input"],
Cell[101738, 3516, 92, 1, 30, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[101879, 3523, 30, 0, 31, "Subsection"],

Cell[CellGroupData[{
Cell[101934, 3527, 34, 0, 43, "Subsubsection"],
Cell[101971, 3529, 128, 3, 20, "Text"],
Cell[102102, 3534, 65, 1, 26, "Input"],
Cell[102170, 3537, 58, 1, 26, "Input"],
Cell[102231, 3540, 46, 1, 26, "Input"],
Cell[102280, 3543, 61, 1, 26, "Input"],
Cell[102344, 3546, 112, 2, 26, "Input"],
Cell[102459, 3550, 52, 1, 26, "Input"],
Cell[102514, 3553, 65, 1, 26, "Input"],
Cell[102582, 3556, 121, 3, 26, "Input"],
Cell[102706, 3561, 66, 1, 26, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[102821, 3568, 35, 0, 31, "Subsection"],

Cell[CellGroupData[{
Cell[102881, 3572, 34, 0, 43, "Subsubsection"],
Cell[102918, 3574, 665, 21, 52, "Text"],
Cell[103586, 3597, 70, 1, 26, "Input"],
Cell[103659, 3600, 67, 1, 26, "Input"],
Cell[103729, 3603, 75, 1, 26, "Input"],
Cell[103807, 3606, 216, 4, 51, "Input"],
Cell[104026, 3612, 112, 2, 26, "Input"],
Cell[104141, 3616, 235, 4, 76, "Input"],
Cell[104379, 3622, 294, 6, 88, "Input"],
Cell[104676, 3630, 125, 3, 38, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[104850, 3639, 35, 0, 31, "Subsection"],

Cell[CellGroupData[{
Cell[104910, 3643, 32, 0, 43, "Subsubsection"],
Cell[104945, 3645, 691, 12, 74, "Text"],
Cell[105639, 3659, 68, 1, 26, "Input"],
Cell[105710, 3662, 142, 2, 38, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[105889, 3669, 35, 0, 29, "Subsubsection"],
Cell[105927, 3671, 467, 16, 41, "Text"],
Cell[106397, 3689, 71, 1, 26, "Input"],
Cell[106471, 3692, 69, 0, 26, "Input"],
Cell[106543, 3694, 75, 0, 26, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[106655, 3699, 34, 0, 29, "Subsubsection"],
Cell[106692, 3701, 675, 10, 74, "Text"],
Cell[107370, 3713, 70, 1, 26, "Input"],
Cell[107443, 3716, 1219, 19, 251, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[108711, 3741, 32, 0, 31, "Subsection"],

Cell[CellGroupData[{
Cell[108768, 3745, 41, 0, 43, "Subsubsection"],
Cell[108812, 3747, 482, 11, 52, "Text"],
Cell[109297, 3760, 74, 1, 26, "Input"],
Cell[109374, 3763, 146, 4, 20, "Text"],
Cell[109523, 3769, 272, 5, 63, "Input"],
Cell[109798, 3776, 143, 3, 46, "Input"],
Cell[109944, 3781, 180, 3, 46, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[110161, 3789, 51, 0, 29, "Subsubsection"],
Cell[110215, 3791, 284, 5, 52, "Text"],
Cell[110502, 3798, 84, 1, 30, "Input"],
Cell[110589, 3801, 375, 11, 54, "Text"],
Cell[110967, 3814, 584, 9, 110, "Input",
  CellTags->"S6.4.1"],
Cell[111554, 3825, 190, 4, 50, "Input",
  CellTags->"S6.4.1"],
Cell[111747, 3831, 564, 11, 150, "Input",
  CellTags->"S6.4.1"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[112360, 3848, 31, 0, 31, "Subsection"],

Cell[CellGroupData[{
Cell[112416, 3852, 28, 0, 43, "Subsubsection"],
Cell[112447, 3854, 152, 4, 34, "Text"],
Cell[112602, 3860, 60, 1, 30, "Input"],
Cell[112665, 3863, 152, 3, 56, "Input"],
Cell[112820, 3868, 58, 1, 30, "Input"],
Cell[112881, 3871, 63, 1, 30, "Input"],
Cell[112947, 3874, 249, 10, 34, "Text"]
}, Closed]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[113257, 3891, 28, 0, 36, "Section"],

Cell[CellGroupData[{
Cell[113310, 3895, 46, 0, 47, "Subsection"],
Cell[113359, 3897, 2842, 115, 155, "Text"],
Cell[116204, 4014, 260, 9, 128, "Input"],
Cell[116467, 4025, 195, 7, 23, "Text",
  CellTags->"Hue"],
Cell[116665, 4034, 237, 6, 71, "Input",
  CellTags->"Hue"],
Cell[116905, 4042, 91, 2, 23, "Input",
  CellTags->"Hue"],
Cell[116999, 4046, 610, 15, 83, "Text"],
Cell[117612, 4063, 49094, 757, 327, 49001, 754, "GraphicsData", 
"PICT", "Graphics"]
}, Closed]],

Cell[CellGroupData[{
Cell[166743, 4825, 39, 0, 31, "Subsection"],
Cell[166785, 4827, 478, 12, 73, "Text"],
Cell[167266, 4841, 44, 0, 30, "Input"],
Cell[167313, 4843, 50, 1, 30, "Input"],
Cell[167366, 4846, 37, 0, 30, "Input"],
Cell[167406, 4848, 147, 6, 34, "Text"],
Cell[167556, 4856, 45, 1, 30, "Input"],
Cell[167604, 4859, 1088, 43, 93, "Text"],
Cell[168695, 4904, 328, 6, 130, "Input"],
Cell[169026, 4912, 1121, 43, 113, "Text"],
Cell[170150, 4957, 65, 1, 30, "Input"],
Cell[170218, 4960, 56, 1, 30, "Input"],
Cell[170277, 4963, 194, 4, 53, "Text"],
Cell[170474, 4969, 243, 6, 105, "Input",
  CellTags->"Hue"],
Cell[170720, 4977, 234, 6, 90, "Input",
  CellTags->"Hue"],
Cell[170957, 4985, 74, 1, 30, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[171080, 4992, 56, 0, 36, "Section"],

Cell[CellGroupData[{
Cell[171161, 4996, 30, 0, 47, "Subsection"],
Cell[171194, 4998, 462, 11, 68, "Text"],
Cell[171659, 5011, 177, 5, 38, "Text"],
Cell[171839, 5018, 1837, 64, 23, "Input"],
Cell[173679, 5084, 1789, 62, 23, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[175505, 5151, 105, 2, 31, "Subsection",
  Evaluatable->False],
Cell[175613, 5155, 746, 34, 38, "Text"],
Cell[176362, 5191, 2776, 125, 113, "Text"],
Cell[179141, 5318, 785, 13, 232, "Input"],
Cell[179929, 5333, 209, 6, 38, "Text"],
Cell[180141, 5341, 144, 3, 39, "Input",
  CellTags->"RasterArray"],
Cell[180288, 5346, 86, 1, 23, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[180411, 5352, 41, 0, 31, "Subsection"],
Cell[180455, 5354, 512, 16, 53, "Text"],
Cell[180970, 5372, 1586, 51, 132, "Text"],
Cell[182559, 5425, 1010, 18, 319, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[183606, 5448, 64, 0, 31, "Subsection"],
Cell[183673, 5450, 1258, 48, 83, "Text"],
Cell[184934, 5500, 602, 11, 215, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[185573, 5516, 64, 0, 31, "Subsection"],
Cell[185640, 5518, 345, 11, 38, "Text"],
Cell[185988, 5531, 304, 6, 119, "Input"],
Cell[186295, 5539, 344, 11, 38, "Text"],
Cell[186642, 5552, 285, 6, 103, "Input",
  CellTags->"EdgeForm"],
Cell[186930, 5560, 477, 17, 38, "Text"],
Cell[187410, 5579, 398, 6, 119, "Input"],
Cell[187811, 5587, 391, 6, 119, "Input"],
Cell[188205, 5595, 76, 1, 23, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[188318, 5601, 37, 0, 31, "Subsection"],
Cell[188358, 5603, 2021, 81, 152, "Text"],
Cell[190382, 5686, 2430, 74, 152, "Input"],
Cell[192815, 5762, 381, 8, 110, "Input"],
Cell[193199, 5772, 530, 18, 74, "Text"],
Cell[193732, 5792, 357, 7, 90, "Input"],
Cell[194092, 5801, 600, 21, 73, "Text"],
Cell[194695, 5824, 164, 4, 50, "Input",
  CellTags->"S2.8.9"],
Cell[194862, 5830, 78, 1, 30, "Input"],
Cell[194943, 5833, 361, 9, 73, "Text"],
Cell[195307, 5844, 100, 2, 30, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[195444, 5851, 91, 1, 31, "Subsection",
  PageBreakAbove->True],
Cell[195538, 5854, 503, 17, 59, "Question"],
Cell[196044, 5873, 293, 6, 104, "Input"],
Cell[196340, 5881, 604, 10, 165, "Input"],
Cell[196947, 5893, 21, 0, 25, "Special3"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[197017, 5899, 55, 0, 36, "Section"],

Cell[CellGroupData[{
Cell[197097, 5903, 26, 0, 47, "Subsection"],
Cell[197126, 5905, 790, 24, 93, "Text"],
Cell[197919, 5931, 100, 2, 31, "Input"],
Cell[198022, 5935, 119, 2, 30, "Input"],
Cell[198144, 5939, 200, 6, 34, "Text"],
Cell[198347, 5947, 175, 3, 52, "Input"],
Cell[198525, 5952, 66, 1, 30, "Input"],
Cell[198594, 5955, 149, 4, 34, "Text"],
Cell[198746, 5961, 144, 3, 30, "Input"],
Cell[198893, 5966, 145, 3, 30, "Input"],
Cell[199041, 5971, 64, 1, 30, "Input"],
Cell[199108, 5974, 258, 5, 53, "Text"],
Cell[199369, 5981, 480, 8, 130, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[199886, 5994, 35, 0, 47, "Subsection"],
Cell[199924, 5996, 758, 20, 83, "Text"],
Cell[200685, 6018, 83, 1, 23, "Input"],
Cell[200771, 6021, 315, 12, 38, "Text"],
Cell[201089, 6035, 148, 3, 39, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[201274, 6043, 31, 0, 31, "Subsection"],
Cell[201308, 6045, 258, 6, 38, "Text"],
Cell[201569, 6053, 228, 5, 38, "Text"],
Cell[201800, 6060, 52, 0, 23, "Text"],
Cell[201855, 6062, 165, 3, 55, "Input"],
Cell[202023, 6067, 190, 6, 38, "Text"],
Cell[202216, 6075, 179, 3, 55, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[202432, 6083, 37, 0, 31, "Subsection"],
Cell[202472, 6085, 1029, 35, 83, "Text"],
Cell[203504, 6122, 576, 19, 68, "Text"],
Cell[204083, 6143, 441, 9, 152, "Input"],
Cell[204527, 6154, 355, 10, 53, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[204919, 6169, 62, 0, 31, "Subsection"],
Cell[204984, 6171, 385, 10, 73, "Text"],
Cell[205372, 6183, 64, 1, 30, "Input"],
Cell[205439, 6186, 197, 4, 70, "Input"],
Cell[205639, 6192, 198, 4, 70, "Input"],
Cell[205840, 6198, 345, 9, 73, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[206222, 6212, 38, 0, 47, "Subsection"],
Cell[206263, 6214, 397, 9, 53, "Text"],
Cell[206663, 6225, 527, 10, 83, "Text"],
Cell[207193, 6237, 308, 7, 53, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[207538, 6249, 39, 0, 31, "Subsection"],
Cell[207580, 6251, 1332, 47, 112, "Text"],
Cell[208915, 6300, 1655, 48, 262, "Input"],
Cell[210573, 6350, 326, 7, 71, "Text"],
Cell[210902, 6359, 115, 4, 34, "Text"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[211066, 6369, 35, 0, 36, "Section"],

Cell[CellGroupData[{
Cell[211126, 6373, 34, 0, 47, "Subsection"],

Cell[CellGroupData[{
Cell[211185, 6377, 31, 0, 43, "Subsubsection"],
Cell[211219, 6379, 503, 17, 53, "Text"],
Cell[211725, 6398, 61, 1, 23, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[211823, 6404, 45, 0, 29, "Subsubsection"],
Cell[211871, 6406, 827, 22, 98, "Text"],
Cell[212701, 6430, 75, 1, 27, "Input"],
Cell[212779, 6433, 1013, 35, 83, "Text"],
Cell[213795, 6470, 157, 3, 46, "Input"],
Cell[213955, 6475, 167, 3, 46, "Input"],
Cell[214125, 6480, 193, 4, 63, "Input"],
Cell[214321, 6486, 84, 1, 23, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[214442, 6492, 85, 1, 29, "Subsubsection"],
Cell[214530, 6495, 953, 22, 113, "Text"],
Cell[215486, 6519, 387, 11, 53, "Text"],
Cell[215876, 6532, 371, 6, 68, "Text"],
Cell[216250, 6540, 77, 1, 23, "Input"],
Cell[216330, 6543, 103, 2, 23, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[216470, 6550, 36, 0, 29, "Subsubsection"],
Cell[216509, 6552, 950, 26, 113, "Text"],
Cell[217462, 6580, 149, 3, 39, "Input"],
Cell[217614, 6585, 185, 4, 55, "Input"],
Cell[217802, 6591, 78, 1, 23, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[217917, 6597, 34, 0, 29, "Subsubsection"],
Cell[217954, 6599, 538, 13, 68, "Text"],
Cell[218495, 6614, 2960, 108, 162, "Text"],
Cell[221458, 6724, 330, 6, 98, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[221825, 6735, 56, 0, 29, "Subsubsection"],
Cell[221884, 6737, 186, 6, 38, "Text"],
Cell[222073, 6745, 81, 1, 23, "Input"],
Cell[222157, 6748, 183, 4, 38, "Text"],
Cell[222343, 6754, 166, 3, 39, "Input"],
Cell[222512, 6759, 186, 3, 55, "Input"],
Cell[222701, 6764, 76, 1, 23, "Input"],
Cell[222780, 6767, 1019, 35, 83, "Text"],
Cell[223802, 6804, 176, 4, 39, "Input"],
Cell[223981, 6810, 278, 6, 87, "Input"],
Cell[224262, 6818, 272, 5, 87, "Input"],
Cell[224537, 6825, 81, 1, 23, "Input"],
Cell[224621, 6828, 393, 10, 53, "Text"],
Cell[225017, 6840, 277, 5, 87, "Input"],
Cell[225297, 6847, 284, 9, 38, "Text"],
Cell[225584, 6858, 413, 7, 135, "Input"],
Cell[226000, 6867, 545, 10, 167, "Input"],
Cell[226548, 6879, 74, 1, 23, "Input"],
Cell[226625, 6882, 286, 10, 38, "Text"],
Cell[226914, 6894, 275, 7, 38, "Text"],
Cell[227192, 6903, 232, 4, 71, "Input"],
Cell[227427, 6909, 187, 4, 55, "Input"],
Cell[227617, 6915, 78, 1, 23, "Input"],
Cell[227698, 6918, 760, 25, 68, "Text"],
Cell[228461, 6945, 159, 3, 58, "Input"],
Cell[228623, 6950, 206, 4, 75, "Input"],
Cell[228832, 6956, 190, 4, 75, "Input"],
Cell[229025, 6962, 81, 1, 23, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[229143, 6968, 31, 0, 29, "Subsubsection"],
Cell[229177, 6970, 997, 33, 98, "Text"],
Cell[230177, 7005, 62, 1, 23, "Input"],
Cell[230242, 7008, 233, 3, 92, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[230512, 7016, 32, 0, 29, "Subsubsection"],
Cell[230547, 7018, 814, 29, 73, "Text"],
Cell[231364, 7049, 364, 7, 72, "Input"],
Cell[231731, 7058, 417, 8, 72, "Input"],
Cell[232151, 7068, 89, 1, 30, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[232289, 7075, 36, 0, 31, "Subsection"],

Cell[CellGroupData[{
Cell[232350, 7079, 31, 0, 43, "Subsubsection"],
Cell[232384, 7081, 141, 3, 38, "Text"],
Cell[232528, 7086, 1191, 50, 68, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[233756, 7141, 39, 0, 29, "Subsubsection"],
Cell[233798, 7143, 887, 26, 98, "Text"],
Cell[234688, 7171, 130, 2, 39, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[234855, 7178, 67, 0, 29, "Subsubsection"],
Cell[234925, 7180, 102, 3, 23, "Text"],
Cell[235030, 7185, 512, 15, 53, "Text"],
Cell[235545, 7202, 133, 2, 39, "Input"],
Cell[235681, 7206, 542, 14, 83, "Text"],
Cell[236226, 7222, 173, 3, 55, "Input"],
Cell[236402, 7227, 806, 28, 68, "Text"],
Cell[237211, 7257, 203, 4, 55, "Input"],
Cell[237417, 7263, 92, 1, 23, "Input"],
Cell[237512, 7266, 207, 4, 55, "Input"],
Cell[237722, 7272, 61, 0, 23, "Text"],
Cell[237786, 7274, 200, 4, 55, "Input"],
Cell[237989, 7280, 75, 1, 23, "Input"],
Cell[238067, 7283, 1350, 56, 53, "Text"],
Cell[239420, 7341, 194, 4, 69, "Input"],
Cell[239617, 7347, 234, 4, 88, "Input"],
Cell[239854, 7353, 258, 5, 99, "Input"],
Cell[240115, 7360, 84, 1, 23, "Input"],
Cell[240202, 7363, 603, 17, 68, "Text"],
Cell[240808, 7382, 189, 4, 39, "Input"],
Cell[241000, 7388, 216, 4, 55, "Input"],
Cell[241219, 7394, 251, 5, 55, "Input"],
Cell[241473, 7401, 75, 1, 23, "Input"],
Cell[241551, 7404, 282, 10, 38, "Text"],
Cell[241836, 7416, 133, 2, 39, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[242006, 7423, 63, 0, 29, "Subsubsection"],
Cell[242072, 7425, 651, 25, 53, "Text"],
Cell[242726, 7452, 142, 2, 39, "Input"],
Cell[242871, 7456, 288, 10, 38, "Text"],
Cell[243162, 7468, 186, 3, 55, "Input"],
Cell[243351, 7473, 2239, 77, 114, "Text"],
Cell[245593, 7552, 176, 4, 39, "Input"],
Cell[245772, 7558, 206, 4, 55, "Input"],
Cell[245981, 7564, 347, 7, 87, "Input"],
Cell[246331, 7573, 75, 1, 23, "Input"],
Cell[246409, 7576, 1125, 40, 83, "Text"],
Cell[247537, 7618, 150, 2, 39, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[247724, 7625, 54, 0, 29, "Subsubsection"],
Cell[247781, 7627, 461, 11, 72, "Text"],
Cell[248245, 7640, 131, 2, 50, "Input"],
Cell[248379, 7644, 433, 13, 72, "Text"],
Cell[248815, 7659, 184, 4, 50, "Input"],
Cell[249002, 7665, 186, 4, 50, "Input"],
Cell[249191, 7671, 68, 1, 30, "Input"],
Cell[249262, 7674, 299, 10, 53, "Text"],
Cell[249564, 7686, 130, 2, 30, "Input"],
Cell[249697, 7690, 1023, 30, 132, "Text"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[250769, 7726, 57, 0, 31, "Subsection"],
Cell[250829, 7728, 1382, 49, 132, "Text"],
Cell[252214, 7779, 155, 2, 50, "Input"],
Cell[252372, 7783, 489, 16, 73, "Text"],
Cell[252864, 7801, 185, 3, 50, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[253086, 7809, 65, 0, 31, "Subsection"],
Cell[253154, 7811, 224, 9, 34, "Text"],
Cell[253381, 7822, 311, 5, 90, "Input"],
Cell[253695, 7829, 702, 28, 54, "Text"],
Cell[254400, 7859, 300, 7, 71, "Text"],
Cell[254703, 7868, 523, 13, 91, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[255263, 7886, 27, 0, 31, "Subsection"],
Cell[255293, 7888, 1301, 53, 83, "Text"],
Cell[256597, 7943, 108, 2, 39, "Input"],
Cell[256708, 7947, 478, 13, 68, "Text"],
Cell[257189, 7962, 152, 3, 39, "Input"],
Cell[257344, 7967, 348, 6, 68, "Text"],
Cell[257695, 7975, 3807, 101, 119, "Input"],
Cell[261505, 8078, 390, 6, 143, "Input"],
Cell[261898, 8086, 686, 14, 98, "Text"],
Cell[262587, 8102, 307, 10, 143, "Input"],
Cell[262897, 8114, 310, 10, 143, "Input"],
Cell[263210, 8126, 165, 3, 63, "Input"],
Cell[263378, 8131, 62, 0, 23, "Input"],
Cell[263443, 8133, 397, 14, 38, "Text"],
Cell[263843, 8149, 183, 3, 63, "Input"],
Cell[264029, 8154, 127, 3, 38, "Text"],
Cell[264159, 8159, 190, 3, 55, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[264386, 8167, 51, 0, 31, "Subsection"],
Cell[264440, 8169, 544, 15, 74, "Text"],
Cell[264987, 8186, 566, 20, 73, "Text"],
Cell[265556, 8208, 408, 7, 132, "Input"],
Cell[265967, 8217, 1013, 18, 272, "Input"],
Cell[266983, 8237, 1415, 58, 93, "Text"],
Cell[268401, 8297, 350, 6, 164, "Input"],
Cell[268754, 8305, 192, 4, 72, "Input"],
Cell[268949, 8311, 295, 9, 53, "Text"],
Cell[269247, 8322, 138, 2, 50, "Input"],
Cell[269388, 8326, 245, 4, 112, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[269682, 8336, 32, 0, 36, "Section"],

Cell[CellGroupData[{
Cell[269739, 8340, 54, 0, 47, "Subsection"],
Cell[269796, 8342, 314, 9, 52, "Text"],
Cell[270113, 8353, 976, 34, 94, "Text"],
Cell[271092, 8389, 151, 3, 70, "Input"],
Cell[271246, 8394, 196, 6, 52, "Text"],
Cell[271445, 8402, 273, 7, 54, "Text"],
Cell[271721, 8411, 253, 4, 90, "Input"],
Cell[271977, 8417, 196, 6, 52, "Text"],
Cell[272176, 8425, 285, 8, 53, "Text"],
Cell[272464, 8435, 47, 0, 30, "Input"],
Cell[272514, 8437, 58, 1, 30, "Input"],
Cell[272575, 8440, 1406, 64, 53, "Text"],
Cell[273984, 8506, 81, 1, 30, "Input"],
Cell[274068, 8509, 196, 6, 52, "Text"],
Cell[274267, 8517, 487, 18, 54, "Text"],
Cell[274757, 8537, 258, 7, 53, "Text"],
Cell[275018, 8546, 126, 2, 50, "Input"],
Cell[275147, 8550, 196, 6, 52, "Text"],
Cell[275346, 8558, 103, 3, 33, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[275486, 8566, 73, 1, 47, "Subsection"],
Cell[275562, 8569, 226, 7, 52, "Text"],
Cell[275791, 8578, 515, 15, 71, "Text"],

Cell[CellGroupData[{
Cell[276331, 8597, 30, 0, 43, "Subsubsection"],
Cell[276364, 8599, 1181, 33, 143, "Text"],
Cell[277548, 8634, 67, 1, 23, "Input"],
Cell[277618, 8637, 214, 4, 46, "Input"],
Cell[277835, 8643, 67, 1, 23, "Input"],
Cell[277905, 8646, 150, 5, 23, "Text"],
Cell[278058, 8653, 108268, 1667, 319, 108175, 1664, "GraphicsData", 
"PICT", "Graphics"]
}, Closed]],

Cell[CellGroupData[{
Cell[386363, 10325, 157, 7, 29, "Subsubsection"],
Cell[386523, 10334, 563, 19, 53, "Text"],
Cell[387089, 10355, 183, 8, 23, "Text"],
Cell[387275, 10365, 126, 3, 39, "Input"],
Cell[387404, 10370, 290, 11, 38, "Text"],
Cell[387697, 10383, 103, 2, 23, "Input"],
Cell[387803, 10387, 159, 5, 23, "Text"],
Cell[387965, 10394, 111, 2, 23, "Input"],
Cell[388079, 10398, 233, 8, 38, "Text"],
Cell[388315, 10408, 232, 4, 46, "Input"],
Cell[388550, 10414, 63, 1, 23, "Input"],
Cell[388616, 10417, 168, 7, 23, "Text"],
Cell[388787, 10426, 108690, 1673, 319, 108597, 1670, "GraphicsData", 
"PICT", "Graphics"]
}, Closed]],

Cell[CellGroupData[{
Cell[497514, 12104, 37, 0, 29, "Subsubsection"],
Cell[497554, 12106, 374, 8, 68, "Text"],
Cell[497931, 12116, 321, 7, 53, "Text"],
Cell[498255, 12125, 108851, 1676, 303, 108758, 1673, "GraphicsData", 
"PICT", "Graphics"],
Cell[607109, 13803, 219, 4, 52, "Text"],
Cell[607331, 13809, 69, 0, 33, "Text"],
Cell[607403, 13811, 65, 1, 25, "Exercises",
  PageBreakAbove->True],
Cell[607471, 13814, 364, 12, 42, "Question"],
Cell[607838, 13828, 194, 4, 50, "Input"],
Cell[608035, 13834, 247, 7, 34, "Text"],
Cell[608285, 13843, 80, 1, 30, "Input"],
Cell[608368, 13846, 345, 9, 44, "Question"],
Cell[608716, 13857, 65, 1, 30, "Input"],
Cell[608784, 13860, 162, 3, 50, "Input"],
Cell[608949, 13865, 85, 1, 23, "Input"],
Cell[609037, 13868, 196, 6, 38, "Text"],
Cell[609236, 13876, 24, 0, 23, "EndExercises"],
Cell[609263, 13878, 74, 2, 23, "Input",
  PageBreakAbove->True,
  PageBreakBelow->True]
}, Closed]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[609398, 13887, 50, 0, 36, "Section"],
Cell[609451, 13889, 604, 15, 90, "Text"],

Cell[CellGroupData[{
Cell[610080, 13908, 36, 0, 47, "Subsection"],

Cell[CellGroupData[{
Cell[610141, 13912, 30, 0, 43, "Subsubsection"],
Cell[610174, 13914, 250, 6, 38, "Text"],
Cell[610427, 13922, 1651, 62, 114, "Text"],
Cell[612081, 13986, 134, 2, 46, "Input"],
Cell[612218, 13990, 324, 11, 40, "Text"],
Cell[612545, 14003, 152, 2, 46, "Input"],
Cell[612700, 14007, 1377, 47, 113, "Text"],
Cell[614080, 14056, 209, 3, 71, "Input"],
Cell[614292, 14061, 98, 2, 27, "Input"],
Cell[614393, 14065, 243, 3, 38, "Text"],
Cell[614639, 14070, 368, 6, 151, "Input"],
Cell[615010, 14078, 98, 2, 27, "Input"],
Cell[615111, 14082, 168, 4, 38, "Text"],
Cell[615282, 14088, 466, 7, 167, "Input"],
Cell[615751, 14097, 98, 2, 27, "Input"],
Cell[615852, 14101, 855, 22, 113, "Text"],
Cell[616710, 14125, 66, 1, 23, "Input"],
Cell[616779, 14128, 636, 11, 215, "Input"],
Cell[617418, 14141, 98, 2, 27, "Input"],
Cell[617519, 14145, 931, 22, 98, "Text"]
}, Closed]],

Cell[CellGroupData[{
Cell[618487, 14172, 50, 0, 29, "Subsubsection"],
Cell[618540, 14174, 62, 1, 23, "Input"],
Cell[618605, 14177, 3201, 58, 1255, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[621843, 14240, 33, 0, 29, "Subsubsection"],
Cell[621879, 14242, 111, 2, 23, "Input"],
Cell[621993, 14246, 135, 3, 46, "Input"],
Cell[622131, 14251, 1036, 19, 359, "Input"],
Cell[623170, 14272, 70, 1, 23, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[623289, 14279, 33, 0, 31, "Subsection"],

Cell[CellGroupData[{
Cell[623347, 14283, 29, 0, 43, "Subsubsection"],
Cell[623379, 14285, 70, 1, 23, "Input"],
Cell[623452, 14288, 231, 4, 71, "Input"],
Cell[623686, 14294, 259, 4, 87, "Input"],
Cell[623948, 14300, 229, 5, 55, "Input"],
Cell[624180, 14307, 2134, 38, 663, "Input"]
}, Closed]],

Cell[CellGroupData[{
Cell[626351, 14350, 33, 0, 29, "Subsubsection"],
Cell[626387, 14352, 62, 1, 23, "Input"],
Cell[626452, 14355, 49, 1, 23, "Input"],
Cell[626504, 14358, 55, 1, 23, "Input"],
Cell[626562, 14361, 241, 11, 23, "Text"],
Cell[626806, 14374, 235, 3, 123, "Input"]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{
Cell[627090, 14383, 41, 0, 31, "Subsection"],
Cell[627134, 14385, 233, 6, 52, "Text"],
Cell[627370, 14393, 155, 3, 63, "Input"],
Cell[627528, 14398, 103, 2, 30, "Input"],
Cell[627634, 14402, 142, 4, 34, "Text"],
Cell[627779, 14408, 173, 3, 63, "Input"],
Cell[627955, 14413, 183, 4, 70, "Input"]
}, Closed]]
}, Closed]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

