(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 8.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       157,          7]
NotebookDataLength[     63393,       1379]
NotebookOptionsPosition[     61851,       1325]
NotebookOutlinePosition[     62523,       1350]
CellTagsIndexPosition[     62480,       1347]
WindowTitle->Wavefront Maps and Profiles of Seidel Aberrations - Source
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["Wavefront Maps and Profiles of Seidel Aberrations", "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False,
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztnVuSozgWQDNiPqa3MJ+zpVlCbaB3WlGVNV9dvYH6TQRUph/ZGIHQmwvG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   "], {{0, 0}, {1800, 25}}, {0, 255},
   ColorFunction->RGBColor],
  ImageSize->{1800, 25},
  PlotRange->{{0, 1800}, {0, 25}}]], "Section",
 CellFrame->{{0, 0}, {0, 0}},
 ShowCellBracket->False],

Cell[BoxData[
 RowBox[{"Manipulate", "[", 
  RowBox[{
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "parameters", ",", "xplot", ",", "yplot", ",", "plotProfile", ",", 
       "bottom", ",", "gra", ",", "gr2a", ",", "parametersBlank", ",", 
       "aberration", ",", "aberrationCylindrical", ",", "aberrationX", ",", 
       "aberrationY", ",", "profilePlotOptions"}], "}"}], ",", 
     "\[IndentingNewLine]", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"r2", "=", 
       RowBox[{"(", 
        RowBox[{
         SuperscriptBox["x", "2"], "+", 
         SuperscriptBox["y", "2"]}], ")"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"parameters", "=", 
       RowBox[{"{", 
        RowBox[{
        "tilt", ",", "tiltAng", ",", "focus", ",", "coma", ",", "comaAng", 
         ",", "ast", ",", "astAng", ",", "sph"}], "}"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"parametersBlank", "=", 
       RowBox[{"{", 
        RowBox[{
        "tilt_", ",", "tiltAng_", ",", "focus_", ",", "coma_", ",", 
         "comaAng_", ",", "ast_", ",", "astAng_", ",", "sph_"}], "}"}]}], ";",
       "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"aberration", "[", "parametersBlank", "]"}], ":=", 
       RowBox[{
        RowBox[{"tilt", " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"x", " ", 
            RowBox[{"Cos", "[", 
             RowBox[{"tiltAng", " ", "Degree"}], "]"}]}], "+", 
           RowBox[{"y", " ", 
            RowBox[{"Sin", "[", 
             RowBox[{"tiltAng", " ", "Degree"}], "]"}]}]}], ")"}]}], "+", 
        RowBox[{"coma", " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"x", " ", "r2", " ", 
            RowBox[{"Cos", "[", 
             RowBox[{"comaAng", " ", "Degree"}], "]"}]}], "+", 
           RowBox[{"y", " ", "r2", " ", 
            RowBox[{"Sin", "[", 
             RowBox[{"comaAng", " ", "Degree"}], "]"}]}]}], ")"}]}], "+", 
        RowBox[{"focus", " ", "r2"}], "+", 
        RowBox[{"ast", " ", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            RowBox[{"x", " ", 
             RowBox[{"Cos", "[", 
              RowBox[{"astAng", " ", "Degree"}], "]"}]}], "+", 
            RowBox[{"y", " ", 
             RowBox[{"Sin", "[", 
              RowBox[{"astAng", " ", "Degree"}], "]"}]}]}], ")"}], "2"]}], 
        "+", 
        RowBox[{"sph", " ", 
         SuperscriptBox["r2", "2"]}]}]}], ";", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"aberrationCylindrical", "[", "parametersBlank", "]"}], ":=", 
       RowBox[{
        RowBox[{"tilt", " ", "\[Rho]", " ", 
         RowBox[{"Cos", "[", 
          RowBox[{"\[Theta]", "-", 
           RowBox[{"tiltAng", " ", "Degree"}]}], "]"}]}], "+", 
        RowBox[{"focus", " ", 
         SuperscriptBox["\[Rho]", "2"]}], "+", 
        RowBox[{"coma", " ", 
         SuperscriptBox["\[Rho]", "3"], " ", 
         RowBox[{"Cos", "[", 
          RowBox[{"\[Theta]", "-", 
           RowBox[{"comaAng", " ", "Degree"}]}], "]"}]}], "+", 
        RowBox[{"ast", " ", 
         SuperscriptBox["\[Rho]", "2"], " ", 
         SuperscriptBox[
          RowBox[{"Cos", "[", 
           RowBox[{"\[Theta]", "-", 
            RowBox[{"astAng", " ", "Degree"}]}], "]"}], "2"]}], "+", 
        RowBox[{"sph", " ", 
         SuperscriptBox["\[Rho]", "4"]}]}]}], ";", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"aberrationX", "[", "parametersBlank", "]"}], ":=", 
       RowBox[{
        RowBox[{"tilt", " ", "x", " ", 
         RowBox[{"Cos", "[", 
          RowBox[{"tiltAng", " ", "Degree"}], "]"}]}], "+", 
        RowBox[{"focus", " ", 
         SuperscriptBox["x", "2"]}], "+", 
        RowBox[{"coma", " ", 
         SuperscriptBox["x", "3"], " ", 
         RowBox[{"Cos", "[", 
          RowBox[{"comaAng", " ", "Degree"}], "]"}]}], "+", 
        RowBox[{"ast", " ", 
         SuperscriptBox["x", "2"], " ", 
         SuperscriptBox[
          RowBox[{"Cos", "[", 
           RowBox[{"astAng", " ", "Degree"}], "]"}], "2"]}], "+", 
        RowBox[{"sph", " ", 
         SuperscriptBox["x", "4"]}]}]}], ";", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"aberrationY", "[", "parametersBlank", "]"}], ":=", 
       RowBox[{
        RowBox[{"tilt", " ", "y", " ", 
         RowBox[{"Sin", "[", 
          RowBox[{"tiltAng", " ", "Degree"}], "]"}]}], "+", 
        RowBox[{"focus", " ", 
         SuperscriptBox["y", "2"]}], "+", 
        RowBox[{"coma", " ", 
         SuperscriptBox["y", "3"], " ", 
         RowBox[{"Sin", "[", 
          RowBox[{"comaAng", " ", "Degree"}], "]"}]}], "+", 
        RowBox[{"ast", " ", 
         SuperscriptBox["y", "2"], " ", 
         SuperscriptBox[
          RowBox[{"Sin", "[", 
           RowBox[{"astAng", " ", "Degree"}], "]"}], "2"]}], "+", 
        RowBox[{"sph", " ", 
         SuperscriptBox["y", "4"]}]}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"profilePlotOptions", ":=", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
         RowBox[{"Frame", "\[Rule]", "True"}], ",", 
         RowBox[{"PlotLabel", "\[Rule]", 
          RowBox[{"Row", "[", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"Style", "[", 
              RowBox[{"\"\<X\>\"", ",", "Italic"}], "]"}], ",", 
             "\"\< profile (red), \>\"", ",", " ", 
             RowBox[{"Style", "[", 
              RowBox[{"\"\<Y\>\"", ",", "Italic"}], "]"}], ",", 
             "\"\< profile (green)\>\""}], "}"}], "]"}]}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"FrameTicks", "\[Rule]", "Automatic"}], ",", 
         RowBox[{"GridLines", "\[Rule]", "Automatic"}], ",", 
         RowBox[{"ImageSize", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"350", ",", "248"}], "}"}]}]}], "}"}]}], ";", 
      "\[IndentingNewLine]", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"xplot", "[", "parametersBlank", "]"}], ":=", 
       RowBox[{"Plot", "[", 
        RowBox[{
         RowBox[{"aberrationX", "[", "parameters", "]"}], ",", 
         RowBox[{"{", 
          RowBox[{"x", ",", 
           RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
         RowBox[{"PlotStyle", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"RGBColor", "[", 
              RowBox[{"1", ",", "0", ",", "0"}], "]"}], ",", 
             RowBox[{"Thickness", "[", ".008", "]"}]}], "}"}], "}"}]}]}], 
        "]"}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"yplot", "[", "parametersBlank", "]"}], ":=", 
       RowBox[{"Plot", "[", 
        RowBox[{
         RowBox[{"aberrationY", "[", "parameters", "]"}], ",", 
         RowBox[{"{", 
          RowBox[{"y", ",", 
           RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
         RowBox[{"PlotStyle", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"RGBColor", "[", 
              RowBox[{"0", ",", "1", ",", "0"}], "]"}], ",", 
             RowBox[{"Thickness", "[", ".008", "]"}]}], "}"}], "}"}]}]}], 
        "]"}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"plotProfile", "[", 
        RowBox[{"parametersBlank", ",", "graphicsPlotType_"}], "]"}], ":=", 
       RowBox[{"Show", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"xplot", "[", "parameters", "]"}], ",", 
           RowBox[{"yplot", "[", "parameters", "]"}]}], "}"}], ",", 
         RowBox[{"Evaluate", "[", "profilePlotOptions", "]"}], ",", 
         RowBox[{"FrameLabel", "\[Rule]", 
          RowBox[{"Style", "[", 
           RowBox[{
            RowBox[{"If", "[", 
             RowBox[{
              RowBox[{
               RowBox[{"graphicsPlotType", "\[Equal]", "densityPlot"}], "||", 
               
               RowBox[{"graphicsPlotType", "\[Equal]", "threeD"}]}], ",", 
              RowBox[{"Row", "[", 
               RowBox[{"{", 
                RowBox[{"\"\<OPD = \>\"", ",", 
                 RowBox[{"N", "[", 
                  RowBox[{
                   RowBox[{"aberration", "[", "parameters", "]"}], ",", "2"}],
                   "]"}]}], "}"}], "]"}], ",", 
              RowBox[{"Row", "[", 
               RowBox[{"{", 
                RowBox[{"\"\<OPD = \>\"", ",", 
                 RowBox[{"aberrationCylindrical", "[", "parameters", "]"}]}], 
                "}"}], "]"}], ",", 
              RowBox[{"Row", "[", 
               RowBox[{"{", 
                RowBox[{"\"\<OPD = \>\"", ",", 
                 RowBox[{"aberrationCylindrical", "[", "parameters", "]"}]}], 
                "}"}], "]"}]}], "]"}], ",", 
            RowBox[{"FontSize", "\[Rule]", "12"}]}], "]"}]}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", "\[IndentingNewLine]", 
      RowBox[{"GraphicsColumn", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"plotProfile", "[", 
           RowBox[{"parameters", ",", "graphicsPlotType"}], "]"}], ",", 
          RowBox[{"Switch", "[", 
           RowBox[{
           "graphicsPlotType", ",", "\[IndentingNewLine]", 
            "\[IndentingNewLine]", "densityPlot", ",", "\[IndentingNewLine]", 
            
            RowBox[{"DensityPlot", "[", 
             RowBox[{
              RowBox[{"If", "[", 
               RowBox[{
                RowBox[{
                 RowBox[{
                  SuperscriptBox["x", "2"], "+", 
                  SuperscriptBox["y", "2"]}], "\[LessEqual]", "1"}], ",", 
                RowBox[{"0.5", " ", 
                 RowBox[{"(", 
                  RowBox[{"1", "+", 
                   RowBox[{"Cos", "[", 
                    RowBox[{"2", " ", "\[Pi]", " ", 
                    RowBox[{"aberration", "[", "parameters", "]"}]}], "]"}]}],
                   ")"}]}], ",", "1"}], "]"}], ",", 
              RowBox[{"{", 
               RowBox[{"x", ",", 
                RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
              RowBox[{"{", 
               RowBox[{"y", ",", 
                RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
              RowBox[{"ColorFunction", "\[Rule]", "GrayLevel"}], ",", 
              RowBox[{"PlotPoints", "\[Rule]", "40"}], ",", 
              RowBox[{"Mesh", "\[Rule]", "False"}], ",", 
              RowBox[{"Frame", "\[Rule]", "True"}]}], "]"}], ",", 
            "\[IndentingNewLine]", "\[IndentingNewLine]", "cylindricalPlot3D",
             ",", "\[IndentingNewLine]", 
            RowBox[{"RevolutionPlot3D", "[", 
             RowBox[{
              RowBox[{"aberrationCylindrical", "[", "parameters", "]"}], ",", 
              
              RowBox[{"{", 
               RowBox[{"\[Rho]", ",", "0", ",", "1"}], "}"}], ",", 
              RowBox[{"{", 
               RowBox[{"\[Theta]", ",", "0", ",", 
                RowBox[{"2", " ", "\[Pi]"}]}], "}"}], ",", 
              RowBox[{"BoxRatios", "\[Rule]", 
               RowBox[{"{", 
                RowBox[{"1", ",", "1", ",", "0.5"}], "}"}]}], ",", 
              RowBox[{"Axes", "\[Rule]", "False"}], ",", 
              RowBox[{"Boxed", "\[Rule]", "False"}], ",", 
              RowBox[{"ColorFunction", "->", 
               RowBox[{"Function", "[", 
                RowBox[{
                 RowBox[{"{", 
                  RowBox[{"x", ",", "y", ",", "z"}], "}"}], ",", 
                 RowBox[{"Hue", "[", "z", "]"}]}], "]"}]}]}], "]"}], ",", 
            "\[IndentingNewLine]", "\[IndentingNewLine]", 
            "cylindricalPlot3DStereo", ",", "\[IndentingNewLine]", 
            RowBox[{"GraphicsRow", "[", 
             RowBox[{
              RowBox[{
               RowBox[{
                RowBox[{"RevolutionPlot3D", "[", 
                 RowBox[{
                  RowBox[{"aberrationCylindrical", "[", "parameters", "]"}], 
                  ",", 
                  RowBox[{"{", 
                   RowBox[{"\[Rho]", ",", "0", ",", "1"}], "}"}], ",", 
                  RowBox[{"{", 
                   RowBox[{"\[Theta]", ",", "0", ",", 
                    RowBox[{"2", " ", "\[Pi]"}]}], "}"}], ",", 
                  RowBox[{"Axes", "\[Rule]", "None"}], ",", 
                  RowBox[{"Boxed", "\[Rule]", "False"}], ",", 
                  RowBox[{"ColorFunction", "->", 
                   RowBox[{"Function", "[", 
                    RowBox[{
                    RowBox[{"{", 
                    RowBox[{"x", ",", "y", ",", "z"}], "}"}], ",", 
                    RowBox[{"Hue", "[", "z", "]"}]}], "]"}]}], ",", 
                  RowBox[{"ImageSize", "\[Rule]", "125"}], ",", 
                  RowBox[{"BoxRatios", "\[Rule]", 
                   RowBox[{"{", 
                    RowBox[{"1", ",", "1", ",", "0.5"}], "}"}]}], ",", 
                  RowBox[{"ViewVertical", "\[Rule]", 
                   RowBox[{"Dynamic", "[", "vV", "]"}]}], ",", 
                  RowBox[{"ViewAngle", "\[Rule]", 
                   RowBox[{"Dynamic", "[", "vA", "]"}]}], ",", "#"}], "]"}], 
                "&"}], "/@", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{"ViewPoint", "\[Rule]", 
                  RowBox[{"Dynamic", "[", 
                   RowBox[{"vp1", ",", 
                    RowBox[{
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"vp1", "=", "#1"}], ";", 
                    RowBox[{"vp2", "=", 
                    RowBox[{"#1", "-", 
                    RowBox[{"{", 
                    RowBox[{"0.1", ",", "0", ",", "0"}], "}"}]}]}]}], ")"}], 
                    "&"}]}], "]"}]}], ",", 
                 RowBox[{"ViewPoint", "\[Rule]", 
                  RowBox[{"Dynamic", "[", 
                   RowBox[{"vp2", ",", 
                    RowBox[{
                    RowBox[{"(", 
                    RowBox[{
                    RowBox[{"vp2", "=", "#1"}], ";", 
                    RowBox[{"vp1", "=", 
                    RowBox[{"#1", "+", 
                    RowBox[{"{", 
                    RowBox[{"0.1", ",", "0", ",", "0"}], "}"}]}]}]}], ")"}], 
                    "&"}]}], "]"}]}]}], "}"}]}], ",", 
              RowBox[{"-", "20"}]}], "]"}], ",", "\[IndentingNewLine]", 
            "\[IndentingNewLine]", "threeD", ",", "\[IndentingNewLine]", 
            RowBox[{"Plot3D", "[", 
             RowBox[{
              RowBox[{"If", "[", 
               RowBox[{
                RowBox[{
                 RowBox[{
                  SuperscriptBox["x", "2"], "+", 
                  SuperscriptBox["y", "2"]}], "\[LessEqual]", "1"}], ",", 
                RowBox[{"aberration", "[", "parameters", "]"}], ",", "1"}], 
               "]"}], ",", 
              RowBox[{"{", 
               RowBox[{"x", ",", 
                RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
              RowBox[{"{", 
               RowBox[{"y", ",", 
                RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
              RowBox[{"PlotPoints", "\[Rule]", "30"}], ",", 
              RowBox[{"Axes", "\[Rule]", "False"}], ",", 
              RowBox[{"BoxRatios", "\[Rule]", 
               RowBox[{"{", 
                RowBox[{"1", ",", "1", ",", "0.5"}], "}"}]}], ",", 
              RowBox[{"Boxed", "\[Rule]", "True"}], ",", 
              RowBox[{"ColorFunction", "->", 
               RowBox[{"Function", "[", 
                RowBox[{
                 RowBox[{"{", 
                  RowBox[{"x", ",", "y", ",", "z"}], "}"}], ",", 
                 RowBox[{"Hue", "[", "z", "]"}]}], "]"}]}]}], "]"}], ",", 
            "\[IndentingNewLine]", "\[IndentingNewLine]", "ShadowPlot3D", ",",
             "\[IndentingNewLine]", 
            RowBox[{
             RowBox[{"gra", "=", 
              RowBox[{"Plot3D", "[", 
               RowBox[{
                RowBox[{"aberration", "[", "parameters", "]"}], ",", 
                RowBox[{"{", 
                 RowBox[{"x", ",", 
                  RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
                RowBox[{"{", 
                 RowBox[{"y", ",", 
                  RowBox[{"-", 
                   SqrtBox[
                    RowBox[{"1", "-", 
                    SuperscriptBox["x", "2"]}]]}], ",", 
                  SqrtBox[
                   RowBox[{"1", "-", 
                    SuperscriptBox["x", "2"]}]]}], "}"}], ",", 
                RowBox[{"PlotPoints", "\[Rule]", "20"}], ",", 
                RowBox[{"Axes", "\[Rule]", "False"}], ",", 
                RowBox[{"ColorFunction", "->", 
                 RowBox[{"Function", "[", 
                  RowBox[{
                   RowBox[{"{", 
                    RowBox[{"x", ",", "y", ",", "z"}], "}"}], ",", 
                   RowBox[{"Hue", "[", "z", "]"}]}], "]"}]}]}], "]"}]}], ";", 
             "\[IndentingNewLine]", 
             RowBox[{"bottom", "=", 
              RowBox[{"3", 
               RowBox[{"Min", "[", 
                RowBox[{"Table", "[", 
                 RowBox[{
                  RowBox[{"aberration", "[", "parameters", "]"}], ",", 
                  RowBox[{"{", 
                   RowBox[{"x", ",", 
                    RowBox[{"-", "1"}], ",", "1", ",", ".1"}], "}"}], ",", 
                  RowBox[{"{", 
                   RowBox[{"y", ",", 
                    RowBox[{"-", 
                    SqrtBox[
                    RowBox[{"1", "-", 
                    SuperscriptBox["x", "2"]}]]}], ",", 
                    SqrtBox[
                    RowBox[{"1", "-", 
                    SuperscriptBox["x", "2"]}]]}], "}"}]}], "]"}], "]"}]}]}], 
             ";", 
             RowBox[{"gr2a", "=", 
              RowBox[{"ReplacePart", "[", 
               RowBox[{"gra", ",", 
                RowBox[{
                 RowBox[{
                  RowBox[{"{", 
                   RowBox[{"1", ",", "1"}], "}"}], "\[Rule]", " ", 
                  RowBox[{"gra", "[", 
                   RowBox[{"[", 
                    RowBox[{"1", ",", "1"}], "]"}], "]"}]}], "/.", 
                 RowBox[{
                  RowBox[{"{", 
                   RowBox[{"x_", ",", "y_", ",", "z_"}], "}"}], "\[Rule]", 
                  RowBox[{"{", 
                   RowBox[{"x", ",", "y", ",", "bottom"}], "}"}]}]}]}], 
               "]"}]}], ";", 
             RowBox[{"Show", "[", 
              RowBox[{"gra", ",", "gr2a", ",", 
               RowBox[{"PlotRange", " ", "\[Rule]", " ", "All"}], ",", 
               RowBox[{"BoxRatios", "->", 
                RowBox[{"{", 
                 RowBox[{"1", ",", "1", ",", ".9"}], "}"}]}], ",", 
               RowBox[{"ImageSize", "\[Rule]", 
                RowBox[{"{", 
                 RowBox[{"250", ",", "300"}], "}"}]}]}], "]"}]}], ",", 
            "\[IndentingNewLine]", "\[IndentingNewLine]", "_", ",", 
            "\[IndentingNewLine]", 
            RowBox[{"plotAberrationDensity", "[", "parameters", "]"}]}], 
           "]"}]}], "}"}], ",", "Center", ",", " ", "25", ",", 
        "\[IndentingNewLine]", "\[IndentingNewLine]", 
        RowBox[{"ImageSize", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"380", ",", "460"}], "}"}]}]}], "]"}]}]}], "]"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"Style", "[", 
    RowBox[{"\"\< aberration coefficients\>\"", ",", "Bold"}], "]"}], ",", 
   "\[IndentingNewLine]", "\"\<\>\"", ",", "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"tilt", ",", 
       RowBox[{"-", "2"}], ",", "\"\<tilt\>\""}], "}"}], ",", 
     RowBox[{"-", "5"}], ",", "5", ",", "0.25", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}], ",", " ", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"tiltAng", ",", "0", ",", "\"\<tilt angle\>\""}], "}"}], ",", 
     RowBox[{"-", "180"}], ",", "180", ",", "5", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}], ",", " ", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", "\"\<\>\"", ",", "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"focus", ",", 
       RowBox[{"-", "6"}], ",", "\"\<focus\>\""}], "}"}], ",", 
     RowBox[{"-", "6"}], ",", "6", ",", "0.25", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}], ",", " ", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", "\"\<\>\"", ",", "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"coma", ",", "0", ",", "\"\<coma\>\""}], "}"}], ",", 
     RowBox[{"-", "5"}], ",", "5", ",", "0.25", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}], ",", " ", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"comaAng", ",", "0", ",", "\"\<coma angle\>\""}], "}"}], ",", 
     RowBox[{"-", "90"}], ",", "90", ",", "5", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}], ",", " ", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", "\"\<\>\"", ",", "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"ast", ",", "0", ",", "\"\<astigmatism\>\""}], "}"}], ",", 
     RowBox[{"-", "5"}], ",", "5", ",", "0.25", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}], ",", " ", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"astAng", ",", "0", ",", "\"\<ast. angle\>\""}], "}"}], ",", 
     RowBox[{"-", "90"}], ",", "90", ",", "5", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}], ",", " ", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", "\"\<\>\"", ",", "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"sph", ",", "6", ",", "\"\<spherical\>\""}], "}"}], ",", 
     RowBox[{"-", "6"}], ",", "6", ",", "0.25", ",", 
     RowBox[{"Appearance", "\[Rule]", "\"\<Labeled\>\""}], ",", " ", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", "\"\<\>\"", ",", "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"vp1", ",", 
       RowBox[{"{", 
        RowBox[{"0", ",", 
         RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "}"}], ",", " ", 
     RowBox[{"ControlType", "\[Rule]", " ", "None"}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"vp2", ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"0", ",", 
          RowBox[{"-", "2"}], ",", "2"}], "}"}], "-", 
        RowBox[{"{", 
         RowBox[{"0.1", ",", "0", ",", "0"}], "}"}]}]}], "}"}], ",", " ", 
     RowBox[{"ControlType", "\[Rule]", " ", "None"}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"vV", ",", 
       RowBox[{"{", 
        RowBox[{"0", ",", 
         RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "}"}], ",", " ", 
     RowBox[{"ControlType", "\[Rule]", " ", "None"}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"vA", ",", 
       RowBox[{"25", " ", "Degree"}]}], "}"}], ",", " ", 
     RowBox[{"ControlType", "\[Rule]", " ", "None"}]}], "}"}], ",", 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "graphicsPlotType", ",", "densityPlot", ",", "\"\<plot type\>\""}], 
      "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"densityPlot", "\[Rule]", "\"\<density\>\""}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"cylindricalPlot3D", "\[Rule]", "\"\<cylindrical\>\""}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"cylindricalPlot3DStereo", "\[Rule]", "\"\<stereo pairs\>\""}],
        ",", "\[IndentingNewLine]", 
       RowBox[{"threeD", "\[Rule]", "\"\<3D\>\""}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"ShadowPlot3D", "->", "\"\<shadow 3D\>\""}]}], "}"}], ",", 
     RowBox[{"ControlType", "\[Rule]", "PopupMenu"}], ",", " ", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "}"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"ControlPlacement", "\[Rule]", "Left"}]}], "]"}]], "Input",
 CellChangeTimes->{
  3.3905876158125*^9, 3.39058770334375*^9, {3.390587756171875*^9, 
   3.390587762546875*^9}, {3.39058780803125*^9, 3.39058784771875*^9}, 
   3.390587881609375*^9, {3.390587986359375*^9, 3.3905880303125*^9}, {
   3.390588110453125*^9, 3.390588127453125*^9}, {3.39058831878125*^9, 
   3.390588362703125*^9}, {3.39058839390625*^9, 3.390588404921875*^9}, {
   3.390588660640625*^9, 3.39058871478125*^9}, {3.390588783703125*^9, 
   3.390588827953125*^9}, {3.390588860875*^9, 3.390588913875*^9}, {
   3.39058896928125*^9, 3.390588984953125*^9}, {3.4020999034375*^9, 
   3.40209994496875*^9}, {3.403440928765625*^9, 3.403440951421875*^9}, {
   3.403440992359375*^9, 3.403440993484375*^9}, {3.4034410365625*^9, 
   3.40344103690625*^9}, {3.403441072890625*^9, 3.4034411520625*^9}, {
   3.403444379578125*^9, 3.403444458375*^9}, {3.403444501625*^9, 
   3.4034445181875*^9}, {3.5118124154900384`*^9, 3.5118124179236426`*^9}, {
   3.511812539900257*^9, 3.511812543254263*^9}, {3.511812683444509*^9, 
   3.511812837229579*^9}, {3.5118128697868366`*^9, 3.5118128863384657`*^9}, 
   3.5118129884562445`*^9, 3.5118134336810265`*^9, {3.5118141149014235`*^9, 
   3.5118141244798403`*^9}, {3.5118146959400434`*^9, 3.511814696595245*^9}, {
   3.5118148769471617`*^9, 3.5118148913271914`*^9}, {3.51181508439793*^9, 
   3.511815086004733*^9}, {3.511826149926444*^9, 3.511826157339326*^9}, {
   3.5118262421142673`*^9, 3.5118262480970707`*^9}, {3.511826298915742*^9, 
   3.511826304809677*^9}, {3.5118263463392353`*^9, 3.511826363554304*^9}, {
   3.511826438771937*^9, 3.5118264437470217`*^9}, {3.5118268167367573`*^9, 
   3.511826832538281*^9}, {3.511826898066306*^9, 3.5118268985655737`*^9}, {
   3.511827052422698*^9, 3.511827061584132*^9}, 3.5118736401640377`*^9, {
   3.5118763171341515`*^9, 3.5118763404384847`*^9}, {3.5118766214815598`*^9, 
   3.5118766224066124`*^9}, {3.51187730666775*^9, 3.5118773084458513`*^9}, {
   3.5118775792263393`*^9, 3.511877607825975*^9}, {3.511877645529132*^9, 
   3.5118777011933155`*^9}, {3.511878228171457*^9, 3.511878242411271*^9}, {
   3.5118987414717503`*^9, 3.5118987518213425`*^9}, {3.5118998962968025`*^9, 
   3.511899912404724*^9}, {3.5119005098218946`*^9, 3.5119005997010355`*^9}, {
   3.511900739881053*^9, 3.5119007518597383`*^9}, 3.511961922091876*^9, {
   3.511962431912904*^9, 3.5119624618965063`*^9}, {3.511963410669283*^9, 
   3.5119634509626417`*^9}, {3.511964467398172*^9, 3.511964474094604*^9}, {
   3.511973162091617*^9, 3.5119731781968317`*^9}, 3.511973230166847*^9, {
   3.5119735315247993`*^9, 3.5119735705343447`*^9}, {3.5119740034521914`*^9, 
   3.511974008902136*^9}, {3.5119741667484417`*^9, 3.5119742015761023`*^9}, {
   3.5119742602575693`*^9, 3.51197431550483*^9}, {3.5119744649071703`*^9, 
   3.511974502782177*^9}, {3.511974591670247*^9, 3.5119746387847567`*^9}, {
   3.511975115658308*^9, 3.511975137682371*^9}, {3.5119753895988693`*^9, 
   3.511975411748109*^9}, {3.511975523776287*^9, 3.511975530135042*^9}, {
   3.511975765924243*^9, 3.511975790315139*^9}, {3.511976519477559*^9, 
   3.51197655030916*^9}, {3.511976607292714*^9, 3.511976658683736*^9}, {
   3.511977041696575*^9, 3.5119771692190657`*^9}, {3.5119772795646973`*^9, 
   3.511977330396433*^9}, {3.511977705250029*^9, 3.511977786498131*^9}, {
   3.5119778673570557`*^9, 3.511977868269005*^9}, 3.511977898379346*^9, 
   3.511977971415811*^9, {3.51197804051939*^9, 3.511978047008102*^9}, {
   3.5119782594334993`*^9, 3.511978267323975*^9}, {3.511978492568726*^9, 
   3.511978503769129*^9}, {3.511978577839986*^9, 3.511978584118198*^9}, {
   3.5119786310857477`*^9, 3.511978676521161*^9}, {3.511979234430902*^9, 
   3.5119792633045883`*^9}, {3.511979346587953*^9, 3.511979452568962*^9}, 
   3.511980191737597*^9, {3.511980607844881*^9, 3.511980616985065*^9}, {
   3.511980649053691*^9, 3.5119806955438023`*^9}, 3.511980737164892*^9, {
   3.511981301852495*^9, 3.511981343491905*^9}, 3.511981378643714*^9, {
   3.511981465772633*^9, 3.511981486715638*^9}, {3.511981564079302*^9, 
   3.511981571306398*^9}, {3.5119816989928217`*^9, 3.511981739120142*^9}, {
   3.511981880794716*^9, 3.511981892135625*^9}, {3.5119831198007298`*^9, 
   3.511983120623684*^9}, {3.5119835116796837`*^9, 3.511983518662097*^9}, {
   3.5119836496133347`*^9, 3.511983655899197*^9}, {3.511984871711842*^9, 
   3.511984874734541*^9}, {3.5119854290285873`*^9, 3.5119854477410507`*^9}, {
   3.514585749310878*^9, 3.5145857521256857`*^9}, {3.514585926772687*^9, 
   3.514585956329234*^9}, {3.5145860367084427`*^9, 3.514586059460896*^9}, {
   3.5145864250781593`*^9, 3.5145864463302813`*^9}, {3.51465120449712*^9, 
   3.514651209392147*^9}, {3.514651306069824*^9, 3.5146513260101023`*^9}, {
   3.514651672043523*^9, 3.51465175584608*^9}, {3.514652547962867*^9, 
   3.5146525565313597`*^9}, {3.514652778749041*^9, 3.5146528422364283`*^9}, {
   3.5146535513510227`*^9, 3.51465355414493*^9}, {3.514653619603953*^9, 
   3.514653624384005*^9}, {3.5146537470017014`*^9, 3.5146537476968946`*^9}, {
   3.51465381100346*^9, 3.514653813097337*^9}, {3.5146548671695843`*^9, 
   3.514654875234096*^9}, {3.514655091752915*^9, 3.514655106728039*^9}, {
   3.514655191678343*^9, 3.51465519241809*^9}, {3.5146553168886147`*^9, 
   3.514655325319631*^9}, {3.514655477566409*^9, 3.5146554831655407`*^9}, {
   3.51465569313062*^9, 3.5146557215653133`*^9}, {3.514655764141292*^9, 
   3.514655764565465*^9}, {3.5146558175820303`*^9, 3.514655818037374*^9}, {
   3.51465585598596*^9, 3.514655856369047*^9}, {3.5146559077010927`*^9, 
   3.514655909285705*^9}, {3.514655979094637*^9, 3.51465598058534*^9}, {
   3.514656025786921*^9, 3.514656026546921*^9}, {3.514656398396264*^9, 
   3.514656428127234*^9}, {3.514989419392631*^9, 3.514989420608651*^9}, {
   3.514990050866725*^9, 3.51499005163776*^9}, {3.514990335733864*^9, 
   3.51499034791675*^9}, 3.5149904122314253`*^9, {3.514992694532916*^9, 
   3.514992702689695*^9}, 3.514992755067176*^9, {3.514994497615985*^9, 
   3.51499449848298*^9}, {3.514994960197442*^9, 3.514994961384247*^9}, 
   3.5149950211790037`*^9, {3.51499581862253*^9, 3.5149958466570883`*^9}, {
   3.5149958968485727`*^9, 3.514995900945168*^9}, {3.514995965215721*^9, 
   3.514995966064403*^9}, 3.5149960081095*^9, {3.514996072836761*^9, 
   3.514996185421815*^9}, {3.51499629076621*^9, 3.514996291747778*^9}, {
   3.514996531925571*^9, 3.514996532902109*^9}, {3.514996771628921*^9, 
   3.5149967733335247`*^9}, {3.515000846265233*^9, 3.515000849276724*^9}, {
   3.515002670703863*^9, 3.5150027051369047`*^9}, 3.515003065236936*^9, {
   3.5150052197117987`*^9, 3.515005222413865*^9}, 3.515005462430373*^9, {
   3.5150055558658047`*^9, 3.5150055569258633`*^9}, {3.5150080801777267`*^9, 
   3.515008104154675*^9}, {3.5150085858688374`*^9, 3.5150086309834747`*^9}, {
   3.5150087321789*^9, 3.51500879455497*^9}, {3.515008839169441*^9, 
   3.515008840371735*^9}, {3.51500889623521*^9, 3.515008897299636*^9}, {
   3.531146906256847*^9, 3.53114692619239*^9}, 3.531146956555347*^9, 
   3.5311469922550163`*^9, {3.5311471116493998`*^9, 3.531147167649682*^9}, {
   3.5311472281278257`*^9, 3.5311472816647778`*^9}, {3.53114731632235*^9, 
   3.53114731792062*^9}, {3.531147366555488*^9, 3.5311473678242807`*^9}, {
   3.531147705397538*^9, 3.531147739360827*^9}, {3.531147780839719*^9, 
   3.5311477820672207`*^9}, {3.531147898062387*^9, 3.531147899291259*^9}, {
   3.531147940555406*^9, 3.531147941606741*^9}, {3.531147972589222*^9, 
   3.531148023574361*^9}, {3.531148112963108*^9, 3.531148392521102*^9}, {
   3.531148462575121*^9, 3.53114846320686*^9}, {3.531148637815125*^9, 
   3.5311486383528423`*^9}, {3.5311486965346327`*^9, 3.531148696839306*^9}, {
   3.5311490802743464`*^9, 3.5311490814168663`*^9}, {3.5311491229426327`*^9, 
   3.531149155276681*^9}, {3.531149365826747*^9, 3.531149449527505*^9}, {
   3.5311494878377457`*^9, 3.5311495187438593`*^9}, {3.531149581571784*^9, 
   3.531149611764597*^9}, {3.531149691980982*^9, 3.531149692329013*^9}, {
   3.531149863145171*^9, 3.53114986771935*^9}, {3.53114991043314*^9, 
   3.5311499111797647`*^9}, {3.531149954595813*^9, 3.531149955676169*^9}, {
   3.53115002989478*^9, 3.531150030591325*^9}, {3.5311501499291477`*^9, 
   3.531150173706122*^9}, {3.53117538513275*^9, 3.531175388727845*^9}, {
   3.531179726620748*^9, 3.531179734795568*^9}, {3.531179895195475*^9, 
   3.531179901107325*^9}, {3.531181868733502*^9, 3.531181869912702*^9}, {
   3.531181907067224*^9, 3.5311819366575203`*^9}, {3.5311820089227047`*^9, 
   3.531182033178711*^9}, 3.531252339519133*^9, {3.5312524003854322`*^9, 
   3.5312524014619513`*^9}, {3.53125334618106*^9, 3.531253354859063*^9}, 
   3.531253424986681*^9, 3.5312534603964033`*^9, 3.531256565509344*^9, 
   3.531256609734684*^9, {3.531264681255824*^9, 3.5312646862714853`*^9}, {
   3.533212681177844*^9, 3.533212707149508*^9}, {3.5332128439883223`*^9, 
   3.533212860237361*^9}, {3.533215839850655*^9, 3.5332160439794416`*^9}, {
   3.533216121090537*^9, 3.533216134401511*^9}, {3.533216275517028*^9, 
   3.533216302190435*^9}, {3.53321649707382*^9, 3.533216503103141*^9}, {
   3.5332165594388227`*^9, 3.533216667763556*^9}, {3.5332167967255077`*^9, 
   3.5332168074817877`*^9}, {3.533216893727111*^9, 3.5332169262899857`*^9}, {
   3.533217221294548*^9, 3.533217244358881*^9}, 3.5332173208254538`*^9, {
   3.533217922474595*^9, 3.533217925835581*^9}, 3.533217962536433*^9, {
   3.533218261273135*^9, 3.53321835737495*^9}, {3.533247538026782*^9, 
   3.5332475555672073`*^9}, 3.5332477994706297`*^9, {3.533248999316382*^9, 
   3.5332492268651037`*^9}, {3.533249442672893*^9, 3.5332494993479967`*^9}, {
   3.533249785690001*^9, 3.533249861299468*^9}, {3.533302589826269*^9, 
   3.5333027422675247`*^9}, {3.533302781530851*^9, 3.533302945895156*^9}, {
   3.5333037806332097`*^9, 3.533303796608224*^9}, {3.5333038289950695`*^9, 
   3.5333038504458947`*^9}, {3.5333038881213436`*^9, 3.533303888339752*^9}, {
   3.533304096477208*^9, 3.5333040990823245`*^9}, {3.533561399962727*^9, 
   3.533561466233533*^9}, {3.5335615220554323`*^9, 3.533561585199335*^9}, {
   3.533561619471387*^9, 3.533561696491055*^9}, 3.5345915099041357`*^9},
 CellID->1666125579]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`ast$$ = 0, $CellContext`astAng$$ = 
    0, $CellContext`coma$$ = 0, $CellContext`comaAng$$ = 
    0, $CellContext`focus$$ = -6, $CellContext`graphicsPlotType$$ = \
$CellContext`densityPlot, $CellContext`sph$$ = 
    6, $CellContext`tilt$$ = -2, $CellContext`tiltAng$$ = 
    0, $CellContext`vA$$ = 25 Degree, $CellContext`vp1$$ = {0, -2, 
    2}, $CellContext`vp2$$ = {-0.1, -2, 2}, $CellContext`vV$$ = {0, -1, 1}, 
    Typeset`show$$ = True, Typeset`bookmarkList$$ = {}, 
    Typeset`bookmarkMode$$ = "Menu", Typeset`animator$$, Typeset`animvar$$ = 
    1, Typeset`name$$ = "\"untitled\"", Typeset`specs$$ = {{
      Hold[
       Style[" aberration coefficients", Bold]], 
      Manipulate`Dump`ThisIsNotAControl}, {
      Hold[""], Manipulate`Dump`ThisIsNotAControl}, {{
       Hold[$CellContext`tilt$$], -2, "tilt"}, -5, 5, 0.25}, {{
       Hold[$CellContext`tiltAng$$], 0, "tilt angle"}, -180, 180, 5}, {
      Hold[""], Manipulate`Dump`ThisIsNotAControl}, {{
       Hold[$CellContext`focus$$], -6, "focus"}, -6, 6, 0.25}, {
      Hold[""], Manipulate`Dump`ThisIsNotAControl}, {{
       Hold[$CellContext`coma$$], 0, "coma"}, -5, 5, 0.25}, {{
       Hold[$CellContext`comaAng$$], 0, "coma angle"}, -90, 90, 5}, {
      Hold[""], Manipulate`Dump`ThisIsNotAControl}, {{
       Hold[$CellContext`ast$$], 0, "astigmatism"}, -5, 5, 0.25}, {{
       Hold[$CellContext`astAng$$], 0, "ast. angle"}, -90, 90, 5}, {
      Hold[""], Manipulate`Dump`ThisIsNotAControl}, {{
       Hold[$CellContext`sph$$], 6, "spherical"}, -6, 6, 0.25}, {
      Hold[""], Manipulate`Dump`ThisIsNotAControl}, {{
       Hold[$CellContext`vp1$$], {0, -2, 2}}}, {{
       Hold[$CellContext`vp2$$], {-0.1, -2, 2}}}, {{
       Hold[$CellContext`vV$$], {0, -1, 1}}}, {{
       Hold[$CellContext`vA$$], 25 Degree}}, {{
       Hold[$CellContext`graphicsPlotType$$], $CellContext`densityPlot, 
       "plot type"}, {$CellContext`densityPlot -> 
       "density", $CellContext`cylindricalPlot3D -> 
       "cylindrical", $CellContext`cylindricalPlot3DStereo -> 
       "stereo pairs", $CellContext`threeD -> "3D", $CellContext`ShadowPlot3D -> 
       "shadow 3D"}}}, Typeset`size$$ = {380., {228., 232.}}, 
    Typeset`update$$ = 0, Typeset`initDone$$, Typeset`skipInitDone$$ = 
    True, $CellContext`tilt$1421$$ = 0, $CellContext`tiltAng$1422$$ = 
    0, $CellContext`focus$1423$$ = 0, $CellContext`coma$1424$$ = 
    0, $CellContext`comaAng$1425$$ = 0, $CellContext`ast$1426$$ = 
    0, $CellContext`astAng$1427$$ = 0, $CellContext`sph$1428$$ = 
    0, $CellContext`graphicsPlotType$1429$$ = False}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, 
      "Variables" :> {$CellContext`ast$$ = 0, $CellContext`astAng$$ = 
        0, $CellContext`coma$$ = 0, $CellContext`comaAng$$ = 
        0, $CellContext`focus$$ = -6, $CellContext`graphicsPlotType$$ = \
$CellContext`densityPlot, $CellContext`sph$$ = 
        6, $CellContext`tilt$$ = -2, $CellContext`tiltAng$$ = 
        0, $CellContext`vA$$ = 
        25 Degree, $CellContext`vp1$$ = {0, -2, 
         2}, $CellContext`vp2$$ = {-0.1, -2, 2}, $CellContext`vV$$ = {0, -1, 
         1}}, "ControllerVariables" :> {
        Hold[$CellContext`tilt$$, $CellContext`tilt$1421$$, 0], 
        Hold[$CellContext`tiltAng$$, $CellContext`tiltAng$1422$$, 0], 
        Hold[$CellContext`focus$$, $CellContext`focus$1423$$, 0], 
        Hold[$CellContext`coma$$, $CellContext`coma$1424$$, 0], 
        Hold[$CellContext`comaAng$$, $CellContext`comaAng$1425$$, 0], 
        Hold[$CellContext`ast$$, $CellContext`ast$1426$$, 0], 
        Hold[$CellContext`astAng$$, $CellContext`astAng$1427$$, 0], 
        Hold[$CellContext`sph$$, $CellContext`sph$1428$$, 0], 
        Hold[$CellContext`graphicsPlotType$$, \
$CellContext`graphicsPlotType$1429$$, False]}, 
      "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, "Body" :> 
      Module[{$CellContext`parameters$, $CellContext`xplot$, \
$CellContext`yplot$, $CellContext`plotProfile$, $CellContext`bottom$, \
$CellContext`gra$, $CellContext`gr2a$, $CellContext`parametersBlank$, \
$CellContext`aberration$, $CellContext`aberrationCylindrical$, \
$CellContext`aberrationX$, $CellContext`aberrationY$, \
$CellContext`profilePlotOptions$}, $CellContext`r2 = $CellContext`x^2 + \
$CellContext`y^2; $CellContext`parameters$ = {$CellContext`tilt$$, \
$CellContext`tiltAng$$, $CellContext`focus$$, $CellContext`coma$$, \
$CellContext`comaAng$$, $CellContext`ast$$, $CellContext`astAng$$, \
$CellContext`sph$$}; $CellContext`parametersBlank$ = {
           Pattern[$CellContext`tilt$$, 
            Blank[]], 
           Pattern[$CellContext`tiltAng$$, 
            Blank[]], 
           Pattern[$CellContext`focus$$, 
            Blank[]], 
           Pattern[$CellContext`coma$$, 
            Blank[]], 
           Pattern[$CellContext`comaAng$$, 
            Blank[]], 
           Pattern[$CellContext`ast$$, 
            Blank[]], 
           Pattern[$CellContext`astAng$$, 
            Blank[]], 
           Pattern[$CellContext`sph$$, 
            
            Blank[]]}; \
$CellContext`aberration$[$CellContext`parametersBlank$] := \
$CellContext`tilt$$ ($CellContext`x 
             Cos[$CellContext`tiltAng$$ Degree] + $CellContext`y 
             Sin[$CellContext`tiltAng$$ 
               Degree]) + $CellContext`coma$$ ($CellContext`x $CellContext`r2 
             Cos[$CellContext`comaAng$$ 
               Degree] + $CellContext`y $CellContext`r2 
             Sin[$CellContext`comaAng$$ 
               Degree]) + $CellContext`focus$$ $CellContext`r2 + \
$CellContext`ast$$ ($CellContext`x 
              Cos[$CellContext`astAng$$ Degree] + $CellContext`y 
              Sin[$CellContext`astAng$$ 
                Degree])^2 + $CellContext`sph$$ $CellContext`r2^2; \
$CellContext`aberrationCylindrical$[$CellContext`parametersBlank$] := \
$CellContext`tilt$$ $CellContext`\[Rho] 
           Cos[$CellContext`\[Theta] - $CellContext`tiltAng$$ 
             Degree] + $CellContext`focus$$ $CellContext`\[Rho]^2 + \
$CellContext`coma$$ $CellContext`\[Rho]^3 
           Cos[$CellContext`\[Theta] - $CellContext`comaAng$$ 
             Degree] + $CellContext`ast$$ $CellContext`\[Rho]^2 
           Cos[$CellContext`\[Theta] - $CellContext`astAng$$ 
              Degree]^2 + $CellContext`sph$$ $CellContext`\[Rho]^4; \
$CellContext`aberrationX$[$CellContext`parametersBlank$] := \
$CellContext`tilt$$ $CellContext`x 
           Cos[$CellContext`tiltAng$$ 
             Degree] + $CellContext`focus$$ $CellContext`x^2 + \
$CellContext`coma$$ $CellContext`x^3 
           Cos[$CellContext`comaAng$$ 
             Degree] + $CellContext`ast$$ $CellContext`x^2 
           Cos[$CellContext`astAng$$ 
              Degree]^2 + $CellContext`sph$$ $CellContext`x^4; \
$CellContext`aberrationY$[$CellContext`parametersBlank$] := \
$CellContext`tilt$$ $CellContext`y 
           Sin[$CellContext`tiltAng$$ 
             Degree] + $CellContext`focus$$ $CellContext`y^2 + \
$CellContext`coma$$ $CellContext`y^3 
           Sin[$CellContext`comaAng$$ 
             Degree] + $CellContext`ast$$ $CellContext`y^2 
           Sin[$CellContext`astAng$$ 
              Degree]^2 + $CellContext`sph$$ $CellContext`y^4; \
$CellContext`profilePlotOptions$ := {
          PlotRange -> All, Frame -> True, PlotLabel -> Row[{
              Style["X", Italic], " profile (red), ", 
              Style["Y", Italic], " profile (green)"}], FrameTicks -> 
           Automatic, GridLines -> Automatic, 
           ImageSize -> {350, 
            248}}; $CellContext`xplot$[$CellContext`parametersBlank$] := 
         Plot[
           $CellContext`aberrationX$[$CellContext`parameters$], \
{$CellContext`x, -1, 1}, PlotStyle -> {{
              RGBColor[1, 0, 0], 
              Thickness[
              0.008]}}]; $CellContext`yplot$[$CellContext`parametersBlank$] := 
         Plot[
           $CellContext`aberrationY$[$CellContext`parameters$], \
{$CellContext`y, -1, 1}, PlotStyle -> {{
              RGBColor[0, 1, 0], 
              Thickness[
              0.008]}}]; \
$CellContext`plotProfile$[$CellContext`parametersBlank$, 
           Pattern[$CellContext`graphicsPlotType$, 
            Blank[]]] := Show[{
            $CellContext`xplot$[$CellContext`parameters$], 
            $CellContext`yplot$[$CellContext`parameters$]}, 
           Evaluate[$CellContext`profilePlotOptions$], FrameLabel -> Style[
             If[
              
              Or[$CellContext`graphicsPlotType$ == $CellContext`densityPlot, \
$CellContext`graphicsPlotType$ == $CellContext`threeD], 
              Row[{"OPD = ", 
                N[
                 $CellContext`aberration$[$CellContext`parameters$], 2]}], 
              Row[{"OPD = ", 
                $CellContext`aberrationCylindrical$[$CellContext`parameters$]}\
], 
              Row[{"OPD = ", 
                $CellContext`aberrationCylindrical$[$CellContext`parameters$]}\
]], FontSize -> 12]]; GraphicsColumn[{
           $CellContext`plotProfile$[$CellContext`parameters$, \
$CellContext`graphicsPlotType$$], 
           Switch[$CellContext`graphicsPlotType$$, $CellContext`densityPlot, 
            DensityPlot[
             
             If[$CellContext`x^2 + $CellContext`y^2 <= 1, 
              0.5 (1 + Cos[
                2 Pi $CellContext`aberration$[$CellContext`parameters$]]), 
              1], {$CellContext`x, -1, 1}, {$CellContext`y, -1, 1}, 
             ColorFunction -> GrayLevel, PlotPoints -> 40, Mesh -> False, 
             Frame -> True], $CellContext`cylindricalPlot3D, 
            RevolutionPlot3D[
             $CellContext`aberrationCylindrical$[$CellContext`parameters$], \
{$CellContext`\[Rho], 0, 1}, {$CellContext`\[Theta], 0, 2 Pi}, 
             BoxRatios -> {1, 1, 0.5}, Axes -> False, Boxed -> False, 
             ColorFunction -> 
             Function[{$CellContext`x, $CellContext`y, $CellContext`z}, 
               Hue[$CellContext`z]]], $CellContext`cylindricalPlot3DStereo, 
            GraphicsRow[
             Map[RevolutionPlot3D[
               $CellContext`aberrationCylindrical$[$CellContext`parameters$], \
{$CellContext`\[Rho], 0, 1}, {$CellContext`\[Theta], 0, 2 Pi}, Axes -> None, 
               Boxed -> False, ColorFunction -> 
               Function[{$CellContext`x, $CellContext`y, $CellContext`z}, 
                 Hue[$CellContext`z]], ImageSize -> 125, 
               BoxRatios -> {1, 1, 0.5}, ViewVertical -> 
               Dynamic[$CellContext`vV$$], ViewAngle -> 
               Dynamic[$CellContext`vA$$], #]& , {
              ViewPoint -> 
               Dynamic[$CellContext`vp1$$, ($CellContext`vp1$$ = #; \
$CellContext`vp2$$ = # - {0.1, 0, 0})& ], ViewPoint -> 
               Dynamic[$CellContext`vp2$$, ($CellContext`vp2$$ = #; \
$CellContext`vp1$$ = # + {0.1, 0, 0})& ]}], -20], $CellContext`threeD, 
            Plot3D[
             If[$CellContext`x^2 + $CellContext`y^2 <= 1, 
              $CellContext`aberration$[$CellContext`parameters$], 
              1], {$CellContext`x, -1, 1}, {$CellContext`y, -1, 1}, 
             PlotPoints -> 30, Axes -> False, BoxRatios -> {1, 1, 0.5}, Boxed -> 
             True, ColorFunction -> 
             Function[{$CellContext`x, $CellContext`y, $CellContext`z}, 
               
               Hue[$CellContext`z]]], $CellContext`ShadowPlot3D, \
$CellContext`gra$ = Plot3D[
               $CellContext`aberration$[$CellContext`parameters$], \
{$CellContext`x, -1, 
                1}, {$CellContext`y, -(1 - $CellContext`x^2)^Rational[1, 2], (
                 1 - $CellContext`x^2)^Rational[1, 2]}, PlotPoints -> 20, 
               Axes -> False, ColorFunction -> 
               Function[{$CellContext`x, $CellContext`y, $CellContext`z}, 
                 Hue[$CellContext`z]]]; $CellContext`bottom$ = 3 Min[
                Table[
                 $CellContext`aberration$[$CellContext`parameters$], \
{$CellContext`x, -1, 1, 
                  0.1}, {$CellContext`y, -(1 - $CellContext`x^2)^
                   Rational[1, 2], (1 - $CellContext`x^2)^
                  Rational[1, 2]}]]; $CellContext`gr2a$ = 
             ReplacePart[$CellContext`gra$, 
               ReplaceAll[{1, 1} -> Part[$CellContext`gra$, 1, 1], {
                  Pattern[$CellContext`x$, 
                   Blank[]], 
                  Pattern[$CellContext`y$, 
                   Blank[]], 
                  Pattern[$CellContext`z$, 
                   
                   Blank[]]} -> {$CellContext`x$, $CellContext`y$, \
$CellContext`bottom$}]]; 
            Show[$CellContext`gra$, $CellContext`gr2a$, PlotRange -> All, 
              BoxRatios -> {1, 1, 0.9}, ImageSize -> {250, 300}], 
            Blank[], 
            $CellContext`plotAberrationDensity[$CellContext`parameters$]]}, 
          Center, 25, ImageSize -> {380, 460}]], "Specifications" :> {
        Style[" aberration coefficients", Bold], 
        "", {{$CellContext`tilt$$, -2, "tilt"}, -5, 5, 0.25, Appearance -> 
         "Labeled", ImageSize -> 
         Tiny}, {{$CellContext`tiltAng$$, 0, "tilt angle"}, -180, 180, 5, 
         Appearance -> "Labeled", ImageSize -> Tiny}, 
        "", {{$CellContext`focus$$, -6, "focus"}, -6, 6, 0.25, Appearance -> 
         "Labeled", ImageSize -> Tiny}, 
        "", {{$CellContext`coma$$, 0, "coma"}, -5, 5, 0.25, Appearance -> 
         "Labeled", ImageSize -> 
         Tiny}, {{$CellContext`comaAng$$, 0, "coma angle"}, -90, 90, 5, 
         Appearance -> "Labeled", ImageSize -> Tiny}, 
        "", {{$CellContext`ast$$, 0, "astigmatism"}, -5, 5, 0.25, Appearance -> 
         "Labeled", ImageSize -> 
         Tiny}, {{$CellContext`astAng$$, 0, "ast. angle"}, -90, 90, 5, 
         Appearance -> "Labeled", ImageSize -> Tiny}, 
        "", {{$CellContext`sph$$, 6, "spherical"}, -6, 6, 0.25, Appearance -> 
         "Labeled", ImageSize -> Tiny}, 
        "", {{$CellContext`vp1$$, {0, -2, 2}}, ControlType -> 
         None}, {{$CellContext`vp2$$, {-0.1, -2, 2}}, ControlType -> 
         None}, {{$CellContext`vV$$, {0, -1, 1}}, ControlType -> 
         None}, {{$CellContext`vA$$, 25 Degree}, ControlType -> 
         None}, {{$CellContext`graphicsPlotType$$, $CellContext`densityPlot, 
          "plot type"}, {$CellContext`densityPlot -> 
          "density", $CellContext`cylindricalPlot3D -> 
          "cylindrical", $CellContext`cylindricalPlot3DStereo -> 
          "stereo pairs", $CellContext`threeD -> 
          "3D", $CellContext`ShadowPlot3D -> "shadow 3D"}, ControlType -> 
         PopupMenu, ImageSize -> Tiny}}, 
      "Options" :> {ControlPlacement -> Left}, 
      "DefaultOptions" :> {ControllerLinking -> True}],
     ImageSizeCache->{605., {256., 261.}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 CellID->63181872],

Cell[CellGroupData[{

Cell["CAPTION", "Section",
 CellFrame->{{0, 0}, {1, 0}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->12,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell["\<\
This Demonstration shows profile and wavefront maps for tilt, focus, and \
fourth-order wavefront aberrations coma, astigmatism, and spherical \
aberration. You can choose the wavefront map to be a density plot or one of \
several 3D plots, including stereo pairs of the cylindrical plot. The OPD \
(optical path difference) is the wavefront aberration calculated from the \
aberration coefficients.\
\>", "Text"]
}, Close]]
}, Open  ]],

Cell[CellGroupData[{

Cell["DETAILS", "Section",
 CellFrame->{{0, 0}, {1, 0}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->12,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[TextData[{
 "The study of wavefront aberrations is very important in the design, \
fabrication, and testing of all optical systems. This Demonstration helps in \
the study of the Seidel monochromatic aberrations, named after L. Seidel, who \
in 1856 gave explicit formulas for calculating them. The Seidel aberrations \
are of the fourth degree in the pupil variables ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 " and the field position when expressed as wavefront aberrations, and are of \
the third degree when expressed as transverse ray aberrations. Because of \
this, these aberrations are either known as third-order or fourth-order \
aberrations, depending upon whether transverse ray or wavefront aberrations \
are being considered. The five Seidel aberrations are coma, astigmatism, \
spherical aberration, distortion, and field curvature. At a given field \
position, distortion and field curvature are simply tilt and focus \
(curvature), respectively. The density plot shows what an interferogram \
testing an optical system with the fourth-order wavefront aberrations present \
would look like. The various 3D plots give a representation of the wavefront \
shape. A three-dimensional image of the wavefront aberration can be seen by \
looking at the stereo pairs of the 3D plots."
}], "Text"],

Cell["References", "Text"],

Cell[TextData[{
 "[1] W. J. Smith, ",
 StyleBox["Modern Optical Engineering",
  FontSlant->"Italic"],
 ", 4th ed., New York: McGraw\[Dash]Hill, 2008."
}], "Text"],

Cell[TextData[{
 "[2] W. T. Welford, ",
 StyleBox["Aberrations of Optical Systems",
  FontSlant->"Italic"],
 ", Bristol: Adam Hilger, 1986."
}], "Text"],

Cell[TextData[{
 "[3] V. N. Mahajan, ",
 StyleBox["Optical Imaging and Aberrations, Part I: Ray Geometrical Optics",
  FontSlant->"Italic"],
 ", Bellingham, WA: SPIE Press, 1998."
}], "Text"],

Cell[TextData[{
 "[4] J. C. Wyant and K. Creath, \"Basice Wavefront Aberration Theory for \
Optical Metrology,\" ",
 StyleBox["Applied Optics and Optical Engineering",
  FontSlant->"Italic"],
 ", Vol. XI (R. R. Shannon and J. C. Wyant, eds.), New York: Academic Press, \
1992,  pp. 11\[Dash]53."
}], "Text"],

Cell[TextData[{
 "[5] J. C. Wyant. \"web",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " and LiveGraphics3D.\" (Dec 22, 2011) ",
 ButtonBox["wyant.optics.arizona.edu/math.htm",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://wyant.optics.arizona.edu/math.htm"], None},
  ButtonNote->"http://wyant.optics.arizona.edu/math.htm"],
 "."
}], "Text"]
}, Close]],

Cell["THIS NOTEBOOK IS THE SOURCE CODE FROM", "Text",
 CellFrame->{{0, 0}, {0, 0}},
 CellMargins->{{48, 10}, {4, 28}},
 CellGroupingRules->{"SectionGrouping", 25},
 CellFrameMargins->{{48, 48}, {6, 5}},
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 FontFamily->"Helvetica",
 FontSize->10,
 FontWeight->"Bold",
 FontColor->RGBColor[0.597406, 0, 0.0527047]],

Cell[TextData[{
 "\[NonBreakingSpace]",
 ButtonBox["James C. Wyant",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/author.html?author=James+C.+Wyant"]\
, None},
  ButtonNote->
   "http://demonstrations.wolfram.com/author.html?author=James+C.+Wyant"]
}], "Text",
 CellMargins->{{48, Inherited}, {0, Inherited}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.6]],

Cell[TextData[{
 "\"",
 ButtonBox["Wavefront Maps and Profiles of Seidel Aberrations",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/\
WavefrontMapsAndProfilesOfSeidelAberrations/"], None},
  ButtonNote->
   "http://demonstrations.wolfram.com/\
WavefrontMapsAndProfilesOfSeidelAberrations/"],
 "\"",
 "\[ParagraphSeparator]\[NonBreakingSpace]",
 ButtonBox["http://demonstrations.wolfram.com/\
WavefrontMapsAndProfilesOfSeidelAberrations/",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/\
WavefrontMapsAndProfilesOfSeidelAberrations/"], None},
  ButtonNote->
   "http://demonstrations.wolfram.com/\
WavefrontMapsAndProfilesOfSeidelAberrations/"],
 "\[ParagraphSeparator]\[NonBreakingSpace]",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
 "\[ParagraphSeparator]\[NonBreakingSpace]",
 "Published: ",
 "January 4, 2012"
}], "Text",
 CellMargins->{{48, Inherited}, {0, Inherited}},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.5]],

Cell[CellGroupData[{

Cell[TextData[{
 "A full-function Wolfram ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " system is required to edit this notebook.\n",
 StyleBox[ButtonBox["GET WOLFRAM MATHEMATICA \[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolfram.com/products/mathematica/"], None},
  ButtonNote->"http://www.wolfram.com/products/mathematica/"],
  FontFamily->"Helvetica",
  FontWeight->"Bold",
  FontSlant->"Italic",
  FontColor->RGBColor[1, 0.42, 0]]
}], "Text",
 CellFrame->True,
 CellMargins->{{48, 68}, {8, 28}},
 CellFrameMargins->12,
 CellFrameColor->RGBColor[0.87, 0.87, 0.87],
 CellChangeTimes->{3.3750111182355957`*^9},
 ParagraphSpacing->{1., 1.},
 FontFamily->"Verdana",
 FontSize->10,
 FontColor->GrayLevel[0.411765],
 Background->RGBColor[1, 1, 1]],

Cell[TextData[{
 "\[Copyright] ",
 StyleBox[ButtonBox["Wolfram Demonstrations Project & Contributors",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None},
  ButtonNote->"http://demonstrations.wolfram.com/"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Terms of Use",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/termsofuse.html"], None},
  ButtonNote->"http://demonstrations.wolfram.com/termsofuse.html"],
  FontColor->GrayLevel[0.6]],
 "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
 StyleBox[ButtonBox["Make a new version of this Demonstration \
\[RightGuillemet]",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/participate/upload.jsp?id=\
WavefrontMapsAndProfilesOfSeidelAberrations"], None},
  ButtonNote->None],
  FontColor->GrayLevel[0.6]]
}], "Text",
 CellFrame->{{0, 0}, {0, 0.5}},
 CellMargins->{{48, 10}, {20, 50}},
 CellFrameMargins->{{6, 0}, {6, 6}},
 CellFrameColor->GrayLevel[0.6],
 FontFamily->"Verdana",
 FontSize->9,
 FontColor->GrayLevel[0.6]]
}, Open  ]]
},
Editable->True,
Saveable->False,
ScreenStyleEnvironment->"Working",
CellInsertionPointCell->None,
WindowSize->{780, 650},
WindowMargins->{{Inherited, Inherited}, {Inherited, 0}},
WindowElements->{
 "StatusArea", "MemoryMonitor", "MagnificationPopUp", "VerticalScrollBar", 
  "MenuBar"},
WindowTitle->"Wavefront Maps and Profiles of Seidel Aberrations - Source",
DockedCells->{},
CellContext->Notebook,
FrontEndVersion->"8.0 for Microsoft Windows (32-bit) (February 23, 2011)",
StyleDefinitions->"Default.nb"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[651, 23, 169, 3, 70, "Section"],
Cell[823, 28, 3700, 64, 70, "Section"],
Cell[4526, 94, 34477, 708, 70, "Input",
 CellID->1666125579]
}, Open  ]],
Cell[CellGroupData[{
Cell[39040, 807, 15254, 288, 70, "Output",
 CellID->63181872],
Cell[CellGroupData[{
Cell[54319, 1099, 209, 6, 70, "Section"],
Cell[54531, 1107, 423, 7, 70, "Text"]
}, Close]]
}, Open  ]],
Cell[CellGroupData[{
Cell[55002, 1120, 209, 6, 70, "Section"],
Cell[55214, 1128, 1408, 23, 70, "Text"],
Cell[56625, 1153, 26, 0, 70, "Text"],
Cell[56654, 1155, 162, 5, 70, "Text"],
Cell[56819, 1162, 152, 5, 70, "Text"],
Cell[56974, 1169, 191, 5, 70, "Text"],
Cell[57168, 1176, 307, 7, 70, "Text"],
Cell[57478, 1185, 360, 11, 70, "Text"]
}, Close]],
Cell[57852, 1199, 355, 9, 70, "Text",
 CellGroupingRules->{"SectionGrouping", 25}],
Cell[58210, 1210, 416, 13, 70, "Text"],
Cell[58629, 1225, 1170, 34, 70, "Text"],
Cell[CellGroupData[{
Cell[59824, 1263, 793, 24, 70, "Text"],
Cell[60620, 1289, 1215, 33, 70, "Text"]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
(* NotebookSignature TQNSTYD8z#ADGB05qSHOX5k8 *)
